// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision:
//  $Date:
//  $Author:

//=============================================================================
//
//  add -surface mode to model proteins with surfaces eg mica, hydroxyapatite
//
//		Kosta Makrodimitris,Ph.D. (KMa) 2006-02
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// RosettaSurface inherit properties and functions from Ligand-Dock modes
// (translation, rotation, monte-carlo, minimize etc)
// It has additional geometric features because of their symmetry
// and performance because of the large number of atoms.

// Surfaces have differences comparing to a ligand or a protein
// and for this reason they should be different objects
// in the future releases of Rosetta.
// 1) Surfaces have a defined symmetry
//    ( CUBIC, ORTHOGONAL, HEXAGONAL etc) and space group.
//    In the most cases their dimensions are defined through the
//    concept of a unit cell // in X,Y,Z directions.
//    This symmetry needs different simulation methodology.
// 2) When the protein contact area that we model is around 100nm2,
//    we need an adequate solid-surface area to be around 1000nm2
//    in the binding plane.
//    So the number of atoms for a surface (2000-10000)
//    is larger than a ligand (100-200)
// 3) The solid-surface is usually modeled as rigid whereas
//    the ligand in the most cases has flexibility,
//    which needs different simulation stategies
// 4) The applications for protein-surface systems
//	  (biosensors, bones, teeth, bioelectronics, semiconductors)
//    are different than the ligand-protein ones.

// CASES
// -Hydroxyapatite & statherin
// -MICA & Top7
// -QUARTZ & Lysozyme
// -GOLD, SILVER & peptides
// -C4, C18 (chromatography) & peptides
// -Self assembled peptides & large proteins
// -Phospholipid Bilayers and membrane proteins

// REFERENCES
// 1)  "Modeling statherin structure binding to
//     hydroxyapatite [001] crystal surface"
//		Kosta Makrodimitris, Jeffrey J. Gray
//      Molecular Mechanics and Simulation (COMPUTERS IN CHEMISTRY)
//	    232nd ACS National Meeting, San Francisco, CA, September 10-14, 2006
//      http://oasys2.confex.com/acs/232nm/techprogram/P986054.HTM
// 2) "Modeling protein structure on solid surfaces: the human
//     salivary statherin on hydroxyapatite"
//     Kosta Makrodimitris, David Masica & Jeffrey J. Gray
//     (2006)in preparation for JACS

// Rosetta Headers
#include "SurfaceMode.h"
#include "surface.h"
#include "param.h"
#include "misc.h"
#include "docking.h"
#include "docking_ns.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "minimize_ns.h"
#include "runlevel.h"
#include "refold.h"

// Numeric Headers
#include <numeric/xyzVector.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>

// C++ Headers
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <iomanip>
#include <math.h>
//#include <cstdio>


namespace surface
{
	 SurfaceMode mode;




//////////////////////////////////////////////////////////////////////////////
/// @begin  protein_surfacemove_symmetry
///
/// @brief
/// move surface to the projected center of geometry of the protein by symmetry
/// @detailed
///
/// @param[in]
/// @param[out] xyz-coordinates of the solid surface recentered
///
/// @global_read
///  surface3pointA0, surface3pointB0, surface3pointC0, vAB0, vAC0
/// @global_write
///  centerG_surface
/// @remarks
///  needs to read 3 atoms from HETATM pdb (defined in the input PDB file)
/// @references
/// Modeling statherin structure binding to hydroxyapatite [001] crystal surface
/// Kosta Makrodimitris, Jeffrey J. Gray
/// 232nd ACS National Meeting, San Francisco, CA, September 10-14, 2006
/// http://oasys2.confex.com/acs/232nm/techprogram/P986054.HTM
///
/// @authors:Kosta Makrodimitris,Ph.D. 2006-02
///
/// @last_modified 2006-03-03
///////////////////////////////////////////////////////////////////////////////


numeric::xyzVector<double>
SurfaceMode::protein_surfacemove_symmetry()
{

  Surface a_surface;
  //--------------------

numeric::xyzVector<double> CA_centerG_protein,
							vAB, vAC, fvAB, fvAC,
							Nproteintosurface, FinalPointPlane1,
							Psurface_Nprotein, NEW_centerG_protein,
							IntersectPoint1, IntersectPoint2,
							findProteinZ, Translate,
							iNproteintosurface,	 surface3pointA,
							surface3pointB, surface3pointC;

double tDistanceB=0.0, tDistanceC=0.0, bb=0.0, cc=0.0,
		ibb=0.0, icc=0.0 ;
std::vector<double>  plane0 ;
//  VARIABLES to print if runlevel_ns::runlevel >= runlevel_ns::inform
numeric::xyzVector<double> P_ABpAC , v_ABpAC, XYZ ;
double cosBAC, thetaBAC,d_ABpAC,d_Psurfprot_cG,
		 xpos,ypos,zpos, floor_ceil_icc,  floor_ceil_ibb;
//const long double pppi = 3.14159265358979323846264338327950288 ;
//?????????????????????????????????????????????????????????????????????????

chain_centroid_CA
(misc::position,1,misc::total_residue,docking::part_centroid(1,1));

// center of geometry of protein	P
CA_centerG_protein.x() = docking::part_centroid(1,1);
CA_centerG_protein.y() = docking::part_centroid(2,1);
CA_centerG_protein.z() = docking::part_centroid(3,1);

ligand_center(docking::part_centroid(1,2),(*ligand::ligand_one));

// center of geometry of surface	S
centerG_surface.x() = docking::part_centroid(1,2);
centerG_surface.y() = docking::part_centroid(2,2);
centerG_surface.z() = docking::part_centroid(3,2);

surface3pointA0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [0] ]->get_coordinates();
surface3pointB0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [1] ]->get_coordinates();
surface3pointC0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [2] ]->get_coordinates();

//vectors of the 2 basis vectors of the surface crystal system
//that are the unit cell dimensions
vAB0 = surface3pointB0 - surface3pointA0 ;
vAC0 = surface3pointC0 - surface3pointA0 ;

//FIX the ABC in THE center of surface
surface3pointA = centerG_surface ;
surface3pointB = centerG_surface + vAB0 ;
surface3pointC = centerG_surface + vAC0 ;

vAB = surface3pointB - surface3pointA ;
vAC = surface3pointC - surface3pointA ;

// find the normal vector to the surface of the vectors AB, AC
Nproteintosurface =
a_surface.normalto3points ( surface3pointA, surface3pointB, surface3pointC );

// vector antiparallel
iNproteintosurface = -1.0 * Nproteintosurface ;

// Find the equation of the plane
plane0 =
a_surface.plane3points ( surface3pointA, surface3pointB, surface3pointC  );

// find the intersection point of the antiparallel normal vector
// and the surface plane
Psurface_Nprotein =
a_surface.point_intersect_plane(plane0,CA_centerG_protein,iNproteintosurface);

// find the vector difference of the normal intersection
// point and the center of geometry of protein
findProteinZ = Psurface_Nprotein - CA_centerG_protein ;

//FIND the 2 first intersection points
// *.........
//	.		 .
//	.		 .
//	.........*

IntersectPoint1 =
a_surface.point2lines3 (  Psurface_Nprotein, centerG_surface,  vAC, vAB );

IntersectPoint2 =
a_surface.point2lines3 (  Psurface_Nprotein, centerG_surface, vAB, vAC );

//FIND the distances form the center of surface paralle AB,AC
tDistanceC = distance( IntersectPoint1, Psurface_Nprotein ) ;
tDistanceB = distance( IntersectPoint2, Psurface_Nprotein ) ;

//BEGIN the criteria to move or not the surface to center the protein
//kma
if (tDistanceB <= ABdistance/2.0 && tDistanceC <= ACdistance/2.0) {

NEW_centerG_protein = CA_centerG_protein ;

Translate.x()=0;
Translate.y()=0;
Translate.z()=0;

//write-screen information finally  KMa
//************************************************************************
if ( runlevel_ns::runlevel >= runlevel_ns::inform ){

v_ABpAC = vAB + vAC ;
P_ABpAC = surface3pointA + v_ABpAC ;
d_ABpAC = distance ( P_ABpAC,surface3pointA ) ;
d_Psurfprot_cG = distance ( Psurface_Nprotein, centerG_surface ) ;

	bb=0;
	cc=0;
	icc=0;
	ibb=0;

std::cout << " $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$="
<<  std::endl;
std::cout << " FINALE tDistanceB = " << tDistanceB << "  tDistanceC="
	<<	tDistanceC <<  std::endl;
std::cout << " AB = " << distance(surface3pointB0 ,surface3pointA0 ) << " AC="
	<<	distance(surface3pointC0 ,surface3pointA0 ) <<  std::endl;
std::cout << "dISTANCE:Pnormaltosurface-protein_cG or tDistanceC + tDistanceB="
<< d_Psurfprot_cG 	<< " dISTANCE AB + AC= " <<	 d_ABpAC << std::endl;
std::cout << "distance of the protein zzzzzs ="
	<< findProteinZ.length()<< std::endl;
std::cout << "distance of the protein INITIALLY vertical="
	<< distance(CA_centerG_protein,Psurface_Nprotein) << std::endl;
std::cout << "meta inside centerG_surface.x() x= " << centerG_surface.x()
<< " y= " << centerG_surface.y() << " z=" << centerG_surface.z() << std::endl;
std::cout <<"CA_centerG_protein.x() x=" << CA_centerG_protein.x() << " y=" <<
	   	CA_centerG_protein.y()<< " z=" << CA_centerG_protein.z() << std::endl;
std::cout << "PLANE A= " << plane0[0] << " B= " <<	 plane0[1] << " C="
<< plane0[2] <<  " d= "<< plane0[3]<<std::endl;
std::cout << "verify P PLANE= "
<< plane0[0] *Psurface_Nprotein.x() + plane0[1] *Psurface_Nprotein.y() +
plane0[2] *Psurface_Nprotein.z() << std::endl;

std::cout << " $$$$$$$$$$$$$$$$$$$$$$$$$= " <<  std::endl;

}//if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
//************************************************************************

return Translate;
}//if (tDistanceB <= ABdistance && tDistanceC <= ACdistance)
// don't move the surface because protein is centered KMa

//MOVE THE SURFACE if the distance to the protein Cgeometry
// (projected to the plane) is larger than the UNIT CELL
//dimensions along both vectors AB,AC

if ( tDistanceC > ACdistance/2.0 ){
	    cc = tDistanceC / ACdistance ;
		icc = floor(cc);

        floor_ceil_icc = cc - icc;

		if (icc >= 1.0){

			if (floor_ceil_icc <= 0.5){
			IntersectPoint1 =
			floor_ceil_icc * ( IntersectPoint1 - Psurface_Nprotein ) / cc
			+ Psurface_Nprotein;
			}
			else{
			IntersectPoint1 =
			(floor_ceil_icc - 1.) * ( IntersectPoint1 - Psurface_Nprotein )/cc
			+ Psurface_Nprotein;
			}
		}
		else{
			IntersectPoint1 =
			( cc - icc - 1. )/ (cc - icc) *
			(IntersectPoint1 - Psurface_Nprotein ) + Psurface_Nprotein;
		}
}



if ( tDistanceB > ABdistance/2.0 ){
		bb = tDistanceB / ABdistance ;
		ibb = floor(bb);

        floor_ceil_ibb = bb - ibb;

		if (ibb >= 1.0){

			if (floor_ceil_ibb <= 0.5){
			IntersectPoint2 =
			floor_ceil_ibb * ( IntersectPoint2 - Psurface_Nprotein ) / bb
			+ Psurface_Nprotein;
			}
			else{
			IntersectPoint2 =
			(floor_ceil_ibb - 1.) * ( IntersectPoint2 - Psurface_Nprotein )/bb
			+ Psurface_Nprotein;
			}
		}
		else{
			IntersectPoint2 =
			( bb - ibb - 1. )/ ( bb - ibb) *
			(IntersectPoint2 - Psurface_Nprotein ) + Psurface_Nprotein;
		}
}



///////////////////////////////////////////////////////////////////////////////
fvAB = IntersectPoint2 - Psurface_Nprotein ;
fvAC = IntersectPoint1 - Psurface_Nprotein ;

// 2nd loop find the final intersection point F of I1'-I1-I2'-F
// the smaller parallelogram inscribed in Tthe larger P-I1-S-I2
tDistanceC = distance( IntersectPoint1, Psurface_Nprotein ) ;
tDistanceB = distance( IntersectPoint2, Psurface_Nprotein ) ;

//FIND the final intersection point
// *.........
//	.		.
//	.	  *	.
//	.........*

FinalPointPlane1 =
a_surface.point2lines3  ( IntersectPoint1, IntersectPoint2, fvAB, fvAC );

/*if ( distance ( FinalPointPlane1, Psurface_Nprotein ) >
	 distance ( Psurface_Nprotein, centerG_surface ) ){
std::cout << "integers do not recenter/periodicity & 180-A angle nohelp"
		  << std::endl;
std::cout << "IBB= " << ibb << " icc= " <<	    icc<< std::endl;
std::cout << "BB= " << bb << " cc= " <<	    cc<< std::endl;
std::cout << "before effort= " << distance ( Psurface_Nprotein, centerG_surface)
	<<" after=" << distance ( FinalPointPlane1, Psurface_Nprotein ) << std::endl;

		return Translate;
}*/

///////////////////////////////////////////////////////////////////////////////

//Translate = NEW_centerG_protein - CA_centerG_protein ;
	Translate = FinalPointPlane1 - centerG_surface  ;

	for ( size_t i = 0; i < ligand::ligand_one->atom_vector.size(); ++i ) {

		XYZ=ligand::ligand_one->atom_vector[i]->get_coordinates();

		XYZ += Translate;

		xpos = XYZ.x();
		ypos = XYZ.y();
		zpos = XYZ.z();

		ligand::ligand_one->atom_vector[i]->set_coordinates(xpos,ypos,zpos);
		}

		(*ligand::ligand_one).recompute_ligand_interface = true;

		ligand_center(docking::part_centroid(1,2),(*ligand::ligand_one));

// NEW center of geometry of surface	S
centerG_surface.x() = docking::part_centroid(1,2);
centerG_surface.y() = docking::part_centroid(2,2);
centerG_surface.z() = docking::part_centroid(3,2);

std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;
std::cout << "integers to move and recenter protein " <<  std::endl;
std::cout << "IBB= " << ibb << " icc= " <<	    icc<< std::endl;
std::cout << "BB= " << bb << " cc= " <<	    cc<< std::endl;
std::cout << " FINALE DistanceAB = " << tDistanceB
		  << "  DistanceAC="    	 <<	tDistanceC <<  std::endl;
std::cout << "TranslationXYZ:  x= " << Translate.x() << " y= " <<
		     Translate.y() << " z=" << Translate.z() << std::endl;
std::cout << "DISTANCE FINAL= " << distance(centerG_surface,Psurface_Nprotein)
<< std::endl;
std::cout << "DISTANCE FINAL 0=" << distance(centerG_surface,FinalPointPlane1)
<< std::endl;
std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;

//write-screen information finally  KMa
//************************************************************************
if ( runlevel_ns::runlevel >= runlevel_ns::inform ){

v_ABpAC = vAB + vAC ;
P_ABpAC = surface3pointA + v_ABpAC ;
d_ABpAC = distance ( P_ABpAC,surface3pointA ) ;

// cosBAC, thetaBAC;
cosBAC = inner_product ( vAB,vAC ) / ( ABdistance * ACdistance  ) ;
thetaBAC = acos ( cosBAC ) ;
 d_Psurfprot_cG = distance ( Psurface_Nprotein, centerG_surface ) ;

std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;

std::cout << " AB = " << distance(surface3pointB0 ,surface3pointA0 ) << " AC="
	<<	distance(surface3pointC0 ,surface3pointA0 ) <<  std::endl;

std::cout << "dISTANCE:Pnormaltosurface-protein_cG or tDistanceC + tDistanceB="
<< d_Psurfprot_cG 	<< " dISTANCE AB + AC= " <<	 d_ABpAC << std::endl;
std::cout << "cosBAC= " << cosBAC << " thetaBAC= " <<	 thetaBAC << std::endl;

std::cout << "findProteinZ x= " << findProteinZ.x() << " y= " <<
		     findProteinZ.y() << " z=" << findProteinZ.z() << std::endl;
std::cout << "distance of the protein zzzzzs ="
	<< findProteinZ.length()<< std::endl;
std::cout << "distance of the protein INITIALLY vertical="
	<< distance(CA_centerG_protein,Psurface_Nprotein) << std::endl;
std::cout << "meta inside centerG_surface.x() x= " << centerG_surface.x()
<< " y= " << centerG_surface.y() << " z=" << centerG_surface.z() << std::endl;
std::cout <<"CA_centerG_protein.x() x=" << CA_centerG_protein.x() << " y=" <<
	   	CA_centerG_protein.y()<< " z=" << CA_centerG_protein.z() << std::endl;
std::cout << "PLANE A= " << plane0[0] << " B= " <<	 plane0[1] << " C="
<< plane0[2] <<  " d= "<< plane0[3]<<std::endl;
std::cout << "verify P PLANE= "
<< plane0[0] *Psurface_Nprotein.x() + plane0[1] *Psurface_Nprotein.y() +
plane0[2] *Psurface_Nprotein.z() << std::endl;

 std::cout << "verify i2 PLANE= " << plane0[0] *IntersectPoint2.x()
			+ plane0[1] *IntersectPoint2.y()
			+ plane0[2] *IntersectPoint2.z() << std::endl;

std::cout << "verify i1 PLANE= " << plane0[0] *IntersectPoint1.x()
				+ plane0[1] *IntersectPoint1.y()
				+ plane0[2] *IntersectPoint1.z() << std::endl;

std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;
}//if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
//************************************************************************


return Translate ;
}



//////////////////////////////////////////////////////////////////////////////
/// @begin proteinmove_surface_symmetry
///
/// @brief
/// move protein to the center of geometry of the surface by symmetry(not used)
/// @detailed
/// It is not used currently since it is easier to move the surface
/// instead of protein by the protein_surfacemove_symmetry()
/// It will be necessary when we dock more than one proteins on the surface.
/// @param  fail - [in/out]? -
/// @param  mode - [in/out]? -
///
/// @global_read surface3pointA0,surface3pointB0,surface3pointC0,vAB0, vAC0
///
/// @global_write
///  centerG_surface
/// @remarks
///  needs to read 3 atoms from HETATM pdb (defined in the input PDB)
/// @references
///
/// @authors:Kosta Makrodimitris,Ph.D. 2006-02
///
/// @last_modified 2006-03-03
//////////////////////////////////////////////////////////////////////////////

numeric::xyzVector<double>
SurfaceMode::proteinmove_surface_symmetry()
{

	Surface a_surface;
  //--------------------
numeric::xyzVector<double> CA_centerG_protein, centerG_surface,
							centerG_surface1,vAB, vAC, fvAB, fvAC,
							Nproteintosurface, FinalPointPlane1,
							Psurface_Nprotein, NEW_centerG_protein,
							IntersectPoint1, IntersectPoint2,
							findProteinZ, Translate,
							iNproteintosurface,	 surface3pointA,
							surface3pointB, surface3pointC;
	FArray1D_float Tra_float( 3 ) ;
double tDistanceB=0.0, tDistanceC=0.0, bb=0.0, cc=0.0,
		ibb=0.0, icc=0.0 ;
std::vector<double>  plane0 ;
//?????????????????????????????????????????????????????????????????????????

chain_centroid_CA
(misc::position,1,misc::total_residue,docking::part_centroid(1,1));

ligand_center(docking::part_centroid(1,2),(*ligand::ligand_one));

// center of geometry of protein	P
CA_centerG_protein.x() = docking::part_centroid(1,1);
CA_centerG_protein.y() = docking::part_centroid(2,1);
CA_centerG_protein.z() = docking::part_centroid(3,1);

// center of geometry of surface	S
centerG_surface.x() = docking::part_centroid(1,2);
centerG_surface.y() = docking::part_centroid(2,2);
centerG_surface.z() = docking::part_centroid(3,2);

surface3pointA0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [0] ]->get_coordinates();
surface3pointB0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [1] ]->get_coordinates();
surface3pointC0 =
ligand::ligand_one->atom_vector[ PDB3atoms_planeCell [2] ]->get_coordinates();

//vectors of the 2 basis vectors of the surface system,the unit cell dimensions
		vAB0 = surface3pointB0 - surface3pointA0 ;
		vAC0 = surface3pointC0 - surface3pointA0 ;

//FIX the ABC in THE center of surface
surface3pointA = centerG_surface ;
surface3pointB = centerG_surface + vAB0 ;
surface3pointC = centerG_surface + vAC0 ;

vAB = surface3pointB - surface3pointA ;
vAC = surface3pointC - surface3pointA ;

// find the normal vector to the surface of the vectors AB, AC
Nproteintosurface =
a_surface.normalto3points ( surface3pointA, surface3pointB, surface3pointC );

// vector antiparallel
iNproteintosurface = -1.0 * Nproteintosurface ;

// Find the equation of the plane
plane0 =
a_surface.plane3points ( surface3pointA, surface3pointB, surface3pointC  );

// find the intersection point of the antiparallel normal vector
// and the surface plane
Psurface_Nprotein =
a_surface.point_intersect_plane(plane0,CA_centerG_protein,iNproteintosurface);

// find the vector difference of the normal intersection
// point and the center of geometry of protein
findProteinZ = Psurface_Nprotein - CA_centerG_protein ;

//FIND the 2 first intersection points
// *.........
//	.		 .
//	.		 .
//	.........*
 IntersectPoint1 =
 a_surface.point2lines3 ( centerG_surface, Psurface_Nprotein,  vAC, vAB );

 IntersectPoint2 =
 a_surface.point2lines3 ( centerG_surface, Psurface_Nprotein,  vAB, vAC );

tDistanceC = distance( IntersectPoint1, centerG_surface ) ;
tDistanceB = distance( IntersectPoint2, centerG_surface ) ;

//KMa
//BEGIN the criteria to move or not the surface to center the protein
if (tDistanceB <= ABdistance && tDistanceC <= ACdistance){

if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
	bb=0;
	cc=0;
	icc=0;
	ibb=0;
std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;
std::cout << "integers to move and recenter " <<  std::endl;
std::cout << "IBB= " << ibb << " icc= " <<	    icc<< std::endl;
std::cout << "BB= " << bb << " cc= " <<	    cc<< std::endl;
std::cout << " FINALE DistanceAB = " << tDistanceB
		  << "  DistanceAC="    	 <<	tDistanceC <<  std::endl;
}

return CA_centerG_protein;
}
// don't move the protein because the protein is centered KMa

//MOVE THE PROTEIN if the distance to the protein Cgeometry
// (projected to the plane) is larger than the UNIT CELL
//dimensions along both vectors AB,AC
if ( tDistanceC > ACdistance ){ cc = tDistanceC / ACdistance ; icc = floor(cc);
IntersectPoint1 =
(cc-icc) * ( IntersectPoint1 - centerG_surface ) / cc + centerG_surface;
}

if ( tDistanceB > ABdistance ) { bb = tDistanceB / ABdistance; ibb = floor(bb);
IntersectPoint2 =
(bb-ibb) * ( IntersectPoint2 - centerG_surface ) / bb + centerG_surface;
}

///////////////////////////////////////////////////////////////////////////////
fvAB = IntersectPoint2 - centerG_surface ;
fvAC = IntersectPoint1 - centerG_surface ;

// 2nd loop find the final intersection point F of I1'-I1-I2'-F
// the smaller parallelogram inscribed in Tthe larger P-I1-S-I2
tDistanceC = distance( IntersectPoint1, centerG_surface ) ;
tDistanceB = distance( IntersectPoint2, centerG_surface ) ;

//FIND the final intersection point
// *.........
//	.		.
//	.	  *	.
//	.........*
 FinalPointPlane1 =
 a_surface.point2lines3  ( IntersectPoint1, IntersectPoint2, fvAB, fvAC );

// ADD the Z-dimension in the protein translation movement
 NEW_centerG_protein = FinalPointPlane1 - findProteinZ ;
// NEW_centerG_protein=-Psurface_Nprotein+FinalPointPlane1+CA_centerG_protein;
//////////////////////////////////////////////////////////////////////////////

Translate = NEW_centerG_protein - CA_centerG_protein ;

Tra_float(1) = Translate.x() ;
Tra_float(2) = Translate.y() ;
Tra_float(3) = Translate.z() ;

for ( int i = 1; i <= misc::ints::total_residue; ++i ) {
 for ( int j = 1; j <= param::MAX_ATOM(); ++j ) {
  for ( int k = 1; k <= 3; ++k ) {
			misc::full_coord(k,j,i) = Tra_float(k) + misc::full_coord(k,j,i) ;
  }}}

for ( int i = 1; i <= misc::ints::total_residue;  ++i ) {
 for ( int j = 1; j <= 3; ++j ) {
			misc::Eposition(j,1,i) =  misc::full_coord(j,1,i); // N
			misc::Eposition(j,2,i) =  misc::full_coord(j,2,i); // CA
			misc::Eposition(j,4,i) =  misc::full_coord(j,3,i); // C --
			misc::Eposition(j,5,i) =  misc::full_coord(j,4,i); //
			misc::Eposition(j,3,i) =  misc::full_coord(j,5,i); //
			misc::centroid(j,i)+= Tra_float(j);
 }}
compute_centroid_parm_eachres(
misc::ints::res, misc::current_pose::res_variant, misc::full_coord );

misc::Ebest_position = misc::Eposition;
misc::best_centroid	= misc::centroid;
misc::best_centroid_parm_eachres = misc::centroid_parm_eachres;
misc::best_full_coord= misc::full_coord;


std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;
std::cout << "integers to move and recenter " <<  std::endl;
std::cout << "IBB= " << ibb << " icc= " <<	    icc<< std::endl;
std::cout << "BB= " << bb << " cc= " <<	    cc<< std::endl;
std::cout << " FINALE DistanceAB = " << tDistanceB
		  << "  DistanceAC="    	 <<	tDistanceC <<  std::endl;
std::cout << "Translation x= " << Tra_float(1) << " y= " <<
		     Tra_float(2) << " z=" << Tra_float(3)<< std::endl;

//write-screen information finally  KMa
//************************************************************************
if ( runlevel_ns::runlevel >= runlevel_ns::inform ){

// runlevel_ns::runlevel >= runlevel_ns::inform VARIABLES
numeric::xyzVector<double> v_ABpAC = vAB + vAC ;
numeric::xyzVector<double> P_ABpAC = surface3pointA + v_ABpAC ;
double d_ABpAC = distance ( P_ABpAC,surface3pointA ) ;
double cosBAC, thetaBAC;
cosBAC = inner_product ( vAB,vAC ) / ( ABdistance * ACdistance  ) ;
thetaBAC = acos ( cosBAC ) ;
double d_Psurfprot_cG = distance ( Psurface_Nprotein, centerG_surface ) ;
//const long double pppi = 3.14159265358979323846264338327950288 ;

std::cout << " ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ " <<  std::endl;
std::cout << "findProteinZ x= " << findProteinZ.x() << " y= " <<
		     findProteinZ.y() << " z=" << findProteinZ.z() << std::endl;
std::cout << "distance of the protein zzzzzs ="
	<< findProteinZ.length()<< std::endl;
std::cout << "distance of the protein centers ="
	<< distance(NEW_centerG_protein,CA_centerG_protein) << std::endl;
std::cout << "d_Psurf to prot_cG= " << d_Psurfprot_cG
	<< " d_AB plus AC= " <<	 d_ABpAC << std::endl;
std::cout << "cosBAC= " << cosBAC << " thetaBAC= " <<	 thetaBAC << std::endl;
std::cout << "centerG_surface.x() x= " << centerG_surface.x() << " y= " <<
		     centerG_surface.y() << " z=" << centerG_surface.z() << std::endl;
std::cout <<"CA_centerG_protein.x() x=" << CA_centerG_protein.x() << " y=" <<
	   	CA_centerG_protein.y()<< " z=" << CA_centerG_protein.z() << std::endl;
std::cout << "PLANE A= " << plane0[0] << " B= " <<	 plane0[1] << " C="
<< plane0[2] <<  " d= "<< plane0[3]<<std::endl;
std::cout << "verify P PLANE= "
<< plane0[0] *Psurface_Nprotein.x() + plane0[1] *Psurface_Nprotein.y() +
plane0[2] *Psurface_Nprotein.z() << std::endl;
std::cout << "verify i2 PLANE= " << plane0[0] *IntersectPoint2.x()
			+ plane0[1] *IntersectPoint2.y()
			+ plane0[2] *IntersectPoint2.z() << std::endl;
std::cout << "verify i1 PLANE= " << plane0[0] *IntersectPoint1.x()
				+ plane0[1] *IntersectPoint1.y()
				+ plane0[2] *IntersectPoint1.z() << std::endl;
std::cout << "verify F PLANE= " << plane0[0] *FinalPointPlane1.x()
				+ plane0[1] *FinalPointPlane1.y()
				+ plane0[2] *FinalPointPlane1.z() << std::endl;
std::cout << " final tDistanceB = " << tDistanceB << " final tDistanceC="
	<<	tDistanceC <<  std::endl;
std::cout << "1 FinalPointPlane1 x= " <<  FinalPointPlane1.x() << " y= " <<
	  FinalPointPlane1 .y() << " z=" <<  FinalPointPlane1.z() << std::endl;
std::cout <<" NEW_centerG_protein x= " << NEW_centerG_protein.x()<< " y=" <<
NEW_centerG_protein .y() << " z=" << NEW_centerG_protein.z() << std::endl;
std::cout << "PLANE distance of the protein FINAL ="
	<< distance(centerG_surface,FinalPointPlane1) << std::endl;
std::cout << "PLANE distance of the protein INITIALLY CG ="
	<< distance(Psurface_Nprotein, centerG_surface) << std::endl;
std::cout << "PLANE distance of the protein INITIALLY TARGET="
	<< distance(Psurface_Nprotein, FinalPointPlane1) << std::endl;
std::cout << "distance of the protein FINAL vertical="
	<< distance(NEW_centerG_protein,FinalPointPlane1) << std::endl;
std::cout << "distance of the protein INITIALLY vertical="
	<< distance(CA_centerG_protein,Psurface_Nprotein) << std::endl;
}// ( runlevel_ns::runlevel >= runlevel_ns::inform ){
//************************************************************************


return NEW_centerG_protein ;
}




}//namespace
