// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.10 $
//  $Date: 2005/10/06 23:09:52 $
//  $Author: aroop $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   rosetta++/antibody_modeling.cc
///
/// @brief  Modules  to graft  in CDR  loops in  a putative  antibody  model.
///         CDR-H3 is modeled based on special antibody H3 specific  fragment
///         libraries. Other CDR  loops are  grafted  in  based  on pdb files
///         containing only loop regions obtained by Arvind Sivasubramanina's
///         antibody homology modeling protocol.
/// @author Aroop Sircar (aroop@jhu.edu)

// Rosetta Headers
#include "aa_name_conversion.h"
#include "after_opts.h"
#include "antibody_modeling.h"
#include "cenlist.h" // snugdock functionality
#include "docking.h" // snugdock functionality
#include "docking_constraints.h"
#include "dock_loop_ensemble.h"
#include "dock_loop_ensemble_ns.h"
#include "docking_minimize.h" // snugdock functionality
#include "docking_movement.h" // snugdock functionality
#include "docking_ns.h"
#include "docking_score.h" // snugdock functionality
#include "docking_scoring.h" // snugdock functionality
#include "files_paths.h"
#include "fold_loops.h"
#include "fragment_class.h"
#include "fragments_pose.h"
#include "initialize.h"
#include "interface.h" // snugdock functionality
#include "jumping_loops.h"
#include "jumping_util.h"
#include "loops_ns.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h" // snugdock functionality
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_docking.h"
#include "pose_idealize.h"
#include "pose_io.h"
#include "pose_rotamer_trials.h" // snugdock functionality
#include "pose_vdw.h"
#include "prof.h"
#include "random_numbers.h"
#include "rotamer_trials.h" // snugdock functionality
#include "recover.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "smallmove.h" // snugdock functionality

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Utility Headers
#include <utility/io/izstream.hh>

///////////////////////////////////////////////////////////////////////////////
/// Command Line to invoke this function directly
/// /home/aroop/simcode/bin/rosetta.gcc aa 1flx _ -pose1 -antibody_modeling
///  -quiet -nstruct 1 -s 1flx.pdb
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling
///
/// @brief switches to antibody modeler or linker building depending on flag
///        read in initialize mode
///
/// @detailed this is mainly a wrapper called by job_distributor. Also handles
///           pose flag settings
///
/// @param[in]
///
/// @global_read antibody_ns
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling()
{

	using namespace pose_ns;
	using namespace runlevel_ns;

	pose_flag_ns::pose_flag_setting = true;

	// Flag to turn on mode for builing Fv linker for Erik Fernandez
	if( antibody_ns::fv_linker_mode)
		antibody_modeling_build_Fv_linker(); // Erik's routine for creating
	                                       //linker for Fv creation
	else
		antibody_modeling_assemble_cdrs(); // Developed to copy X-tal H3 to
                                       // WAM model

	pose_flag_ns::pose_flag_setting = false;

	// misc based monte carlo reset
	monte_carlo_accept_low();
	monte_carlo_accept_best();
	mc_update_scorefxns();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// Copy loop segment from scorce to starting structure
/// Loop hardcoded by standard chothia H3 numbering of 95H->102H
///
/// Command Line to invoke this function directly
/// rosetta.gcc aa FR01 _ -antibody_modeler -quiet -l1 -l2 -l3 -h1 -h2
/// -H3_filter -nstruct 1 -use_pdb_numbering -ccd_closure -s FR01
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_assemble_cdrs
///
/// @brief wrapper to put in differnt cdrs into the template pose based on
///        command line options
///
/// @detailed command line options read in previously enable grafting of
///           specific cdrs. Only H3 is modeled from scratch.
///
/// @param[in]
///
/// @global_read pose_from_pdb reads in pdb file, antibody_ns for different
///              options
///
/// @global_write pose is updated
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 03/25/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_assemble_cdrs()
{
	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace antibody_ns;
	using namespace files_paths;

	Pose framework_pose;
	// check to see if loops already grafted in
	if( !already_grafted ) {
		bool const fullatom( true );
		bool const ideal_pose( false );
		bool const read_all_chains( true );
		std::string framework_pdb_filename = dle_ns::pdb_filename
			+ ".pdb";
		bool local_multi_chain_setting = files_paths::multi_chain;
		pose_from_pdb( framework_pose, start_path + framework_pdb_filename,
									 fullatom, ideal_pose, read_all_chains );
		files_paths::multi_chain = local_multi_chain_setting;

		// framework_pose.dump_pdb( "01_start.pdb" ); // aroop_temp remove

		if( graft_l1 )
			antibody_modeling_graft_a_cdr( framework_pose, "l1");
		if( graft_l2 )
			antibody_modeling_graft_a_cdr( framework_pose, "l2");
		if( graft_l3)
			antibody_modeling_graft_a_cdr( framework_pose, "l3");
		if( graft_h1 )
			antibody_modeling_graft_a_cdr( framework_pose, "h1");
		if( graft_h2 )
			antibody_modeling_graft_a_cdr( framework_pose, "h2");
		if( graft_h3 )
			antibody_modeling_graft_a_cdr( framework_pose, "h3");

		// framework_pose.dump_pdb( "02_grafted.pdb" ); // aroop_temp remove

		// Hack to reset all misc style arrays by reloading
		// initial pdb
		{
			Pose reset_pose;
			pose_from_pdb( reset_pose, start_path + framework_pdb_filename, fullatom,
										 ideal_pose, read_all_chains );
			files_paths::multi_chain = local_multi_chain_setting;
		}

		// saving starting sidechains
		dle_ns::starting_sidechain_pose = framework_pose;

		// saving unbound rotamers
		dle_save_unbound(
      dle_ns::starting_sidechain_pose );

		// No repacking needed while clustering
		if( !antibody_cluster && !norepack_antibody )
			dle_trim_and_repack( framework_pose,cdr_ns::cdr_h3_begin,
																					cdr_ns::cdr_h3_end );

		// storing grafted pose
		grafted_pose = framework_pose;
		already_grafted = true;
	}
	framework_pose = grafted_pose;
	if( model_h3 ) {
		antibody_modeling_model_H3( framework_pose );
		FArray1D_bool allow_repack( framework_pose.total_residue(), true );
		int repack_cycles(1);
		if( antibody_refine && !snug_fit )
			repack_cycles = 3;
		if( !runlevel_ns::benchmark)
			dle_extreme_repack( framework_pose,
																				 repack_cycles,
																				 allow_repack,
																				 true, // rt_min
																				 true, // rotamer_trials
																				 false, // force_one_repack
																				 true ); // use_unbounds

		// framework_pose.dump_pdb( "05_h3.pdb" ); // aroop_temp remove

		// docking local refine of L-H & simultaneous H3 minimization
		if( snug_fit ) {
			antibody_modeling_refineLH( framework_pose );
			antibody_modeling_orient_to_native( framework_pose );
			// framework_pose.dump_pdb( "06_snug.pdb" ); // aroop_temp remove
		}

		if( dle_ns::dle_model_H3 )
			dle_ns::dle_H3_filter = true;
		// relaxing CDR-H3
		// if( flank_relax  && antibody_ns::freeze_h3 )
		//	dle_loop_fa_relax( framework_pose,
		//	  cdr_ns::cdr_h3_begin - antibody_ns::nter - antibody_ns::h3_flank,
		//	    cdr_ns::cdr_h3_end + antibody_ns::base + antibody_ns::h3_flank);
		//else
		dle_loop_fa_relax( framework_pose,
			 cdr_ns::cdr_h3_begin - antibody_ns::nter,
			   cdr_ns::cdr_h3_end + antibody_ns::base );
		if( dle_ns::dle_model_H3 )
			dle_ns::dle_H3_filter = false;
		// framework_pose.dump_pdb( "07_relax_h3.pdb" ); // aroop_temp remove


		if( relax_cdrs ) {
			antibody_modeling_relax_cdrs( framework_pose );
			// framework_pose.dump_pdb( "08_relax_cdr.pdb" ); // aroop_temp remove
		}

		int const modeled_loop_begin( cdr_ns::cdr_h3_begin - nter );
		int const modeled_loop_end( cdr_ns::cdr_h3_end + base );
		// variable to be consistent with flexible docking
		int counter(0);
		dle_write_one_loopfile( framework_pose,
																					 modeled_loop_begin,
																					 modeled_loop_end,
																					 counter );
	}

	// repacking clashing sidechains
	// dle_bad_Erep_repack( framework_pose );

	Score_weight_map weight_map( score12 );
	framework_pose.score( weight_map ); // ensures score sync
	Monte_carlo mc( framework_pose, weight_map, 2.0 ); // temp = 2.0
	mc.reset( framework_pose ); // pose based Monte Carlo Reset

	framework_pose.copy_to_misc();

	update_sequence();
	save_START();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_convert_to_sequential_res_from_chothia_res
///
/// @brief converts chothia numbering into rosetta sequential numbering
///
/// @detailed for each CDR there is a different but consisten chothia numbering
///           scheme. One has to find invert these constant numbers to obtain
///           rosetta sequential numbering for the loop in question. The
///           Chothia numbering defines the loops as follows:
///
///           Chothia numbered pdbs have the CDRs numbered as follows:
///           CDR L1: Chain L : Residues 24-34
///           CDR L2: Chain L : Residues 50-56
///           CDR L3: Chain L : Residues 89-97
///           CDR H1: Chain H : Residues 26-35
///           CDR H2: Chain H : Residues 50-65
///           CDR H3: Chain H : Residues 95-102
///
/// @param[in] cdr_loop_name: informs this function which loop definition
///            is desired. This is the only input. Others are output
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_convert_to_sequential_res_from_chothia_res(
	const std::string & cdr_loop_name,
	char & pdb_chain_id,
	int & pdb_loop_begin,
	int & pdb_loop_end
)
{
	if(cdr_loop_name == "l1")
		{
			pdb_chain_id = 'L';
			pdb_loop_begin = 24;
			pdb_loop_end = 34;
		}
	if(cdr_loop_name == "l2")
		{
			pdb_chain_id = 'L';
			pdb_loop_begin = 50;
			pdb_loop_end = 56;
		}
	if(cdr_loop_name == "l3")
		{
			pdb_chain_id = 'L';
			pdb_loop_begin = 89;
			pdb_loop_end = 97;
		}
	if(cdr_loop_name == "h1")
		{
			pdb_chain_id = 'H';
			pdb_loop_begin = 26;
			pdb_loop_end = 35;
		}
	if(cdr_loop_name == "h2")
		{
			pdb_chain_id = 'H';
			pdb_loop_begin = 50;
			pdb_loop_end = 65;
		}
	if(cdr_loop_name == "h3")
		{
			pdb_chain_id = 'H';
			pdb_loop_begin = 95;
			pdb_loop_end = 103; // Actually its H 102, however H3 lengths vary and
                          // some do not have 102. Later on we use the end of
                          // H chain as res(H,103) minus one
		}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_graft_a_cdr
///
/// @brief graft a particular CDR loop into the framework pose
///
/// @detailed Each CDR is grafted into the framework_pose. The CDR is read in
///           from files defined. Then CCD is used to close the loops. This
///           leads to minor diversity. The sidechains are also repacked for
///           the CDR in question.
///
/// @param[in] cdr_loop_name: informs this function which loop to graft in
///            framework_pose: the main framework into which grafting takes
///                            place
///
/// @global_read Default framewokrs are read in from framework files to take
///              care of the final omega angle in the loop.
///
/// @global_write pose is updated with the loop grafted in
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 03/15/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_graft_a_cdr(
  pose_ns::Pose & framework_pose,
	const std::string & cdr_loop_name
	)
{
	using namespace pose_ns;
	using namespace files_paths;
	// pose_from_pdb changes this and hence needs to be stored
	bool local_multi_chain_setting = files_paths::multi_chain;

	int pdb_loop_begin(0), pdb_loop_end(0);
	char pdb_chain_id = '-';
	antibody_modeling_convert_to_sequential_res_from_chothia_res( cdr_loop_name,
	  pdb_chain_id, pdb_loop_begin, pdb_loop_end);

	int nres( framework_pose.total_residue() );
	int framework_loop_begin;
	res_num_from_pdb_res_num_chain( framework_loop_begin, pdb_loop_begin,
																	pdb_chain_id );
	framework_loop_begin = framework_loop_begin - antibody_ns::deep_graft;
	std::cout << cdr_loop_name << " Loop begin : "
						<< framework_loop_begin << std::endl;
	int framework_loop_end;
	res_num_from_pdb_res_num_chain( framework_loop_end, pdb_loop_end,
                                  pdb_chain_id );
	// Detailed explanation provided
	// in _convert_to_sequential_res_from_chothia_res
	if( cdr_loop_name == "h3" )
		framework_loop_end = framework_loop_end - 1;
	framework_loop_end = framework_loop_end + antibody_ns::deep_graft;
	std::cout << cdr_loop_name << " Loop end : "
						<< framework_loop_end << std::endl;
	int framework_loop_size = ( framework_loop_end - framework_loop_begin ) + 1;
	int cutpoint = framework_loop_begin + int(framework_loop_size/2);

	Pose cdr_pose;
	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const read_all_chains( true );
	std::string cdr_pdb_filename;
	stringafteroption(cdr_loop_name, cdr_loop_name, cdr_pdb_filename);
	if( cdr_pdb_filename[0] == '-' ) {
		cdr_pdb_filename = cdr_loop_name;
		std::cout << "[STR  OPT]Not using new value for [-"
							<< cdr_loop_name << "]." << std::endl;
		std::cout << "[STR  OPT]Using default value for [-"
							<< cdr_loop_name << "] " << cdr_loop_name << "." << std::endl;
	}
	cdr_pdb_filename = cdr_pdb_filename + ".pdb";
	if( !pose_from_pdb( cdr_pose, start_path + cdr_pdb_filename,
											fullatom, ideal_pose, read_all_chains )) {
		std::cout << "[Error]: Could not open cdr loop file: "
							<< cdr_pdb_filename << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}
	files_paths::multi_chain = local_multi_chain_setting;
	cdr_pose.copy_to_misc();

	int cdr_loop_begin;
	res_num_from_pdb_res_num_chain( cdr_loop_begin,
																	pdb_loop_begin, pdb_chain_id );
	cdr_loop_begin = cdr_loop_begin - antibody_ns::deep_graft;
	std::cout << cdr_loop_name << " Loop begin source : "
						<< cdr_loop_begin << std::endl;
	int cdr_loop_end;
	res_num_from_pdb_res_num_chain( cdr_loop_end, pdb_loop_end, pdb_chain_id );
	// Detailed explanation provided
	// in _convert_to_sequential_res_from_chothia_res
	if( cdr_loop_name == "h3" )
		cdr_loop_end = cdr_loop_end - 1;
	cdr_loop_end = cdr_loop_end + antibody_ns::deep_graft;
	std::cout << cdr_loop_name << " Loop end source : " << cdr_loop_end
						<< std::endl;

	// grafting of cdrs by superimposition of stem residues
	if( antibody_ns::superimpose_cdr) {
		files_paths::multi_chain = local_multi_chain_setting;
		framework_pose.copy_to_misc();
		// updating backbone bonds and torsions
		framework_pose.update_backbone_bonds_and_torsions();
		antibody_modeling_graft_cdr_by_superimpostion(framework_pose, cdr_pose,
			framework_loop_begin, framework_loop_end,	cdr_loop_begin, cdr_loop_end);
		return;
	}

	Pose unaligned_framework_monomer;
  int unaligned_cdr_loop_begin(0), unaligned_cdr_loop_end(0);

	std::string unaligned_framework_monomer_filename;
	if( antibody_ns::deep_graft == 0 ) {
		if( (cdr_loop_name == "l1") || (cdr_loop_name == "l2") ||
				(cdr_loop_name == "l3") )
			unaligned_framework_monomer =
				antibody_ns::unaligned_framework_light_monomer;
		if( (cdr_loop_name == "h1") || (cdr_loop_name == "h2") )
			unaligned_framework_monomer =
				antibody_ns::unaligned_framework_heavy_monomer;
		unaligned_framework_monomer.copy_to_misc();

		res_num_from_pdb_res_num_chain( unaligned_cdr_loop_begin,
																		pdb_loop_begin, pdb_chain_id );
		std::cout << cdr_loop_name << " Unaligned loop begin source : "
							<< unaligned_cdr_loop_begin << std::endl;
		res_num_from_pdb_res_num_chain( unaligned_cdr_loop_end, pdb_loop_end,
																		pdb_chain_id );
		// Detailed explanation provided
		// in _convert_to_sequential_res_from_chothia_res
		if( cdr_loop_name == "h3" )
			unaligned_cdr_loop_end = unaligned_cdr_loop_end - 1;
		std::cout << cdr_loop_name << " Unaligned loop end source : "
							<< unaligned_cdr_loop_end << std::endl;
	}

	files_paths::multi_chain = local_multi_chain_setting;
	framework_pose.copy_to_misc();
	// updating backbone bonds and torsions
	framework_pose.update_backbone_bonds_and_torsions();

	//setup a fold_tree with jump from loop_begin-1 to loop_end+1
	framework_pose.one_jump_tree( nres, framework_loop_begin-2,
																framework_loop_end+2, cutpoint,1);
	// Insert init phi-psi angles
	insert_init_frag( framework_pose,
										framework_loop_begin+antibody_ns::deep_graft,
										framework_loop_size-(2*antibody_ns::deep_graft) );
	                  // 2 for each terminal
	if( antibody_ns::deep_graft == 0 ) {
		framework_pose.set_psi(framework_loop_begin-1, cdr_pose.psi(
													   cdr_loop_begin-1));
		framework_pose.set_omega(framework_loop_begin-1, cdr_pose.omega(
													     cdr_loop_begin-1));
	}
	else {
		if ( !dle_ns::freeze_bonds )
			framework_pose.insert_ideal_bonds( framework_loop_begin - 1 +
				antibody_ns::deep_graft, framework_loop_end );
	}

	Pose rename_pose; // copy of framework pose before copying in segments.
                    // This will be used to reset parameters like name
                    // res, res_variant after copying segments
	rename_pose = framework_pose; // storing initial information

	if( antibody_ns::deep_graft == 0 ) {
		framework_pose.copy_segment( 1, unaligned_framework_monomer,
			framework_loop_begin-1, unaligned_cdr_loop_begin-1,
				"true" ); // update jumps
		framework_pose.copy_segment( 1, unaligned_framework_monomer,
    	framework_loop_end+1, unaligned_cdr_loop_end+1, "true");
	}
	framework_pose.copy_segment( framework_loop_size, cdr_pose,
		framework_loop_begin, cdr_loop_begin, "true");

	for(int i = framework_loop_begin-1; i <= framework_loop_end+1; i++)	{
		framework_pose.set_name( i, rename_pose.name(i));
		framework_pose.set_res( i, rename_pose.res(i));
		framework_pose.set_res_variant( i, rename_pose.res_variant(i));
	}


	// Loop closing module
	Pose pre_ccd; // pose before ccd_closure
	pre_ccd = framework_pose;
	// closing grafted loops by ccd
	if( antibody_ns::fast_ccd ) {
		// Deterministic algorithm but gives poor rmsg to native
		// outputs from fast_ccd_loop_closure not used anywhere
		float fdev, bdev, torsion_delta, rama_delta;
		float ccd_threshold(1.0); // around a bond length
		fast_ccd_loop_closure( framework_pose,
			framework_loop_begin, framework_loop_end, cutpoint, 100, ccd_threshold,
				false /*rama check*/, 2.0, 0.1, 0.1, 0.1, fdev, bdev, torsion_delta,
          rama_delta);
	}
	else {
		// stochastic but gives better rmsg values to native
		ccd_moves( 50 /*total_moves*/, framework_pose, framework_loop_begin,
							 framework_loop_end, cutpoint);
	}
	// rmsg check to ensure that loop does not move very far
	// a small gap is preferred to a drastically altered loop
	// float ccd_rmsg(0); // rmsg of loop between pre and post ccd closure
	// ccd_rmsg = dle_loop_rmsg( pre_ccd, framework_pose,
	// 						 framework_loop_begin,framework_loop_end);
	//if( ccd_rmsg > 5 ) // Under test for right parameter
	//		framework_pose = pre_ccd; // forget ccd and be satisfied with small gap

	// updating backbone bonds and torsions
	framework_pose.update_backbone_bonds_and_torsions();

	FArray1D_bool allow_repack( nres, false );
	for( int i = 1; i <= nres; i++ )
		if( (i >= framework_loop_begin) && ( i <= framework_loop_end ) )
			allow_repack(i) = true ;
	framework_pose.repack( allow_repack, false /*include_current*/ );
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_model_H3
///
/// @brief wrapper script to build, cluster, refine putative H3's
///
/// @detailed In build mode, a user defined number of H3's are built. In
///           cluster mode, a rms table and score file are generated to serve
///           as input to the statistical package R for hierarchial clustering.
///           Finally in refine mode, the top 10 clusters by score & top 10 by
///           size are exhaustively refined
///
/// @param[in] framework_pose: the main framework into which H3 is modeled
///
/// @global_read antibody_ns, hfr.pdb
///
/// @global_write pose is updated with the loop grafted in
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 03/28/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_model_H3(
  pose_ns::Pose & framework_pose
)
{
	using namespace pose_ns;
	using namespace antibody_ns;
	// using namespace design;

	dle_ns::dle_calc_rmsg_flag = true;
	std::string cdr_loop_name = "h3";

	int framework_loop_begin(cdr_ns::cdr_h3_begin);
	int framework_loop_end(cdr_ns::cdr_h3_end);
	int nres( framework_pose.total_residue() );
	int framework_loop_size = ( framework_loop_end - framework_loop_begin ) + 1;

	if( H3_base_mode ) {
		dle_CDR_H3_filter( framework_pose,framework_loop_begin,
																			framework_loop_size);
		return;
	}

	// updating backbone bonds and torsions
	framework_pose.update_backbone_bonds_and_torsions();

	if( nter == 0 ) {
		// int cutpoint = framework_loop_begin + int(framework_loop_size/2);
		int cutpoint = framework_loop_begin + 1;
		int unaligned_cdr_loop_begin(0), unaligned_cdr_loop_end(0);
		unaligned_cdr_loop_begin = unaligned_h3_loop_begin;
		unaligned_cdr_loop_end = unaligned_h3_loop_end;

		//setup a fold_tree with jump from loop_begin-1 to loop_end+1
		framework_pose.one_jump_tree( nres, framework_loop_begin-2,
																	framework_loop_end+2, cutpoint,1);
		if( !dle_ns::refine_input_loop ) {
			insert_init_frag( framework_pose, framework_loop_begin,
												framework_loop_size ); // Insert init phi-psi angles
			framework_pose.set_psi(framework_loop_begin-1,
			  unaligned_framework_heavy_monomer.psi( unaligned_cdr_loop_begin-1));
			framework_pose.set_omega(framework_loop_begin-1,
			  unaligned_framework_heavy_monomer.omega( unaligned_cdr_loop_begin-1));
			if ( !dle_ns::freeze_bonds )
				framework_pose.insert_ideal_bonds( framework_loop_begin-1,
					  															 framework_loop_end );
			framework_pose.copy_segment( 1, unaligned_framework_heavy_monomer,
				framework_loop_begin-1, unaligned_cdr_loop_begin-1, "true");
			framework_pose.copy_segment( 1, unaligned_framework_heavy_monomer,
			  framework_loop_end+1, unaligned_cdr_loop_end+1, "true");
		}
	}

	//setup a fold_tree with jump from loop_begin-1 to loop_end+1
	int modified_framework_loop_begin = framework_loop_begin - nter;
	int modified_framework_loop_end = framework_loop_end + base;
	int modified_framework_loop_size = ( modified_framework_loop_end -
																			 modified_framework_loop_begin ) + 1;
	int modified_cutpoint = modified_framework_loop_begin + 1;
	//int modified_cutpoint = modified_framework_loop_begin +
	//	int( modified_framework_loop_size / 2 );
	framework_pose.one_jump_tree( nres, modified_framework_loop_begin-1,
																modified_framework_loop_end+1,
																modified_cutpoint,1);
	framework_pose.set_allow_jump_move(1,false);
	// Insert init phi-psi angles
	if( nter != 0 ) {
		if( !dle_ns::refine_input_loop ) {
			insert_init_frag( framework_pose, modified_framework_loop_begin,
												modified_framework_loop_size );
		}
		if ( !dle_ns::freeze_bonds )
			framework_pose.insert_ideal_bonds( modified_framework_loop_begin,
																				 modified_framework_loop_end );
	}

	/*
	// enabling thorough rotamer sampling of H3 and neighbors
	RotamerOptions saved_rotamer_options = active_rotamer_options;
	{
		using namespace exchi_flags;
		active_rotamer_options.ex1 = true;
		active_rotamer_options.ex1aro = true;
		active_rotamer_options.ex1_sample_level = EX_TWO_FULL_STEP_STDDEVS;
		active_rotamer_options.ex1aro_exposed = true;
		active_rotamer_options.exOH = true;
		active_rotamer_options.ex2 = true;
		active_rotamer_options.ex2_sample_level = EX_TWO_FULL_STEP_STDDEVS;
		active_rotamer_options.ex2aro_exposed = true;
		active_rotamer_options.ex3 = true;
		active_rotamer_options.ex3_sample_level = EX_ONE_STDDEV;
		active_rotamer_options.ex4 = true;
		active_rotamer_options.ex4_sample_level = EX_ONE_STDDEV;
		active_rotamer_options.setup_exflags();
	}*/
	int counter(0); // variable to be consistent with flexible docking
	//std::vector< Fragment > h3_library;

	if( antibody_build ) {
		Pose starting_framework_pose;
		starting_framework_pose = framework_pose;
		float cut_separation( 10.00 );
		while( cut_separation > 1.9 ) {
			framework_pose = starting_framework_pose;
			if( nter != 0 )
				antibody_modeling_insert_ter( framework_pose, "nter");
			if( base != 0 )
				antibody_modeling_insert_ter( framework_pose, "base");
			if( dle_ns::dle_model_H3 )
				dle_ns::dle_H3_filter = true;
			dle_build_one_loop( framework_pose, framework_loop_begin,
																				 framework_loop_end,
																				 h3_library, counter );
			cut_separation = dle_cutpoint_separation(framework_pose);
		}
		// framework_pose.dump_pdb( "03_cetroid_h3.pdb" ); // aroop_temp remove
		/*
		if( antibody_ns::min_base_relax ) {
			antibody_ns::min_base_relax = false; // triggers higer amplitude
			dle_loop_fa_relax( framework_pose,
				modified_framework_loop_begin, modified_framework_loop_end-4 );
			antibody_ns::min_base_relax = true; // triggers lower amplitude
			dle_loop_fa_relax( framework_pose,
				modified_framework_loop_end-3, modified_framework_loop_end );
			antibody_ns::min_base_relax = false; // triggers higer amplitude
		}
		else
			dle_loop_fa_relax( framework_pose,
				modified_framework_loop_begin, modified_framework_loop_end );
		// framework_pose.dump_pdb( "04_fullatom_h3.pdb" ); // aroop_temp remove
		*/
		dle_ns::dle_H3_filter = false;
	} // MIGHT HAVE TO CHANGE -3 TO ACCOMODATE OMEGA BONDS ETC
	else if( antibody_cluster ) {
		dle_cluster_loop_library( framework_pose,
			modified_framework_loop_begin, modified_framework_loop_end,
			  h3_library, counter );
	}
	else if( antibody_refine ){
		antibody_modeling_refine_H3(framework_pose, modified_framework_loop_begin,
																modified_framework_loop_end);
	}
	else if( antibody_insert_all_h3 ) {
		antibody_modeling_insert_h3(framework_pose, modified_framework_loop_begin,
																modified_framework_loop_end, h3_library);
	}
	else if( dle_ns::refine_input_loop ) {
			int const relax_cycles(1);  // 1 cycle around 10 min: Default 30 CHANGE TO 30 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			Score_weight_map weight_map( score12 );
			Monte_carlo relaxed_h3_mc ( framework_pose, weight_map, 2.0 );
			for(int i = 1; i <= relax_cycles; i++){
				if( dle_ns::dle_model_H3 )
					dle_ns::dle_H3_filter = true;
				dle_loop_fa_relax( framework_pose,
					modified_framework_loop_begin, modified_framework_loop_end );
				dle_ns::dle_H3_filter = false;
				relaxed_h3_mc.boltzmann(framework_pose);
			}
			// Choose the lowest scoring decoy ever observed
			framework_pose = relaxed_h3_mc.low_pose();
			// Output a loopfile for the decoy
			dle_write_one_loopfile( framework_pose,
																						 modified_framework_loop_begin,
																						 modified_framework_loop_end,
																						 counter );
	}
	else {
		std::cout << std::endl
							<< "Missing options. Options are: " << std::endl
							<< "-build_loop" << std::endl
							<< "-cluster_loop" << std::endl
							<< "-refine_loop" << std::endl
							<< "-insert_loop" << std::endl
							<< "-refine_input"<< std::endl;
		// exit if neither build nor read nor cluster
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// // Restoring initial rotamer options
	// active_rotamer_options = saved_rotamer_options;
	// active_rotamer_options.setup_exflags();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_initialize_pose
///
/// @brief various options needed by antibody modeling mode
///
/// @detailed In build mode fragment files have to be read in. In refine mode,
///           the loop library has to be read in and as well in cluster mode.
///
/// @param[in] framework_pose on which all operations are carried out
///            mode = rosetta mode being run (Antibody Modeler)
///
/// @global_read fragment files & loop file (not in build mode)
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 12/26/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_initialize_pose(
	pose_ns::Pose & start_pose,
	std::string const mode
)
{
	using namespace files_paths;
	using namespace dle_ns;
	using namespace antibody_ns;

	std::string filename;
	bool ideal_pos (false);
	bool fullatom (true);
	bool read_all_chains(true);
	set_pose_flag (true);

	filename = start_path + protein_name + ".pdb";
	bool local_multi_chain_setting = multi_chain;
	pose_from_pdb(start_pose, filename, fullatom, ideal_pos, read_all_chains);
	multi_chain = local_multi_chain_setting;
	start_pose.copy_to_misc();

	// snugdock options
	if( docking::antibody_modeling_dock_n_snug_flag ) {
		monte_carlo_accept_low();
		antibody_modeling_detect_L_end( start_pose );
		snugloop = truefalseoption( "snugloop" );
		snugh3 = truefalseoption( "snugh3" );
		if( snugloop || snugh3 ) {
			freeze_h3 = false;
			flank_relax = false;
			deep_graft = 0;
			antibody_modeling_add_all_cdrs( start_pose );
			dle_loops = all_cdrs;
		}
		snugnative = truefalseoption( "snugnative" );
		if( snugnative ) {
			// storing native pdb filename
			stringafteroption("snugnative", stringafteroption("s"),
												native_cdr::native_pdb_filename);
			if( native_cdr::native_pdb_filename[0] == '-' ) {
				native_cdr::native_pdb_filename = stringafteroption("s");
				std::cout << "[STR  OPT]Not using new value for [-native]."
									<< std::endl;
				std::cout << "[STR  OPT]Using default value for [-native] "
									<< dle_ns::pdb_filename << "." << std::endl;
			}
			native_cdr::native_pdb_filename = native_cdr::native_pdb_filename
				                                +	".pdb";
			// load native pdb and native cdr loop definitions
			antibody_modeling_load_native();;
		}
		save_START();
		if (mode != "score")
			set_pose_flag (false);
		return;
	}

	// relax flanking regions of h3
	flank_relax = truefalseoption( "flank_relax" );
	// choosing number of flanking residues
	if( flank_relax )
		intafteroption( "flank_relax", h3_flank, h3_flank );

	// default value of flanking residues to be used during grafting
	int deep_default = 0;
	// graft cdr by superimposing stems of cdr
	// deep defaults to 1 if this is set to true
	// literature suggests that deep should be 2
	// deep 2 indicates 3 flanking residues
	superimpose_cdr = truefalseoption("superimpose");
	if( superimpose_cdr )
		deep_default = 1;

	// specify no. of flanking cdr residues to be grafted in
	intafteroption( "deep", deep_default, deep_graft );

	// freeze h3 after full atom relax
	freeze_h3 = truefalseoption( "freeze_h3" );

	antibody_modeling_add_all_cdrs( start_pose );

	pdb_filename = stringafteroption("s"); // four letter pdb filename
	fv_linker_mode = truefalseoption("fv_linker");
	graft_l1 = truefalseoption("l1"); // graft in l1 from default file l1.pdb
	graft_l2 = truefalseoption("l2"); // graft in l2 from default file l2.pdb
	graft_l3 = truefalseoption("l3"); // graft in l3 from default file l3.pdb
	graft_h1 = truefalseoption("h1"); // graft in h1 from default file h1.pdb
	graft_h2 = truefalseoption("h2"); // graft in h2 from default file h2.pdb
	// graft in h3 from default file h3.pdb
	graft_h3 = truefalseoption("graft_h3");
	model_h3 = truefalseoption("h3"); // model h3 loop from fragments

	dle_common_options();

	// extensive repacking option
  norepack_antibody = truefalseoption( "norepack_antibody" );

	// force hbond computation
	compute_hbond = truefalseoption( "compute_hbond" );

	// enable docking local refine of L-H & simultaneous H3 minimization
	snug_fit = truefalseoption( "snug_fit" );

	// use fast_ccd_frag_close or default ccd_moves
	fast_ccd = truefalseoption("fast_ccd");
	if( graft_l1 || graft_l2 || graft_l3 )
		stringafteroption("lfr", "lfr.pdb",
											unaligned_framework_light_monomer_filename);
	if( graft_h1 || graft_l2 || model_h3 )
		stringafteroption("hfr", "hfr.pdb",
											unaligned_framework_heavy_monomer_filename);
	// load the unaligned frameworks
	antibody_modeling_store_unaligned_framework_info();

	// dummy variable not used here but used in dle
	int dummy_counter(0);
	int nres( start_pose.total_residue() );

	// int pdb_loop_begin(0), pdb_loop_end(0);
	// char pdb_chain_id = '-';
	// antibody_modeling_convert_to_sequential_res_from_chothia_res( "h3",
	//  pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	// int framework_loop_begin;
	// res_num_from_pdb_res_num_chain( framework_loop_begin, pdb_loop_begin,
	//	 															pdb_chain_id );
	// std::cout << "H3" << " Loop begin : " << framework_loop_begin << std::endl;
	// int framework_loop_end;
	// res_num_from_pdb_res_num_chain( framework_loop_end, pdb_loop_end,
	//  															pdb_chain_id );
	// // explation provided in convert_to_sequential_res_from_chothia_res
	// framework_loop_end = framework_loop_end - 1;
	// std::cout << "H3" << " Loop end : " << framework_loop_end << std::endl;
  //
	// // storing in antibody_ns
	// cdr_ns::cdr_h3_begin = framework_loop_begin;
	// cdr_ns::cdr_h3_end = framework_loop_end;

	if( model_h3 ) {
		antibody_ns::use_native = truefalseoption("native");
		if( antibody_ns::use_native ) {
			dle_ns::idealize_native =
				truefalseoption("idealize_native");

			// storing native pdb filename
			stringafteroption("native", dle_ns::pdb_filename,
												native_cdr::native_pdb_filename);
			if( native_cdr::native_pdb_filename[0] == '-' ) {
				native_cdr::native_pdb_filename = dle_ns::pdb_filename;
				std::cout << "[STR  OPT]Not using new value for [-native]."
									<< std::endl;
				std::cout << "[STR  OPT]Using default value for [-native] "
									<< dle_ns::pdb_filename << "." << std::endl;
			}
			native_cdr::native_pdb_filename = native_cdr::native_pdb_filename
				                                +	".pdb";
			// load native pdb and native cdr loop definitions
			antibody_modeling_load_native();
		}

		// relax all cdrs simultaneously after h3 modeling
		relax_cdrs = truefalseoption( "relax_cdrs" );

		// relax all cdrs while docking
		relax_dock = truefalseoption( "relax_dock" );

		// deciding to read in full/only h3 fragments
		loop_frags = truefalseoption("loop_frags");
		use_nter_frags = truefalseoption("nter");
		dont_use_cter_frags = truefalseoption("no_cter");
		if( dont_use_cter_frags ) {
			base = 0;
			h3_base_frag_size = 0;
		}
		if( !use_nter_frags ) {
			nter = 0;
			h3_nter_frag_size = 0;
		}
		int antibody_frag_res(nres);
		if( loop_frags ) {
			// loop_size + one flanking residues on either side
			antibody_frag_res = (((cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1)
												 + 2);
			// negetive of (loop_begin - 1 flanking residue -
			// - 1) (The last minus one follows from the fact
			// that if the number of residues in fragment
			// file is equal to total residues of protein then
			// the offset is zero and NOT one
			loop_frag_offset = 0 - ((cdr_ns::cdr_h3_begin - 1)
															- 1);
		}
		// use random cutpoint instead of cut in the middle
		h3_random_cut = truefalseoption( "h3_random_cut" );
		// lower amplitude of base minimization
		min_base_relax = truefalseoption( "base_relax" );
		// bias use of antibody fragments
		bias_frags = truefalseoption( "bias_frags" );
		// decide bias weight of antibody fragments
		if( bias_frags )
			realafteroption( "bias_frags", 3.00, bias_fraction );
		// sets the secondary structure of the base region
		// to a strand. This results in lower amplitudes
		// during relaxation
		enforce_base_strand = truefalseoption( "base_strand" );
		// select bases based on only kink/extended match
		select_all_kink_match = truefalseoption( "all_kink" );
		if( antibody_build ) {
			read_fragments_simple( pdb_filename.erase(4) + "_", antibody_frag_res );
			if( nter != 0 )
				antibody_modeling_read_H3_ter_fragment( start_pose, "nter" );
			if( base != 0 )
				antibody_modeling_read_H3_ter_fragment( start_pose, "base" );
		}
		else if( antibody_cluster ) {
			dle_read_in_one_loop_library(cdr_ns::cdr_h3_begin - nter,
				cdr_ns::cdr_h3_end + base, dummy_counter, h3_library,
				  "unrefined_unclustered");
		}
		else if( antibody_refine ) {
			antibody_modeling_grab_top_H3_models(cdr_ns::cdr_h3_begin - nter,
																					 cdr_ns::cdr_h3_end + base,
																					 h3_library);
			//dle_read_in_one_loop_library(cdr_ns::cdr_h3_begin
      //  - nter,	cdr_ns::cdr_h3_end + base, counter, dock_loop[counter],
			//		"unrefined");
		}
		else if( antibody_insert_all_h3 ) {
			dle_read_in_one_loop_library(cdr_ns::cdr_h3_begin - nter,
																									cdr_ns::cdr_h3_end + base,
																									dummy_counter, h3_library,
																									"unrefined_unclustered");
		}
		else if(truefalseoption("base"))
			H3_base_mode = true; // only calculate H3 base dihedrals
	}

	// outputs all decoys without applying filters
	files_paths::disable_output_decoy_filters = (antibody_build ||
		antibody_cluster || antibody_refine	|| output_with_no_filters ||
      refine_input_loop) && (!decoy_filter);

	save_START();
	if (mode != "score")
		set_pose_flag (false);
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_grab_top_H3_models
///
/// @brief grab the top 10 H3 loops by cluster size and then the top 10 H3
///        loops by cluster score for final refinement.
///
/// @detailed same
///
/// @param[in] loop_begin: seq numbering of H3 begin
///            loop_end: seq numbering of H3 end
///            h3_library: this is an output
///
/// @global_read complete loop file and then accordingly filterd based on
///              clustered files centers.bysize and centers.byscore
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_grab_top_H3_models(
  int loop_begin,
	int loop_end,
	std::vector< Fragment > & h3_library
)
{
	using namespace docking;
	using namespace files_paths;
	using namespace runlevel_ns;

	std::vector<Fragment> complete_loop_library;

	int dummy_counter = 1; // Start at 1 instead of zero for easy readability
	// String holder to contain the counter converted from int to string
	std::stringstream filename_counter;
	filename_counter << dummy_counter; // int counter fed to string
	// file is read in from where other contraints are supposed to exist
	std::string loop_library_filename= constraints_path
		+ protein_name + "_loop_library_0" + filename_counter.str();

	// Read the file defined by command line option
	utility::io::izstream loop_library_stream( loop_library_filename );

	// Check to see if file exists
	if ( !loop_library_stream ) {
		std::cout << "[Error]: Could not open loop library file: "
							<< loop_library_filename << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	int loop_size = (loop_end - loop_begin) + 1;
	float phi(100.0), psi(100.0), omega(100.0);

	bool end_not_reached(true);
	while(end_not_reached){
		if(loop_library_stream.eof()) {
			end_not_reached = false;
			continue;
		}
		Fragment f( loop_size );
		for ( int i = 1; i <= loop_size; ++i ) {
			loop_library_stream >> phi >> psi >> omega >> std::skipws;
			f.phi       ( i ) = phi;
			f.psi       ( i ) = psi;
			f.omega     ( i ) = omega;
			if( (i >= loop_size-3) && antibody_ns::enforce_base_strand )
				f.secstruct ( i ) = 'E'; // enforce strand in base
			else
				f.secstruct ( i ) = 'L';
		}
		complete_loop_library.push_back( f );
	}

	// filter complete loop_library
	std::string centerbysize_filename = constraints_path  + "centers.bysize";
	int cluster_counter(0);
	int decoy_number(0);
	int cluster_size(0);
	float cluster_score(0);
	int no_of_h3(0); // counter to get top 10 by cluster size & top 10 by score

	// Read the file containing cluster information
	utility::io::izstream centerbysize_stream( centerbysize_filename );

	// Check to see if file exists
	if ( !centerbysize_stream ) {
		std::cout << "[Error]: Could not open loop library file: "
							<< centerbysize_filename << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	end_not_reached = true;
	while( end_not_reached ){
		centerbysize_stream >> cluster_counter >> decoy_number >> cluster_size
												>> cluster_score >> std::skipws;
		if(centerbysize_stream.eof()) {
			end_not_reached = false;
			continue;
		}
		Fragment f( loop_size );
		f = complete_loop_library[decoy_number-1];
		h3_library.push_back( f );
		no_of_h3++;
		if(no_of_h3 == 10) {
			end_not_reached = false;
			continue;
		}
	}

	std::string centerbyscore_filename = constraints_path  + "centers.byscore";
	// Read the file containing cluster information
	utility::io::izstream centerbyscore_stream( centerbyscore_filename );

	// Check to see if file exists
	if ( !centerbyscore_stream ) {
		std::cout << "[Error]: Could not open loop library file: "
							<< centerbyscore_filename << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	end_not_reached = true; // flag to trigger next read cycle
	while( end_not_reached ){
		centerbyscore_stream >> cluster_counter >> decoy_number >> cluster_size
												 >> cluster_score >> std::skipws;
		if(centerbyscore_stream.eof()) {
				end_not_reached = false;
				continue;
		}
		Fragment f( loop_size );
		f = complete_loop_library[decoy_number-1];
		h3_library.push_back( f );
		no_of_h3++;
		if(no_of_h3 == 20) {
			end_not_reached = false;
			continue;
		}
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_refine_H3
///
/// @brief refine the members of loop library h3_library
///
/// @detailed Exhaustively refine over 100 simulations per loop and then take
///           low.
///
/// @param[in] loop_begin: seq numbering of H3 begin
///            loop_end: seq numbering of H3 end
///            framework_pose: where the loops from h3_lib are fed in and
///                            refined
///
/// @global_read makepdb_number for parallelization
///              antibody_ns::h3_library = library of fragments containing
///                                        different h3 conformations
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 03/21/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_refine_H3(
	pose_ns::Pose & framework_pose,
	int loop_begin,
	int loop_end
)
{
	using namespace pose_ns;
	using namespace makepdb_decoy;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	framework_pose.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
	// Initial fold tree
	Fold_tree const input_tree( framework_pose.fold_tree() );

	int nres(framework_pose.total_residue());
	// N_TER has been fixed
	// int local_loop_begin(loop_begin + antibody_ns::h3_nter_frag_size);
	// Kink/Extended C_TER has been fixed
	// int local_loop_end(loop_end - antibody_ns::h3_base_frag_size);
	int loop_size = ( loop_end - loop_begin ) + 1;
	// int local_loop_size = ( local_loop_end - local_loop_begin ) + 1;
	int cutpoint = loop_begin + int(loop_size/2);
	if( dle_ns::decoy_loop_cutpoint == 0 )
		dle_ns::decoy_loop_cutpoint = cutpoint;
	else
		cutpoint = dle_ns::decoy_loop_cutpoint;
	// int local_cutpoint = local_loop_begin + int( local_loop_size / 2 );

	// Setting up fold tree
	framework_pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint);
	framework_pose.set_allow_jump_move(1,false);
	// Disallowing all movements
	framework_pose.set_allow_bb_move( false );
	framework_pose.set_allow_chi_move( false );
	// Allowing movements
	for(int i = loop_begin; i <= loop_end; i++)
		framework_pose.set_allow_bb_move( i, true );

	Score_weight_map weight_map( score12 );

	Pose dummy;
	dummy = framework_pose;

	int filecounter(0);
	int const relax_cycles(1);  // 1 cycle around 10 min
                              // rms change suggests 10 cycles are enough
	                            // score change suggests 25 cycles are required
	for ( std::vector< Fragment >::const_iterator
    it=antibody_ns::h3_library.begin(), it_end = antibody_ns::h3_library.end();
      it != it_end; ++it ) {
		filecounter++;
		if( !((makepdb_number == filecounter) ||
					(int((makepdb_number%antibody_ns::h3_library.size())) ==
					 filecounter)) )
			continue;
		it->insert( dummy, loop_begin ); // inserting the loop in the decoy
		FArray1D_bool allow_repack( nres, false );
		dle_loop_neighbor(dummy, loop_begin-1, loop_end+1,
																		 allow_repack);
		framework_pose.set_allow_chi_move( allow_repack );
		dle_extreme_repack( dummy,
																			 1, // repack_cycles
																			 allow_repack,
																			 true, // rt_min
																			 true, // rotamer_trials
																			 true, // force one repack
																			 true ); // use_unbounds
		dummy.score( weight_map );
		// dummy.one_jump_tree( nres, local_loop_begin - 1, local_loop_end + 1,
		// 										 local_cutpoint );
		dummy.one_jump_tree( nres, loop_begin - 1, loop_end + 1,
												 cutpoint );
		framework_pose.set_allow_jump_move(1,false);

		Monte_carlo relaxed_h3_mc ( dummy, weight_map, 2.0 );
		for(int i = 1; i <= relax_cycles; i++){
			dummy.copy_to_misc(); // aroop_temp remove
			                      // incorporated to make following debug line work
			std::cout << "Decoy : " << I(4, 4,  makepdb_number)
								<< " Cycle : " << I(4, 4, i)
								<< " Score : " << F(8, 2,dummy.score(weight_map))
								<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
								<< " OUTER "
								<< std::endl; // aroop_temp remove
			//std::cout << "H3 Relax Cycle Number: " << I(4, i) << std::endl;
			// enabling H3 filter in relaxation
			if(dle_ns::dle_model_H3)
				dle_ns::dle_H3_filter = true;
			prof::reset( true );
			if( antibody_ns::min_base_relax ) {
				antibody_ns::min_base_relax = false; // triggers higer amplitude
				dle_loop_fa_relax( dummy, loop_begin,
																					loop_end-4 );
				antibody_ns::min_base_relax = true; // triggers lower amplitude
				dle_loop_fa_relax( dummy, loop_end-3,
																					loop_end );
			}
			else
				dle_loop_fa_relax( dummy, loop_begin,
																					loop_end );
			dle_ns::dle_H3_filter = false;
			prof::show();
			relaxed_h3_mc.boltzmann(dummy);
		}
		// Choose the lowest scoring decoy ever observed
		dummy = relaxed_h3_mc.low_pose();
	}
	framework_pose = dummy; // Assign lowest to be return

	// Restoring pose stuff
	framework_pose.set_fold_tree( input_tree ); // initial tree
	framework_pose.set_allow_jump_move( jump_move_old );//restore the jump move
	framework_pose.set_allow_bb_move( bb_move_old );
	framework_pose.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// Loop generation to create linker for Erik Fernandez
/// Loop tries to connect C-ter of L and N-ter of H chain of Ab to create FV
///
/// Command Line to invoke this function directly
/// /home/aroop/simcode/bin/rosetta.gcc aa 1flx _ -antibody_modeler -fv_linker
/// -dock_keyres loopfile -quiet -s 1flx.pdb
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_build_Fv_linker
///
/// @brief build linkers for Fv
///
/// @detailed Designed for Erik Fernandez. This is a very ambitious routine
///           that tries to come up with putative models for huge linkers
///           connecting two ends of a Fv region. Not paralleized and does not
///           use Rosetta machienary to spit out files. Just goes into one big
///           ineffective loop and then prints scores and dumps pdb files with
///           different loops
///
/// @param[in]
///
/// @global_read fragment files
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_build_Fv_linker()
{

	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace fv_loop_insert_ns;
	using namespace dle_ns;

	pose_flag_ns::pose_flag_setting = true; // switching on pose mode
	Score_weight_map centroid_weight_map( score4L );
	Score_weight_map fullatom_weight_map( score12 );

	Pose pdb_pose;
	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const read_all_chains( true );
	std::string pdb_file = files_paths::start_path + pdb_filename;
	pose_from_pdb( pdb_pose, pdb_file, fullatom, ideal_pose, read_all_chains );

	Pose pose_save; // place holder for native side chains
	pose_save = pdb_pose;
	pdb_pose.set_fullatom_flag( false );

	int nres( pdb_pose.total_residue() );
	pdb_pose.copy_to_misc();
	dle_load_and_define_loops();
	fv_loop_insert_begin = dle_loops.begin()->start();
	fv_loop_insert_end = dle_loops.begin()->stop();
	std::cout << "Loop Begin : " << fv_loop_insert_begin << std::endl;
	std::cout << "Loop End   : " << fv_loop_insert_end << std::endl;

	read_fragments_simple( dle_ns::pdb_filename.erase(4) + "_",
												 nres );
	std::cout << "Building Loops..." << std::endl;
	dle_build_one_loop( pdb_pose, fv_loop_insert_begin,
																		 fv_loop_insert_end, dock_loop[0], 0);
	pdb_pose.one_jump_tree( nres, fv_loop_insert_begin-1, fv_loop_insert_end+1,
													fv_loop_insert_begin+1); // loop tree
	pdb_pose.set_allow_jump_move(1,false); // fixed stems
	// reset torsions and bonds in the loop region
	// Insert ideal bond lengths
	if ( !dle_ns::freeze_bonds )
		pdb_pose.insert_ideal_bonds( fv_loop_insert_begin, fv_loop_insert_end );
	std::cout << "Finished Making Loops..." << std::endl;

	Pose dummy;
	dummy = pdb_pose;


	int filecounter = 1;

	for ( std::vector< Fragment >::const_iterator it=dock_loop[0].begin(),
					it_end = dock_loop[0].end(); it != it_end; ++it )
		{
			dummy = pdb_pose;
			dummy.set_fullatom_flag( false );
			it->insert( dummy, fv_loop_insert_begin );
			dummy.score( centroid_weight_map );
			// fullatom mode
			dummy.set_fullatom_flag( true, false );
			// copy back the starting sidechain first
			dummy.recover_sidechain( pose_save );
			// Repacking sidechains
			FArray1D_bool allow_repack( nres, false );
			dle_loop_neighbor(dummy, fv_loop_insert_begin,
																			 fv_loop_insert_end, allow_repack);
			// dummy.repack( allow_repack, true /*include_current*/ );
			dle_pack_with_unbound( dummy, allow_repack,
																						true /*include_current*/ );
			dle_loop_fa_relax( dummy, fv_loop_insert_begin,
																				fv_loop_insert_end );
			std::ostringstream filename;
			filename << I( 4, 4, filecounter++);
			dummy.dump_pdb(dle_ns::pdb_filename.erase(4) + "_" +
										 filename.str()+".pdb");
			std::cout << "File: " << dle_ns::pdb_filename.erase(4) +
				"_" + filename.str() << ".pdb\tScore : " <<
				dummy.score(fullatom_weight_map) << std::endl;
		}

	return;

}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_contact_score
///
/// @brief a score that rewards contact of a loop with the main body of a
///        protein
///
/// @detailed Designed for Erik Fernandez.If loop residues are within 5A of the
///           bulk of the protein they are rewarded. Penalties are awarded for
///           losing contact
///
/// @param[in] none, antibody_linker_contact_score is an output
///
/// @global_read fv_loops_insert_ns - loop definitions
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_contact_score( float & antibody_linker_contact_score )
{

	using namespace pose_ns;
	using namespace fv_loop_insert_ns;
	using namespace scorefxns;
	using namespace scores;

	if( !files_paths::antibody_modeler ) // not used anywhere other than
                                       // antibody modeling mode
		return;

	bool pose_flag_save;
	pose_flag_save = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true;

	Pose linked_fv; // fv with linker
	bool const ideal_pose( false ); // pose is non-ideal
	bool const coords_init( true ); // co-ordinates are initialized
	bool const fullatom( false ); // fullatom status
	// copy from misc
	pose_from_misc( linked_fv, fullatom, ideal_pose, coords_init );
	int const nres( linked_fv.total_residue() );

	pose_update_cendist( linked_fv );
	FArray2D_float const & cendist ( linked_fv.get_2D_score( CENDIST ) );
	// flags to indicate interacting residues
	FArray1D_bool is_interacting( nres, false );

	for ( int i=fv_loop_insert_begin+2; i<=fv_loop_insert_end-2; ++i ) {
		for ( int j=1; j<=nres; ++j ) {
			if ( j >= fv_loop_insert_begin-6 && j <= fv_loop_insert_end+6 ) continue;
			if ( is_interacting(i) && is_interacting(j) ) continue;
			if ( cendist(i,j) < 25 ) { // if under 5A
				is_interacting(i) = true;
				is_interacting(j) = true;
			}
		}
	}

	int total_int_res(0); // total number of interacting residues
	for( int i=1; i<=nres; i++ ) {
		if( is_interacting(i) )
			total_int_res++;
	}

	loop_contact_score = ( 20 - total_int_res ) * 0.5;

	// extra penalties for losing contact
	if ( total_int_res == 0 ) loop_contact_score += 2.0;
	if ( total_int_res == 1 ) loop_contact_score += 1.0;
	if ( total_int_res == 2 ) loop_contact_score += 0.5;

	antibody_linker_contact_score = loop_contact_weight * loop_contact_score;

	pose_flag_ns::pose_flag_setting = pose_flag_save;
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_CDR_H3_prediction
///
/// @brief tries to predict if a given sequence will form a kinked/extended
///        base
///
/// @detailed Predicts based on rules laid down by Hiroki Shirai
///
/// @param[in]
///
/// @global_read fragment files
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
antibody_modeling_CDR_H3_prediction(
  const pose_ns::Pose & pose,
	bool & is_kinked,
	bool & is_extended,
	std::vector <char> & aa_1name, // loop residue 1 letter codes
	const std::string & insert_region
)
{
	using namespace pose_ns;
	using namespace dle_ns;
	using namespace antibody_ns;

	int loop_begin(0), loop_end(0);

	if( insert_region == "base" ) {
		loop_begin = cdr_ns::cdr_h3_begin;
		loop_end = cdr_ns::cdr_h3_end;
	}
	else if ( insert_region == "nter" ) {
		loop_begin = cdr_ns::cdr_h3_begin-3;
		loop_end = cdr_ns::cdr_h3_end+1;
	}
	int size = ( loop_end - loop_begin ) + 1;

	Pose h3_loop;
	bool is_H3(false);

	// chop out the loop
	h3_loop.simple_fold_tree( size+3);
	h3_loop.copy_segment( size+3, pose, 1,loop_begin-2);

	// extract letter residue codes for the chopped loop
	std::vector <std::string> aa_name; // loop residue 3 letter codes
	std::string aa_3let_code; // amino acid 3 letter code
	char aa_1let_code; // amino acid 1 char code
	for(int ii = 1; ii <= size+3; ii++)	{
		name_from_num( h3_loop.res(ii), aa_3let_code);
		res1_from_num( h3_loop.res(ii), aa_1let_code);
		aa_1name.push_back( aa_1let_code);
		aa_name.push_back( aa_3let_code );
	}

	if( insert_region == "nter" )
		return( is_H3 ); // N-terminal of H3 does not have any
	                   // kink/extended annotation

	// Rule 1a for standard kink
	if (aa_name[aa_name.size()-3] != "ASP") {
		is_kinked = true;
		is_H3 = true;
	}


	// Rule 1b for standard extended form
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] != "LYS") &&
		 (aa_name[1] != "ARG") ) && (is_H3 != true)) {
		is_extended = true;
		is_H3 = true;
	}

	if(!is_H3) {
		// Rule 1b extension for special kinked form
		bool is_basic(false); // Special basic residue exception flag
		for(int ii = 2; ii <= int(aa_name.size() - 5); ii++) {
			if(aa_name[ii] == "ARG" || aa_name[ii] == "LYS") {
				is_basic = true;
				break;
			}
		}

		if(!is_basic) {
			int rosetta_number_of_L49;
			Pose misc_pose; // horrible hack to ensure res_num_from_pdb works
			// saves current misc into misc_pose
			// copies full pose into misc
			// computes residues
			// puts pack misc_pose into misc
			bool const ideal_pos( false );
			bool const coords_init( true );
			bool const fullatom( false );
			pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );
			pose.copy_to_misc();
			res_num_from_pdb_res_num_chain( rosetta_number_of_L49, 49, light_chain);
			std::string let3_code_L49;
			name_from_num( pose.res(rosetta_number_of_L49), let3_code_L49);
			misc_pose.copy_to_misc();
			if( let3_code_L49 == "ARG" || let3_code_L49 == "LYS")
				is_basic = true;
		}
		if( is_basic) {
			is_kinked = true;
			is_H3 = true;
		}
	}

	// Rule 1c for kinked form with salt bridge
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] == "LYS") ||
		 (aa_name[1] == "ARG")) && ( (aa_name[0] != "LYS") &&
     (aa_name[0] != "ARG") ) && (is_H3 != true) ) {
		is_kinked = true;
		is_H3 = true;
		if(!is_H3) {
			bool is_basic(false); // Special basic residue exception flag
			int rosetta_number_of_L46;
			Pose misc_pose; // horrible hack to ensure res_num_from_pdb works
			// saves current misc into misc_pose
			// copies full pose into misc
			// computes residues
			// puts pack misc_pose into misc
			bool const ideal_pos( false );
			bool const coords_init( true );
			bool const fullatom( false );
			pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );
			pose.copy_to_misc();
			res_num_from_pdb_res_num_chain( rosetta_number_of_L46, 46, light_chain );
			std::string let3_code_L46;
			name_from_num( pose.res(rosetta_number_of_L46), let3_code_L46);
			misc_pose.copy_to_misc();
			if( let3_code_L46 == "ARG" || let3_code_L46 == "LYS")
				is_basic = true;
			if( is_basic ) {
				is_extended = true;
				is_H3 = true;
			}
		}
	}

	// Rule 1d for extened form with salt bridge
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] == "LYS") ||
     (aa_name[1] == "ARG")) && ( (aa_name[0] == "LYS") ||
     (aa_name[0] == "ARG") ) && (is_H3 != true) ) {
		is_extended = true;
		is_H3 = true;
	}

	return( is_H3 );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_insert_ter
///
/// @brief insert "base" or "nter" fragments
///
/// @detailed insert frags and keep them fixed
///
/// @param[in] framework_pose: template antibody
///            insert_region == "nter" or "base"
///
/// @global_read antibody_ns:: H3_base_library/H3_nter_library
///
/// @global_write framework_pose updated
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_insert_ter(
	pose_ns::Pose & framework_pose,
	const std::string & insert_region
)
{
	using namespace pose_ns;
	using namespace antibody_ns;

	int loop_begin(0), loop_end(0), cutpoint(0), random_H3_ter(0);
	std::vector< Fragment >::const_iterator H3_ter;

	if( insert_region == "base" ) {
		loop_begin = cdr_ns::cdr_h3_begin;
		//cutpoint = cdr_ns::cdr_h3_begin;
		cutpoint = cdr_ns::cdr_h3_begin + 1; // aroop_temp remove
		random_H3_ter = int(ran3() * H3_base_library.size ());
		H3_ter = H3_base_library.begin();
	}
	else if( insert_region == "nter" ) {
		loop_begin = cdr_ns::cdr_h3_begin-3;
		cutpoint = cdr_ns::cdr_h3_begin+1;
		random_H3_ter = int(ran3() * H3_nter_library.size ());
		H3_ter = H3_nter_library.begin();
	}
	loop_end = cdr_ns::cdr_h3_end + 1;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	framework_pose.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
	// Initial fold tree
	Fold_tree const input_tree( framework_pose.fold_tree() );

	// allow backbone moves
	for( int i = loop_begin; i <= loop_end; i++ )
		framework_pose.set_allow_bb_move( i, true );

	int nres( framework_pose.total_residue() );
	framework_pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint, 1);
	framework_pose.set_allow_jump_move(1,false);

	// choosing a base randomly
	H3_ter = H3_ter + random_H3_ter;

	//inserting base dihedrals
	if( (cdr_ns::cdr_h3_end-2) <= cdr_ns::cdr_h3_begin )
		std::cout << "H3 LOOP IS TOO SHORT: CAN NOT USE N-TERM INFORMATION"
							<< std::endl;
	else if( insert_region == "base")
		H3_ter->insert( framework_pose, cdr_ns::cdr_h3_end-2 );
	else if( insert_region == "nter")
		H3_ter->insert( framework_pose, cdr_ns::cdr_h3_begin-3 );

	// Restoring pose stuff
	framework_pose.set_fold_tree( input_tree ); // initial tree
	framework_pose.set_allow_jump_move( jump_move_old );//restore the jump move
	framework_pose.set_allow_bb_move( bb_move_old );
	framework_pose.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_read_H3_ter_fragment
///
/// @brief read H3_CTERM or H3_NTERM and keep pertinent fragments
///
/// @detailed Depending if the insert_region is "base" or "nter" different
///           fragments are chosed keeping into consideration sequence of
///           fragment and template, base topology - kinked/extended, H3
///           length match etc
///
/// @param[in] insert_region == "base" or "nter"
///
/// @global_read H3_CTERM/H3_NTERM fragment files on disk
///              h3 stem fragment size from antibody_ns
///              select_all_kink_match_from antibody_ns
///
/// @global_write H3_base_library / H3_nter_library in antibody_ns
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 06/26/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_read_H3_ter_fragment(
  pose_ns::Pose & pose_in,
	const std::string & insert_region
)
{
	using namespace files_paths;
	using namespace runlevel_ns;
	using namespace antibody_ns;

	bool is_kinked( false );
	bool is_extended( false );
	std::vector <char> aa_1name; // loop residue 1 letter codes
	antibody_modeling_CDR_H3_prediction( pose_in, is_kinked, is_extended,
																			 aa_1name, insert_region);

	// used only when no length & kink match are found
	std::vector< Fragment > H3_base_library_seq_kink;

	std::string H3_ter_library_filename;
	if( insert_region == "base") {
		// file is read in from where other contraints are supposed to exist
		H3_ter_library_filename = constraints_path  + "H3_CTERM";
	}
	else if( insert_region == "nter") {
		// file is read in from where other contraints are supposed to exist
		H3_ter_library_filename = constraints_path  + "H3_NTERM";
	}

	// Read the file defined by command line option
	utility::io::izstream H3_ter_library_stream( H3_ter_library_filename );

	// Check to see if file exists
	if ( !H3_ter_library_stream ) {
		std::cout << "[Error]: Could not open H3 base library file: "
							<< H3_ter_library_filename << std::endl
							<< "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	std::string pdb_name;
	std::string res_no;
	char res_name;
	float phi(0.0);
	float psi(0.0);
	float omega(0.0);
	int H3_length(0);
	float resolution(0.0);
	std::string base_type;

	int pdb_H3_length = (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1;

	bool end_not_reached(true);
	while(end_not_reached){
		bool seq_match( true );
		bool kink_match( false );
		int h3_stem_frag_size = (insert_region == "nter") ? h3_nter_frag_size :
			                      h3_base_frag_size;
		Fragment f( h3_stem_frag_size );
		for ( int i = 1; i <= h3_stem_frag_size; ++i ) {
			H3_ter_library_stream >> pdb_name
														>> res_no
														>> res_name
														>> omega
														>> phi
														>> psi
														>> H3_length
														>> resolution
														>> base_type
														>> std::skipws;
			if(H3_ter_library_stream.eof()) {
				end_not_reached = false;
				break;
			}
			if( insert_region == "base") {
				if( res_name != aa_1name[aa_1name.size() - 5 + i] )
					seq_match = false;
			}
			else if( insert_region == "nter") {
				if( res_name != aa_1name[i + 1] )
					seq_match = false;
			}
			f.phi       ( i ) = phi;
			f.psi       ( i ) = psi;
			f.omega     ( i ) = omega;
			if( insert_region == "base" && enforce_base_strand ) {
				// activated by command line -base_strand
				// sets strand secondary structure annotation
				// for the base region
				f.secstruct ( i ) = 'E';
			}
			else
				f.secstruct ( i ) = 'L';
		}
		if( insert_region == "base") {
			if( is_kinked && base_type == "KINK" )
				kink_match = true;
			else if( is_extended && base_type == "EXTENDED" )
				kink_match = true;
			else if( !is_kinked && !is_extended && base_type == "NEUTRAL" )
				kink_match = true;
			// choose all matching bases if appropriate flag set
			if( select_all_kink_match ) {
				if(end_not_reached && kink_match == true)
					H3_base_library.push_back( f );
			}
			else if(end_not_reached && (H3_length == pdb_H3_length) /* &&
							seq_match == true */ && kink_match == true)
				H3_base_library.push_back( f );
			if(end_not_reached /* && (H3_length == pdb_H3_length) */
				 && (seq_match == true) && kink_match == true)
				H3_base_library_seq_kink.push_back( f );
		}
	  else if( insert_region == "nter")
			if(end_not_reached && (seq_match == true) && (H3_length== pdb_H3_length))
				H3_nter_library.push_back( f );
	}

	H3_ter_library_stream.close();
	H3_ter_library_stream.clear();

	// if no match found based on sequence and kink match criterion
	// then choose based on size and kink match criterion
	if( H3_base_library.size() == 0 )
		H3_base_library = H3_base_library_seq_kink;

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_load_native
///
/// @brief reads in native and cdr definitions and stores in native_cdr
///        namespace
///
/// @detailed Stores the native for later comparison of CDR-H3 rmsg to native
///           in the score file. This function is enabled if the command line
///           option -native is specified. A suboption can also be enabled by
///           command line option -idealize_native. This results in the
///           idealization of the native CDR-H3 as well as repacked ideal nat
///
/// @param[in] reads in native pdb from file on disk
///
/// @global_read
///
/// @global_write fills up various stuff in native_cdr namespace
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/16/2007
///
/// @last_modified 01/30/2008
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_load_native()
{
	using namespace pose_ns;
	using namespace native_cdr;

	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const coords_init( true );
	bool const read_all_chains( true );
	bool local_multi_chain_setting = files_paths::multi_chain;
	// place to store current misc to be able to reset misc
	// because pose_from_pdb disrupts current misc status
	Pose current_pose_in_misc;
	pose_from_misc( current_pose_in_misc, fullatom, ideal_pose, coords_init );
	pose_from_pdb( native_fv, files_paths::start_path + native_pdb_filename,
								 fullatom, ideal_pose, read_all_chains );
	files_paths::multi_chain = local_multi_chain_setting;
	native_fv.copy_to_misc(); // explicit copy to misc

	Pose start_native_pose;
	start_native_pose = native_fv;

	int pdb_loop_begin(0), pdb_loop_end(0);
	char pdb_chain_id = '-';

	// storing native l1 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "l1",
		pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_l1_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_l1_end, pdb_loop_end, pdb_chain_id );

	// storing native l2 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "l2",
    pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_l2_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_l2_end, pdb_loop_end, pdb_chain_id );

	// storing native l3 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "l3",
    pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_l3_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_l3_end, pdb_loop_end, pdb_chain_id );

	// storing native h1 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "h1",
    pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_h1_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_h1_end, pdb_loop_end, pdb_chain_id );

	// storing native h2 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "h2",
    pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_h2_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_h2_end, pdb_loop_end, pdb_chain_id );

	// storing native h3 information
	antibody_modeling_convert_to_sequential_res_from_chothia_res( "h3",
    pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( native_h3_begin,pdb_loop_begin,pdb_chain_id);
	res_num_from_pdb_res_num_chain( native_h3_end, pdb_loop_end, pdb_chain_id );
	// strange number in h3 is required because in some CDR-H3's
	// the last residue of H3, viz 102:H is missing
	native_h3_end = native_h3_end - 1;

	// storing chain_end information
	// (to make this work had to make changes to enzyme.cc)
	native_chain_L_end = docking::part_end(1);
	// hack to save L_end during snugdock
	if( docking::antibody_modeling_dock_n_snug_flag ) {
		int const start_L_end( antibody_ns::L_end );
		antibody_ns::L_end = 0;
		antibody_modeling_detect_L_end( native_fv );
		native_chain_L_end = antibody_ns::L_end;
		antibody_ns::L_end = start_L_end;
	}

	// storing framework definitions
	antibody_modeling_define_framework( native_fv );

	// output native scores
	if( !docking::antibody_modeling_dock_n_snug_flag )
	{
		int nres = native_fv.total_residue();
		// compute native scores
		Score_weight_map weight_map( score12 );
		native_fv.score( weight_map );

		// output native score
		int file_extension_position = native_pdb_filename.find_last_of(".");
		std::string native_pdb_code = native_pdb_filename.substr(0,
																	file_extension_position);

		// Try reading native if it exists
		utility::io::izstream native_stream( native_pdb_code + "_native.pdb" );

		// If native already created
		if( native_stream )
			return;

		scorefile_output( native_pdb_code + "_native.pdb", "native", fullatom );
		// output native pdb
		native_fv.dump_scored_pdb( native_pdb_code + "_native.pdb", weight_map );

		Pose idealized_native, idealize_ccd;
		if( dle_ns::idealize_native ) {
			idealized_native = native_fv;
			int cutpoint = (native_h3_begin-1) + int((((native_h3_end + 1) -
																								 (native_h3_begin-1)) + 1)/2);
			idealized_native.one_jump_tree( nres, native_h3_begin - 2, native_h3_end
																			+ 2, cutpoint );
			idealized_native.set_allow_jump_move(1,false); // fixed stems

			for ( int i=1; i<= nres; ++i ) {
				if( (i >= native_h3_begin - 1) && (i <= native_h3_end + 1) ) {
					idealized_native.set_allow_bb_move(i, true);
					idealized_native.set_allow_chi_move(i, true);
				}
				else {
					idealized_native.set_allow_bb_move(i, false);
					idealized_native.set_allow_chi_move(i, false);
				}
			}
			idealize_pose( idealized_native,
										 false ); // ignore_allow_move

			for ( int i=1; i<= nres; ++i )
				idealized_native.set_allow_chi_move(i, true);

			// score idealized native
			idealized_native.score( weight_map );
			scorefile_output( native_pdb_code + "_ideal.pdb", "ideal_native",
												fullatom );
			// output idealized native pdb
			idealized_native.dump_scored_pdb( native_pdb_code + "_ideal.pdb",
																				weight_map );

			// close idealized native
			idealize_ccd = idealized_native;
			float fdev, bdev, torsion_delta, rama_delta;
			// changes cycles from 100 to 500
			fast_ccd_loop_closure( idealize_ccd, native_h3_begin-1, native_h3_end+1,
        cutpoint, 100, 0.01, true, 2.0, 10, 50, 75, fdev, bdev,
          torsion_delta, rama_delta);
			idealize_ccd.score( weight_map );
			scorefile_output( native_pdb_code + "_ideal_ccd.pdb", "ideal_ccd",
												fullatom );
			// output idealized native pdb
			idealize_ccd.dump_scored_pdb( native_pdb_code + "_ideal_ccd.pdb",
																		weight_map );
		}

		// repack native
		FArray1D_bool allow_repack( nres, true );
		if( !runlevel_ns::benchmark )
			dle_extreme_repack( native_fv,
																				 10, // repack cycles
																				 allow_repack,
																				 true, // rt_min
																				 true,// rotamer_trials
																				 false ); // force_one_repack
		native_fv.score( weight_map );
		// output repacked native score
		scorefile_output(native_pdb_code + "_rpk_native.pdb", "repacked_native",
										 fullatom );
		// output repacked native pdb
		native_fv.dump_scored_pdb( native_pdb_code + "_rpk_native.pdb",
															 weight_map );

		if( dle_ns::idealize_native ) {
			// repack idealized native
			dle_extreme_repack( idealized_native,
																				 10, // repack cycles
																				 allow_repack,
																				 true, // rt_min
																				 true,// rotamer_trials
																				 false ); // force_one_repack
			// score idealized native
			idealized_native.score( weight_map );
			scorefile_output( native_pdb_code + "_ideal_rpk.pdb",
												"ideal_rpk_native", fullatom );
			// output idealized native pdb
			idealized_native.dump_scored_pdb( native_pdb_code +
																				"_ideal_rpk.pdb", weight_map );

			// repack idealized ccd native
			dle_extreme_repack( idealize_ccd,
																				 10, // repack cycles
																				 allow_repack,
																				 true, // rt_min
																				 true, // rotamer_trials
																				 false ); // force_one_repack
			// score idealized native
			idealize_ccd.score( weight_map );
			scorefile_output( native_pdb_code + "_ideal_ccd_rpk.pdb",
												"ideal_ccd_rpk", fullatom );
			// output idealized native pdb
			idealize_ccd.dump_scored_pdb( native_pdb_code +
																		"_ideal_ccd_rpk.pdb", weight_map );
		}

		// resetting native
		native_fv = start_native_pose;
	}

	current_pose_in_misc.copy_to_misc(); // reseting misc to original state
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_define_framework
///
/// @brief stores the sequential residue numbers of the framework
///
/// @detailed stores the sequential residue numbers of both the light chain and
///           heavy chain framework. The framework consists of non-svr and non-
///           cdr regions. These are used later for alignment for rms
///           computation
///           Non SVR region - non CDR region
///           Light Chain:
///           ZONE 01 L  4-L  6
///           ZONE 02 L 10-L 23
///           ZONE 03 L 35-L 38
///           ZONE 04 L 45-L 49
///           ZONE 05 L 57-L 66
///           ZONE 06 L 71-L 88
///           ZONE 07 L 98-L104
///           Heavy Chain:
///           ZONE 01 H  5-H  6
///           ZONE 02 H 10-H 25
///           ZONE 03 H 36-H 39
///           ZONE 04 H 46-H 49
///           ZONE 05 H 66-H 94
///           ZONE 06 H103-H110
///
/// @param[in] pose_in: Pose based on which the framework residues are
///            computed
///
/// @global_read
///
/// @global_write fills up various stuff in native_cdr namespace
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/09/2007
///
/// @last_modified 01/28/2008
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_define_framework(
  pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace native_cdr;

	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const coords_init( true );
	bool local_multi_chain_setting = files_paths::multi_chain;
	// place to store current misc to be able to reset misc
	// because pose_from_pdb disrupts current misc status
	Pose current_pose_in_misc;
	pose_from_misc( current_pose_in_misc, fullatom, ideal_pose, coords_init );
	files_paths::multi_chain = local_multi_chain_setting;
	pose_in.copy_to_misc(); // explicit copy to misc

	// storing native L frmk zone 1 information
	res_num_from_pdb_res_num_chain( L_framework_begin[0], 4,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[0], 6, 'L' );

	// storing native L frmk zone 2 information
	res_num_from_pdb_res_num_chain( L_framework_begin[1], 10,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[1], 23, 'L' );

	// storing native L frmk zone 3 information
	res_num_from_pdb_res_num_chain( L_framework_begin[2], 35,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[2], 38, 'L' );

	// storing native L frmk zone 4 information
	res_num_from_pdb_res_num_chain( L_framework_begin[3], 45,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[3], 49, 'L' );

	// storing native L frmk zone 5 information
	res_num_from_pdb_res_num_chain( L_framework_begin[4], 57,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[4], 66, 'L' );

	// storing native L frmk zone 6 information
	res_num_from_pdb_res_num_chain( L_framework_begin[5], 71,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[5], 88, 'L' );

	// storing native L frmk zone 7 information
	res_num_from_pdb_res_num_chain( L_framework_begin[6], 98,'L');
	res_num_from_pdb_res_num_chain( L_framework_end[6], 104, 'L' );

	// storing native H frmk zone 1 information
	res_num_from_pdb_res_num_chain( H_framework_begin[0], 5,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[0], 6, 'H' );

	// storing native H frmk zone 2 information
	res_num_from_pdb_res_num_chain( H_framework_begin[1], 10,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[1], 25, 'H' );

	// storing native H frmk zone 3 information
	res_num_from_pdb_res_num_chain( H_framework_begin[2], 36,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[2], 39, 'H' );

	// storing native H frmk zone 4 information
	res_num_from_pdb_res_num_chain( H_framework_begin[3], 46,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[3], 49, 'H' );

	// storing native H frmk zone 5 information
	res_num_from_pdb_res_num_chain( H_framework_begin[4], 66,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[4], 94, 'H' );

	// storing native H frmk zone 6 information
	res_num_from_pdb_res_num_chain( H_framework_begin[5], 103,'H');
	res_num_from_pdb_res_num_chain( H_framework_end[5], 110, 'H' );

	current_pose_in_misc.copy_to_misc(); // reseting misc to original state

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_rmsg
///
/// @brief compute the rms global for CDR-H3 after superimposition of heavy
///        chain framework
///
/// @detailed the first step is the alignment of the framework of the native
///           heavy and the current decoy. Then the rms of the CA atoms of
///           the loop are computed to find out the global rms of the CDR-H3
///
/// @param[in]
///
/// @global_read native_cdr from dle_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/16/2007
///
/// @last_modified 02/20/2007
///////////////////////////////////////////////////////////////////////////////
float
antibody_modeling_h3_rmsg()
{
	using namespace pose_ns;
	using namespace native_cdr;

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int const nres( pose_decoy.total_residue() ); // Total residues
	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(6); // Heavy chain broken into six alignable
                                // regions by svrs and cdrs
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	// defining alignable regions
	for( int i = 0; i < 6; i++ ) {
		alignstart(i+1) = H_framework_begin[i];
		alignend(i+1) = H_framework_end[i] ;
	}

	/* // defining alignable regions
	alignstart(1) = native_chain_L_end + 1;
	alignend(1) = native_h1_begin - 1;
	alignstart(2) = native_h1_end + 1;
	alignend(2) = native_h2_begin - 1;
	alignstart(3) = native_h2_end + 1;
	alignend(3) = native_h3_begin - 1;
	alignstart(4) = native_h3_end + 2;
	alignend(4) = nres; */

	// copy coords into double arrays
	FArray3D_float full1 ( pose_decoy.full_coord() );
	FArray3D_float Epos1 ( pose_decoy.Eposition() );
	FArray3D_float const & Epos2 ( native_fv.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
								alignstart,alignend,aligned_regions,alignstart,
								alignend,local_rms);

	Pose decoy_copy; // copy needed for reorienting, cannot re-orient input
	                 // decoy2 have enforced const decoy2 to prevent accidental
                   // re-orient
	decoy_copy = pose_decoy;

	decoy_copy.set_coords( false /*ideal_pos*/, Epos1, full1 );
	decoy_copy.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	// grab the Eposition array from orientalbe pose
	FArray3D_float const decoy_copy_coord(decoy_copy.Eposition());
	// grab the Eposition array from native pose
	FArray3D_float const native_fv_coord(native_fv.Eposition());

	float loop_rmsg(0.00); // Global Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=cdr_ns::cdr_h3_begin,j = native_h3_begin;
				i <= cdr_ns::cdr_h3_end; ++i,++j ) {
		++count;
		loop_rmsg += distance_squared(numeric::xyzVector_float(
                 &decoy_copy_coord(1,CA,i)), numeric::xyzVector_float(
                 &native_fv_coord(1,CA,j)));
	}

	loop_rmsg  = std::sqrt( loop_rmsg  / count  );

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( loop_rmsg );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_L_rms
///
/// @brief compute the rms of the L chain CA atoms to respective native atoms
///
/// @detailed the first step is the alignment of the framework of the native
///           heavy and the current decoy. Then the rms of the CA atoms of
///           the L chain are computed to find out the rms
///
/// @param[in]
///
/// @global_read native_cdr from dle_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/19/2007
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
float
antibody_modeling_L_rms()
{
	using namespace pose_ns;
	using namespace native_cdr;

	// in the absence of local perturbation between L & H chain
	// if( !antibody_ns::snug_fit )
	//  return( 0.00 );

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int const nres( pose_decoy.total_residue() ); // Total residues
	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(6); // Heavy chain broken into six alignable
                                // regions by svrs and cdrs
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	// defining alignable regions
	for( int i = 0; i < 6; i++ ) {
		alignstart(i+1) = H_framework_begin[i];
		alignend(i+1) = H_framework_end[i] ;
	}

	// find out the rms regions
	FArray1D_int rmsstart, rmsend;
	int const rms_regions(7); // Light chain broken into seven alignable
                            // regions by svrs and cdrs
	rmsstart.dimension( rms_regions );
	rmsend.dimension( rms_regions );

	// defining alignable regions
	for( int i = 0; i < 7; i++ ) {
		rmsstart(i+1) = L_framework_begin[i];
		rmsend(i+1) = L_framework_end[i] ;
	}

	/*// defining alignable regions
	alignstart(1) = native_chain_L_end + 1;
	alignend(1) = native_h1_begin - 1;
	alignstart(2) = native_h1_end + 1;
	alignend(2) = native_h2_begin - 1;
	alignstart(3) = native_h2_end + 1;
	alignend(3) = native_h3_begin - 1;
	alignstart(4) = native_h3_end + 2;
	alignend(4) = nres;*/

	// copy coords into double arrays
	FArray3D_float full1 ( pose_decoy.full_coord() );
	FArray3D_float Epos1 ( pose_decoy.Eposition() );
	FArray3D_float const & Epos2 ( native_fv.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float L_rms(0.00);
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
								alignstart,alignend,rms_regions,rmsstart,
								rmsend,L_rms);

	Pose decoy_copy; // copy needed for reorienting, cannot re-orient input
	                 // decoy2 have enforced const decoy2 to prevent accidental
                   // re-orient
	decoy_copy = pose_decoy;

	decoy_copy.set_coords( false /*ideal_pos*/, Epos1, full1 );
	decoy_copy.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	// grab the Eposition array from orientalbe pose
	FArray3D_float const decoy_copy_coord(decoy_copy.Eposition());
	// grab the Eposition array from native pose
	FArray3D_float const native_fv_coord(native_fv.Eposition());

	/*
	float L_rms(0.00); // Rms of Light Chain (L)
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=1,j = 1; j <= native_chain_L_end; ++i,++j ) {
		++count;
		L_rms += distance_squared(numeric::xyzVector_float(
					   &decoy_copy_coord(1,CA,i)), numeric::xyzVector_float(
             &native_fv_coord(1,CA,j)));
	}

	for( int i = 0; i < 7; i++ ){
		for( int j = L_framework_begin[i]; j <= L_framework_end[i]; j++ ) {
			++count;
			L_rms += distance_squared(numeric::xyzVector_float(
							 &decoy_copy_coord(1,CA,j)), numeric::xyzVector_float(
    					 &native_fv_coord(1,CA,j)));
		}
	}

	L_rms  = std::sqrt( L_rms  / count  );
	*/

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( L_rms );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_base_rmsg
///
/// @brief compute the rms global for base region of CDR-H3 after superposition
///        of heavy chain framework, ie the kink/extended region
///
/// @detailed the first step is the alignment of the framework of the native
///           heavy and the current decoy. Then the rms of the CA atoms of
///           the loop are computed to find out the global rms of base
///
/// @param[in]
///
/// @global_read native_cdr from dle_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2007
///
/// @last_modified 04/27/2007
///////////////////////////////////////////////////////////////////////////////
float
antibody_modeling_h3_base_rmsg()
{
	using namespace pose_ns;
	using namespace native_cdr;

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int const nres( pose_decoy.total_residue() ); // Total residues
	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(6); // Heavy chain broken into six alignable
                                // regions by svrs and cdrs
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	// defining alignable regions
	for( int i = 0; i < 6; i++ ) {
		alignstart(i+1) = H_framework_begin[i];
		alignend(i+1) = H_framework_end[i] ;
	}

	/*// defining alignable regions
	alignstart(1) = native_chain_L_end + 1;
	alignend(1) = native_h1_begin - 1;
	alignstart(2) = native_h1_end + 1;
	alignend(2) = native_h2_begin - 1;
	alignstart(3) = native_h2_end + 1;
	alignend(3) = native_h3_begin - 1;
	alignstart(4) = native_h3_end + 2;
	alignend(4) = nres;*/

	// copy coords into double arrays
	FArray3D_float full1 ( pose_decoy.full_coord() );
	FArray3D_float Epos1 ( pose_decoy.Eposition() );
	FArray3D_float const & Epos2 ( native_fv.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
								alignstart,alignend,aligned_regions,alignstart,
								alignend,local_rms);

	Pose decoy_copy; // copy needed for reorienting, cannot re-orient input
	                 // decoy2 have enforced const decoy2 to prevent accidental
                   // re-orient
	decoy_copy = pose_decoy;

	decoy_copy.set_coords( false /*ideal_pos*/, Epos1, full1 );
	decoy_copy.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	// grab the Eposition array from orientalbe pose
	FArray3D_float const decoy_copy_coord(decoy_copy.Eposition());
	// grab the Eposition array from native pose
	FArray3D_float const native_fv_coord(native_fv.Eposition());

	float base_rmsg(0.00); // Global Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=cdr_ns::cdr_h3_end-2,j = native_h3_end-2;
				i <= cdr_ns::cdr_h3_end+1; ++i,++j ) {
		++count;
		base_rmsg += distance_squared(numeric::xyzVector_float(
                 &decoy_copy_coord(1,CA,i)), numeric::xyzVector_float(
                 &native_fv_coord(1,CA,j)));
	}

	base_rmsg  = std::sqrt( base_rmsg  / count  );

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( base_rmsg );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_non_base_rmsg
///
/// @brief compute the rms global for non_base region of CDR-H3 after
///        superimposition of heavy chain framework, ie the non
///        kink/extended region
///
/// @detailed the first step is the alignment of the framework of the native
///           heavy and the current decoy. Then the rms of the CA atoms of
///           the loop are computed to find out the global rms of non_base
///
/// @param[in]
///
/// @global_read native_cdr from dle_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2007
///
/// @last_modified 04/27/2007
///////////////////////////////////////////////////////////////////////////////
float
antibody_modeling_h3_non_base_rmsg()
{
	using namespace pose_ns;
	using namespace native_cdr;

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int const nres( pose_decoy.total_residue() ); // Total residues
	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(6); // Heavy chain broken into six alignable
                                // regions by svrs and cdrs
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	// defining alignable regions
	for( int i = 0; i < 6; i++ ) {
		alignstart(i+1) = H_framework_begin[i];
		alignend(i+1) = H_framework_end[i] ;
	}

	/*// defining alignable regions
	alignstart(1) = native_chain_L_end + 1;
	alignend(1) = native_h1_begin - 1;
	alignstart(2) = native_h1_end + 1;
	alignend(2) = native_h2_begin - 1;
	alignstart(3) = native_h2_end + 1;
	alignend(3) = native_h3_begin - 1;
	alignstart(4) = native_h3_end + 2;
	alignend(4) = nres; */

	// copy coords into double arrays
	FArray3D_float full1 ( pose_decoy.full_coord() );
	FArray3D_float Epos1 ( pose_decoy.Eposition() );
	FArray3D_float const & Epos2 ( native_fv.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
								alignstart,alignend,aligned_regions,alignstart,
								alignend,local_rms);

	Pose decoy_copy; // copy needed for reorienting, cannot re-orient input
	                 // decoy2 have enforced const decoy2 to prevent accidental
                   // re-orient
	decoy_copy = pose_decoy;

	decoy_copy.set_coords( false /*ideal_pos*/, Epos1, full1 );
	decoy_copy.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	// grab the Eposition array from orientalbe pose
	FArray3D_float const decoy_copy_coord(decoy_copy.Eposition());
	// grab the Eposition array from native pose
	FArray3D_float const native_fv_coord(native_fv.Eposition());

	float non_base_rmsg(0.00); // Global Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=cdr_ns::cdr_h3_begin,j = native_h3_begin;
				i <= cdr_ns::cdr_h3_end-3; ++i,++j ) {
		++count;
		non_base_rmsg += distance_squared(numeric::xyzVector_float(
                 &decoy_copy_coord(1,CA,i)), numeric::xyzVector_float(
                 &native_fv_coord(1,CA,j)));
	}

	non_base_rmsg  = std::sqrt( non_base_rmsg  / count  );

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( non_base_rmsg );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_change_to_all_cdr_fold_tree
///
/// @brief changes the current fold tree of a pose to one that encompasses all
///        the cdr loops
///
/// @detailed takes into account that cdr l1 to h3 have been put in by placing
///           cutpoints in the middle of each. Also sets jumps to false since
///           the stems of the loops, ie the jumpoints do not move relative to
///           each other
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/28/2007
///
/// @last_modified 08/10/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_change_to_all_cdr_fold_tree(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;

	pose_in.simple_fold_tree( pose_in.total_residue() );
	Fold_tree f;
	f.clear();

	Loops tmp_loops = antibody_ns::all_cdrs;
	tmp_loops.sequential_order();

	int jump_num = 0;
	for( Loops::const_iterator it=tmp_loops.begin(), it_end=tmp_loops.end(),
				 it_next; it < it_end; ++it ) {
		it_next = it;
		it_next++;

		if( it == tmp_loops.begin() )
			f.add_edge( 1, it->start()-1, pose_param::PEPTIDE );

		jump_num++;
		f.add_edge( it->start()-1, it->stop()+1, jump_num );
		f.add_edge( it->start()-1, it->cut(),  pose_param::PEPTIDE );
		f.add_edge( it->cut()+1, it->stop()+1, pose_param::PEPTIDE );
		if( it == (it_end-1) )
			f.add_edge( it->stop()+1, pose_in.total_residue(), pose_param::PEPTIDE);
		else
			f.add_edge( it->stop()+1, it_next->start()-1, pose_param::PEPTIDE );
	}

	f.reorder(1);
  pose_in.set_fold_tree(f);

	for( int i = 1; i <= f.get_num_jump(); i++ )
		pose_in.set_allow_jump_move( i, false );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_store_unaligned_framework_info
///
/// @brief store the heavy chain and light chain pdbs from which the respective
///        templates were created
///
/// @detailed stores the information of the aforementioned pdbs. These are
///           used so that they can be used later to compute the loop_begin-1
///           residue's psi and omega angle. Otherwise this information is lost
///           because the coordinates of the loop residues are blanked out
///
/// @param[in] pdb on disc which are named as default hfr.pdb for heavy chain
///            framework and lfr.pdb for light chain framework
///
/// @global_read antibody_ns:: file names of respective files
///
/// @global_write antibody_ns:; stores these poses
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/02/2007
///
/// @last_modified 01/30/2008
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_store_unaligned_framework_info()
{
	using namespace pose_ns;
	using namespace antibody_ns;
	using namespace files_paths;

	if( !( graft_l1 || graft_l2 ||graft_l3 || graft_h1 || graft_h2 || model_h3)
			&& dle_ns::refine_input_loop )
		return;

	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const coords_init( true ); // co-ordinates are initialized
	bool const read_all_chains( true );
	bool const initial_multi_chain_setting = files_paths::multi_chain;
	files_paths::multi_chain = true;
	Pose saved_misc;
	 // copy from misc
	pose_from_misc( saved_misc, fullatom, ideal_pose, coords_init );

	if( graft_l1 || graft_l2 || graft_l3 )
		pose_from_pdb( unaligned_framework_light_monomer,
									 start_path + unaligned_framework_light_monomer_filename,
									 fullatom, ideal_pose, read_all_chains );
	files_paths::multi_chain = true; // resetting

	if( graft_h1 || graft_h2 || model_h3 )
	pose_from_pdb( unaligned_framework_heavy_monomer,
								 start_path + unaligned_framework_heavy_monomer_filename,
								 fullatom, ideal_pose, read_all_chains );

	// store h3 info for unaligned heavy chain monomer
	std::string cdr_loop_name = "h3";
	int pdb_loop_begin(0), pdb_loop_end(0);
	char pdb_chain_id = '-';
	antibody_modeling_convert_to_sequential_res_from_chothia_res( cdr_loop_name,
	  pdb_chain_id, pdb_loop_begin, pdb_loop_end);
	res_num_from_pdb_res_num_chain( unaligned_h3_loop_begin, pdb_loop_begin,
																	pdb_chain_id );
	std::cout << cdr_loop_name << " Unaligned loop begin source : "
						<< unaligned_h3_loop_begin << std::endl;
	res_num_from_pdb_res_num_chain( unaligned_h3_loop_end, pdb_loop_end,
																	pdb_chain_id );
	// explation provided in convert_to_sequential_res_from_chothia_res
	unaligned_h3_loop_end = unaligned_h3_loop_end - 1;

	std::cout << cdr_loop_name << " Unaligned loop end source : "
						<< unaligned_h3_loop_end << std::endl;

	// Resetting stuff
	files_paths::multi_chain = initial_multi_chain_setting;
	saved_misc.copy_to_misc();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_graft_cdr_by_superimposition
///
/// @brief graft a cdr by first aligning the stem regions of the cdr and then
///        pasting it in
///
/// @detailed This is in contrast to the other method wherein bond lengths and
///           bond angles are copied over from the cdr pose and then the
///           framework is refolded. Here after aligning, the coordinates are
///           copied from the loop into the template for the loop region. Then
///           the backbone torsions etc are updated from the cartesian
///           coordinates. Latest developments incorporate fullatom grafting
///
/// @param[in] framework_pose = template (this is the output)
///            cdr_pose = pose containing the cdr loop
///            framework_loop_begin/end = seq nos of the loop in the template
///            cdr_loop_begin/end = seq nos of the loop in the loop pose
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/14/2007
///
/// @last_modified 08/10/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_graft_cdr_by_superimpostion(
	pose_ns::Pose & framework_pose,
	pose_ns::Pose & cdr_pose,
	int framework_loop_begin,
	int framework_loop_end,
	int cdr_loop_begin,
	int cdr_loop_end
)
{
	using namespace pose_ns;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	framework_pose.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
	// storing docking fold tree
	Fold_tree const in_tree( framework_pose.fold_tree() );

	int const nres( framework_pose.total_residue() ); // Total residues
	int loop_size = ( framework_loop_end - framework_loop_begin ) + 1;
	int cdr_size = (cdr_loop_end - cdr_loop_begin ) + 1;
	assert( loop_size == cdr_size );

	// creation of chimeric pose is a sort of hack
	// one is interested only in the loop region of this chimeric pose
	// the chimeric pose merely provides a pose of a similar size,
	// which translates to Eposition arrays of similar dimension
	// Eposition with identical dimensions are required by orient_region
	Pose chimeric_pose, orientable_pose;
	orientable_pose = framework_pose;
	chimeric_pose = framework_pose;
	orientable_pose.simple_fold_tree(nres);
	chimeric_pose.simple_fold_tree(nres);
	chimeric_pose.copy_segment(loop_size+2, cdr_pose, framework_loop_begin-1,
														 cdr_loop_begin-1, false );

	// find out the aligned regions
	FArray1D_int alignstart, alignend;
	int const aligned_regions(2); // 2 Stems on either side for one loop
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	alignstart(1) = framework_loop_begin - 1;
	alignend(1) = (framework_loop_begin - 1) + antibody_ns::deep_graft;
	alignstart(2) = (framework_loop_end + 1) - antibody_ns::deep_graft;
	alignend(2) = framework_loop_end + 1;

	// find out the rms regions
	// this is NOT used, but then orient_rms needs these variables
	FArray1D_int rms_start, rms_end;
	int const rms_regions(2); // 2 Stems on either side for one loop
	rms_start.dimension( rms_regions );
	rms_end.dimension( rms_regions );
	rms_start(1) = 1;
	rms_end(1) = framework_loop_begin - 2;
	rms_start(2) = framework_loop_end + 2;
	rms_end(2) = nres;

	// copy coords into double arrays
	FArray3D_float orientable_full ( orientable_pose.full_coord() );
	FArray3D_float orientable_Epos ( orientable_pose.Eposition() );
	FArray3D_float chimeric_Epos ( chimeric_pose.Eposition() );
	FArray3D_float const & const_chimeric_Epos ( chimeric_pose.Eposition() );
	FArray3D_float chimeric_full ( chimeric_pose.full_coord() );

	float nter_rms;
	// align the stems of the loop
	orient_region(nres,orientable_Epos,orientable_full,const_chimeric_Epos,
		aligned_regions, alignstart, alignend, rms_regions, rms_start,
      rms_end,nter_rms);

	// oriented pose created from oriented coords
	orientable_pose.set_coords( false, orientable_Epos, orientable_full );
	// update oriented pose with loop coords
	orientable_pose.set_segment_full_coord( loop_size, framework_loop_begin,
		chimeric_full(1,1,framework_loop_begin));
	// update torsions from new coords
	orientable_pose.update_backbone_bonds_and_torsions();

	// Setting up fold tree for copying into framework pose
	int flanking_region(5);
	int local_begin = framework_loop_begin - flanking_region;
	int local_end = framework_loop_end + flanking_region;
	int local_size = (local_end - local_begin) + 1;
	int local_cutpoint = local_begin + int(local_size/2);

 	framework_pose.one_jump_tree( nres, local_begin-1,
																local_end+1, local_cutpoint,1);
	framework_pose.set_allow_jump_move(1,false);
	framework_pose.copy_segment(local_size, orientable_pose, local_begin,
															local_begin, false);

	// Restoring pose stuff
	framework_pose.set_fold_tree( in_tree ); // initial tree
	framework_pose.set_allow_jump_move( jump_move_old );//restore the jump move
	framework_pose.set_allow_bb_move( bb_move_old );
	framework_pose.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_insert_h3
///
/// @brief insert all h3 loops from the h3 library
///
/// @detailed This routine inserts a loop from a loop library. The loop to be
///           inserted is decided by the decoy number.
///
/// @param[in] pose_in = template (this is the output)
///            loop_begin = starting residue to be inserted
///            loop_end = last residue to be inserted
///            loop_library = library containing loop conformations
///
/// @global_read makepdb_number so that one can use the power of parallel
///              processing
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/14/2007
///
/// @last_modified 03/15/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_insert_h3(
  pose_ns::Pose & pose_in,
	int loop_begin,
	int loop_end,
	std::vector< Fragment > & loop_library
)
{

	using namespace pose_ns;
	using namespace files_paths;
	using namespace makepdb_decoy;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	// Initial fold tree
	Fold_tree const input_tree( pose_in.fold_tree() );


	// setting up loop tree
	int nres( pose_in.total_residue() );
	int loop_size = ( loop_end - loop_begin ) + 1;
	int cutpoint = loop_begin + int(loop_size/2);
	pose_in.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint); //loop tree
	pose_in.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		pose_in.insert_ideal_bonds( loop_begin, loop_end ); // idealizing the loop

	// Making sure that backbone bonds and torsions are updated
	// updating backbone bonds and torsions
	pose_in.update_backbone_bonds_and_torsions();

	// Allowing movements in loop backbone
	{
		FArray1D_bool allow_bb_move( nres, false );
		pose_in.set_allow_bb_move( allow_bb_move );
		for(int i = loop_begin; i <= loop_end; i++)
			pose_in.set_allow_bb_move( i, true );
	}

	int filecounter(0);
	for ( std::vector< Fragment >::const_iterator it=loop_library.begin(),
					it_end = loop_library.end(); it != it_end; ++it ) {
		filecounter++;
		if(!(makepdb_number == filecounter))
			continue;
		if( filecounter > makepdb_number ){
			return;
		}
		it->insert( pose_in, loop_begin ); // inserting the loop in the decoy

		// Repack sidechains
		{
			FArray1D_bool allow_repack( nres, false );
			dle_loop_neighbor(pose_in, loop_begin-1,
																			 loop_end+1, allow_repack);
			pose_in.set_allow_chi_move( allow_repack );
			// pose_in.repack( allow_repack, true /*include_current*/ );
			dle_pack_with_unbound( pose_in, allow_repack,
																						true /*include_current*/ );
		}
	}

	// Restoring pose stuff
	pose_in.set_fold_tree( input_tree ); // initial tree
	pose_in.set_allow_jump_move( jump_move_old );//restore the jump move
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );


	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_match
///
/// @brief checks if a decoy in current misc passes the H3 filter
///
/// @detailed Takes in a misc and copies it to a pose. Then uses this pose to
///           check if the pose passes the H3 filter kink/extended sequence
///           rules of Shirai.
///
/// @param[out] h3_match: 0: Decoy does not pass filter
///                       1: Decoy passes filter
///                       2: Error - called in a mode in which this module
///                                  should not have been called
///
/// @global_read current misc, current pose flag status
///
/// @global_write restores misc
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/01/2007
///
/// @last_modified 04/01/2007
///////////////////////////////////////////////////////////////////////////////
int
antibody_modeling_h3_match()
{
	using namespace pose_ns;
	int h3_match(0);

	if((!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
		 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
		 && !dle_ns::refine_input_loop) ||
		 !dle_ns::dle_model_H3 )
		return(2); // return error value if none of
	             // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int loop_size = (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1;
	if( dle_CDR_H3_filter( pose_decoy,	cdr_ns::cdr_h3_begin,
																				loop_size) )
		h3_match = 1;
	else
		h3_match = 0;


	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( h3_match );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_NE1_n_plus_1_O_n_minus_2_distance
///
/// @brief computes the distance between atom NE1 of TRP at (n+1) of H3 & atom
///        O of residue (n-2) of the H3
///
/// @detailed Takes in a misc and copies it to a pose. Then uses this pose to
///           compute the distance between the NE1 atom of the n+1 TRP
///           of the CDR-H3 and the backbone oxygen atom of the n-2 residue of
///           the same CDR-H3 loop
///
/// @param[out] distance between the mentioned atoms
///
/// @global_read current misc, current pose flag status
///
/// @global_write restores misc
///
/// @remarks Shirai paper suggests that majority of distances in
///          kink: 2 Ang - 5.5 Ang With some outliers upto 7.5 Ang
///          extended: 7 Ang - 9 Ang
///
/// @references H. Shirai et al FEBS Letters 399 (1996) 1-8
///
/// @authors Aroop 04/26/2007
///
/// @last_modified 04/28/2007
///////////////////////////////////////////////////////////////////////////////
float
antibody_modeling_h3_NE1_n_plus_1_O_n_minus_2_distance()
{
	using namespace pose_ns;

	if((!antibody_ns::antibody_build && !antibody_ns::antibody_refine &&
			!antibody_ns::antibody_insert_all_h3 &&
			!dle_ns::refine_input_loop) ||
		 !dle_ns::dle_model_H3 ||
		 antibody_ns::antibody_cluster )
		return(0.00); // return error value if none of
	             // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	int loop_size = (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1;

	Pose h3_loop;
	// chop out the loop
	h3_loop = pose_decoy;
	h3_loop.delete_segment(1, cdr_ns::cdr_h3_begin - 3);
	h3_loop.delete_segment(loop_size+4, h3_loop.total_residue() );

	// extract 3 letter residue codes for the chopped loop
	std::vector <std::string> aa_name; // loop residue 3 letter codes
	std::string aa_3let_code; // amino acid 3 letter code
	for(int ii = 1; ii <= loop_size+3; ii++)	{
		name_from_num( h3_loop.res(ii), aa_3let_code);
		aa_name.push_back( aa_3let_code );
	}

	// return error if n+1 is not TRP
	if( aa_name[aa_name.size() - 1] != "TRP" ) {
		// Resetting misc
		pose_misc.copy_to_misc();
		// Resetting pose flag
		pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

		return( 0.00 );
	}

	const FArray3D_float & h3_loop_coord( h3_loop.full_coord() );
	int const NE1(9);   // NE1 atom of TRP
	int const O(4);   // O atom
	// calculate NE1-O distance
	float NE1_O_distance( 0.00 );
	NE1_O_distance = std::sqrt(	distance_squared(numeric::xyzVector_float(
								 &h3_loop_coord(1,NE1,aa_name.size())), numeric::xyzVector_float(
                 &h3_loop_coord(1,O,aa_name.size()-3))));

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( NE1_O_distance );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_hydrogen_ladder
///
/// @brief computes the number of required hydrogen bonds present in decoy H3
///
/// @detailed Calculates the number of hydrogen bonds - based on distance
///           assuming that in the hydrogen bond formed by X-H...Y
///           H...Y distance is in between 1.6 and 2 Angstroms
///           Different base conformations require different bonding
///           Kink : NH(0)---O(n)  ;NH(n-1)---O(0)
///                  NH(2)---O(n-3);NH(n-3)---O(2)
///                  NH(4)---O(n-5);NH(n-5)---O(4)
///           Bulge: NH(0)---O(n)  ;NH(n-1)---O(0)
///                  NH(2)---O(n-4);NH(n-4)---O(2)
///                  NH(4)---O(n-6);NH(n-6)---O(4)
///           Extnd: NH(0)---O(n)  ;NH(n)-----O(0)
///                  NH(2)---O(n-2);NH(n-2)---O(2)
///                  NH(4)---O(n-4);NH(n-4)---O(4)
///           Thus the maximum number of possible bonds is six. However, this
///           can be computed only if the number of residues in H3(95-102) is
///           at least ten.
///
/// @param[out] number of hydrogen bonds ( max 6)
///             if H3 < 10 res then default return is -2
///             if fails due to presence of Proline output is -3
///
/// @global_read current misc, current pose flag status
///
/// @global_write restores misc
///
/// @remarks
///
/// @references H. Shirai et al FEBS Letters 399 (1996) 1-8
///
/// @authors Aroop 04/26/2007
///
/// @last_modified 04/26/2007
///////////////////////////////////////////////////////////////////////////////
int
antibody_modeling_h3_hydrogen_ladder()
{
	using namespace pose_ns;

	if((!antibody_ns::antibody_build && !antibody_ns::antibody_refine &&
			!antibody_ns::antibody_insert_all_h3 &&
			!dle_ns::refine_input_loop) ||
		 !dle_ns::dle_model_H3 ||
		 antibody_ns::antibody_cluster )
		return(-1); // return error value if none of
	              // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	bool is_kinked( false );
	bool is_extended( false );
	bool is_bulged( false );
	std::vector <char> aa_1name; // loop residue 1 letter codes
	antibody_modeling_CDR_H3_prediction( pose_decoy, is_kinked, is_extended,
																			 aa_1name, "base" );


	int loop_size = (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1;
	int H_bonds(0);

	// if loop is short then hydrogen bonds are not formed properly
	if( (loop_size < 10) && !antibody_ns::compute_hbond ) {
		H_bonds = -2; // Normally maximum possible is 6
                 // 7 indicates that it was not computed

		// Resetting misc
		pose_misc.copy_to_misc();
		// Resetting pose flag
		pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

		return( H_bonds );
	}

	Pose h3_loop;
	h3_loop = pose_decoy;
	h3_loop.delete_segment(1, cdr_ns::cdr_h3_begin - 3);
	h3_loop.delete_segment(loop_size+4, h3_loop.total_residue() );

	// extract 3 letter residue codes for the chopped loop
	std::vector <std::string> aa_name; // loop residue 3 letter codes
	std::string aa_3let_code; // amino acid 3 letter code
	for(int ii = 1; ii <= loop_size+3; ii++)	{
		name_from_num( h3_loop.res(ii), aa_3let_code);
		aa_name.push_back( aa_3let_code );
	}

	// Bulge Detection: Shirai Rule ii
	if( (aa_name[aa_name.size()-5]=="TRP") || ((aa_name[aa_name.size()-4]=="GLY")
			&& ((aa_name[aa_name.size()-5]=="PHE") ||
      (aa_name[aa_name.size()-5]=="MET")))) {
		is_bulged = true;
		is_kinked = false;
	}

	// Proline Detection: Shirai Rule iii(a)
	int m(0); // special bulge number
	if( is_kinked )
		m = aa_name.size() - 1 - 2;
	if( is_bulged )
		m = aa_name.size() - 1 - 3;
	if( is_extended )
		m = aa_name.size() - 1 - 1;

	if( ((aa_name[3]=="PRO") || (aa_name[5]=="PRO") || (aa_name[m-1-3]=="PRO") ||
			 (aa_name[m-1-1]=="PRO"))
			&& !antibody_ns::compute_hbond	) {
		H_bonds = -3;

		// Resetting misc
		pose_misc.copy_to_misc();
		// Resetting pose flag
		pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

		return( H_bonds);
	}

	const FArray3D_float & h3_loop_coord( h3_loop.full_coord() );

	if( is_kinked ) {
		if( antibody_modeling_is_Hbond( 2, aa_name.size() - 1, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if( antibody_modeling_is_Hbond( aa_name.size() - 2, 2, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if((loop_size >= 10)) {
			if( antibody_modeling_is_Hbond( 4, aa_name.size() - 4, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 4, 4, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( 6, aa_name.size() - 6, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 6, 6, h3_loop_coord,
																			aa_name) )
				H_bonds++;
		}
	}
	else if( is_bulged ) {
		if( antibody_modeling_is_Hbond( 2, aa_name.size() - 1, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if( antibody_modeling_is_Hbond( aa_name.size() - 2, 2, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if((loop_size >= 10)) {
			if( antibody_modeling_is_Hbond( 4, aa_name.size() - 5, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 5, 4, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( 6, aa_name.size() - 7, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 7, 6, h3_loop_coord,
																			aa_name) )
				H_bonds++;
		}
	}
	else if( is_extended ) {
		if( antibody_modeling_is_Hbond( 2, aa_name.size() - 1, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if( antibody_modeling_is_Hbond( aa_name.size() - 1, 2, h3_loop_coord,
																		aa_name) )
			H_bonds++;
		if((loop_size >= 10)) {
			if( antibody_modeling_is_Hbond( 4, aa_name.size() - 3, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 3, 4, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( 6, aa_name.size() - 5, h3_loop_coord,
																			aa_name) )
				H_bonds++;
			if( antibody_modeling_is_Hbond( aa_name.size() - 5, 6, h3_loop_coord,
																			aa_name) )
				H_bonds++;
		}
	}

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( H_bonds );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_is_Hbond
///
/// @brief detects if the input hydrogen and the oxygen form a hydrogen bond
///
/// @detailed Calculates the distance between a given hydrogen atom & an oxygen
///           atom. If the distance is between 1.6 A and 3.0 A then returns a
///           true value to indicate the presence of a hydrogen bond
///
/// @parma[in] hydrogen_res: the residue number of the hydrogen atom
///            oxygen_res: the residue number of the oxygen atom
///            h3_loop_coord: the full atom coordinates of the structure
///            aa_name: a vector containing the 3 let aa code for the structure
///
/// @param[out] bool indicator for presence of hydrogen bond
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references L. Benco, D. Tunega, J. Hafner, and H. Lischka Upper Limit of
///             the O-H···O Hydrogen Bond. Ab Initio Study of the Kaolinite
///             Structure. J. Phys. Chem. B 2001;105(44):10812 -10817
///
/// @authors Aroop 04/27/2007
///
/// @last_modified 04/27/2007
///////////////////////////////////////////////////////////////////////////////
bool
antibody_modeling_is_Hbond(
	const int hydrogen_res,
	const int oxygen_res,
	const FArray3D_float & h3_loop_coord,
	std::vector <std::string> aa_name
)
{
	bool is_Hbond( false );

	int H(0);   // H atom position
	H = antibody_modeling_H_atom_number(aa_name[hydrogen_res -1]);

	// Proline exception
	if( H == 0 ) // only for Proline
		return( is_Hbond );

	int const O(4);   // O atom position

	float H_O_distance( 100.00);
	H_O_distance = std::sqrt(	distance_squared(numeric::xyzVector_float(
								 &h3_loop_coord(1, H, hydrogen_res)), numeric::xyzVector_float(
								 &h3_loop_coord(1, O, oxygen_res))));

	if( H_O_distance <= 3.0 )
		is_Hbond = true;

	return( is_Hbond );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_H_atom_number
///
/// @brief given an amino acid returns the sequential number of the H (N-H)
///
/// @detailed Depending on the sidechain the sequential number of the H atom
///           differs in the full coordinate array. This function returns this
///           sequential number
/// @param[out] the sequential position of the hydrogen atom
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2007
///
/// @last_modified 04/27/2007
///////////////////////////////////////////////////////////////////////////////
int
antibody_modeling_H_atom_number(
	const std::string & aa_3name
)
{
	int H_atom_number(0);

	if( aa_3name == "GLY" )
		H_atom_number = 5;
	else if( aa_3name == "LEU" )
		H_atom_number = 9;
	else if( aa_3name == "PHE" )
		H_atom_number = 12;
	else if( aa_3name == "LYS" )
		H_atom_number = 10;
	else if( aa_3name == "GLU" )
		H_atom_number = 10;
	else if( aa_3name == "PRO" )
		H_atom_number = 0;
	else if( aa_3name == "ILE" )
		H_atom_number = 9;
	else if( aa_3name == "TYR" )
		H_atom_number = 13;
	else if( aa_3name == "ARG" )
		H_atom_number = 12;
	else if( aa_3name == "ASP" )
		H_atom_number = 9;
	else if( aa_3name == "ALA" )
		H_atom_number = 6;
	else if( aa_3name == "MET" )
		H_atom_number = 9;
	else if( aa_3name == "TRP" )
		H_atom_number = 15;
	else if( aa_3name == "GLN" )
		H_atom_number = 10;
	else if( aa_3name == "SER" )
		H_atom_number = 7;
	else if( aa_3name == "VAL" )
		H_atom_number = 8;
	else if( aa_3name == "CYS" )
		H_atom_number = 7;
	else if( aa_3name == "HIS" )
		H_atom_number = 11;
	else if( aa_3name == "ASN" )
		H_atom_number = 9;
	else if( aa_3name == "THR" )
		H_atom_number = 8;

	return( H_atom_number );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_h3_stem_match
///
/// @brief for a kink matches rules for stem res( n-4, n-3, n-2, n-1)
///        for an extended matches base sp type( n-2, n-1, n, n+1)
///
/// @detailed matches based on Shirai rules ii(a), ii(b), ii(c)
///
/// @param[out] stem_match : -1 error
///                        :  1 correct match
///                        :  0 incorrect match
///                        :  2 sequence does not match any rule
///
///
/// @global_read current misc, current pose flag status
///
/// @global_write restores misc
///
/// @remarks
///
/// @references H. Shirai et al FEBS Letters 455 (1999) 188-197
///
/// @authors Aroop 04/27/2007
///
/// @last_modified 04/27/2007
///////////////////////////////////////////////////////////////////////////////
int
antibody_modeling_h3_stem_match()
{
	using namespace pose_ns;

	if((!antibody_ns::antibody_build && !antibody_ns::antibody_refine &&
			!antibody_ns::antibody_insert_all_h3 &&
			!dle_ns::refine_input_loop) ||
		 !dle_ns::dle_model_H3 ||
		 antibody_ns::antibody_cluster )
		return(-1); // return error value if none of
	              // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	int stem_match(0); // default false setting

	Pose pose_decoy; // decoy to orient to native
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	bool is_kinked( false );
	bool is_extended( false );
	std::vector <char> aa_1name; // loop residue 1 letter codes
	antibody_modeling_CDR_H3_prediction( pose_decoy, is_kinked, is_extended,
																			 aa_1name, "base" );


	int loop_size = (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1;

	Pose h3_loop;
	h3_loop = pose_decoy;
	h3_loop.delete_segment(1, cdr_ns::cdr_h3_begin - 3);
	h3_loop.delete_segment(loop_size+4, h3_loop.total_residue() );

	// extract 3 letter residue codes for the chopped loop
	std::vector <std::string> aa_name; // loop residue 3 letter codes
	std::string aa_3let_code; // amino acid 3 letter code
	for(int ii = 1; ii <= loop_size+3; ii++)	{
		name_from_num( h3_loop.res(ii), aa_3let_code);
		aa_name.push_back( aa_3let_code );
	}

	int const CA(2);
	const FArray3D_float & h3_loop_coord( h3_loop.Eposition() );

	bool rule_tested( false );

	float dihedral(0.00);
	// Rule ii(a)
	if( is_kinked && (aa_name[aa_name.size()-5]=="GLY")){
		float gly_phi = h3_loop.phi(aa_name.size()-4);
		float gly_psi = h3_loop.psi(aa_name.size()-4);
		// make_periodic
		if( gly_phi < 0 )
			gly_phi = gly_phi + 360.00;
		if( gly_psi < 0 )
			gly_psi = gly_psi + 360.00;

		if( (((gly_phi >= 45.00) && (gly_phi <= 135.00)) && ((gly_psi >= 140.00) &&
				(gly_psi <= 240.00))) || (((gly_phi >= 200.00) && (gly_phi <= 220.00))
				&& ((gly_psi >= 140.00) && (gly_psi <= 160.00)))) {

			// calculate dihedral angle
			dihedral_bk(h3_loop_coord(1,CA,aa_name.size()-2),
									h3_loop_coord(1,CA,aa_name.size()-3),
									h3_loop_coord(1,CA,aa_name.size()-4),
									h3_loop_coord(1,CA,aa_name.size()-5),
									dihedral);

			if( (dihedral >= -100.00) && (dihedral <= -10.00) )
				stem_match = 1;

			rule_tested = true;
		}
	}

	// Rule ii(b)
	if( (stem_match==0) && is_kinked && ((aa_name[aa_name.size()-5]=="TRP") ||
			((aa_name[aa_name.size()-4]=="GLY") && (
		(aa_name[aa_name.size()-5]=="PHE")||(aa_name[aa_name.size()-5]=="MET"))))){
		// calculate dihedral angle
		dihedral_bk(h3_loop_coord(1,CA,aa_name.size()-2),
								h3_loop_coord(1,CA,aa_name.size()-3),
								h3_loop_coord(1,CA,aa_name.size()-4),
								h3_loop_coord(1,CA,aa_name.size()-5),
								dihedral);

		if( (dihedral >= -10.00) && (dihedral <= 50.00) )
			stem_match = 1;

		rule_tested = true;
	}

	if( is_extended ){
		if( (aa_name[1]=="GLY") || (aa_name[1]=="ARG") ) {
			// calculate dihedral angle
			dihedral_bk(h3_loop_coord(1,CA,aa_name.size()),
									h3_loop_coord(1,CA,aa_name.size()-1),
									h3_loop_coord(1,CA,aa_name.size()-2),
									h3_loop_coord(1,CA,aa_name.size()-3),
									dihedral);

			if( (dihedral >= -150.00) && (dihedral <= -140.00) )
				stem_match = 1;

			rule_tested = true;
		}
		else {
				// calculate dihedral angle
			dihedral_bk(h3_loop_coord(1,CA,aa_name.size()),
									h3_loop_coord(1,CA,aa_name.size()-1),
									h3_loop_coord(1,CA,aa_name.size()-2),
									h3_loop_coord(1,CA,aa_name.size()-3),
									dihedral);

			if( (dihedral >= 150.00) && (dihedral <= 170.00) )
				stem_match = 1;

			rule_tested = true;
		}
	}

	// If sequence itself does not match any rule
	// treat this as a success
	if( (stem_match == 0) && !rule_tested )
		stem_match = 2;


	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( stem_match );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_orinent_to_native
///
/// @brief orient the decoy based on heavy framework alignment
///
/// @detailed Care is taken to superimpose only over non-loop residues of heavy
///           chain for a more accurate result.Superimposition is done on the
///           native
///
/// @param[in] pose to be superimposed
///
/// @global_read native_cdr from dle_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/19/2007
///
/// @last_modified 07/19/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_orient_to_native(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	using namespace native_cdr;

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return; // return if none of
	          // the above modes have been initialized

	Pose orientable_pose;
	orientable_pose = pose;

	int const nres( orientable_pose.total_residue() ); // Total residues

	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(6); // Heavy chain broken into six alignable
                                // regions by svrs and cdrs
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	// defining alignable regions
	for( int i = 0; i < 6; i++ ) {
		alignstart(i+1) = H_framework_begin[i];
		alignend(i+1) = H_framework_end[i] ;
	}
	/*
	alignstart(1) = native_chain_L_end + 1;
	alignend(1) = native_h1_begin - 1;
	alignstart(2) = native_h1_end + 1;
	alignend(2) = native_h2_begin - 1;
	alignstart(3) = native_h2_end + 1;
	alignend(3) = native_h3_begin - 1;
	alignstart(4) = native_h3_end + 2;
	alignend(4) = nres;
	*/

	// copy coords into double arrays
	FArray3D_float full1 ( orientable_pose.full_coord() );
	FArray3D_float Epos1 ( orientable_pose.Eposition() );
	FArray3D_float const & Epos2 ( native_fv.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms; // dummy - not used
	orient_region( nres, Epos1, full1, Epos2, aligned_regions, alignstart,
								 alignend, aligned_regions, alignstart, alignend, local_rms );

	orientable_pose.set_coords( false /*ideal_pos*/, Epos1, full1 );
	orientable_pose.update_backbone_bonds_and_torsions();

	pose = orientable_pose;

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_refineLH
///
/// @brief wrapper to carry out L-H docking local refine type moves with
///        simultaneous loop relaxation
///
/// @detailed This is a wrapper function that sets various variables, runs the
///           heart of the routine:
///           (viz. pose_docking_standard_protocol( pose_in );
///           and then resets all the variables to the initial state
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/17/2007
///
/// @last_modified 02/05/2008
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_refineLH(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace docking;
	using namespace dle_ns;

	pose_in.update_backbone_bonds_and_torsions();
	pose_in.copy_to_misc();

	// Saving initial settings
	bool const initial_dle_loops_flag( dle_loops_flag );
	bool const initial_local_refine( docking_local_refine );
	bool const initial_loop_ensemble_flag( dle_flag );
	bool const initial_simul_min( simultaneous_minimization );
	bool const initial_rt_min( dock_rtmin );
	bool const initial_model_H3( dle_model_H3 );
	bool const initial_dle_refine( dle_refine );
	bool const initial_dock_mcm( docking_mcm_flag );
	bool const initial_prepack( prepack_mode );
	bool const initial_dock_fullatom( docking_fullatom_flag );
	bool const initial_fullatom_loop_relax( fullatom_loop_relax );
	bool const initial_minimize_backbone( minimize_backbone );
	bool const initial_loops_fast( loops_ns::fast );
	int initial_dock_flex_loop_begin[DOCK_MAX_LOOPS];
	int initial_dock_flex_loop_end[DOCK_MAX_LOOPS];

	for(int i = 0; i < DOCK_MAX_LOOPS; i++ ) {
		initial_dock_flex_loop_begin[i] = dock_flex_loop_begin[i];
		initial_dock_flex_loop_end[i] = dock_flex_loop_end[i];
	}

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
	// Initial fold tree
	Fold_tree const input_tree( pose_in.fold_tree() );

	Loops initial_dle_loops;
	initial_dle_loops = dle_ns::dle_loops;
	int const initial_total_loops(dle_ns::dock_flex_total_loops);


	if( antibody_ns::relax_dock || !antibody_ns::freeze_h3 ){
		dle_loops_flag = true;
		dle_flag = true;
		simultaneous_minimization = true;
		dle_model_H3 = true;
		dle_refine = false;
		fullatom_loop_relax = true;
	}

	docking_local_refine = true;
	if( runlevel_ns::benchmark )
		dock_rtmin = false;
	else
		dock_rtmin = true;
	docking_mcm_flag = true;
	prepack_mode = false;
	docking_fullatom_flag = true;
	minimize_backbone = false;
	loops_ns::fast = true;

	int nres( pose_in.total_residue() );
	pose_docking_build_simple_tree( pose_in, nres, docking::part_end(1) );

	// storing in loop class for snug fitting
	if( antibody_ns::relax_dock ) {
		using namespace dle_ns;
		using namespace cdr_ns;
		using namespace antibody_ns;

		dle_loops = all_cdrs;

		if( !freeze_h3 ) {
			// delete h3 so that h3 with different cutpoint
			// can be entered subsequently
			if( flank_relax )
				dle_loops.delete_loop( cdr_h3_begin - h3_flank,
																						  cdr_h3_end + h3_flank );
			else
				dle_loops.delete_loop( cdr_h3_begin, cdr_h3_end );

			// re-entering h3 data with proper cutpoint
			int h3_size = (( cdr_h3_end + 1 ) - ( cdr_h3_begin )) + 1;
			int h3_cut(0);
			if( decoy_loop_cutpoint == 0 )
				h3_cut = cdr_h3_begin + int( h3_size / 2 );
			else
				h3_cut = decoy_loop_cutpoint;
			if( flank_relax )
				dle_loops.add_loop( cdr_h3_begin - h3_flank, cdr_h3_end
          + h3_flank + 1, h3_cut, loop_frag_offset, 0 );
			else
				dle_loops.add_loop(cdr_h3_begin, cdr_h3_end+1, h3_cut,
																					loop_frag_offset, 0);
		}
		dock_flex_total_loops = 0;

		for( Loops::const_iterator it=dle_loops.begin(),
					 it_end=dle_loops.end(); it < it_end; ++it ) {
			dock_flex_loop_begin[dock_flex_total_loops] = it->start();
			dock_flex_loop_end[dock_flex_total_loops] = it->stop();
			dock_flex_total_loops++;
		}
	}
	else if( !antibody_ns::freeze_h3 ) {
		using namespace dle_ns;
		using namespace cdr_ns;
		using namespace antibody_ns;

		Loops one_loop;
		int h3_size = (( cdr_h3_end + 1 ) - ( cdr_h3_begin )) + 1;
		int h3_cut(0);
		if( decoy_loop_cutpoint == 0 )
			h3_cut = cdr_h3_begin + int( h3_size / 2 );
		else
			h3_cut = decoy_loop_cutpoint;
		if( flank_relax )
			one_loop.add_loop( cdr_h3_begin - h3_flank, cdr_h3_end + h3_flank + 1,
                         loop_frag_offset, 0 );
		else
			one_loop.add_loop(cdr_h3_begin, cdr_h3_end+1, h3_cut,loop_frag_offset,0);
		dle_loops = one_loop;

		// populate other necessary array
		if( flank_relax ) {
			dock_flex_loop_begin[0] = cdr_h3_begin - h3_flank;
			dock_flex_loop_end[0] = cdr_h3_end + h3_flank + 1;
		}
		else {
			dock_flex_loop_begin[0] = cdr_h3_begin;
			dock_flex_loop_end[0] = cdr_h3_end + 1;
		}
		// increase loop count by one
		dock_flex_total_loops = 1;
	}

	// minimize parameters
	minimize_exclude_sstype( false, false );
	minimize_set_vary_phipsi( true );
	minimize_set_vary_omega( false );
	minimize_set_vary_chi( true );
	minimize_set_vary_rb_trans( true );
	minimize_set_vary_rb_angle( true );
	minimize_set_local_min( false, 0 );
	pose_set_use_nblist( true );

	// CORE of the function
	pose_docking_standard_protocol( pose_in );

	// Deleting loop
	if( antibody_ns::relax_dock ) {
		using namespace dle_ns;
		for( Loops::const_iterator it=dle_loops.begin(),
					 it_end=dle_loops.end(); it < it_end; ++it ) {
			dle_loops.delete_loop( it->start(), it->stop() );
			dock_flex_total_loops--;
		}
	}
	else if( !antibody_ns::freeze_h3 ) {
		using namespace dle_ns;
		using namespace cdr_ns;
		dle_loops.delete_loop( cdr_h3_begin, cdr_h3_end + 1);
		// decrease loop count by one
		dock_flex_total_loops -= 1;
	}

	pose_in.update_backbone_bonds_and_torsions();
	pose_in.copy_to_misc();

	// Resetting
	dle_loops_flag = initial_dle_loops_flag;
	docking_local_refine = initial_local_refine;
	dle_flag = initial_loop_ensemble_flag;
	simultaneous_minimization = initial_simul_min;
	dock_rtmin = initial_rt_min;
	dle_model_H3 = initial_model_H3;
	dle_refine = initial_dle_refine;
	docking_mcm_flag = initial_dock_mcm;
	prepack_mode = initial_prepack;
	docking_fullatom_flag = initial_dock_fullatom;
	fullatom_loop_relax = initial_fullatom_loop_relax;
	minimize_backbone = initial_minimize_backbone;
	loops_ns::fast = initial_loops_fast;

	for(int i = 0; i < DOCK_MAX_LOOPS; i++ ) {
		dock_flex_loop_begin[i] = initial_dock_flex_loop_begin[i];
		dock_flex_loop_end[i] = initial_dock_flex_loop_end[i];
	}

	dle_loops = initial_dle_loops;
	dock_flex_total_loops  = initial_total_loops;

	// Restoring pose stuff
	pose_in.set_fold_tree( input_tree ); // initial tree
	pose_in.set_allow_jump_move( jump_move_old );//restore the jump move
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_add_all_cdrs
///
/// @brief add all cdrs to a loop class
///
/// @detailed adds all the cdrs to a loop class. The deep graft parameter is
///           taken into account while defining loop ends. Cutpoint is in the
///           middle for all cdrs expect H3. For H3 if there is a previously
///           defined cutpoint, that is used, if not - then the middle is used.
///           if command line option "flank_relax" is set, then sets the h3
///           limits to encompass additional five flanking residues on each
///           side
///
/// @param[out]
///
/// @global_read antibody_ns::deep_graft,
///              dle_ns::decoy_loop_cutpoint
///
/// @global_write all_cdrs in antibody_ns
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/10/2007
///
/// @last_modified 12/26/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_add_all_cdrs(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace antibody_ns;
	using namespace dle_ns;
	using namespace cdr_ns;

	// saving current misc
	bool local_multi_chain_setting = files_paths::multi_chain;
	Pose misc_pose;
	bool const ideal_pos( false );
	bool const coords_init( true );
	bool const fullatom( false );
	pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );
	files_paths::multi_chain = local_multi_chain_setting;

	pose_in.copy_to_misc();

	int pdb_loop_begin(0), pdb_loop_end(0), begin(0), end(0), size(0), cut(0);
	char pdb_chain_id = '-';

	std::string cdr_name[6] = { "l1", "l2", "l3", "h1", "h2", "h3" };

	int cdr_total( 6 );
	if( freeze_h3 ) {
		// eliminate h3 from loop count
		cdr_total = 5;
		// only add h3 definition
		antibody_modeling_convert_to_sequential_res_from_chothia_res( cdr_name[5],
		  pdb_chain_id, pdb_loop_begin, pdb_loop_end);
		res_num_from_pdb_res_num_chain( begin, pdb_loop_begin,	pdb_chain_id );
		begin = begin - antibody_ns::deep_graft;
		res_num_from_pdb_res_num_chain( end, pdb_loop_end, pdb_chain_id );
		end = end - 1;
		cdr_h3_begin = begin;
		cdr_h3_end = end;
		if( flank_relax ) {
			begin = begin - h3_flank;
			end = end + h3_flank;
		}
		size = ( end - begin ) + 1;
		cut = begin + int( size / 2 );
		if( decoy_loop_cutpoint != 0  )
			cut = decoy_loop_cutpoint;
		else
			decoy_loop_cutpoint = cut;
		std::cout << cdr_name[5] << " " << begin << " "
							<< end << " "
							<< cut << std::endl;
	}
	for( int i = 0; i < cdr_total; i++ ) {
		antibody_modeling_convert_to_sequential_res_from_chothia_res( cdr_name[i],
		  pdb_chain_id, pdb_loop_begin, pdb_loop_end);
		res_num_from_pdb_res_num_chain( begin, pdb_loop_begin,	pdb_chain_id );
		begin = begin - antibody_ns::deep_graft;
		res_num_from_pdb_res_num_chain( end, pdb_loop_end, pdb_chain_id );
		// Detailed explanation provided
		// in _convert_to_sequential_res_from_chothia_res
		if( cdr_name[i] == "h3" )
			end = end - 1;

		if( cdr_name[i] == "l1" ) {
			cdr_l1_begin = begin;
			cdr_l1_end = end;
		}
		else if( cdr_name[i] == "l2" ) {
			cdr_l2_begin = begin;
			cdr_l2_end = end;
		}
		else if( cdr_name[i] == "l3" ) {
			cdr_l3_begin = begin;
			cdr_l3_end = end;
		}
		else if( cdr_name[i] == "h1" ) {
			cdr_h1_begin = begin;
			cdr_h1_end = end;
		}
		else if( cdr_name[i] == "h2" ) {
			cdr_h2_begin = begin;
			cdr_h2_end = end;
		}
		else if( cdr_name[i] == "h3" ) {
			cdr_h3_begin = begin;
			cdr_h3_end = end;
		}

		// end = end + antibody_ns::deep_graft;
		if( flank_relax && cdr_name[i] == "h3" ) {
			begin = begin - h3_flank;
			end = end + h3_flank;
		}
		size = ( end - begin ) + 1;
		cut = begin + int( size / 2 );
		if( decoy_loop_cutpoint != 0  && ( cdr_name[i] == "h3" ) )
			cut = decoy_loop_cutpoint;
		else if( cdr_name[i] == "h3" )
			decoy_loop_cutpoint = cut;
		all_cdrs.add_loop( begin, end, cut, 0, false);
		std::cout << cdr_name[i] << " " << begin << " "
							<< end << " "
							<< cut << std::endl;
	}

	// resetting misc back to initial state
	misc_pose.copy_to_misc();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_relax_cdrs
///
/// @brief relaxes all cdrs simultaneously
///
/// @detailed based on the all_cdrs loop definiton, minimizes only those
///           regions. A standard dfpmin is utilized with score12 and chain-
///           break and chain-overlap set. The allow_bb/chi arrays are changed
///           accordingly but then are reset to their initial states before
///           exiting the routine. Similarly the fold tree and jump movements
///           are restored to their initial states
///
/// @param[out]
///
/// @global_read all_cdrs in antibody_ns
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/10/2007
///
/// @last_modified 08/14/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_relax_cdrs(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace antibody_ns;
	using namespace dle_ns;

	// setting old style globals
	pose_in.update_backbone_bonds_and_torsions();
	pose_in.copy_to_misc();

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move(bb_move_old, chi_move_old, jump_move_old);
	// Initial fold tree
	Fold_tree const input_tree( pose_in.fold_tree() );

	// changing to all cdr fold tree
	antibody_modeling_change_to_all_cdr_fold_tree( pose_in );
	int const nres = pose_in.total_residue();

	// Enabling bb_moves
	pose_in.set_allow_bb_move( false );
	pose_loops_set_allow_bb_move( pose_in, all_cdrs );

	// Enabling chi moves
	bool include_neighbor( true );
	FArray1D_bool allow_chi_move( nres, false );
	pose_loops_select_loop_residues( pose_in, all_cdrs, include_neighbor,
																	 allow_chi_move );
	pose_in.set_allow_chi_move( allow_chi_move );

	// setup scoring
	int n_cutpoints( all_cdrs.num_loop());
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D_float cut_weight( n_cutpoints, 1.0 );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );

	// minimizing
	pose_in.main_minimize( weight_map, "dfpmin" );
	pose_in.update_backbone_bonds_and_torsions();
	pose_in.copy_to_misc();

	// repacking neighbors
	dle_extreme_repack( pose_in,
																		 1, // relax_cycles
																		 allow_chi_move,
																		 true, // rt_min
																		 true, // rotamer_trials
																		 false, // force_one_repack
																		 true ); // use_unbounds
	pose_in.update_backbone_bonds_and_torsions();
	pose_in.copy_to_misc();

	// Restoring pose stuff
	pose_in.set_fold_tree( input_tree ); // initial tree
	pose_in.set_allow_jump_move( jump_move_old );//restore the jump move
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock
///
/// @brief perturb light and heavy chain relative orientation while
///        simultaneously docking with antigen
///
/// @detailed based on pose_docking_mcm_protocol() in pose_docking.cc
///           Carries out simultaneous optimization while keeping track of both
///           interfaces ( between docking partners separated by TER and also
///           inbetween the light and heavy chains )
///
/// @param[out]
///
/// @global_read assumes that normal docking tree was fed in with pose
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/19/2007
///
/// @last_modified 12/26/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_snugdock(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace docking;


	pose_in.update_backbone_bonds_and_torsions();
	// refining H3 before snugdock
	if( antibody_ns::snugh3 ) {
		set_smallmove_size(8.0,10.0,16.0); //helix,strand,other
		dle_ns::refine_more = true; // default::true
		dle_refine_modeled_loop( pose_in );
		dle_ns::refine_more = false;
		// resetting for smaller subsequent relaxation
		set_smallmove_size(1.0,5.0,6.0); //helix,strand,other
	}
	pose_in.update_backbone_bonds_and_torsions();

	// setting multi dock fold tree
	if( antibody_ns::snugloop )
		antibody_modeling_snugdock_loop_tree( pose_in, antibody_ns::all_cdrs );
	else
		antibody_modeling_set_snugdock_foldtree( pose_in );

	// populate both VL-VH & Ab-Ag interfaces in intrf list
	antibody_ns::LH_AbAg_interface = true;
	pose_update_cendist( pose_in );
	antibody_modeling_detect_snugdock_interf_res();

	// MC move
	/* 0.1 angstrom default */
	const float trans_magnitude = get_dock_mcm_trans_magnitude();
	/* 5.0 degrees default */
	float rot_magnitude   = get_dock_mcm_rot_magnitude();

	// rb minimization
	const float loose_func_tol = 1.0;  /* absolute tolerance */
	const float tight_func_tol = 0.02; /* absolute tolerance */
	const std::string min_type = "dfpmin_atol";
	// monte carlo
	const float minimization_threshold = 15.0; /* score unit */
	// scorefxn to minimize
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score10d_min );

	if( antibody_ns::snugloop )
		dle_modify_weightmap( pose_in, weight_map);
	mc_global_track::mc_score::score = pose_in.score( score10d );

	// populate only one interfaces in intrf list
	antibody_ns::LH_AbAg_interface = false;
	pose_update_cendist( pose_in );
	docking_detect_interf_res();
	// hack to pass fab filter
	bool mcm_filter = docking_mcm_filter(delta_before_mcm, score_before_mcm);
	// populate both VL-VH & Ab-Ag interfaces in intrf list
	antibody_ns::LH_AbAg_interface = true;
	pose_update_cendist( pose_in );
	antibody_modeling_detect_snugdock_interf_res();

	if ( mcm_filter ) {
		antibody_modeling_snugdock_minize_trial( pose_in, min_type, weight_map,
																						 loose_func_tol);
		mc_global_track::mc_score::score = pose_in.score( score10d );

		// populate only one interfaces in intrf list
		antibody_ns::LH_AbAg_interface = false;
		pose_update_cendist( pose_in );
		docking_detect_interf_res();
		// hack to pass fab filter
		mcm_filter = docking_mcm_filter(delta_before_mcm, score_before_mcm);
		// populate both VL-VH & Ab-Ag interfaces in intrf list
		antibody_ns::LH_AbAg_interface = true;
		pose_update_cendist( pose_in );
		antibody_modeling_detect_snugdock_interf_res();

		if ( mcm_filter ) {
			const int cycles = 4; /* 4 cycles mcm first */
			antibody_modeling_snugdock_monte_carlo_minimize( pose_in, cycles,
        min_type,	weight_map, trans_magnitude,rot_magnitude,
          minimization_threshold, loose_func_tol );
			mc_global_track::mc_score::score = pose_in.score( score10d );

			// populate only one interfaces in intrf list
			antibody_ns::LH_AbAg_interface = false;
			pose_update_cendist( pose_in );
			docking_detect_interf_res();
			// hack to pass fab filter
			mcm_filter = docking_mcm_filter(delta_before_mcm, score_before_mcm);
			// populate both VL-VH & Ab-Ag interfaces in intrf list
			antibody_ns::LH_AbAg_interface = true;
			pose_update_cendist( pose_in );
			antibody_modeling_detect_snugdock_interf_res();

			if ( ! runlevel_ns::benchmark && mcm_filter ) {
				int cycles = 45; /* 45 cycles of extensive mcm */
				antibody_modeling_snugdock_monte_carlo_minimize( pose_in, cycles,
          min_type,	weight_map, trans_magnitude, rot_magnitude,
            minimization_threshold,	loose_func_tol );
				// minimize to a strict tolerance
				antibody_modeling_snugdock_minize_trial( pose_in, min_type, weight_map,
																								 tight_func_tol );

			} /* after_five_mcm */

		} /* after_one_min */

	} /* before_min */

	// populate only one interfaces in intrf list
	antibody_ns::LH_AbAg_interface = false;
	pose_update_cendist( pose_in );
	docking_detect_interf_res();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock_minimize_trial
///
/// @brief minimize trial for simultaneous dock of light & heavy chains as well
///        as the antigen
///
/// @detailed based on pose_docking_minimize_trial() in pose_docking.cc
///           Sets the rigid body center of a docking jump, minimizes on that
///           docking jump. This is carried out for both docking jumps. Finally
///           residues at both the interfaces ( for the two sets of docking
///           partners ) are detected and stored
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/20/2007
///
/// @last_modified 12/20/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_snugdock_minize_trial(
	pose_ns::Pose & docking_pose,
	const std::string & min_type,
	const pose_ns::Score_weight_map & wt_map,
	const float func_tol
)
{
	std::cout << std::endl;
	std::cout << "***** snugdock_minimize_trial *****" << std:: endl;

	using namespace pose_ns;

	const float temperature( 0.8 );
	Monte_carlo mc( docking_pose, wt_map, temperature );

	// populate intrf residues for only one interface
	antibody_ns::LH_AbAg_interface = false;
	pose_update_cendist( docking_pose );
	docking_detect_interf_res();

	antibody_modeling_snugdock_set_rb_center ( docking_pose, 1);
	pose_docking_minimize_rb_position( docking_pose, min_type, wt_map, func_tol);

	antibody_modeling_snugdock_set_rb_center ( docking_pose, 2);
	pose_docking_minimize_rb_position( docking_pose, min_type, wt_map, func_tol);

	// populate both VL-VH & Ab-Ag interfaces in intrf list
	antibody_ns::LH_AbAg_interface = true;
	pose_update_cendist( docking_pose );
	antibody_modeling_detect_snugdock_interf_res();

	mc.boltzmann( docking_pose );
	mc.show_scores();

	docking_pose = mc.low_pose();

	std::cout << "***** snugdock_minimize_trial *****" << std:: endl;
	std::cout << std::endl;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock_set_rb_center
///
/// @brief set the rigid body center for the docking jump specified
///
/// @detailed
///
/// @param[in]  pose: protein-protein complex (antibody-antigen complex)
///             rb_jump_num: rigid body jump number
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/20/2007
///
/// @last_modified 12/20/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_snugdock_set_rb_center(
  pose_ns::Pose & pose,
	int rb_jump_num
)
{
	int const begin1 ( docking::part_begin(1) );
	int const end1 ( docking::part_end(1) );
	int const begin2( docking::part_begin(2) );
	int const end2( docking::part_end(2) );

	// changes for jump between light and heavy chain
	if( rb_jump_num == 2 ) {
		docking::part_begin(1) = 1;
		docking::part_end(1) = antibody_ns::L_end;
		docking::part_begin(2) = antibody_ns::L_end + 1;
		docking::part_end(2) = end1;
	}
	/* calculate center of interface residues */
	FArray1D_float center(3);

	pose_update_cendist( pose );

	calculate_interface_centroid( center );
	/* set as the jump rb_center */
	numeric::xyzVector_double tmp_center( &center(1) );
	pose.set_jump_rb_center( rb_jump_num, tmp_center );

	// reseting docking variables
	docking::part_begin(1) = begin1;
	docking::part_end(1) = end1;
	docking::part_begin(2) = begin2;
	docking::part_end(2) = end2;

}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock_monte_carlo_minimize
///
/// @brief multi monte carlo inter minimize of light and heavy chains as well
///        as minimize antigen rigid body position
///
/// @detailed Adds on top of normal docking a second docking move. After
///           setting the rigid body center of the first docking move and then
///           carrying out a gaussian rigid body move of the first docking
///           jump, then does the same thing for the second docking jump. Also
///           makes sure that both interfaces are detected by calling appropri-
///           -ate functions.
/// @param[in] 	docking_pose   : protein-protein complex ( antibody-antigen)
///             cycles         : number of cycles
///             min_type       : minimization type
///             wt_map         : weight map used
///             trans_magnitude: amplitude of translational perturbation
///             rot_magnitude  : amplitude of rotational perturbation
///             minimization_threshold: minization threshold
///             func_tol       : functional tolerance
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/20/2007
///
/// @last_modified 12/20/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_snugdock_monte_carlo_minimize(
	pose_ns::Pose & docking_pose,
	const int cycles,
	const std::string & min_type,
	const pose_ns::Score_weight_map & wt_map,
	const float trans_magnitude,
	const float rot_magnitude,
	const float minimization_threshold,
	const float func_tol
)
{
	std::cout << "***** snugdock_monte_carlo_minimize *****" << std::endl;

	using namespace pose_ns;

	// create a monte_carlo object
	const float temperature ( 0.8 );
	Monte_carlo mc ( docking_pose, wt_map, temperature );

	// start MCM
	int n_accepted = 0; // diagnostic counter

	const bool interface_only = true;
	const bool include_current_sidechain = true;
	// for rotamer_trials
	const float energycut = get_energycut();
	const int nres = docking_pose.total_residue();
	const int rottrial_cycles = 1;

	for( int i=1; i<=cycles; ++i ) {
		// populate only one interfaces in intrf list
		antibody_ns::LH_AbAg_interface = false;
		pose_update_cendist( docking_pose );
		docking_detect_interf_res();

		antibody_modeling_snugdock_set_rb_center( docking_pose, 1 /*jump*/ );
		pose_docking_gaussian_rigid_body_move( docking_pose, 1 /*jump*/,
																					 trans_magnitude, rot_magnitude );
		antibody_modeling_snugdock_set_rb_center( docking_pose, 2 /*jump*/ );
		pose_docking_gaussian_rigid_body_move( docking_pose, 2 /*jump*/,
																					 trans_magnitude, rot_magnitude );

		// populate both VL-VH & Ab-Ag interfaces in intrf list
		antibody_ns::LH_AbAg_interface = true;
		pose_update_cendist( docking_pose );
    antibody_modeling_detect_snugdock_interf_res();

		bool save_docking_interface_pack = get_docking_interface_pack();

		if ( ( i % 8 == 0 ) || ( files_paths::antibody_modeler &&
													( i == cycles ))) { // default 8
			// full repacking
			pose_docking_repack( docking_pose, interface_only,
													 include_current_sidechain );
			// rotamer_trials_minimization ?
			if ( docking::dock_rtmin ) {
				set_docking_interface_pack( interface_only );
				score_set_try_rotamers( true );
				score_set_minimize_rot( true );
				docking_pose.score( wt_map );
				score_set_minimize_rot( false );
				score_set_try_rotamers( false );
			} else { // or just scoring with rotamer_trials off
				score_enable_rotamer_trials( false );
				docking_pose.score( wt_map );
				score_enable_rotamer_trials( true );
			}
		} else {
			//rotamer_trials for the whole protein
			//chu 06/11/21 -- be dependent on energycut, otherwise it will be slow.
			set_docking_interface_pack( ! interface_only );
			score_set_try_rotamers( true );
			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
				mc.best_pose().get_1D_score( pose_ns::RESENERGY ), rottrial_cycles);
			docking_pose.score( wt_map );
			score_set_try_rotamers( false );
		}

		set_docking_interface_pack( save_docking_interface_pack );
		const float current_score ( docking_pose.get_0D_score( SCORE ) );
		const float best_score ( mc.best_pose().get_0D_score( SCORE ) );

		if ( current_score - best_score < minimization_threshold )
			pose_docking_minimize_rb_position(docking_pose,min_type,wt_map,func_tol);

 		const bool accepted = mc.boltzmann( docking_pose );

		if( accepted )
			++n_accepted;

		std::cout << "Cycle: " << i << " -- ";
		mc.show_scores();
	}

	std::cout << "snugdock_mcm --" << n_accepted << " out of " << cycles
						<< " monte carlo cycles accepted" << std::endl;

	docking_pose = mc.low_pose();

	std::cout << "***** snugdock_monte_carlo_minimize *****" << std:: endl;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock_calc_interface
///
/// @brief detect interface residues for the light-heavy interface and the
///        antibody-antigen interface
///
/// @detailed In addition to detecting the interface residues for a docking
///           partner separated by TER, also incorporates interface of the
///           second docking jump. This was specially designed for the incorpo-
///           -ration of the light and the heavy chain interface
///
/// @param[in] pose_in: protein structure ( antibody-antigen complex)
///            int_res: interface residue array which will have true for
///                     interface residues
///
/// @param[out] int_res = array with interface residues set as true
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/20/2007
///
/// @last_modified 12/20/2007
///////////////////////////////////////////////////////////////////////////////
void
anitbody_modeling_snugdock_calc_interface(
	pose_ns::Pose & pose,
	FArray1DB_bool & int_res
)
{
	using namespace pose_ns;

	int const total_residue( pose.total_residue() );
	assert( int(int_res.size1()) == total_residue );

	pose_update_cendist( pose );
	Fold_tree const & fold_tree ( pose.fold_tree() );
	FArray1D_bool partner( pose.total_residue(), false );

	fold_tree.partition_by_jump( 1, partner );

	FArray2D_float const & cendist ( pose.get_2D_score( CENDIST ) );

	// Computing interface for Ab-Ag interface
	for ( int i=1; i<=total_residue; ++i ) {
		for ( int j=i+1; j<=total_residue; ++j ) {
			if ( partner(i) == partner(j) ) continue;
			if ( int_res(i) && int_res(j) ) continue;
			if ( cendist(i,j) < 64 ) int_res(i) = int_res(j) = true;
		}
	}

	// Computing interface for light & heavy chain interface
	for ( int i=1; i<=total_residue; ++i )
		partner(i) = false;
	fold_tree.partition_by_jump( 2, partner );

	for ( int i=1; i<=total_residue; ++i ) {
		for ( int j=i+1; j<=total_residue; ++j ) {
			if ( partner(i) == partner(j) ) continue;
			if ( int_res(i) && int_res(j) ) continue;
			if ( cendist(i,j) < 64 ) int_res(i) = int_res(j) = true;
		}
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_detect_L_end
///
/// @brief detects the sequential residue number of the light chain
///
/// @detailed Since there is no TER at the end of the light chain, uses change
///           in chain identifier to find out the light chain. Works even if
///           there are missing residues in the light chain.
///
/// @param[in] pose_in: protein structure ( antibody-antigen complex)
///
/// @param[out] integer specifying the sequential residue number of the last
///             residue of the light chain
///
/// @global_read dle_ns::light_chain : light chain identifier
///                                                   (char)
///
/// @global_write antibody_ns::L_end
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/19/2007
///
/// @last_modified 12/26/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_detect_L_end(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace antibody_ns;

	// if already computed, no need to compute again
	if( L_end != 0 )
		return;

	int current_res(-1);
	int const nres( pose_in.total_residue() );
	int residue_counter(0);

	for(int i = 1; i < nres; i++) {
		res_num_from_pdb_res_num_chain( current_res, i, dle_ns::light_chain );
		if(current_res == -1) {
			if(residue_counter == 0)
				continue;
			else {
				L_end = residue_counter;
				continue;
			}
		}
		else
			residue_counter++;
	}
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_set_snugdock_foldtree
///
/// @brief generates the fold tree for simultaneous light and heavy chain
///        minimization and antigen docking
///
/// @detailed Creates multi-dock fold tree. The jump positions are at residues
///           closest to the center of mass for each docking partner
///
/// @param[in] pose_in: protein structure ( antibody-antigen complex)
///
/// @param[out]
///
/// @global_read docking::part_end(1)
///
/// @global_write updated fold tree in pose_in
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/19/2007
///
/// @last_modified 12/25/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_set_snugdock_foldtree(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace antibody_ns;

	int const Ab_end( docking::part_end(1) );
	int const nres( pose_in.total_residue() );

	// creating fold tree for multi dock
	int const jump_pos1a ( pose_in.residue_center_of_mass( 1, Ab_end) );
 	int const jump_pos1b ( pose_in.residue_center_of_mass( Ab_end + 1,	nres) );
	int const jump_pos2a ( pose_in.residue_center_of_mass( 1, L_end) );
	int const jump_pos2b ( pose_in.residue_center_of_mass( L_end + 1, Ab_end ) );

	Fold_tree f;
	f.clear();
	f.add_edge( jump_pos1a, jump_pos1b, 1 );
	f.add_edge( jump_pos2a, jump_pos2b, 2 );
	if( jump_pos1a < jump_pos2a ) {
		f.add_edge(1, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( (jump_pos1a > jump_pos2a) && (jump_pos1a < L_end ) ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( (jump_pos1a > L_end + 1 ) && (jump_pos1a < jump_pos2b ) ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( jump_pos1a > jump_pos2b ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, Ab_end, pose_param::PEPTIDE );
	}
	else {
		std::cout << "!!!!!!!BAD MULTI-DOCK FOLD TREE!!!!!!!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	f.add_edge(jump_pos1b, Ab_end + 1, pose_param::PEPTIDE );
	f.add_edge(jump_pos1b, nres, pose_param::PEPTIDE );
	f.reorder(1);
	pose_in.set_fold_tree( f );
	pose_in.set_allow_jump_move( 1, true );
	pose_in.set_allow_jump_move( 2, true );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_snugdock_loop_tree
///
/// @brief updates pose with fold tree having two flexible jumps for 1.docking
///        VL-VH and 2.antigen. Also has fixed jumps for loops to be refined
///
/// @detailed Creates multi-dock fold tree with jumps for loops. Makes sure
///           that jump-positions of the dock jumps do not lie within the loops
///           ( within begin-1 and end+1 ). If such a jump-position occurs then
///           moves the jump to 2 residues before the start of the loop. The
///           dock jumps are flexible jumps whereas the loop jumps are fixed
///
/// @param[in] pose: protein structure ( antibody-antigen complex)
///            loops: loop definition ( cdr loops )
///
/// @param[out] updated fold tree in pose
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/21/2007
///
/// @last_modified 12/25/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_snugdock_loop_tree(
  pose_ns::Pose & pose,
	const pose_ns::Loops & loops
)
{
  using namespace pose_ns;
	using namespace antibody_ns;

	int const Ab_end( docking::part_end(1) );
	int const nres( pose.total_residue() );

	// creating fold tree for multi dock
	int jump_pos1a ( pose.residue_center_of_mass( 1, Ab_end) );
 	int jump_pos1b ( pose.residue_center_of_mass( Ab_end + 1,	nres) );
	int jump_pos2a ( pose.residue_center_of_mass( 1, L_end) );
	int jump_pos2b ( pose.residue_center_of_mass( L_end + 1, Ab_end ) );

  // make sure rb jumps do not reside in the loop region
	for( Loops::const_iterator it= loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		if( ( jump_pos1a >= (it->start()-1) ) && ( jump_pos1a <= (it->stop()+1) ) )
			jump_pos1a = it->start() - 2;
		if( ( jump_pos1b >= (it->start()-1) ) && ( jump_pos1b <= (it->stop()+1) ) )
			jump_pos1b = it->start() - 2;
		if( ( jump_pos2a >= (it->start()-1) ) && ( jump_pos2a <= (it->stop()+1) ) )
			jump_pos2a = it->start() - 2;
		if( ( jump_pos2b >= (it->start()-1) ) && ( jump_pos2b <= (it->stop()+1) ) )
			jump_pos2b = it->start() - 2;
	}

	// error checking for same jump points
	if( jump_pos1a == jump_pos2a )
		jump_pos1a = jump_pos1a - 1;
	if( jump_pos1a == jump_pos2b )
		jump_pos1a = jump_pos1a - 1;

	// re-checking for jump_pos1a if jump_pos1 was moved in loop by err chking
	for( Loops::const_iterator it= loops.begin(), it_end=loops.end();
			 it != it_end; ++it )
		if ( jump_pos1a >= it->start() && jump_pos1a <= it->stop())
			jump_pos1a = it->start() - 2;

	Fold_tree f;
	f.clear();
	f.add_edge( jump_pos1a, jump_pos1b, 1 );
	f.add_edge( jump_pos2a, jump_pos2b, 2 );
	if( jump_pos1a < jump_pos2a ) {
		f.add_edge(1, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( (jump_pos1a > jump_pos2a) && (jump_pos1a < L_end ) ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( (jump_pos1a > L_end + 1 ) && (jump_pos1a < jump_pos2b ) ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, Ab_end, pose_param::PEPTIDE );
	}
	else if( jump_pos1a > jump_pos2b ) {
		f.add_edge(1, jump_pos2a, pose_param::PEPTIDE );
		f.add_edge(jump_pos2a, L_end, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, L_end + 1, pose_param::PEPTIDE );
		f.add_edge(jump_pos2b, jump_pos1a, pose_param::PEPTIDE );
		f.add_edge(jump_pos1a, Ab_end, pose_param::PEPTIDE );
	}
	else {
		std::cout << "!!!!!!!BAD MULTI-DOCK FOLD TREE!!!!!!!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	f.add_edge(jump_pos1b, Ab_end + 1, pose_param::PEPTIDE );
	f.add_edge(jump_pos1b, nres, pose_param::PEPTIDE );
	f.reorder(1);

  // add the loop jump into the current tree, delete some old edge accordingly
	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		int const loop_start ( it->start() );
		int const loop_stop ( it->stop() );
		int const loop_cutpoint ( it->cut() );
		int edge_start(0), edge_stop(0);
		bool edge_found = false;
		int const num_jump = f.get_num_jump();
		for( Fold_tree::const_iterator it2=f.begin(), it2_end=f.end();
			 it2 !=it2_end; ++it2) {
			edge_start = min( it2->start, it2->stop );
			edge_stop = max( it2->start, it2->stop );
			if ( ! it2->is_jump() && loop_start>edge_start && loop_stop<edge_stop )
				{ edge_found = true; goto L100; }
		}
	L100:
		f.delete_unordered_edge( edge_start, edge_stop, pose_param::PEPTIDE);
		f.add_edge( loop_start-1, loop_stop+1, num_jump+1 );
		f.add_edge( edge_start, loop_start-1, pose_param::PEPTIDE );
		f.add_edge( loop_start-1, loop_cutpoint, pose_param::PEPTIDE );
		f.add_edge( loop_cutpoint+1, loop_stop+1, pose_param::PEPTIDE );
		f.add_edge( loop_stop+1, edge_stop, pose_param::PEPTIDE );
	}
	assert( f.get_num_jump() == loops.num_loop() + 1 ); //

  f.reorder(1);
  pose.set_fold_tree(f);

  pose.set_allow_jump_move(1, true); //Ab-Ag jump
	pose.set_allow_jump_move(2, true); //VL-VH jump
	for ( int i = 3; i<= f.get_num_jump(); ++i ) {
		pose.set_allow_jump_move(i, false); // loop jump
	}
	std::cout << "anitbody_modeling_snugdock_loop_tree:\n" << pose.fold_tree();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin antibody_modeling_detect_snugdock_interf_res
///
/// @brief set up snugdock dual interface residue list
///
/// @detailed This function detects and saves a record of the interfac residues
///           of both light-heavy chain interface & antibody-antigen interfac &
///           simultaneoudly sums the docking pair interaction score.
///           (adapted from 6/28/01 JJG rev. 8/2/1)
///
///           Interface residues defined as those with centroid distances of
///           less than 6 Angstroms across the interface
///
///           cendist array must be precalculated !!!!!
///
///           a 8A definition interface residue list is calculated for side-
///           chain repacking and cheap electrostatics.
///
/// @global_read cendist array precalculated in cenlist.h
/// @global_write interface array in interface.h
///
/// @remarks
///
/// @references
///
/// @authors Aroop 12/25/2007
///
/// @last_modified 12/25/2007
///////////////////////////////////////////////////////////////////////////////
void
antibody_modeling_detect_snugdock_interf_res()
{
	// save the int_pair_list matrix
	// save the int_res_list array

	using namespace cenlist_ns;
	using namespace docking;
	using namespace interface;
	using namespace misc;
	using namespace runlevel_ns;

	int_res  = false;
	int_res8 = false;
	for( int i=1; i <= max_docking_sites; ++i ) {
		int_res_list(i).clear();
		int_pair_list(i).clear();
		int_res_list8(i).clear();
		int_pair_list8(i).clear();
	}

	for ( int i = part_begin(1), iend = part_end(1), jend = part_end(2);
				i <= iend; ++i ) {
		for ( int j = part_begin(2); j <= jend; ++j ) {
			if ( cendist(i,j) < 64.0 ) { // if under 8A
//     predefine int_res_list8(2)[X]
				if ( !int_res8(j) ) {
					int_res_list8(2).push_back(j);
				}
//     predefine int_pair_list8
				int_res8(i) = true;
				int_res8(j) = true;
				int_pair_list8(1).push_back(i);
				int_pair_list8(2).push_back(j);

				if ( cendist(i,j) < 36.0 ) { // check if under 6A
//     predefine int_res_list(2)[X]
					if ( !int_res(j) ) {
						int_res_list(2).push_back(j);
					}
//     predefine int_pair_list
					int_res(i) = true;
					int_res(j) = true;
					int_pair_list(1).push_back(i);
					int_pair_list(2).push_back(j);
				}            // 6A
			}               // 8A
		}                  // partner2

//     predefine int_res_list(1)[X]
		if ( int_res(i) ) {
			int_res_list(1).push_back(i);
		}
//     predefine int_res_list8(1)[X]
		if ( int_res8(i) ) {
			int_res_list8(1).push_back(i);
		}

	}        // partner1

	for ( int i = 1, iend = antibody_ns::L_end, jend = part_end(1);
				i <= iend; ++i ) {
		for ( int j = antibody_ns::L_end + 1; j <= jend; ++j ) {
			if ( cendist(i,j) < 64.0 ) { // if under 8A
//     predefine int_res_list8(2)[X]
				if ( !int_res8(j) ) {
					int_res_list8(2).push_back(j);
				}
//     predefine int_pair_list8
				int_res8(i) = true;
				int_res8(j) = true;
				int_pair_list8(1).push_back(i);
				int_pair_list8(2).push_back(j);

				if ( cendist(i,j) < 36.0 ) { // check if under 6A
//     predefine int_res_list(2)[X]
					if ( !int_res(j) ) {
						int_res_list(2).push_back(j);
					}
//     predefine int_pair_list
					int_res(i) = true;
					int_res(j) = true;
					int_pair_list(1).push_back(i);
					int_pair_list(2).push_back(j);
				}            // 6A
			}               // 8A
		}                  // partner2

//     predefine int_res_list(1)[X]
		if ( int_res(i) ) {
			int_res_list(1).push_back(i);
		}
//     predefine int_res_list8(1)[X]
		if ( int_res8(i) ) {
			int_res_list8(1).push_back(i);
		}

	}        // partner1

	assert( int_pair_list(1).size() == int_pair_list(2).size() );
	assert( int_pair_list8(1).size() == int_pair_list8(2).size() );

	if ( runlevel >= yap ) {
		std::cout << "interface sizes under 6A: " << int_res_list(1).size()
							<< ' ' << int_res_list(2).size() << std::endl;
		std::cout << "interface pairs under 6A: " << int_pair_list(1).size()
							<< std::endl;
		std::cout << "interface sizes under 8A: " << int_res_list8(1).size()
							<< ' ' << int_res_list8(2).size() << std::endl;
		std::cout << "interface pairs under 8A: " << int_pair_list8(1).size()
							<< std::endl;
	}

}

