// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7212 $
//  $Date: 2006-01-04 07:49:01 -0800 (Wed, 04 Jan 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "atom_tree_minimize.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "all_atom.h" // all_atom_cutoff
#include "all_atom_type.h" // all_atom_type
#include "bond_angle.h"
#include "cenlist.h" // cendist_cutoff2
#include "count_pair.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "gb_elec.h"
#include "gb_elec_ns.h"
#include "geometric_solvation.h"
#include "gl_graphics.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "jumping_diagnostics.h"
#include "jumping_minimize.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "kin_atom.h"
#include "minimize.h"
#include "minimize_ns.h" // deriv_precalc
#include "min_debug_ns.h"
#include "nblist.h"
#include "pack.h"
#include "pairenergy.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pose.h"
#include "pose_constraints.h"
#include "pose_dna.h"
#include "pose_docking.h"
#include "pose_vdw.h"
#include "prof.h"
#include "ramachandran.h"
#include "relax_structure.h" // get_score_variant()
#include "rotamer_trials.h"
#include "runlevel.h"
#include "scale_res_energy.h"
#include "score.h"
#include "score_ns.h"
#include "sidechain_bond_angles.h"
#include "tether.h"
#include "timer.h"
#include "template_pack.h"
#include "util_basic.h"
#include "util_vector.h"
#include "water_ns.h" // PBHACK
#include "water.h" // PBHACK

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>


// Namespaces
namespace min_map_ns{
	kin::Minimizer_map min_map;
}

// Using




///////////////////////////////////////////////////////////////////////////////
// for use in fullatom_energy: nblist

kin::Minimizer_map const &
retrieve_min_map()
{
	return min_map_ns::min_map;
}


///////////////////////////////////////////////////////////////////////////////
void
write_big_nblist_warning(
												 float const score_delta,
												 std::string const & setting
)
{
#ifndef BOINC
	std::cout <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!!\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!!\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"pose_minimize:: Big score_delta when turning " << setting <<
		" the nblist: " << score_delta << '\n' <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!!\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!! DANGER!!!!\n" <<
		"======================================================\n" <<
		"======================================================\n" <<
		"======================================================" <<
		std::endl;
#endif
}


///////////////////////////////////////////////////////////////////////////////

void
atom_tree_minimize(
	pose_ns::Pose & pose,
	std::string const & min_type,
	std::string const & move_type,
	Scoring_Function score_fxn
)
{
	using min_map_ns::min_map; // global object

	// params
	float const warning_cutoff = { 1.0 }; // energy cutoff for warnings
	score_set_use_subset_energy( false ); // no subset energy

	/////////////////////////////////////////////////
	// score with rotamer trials if try_rotamer==true
	// this should happen before the min_map setup
	pose.score( score_fxn );

	debug_rotamer_trials( pose, score_fxn );

	if ( score_get_try_rotamers() ) {
// 	if ( param_pack::gen_born && score_get_try_rotamers() ) {
		// we should update the born radii and the rsd/pair energies
		// since rotamer_trials is an approximation for gb
		// actually, always recompute score if we did rotamer-trials
		pose.new_score_pose();
		pose.score( score_fxn );
	}

	score_enable_rotamer_trials( false );

	float const start_score( pose.get_0D_score( pose_ns::SCORE ) );

	if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
		std::cout << "STARTMIN_" << pose.show_scores() << " atom_tree!" <<
			std::endl;
	}


	//////////////////////////
	// setup the minimizer map
	min_map.setup( pose, get_use_nblist_during_minimization() );
	int const nangles( min_map.nangles() );
	if ( nangles == 0 ) {
		std::cout << "minimize:: no angles to minimize" << std::endl;
		return;
	}


	// set minimizing flag --> triggers use of nblist
	set_currently_minimizing( true );

	// no radii updates:
	if ( param_pack::gen_born ) {
		set_update_born_radii( false );
	}

	// check score w/ nblist on
	if ( get_use_nblist() ) {
		// rescore w/ nblist on
		pose.score( score_fxn );
		float const score_delta( pose.get_0D_score( pose_ns::SCORE ) -
														 start_score );
		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "score-delta after turning on nblist: " <<
				score_delta << std::endl;
		}
		if ( std::abs( score_delta ) > 1.0 ) {
			write_big_nblist_warning( score_delta, "on" );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//car MINIMIZE
	minimize_set_func(2);
	float fret;
	int iter;
	bool gfrag( true );
	FArray1D_float phipsi( nangles );
	float const minimize_tolerance( minimize_get_tolerance() );

	// pack the angles
	min_map.pack_phipsi( pose, phipsi );

	// save starting function value for debugging...
	float const start_func = func( gfrag, phipsi );

	if ( std::abs( start_func - start_score ) > 0.1 ) {
		std::cout << "start_score/func mismatch: " << start_score << ' ' <<
			start_func << std::endl;
	}

	// Do an ediff check before minimizing.
	// Note, we always pass the check if no one
	// has bothered to add an extra EDIFF_SCORE
	// to the pose before this point.
	if ( !pose_ediff_check( pose ) ) {
		GRAPHICS_LOG( "atom_tree_minimize: "+min_type );
		start_timer("minimize");
		if ( min_type == "linmin" ) {
			FArray1D_float dE_dphipsi( nangles );
			for ( int j = 1; j <= 1; ++j ) {
				dfunc( phipsi, dE_dphipsi, nangles );
				linmin( phipsi, dE_dphipsi, nangles, fret, gfrag );
				if ( !gfrag ) break;
			}
		} else if ( min_type == "dfpmin" ) {
			dfpmin( phipsi, nangles, minimize_tolerance, iter, fret,
							gfrag);
		} else if ( min_type == "dfpmin_atol" ) { /* absolute tolerence */
			dfpmin_atol( phipsi, nangles, minimize_tolerance, iter, fret,
									 gfrag);
		} else if ( min_type == "frpmin" ) {
			frprmn( phipsi, nangles, minimize_tolerance, iter, fret, gfrag );
		} else if ( min_type == "frpmin_atol" ) { /* absolute tolerance */
			frprmn_atol( phipsi, nangles, minimize_tolerance, iter, fret,
									 gfrag);
		} else {
			std::cout << "min_type not defined in function minimize (debump.cc)"
								<< std::endl << "min_type: " << min_type << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}


		// check crazy gfrag variable
		if ( !gfrag ) {
			std::cout << "gfrag tripped!!" << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	// store final func value, scoring is with nblist if use_nblist() == true
	float const end_func = func( gfrag, phipsi );
	float const end_func_gb( param_pack::gen_born && pose.fullatom() &&
													 scorefxns::fa_gb_elec_weight != 0.0 ?
													 pose.get_0D_score( pose_ns::GB ) : 0.0 );


	// turn off minimizing flag
	set_currently_minimizing(false); // set to true in unpack_phipsi


	// no radii updates:
	if ( param_pack::gen_born ) {
		set_update_born_radii( true );
	}


	// zero out the deltas !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	pose.fold_in_rb_deltas();


	// status output
	float const end_score = pose.score( score_fxn );
	float const end_score_gb( param_pack::gen_born && pose.fullatom() &&
														scorefxns::fa_gb_elec_weight != 0.0 ?
														pose.get_0D_score( pose_ns::GB ) : 0.0 );

	if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
		std::cout << "ENDMIN_" << pose.show_scores() << std::endl;

		std::cout << "minimize:: time= " << get_timer("minimize") <<
			" start_score= " << start_score << " end_score= " << end_score <<
			" start_func= " << start_func << " end_func= " << end_func <<
			" end_gb1= " << end_func_gb << " end_gb2= " << end_score_gb << std::endl;
	}
	if ( true ) {
		if ( end_score - start_score > warning_cutoff ) {
			pose.new_score_pose();
			float const total_score = pose.score( score_fxn );
			std::cout << "WARNING!! " << move_type << " score increase:" <<
			 SS( start_score ) << SS( end_score ) << SS( start_func ) <<
			 SS( end_func ) << SS( total_score ) << std::endl;
		}
	}


	// restore!
	score_enable_rotamer_trials(true);
}

///////////////////////////////////////////////////////////////////////////////
std::string
sv(
	 numeric::xyzVector_float const & v
	 )
{
	std::ostringstream os;
	os << F(9,3,v(1)) << F(9,3,v(2)) << F(9,3,v(3));
	return os.str();
}


//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_dfunc_vdw
///
/// @brief
///
/// @detailed
///car note that this calculates the deriv for all torsion angles even
///car those that are fixed. Because of the way that the derivative is
///car calculated, I don't believe this is a significant slow down (ie
///car have to run over all the atom pairs twice, regardless of the number
///car of torsion angles)
///
///car multiple neighborlists:
///car cendist       centroid distances in current structure, cutoff for vdw
///car dis2_tether   centroid distances in tether structure, cutoff for tether
///
///db computes the derivative of E  with respect to each
///db of the torsion angles.  Using the chain rule, we have
///db
///db dE/d phi = dE/dr *  dr/dphi
///db
///db dr/dphi  = Eab x (V-Vb) . (V' - V)/|V-V'|
///db
///db (the first cross product is the displacement of V upon a rotation dphi
///db around the unit vector Eab, Vb is the coordinates of the second atom in
///db the bond)
///db
///car dE/dR = 2r  (for vdw at least)
///db since | V-V'| = r,
///db
///db dE/ dphi = 2 Eab x (V-Vb) . (V' - V)
///db
///db  note that Eab and Vb are different for each torsion angle, but V'
///db and V are the same.  rearranging:
///db
///db = -  2 Eab X Vb . (V' - V) - 2 Eab . (V' x V).
///db
///db now we need the averages over all Vi of the difference and the
///db crossproduct of V and V'.
///
///car below, Eab x Vb is 'vec'
///car        Eab      is 'unit'
///car        (V'-V)   is 'f2'
///car        'F2tot' = f2*dE_dR (cumulative)
///car        (V' X V) is 'f1' ('F1_xxxE' is cumulative for potential xxx)
///car        eval_dE_dR actually returns dE_dR/r
///
///car if two atoms are fixed relatively in cartesian space, then dr/dphi = 0
///car and there is no contribution to the derivative
///
/// @param  phipsi - [in/out]? -
/// @param  dE_dphipsi - [in/out]? -
/// @param  nangles - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
//car recursive derivative calculation
//car see Braun and Go, JMB 1985 186:611-26
//car     Guntert,Braun & Wuthrich JMB 1991 217:517-53

void
atom_tree_dfunc_vdw(
	pose_ns::Pose & pose,
	FArray1DB_float & phipsi,
	FArray1DB_float & dE_dphipsi
)
{
	using namespace param;
	using namespace kin;
	using namespace param_torsion;
	using namespace scorefxns;   // scorefxn weights
	float const deg2rad( numeric::conversions::radians(1.0) );

	using min_map_ns::min_map; // lives in this .cc file

	PROF_START( prof::ATOM_TREE_DFUNC_VDW );

	//std::cout << "dfunc_vdw: " << pose.show_scores() << std::endl;

	if( false ) {//dump pdb
		static int counter (0);
		pose.dump_pdb("dump_"+lead_zero_string_of(++counter,4)+".pdb");
	}

	// clear all the F1's and F2's
	min_map.zero_torsion_vectors();


	// puts the torsions from phipsi into pose
	min_map.unpack_phipsi( pose, phipsi );


	// copy pose data to misc::
	// necessary for things like hbonds, etc that still look at misc
	pose.copy_to_misc();


	// dump pdb for debugging
	if ( get_dfunc_dump() ) {
		dump_phil_pdb();
	}


	//car if not fullatom mode, put position array+centroids in full_coord array
	bool const fullatom ( pose.fullatom() );
	assert( fullatom );
	if ( !fullatom ) minimize_update_fullcoord();


	// dunbrack derivative
	FArray2D_float dunbrack_deriv;
	if ( fullatom && fa_dun_weight != 0. ) {
		dunbrack_deriv.dimension( total_torsion, MAX_RES()() );
		pose_eval_dunbrack_deriv( pose, dunbrack_deriv);
	}


	// evaluate bb torsion tether deriv
	FArray2D_float torsion_tether_deriv;
	if ( phipsi_weight != 0.0 || omega_weight != 0.0 )	{
		torsion_tether_deriv.dimension( total_bb_torsion, MAX_RES()() );
		float tmp1,tmp2;
		if ( phipsi_weight != 0.0 )
			calculate_phipsi_torsion_tether_score(tmp1, torsion_tether_deriv);
		if (  omega_weight != 0.0 )
			calculate_omega_torsion_tether_score(tmp2, torsion_tether_deriv);
	}

	//  get derivative of all atom pair potentials
	//  this includes fa_pair and hbonds
	//
	PROF_START( prof::ATOMPAIR_DERIV );
	atom_tree_get_atompairE_deriv( pose, min_map, fullatom );
	PROF_STOP ( prof::ATOMPAIR_DERIV );


	// this should only be done once, after all torsion F1,F2's have
	// been filled in
	min_map.link_torsion_vectors();

	// precalculate all bond angle derivatives
	if (bond_angle_weight != 0.0) {
		precalc_bond_angle_deriv(pose);
	}

	// now loop over the torsions in the map
	int imap( 1 ); // for indexing into de_dphipsi( imap )
	for ( kin::Minimizer_map::Torsion_iterator tor_iter=min_map.torsions_begin(),
					tor_end = min_map.torsions_end(); tor_iter != tor_end;
				++tor_iter, ++imap ) {
		Torsion_node const & tor( **tor_iter );

		// NOTE: deriv is in the units of the degree of freedom as
		// represented internally, without any scale factors applied
		// ie the units returned by pose.get_atom_tree_torsion(...)
		//
		//
		// type  -- units
		// -------------------
		// PHI   -- radians
		// THETA -- radians
		// D     -- angstroms
		// RB1-3 -- angstroms
		// RB4-6 -- degrees    (!)

		float deriv = 0.0;

		kin::Atom const * atom
			( pose.get_atom_tree_atom( tor.atomno(),tor.rsd() ));

		// this function determines the axis and center of rotation/translation
		numeric::xyzVector_float axis, end_pos;
		kin::Kin_torsion_type const type( tor.type() );
		atom->get_torsion_axis_and_end_pos( Coords_FArray_const(pose.full_coord()),
																				axis, end_pos, type );


		// convert combined F1,F2 to an angular derivative
		//  using Abe Go trick
		//
		if ( type == PHI || type == THETA || type == RB4 || type == RB5 ||
				 type == RB6 ) {
			// rotation about an axis
			// note: assumes we are dealing with RADIANS
			// scale factor below handles the fact that RB4-6 are degrees
			float scale_factor( ( type == PHI || type == THETA ) ? 1 : deg2rad );

			if ( type == THETA ) {
				// need to think about this more carefully
				using numeric::constants::f::pi;
				float const theta( phipsi( imap ) / min_map.torsion_scale_factor(tor));
				int const theta_mod
					( ( static_cast< int >( std::floor( theta/pi )))%2);
				if ( theta_mod == 1 || theta_mod == -1 ) {
					scale_factor *= -1.0f;
				}
			}

			deriv -= scale_factor * ( dot( axis, tor.F1() ) +
																dot( cross( axis, end_pos ), tor.F2() ) );
		} else {
			// translation along an axis
			deriv += dot( axis, tor.F2() );
		}

		/////////////////////////////////////////////////////////////////
		// derivatives of this degree of freedom


		// calc new-style torsion/angle/bond degree of freedom deriv's
		deriv += calculate_torsion_constraint_deriv( tor, pose );

		// simple bond angle terms
		if ( sidechain_bond_angle_weight != 0.0 ) {
			deriv += sidechain_bond_angle_weight *
				eval_bond_angle_deriv( tor.torsion_id(), pose );
		}

		if ( bond_angle_weight != 0.0 ) {
			deriv += bond_angle_weight * bond_angle_deriv(pose, tor);
		}

		{ // scope
			// figure out the standard rosetta torsion type of this guy
			int torsion, i;
			tor.get_rosetta_type( torsion, i );

			if ( torsion ) {
				// PHI torsions are stored internally as radians
				// so deriv is in these units
				assert( type == PHI );
				float const scale_factor( numeric::conversions::degrees(1.0) );

				// dunbrack: bb or chi
				if ( torsion_type_is_bb(torsion) || torsion_type_is_chi(torsion) ) {
					float const fa_dun_factor
						( param_pack::pack_wts.Wdun() * fa_dun_weight );
					if ( fullatom && fa_dun_factor != 0.0 ) {
						deriv += scale_factor * dunbrack_deriv(torsion,i) * fa_dun_factor;
					}
				}


				// bb only terms: rama and fa_prob
				if ( torsion_type_is_bb(torsion) ) {
					float const fa_prob_factor
						( param_pack::pack_wts.Wone() * fa_prob1b_weight );


					// ramachandran
					if ( ramachandran_weight != 0. ) {
						float const rama_deriv
							( get_rama_score_residue_deriv( pose.res(i), pose.phi(i),
								pose.psi(i), pose.secstruct(i),torsion) );
						deriv += scale_factor * rama_deriv * ramachandran_weight;
					}

					// fa_prob
					if ( fa_prob_factor != 0.0 ) {
						float const fa_prob_deriv
							( get_Paa_pp_deriv( pose.res(i), pose.phi(i), pose.psi(i),
																	torsion ) );
						deriv += scale_factor * fa_prob_deriv * fa_prob_factor;
					}

					// old-style torsion tethering
					if ( phipsi_weight != 0.0 ) { // torsion tethers
						if ( torsion == phi_torsion || torsion == psi_torsion ) {
							deriv += scale_factor * phipsi_weight *
								torsion_tether_deriv(torsion,i);
						}
					}
					if ( omega_weight != 0.0 ) { // torsion tethers
						if ( torsion == omega_torsion ) {
							deriv += scale_factor * omega_weight *
								torsion_tether_deriv(torsion,i);
						}
					}
				}


				//////////////////////////////////////////////////////////////////////
				//////////////////////////////////////////////////////////////////////
				// HACK to hold omega fixed !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
				//////////////////////////////////////////////////////////////////////
				//////////////////////////////////////////////////////////////////////

				if ( torsion == omega_torsion && !minimize_ns::minimize_vary_omega ) {
					deriv = 0.0;
				}

			} // if ( torsion )
		} // scope


		deriv /= min_map.torsion_scale_factor( tor );

		dE_dphipsi(imap) = deriv;
	} // loop over map


	if ( minimize_get_deriv_check() ) {
		numerical_derivative_check( min_map, phipsi, dE_dphipsi,min_map.nangles());
	}

	PROF_STOP( prof::ATOM_TREE_DFUNC_VDW );

}


//////////////////////////////////////////////////////////////////////////////
float
calculate_torsion_constraint_deriv(
	kin::Torsion_node const & tor,
	pose_ns::Pose const & pose
)
{
	if ( !pose.constraints_exist() ) return 0.0;

	using namespace scorefxns; // DANGER!!
	using namespace param_torsion;

	float deriv( 0.0 );

	cst_set_ns::Cst_set const & cst_set( pose.constraints() );

	// new-style atom_tree torsion
	if ( kin_1D_cst_weight != 0.0 ) {
		kin::Torsion_id const & id( tor.torsion_id() );
		float const current_value( pose.get_atom_tree_torsion( id ) );
		deriv += kin_1D_cst_weight *
			cst_set.kin_torsion_deriv( id, current_value );
	}

	// rosetta-style phi/psi/omega torsion
	if ( phipsi_cst_weight !=  0.0 || omega_cst_weight != 0.0 ||
			 chi_cst_weight != 0.0 ) {
		int seqpos, torsion;
		tor.get_rosetta_type( torsion, seqpos );
		if ( torsion ) {
			// need scale_factor since the angles/deriv values are in radians
			float const scale_factor( numeric::conversions::degrees(1.0) );
			float const weight
				( ( torsion == phi_torsion || torsion ==psi_torsion ) ? phipsi_weight :
					( torsion_type_is_chi( torsion ) ? chi_cst_weight :
						( torsion == omega_torsion ? omega_cst_weight : 0.0f ) ) );
			if ( weight != 0.0 ) {
				float const current_value
					( pose.get_torsion_by_number( seqpos, torsion) );
				deriv += scale_factor * cst_set.rosetta_torsion_deriv( seqpos, torsion,
																															 current_value );
			}
		} // valid rosetta torsion
	} // non-zero weights
	return deriv;
}

///////////////////////////////////////////////////////////////////////////////
void
numerical_derivative_check(
	kin::Minimizer_map const & min_map,
	FArray1DB_float & phipsi, // should be const!
	FArray1DB_float & dE_dphipsi, // should be const!
	int nangles
)
{
	/////////////////////////////////////////////////////////////////////////////
	// NUMERICAL DERIVATIVE CHECK
	/////////////////////////////////////////////////////////////////////////////
	// how to analyze this:
	//
	// in gnuplot, look at numerical vs analytical derivs:
	// plot '< grep ratio a3.log ' u 6:7,x
	//
	// also sort by deriv_dev lines:
	//
	// by magnitude of deviation
	// grep deriv_dev a3.log | sort -g +8
	//
	// or by ratio of deviation to actual
	//
	// grep deriv_dev a3.log | sort -g +9

	float const increment = 0.0005; // PB -- 3/02
//  	increment = 0.001;
	int const n_increment = 5;
	FArray2D_float dE_dphipsi_numeric( nangles, n_increment );

	// setup for saving diagnostics
	min_debug::nangles = nangles;
	if ( nangles > int( min_debug::abs_deriv_dev.size1() ) ) {
		min_debug::abs_deriv_dev.dimension( nangles );
		min_debug::rel_deriv_dev.dimension( nangles );
	}

	bool gfrag = true;
	FArray1D_float phipsi_ori( nangles );
	for ( int i = 1; i <= nangles; ++i ) {
		phipsi_ori(i) = phipsi(i);
	}

	float f00 = func(gfrag,phipsi);
	int ii( 1 ); // for indexing into de_dphipsi( imap )
	for ( kin::Minimizer_map::Torsion_const_iterator
					tor_iter = min_map.torsions_begin(),
					tor_end = min_map.torsions_end(); tor_iter != tor_end;
				++tor_iter, ++ii ) {
		kin::Torsion_node const & tor( **tor_iter );

		bool force_zero_deriv( false );
		{ // check for omega
			int torsion,seqpos;
			tor.get_rosetta_type( torsion, seqpos );
			force_zero_deriv = ( torsion == param_torsion::omega_torsion &&
													 !minimize_ns::minimize_vary_omega );
		}

		if ( ii%10==0 )
			std::cout << "checking derivatives: angle= " << ii << " nangles= " <<
				nangles << std::endl;

		float deriv_dev = 10000.0;
		for ( int j = 1,factor=1; j <= n_increment; ++j ) {
			factor*=2;

			float deriv(0.0), f11( 0.0 ), f22( 0.0 );
			if ( !force_zero_deriv ) { // not a fixed omega angle
				phipsi_ori(ii) = phipsi(ii) + factor * increment;
				f11 = func(gfrag,phipsi_ori);
				phipsi_ori(ii) = phipsi(ii) - factor * increment;
				f22 = func(gfrag,phipsi_ori);
				deriv = ( f11 - f22 ) / ( factor * 2 * increment );
			}
			dE_dphipsi_numeric( ii, j ) = deriv;

			deriv_dev = std::min( deriv_dev, std::abs( deriv  - dE_dphipsi(ii) ) );

			phipsi_ori(ii) = phipsi(ii);

			float const ratio( std::abs( dE_dphipsi(ii) ) < 0.001 ? 0.0 :
												 deriv / dE_dphipsi(ii) );

			if ( runlevel_ns::runlevel > runlevel_ns::standard &&
					 ( std::abs(dE_dphipsi(ii)) > 0.001 || std::abs(deriv) > 0.01 ) ) {
				// if you change this output, please also change the comments
				// at the beginning of this section
				std::cout << "ratio" << I( 4, tor.type() ) << I( 4, j ) <<
					I( 4, tor.rsd() ) << I( 4, tor.atomno() ) <<
					F( 10, 4, deriv ) <<
					F( 10, 4, dE_dphipsi(ii) ) <<
					F( 10, 4, ratio ) <<
					F( 10, 4, 0.0 ) <<
					F( 10, 4, 0.0 ) <<
					F( 10, 4, f11 ) <<
					F( 10, 4, f00 ) <<
					F( 10, 4, f22 ) <<
					F( 10, 4, phipsi(ii) ) << std::endl;
			}
		}
		if ( true ) {

			float const ratio( std::abs( dE_dphipsi(ii) ) < 0.001 ? 0.0 :
												 deriv_dev / std::abs( dE_dphipsi(ii) ) );

			min_debug::rel_deriv_dev( ii ) = ratio;
			min_debug::abs_deriv_dev( ii ) = deriv_dev;


			if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
				// if you change this output, please also change the comments
				// at the beginning of this section
				std::cout << "deriv_dev:" << SS(ii) << SS(nangles) << SS(f00) <<
					SS( tor.type() ) << SS( tor.atomno() ) << SS( tor.rsd() ) <<
					SS( dE_dphipsi(ii) ) << SS( deriv_dev ) << SS(ratio) << std::endl;
			}
		}
	}

	// calculate magnitudes, dot products of gradient vectors
	FArray1D_float norm_numeric(n_increment,0.0), dot(n_increment,0.0);
	float norm(0.0);
	for ( int i=1; i<= nangles; ++i ) {
		norm += dE_dphipsi(i) * dE_dphipsi(i);
		for ( int j=1; j<= n_increment; ++j ) {
			dot(j) += dE_dphipsi(i) * dE_dphipsi_numeric(i,j);
			norm_numeric(j) += dE_dphipsi_numeric(i,j) * dE_dphipsi_numeric(i,j);
		}
	}
	norm = std::sqrt( norm );

	min_debug::best_cos_theta = -10.0;
	min_debug::best_abs_log_norm_ratio = 200.0;
	min_debug::best_norm_analytic = 999.9;
	min_debug::best_norm_numeric  = 999.9;

	for ( int j=1; j<= n_increment; ++j ) {
		norm_numeric(j) = std::sqrt( norm_numeric(j) );

		// handle strange cases
		float log_norm_ratio;
		if ( norm < 0.001 && norm_numeric(j) < 0.001 ) {
				log_norm_ratio = 1.0;
		} else if ( norm < 0.001 ) {
			log_norm_ratio = 100.0;
		} else if ( norm_numeric(j) < 0.001 ) {
			log_norm_ratio = -100.0;
		} else {
			log_norm_ratio = std::log( norm_numeric(j) / norm );
		}

		float const cos_theta( dot(j) / ( norm * norm_numeric(j)) );

		std::cout <<
			" norm: " << j << ' ' << norm <<
			" norm_numeric: " << norm_numeric(j) <<
			" cos_theta: " << cos_theta <<
			" log_norm_ratio: " << log_norm_ratio << std::endl;

		min_debug::best_cos_theta = std::max( min_debug::best_cos_theta,
																					cos_theta );
		if ( std::abs( log_norm_ratio ) < min_debug::best_abs_log_norm_ratio ) {
			min_debug::best_abs_log_norm_ratio = std::abs( log_norm_ratio );
			min_debug::best_norm_analytic = norm;
			min_debug::best_norm_numeric = norm_numeric(j);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
void
pose_eval_dunbrack_deriv(
	pose_ns::Pose & pose,
	FArray2D_float & dunbrack_deriv
	)
{
	assert( pose.fullatom() );

	// so all scoring becomes a lot simpler
	FArray2D_int rotarray( param::MAX_CHI, param::MAX_RES()() );
	FArray2D_float chiarray( param::MAX_CHI, param::MAX_RES()() );

	get_chi_and_rot_from_coords( pose.total_residue(), pose.res(),
		pose.res_variant(), pose.full_coord(), chiarray, rotarray );

	// prob dont need 1,total_residue
	eval_total_dunbrack_deriv( 1, pose.total_residue(), chiarray, rotarray,
														 dunbrack_deriv );
}



//////////////////////////////////////////////////////////////////////////////
int
get_fa_pair_base_atom(
	int const aa
)
{
	int const aav(1);
	assert( param_aa::is_protein(aa));
	int chino;
	get_wcentroid_closest_chi( chino, aa );
	if ( chino == 0 ) {
		if ( aa == param_aa::aa_gly ) {
			return 2; // CA
		} else {
			return 5; // CB
		}
	} else {
		return aaproperties_pack::chi_atoms(4,chino,aa,aav);
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_get_atompairE_deriv
///
/// @brief
///
/// @detailed
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  F1_atompairE - [in/out]? -
/// @param  F2_atompairE - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
atom_tree_get_atompairE_deriv(
	pose_ns::Pose & pose,
	kin::Minimizer_map & min_map,
	bool const fullatom
)
{
	bool const use_nblist( min_map.use_nblist() );
	using namespace pose_ns;
	using namespace kin;

	//car precalculate constants (weights and constants in derivatives)
	//car constants from form of derivatives
	//car  except 0.8 for vdw which is weight applied inside vdw_compute
	// these silly things are used inside the function eval_de_dr
	minimize_ns::deriv_precalc::vdw_factor = -4.*0.8*scorefxns::vdw_weight;
	minimize_ns::deriv_precalc::fa_rep_factor = param_pack::pack_wts.Wrep() *
		scorefxns::fa_rep_weight;
	minimize_ns::deriv_precalc::fa_atr_factor = param_pack::pack_wts.Watr() *
		scorefxns::fa_atr_weight;
	minimize_ns::deriv_precalc::fa_solv_factor = param_pack::pack_wts.Wsol() *
		scorefxns::fa_solv_weight;

	float const fa_pair_factor
		( param_pack::pack_wts.Wpair() * scorefxns::fa_pair_weight );

	// probably this is unnecessary since if two atoms are fixed relative
	// to one another then they wont be on each other's nblists:
	FArray1D_int const & jmp_domain_map ( pose.get_domain_map() );//

	//car obtain cendist to determine which atoms can be skipped
	if ( !use_nblist ) {
		pose_update_cendist( pose );
	}
	Score_state cendist_state; // may not be updated:
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST, cendist_state ));

	// other useful pose stuff
	int const total_residue( pose.total_residue() );
	FArray1D_int const & res( pose.res() );
	FArray1D_int const & res_variant( pose.res_variant() );
	FArray3D_float const & full_coord( pose.full_coord() );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// initialization for some derivative terms
	//

	// these may not be dimensioned if not used
	FArray2D_float actcoord;
	FArray1D_int neighbors;
	FArray2D_bool neighborlist;
	bool const calc_fa_pair( fullatom && fa_pair_factor != 0.0 );
	bool const update_neighbor_info
		( fullatom && ( calc_fa_pair || !use_nblist || water_exists() ));
	bool const eval_water_deriv
		(water_exists() && param_pack::pack_wts.Wh2o() !=0.0 && scorefxns::fa_h2o_weight !=0.0);

	{
		// fullatom neighbor info, for any terms that might need it
		if ( update_neighbor_info ) {
			actcoord.dimension( 3, param::MAX_RES()() );
			neighbors.dimension( param::MAX_RES()() );
			neighborlist.dimension( param::MAX_RES()(), param::MAX_RES()() );
			PROF_START( prof::DFUNC_NB );
			make_neighbor_info( res, total_residue, pose.full_coord(),
													neighborlist, neighbors);
			PROF_STOP ( prof::DFUNC_NB );
		}

		// hbonds
		if ( param_pack::pack_wts.hbond_wts_nonzero() ) {
			float sr_hbenergy, lr_hbenergy;
			PROF_START( prof::DFUNC_HB );
			evaluate_hbenergy( hbonds::hbderiv_ABE_GO, sr_hbenergy, lr_hbenergy ); // JSS just to fill arrays; return values unused.
			PROF_STOP ( prof::DFUNC_HB );
		}

		// geometric solvation
		if ( get_geometric_sol_flag() ) {
			fill_geo_sol_arrays(true,res,res_variant,pose.full_coord(),total_residue,neighborlist,true/*update_deriv*/);
		}

		// water hbonds
		if ( eval_water_deriv ) {
			fill_h2ototalE_arrays( res, res_variant, pose.full_coord(),total_residue,
														 neighborlist, hbonds::hbderiv_ABE_GO );
		}

		// pair term
		if ( calc_fa_pair ) {
			for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
				int const aa = res(seqpos);
				if ( param_aa::is_protein( aa ) ) {
					put_wcentroid( full_coord(1,1,seqpos), actcoord(1,seqpos), aa);
				}
			}
		}
	}



	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// loop through the moving torsions


	for ( Minimizer_map::Torsion_iterator tor_iter = min_map.torsions_begin(),
					tor_end = min_map.torsions_end(); tor_iter != tor_end; ++tor_iter ) {
		Torsion_node & tor( **tor_iter );

		// loop through atoms first moved by this torsion
		for ( std::vector< Atom_node* >::const_iterator atom1_iter =
						tor.atoms_begin(), atom1_end = tor.atoms_end();
					atom1_iter != atom1_end; ++atom1_iter ) {
			Atom_node* const atom1( *atom1_iter );
			int const rsd1( atom1->rsd() );
			int const atomno1( atom1->atomno() );
			int const map1( jmp_domain_map( rsd1 ) );
			int const aa1 ( res( rsd1 ) );
			int const aav1( res_variant( rsd1 ) );

			/////////////////////////////////////////////////////////////////////////
			/////////////////////////////////////////////////////////////////////////
			// here you can put derivatives that just need this atom

			// new constraint derivative
			{
				fill_atom_cst_F1_F2( pose, scorefxns::kin_3D_cst_weight,
					scorefxns::atompair_cst_weight, scorefxns::coord_cst_weight,
					scorefxns::chainbreak_cst_weight,
					rsd1, atomno1, tor.F1(), tor.F2() );
			}

			// hbonds
			if ( param_pack::pack_wts.hbond_wts_nonzero() ) {
				compute_atom_hbond_deriv( atomno1, rsd1, tor.F1(), tor.F2() );
			}

			// water hbonds
			if ( eval_water_deriv ) {
				compute_atom_h2o_hb_deriv( atomno1, rsd1, scorefxns::fa_h2o_weight *
																	 param_pack::pack_wts.Wh2o(), tor.F1(), tor.F2() );
			}

			// geometric solvation
			if ( get_geometric_sol_flag() ) {
				compute_atom_geom_sol_deriv( atomno1, rsd1, tor.F1(), tor.F2() );
			}

			// gen_born
			if ( param_pack::gen_born &&
					 param_pack::pack_wts.Wgb_elec() * scorefxns::fa_gb_elec_weight != 0.0 ) {
				compute_atom_gen_born_deriv( atomno1, rsd1,
					param_pack::pack_wts.Wgb_elec() * scorefxns::fa_gb_elec_weight, pose,
					tor.F1(), tor.F2() );
			}

			// fa pair term
			if ( calc_fa_pair ) {
				if ( param_aa::is_protein( aa1 ) &&
						 atomno1 == get_fa_pair_base_atom( aa1 ) ) {
					get_atom_fa_pair_deriv( rsd1, total_residue, res, actcoord,
																	neighborlist, fa_pair_factor,
																	tor.F1(), tor.F2() );
				}
			}

			// DNA knowledge-based derivs
			if ( param_aa::is_DNA( aa1 ) &&
					 scorefxns::dna_bp_weight != 0.0 &&
					 atomno1 == get_dna_bp_atom( aa1, aav1 ) ) {
				calc_dna_bp_deriv( rsd1, pose, tor.F1(), tor.F2() );
			}
			if ( param_aa::is_DNA( aa1 ) &&
					 scorefxns::dna_bs_weight != 0.0 &&
					 atomno1 == get_dna_bp_atom( aa1, aav1 ) ) {
				calc_new_dna_bs_deriv( rsd1, pose, tor.F1(), tor.F2() );
			}

			/////////////////////////////////////////////////////////////////////////
			// do we need to do atom pairs?
			if ( fullatom &&
					 ( std::abs( param_pack::pack_wts.Watr()*scorefxns::fa_atr_weight ) +
						 std::abs( param_pack::pack_wts.Wrep()*scorefxns::fa_rep_weight ) +
						 std::abs( param_pack::pack_wts.Wsol()*scorefxns::fa_solv_weight ) ) < 1e-2 ){
				continue;
			}


			/////////////////////////////////////////////////////////////////////////
			// now we loop over neighboring atoms

			std::vector< Atom_node* >::const_iterator atom2_begin;
			std::vector< Atom_node* >::const_iterator atom2_end;
			min_map.get_nbr_bounds( atom1, atom2_begin, atom2_end );

			for ( std::vector< Atom_node* >::const_iterator atom2_iter = atom2_begin;
						atom2_iter != atom2_end; ++atom2_iter ) {
				Atom_node* const atom2( *atom2_iter );

				if ( atom1 == atom2 ) continue;

				int const atomno2 = atom2->atomno();
				int const rsd2 = atom2->rsd();

				if ( map1 == jmp_domain_map( rsd2 ) && map1 != 0 ) continue;

				if ( !min_map.use_nblist() ) {
					if ( fullatom && update_neighbor_info &&
							 !neighborlist( rsd1, rsd2 ) ) continue;
					if ( !fullatom &&
							 cendist( rsd1, rsd2 ) > cenlist_ns::cen_dist_cutoff2 ) continue;
				}

				FArray1D_float f1(3,0.0), f2(3,0.0);

				float dE_dR;
				eval_dE_dR( rsd1, atomno1, rsd2, atomno2,
				 pose.res( rsd1 ), pose.res( rsd2 ),
				 pose.res_variant( rsd1 ), pose.res_variant( rsd2 ),
				 fullatom, dE_dR, f1, f2 );

				if ( dE_dR != 0.0 ) {
					for ( int k = 1; k <= 3; ++k ) {
						tor.F1()(k) += dE_dR*f1(k);
						tor.F2()(k) += dE_dR*f2(k);
					}
				}
			} // atom2
		} // atom1
	} // tor


}


//////////////////////////////////////////////////////////////////////////////
// this assumes that the radii are not changing, since in that case
// we can neglect interactions between residues that aren't moving wrt1another

void
compute_atom_gen_born_deriv(
	int const ii, // atomno
	int const i, // rsd
	float const gb_weight,
	pose_ns::Pose const & pose,
	numeric::xyzVector_float & F1, // accumulate contributions
	numeric::xyzVector_float & F2
)
{
	using numeric::xyzVector_float;
	using namespace gb_elec;
	using namespace param;
	using namespace param_aa;

	assert( param_pack::gen_born );

	// pose stuff
	int const nres( pose.total_residue() );
	FArray1D_int const & domain_map( pose.get_domain_map() );
	FArray3D_float const & coords( pose.full_coord() );
	int const resi ( pose.res( i ) );
	int const resvi( pose.res_variant( i ) );
	int const i_map( domain_map( i ) );
	bool const i_fixed( i_map != 0 );
	xyzVector_float xyzi( &coords(1,ii,i) );

	// gb stuff
	float const Ep = 4.0; // dielectric for protein
	float const Ew = 80.0; // dielectric for water
	float const tau = (1.0/Ep - 1.0/Ew);
	float const q1 = aaproperties_pack::atomic_charge( ii, resi, resvi );
	float const b1 = template_pack::born_radius( ii, i );

	for ( int j=1; j<= nres; ++j ) {
		int const resj ( pose.res(j) );
		int const resvj( pose.res_variant(j) );
		if ( i_fixed && domain_map(j) == i_map ) continue; // DANGER

		for ( int jj=1, jj_end=aaproperties_pack::natoms(resj,resvj); jj<= jj_end;
					++jj ) {
			if ( !count_pair_gb( ii, i, resi, resvi, jj, j, resj, resvj ) ) continue;
			bool const same_atom( j == i && jj == ii );

// 			++gb_atom_count;
// 			++gb_dfunc_pair_count( i, j );
// 			++gb_dfunc_pair_count( j, i );

// 			if ( i==1 && j==1 ) {
// 				std::cout << "11_dfunc: " << ii << ' ' << jj << std::endl;
// 			}

			xyzVector_float const xyzj( &coords(1,jj,j) );
			xyzVector_float const f2( xyzi - xyzj );

			float const dis2( f2.length_squared() );
			float const dis = std::sqrt( dis2 );

			///jjh First the dielectric screening term
			float const q2 = aaproperties_pack::atomic_charge( jj, resj, resvj );
			float const b2 = template_pack::born_radius( jj, j );

			float const exparg = (-dis2)/(4.0*b1*b2);
			float const expon = std::exp( exparg );
			float const denom  = 1.0 / ( std::sqrt( dis2 + b1*b2*expon ) );
			float const deriv_denom = denom * denom * denom;

			float dE_dR
				( +166.0*tau*q1*q2*dis * ( 2.0 - 0.5*expon ) * deriv_denom );

			float const r1
				( gb_radius(aaproperties_pack::fullatom_type(ii,resi,resvi)) );
			float const r2
				( gb_radius(aaproperties_pack::fullatom_type(jj,resj,resvj)) );

			float const dE_dR_coul( gb_shell_intxn_deriv( q1, r1, q2, r2, dis ) );

			dE_dR += 332.0*dE_dR_coul/Ep;

			if ( same_atom ) dE_dR *= 0.5;

			if ( dis > 0.0 ) {
				float const factor( gb_weight * dE_dR / dis );
				xyzVector_float f1( cross( xyzi, xyzj ) );
				F1 += factor * f1;
				F2 += factor * f2;
			}
		}
	}
}


//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_get_total_pair_deriv
///
/// @brief
///
/// @detailed
///    Gets the F1 and F2 pair energy derivative vectors from the
///  Braun and Go recursive torsion derivative scheme for every
///  torsion angle on the protein backbone [in the range first_res-last_res]
///    These results can be
///  subsequently combined with any subset of torsion angles in imap
///  to get any set of dpairE_dtorsion.
///
///  ...lots of this is copied from code by DB and Bill Wedemeyer
///tsa
///
/// @param first_res - [in/out]? -
/// @param last_res - [in/out]? -
/// @param F1_pair - [in/out]? -
/// @param F2_pair - [in/out]? -
//
//
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_atom_fa_pair_deriv(
	int const seqpos1,
	int const total_residue,
	FArray1D_int const & res,
	FArray2D_float const & actcoord,
	FArray2D_bool const & neighborlist,
	float const weight,
	numeric::xyzVector_float & F1,
	numeric::xyzVector_float & F2
)
{
	using namespace param;
	using namespace param_aa;
	using namespace param_torsion;

	int const aa1 = res(seqpos1);
	for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
		if ( seqpos1 == seqpos2 || ! neighborlist( seqpos1, seqpos2 ) ) continue;
		int const aa2 = res( seqpos2 );
		FArray1D_float f1(3,0.0), f2(3,0.0);
		calc_and_sum_pair_deriv(aa1,aa2,seqpos1,seqpos2,
														actcoord(1,seqpos1), actcoord(1,seqpos2),
														f1,f2);
		for ( int k=1; k<= 3; ++k ) {
			F1(k) += weight * f1(k);
			F2(k) += weight * f2(k);
		}
	} // seqpos2
}

///////////////////////////////////////////////////////////////////////////////
float
atom_tree_func_vdw(
	pose_ns::Pose & pose,
	FArray1DB_float const & phipsi
)
{
	PROF_START( prof::ATOM_TREE_FUNC_VDW );
	min_map_ns::min_map.unpack_phipsi( pose, phipsi );

	// I think this refers to the routine "scorefxn()" in score.cc
	float const score( pose.score( scorefxn ) );
	PROF_STOP( prof::ATOM_TREE_FUNC_VDW );

	return score;
}



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// should move this routine, later
namespace kin {
void
Minimizer_map::update_nblist(
	pose_ns::Pose & pose
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace param_torsion;
	using numeric::xyzVector_float;
	using namespace pose_ns;

	PROF_START( prof::MIN_MAP_UPDATE_NBLIST );

	bool const fullatom ( pose.fullatom() );

	// update cendist if necessary
	if ( !fullatom ) pose_update_cendist( pose );
	Score_state cendist_state;
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST,cendist_state ) );
	if ( !fullatom ) assert( cendist_state == GOOD );

	// fullatom nbr information:
	FArray2D_bool nbres;
	FArray1D_int nbrs;
	if ( fullatom ) {
		nbres.dimension( param::MAX_RES()(), param::MAX_RES()() );
		nbrs.dimension( param::MAX_RES()() );
		PROF_START( prof::MMUN_NB_INFO );
		make_neighbor_info( pose.res(), pose.total_residue(), pose.full_coord(),
												nbres,nbrs);
		PROF_STOP ( prof::MMUN_NB_INFO );
	}

	// get domain map
	pose.set_atom_tree_moved();
	FArray1D_int const & jmp_domain_map ( pose.get_domain_map() );

	// NOTE: we traverse only half of the atom pairs!!!!!!!!!!!!!!!!!!!!!!!!!
	int total_nbrs(0);
	for ( int i = 1; i <= pose.total_residue(); ++i ) {
		int const aa1 = pose.res(i);
		int const aav1 = pose.res_variant(i);
		int const imap = jmp_domain_map(i);
		bool const ifixed ( imap != 0 );

		for ( int j = i; j <= pose.total_residue(); ++j ) {
			if ( ( imap == jmp_domain_map(j) && ifixed ) ||
					 (  fullatom && !nbres(i,j) ) ||
					 ( !fullatom && cendist(i,j) > cenlist_ns::cen_dist_cutoff2 ) ) {
				continue;
			}

			int const aa2 = pose.res(j);
			int const aav2 = pose.res_variant(j);
			int const iiend( fullatom ? natoms(aa1,aav1) : 6 );
			for ( int ii = 1; ii <= iiend; ++ii ) {
				if ( ! fullatom && aa1 == aa_gly && ii == 5 ) continue;
				{ // Scope
	  		xyzVector_float full_coord_ii_i( &(pose.full_coord()(1,ii,i) ) );

				int const type1 = all_atom_type(aa1,aav1,ii,fullatom);
				int const jjend( fullatom ? natoms(aa2,aav2) : 6 );
				for ( int jj = 1; jj <= jjend; ++jj ) {
					float cp_weight;
					if ( ( i == j && jj <= ii ) ||
							 !count_pair(i,ii,aa1,aav1,j,jj,aa2,aav2,fullatom,cp_weight) ||
							 ( !fullatom && aa2 == aa_gly && jj == 5 ) ) continue;

					float dist_sq;
					if ( ! fullatom && ii == 6 && jj == 6 ) {
						dist_sq = cendist(i,j);
					} else {
						dist_sq = distance_squared(
						 full_coord_ii_i,
						 xyzVector_float( &(pose.full_coord()(1,jj,j) ) ) );
					}
					int const type2 = all_atom_type(aa2,aav2,jj,fullatom);
					if ( dist_sq < all_atom_cutoff(type1,type2,fullatom) ) {
						this->add_to_nblist( ii, i, jj, j );
						++total_nbrs;
						assert( j>= i);
						if ( false || type1 == water::type_h2o ||
								 type2 == water::type_h2o ) {
							std::cout << "nb" << I( 4, i ) << I( 4, ii ) << I( 4, j ) <<
								I( 4, jj ) << F( 7, 3, dist_sq ) <<
								F( 7, 3, all_atom_cutoff(type1,type2,fullatom) ) <<
								std::endl;
						}
					}
				} // jj
				} // Scope
			} // ii
		} // j
	} // i
	//std::cout << "total_nbrs: " << total_nbrs << std::endl;
	PROF_STOP( prof::MIN_MAP_UPDATE_NBLIST );
}
}// namespace kin

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
