// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20369 $
//  $Date: 2008-02-13 23:38:22 -0500 (Wed, 13 Feb 2008) $
//  $Author: bblum $

#ifndef INCLUDED_barcode_classes
#define INCLUDED_barcode_classes


// Rosetta Headers
#include "util_basic.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/all.fwd.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
//#include <iosfwd>
#include <cassert>
#include <vector>
#include <string>
#include <map>


///////////////////////////////////////////////////////////////////////////
/// this file has the classes for barcode constraints
/// there are four:
/// "constraint", "flavor", "feature", and "feature_set"
///
/// a constraint is a single constraint: PHI = xxx, chi1 = xxx, etc
///
/// a flavor is a set of constraints to be applied together
///
/// a feature is a group of flavors (alternatives), a single one of which
///  is applied at a time. For example, a single feature might correspond
///  to the side chain of residue 44. flavor1 could be chi1=56,chi2=0
///  flavor2: chi1=120,chi2=-120, flavor3= rot1=1,rot2=2, etc
///  each flavor has an associated frequency which can be used to
///  select a single one probabilistically.
///
/// a feature_set is a set of features, independently chosen and applied
///  during a run.
///
///////////////////////////////////////////////////////////////////////////


// hide everything inside a namespace:
namespace barcode_classes {

	namespace barcode_param {
		extern int const PHI_TORSION;
		extern int const PSI_TORSION;
		extern int const OMEGA_TORSION;
		extern int const CHI1_TORSION; // marks the beginning of the chi,rot torsions
		extern int const ROT1_TORSION; //  = CHI1_TORSION+4
		extern int const BB_BIG_BIN_TORSION;
		extern int const SS_TORSION;
		extern int const PHI_RANGE_TORSION;
		extern int const BB_SMALL_BIN_TORSION;
		extern int const SC_BIN_TORSION;
		extern int const BB_CLUSTER_TORSION;
		extern int const SSPAIR_TORSION;
		extern int const CONTACT_TORSION;
		extern int const DISULF_TORSION;

		extern int const CONTACT_ENERGY_TORSION;
		extern int const BETA_PAIR_ENERGY_TORSION;
		extern int const SS_ENERGY_TORSION;
		extern int const BIG_BIN_ENERGY_TORSION;
		extern int const DSSP_ENERGY_TORSION;
		extern int const BBLUM_ENERGY_TORSION;
		extern int const ENV_ENERGY_TORSION;
		
		extern int const RNA_BP_JUMP_TORSION;
	}

	//------------------------------------------------------------------------------
	// simple classes:

	//////////////////////////////////////////// Bottom level: single constraint
	// I really should have a hierarchy of different constraint classes,
	// and use inheritance, that way value wouldn't always have to be a float (eg)
	//
	// but this is already more OO than I'm used to

	class constraint {
	public:
		static int const ANTIPARALLEL = 1;
		static int const PARALLEL = 2;
	public:
		int residue;
		int torsion; // see barcode_param
		float weight;
		// have multiple value types for different constraints
		// this is a little wasteful and not very pretty...
		// probably should use inheritance
		float float_value;
		char char_value;
		char char_value1;
		char char_value2;
		int int_value;
		int int_value1;
		int int_value2;
		int int_value3;
		int int_value4;
		float float_value1;
		float float_value2;
		float float_value3;

		// constructor
		constraint ():
			float_value(0),
			char_value('-'),
			int_value(0),
			int_value1(0),
			int_value2(0),
			int_value3(0),
			int_value4(0),
			float_value1(0),
			float_value2(0),
			float_value3(0)
		{}

		// these helper functions tell others what type of constraint we are
		inline bool bb_big_bin_constraint() const;
		inline bool sc_constraint() const;
		inline bool chi_constraint() const;
		inline bool rot_constraint() const;
		inline bool ss_constraint() const;
		inline bool bb_tether_constraint() const;
		inline bool bb_small_bin_constraint() const;
		inline bool sc_bin_constraint() const;
		inline bool bb_cluster_constraint() const;
		inline bool sspair_constraint() const;
		inline bool sspair_constraint_bonus() const;
		inline bool contact_constraint() const;
		inline bool disulf_constraint() const;
		inline bool beta_pair_constraint() const;
		inline bool beta_pair_constraint_bonus() const;

		inline int chi_number() const;

		// scoring routines
		inline float bb_score(
			float phi,
			float psi,
			float omega,
			char secstruct
		) const;

		inline bool rot_match(
			const FArray1DB_int & rot,
			const int nchi_aa_aav
		) const;

		inline bool sc_bin_match(
			const FArray1DB_float & chi,
			const int nchi_aa_aav
		) const;

		inline float sspair_constraint_score(
			int const pos1,
			int const pos2,
			float const theta
		) const;

		inline float contact_score(
			int const nres_in,
			FArray2D_float const & cendist
		) const;

		std::string to_string() const;

	};

	typedef std::vector< constraint >::iterator constraint_iterator;
	typedef std::vector< constraint >::const_iterator constraint_const_iterator;

	// a < operator so we can sort a list of constraints:
	bool operator <( constraint const & a, constraint const & b );
	// stream inserter
	std::ostream & operator <<( std::ostream & os, constraint const & c );
	// stream extractor
	std::istream & operator >>( std::istream & is, constraint & c );

	// inline definitions
	inline bool constraint::bb_big_bin_constraint() const {
		return ( torsion == barcode_param::BB_BIG_BIN_TORSION );
	}

	inline bool constraint::sc_constraint() const {
		// sc_constraint includes chi, rot, and sc_bin constraints => 4+4+1 = 9
		return ( torsion >= barcode_param::CHI1_TORSION &&
						 torsion <  barcode_param::CHI1_TORSION+9 );
	}

	inline bool constraint::chi_constraint() const {
		return ( torsion >= barcode_param::CHI1_TORSION &&
						 torsion <  barcode_param::ROT1_TORSION );
	}

	inline bool constraint::rot_constraint() const {
		return ( torsion >= barcode_param::ROT1_TORSION &&
						 torsion <  barcode_param::ROT1_TORSION+4 );
	}

	inline bool constraint::ss_constraint() const {
		return ( torsion == barcode_param::SS_TORSION );
	}

	inline bool constraint::bb_tether_constraint() const {
		return ( torsion == barcode_param::PHI_TORSION ||
						 torsion == barcode_param::PSI_TORSION ||
						 torsion == barcode_param::OMEGA_TORSION );
	}

	inline bool constraint::bb_small_bin_constraint() const {
		return ( torsion == barcode_param::BB_SMALL_BIN_TORSION );
	}

	inline bool constraint::sc_bin_constraint() const {
		return ( torsion == barcode_param::SC_BIN_TORSION );
	}

	inline bool constraint::bb_cluster_constraint() const {
		return ( torsion == barcode_param::BB_CLUSTER_TORSION );
	}

	inline bool constraint::sspair_constraint() const {
		return ( torsion == barcode_param::SSPAIR_TORSION );
	}

	inline bool constraint::sspair_constraint_bonus() const {
		return ( torsion == barcode_param::SSPAIR_TORSION & weight < 0.0);
	}

	inline bool constraint::contact_constraint() const {
		return ( torsion == barcode_param::CONTACT_TORSION );
	}

	inline bool constraint::disulf_constraint() const {
		return ( torsion == barcode_param::DISULF_TORSION );
	}

	inline bool constraint::beta_pair_constraint() const {
		return ( torsion == barcode_param::BETA_PAIR_ENERGY_TORSION );
	}

	inline bool constraint::beta_pair_constraint_bonus() const {
		return ( torsion == barcode_param::BETA_PAIR_ENERGY_TORSION & weight < 0.0);
	}

	// which chi number does this constraint apply to?
	// for rot constraints, this means all chis up to and including this
	// constraint
	inline int constraint::chi_number() const {
		if ( chi_constraint() ) {
			return torsion - barcode_param::CHI1_TORSION + 1;
		} else if ( rot_constraint() ) {
			return torsion - barcode_param::ROT1_TORSION + 1;
		} else {
			return 0;
		}
	}

	//////////////////////////
	// useful helper function:
	inline bool angle_bin_match(
		const float a,
		const float a_min,
		const float a_max
	)
	{
		return ( ( ( a_min <  a_max ) && ( a_min <= a && a <= a_max ) ) ||
						 ( ( a_min >= a_max ) && ( a_min <= a || a <= a_max ) ) );
	}

	// score for backbone bin and SS constraint
	inline float constraint::bb_score( float phi, float psi, float omega, char secstruct ) const
	{
		float score = 0.0;
		if ( torsion == barcode_param::BB_BIG_BIN_TORSION ) { // magic number!
			// constrain to (-180,180]
			periodic_range( phi  , 360.0 );
			periodic_range( psi  , 360.0 );
			periodic_range( omega, 360.0 );

			// convert to bin
			char bb_bin = ' '; // dummy
			// the stream extractor in barcode_classes.cc has a copy of these
			// five character values, so they should be kept in sync
			if ( std::abs( omega ) < 90 ) {
				bb_bin = 'O'; // cis-omega
			} else if ( phi >= 0.0 ) {
				if ( -100 < psi && psi <= 100 ) {
					bb_bin = 'G'; // alpha-L
				} else {
					bb_bin = 'E'; // E
				}
			} else {
				if ( -125 < psi && psi <= 50 ) {
					bb_bin = 'A'; // helical
				} else {
					bb_bin = 'B'; // extended
				}
			}
			assert( bb_bin != ' ' );
			if ( char_value != bb_bin ) score = weight;
		} else if ( torsion == barcode_param::SS_TORSION ) {
			if ( char_value != secstruct ) score = weight;
		} else if ( torsion == barcode_param::BB_SMALL_BIN_TORSION ) {
			// constrain to (-180,180]
			periodic_range( phi, 360.0 );
			periodic_range( psi, 360.0 );

			// recopying: slower but clearer
			float const phi_min = float_value;
			float const phi_max = float_value1;
			float const psi_min = float_value2;
			float const psi_max = float_value3;

			if ( angle_bin_match( phi, phi_min, phi_max ) &&
					 angle_bin_match( psi, psi_min, psi_max ) ) {
				score = weight; //in this case negative
			}
		} else if ( torsion == barcode_param::BB_CLUSTER_TORSION ) {
			// recopying: slower but clearer
			float const phi_cen = float_value;
			float const psi_cen = float_value1;
//			float const omega_cen = float_value2;
			float const threshold = float_value3;
			float phi_dev   = subtract_degree_angles( phi,   phi_cen  );
			float psi_dev   = subtract_degree_angles( psi,   psi_cen  );
//			float omega_dev = subtract_degree_angles( omega, omega_cen);
			score = 0.0;
			if ( phi_dev > threshold ) {
				//score+= (phi_dev - threshold)*(phi_dev-threshold);
				score += 1*(phi_dev - threshold);
			}
			if ( psi_dev > threshold ) {
				//				score+= (psi_dev - threshold)*(psi_dev-threshold);
				score += 1*(psi_dev - threshold);
			}
			score *= weight;
		}
		return score;
	}

	/////////////////////////////////////////////////////////////////////////////////
	inline
	bool
	constraint::rot_match(
		const FArray1DB_int & rot,
		const int nchi_aa_aav
	) const
	{
		assert( rot_constraint() );
		bool match(true);
		const int nchi( std::min( chi_number(), nchi_aa_aav ) );
		FArray1D_float target_rot(4);
		target_rot(1) = float_value;
		target_rot(2) = float_value1;
		target_rot(3) = float_value2;
		target_rot(4) = float_value3;
		for ( int chino = 1; chino <= nchi; ++chino ) {
			if ( std::abs( target_rot(chino) - rot(chino) ) > 0.5 ) {
				match = false;
				break;
			}
		}
		return match;
	}

	/////////////////////////////////////////////////////////////////////////////////
	inline
	bool
	constraint::sc_bin_match(
		const FArray1DB_float & chi,
		const int nchi_aa_aav
	) const
	{
		assert( sc_bin_constraint() );

		float chi1 = periodic_range( chi(1), 360.0 );
		float chi2 = periodic_range( chi(2), 360.0 );

		// recopying: slower but clearer
		const float chi1_min = float_value;
		const float chi1_max = float_value1;
		const float chi2_min = float_value2;
		const float chi2_max = float_value3;

		return ( ( nchi_aa_aav < 1 || angle_bin_match( chi1, chi1_min, chi1_max ) ) &&
						 ( nchi_aa_aav < 2 || angle_bin_match( chi2, chi2_min, chi2_max ) ) );
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	float
	constraint::sspair_constraint_score(
		int const pos1,
		int const pos2,
		float const theta
	) const
	{
		assert( sspair_constraint() );

		if ( theta < 0.0 || theta > 180.0 ) {
			std::cout << "WARNING:: sspair_constraint_score: theta should lie " <<
				" between 0 and 180 degrees! theta = "<< theta << std::endl;
			return 0.0;
		}

		int const strand1_begin( int_value1 );
		int const strand1_end  ( int_value2 );
		int const strand2_begin( int_value3 );
		int const strand2_end  ( int_value4 );
		int const orientation( int_value );

		if ( ( ( pos1 >= strand1_begin && pos1 <= strand1_end &&
						 pos2 >= strand2_begin && pos2 <= strand2_end ) ||
					 ( pos1 >= strand2_begin && pos1 <= strand2_end &&
						 pos2 >= strand1_begin && pos2 <= strand1_end ) ) &&
				 ( ( orientation == PARALLEL && theta <= 90.0 ) ||
					 ( orientation == ANTIPARALLEL && theta > 90.0 ) ) ) {
			return weight; // bonus or penalty
		} else {
			return 0.0;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	float
	constraint::contact_score(
		int const nres_in,
		FArray2D_float const & cendist
	) const
	{
		assert( contact_constraint() );

		float const threshold2( float_value * float_value );
		int const range1_begin( int_value );
		int const range1_end  ( static_cast< int >( float_value1 ) );
		int const range2_begin( static_cast< int >( float_value2 ) );
		int const range2_end  ( static_cast< int >( float_value3 ) );

		assert( std::abs( range1_end - float_value1 ) < 0.01 ); // sanity
		if ( range1_begin < 1     || range2_begin < 1 ||
				 range1_end > nres_in || range2_end > nres_in ) {
			std::cout << "bad ranges in contact constraint: " << nres_in << ' ' <<
				range1_begin << ' ' << range1_end << ' ' <<
				range2_begin << ' ' << range2_end << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		bool matched( false );
		for ( int i=range1_begin; i<= range1_end; ++i ) {
			for ( int j=range2_begin; j<= range2_end; ++j ) {
				if ( cendist(i,j) <= threshold2 ) {
					matched = true;
					break;
				}
			}
			if ( matched ) break;
		}

		if ( matched ) {
			return weight;
		} else {
			return 0.0;
		}
	}

	/////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////// set of constraints to be applied together:

	class flavor {
	public:
		std::vector< constraint > constraints;
		float frequency;

		flavor() {}

		flavor( float const f ):
			frequency ( f )
		{}

		friend std::ostream & operator<<(std::ostream & out, const flavor& flav);

	};
	typedef std::vector< flavor >::iterator flavor_iterator;


	/////////////////////////////////////////// a single feature, can have multiple
	/////////////////////////////////////////// states (flavors)

	class feature {
	private:
		std::vector< flavor > flavors;
		//flavor_iterator current_flavor;

		unsigned int current_flavor_index;

	public:
		feature():
			flavors( 1, flavor( 1.0 ) ),
			//current_flavor( flavors.begin() )
			current_flavor_index(0)
		{}

		flavor_iterator begin() {return flavors.begin();}
		flavor_iterator end() {return flavors.end();}

		bool increment() {
			bool flipped = false;
// 			++current_flavor;
// 			if ( current_flavor == flavors.end() ) {
// 				current_flavor = flavors.begin();
// 				flipped = true;
// 			}

			++current_flavor_index;
			if( current_flavor_index >= flavors.size() ) {
				current_flavor_index = 0;
				flipped = true;
			}
			return flipped;
		}

		float current_frequency() {
			//return current_flavor->frequency;
			return flavors[current_flavor_index].frequency;
		}

		unsigned int current_index() {
			return current_flavor_index;
		}

		void set_current_index(unsigned int index) {
			if(flavors.size() > index)
				current_flavor_index = index;
		}

		void new_flavor( const flavor & f ) {
			// figure out where the current_flavor pointer is
			//			int current_pos( 0 );
			// 			if ( current_flavor >= flavors.begin() &&
			// 					 current_flavor < flavors.end() ) {
			// 				while ( current_flavor != flavors.begin() ) {
			// 					++current_pos;
			// 					--current_flavor;
			// 				}
			// 			}

			// this is tricky: push_back call may trigger re-allocation
			// of space, invalidating the pointer current_flavor
			flavors.push_back(f);

			// 			current_flavor = flavors.begin();
			// 			current_flavor += current_pos;

			flavor_iterator it = flavors.begin();
			it->frequency -= f.frequency;

			// debugging:
			if ( it->frequency < -1.0e-3 ) {
				// error message
				std::cout << "WARNING: flavor frequency sum exceeds 1" << std::endl;
				//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		void new_flavors( const std::vector<flavor> & vFlavors ) {
			for( std::vector<flavor>::const_iterator i = vFlavors.begin(); i != vFlavors.end(); ++i ) {
				new_flavor(*i);
			} // i
		} // new flavors

		void reset() {
			//current_flavor = flavors.begin();
			current_flavor_index = 0;
		}

		void
			add_active_constraints(
				std::vector< constraint > & active_list
		) {
		// add all the current flavor's constraints to cst_list

		// if we are in the off-state, this loop will not execute
		// (current_flavor->constraints will be an empty list)
			flavor& current_flavor = flavors[current_flavor_index];
// 			for ( constraint_iterator it = current_flavor->constraints.begin();
// 						it != current_flavor->constraints.end(); ++it ) {
 			for ( constraint_iterator it = current_flavor.constraints.begin();
 						it != current_flavor.constraints.end(); ++it ) {
				active_list.push_back ( *it ); // makes a copy
			}
		}

		void add_all_constraints( std::vector< constraint > & active_list ) {
			// add all the constraints to cst_list
			for ( flavor_iterator it1 = begin(); it1 != end(); ++it1 ) {
				for ( constraint_iterator it = it1->constraints.begin(),
								it_end = it1->constraints.end(); it != it_end; ++it ) {
					active_list.push_back ( *it ); // makes a copy
					std::cout << "add_all_const "<< *it << std::endl;
				}
			}
		}

		int size() { return flavors.size(); }

		friend std::ostream & operator<<(std::ostream & out, const feature& feat);

	};
		

	typedef std::map< std::string, feature >::iterator feature_iterator;
	//	typedef std::map< std::string, feature >::const_iterator feature_const_iterator; // people should learn about const correctness

	//////////////////////////////////////////TOP level object
	class feature_set {

	private:

		std::map< std::string, feature > features; // from tag to feature
		std::vector< constraint > active_constraints;

		std::map< int, constraint_iterator > sc_begin_map;
		std::map< int, constraint_iterator > sc_end_map;

		int nres; // how long we think the protein is
		int mode; // mode for incrementing counters

		feature_iterator current_feature;

		float
		flavordist(
			flavor f,
			FArray1D_float const & phi,
			FArray1D_float const & psi
		);

		// should sc constraints be applied within get_rotamers?
		bool force_rotamer;

	public:

		feature_set():
			nres(0),
			force_rotamer(false)
		{} // silly, but maybe we want something else later?


		// called when feature states have changed, also sets up sc_begin_map, sc_end_map;
		// called by increment_counters

		void update_active_constraints();
		void loop_update_active_constraints(
				int const & loop_begin,
				int const & loop_end,
				std::vector< int > const & free_res,
				int const & loop_begin_extension,
				int const & loop_end_extension
		);

		// called before we start each decoy
		// should be called before we do any scoring or anything
		void increment_counters();

		//sets mode and nres
		void clear(); // clears the features vector, != reset
		void read_file( std::string const & filename, int const nres, int const mode, bool const verbose );
		void analyze_fragments( int const nres, int const mode );

		void show_data(std::ostream& out);

		void
		reset()
		{
			for ( feature_iterator it = features.begin(); it != features.end(); ++it ) {
				it->second.reset(); // reset that feature --> goes to off-state
			}
			current_feature = features.begin(); // for modes 2,4
			update_active_constraints();
		}

		void
		fill_bb_tether_arrays(
			FArray1Da_float tether_angle_res_weight,
			FArray1Da_float phi_tether,
			FArray1Da_float psi_tether
		);

		void
		fill_barcode_energy_arrays();

		void
		update_pairing_list_RNA(); //lives in pose_rna_jumping.cc

		// return number of constraints
		inline
		int
		size()
		{
			return active_constraints.size();
		}


		// return pointers to beginning and end of constraints
		inline constraint_iterator begin() { return active_constraints.begin(); }
		inline constraint_iterator end() { return active_constraints.end(); }
		inline constraint_const_iterator begin() const { return active_constraints.begin(); }
		inline constraint_const_iterator end() const { return active_constraints.end(); }

		inline std::map< std::string, feature >::iterator feature_begin() { return features.begin(); }
		inline std::map< std::string, feature >::iterator feature_end() { return features.end(); }

		inline unsigned int num_features() { return features.size(); }

		// return pointers that define the range of sc constraints for each residue
		constraint_iterator
		sc_begin( int rsd ) {
			assert( 1 <= rsd && rsd <= nres );
			return sc_begin_map[ rsd ];
		}

		constraint_iterator
		sc_end( int rsd ) {
			assert( 1 <= rsd && rsd <= nres );
			return sc_end_map[ rsd ];
		}

		// output info on current (active) constraints to ostream
		void show_constraints( utility::io::orstream & out );

		std::string concise_flavor_output();

		// added by sheffler 11/5/04
		// computes current barcode WRT this feature_set
		// ONLY WORKS FOR BB_SMALL_BIN FEATURES!!!
		std::map< std::string, int > get_closest_barcode();

		std::map< std::string, int >
		get_closest_barcode(
			FArray1D_float const & phi,
			FArray1D_float const & psi
		);

		void
		add_new_bbcluster_flavor(
			FArray1D_float const & phi,
			FArray1D_float const & psi
		);

		void dump_barcode_file();

		void set_force_rotamer( const bool setting );
		bool get_force_rotamer() const;


		//////////////////////////////
		// support sspair constraints:
		bool
		sspair_constraints_exist() const;

		int
		num_bonus_sspair_constraints() const;

		float
		sspair_constraint_score(
			int const pos1,
			int const pos2,
			float const theta
		) const;
		//////////////////////////////
		// support disulf constraints:
		// (for rvernon's pose_disulf), not the traditional version

		bool
		disulf_constraints_exist() const;


		//////////////////////////////
		// support beta_pair constraints
		bool
		beta_pair_constraints_exist() const;

		int
		num_bonus_beta_pair_constraints() const;



	};

} // namespace barcode_classes

#endif
