// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19866 $
//  $Date: 2008-01-23 14:54:04 -0500 (Wed, 23 Jan 2008) $
//  $Author: possu $


// Rosetta Headers
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h"
#include "bond_angle.h"
#include "kin_atom_tree.h"
#include "kin_min.h"
#include "param_aa.h"
#include "param.h"
#include "pose.h"
#include "read_paths.h"

// Utility Headers
#include <utility/io/izstream.hh>


namespace bond_angle {

bool initialized(false);
utility::vector1<int> CBpos;
utility::vector1<utility::vector0<float> > Ktheta;
utility::vector1<utility::vector0<float> > Theta0;
utility::vector1<float> Ktheta_bb;
utility::vector1<float> Theta0_bb;
utility::vector1<float> Energy0_bb;
utility::vector1<utility::vector0<float> > thetaCB;
utility::vector1<utility::vector0<float> > phiCB;
utility::vector1<utility::vector0<float> > thetaHA;
utility::vector1<utility::vector0<float> > phiHA;
utility::vector1<utility::vector1<utility::vector0<float> > > dE_datom_tree;
utility::vector1<utility::vector1<bool> > dE_datom_tree_valid;

////////////////////////////////////////////////////////////////////////////////
/// @begin read_bond_angle_params_default
///
/// @brief
/// set energy parameters for bond angles around CA atom to default values
///
/// @detailed
/// Sets parameters from the CHARMM27 force field. Most residues have the
/// same parameters. Glycine and proline are different.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// The bond angles, in order, are: N-CA-C, N-CA-CB, N-CA-HA, C-CA-CB, C-CA-HA
/// CB-CA-HA. In glycine, the CB parameters refer to HA2.
///
/// *_bb Parameters were derived from fits of the total energy from structures
/// with CB/HA positions optimized for a fixed N-CA-C angle. Only energies less
/// than 5 kcal/mole were used for the fits.
///
/// The phi/theta parameters are derived from minimizing the polar coordinates
/// of CB and HA while varrying the N-CA-C angle. The minimization was done in
/// an R script which used the CHARMM bond angle parameters. The quadratic fits
/// were to those angles which produced energies of less then 20 kcal/mole, or
/// roughly +/- 30 degress from the optimal N-CA-C bond angle.
///
/// @refrences
/// http://www.pharmacy.umaryland.edu/faculty/amackere/force_fields.htm
///
/// @authors Colin A. Smith
///
/// @last_modified March 12, 2007
////////////////////////////////////////////////////////////////////////////////
void
read_bond_angle_params_default()
{
	using numeric::conversions::radians;

	int const numaa = 20;

	initialized = true;

	CBpos.resize(numaa, 5);
	Ktheta.resize(numaa);
	Theta0.resize(numaa);
	Ktheta_bb.resize(numaa);
	Theta0_bb.resize(numaa);
	Energy0_bb.resize(numaa);
	thetaCB.resize(numaa);
	phiCB.resize(numaa);
	thetaHA.resize(numaa);
	phiHA.resize(numaa);

	for (int aa = 1; aa <= numaa; aa++) {

		if (aa == param_aa::aa_gly) CBpos[aa] = 6; // 2HA is the pseudo CB for glycine

		Ktheta[aa].resize(6);
		Theta0[aa].resize(6);
		thetaCB[aa].resize(3);
		phiCB[aa].resize(3);
		thetaHA[aa].resize(3);
		phiHA[aa].resize(3);

		if (aa == param_aa::aa_gly) {
			Ktheta[aa][0] = 50; // N-CA-C
			Ktheta[aa][1] = 48; // N-CA-CB
			Ktheta[aa][2] = 48; // N-CA-HA
			Ktheta[aa][3] = 50; // C-CA-CB
			Ktheta[aa][4] = 50; // C-CA-HA
			Ktheta[aa][5] = 36; // CB-CA-HA
			Theta0[aa][0] = radians(107.0);
			Theta0[aa][1] = radians(108.0);
			Theta0[aa][2] = radians(108.0);
			Theta0[aa][3] = radians(109.5);
			Theta0[aa][4] = radians(109.5);
			Theta0[aa][5] = radians(115.0);

			Ktheta_bb[aa] = 57.34891;
			Theta0_bb[aa] = radians(106.7951);
			Energy0_bb[aa] = 0.001818123;

			thetaCB[aa][0] = 1.4435633; // thetaCB a (radians)
			thetaCB[aa][1] = 0.42161151; // thetaCB b (no units)
			thetaCB[aa][2] = -0.050956424; // thetaCB c (radians^-1)
			phiCB[aa][0] = 1.8551464; // phiCB a
			phiCB[aa][1] = 0.19712466; // phiCB b
			phiCB[aa][2] = -0.097477479; // phiCB c
			thetaHA[aa][0] = -1.4435633; // thetaHA a
			thetaHA[aa][1] = -0.42161151; // thetaHA b
			thetaHA[aa][2] = 0.050956426; // thetaHA c
			phiHA[aa][0] = 1.8551465; // phiHA a
			phiHA[aa][1] = 0.19712464; // phiHA b
			phiHA[aa][2] = -0.097477476; // phiHA c
		} else if (aa == param_aa::aa_pro) {
			Ktheta[aa][0] = 50;
			Ktheta[aa][1] = 70;
			Ktheta[aa][2] = 48;
			Ktheta[aa][3] = 52;
			Ktheta[aa][4] = 50;
			Ktheta[aa][5] = 35;
			Theta0[aa][0] = radians(108.2);
			Theta0[aa][1] = radians(110.8);
			Theta0[aa][2] = radians(112.0);
			Theta0[aa][3] = radians(112.3);
			Theta0[aa][4] = radians(112.0);
			Theta0[aa][5] = radians(118.0);

			Ktheta_bb[aa] = 59.50675;
			Theta0_bb[aa] = radians(105.5814);
			Energy0_bb[aa] = 0.699624184;

			thetaCB[aa][0] = 1.5283285;
			thetaCB[aa][1] = 0.32031172;
			thetaCB[aa][2] = -0.018854912;
			phiCB[aa][0] = 1.8868616;
			phiCB[aa][1] = 0.14786939;
			phiCB[aa][2] = -0.076313007;
			thetaHA[aa][0] = -1.3019436;
			thetaHA[aa][1] = -0.59808661;
			thetaHA[aa][2] = 0.10458026;
			phiHA[aa][0] = 1.9113772;
			phiHA[aa][1] = 0.18404484;
			phiHA[aa][2] = -0.10151666;
		} else {
			Ktheta[aa][0] = 50;
			Ktheta[aa][1] = 70;
			Ktheta[aa][2] = 48;
			Ktheta[aa][3] = 52;
			Ktheta[aa][4] = 50;
			Ktheta[aa][5] = 35;
			Theta0[aa][0] = radians(107.0);
			Theta0[aa][1] = radians(113.5);
			Theta0[aa][2] = radians(108.0);
			Theta0[aa][3] = radians(108.0);
			Theta0[aa][4] = radians(109.5);
			Theta0[aa][5] = radians(111.0);

			Ktheta_bb[aa] = 58.86439;
			Theta0_bb[aa] = radians(106.7849);
			Energy0_bb[aa] = 0.001745961;

			thetaCB[aa][0] = 1.2902338;
			thetaCB[aa][1] = 0.48783192;
			thetaCB[aa][2] = -0.035852417;
			phiCB[aa][0] = 1.9573143;
			phiCB[aa][1] = 0.16424161;
			phiCB[aa][2] = -0.081428465;
			thetaHA[aa][0] = -1.2682427;
			thetaHA[aa][1] = -0.63868211;
			thetaHA[aa][2] = 0.11661934;
			phiHA[aa][0] = 1.9125061;
			phiHA[aa][1] = 0.16459203;
			phiHA[aa][2] = -0.09642947;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_bond_angle_params
///
/// @brief
/// read energy parameters for bond angles around CA atom from a file
///
/// @detailed
///
/// @param[in] filename - file name of parameter file located in data directory
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 12, 2007
////////////////////////////////////////////////////////////////////////////////
void
read_bond_angle_params(
	std::string const & filename
)
{
	utility::io::izstream & param_stream( open_data_file( filename ) );
	std::string aa_types;
	int const numaa = 20;

	CBpos.resize(numaa, 5);
	Ktheta.resize(numaa);
	Theta0.resize(numaa);
	Ktheta_bb.resize(numaa);
	Theta0_bb.resize(numaa);
	Energy0_bb.resize(numaa);
	thetaCB.resize(numaa);
	phiCB.resize(numaa);
	thetaHA.resize(numaa);
	phiHA.resize(numaa);

	utility::vector0<float> Ktheta_local(6);
	utility::vector0<float> Theta0_local(6);
	float Ktheta_bb_local;
	float Theta0_bb_local;
	float Energy0_bb_local;
	utility::vector0<float> thetaCB_local(3);
	utility::vector0<float> phiCB_local(3);
	utility::vector0<float> thetaHA_local(3);
	utility::vector0<float> phiHA_local(3);

	while (param_stream >> aa_types) {

		if (param_stream >> Ktheta_local[0] >> Theta0_local[0]
		                 >> Ktheta_local[1] >> Theta0_local[1]
		                 >> Ktheta_local[2] >> Theta0_local[2]
		                 >> Ktheta_local[3] >> Theta0_local[3]
		                 >> Ktheta_local[4] >> Theta0_local[4]
		                 >> Ktheta_local[5] >> Theta0_local[5]
		                 >> Ktheta_bb_local >> Theta0_bb_local >> Energy0_bb_local
		                 >> thetaCB_local[0] >> thetaCB_local[1] >> thetaCB_local[2]
		                 >> phiCB_local[0] >> phiCB_local[1] >> phiCB_local[2]
		                 >> thetaHA_local[0] >> thetaHA_local[1] >> thetaHA_local[2]
		                 >> phiHA_local[0] >> phiHA_local[1] >> phiHA_local[2]) {

			for (size_t i = 0; i < Theta0_local.size(); i++) {
				numeric::conversions::to_radians(Theta0_local[i]);
			}
			numeric::conversions::to_radians(Theta0_bb_local);

			for (size_t i = 0; i+3 <= aa_types.size(); i += 4) {

				int aa;
				num_from_res3(aa_types.substr(i, 3), aa);
				if (aa == 0) {
					std::cout << "Error: No amino acids with name " << aa_types.substr(i, 3) << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				if (aa == param_aa::aa_gly) CBpos[aa] = 6; // 2HA is the pseudo CB for glycine

				Ktheta[aa] = Ktheta_local;
				Theta0[aa] = Theta0_local;
				Ktheta_bb[aa] = Ktheta_bb_local;
				Theta0_bb[aa] = Theta0_bb_local;
				Energy0_bb[aa] = Energy0_bb_local;
				thetaCB[aa] = thetaCB_local;
				phiCB[aa] = phiCB_local;
				thetaHA[aa] = thetaHA_local;
				phiHA[aa] = phiHA_local;
			}
		} else {
			std::cout << "Error reading bond angle parameters for amino acid type(s):" << std::endl
			          << " " << aa_types << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	param_stream.close();
}

} // namespace bond_angle

////////////////////////////////////////////////////////////////////////////////
/// @begin get_bond_angleE
///
/// @brief
/// get energy for CA bond angles
///
/// @detailed
/// Returns the energy for CA bond angles as defined by the CHARMM27 force
/// field.
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] coord - amino acid atom coordinates (2D array)
///
/// @global_read
///
/// @global_write
///
/// @return
/// the bond angle energy in kcal/mole
///
/// @remarks
/// The energy formula is: V(Theta) = Ktheta(Theta - Theta0)^2
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 11, 2006
////////////////////////////////////////////////////////////////////////////////
float
get_bond_angleE(
	int const aa, // amino acid type
	int const aav, // amino acid variants
	FArray2Da_float coord // coordinates
)
{
	using numeric::conversions::radians;

	float angleE = 0.0, n_dist, c_dist, cb_dist, ha_dist;

	bond_angle::initialize_bond_angle_params();

	if ( !param_aa::is_protein(aa) && !param_aa::is_nonnatural(aa) ) return angleE;
	if (aav) {} // avoid unused parameter warning

	coord.dimension( 3, param::MAX_ATOM() );

	int const CBpos = bond_angle::CBpos[aa];
	int const HApos = aaproperties_pack::HApos(aa,aav);

	distance2_bk(coord(1,1), coord(1,2), n_dist);
	distance2_bk(coord(1,3), coord(1,2), c_dist);
	distance2_bk(coord(1,CBpos), coord(1,2), cb_dist);
	distance2_bk(coord(1,HApos), coord(1,2), ha_dist);

	// hack to allow proline motion: CB/HA angles are ignored
	if (aa == param_aa::aa_pro) {
		if (n_dist > 0 && c_dist > 0) {
			float const angle = vec_angle(coord(1,1), coord(1,2), coord(1,3)); // N-CA-C
			float const anglediff = radians(angle) - bond_angle::Theta0_bb[aa];
			angleE = bond_angle::Ktheta_bb[aa]*anglediff*anglediff + bond_angle::Energy0_bb[aa];
		}
		
		return angleE;
	}

	if (n_dist > 0 && c_dist > 0) {
		float const angle = vec_angle(coord(1,1), coord(1,2), coord(1,3)); // N-CA-C
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][0];
		angleE += bond_angle::Ktheta[aa][0]*anglediff*anglediff;
	}
	if (n_dist > 0 && c_dist > 0) {
		float const angle = vec_angle(coord(1,1), coord(1,2), coord(1,CBpos)); // N-CA-CB
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][1];
		angleE += bond_angle::Ktheta[aa][1]*anglediff*anglediff;
	}
	if (n_dist > 0 && ha_dist > 0) {
		float const angle = vec_angle(coord(1,1), coord(1,2), coord(1,HApos)); // N-CA-HA
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][2];
		angleE += bond_angle::Ktheta[aa][2]*anglediff*anglediff;
	}
	if (c_dist > 0 && cb_dist > 0) {
		float const angle = vec_angle(coord(1,3), coord(1,2), coord(1,CBpos)); // C-CA-CB
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][3];
		angleE += bond_angle::Ktheta[aa][3]*anglediff*anglediff;
	}
	if (c_dist > 0 && ha_dist > 0) {
		float const angle = vec_angle(coord(1,3), coord(1,2), coord(1,HApos)); // C-CA-HA
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][4];
		angleE += bond_angle::Ktheta[aa][4]*anglediff*anglediff;
	}
	if (cb_dist > 0 && ha_dist > 0) {
		float const angle = vec_angle(coord(1,CBpos), coord(1,2), coord(1,HApos)); // CB-CA-HA
		float const anglediff = radians(angle) - bond_angle::Theta0[aa][5];
		angleE += bond_angle::Ktheta[aa][5]*anglediff*anglediff;
	}

	return angleE;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_bond_angleE_Ep
///
/// @brief
/// get bond angle energy for a residue in an Eposition array
///
/// @detailed
/// Only the N-CA-C bond angle is used to compute the energy
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] coord - amino acid atom coordinates (2D array)
///
/// @global_read
///
/// @global_write
///
/// @return
/// bond angle energy in kcal/mole
///
/// @remarks
/// The energy formula is: V(Theta) = Ktheta(Theta - Theta0)^2 + Energy0
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 25, 2006
////////////////////////////////////////////////////////////////////////////////
float
get_bond_angleE_Ep(
	int const aa, // amino acid type
	int const aav, // amino acid variants
	FArray2Da_float coord // coordinates
)
{
	using numeric::conversions::radians;

	bond_angle::initialize_bond_angle_params();

	if ( !param_aa::is_protein(aa) && !param_aa::is_nonnatural(aa) ) return 0;
	if (aav) {} // avoid unused parameter warning

	coord.dimension( 3, 5 );

	float const angle = vec_angle(coord(1,1), coord(1,2), coord(1,4)); // N-CA-C
	float const anglediff = radians(angle) - bond_angle::Theta0_bb[aa];

	return bond_angle::Ktheta_bb[aa]*anglediff*anglediff + bond_angle::Energy0_bb[aa];
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_cbha_angles
///
/// @brief
/// calculate CB and HA polar coordinates that minimize bond energy
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] phiC - N-CA-C bond angle
/// @param[out] thetaCB - CB theta polar coordinate (radians)
/// @param[out] phiCB - CB phi polar coordinate (radians)
/// @param[out] thetaHA - HA theta polar coordinate (radians)
/// @param[out] phiHA - HA phi polar coordinate (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void get_cbha_angles(
	int const aa,
	int const aav,
	float const phiC,
	float & thetaCB,
	float & phiCB,
	float & thetaHA,
	float & phiHA
)
{
	if ( !param_aa::is_protein(aa) ) {
		std::cout << "WARNING: Attempt to get CB & HA angles for non-amino acid"
		          << std::endl;
		return;
	}
	if (aav) {} // avoid unused parameter warning

	bond_angle::initialize_bond_angle_params();

	thetaCB = bond_angle::thetaCB[aa][0] + bond_angle::thetaCB[aa][1]*phiC +
	          bond_angle::thetaCB[aa][2]*phiC*phiC;
	phiCB = bond_angle::phiCB[aa][0] + bond_angle::phiCB[aa][1]*phiC +
	        bond_angle::phiCB[aa][2]*phiC*phiC;
	thetaHA = bond_angle::thetaHA[aa][0] + bond_angle::thetaHA[aa][1]*phiC +
	          bond_angle::thetaHA[aa][2]*phiC*phiC;
	phiHA = bond_angle::phiHA[aa][0] + bond_angle::phiHA[aa][1]*phiC +
	        bond_angle::phiHA[aa][2]*phiC*phiC;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_cbha_angles
///
/// @brief
/// calculate CB and HA polar coordinates that minimize bond energy
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] res_coor - residue atom coordinates
/// @param[out] thetaCB - CB theta polar coordinate (radians)
/// @param[out] phiCB - CB phi polar coordinate (radians)
/// @param[out] thetaHA - HA theta polar coordinate (radians)
/// @param[out] phiHA - HA phi polar coordinate (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void get_cbha_angles(
	int const aa,
	int const aav,
	FArray2Da_float res_coor,
	float & thetaCB,
	float & phiCB,
	float & thetaHA,
	float & phiHA
)
{
	using numeric::conversions::radians;

	float phiC;

	res_coor.dimension(3, param::MAX_ATOM());

	// N-CA-C bond angle
	phiC = radians(vec_angle(res_coor(1,1), res_coor(1,2), res_coor(1,3)));
	
	get_cbha_angles(aa, aav, phiC, thetaCB, phiCB, thetaHA, phiHA);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin print_idealzed_ca_angles
///
/// @brief
/// print out values of Rosetta idealized CA bond angles
///
/// @detailed
/// this function generates a Pose object with all amino acid types and measures
/// bond angles around each
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 12, 2007
////////////////////////////////////////////////////////////////////////////////
void print_idealzed_ca_angles(
	std::ostream & out
)
{
	pose_ns::Pose pose;

	pose.simple_fold_tree(20);

	for (int i = 1; i <= 20; i++) {
		pose.set_phi(i, 180);
		pose.set_psi(i, 180);
		pose.set_omega(i, 180);
		pose.set_res(i, i);
		pose.set_res_variant(i, 1);
	}
	pose.set_fullatom_flag(true, true);

	FArray3D_float const & coord = pose.full_coord();
	int const aav = 1;

	bond_angle::initialize_bond_angle_params();

	// Output Rosetta Template CA Bond Angles
	out << "Res\tN-CA-C   \tN-CA-CB   \tN-CA-HA   \tC-CA-CB   \tC-CA-HA   \tCB-CA-HA" << std::endl;
	for (int aa = 1; aa <= 20; aa++) {
		using aaproperties_pack::HApos;
		int const CBpos = bond_angle::CBpos[aa];
		out << param_aa::aa_name3(aa) << "\t" << std::setprecision(9)
		    << vec_angle(coord(1,1,aa), coord(1,2,aa), coord(1,3,aa)) << "\t" // N-CA-C
		    << vec_angle(coord(1,1,aa), coord(1,2,aa), coord(1,CBpos,aa)) << "\t" // N-CA-CB
		    << vec_angle(coord(1,1,aa), coord(1,2,aa), coord(1,HApos(aa,aav),aa)) << "\t" // N-CA-HA
		    << vec_angle(coord(1,3,aa), coord(1,2,aa), coord(1,CBpos,aa)) << "\t" // C-CA-CB
		    << vec_angle(coord(1,3,aa), coord(1,2,aa), coord(1,HApos(aa,aav),aa)) << "\t" // C-CA-HA
		    << vec_angle(coord(1,CBpos,aa), coord(1,2,aa), coord(1,HApos(aa,aav),aa)) // CB-CA-HA
		    << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin precalc_bond_angle_deriv
///
/// @brief
/// pre-calculate bond angle energy derivatives for a pose/atom tree object
///
/// @detailed
///
/// @param[in] pose - pose object using atom tree
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 19, 2007
////////////////////////////////////////////////////////////////////////////////
void
precalc_bond_angle_deriv(
	pose_ns::Pose & pose
)
{
	using bond_angle::dE_datom_tree;
	using bond_angle::dE_datom_tree_valid;

	kin::Minimizer_map const & min_map(retrieve_min_map());
	
	bond_angle::initialize_bond_angle_params();
	
	// Invalidate previous derivatives
	for (size_t i = 1; i <= dE_datom_tree_valid.size(); i++) {
		dE_datom_tree_valid[i].assign(dE_datom_tree_valid[i].size(), false);
	}
	
	dE_datom_tree.resize(pose.total_residue());
	dE_datom_tree_valid.resize(pose.total_residue());
	
	// loop over the torsions in the map
	int imap(1);
	for (kin::Minimizer_map::Torsion_const_iterator tor_iter=min_map.torsions_begin(),
	     tor_end = min_map.torsions_end(); tor_iter != tor_end; ++tor_iter, ++imap ) {
	
		kin::Torsion_node const & tor( **tor_iter );
		kin::Kin_torsion_type const type( tor.type() );
		
		// Check to see if this is a bond angle or torsion offset
		if (type == kin::THETA ||
		    (type == kin::PHI && pose.get_atom_tree_atom(tor.atom_id())->previous_sibling())) {
		
			kin::Atom const & center_atom(*pose.get_atom_tree_atom(tor.atom_id())->parent);
			int const center_rsd = center_atom.atom_id.rsd();
			int const center_atomno = center_atom.atom_id.atomno();
			
			// Only do each atom center once
			if (static_cast<signed>(dE_datom_tree_valid[center_rsd].size()) < center_atomno ||
			    !dE_datom_tree_valid[center_rsd][center_atomno]) {
				
				// Resize the per-residue atom vectors
				if (static_cast<signed>(dE_datom_tree[center_rsd].size()) < center_atomno) {
					dE_datom_tree[center_rsd].resize(center_atomno);
					dE_datom_tree_valid[center_rsd].resize(center_atomno, false);
				}
				
				bond_angle_deriv_quad(pose, center_atom, dE_datom_tree[center_rsd][center_atomno]);
				
				// Record that we've already calculated the bond angle derivatives
				dE_datom_tree_valid[center_rsd][center_atomno] = true;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin bond_angle_deriv_quad
///
/// @brief
/// calculate bond angle energy derivatives about an atom with 4 neighbors
///
/// @detailed
///
/// @param[in] pose - pose object
/// @param[in] center_atom - center_atom in atom tree
/// @param[out] deriv - dE_dthetaA, dE_dphiB, dE_dthetaB, dE_dphiC, dE_dthetaC
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// The derivatives were auto-generated with this R command:
///
/// deriv(expression(Ktheta_0_ * (thetaA - Theta0_0_)^2 +
///                  Ktheta_1_ * (thetaB - Theta0_1_)^2 +
///                  Ktheta_2_ * (thetaC - Theta0_2_)^2 +
///                  Ktheta_3_ * (acos(sin(thetaA) * sin(thetaB) * cos(-phiB) +
///                                    cos(thetaA) * cos(thetaB)) - Theta0_3_)^2 +
///                  Ktheta_4_ * (acos(sin(thetaA) * sin(thetaC) * cos(phiB + phiC) +
///                                    cos(thetaA) * cos(thetaC)) - Theta0_4_)^2 +
///                  Ktheta_5_ * (acos(sin(thetaB) * sin(thetaC) * cos(-phiC) +
///                                    cos(thetaB) * cos(thetaC)) - Theta0_5_)^2),
///       c("thetaA", "phiB", "thetaB", "phiC", "thetaC"))
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 19, 2007
////////////////////////////////////////////////////////////////////////////////
void
bond_angle_deriv_quad(
	pose_ns::Pose & pose,
	kin::Atom const & center_atom,
	utility::vector0<float> & deriv
)
{
	using std::sin;
	using std::cos;
	using std::acos;
	
	int const atomno(center_atom.atom_id.atomno());
	
	// This only handles CA atoms right now
	if (atomno != 2) {
		return;
	}
	
	int const rsd(center_atom.atom_id.rsd());
	int const aatype(pose.res(rsd));
	int const aavariant(pose.res_variant(rsd));
	
	assert(aaproperties_pack::nbonded_neighbors(atomno, aatype, aavariant) == 4);
	
	int const atomnoA(aaproperties_pack::bonded_neighbor(2, atomno, aatype, aavariant));
	int const atomnoB(aaproperties_pack::bonded_neighbor(3, atomno, aatype, aavariant));
	int const atomnoC(aaproperties_pack::bonded_neighbor(4, atomno, aatype, aavariant));
	
	// Extra checks to make sure generalization is working
	int const CBpos = bond_angle::CBpos[aatype];
	int const HApos = aaproperties_pack::HApos(aatype, aavariant);
	assert(atomnoA == 3 && atomnoB == CBpos && atomnoC == HApos);
	
	float thetaA = -1, phiB = -1, thetaB = -1, phiC = -1, thetaC = -1;
	
	// A is the atom in the trunk of the tree
	// B takes its phi offset from A
	// C takes its phi offset from B
	// theta degrees of freedom are redefined to their supplement for computational simplicity
	// DANGER! This code assumes that no jumps separate the relevant atoms in the tree
	for (int i = 0; i < center_atom.nchildren(); i++) {
		kin::Atom const & child(*center_atom.child(i));
		if (child.atom_id.atomno() == atomnoA) {
			thetaA = numeric::constants::f::pi - child.get_torsion(kin::THETA);
		} else if (child.atom_id.atomno() == atomnoB) {
			phiB = child.get_torsion(kin::PHI);
			thetaB = numeric::constants::f::pi - child.get_torsion(kin::THETA);
		} else if (child.atom_id.atomno() == atomnoC) {
			phiC = child.get_torsion(kin::PHI);
			thetaC = numeric::constants::f::pi - child.get_torsion(kin::THETA);
		}
	}
	
	// Check to see if we missed any degrees of freedom
	assert(thetaA != -1 && phiB != -1 && thetaB != -1 && phiC != -1 && thetaC != -1);
	
	utility::vector0<float> const & Ktheta(bond_angle::Ktheta[aatype]);
	utility::vector0<float> const & Theta0(bond_angle::Theta0[aatype]);
	
	float const expr1 = thetaA - Theta0[0];
	float const expr4 = thetaB - Theta0[1];
	float const expr8 = thetaC - Theta0[2];
	float const expr12 = sin(thetaA);
	float const expr13 = sin(thetaB);
	float const expr14 = expr12 * expr13;
	float const expr15 = -phiB;
	float const expr16 = cos(expr15);
	float const expr18 = cos(thetaA);
	float const expr19 = cos(thetaB);
	float const expr21 = expr14 * expr16 + expr18 * expr19;
	float const expr23 = acos(expr21) - Theta0[3];
	float const expr27 = sin(thetaC);
	float const expr28 = expr12 * expr27;
	float const expr29 = phiB + phiC;
	float const expr30 = cos(expr29);
	float const expr32 = cos(thetaC);
	float const expr34 = expr28 * expr30 + expr18 * expr32;
	float const expr36 = acos(expr34) - Theta0[4];
	float const expr40 = expr13 * expr27;
	float const expr41 = -phiC;
	float const expr42 = cos(expr41);
	float const expr45 = expr40 * expr42 + expr19 * expr32;
	float const expr47 = acos(expr45) - Theta0[5];
	float const expr53 = expr18 * expr13;
	float const expr55 = expr12 * expr19;
	float const expr59 = sqrt(1 - expr21*expr21);
	float const expr65 = expr18 * expr27;
	float const expr67 = expr12 * expr32;
	float const expr71 = sqrt(1 - expr34*expr34);
	float const expr82 = Ktheta[4] * (2 * (expr28 * sin(expr29)/expr71 * expr36));
	float const expr99 = expr19 * expr27;
	float const expr101 = expr13 * expr32;
	float const expr105 = sqrt(1 - expr45*expr45);

	deriv.resize(5);

	// deriv[0], deriv[2], and deriv[4] have a sign swap to handle supplement redefinition
	// dE_dthetaA
	deriv[0] = -(Ktheta[0] * (2 * expr1) - Ktheta[3] *
	    (2 * ((expr53 * expr16 - expr55)/expr59 * expr23)) -
	    Ktheta[4] * (2 * ((expr65 * expr30 - expr67)/expr71 *
	        expr36)));
	// dE_dphiB
	deriv[1] = expr82 - Ktheta[3] * (2 * (expr14 *
	    sin(expr15)/expr59 * expr23));
	// dE_dthetaB
	deriv[2] = -(Ktheta[1] * (2 * expr4) - Ktheta[3] *
	    (2 * ((expr55 * expr16 - expr53)/expr59 * expr23)) -
	    Ktheta[5] * (2 * ((expr99 * expr42 - expr101)/expr105 *
	        expr47)));
	// dE_dphiC
	deriv[3] = expr82 - Ktheta[5] * (2 * (expr40 *
	    sin(expr41)/expr105 * expr47));
	// dE_dthetaC
	deriv[4] = -(Ktheta[2] * (2 * expr8) - Ktheta[4] *
	    (2 * ((expr67 * expr30 - expr65)/expr71 * expr36)) -
	    Ktheta[5] * (2 * ((expr101 * expr42 - expr99)/expr105 *
	        expr47)));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin bond_angle_deriv
///
/// @brief
/// fetch pre calculated bond angle derivatives
///
/// @detailed
///
/// @param[in] pose - pose object
/// @param[in] tor - degree of freedom
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 19, 2007
////////////////////////////////////////////////////////////////////////////////
float
bond_angle_deriv(
	pose_ns::Pose & pose,
	kin::Torsion_node const & tor
)
{
	using bond_angle::dE_datom_tree;
	using bond_angle::dE_datom_tree_valid;

	kin::Atom const & atom(*pose.get_atom_tree_atom(tor.atom_id()));
	kin::Atom const & center_atom(*atom.parent);
	
	int const atomno(center_atom.atom_id.atomno());
	
	// This only handles CA atoms right now
	if (atomno != 2) {
		return 0;
	}
	
	int const rsd(center_atom.atom_id.rsd());
	int const aatype(pose.res(rsd));
	int const aavariant(pose.res_variant(rsd));
	
	assert(aaproperties_pack::nbonded_neighbors(atomno, aatype, aavariant) == 4);
	
	int const atomnoA(aaproperties_pack::bonded_neighbor(2, atomno, aatype, aavariant));
	int const atomnoB(aaproperties_pack::bonded_neighbor(3, atomno, aatype, aavariant));
	int const atomnoC(aaproperties_pack::bonded_neighbor(4, atomno, aatype, aavariant));
	
	// Extra checks to make sure generalization is working
	int const CBpos = bond_angle::CBpos[aatype];
	int const HApos = aaproperties_pack::HApos(aatype, aavariant);
	assert(atomnoA == 3 && atomnoB == CBpos && atomnoC == HApos);
	
	// This assertion appears to die without a valid reason
	//assert(dE_datom_tree_valid[rsd][atomno] == true && dE_datom_tree[rsd][atomno].size() == 5);
	
	if (atom.atom_id.atomno() == atomnoA) {
		if (tor.type() == kin::THETA) return dE_datom_tree[rsd][atomno][0];
	} else if (atom.atom_id.atomno() == atomnoB) {
		if (tor.type() == kin::PHI) return dE_datom_tree[rsd][atomno][1];
		if (tor.type() == kin::THETA) return dE_datom_tree[rsd][atomno][2];
	} else if (atom.atom_id.atomno() == atomnoC) {
		if (tor.type() == kin::PHI) return dE_datom_tree[rsd][atomno][3];
		if (tor.type() == kin::THETA) return dE_datom_tree[rsd][atomno][4];
	}
	
	return 0;
}
