// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   ccd_functions.cc
/// @brief  Various cyclic coordinate descent functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <ccd_functions.hh>

// Rosetta headers
#include <after_opts.h>
#include <bonds_class.h>
#include <jumping_loops.h>
#include <jumping_refold.h>
#include <misc.h>
#include <ramachandran.h>
#include <refold_ns.h>
#include <pose.h>
#include <random_numbers.h>
#include <refold.h>
#include <util_vector.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3DB.hh>


namespace epigraft {
namespace design {


/// @brief original rosetta ccd moves, moves per-residue (always phi->psi), ideal bond geometry only
void
original_ccd_moves(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
)
{

	// debug:
	pose.copy_to_misc();
	assert( misc_in_sync( pose ) );

	/////////
	// params
	/////////
	bool const local_verbose = false;
	int const n2c = { 1 };
	int const c2n = { -1 };
	double const max_angle_delta_helix = { 1.0 }; // per-cycle max move params
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	float const ccd_temperature = { 0.25 }; // for boltzmann rama-checking

	// which residues are insertable?
	// setup arrays for choosing residues at random weighted more toward loop
	// residues
	int total_insert;
	const FArray1D_int & insert_map ( pose.get_insert_map( total_insert ) );
	float const H_weight ( 0.5 );
	float const E_weight ( 1.0 );
	float const L_weight ( 8.5 );
	float total_weight(0.0);
	FArray1D_float weight_map ( total_insert );
	int loop_total_insert(0), first_insert(0);
	for ( int i=1,pos; i<= total_insert; ++i ) {
		pos = insert_map(i);
		if ( pos >= loop_begin && pos <= loop_end ) {
			++loop_total_insert;
			if ( first_insert == 0 ) {
				first_insert = i;
			}

			char ss ( pose.secstruct( pos ) );
			if ( ss == 'H' ) {
				total_weight += H_weight;
			} else if ( ss == 'E' ) {
				total_weight += E_weight;
			} else {
				assert ( ss == 'L' );
				total_weight += L_weight;
			}
			weight_map(loop_total_insert) = total_weight;
		}
	}
	if ( loop_total_insert <= 0 ) {
		std::cout << "no insertable residues in the loop!" << std::endl;
		return;
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		const int pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	////////////////////////////
	// now start cycling through
	////////////////////////////

	int ntries(0);
	int nmoves(0);

	while ( nmoves < total_moves ) {
		++ntries;
		if ( ntries > 5*total_moves ) {
			break;
		}

		// choose a residue at random:
		const float weight ( ran3() * total_weight );
		int ipos;
		for ( ipos=1; ipos<= loop_total_insert; ++ipos ) {
				if ( weight <= weight_map(ipos) ) break;
			}
		if ( ipos > loop_total_insert ) {
			std::cout << "ccd_moves: bad weight? " << weight << ' ' <<
				total_weight << ' ' << weight_map( loop_total_insert ) << std::endl;
			ipos = loop_total_insert;
		}

		const int pos = insert_map( first_insert + ipos - 1 );
		const int direction ( pos <= cutpoint ? n2c : c2n );

		// get fixed and moving positions for this direction:
		if ( direction == n2c ) {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
		}
		get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M);

		// as we change torsion angles through the loop the moving
		// atoms in M will be transformed along with the downstream segment
		// of the loop, by the routine refold_loop_torsions(...)

		float max_angle_delta;
		if ( misc::secstruct(pos) == 'H' ) {
			max_angle_delta = max_angle_delta_helix;
		} else if ( misc::secstruct(pos) == 'E' ) {
			max_angle_delta = max_angle_delta_strand;
		} else {
			max_angle_delta = max_angle_delta_loop;
		}

		// save values for rama score eval, and in case we rama-reject
		const float starting_phi ( misc::phi(pos) );
		const float starting_psi ( misc::psi(pos) );

		// save in case we rama-reject!
		// re-use these index ranges later
		const int Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
		const int Epos_l_end   ( Epos_l_begin +
														 3 * param::MAX_POS * (loop_end-loop_begin+1) );
		assert ( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
		for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
			save_Eposition[l] = misc::Eposition[l];
		}

		for ( int torsion = 1; torsion <= 2; ++torsion ) { // 1 is phi, 2 is psi

			double alpha,dev;
			calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
			double const alpha_orig( alpha );

			// impose per-move deltas
			if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
			if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

			// just rebuilds N,CA,C in Eposition
			refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
													 misc::Eposition, M );

			// update torsions in misc (NOT in pose yet, only when we accept!)
			if ( torsion == 1 ) {
				misc::phi(pos) += alpha;
				misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
			} else {
				misc::psi(pos) += alpha;
				misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
			}

			if ( local_verbose ) {
				double tmp_dev = 0.0;
				for ( int i = 1; i <= 3; ++i ) {
					for ( int j = 1; j <= 3; ++j ) {
						tmp_dev += square( M(j,i) - F(j,i) );
					}
				}
				std::cout << "pos: " << I( 3, pos ) <<
					" alpha: " << fmt::F( 7, 3, alpha ) <<
					" dev1: " << fmt::F( 13, 9, dev ) <<
					" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

				if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
					std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
						tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
				}



			}
		}               // torsion = 1,2


		if ( true ) { // always rama-checking
			float tmp1,tmp2;
			//////////////////////////////////////
			// evaluate the rama score difference:
			float old_rama_score,new_rama_score;
			eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
															 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

			eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
															 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

			if ( new_rama_score > old_rama_score ) {
				const float boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
				const float probability ( std::exp(max(-40.0f,boltz_factor) ) );
				if ( ran3() >= probability ) {
					// undo the change:
					misc::phi(pos) = starting_phi;
					misc::psi(pos) = starting_psi;
					// copy saved Eposition
					for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
						misc::Eposition[l] = save_Eposition[l];
					}
					continue; // go to next position
				} // rama reject
			} // score got worse
		} // if ( rama_check )

		////////////////////////////////////////////////////
		// if we get here we have accepted the phi/psi move:
		////////////////////////////////////////////////////
		pose.set_phi(pos,misc::phi(pos));
		pose.set_psi(pos,misc::psi(pos));
		++nmoves;
	}

}


/// @brief original rosetta ccd moves where rama score is not allowed to increase
/// @brief moves per-residue (always phi->psi), ideal bond geometry only
void
rama_limited_original_ccd_moves(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
)
{

	// debug:
	pose.copy_to_misc();
	assert( misc_in_sync( pose ) );

	/////////
	// params
	/////////
	bool const local_verbose = false;
	int const n2c = { 1 };
	int const c2n = { -1 };
	double const max_angle_delta_helix = { 1.0 }; // per-cycle max move params
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };

	// which residues are insertable?
	// setup arrays for choosing residues at random weighted more toward loop
	// residues
	int total_insert;
	const FArray1D_int & insert_map ( pose.get_insert_map( total_insert ) );
	float const H_weight ( 0.5 );
	float const E_weight ( 1.0 );
	float const L_weight ( 8.5 );
	float total_weight(0.0);
	FArray1D_float weight_map ( total_insert );
	int loop_total_insert(0), first_insert(0);
	for ( int i=1,pos; i<= total_insert; ++i ) {
		pos = insert_map(i);
		if ( pos >= loop_begin && pos <= loop_end ) {
			++loop_total_insert;
			if ( first_insert == 0 ) {
				first_insert = i;
			}

			char ss ( pose.secstruct( pos ) );
			if ( ss == 'H' ) {
				total_weight += H_weight;
			} else if ( ss == 'E' ) {
				total_weight += E_weight;
			} else {
				assert ( ss == 'L' );
				total_weight += L_weight;
			}
			weight_map(loop_total_insert) = total_weight;
		}
	}
	if ( loop_total_insert <= 0 ) {
		std::cout << "no insertable residues in the loop!" << std::endl;
		return;
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		const int pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	////////////////////////////
	// now start cycling through
	////////////////////////////

	int ntries(0);
	int nmoves(0);

	while ( nmoves < total_moves ) {
		++ntries;
		if ( ntries > 5*total_moves ) {
			break;
		}

		// choose a residue at random:
		const float weight ( ran3() * total_weight );
		int ipos;
		for ( ipos=1; ipos<= loop_total_insert; ++ipos ) {
				if ( weight <= weight_map(ipos) ) break;
			}
		if ( ipos > loop_total_insert ) {
			std::cout << "ccd_moves: bad weight? " << weight << ' ' <<
				total_weight << ' ' << weight_map( loop_total_insert ) << std::endl;
			ipos = loop_total_insert;
		}

		const int pos = insert_map( first_insert + ipos - 1 );
		const int direction ( pos <= cutpoint ? n2c : c2n );

		// get fixed and moving positions for this direction:
		if ( direction == n2c ) {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
		}
		get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M);

		// as we change torsion angles through the loop the moving
		// atoms in M will be transformed along with the downstream segment
		// of the loop, by the routine refold_loop_torsions(...)

		float max_angle_delta;
		if ( misc::secstruct(pos) == 'H' ) {
			max_angle_delta = max_angle_delta_helix;
		} else if ( misc::secstruct(pos) == 'E' ) {
			max_angle_delta = max_angle_delta_strand;
		} else {
			max_angle_delta = max_angle_delta_loop;
		}

		// save values for rama score eval, and in case we rama-reject
		const float starting_phi ( misc::phi(pos) );
		const float starting_psi ( misc::psi(pos) );

		// save in case we rama-reject!
		// re-use these index ranges later
		const int Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
		const int Epos_l_end   ( Epos_l_begin +
														 3 * param::MAX_POS * (loop_end-loop_begin+1) );
		assert ( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
		for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
			save_Eposition[l] = misc::Eposition[l];
		}

		for ( int torsion = 1; torsion <= 2; ++torsion ) { // 1 is phi, 2 is psi

			double alpha,dev;
			calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
			double const alpha_orig( alpha );

			// impose per-move deltas
			if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
			if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

			// just rebuilds N,CA,C in Eposition
			refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
													 misc::Eposition, M );

			// update torsions in misc (NOT in pose yet, only when we accept!)
			if ( torsion == 1 ) {
				misc::phi(pos) += alpha;
				misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
			} else {
				misc::psi(pos) += alpha;
				misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
			}

			if ( local_verbose ) {
				double tmp_dev = 0.0;
				for ( int i = 1; i <= 3; ++i ) {
					for ( int j = 1; j <= 3; ++j ) {
						tmp_dev += square( M(j,i) - F(j,i) );
					}
				}
				std::cout << "pos: " << I( 3, pos ) <<
					" alpha: " << fmt::F( 7, 3, alpha ) <<
					" dev1: " << fmt::F( 13, 9, dev ) <<
					" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

				if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
					std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
						tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
				}



			}
		}               // torsion = 1,2


		if ( true ) { // always rama-checking
			float tmp1,tmp2;
			//////////////////////////////////////
			// evaluate the rama score difference:
			float old_rama_score,new_rama_score;
			eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
															 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

			eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
															 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

			if ( new_rama_score > old_rama_score ) {
				for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
					misc::Eposition[l] = save_Eposition[l];
				}
				continue;
			} // score got worse
		} // if ( rama_check )

		////////////////////////////////////////////////////
		// if we get here we have accepted the phi/psi move:
		////////////////////////////////////////////////////
		pose.set_phi(pos,misc::phi(pos));
		pose.set_psi(pos,misc::psi(pos));
		++nmoves;
	}

}


/// @brief ccd moves that take into account the nonideal bonds of the cutpoint
/// @brief when attempting to close, moves per-residue (directionality dependent)
void
ccd_moves_obeying_nonideality(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
)
{

	// debug:
	pose.copy_to_misc();
	assert( misc_in_sync( pose ) );

	/////////
	// params
	/////////
	bool const local_verbose = false;
	int const n2c = { 1 };
	int const c2n = { -1 };
	double const max_angle_delta_helix = { 1.0 }; // per-cycle max move params
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	float const ccd_temperature = { 0.25 }; // for boltzmann rama-checking

	// which residues are insertable?
	// setup arrays for choosing residues at random weighted more toward loop
	// residues
	int total_insert;
	const FArray1D_int & insert_map ( pose.get_insert_map( total_insert ) );
	float const H_weight ( 0.5 );
	float const E_weight ( 1.0 );
	float const L_weight ( 8.5 );
	float total_weight(0.0);
	FArray1D_float weight_map ( total_insert );
	int loop_total_insert(0), first_insert(0);
	for ( int i=1,pos; i<= total_insert; ++i ) {
		pos = insert_map(i);
		if ( pos >= loop_begin && pos <= loop_end ) {
			++loop_total_insert;
			if ( first_insert == 0 ) {
				first_insert = i;
			}

			char ss ( pose.secstruct( pos ) );
			if ( ss == 'H' ) {
				total_weight += H_weight;
			} else if ( ss == 'E' ) {
				total_weight += E_weight;
			} else {
				assert ( ss == 'L' );
				total_weight += L_weight;
			}
			weight_map(loop_total_insert) = total_weight;
		}
	}
	if ( loop_total_insert <= 0 ) {
		std::cout << "no insertable residues in the loop!" << std::endl;
		return;
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		const int pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	// torsion order
	int torsion_order[ 2 ]; // value of 1 in array is phi, 2 is psi

	////////////////////////////
	// now start cycling through
	////////////////////////////

	int ntries(0);
	int nmoves(0);

	while ( nmoves < total_moves ) {
		++ntries;
		if ( ntries > 5*total_moves ) {
			break;
		}

		// choose a residue at random:
		const float weight ( ran3() * total_weight );
		int ipos;
		for ( ipos=1; ipos<= loop_total_insert; ++ipos ) {
				if ( weight <= weight_map(ipos) ) break;
			}
		if ( ipos > loop_total_insert ) {
			std::cout << "ccd_moves: bad weight? " << weight << ' ' <<
				total_weight << ' ' << weight_map( loop_total_insert ) << std::endl;
			ipos = loop_total_insert;
		}

		const int pos = insert_map( first_insert + ipos - 1 );
		const int direction ( pos <= cutpoint ? n2c : c2n );

		// get fixed and moving positions for this direction and set torsion order:
		if ( direction == n2c ) {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
			torsion_order[ 0 ] = 1; // phi first
			torsion_order[ 1 ] = 2; // psi second
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
			torsion_order[ 0 ] = 2; // psi first
			torsion_order[ 1 ] = 1; // phi second
		}
		get_overlap( pose, misc::Eposition, cutpoint, direction, M);

		// as we change torsion angles through the loop the moving
		// atoms in M will be transformed along with the downstream segment
		// of the loop, by the routine refold_loop_torsions(...)

		float max_angle_delta;
		if ( misc::secstruct(pos) == 'H' ) {
			max_angle_delta = max_angle_delta_helix;
		} else if ( misc::secstruct(pos) == 'E' ) {
			max_angle_delta = max_angle_delta_strand;
		} else {
			max_angle_delta = max_angle_delta_loop;
		}

		// save values for rama score eval, and in case we rama-reject
		const float starting_phi ( misc::phi(pos) );
		const float starting_psi ( misc::psi(pos) );

		// save in case we rama-reject!
		// re-use these index ranges later
		const int Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
		const int Epos_l_end   ( Epos_l_begin +
														 3 * param::MAX_POS * (loop_end-loop_begin+1) );
		assert ( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
		for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
			save_Eposition[l] = misc::Eposition[l];
		}

		for ( int order = 0; order < 2; ++order ) {

			int const torsion = torsion_order[ order ]; // value of 1 in torsion_order array is phi, 2 is psi

			double alpha,dev;
			calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
			double const alpha_orig( alpha );

			// impose per-move deltas
			if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
			if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

			// just rebuilds N,CA,C in Eposition
			refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
													 misc::Eposition, M );

			// update torsions in misc (NOT in pose yet, only when we accept!)
			if ( torsion == 1 ) {
				misc::phi(pos) += alpha;
				misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
			} else {
				misc::psi(pos) += alpha;
				misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
			}

			if ( local_verbose ) {
				double tmp_dev = 0.0;
				for ( int i = 1; i <= 3; ++i ) {
					for ( int j = 1; j <= 3; ++j ) {
						tmp_dev += square( M(j,i) - F(j,i) );
					}
				}
				std::cout << "pos: " << I( 3, pos ) <<
					" alpha: " << fmt::F( 7, 3, alpha ) <<
					" dev1: " << fmt::F( 13, 9, dev ) <<
					" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

				if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
					std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
						tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
				}



			}
		} // torsion


		if ( true ) { // always rama-checking
			float tmp1,tmp2;
			//////////////////////////////////////
			// evaluate the rama score difference:
			float old_rama_score,new_rama_score;
			eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
															 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

			eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
															 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

			if ( new_rama_score > old_rama_score ) {
				const float boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
				const float probability ( std::exp(max(-40.0f,boltz_factor) ) );
				if ( ran3() >= probability ) {
					// undo the change:
					misc::phi(pos) = starting_phi;
					misc::psi(pos) = starting_psi;
					// copy saved Eposition
					for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
						misc::Eposition[l] = save_Eposition[l];
					}
					continue; // go to next position
				} // rama reject
			} // score got worse
		} // if ( rama_check )

		////////////////////////////////////////////////////
		// if we get here we have accepted the phi/psi move:
		////////////////////////////////////////////////////
		pose.set_phi(pos,misc::phi(pos));
		pose.set_psi(pos,misc::psi(pos));
		++nmoves;
	}

}


/// @brief ccd moves that take into account the nonideal bonds of the cutpoint
/// @brief when attempting to close, moves per-torsion
void
per_torsion_ccd_moves_obeying_nonideality(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
)
{

	// debug:
	pose.copy_to_misc();
	assert( misc_in_sync( pose ) );

	/////////
	// params
	/////////
	bool const local_verbose = false;
	int const n2c = { 1 };
	int const c2n = { -1 };
	double const max_angle_delta_helix = { 1.0 }; // per-cycle max move params
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	float const ccd_temperature = { 0.25 }; // for boltzmann rama-checking

	// which residues are insertable?
	// setup arrays for choosing residues at random weighted more toward loop
	// residues
	int total_insert;
	const FArray1D_int & insert_map ( pose.get_insert_map( total_insert ) );
	float const H_weight ( 0.5 );
	float const E_weight ( 1.0 );
	float const L_weight ( 8.5 );
	float total_weight(0.0);
	FArray1D_float weight_map ( total_insert );
	int loop_total_insert(0), first_insert(0);
	for ( int i=1,pos; i<= total_insert; ++i ) {
		pos = insert_map(i);
		if ( pos >= loop_begin && pos <= loop_end ) {
			++loop_total_insert;
			if ( first_insert == 0 ) {
				first_insert = i;
			}

			char ss ( pose.secstruct( pos ) );
			if ( ss == 'H' ) {
				total_weight += H_weight;
			} else if ( ss == 'E' ) {
				total_weight += E_weight;
			} else {
				assert ( ss == 'L' );
				total_weight += L_weight;
			}
			weight_map(loop_total_insert) = total_weight;
		}
	}
	if ( loop_total_insert <= 0 ) {
		std::cout << "no insertable residues in the loop!" << std::endl;
		return;
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		const int pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	////////////////////////////
	// now start cycling through
	////////////////////////////

	int ntries(0);
	int nmoves(0);

	while ( nmoves < total_moves ) {
		++ntries;
		if ( ntries > 5*total_moves ) {
			break;
		}

		// choose a residue at random:
		const float weight ( ran3() * total_weight );
		int ipos;
		for ( ipos=1; ipos<= loop_total_insert; ++ipos ) {
				if ( weight <= weight_map(ipos) ) break;
			}
		if ( ipos > loop_total_insert ) {
			std::cout << "ccd_moves: bad weight? " << weight << ' ' <<
				total_weight << ' ' << weight_map( loop_total_insert ) << std::endl;
			ipos = loop_total_insert;
		}

		const int pos = insert_map( first_insert + ipos - 1 );
		const int direction ( pos <= cutpoint ? n2c : c2n );

		// get fixed and moving positions for this direction and set torsion order:
		if ( direction == n2c ) {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
		}
		get_overlap( pose, misc::Eposition, cutpoint, direction, M);

		// as we change torsion angles through the loop the moving
		// atoms in M will be transformed along with the downstream segment
		// of the loop, by the routine refold_loop_torsions(...)

		float max_angle_delta;
		if ( misc::secstruct(pos) == 'H' ) {
			max_angle_delta = max_angle_delta_helix;
		} else if ( misc::secstruct(pos) == 'E' ) {
			max_angle_delta = max_angle_delta_strand;
		} else {
			max_angle_delta = max_angle_delta_loop;
		}

		// save values for rama score eval, and in case we rama-reject
		const float starting_phi ( misc::phi(pos) );
		const float starting_psi ( misc::psi(pos) );

		// save in case we rama-reject!
		// re-use these index ranges later
		const int Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
		const int Epos_l_end   ( Epos_l_begin +
														 3 * param::MAX_POS * (loop_end-loop_begin+1) );
		assert ( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
		for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
			save_Eposition[l] = misc::Eposition[l];
		}


		// pick random torsion and attempt
		int const torsion = (int)(ran3() * 2 ) + 1; // 1 is phi, 2 is psi

		double alpha,dev;
		calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
		double const alpha_orig( alpha );

		// impose per-move deltas
		if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
		if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

		// just rebuilds N,CA,C in Eposition
		refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
												 misc::Eposition, M );

		// update torsions in misc (NOT in pose yet, only when we accept!)
		if ( torsion == 1 ) {
			misc::phi(pos) += alpha;
			misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
		} else {
			misc::psi(pos) += alpha;
			misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
		}

		if ( local_verbose ) {
			double tmp_dev = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					tmp_dev += square( M(j,i) - F(j,i) );
				}
			}
			std::cout << "pos: " << I( 3, pos ) <<
				" alpha: " << fmt::F( 7, 3, alpha ) <<
				" dev1: " << fmt::F( 13, 9, dev ) <<
				" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

			if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
				std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
					tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
			}
		}


		if ( true ) { // always rama-checking
			float tmp1,tmp2;
			//////////////////////////////////////
			// evaluate the rama score difference:
			float old_rama_score,new_rama_score;
			eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
															 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

			eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
															 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

			if ( new_rama_score > old_rama_score ) {
				const float boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
				const float probability ( std::exp(max(-40.0f,boltz_factor) ) );
				if ( ran3() >= probability ) {
					// undo the change:
					misc::phi(pos) = starting_phi;
					misc::psi(pos) = starting_psi;
					// copy saved Eposition
					for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
						misc::Eposition[l] = save_Eposition[l];
					}
					continue; // go to next position
				} // rama reject
			} // score got worse
		} // if ( rama_check )

		////////////////////////////////////////////////////
		// if we get here we have accepted the phi/psi move:
		////////////////////////////////////////////////////
		pose.set_phi(pos,misc::phi(pos));
		pose.set_psi(pos,misc::psi(pos));
		++nmoves;
	}

}


/// @brief ccd loop closure obeying nonideal bond geometry, moves per-residue
/// @note tolerance in Angstroms, forward and backward splice RMS over N,CA,C must
/// @note be less than tolerance for an early return. otherwise goes through the
/// @note loop ii_cycles, each time both forward and backward
/// @return  the number of cycles actually taken
int
fast_ccd_loop_closure_obeying_nonideality(
	Pose & pose,
	int const & loop_begin,
	int const & loop_end,
	int const & cutpoint,
	int const & ii_cycles,
	float const & tolerance,
	bool const & rama_check,
	float const & max_rama_score_increase,
	float const & max_total_delta_helix,
	float const & max_total_delta_strand,
	float const & max_total_delta_loop,
	float & forward_deviation, // output
	float & backward_deviation, // output
	float & torsion_delta,
	float & rama_delta
)
{
	pose.copy_to_misc();
	assert( misc_in_sync( pose ));

	/////////
	// params
	/////////
	bool const local_verbose = false;
	int const n2c = { 1 };
	int const c2n = { -1 };
	// per-cycle max move params
	double const max_angle_delta_helix = { 1.0 };
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	// for rama-checking with Boltzman criterion (should be higher?)
	//	float const ccd_temperature = { 0.25 };
	//read temperature from command-line....this will be updated later.
	static float ccd_temperature = realafteroption( "ccd_temperature", 0.25 );



	///////
	// POSE
	FArray1D_bool allow_bb_move( loop_end, false );
	for ( int i=loop_begin; i<= loop_end; ++i ) {
		allow_bb_move( i ) = pose.get_allow_bb_move( i );
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		int const pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	//////////////////////////////////////
	// save starting torsions, rama scores
	FArray1D_float start_phi(loop_end), start_psi(loop_end),
		start_rama_score(loop_end);
	float tmp1,tmp2;
	for ( int i=loop_begin; i<= loop_end; ++i ) {
		start_phi(i) = misc::phi(i);
		start_psi(i) = misc::psi(i);
		eval_rama_score_residue( misc::res(i), misc::phi(i), misc::psi(i),
										 misc::secstruct(i), start_rama_score(i), tmp1, tmp2 );
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	// torsion order
	int torsion_order[ 2 ]; // value of 1 in array is phi, 2 is psi

	// for reporting how many cycles we actually took
	int total_cycles = ii_cycles;

	////////////////////////////
	// now start cycling through
	////////////////////////////
	for ( int ii = 1; ii <= ii_cycles; ++ii ) {
		// first forward, then backward
		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			int direction,start_pos,stop_pos,increment;
			if ( repeat == 1 ) {
				direction = n2c;
				start_pos = loop_begin;
				stop_pos = std::min(loop_end,cutpoint);
				increment = 1;
			} else {
				direction = c2n;
				start_pos = loop_end;
				stop_pos = std::max(loop_begin,cutpoint+1);
				increment = -1;
			}

			// get fixed and moving positions for this direction:
			if ( direction == n2c ) {
				for ( int i = 1; i <= 3; ++i ) {
					F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
					F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
					F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
				}
				torsion_order[ 0 ] = 1; // phi first
				torsion_order[ 1 ] = 2; // psi second
			} else {
				for ( int i = 1; i <= 3; ++i ) {
					F(i,1) = misc::Eposition(i,1,cutpoint); // N
					F(i,2) = misc::Eposition(i,2,cutpoint); // CA
					F(i,3) = misc::Eposition(i,4,cutpoint); // C
				}
				torsion_order[ 0 ] = 2; // psi first
				torsion_order[ 1 ] = 1; // phi second
			}
			get_overlap( pose, misc::Eposition, cutpoint, direction, M );
			// as we change torsion angles through the loop the moving
			// atoms in M will be transformed along with the downstream segment
			// of the loop, by the routine refold_loop_torsions(...)

			for ( int pos = start_pos; pos*increment <= stop_pos*increment;
						pos += increment ) {

				if ( !allow_bb_move( pos ) ) continue;

				float max_angle_delta, max_total_delta;
				if ( misc::secstruct(pos) == 'H' ) {
					max_angle_delta = max_angle_delta_helix;
					max_total_delta = max_total_delta_helix;
				} else if ( misc::secstruct(pos) == 'E' ) {
					max_angle_delta = max_angle_delta_strand;
					max_total_delta = max_total_delta_strand;
				} else {
					max_angle_delta = max_angle_delta_loop;
					max_total_delta = max_total_delta_loop;
				}

				if ( max_total_delta <= 0.01 ) {
					//std::cout << "cant move this residue " << pos << std::endl;
					continue;
				}

				// save values for rama score eval, and in case we rama-reject
				float const starting_phi ( misc::phi(pos) );
				float const starting_psi ( misc::psi(pos) );

				// save in case we rama-reject!
				// re-use these index ranges later
				int const Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
				int const Epos_l_end   ( Epos_l_begin +
																 3 * param::MAX_POS * (loop_end-loop_begin+1) );
				assert( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
				for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
					save_Eposition[l] = misc::Eposition[l];
				}

				for ( int order = 0; order < 2; ++order ) {

					int const torsion = torsion_order[ order ]; // value of 1 in torsion_order array is phi, 2 is psi

					double alpha,dev;
					calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
//					double const alpha_orig( alpha );

					// impose per-move deltas
					if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
					if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

					// check for total movement during closure run:
					float const total_delta
						( ( torsion == 1 ) ? subtract_degree_angles( start_phi(pos),
																	 misc::phi(pos) + alpha )
							                 : subtract_degree_angles( start_psi(pos),
																	 misc::psi(pos) + alpha ) );

					if ( total_delta > max_total_delta ) {
						// this logic is a little tricky: if adding alpha to the previous
						// delta pushes us past 180 from start, then it wont work, so check for
						// that (note that if max_total_delta > 180 we wont even get here ):
						assert( alpha + max_total_delta < 180 );
						if ( alpha > 0 ) {
							alpha -= ( total_delta - max_total_delta + 0.01 );
						} else {
							alpha += ( total_delta - max_total_delta + 0.01 );
						}
					}

					// just rebuilds N,CA,C in Eposition
					if ( local_verbose )
						std::cout << "refold_loop_torsion: " << alpha << ' ' << pos <<
							' ' << torsion << ' ' << direction << ' ' << cutpoint << std::endl;

					refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
															 misc::Eposition, M );

					//dump_pdb("refold_loop_torsion");

					// update torsions in misc (NOT in pose yet, only when we accept!)
					if ( torsion == 1 ) {
						misc::phi(pos) += alpha;
						misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
					} else {
						misc::psi(pos) += alpha;
						misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
					}

				}               // torsion = 1,2


				if ( rama_check ) {
					//////////////////////////////////////
					// evaluate the rama score difference:
					float old_rama_score,new_rama_score;
					eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
																	 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

					eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
																	 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

					if ( new_rama_score > old_rama_score ) {
						float const boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
						float const probability ( std::exp(std::max(-40.0f,boltz_factor) ) );
						if ( new_rama_score - start_rama_score( pos ) > max_rama_score_increase ||
								 ran3() >= probability ) {
							// undo the change:
							misc::phi(pos) = starting_phi;
							misc::psi(pos) = starting_psi;
							// copy saved Eposition
							for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
								misc::Eposition[l] = save_Eposition[l];
							}
							continue; // go to next position
						} // rama reject
					} // score got worse
				} // if ( rama_check )

				////////////////////////////////////////////////////
				// if we get here we have accepted the phi/psi move:
				////////////////////////////////////////////////////
				pose.set_phi(pos,misc::phi(pos));
				pose.set_psi(pos,misc::psi(pos));
			} // pos = start_pos,stop_pos
		} // repeat = 1,2   1=n2c; 2=c2n

		if ( mod(ii,5) == 0 || ii == ii_cycles ) {
			// check overlap deviations to see if loop is closed
			// every 5 cycles or on the last one.
			//
			// forward_deviation:
			int direction = n2c;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
			get_overlap( pose, misc::Eposition, cutpoint,direction,M);
			forward_deviation = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					forward_deviation += square( M(j,i) - F(j,i) );
				}
			}
			forward_deviation = sqrt( forward_deviation / 3 );
			// backward_deviation:
			direction = c2n;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
			get_overlap( pose, misc::Eposition, cutpoint,direction,M);
			backward_deviation = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					backward_deviation += square( M(j,i) - F(j,i) );
				}
			}
			backward_deviation = sqrt( backward_deviation / 3 );
			if ( forward_deviation < tolerance && backward_deviation < tolerance ) {
				if ( local_verbose )
					std::cout << "closed early: ii= " << ii << ' ' << tolerance
										<< std::endl;
				total_cycles = ii;
				break;
			}
		}                  // check deviations
	}                     // ii=1,ii_cycles

	// calculate torsion dev, rama delta
	torsion_delta = 0.0;
	rama_delta = 0.0;

	for ( int i=loop_begin; i<= loop_end; ++i ) {
		torsion_delta += std::abs( subtract_degree_angles( start_phi(i), misc::phi(i) ));
		torsion_delta += std::abs( subtract_degree_angles( start_psi(i), misc::psi(i) ));
		//std::cout << "fast_ccd_closure phi: start, final, abs_delta " << I( 3, i ) << " " << start_phi(i)  << " " << misc::phi(i)  << " " << std::abs( subtract_degree_angles( start_phi(i), misc::phi(i) )) << std::endl;
		//std::cout << "fast_ccd_closure psi: start, final, abs_delta " << I( 3, i ) << " " << start_psi(i)  << " " << misc::psi(i)  << " " << std::abs( subtract_degree_angles( start_psi(i), misc::psi(i) )) << std::endl;
		float final_rama_score;
		eval_rama_score_residue( misc::res(i), misc::phi(i), misc::psi(i),
														 misc::secstruct(i), final_rama_score, tmp1, tmp2 );
		rama_delta += ( final_rama_score - start_rama_score(i) );
	}
	torsion_delta /= ( loop_end - loop_begin + 1);
	rama_delta /= ( loop_end - loop_begin + 1);
	//std::cout << "fast_ccd_closure avg rama_delta over loop = " << rama_delta << std::endl;

	return total_cycles;
}


/// @brief get overlap position, obeys nonideal bond geometry
void
get_overlap(
	Pose const & pose,
	FArray3DB_float const & Eposition,// dont get from pose -- dont want a refold
	int const & cutpoint,
	int const & dir,
	FArray2Da_double M // output
)
{
	static FArray2D_float overlap_xyz(3,6);

	build_overlap_coords( pose, cutpoint, Eposition, overlap_xyz, true, dir );

	int const n2c(1);
	int const j_offset( dir == n2c ? 3 : 0 );
	for ( int k=1; k<=3; ++k ) {
		for ( int j=1; j<= 3; ++j ) {
			M(k,j) = overlap_xyz(k,j+j_offset);
		}
	}
}


/// @brief build overlap coordinates, obeys nonideal geometry
void
build_overlap_coords(
	Pose const & pose,
	int const & cutpoint,
	FArray3DB_float const & coords,
	FArray2D_float & overlap_xyz,
	bool const & Epos_index,
	int const & dir
)
{
	float const & psi = pose.psi( cutpoint );
	float const & omega = pose.omega( cutpoint );
	float const & phi = pose.phi( cutpoint + 1 );

	int const N_index(1), CA_index(2), C_index( Epos_index ? 4 : 3 );
	int const n2c(1), c2n(-1);
	//float const omega(180.0);

	//FArray3D_float const & coords( pose.full_coord() );
	FArray1D_float c_xyz(3), n_xyz(3), ca_xyz(3);

	build_C_coords( pose, cutpoint + 1, phi, coords(1,N_index,cutpoint+1),
	                coords(1,CA_index,cutpoint+1), coords(1,C_index,cutpoint+1), c_xyz );

	build_N_CA_coords( pose, cutpoint, psi, omega,
	                   coords(1,N_index,cutpoint), coords(1,CA_index,cutpoint),
	                   coords(1,C_index,cutpoint), n_xyz, ca_xyz );

	FArray2D_float Mgl( 4, 4 );
	if ( !dir || dir == c2n ) {
		// build overlap_xyz 1-3
		// by mapping the existing coords for n,ca,c of cutpoint
		// by a transformation that maps c,n_xyz,ca_xyz to c_xyz,n,ca
		get_GL_matrix( ca_xyz,
									 n_xyz,
									 coords(1,  C_index, cutpoint),
									 coords(1, CA_index, cutpoint+1),
									 coords(1,  N_index, cutpoint+1),
									 c_xyz,
									 Mgl);
		GL_rot( Mgl, coords(1,  N_index, cutpoint ), overlap_xyz(1,1) );
		GL_rot( Mgl, coords(1, CA_index, cutpoint ), overlap_xyz(1,2) );
		GL_rot( Mgl, coords(1,  C_index, cutpoint ), overlap_xyz(1,3) );
	}

	if ( !dir || dir == n2c ) {
		// build overlap_xyz 4-6 -- could just invert the transform??
		get_GL_matrix( coords(1, CA_index, cutpoint+1),
									 coords(1,  N_index, cutpoint+1),
									 c_xyz,
									 ca_xyz,
									 n_xyz,
									 coords(1,  C_index, cutpoint),
									 Mgl);
		GL_rot( Mgl, coords(1,  N_index, cutpoint+1 ), overlap_xyz(1,4) );
		GL_rot( Mgl, coords(1, CA_index, cutpoint+1 ), overlap_xyz(1,5) );
		GL_rot( Mgl, coords(1,  C_index, cutpoint+1 ), overlap_xyz(1,6) );
	}

	return;
}


/// @brief builds coordinates of backbone C, obeys nonideal bond geometry
/// @note  uses: the n_xyz position, the n->ca bond vector, the n,ca,c oriented plane
/// @warning  pos is the position one residue after the C, ie the position to whom phi,n,ca,c belong!!!
void
build_C_coords(
	Pose const & pose,
	int const & pos,
	float const & phi,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float c_xyz_out
)
{
	using namespace refold_ns;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	c_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	subvec( n_xyz,ca_xyz,X(1,ph)); // order is tricky: we are folding c2n
	subvec(ca_xyz, c_xyz,X(1,ps)); // in the call to build_atom!

// the X-axis of this coord system (M(*,1)) is parallel to
// X(1,ph), ie along the n->ca bond vector

	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));

	bonds_class::Bonds const & pose_bonds = pose.bonds();
	build_atom( M, n_xyz, pose_bonds.cT()(c,c2n,pos-1),
	            pose_bonds.sT()(c,c2n,pos-1), phi,
	            pose_bonds.D ()(c,c2n,pos-1), c_xyz_out, X(1,om));
}


/// @brief builds coordinates of backbone N,CA of next position forward in sequence
/// @warning  pos is the position to whom psi, omega, and n,ca,c_xyz belong ie one
/// @wrarnin  residue before the residue for whom we are actually building coordinates!!
void
build_N_CA_coords(
	Pose const & pose,
	int const & pos,
	float const & psi,
	float const & omega,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz_out,
	FArray1Da_float ca_xyz_out
)
{
	using namespace refold_ns;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	n_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	subvec(ca_xyz, n_xyz,X(1,ph)); // order is tricky: we are folding n2c
	subvec( c_xyz,ca_xyz,X(1,ps)); // in the call to build_atom!

	bonds_class::Bonds const & pose_bonds = pose.bonds();
	int const bonds_pos( pos+1 );
	// N
	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom( M, c_xyz, pose_bonds.cT()(n,n2c,bonds_pos),
	            pose_bonds.sT()(n,n2c,bonds_pos), psi, pose_bonds.D()(n,n2c,bonds_pos),
	            n_xyz_out, X(1,om));
	// CA
	refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom( M, n_xyz_out, pose_bonds.cT()(ca,n2c,bonds_pos),
	            pose_bonds.sT()(ca,n2c,bonds_pos), omega, pose_bonds.D()(ca,n2c,bonds_pos),
	            ca_xyz_out, X(1,ph) );
}


} // namespace design
} // namespace epigraft
