// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 12:37:52 -0500 (Fri, 10 Mar 2006) $
//  $Author: stuartm $


// Rosetta Headers
#include "connect_templates.h"
#include "aaproperties_pack.h"
#include "files_paths.h"
#include "jumping_loops.h"
#include "jumping_ns.h"
#include "jumping_refold.h"
#include "loops.h"
#include "loops_ns.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "tree.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
///@begin init_jmp_loops
///
///@brief calls the connect_templates subrountines and then calls links_to_tree
///
///@detailed calls setup_links to connect Template Regions, calls links_to_tree
///
///@global_read
///        current_pose   misc.h
///
///@global_write
///
///@remarks
///
///@refrences
///
///@authors firas 6/3/2004
///
///@last_modified
/////////////////////////////////////////////////////////////////////////////////


void
init_jmp_loops()
{
  using namespace misc;
  using namespace param;

  FArray2D_int edgeList( 2, MAX_RES()() );
  FArray1D_int breakpoints( MAX_RES()() );
  int numOfEdges, numOfBreaks;

  setup_links(edgeList,numOfEdges,breakpoints,numOfBreaks);

  for ( int i = 1; i <= numOfBreaks; ++i ) {
    std::cerr << "Breakpoint at residue: " << breakpoints(i) << "\n";
  }

  for ( int i = 1; i <= numOfEdges; ++i ) {
    std::cerr << "EDGE: Residues " << edgeList(1,i) << " and " <<
      edgeList(2,i) << " are connected \n";
  }

  links_to_tree(edgeList,numOfEdges,breakpoints,numOfBreaks,position,total_residue);
  set_jumping_flag(true);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin links_to_tree
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car, firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
links_to_tree(
	FArray2DB_int const & links,
	int const nlinks,
	FArray1DB_int const & cuts,
	int const ncuts,
	FArray3Da_float Eposition,
	int const total_residue
)
{
	using namespace param;
	using namespace jumping;

	Eposition.dimension( 3, MAX_POS, MAX_RES() );

	FArray2D_int endpoints( 2, MAX_RES()() ); // 2 = type (1 cut, 0  link) 1= res#

	// construct list of all cut and link points, sort

	int i = 1;
	endpoints(1,i)=1;
	endpoints(2,i)=1;
	for ( int icut = 1; icut <= ncuts; ++icut ) {
		++i;
		endpoints(1,i) = cuts(icut);
		endpoints(2,i)= 1;
		++i;
		endpoints(1,i) = cuts(icut)+1;
		endpoints(2,i) = 1;
	}
	for ( int ilink = 1; ilink <= nlinks; ++ilink ) {
		++i;
		endpoints(1,i) = links(1,ilink);
		endpoints(2,i) = 0;
		++i;
		endpoints(1,i) = links(2,ilink);
		endpoints(2,i) = 0;
	}
	++i;
	endpoints(1,i)=total_residue;
	endpoints(2,i)=1;

	hpsort_array(i,endpoints,2,1);

	fold_tree(1,0)=0;          //empty graph
	for ( int j = 2; j <= i; ++j ) {
		if ( endpoints(1,j) - endpoints(1,j-1) != 0 ) { // not ident.
			if ( endpoints(2,j-1) == 1 ) { // j-1 is a cut
				if ( endpoints(2,j) == 0 ) { // j is a link
					add_edge(fold_tree,endpoints(1,j-1),endpoints(1,j),-1);
					 //j must be cut
				} else if ( endpoints(1,j) - endpoints(1,j-1) != 1 ) { // not seq
					add_edge(fold_tree,endpoints(1,j-1),endpoints(1,j),-1);
				}
			} else {              // j-1 is a link
				add_edge(fold_tree,endpoints(1,j-1),endpoints(1,j),-1);
				 // j must be link
			}
		}
	}
	for ( int j = 1; j <= nlinks; ++j ) {
		add_edge(fold_tree,links(1,j),links(2,j),j);
		// forward
		calculate_jump(Eposition(1,1,links(1,j)),Eposition(1,1,links(2,j)),
			 jump_transform(1,1,1,j));
		// backward
		calculate_jump(Eposition(1,1,links(2,j)),Eposition(1,1,links(1,j)),
			 jump_transform(1,1,2,j));
	}
	reorder_tree(fold_tree,1);
	new_fold_tree();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin connect_template
///
/// @brief connects the closest Template regions together
///
/// @detailed Given a Template, and a list of connectable residues for each
///          Template Region, connects the closest Template regions together
///          until they are all linked up together as one big Template Region.
///          Local cyclic links are not allowed.
///
/// @param[in] connectableResidues - an array of all the residues that are in
///                                 Template Regions and are allowed to connect
///                                 to other Template Regions. Length N.
/// @param[in] N - number of connectableResidues
/// @param[in] D - length of distances array that will be created
/// @param[in] Template Regions - denotes which residues are within the same
///                              Template Region. All rest set to 0. Length M.
/// @param[out] edgeList - array of final connected residues
/// @param[out] number_of_edges - number of residues connected in edgeList
/// @param[out] breaks - array of all the breakpoints
/// @param[out] b_count - number of breakpoints
///
/// @return a list of linked edges: edgeList
///        also writes out Edgelist and Breakpoints to a file: target.edges
///
/// @global_read
///        current_pose   misc.h
///
/// @global_write
///
/// @remarks
///   this program is great!
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
connect_template(
	FArray1DB_int & connectableResidues,
	int & N,                // previously linkCounter
	int & D,                // instead of (total_res)^2, it's smaller
	FArray1DB_int & TemplateRegions,
	FArray2DB_int & edgeList,
	int & number_of_edges,
	FArray1DB_int & breaks,
	int & b_count
)
{
	using namespace param;
	using namespace misc;
	using namespace files_paths;
	using namespace loops_ns;

	//local

	int M;            //M is the number of C-alphas
	int numTR;        //numTR is the number of Template Regions
	int edgeIndex;
	int LODM;         // Length Of Distance Matrix (after it has been created)
										// Since residues within the same Template Region cannot
										// be connected, this makes the LODM less than D
										// so it will have a smaller matrix to sort
	FArray2D_float CA_chain( 3, total_residue);
	FArray2D_float distances( 4, D);
	std::string fullname;

	//rename values
	numTR = num_loop;
	M = total_residue;

	//setup C-alpha chain

	for ( int i = 1; i <= M; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			CA_chain(j,i) = Eposition(j,2,i);
		}
	}

	//Create the data structures

	createDistances(CA_chain, M, distances, TemplateRegions,
			connectableResidues, N, LODM);

	//SORT THE DISTANCES

	sortDistances(LODM,distances);

	//Connect all the template regions

	connect_template_regions(distances,LODM,TemplateRegions,M,N,numTR,
				number_of_edges);

	//Return a list of all connected edges

	edgeIndex = 1;
	fullname = start_path + start_file + ".edges";
	user_x.clear();
	user_x.open( fullname.c_str() );
	for ( int i = 1; i <= LODM; ++i ) {
		if ( distances(4,i) == 1 ) {        //this edge was connected
			std::cerr << "EDGE: Residue " << distances(1,i) <<
				" connected to residue " << distances(2,i) << '\n';
			//  Writes out edge pairs in filename.edges (just the numbers)
			user_x << "\t"  <<  static_cast< int >(distances(1,i)) << "\t" << static_cast< int >(distances(2,i)) << '\n';

			// this is to keep an array of the edges
			// so that this subroutine can pass that out if needed
			createEdgeList(distances,LODM,i,edgeList,number_of_edges,edgeIndex);
			++edgeIndex;
		}
	}

	for ( int i = 1; i <= b_count; ++i ) {
		user_x << "\t" << breaks(i) << "\n";
	}

	user_x.close();
	user_x.clear();
	--edgeIndex;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin setup_links
///
/// @brief Prepares necessary arrays for what is allowed to connect
///
/// @detailed Find out which residues are part of which Template Region
///
/// @param[in] edgeList - array of final connected residues
/// @param[in] LOEL - current Length Of the Edge List
/// @param[in] breakpointsList - array of all the breakpoints
/// @param[in] LOBL - current Length Of the Breakpoints List
///
/// @global_read
///        loop_int loops_ns.h
///
/// @global_write
///
/// @remarks
///   prints out all the breakpoints
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
setup_links(
	FArray2DB_int & edgeList,
	int & LOEL,
	FArray1DB_int & breakpointsList,
	int & LOBL
)
{
	using namespace param;
	using namespace misc;
	using namespace files_paths;
	using namespace loops_ns;

	//local

	int D, beginTemplate, endTemplate, center, linkCounter, b_count;
	FArray1D_int breaks( total_residue );
	FArray1D_int Template( total_residue ); //had to change to uppercase
	FArray1D_int connectableResidues( total_residue );

	linkCounter = 1;     // index of connectableResidues array
	int j = 1;               // index of Loop Regions
	beginTemplate = 1;
	b_count = 1;         // index of break array (breakpoints)

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( (loop_begin(j) == i) && (i != 1)) {
			endTemplate = i - 1;    //the template ended on previous residue
			// now calculate which residues in the template are connectable
			if ( beginTemplate == endTemplate ) {
				// this is for the case where there is only one residue
				// seperating two Loop Regions. Therefore only it will link.
				connectableResidues(linkCounter) = endTemplate;
				Template(endTemplate) = endTemplate;
				++linkCounter;
				goto L1;
			}
			//get secondary structure info and make connections based on that
			getSecStructAndAssign(beginTemplate, endTemplate,Template,
														connectableResidues,linkCounter,breaks,b_count);
		}
	L1:
		Template(i) = 0;
		if ( loop_end(j) == i) {
			beginTemplate = i + 1;
			++j;
		}
	}

	if ( loop_end(j-1) < total_residue) {
		// the last Loop Region is before the end of the chain
		// so there is one last template region to do.
		endTemplate = total_residue;
		if (beginTemplate == endTemplate) {
			// this is for the case where there is only one residue
			// as the last Template Region
			connectableResidues(linkCounter) = endTemplate;
			Template(endTemplate) = endTemplate;
			++linkCounter;
			goto L2;
		}
		// get secondary structure info and make connections based on that
		getSecStructAndAssign(beginTemplate, endTemplate, Template,
													connectableResidues,linkCounter,breaks,b_count);
	L2:;
	}

	//  print out all the breakpoints: the middle residue in each
	//  Loop Region as well as the breakpoints within Template Regions
	//  that were split up due to their secondary structure

	j=1;
	--b_count;
	LOBL = 1;

	for ( int i = 1; i <= num_loop; ++i ) {
		if ((loop_begin(i) != 1) && (loop_end(i) != total_residue)) {
			// do not want breakpoints for Loop Regions at the ends
			beginTemplate = loop_begin(i);
			endTemplate = loop_end(i);
			center = static_cast< int > (beginTemplate + endTemplate)/2;
			while ((j <= b_count) && (center > breaks(j))) {
				//then need to print out Template Region breakpoint
				// before printing out the breakpoint between LoopRegions
				std::cerr << "Breakpoint in TR at:" << breaks(j) << "\n";
				breakpointsList(LOBL) = breaks(j);
				++j;
				++LOBL;
			}
			std::cerr << "Breakpoint of Loop Region " << i << " is " << center << "\n";
			breakpointsList(LOBL) = center;
			++LOBL;
		}
	}

	while (j <= b_count) {
		//there are Template Region Breakpoints after the second
		// to last LoopRegion (since we do not care about last LR)
		std::cerr << "Breakpoint in TR at: " << breaks(j) << "\n";
		breakpointsList(LOBL) = breaks(j);
		++j;
		++LOBL;
	}
	--LOBL;

	//Setup variables for connect_template subroutine

	--linkCounter;

	D = ((total_residue) * (total_residue - 1)) / 2;

	connect_template(connectableResidues,linkCounter,D,Template,
			 edgeList,LOEL,breakpointsList,LOBL);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sortDistances
///
/// @brief sorts a (4 by n) array on the third element (distance)
///
/// @detailed car heapsort algorithm from numerical recipes
///          car modified here from hpsort to use multidimensional ra
///
/// @param[in] n - the total number of distances
/// @param[in,out] ra - distances array
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car, firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
sortDistances(
	int n,
	FArray2Da_float ra
)
{
	ra.dimension( 4, n );

	int const size = { 4 }; // first dim of ra
													// set at 4: res1,res2,distance,linked?
	int const key = { 3 }; // key to sort on
												 // set to sort on Distance (3)
	int i,ir,j,l,k;
	FArray1D_float rra( size );

	if ( n < 2 ) return;
	l = n/2+1;
	ir = n;
L10:
	if ( l > 1 ) {
		--l;
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,l);
		}
	} else {
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,ir);
		}
		for ( k = 1; k <= size; ++k ) {
			ra(k,ir) = ra(k,1);
		}
		--ir;
		if ( ir == 1 ) {
			for ( k = 1; k <= size; ++k ) {
				ra(k,1) = rra(k);
			}
			return;
		}
	}
	i = l;
	j = l+l;
L20:
	if ( j <= ir ) {
		if ( j < ir ) {
			if ( ra(key,j) < ra(key,j+1) ) ++j;
		}
		if ( rra(key) < ra(key,j) ) {
			for ( k = 1; k <= size; ++k ) {
				ra(k,i) = ra(k,j);
			}
			i = j;
			j += j;
		} else {
			j = ir+1;
		}
		goto L20;
	}
	for ( k = 1; k <= size; ++k ) {
		ra(k,i) = rra(k);
	}
	goto L10;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin createEdgeList
///
/// @brief puts two residues that are connected in the edgeList array for output
///
/// @detailed
///
/// @param[in] distMatrix - the distance matrix
/// @param[in] LODM - the length of the distance matrix
/// @param[in] a - the current position in the distance matrix
/// @param[in] edgeList - array of final connected residues
/// @param[in] N - the number of edges in the edgeList
/// @param[in] index - which edge we are currently writing
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
createEdgeList(
	FArray2DB_float & distMatrix,
	int & LODM,
	int & a,
	FArray2DB_int & edgeList,
	int & N,
	int & index
)
{
	edgeList(1,index) = static_cast< int >(distMatrix(1,a));
	edgeList(2,index) = static_cast< int >(distMatrix(2,a));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin createDistances
///
/// @brief calculates all the distances between connectable residues
///
/// @detailed
///
/// @param[in] chain - c-alpha chain
/// @param[in] M - the length of the c-alpha chain (total_residue)
/// @param[out] distances - distances between residues
/// @param[in] TR - TemplateRegions array
/// @param[in] connectR - connectableResidues
/// @param[in] N - number of connectable Residues
/// @param[out] actual_length_of_distances_matrix - final length of distances
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
createDistances(
	FArray2DB_float & chain,
	int & M,
	FArray2DB_float & distances,
	FArray1DB_int & TR,
	FArray1DB_int & connectR,
	int & N,
	int & actual_length_of_distances_matrix
)
{
	//local

	int current, next, compare;
	float dist;            //distance between two residues

	//create distances matrix

	int c = 1;               //counter

	for ( int i = 1; i < N; ++i ) {
		current = connectR(i);
		next = i + 1;
		for ( int j = next; j <= N; ++j ) {
			compare = connectR(j);
			if (TR(current) != TR(compare)) {
				//they are not in the same TemplateRegion
				//so calculate the distance between them
				calc_distance(chain, M, current, compare, dist);
				distances(1,c)=current;  //residue1
				distances(2,c)=compare;  //residue2
				distances(3,c)=dist;     //distance btwn res1 and res2
				distances(4,c)=0;        //denotes they are not connected yet
				++c;
			}
		}
	}

	actual_length_of_distances_matrix = c - 1;

	//denotes how many distances were calculated so that this value
	//can be passed to the sorting algorithm instead of N(N-1)/2
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_distance
///
/// @brief calculates the square of the distance between two residues
///
/// @detailed
///
/// @param[in] chain - c-alpha chain
/// @param[in] M - the length of the c-alpha chain (total_residue)
/// @param[in] res1 - first residue
/// @param[in] res2 - second residue
/// @param[out] distance - distance squared between both residues
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_distance(
	FArray2DB_float & chain,
	int & M,
	int & res1,
	int & res2,
	float & distance
)
{

	distance = ((chain(1,res1) - chain(1,res2))*(chain(1,res1) - chain(1,res2)))
		+ ((chain(2,res1) - chain(2,res2))*(chain(2,res1) - chain(2,res2)))
		+ ((chain(3,res1) - chain(3,res2))*(chain(3,res1) - chain(3,res2)));

}

////////////////////////////////////////////////////////////////////////////////
/// @begin connect_template_regions
///
/// @brief connects all the closest Template Regions
///
/// @detailed starting with the shortest distances, connects residues that are
///          not in the same Template Region and then count that as 1 Region
///
/// @param[in] sorted_dist - the sorted distance matrix
/// @param[in] L - the number of distances
/// @param[in,out] TR - the Template Regions array
/// @param[in] M - the number of residues
/// @param[in] N - the total # of connectable residues
/// @param[in] numTR - the number of Template Regions
/// @param[out] number_of_edges - the final number of edges linked
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
connect_template_regions(
	FArray2DB_float & sorted_dist,
	int & L,
	FArray1DB_int & TR,
	int & M,
	int & N,
	int & numTR,
	int & number_of_edges
)
{
	int previous_TR, tally;

	number_of_edges = 0;
	int current = -1;


	for ( int i = 1; i <= L; ++i ) {  //starting with shortest distance
		if (TR(static_cast< int >(sorted_dist(1,i))) !=
				TR(static_cast< int >(sorted_dist(2,i)))) {
			// then the two residues are not in the same Template Region
			// so they can be connected
			sorted_dist(4,i) = 1;       //denote that they are linked
			++number_of_edges;
			previous_TR = TR(static_cast< int >(sorted_dist(2,i)));
			for ( int j = 1; j <= M; ++j ) {
				if (TR(j) == previous_TR) {
					//need to change all the residues that were in
					//residue 2's Template Region to now be in res1's TR
					//(ie res(1,4,5) are in TR 1 and res(2,8) are in TR 2
					//and we are connecting res 2 -> res 5. So we want to
					//change res1 -> TR2, res4 -> TR2, and res5 -> TR2
					// so that now res(1,2,4,5,8) are all in TR 2)
					TR(j) = TR(static_cast< int >(sorted_dist(1,i)));
				}
			}
			current = TR(static_cast< int >(sorted_dist(1,i)));//note an existing TR
		}

		// Next statement checks to see if they are all in the same
		// Template Region. Once that happens, we can end.
		// This check is not useful early on, so we do not do it
		// until it has tried to connect the minimum number of nodes

		if ( i >= (numTR-1) ) {    //check to see if all in same TR
			tally = 0;
			for ( int j = 1; j <= M; ++j ) {
				if (TR(j) != 0) {             // check it
					if (TR(j) == current) {     // tally it
						++tally;
					}
				}
			}
			if (tally == N) {       // everything is the same Template Region
				return;               // so we can end
			}
		}
	}
	std::cerr << "IT DID NOT END UP WITH ALL NODES CONNECTED \n";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin getSecStructAndAssign
///
/// @brief This extracts the secondary structure information for the given
///       Template Region and only allows Helix or Strand members to be linked
///
/// @detailed  The only way non-helix or non-strand residues are connectable is if
///           that template region contains NO Helix or strand members
///   Different Secondary Structure members within the same Template Region
///   are seperated so that each secondary structure is their own region.
///   ie [34-56] is a Template Region with [36-40] being helix and [50-56]
///       being strand. Now [36-40] is part of Region 36 and [50-56] is part
///       of region 50 (the helix and strand regions can now be connected)
///   For longer regions, only the middle X residues (where X is the
///       parameter: middle_width ) are connectable
///   RESTRICTIONS:
///   There need to be at least Z E (strands) in a row to be considered a
///   region and there needs to be at least Y H (helices) in a row.
///       If this is not the case, they are considered L residues
///       (where Z is the parameter: min_strand and Y: min_helix)
///   In the middle of one type of secondary structure, there needs to be
///   more than "L_break" between them in order to seperate the regions.
///   ie if L_break = 1, EELEE is 1 region EELLEE is 2 different regions.
///       EELHH is two different regions as well (since not the same ss)
///   There is a special case for Helices and L breaks involving Prolines.
///
/// @param[in] beginTemplate - the starting point of the Template Region
/// @param[in] endTemplate - the end point of the Template Region
/// @param[in,out] Template - the Template Regions array
/// @param[in,out] connectableResidues - which residues are connectable
/// @param[in] linkCounter - the number of residues that have been connected
/// @param[in,out] breakpoint - the breakpoints array
/// @param[in,out] b_count - the breakpoints counter
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     this function is muy complicado!
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
getSecStructAndAssign(
	int & beginTemplate,
	int & endTemplate,
	FArray1DB_int & Template,
	FArray1DB_int & connectableResidues,
	int & linkCounter,
	FArray1DB_int & breakpoint,
	int & b_count
)
{
	using namespace param;
	using namespace misc;
	using namespace files_paths;
	using namespace loops_ns;

	int const middle_width = { 3 }; //the number of middle residues that
																	//are allowed to connect to others
	int const min_helix = { 3 };    //the minimum number of Hs
																	//to be considered a Helix Region
	int const min_strand = { 2 };   //the minimum number of Es
																	//to be considered a Strand Region
	int const L_break = { 1 };      //the number of Ls needed to break a ss run
					// ie HHLHH would be considered one structure if L_break = 2
					// since it would need at least 2 Ls to break it: HHLLHHH for ex.

	int first;
	int length = endTemplate - beginTemplate;

	bool ssH_exists = false;
	bool ssE_exists = false;
	bool ssL_exists = false;
	bool connected = false;
	bool p_exists = false;

	int helix_counter = 0;
	int strand_counter = 0;
	int loop_counter = 0;
	int firstStrand;
	int lastStrand;
	int firstHelix;
	int lastHelix;

	for ( int a = 0; a <= length; ++a ) {
		if (secstruct(beginTemplate+a) == 'H') {
			if (ssE_exists) {              //the preceding region was strands
				// we need to do all the connections for that E region
				if (strand_counter >= min_strand) {  //it is linkable
					if (! ssL_exists) {     // then there was no L
						// btwn the E and the H: ie EEEH so we need to note
						// that the lastStrand of E was the preceding residue
						lastStrand = beginTemplate + a - 1;      // the last E
					}
					connectRegion(firstStrand,strand_counter,lastStrand,linkCounter,
												Template,connectableResidues,middle_width);
					connected = true;
					breakpoint(b_count) =static_cast< int >(lastStrand +
																									beginTemplate + a)/2;
					++b_count;
				}   // otherwise this means that there was not the min
						// number of E residues, so pretend they were Ls
				ssE_exists = false;
				strand_counter = 0;
				// and of course start the new Helix Region
				ssH_exists = true;
				++helix_counter;
				firstHelix = beginTemplate + a;
			}
			else if (ssH_exists) {         //we are in the middle of H region
				if (ssL_exists) {           // perhaps we want to seperate into
					// two different helix regions. Depends on L_break param
					// and if there is a Proline or Gly at the break
					prolineExists(beginTemplate+a,p_exists);
					if ((loop_counter >= L_break) && p_exists) {
						// we need a new helix region, connect the last one
						connectRegion(firstHelix,helix_counter,lastHelix,
													linkCounter,Template,connectableResidues,middle_width);
						connected = true;
						breakpoint(b_count) = lastHelix + 1
							+ static_cast< int >(loop_counter/2);
						++b_count;
						//set parameters to start NEW helix region
						helix_counter = 0;
						firstHelix = beginTemplate + a;
					}
					else { // otherwise that means there were not enough
								 // Ls between the H residues to break the ss into 2
								 // different regions, so just pretend it is an H
						helix_counter += loop_counter;
						//count each L that was considered an H in the helix_counter
					}
				}
				++helix_counter;
			}
			else {
				ssH_exists = true;
				++helix_counter;
				firstHelix = beginTemplate + a;
			}
			ssL_exists = false;
			loop_counter = 0;
		}
		else if (secstruct(beginTemplate+a) == 'E') {
			if (ssH_exists) {   // the preceding region was helices
				// we need to do all the connections for that H region
				if (helix_counter >= min_helix) {   // it is linkable
					if (! ssL_exists) {    // then there was no L
						// btwn the H and the E: ie HHHHE so we need to note
						// that the lastHelix of H was the preceding residue
						lastHelix = beginTemplate + a - 1;      // the last H
					}
					connectRegion(firstHelix,helix_counter,lastHelix,linkCounter,
												Template,connectableResidues, middle_width);
					connected = true;
					breakpoint(b_count) = static_cast< int >(lastHelix +
																									 beginTemplate + a)/2;
					++b_count;
				}  // otherwise this means that there was not the
				// minimum number of Hs, so pretend they were Ls
				ssH_exists = false;
				helix_counter = 0;
				// and of course start the new Strand Region
				ssE_exists = true;
				++strand_counter;
				firstStrand = beginTemplate + a;
			}
			else if (ssE_exists) { //we are in the middle of E region
				if (ssL_exists) { // perhaps we want to seperate into
					// two different Strand regions. Depends on L_break
					if (loop_counter >= L_break) {
						// we need a new strand region, connect the last one
						connectRegion(firstStrand,strand_counter,lastStrand,
													linkCounter,Template,connectableResidues,middle_width);
						connected = true;
						breakpoint(b_count) = lastStrand + 1
							+ static_cast< int >(loop_counter/2);
						++b_count;
						// set parameters to start NEW strand region
						strand_counter = 0;
						firstStrand = beginTemplate + a;
					}
					else { // otherwise that means there were not enough
								 // Ls between the E residues to break the ss into 2
								 // different regions, so just pretend it is an E
						strand_counter = strand_counter + loop_counter;
						// count each L that was considered an E in the strand_counter
					}
				}
				++strand_counter;
			}
			else {
				ssE_exists = true;
				++strand_counter;
				firstStrand = beginTemplate + a;
			}
			ssL_exists = false;
			loop_counter = 0;
		}
		else if (secstruct(beginTemplate+a) == 'L') {
			if (ssE_exists && (! ssL_exists)) {
				// previous residue was an E
				if (strand_counter < min_strand) {
					// this means there was not enough Es, just ignore
					ssE_exists = false;
					strand_counter = 0;
				}
				else {
					lastStrand = beginTemplate + a - 1;
				}
			}
			if (ssH_exists && (! ssL_exists)) {
				//previous residue was an H
				if (helix_counter < min_helix) {
					// this means there was not enough Hs, just ignore
					ssH_exists = false;
					helix_counter = 0;
				}
				else {
					lastHelix = beginTemplate + a - 1;
				}
			}
			ssL_exists = true;
			++loop_counter;
		}
		else {
			std::cerr << "we got a sec struct other than H,E,or L: " <<
				secstruct(beginTemplate+a) << '\n';
		}
	}

	//Still might need to connect the last Secondary Structure Region

	if (! ssL_exists) {    // just ended on a secondary structure
		// so we just need to connect that region up
		if (ssH_exists) {       // the last residue was an H
			if (helix_counter >= min_helix) { // it is linkable
				lastHelix = endTemplate;
				connectRegion(firstHelix,helix_counter,lastHelix,linkCounter,
											Template,connectableResidues, middle_width);
				connected = true;
			}
		}               // then it was not linkable, so ignore
		else if (ssE_exists) {   // the last residue was an E
			if (strand_counter >= min_strand) { // it is linkable
				lastStrand = endTemplate;
				connectRegion(firstStrand,strand_counter,lastStrand,
											linkCounter,Template,connectableResidues, middle_width);
				connected = true;
			}               // then it was not linkable, so ignore
		}
	}
	else {
		// we already have the last residue correctly
		// so we just need to connect that region up
		if (ssH_exists) {       // the last residue was an H
			if (helix_counter >= min_helix) { // it is linkable
				connectRegion(firstHelix,helix_counter,lastHelix,
											linkCounter,Template,connectableResidues, middle_width);
				connected = true;
			}               // then it was not linkable, so ignore
		}
		else if (ssE_exists) {   // the last residue was an E
			if (strand_counter >= min_strand) { // it is linkable
				connectRegion(firstStrand,strand_counter,lastStrand,
											linkCounter,Template,connectableResidues, middle_width);
				connected = true;
			}               // then it was not linkable, so ignore
		}
	}

	//Need to check if there were no Helix or Strands in this Template
	//Region (or if they were not long enough to be considered sec structs)
	//If this is the case, then we just use middle residues of the Region.

	if (! connected) {    // nothing was connected, use middle res
		if (length <= middle_width) {
			// there are no middle residues, so all can be connected
			for ( int a = 0; a < length; ++a ) {
				connectableResidues(linkCounter)= beginTemplate + a;
				Template(beginTemplate + a) = beginTemplate;
				++linkCounter;
			}
		}
		else {   // then pick the middle residues defined by middle_width
			first = 0;
			findCenter(beginTemplate,endTemplate,middle_width,first);
			for ( int a = 0; a < middle_width; ++a ) {
				connectableResidues(linkCounter) = first + a;
				Template(first + a) = first;
				++linkCounter;
			}
		}
		if (! ssL_exists) {
			std::cerr << "This Template Region had neither H,L, nor E" <<
				"as secondary structure elements" <<
				"This is Region between" << beginTemplate << " and " <<
				endTemplate << " only middle" << middle_width <<
				" residues were considered linkable" << "\n";
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin connectRegion
///
/// @brief connects the middle residues on the current region
///
/// @detailed Given a starting point, endpoint, and length of residues
///          (all of the same secondary structure) it makes the center
///          residues connectable and puts them in the linkables array.
///
/// @param[in] first - the first residue of the Region
/// @param[in] numOf - the number of residues in the Region
/// @param[in] last - the last residue of the Region
/// @param[in] linkCounter - the number of residues that have been connected
/// @param[in,out] Template - the Template Regions array
/// @param[in,out] linkables - usually named: connectableResidues
/// @param[in] center_width - the parameter middle_width
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////////

void
connectRegion(
	int & first,
	int & numOf,
	int & last,
	int & linkCounter,
	FArray1DB_int & Template,
	FArray1DB_int & linkables,
	int const center_width
)
{
	using namespace misc;

	int center_start;

	if (first > last) {          //something is wrong
		std::cerr << "End of Region is less than Start of it!" << "\n";
	}

	if (numOf <= center_width) {
		// there are no middle residues, so all can be connected
		for ( int b = 0; b < numOf; ++b ) {
			linkables(linkCounter) = first + b;
			Template(first + b) = first;
			++linkCounter;
		}
	}
	else {  // find the middle residues and connect only those
		findCenter(first,last,center_width,center_start);
		for ( int b = 0; b < center_width; ++b ) {
			linkables(linkCounter) = center_start + b;
			Template(center_start + b) = center_start;
			++linkCounter;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin findCenter
///
/// @brief finds the center of the region, returns the first connectable residue
///
/// @detailed Given the beginning and end of a region, as well as the width
///          of the center (how many connectable residues there are)... this
///          returns the first connectable residue (then center_width
///          determines how many more to connect)
///
/// @param[in] begin - the first residue of the Region
/// @param[in] ending - the last residue of the Region
/// @param[in] length_of_middle - the parameter middle_width
/// @param[out] startPoint - the first connectable residue in the current region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////////
void
findCenter(
	int const begin,
	int const ending,
	int const length_of_middle,
	int & startPoint
)
{
	//determine whether the length of the region is odd or even

	int mid_counter;
	int sum = begin + ending;

	bool region_even = false;
	if (! integer_is_even(sum)) {
		region_even = true;
	}

	if (! region_even) {
		startPoint = (begin + ending)/2;
		mid_counter = length_of_middle - 1;   //the startPoint is counted
		while (mid_counter > 0) {
			startPoint = startPoint - 1;   //make it the one before the current
			mid_counter = mid_counter - 2;  //decrease by 2 (the one b4 and next)
		}
	}

	if (region_even) {
		startPoint = static_cast< int> (begin + ending)/2;
		mid_counter = length_of_middle - 2;   //we already have 2 denoted
		while (mid_counter > 0) {
			startPoint = startPoint - 1;      //go back one residue
			mid_counter = mid_counter - 2;    //decrease for both residues
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin even_integer
///
/// @brief
///
/// @detailed
///bs returns true if an input integer is even (or 0), false if odd
///bs gives nonsense results if the input is a real number!!!
///
/// @param[in] - num -  must be integer
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bs
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////////
bool
integer_is_even( int & num )
{
	bool integer_is_even = false;
	float testnum = num/2.0 - static_cast< int >(num/2.0);
	if ( testnum == 0.0 ) integer_is_even = true;
	return integer_is_even;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin findCenter
///
/// @brief Checks to see if residue (or residues on either side) are P or G
///
/// @detailed Returns TRUE if either residues are Pro or Gly.
///          ie if residue 56 was an L within a helix region: HHHHLHHHHH, then
///          this returns TRUE if either of residues 55,56,or 57 are Pro or Gly
///          This way, we only divide a Template Helix Region that contains Ls
///          within it into TWO Template Helix Regions if the break is caused
///          by a Proline or Glycine
///
/// @param[in] current_res - the current residue
/// @param[out] exists - true or false return value
///
/// @return true if current_res or residues on either side are Pro or Gly
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors firas 6/3/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////////
void
prolineExists(
	int const current_res,
	bool & exists
)
{
	using namespace param;
	using namespace param_aa;
	using namespace misc;
	using namespace aaproperties_pack;

	exists = false;
	for ( int i = (current_res-2); i <= current_res; ++i ) {
		if ((res(i) == aa_pro) || (res(i) == aa_gly)) {
			exists = true;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hpsort_array
///
/// @brief sorts a 'size by n' array 'ra' on 'key'
///
/// @detailed car heapsort algorithm from numerical recipes
///          car modified here from hpsort to use multidimensional ra
///
/// @param[in] n - the dimension of the ra matrix
/// @param[in,out] ra - distances array
/// @param[in] size - the second dimension of the ra matrix
/// @param[in] key - what to sort on
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car, firas 6/3/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hpsort_array(
	int const n,
	FArray2DB_int & ra,
	int const size,
	int const key
)
{

//car heapsort algorithm from numerical recipes
//car modified here from hpsort to use multidimensional ra

	int i,ir,j,l,k;
	FArray1D_int rra( size );

	if ( n < 2 ) return;
	l = n/2+1;
	ir = n;
L10:
	if ( l > 1 ) {
		--l;
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,l);
		}
	} else {
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,ir);
		}
		for ( k = 1; k <= size; ++k ) {
			ra(k,ir) = ra(k,1);
		}
		--ir;
		if ( ir == 1 ) {
			for ( k = 1; k <= size; ++k ) {
	ra(k,1) = rra(k);
			}
			return;
		}
	}
	i = l;
	j = l+l;
L20:
	if ( j <= ir ) {
		if ( j < ir ) {
			if ( ra(key,j) < ra(key,j+1) ) ++j;
		}
		if ( rra(key) < ra(key,j) ) {
			for ( k = 1; k <= size; ++k ) {
				ra(k,i) = ra(k,j);
			}
			i = j;
			j += j;
		} else {
			j = ir+1;
		}
		goto L20;
	}
	for ( k = 1; k <= size; ++k ) {
		ra(k,i) = rra(k);
	}
	goto L10;
}
