// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15298 $
//  $Date: 2007-06-03 04:16:42 -0400 (Sun, 03 Jun 2007) $
//  $Author: johnk $


// Rosetta Headers
#include "crankshaft.h"
#include "after_opts.h"
#include "chuck.h"
#include "flexibility.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "loops.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "param.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "runlevel.h"
#include "status.h"
#include "wobble.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>


//db  regions of the sequence in which the fragments are highly variable are
//db  identified using Jack's fragment_diversity function.
//db  local maxima in the fragment variability (entropy)
//db  are identified and pairs of these maxima, or in some cases a single
//db  maxima and a loop position are selected by choose_crankshaft.
//db  crankshaft_move then finds a fragment that causes a not too large
//db  increase in msd for the first region,
//db  wobbles nwobble1 residues following the fragment,
//db  and then wobbles nwobble2 residues in the 2nd region to further
//db  reduce the msd.

//db  in the tests I did, nwobble1 was set to 1 and nwobble2 was set to
//db  2 or 3.

//db  the functions main_trial_crank and main_trial_crank_min are exaxtly
//db  analogous to their wobble counterparts in torsion_bbmove_trials and
//db  torsion_bbmove_trials_min.

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_crankshaft
///
/// @brief
///db identifies ranges for the starting points for two moves that
///db bracket one or two relatively consistent regions in the fragment
///db libraries. ranges are returned rather than specific residues to
///db avoid repeated attempts of exactly the same move.
///
/// @detailed
///
/// @param  minfrag_begin1 - [in/out]? -
/// @param  maxfrag_begin1 - [in/out]? -
/// @param  minfrag_begin2 - [in/out]? -
/// @param  maxfrag_begin2 - [in/out]? -
/// @param  size - [in/out]? - size of frag to insert
/// @param  delta - [in/out]? - spacing/size of each region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_crankshaft(
	int & minfrag_begin1,
	int & maxfrag_begin1,
	int & minfrag_begin2,
	int & maxfrag_begin2,
	int size, // size of frag to insert
	int & delta // spacing/size of each region
)
{
	using namespace flexibility;
	using namespace misc;

//car local
	int first_try,site1,site2;
	int site1L,site2L,loop1;
	int temp1,temp2;

L30:
	first_try  = static_cast< int >(ran3()*(total_residue-30))+15;
	if ( first_try <= local_maxima(1) || first_try >=local_maxima(num_local_maxima) ) goto L30;

	find_closest_maxima(first_try,site1,site2);
//db     if chosen residue is in helix or strand, pivot points will be
//db     flanking local maxima of the fragment variability function

//db     if chosen residue is in loop, pivot points will be this residue
//db     and either the local maxima before or after this residue

//db     sometimes also make pivot points the flanking maxima (allows
//db     moves in contiguous L segments)

	if ( secstruct(first_try) == 'L' ) {
		if ( ran3() < 0.75 ) {
			if ( std::abs(first_try - site1) > std::abs(first_try-site2) ) {
				site2 = first_try;
			} else {
				site1 = first_try;
			}
		}
	} else {

//db     half the time change a pivot point to the closest loop
//db     residue
		if ( ran3() > 0.5 ) {
			find_closest_loop(first_try,loop1);
			if ( first_try > loop1 ) {
				site1 = loop1;
			} else {
				site2 = loop1;
			}
		}

	}


//db  make sure sites are currently 'L' positions
//car remove this cause it causes problems on rare occassions
//      find_closest_loop(site1,site1L)
//      find_closest_loop(site2,site2L)

	site1L = site1;
	site2L = site2;
	if ( std::abs(site1 - site2) < 6 ) {
//		std::cout << "not worth it " << site1 << ' ' << site1L << ' ' <<
//		 site2 << ' ' << site2L << std::endl;
		goto L30;
	}
//db  no point in making a crankshaft move if constant region is very small

	minfrag_begin1 = site1L-delta-size/2;
	maxfrag_begin1 = site1L+delta-size/2;

	minfrag_begin2 = site2L-delta-size/2;
	maxfrag_begin2 = site2L+delta-size/2;

	if ( minfrag_begin2 -  maxfrag_begin1  < 2 ) {
		++minfrag_begin2;
		--maxfrag_begin1;
	}

	if ( minfrag_begin1 < 10 ) goto L30;
	if ( minfrag_begin2 > (total_residue-10) ) goto L30;

//db  switch order so that fragment is added at second region sometimes

	if ( ran3() > 0.5 ) {
		temp1 = minfrag_begin1;
		temp2 = maxfrag_begin1;
		minfrag_begin1 = minfrag_begin2;
		maxfrag_begin1 = maxfrag_begin2;
		minfrag_begin2 = temp1;
		maxfrag_begin2 = temp2;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_closest_maxima
///
/// @brief
///
/// @detailed
///
/// @param  first_try - [in/out]? -
/// @param  maxima1 - [in/out]? -
/// @param  maxima2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_closest_maxima(
	int & first_try,
	int & maxima1,
	int & maxima2
)
{
	using namespace flexibility;

//car local
	int i;

	for ( i = 1; i <= num_local_maxima; ++i ) {
		if ( local_maxima(i) > first_try ) goto L40;
	}
L40:

	maxima1 = local_maxima(i-1);
	maxima2 = local_maxima(i);

	if ( (maxima2-maxima1) < 5 ) {
		if ( i <= 2 ) {
			maxima2 = local_maxima(i+1);
			goto L55;
		}

		if ( i >= num_local_maxima-1 ) {
			maxima1 = local_maxima(i-2);
			goto L55;
		}

		if ( ran3() > 0.5 ) {
			maxima1 = local_maxima(i-2);
		} else {
			maxima2 = local_maxima(i+1);
		}
	}

//db     small fraction of time choose once removed pivot point
//db     to allow for moves of two consecutive ss elements

	if ( ( ran3() > 0.8 ) && ( i > 2 ) ) maxima1 = local_maxima(i-2);

	if ( ( ran3() > 0.8 ) && ( i < num_local_maxima ) ) maxima2 = local_maxima(i+1);

L55:;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_closest_loop
///
/// @brief
///
/// @detailed
///
/// @param  first_try - [in/out]? -
/// @param  loop - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_closest_loop(
	int & first_try,
	int & loop
)
{
	using namespace misc;

	for ( int i = 0; i <= total_residue; ++i ) {
		if ( (first_try + i) < total_residue ) {
			if ( secstruct(first_try + i) == 'L' ) {
				loop = first_try+i;
//				std::cout << first_try << ' ' << secstruct(first_try) << ' ' << i << std::endl;
				return;
			}
		}

		if ( (first_try - i) > 1 ) {
			if ( secstruct(first_try - i) == 'L' ) {
				loop = first_try-i-1;
//				std::cout << first_try << ' ' << secstruct(first_try) << ' ' << i << std::endl;
				return;
			}
		}
	}

//db  this only happens if there is no loop in the secstruct array,
//db  ie almost never
	loop = first_try;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fragment_diversity
///
/// @brief
///
/// @detailed
///js   This function calculates the diversity of phi-psi angles
///js   for each residue based on the 3mer fragment set.  The bins
///js   are 10x10 degree squares in phi-psi space.  The measure of
///js   diversity is the "entropy" of this distribution:
///js   entropy = sum over all bins of p log (p),
///js   where p is the probability of that residues being in that bin.
///js   the results are stored in the array entropy(), which itself is
///js   declared in flexibility.h
///
///js   entropy(i) is measure of the number of conformations avaliable to
///js   residue i.  It can be thought of as the log of the number of
///js   conformations.
///js   It is the entropy of the distribution of phi-psi angles.  It will depend
///js   on the binwidth.
///js   If we want to attempt to move residues with a frequency proportional to
///js   their flexibility, we might be interested in
///js      exp(entropy(i)/(sum_i exp(entropy(i)).
///js   This would be a sort of measure of the relative number of
///js   conformations available to each residue.  This quantity is stored in
///js   flexibility_map()
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fragment_diversity()
{
	using namespace flexibility;
	using namespace fragments;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

	int first,last;
	FArray2D_int phipsi_distribution( 36, 36, 0 );
	FArray1D_float entropy( MAX_RES()() );
	int entries;
	float p;
	int offset;

	int const width = 10, minval = -180; // used for binning up dihedral angles
	int const size = 3;

	int const size_bin = get_index_by_frag_size(size);

	std::cout << "calculating fragment_diversity..." << std::endl;
	for ( int i = 1; i <= total_residue; ++i ) { // residue of interest
		entries = 0;
		first = std::max(1,i-size+1);
		last = std::min(i,total_residue-size+1);
		entropy(i) = 0;
		for ( int frag_begin = first; frag_begin <= last; ++frag_begin ) {
		 // loop over frags  that contain i
			offset = i-frag_begin; // position of i in fragment
			for ( int neighbor = 1, max_neigh = MAX_NEIGH(); neighbor <= max_neigh; ++neighbor ) {
			 // loop over each fragment
				int div_bin_phi = dihedral_bin(width,minval,
				align_phi(frag_begin,neighbor,offset,size_bin));
				int div_bin_psi = dihedral_bin(width,minval,
				align_psi(frag_begin,neighbor,offset,size_bin));
				if ( false ) {
					std::cout << frag_begin << ' ' << neighbor << ' ' << offset << ' ' <<
					 size_bin << ' ' << entries << std::endl;
					std::cout << i << ' ' << neighbor << ' ' << offset << ' ' <<
					 frag_begin << ' ' << div_bin_phi << ' ' << div_bin_psi << std::endl;
					std::cout << i <<
					 align_phi(frag_begin,neighbor,offset,size_bin) <<
					 align_psi(frag_begin,neighbor,offset,size_bin) << std::endl;
				}
				++phipsi_distribution(div_bin_phi,div_bin_psi);
				++entries;
			}
		}
		for ( int div_bin_phi = 1; div_bin_phi <= 36; ++div_bin_phi ) {
			for ( int div_bin_psi = 1; div_bin_psi <= 36; ++div_bin_psi ) {
				p = float(phipsi_distribution(div_bin_phi,div_bin_psi))/float(entries);
				if ( p != 0 ) {
					entropy(i) -= p * std::log(p);
					phipsi_distribution(div_bin_phi,div_bin_psi) = 0;
				}
			}
		}
	}

	float total = 0.0;
//js debugging hack!!!!!!!!
	entropy(1) = 5; // problem because has no phi...
//js debugging hack!!!!!!!!

	for ( int i = 1; i <= total_residue; ++i ) {
		total += std::exp(entropy(i));
		if ( runlevel >= inform) std::cout << i << SS( entropy(i) ) << std::endl;
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		flexibility_map(i) = std::exp(entropy(i))/total;
	}


	for ( int j = 3; j <= total_residue-2; ++j ) {

		insert_probability(j) = ( 0.5*entropy(j-2) + entropy(j-1) + entropy(j) +
		 entropy(j+1) + 0.5*entropy(j+2) )/4.;

	}
	assert( ( total_residue >= 2 ) );
	insert_probability(1) = (entropy(1)+entropy(2))/2.;
	insert_probability(2) = (entropy(1)+entropy(2)+entropy(3))/3.;
	insert_probability(total_residue) = (entropy(total_residue-1)+
	 entropy(total_residue))/2.;
	insert_probability(total_residue-1) = (entropy(total_residue-2)+
	 entropy(total_residue-1)+entropy(total_residue))/3.;


//db     identify local maxima

	num_local_maxima = 0;
	for ( int j = 4; j <= total_residue - 4; ++j ) {

//car require maxima to be >4 residues apart:
		if ( num_local_maxima > 0 ) {
			if ( j-local_maxima(num_local_maxima) <= 4 ) goto L66;
		}
		if ( ( insert_probability(j) >
		 ((insert_probability(j-1)+insert_probability(j-2))/2. + 0.01) ) &&
		 ( insert_probability(j) >
		 ((insert_probability(j+1)+insert_probability(j+2))/2. + 0.01) ) ) {
			++num_local_maxima;
			local_maxima(num_local_maxima) = j;
			loc_max.push_back( j );
		}
L66:;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dihedral_bin
///
/// @brief
///
/// @detailed
///
/// @param  width - [in/out]? -
/// @param  minimum - [in/out]? -
/// @param  angle - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
dihedral_bin(
	int const width,
	int const minimum,
	float const angle
)
{
	float const int_angle = angle - minimum;
	int dihedral_bin = static_cast< int >( int_angle / width );
	dihedral_bin = 1 + mod( dihedral_bin, 36 );
	if ( dihedral_bin < 1 ) {
		std::cout << "dihedral_bin: unusual angle" << std::endl;
		dihedral_bin = 36;
	}

	return dihedral_bin;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin crankshaft_move
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  nwobble1 - [in/out]? -
/// @param  nwobble2 - [in/out]? -
/// @param  crank_begin - [in/out]? -
/// @param  crank_end - [in/out]? -
/// @param  wobble_cost - [in/out]? -
/// @param  bump_score - [in/out]? -
/// @param  gfrag - [in/out]? - if true, powell minimized sucessfully
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
crankshaft_move(
	int size,
	float cutoff_max,
	int nwobble1,
	int nwobble2,
	int & crank_begin,
	int & crank_end,
	float & wobble_cost,
	float & bump_score,
	bool & gfrag // if true, crankshaft_move was successful
)
{
	using namespace param;

//car this should pass out crank_begin and crank_end !!!!! CAROL READ THIS

//db  call choose_crankshaft to find two regions of high fragment variability
//db  one of which is sometimes replaced by a loop region

//db put a chuck frag in the first region, wobble nwobble residues on the
//db short side, and then wobble 3 residues in the second region

//car update global position array as well as phi,psi,omega

//car 1) pick a random position and random chuck fragment passing the cutoff,
//car 2) if wobble fails, try the next frag in the list,
//car 3) if all frags in the list tried, try the adjacent position
//car 4) if no fragments found within +/- 5 of the original random position,
//car     choose another random position

//car wobble cutoff set to 10.0; 3-10% acceptance of moves with a wobble
//car cost of 9-10

//car recommended chuck cutoff (cutoff_max)=60;
//car ~65% of moves use an actual cutoff of 60/4=15

// local
	int size_bin;

	int ntries_begin; // number of insertion points tried relative to
	 // init_frag_begin
	int ntries_frag; // number of frags in chuck list tried
	int nfail; // total number of insertion points tried
	int c_counter,w_counter; // total number of chuck and wobble tries
	int b_counter; // number of frags bump checked
	int min_frag_begin1,min_frag_begin2;
	int max_frag_begin1,max_frag_begin2;
	int frag_begin,frag_end; // site frag insertion in region 1
	int init_frag_begin;

	FArray1D_float cut_cost( MAX_NEIGH()() ); // cost of frags in chuck list
	FArray1D_int cut_frag( MAX_NEIGH()() ); // list of frags that pass chuck cutoff
	int cut_count; // length of chuck list
	int chuck_frag; // pointer into list of chuck fragments
	int best_frag; // pointer into neighbor list, best chuck frag
	float chuck_cost; // cost of selected chuck frag (ie best_frag)
	float chuck_cutoff; // local copy of chuck cutoff, gets set to
	 // cutoff actually used
	float bump_score_start;
	float wobble_cutoff;
	float rama_score,first_wobble_cost;
	int wobble_begin,wobble_end; // site of 2nd wobble (ie in region2)
	int first_wobble_begin; // site of first wobble
	int fold_begin,fold_end; // regions folded by add_fast_wobble
	int dir; // refold direction
	int max_sites(0);
	bool yes_no;

//db  removed loop option for now; need to make sure that two crank regions
//db  are within loop
	if ( get_loop_flag() ) {
		std::cout << "no crank moves in loop mode" << std::endl;
		return;
	}

	w_counter = 0;
	c_counter = 0;
	b_counter = 0;
	size_bin = get_index_by_frag_size(size);
	wobble_cutoff = 10.0;
	gfrag = true;
	ntries_begin = 0;
	nfail = 0;
	int max_fail = nfail;  // later we will guestimate how many sites are possible

	int const factor = { 3 }; //choose_double will estimate how many sites can be inserted into; we will increase the allowed number of attempts by this factor

L171:; // return here if chuck fails, try next insertion point
	refold_set_direction(0);
	if (nfail > max_fail) {
// 		std::cout << "nfail " << nfail << " maxfail " << max_fail << std::endl;
		gfrag = false;
		return;
	}
	++nfail;

	chuck_cutoff = cutoff_max; // local copy, cause may change
	ntries_frag = 0;
	++ntries_begin;

//db  choose pivot points for crankshaft move
//      choose_crankshaft(min_frag_begin1,max_frag_begin1,
//       min_frag_begin2,max_frag_begin2,size,1);
// size of first frag insert must include wobble

	if( use_new_crankshaft() ) {
	yes_no = choose_crankshaft_trial(max_sites, min_frag_begin1,max_frag_begin1,min_frag_begin2,max_frag_begin2,size+nwobble1,nwobble2,1,4,40);
	if( !yes_no ){
		min_frag_begin1 = -1;
		max_frag_begin1 = -1;
		min_frag_begin2 = -1;
		max_frag_begin2 = -1;
	}
	max_fail = factor * max_sites;
	} else {
	max_fail = factor * choose_double(min_frag_begin1,max_frag_begin1,
	 min_frag_begin2,max_frag_begin2,size+nwobble1,nwobble2,1,4,40);
	}

//car catch case where not enough space for a crankshaft move
//car this is a flagged case returned by choose_double
	if ( min_frag_begin1 < 1 ) {
		gfrag = false;
		return;
	}

//car choose a place to wobble three residues in region 2:
//db in later versions, may want to do a fragment insertion between
//db min_frag_begin2 and max_frag_begin2

	wobble_begin = min_frag_begin2 +
	 static_cast< int >((max_frag_begin2-min_frag_begin2-1)*ran3());
	wobble_end = wobble_begin + nwobble2 - 1;

//car choose a frag insertion in region 1:
	find_chuck_moves(size,chuck_cutoff,min_frag_begin1,max_frag_begin1,cut_cost,
	 cut_frag,cut_count,frag_begin);
	frag_end = frag_begin + size - 1;
	++c_counter;
//	std::cout << "chuck_moves " << frag_begin << ' ' << min_frag_begin1 << ' ' <<
//	 max_frag_begin1 << ' ' << chuck_cutoff << ' ' << cut_count << ' ' <<
//	 cut_cost(1) << ' ' << ntries_begin << std::endl;

	if ( ntries_begin == 1 ) init_frag_begin = frag_begin;
	if ( ntries_begin == 11 ) ntries_begin = 0; // pick randomly next time
	if ( cut_count == 0 ) goto L171; // no frags at this position, try next

	chuck_frag = static_cast< int >(cut_count*(ran3())+1);
	 // pick random point in list
	chuck_cost = cut_cost(chuck_frag);
	best_frag = cut_frag(chuck_frag);

L180:; // return here to try next frag in list
//	std::cout << "chuck " <<
//	 frag_begin << ' ' << best_frag << ' ' << chuck_cost << ' ' <<
//	 chuck_cutoff << std::endl;
	++ntries_frag;

//car BUMP SCREEN
//car define entire crank region to set up the bump screen
	if ( wobble_begin > frag_begin ) {
		crank_end = wobble_end;
		crank_begin = frag_begin;
	} else {
		crank_begin = wobble_begin;
		crank_end = frag_end;
	}
	screen_move_best(crank_begin,crank_end,bump_score_start);
	insert_frag(size,size_bin,frag_begin,best_frag);
	refold(frag_begin,frag_end);
	refold_get_dir(dir);

	screen_move_trial(frag_begin,frag_end,bump_score);


//car FIRST WOBBLE
// 	std::cout << " befwo " << I( 3, frag_begin ) << ' ' <<
// 	 I( 3, best_frag ) << F( 9, 3, chuck_cost ) << F( 9, 3, chuck_cutoff ) <<
// 	 F( 9, 3, bump_score ) << std::endl;
	add_fast_wobble(frag_begin,size,nwobble1,0,rama_score,first_wobble_cost,gfrag,
									fold_begin,fold_end, false /*allow_use_ccd*/);
	if (fold_begin < 1) goto L300; //warning sign!
	if ( !gfrag ) return;

//   if ( !gfrag ) std::cout << "gfrag tripped" << std::endl;
//   std::cout << "firstwobble" << SS( chuck_cost ) << SS( wobble_cost ) <<
//	  ' ' << dir << ' ' << frag_begin << ' ' << frag_end << std::endl;
//    if ( first_wobble_cost > 2*chuck_cost ) {
//       std::cout << "wblfail" << I( 5, frag_begin ) <<
//  		 F( 9, 3, phi(frag_begin) ) <<
//  		 F( 9, 3, align_phi(frag_begin,best_frag,0,size_bin) ) <<
//  		 F( 9, 3, phi(frag_end+1) ) << F( 9, 3, best_phi(frag_end+1) ) << std::endl;
//    }
	if ( first_wobble_cost > chuck_cutoff ) goto L300;

	if ( dir == 1 ) {
		first_wobble_begin = frag_end+1;
	} else {
		first_wobble_begin = frag_begin - nwobble1;
	}


//car BUMP SCREEN
	if ( wobble_begin > frag_begin ) {
		crank_end = wobble_end;
		crank_begin = std::min(frag_begin,first_wobble_begin);
		refold_set_direction(1);
	} else {
		crank_begin = wobble_begin;
		crank_end = std::max(frag_end,first_wobble_begin+nwobble1-1);
		refold_set_direction(-1);
	}


	refold(crank_begin,crank_end);
	screen_move_trial(crank_begin,crank_end,bump_score);
// 	std::cout << " chuck " << I( 3, frag_begin ) << ' ' <<
// 	 I( 3, best_frag ) << F( 9, 3, chuck_cost ) <<
// 	 F( 9, 3, first_wobble_cost ) << F( 9, 3, bump_score ) <<
// 	 F( 9, 3, bump_score_start ) << std::endl;

	++b_counter;
	if ( bump_score - bump_score_start > 10 ) goto L300; // try another frag

//car SECOND WOBBLE (ie 3 res wobble in region2)
//car second wobble occurs _within_ crank region which includes both
//car regions identified by choose_double
	add_fast_wobble(crank_begin,crank_end-crank_begin+1,nwobble2,-nwobble2,
	 rama_score,wobble_cost,gfrag,fold_begin,fold_end);
	if (fold_begin < 1) goto L300; //warning sign!

	refold_set_direction(0); // restore auto direction before exit
	if ( !gfrag ) return;
	++w_counter;

//       refold(fold_begin,fold_end);
//       refold_get_dir(dir);
//       if ( dir == -1 ) {
//          calc_msd_best(1,first_wobble_begin-1,msd5);
//       } else {
//          calc_msd_best(fold_end+1,total_residue,msd5);
//       }
//       std::cout << "wobble " << I( 3, frag_begin ) << ' ' <<
//        I( 3, best_frag ) << ' ' << I( 3, wobble_begin ) << ' ' <<
// 			 I( 3, fold_begin ) << F( 10, 3, chuck_cost ) <<
//        F( 10, 3, first_wobble_cost ) << F( 10, 3, wobble_cost ) <<
// 				//			 F( 10, 3, msd5 ) << std::endl;
	if ( wobble_cost > wobble_cutoff ) goto L300; // try a different frag

//	std::cout << " final " << I( 3, frag_begin ) << ' ' <<
// 	 I( 3, best_frag ) << ' ' << I( 3, c_counter ) <<
// 	 F( 7, 3, chuck_cost ) <<
// 	 F( 7, 3, first_wobble_cost ) << I( 3, w_counter ) <<
// 	 F( 7, 3, wobble_cost ) << F( 7, 1, wobble_cutoff ) <<
// 		I( 3, b_counter ) << F( 6, 2, bump_score ) << " " << nfail << std::endl;


//car set return values:
	crank_begin = fold_begin;
	crank_end = fold_end;
	refold_set_direction(0); // make sure its auto direction before exit
	return;

//car jump here to abort move and decide to try either another frag at
//car the same position or a new position
L300:
	resetphipsi();
	refold_set_direction(0);
	if ( ntries_frag < std::min(5,cut_count) ) {
		++chuck_frag;
		if ( chuck_frag > cut_count ) chuck_frag = 1;
		chuck_cost = cut_cost(chuck_frag);
		best_frag = cut_frag(chuck_frag);
		goto L180; // try another frag at same position
	} else {
		goto L171; // try a new insertion position
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_crank_trial
///
/// @brief
///db   analogous to main_wobble_trial.  chuck move followed by wobble
///     (powell minimization of phi/psi) to minimize change in MSD of
///      the position array
///
/// @detailed
///
/// @param  size - [in/out]? - fragment size
/// @param  cutoff_max - [in/out]? - chuck cutoff, if <0  chuck sets cutoff
///        automatic (rec value: 60)
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? - cycle_number to send to mc
///        (>100000 enables convergence check)
/// @param  nwobble1 - [in/out]? - number of residues to wobble after db first insertion
/// @param  nwobble2 - [in/out]? - number of residues to be wobbled in second segment
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_crank_trial(
	int size,
	float cutoff_max,
	Scoring_Function score_fxn,
	int & cycle_number,
	int nwobble1,
	int nwobble2
)
{
	int crank_begin,crank_end;
	float wobble_cost,bump_score;
	int total_insert;
	bool gfrag( true );

	crankshaft_move(size,cutoff_max,nwobble1,nwobble2,crank_begin,crank_end,
	 wobble_cost,bump_score,gfrag);
	if ( !gfrag ) return;
	total_insert = crank_end-crank_begin+1;
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("crank",crank_begin,total_insert);
	monte_carlo(cycle_number);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin main_crank_min_trial
///
/// @brief
///db     analogous to main_wobble_min_trial
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  nwobble1 - [in/out]? -
/// @param  nwobble2 - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_crank_min_trial(
	int size,
	float cutoff_max,
	Scoring_Function score_fxn,
	int & cycle_number,
	int nwobble1,
	int nwobble2,
	std::string const & type
)
{

	std::string const move_type( "crankmin" );

	bool gfrag( true ); // if true, powell minimized sucessfully
	int crank_begin,crank_end;
	float wobble_cost,bump_score;
	int total_insert;

	crankshaft_move(size,cutoff_max,nwobble1,nwobble2,crank_begin,crank_end,
	 wobble_cost,bump_score,gfrag);
	if ( !gfrag ) return;

	total_insert = crank_end-crank_begin+1;

	if (get_minimize_exclude_frag_in_crank()) minimize_exclude_frag( crank_begin, total_insert );
//      dump_pdb("crank.pdb");
	minimize( type, move_type, score_fxn, crank_begin, crank_end, gfrag );
//      dump_pdb("min.pdb");
//      error_stop( "" );
	if ( !gfrag ) return;

	save_status_info( move_type, crank_begin, total_insert );
	monte_carlo(cycle_number);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_double
///
/// @brief
///car choose a frag insertion site (minfrag_begin1, max_frag_begin1)
///car and a second site where a three res wobble can occur based on ss end,
///car position of 'L' or frag-diversity max
///
/// @detailed
///
/// @param  minfrag_begin1 - [out] -
/// @param  maxfrag_begin1 - [out] -
/// @param  minfrag_begin2 - [out] -
/// @param  maxfrag_begin2 - [out] -
/// @param  size1 - [in] -size of insertion in region1
/// @param  size2 - [in] - size of insertion in region2
/// @param  delta - [in] -size of each region (maxbegin-minbegin+1)
/// @param  min_spacing - [in] -min # of unmodified residues between two modified regions
/// @param  max_spacing - [in] -max # of unmodified residues between two modified regions
///
/// @global_read
///    fragment diversity in flexibility namespace
///
/// @global_write
///
/// @remarks
/// max_spacing not guaranteed--depends on method used to select site2
///
/// returns an estimate of how many insertion points are possible
///
/// @references
///
/// @authors  car 8/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
choose_double(
	int & minfrag_begin1,
	int & maxfrag_begin1,
	int & minfrag_begin2,
	int & maxfrag_begin2,
	int size1, // size of insertion in region 1
	int size2, // size of insertion in region 2
	int delta, // size of each region (maxbegin-minbegin+1) = 2*delta +1
	int min_spacing, // min # of unmodified residues between two modified regions
	int max_spacing  // max # of unmodified residues between two modified regions
									 // max not guaranteed--depends on method used to select site2
)
{
	using namespace flexibility;
	using namespace misc;
	//	using namespace protein_maps;
	using namespace param;

//car parameters
	int const end_buffer = { 15 }; // how close to the end a site can be
	 // min_frag_begin can be end_buffer_min-delta
	 // int const min_spacing = { 6 }; // site2>=site1+min_spacing
								 // min_spacing-2*delta-1  is min residues between regions
	 // int const max_spacing = { 20 }; // used only for random selection


//car local
	int i;
	int site1,site2;  // site is the center of the frag_begin region
	int dir;
	float temp;
	char ss1,ss2;
	int nfail = 0;
	int const MAXFAIL = 100;

//------------------------------------------------------------------------------// dont use protein_maps version directly, for pose compatibility
	FArray1D_bool allow_insert( MAX_RES()() );
	FArray1D_int insert_map( MAX_RES()() );
	int total_insert;

	retrieve_allow_insert(allow_insert,total_residue);
	retrieve_insertmap(insert_map,total_insert);

	int max_sites = total_insert -
		(end_buffer+size1+min_spacing+size2+end_buffer);
// must fit in total_residue:  Nend_buffer.site.min_spacing.Cend_buffer
	if ( max_sites < 1 ) {
//car not enough space for a crank move
		minfrag_begin1 = -1;
		maxfrag_begin1 = -1;

		minfrag_begin2 = -1;
		maxfrag_begin2 = -1;
		return max_sites;
	}

	int min_site1 = end_buffer + delta + 1;
	int min_site2 = end_buffer + delta + 1;
	int max_site1 = total_residue - (end_buffer + size1 + delta); //leave room for frag
	int max_site2 = total_residue - (end_buffer + size2 + delta); //leave room for frag
	int ntry = 0;

L30:
	nfail++;
	if ( nfail > MAXFAIL ) {
		// something went wrong -- maybe protein too small. (length of 44 residues is a problem)
		minfrag_begin1 = -1;
		maxfrag_begin1 = -1;

		minfrag_begin2 = -1;
		maxfrag_begin2 = -1;
		return max_sites;
	}

	site1 = insert_map(static_cast< int >(ran3()*(total_insert-size1+1))+1);
	if ( site1 < min_site1 || site1 > max_site1 ) goto L30;

	int in_reg_secstruct = 1;
	for ( int j = site1 - 2; j <= site1 + 2; ++j ) {
		if ( secstruct(j) == 'L' ) in_reg_secstruct = 0;
	}
	if ( in_reg_secstruct == 1 ) {
		++ntry;
		if ( ntry < 3 ) goto L30;
	}

	minfrag_begin1 = site1 - delta;
	maxfrag_begin1 = site1 + delta;
	assert( minfrag_begin1 > end_buffer );
	assert( maxfrag_begin1 < total_residue - (end_buffer + size1 -1));

//car check that insertion is allowed here
	for ( int j = minfrag_begin1, je = maxfrag_begin1 + size1 - 1; j <= je ; ++j ) {
		if ( ! allow_insert(j) ) goto L30;
	}

	//car pick a direction to look for the second site:
	if ( site1 < 2*end_buffer ) { //close to N-term
		dir = 1; // look C-terminally
	} else if ( site1 > total_residue- (2 * end_buffer + size1) ) { //close to Cterm
		dir = -1; // look N-terminally
	} else if ( ran3() < 0.5 ) {
		dir = 1; // pick randomly
	} else {
		dir = -1;
	}
	int size;
	if ( dir == 1 ) {
		size = size1;  //frag1 is relevant for computing spacing
	} else {
		size = size2;  //frag2 is relevant for computing spacing
	}
	int closest_site2 = site1 + dir * (size + min_spacing + 2*delta);
	if ( closest_site2 > max_site2 || closest_site2 < min_site2) goto L30;
	int farthest_site2 = std::max(min_site2, std::min(max_site2, site1 + dir * (size + max_spacing + 2*delta)));

//car pick a second site:
	temp = ran3();
	if ( temp < 0.5 ) { // pick ss end
		site2 = closest_site2;
		ss1 = secstruct(site2);
		ss2 = ss1;
		while ( ss2 == ss1 && ss2 != 'L' ) {
			site2 += dir;
			if ( site2 == farthest_site2+dir ) goto L30; //out of range
			ss2 = secstruct(site2);
		}

	} else if ( temp < 0.90 ) { // pick local maxima in frag diversity
		//car ensure min_spacing, but not max_spacing!!
		if ( dir == 1 ) {
			for ( i = 1; i <= num_local_maxima; ++i ) {
				if ( local_maxima(i) >= closest_site2 ) goto L40;
			}
		} else {
			for ( i = num_local_maxima; i >= 1; --i ) {
				if ( local_maxima(i) <= closest_site2 ) goto L40;
			}
		}
//car if you get to this point, there is no max between the site1 and
//car the protein endpoint
		goto L30;
L40:
		site2 = local_maxima(i);
		if ( site2 < min_site2 ||
				 site2 > max_site2) goto L30;  //check min&max, not closest/farthest

	} else {                     // pick randomly
		//car pick random separation distance
		int min_site_spacing = std::abs( site1 - closest_site2 ) - 1;
		int max_site_spacing = std::abs( site1 - farthest_site2 ) - 1;
		site2 = site1 + dir *
			( static_cast< int >( ran3() * (max_site_spacing-min_site_spacing) )
				+ min_site_spacing );
	}

	//site2 = site2 + int(ran3()*5) - 2;

	minfrag_begin2 = site2-delta;
	maxfrag_begin2 = site2+delta;
	//car check that insertion is allowed here
	for ( int j = minfrag_begin2, je = maxfrag_begin2 + size2 - 1; j <= je; ++j ) {
		if ( ! allow_insert(j) ) goto L30;
	}

	assert( minfrag_begin2 > end_buffer );
	assert( maxfrag_begin2 < total_residue - (end_buffer + size2 -1));
	assert( std::max( (minfrag_begin2 - maxfrag_begin1),
							 (minfrag_begin1 - maxfrag_begin2) ) + 1
					>  min_spacing + size - 1 );
	return max_sites;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin choose_crankshaft_trial
///
/// @brief
///db identifies ranges for the starting points for two moves that
///db bracket one or two relatively consistent regions in the fragment
///db libraries. ranges are returned rather than specific residues to
///db avoid repeated attempts of exactly the same move.
///
/// @detailed
///
/// @param  minfrag_begin1 - [in/out]? -
/// @param  maxfrag_begin1 - [in/out]? -
/// @param  minfrag_begin2 - [in/out]? -
/// @param  maxfrag_begin2 - [in/out]? -
/// @param  size - [in/out]? - size of frag to insert
/// @param  delta - [in/out]? - spacing/size of each region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 23rd 2006
/////////////////////////////////////////////////////////////////////////////////
bool
choose_crankshaft_trial(
	int & maxsites,
	int & minfrag_begin1,
	int & maxfrag_begin1,
	int & minfrag_begin2,
	int & maxfrag_begin2,
	int size1, // size of insertion in region 1
	int size2, // size of insertion in region 2
	int delta, // size of each region (maxbegin-minbegin+1) = 2*delta +1
	int min_spacing, // min # of unmodified residues between two modified regions
	int max_spacing  // max # of unmodified residues between two modified regions
									 // max not guaranteed--depends on method used to select site2
)
{
	using namespace flexibility;
	using namespace misc;
	using namespace protein_maps;

	int const end_buffer( 15 );
	int const MAX_FAIL( 100 );
	bool sites_picked( false );
	int max_fail_count( 0 );
	int dir;
	int site1 ( end_buffer );
	int site2 ( total_residue - end_buffer );

	std::vector< int > loc_max_temp;
	std::vector< int >::iterator loc_max_it;

   maxsites = total_insert - (end_buffer + size1 + min_spacing + size2 + end_buffer);


	if (maxsites < 1 ){
		minfrag_begin1 = -1;
		maxfrag_begin1 = -1;

		minfrag_begin2 = -1;
		maxfrag_begin2 = -1;
		return false;
	}

	do {
		++max_fail_count;

	int min_site1 = end_buffer + delta + 1;
	int min_site2 = end_buffer + delta + 1;
	int max_site1 = total_residue - (end_buffer + size1 + delta); //leave room for frag
	int max_site2 = total_residue - (end_buffer + size2 + delta); //leave room for frag

	for (int i = 0; i < static_cast<int>(loc_max.size()); ++i){
		if ( loc_max.at( i ) >= min_site1 && loc_max.at( i ) <= max_site1 ) loc_max_temp.push_back( loc_max.at( i ) );
	}

	bool in_loc_max ( false );
	int jitter_window( 4 );
	float temp = ran3();

//vats Pick site1

	if ( temp < 0.33 ){
		if ( loc_max_temp.size() > 0 ) {
			site1 = loc_max_temp.at( int(ran3()*loc_max_temp.size()) );
			in_loc_max = true;
		}
	} else {
		site1 = static_cast< int > ( min_site1 + ran3()*(max_site1 - min_site1) );
	}

	if ( ran3() < 0.5 ){
		if (!in_loc_max && secstruct(site1) != 'L') {
			if (ran3() < 0.5 ){
				while(site1 <= max_site1 && secstruct(site1) != 'L') {
					++site1;
				}
			} else{
				while(site1 >= min_site1 && secstruct(site1) != 'L'){
					--site1;
				}
			}
		}
	}

	site1 = site1 + int(ran3()*jitter_window);


	if ( site1 <= min_site1 || site1 >= max_site1 ) continue;

	//car pick a direction to look for the second site:
	if ( site1 < 2*end_buffer ) { //close to N-term
		dir = 1; // look C-terminally
	} else if ( site1 > total_residue- (2 * end_buffer + size1) ) { //close to Cterm
		dir = -1; // look N-terminally
	} else if ( ran3() < 0.5 ) {
		dir = 1; // pick randomly
	} else {
		dir = -1;
	}

	minfrag_begin1 = site1 - delta;
	maxfrag_begin1 = site1 + delta;
	assert( minfrag_begin1 > end_buffer );
	assert( maxfrag_begin1 < total_residue - (end_buffer + size1 -1));

	for ( int j = minfrag_begin1, je = maxfrag_begin1 + size1 - 1; j <= je ; ++j ) {
		if ( ! allow_insert(j) ) {
			continue;
			//		return max_sites;
		}
	}

	int size;
	if ( dir == 1 ) {
		size = size1;  //frag1 is relevant for computing spacing
	} else {
		size = size2;  //frag2 is relevant for computing spacing
	}

	int closest_site2 = site1 + dir * (size + min_spacing + 2*delta);
	int farthest_site2 = std::max(min_site2, std::min(max_site2, site1 + dir * (size + max_spacing + 2*delta)));


	if ( dir == 1 ){
		if ( closest_site2 > max_site2 ) {
			continue;
			//			return max_sites;
		} else {
			if( farthest_site2 > max_site2 ) {
				farthest_site2 = max_site2 - 1;
				assert( farthest_site2 >= closest_site2 );
			}
		}
	} else {
		if( closest_site2 < min_site2 ) {
			continue;
			//			return max_sites;
		} else {
			if( farthest_site2 < min_site2 ) {
				farthest_site2 = min_site2 + 1;
				assert( farthest_site2 <= closest_site2 );
			}
		}
	}

//vats Now pick site2
	temp = ran3();


	if( temp < 0.33 ){
		site2 = closest_site2;
		if( dir == 1 ){
			 while( secstruct(site2) !='L' && site2 <= farthest_site2 ){
				++site2;
			}
		} else {
			while( secstruct(site2) !='L' && site2 >= farthest_site2 ){
				--site2;
			}
		}
	} else if ( temp < 0.66 ) {
		site2 = closest_site2;
		if( dir == 1 ){
			while( site2 <= farthest_site2 ){
				loc_max_it = find(loc_max_temp.begin(),loc_max_temp.end(),site2);
				++site2;
				}
			if( loc_max_it == loc_max_temp.end() ){
				site2 = static_cast< int >( closest_site2 + ran3()*(farthest_site2 - closest_site2) );
			} else {
				site2 = *(loc_max_it);
			}
		} else {
			while( site2 >= farthest_site2 ){
				loc_max_it = find(loc_max_temp.begin(),loc_max_temp.end(),site2);
				--site2;
			}
			if ( loc_max_it == loc_max_temp.end() ){
				site2 = static_cast< int >( closest_site2 - ran3()*(closest_site2 - farthest_site2) );
			} else {
				site2 = *(loc_max_it);
			}
		}

	} else {
		if ( dir == 1 ) {
			site2 = static_cast< int >( closest_site2 + ran3()*(farthest_site2 - closest_site2) );
		} else {
			site2 = static_cast< int >( closest_site2 - ran3()*(closest_site2 - farthest_site2) );
		}
	}


/////////////////////////////////////////////////////////////////////////////

	if( site1 < min_site1 || site1 > max_site1 ) continue;
	if( site2 < min_site1 || site2 > max_site2 ) continue;

	minfrag_begin2 = site2 - delta;
	maxfrag_begin2 = site2 + delta;

	for ( int j = minfrag_begin2, je = maxfrag_begin2 + size2 - 1; j <= je; ++j ) {
		if ( !allow_insert( j ) ) continue;
	}


	//	assert( std::max( (minfrag_begin2 - maxfrag_begin1),(minfrag_begin1 - maxfrag_begin2) ) + 1	>  min_spacing + size - 1 );
	if( std::max( (minfrag_begin2 - maxfrag_begin1),(minfrag_begin1 - maxfrag_begin2) ) + 1	>  min_spacing + size - 1 ) sites_picked = true;

	} while ( !sites_picked || ( max_fail_count <= MAX_FAIL ) );

	return sites_picked;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin use_crankshaft
//
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 10th 2006
////////////////////////////////////////////////////////////////////////////////
bool
use_new_crankshaft()
{
	static bool apply_new_crankshaft( false );
	static bool init( false );

	if ( !init ) {
		apply_new_crankshaft = truefalseoption("use_crankshaft");
		init = true;
	}
	return apply_new_crankshaft;
}
