// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7523 $
//  $Date: 2006-02-20 15:10:59 -0800 (Mon, 20 Feb 2006) $
//  $Author: jiangl $

//////////////////////////////////////////////

// Rosetta Headers
#include "cst_descriptor.h"
#include "cst_countpair.h"
#include "cst_packer.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "misc.h"
#include "ligand.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param_aa.h"
#include "pdb.h"
#include "pose.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>
#include <list>
#include <string>

namespace cst_descriptor_ns {

	////////////////////////////////
	//  constraint descriptor
	////////////////////////////////
	// set up constraint set for handing the input cst
	// and pass it to pose_cst and packer_cst
	// Lin Jiang

  /////////////////////////////////////////////////////////////////////////////
  //lin read constraint file, and add the constraints into Cst_set class
  //lin return true if read the file normally
	//
	// Constriant format:
	//
	// reading begins with "CST::BEGIN" and end with "CST::END"
	//
	// # define template_atoms
	// # ATOM_MAP  0:template_atom_A,  1:template_atom_B
	// #define the atom:
	//   TEMPLATE::   ATOM_MAP: 0 atom_id:  0  0  0
	//   TEMPLATE::   ATOM_MAP: 0 atom_name: ****,
	// #define the res:
	//   TEMPLATE::   ATOM_MAP: 0 seqpos:   0 0 0
	//   TEMPLATE::   ATOM_MAP: 0 posfile:  ****
	//   TEMPLATE::   ATOM_MAP: 0 poslist:  0
	// #define the aa:
	//   TEMPLATE::   ATOM_MAP: 0 ligand:  1
	//   TEMPLATE::   ATOM_MAP: 0 residue: 1
	//
	// #define the coord constraint which constrant the atom to the coord(x,y,z)
	// #                              x       y       z
	//   TEMPLATE::   FIX_COORD:     0.000   0.000   0.000
	//
	// # define constraint
	// # distance constraint         d delta_d weight is_covalent (0:false, 1:true)
	//   CONSTRAINT:: distanceAB:   0.00  0.00  0.00  0
	// # angle constraint         ang delta_ang weight periodic
	//   CONSTRAINT::    angle_A:   0.00  0.00  0.00  360.
	//   CONSTRAINT::    angle_B:   0.00  0.00  0.00  360.
	//   CONSTRAINT::  torsion_A:   0.00  0.00  0.00  360.
	//   CONSTRAINT:: torsion_AB:   0.00  0.00  0.00  360.
	//   CONSTRAINT::  torsion_B:   0.00  0.00  0.00  360.
	//
	// # define conformer
	// # distance conformer       start_val conf_sd conf_N(how many steps are tried within the variation)
	//   CONFORMER::  distanceAB:    0.00      0.00     0
	// # angle conformer          start_val conf_sd conf_N ang_N(rotate the start angle by 360.0/ang_N)
	//   CONFORMER::    angle_A:     0.00      0.00     0     0
	//   CONFORMER::    angle_B:     0.00      0.00     0     0
	//   CONFORMER::  torsion_A:     0.00      0.00     0     0
	//   CONFORMER:: torsion_AB:     0.00      0.00     0     0
	//   CONFORMER::  torsion_B:     0.00      0.00     0     0

  bool
  cst_set_descriptor::read_cstfile( std::string const & filename_in ) {

    std::ifstream file;
    std::istringstream line_stream;
    file.clear();
    file.open( filename_in.c_str() );

    if ( !file ) {
      std::cout << "cst_descriptor::read_cstfile - unable to open cstfile " << filename_in  << std::endl;
      return false;
    } else {
      std::cout << "cst_descriptor::read_cstfile: "<< filename_in  << std::endl;
    }

    //cst_list.clear();

    std::string line, key(""), tag("");
    bool start( false);
    int map_id;
    float coor_x,coor_y,coor_z;
    cst_descriptor cst_index;

    while(file) {
      key = "";
      tag = "";
      getline(file, line);
      line_stream.clear();
      line_stream.str(line);
      line_stream.seekg( std::ios_base::beg );
      line_stream >> key;

      if( key == "CST::BEGIN" ) {
				start=true;
				cst_index.erase();
      }
      if( key == "CST::END" ) {
				start=false;
				cst_list.push_back(cst_index);
      }

      if( !start ) continue;

      //lin read template information
      if( key == "TEMPLATE::" ) {

				line_stream >> tag ;

				//lin read the template map id
				//lin coordinate constraint map id is maped to fix_coord
				//lin pairwise constraint map id is 1, 2
				if( tag == "ATOM_MAP:" ) {
					line_stream >> map_id;
					if ( map_id <=1 ) {
						//std::cout<<"read template A:"  << std::endl;
						cst_index.template_A.read( line_stream );
					} else if ( map_id == 2 ) {
						//std::cout<<"read template B:" << std::endl;
						cst_index.template_B.read( line_stream );
					}
				}

				//read the fix coordinates
				if( key == "FIX_COORD:" ) {
					line_stream >> coor_x >> coor_y >> coor_z ;
					cst_index.coord.assign(coor_x, coor_y, coor_z);
					cst_index.fix_coord=true;
				}
      }

      //lin read constraint information
      if( key == "CONSTRAINT::" ) {
				//    std::cout<<"read constraint:" << std::endl;
				cst_index.cst_parameter.read( line_stream );
				// std::cout<< cst_index.cst_parameter<<std::endl;
      }

      //lin read conformer information
      if( key == "CONFORMER::" ) {
				//    std::cout<<"read constraint:" << std::endl;
				cst_index.cst_conformer.read( line_stream );
				// std::cout<< cst_index.cst_parameter<<std::endl;
      }
    }

    return true;
  }

	/////////////////////////////////////////////////////////////////////////////
	//lin read the seq position from a file
  void read_posfile( std::string const & filename_in, std::vector<int> &poslist ) {

    using namespace files_paths;

    std::ifstream file;
    std::istringstream line_stream;
    std::string filename;
    if ( filename_in == "none" ) return;
    file.clear();
    if ( filename_in == "auto") {
      filename = start_path + start_file + ".pos";
    } else {
      filename = start_path + filename_in;
    }
    file.open( filename.c_str() );

    if ( !file ) {
      std::cout << "read_posefile - unable to open cstfile " << filename  << std::endl;
      return;
    } else {
      std::cout << "read_posfile: "<< filename  << std::endl;
    }

    poslist.clear();
    int seqpos;
    while ( !line_stream.fail() ) {
      line_stream >> seqpos ;
      if( seqpos > 0 ) poslist.push_back(seqpos);
    }
  }

	/////////////////////////////////////////////////////////////////////////////
  bool
  template_residue::insert_atype_list(int const atom_type) {
		if( atom_type > 0 && std::find(atype_list.begin(), atype_list.end(), atom_type )
				== atype_list.end() ) {
			atype_list.push_back(atom_type);
			return true;
		} else {
			return false;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
  bool
  template_residue::insert_seqpos_list(int const seqpos){
		if( seqpos > 0 && std::find(seqpos_list.begin(), seqpos_list.end(), seqpos )
				== seqpos_list.end() ) {
			seqpos_list.push_back(seqpos);
			return true;
		} else {
			return false;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
  bool
  template_residue::insert_res_list(int const aa_type){
		if( aa_type > 0 && std::find(res_list.begin(), res_list.end(), aa_type )
				== res_list.end() ) {
			res_list.push_back(aa_type);
			//std::cout<<"rescode="<<aa_type<<std::endl;
			return true;
		} else {
			return false;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
  bool
	template_residue::insert_template_id_list(numeric::xyzVector_int const & b) {
		if( b.x() > 0 && std::find(template_id_list.begin(), template_id_list.end(), b )
				== template_id_list.end() ) {
			template_id_list.push_back(b);
			return true;
		} else {
			return false;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
  void
  template_residue::read( std::istringstream &line_stream ) {

    using namespace param;

    std::string tag(""),aa3("");
    int seqpos, res_tmp, ta1, ta2, ta3, atom_type,atom_cent,aa_type;
    char aachar( ' ' ), chain_id( ' ' );
    std::string atom_name(""), posfile("");
    numeric::xyzVector_int b(0);

    while ( !line_stream.fail() ) {
      line_stream >> tag ;

      //read the template atom id
      if( tag == "atom_id:" ) {
				line_stream >> ta1 >> ta2 >> ta3 ;
				b.assign( ta1, ta2, ta3 );
				//std::cout<<"atom_id:"<<b.x()<<" "<<b.y()<<" "<<b.z()<<" "<<std::endl;
				insert_template_id_list(b);
				ta1 = 0;
      } else if( tag == "atom_name:" ) {
				line_stream >> skip(1) >> bite(4, atom_name) >> skip(1) ;
				//std::cout<<"atom_name:|"<<atom_name<<"|"<<std::endl;
				if( atom_name != "" ) {
					atom_num_from_atom_char(atom_name,atom_type,
																	atom_cent);
					insert_atype_list(atom_type);
					atom_name = "";
					atom_type = 0;
				}
      }

      //read the template seqpos
      if( tag == "seqpos:" ) {
				while ( !line_stream.fail() ) {
					line_stream >> seqpos ;
					insert_seqpos_list(seqpos);
					seqpos = 0;
				}
      }
      if( tag == "posfile:" ) {
				line_stream >> posfile;
				read_posfile( posfile, seqpos_list );
				posfile = "" ;
      }
      if( tag == "pdbres:" ) {
				line_stream >> skip(1) >> bite( chain_id);
				while ( !line_stream.fail() ) {
					line_stream >> res_tmp ;
					if( res_tmp != 0 ){
						pdbres2rosettares( chain_id, res_tmp, seqpos );
					} else {
						seqpos = res_tmp;
					}
					insert_seqpos_list(seqpos);
					seqpos = 0;
				}
      }

      //read the template aa, asume aav = 1
      if( tag == "residue1:" ) {
				line_stream >> skip( 1 ) ;
				for ( int i = 1, ie = MAX_AA(); i <= ie; ++i ) {
					line_stream >> bite( aachar );
					if ( aachar != ' ' ) {
						num_from_res1(aachar,aa_type);
						insert_res_list(aa_type);
					}
					aa_type = 0;
					aachar = ' ';
				}
      }

      //read the template aa, asume aav = 1
			if( tag == "residue3:" ) {
				while ( !line_stream.fail() ) {
					line_stream >> skip(1) >> bite(3,aa3) ;
					num_from_res3( aa3, aa_type );
					insert_res_list(aa_type);
				}
				aa3 = "";
      }

      tag=" ";
    }
  }

	/////////////////////////////////////////////////////////////////////////////
  void
  cst_parameters::read( std::istringstream &line_stream ) {

    std::string tag(""), func_name("");

    while ( !line_stream.fail() ) {

      line_stream >> tag ;

      if( tag == "distanceAB:") {
				move_disAB=true;
				line_stream >> disAB >> disAB_sd >> disAB_k >> is_covalent ;
      }
      if( tag == "angle_A:") {
				move_angA=true;
				line_stream >> angA >> angA_sd >> angA_k >> angA_periodic ;
      }
      if( tag == "angle_B:") {
				move_angB=true;
				line_stream >> angB >> angB_sd >> angB_k >> angB_periodic ;
     }
      if( tag == "torsion_A:") {
				move_dihA=true;
				line_stream >> dihA >> dihA_sd >> dihA_k >> dihA_periodic >> dihA_func ;
				if( dihA_func != "PERIODIC" ) dihA_func = "HARMONIC";//default HARMONIC
      }
      if( tag == "torsion_B:") {
				move_dihB=true;
				line_stream >> dihB >> dihB_sd >> dihB_k >> dihB_periodic >> dihB_func ;
				if( dihA_func != "PERIODIC" ) dihA_func = "HARMONIC";
      }
      if( tag == "torsion_AB:") {
				move_dihAB=true;
				line_stream >> dihAB >> dihAB_sd >> dihAB_k >> dihAB_periodic >> dihAB_func ;
				if( dihA_func != "PERIODIC" ) dihA_func = "HARMONIC";
      }

      tag = "";
    }
  }


	/////////////////////////////////////////////////////////////////////////////
  void
  cst_conformers::read( std::istringstream &line_stream ) {

		//float const deg2rad( numeric::conversions::radians(1.0) );
    std::string tag("");
    int val_N, conf_N;
    float val, conf_sd;

    while ( !line_stream.fail() ) {

      line_stream >> tag ;

      if( tag == "distanceAB:") {
				line_stream >> val >> conf_sd >> conf_N ;
				disAB=val;
				disAB_conf_sd=conf_sd;
				disAB_conf_N=conf_N;
      }
      if( tag == "angle_A:") {
				line_stream >> val >> conf_sd >> conf_N >> val_N ;
				angA=val;
				angA_conf_sd=conf_sd;
				angA_conf_N=conf_N;
				angA_N=val_N;
      }
      if( tag == "angle_B:") {
				line_stream >> val >> conf_sd >> conf_N >> val_N ;
				angB=val;
				angB_conf_sd=conf_sd;
				angB_conf_N=conf_N;
				angB_N=val_N;
      }
      if( tag == "torsion_A:") {
				line_stream >> val >> conf_sd >> conf_N >> val_N ;

				dihA_conf_N=conf_N;
				dihA_N=val_N;
      }
      if( tag == "torsion_B:") {
				line_stream >> val >> conf_sd >> conf_N >> val_N ;
				dihB=val;
				dihB_conf_sd=conf_sd;
				dihB_conf_N=conf_N;
				dihB_N=val_N;
      }
      if( tag == "torsion_AB:") {
				line_stream >> val >> conf_sd >> conf_N >> val_N ;
				dihAB=val;
				dihAB_conf_sd=conf_sd;
				dihAB_conf_N=conf_N;
				dihAB_N=val_N;
      }

      tag = "";
    }
  }


	/////////////////////////////////////////////////////////////////////////////
	//lin is the constraint with anchor between protein and ligand
	bool cst_descriptor::is_local_ligand_cst() const {

		using namespace param_aa;
		return ( stat_cst() && ( ( is_ligand( template_A.res_list[0]) &&
															 is_protein( template_B.res_list[0]) ) ||
														 ( is_ligand( template_B.res_list[0]) &&
															 is_protein( template_A.res_list[0]) ) ) ) ;
	}

	/////////////////////////////////////////////////////////////////////////////
  bool cst_descriptor::is_local_ligand_cst( int const LG, bool const attach_by_file /*false by default*/ ) const {

		using namespace param_aa;
		if( LG == param_aa::ligand_aa_vector[1] ) {
			return ( stat_cst() && ( ( template_A.res_list[0] == LG
															 && is_protein( template_B.res_list[0] ) ) ||
															 ( template_B.res_list[0] == LG
															 && is_protein( template_A.res_list[0] ) ) ) ) ;
		} else {
			// modify the logic slightly: it used to always attach lig2 to lig1, but we want to change the
			// behavior so that it uses the order from the cst file, i.e., if a cst exists between lig2 and protein and
			// it comes before any cst between lig2 and lig1, we will attach lig2 to protein. However, we still keep
			// the original behavior that lig1 is always attached to protein no matter how is defined is the cst file
			// chu 2008/01/16
			if ( attach_by_file ) {
				return ( stat_cst() && ( ( template_A.res_list[0] == LG || template_B.res_list[0] == LG ) ) );
			} else { 	//lig2 attached to lig1
				return ( stat_cst() && ( ( template_A.res_list[0] == LG
							&& template_B.res_list[0] == param_aa::ligand_aa_vector[1] ) ||
						( template_B.res_list[0] == LG
							&& template_A.res_list[0] == param_aa::ligand_aa_vector[1] ) ) ) ;
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin check the seqpos is the catalytic position defined in catalytic_descriptor
	bool cst_set_descriptor::is_catpos( int const seqpos ) const {

		using namespace param_aa;

		for( int ii=0, ie=cst_list.size(); ii<ie; ii++ ) {
			if( cst_list[ii].is_local_ligand_cst()  ) {
				if( has_ligand_no( cst_list[ii].template_A.res_list[0] ) >= 1 ) {
					if( seqpos == cst_list[ii].template_B.seqpos_list[0] ){
						return true;
					}
				} else {
					if( seqpos == cst_list[ii].template_A.seqpos_list[0] ){
						return true;
					}
				}
			}
		}
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin check the seqpos is the catalytic position defined in catalytic_descriptor
	bool cst_set_descriptor::is_cstpos( int const seqpos ) const {

		using namespace param_aa;

		for( int ii=0, ie=cst_list.size(); ii<ie; ii++ ) {
			if( cst_list[ii].stat_cst()  ) {
				if( seqpos == cst_list[ii].template_A.seqpos_list[0] ||
						seqpos == cst_list[ii].template_B.seqpos_list[0] ){
						return true;
				}
			}
		}
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin get the anchor inforamtion from anchor constraint
	void cst_set_descriptor::get_ligand_anchor_atoms(
		pose_ns::Pose  & pose,
		int & ligaa,
		FArray2Da_float lig_coord,
		int & anchor_atomno,
		int & anchor_rsd,
		int & lig_root_atomno,
		bool & attach_by_jump ) const {

		lig_coord.dimension( 3, param::MAX_ATOM() );

		//find the stat constraint to attach ligand
		cst_descriptor anchor_cst;
		for( int ii=0, ie=cst_list.size(); ii<ie; ii++ ) {
			if( cst_list[ii].is_local_ligand_cst(ligaa, true) ) {
				anchor_cst = cst_list[ii];
				break;
			}
		}

		//lin attach by jump
		attach_by_jump=enzyme::attach_ligaa_by_jump;

		if( anchor_cst.empty() ) {

			// find closest C-alpha to ligand anchor atom
			using numeric::xyzVector_float;
			lig_root_atomno = 1; // 1st hetero heavyatom in pdb file
			xyzVector_float const lig_root_xyz( &( lig_coord( 1, lig_root_atomno )));
			float min_d( 1000.0f );

			anchor_atomno = 1; // chu change from CA to N to be consistent with atom_tree get_anchor_atom convention
			for ( int i=1; i<= pose.total_residue(); ++i ) {
				float const d = distance( xyzVector_float( &(pose.full_coord()(1,2,i))),
                                lig_root_xyz );
				if ( d<min_d ) {
					min_d = d;
					anchor_rsd = i;
				}
			}
		} else {

			//use the stat constraint to attach ligand
			if( anchor_cst.template_A.res_list[0] == ligaa) {
				lig_root_atomno = anchor_cst.template_A.template_id_list[0].x();
				anchor_atomno = anchor_cst.template_B.template_id_list[0].x();;
				anchor_rsd = anchor_cst.template_B.seqpos_list[0];
			} else {
				lig_root_atomno = anchor_cst.template_B.template_id_list[0].x();
				anchor_atomno = anchor_cst.template_A.template_id_list[0].x();;
				anchor_rsd = anchor_cst.template_A.seqpos_list[0];
			}
		}

		std::cout<<"get_ligand_anchor_atoms: anchor: atomno=" <<anchor_atomno
						 <<" rsd= "<<anchor_rsd<<" lig_root_atomno= "<<lig_root_atomno
						 <<" attach_by_jump= "<<attach_by_jump<<std::endl;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin constraint setup for cst_set and pose in order to do pose type minimization
	void cst_set_descriptor::fill_cst_allow_move_set(
					packer_cst_ns::Packer_cst_set & packer_cst,
					cst_allow_move_ns::Allow_move_set & allow_move ) const {

		//fill cst set and pose allow move
		for( int ii=0, ie=cst_list.size(); ii<ie; ii++ ) {
			cst_list[ii].fill_cst_allow_move_set( packer_cst, allow_move );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin constraint setup of single constraint
	void cst_descriptor::fill_cst_allow_move_set(
					 packer_cst_ns::Packer_cst_set & packer_cst,
					 cst_allow_move_ns::Allow_move_set & allow_move ) const {

		// method for filling the cst set
		int seqpos1,seqpos2;
		int aa1,aav1,aa2,aav2;

		if( fix_coord ) return;//method ?

		for(int mm=0, me=template_A.seqpos_list.size(); mm<me;mm++) {
			seqpos1=template_A.seqpos_list[mm];
			for(int nn=0, ne=template_B.seqpos_list.size(); nn<ne;nn++) {
				seqpos2=template_B.seqpos_list[nn];

				for(int ii=0, ie=template_A.template_atom_list.size(); ii<ie;ii++) {
					aa1=template_A.template_atom_list[ii].aa;
					aav1=template_A.template_atom_list[ii].aav;

					//define the all atom_ids involved in the constraint
					kin::Fullatom_id a1(seqpos1, template_A.template_atom_list[ii].
															template_id.x(), aa1, aav1);
					kin::Fullatom_id a2(seqpos1, template_A.template_atom_list[ii].
															template_id.y(), aa1, aav1);
					kin::Fullatom_id a3(seqpos1, template_A.template_atom_list[ii].
															template_id.z(), aa1, aav1);

					for (int jj=0, je=template_B.template_atom_list.size(); jj<je;jj++){
						aa2=template_B.template_atom_list[jj].aa;
						aav2=template_B.template_atom_list[jj].aav;

						kin::Fullatom_id b1(seqpos2, template_B.template_atom_list[jj].
																template_id.x(), aa2, aav2);
						kin::Fullatom_id b2(seqpos2, template_B.template_atom_list[jj].
																template_id.y(), aa2, aav2);
						kin::Fullatom_id b3(seqpos2, template_B.template_atom_list[jj].
																template_id.z(), aa2, aav2);

						// method for filling the packer cst set, allow move

						//lin fill the packer cst set as the same group
						fill_packer_cst_set( a1, a2, a3, b1, b2, b3, packer_cst, ii!=0||jj!=0 );

						//lin fill the pose cst set
						if( !enzyme::cst_conformer || ( enzyme::cst_conformer && ii == 0 && jj == 0 ) )
								fill_allow_move_set( a1, a2, a3, b1, b2, b3, allow_move );
					}
				}
			}
		}

		//packer_cst.clean_up();
	}

	//lin packer constraint setup of single constraint
	void cst_descriptor::fill_packer_cst_set(
					 kin::Fullatom_id const & a1,
					 kin::Fullatom_id const & a2,
					 kin::Fullatom_id const & a3,
					 kin::Fullatom_id const & b1,
					 kin::Fullatom_id const & b2,
					 kin::Fullatom_id const & b3,
				   packer_cst_ns::Packer_cst_set & packer_cst,
					 bool same_group ) const { //lin default is false, if cst is same group, use true

		float const deg2rad( numeric::conversions::radians(1.0) );
		float const pi( numeric::constants::d::pi );

		std::vector< packer_cst_ns::Rsd_rsd_cst* > cst_plist;

		//distance AB
		if( cst_parameter.disAB_k != 0 ) {
			float weight_scale (1.0);
			if( cst_parameter.is_covalent ) weight_scale=enzyme::covalent_cst_scale;
			cst_set_ns::Cst distance_cst( cst_parameter.disAB, cst_parameter.disAB_sd,
											cst_parameter.disAB_k*weight_scale, cst_parameter.disAB_k );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_distance_cst( a1, b1, distance_cst );
			cst_plist.push_back( rr_cst_p );
		}

		//angle A
		if( cst_parameter.angA_k != 0 ) {
			cst_set_ns::Angle_cst angle_cst( a2, a1, b1,
																			 std::abs( periodic_range( cst_parameter.angA * deg2rad, 2.0*pi ) ),
																			 std::abs( periodic_range( cst_parameter.angA_sd * deg2rad, 2.0*pi ) ),
																			 cst_parameter.angA_periodic * deg2rad,
																			 cst_parameter.angA_k );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_angle_cst( angle_cst );
			cst_plist.push_back( rr_cst_p );
		}

		//angle B
		if( cst_parameter.angB_k != 0 ) {
			cst_set_ns::Angle_cst angle_cst( a1, b1, b2,
																			 std::abs( periodic_range( cst_parameter.angB * deg2rad, 2.0*pi ) ),
																			 std::abs( periodic_range( cst_parameter.angB_sd * deg2rad, 2.0*pi ) ),
																			 cst_parameter.angB_periodic * deg2rad,
																			 cst_parameter.angB_k );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_angle_cst( angle_cst );
			cst_plist.push_back( rr_cst_p );
		}

		//torsion A
		if( cst_parameter.dihA_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst( a3, a2, a1, b1,
																					 periodic_range( cst_parameter.dihA * deg2rad, 2.0*pi ),
																					 periodic_range( cst_parameter.dihA_sd * deg2rad, 2.0*pi ),
																					 cst_parameter.dihA_periodic * deg2rad,
																					 cst_parameter.dihA_k,
																					 cst_parameter.dihA_func );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_torsion_cst( torsion_cst );
			cst_plist.push_back( rr_cst_p );
		}

		//torsion AB
		if( cst_parameter.dihAB_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst( a2, a1, b1, b2,
																					 periodic_range( cst_parameter.dihAB * deg2rad, 2.0*pi ),
																					 periodic_range( cst_parameter.dihAB_sd * deg2rad, 2.0*pi ),
																					 cst_parameter.dihAB_periodic * deg2rad,
																					 cst_parameter.dihAB_k,
																					 cst_parameter.dihAB_func );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_torsion_cst( torsion_cst );
			cst_plist.push_back( rr_cst_p );
		}

		//torsion B
		if( cst_parameter.dihB_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst( a1, b1, b2, b3,
																					 periodic_range( cst_parameter.dihB * deg2rad, 2.0*pi ),
																					 periodic_range( cst_parameter.dihB_sd * deg2rad, 2.0*pi ),
																					 cst_parameter.dihB_periodic * deg2rad,
																					 cst_parameter.dihB_k,
																					 cst_parameter.dihB_func );
			packer_cst_ns::Rsd_rsd_cst* rr_cst_p;
			rr_cst_p = new packer_cst_ns::Rsd_rsd_torsion_cst( torsion_cst );
			cst_plist.push_back( rr_cst_p );
		}

		packer_cst.add_rr_constraint_list( a1, b1, cst_plist, same_group );
	}


	//lin pose constraint setup of single constraint
	void cst_descriptor::fill_pose_cst_set(
					 kin::Fullatom_id const & a1,
					 kin::Fullatom_id const & a2,
					 kin::Fullatom_id const & a3,
					 kin::Fullatom_id const & b1,
					 kin::Fullatom_id const & b2,
					 kin::Fullatom_id const & b3,
					 cst_set_ns::Cst_set & cst_set
					 ) const {

		float const deg2rad( numeric::conversions::radians(1.0) );
		float const pi( numeric::constants::d::pi );

		//distance AB
		if( cst_parameter.disAB_k != 0 ) {
			cst_set_ns::Cst distance_cst( cst_parameter.disAB,
									 cst_parameter.disAB_sd, cst_parameter.disAB_k );
			cst_set.add_atompair_constraint( a1, b1, distance_cst );
		}

		//angle A
		if( cst_parameter.angA_k != 0 ) {
			cst_set_ns::Angle_cst angle_cst
				( a2, a1, b1,
					std::abs( periodic_range( cst_parameter.angA * deg2rad, 2.0*pi ) ),
					std::abs( periodic_range( cst_parameter.angA_sd * deg2rad, 2.0*pi ) ),
					cst_parameter.angA_periodic * deg2rad,
					cst_parameter.angA_k );
			cst_set.add_atom_angle_constraint( angle_cst );
		}

		//angle B
		if( cst_parameter.angB_k != 0 ) {
			cst_set_ns::Angle_cst angle_cst
				( a1, b1, b2,
					std::abs( periodic_range( cst_parameter.angB * deg2rad, 2.0*pi ) ),
					std::abs( periodic_range( cst_parameter.angB_sd * deg2rad, 2.0*pi ) ),
					cst_parameter.angB_periodic * deg2rad,
					cst_parameter.angB_k );
			cst_set.add_atom_angle_constraint( angle_cst );
		}

		//torsion A
		if( cst_parameter.dihA_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst
				( a3, a2, a1, b1,
					periodic_range( cst_parameter.dihA * deg2rad, 2.0*pi ),
					periodic_range( cst_parameter.dihA_sd * deg2rad, 2.0*pi ),
					cst_parameter.dihA_periodic * deg2rad,
					cst_parameter.dihA_k,
					cst_parameter.dihA_func );
			cst_set.add_atom_torsion_constraint( torsion_cst );
		}

		//torsion AB
		if( cst_parameter.dihAB_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst
				( a2, a1, b1, b2,
					periodic_range( cst_parameter.dihAB * deg2rad, 2.0*pi ),
					periodic_range( cst_parameter.dihAB_sd * deg2rad, 2.0*pi ),
					cst_parameter.dihAB_periodic * deg2rad,
					cst_parameter.dihAB_k,
					cst_parameter.dihAB_func );
			cst_set.add_atom_torsion_constraint( torsion_cst );
		}

		//torsion B
		if( cst_parameter.dihB_k != 0 ) {
			cst_set_ns::Torsion_cst torsion_cst
				( a1, b1, b2, b3,
					periodic_range( cst_parameter.dihB * deg2rad, 2.0*pi ),
					periodic_range( cst_parameter.dihB_sd * deg2rad, 2.0*pi ),
					cst_parameter.dihB_periodic * deg2rad,
					cst_parameter.dihB_k,
					cst_parameter.dihB_func );
			cst_set.add_atom_torsion_constraint( torsion_cst );
		}
	}

	//lin pose constraint setup of single constraint
	void cst_descriptor::fill_allow_move_set(
					 kin::Fullatom_id const & a1,
					 kin::Fullatom_id const & a2,
					 kin::Fullatom_id const & a3,
					 kin::Fullatom_id const & b1,
					 kin::Fullatom_id const & b2,
					 kin::Fullatom_id const & b3,
					 cst_allow_move_ns::Allow_move_set & allow_move
					 ) const {

    float const deg2rad( numeric::conversions::radians(1.0) );
		float const pi( numeric::constants::d::pi );

		//distance AB
		if( cst_parameter.move_disAB ) {
			allow_move.add_distance_atoms( a1, b1, cst_conformer.disAB,
																		 cst_conformer.disAB_conf_sd,
																		 cst_conformer.disAB_conf_N );
		}

		//angle A
		if( cst_parameter.move_angA ) {
			allow_move.add_angle_atoms( a2, a1, b1,
																	periodic_range( cst_conformer.angA*deg2rad, 2.0*pi ),
																	periodic_range( cst_conformer.angA_conf_sd*deg2rad, 2.0*pi ),
																	cst_conformer.angA_conf_N,cst_conformer.angA_N );
		}

		//angle B
		if( cst_parameter.move_angB ) {
		  allow_move.add_angle_atoms( a1, b1, b2,
																	periodic_range( cst_conformer.angB*deg2rad, 2.0*pi ),
																	periodic_range( cst_conformer.angB_conf_sd*deg2rad, 2.0*pi ),
																	cst_conformer.angB_conf_N,cst_conformer.angB_N );
		}

		//torsion A
		if( cst_parameter.move_dihA ) {
			allow_move.add_torsion_atoms( a3 ,a2, a1, b1,
																	periodic_range( cst_conformer.dihA*deg2rad, 2.0*pi ),
																	periodic_range( cst_conformer.dihA_conf_sd*deg2rad, 2.0*pi ),
																	cst_conformer.dihA_conf_N,cst_conformer.dihA_N );
		}

		//torsion AB
		if( cst_parameter.move_dihAB ) {
			allow_move.add_torsion_atoms( a2, a1, b1, b2,
																	periodic_range( cst_conformer.dihAB*deg2rad, 2.0*pi ),
																	periodic_range( cst_conformer.dihAB_conf_sd*deg2rad, 2.0*pi ),
																	cst_conformer.dihAB_conf_N,cst_conformer.dihAB_N );
		}

		//torsion B
		if( cst_parameter.move_dihB ) {
			allow_move.add_torsion_atoms( a1, b1, b2, b3,
																	periodic_range( cst_conformer.dihB*deg2rad, 2.0*pi ),
																	periodic_range( cst_conformer.dihB_conf_sd*deg2rad, 2.0*pi ),
																	cst_conformer.dihB_conf_N,cst_conformer.dihB_N );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	//lin detect covalent bond from the constraint set
	void cst_descriptor::find_covalent_pair() {

    using namespace cst_countpair_ns;

		if(fix_coord) return;

		for(int ii=0, ie=template_A.template_atom_list.size(); ii<ie;ii++) {
			for(int jj=0, je=template_B.template_atom_list.size(); jj<je;jj++) {
					kin::Fullatom_id a(template_A.seqpos_list[0],
									template_A.template_atom_list[ii].template_id.x(),
									template_A.template_atom_list[ii].aa,
									template_A.template_atom_list[ii].aav);
					kin::Fullatom_id b(template_B.seqpos_list[0],
									template_B.template_atom_list[jj].template_id.x(),
									template_B.template_atom_list[jj].aa,
									template_B.template_atom_list[jj].aav);
					if( enzyme::covalent_from_cstfile ) {
						if( cst_parameter.is_covalent ) {
							covalent_bonds.push_back( std::make_pair(std::min(a,b), std::max(a,b) ) );
						}
	  			} else {
						if( is_covalent_bond( a, b, cst_parameter.disAB ) ) {
							cst_parameter.is_covalent=true;
							if( enzyme::safety_check ) {
								std::cout<<"Warning: atom pair " << a << ' ' << b <<
								" input disAB= "<< cst_parameter.disAB <<
								" detect as covalent bond!!! "<<std::endl;
							}
							covalent_bonds.push_back( std::make_pair(std::min(a,b), std::max(a,b) ) );
						}
					}
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
  void
  template_atom::cleanup( ) {

    using namespace aaproperties_pack;

    float const convert_to_degree = { 180.0/3.14159265359 };

    int b1=template_id.x();
    int b2=template_id.y();
    int b3=template_id.z();

		assert ( b1 != 0 );

		if( b2 == 0 || b3 == 0 ) {
			bond12=0; bond23=0; ang123=0;
			return;
		}

    type= fullatom_type(b1,aa,aav);
		//std::cout<<"tag1: "<<b1<<" "<<aa<<" "<<aav<<" "<<type<<std::endl;

		//calculate the bond length of b1->b2, b2->b3
    distance_bk(icoor(1,b1,aa,aav),icoor(1,b2,aa,aav),bond12);
    distance_bk(icoor(1,b2,aa,aav),icoor(1,b3,aa,aav),bond23);
    angle_bk(icoor(1,b2,aa,aav),icoor(1,b1,aa,aav),
	   icoor(1,b2,aa,aav),icoor(1,b3,aa,aav),ang123);
    ang123 *= convert_to_degree;
  }

	/////////////////////////////////////////////////////////////////////////////
  bool
  template_residue::cleanup( int total_res ) {

    using namespace aaproperties_pack;

    if( template_id_list.empty() && atype_list.empty() ) {
      std::cout << "WARNING:: template is not specified, should give either atom_id or atom_name"<<std::endl;
      return false;
    }

    if( res_list.empty() ) {
			std::cout << "WARNING:: map_atom don't specify the residue type!!!!!" << std::endl;
			if( cst_set_ns::debug_output ) {
				std::cout << *this;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			} else {
				return false;
			}
    }
    if( seqpos_list.empty() ) {
      if( param_aa::is_ligand(res_list[0]) ) {
				seqpos_list.push_back( total_res + param_aa::has_ligand_no(res_list[0]) );
      } else {
				std::cout << "WARNING:: template don't specify the seq positon!!!!!!" <<std::endl;
				if( cst_set_ns::debug_output ) {
					std::cout << *this;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				} else {
					return false;
				}
      }
    }

		//erase the template_atom_list
		template_atom_list.clear();

		if( !template_id_list.empty() ) {
			for( int ii=0, ie=res_list.size(); ii<ie; ii++ ) {
				int aa=res_list[ii];
				int aav_end=1;
				//lin try_both_his_tautomer aav=1,2
				if( design::try_both_his_tautomers && aa==param_aa::aa_his ) {
					aav_end = 2;
				}
				for( int jj=0, je=template_id_list.size(); jj<je; jj++ ) {
					for( int aav=1;aav<=aav_end;aav++) {
						int atom = template_id_list[jj].x();
						insert_atype_list( fullatom_type(atom,aa,aav) );
						template_atom atom_index(template_id_list[jj],aa,aav);
						//std::cout<<atom_index<<std::endl;
						template_atom_list.push_back(atom_index);
					}
				}
			}
		} else if ( !atype_list.empty() ) {
			for( int ii=0, ie=res_list.size(); ii<ie; ii++ ) {
				int aa=res_list[ii];
				int aav_end=1;
				//lin try_both_his_tautomer aav=1,2
				if( design::try_both_his_tautomers && aa==param_aa::aa_his ) {
					aav_end = 2;
				}
				for( int jj=0, je=atype_list.size(); jj<je; jj++ ) {
					for( int aav=1;aav<=aav_end;aav++) {
						int atom_type = atype_list[jj];
						//std::cout<<"natoms("<<aa<<","<<aav<<")="<<natoms(aa,aav)<<std::endl;
						for ( int atom = natoms(aa,aav); atom >= 1; --atom ) {
							if( atom_type == fullatom_type(atom,aa,aav) ) {
								int b1,b2,b3;
								b1=atom;
								b2=atom_base(b1, aa, aav);
								b3=atom_base(b2, aa, aav);
								template_atom atom_index(b1,b2,b3,aa,aav);
								//std::cout<<atom_index<<std::endl;

								insert_template_id_list(atom_index.template_id);
								template_atom_list.push_back(atom_index);
							}
						}
					}
				}
			}
		} else {
			std::cout << "WARNING:: constraint should  specify the template atom type or atom id" << std::endl;
			return false;
		}

    if( !template_atom_list.empty() ) return true;
    else return false;
  }

	/////////////////////////////////////////////////////////////////////////////
  bool
  cst_descriptor::cleanup( int total_res ) {
    bool nonemptyA,nonemptyB;
    nonemptyA=template_A.cleanup( total_res );
		//std::cout<<"cleanup template A:\n"<<template_A<<std::endl;
    nonemptyB=template_B.cleanup( total_res );
		//std::cout<<"cleanup template B:\n"<<template_B<<std::endl;

		//detect the covalent bonding pair in the cst
		find_covalent_pair();

    if( !nonemptyA ) {
      std::cout<< " WARNING:: template A empty" << std::endl;
      return false;
    }

    if( !nonemptyB && !fix_coord ) {
      std::cout<< " WARNING:: template B and fix coord empty" << std::endl;
      return false;
    }

//     if( cst_parameter.empty() ) {
//       std::cout << " ERROR:: zero constraint "<< std::endl;
//       return false;
//     }

    return true;

  }

	/////////////////////////////////////////////////////////////////////////////
  void
  cst_set_descriptor::cleanup( int total_res ) {
    for( int ii=0, ie=cst_list.size(); ii<ie; ii++ ) {
      if( !cst_list[ii].cleanup( total_res ) ) {
 				if( cst_set_ns::debug_output )
 					std::cout << "cst_set_descriptor::cleanup uncompleted, and delete it:\n"
										<< cst_list[ii] <<std::endl;
				cst_list.erase(cst_list.begin()+ii);
			}
			//cst_list[ii].build_templateB_atoms();
    }
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin debug output
  std::ostream& operator<< ( std::ostream& s, template_atom const & d ) {
    s << "\t\t\t\tTEMPALTE ATOM INFORMATION:" << std::endl;
    s << "\t\t\t\tres_code: aa=" << d.aa<< " aav=" <<d.aav << std::endl;
    s << "\t\t\t\ttemplate_id: "<<d.template_id.x()<<" "<<d.template_id.y()<<" "<<d.template_id.z()<<std::endl;
    s << "\t\t\t\tbond info: bond12="<<d.bond12<<" bond23="<<d.bond23<<" ang123="<<d.ang123<<std::endl;
    return s;
  }

  std::ostream& operator<< ( std::ostream& s, template_residue const & d ) {
    s << "\t\tTEMPALTE RESIDUE INFORMATION:" << std::endl;

    s << "\t\t\tmap_atom_list: atom_type=";
    for(int ii=0,ie=d.atype_list.size(); ii<ie; ii++) s <<" "<<d.atype_list[ii];
    s << std::endl;

    s << "\t\t\tmap_atom_list: template_id=";
    for(int ii=0,ie=d.template_id_list.size(); ii<ie; ii++)
      s <<" [ "<<d.template_id_list[ii].x()<<", "<<d.template_id_list[ii].y()
	<<", "<<d.template_id_list[ii].z()<<" ]";
    s << std::endl;

    s << "\t\t\tmap_seqpose_list: seqpos= ";
    for(int ii=0, ie=d.seqpos_list.size(); ii<ie; ii++)
      s << d.seqpos_list[ii]<< ", " ;
    s << std::endl;

    s << "\t\t\tmap_reslist: rescode= ";
    char aa;
    for(int ii=0, ie=d.res_list.size(); ii<ie; ii++) {
      res1_from_num(d.res_list[ii],aa);
      std::cout<<aa;
    }
    s << std::endl;

    s << "\t\t\ttemplate_atom_list: " <<std::endl;
    for(int ii=0, ie=d.template_atom_list.size(); ii<ie; ii++) {
      s << d.template_atom_list[ii];
    }

    return s;
  }

	/////////////////////////////////////////////////////////////////////////////
  std::ostream& operator<< ( std::ostream& s, cst_parameters const & d ) {
    s << "\t\tCST PARAMETER INFORMATION:" << std::endl;
    s << "\t\t\tdistance_AB=" << d.disAB << " sd="<<d.disAB_sd <<" k=" <<d.disAB_k
      << " is_covalent= "<<d.is_covalent << " allow_move= " <<d.move_disAB<< std::endl;
    s << "\t\t\tangle_A=" << d.angA << " sd="<<d.angA_sd <<" k=" <<d.angA_k
      << " periodic= "<<d.angA_periodic << " allow_move= " <<d.move_angA<<std::endl;
    s << "\t\t\tangle_B=" << d.angB << " sd="<<d.angB_sd <<" k=" <<d.angB_k
      << " periodic= "<<d.angB_periodic << " allow_move= " <<d.move_angB<<std::endl;
    s << "\t\t\ttorsion_A=" << d.dihA << " sd="<<d.dihA_sd <<" k=" <<d.dihA_k
      << " periodic= "<<d.dihA_periodic << " func= " << d.dihA_func
			<< " allow_move= " <<d.move_dihA<<std::endl;
    s << "\t\t\ttorsion_B=" << d.dihB << " sd="<<d.dihB_sd <<" k=" <<d.dihB_k
      << " periodic= "<<d.dihB_periodic << " func= " << d.dihB_func
			<< " allow_move= " <<d.move_dihB<<std::endl;
    s << "\t\t\ttorsion_AB=" << d.dihAB << " sd="<<d.dihAB_sd <<" k=" <<d.dihAB_k
      << " periodic= "<<d.dihAB_periodic << " func= " << d.dihAB_func
			<< " allow_move= " <<d.move_dihAB<<std::endl;
    return s;
  }

	/////////////////////////////////////////////////////////////////////////////
  std::ostream& operator<< ( std::ostream& s, cst_conformers const & d ) {
		//float const rad2deg( numeric::conversions::degrees(1.0) );
    s << "\t\tCST CONFORMER INFORMATION:" << std::endl;
    s << "\t\t\tdistance_AB=" << d.disAB << " conf_sd="<<d.disAB_conf_sd<<" conf_N="
      <<d.disAB_conf_N <<std::endl;
    s << "\t\t\tangle_A=" << d.angA <<" conf_sd=" <<d.angA_conf_sd<<" conf_N="
			<<d.angA_conf_N <<" ang_N="<< d.angA_N <<std::endl;
    s << "\t\t\tangle_B=" << d.angB <<" conf_sd=" <<d.angB_conf_sd<<" conf_N="
			<<d.angB_conf_N <<" ang_N=" << d.angB_N <<std::endl;
    s << "\t\t\ttorsion_A=" << d.dihA <<" conf_sd=" <<d.dihA_conf_sd<<" conf_N="
			<<d.dihA_conf_N <<" ang_N=" << d.dihA_N <<std::endl;
    s << "\t\t\ttorsion_B=" << d.dihB <<" conf_sd=" <<d.dihB_conf_sd<<" conf_N="
			<<d.dihB_conf_N <<" ang_N=" <<d.dihB_N <<std::endl;
    s << "\t\t\ttorsion_AB=" << d.dihAB <<" conf_sd=" <<d.dihAB_conf_sd<<" conf_N="
			<<d.dihAB_conf_N <<" ang_N="<<d.dihAB_N <<std::endl;
    return s;
  }

	/////////////////////////////////////////////////////////////////////////////
  std::ostream& operator<< ( std::ostream& s, cst_descriptor const & d ) {
    s << "\tCONSTRAINT INFORMATION:" << std::endl;
    s << "\t\tTEMPLATE_A:\n"<<d.template_A<<std::endl;
    s << "\t\tTEMPLATE_B:\n"<<d.template_B<<std::endl;
    if ( d.fix_coord ) {
      s << "\t\tFIX_COORD:\n\t\t\t\t"<<d.coord.x()<<" "<<d.coord.y()<<" "<<d.coord.z()<<std::endl;
    }
		s << "\n\t\tCST_PARAMETER:\n"<<d.cst_parameter<<std::endl;
		s << "\n\t\tCST_CONFORMER:\n"<<d.cst_conformer<<std::endl;
    return s;
  }

	/////////////////////////////////////////////////////////////////////////////
  std::ostream& operator<< ( std::ostream& s, cst_set_descriptor const & d ) {

    s << "SET of CONSTRAINTS:" << std::endl;
    for( int ii=0, ie=d.cst_list.size(); ii<ie; ii++ ) {
      s << "CONSTRAINT No. " << ii <<std::endl;
      s << d.cst_list[ii];
    }
    s << std::endl;
    return s;
  }

	//lin read the old enzyme constraint format
	bool cst_set_descriptor::read_enzyme_constraint(std::string const & filename_in )
	{

		//local variables
		std::string pdb_card; // first word of each line in a pdb file
		std::string atom_name; // atom name such as CA
		std::string res_name; // amino acid in three letter code
		int pdb_number;
		int pdb_res; // residue number in pdb file
		float xpos,ypos,zpos; // coordinates
		float occ,temp;
		int atom_type_tmp;
		std::vector<int> res_list_tmp;

		int constraint_count, virtual_atom_count;
		int ta1_id, ta2_id, ta3_id;
		bool stat_constraint,use_base_center;
		float weight, dis, dis_sd, dis_k, ang, ang_sd, ang_k, dih, dih_sd, dih_k;
		int dis_N, ang_N, dih_N, base_ang_N, base_dih_N, base2_dih_N;
		float base_ang, base_ang_sd, base_ang_k, base_dih, base_dih_sd, base_dih_k;
		float base2_dih, base2_dih_sd, base2_dih_k;

		std::ifstream file;
		std::istringstream line_stream;
		file.clear();
		file.open( filename_in.c_str() );
	//		utility::io::izstream file( filename_in );
		if ( !file ) {
      std::cout << "cst_set_descriptor::read_enzyme_constraint - unable to open cstfile " << filename_in  << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    } else {
      std::cout << "cst_set_descriptor::read_enzyme_constraint " << filename_in  << std::endl;
		}

		std::string line;
		constraint_count = 0;
		virtual_atom_count = 0;
		//cst_list.clear();		//clear constraint list

		while (file){
			getline(file, line);
			line_stream.clear();
			line_stream.str(line);
			line_stream.seekg( std::ios_base::beg );
			line_stream >> bite( 6, pdb_card ) >> bite( 6, pdb_number ) >>
				bite( 4, atom_name );
			if (pdb_card=="HETATM"){ // is hetero
				atom_from_virtual_name( atom_name, atom_type_tmp, res_list_tmp );
				//std::cout<<"DEBUG: "<< atom_type_tmp<<std::endl;
				if ( atom_type_tmp > 0 ) { // is virtual
					std::cout<<"ATOM NAME "<<atom_name<<std::endl;
					line_stream >> skip(1) >> bite( 3, res_name ) >>
						skip( 4 ) >> bite( 2, pdb_res ) >>
						skip(3) >> bite( 1, stat_constraint ) >>
						bite( 8, xpos ) >> bite( 8, ypos ) >> bite( 8, zpos ) >>
						bite( 6, occ ) >> bite( 6, temp ) >>
						skip( 3 ) >> bite( 1, use_base_center ) >> bite( 8, weight ) >>
						bite ( 4, ta1_id ) >> bite ( 4, ta2_id ) >> bite ( 4, ta3_id ) >>
						bite ( 6, dis ) >> bite ( 6, dis_sd) >> bite ( 6, dis_k) >> bite( 6, dis_N ) >>
						bite ( 6, ang ) >> bite ( 6, ang_sd) >> bite ( 6, ang_k) >> bite( 6, ang_N ) >>
						bite ( 6, dih ) >> bite ( 6, dih_sd) >> bite ( 6, dih_k) >> bite( 6, dih_N ) >>
						bite ( 6, base_ang ) >> bite ( 6, base_ang_sd) >> bite ( 6, base_ang_k) >> bite( 6, base_ang_N ) >>
						bite ( 6, base_dih ) >> bite ( 6, base_dih_sd) >> bite ( 6, base_dih_k) >> bite( 6, base_dih_N ) >>
						bite ( 6, base2_dih ) >> bite ( 6, base2_dih_sd) >> bite ( 6, base2_dih_k) >> bite( 6, base2_dih_N ) >>
						skip;
					virtual_atom_count++;
					if (pdb_res !=0) { // is constraint
						cst_descriptor cst_index;
						numeric::xyzVector_int b(0);
						b.assign( ta1_id, ta2_id, ta3_id );
						cst_index.template_A.insert_template_id_list(b);
						cst_index.template_A.insert_res_list(param_aa::ligand_aa_vector[1]);
						cst_index.template_A.insert_seqpos_list
							( misc::ints::total_residue + param_aa::has_ligand_no(param_aa::ligand_aa_vector[1]) );

						cst_index.cst_parameter.set_all_move(true);
						cst_index.cst_parameter.is_covalent = bool(dis_N);
						if( dis_k != 0.0 ) {
							cst_index.cst_parameter.disAB = dis;
							cst_index.cst_parameter.disAB_sd = dis_sd;
							cst_index.cst_parameter.disAB_k = weight*dis_k;
						}
						if( ang_k != 0.0 ) {
							cst_index.cst_parameter.angA = ang ;
							if( ang_sd == 180.0 ) ang_sd = 10.0 ;
							cst_index.cst_parameter.angA_sd = ang_sd ;
							cst_index.cst_parameter.angA_k = weight*ang_k;
							cst_index.cst_parameter.angA_periodic = 360./float(ang_N+1);
						}
						if( base_ang_k != 0.0 ) {
							cst_index.cst_parameter.angB = base_ang ;
							if( base_ang_sd == 180.0 ) base_ang_sd = 10.0 ;
							cst_index.cst_parameter.angB_sd = base_ang_sd ;
							cst_index.cst_parameter.angB_k = weight*base_ang_k;
							cst_index.cst_parameter.angB_periodic = 360./float(base_ang_N+1);
						}
						if( dih_k != 0.0 ) {
							cst_index.cst_parameter.dihA = dih ;
							if( dih_sd == 180.0 ) dih_sd = 10.0 ;
							cst_index.cst_parameter.dihA_sd = dih_sd ;
							cst_index.cst_parameter.dihA_k = weight*dih_k;
							cst_index.cst_parameter.dihA_periodic = 360./float(dih_N+1);
							cst_index.cst_parameter.dihA_func = "HARMONIC";
						}
						if( base_dih_k != 0.0 ) {
							cst_index.cst_parameter.dihAB = base_dih ;
							if( base_dih_sd == 180.0 ) base_dih_sd = 10.0 ;
							cst_index.cst_parameter.dihAB_sd = base_dih_sd ;
							cst_index.cst_parameter.dihAB_k = weight*base_dih_k;
							cst_index.cst_parameter.dihAB_periodic = 360./float(base_dih_N+1);
							cst_index.cst_parameter.dihAB_func = "HARMONIC";
						}
						if( base2_dih_k != 0.0 ) {
							cst_index.cst_parameter.dihB = base2_dih ;
							if( base2_dih_sd == 180.0 ) base2_dih_sd = 10.0 ;
							cst_index.cst_parameter.dihB_sd = base2_dih_sd ;
							cst_index.cst_parameter.dihB_k = weight*base2_dih_k;
							cst_index.cst_parameter.dihB_periodic = 360./float(base2_dih_N+1);
							cst_index.cst_parameter.dihB_func = "HARMONIC";
						}
						if( enzyme::safety_check ) {
							//std::cout << "DEBUG: PDB_RES " << pdb_res << std::endl;
							std::cout << "occ "<<occ<<" temp "<<temp<<" weight " << weight << " stat " << stat_constraint <<
							" center_base " << use_base_center <<
							" ta1 " << ta1_id << " ta2 " << ta2_id << " ta3 " << ta3_id <<
							" dis " << dis << " dis_sd " << dis_sd << " dis_k " << dis_k << " dis_N " << dis_N <<
							" ang " << ang << " ang_sd " << ang_sd << " ang_k " << ang_k << " ang_N " << ang_N <<
							" dih " << dih << " dih_sd " << dih_sd << " dih_k " << dih_k << " dih_N " << dih_N <<
							" base_ang " << base_ang << " base_ang_sd " << base_ang_sd << " base_ang_k " << base_ang_k << " base_ang_N " << base_ang_N <<
							" base_dih " << base_dih << " base_dih_sd " << base_dih_sd << " base_dih_k " << base_dih_k << " base_dihedral_N " << base_dih_N <<
							" base2_dih " << base2_dih << " base2_dih_sd " << base2_dih_sd << " base2_dih_k " << base2_dih_k << " base2_dihedral_N " << base2_dih_N <<
							std::endl;
						}
						for(int ii=0, ie=res_list_tmp.size(); ii < ie; ii++ ) {
							cst_index.template_B.insert_res_list( res_list_tmp[ii] );
						}
						cst_index.template_B.insert_atype_list( atom_type_tmp );
						constraint_count++;
						cst_list.push_back(cst_index);
					} // end is constraint
				} // end is virtual
			} // end is hetero
		} // end reading file

		file.close();
		return true;
	}

	/// REMARK BACKBONE TEMPLATE A ASP xxxx MATCH MOTIF B HIS xxxxx  1
	/// REMARK BACKBONE TEMPLATE A ILE xxxx MATCH MOTIF B HIS xxxxx  2
	/// REMARK BACKBONE TEMPLATE A CYS xxxx MATCH MOTIF B HIS xxxxx  3
	void cst_set_descriptor::read_catalytic_map( std::string const & filename_in )
	{
		int seqpos=0;
		int seqpos1, seqpos2, aa;
		unsigned cst_id;
		std::string pdb_card, tag, cata_aa1, cata_aa2;
		char chain_id1,chain_id2;
		std::string filename;
		std::ifstream file;
		std::string line;
		std::istringstream line_stream;

		file.clear();
		file.open( filename_in.c_str() );
		if ( !file ) {
			std::cout << "cst_set_descriptor::get_catalytic_map - unable to open pdb file " << filename_in  << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		else {
			std::cout<<"cst_set_descriptor::get_catalytic_map read the header from the file " << filename_in <<std::endl;
		}

/// REMARK BACKBONE TEMPLATE A ASP xxxx MATCH MOTIF B HIS xxxxx  1
/// REMARK BACKBONE TEMPLATE A LYS   19 MATCH MOTIF B HIS   142  1
		while ( file) {
			getline(file, line);
			line_stream.clear();
			line_stream.str(line);
			line_stream.seekg( std::ios_base::beg );
			line_stream >> pdb_card  >> skip( 10 ) >> tag ;

			if( pdb_card == "REMARK" && tag == "TEMPLATE" ) {
				cstmap_string.push_back(line);
				line_stream >> skip(1) >> bite(1,chain_id1)  >> skip(1) >> bite(3,cata_aa1)
										>> skip(1) >> bite(4,seqpos1) >> skip (13) >> bite(1,chain_id2)
										>> skip(1) >> bite (3,cata_aa2) >> skip(1) >> bite(5,seqpos2)
										>> skip(1) >> bite(2,cst_id) >> skip ;
				cst_id--; //begin with 1

				if( cst_id >= cst_list.size() ) continue;

				if( cst_set_ns::debug_output ) {
					std::cout << "catatlytic_map: res1 "<<seqpos1<<" "<<chain_id1<<" "<<cata_aa1
									<<" res2 "<<seqpos2<<" "<<chain_id2<<" "<<cata_aa2
									<<" virtual_cst_id: " <<cst_id<<std::endl;
				}
				num_from_res3(cata_aa1,aa);
				cst_list[cst_id].template_A.insert_res_list(aa);
				if( !param_aa::is_ligand( aa ) && files_paths::use_pdb_numbering ) {
					pdbres2rosettares( chain_id1, seqpos1, seqpos );
				} else {
					seqpos = seqpos1;
				}
				cst_list[cst_id].template_A.insert_seqpos_list(seqpos);

				num_from_res3(cata_aa2,aa);
				cst_list[cst_id].template_B.insert_res_list(aa);
				if( !param_aa::is_ligand( aa ) && files_paths::use_pdb_numbering ) {
					pdbres2rosettares( chain_id2, seqpos2, seqpos );
				} else {
					seqpos = seqpos2;
				}
				cst_list[cst_id].template_B.insert_seqpos_list(seqpos);

			} // end read remark line
		} //end read file
		file.close();
		return;
	}

	/// REMARK BACKBONE TEMPLATE A ASP xxxx MATCH MOTIF B HIS xxxxx  1
	/// REMARK BACKBONE TEMPLATE A ILE xxxx MATCH MOTIF B HIS xxxxx  2
	/// REMARK BACKBONE TEMPLATE A CYS xxxx MATCH MOTIF B HIS xxxxx  3
	void cst_set_descriptor::read_old_catalytic_map( std::string const & filename_in )
	{
		int seqpos=0;
		int seqpos1, aa;
		unsigned cst_id;
		std::string pdb_card, tag, cata_aa;
		char chain_id;
		std::string filename;
		std::ifstream file;
		std::string line;
		std::istringstream line_stream;

		file.clear();
		file.open( filename_in.c_str() );
		if ( !file ) {
			std::cout << "cst_set_descriptor::get_catalytic_map - unable to open pdb file " << filename_in  << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		else {
			std::cout<<"cst_set_descriptor::get_catalytic_map read the header from the file " << filename_in <<std::endl;
		}

/// REMARK BACKBONE TEMPLATE A ***   19 MATCH MOTIF * HIS xxxxx  1
		while ( file) {
			getline(file, line);
			line_stream.clear();
			line_stream.str(line);
			line_stream.seekg( std::ios_base::beg );
			line_stream >> pdb_card  >> skip( 10 ) >> tag ;

			if( pdb_card == "REMARK" && tag == "TEMPLATE" ) {
				cstmap_string.push_back(line);
				line_stream >> skip(1) >> bite(1,chain_id)  >> skip(5) >> bite(4,seqpos1)
										>> skip (15) >> bite (3,cata_aa) >> skip(7) >> bite(2,cst_id) >> skip ;
				cst_id--; //begin with 1

				if( cst_id >= cst_list.size() ) continue;

				//std::cout << "DEBUG: res "<<seqpos1<<" "<<chain_id<<" "<<cata_aa
				//					<<" virtual_cst_id: " <<cst_id<<std::endl;
				//lin assume the template_A is ligand
				num_from_res3(cata_aa,aa);
				if( aa > 0 ) {
				  cst_list[cst_id].template_B.res_list.clear();
				  cst_list[cst_id].template_B.insert_res_list(aa);
				}
				if( !param_aa::is_ligand( aa ) && files_paths::use_pdb_numbering ) {
					pdbres2rosettares( chain_id, seqpos1, seqpos );
				} else {
					seqpos = seqpos1;
				}
				cst_list[cst_id].template_B.insert_seqpos_list(seqpos);

			} // end read remark line
		} //end read file
		file.close();
		return;
	}

	void cst_set_descriptor::write_catalytic_map( std::ostream & out ) const {
		for( int ii=0, ie=cstmap_string.size(); ii<ie; ii++ ) {
			out << cstmap_string[ii] << std::endl;
		}
	}

	void
	atom_from_virtual_name(
     std::string const & name, int & type,
		 std::vector<int> & res_list ) {

		using namespace param_aa;

		int act_tmp;

		if( name == "VOOC" ) {//  1 ASP/GLU
			atom_num_from_atom_char( "OOC ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_asp);
			res_list.push_back(aa_glu);
		} else if( name == "VCOO" ) {//  2 ASP/GLU
			atom_num_from_atom_char( "COO ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_asp);
			res_list.push_back(aa_glu);
		} else if( name == "VOCN" ) {//  3 ASN/GLN
			atom_num_from_atom_char( "ONH2", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_asn);
			res_list.push_back(aa_gln);
		} else if( name == "VNOC" ) {//  4 ASN/GLN
			atom_num_from_atom_char( "NH2O", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_asn);
			res_list.push_back(aa_gln);
		} else if( name == "VCON" ) {//  5 ASN/GLN
			atom_num_from_atom_char( "CNH2", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_asn);
			res_list.push_back(aa_gln);
		} else if( name == "VSOG" ) {//  6 SER/THR OG
			atom_num_from_atom_char( "OH  ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_ser);
			res_list.push_back(aa_thr);
		} else if( name == "VSCB" ) {//  7 SER/THR CB
			atom_num_from_atom_char( "CH2 ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_ser);
			res_list.push_back(aa_thr);
		} else if( name == "VCSG" ) {//  8 CYS SG
			atom_num_from_atom_char( "S   ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_cys);
		} else if( name == "VCCB" ) {//  9 CYS CB
			atom_num_from_atom_char( "CH2 ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_cys);
		} else if( name == "VRNH" || name == "VRNE") {// 10,11 ARG NH
			atom_num_from_atom_char( "Narg", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_arg);
		} else if( name == "VKNZ" ) {// 12 LYS NZ
			atom_num_from_atom_char( "Nlys", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_lys);
  	//if( name == "VKCE" ) {// 13 LYS CE not used any more
		} else if( name == "VHND" ) {// 14 HIS ND
			atom_num_from_atom_char( "Nhis", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
		} else if( name == "VHNE" ) {// 15 HIS NE
			atom_num_from_atom_char( "Ntrp", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
		} else if( name == "VHCB" ) {// 16 HIS CB
			atom_num_from_atom_char( "CH2 ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
		} else if( name == "VHPO" ) {// 17 HPOL
			atom_num_from_atom_char( "Hpol", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
			res_list.push_back(aa_lys);
			res_list.push_back(aa_trp);
			res_list.push_back(aa_tyr);
			res_list.push_back(aa_ser);
			res_list.push_back(aa_thr);
			res_list.push_back(aa_asn);
			res_list.push_back(aa_gln);
		} else if( name == "VHAR" ) { // 18 HARO
			atom_num_from_atom_char( "Haro", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
			res_list.push_back(aa_trp);
			res_list.push_back(aa_phe);
			res_list.push_back(aa_tyr);
		} else if( name == "VYOH" ) {// 19 TYR OH
			atom_num_from_atom_char( "OH  ", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_tyr);
		} else if( name == "VARO" ) {// 20 AROC
			atom_num_from_atom_char( "aroC", type, act_tmp );
			res_list.clear();
			res_list.push_back(aa_his);
			res_list.push_back(aa_trp);
			res_list.push_back(aa_phe);
			res_list.push_back(aa_tyr);
		} else {
			//std::cout<<" Warning: unkown virtual name "<< name <<std::endl;
			type=0;
		}
	}

	void pdbres2rosettares( char const chain_id, int const pdbres, int & seqpos ){
		using namespace pdb;

		assert( files_paths::use_pdb_numbering );
		for( int ii=1, ie=misc::ints::total_residue; ii<=ie; ii++ ) {
			if( res_chain( ii ) == chain_id && pdb_res_num( ii ) == pdbres ) {
				seqpos = ii;
				return;
			}
		}
		std::cout<<"ERROR: pdbres2rosettares: not found res: "
						 <<chain_id<<" "<<pdbres<<std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string get_filename( std::string const & filename_in, std::string const & default_name ) {
    using namespace files_paths;

		std::string const filename ( start_path + filename_in );

    std::ifstream file;
    file.clear();
    file.open( filename.c_str() );

		if( !file ) {
      return default_name;
    } else {
			file.close();
      return start_path + filename_in;
    }
	}


}
