// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10231 $
//  $Date: 2006-09-07 12:59:56 -0700 (Thu, 07 Sep 2006) $
//  $Author: jiangl $


//////////////////////////////////////////////

// Rosetta Headers
#include "cst_functions.h"

// ObjexxFCL Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cassert>
#include <cmath>
#include <iostream>

namespace cst_functions_ns {

  ////////////////////////////////////////
  // Cst function manager
	CstE_manager CstE_DB;
	std::vector< Cst_function* > CstE_manager::func_map_;
  bool CstE_manager::func_map_init( false );

  void cst_memory_cleanup() { CstE_DB.erase(); }

  /////////////////////////////////////////////////////////////////////////////
  // static function
  void CstE_manager::initialize_func_map() {
    if ( func_map_init ) return;
    erase();
    func_map_init = true;
    func_map_.push_back( new Cst_harmonic_function );
    func_map_.push_back( new Cst_periodic_function );
		//std::cout<<"CST_FUNC_DB:: func_map_list: "<<func_map_[0]->func_name()
		//				 <<" "<< func_map_[1]->func_name()<<std::endl;
  }

  // constructor:
  CstE_manager::CstE_manager() {}

  // destructor
  CstE_manager::~CstE_manager() {}

  const Cst_function*
  CstE_manager::get_cst_function( std::string const & name )
  {
    initialize_func_map();
    for( int ii=0,ie=func_map_.size(); ii<ie; ++ii ) {
      if( func_map_[ii]->func_name()==name ){
				return func_map_[ii];
      }
    }
    std::cout<<"CstE_manager::GetCstFunction "<<name<<" doesnt exist"<<std::endl;
    assert(false);
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return NULL;
  }

  const Cst_function*
  CstE_manager::get_cst_function( Cst_func_type const type )
  {
    initialize_func_map();
    for( int ii=0,ie=func_map_.size(); ii<ie; ++ii ) {
      if( func_map_[ii]->func_type()==type ){
				return func_map_[ii];
      }
    }
    std::cout<<"CstE_manager::GetCstFunction "<<type<<" doesnt exist"<<std::endl;
    assert(false);
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return NULL;
  }

	const Cst_function*
  GetCstFunction( std::string const & name )
  {
		return CstE_DB.get_cst_function( name );
  }

	const Cst_function*
  GetCstFunction( Cst_func_type const type )
  {
		return CstE_DB.get_cst_function( type );
  }

  /////////////////////////////////////////////////////////////////
	// general harmonic function:
  //        x is the delta value ( val - val0 )
  //        y = k1 * ( x + sd ) * ( x + sd )    if x < -sd
  //        y = 0                               if  -sd <= x <= sd
	//        y = k2 * ( x - sd ) * ( x - sd )    if x > sd
  float Cst_harmonic_function::cstE(
	       float const delta_val,
	       float const sd,
	       float const k1,
	       float const k2
	       ) const {
    assert( sd >=0 );

    if( k1 == 0 && k2 == 0 ) return 0;

    if( delta_val > sd ) {
      return k2 * ( delta_val - sd ) * ( delta_val - sd );
    } else if (  delta_val < -sd ) {
      return k1 * ( delta_val + sd ) * ( delta_val + sd );
    } else {
      return 0;
    }
  }

  ////////////////////////////////////////////////////////////////////
	// general harmonic function:
  //        x is the delta value ( val - val0 )
  //        y = k1 * ( x + sd ) * ( x + sd )    if x < -sd
  //        y = 0                               if  -sd <= x <= sd
	//        y = k2 * ( x - sd ) * ( x - sd )    if x > sd
  float Cst_harmonic_function::cstE_deriv(
         float const delta_val,
	       float const sd,
	       float const k1,
	       float const k2
	       ) const {

    assert( sd >=0  );

    if( k1 == 0 && k2 == 0 ) return 0;

    if( delta_val > sd ) {
      return 2.0f * k2 * ( delta_val - sd );
    } else if (  delta_val < -sd ) {
      return 2.0f * k1 * ( delta_val + sd );
    } else {
      return 0;
    }
  }


  ////////////////////////////////////////////////////////////////////
	// periodic function:
  // y = 0.5 * k * ( 1 - cos( n * (x - x0) ) )
  float Cst_periodic_function::cstE(
	       float const val,
				 float const val0,
	       float const n_periodic,
	       float const k
	       ) const {

    if( k == 0 || n_periodic == 0 ) return 0;
		//std::cout<<"periodic: "<<val<<" "<<val0<<" "<<n_periodic<<" "<<k<<std::endl;
		return 0.5f * k * ( 1 - cos( n_periodic * ( val - val0 ) ) );
		// compare with charmm formula y = k * ( 1 + cos( n * x - x0 ) )
		//return k * ( 1 + cos( n_periodic * val - val0 ) );

  }

  ////////////////////////////////////////////////////////////////////
	// periodic function:
  // y = k * ( 1 - cos( n * (x - x0) ) )
  float Cst_periodic_function::cstE_deriv(
         float const val,
				 float const val0,
	       float const n_periodic,
	       float const k
	       ) const {

    if( k == 0 || n_periodic == 0 ) return 0;
		return 0.5f * k * n_periodic * sin ( n_periodic * ( val - val0 ) );
		// compare with charmm formula y = k * ( 1 + cos( n * x - x0 ) )
		//return k * n_periodic * sin ( n_periodic * val - val0 );
  }


}

