// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19111 $
//  $Date: 2007-12-13 19:13:18 -0500 (Thu, 13 Dec 2007) $
//  $Author: johnk $

//////////////////////////////////////////////
#ifndef INCLUDED_cst_set
#define INCLUDED_cst_set

// Rosetta Headers
#include "cst_functions.h"
#include "kin_coords.h"
#include "pose_fwd.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.io.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <map>
#include <utility>

// Forward Declarations
namespace packer_cst_ns { class Packer_cst_set; }

//////////////////////////////////
// the Cst_set object holds data that can be used to constrain
// a structure
//



namespace cst_set_ns {

	//lin debug flag
	extern bool debug_output;

	//lin move to the cst_functions.cc
// 	////////////////////////////////////////////////////////////////////
// 	//lin a single constraint potential and derivative
// 	////////////////////////////////////////////////////////////////////
// 	inline float cst_E(
// 						float const delta_val,
// 						float const val_sd,
// 						float const k
// 						)
// 	{
// 		assert( val_sd >=0 );

// 		if( k == 0 ) return 0;

// 		float const delta_tmp( std::abs( delta_val ) );
// 		float val_diff = delta_tmp - val_sd;

// 		if (val_diff > 0) {
// 			return k * val_diff * val_diff;
// 		} else {
// 			return 0;
// 		}
// 	}

// 	inline float cst_E_deriv(
// 									float const delta_val,
// 									float const val_sd,
// 									float const k
// 									)
// 	{
// 		assert( val_sd >=0  );

// 		if( k == 0 ) return 0;

// 		if( delta_val > val_sd ) {
// 			return 2.0f * k * ( delta_val - val_sd );
// 		} else if ( delta_val < -val_sd ) {
// 			return 2.0f * k * ( delta_val + val_sd );
// 		} else {
// 			return 0;
// 		}
// 	}

	////////////////////////////////////////////////////////////////////
	// a single atompair distance constraint
	//
	// two methods: score(r) and dscore_dr(r), both function of distance
	class Cst {
	public:
		//lin score
		float score( float const r ) const;

		float dscore_dr( float const r ) const;

	  float score(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2
			) const ;

		// constructor
		Cst():
			r0( 0.0 ),
			r_sd( 0.0 ),
			weight1( 1.0 ),
			weight2( 1.0 ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{ }
		Cst( float const r0_in ):
			r0( r0_in ),
			r_sd( 0.0 ),
			weight1( 1.0 ),
			weight2( 1.0 ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{}
		Cst( float const r0_in, float const weight_in ):
			r0( r0_in ),
			r_sd( 0.0 ),
			weight1( weight_in ),
			weight2( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{}
		Cst( float const r0_in, float const r_sd_in, float const weight_in ):
			r0( r0_in ),
			r_sd( r_sd_in ),
			weight1( weight_in ),
			weight2( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{}
		Cst( float const r0_in, float const r_sd_in, float const weight1_in, float const weight2_in ):
			r0( r0_in ),
			r_sd( r_sd_in ),
			weight1( weight1_in ),
			weight2( weight2_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{}

		inline void set_sd( float const value ) { r_sd = value; };

    friend std::ostream& operator<< ( std::ostream& s, const Cst& d );
    friend bool operator== ( const Cst& d1, const Cst& d2 );

	private:
		float r0;
		float r_sd;
		float weight1;
		float weight2;
		const cst_functions_ns::Cst_function* func_p;
	};

	typedef std::map< kin::Atom_id, Cst > Cst_map;
	typedef std::map< kin::Atom_id, Cst_map > Cst_map_map;


	/////////////////////////////////////////////////////////////////////////////
	// Rsd_rsd_cst
	/////////////////////////////////////////////////////////////////////////////

	// for packing and rotamer_trials it's convenient to lump constraints
	// by residue pair

	// in the future would like to generalize this to include
	// angle and torsion constraints -- right now just a wrapper for distance
	// constraints
	//
	// probably will necessitate a class hierarchy
	//
	// I'm envisioning the Coords object being passed in could
	// be a special one that is compatible packing/rotamer_trials
	// situations, ie you could specify a subset of the coords from
	// a rotcoord array and use background xyz for the rest...

	class Rsd_rsd_cst {
	public:
// 		inline
// 		float
// 		score(
// 			int pos1,
// 			int pos2,
// 			kin::Coords const & coords
// 		);

		// constructor
		Rsd_rsd_cst(
			int const a1,
			int const a2,
			Cst const & cst_in
		):
			atomno1( a1 ),
			atomno2( a2 ),
			cst( cst_in )
		{}

    friend std::ostream& operator<< ( std::ostream& s, const Rsd_rsd_cst& d );

		//private:
		int atomno1, atomno2;
		Cst cst;
	};


	/////////////////////////////////////////////////////////////////////////////
	// Torsion_cst
	/////////////////////////////////////////////////////////////////////////////

	// simple quadratic (unsigned!!!!) torsion-angle potential
	//
	// values in radians
	class Torsion_cst {
	public:
		// score
		float
		score(
			kin::Coords const & coords
		) const;


		// score
		float
		score(
			numeric::xyzVector_float const & atom1,
			numeric::xyzVector_float const & atom2,
			numeric::xyzVector_float const & atom3,
			numeric::xyzVector_float const & atom4
		) const  ;

    // accessor for private atoms
    void
    extract_atom_ids(
      kin::Atom_id & a1,
      kin::Atom_id & a2,
      kin::Atom_id & a3,
      kin::Atom_id & a4
    ) const;

		// atom deriv
		void
		fill_f1_f2(
			kin::Atom_id const atom,
			kin::Coords const & coords,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;


		//c-tor
		Torsion_cst(
			kin::Atom_id const & a1,
			kin::Atom_id const & a2,
			kin::Atom_id const & a3,
			kin::Atom_id const & a4,
			float const theta0_in,
			float const weight_in
			):
			atom1(a1),
			atom2(a2),
			atom3(a3),
			atom4(a4),
			theta0( theta0_in ),
			theta_sd( 0.0 ),
			periodic_val( 2.0*numeric::constants::f::pi ),
			weight( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{
			if ( theta0 < - numeric::constants::f::pi || theta0 > numeric::constants::f::pi ) {
				std::cout << "angle constraints should be in radians: (-pi,pi)" << theta0 <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		//c-tor
		Torsion_cst(
			kin::Atom_id const & a1,
			kin::Atom_id const & a2,
			kin::Atom_id const & a3,
			kin::Atom_id const & a4,
			float const theta0_in,
			float const theta_sd_in,
			float const periodic_value_in,
			float const weight_in
			):
			atom1(a1),
			atom2(a2),
			atom3(a3),
			atom4(a4),
			theta0( theta0_in ),
			theta_sd( theta_sd_in ),
			periodic_val( periodic_value_in ),
			weight( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{
			if ( theta0 < - numeric::constants::f::pi || theta0 > numeric::constants::f::pi ) {
				std::cout << "angle constraints should be in radians: (-pi,pi)" << theta0 <<
					std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

		//c-tor
		Torsion_cst(
			kin::Atom_id const & a1,
			kin::Atom_id const & a2,
			kin::Atom_id const & a3,
			kin::Atom_id const & a4,
			float const theta0_in,
			float const theta_sd_in,
			float const periodic_value_in,
			float const weight_in,
			std::string const & func_name
			):
			atom1(a1),
			atom2(a2),
			atom3(a3),
			atom4(a4),
			theta0( theta0_in ),
			periodic_val( periodic_value_in ),
			weight( weight_in )
		{
			using namespace cst_functions_ns;

			func_p = GetCstFunction(func_name) ;
			if( func_p->func_type() == HARMONIC ) {
				theta_sd = theta_sd_in ;
			} else {
				theta_sd = 0.0;
			}

			if ( theta0 < - numeric::constants::f::pi || theta0 > numeric::constants::f::pi ) {
				std::cout << "angle constraints should be in radians: (-pi,pi)" << theta0 <<
					std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

 		inline void set_sd( float const value ) { theta_sd = value; };

		friend std::ostream& operator<< ( std::ostream& s, const Torsion_cst& d );
    friend bool operator== ( const Torsion_cst& d1, const Torsion_cst& d2 );

	private:
		// functions

		// this guy doesnt need member data --> static
		static
		void
		helper(
			numeric::xyzVector_float const & M,
			numeric::xyzVector_float const & v,
			numeric::xyzVector_float const & w,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		);

		static
		void
		p1_cosine_deriv(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2,
			numeric::xyzVector_float const & p3,
			numeric::xyzVector_float const & p4,
			float & x,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		);

		static
		void
		p2_cosine_deriv(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2,
			numeric::xyzVector_float const & p3,
			numeric::xyzVector_float const & p4,
			float & x,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		);


		// data
		kin::Atom_id atom1, atom2, atom3, atom4;
		float theta0; // well center: (0,2pi)
		float theta_sd; // tolerant derivation
		float periodic_val; // periodic value in degree
		float weight;
		const cst_functions_ns::Cst_function* func_p;

	};

	/////////////////////////////////////////////////////////////////////////////
	// Angle_cst
	/////////////////////////////////////////////////////////////////////////////

	class Angle_cst {
	public:
		// score
		float
		score(
			kin::Coords const & coords
		) const;

		// score
		float
		score(
			numeric::xyzVector_float const & atom1,
			numeric::xyzVector_float const & atom2,
			numeric::xyzVector_float const & atom3
		) const;

		// accessor for private atoms
		void
		extract_atom_ids(
			kin::Atom_id & a1,
			kin::Atom_id & a2,
			kin::Atom_id & a3
		) const;

		// atom deriv
		void
		fill_f1_f2(
			kin::Atom_id const atom,
			kin::Coords const & coords,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;


		//c-tor
		Angle_cst(
			kin::Atom_id const & a1,
			kin::Atom_id const & a2,
			kin::Atom_id const & a3,
			float const theta0_in,
			float const weight_in
			):
			atom1(a1),
			atom2(a2),
			atom3(a3),
			theta0( theta0_in ),
			theta_sd( 0.0 ),
			periodic_val( numeric::constants::f::pi ),
			weight( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{
			if ( theta0 < 0 || theta0 > numeric::constants::f::pi ) {
				std::cout << "angle constraints should be in radians: (0,pi)" << theta0 <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		//c-tor
		Angle_cst(
			kin::Atom_id const & a1,
			kin::Atom_id const & a2,
			kin::Atom_id const & a3,
			float const theta0_in,
			float const theta_sd_in,
			float const periodic_value_in,
			float const weight_in
			):
			atom1(a1),
			atom2(a2),
			atom3(a3),
			theta0( theta0_in ),
			theta_sd( theta_sd_in ),
			periodic_val( periodic_value_in ),
			weight( weight_in ),
			func_p( GetCstFunction(cst_functions_ns::HARMONIC) )
		{
			if ( theta0 < 0 || theta0 > numeric::constants::f::pi ) {
				std::cout << "angle constraints should be in radians: (0,pi)" << theta0 <<
					std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

 		inline void set_sd( float const value ) { theta_sd = value; };

    friend std::ostream& operator<< ( std::ostream& s, const Angle_cst& d );
    friend bool operator== ( const Angle_cst& d1, const Angle_cst& d2 );

	private:
		// functions

		// this guy doesnt need member data --> static
		static
		void
		p1_theta_deriv(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2,
			numeric::xyzVector_float const & p3,
			numeric::xyzVector_float & f1,
			numeric::xyzVector_float & f2
		);


		static
		void
		helper(
			numeric::xyzVector_float const & M,
			numeric::xyzVector_float const & w,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		);


		void
		p1_deriv(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2,
			numeric::xyzVector_float const & p3,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;


		void
		p2_deriv(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2,
			numeric::xyzVector_float const & p3,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;

		// data
		kin::Atom_id atom1, atom2, atom3;
		float theta0; // well center: (0,2pi)
		float theta_sd; // tolerant derivation
		float periodic_val; // periodic value in degree
		float weight;
		const cst_functions_ns::Cst_function* func_p;
	};

	/////////////////////////////////////////////////////////////////////////////
	// a set of constraints that can be applied to a pose
	// during minimization and scoring
	//
	// right now:
	// 1. atom-atom distance constraints
	// 2. torsion-angle constraints
	// 3. 3-space constraints: tether an atom to a position in space
	//    this only makes sense if the pose is staying fixed in relative space
	//    for example, maybe during a loop-modeling application, or side-chain
	//    only stuff. I'm using it right now for idealizing side-chains

	class Cst_set {
	public:
		///////////////////////////////////////////////////////////////////////////
		// atompair constraints
		void
		add_atompair_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			Cst const & cst,
			bool const verbose = true
		);

		void
		add_atompair_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			float const r0,
			float const weight,
			bool const verbose = true
		);

		void
	  add_atompair_constraint(
		  kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			float const r0,
			float const r0_sd,
			float const weight,
			bool const verbose = true
		);

		void
		remove_atompair_constraints_to_atom(
			kin::Atom_id const & atom1
		);

		int
		count_atompair_constraints() const;

		// do we have any atompair constraints involving this atom?
		inline
		bool
		atompair_constraint_exists(
			kin::Atom_id const & atom
			) const {
			return ( cst_map_by_atom.count( atom ) > 0 );
		}


		inline
		bool
		residue_pair_constraint_exists(
			int const pos1,
			int const pos2
			) const {
			std::pair< int, int >
				key( std::min( pos1, pos2 ), std::max( pos1, pos2 ) );
			return ( cst_map_by_rsd_pair.count( key ) > 0 );
		}

		float
		get_res_res_cstE(
			int const res1,
			int const res2,
			int const aa1,
			int const aav1,
			int const aa2,
			int const aav2,
			FArray2Da_float coord1,
			FArray2Da_float coord2,
			bool const bb1,
			bool const sc1,
			bool const bb2,
			bool const sc2
		) const;


		float
		atompair_cst_score(
			kin::Coords const & coords
		) const;


		void
		atompair_cst_atom_deriv(
			kin::Atom_id const & atom,
			kin::Coords const & coords,
			FArray1D_int const & domain_map,
			float const atompair_cst_weight,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;

		///////////////////////////////////////////////////////////////////////////
		// coordinate constraints
		void add_coordinate_constraint(
			kin::Atom_id const & atom,
			numeric::xyzVector_float const & xyz
		);


		void
		clear_coordinate_constraints()
		{ coord_cst_.clear(); }


		int
		count_coordinate_constraints() const;


		inline
		bool
		coordinate_constraint_exists(
			kin::Atom_id const & atom
		) const
		{ return ( coord_cst_.count( atom ) > 0 ); }


		inline
		numeric::xyzVector_float const &
		coordinate_constraint(
			kin::Atom_id const & atom
		) const;


		float
		coord_cst_score(
			kin::Coords const & coords
		) const;


		void
		coord_cst_atom_deriv(
			kin::Atom_id const & atom,
			kin::Coords const & coords,
			float const coord_cst_weight,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;


		///////////////////////////////////////////////////////////////////////////
		// torsion constraints
		void
		add_rosetta_torsion_constraint(
			int const pos,
			int const torsion,
			float const value
		);

		inline
		bool
		has_kin_torsion_constraint( kin::Torsion_id const & torsion	) const;

		void
		add_kin_torsion_constraint(
			kin::Torsion_id const & torsion,
			float const value,
			float const weight
		);


		int
		count_chi_constraints() const;


		void
		torsion1D_score(
			pose_ns::Pose const & pose,
			float & phipsi_score,
			float & omega_score,
			float & chi_score,
			float & kin_torsion_score
		) const;

		float
		kin_torsion_deriv(
			kin::Torsion_id const & id,
			float const value
		) const;


		float
		rosetta_torsion_deriv(
			int const seqpos,
			int const torsion,
			float const value
		) const;


		///////////////////////////////////////////////////////////////////////////
		// routines for handling angle constraints
		float
		atom_angle_score(
			kin::Coords const & coords
		) const;


		void
		atom_angle_deriv(
			kin::Atom_id const & atom,
			kin::Coords const & coords,
			float const weight,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;

		void
		add_atom_angle_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			float theta0,
			float weight
		);

    void
    add_atom_angle_constraint(
			Angle_cst const & angle_cst
		);

		void
		add_atom_angle_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			float theta0,
			float theta_sd,
			float periodic_value,
			float weight
		);

		///////////////////////////////////////////////////////////////////////////
		// routines for handling geometric (non-DOF) torsion angle constraints
		float
		atom_torsion_score(
			kin::Coords const & coords
		) const;


		void
		atom_torsion_deriv(
			kin::Atom_id const & atom,
			kin::Coords const & coords,
			float const weight,
			numeric::xyzVector_float & F1,
			numeric::xyzVector_float & F2
		) const;

		void
		add_atom_torsion_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			kin::Atom_id const & atom4,
			float theta0,
			float weight
		);

    void
    add_atom_torsion_constraint(
			Torsion_cst const & torsion_cst
    );

		void
		add_atom_torsion_constraint(
			kin::Atom_id const & atom1,
			kin::Atom_id const & atom2,
			kin::Atom_id const & atom3,
			kin::Atom_id const & atom4,
			float theta0,
			float theta_sd,
			float periodic_value,
			float weight
		);

		///////////////////////////////////////////////////////////////////////////
		// manage chainbreak data
		void
		add_chainbreak( int const seqpos, float const weight );

		inline
		bool
		is_chainbreak( int const seqpos ) const;

		inline
		float
		chainbreak_weight( int const seqpos ) const;

		inline
		int
		clear_chainbreak( int const seqpos );

		inline
		void
		clear_all_chainbreaks( );

		// constructor
		Cst_set(){ packer_atompair_weight = 0.0; ptr_references_ = 1; }

		void add_ptr_reference() const;
		void remove_ptr_reference() const;

		// method to assume another Cst_set's constraints
		Cst_set &
		collect( Cst_set const & other_cst );

		//lin add the cst_set to packer_cst
		void
		get_packer_cst(
         pose_ns::Pose const & pose,
				 packer_cst_ns::Packer_cst_set & packer_cst ) const;

		void
		get_packer_cst( packer_cst_ns::Packer_cst_set & packer_cst ) const;

		//lin method for erase atom constraint: dis, angle and torsion cst
		inline void
		erase_atom_constraints() {
			cst_map_by_atom.clear();
			angle_cst_list.clear();
			angle_cst_count.clear();
			torsion_cst_list.clear();
			torsion_cst_count.clear();
		}

		void zero_atompair_cst_sd();

		// stream I/O
		friend std::ostream & operator <<(std::ostream & os, const Cst_set & cs);

	private:
		// atompair constraints
		Cst_map_map cst_map_by_atom;
    std::map< std::pair< int, int >,
              std::vector< Rsd_rsd_cst > > cst_map_by_rsd_pair;
    float packer_atompair_weight;

		// torsion constraints, indexed by <pos,torsion> pairs: (rosetta torsions)
		std::map< std::pair< int, int >, float > rosetta_torsion_cst_;

		// torsion constraints, indexed by kin::Torsion_id info
		std::map< kin::Torsion_id, std::pair< float, float > > kin_torsion_cst_;

		// coordinate constraints
		std::map< kin::Atom_id, numeric::xyzVector_float > coord_cst_;

		// angle constraints
		std::vector< Angle_cst > angle_cst_list;
		std::map< kin::Atom_id, int > angle_cst_count;

		// torsion constraints
		std::vector< Torsion_cst > torsion_cst_list;
		std::map< kin::Atom_id, int > torsion_cst_count;

		// chainbreak constraints -- mapping from cutpoint to weight
		std::map< int, float > chainbreak_weight_;

		// The number of pointers referring to this cst_set. This is 1 if the cst_set has an
		// actual variable name, and is incremented as this cst_set is used in references.
		// This is needed for garbage collection of dynamically allocated "anonymous" cst_sets
		mutable int ptr_references_;

	};


	/////////////////////////////////////////////////////////////////////////////
	// inline functions
	/////////////////////////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////////////////
	inline
	bool
	Cst_set::has_kin_torsion_constraint(
			kin::Torsion_id const & torsion
	) const
	{
		return ( kin_torsion_cst_.count( torsion ) > 0);
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	bool
	Cst_set::is_chainbreak( int const seqpos ) const
	{
		return chainbreak_weight_.count( seqpos );
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	float
	Cst_set::chainbreak_weight( int const seqpos ) const
	{
		assert( chainbreak_weight_.count( seqpos ) );
		return chainbreak_weight_.find( seqpos )->second;
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	int
	Cst_set::clear_chainbreak( int const seqpos )
	{
		return chainbreak_weight_.erase( seqpos );
	}

	/////////////////////////////////////////////////////////////////////////////
	inline
	void
	Cst_set::clear_all_chainbreaks( )
	{
		chainbreak_weight_.clear( );
	}

	// get the map of constraints for this atom:
// 	inline const Cst_map & Cst_set::get_cst_map_by_atom(
// 		kin::Atom_id const & atom
// 	) const
// 	{
// 		Cst_map_map::const_iterator it ( cst_map_by_atom.find( atom ) );
// 		if ( it == cst_map_by_atom.end() ) {
// 			std::cout << "atom not in map, check with has_atom() first!!!" << std::endl;
// 			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 		}
// 		return it->second;
// 	}

	// access a previously stored torsion constraint value
// 	inline float Cst_set::torsion_constraint(
// 		int const pos,
// 		int const torsion
// 	) const
// 	{
// 		std::pair< int, int > key( pos, torsion );
// 		std::map< std::pair< int, int >, float >::const_iterator
// 			it( torsion_cst_.find( key ) );
// 		if ( it == torsion_cst_.end() ) {
// 			std::cout << "torsion constraint does not exist!" << std::endl;
// 			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 		}
// 		return it->second;
// 	}

	// get target position for coordinate constraint
	inline numeric::xyzVector_float const & Cst_set::coordinate_constraint(
		kin::Atom_id const & atom
	) const
	{
		std::map< kin::Atom_id, numeric::xyzVector_float >::const_iterator it
			( coord_cst_.find( atom ) );
		if ( it == coord_cst_.end() ) {
			std::cout << "coordinate constraint doesnt exist" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return it->second;
	}



	// check for existence of a torsion constraint value
// 	inline bool Cst_set::torsion_constraint_exists(
// 		int const pos,
// 		int const torsion
// 	) const
// 	{
// 		std::pair< int, int > key( pos, torsion );
// 		return ( torsion_cst_.count( key ) != 0 );
// 	}

	// helper function
	void insert_atom_pair_constraint(
		Cst_map_map & cst_map_map,
		kin::Atom_id const & pair1,
		kin::Atom_id const & pair2,
		Cst const & cst,
		bool const verbose = true
	);

} // namespace cst_set_ns

#endif
