// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "decoystats_interface.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "decoystats.h"
#include "decoystats_classes.h"
#include "decoystats_ns.h"
#include "decoystats_interface_classes.h"
#include "design.h"
#include "design_structure.h"
#include "docking.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "fullatom_energies.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "make_pdb.h"
#include "misc.h"
#include "namespace_fullatom_flag.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "read_aaproperties.h"
#include "score.h"
#include "template_pack.h"
#include "util_vector.h"
#include "void.h"
#include "void_ns.h"
#include "water_ns.h"

// Numeric Headers
#include <numeric/constants.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <list>
#include <sstream>
#include <string>
#include <vector>

static int const MAX_QUANTIFIERS = 30;

//////////////////////////////////////////////////////////////////////////////
/// @begin set_interface_ds_flag
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_interface_ds_flag( bool setting )
{
	using namespace decoystats;

	interface_ds_flag = setting;
	set_decoystats_flag(true);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_interface_ds_flag
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_interface_ds_flag()
{
	using namespace decoystats;

	return interface_ds_flag;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::interface_ds
///
/// @brief: constructor for class interface_ds
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/14/05
/////////////////////////////////////////////////////////////////////////////////
interface_ds::interface_ds( pose_ns::Pose * input_pose_ptr ) {

	using namespace param;

	pose = input_pose_ptr;

	interface_res.dimension( pose->total_residue() );
	core_interface_res.dimension( pose->total_residue() );
	interface_atom.dimension( param::MAX_ATOM(), pose->total_residue() );
	res_sasa_diff.dimension( pose->total_residue() );
	atom_sasa_diff.dimension( param::MAX_ATOM()(), pose->total_residue() );
	energy_quantifiers.resize(MAX_QUANTIFIERS);

	energies_initialized = false;

	pose->copy_to_misc();

	// make sure energies and SASAs and decoystats Hbond arrays are up to date
	score_set_new_pose();
	score12();
	copy_hbenergies();
	calc_aprox_sasa();
	calc_sasa_pack_score(pose->total_residue(), pose->res(), pose->res_variant(), pose->full_coord());
	calc_sasa_prob();

	define_interface();
	setup_energetic_quantifiers();
	compute_quantifiers();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::define_interface
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::define_interface() {

	using namespace decoystats;
	using namespace param;

	interface_res=false;
	core_interface_res=false;
	interface_atom=false;

	res_sasa_complex.dimension( pose->total_residue() );
	atom_sasa_complex.dimension( MAX_ATOM()(), pose->total_residue() );

	FArray1D_float res_sasa_unbound( pose->total_residue() );
	FArray2D_float atom_sasa_unbound( MAX_ATOM()(), pose->total_residue() );

	float const probe_radius = { 1.4 };

	// jk NOTE: interface SASA numbers will be computed IGNORING HYDROGENS AND WATERS

	// compute the SASA for the complex
	calc_per_atom_sasa( atom_sasa_complex, res_sasa_complex,
    probe_radius, true, false );

	// compute the SASA for the unbound
	calc_per_atom_sasa( atom_sasa_unbound, res_sasa_unbound,
	  probe_radius, true, true );

	// compute SASA differences
	res_sasa_diff = res_sasa_unbound - res_sasa_complex;
	atom_sasa_diff = atom_sasa_unbound - atom_sasa_complex;

	// jk Residue must lose this much SASA to be an interface residue
	float const res_thres = { 0.3 };

	// jk Residue must have less than this fraction of the
	// jk exposed total SASA to be "core"
	float const core_res_thres = { 0.10 };

	// jk Atom must lose this much SASA to be an interface atom
	float const atom_thres = { 0.1 };

	num_interface_res = 0;
	num_core_interface_res = 0;
	num_interface_atom = 0;
	num_interface_water = 0;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= pose->total_residue(); ++i ) {
		if ( res_sasa_diff(i) > res_thres ) {
			interface_res(i)=true;
			++num_interface_res;
			if ( (res_sasa_complex(i)/rsd_exposed_sasa(pose->res(i))) < core_res_thres ) {
				core_interface_res(i)=true;
				++num_core_interface_res;
			}
		}
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( atom_sasa_diff(j,i) > atom_thres ) {
				interface_atom(j,i)=true;
				++num_interface_atom;
			}
		}
	}

	// jk make sure SASApack / SASAprob are up to date
	// note: calc_sasa_pack_score will call calc_sasa_prob,
	// so sasaprob will be updated too.
	//	calc_sasa_pack_score();

	if ( design::explicit_h2o || water::use_hetero_h2o )
		count_interface_waters( probe_radius, atom_thres );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::find_CBCB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::find_CBCB() {

	using namespace docking;
	using namespace param_aa;

	std::ofstream cbcbOUT("cbcb.txt",std::ios_base::app);

	for ( int i = part_begin(1); i <= part_end(1); ++i ) {
		int const aa1 = pose->res(i);
		if ( interface_res(i) && ( aa1 != aa_gly ) ) {
			for ( int j = part_begin(2); j <= part_end(2); ++j ) {
				int const aa2 = pose->res(j);
				if ( interface_res(j) && ( aa2 != aa_gly ) ) {
					float dist=0;
					distance_bk(pose->full_coord()(1,5,i),pose->full_coord()(1,5,j),dist);
					cbcbOUT << dist << std::endl;
				}
			}
		}
	}

	cbcbOUT.close();
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::count_interface_waters
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::count_interface_waters(
  float const probe_radius,
	float const interface_diff_thres
) {

	using namespace aaproperties_pack;
	using namespace param;

	// jk A water atom must have less than this much SASA to be considered buried
	float const burial_sasa_thres = { 5. };

	FArray1D_float water_res_sasa_complex( pose->total_residue() );
	FArray2D_float water_atom_sasa_complex( MAX_ATOM()(), pose->total_residue() );

	FArray1D_float water_res_sasa_unbound( pose->total_residue() );
	FArray2D_float water_atom_sasa_unbound( MAX_ATOM()(), pose->total_residue() );

	FArray2D_float water_atom_sasa_diff( MAX_ATOM()(), pose->total_residue() );

	// jk NOTE: interface SASA numbers will be computed WITH HYDROGENS AND WATERS

	// compute the SASA for the complex
	calc_per_atom_sasa( water_atom_sasa_complex, water_res_sasa_complex,
    probe_radius, false, false );

	// compute the SASA for the unbound
	calc_per_atom_sasa( water_atom_sasa_unbound, water_res_sasa_unbound,
	  probe_radius, false, true );

	// compute SASA differences
	water_atom_sasa_diff = water_atom_sasa_unbound - water_atom_sasa_complex;

	num_interface_water = 0;
	for ( int i = 1, max_atom = MAX_ATOM(); i <= pose->total_residue(); ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			int const jt = fullatom_type(j,pose->res(i),pose->res_variant(i));
			if ( jt == 26 ) {
				if ( ( water_atom_sasa_complex(j,i) < burial_sasa_thres ) &&
						 ( water_atom_sasa_diff(j,i) > interface_diff_thres ) ) {
					++num_interface_water;
				}
			}
		}
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::report_core_uns
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::report_core_uns(
  std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list
)
{

	int bb_unsatisfied=0;
	int sc_unsatisfied=0;
	float uns_score=0;

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			uns != uns_list.end(); ++uns ) {

		int resnum=uns->get_seqpos();
		if ( core_interface_res(resnum) ) {

			uns_score+=uns->weight();

			std::string const uns_type = decide_uns_type( *uns );
			if ( ! get_ds_outpdbonly_flag() ) uns->write_to_cout(tag, uns_type);
			uns->write_to_PDB(tag, uns_type);

			// jk require burial to a 1.0 A probe if explicit water is not used,
			// jk otherwise relax the criterion to "buried to a 1.4 A probe"
			if ( ! uns->satisfied() ) {
				if ( ( uns->buried10() ) || ( ( design::explicit_h2o ) && uns->buried() ) ) {
					if ( uns->backbone() ) {
						++bb_unsatisfied;
					} else {
						++sc_unsatisfied;
					}
				}
			}

		}
	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"bbhb_uns",float(bb_unsatisfied));
	add_decoy_score(tag+"schb_uns",float(sc_unsatisfied));
	add_decoy_score(tag+"uns_score",uns_score);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::report_core_uns
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::report_core_uns(
  std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{

	// jk Overloading: if group_uns is passed in do it after doing the uns as per previous subroutine
	report_core_uns(tag, uns_list);

	int num_group_uns=0;
	float group_uns_score=0;

	for( std::list<unsatisfied_buried_group>::iterator group_uns = group_uns_list.begin();
			group_uns != group_uns_list.end(); ++group_uns ) {

		int resnum=group_uns->get_seqpos();
		if ( core_interface_res(resnum) ) {

			++num_group_uns;
			group_uns_score+=group_uns->weight();

			if ( ! get_ds_outpdbonly_flag() ) group_uns->write_to_cout(tag);
			group_uns->write_to_PDB(tag);

		}

	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"group_uns",float(num_group_uns));
	add_decoy_score(tag+"group_uns_score",group_uns_score);

	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin compute_delta_UNS
///
/// @brief
///    find unsatified buried polars in a protein-protein interface
///
/// @detailed
///    protocol involved repacking the interface with solvated rotamers,
///    and comparing UNS to those found in unbound structure when non-Gly interface
///    residues are all replaced with Ala
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors John Karanicolas
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_delta_UNS(
  bool const do_repack,
	bool const compare_to_polyAla,
	bool const refine_loops
){

	using namespace decoystats;
	using namespace design;
	using namespace docking;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace param;
	using namespace param_aa;

	// save a local copy of the original setting of these vars
	bool const prev_full_atom = full_atom;
	std::string const prev_resfile = resfile;
	bool const prev_output_coord = output_coord;
	bool const prev_query_defined = query_defined;
	bool const prev_repack_inter = repack_inter;
	bool const prev_design_inter = design_inter;
	bool const prev_build_Ala_interface = build_Ala_interface;

	full_atom = true;
	resfile = "none";
	output_coord = false;
	query_defined = false;

	pose->copy_to_misc();

	// Collect a list of interface residues
	repack_inter = true;
	design_inter = true;
	interface_residues();

	if ( refine_loops ) {

		std::cout << "STOP:: refine_loops portion of compute_delta_UNS is not yet implemented"
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

		// jk Relax the loops (via pose), only on the first chain
		// jk Define each loop as 7 residues, use overlapping loops
		pose_ns::Loops interface_loops;
		for ( int i = 1; i <= misc::domain_end(1); ++i ) {
			if ( interface_residue(i) ) {
				int cutsite=i;
				int start=i-3;
				int stop=i+3;
				if ( start < 2 ) start = 2;
				if ( stop > ( misc::domain_end(1)-1 ) ) stop = misc::domain_end(1)-1;
				interface_loops.add_loop(start, stop, cutsite);
			}
		}

		pose_ns::Pose interface_pose;
		set_pose_flag(true);
		pose_from_misc( interface_pose, true, false, true );
		interface_pose.simple_fold_tree( interface_pose.total_residue() );
		loop_refinement(interface_pose, interface_loops);
		interface_pose.copy_to_misc();
		set_pose_flag(false);

	}

	//apl -- test, see if we need to reuse the same task between each packrot call
	pose_ns::Pose temp_pose;
	pose_from_misc( temp_pose, true, false, true );
	PackerTask Task( temp_pose );

	if ( do_repack || compare_to_polyAla ) {
		// Create PackerTask and setup values before pass into pack_rotamers
		FArray2D_int extra_rot( MAX_CHI, pose->total_residue() );
		FArray2D_float extra_chi( MAX_CHI, pose->total_residue() );
		Task.set_task("design",false,interface_residue,true,false,extra_rot, extra_chi);
		build_Ala_interface = false;
		Task.setup_residues_to_vary();
		if ( do_repack ) {
			// Repacking the starting interface
			std::cout << "repacking starting interface" << std::endl;
			pose_ns::Pose repack_pose;
			repack_pose = temp_pose;
			pose_from_misc( repack_pose, true, false, true );
			pack_rotamers( repack_pose, Task );
			repack_pose.copy_to_misc();
		}
	}

	// Collect a list of UNS in the bound structure
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(pose->res(),pose->res_variant(),pose->full_coord(),pose->total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(bound_uns_list,bound_group_uns_list);

	// Save bound (potentially repacked) interface pose
	pose_ns::Pose repacked_pose;
	repacked_pose = *pose;

	// Translate subunits apart (moving the second, keeping the first fixed)
	float const translation_dist=50.;
	translate_subunits(translation_dist);

	// Collect a list of UNS in the unbound structure
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(pose->res(),pose->res_variant(),pose->full_coord(),pose->total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(unbound_uns_list,unbound_group_uns_list);

	// Build a list of those unique to the bound state (lists are declared in decoystats_ns)
	//	std::list < unsatisfied_buried_polar > interface_uns_list;
	//	std::list < unsatisfied_buried_group > interface_group_uns_list;
	interface_delta_uns_list = find_unique_uns(bound_uns_list,unbound_uns_list);
	interface_delta_group_uns_list = find_unique_uns(bound_group_uns_list,unbound_group_uns_list);

	// Repack the same residues in the unbound,
	// optionally substituting Ala at all interface positions
	if ( do_repack || compare_to_polyAla ) {
		if ( compare_to_polyAla ) {
			std::cout << "repacking unbound state using only Ala" << std::endl;
			build_Ala_interface = true;
		} else {
			std::cout << "repacking unbound state" << std::endl;
		}
		Task.setup_residues_to_vary();
		pose_ns::Pose repack_pose;
		pose_from_misc( repack_pose, true, false, true );
		pack_rotamers( repack_pose, Task );
		repack_pose.copy_to_misc();
	}

	// Collect a list of UNS in the unbound structure
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(pose->res(),pose->res_variant(),pose->full_coord(),pose->total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(unbound_uns_list, unbound_group_uns_list);

	// Build a list of those unique to the bound state (lists are declared in decoystats_ns)
	interface_strict_uns_list = find_unique_uns(bound_uns_list,unbound_uns_list);
	interface_strict_group_uns_list = find_unique_uns(bound_group_uns_list,unbound_group_uns_list);

	// Read a file that specifies exempt uns or group_uns
	std::string exemption_file;
	stringafteroption( "gu_exempt", "none", exemption_file );

	if ( exemption_file != "none" ) {
		utility::io::izstream pdb_stream;
		pdb_stream.open(exemption_file);
		read_uns_from_pdb(pdb_stream,exempt_uns_list,exempt_group_uns_list);
 		interface_delta_uns_list = find_unique_uns(interface_delta_uns_list,exempt_uns_list);
		interface_delta_group_uns_list = find_unique_uns(interface_delta_group_uns_list,exempt_group_uns_list);
 		interface_strict_uns_list = find_unique_uns(interface_strict_uns_list,exempt_uns_list);
		interface_strict_group_uns_list = find_unique_uns(interface_strict_group_uns_list,exempt_group_uns_list);
	}

	// Retrieve the repacked (possibly solvated) interface
	*pose = repacked_pose;
	pose->copy_to_misc();

	// Update score info
	score_set_new_pose();
	score12();
	copy_hbenergies();
	calc_aprox_sasa();
	calc_sasa_pack_score(pose->total_residue(), pose->res(),pose->res_variant(), pose->full_coord());

	// restore state of globals
	full_atom = prev_full_atom;
	resfile = prev_resfile;
	output_coord = prev_output_coord;
	query_defined = prev_query_defined;
	repack_inter = prev_repack_inter;
	design_inter = prev_design_inter;
	build_Ala_interface = prev_build_Ala_interface;

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_quantifiers
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_quantifiers() {

	using namespace decoystats;

	if ( ! design::identify_interface_UNS_flag ) {
		compute_delta_UNS( false, false, false );
	}

	compute_delta_sasa();
	compute_composition();
	compute_energies();

	if ( tight_core_analysis ) {
		compute_tight_core_packing();
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::write_all_quantifiers_to_PDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::write_all_quantifiers_to_PDB()
{

	using namespace decoystats;
	using namespace energy_quantifier_ns;
	using namespace param;

	// jk measures of interface size
	add_decoy_score("delta_sasa",delta_sasa);
	add_decoy_score("num_interface_res",num_interface_res);
	add_decoy_score("num_interface_atoms",num_interface_atom);
	add_decoy_score("num_interface_waters",num_interface_water);
	add_decoy_score("num_core_interface_res",num_core_interface_res);

	// jk measures of composition
	add_decoy_score("interface_num_aromatic_atoms", num_interface_aromatics);
	add_decoy_score("interface_aromatic_sasa", sasa_interface_aromatics);
	float const fraction_polar = static_cast<float>(num_polar_atoms) / static_cast<float>(num_atoms);
	add_decoy_score("interface_fraction_polar", fraction_polar);
	add_decoy_score("met_sc_atoms", met_sc_atoms);

	// jk measures of packing
	if ( tight_core_analysis ) {
		add_decoy_score("interface_num_tight_core_atoms",interface_num_tight);
		float const tight_per_sasa = 100.*static_cast<float>(interface_num_tight)/delta_sasa;
		add_decoy_score("num_tight_core_atoms_per_sasa",tight_per_sasa);
		float const fraction_tight = static_cast<float>(interface_num_tight) / static_cast<float>(num_atoms);
		add_decoy_score("interface_fraction_tight", fraction_tight);
	}

	// jk energetics
	for( std::vector<interface_energy_quantifier>::iterator qcurr = energy_quantifiers.begin();
			qcurr != energy_quantifiers.end(); ++qcurr ) {
		if ( qcurr->is_initialized() ) {
			qcurr->write_to_PDB( num_interface_res, num_core_interface_res );
		}
	}

	// jk Rosetta predicted binding energy
	float delta_score = 0.;
	delta_score += (energy_quantifiers[ATR]).report_delta_val();
	delta_score += (energy_quantifiers[REP]).report_delta_val();
	delta_score += (energy_quantifiers[SOL]).report_delta_val();
	delta_score += (energy_quantifiers[HB]).report_delta_val();
	delta_score += (energy_quantifiers[PAIR]).report_delta_val();
	delta_score += (energy_quantifiers[GB]).report_delta_val();
	delta_score += (energy_quantifiers[CST]).report_delta_val();
	delta_score += (energy_quantifiers[PLANE]).report_delta_val();
	delta_score += (energy_quantifiers[H2O]).report_delta_val();
	delta_score += (energy_quantifiers[H2OHB]).report_delta_val();
	add_decoy_score("delta_score", delta_score);

	// jk an extra packing measure
	add_decoy_score("delta_atr_per_sasa",
	  1000.* (energy_quantifiers[ATR]).report_delta_val() / delta_sasa);

	// jk Uns lists
	report_unsatisfied_hbonds("BOUND UNS",bound_uns_list);
	report_unsatisfied_hbonds("UNBOUND UNS",unbound_uns_list);
	report_unsatisfied_hbonds("INT",interface_strict_uns_list,interface_strict_group_uns_list);
	report_unsatisfied_hbonds("DELTA",interface_delta_uns_list,interface_delta_group_uns_list);

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_delta_sasa
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::compute_delta_sasa()
{

	// jk find total SASA lost upon complexation
	delta_sasa=0.;
	for ( int i = 1; i <= pose->total_residue(); ++i ) {
		delta_sasa += res_sasa_diff(i);
	}

	return delta_sasa;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_composition
///
/// @brief
///    Count the number of aromatic atoms buried by the interface and
///    their interface SASA, also write the fraction of interface atoms
///    which are polar
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_composition()
{

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace param;
	using namespace param_aa;

	// jk initialization
	num_interface_aromatics=0;
	sasa_interface_aromatics=0.;
	num_polar_atoms=0;
	num_atoms=0;
	met_sc_atoms=0;

	for ( int i = 1; i <= pose->total_residue(); ++i ) {
		int const aa = pose->res(i);
		int const aav = pose->res_variant(i);
		int const max_atom = nheavyatoms(aa,aav);
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( interface_atom(j,i) ) {
				++num_atoms;
				int const atom_type = fullatom_type( j, aa, aav );
				if ( ( atom_type == 6 ) &&
						 ( aa != aa_his ) ) {
					// jk if atom is aromatic, but not His
					++num_interface_aromatics;
					sasa_interface_aromatics += atom_sasa_diff(j,i);
				} else if ( ( aa == aa_met ) && ( j >= 5 ) ) {
					// jk Met sidechain
					++met_sc_atoms;
				} else if ( atomtype_sasapolarity(atom_type) == POLAR ) {
					// jk if atom is polar
					++num_polar_atoms;
				}
			}
		}
	}

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::setup_energetic_quantifiers
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::setup_energetic_quantifiers()
{

	using namespace energy_quantifier_ns;
	using namespace fullatom_energies;
	using namespace pdbstatistics_pack;
	using namespace template_pack;
	using namespace void_ns;

	// jk When adding new energy quantifiers, do it by first adding
	// jk to the enum in decoystats_interface_classes.h

	{
		interface_energy_quantifier & q = energy_quantifiers[ATR];
		q.reset();
		q.initialize("atr");
		q.allow_delta(atr_pair);
		q.allow_interface_sum(atrenergy);
		q.allow_VsPDB(atrenergy,atr_avg);
		q.allow_max_VsPDB();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[REP];
		q.reset();
		q.initialize("rep");
		q.allow_delta(rep_pair);
		q.allow_interface_sum(repenergy);
		q.allow_VsPDB(repenergy,rep_avg);
		q.allow_max();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SOL];
		q.reset();
		q.initialize("sol");
		q.allow_delta(sol_pair);
		q.allow_interface_sum(solenergy);
		q.allow_VsPDB(solenergy,sol_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[HB];
		q.reset();
		q.initialize("hbond");
		q.allow_delta(hbond_pair);
		q.allow_interface_sum(hbenergy);
		q.allow_VsPDB(hbenergy,hb_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PROB];
		q.reset();
		q.initialize("prob");
		q.allow_interface_sum(probenergy);
		q.allow_VsPDB(probenergy,one_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[DUN];
		q.reset();
		q.initialize("dun");
		q.allow_interface_sum(dunenergy);
		q.allow_VsPDB(dunenergy,dun_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PAIR];
		q.reset();
		q.initialize("pair");
		q.allow_delta(pair_pair);
		q.allow_interface_sum(pair_energy);
		q.allow_VsPDB(pair_energy,pair_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[INTRARES];
		q.reset();
		q.initialize("intrares");
		q.allow_interface_sum(intraresenergy);
		q.allow_VsPDB(intraresenergy,intrares_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[GB];
		q.reset();
		q.initialize("gb");
		q.allow_delta(gb_elec_pair);
		q.allow_interface_sum(gb_elecenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[CST];
		q.reset();
		q.initialize("cst");
		q.allow_delta(cst_pair);
		q.allow_interface_sum(cstenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PLANE];
		q.reset();
		q.initialize("plane");
		q.allow_delta(plane_plane);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[H2O];
		q.reset();
		q.initialize("h2o");
		q.allow_delta(h2o_pair);
		q.allow_interface_sum(h2oenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[H2OHB];
		q.reset();
		q.initialize("h2o_hb");
		q.allow_delta(h2o_hb_pair);
		q.allow_interface_sum(h2ohbenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[WSOL];
		q.reset();
		q.initialize("wsol");
		q.allow_interface_sum(wsolenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[UNF];
		q.reset();
		q.initialize("unf");
		q.allow_interface_sum(unfenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[TOT];
		q.reset();
		q.initialize("total");
		q.allow_interface_sum(resenergy);
		q.allow_VsPDB(resenergy,tot_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SASAPACK];
		q.reset();
		q.initialize("SASApackVsPDB");
		q.allow_interface_sum(rsd_sasa_pack_score);
		q.allow_max();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SASAPROB];
		q.reset();
		q.initialize("SASAprobVsPDB");
		q.allow_interface_sum(rsd_sasaprob);
		q.allow_min();
	}

	energies_initialized = true;

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_energies()
{

	for( std::vector<interface_energy_quantifier>::iterator qcurr = energy_quantifiers.begin();
			qcurr != energy_quantifiers.end(); ++qcurr ) {
		if ( qcurr->is_initialized() ) {
			qcurr->compute_energies( interface_res, core_interface_res );
		}
	}

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_tight_core_packing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
interface_ds::compute_tight_core_packing()
{

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace param;

	FArray2D_bool tight_in_monomers( MAX_ATOM(), pose->total_residue() );

	// jk max SASA5 which still qualifies as tight packing
	realafteroption("tight_core_thres",4.0,tight_core_thres);

	int unbound_tight=find_tight_core_packing(true);
	tight_in_monomers=tightly_packed;

	int bound_tight=find_tight_core_packing(false);
	interface_num_tight=bound_tight-unbound_tight;

	// jk list the tightly packed interface atoms (to the output pdb)
	//	for ( int i = 1, max_atom = MAX_ATOM(); i <= pose->total_residue(); ++i ) {
	//		int const aa = pose->res(i);
	//		int const aav = pose->res_variant(i);
	//		for ( int j = 1; j <= max_atom; ++j ) {
	//			if ( tightly_packed(j,i) && ! tight_in_monomers(j,i) ) {
	//				// jk we'll write to an output stream
	//				std::ostringstream tight_int_atom_out_stream;
	//				tight_int_atom_out_stream << "TIGHTINTATOM " << atom_name(j,aa,aav) <<
	//					' ' << misc::residue1(i) << ' ' << I( 4, i );
	//				decoystats_store_output_line( tight_int_atom_out_stream.str() );
	//			}
	//		}
	//	}

	reset_noninterface_bfactors(50.);

	return interface_num_tight;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::list_interface_res
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::list_interface_res() {

	using namespace param;

	for ( int i = 1; i <= pose->total_residue(); ++i ) {
		if ( interface_res(i) ) {
			// jk we'll write to an output stream
			std::ostringstream int_res_out_stream;
			int_res_out_stream << "INTRES " << misc::residue1(i) << ' ' << I( 4, i );
			decoystats_store_output_line( int_res_out_stream.str() );
		}
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::list_interface_atoms
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::list_interface_atoms() {

	using namespace aaproperties_pack;
	using namespace param;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= pose->total_residue(); ++i ) {
		int const aa = pose->res(i);
		int const aav = pose->res_variant(i);
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( interface_atom(j,i) ) {
				// jk we'll write to an output stream
				std::ostringstream int_atom_out_stream;
				int_atom_out_stream << "INTATOM " << atom_name(j,aa,aav) << ' ' << misc::residue1(i) << ' ' << I( 4, i );
				decoystats_store_output_line( int_atom_out_stream.str() );
			}
		}
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::reset_noninterface_bfactor
///
/// @brief
///   jk Set ds_factor to the specified value for all non-interface atoms
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::reset_noninterface_bfactors(
  float const reset_val
)
{

	using namespace param;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= pose->total_residue(); ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( ! interface_atom(j,i) ) {
				make_pdb_ns::bfactor(j,i)=reset_val;
			}
		}
	}

	return;
}

