// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19866 $
//  $Date: 2008-01-23 14:54:04 -0500 (Wed, 23 Jan 2008) $
//  $Author: possu $


// Rosetta Headers
#include "design_structure.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "analyze_interface_ddg.h"
#include "analyze_interface_ddg_ns.h"
#include "are_they_neighbors.h"
#include "cluster_mutE.h"
#include "constraints.h"
#include "counters.h"
#include "crankshaft.h"
#include "cst_set.h"
#include "decoystats.h"
#include "decoystats_classes.h"
#include "design.h"
#include "diagnostics_rosetta.h"
#include "dna.h"
#include "DnaPose.h"
#include "dna_ns.h"
#include "dna_motifs.h"
#include "docking.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "dock_loops.h"
#include "dock_pivot.h"
#include "dock_structure.h"
#include "docking_constraints.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "etable.h"
#include "favor_residue_ns.h"
#include "files_paths.h"
#include "fold_loops.h"
#include "fragments.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "fullatom_setup.h"
#include "grid_docking.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "int_fullatom_energies.h"
#include "job_distributor.h"
#include "jumping_refold.h"
#include "ligand.h"
#include "loops.h"
#include "loops_ns.h"
#include "make_pdb.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "multistate_design.h"
#include "namespace_fullatom_flag.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdb.h"
#include "pose.h"
#include "pose_constraints.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_peptide_docking.h"
#include "pose_io.h"
#include "pose_loops.h"
#include "current_pose.h"
#include "random_numbers.h"
#include "read_aa_ss.h"
#include "recover.h"
#include "refold.h"
#include "RotamerOptions.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "scale_res_energy.h"
#include "score.h"
#include "score_ns.h"
#include "smallmove.h"
#include "start.h"
#include "status.h"
#include "torsion_bbmove_trials.h"
#include "vdw.h"
#include "void.h"
#include "weights_manager.h"
#include "DesignMap.h"
#include "PackerTask.h"

// Numeric Headers
#include <numeric/xyzVector.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <ctime>
#include <fstream>
#include <iostream>
#include <list>
#include <string>
#include <vector>

// Namespaces

namespace design_profile {
	using namespace files_paths;
	using namespace param;
	using namespace param_aa;
	FArray2D_float profile( MAX_RES(), MAX_AA() );
	int nseq;
	bool use_profile = { true };
	std::string designfile;
}

namespace rdesign_control {
	float init_temp = { 2.0 };
	float final_temp = { 0.8 };
	int cycles1 = { 1 };
	int cycles2 = { 20 };
	int cycles3 = { 20 };
	int nloop1 = { 1 };
	int nloop2 = { 25 };
	int nloop3 = { 25 };
	float helix1 = { 2.0 };
	float helix2 = { 2.0 };
	float helix3 = { 2.0 };
	float strand1 = { 5.0 };
	float strand2 = { 3.0 };
	float strand3 = { 2.0 };
	float loop1 = { 5.0 };
	float loop2 = { 3.0 };
	float loop3 = { 2.0 };
}

//forward declarations
void disable_repack_prolines( PackerTask & task );

////////////////////////////////////////////////////////////////////////////////
/// @begin design_structure
///
/// @brief
///bk can do different design possibilites, varying from just repacking
///bk to redesigning with backbone movement
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
design_structure()
{
	using namespace design;

	std::string mode;
//------------------------------------------------------------------------------
	std::cout << "\nStart of design structure\n" << std::endl;

	if ( ! dna_interface ) {
		set_fullatom_flag(true);
	}

	if ( desock ) {             // jk design and dock (dock_mcm only)
		mode = "design";
		design_and_dock_mcm();
		return;
	}
//SJF
  if( flex_peptide ) {
    design_flex_peptide();
    return;
  }


	if ( design_inter ) {
		mode = "design";
		design_interface(mode); // redesign protein-protein interface
		return;
	}

	if ( dock_des_min_inter ) {
		dock_design_minimize_interface();
		return;
	}

	if ( multistate ) {
		mode = "design";
		multistate_design(mode);
		return;
	}

	if ( onlypack ) {         // repack structure on fixed backbone
		mode = "packrot";
		fixed_backbone(mode);
		return;
	}

	if ( dna_interface ) {
		DnaPose dnapose;
		if ( optE ) { get_optE(); return; } //(some DNA-specific setup has occured)
		dnapose.design();
		return;
	}

	if ( optE ) {             // output energies for weight optimization
		get_optE();
		return;
	}

	if ( fixbb ) {       // design sequence on fixed backbone
		mode = "design";
		fixed_backbone(mode);
		return;
	}

	if ( mvbb ) {             // move the protein backbone
		move_backbone();
		return;
	}

	if ( tail ) {             // flexible tail for interface
		move_tail( false );
		return;
	}

	if ( tail_fix_helix ) {       // flexible tail (but fix helices) for interface
		move_tail( true );
		return;
	}

	if ( design_min_inter ) {       // local bb minimization at interface
		design_minimize_interface();
		return;
	}

	shakenbake = false; //// not implemented
	if ( shakenbake ) {       // really move the backbone
		shakenbake_backbone(false);
		return;
	}

	if ( alter_spec ) {      // alter binding specificity
		altered_specificity();
		return;
	}

	if ( mutate_cluster ){   // mutate clusters
		mutated_clusters();
		return;
	}

	if ( design_loops ) {
		design_for_loops();
		return;
	}

	if ( point_mutation )
	{
		point_mut();
		return;
	}


	if ( mut_list ) { // read in a list of mutations and
										// if compensate flag true - design neighbors as well
										// if compensate flag false - just force mutation
										// output a list of ddG and compensating mutations
		design_from_mut_list();
		return;
	}

	//lin rational design code
	if ( cluster_design ) {
		design_from_cluster();
		return;
	}
	std::cout << "More flags needed for design mode" << std::endl;
	std::cout << "For example, do you want to move the backbone" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin design_for_loops
///
///
/// @two options:design_loops_dock:
/// brief iterate between small docking, loop moves, and design
/// @detailed
///	only allows small docking and loop movements ... not for preliminary docking steps
/// some redundancy in repacking interface
/// design_loops_hold:
/// design 1 loop for a single chain protein, no docking movement
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// need -design and -loops flags to run
///
/// @refrences
///
/// @authors Glenn Butterfoss
///
/// @last_modified 24 Jan 2005
////////////////////////////////////////////////////////////////////////////////
void
design_for_loops()
{
	using namespace design;
	using namespace misc;
	using namespace rdesign_control;
	using namespace runlevel_ns;
	using namespace param;
	using namespace param_aa;
	using namespace loops_ns;

	if ( ! get_loop_flag() ) return;

	bool exist;
	float gamma, temperature;
	float new_gly_Waa = -6.0;
	int total_outer_cycles, total_inner_cycles;

	if ( benchmark ) {
		total_outer_cycles = 1;
		total_inner_cycles = 2; //design at 1
	} else {
		total_outer_cycles = 3;
		total_inner_cycles = 7; //design at 1, 4, and 7 of inner_cycle
	}

	int const max_min_trial = 5; //number of calls to main_min_trial
	float const loop_weight_here = 600; //too high?

	bool apply_docking = design::design_loops_flags::design_loops_dock;

	// input for docking_monte_carlo_minimize for design_loops docking mode
	int const dock_mcm_cycles = 6; //repacks everything every 8 cycles ... so run 6
	float const dock_mcm_trans = 0.005; // size of translation moves
	float const dock_mcm_rot = 0.002; // size of rotation moves
	float const dock_mcm_min = -5.0; // go into minimize if (score - best_score) is within value
	float const dock_mcm_tol = 1.0; // tolerance on the minimization function

	// setup general stuff
	set_fullatom_flag(true);
	monte_carlo_set_simannealing(true);
	classical_constraints::BOUNDARY::set_max_seqSep(total_residue);
	minimize_exclude_sstype(false,false);
	score_set_lj_weight(1.0);
	minimize_set_local_min(true,5);
	set_use_nblist(true);
	gamma = std::pow( (final_temp/init_temp), (1.0f/(total_outer_cycles*total_inner_cycles)) );
	temperature = init_temp;
	monte_carlo_set_temp(temperature);
	// setup main_smallmove_trial
	set_smallmove_size(1.0,1.0,1.0);
	score_set_try_rotamers(true);
	// setup loops
	score_set_loop_weight(loop_weight_here);
	set_allow_insert_by_looplength(total_residue+1,true,exist);

	if ( truefalseoption("prog_grid_dock") ) {
			progressive_grid_dock_move_from_misc();
			monte_carlo_reset();
	}

	std::cout << "  step: starting energy: " << std::endl;
	dlogfile_output(-1);
	std::cout << "  step: starting sequence: " << std::endl;
	write_sequence(-2);

	if ( apply_docking ){
		// loop stuff for dockign step
		set_dock_loopmove_flag(true);
		// penalize gly during design
		set_new_Waa_value( aa_gly, new_gly_Waa );
		//first design for docking submode
		std::cout << "  step: initial design: " << std::endl;
		main_design_force_accept(score12,"design");
		write_sequence(0);
		dlogfile_output(-1);
		std::cout << "  step: initial loop min: " << std::endl;
		main_minimize_trial(score12,"dfpmin");
		dlogfile_output(-1);
	}

	// call them cycles instead of loops ... because loops are something else
	for ( int outer_cycle = 1; outer_cycle <= total_outer_cycles; ++outer_cycle ) {
		for ( int inner_cycle = 1; inner_cycle <= total_inner_cycles; ++inner_cycle ) {

			temperature *= gamma;
			monte_carlo_set_temp(temperature);

			std::cout << "  begin inner cycle " << inner_cycle << " of " << total_inner_cycles;
			std::cout << "  outer cycle " << outer_cycle << " of " << total_outer_cycles << std::endl;

			std::cout << "  step: minimize loops: " << std::endl;

			main_minimize_trial(score12,"dfpmin");
			for ( int number_min_trial = 1; number_min_trial <= max_min_trial; ++number_min_trial ) {
				main_small_min_trial(1,score12,number_min_trial,"dfpmin");
				main_shear_min_trial(1,score12,number_min_trial,"dfpmin");
				dlogfile_output( number_min_trial );
			}

			if ( mod( inner_cycle + 2 ,3) == 0 ) {
				std::cout << "  step: design: " << std::endl;
				main_design_trial(score12,"design");
				dlogfile_output(outer_cycle);
				write_sequence(outer_cycle);

			} else if( apply_docking ){
				// docking_monte_carlo_minimize also repacks everything within 6 Ang of interface
				std::cout << "  step: pack interface: " << std::endl;
				main_design_trial(score12,"pack_interface");
				dlogfile_output(outer_cycle);
			} else {
				std::cout << "  step: pack residues close to the loop: " << std::endl;
				main_design_trial(score12,"pack_loop");
				dlogfile_output(outer_cycle);
			}

			if ( apply_docking ){
				std::cout << "  step: dock_min: " << std::endl;
				// note: seems better when cutoff for complex min is lowered
				docking_monte_carlo_minimize(dock_mcm_cycles,"dfpmin",score12,dock_mcm_trans,dock_mcm_rot,dock_mcm_min,dock_mcm_tol);
			}
				dlogfile_output(outer_cycle);

			//compare_fullatom_energy();
			//some small differences in REP
			//monte_carlo_reset() doesn't help

		}

		dlogfile_output(outer_cycle);
	}

	//reset things
	recover_LOW(score12);
	if ( apply_docking ){
		reset_Waa_value( aa_gly );
	}

	std::cout << "FINAL:" << std::endl;

	// report final splice
	// FArray1D_float splice_rms( 1000 ); // to get splice distances
	// loop_get_all_splicerms(splice_rms,num_loop);
	// std::cout << "splice_rms ";
	// for ( int splice_loop_counter = 1; splice_loop_counter <= num_loop; ++splice_loop_counter ) {
	// 	std::cout << ' ' <<  F( 6, 2, splice_rms(splice_loop_counter));
	// }

	std::cout << std::endl;
	dlogfile_output(-1);
	write_sequence(0);
	output_trial_counters( std::cout );

	score_set_loop_weight(0.0);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin move_tail
///
/// @brief variation of flex_bb design to optimize a flexible tail
///
/// @detailed
///       ... tail region is given at the command line and the various
///       residue/rotamer degrees of freedom are given with -resfile
///       ... not great for internal loops
///       (to be optimized for interface design ... still in testing)
///
/// @authors Glenn Butterfoss
/////////////////////////////////////////////////////////////////////////////////
void
move_tail( bool fix_helix )
{
//bk can design and move the backbone simultaneously

	using namespace design;
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pose_ns;
	using namespace rdesign_control;
	using namespace runlevel_ns;

	//local
	int tail_domain;
	FArray1D_int domain_begin( MAX_SEGMENTS ); //glb res to consider in modified energy
	float temperature, cutoff, helix_move_size, strand_move_size, loop_move_size, new_gly_Waa;
	bool exist;


	//glb checking tail definitions
	if ( tail_begin == 0 || tail_end == 0 ) {
		std::cout << "ERROR tail range not defined" << std::endl;
		return;
	}

	//glb penalize gly
	new_gly_Waa = -9.0;
	set_new_Waa_value( aa_gly, new_gly_Waa );

 	// set move sizes
	if ( fix_helix ) {
		helix_move_size = 0.00001; //glb (stalls on zero sometimes ... don't know why)
	} else {
		helix_move_size = 2.0;
	}
	loop_move_size = 15.0;
	strand_move_size = 15.0;
	set_smallmove_size(helix_move_size,strand_move_size,loop_move_size);

	monte_carlo_set_simannealing(true);
	set_fullatom_flag(true);
	classical_constraints::BOUNDARY::set_max_seqSep(total_residue);
	choose_frag_set_top_N_frags(200);
	minimize_exclude_sstype(true,false);
	score_set_lj_weight(1.0);
	minimize_set_local_min(true,5);
	set_use_nblist(true);
	set_scale_res_energy_flag( true, true ); //glb
	set_scale_res_energy_weight( 1.0 ); //glb
	set_scale_res_rep_energy_weight( 1.0 ); //glb

	// glb tail uses sequential numbering but read/write with pdb numbering
	// glb now figure out chains
	tail_domain = 0;
	for ( int i = 1; i <= total_domains; ++i ) {
		if ( i == 1 ) {
			domain_begin(i) = 1;
		} else {
			int j;
			j = i-1;
			domain_begin(i) = domain_end(j) + 1;
		}
		if ( (domain_begin(i) <= tail_begin && tail_begin <= domain_end(i)) &&
				(domain_begin(i) <= tail_end   && tail_end <= domain_end(i)) ) {
			tail_domain = i;
		}
	}

	if (!tail_domain) {
		std::cout << "problems with tail range" << std::endl;
		return;
	}
	std::cout << "total_domains " << total_domains << std::endl;

	// select res to be scaled -- same chain but not tail
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = i; j <= total_residue; ++j ) {

			set_scale_res_energy(i,j,false);
			set_scale_res_rep_energy(i,j,false);

			if (tail_begin <= i && i <= tail_end) {
				if (domain_begin(tail_domain) <= j && j <= domain_end(tail_domain)) {
					if (tail_begin <= j && j <= tail_end) {
						set_scale_res_energy(i,j,false);
						set_scale_res_rep_energy(i,j,false);
					} else {
						set_scale_res_energy(i,j,true);
						set_scale_res_rep_energy(i,j,false);
					}
				} else {
					set_scale_res_energy(i,j,false);
					set_scale_res_rep_energy(i,j,true);
				}
			}
		}
	}

	//glb define tail with inserts
	std::cout << "setting up the tail" << std::endl;
	std::cout << SS( tail_begin ) << " to " << SS( tail_end ) << std::endl;

	set_allow_insert_by_region(tail_begin,tail_end,true,exist);
	temperature = init_temp;
	monte_carlo_set_temp(temperature);

	cutoff = 60.0;

	// glb pre relax
	// glb score0 - centroid scores clash only
	std::cout << "starting centroid relax..." << std::endl;
	main_minimize_trial(score0,"linmin");

	for ( int i = 1; i <= nloop2; ++i ) {
		set_scale_res_rep_energy_weight( 0.4 ); //glb
		main_small_min_trial(5,score0,i,"linmin");
		main_shear_min_trial(10,score0,i,"linmin");
		main_wobble_min_trial(3,cutoff,score0,i,1,-1,"linmin");
		dlogfile_output(i);
	}

	recover_LOW(score0);
	new_score_function(score12);
	dlogfile_output(-2);

	FArray1D_bool positions_to_minimize( MAX_RES()() );
	bool const just_use_bb_heavy_atoms( true ); //for constraints

	float const fa_rep = 1.0;
	float const min_tolerance = 0.0000001;
	float const cst_weight = 2.0;
	int output_counter = 1;

	pose_ns::Pose pose;
	pose_ns::Score_weight_map weight_map( score12 );
	cst_set_ns::Cst_set cst_set;

	//set up tail as loops
	num_loop = 1;
	loop_begin(1) = tail_begin;
	loop_end(1) = tail_end;

	get_loop_pack_residues(positions_to_minimize,"interface");

	dlogfile_output(-1);

	for(int ii = 1; ii <= 4; ++ii){

		main_design_trial(score12,"design");
		write_sequence(0);
		dlogfile_output(++output_counter);

		//setup pose
		set_pose_flag(true);
		pose.set_allow_bb_move(true);
		pose.set_allow_chi_move(true);
		fullatom_nonideal_initialized_pose_from_misc( pose );
		pose.refold_sidechains_from_chi();

		//setup cst during 1st iter
		if( 1 == ii ){
  		fill_cst_set( cst_set, pose, just_use_bb_heavy_atoms );
  		pose.set_constraints( cst_set );
		}

		//set weights and constraints
		weight_map.set_weight( FA_REP, fa_rep );
		weight_map.set_weight( ATOMPAIR_CST,
 	 	 500.0 * cst_weight / cst_set.count_atompair_constraints() );

		//minimize
		assert( misc_in_sync( pose ));

		pose_setup_minimizer( min_tolerance );
		pose.set_allow_chi_move( positions_to_minimize ); // pass FArray
		pose.set_allow_bb_move( positions_to_minimize ); // pass FArray
		pose.set_allow_jump_move( false );
		pose.main_minimize( weight_map, "dfpmin" );
		assert( misc_in_sync( pose ));
		pose.copy_to_misc();  // make sure complex is in misc

		set_pose_flag(false);
		monte_carlo_reset(); //reset to retain min with changing weights
		dlogfile_output(++output_counter);

	}
	reset_Waa_value( aa_gly );

	std::cout << "FINAL:" << std::endl;
	recover_LOW(score12);
	dlogfile_output(-1);
	write_sequence(0);
	output_trial_counters( std::cout );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin move_backbone
///
/// @brief
///bk can design and move the backbone simultaneously
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
move_backbone()
{
	using namespace design;
	using namespace misc;
	using namespace rdesign_control;
	using namespace runlevel_ns;

	int nloop,cycles,size;
	float gamma,temperature,cutoff;

	rdesign_setup();
	if ( benchmark ) {
		nloop2 = 1;
		nloop3 = 1;
		cycles2 = 1;
		cycles3 = 1;
	}

	monte_carlo_set_simannealing(true); // should this be simannealing?
	set_fullatom_flag(true);
	classical_constraints::BOUNDARY::set_max_seqSep(total_residue);
	choose_frag_set_top_N_frags(200);
	minimize_exclude_sstype(true,false);
	score_set_lj_weight(1.0);
	minimize_set_local_min(true,5);
	set_use_nblist(true);

	nloop = nloop1 + nloop2 + nloop3;
	cycles = cycles1 + cycles2 + cycles3;
	gamma = std::pow( (final_temp/init_temp), (1.0f/(nloop*cycles)) );
	temperature = init_temp;
	monte_carlo_set_temp(temperature);

	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "linmin moves..." << std::endl;
	set_smallmove_size(helix2,strand2,loop2);
	size = 3;
	cutoff = 60.0;

	recover_LOW(score12);
	dlogfile_output(-2);

//      main_minimize_trial(score12,"linmin");
//      dlogfile_output(-1);

	if ( !fixseq ) {
		write_sequence(-2);
		main_design_force_accept(score12,"design");
		write_sequence(-1);
		dlogfile_output(-1);
	} else {
		main_design_trial(score12,"packrot");
		dlogfile_output(-1);
	}

	main_minimize_trial(score12,"linmin");
	dlogfile_output(0);

	reset_trial_counters();

	for ( int jk = 1; jk <= nloop2; ++jk ) {
		for ( int j = 1; j <= cycles2; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
			main_small_min_trial(5,score12,j*jk,"linmin");
			set_smallmove_size(2.0,15.0,15.0);
			main_shear_min_trial(5,score12,j,"linmin");
			set_smallmove_size(helix2,strand2,loop2);
			main_crank_min_trial(3,cutoff,score12,j,3,2,"linmin");
			main_wobble_min_trial(1,cutoff,score12,j,1,0,"linmin");
			if ( mod(j,5) == 0 ) {
				main_design_trial(score12,"packrot");
			}
		}
		if ( mod(jk,6) == 0 && !fixseq ) {
			main_design_trial(score12,"design");
			write_sequence(jk);
		} else {
			main_design_trial(score12,"packrot");
		}
		dlogfile_output(jk);
	}                     // jk

	output_trial_counters( std::cout );

	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "-----------------------------------------------------" << std::endl;
	std::cout << "dfpmin moves..." << std::endl;
	set_smallmove_size(helix3,strand3,loop3);
	recover_LOW(score12);
	dlogfile_output(0);

	main_minimize_trial(score12,"dfpmin");
	dlogfile_output(0);

	reset_trial_counters();
	for ( int jk = 1; jk <= nloop3; ++jk ) {
		for ( int j = 1; j <= cycles3; ++j ) {
			temperature *= gamma;
			monte_carlo_set_temp(temperature);
			main_small_min_trial(5,score12,j*jk,"dfpmin");
			set_smallmove_size(2.0,15.0,15.0);
			main_shear_min_trial(5,score12,j,"dfpmin");
			set_smallmove_size(helix3,strand3,loop3);
			main_crank_min_trial(3,cutoff,score12,j,3,2,"dfpmin");
			main_wobble_min_trial(1,cutoff,score12,j,1,0,"dfpmin");
			if ( mod(j,5) == 0 ) {
				main_design_trial(score12,"packrot");
			}
		}
		if ( mod(jk,6) == 0 && !fixseq ) {
			main_design_trial(score12,"design");
			write_sequence(jk);
		} else {
//			main_design_trial(score12,"packrot");
		}
		dlogfile_output(jk);
	}                     // jk

	output_trial_counters( std::cout );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin design_minimize_interface
///
/// @brief interate between design and small pose-based flexible bb minimiziation
///
/// @detailed uses pose for min, goes through several steps of ramping full atom
///           repulsion and decreasing backbone constraints during minimiziation.
///           default parameters optimized by glb
///
//////////////////////////////////////////////////////////////////////////////
void
design_minimize_interface()
{
	float const start_fa_rep = 0.85;
	float const	start_min_tolerance = 0.01;
	float const	start_cst_weight = 20;

	float const final_fa_rep = 1.0;
	float const	final_min_tolerance = 0.0000001;
	float const	final_cst_weight = 2;

	int const num_iter = 8; //total # if iterations
	int const num_ramp = 6; //# of total iter to ramp weights

	float const new_gly_Waa = -6.0;

	bool const constrain_from_initial_coords = true;

	set_new_Waa_value( param_aa::aa_gly, new_gly_Waa );

	if (truefalseoption( "prog_grid_dock" ) ) {
		progressive_grid_dock_move_from_misc();
		monte_carlo_reset();
	}

	design_minimize_interface(
		start_fa_rep, start_min_tolerance, start_cst_weight,
		final_fa_rep, final_min_tolerance, final_cst_weight,
		num_iter, num_ramp,
		constrain_from_initial_coords
	);

	reset_Waa_value( param_aa::aa_gly );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin design_minimize_interface
///
/// @brief interate between design and small pose-based flexible bb minimiziation
///
/// @detailed
///
/// @param - start_fa_rep - [in] - the fa_rep weight to start at
/// @param - start_min_tolerance - [in] - the tolerance to start at
/// @param - start_cst_weight - [in] - the weight for stay-put constaints to start at
/// @param - final_fa_rep - [in] - the fa_rep weight to conclude at
/// @param - final_min_tolerance - [in] - the tolerance to conclude at
/// @param - final_cst_weight - [in] - the weight for the stay-put constraints to end at
/// @param - num_iter - [in] - the number of design/minimize cycles to perform
/// @param - num_ramp - [in] - the number of the design/minimize cycles in which
///   the weights are ramped; should not be more than num_iter
/// @param - constrain_from_initial_coords - [in] - true: set stay-put constraints
///   from the coordinates of the input structure; false: reset the stay-put
///   constraints after each round of design
///
//////////////////////////////////////////////////////////////////////////////
void
design_minimize_interface(
	float const start_fa_rep,
	float const	start_min_tolerance,
	float const	start_cst_weight,
	float const final_fa_rep,
	float const	final_min_tolerance,
	float const	final_cst_weight,
	int const num_iter,
	int const num_ramp,
	bool const constrain_from_initial_coords
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace files_paths;
	using namespace misc;
	using namespace rdesign_control;
	using namespace runlevel_ns;
	using namespace param;
	using namespace param_aa;
	using namespace loops_ns;
	using namespace pose_ns;
	using namespace analyze_interface_ddg_ns;

	pose_ns::Pose pose;
	pose_ns::Score_weight_map weight_map( score12 );

	//	bool const fullatom( true );
	//	bool const ideal_pose( false ); // non-ideal backbone geometry
	//	bool const coords_init( true ); // use the coords from misc
	bool const just_use_bb_heavy_atoms( true ); //for constraints

	float min_tolerance;
	float fa_rep;
	float cst_weight;

	int output_counter = -1;

	//set up constants linear scaling
	float const fa_rep_slope = ( final_fa_rep - start_fa_rep) /
	 (static_cast<float>(num_ramp) - 1.0);
	float const fa_rep_inter = start_fa_rep - fa_rep_slope;

	float const min_tol_slope = ( final_min_tolerance - start_min_tolerance) /
	 (static_cast<float>(num_ramp) - 1.0);
	float const min_tol_inter = start_min_tolerance - min_tol_slope;

	float const cst_weight_slope = ( final_cst_weight - start_cst_weight) /
	 (static_cast<float>(num_ramp) - 1.0);
	float const cst_weight_inter = start_cst_weight - cst_weight_slope;

	//set up things
	set_fullatom_flag(true);
	minimize_set_vary_omega( true );

	write_sequence(0);
	dlogfile_output(output_counter);
	cst_set_ns::Cst_set cst_set_beginning;

	for(int ii = 1; ii <= num_iter; ++ii){
		//design
		main_design_trial(score12,"design_restrict_interface");
		write_sequence(0);
		dlogfile_output(++output_counter);

		if ( get_fullatom_totalE() > 500 ) {
			//dump_fullatom_pdb("rejected_dmi.pdb");
			return;
		}

		//minimize setup params
		if(ii <= num_ramp){
			fa_rep = fa_rep_slope * ii + fa_rep_inter;
			min_tolerance = min_tol_slope * ii + min_tol_inter;
			cst_weight = cst_weight_slope * ii + cst_weight_inter;
		}else{
			fa_rep = final_fa_rep;
			min_tolerance = final_min_tolerance;
			cst_weight = final_cst_weight;
		}
		std::cout << "minimize interface: " <<
		 " full atom repulsion = " << fa_rep <<
		 " tolerance = " << min_tolerance <<
		 " cst_weight = " << cst_weight << std::endl;

		//setup pose
		set_pose_flag(true);
		pose.set_allow_bb_move(true);
		pose.set_allow_chi_move(true);
		fullatom_nonideal_initialized_pose_from_misc( pose );
 		pose.refold_sidechains_from_chi();

		cst_set_ns::Cst_set cst_set_middle;
		cst_set_ns::Cst_set * cst_set;
		if ( constrain_from_initial_coords )
		{
			if( 1 == ii )
			{
				fill_cst_set( cst_set_beginning, pose, just_use_bb_heavy_atoms );
			}
			cst_set = &cst_set_beginning;
		}
		else
		{
			fill_cst_set( cst_set_middle, pose, just_use_bb_heavy_atoms );
			cst_set = &cst_set_middle;
		}
		pose.set_constraints( *cst_set );


		//set weights and constraints
		weight_map.set_weight( FA_REP, fa_rep );
		weight_map.set_weight( ATOMPAIR_CST,
		 500.0 * cst_weight / cst_set->count_atompair_constraints() );

		//minimize
		pose_mininmize_interface(pose,min_tolerance,weight_map,"dfpmin");
		set_pose_flag(false);

		mc_global_track::mc_score::score = score12(); //find the score with the score12 weights;
		//if ii == 1, this is the first call to monte_carlo since nblist turned on; therefore reset
		if ( ii == 1 ) { monte_carlo_reset();}
		else { monte_carlo(1); } //avoid loss of LOW in call to reset for subsequent design/minimize rounds
		dlogfile_output(++output_counter);

		//make_named_pdb( output_file + "_dmistep_" + lead_zero_string_of(
		//	ii, 4 ) + ".pdb", true );
		pose.reset_constraints(); //to avoid a dangling reference
	}

	dlogfile_output(++output_counter);
}

////////////////
// SJF setting up the docking flags for peptide
// essentially, docking_local_refine with bb dofs of the peptide
////////////////
void
setup_peptide_ligand_docking_flags()
{
  using namespace docking;

  minimize_backbone=true; //SJF for protein-peptide applications one would usually need -nominimize1 in the command line
  norepack1=false;
  norepack2=false;
  prepack_mode=false;
  docking_local_refine=true;
  docking_fullatom_flag=true;
  dock_rtmin=true;
  docking_mcm_flag=true;
/*
// SJF copied from pose_docking_main
  minimize_exclude_sstype( false, false );
  minimize_set_vary_phipsi( true );
  minimize_set_vary_omega( false );
  minimize_set_vary_chi( true );
  minimize_set_vary_rb_trans( true );
  minimize_set_vary_rb_angle( true );
  minimize_set_local_min( false, 0 );
  pose_set_use_nblist( false );
*/
}

////////////////////////////////////////////////////////////////////////////////
/// @begin design_and_dock_mcm
///
/// @brief
///jk iteratively design and dock_mcm
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
design_and_dock_mcm()
{
  using namespace design;
  using namespace files_paths;
  using namespace fullatom_flag;
  using namespace misc;
  using namespace param;
  using namespace pdb;

  // jk Variables to describe general control flow
  const int nloop = 2;

  // jk DOCKING SETUP (adapted from dock_structure)
  calc_docking_rmsd();
  docking_store_initial_rms();
  set_docking_interface_pack( true ); // pack only the interface
  select_rotamer_set( "large" );
  set_fullatom_flag(true);
  pivot_repack(true); // repacks residues perturbed by pivot
  docking_repack(true); // repacks interface residues
  mc_global_track::mc_score::score = score12();
  monte_carlo_reset();

  // jk DESIGN SETUP (adapted from design_interface)
  bool file_exists,rotamers_exist,make_output_file;

  FArray1D_int neighbors( MAX_RES()() );
  FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );
  FArray1D_char sschar( MAX_RES()() );           //yl

  //yl create local pose and prepare the Epositions for misc
  pose_ns::Pose pose;

  full_atom = true;

// jk Comments below taken from design_interface subroutine

//bk this procedure creates its own output files (separate from the
//bk main rosetta functions) because multiple designs can be output for
//bk one starting structure.  If the design procedure has already begun for
//bk this starting structure, return from here so that rosetta can begin on
//bk the next structure.
  ss_from_phipsi(sschar,phi,psi,total_residue);
  check_design_pdb_exists(1,file_exists);
  if ( file_exists && !overwrite_pdbs && !touch_before_reading ) {
    std::cout << "this file has already been started - move to next" << std::endl;
    return;
  }
  if ( touch_before_starting ) {
    create_design_pdb_file(1);
  }

// jk Comments below taken from design_interface subroutine
//bk identify interface residues, and set interface_residue to true so that
//bk they will be redesigned
//bk Only the interface between the first two chains in a pdb is
//bk considered here!!!

//jk When the -fix_target_seq flag is used,
//jk the int passed in refers to the chain with fixed sequence
//jk note: this chain will be repacked

//jk When the -repack_interface flag is used, the both chains are repacked

  interface_residues();

  rotamers_exist = include_inputchi;
  make_output_file = true;

  // jk CALLS TO DOCKING AND DESIGN HERE
  for ( int i = 1; i <= nloop; ++i ) {
    std::cout << "DESOCK STARTING DESIGN CYCLE " << i << std::endl;
    fullatom_nonideal_initialized_pose_from_misc( pose );

    PackerTask Task( pose );
    Task.set_task("design",make_output_file,interface_residue,rotamers_exist);
    //bk set variables that specify which residues to vary
    Task.setup_residues_to_vary();
    pack_rotamers( pose,Task );
    pose.copy_to_misc();
    make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);
    std::cout << "DESOCK STARTING DOCKING CYCLE " << i << std::endl;
    docking_mcm_protocol();
    // For dock_min instead of dock_mcm
    //    docking_rigidbdy_minimize_trial("dfpmin",score12,0.02);
  }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_design_minimize_interface
///
////////////////////////////////////////////////////////////////////////////////
void
dock_design_minimize_interface()
{
	//apl very similar to John K's desock protocol
	//apl relying upon Glenn B's design_minimize_interface protocol
	using namespace design;
	using namespace exchi_flags;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace int_fullatom_energies;
	using namespace misc;
	using namespace param;
	using namespace pdb;

	files_paths::output_coord = false;

	RotamerOptions saved_rotamer_options = active_rotamer_options;

	time_t starttime = time(NULL);
	time_t currtime;

	pose_ns::Pose starting_pose;
	fullatom_nonideal_initialized_pose_from_misc( starting_pose );
	//bool global_design_trials = design_trials; //do not allow allow_insert to become true inside packer
	//design_trials = false;

	// start conditions (docking flags)
	docking::docking_randomize1     = truefalseoption("randomize1");
	docking::docking_randomize2     = truefalseoption("randomize2");
	docking::docking_small_perturbation = truefalseoption("dock_pert") ||
		(!docking::docking_randomize1 && !docking::docking_randomize2);

	bool const two_rounds_ddmi( truefalseoption("two_rounds_ddmi"));

	float const tolerance = 18;
	float const init_score4 = score4();
	std::cerr << "init score4" << init_score4 << std::endl;

	float const first_round_dg_cutoff =
		( truefalseoption("ddmi_dg_round1_thresh") ? realafteroption("ddmi_dg_round1_thresh") : 0.0f );
	float const second_round_dg_cutoff =
		( truefalseoption("ddmi_dg_thresh") ? -1 * realafteroption("ddmi_dg_thresh") : -4.0f );

	//currtime = time( NULL );
	//if ( ((double) currtime - starttime) / 60  > timelimit ) break;

	int docking_attempts = 0;

	set_fullatom_flag( false );
	docking::docking_fullatom_flag = false;
	while ( true )
	{
		currtime = time( NULL );
		if ( ((double) currtime - starttime) / 60  > timelimit ) break;


		pose_ns::Pose docking_pose;
		docking_pose = starting_pose;
		pose_docking_build_simple_tree( docking_pose, total_residue, docking::part_end(1) );
		set_pose_flag( true );
		docking_pose.set_fullatom_flag(false);
		//set_fullatom_flag(false);

		++docking_attempts;
		std::cout << "DOCK_DES_MIN_INTER: STARTING DOCKING CYCLE " << docking_attempts << std::endl;

		// grid dock
		if ( truefalseoption("prog_grid_dock") ) {
				progressive_grid_dock_move( docking_pose );
		}

		//std::cout << "APL: STARTING DOCK_STRUCTURE" << std::endl;
		pose_docking_standard_protocol( docking_pose );

		//std::cerr << "Docking_pose.score(score4): " << docking_pose.score( score4 ) << std::endl;

		docking_pose.copy_to_misc();
		set_pose_flag( false );
		monte_carlo_reset();

		mc_global_track::mc_score::score = score4();
		//std::cerr << "Docked Score: " << mc_global_track::mc_score::score << std::endl;
		if ( mc_global_track::mc_score::score > (init_score4 + tolerance) )
		{
			continue; //reject poorly docked structures
		}

		//re-dock if site constraints unmet
		docking_evaluate_site_score();
		if ( docking_site_constr_filter() )
		{
			// keep track of last docked structure
			//pose_from_misc( starting_pose, fullatom, ideal_pose, coords_init );
			//std::cout << "Docking met site constraints filter; starting design" << std::endl;
			break;
		}

		//dump_pdb(output_file + "_docksite_" + lead_zero_string_of(docking_attempts,4) + ".pdb");

	}

	//currtime = time( NULL );
	//if ( ((double) currtime - starttime) / 60  > timelimit ) break;

	std::cout << "APL: STARTING DESMININTER" << std::endl;
	//design_trials = global_design_trials;

	if ( two_rounds_ddmi )
	{
		active_rotamer_options.ex1 = true; active_rotamer_options.ex1_sample_level = EX_TWO_FULL_STEP_STDDEVS;
		active_rotamer_options.exOH = true;
		active_rotamer_options.ex2 = true; active_rotamer_options.ex2_sample_level = EX_ONE_STDDEV;
		active_rotamer_options.ex3 = false; active_rotamer_options.ex3_sample_level = NO_EXTRA_CHI_SAMPLES;
		active_rotamer_options.ex4 = false; active_rotamer_options.ex4_sample_level = NO_EXTRA_CHI_SAMPLES;
		active_rotamer_options.setup_exflags();
	}

	set_fullatom_flag( true );
	mc_global_track::mc_score::score = score12();

	{//scope
	float const r1_start_fa_rep = 0.85;
	float const	r1_start_min_tolerance = 0.01;
	float const	r1_start_cst_weight = 10;

	float const r1_final_fa_rep = 1.0;
	float const	r1_final_min_tolerance = 0.0000001;
	float const	r1_final_cst_weight = 0.05;

	int const r1_num_iter = 8; //total # if iterations
	int const r1_num_ramp = 6; //# of total iter to ramp weights
	bool const r1_constrain_from_initial_coords = false;

	design_minimize_interface(
		r1_start_fa_rep, r1_start_min_tolerance, r1_start_cst_weight,
		r1_final_fa_rep, r1_final_min_tolerance, r1_final_cst_weight,
		r1_num_iter, r1_num_ramp,
		r1_constrain_from_initial_coords
	);
	}
	recover_LOW( score12 );


	if ( two_rounds_ddmi )
	{
		int dgUns1( 0 );
		float dSASA1( 0.0f );
		float  dG_bind_round1;
		dg_for_complex( dG_bind_round1, dgUns1, dSASA1 );

		std::cout << "dG_bind_round1 " << dG_bind_round1 << std::endl;
		if ( dG_bind_round1 > first_round_dg_cutoff ) return;  //reject poorly bound complexes

		active_rotamer_options = saved_rotamer_options;

		std::cout << "APL: STARTING ROUND 2 DESMININTER" << std::endl;

		{//scope
		float const r2_start_fa_rep = 1.0;
		float const	r2_start_min_tolerance = 0.01;
		float const	r2_start_cst_weight = 2;

		float const r2_final_fa_rep = 1.0;
		float const	r2_final_min_tolerance = 0.0000001;
		float const	r2_final_cst_weight = 0.05;

		int const r2_num_iter = 3; //total # if iterations
		int const r2_num_ramp = 2; //# of total iter to ramp weights
		bool const r2_constrain_from_initial_coords = false;

		design_minimize_interface(
			r2_start_fa_rep, r2_start_min_tolerance, r2_start_cst_weight,
			r2_final_fa_rep, r2_final_min_tolerance, r2_final_cst_weight,
			r2_num_iter, r2_num_ramp,
			r2_constrain_from_initial_coords
		);
		}

		recover_LOW( score12 );
	}



	int dUns2( 0 );
	float dG_bind_round2( 0.0f), dSASA2( 0.0f);

	dg_for_complex( dG_bind_round2, dUns2, dSASA2 );

	bool useUnsFilter( false ), use_dG_dSASA_ratio_filter( false );
	int dUnsFilter( 0 );
	float dG_dSASA_ratio_filter( 0.0f );

	std::cout << "Structure " << main_job_distributor::jd->get_curr_outnum();
	std::cout << " dG_bind_final_round " << dG_bind_round2 << std::endl;
	if ( truefalseoption("ddmi_dUns_filter") )
	{
		useUnsFilter = true;
		dUnsFilter = intafteroption("ddmi_dUns_filter");
	}

	if ( truefalseoption("ddmi_dG_dSASA_ratio_filter") )
	{
		use_dG_dSASA_ratio_filter = true;
		dG_dSASA_ratio_filter = realafteroption("ddmi_dG_dSASA_ratio_filter");
	}

	if ( dG_bind_round2 < second_round_dg_cutoff &&
		( ! useUnsFilter || dUns2 < dUnsFilter) &&
		( ! use_dG_dSASA_ratio_filter || (dG_bind_round2 / dSASA2) > dG_dSASA_ratio_filter) )
	{
		design_output( total_residue, res, res_variant, full_coord, main_job_distributor::jd->get_curr_outnum() );
		//std::cerr << "outputting coord" << std::endl;
		//files_paths::output_coord = true; //let the job_distributor output the structure.
	}
}

void
point_mut()
{
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );
	PackerTask task( pose );

	int const pdb_res_to_mutate( intafteroption("point_mutation") );
	//find the residue in rosetta numbering that corresponds to the
	//residue in the pdb numbering; O(N) operation that should be O(1)
	int residue_to_mutate( 0 );
	for (int ii = 1; ii <= pose.total_residue(); ++ii)
	{
		if ( pdb::pdb_res_num( ii ) == pdb_res_to_mutate )
		{
			residue_to_mutate = ii;
			break;
		}
	}
	if ( residue_to_mutate == 0 )
	{
		std::cout << "ERROR residue to mutate not found in input structure." << std::endl;
		std::cout << "Are you using PDB numbering? (you should be.) Are there multiple chains?" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string new_aa_at_res( stringafteroption("new_aa") );
	if ( new_aa_at_res.length() != 1 ) {
		std::cout << "ERROR new_aa should be a single 1-letter character" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	char const new_aa_char = new_aa_at_res[ 0 ]; //first character
	int new_aa;
	num_from_res1( new_aa_char, new_aa );

	//redesign residue_to_mutate; allow neighbors to repack
	task.get_designmap().set( residue_to_mutate, new_aa );
	for (int ii = 1; ii <= pose.total_residue(); ++ii)
	{
		if ( ii == residue_to_mutate ) continue;

		bool are_neighbors;
		float dis2;

		are_they_neighbors( pose.res( residue_to_mutate ), pose.res( ii ),
			pose.full_coord()(1,1,residue_to_mutate), pose.full_coord()(1,1,ii),
			dis2, are_neighbors );

		if ( are_neighbors )
		{
			task.get_designmap().set( ii, pose.res( ii ) );
		}
	}

	pack_rotamers( pose, task );
	pose.copy_to_misc();
	update_sequence();
	monte_carlo( -1 ); // force the job_distributor to output the new structure
}



//////////////////////////////////////////////////////////////////////////////
/// @ begin dg_for_complex
/// @ breif takes complex appart; repacks residues on each side
///
/// @ param dG_bind - [out] - the delta G of binding
/// @ param dgUns - [out] - the number uns groups created in binding
/// @ param dSASA - [out] - the change in the solvent accessible
///    surface area during binding (negative when surface area gets buried)
//////////////////////////////////////////////////////////////////////////////

void
dg_for_complex(
	float & dG_bind,
	int & dgUns,
	float & dSASA
)
{
	using namespace design;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	//dump_fullatom_pdb("dg_for_complex_begin.pdb");

	scorefxn();
	float const dG_bound = get_fullatom_totalE();

	int n_guns_bound( 0 );

	{ //scope
	//count group uns of complex
	copy_hbenergies();
	std::list < unsatisfied_buried_polar > uns_list;
	std::list < unsatisfied_buried_group > group_uns_list;
	find_unsatisfied_hbonds(uns_list, group_uns_list);
	n_guns_bound = group_uns_list.size();
	}

	float sasa_bound = calc_sasa();

	pose_ns::Pose dmipose;
	fullatom_nonideal_initialized_pose_from_misc( dmipose );

	interface_residues();

	//prepare the task using information about the complex before
	//taking the complex apart
	PackerTask interface_residue_relax_task( dmipose );
	bool const make_output_file( false );
	bool const rotamers_exist( true );

	interface_residue_relax_task.set_task(
		"packrot", make_output_file, interface_residue, rotamers_exist );
	interface_residue_relax_task.setup_residues_to_vary();
	disable_repack_prolines( interface_residue_relax_task );

	//stolen from docking.cc
	//pull the complex apart
	const float away_dist = { 99.0 };
	const float zero_1 ( 0.0f );
	const FArray1D_float zero_3( 3, 0.0f );
	FArray1D_float T( 3 );
	create_docking_trans_vector( T, -away_dist, zero_1, zero_1 );
	apply_rigid_body( T, zero_3 );


	include_inputchi = true;
	//use task defined by interface residues before complex was pulled apart
	//pack_rotamers( res, full_coord, res_variant, total_residue, phi, psi, task );

	pose_ns::Pose complex_apart;
	fullatom_nonideal_initialized_pose_from_misc( complex_apart );
	pack_rotamers( complex_apart, interface_residue_relax_task );

	complex_apart.copy_to_misc();
	scorefxn();
	float const dG_unbound = get_fullatom_totalE();


	int n_guns_apart( 0 );
	{//scope
	//couht group uns after complex separated and relaxed
	copy_hbenergies();
	std::list < unsatisfied_buried_polar > uns_list;
	std::list < unsatisfied_buried_group > group_uns_list;
	find_unsatisfied_hbonds(uns_list, group_uns_list);
	n_guns_apart = group_uns_list.size();
	}

	float sasa_apart = calc_sasa();

	//std::cout << "dG_bound: " << dG_bound << ", dG_unbound: " << dG_unbound << std::endl;
	//
	//std::cout << "n_guns_bound: " << n_guns_bound << ", n_guns_apart: " << n_guns_apart;
	//std::cout << ", dUns " << n_uns_bound - n_uns_apart << std::endl;
	//
	//dump_fullatom_pdb("dg_for_complex_apart2.pdb");

	//create_docking_trans_vector( T, away_dist, zero_1, zero_1)
	//apply_rigid_body( T, zero_3 ); //pretend to move the complex back together


	dmipose.copy_to_misc();
	monte_carlo_reset();
	update_nblist();
	scorefxn();

	dG_bind = dG_bound - dG_unbound;
	dgUns = n_guns_bound - n_guns_apart;
	dSASA = sasa_bound - sasa_apart;

}


//assumes packerTask created from misc;
//avoids "relaxing" prolines away from ideal bond geometry due to
//an oddity in proline creation code.
void
disable_repack_prolines( PackerTask & task )
{
	using namespace misc;
	using namespace param_aa;

	for (int ii = 1; ii <= total_residue; ++ii)
	{
		if ( task.get_designmap().repack_residue(ii) && task.get_designmap().get(ii,aa_pro))
		{
			task.get_designmap().disable( ii, aa_pro );
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rdesign_setup
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rdesign_setup()
{
	using namespace files_paths;
	using namespace rdesign_control;

//car set temperature range...

	std::string filename;

//car format for refinement setup file
//car name: 2ptl_.refine
//car lines in all caps are comments:
//car ---------------------
//car TEMPERATURE
//car 5.0
//car 0.5
//car REGULAR MOVES
//car 40
//car 500
//car 0.0 5.0 5.0
//car linmin MOVES
//car 10
//car 100
//car 0.0 3.0 3.0
//car dfpmin MOVES
//car 10
//car 100
//car 0.0 1.0 1.0
//car ---------------------

	filename = start_path + protein_name_prefix + protein_name + protein_chain + ".refine";
	std::cout << "Looking for refinement input file: " << filename << std::endl;
	start_x.clear();
	start_x.open( filename );
	if ( !start_x ) {
		std::cout << "unable to find refine input file: "<< start_x.filename() << std::endl;
		std::cout << "using default values" << std::endl;
		start_x.clear();
		return;
	}
	start_x >> skip; // temperatures
	start_x >> init_temp >> skip;
	start_x >> final_temp >> skip;
	start_x >> skip; // regular moves
	start_x >> nloop1 >> skip;
	start_x >> cycles1 >> skip;
	start_x >> bite( 3, helix1 ) >> skip( 1 ) >>
	 bite( 3, strand1 ) >> skip( 1 ) >> bite( 3, loop1 ) >> skip;
	start_x >> skip; // linmin moves
	start_x >> nloop2 >> skip;
	start_x >> cycles2 >> skip;
	start_x >> bite( 3, helix2 ) >> skip( 1 ) >>
	 bite( 3, strand2 ) >> skip( 1 ) >> bite( 3, loop2 ) >> skip;
	start_x >> skip; // dfpmin moves
	start_x >> nloop3 >> skip;
	start_x >> cycles3 >> skip;
	start_x >> bite( 3, helix3 ) >> skip( 1 ) >>
	 bite( 3, strand3 ) >> skip( 1 ) >> bite( 3, loop3 ) >> skip;

	start_x.close();
	start_x.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin shakenbake_backbone
///
/// @brief
///ctsa  aggressive backbone movement, optionally including design
///
/// @detailed
///ctsa  This is still an experimental protocol; it's neither finished
///ctsa  nor checked-in for general use.
///
/// @param  is_seq_fixed - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
shakenbake_backbone( bool is_seq_fixed )
{
	using namespace misc;
	using namespace param;

// constants
	bool const use_feedback_potential = { true };
//	float const rabid_wobble_cutoff = { 2000.0 };
	 // msd cutoff for a rabid wobble move

// local
	int cycles_per_residue,cycles,subcycles;
	float start_pack_temp,end_pack_temp;
	float cutoff,pack_lj;
//------------------------------------------------------------------------------

// start shakin'
//      centroid_expand()

// setup fullatom parameters
	cycles_per_residue = 1;
	cycles = cycles_per_residue * total_residue + 30;
	subcycles = 10;
	pack_lj = 1.0;
	cutoff = 60.; // msd cutoff for wobble/crank
	start_pack_temp = 1.4;
	end_pack_temp = 0.8;

//car set options in case of loop mode or constraints
	classical_constraints::BOUNDARY::set_max_seqSep(total_residue);
	score_set_cst_weight(1.0);
	score_set_cst_mode(3);
	score_set_dpl_weight(5.0);

	if ( !is_seq_fixed ) {
		rdesign_setup();
	}
	set_fullatom_flag(true);
	monte_carlo_set_simannealing(true);
	choose_frag_set_top_N_frags( MAX_NEIGH() );
	minimize_exclude_sstype(false,false);
	minimize_set_local_min(true,10);
	set_use_nblist(false); // nb list can cause clashes when large movements
	// take place in fragment insertion or minimization

//  change fullatom potential
	if ( use_feedback_potential ) {
		set_feedback_potential();
	}

	score_set_lj_weight(1.0);


}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_feedback_potential
///
/// @brief
///ctsa    experimental parameterization method, derived from Docta Brian's,
///ctsa  based on training for the native aa in a packing feedback loop together
///ctsa  with an aa natural frequency energy
///
/// @detailed
///ctsa    this isn't worked out for general use, and for the moment I don't plan
///ctsa  to keep the cvs version of the parameterization up to date with the latest
///ctsa  chnages in the rosetta potential.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_feedback_potential()
{

	using namespace design;
	using namespace param_aa;
	using namespace param_pack;

	if ( !active_rotamer_options.ex1 && !active_rotamer_options.ex2 ) {

		pack_wts.set_Watr(0.800);
		pack_wts.set_Wsol(0.800);
		pack_wts.set_Wpair(0.542);
		pack_wts.set_Wplane_total(0.0);
		pack_wts.set_Wdun(0.857);
		pack_wts.set_Wrep(0.800);
		pack_wts.set_Wintra(0.800); // as pack_wts.Wrep
		pack_wts.set_Wone(1.049);
		pack_wts.set_Whbond_bbsc(0.0, 1.388, 1.388); //bb sc bb_sc

		pack_wts.set_Wlig_atr(0.800); //cmj changed from 0.800, changed back from 2.5
		pack_wts.set_Wlig_sol(0.800);
		pack_wts.set_Wlig_cou(0.542); //cmj check
		pack_wts.set_Wlig_rep(0.800);
		pack_wts.set_Wlig_hb(1.388);
		pack_wts.set_Wlig_vir(1.000);

		pack_wts.set_Waa( aa_ala, 0.255);
		pack_wts.set_Waa( aa_cys, 0.);
		pack_wts.set_Waa( aa_asp, 2.078);
		pack_wts.set_Waa( aa_glu, 2.065);
		pack_wts.set_Waa( aa_phe,-3.233);
		pack_wts.set_Waa( aa_gly, 1.27);
		pack_wts.set_Waa( aa_his,-1.917);
		pack_wts.set_Waa( aa_ile,-0.663);
		pack_wts.set_Waa( aa_lys, 1.556);
		pack_wts.set_Waa( aa_leu,-0.304);
		pack_wts.set_Waa( aa_lys, 0.024);
		pack_wts.set_Waa( aa_asn, 0.694);
		pack_wts.set_Waa( aa_pro, 0.228);
		pack_wts.set_Waa( aa_gln, 0.964);
		pack_wts.set_Waa( aa_arg, 2.293);
		pack_wts.set_Waa( aa_ser, 0.965);
		pack_wts.set_Waa( aa_thr, 0.786);
		pack_wts.set_Waa( aa_val,-0.217);
		pack_wts.set_Waa( aa_trp,-4.222);
		pack_wts.set_Waa( aa_tyr,-2.622);
  //pack_wts.print();//JSS debug

	} else if ( active_rotamer_options.ex1 && ! active_rotamer_options.ex2 ) {

		pack_wts.set_Watr(0.800);
		pack_wts.set_Wsol(0.800);
		pack_wts.set_Wpair(0.529);
		pack_wts.set_Wdun(0.799);
		pack_wts.set_Wrep(0.800);
		pack_wts.set_Wintra(0.800); // as pack_wts.set_Wrep
		pack_wts.set_Wone(1.009);
		pack_wts.set_Whbond_bbsc(0.0, 1.411, 1.411); //bb sc bb_sc

		pack_wts.set_Wlig_atr(0.800); //cmj changed from 0.800, changed back from 2.5
		pack_wts.set_Wlig_sol(0.800);
		pack_wts.set_Wlig_cou(0.529); //cmj check
		pack_wts.set_Wlig_rep(0.800);
		pack_wts.set_Wlig_hb(1.411);
		pack_wts.set_Wlig_vir(1.000);

		pack_wts.set_Waa( aa_ala, 0.458);
		pack_wts.set_Waa( aa_cys, 0.);
		pack_wts.set_Waa( aa_asp, 2.196);
		pack_wts.set_Waa( aa_glu, 2.087);
		pack_wts.set_Waa( aa_phe,-3.626);
		pack_wts.set_Waa( aa_gly, 1.444);
		pack_wts.set_Waa( aa_his,-2.006);
		pack_wts.set_Waa( aa_ile,-0.647);
		pack_wts.set_Waa( aa_lys, 1.52);
		pack_wts.set_Waa( aa_leu,-0.395);
		pack_wts.set_Waa( aa_met,-0.163);
		pack_wts.set_Waa( aa_asn, 0.811);
		pack_wts.set_Waa( aa_pro, 0.371);
		pack_wts.set_Waa( aa_asn, 1.027);
		pack_wts.set_Waa( aa_arg, 2.212);
		pack_wts.set_Waa( aa_ser, 1.097);
		pack_wts.set_Waa( aa_thr, 0.93);
		pack_wts.set_Waa( aa_val,-0.165);
		pack_wts.set_Waa( aa_trp,-4.323);
		pack_wts.set_Waa( aa_tyr,-2.826);
  //pack_wts.print();//JSS debug
	} else {
		std::cout << "No potential for current rotamer set" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin dlogfile_output
///
/// @brief
///
/// @detailed
///
/// @param  jk - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dlogfile_output( int jk )
{
	using namespace counters;
	using namespace misc;
	using namespace scores;
	using namespace start;

//      float fav,alw,gen;
	float rms_to_start;

	retrieve_best_pose();
//      eval_procheck_rama(phi,psi,res,total_residue,fav,alw,gen);
	fast_rms_x(position,start_position,rms_to_start);

	if ( jk <= 0 ) {
		std::cout << "jk nlowacc   score    low_sc st_rms   low_rms    atr    rep" << std::endl;
	}
	std::cout << I( 2, jk ) << ' ' << I( 5, n_low_accept ) << "  " <<
	 F( 9, 2, scorefxn() ) << ' '; // Sequence point
	std::cout <<
	 F( 9, 2, mc_global_track::mc_score::low_score ) << ' ' <<
	 F( 5, 2, rms_to_start ) << ' ' <<
	 F( 5, 2, mc_global_track::diagnose::low_rms ) << "   " <<
	 F( 7, 2, fa_atr_score ) << "  " <<
	 F( 7, 2, fa_rep_score ) << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_design_trial
///
/// @brief
///bk design a new sequence
///
/// @detailed
///  options for setup design region:
///  1.design_loops_dock:design the whole protein
///  2.design_loops_hold:design only the loop region and pack the residues around the loop
///   (i) if there is no resfile supplied,automatically design the loop positions and neighbors
///   (ii)if there is a resfile,then use resfile to determine which region for design and repacking
///
/// @param  scoring_function - [in/out]? -
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_design_trial(
	Scoring_Function scoring_function,
	std::string const & mode
)
{
	using namespace misc;
	using namespace param;
	using namespace design;
	using namespace loops_ns;
	using namespace start;

	FArray1D_bool residues_to_pack( MAX_RES()() );
	FArray1D_bool contact_residue( MAX_RES()() );
	//region_design to be true for design_loops_hold submode
	//false for design_loops_dock submode

	bool region_design = design::design_loops_flags::design_loops_hold;
	bool rotamers_exist; //make_output_file;

	resetphipsi();
	refold(1,total_residue);

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );
	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );

	//glb pack_mode (rather than mode) is passed to pack_rotamers
	std::string pack_mode( mode );
	static bool flip_flop = { true };

//bk only start from current sequence in every other design run
	if ( mode == "design" ) {
		if ( flip_flop ) {
			flip_flop = false;
		} else {
			flip_flop = true;
		}
		rotamers_exist = flip_flop;
	} else {
		rotamers_exist = true;
	}


	if ( region_design ){
		contact_residue = false;
		get_loop_pack_residues(contact_residue,"all");
		if ( mode == "pack_loop"){
			pack_mode = "packrot";
			for (int i = 1;i <= total_residue; ++i){
				residues_to_pack(i) = contact_residue(i);
			}
		}else if ( files_paths::resfile != "none"){
			for (int i = 1;i <= total_residue; ++i){
				residues_to_pack(i) = contact_residue(i);
			}
		}else{
			use_design_matrix = true;
			design_matrix = false;
			for (int i = 1; i <= num_loop; ++i){
				for (int j = loop_begin(i),je = loop_end(i); j <= je; ++j){
					for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa){
						if ( aa != 2 ){//don't design cys
							design_matrix(aa,j) = true;
						}
					}
				}
			}
			for (int i = 1;i <= total_residue;++i){
				if (contact_residue(i)){
					design_matrix(res(i),i) = true;
				}else{
					residues_to_pack(i) = false;
				}
			}
		}
	}else{
			for ( int i = 1; i <= total_residue; ++i ) {
			residues_to_pack(i) = true;
		}
	}

	// design_restrict_interface:
  // res      at interface - do what resfile says
	// res  not at interface - if design return to starting aa and repack
	//                         else no pack or design
	if( mode == "design_restrict_interface" ) {

		pack_mode = "design";
		interface_residues();

		for ( int ii = 1; ii <= total_residue; ++ii ) {
			residues_to_pack(ii) = interface_residue(ii);
			if ( !interface_residue(ii) ){
				//std::cout << " not interface " << ii << std::endl;
				residues_to_pack(ii) = false;
				if ( res(ii) != start_res(ii) ){
					Task.get_designmap().set(ii,start_res(ii)); //no redesign
					residues_to_pack(ii) = true;
				}
			}
		}
	}

	//glb set residues to be packed in pack_interface mode
	if ( mode == "pack_interface" ) {
		interface_residues();
		pack_mode = "packrot";
		//std::cout << "repacking interface residues: ";
		for ( int i = 1; i <= total_residue; ++i ) {
			residues_to_pack(i) = interface_residue(i);
			//if (residues_to_pack(i) ) {
			//	std::cout << " " << i << " ";
			//}
		}
		//std::cout << std::endl;rotamers_exist,

	}


//bk with mode set to design, pack returns a new sequence
//	make_output_file = false;

	Task.set_task(pack_mode,false,residues_to_pack,rotamers_exist);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task);
	pose.copy_to_misc();

//bk update sequence arrays
	update_sequence();
	update_nblist();

	mc_global_track::mc_score::score = scoring_function();
	monte_carlo(1);

	if ( mode == "design" ) {

		std::cout << "monte carlo id for design run " << get_monte_carlo_accept() <<
		 std::endl;
		std::cout << "rotamers_exist flag " << rotamers_exist << std::endl;
		std::cout << "score vs  best_score" << SS( mc_global_track::mc_score::score ) << SS( mc_global_track::mc_score::best_score ) <<
		 std::endl;

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_design_force_accept
///
/// @brief
///bk design a new sequence
///
/// @detailed
///
/// @param  scoring_function - [in/out]? -
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_design_force_accept(
	Scoring_Function scoring_function,
	std::string const & mode
)
{

	using namespace misc;
	using namespace param;

	FArray1D_bool which_residue( MAX_RES()() );
	bool rotamers_exist = true;

	resetphipsi();
	refold(1,total_residue);
	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );

	for ( int i = 1; i <= total_residue; ++i ) {
		which_residue(i) = true;
	}

	Task.set_task(mode,false,which_residue,rotamers_exist);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task);
	pose.copy_to_misc();

//bk update sequence arrays
	update_sequence();
	update_nblist();

	mc_global_track::mc_score::score = scoring_function();
	monte_carlo_accept_best();
	monte_carlo_accept_low();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin write_sequence
///
/// @brief
///
/// @detailed
///
/// @param  jk - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_sequence( int jk )
{
	using namespace misc;
	using namespace param;
	using namespace start;

	int nsame = 0;
	FArray1D_char caa( total_residue, ' ' );

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( res(i) == start_res(i) ) {
			++nsame;
		}
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		res1_from_num(res(i),caa(i));
	}

	std::cout << "seq " << jk;
	for ( int i = 1; i <= total_residue; ++i ) {
		std::cout << caa(i);
	}
	std::cout << SS( float(nsame)/float(total_residue) ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_optE
///
/// @brief
///bk output energies of rotamers tried in native proteins,
///bk file is used for optimizing weights
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_optE()
{
	using namespace design;
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace param_pack;

	static bool open = { false };

	if ( !open ) {
		opte_x.clear();
		opte_x.open( (score_path + Eout).c_str() );
		if ( !opte_x ) {
			std::cout << "having trouble opening output file for rot E" <<
			 "trying to open " << score_path << Eout << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		open = true;
	}

	// jk  read weights from a file, if desired
	if ( weightfile::use_weightfile ) {
		ReadWeights( weightfile::weight_fname );
	} else {
		RetrieveWeightsToCurrent( PW_OPTE );
	}

	//mj we only use either the pair term or generalized born electrostatocs
	//mj so we ensure that in the optE output the other column is zero
	if ( gen_born) {
		pack_wts.set_Wgb_elec(1.0);
		pack_wts.set_Wpair(0.0);
	}
	else {
		pack_wts.set_Wgb_elec(0.0);
		pack_wts.set_Wpair(1.0);
	}
	//mj end

	if ( optE_inter ) interface_residues();

	fullatom_energy_full(res, res_variant, full_coord, total_residue);
//bk   rotamer trials senses optE is true and outputs rotamer energies to a file
	rotamer_trials();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin fixed_backbone
///
/// @brief
///
/// @detailed
///
/// @param  packmode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fixed_backbone( std::string const & packmode )
{
	using namespace design;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pdb;

	FArray1D_int res_orig( MAX_RES()() );
	FArray1D_string res_orig3( MAX_RES()() );
	FArray1D_bool printresinfo( MAX_RES()() );
	FArray1D_char sschar( MAX_RES()() );
	FArray1D_bool which_residue( MAX_RES()() );
	bool rotamers_exist, make_output_file = false, file_exists;
	FArray1D_int neighbors( MAX_RES()() );

	FArray1D_int cluster_sizes( MAX_RES()() );
	FArray2D_int cluster_members( MAX_RES()(), MAX_RES()() );
	int nclusters, cluster;
	bool make_only_one_cluster;

	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );
	DesignMap design_map( MAX_RES(), misc::res );

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	full_atom = true;

//bk this procedure creates its own output files (separate from the
//bk main rosetta functions) because multiple designs can be output for
//bk one starting structure.  If the design procedure has already begun for
//bk this starting structure, return from here so that rosetta can begin on
//bk the next structure.
	check_design_pdb_exists(1,file_exists);
	if ( file_exists && !overwrite_pdbs && !touch_before_reading ) {
		std::cout << "this file has already been started - move to next" << std::endl;
		return;
	}
	if ( touch_before_starting ) {
		create_design_pdb_file(1);
	}

//bk save original sequence
	for ( int i = 1; i <= total_residue; ++i ) {
		res_orig(i) = res(i);
	}

	rotamers_exist = include_inputchi; // don't automatically include native rot

//bk examine flag which determines if whole protein will be designed
//bk simultaneously of if chain will be walked through and designed
//bk in small clusters.  The second method is preferable if you have large
//bk numbers of rotamers
	if ( design_in_pieces ) {
		make_only_one_cluster = false;
	} else {
		make_only_one_cluster = true;
	}
	get_clusters(pose,nclusters,cluster_sizes,
		cluster_members,make_only_one_cluster);

	for ( cluster = 1; cluster <= nclusters; ++cluster ) {

		//bk specify which residues to redesign - may be modified
		//bk if packer finds a resfile
		for ( int i = 1; i <= total_residue; ++i ) {
			which_residue(i) = false;
		}
		for ( int i = 1, ie = cluster_sizes(cluster); i <= ie; ++i ) {
			which_residue(cluster_members(i,cluster)) = true;
		}

		if ( cluster == nclusters ) make_output_file = true;

		//bk with mode set to design, pack returns a new sequence

		// *** This is the default behavior ***
		//yl, Create PackerTask and setup values before pass into pack_rotamers
		PackerTask Task( pose );

		//bk set variables that specify which residues to vary
		Task.set_task(packmode,make_output_file,which_residue,rotamers_exist);
		Task.setup_residues_to_vary();
		pack_rotamers( pose, Task );
		pose.copy_to_misc();

//bk update sequence arrays
		for ( int i = 1; i <= total_residue; ++i ) {
			name_from_num( res(i), residue3(i) );
			res1_from_num( res(i), residue1(i) );
		}

//bk score new sequence
		score_set_try_rotamers(false);
		mc_global_track::mc_score::score = scorefxn();

		make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

//bk output list comparing design sequence to native sequence
		if ( cluster == nclusters && sqc != "none" ) {
			sqc_x.clear();
			sqc_x.open( (score_path + sqc).c_str(),
			 std::ios_base::out|std::ios_base::app );
			if ( !sqc_x ) {
				std::cout << "trouble opening output file for seqcompare" <<
				 " trying to open " << score_path << sqc << std::endl;
				std::cout << "sqc " << sqc << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__); //Objexx: Added
			}

			ss_from_phipsi(sschar,phi,psi,total_residue);

			get_design_matrix(design_map,total_residue);

			for ( int i = 1; i <= total_residue; ++i ) {
				printresinfo(i) = false;
				name_from_num( res_orig(i), res_orig3(i) );
				name_from_num( res(i), residue3(i) );
				for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
					if ( is_protein(aa) || is_nonnatural(aa) ) {
						if ( ( aa != res(i) ) && ( design_map.get(i,aa) ) ) {
							printresinfo(i) = true;
						}
					}
				} // for
				if ( printresinfo(i) ) {
					sqc_x
					 << start_file.substr(0,4) << "  "
					 << res_chain(i) << " "
					 << I( 4, pdb_res_num(i) ) << ' '
					 << A( 3, res_orig3(i) ) << ' '
					 << A( 3, residue3(i) ) << ' '
					 << I( 5, neighbors(i)) << ' '
					 << A( 2, sschar(i) ) << ' '
					 << '\n';
				}
			} // for
			sqc_x.close();
			sqc_x.clear();
		}

	}    // loop through clusters
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_design_pdb_name
///
/// @brief
///
/// @detailed
///
/// @param  run - [in/out]? -
/// @param  fullname - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_design_pdb_name(
	int const run,
	std::string & fullname
)
{
	using namespace files_paths;

	int const num_length = { 4 };

//bk retrieve the root of the output name, if pdbout='des' then
//bk the output name was not specified on the command line
//bk and use the default name.

	fullname = pdb_out_path;
	if ( pdbout == "des" ) {
		fullname += output_file;
	} else {
		fullname += pdbout;
	}
	fullname += '_' + lead_zero_string_of( run, num_length ) + ".pdb";
  if (output_pdb_gz) fullname += ".gz";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_design_pdb_exists
///
/// @brief
///
/// @detailed
///
/// @param  run - [in/out]? -
/// @param  file_exists - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_design_pdb_exists(
	int run,
	bool & file_exists
)
{
	using namespace files_paths;

	std::string fullname;

	get_design_pdb_name(run,fullname);

  utility::io::izstream pdb_stream( fullname.c_str() );
  file_exists = pdb_stream.good() ||
	  ( files_paths::concat_outpdbs && cat_filename_exists( fullname ) ); // SJF
	pdb_stream.close();
	pdb_stream.clear();

	if ( ! file_exists ) {
		utility::io::izstream tmp_pdb_stream( (fullname+".in_progress").c_str() );
		file_exists = tmp_pdb_stream.good();
		tmp_pdb_stream.close();
		tmp_pdb_stream.clear();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_design_pdb_file
///
/// @brief
///
/// @detailed
///
/// @param  run - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_design_pdb_file( int run )
{
	using namespace files_paths;

	std::string fullname;

	get_design_pdb_name(run,fullname);

	utility::io::ozstream pdb_out_zstream;

	if ( overwrite_pdbs || touch_before_reading ) { // Overwrite existing file
		pdb_out_zstream.open( (fullname+".in_progress").c_str() );
		if ( !pdb_out_zstream ) {
			std::cout << "STOPPING: trouble creating output pdbfile" << std::endl;
			std::cout << "trying " << pdb_out_zstream.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else { // Must be new file
		utility::io::izstream tmp_pdb_out_izstream( fullname.c_str() );
		if ( tmp_pdb_out_izstream ) { // File exists
			std::cout << "STOPPING: output pdbfile already exists: " << std::endl;
			std::cout << tmp_pdb_out_izstream.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		utility::io::izstream pdb_out_izstream( (fullname+".in_progress").c_str() );
		if ( pdb_out_izstream ) { // File exists
			std::cout << "STOPPING: output pdbfile already exists: " << std::endl;
			std::cout << pdb_out_izstream.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		// cjk - touch a PDB file as a placeholder
		pdb_out_zstream.open( fullname.c_str() );
	}

	pdb_out_zstream.close();
	pdb_out_zstream.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin open_design_pdb_file
///
/// @brief
///
/// @detailed
///
/// @param  run - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
//void
//open_design_pdb_file( int run )
//{
//	using namespace files_paths;
//
//	std::string fullname;
//
//	get_design_pdb_name(run,fullname);
//	pdb_out_x.clear();
//	pdb_out_x.open( fullname.c_str() );
//	if ( !pdb_out_x ) {
//		std::cout << "trouble opening output pdbfile" << std::endl;
//		std::cout << "trying " << fullname << std::endl;
//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
//	}
//}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_design_profile
///
/// @brief outputs a fasta format file of all designed sequences
///       calculates a profile of all designed sequences
/// @detailed
///
/// @param[in] sequence   in - current sequence
/// @param[in] num        in - run number
///
/// @global_read
/// total_residue misc.h
/// start_res     start.h
/// sqc_x, pdb_out,output_file  files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/05/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_design_profile(
	FArray1Da_int sequence,
	int total_residue,
	int & num
)
{
	using namespace design_profile;
	using namespace files_paths;
	using namespace param;
	using namespace start;

	if ( !use_profile ) return;

	sequence.dimension( total_residue );

	static bool init = { false };

	if ( !init ) {
		use_profile = truefalseoption( "profile" );
		if ( !use_profile ) return;
		std::string filename;
		if ( pdbout == "des" ) {
			filename = output_file;
		} else {
			strip_path( pdbout, filename );
		}
		designfile = score_path + filename + "_design.fasta";
		std::cout << 'N' << designfile << 'N' << std::endl;
		sqc_x.clear();
		sqc_x.open( designfile.c_str() );
		if ( !sqc_x ) {
			std::cout << "STOPPING:: unable to open design fasta file:" << std::endl;
			std::cout << designfile << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		std::string tag( start_file );
		output_fasta(sqc_x,start_res,total_residue,tag);
		sqc_x.close();
		sqc_x.clear();
		init = true;
		for ( int i = 1; i <= total_residue; ++i ) {
			for ( int j = 1; j <= MAX_AA(); ++j ) {
				if ( is_protein(j) || is_nonnatural(j) ) {
					profile(i,j) = 0.0;
				}
			}
			profile(i,start_res(i)) = 1.0;
		}
		nseq = 1;
	}

	sqc_x.clear();
	sqc_x.open( designfile.c_str(), std::ios_base::out|std::ios_base::app );
	if ( !sqc_x ) {
		std::cout << "STOPPING:: unable to open design fasta file:" << std::endl;
		std::cout << designfile << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string tag;
	get_design_pdb_name(num,tag);
	strip_path( tag, tag );
	output_fasta(sqc_x,sequence,total_residue,tag);
	sqc_x.close();
	sqc_x.clear();
	for ( int i = 1; i <= total_residue; ++i ) {
		profile(i,sequence(i)) += 1.0;
	}
	++nseq;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_fasta
///
/// @brief writes sequence in a single line in one-letter code
///
/// @detailed
///
/// @param[in] iunit   in - file unit to write to
/// @param[in] sequence   in - current sequence
/// @param[in] length   in - length of sequence vector
/// @param[in] label   in - label for sequence
///
/// @global_read
/// aa_name1 param_aa.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/5/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_fasta(
	std::ostream & iunit,
	FArray1Da_int sequence,
	int & length,
	std::string & label
)
{
	using namespace param_aa;

	sequence.dimension( length );

//car parameters
	int const line_length = { 60 };

	iunit << "> " << label << std::endl;
	for ( int i = 0; i <= length; i += line_length ) {
		int last = std::min(line_length,length-i);
		for ( int j = 1; j <= last; ++j ) {
			iunit << aa_name1(sequence(i+j));
		}
		iunit << std::endl;
	}

}

//////////////////////////////////////////////////////////////////////////////////
/// @begin output_design_profile
///
/// @brief writes out a checkpoint style profile matrix
///
/// @detailed
///
/// @global_read
/// total_residue misc.h
/// sqc_x,
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/5/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_design_profile(int total_residue)
{
	using namespace design_profile;
	using namespace files_paths;
	using namespace param;

	if ( !use_profile ) return;

	std::string::size_type const indexExt = designfile.find( ".fasta" );
	if ( indexExt == std::string::npos ) {
		std::cout << "WARNING: improper design filename: " << designfile << std::endl;
		return;
	}

	std::string const fullname( designfile.substr( 0, indexExt + 1 ) + "checkpoint" );

	sqc_x.clear();
	sqc_x.open( fullname.c_str(), std::ios_base::out );
	if ( !sqc_x ) {
		std::cout << "WARNING:: unable to open profile file for writing:" << std::endl;
		std::cout << fullname << std::endl;
		sqc_x.clear();
		return;
	}
	sqc_x << total_residue << std::endl;
	for ( int i = 1; i <= total_residue; ++i ) {
		int j;
		sqc_x << misc::residue1(i);
		for ( j = 1; j <= MAX_AA(); ++j ) {
			if ( is_protein(j) || is_nonnatural(j) ) {
				sqc_x << F( 7, 4, profile(i,j)/nseq );
			}
		} sqc_x << std::endl;
		std::cout << misc::residue1(i);
		for ( j = 1; j <= MAX_AA(); ++j ) {
			if ( is_protein(j) || is_nonnatural(j) ) {
				std::cout << F( 7, 4, profile(i,j)/nseq );
			}
		} std::cout << std::endl;
	}
	sqc_x.close();
	sqc_x.clear();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_clusters
///
/// @brief
///
/// @detailed
///
/// @param  aan - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  nres - [in/out]? -
/// @param  xyz - [in/out]? -
/// @param  nclusters - [in/out]? -
/// @param  cluster_sizes - [in/out]? -
/// @param  cluster_members - [in/out]? -
/// @param  make_only_one_cluster - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_clusters(
	pose_ns::Pose & pose,
	int & nclusters,
	FArray1Da_int cluster_sizes,
	FArray2Da_int cluster_members,
	bool & make_only_one_cluster
)
{
	using namespace aaproperties_pack;
	using namespace param;

	cluster_sizes.dimension( MAX_RES() );
	cluster_members.dimension( MAX_RES(), MAX_RES() );

	if ( make_only_one_cluster ) {
		nclusters = 1;
		cluster_sizes(1) = pose.total_residue();
		for ( int seqpos1 = 1; seqpos1 <= pose.total_residue(); ++seqpos1 ) {
			cluster_members(seqpos1,1) = seqpos1;
		}
		return;
	}

	//aan.dimension( MAX_RES() );
	//aav.dimension( MAX_RES() );
	//xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	nclusters = 0;
	for ( int seqpos1 = 1; seqpos1 <= pose.total_residue(); ++seqpos1 ) {
		int const cluster = seqpos1;
		cluster_sizes(cluster) = 0;
		for ( int seqpos2 = 1; seqpos2 <= pose.total_residue(); ++seqpos2 ) {
			cluster_members(seqpos2,cluster) = 0;
		}
	}

	for ( int seqpos1 = 1; seqpos1 <= pose.total_residue(); ++seqpos1 ) {
		int const cluster = seqpos1;
		++nclusters;
		for ( int seqpos2 = 1, atm1e = nheavyatoms( pose.res(seqpos1), pose.res_variant(seqpos1) ); seqpos2 <= pose.total_residue(); ++seqpos2 ) {
			int nlow;
			if ( pose.res(seqpos1) == 6 ) {
				nlow = 1;
			} else {
				nlow = 5;
			}
			for ( int atm1 = nlow, atm2e = nheavyatoms( pose.res(seqpos2), pose.res_variant(seqpos2) ); atm1 <= atm1e; ++atm1 ) {
				if ( pose.res(seqpos2) == 6 ) {
					nlow = 1;
				} else {
					nlow = 5;
				}
				for ( int atm2 = nlow; atm2 <= atm2e; ++atm2 ) {
					float dis;
					distance_bk( pose.full_coord()(1,atm1,seqpos1), pose.full_coord()(1,atm2,seqpos2), dis );
					if ( dis < 5.0 ) {
						++cluster_sizes(cluster);
						cluster_members(cluster_sizes(cluster),cluster) = seqpos2;
						goto L10;
					}
				}  // atm2
			}  // atm1
L10:;
		}  // seqpos2
	}  // seqpos1

}

/////////////////////
// SJF designs a peptide where the N-terminus has bb dofs but the c-terminus is anchored
// based on design_interface
//
void
design_flex_peptide()
{

	using namespace design;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace misc;
	using namespace param;
	using namespace pdb;
  using numeric::xyzVector_float;

  calc_docking_rmsd();
  docking_store_initial_rms();
  set_docking_interface_pack( true ); // pack only the interface
  select_rotamer_set( "large" );
  set_fullatom_flag(true);
  pivot_repack(true); // repacks residues perturbed by pivot
  docking_repack(true); // repacks interface residues
  mc_global_track::mc_score::score = score12();
  monte_carlo_reset();

//      int atm1, atm2;
	bool rotamers_exist, make_output_file;
  int const nloop=2; // number of design - bb moves iterations
	FArray1D_char sschar( MAX_RES()() );           //yl
	FArray1D_int neighbors( MAX_RES()() );
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	full_atom = true;
 // set_pose_flag( true );

  int file_counter = main_job_distributor::jd->get_curr_outnum();

	//yl, Create PackerTask and setup values before pass into pack_rotamers

	ss_from_phipsi(sschar,phi,psi,total_residue);

//bk identify interface residues, and set interface_residue to true so that
//bk they will be redesigned
//bk Only the interface between the first two chains in a pdb is
//bk considered here!!!

//jk When the -fix_target_seq flag is used,
//jk the int passed in refers to the chain with fixed sequence
//jk note: this chain will be repacked

//jk When the -repack_interface flag is used, the both chains are repacked

	interface_residues();

//bk if true start design simulation from the sequence and coordinates
//bk observed in the pdb file
	rotamers_exist = include_inputchi; // include_inputchi from command line
	make_output_file = false; // SJF DEBUGGING I can't get pose_to_design_output to work properly I'm circumventing it
     setup_peptide_bb_dofs( pose );
    setup_peptide_foldtree( pose );

  docking::docking_local_refine = true; // make only small bb moves
  for( int i = 1; i <= nloop; ++i) {
    std::cout << "Flexible-peptide Design iteration " << i << std::endl;
    fullatom_nonideal_initialized_pose_from_misc( pose );

  	interface_residues();
	  PackerTask Task( pose );
   	Task.set_task("design",make_output_file,interface_residue,rotamers_exist);
	  //bk set variables that specify which residues to vary
	  Task.setup_residues_to_vary();

	  pack_rotamers( pose, Task);
	  pose.copy_to_misc();

	  make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

    std::cout << "Flexible-peptide BB move iteration " << i << std::endl;
    pose.set_allow_chi_move( interface_residue );
    pose_docking_peptide_std_protocol( pose );
  }

  pose.pose_to_design_output( file_counter );
}




////////////////////////////////////////////////////////////////////////////////
/// @begin design_interface
///
/// @brief
///bk identify residues at the interface of two chains, and redesign or
///bk repack them using pack_rotamers
///
/// @detailed
///
/// @param  packmode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
design_interface( std::string const & packmode )
{

	using namespace design;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace misc;
	using namespace param;
	using namespace pdb;

//      int atm1, atm2;
	FArray1D_int res_orig( MAX_RES()() );
	bool file_exists,rotamers_exist,make_output_file;
	FArray1D_string orig_res3( MAX_RES()() );
	FArray1D_int neighbors( MAX_RES()() );
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );
	FArray1D_string res_orig3( MAX_RES()() );   //yl added initialize for new output format
	FArray1D_char sschar( MAX_RES()() );           //yl

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );

	full_atom = true;

//bk this procedure creates its own output files (separate from the
//bk main rosetta functions) because multiple designs can be output for
//bk one starting structure.  If the design procedure has already begun for
//bk this starting structure, return from here so that rosetta can begin on
//bk the next structure.
	ss_from_phipsi(sschar,phi,psi,total_residue);
	check_design_pdb_exists(1,file_exists);
	if ( file_exists && !overwrite_pdbs && !touch_before_reading ) {
		std::cout << "this file has already been started - move to next" << std::endl;
		return;
	}
	if ( touch_before_starting ) {
		create_design_pdb_file(1);
	}

//bk save original sequence
	for ( int i = 1; i <= total_residue; ++i ) {
		res_orig(i) = res(i);
	}

//bk identify interface residues, and set interface_residue to true so that
//bk they will be redesigned
//bk Only the interface between the first two chains in a pdb is
//bk considered here!!!

//jk When the -fix_target_seq flag is used,
//jk the int passed in refers to the chain with fixed sequence
//jk note: this chain will be repacked

//jk When the -repack_interface flag is used, the both chains are repacked

	interface_residues();



//bk if true start design simulation from the sequence and coordinates
//bk observed in the pdb file
	rotamers_exist = include_inputchi; // include_inputchi from command line
	make_output_file = true; // output a pdb file


	Task.set_task(packmode,true,interface_residue,rotamers_exist);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task);
	pose.copy_to_misc();

	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

//bk   update sequence arrays
	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), residue3(i) );
		name_from_num( res_orig(i), orig_res3(i) );
		name_from_num( res_orig(i), res_orig3(i) ); // add for change the redesign output format
	}

//bk   output list comparing design sequence to native sequence
	utility::io::ozstream interface_stream( "redesign",
	 std::ios_base::out|std::ios_base::app );
	if ( !interface_stream ) {
		std::cout << "having trouble opening output file for design_interface " <<
		 "trying to open interface_residues" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( interface_residue(i) ) {
			interface_stream << start_file.substr(0,4) << "  "
											 << res_chain(i) << " "
											 << I( 4, pdb_res_num(i) ) << ' '
											 << A( 3, res_orig3(i) ) << ' '
											 << A( 3, residue3(i) ) << ' '
											 << I( 5, neighbors(i)) << ' '
											 << A( 2, sschar(i) ) << ' '
											 << '\n'; //std::endl;
		}
	}
	interface_stream.close();
	interface_stream.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_residues()
///
/// @brief
/// identify residues at the interface of two chains
///
/// @detailed
/// apl -- could be much faster; stop gap speedup -- square distance comps
///
/// @param
///
/// @global_read
///
/// @global_write   interface_residue (design.h), true if interface residue
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
interface_residues()
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

	if( truefalseoption( "set_interface_cutoff" ) )	{
		realafteroption("set_interface_cutoff", interface_residue_cutoff, interface_residue_cutoff);
		std::cout << "interface cutoff reset to:" <<  interface_residue_cutoff << std::endl;
	}
	float const interface_cutoff = interface_residue_cutoff;

//bk initialize
	interface_residue = false;
	n_interface_residues = 0;
	interface_residue_list = 0;

	float const interface_cutoff2 = interface_cutoff * interface_cutoff;

//bk set interface_residue to true for interface residues
//bk Only the interface between the first two partners is considered here!!!
	for ( int res1 = 1, res2e = domain_end(2); res1 <= domain_end(1); ++res1 ) {

		for ( int res2 = domain_end(1) + 1; res2 <= res2e; ++res2 ) {

			int start_atm1, end_atm1;
			if ( res(res1) == aa_gly ) {
				start_atm1 = 2;
				end_atm1 = 2;
			} else {
				start_atm1 = 5;
				end_atm1 = nheavyatoms( res(res1), res_variant(res1) );
			}
			int start_atm2, end_atm2;
			if ( res(res2) == aa_gly ) {
				start_atm2 = 2;
				end_atm2 = 2;
			} else {
				start_atm2 = 5;
				end_atm2 = nheavyatoms( res(res2), res_variant(res2) );
			}

			//if ( interface_residue( res1 ) && interface_residue( res2 ) ) continue;

			float dis2;
			bool pair_at_interface = false;
			for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
				for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
					distance2_bk( full_coord(1,atm1,res1), full_coord(1,atm2,res2), dis2 );
					if ( dis2 < interface_cutoff2 ) {

						if ( !interface_residue(res1) ) {
							++n_interface_residues;
							interface_residue_list(n_interface_residues) = res1;
						}
						if ( !interface_residue(res2) ) {
							++n_interface_residues;
							interface_residue_list(n_interface_residues) = res2;
						}
						interface_residue(res1) = true;
						interface_residue(res2) = true;
						pair_at_interface = true;
						break;
					}
				} // atm2
				if (pair_at_interface) break;
			} // atm1
		} // res2
	} // res1
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_pack_residues
///
/// @brief
/// identify residues around the design loops for repacking
///
/// @detailed
/// has two options:
/// "all" - get all residues arround loop regions
/// "interface" - get residues only from oppisite partner which are near loops
///               regions
/////////////////////////////////////////////////////////////////////////////////
void
get_loop_pack_residues(
	FArray1Da_bool contact_residue,
	std::string const & mode
)
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace loops_ns;

	contact_residue.dimension( total_residue );
	int	p1_start, p1_end, p2_start, p2_end;
	int res2_start = 0;
	int res2_end = 0;
	float const distance_cutoff = { 5.5 };
	float dist = 0.0;

	if( !(mode == "interface" || mode == "all") ){
		std::cout << "ERROR: Calling get_loop_pack_residues with improper command: ";
		std::cout << mode << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if(mode == "interface"){
		get_complex_boundaries(p1_start, p1_end, p2_start, p2_end);
	}

	//set the residues in the loop to be true first

	for (int i = 1; i <= num_loop; ++i){
		for (int j = loop_begin(i),je = loop_end(i); j <= je; ++j){
			contact_residue(j) = true;
		}
	}

	//set the residues around each loop to be true
	for (int i = 1; i <= num_loop; ++i){

		//set up range of inner loop
		//interface mode: pick opposite chain
		if(mode == "interface"){
			if(loop_begin(i) >= p1_start && loop_end(i) <= p1_end ){
				res2_start = p2_start;
				res2_end = p2_end;
			}else if(loop_begin(i) >= p2_start && loop_end(i) <= p2_end ){
				res2_start = p1_start;
				res2_end = p1_end;
			}else{
				std::cout << "ERROR: loop spans two proteins" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}else if(mode == "all"){
			res2_start = 1;
			res2_end = total_residue;
		}

		for (int res1 = loop_begin(i),res1_end = loop_end(i); res1 <= res1_end; ++res1){
			for (int res2 = res2_start; res2 <= res2_end; ++res2){
				if (!contact_residue(res2)){
					int start_atm1, end_atm1;
					if ( res(res1) == aa_gly ) {
						start_atm1 = 2;
						end_atm1 = 2;
					} else {
						start_atm1 = 5;
						end_atm1 = nheavyatoms( res(res1), res_variant(res1) );
					}
					int start_atm2, end_atm2;
					if ( res(res2) == aa_gly ) {
						start_atm2 = 2;
						end_atm2 = 2;
					} else {
						start_atm2 = 5;
						end_atm2 = nheavyatoms( res(res2), res_variant(res2) );
					}
					for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
						for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
							distance_bk( full_coord(1,atm1,res1), full_coord(1,atm2,res2), dist );
							if ( dist < distance_cutoff ) {
								contact_residue(res2) = true;
							}
						}
					}
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin altered_specificity()
///
/// @brief
/// This function is designed to alter the binding specificity between
/// proteins.
///
/// @detailed
/// This routine automates the second site suppressor strategy used
/// by Tanja Kortemme to  redesign specificity between IM7 and E7 (NSB 2004).
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks  mutlist outfile generated which can be used with
///        analyze_interface_ddg.  To rename mutlist use -alter_spec_mutlist
///        at the command line followed by file name
///
///        Energy files detailing the sc-sc and sc-bb energies for point
///        mutations introduced at the interface (point_mut_energies1, and
///        sc-sc and sc-bb energies for the "recovery" from these interface
///        point mutations (mut_mut_energies) can be generated. Type
///        -point_mut_energies filename and -mut_mut_energies filename at the
///        command line.
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
altered_specificity()
{
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace fullatom_flag;
	using namespace param_pack;
	using namespace analyze_interface_ddg_ns;
	using namespace files_paths;
	using namespace design;
	using namespace pdb;
	using namespace scale_res_energy_ns;

	// local variables for storing info about point mutants
	int const MAX_PMUT = { 2000 };
	int n_pmut = 0;
	FArray1D_int pmut_aa( MAX_PMUT, 0 );
	FArray1D_int pmut_seqpos( MAX_PMUT, 0 );
	FArray1D_float pmut_ddG( MAX_PMUT, 0.0 );
	FArray1D_int mut_pos(1);
	FArray1D_int mut_aa(1);

	// local variables for storing info about redesigns
	Dimension n_mut_structs; // total number of mutated structures
	FArray2D_int mutations_all_res( n_mut_structs, MAX_MUT_PER_STRUCT, 0 );
	FArray2D_int mutations_all_aa( n_mut_structs, MAX_MUT_PER_STRUCT, 0 );
	FArray1D_int n_mut_per_struct( n_mut_structs, 0 );
	FArray2D_char wt_res( n_mut_structs, MAX_MUT_PER_STRUCT, ' ' );
	FArray2D_char mut_res( n_mut_structs, MAX_MUT_PER_STRUCT, ' ' );
	FArray2D_char mut_file_res_chain( n_mut_structs, MAX_MUT_PER_STRUCT, ' ' );
	char aa_1letter;
	int point_mut_number = 1, struct_number, n_counter;

	// flags for pack_rotamers
	FArray1D_bool allow_repack( MAX_RES()(), false );  // dummy variable in this instance
	bool make_output_file = false;
	bool include_current = true;    //use side chains with native coord
	std::string packmode( "design" );

	// variables for saving protein information
	FArray3D_float full_coord_orig( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_orig( MAX_RES()() );
	FArray1D_float psi_orig( MAX_RES()() );
	FArray1D_int res_orig( MAX_RES()() );
	FArray1D_int res_variant_orig( MAX_RES()() );
	FArray1D_char residue1_orig( MAX_RES(), ' ' );
	FArray1D_string residue3_orig( MAX_RES(), std::string( 3, ' ' ) );
	int total_residue_orig;

	full_atom = true;
	select_rotamer_set( "large" ); // large rotamer set important for negative design

	//bk set which point mutations will be screened
	std::vector< int > scan_pmut;
	set_pmut_to_scan(scan_pmut);

	//bk identify point mutants that destabilize binding
	for ( std::vector< int >::size_type imut = 0;
				imut != scan_pmut.size(); ++imut ) {
		int seqpos = scan_pmut[imut];
		for ( int mutaa = 1; mutaa <= MAX_AUTH_AA; ++mutaa ) {
			if ( mutaa != res(seqpos) ) {
				int nmut = 1;
				mut_pos(1) = seqpos;
				mut_aa(1) = mutaa;
				float ddG_mut = get_ddG_bind_mut (nmut,mut_pos,mut_aa);
				if ( ddG_mut > 1.0 ) {
					++n_pmut;
					pmut_aa(n_pmut) = mutaa;
					pmut_seqpos(n_pmut) = seqpos;
					pmut_ddG(n_pmut) = ddG_mut;
				}
			}
		}  // loop over amino acids
	}  // sequence positions

	//bk save original protein
	copy_xyz_aan_aav_phi_psi(full_coord,full_coord_orig,phi,phi_orig,psi,
													 psi_orig,res,res_orig,res_variant,res_variant_orig,residue1,residue1_orig,
													 residue3,residue3_orig,Eposition,total_residue_orig,1,total_residue);

	//bk set scaling term so that energies across the interface are weighted stronger
	set_scale_res_energy_flag( true, true );
	for ( int seqpos1 = 1, seqpos1e = domain_end(1); seqpos1 <= seqpos1e; ++seqpos1 ) {
		for ( int seqpos2 = domain_end(1)+1; seqpos2 <= total_residue; ++seqpos2 ) {
			scale_energy_matrix( seqpos1, seqpos2 ) = true;
			scale_energy_matrix( seqpos2, seqpos1 ) = true;
		}
	}
	set_scale_res_energy_weight(2.);

	//bk give native residues a bonus so that less residues are mutated
	set_favor_native_residue( true );
	set_native_bonus(-0.6);

	//bk read in a list of residues that are not to be touched when redesigning
	FArray1D_bool fix_res( MAX_RES()(), false );
	get_fixlist( fix_res );

	// Set number of mutated structures: Allocates arrays
	n_mut_structs = n_pmut * 3;

	//bk redesign neighbor residues to accomodate each point mutant
	int i_mut_structs = 0;
	for ( int pmut = 1; pmut <= n_pmut; ++pmut ) {
		int seqpos1 = pmut_seqpos(pmut);
		int aa1 = pmut_aa(pmut);

		// set residues to be redesigned
		design_matrix = false;   //initialize
		FArray1D_bool contacts( MAX_RES()() );
		identify_contacts(seqpos1,full_coord,res,res_variant,total_residue,contacts);
		for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
			if ( seqpos2 == seqpos1 ) {
				design_matrix(aa1,seqpos2) = true;
			} else if ( contacts(seqpos2) && !fix_res(seqpos2) ) {
				for ( int aa2 = 1; aa2 <= MAX_AA(); ++aa2 ) {
					if ( is_protein(aa2) || is_nonnatural(aa2) ) {
						if ( aa2 != aa_cys ) {
							design_matrix(aa2,seqpos2) = true;
						}
					}
				}
			}
		}
		//yl create local pose and prepare the Epositions for misc
		pose_ns::Pose pose;
		fullatom_nonideal_initialized_pose_from_misc( pose );

		PackerTask Task( pose );
		Task.set_task(packmode,make_output_file,allow_repack,include_current);
		Task.setup_residues_to_vary();

		pack_rotamers(pose, Task );
		pose.copy_to_misc();

//ds fill arrays and matrices which will be used to generate mutation list
//ds file, mutlist.  This file can be used with analyze_interface_ddg to
//ds determine the binding energy for the wt-wt, mut-mut, mut-wt and wt-mut.

		for ( int wc = 1; wc <= 3; ++wc ) {
			int res_start; int res_end;
			if ( wc == 1 ) {
				res_start = 1; res_end = total_residue;       // mut-mut data
			} else if ( wc == 2 ) {
				res_start = 1; res_end = domain_end(1);       // mut-wt data
			} else {
				res_start = domain_end(1) + 1; res_end = domain_end(2);  //wt-mut data
			}

			++i_mut_structs;

			for ( int n = res_start; n <= res_end; ++n ) {
				if ( res_orig(n) != res(n) ) {
					++n_mut_per_struct(i_mut_structs);
					if ( n_mut_per_struct(i_mut_structs) > MAX_MUT_PER_STRUCT ) {
						std::cout << "n_mut_per_struct exceededmax_mut_per_struct" << std::endl;
						std::cout << "MAX_MUT_PER_STRUCT= " << MAX_MUT_PER_STRUCT << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}

					mutations_all_res(i_mut_structs,n_mut_per_struct(i_mut_structs)) = pdb_res_num(n);

					mutations_all_aa(i_mut_structs,n_mut_per_struct(i_mut_structs)) = res(n);
					res1_from_num(res_orig(n),aa_1letter);
					wt_res(i_mut_structs,n_mut_per_struct(i_mut_structs)) = aa_1letter;
					res1_from_num(res(n),aa_1letter);
					mut_res(i_mut_structs,n_mut_per_struct(i_mut_structs)) = aa_1letter;

					mut_file_res_chain(i_mut_structs,n_mut_per_struct(i_mut_structs)) = res_chain(n);
				}         // look for mutations by comparing sequences
			}            // loop through sequence
		}

		//bk return protein to original coordinates/sequence
		copy_xyz_aan_aav_phi_psi(full_coord_orig,full_coord,phi_orig,phi,psi_orig,
														 psi,res_orig,res,res_variant_orig,res_variant,residue1_orig,residue1,
														 residue3_orig,residue3,Eposition,total_residue,1,total_residue_orig);

	}

	//ds write the mutlist file to be used with analyze_interface_ddg
	std::ofstream asm_stream( ( score_path + alter_spec_mutfile ).c_str() );
	if ( !asm_stream ) {
		std::cout << "having trouble opening output file for mutlist" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	struct_number = 1;
	point_mut_number = 1;
	n_counter = 1;
	asm_stream << "START" << std::endl;
	asm_stream << I( 5, n_mut_structs ) << std::endl;
	for ( int n = 1; n <= n_mut_structs; ++n ) {
		if ( n == n_counter ) {
			asm_stream << I( 3, n_mut_per_struct(n) ) << space( 20 ) <<
			 I( 3, struct_number ) << space( 5 ) << I( 3, point_mut_number ) << std::endl;
			++point_mut_number;
			n_counter += 3;
		} else {
			asm_stream << I( 3, n_mut_per_struct(n) ) << space( 20 ) <<
			 I( 3, struct_number ) << std::endl;
		}
		++struct_number;

		for ( int d = 1, de = n_mut_per_struct(n); d <= de; ++d ) {
			asm_stream << space( 9 ) << I( 4, mutations_all_res(n,d) ) << "  "
			 << mut_file_res_chain(n,d) << "  " << wt_res(n,d) << "  "
			 << mut_res(n,d) << std::endl;
		}
	}
	asm_stream.close(); // mutlist to use with analyze_interface_ddg
	asm_stream.clear();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin mutated_clusters()
///
/// @brief
///
///
/// @detailed
/// This routine automatically mutate different clusters and generate a file
/// "etable_mut" which contains energies for all the mutations.If you want pdb
/// files for each mutations, put -output_pdb in command line.
/// Also you can use -favor_native_residue flag if you want less mutation.
///
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Xiaozhen Hu  07/16/2005
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
mutated_clusters()
{
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace fullatom_flag;
	using namespace param_pack;
	using namespace analyze_interface_ddg_ns;
	using namespace files_paths;
	using namespace design;
	using namespace favor_residue;
	using namespace pdb;
	using namespace scale_res_energy_ns;
	using namespace scorefxns;

	typedef std::vector< cluster_mutE* > Clusters;
	typedef std::vector< cluster_mutE* >::iterator Itertype;

	utility::io::ozstream etable_stream( pdb_out_path + "etable_mut");
 	FArray1D_bool contacts( MAX_RES()() );
	Clusters mut_clusters;
	Itertype iter;
	std::vector<float> energyItems( 9 );

	//local variables
	float score,wtE,mut_E;
	int mutnum,cluster_num;
	char newaa;
	float solvationE,attractiveE,repulsiveE,referenceE,protonationE,dunbrackE,Paa_ppE;
	float sc_bb_hbondE,pairprobE,planeprobE,h2oE,h2ohbE,h2osolE,intra_resE;
	float srbb_hbE,lrbb_hbE, gb_elecE, cstE, bondangleE;
	float wt_solvationE,wt_attractiveE,wt_repulsiveE,wt_referenceE,wt_protonationE,wt_dunbrackE,wt_Paa_ppE;
	float wt_sc_bb_hbondE,wt_pairprobE,wt_planeprobE,wt_h2oE,wt_h2ohbE,wt_h2osolE,wt_intra_resE;
	float wt_srbb_hbE,wt_lrbb_hbE, wt_gb_elecE, wt_cstE, wt_bondangleE;
  float elecE;

	// flags for pack_rotamers
	FArray1D_bool allow_repack( MAX_RES()(), true );  // dummy variable in this instance
	bool make_output_file = false;
	bool include_current = true;    //use side chains with native coord
	std::string packmode( "design" );

	// variables for saving protein information
	FArray3D_float full_coord_orig( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_orig( MAX_RES()() );
	FArray1D_float psi_orig( MAX_RES()() );
	FArray1D_int res_orig( MAX_RES()() );
	FArray1D_int res_variant_orig( MAX_RES()() );
	FArray1D_char residue1_orig( MAX_RES(), ' ' );
	FArray1D_string residue3_orig( MAX_RES(), std::string( 3, ' ' ) );
	int total_residue_orig;

	//variables for saving repacked wt protein
	FArray3D_float full_coord_relax( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_relax( MAX_RES()() );
	FArray1D_float psi_relax( MAX_RES()() );
	FArray1D_int res_relax( MAX_RES()() );
	FArray1D_int res_variant_relax( MAX_RES()() );
	int total_residue_relax;

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	full_atom = true;
	use_design_matrix = true;
	select_rotamer_set( "large" ); // large rotamer set important for negative design

	cluster_num = 0;

 	//bk save original protein
 	copy_xyz_aan_aav_phi_psi(full_coord,full_coord_orig,phi,phi_orig,psi,
													 psi_orig,res,res_orig,res_variant,res_variant_orig,
													 residue1,residue1_orig,residue3,residue3_orig,
													 Eposition,total_residue_orig,1,total_residue);

	//determine wild type energy, allow all residues to repack

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );
	Task.set_task("packrot", make_output_file, allow_repack, include_current);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();
	pack_rotamers( pose, Task );
	pose.copy_to_misc();

	//save this repacked structure, used as starting structure for all mutations
	copy_xyz_aan_aav_phi_psi(full_coord,full_coord_relax,phi,phi_relax,psi,
													 psi_relax,res,res_relax,res_variant,res_variant_relax,
													 residue1,residue1_orig,residue3,residue3_orig,
													 Eposition,total_residue_relax,1,total_residue);

	score_set_new_pose();
	score = score12();
	wtE = get_fullatom_totalE();

 	etable_stream<<"wild type :"<<'\n';

	output_residueE_partI(etable_stream,total_residue,res);

	etable_stream<<'\n';

sum_fullatom_energies(total_residue,wt_attractiveE,wt_repulsiveE,wt_solvationE,
  wt_referenceE,wt_protonationE,wt_dunbrackE,wt_Paa_ppE,wt_pairprobE,wt_planeprobE,wt_h2oE,wt_h2ohbE,wt_h2osolE,
  wt_srbb_hbE,wt_lrbb_hbE,wt_sc_bb_hbondE,wt_intra_resE,wt_gb_elecE,elecE,
  wt_cstE,wt_bondangleE);

	if ( output_pdb ) {
		make_named_pdb( output_file + "_wt.pdb", true );
	}

	// give native residues a bonus so that less residues are mutated
	if (favor_native_residue) {
		realafteroption("favor_native_residue",-1.5,native_bonus);
		std::cout << "favor_native_residue set to true and native_bonus set to " << native_bonus
		          << std::endl;
	}

 //  keep CYS fixed in redesign
 	FArray1D_bool fix_res( MAX_RES()(), false );

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( res( seqpos ) == aa_cys )
			fix_res( seqpos ) = true;
	}

 	//bk redesign neighbor residues to accomodate each point mutant
	//loop through all aa positions in the whole protein

	for ( int seqpos1 = 1; seqpos1 <= total_residue; ++seqpos1 ) {

		int aa1 = res(seqpos1);
		if ( aa1 != aa_cys ) {

 		// set residues to be redesigned
		design_matrix = false;   //initialize
		contacts = false;

		for (int i = 1;i <= MAX_AUTH_AA; ++i)
			design_matrix(i,seqpos1) = true;

		for (int i = 1;i <= total_residue; ++i)
			design_matrix(res_orig(i),i) = true;

		//for each positions, find its contact neighbors
		//redesign this position and all its neighbors and allow other places repacking
 		identify_contacts(seqpos1,full_coord,res,res_variant,total_residue,contacts);
 		for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
 			if (seqpos2 == seqpos1) {
 				design_matrix(aa1,seqpos2) = true;
 			} else if ( contacts(seqpos2) && !fix_res(seqpos2) ) {
 				for ( int aa2 = 1; aa2 <= MAX_AUTH_AA; ++aa2 ) {
 					if ( aa2 != aa_cys ) {
 						design_matrix(aa2,seqpos2) = true;
 					}//end if
 				}//end for
 			}//end of elseif
 		}//end for

		fullatom_nonideal_initialized_pose_from_misc( pose );
		PackerTask Task2( pose );
		Task2.set_task(packmode, make_output_file, allow_repack, include_current);
		//bk set variables that specify which residues to vary
		Task2.setup_residues_to_vary();
		pack_rotamers( pose, Task2);
		pose.copy_to_misc();

		//set the score for the whole protein
		score_set_new_pose();
		score = score12();
		mut_E = get_fullatom_totalE();

sum_fullatom_energies(total_residue,attractiveE,repulsiveE,solvationE,
                      referenceE,protonationE,dunbrackE,Paa_ppE,pairprobE,planeprobE,h2oE,h2ohbE,h2osolE,
                      srbb_hbE,lrbb_hbE,sc_bb_hbondE,intra_resE,gb_elecE,elecE,cstE,bondangleE);

		int index = 0;
		energyItems[index] = attractiveE - wt_attractiveE ;
		energyItems[++index] = repulsiveE - wt_repulsiveE ;
		energyItems[++index] = solvationE - wt_solvationE ;
		energyItems[++index] = Paa_ppE - wt_Paa_ppE ;
		energyItems[++index] = dunbrackE - wt_dunbrackE ;
		energyItems[++index] = sc_bb_hbondE - wt_sc_bb_hbondE ;
		energyItems[++index] = pairprobE - wt_pairprobE ;
		energyItems[++index] = referenceE - wt_referenceE ;
		energyItems[++index] = mut_E - wtE ;

		mutnum = 0;

		etable_stream<<"*****************************************************************"
								 <<"*****************************************************************";
		etable_stream<<"\n            Cluster center : "
								 <<pdb_res_num(seqpos1)
								 <<", contact residues : ";

		for (int i = 1;i<= total_residue;++i){
			if (contacts(i))
				etable_stream<<pdb_res_num(i)<<' ';
		}

		etable_stream<<'\n';

		std::string mut_type;

		for ( int i = 1;i <= total_residue; ++i ) {
			if ( res(i) != res_orig(i) ) {
				++mutnum;
				res1_from_num(res(i),newaa);
				etable_stream<<"            Chain: "
										 <<res_chain(i)<<"   "
										 <<residue1_orig(i)
										 <<pdb_res_num(i)
										 <<newaa<<'\n';
			  mut_type +=( res_chain(i) == ' ' ? '-' : res_chain(i) )
					+ std::string( ":" )
					+ residue1_orig(i)
					+ string_of( pdb_res_num(i) )
					+ newaa
					+ std::string(" ");
			}
		}

		//calculate energy for each mutation and generate a file which contains all the
		//energy terms.Sorted by total energy.

		if ( mutnum != 0 ) {
			++cluster_num;
			etable_stream << "\n            " << mutnum << " mutations for this cluster" << '\n';
			etable_stream << "Mut#: " << cluster_num << '\n';
			output_residueE_partI(etable_stream,total_residue,res);
			etable_stream<<'\n';
		} else {
			etable_stream<<"\n            No mutations suggested for this cluster.\n";
		}

		//save this mutID and mutations and energies for later use to sort
		if ( mutnum != 0 ) {
			mut_clusters.push_back( new cluster_mutE(mutnum,mut_type,energyItems) );
		}

		//output structures if the output_pdb is turned on
		if ( output_pdb ) {
			make_named_pdb( output_file + "_mut_" + lead_zero_string_of(cluster_num,4) + ".pdb", true );
		}

 	//bk return protein to saved relaxed coordinates/sequence
 		copy_xyz_aan_aav_phi_psi(full_coord_relax,full_coord,phi_relax,phi,psi_relax,
														 psi,res_relax,res,res_variant_relax,res_variant,
														 residue1_orig,residue1,residue3_orig,residue3,
														 Eposition,total_residue,1,total_residue_relax);

		}
	}

	//sort the energies and output
	etable_stream << "\n*****************************************************************"
							 <<"*****************************************************************\n";
	etable_stream << " Wild type energy :" << F(8,2,wtE) << '\n';
	etable_stream << " Energy table sorted by mut energy, number of mutations , mutID\n\n";
	etable_stream << " MutID dEatr   dErep   dEsol   dEaa    dEdun   dEhbnd  dEpair  dEref    ddg   #point mutations(Chain:wt res_num mut)\n";
	etable_stream << "--------------------------------------------------------------------------------------------------------------------\n";

	std::sort( mut_clusters.begin(), mut_clusters.end(), display_order );

	for ( iter = mut_clusters.begin(); iter != mut_clusters.end(); ++iter ) {
		etable_stream << *(*iter) << '\n';
	}

	etable_stream << "--------------------------------------------------------------------------------------------------------------------\n";

	etable_stream.close();
	etable_stream.clear();

	//put original protein back,maybe unnecessary
 	copy_xyz_aan_aav_phi_psi(full_coord_orig,full_coord,phi_orig,phi,psi_orig,
													 psi,res_orig,res,res_variant_orig,res_variant,
													 residue1_orig,residue1,residue3_orig,residue3,
													 Eposition,total_residue,1,total_residue_orig);
	//clear memory
	int nclusters = mut_clusters.size();
	for ( int i = 0; i < nclusters; ++i ) delete mut_clusters[i];
	mut_clusters.clear();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin fill_mutation_arrays(mutations_all_res,mutations_all_aa,
///                                      n_mut_structs,n_mut_per_struct,n)
///
/// @brief
///
/// @detailed
///
/// @param[in]   fma_mut_all_res - in
///         fma_mut_all_aa - in
///         fma_n_mut_structs - in
///         fma_n_mut_per_struct - in
///         fma_pdb_res_chain_to_rosetta - in
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

//      fill_mutation_arrays(fma_mut_all_res,fma_mut_all_aa,
//                           fma_n_mut_structs,fma_n_mut_per_struct,
//                           fma_pdb_res_chain_to_rosetta)
//
//      include 'param.h'
//      include 'analyze_interface_ddg.h'
//
//
//      int fma_mut_all_res(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//                            // mutated positions for all structures
//      int fma_mut_all_aa(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//                            // mutated amino acids for all structures
//      int fma_n_mut_structs     // total number of mutated structures
//      int fma_n_mut_per_struct(MAX_MUT_STRUCTS);
//                            // number of mutations per structure
//      int fma_pdb_res_chain_to_rosetta(MAX_RES(),MAX_CHAINNUMS);
//
//      int alsp_mut_all_res(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//                            // mutated positions for all structures
//      int alsp_mut_all_aa(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//                            // mutated amino acids for all structures
//      int alsp_n_mut_structs;     // total number of mutated structures
//      int alsp_n_mut_per_struct(MAX_MUT_STRUCTS);
//                            // number of mutations per structure
//      int n(MAX_RES(),MAX_CHAINNUMS),d,w,s,m;
//
//      common / mutation_arrays_for_ddGbind / alsp_mut_all_res,
//     *  alsp_mut_all_aa,alsp_n_mut_structs,alsp_n_mut_per_struct,n
//
//      alsp_n_mut_structs = fma_n_mut_structs;
//
//      for ( int d = 1; d <= MAX_MUT_STRUCTS; ++d ) {
//         alsp_n_mut_per_struct(d) = fma_n_mut_per_struct(d);
//         for ( int s = 1; s <= MAX_MUT_PER_STRUCT; ++s ) {
//            alsp_mut_all_res(d,s) = fma_mut_all_res(d,s);
//            alsp_mut_all_aa(d,s) = fma_mut_all_aa(d,s);
//         }
//      }
//
//      for ( int w = 1; w <= MAX_RES(); ++w ) {
//         for ( int m = 1; m <= MAX_CHAINNUMS; ++m ) {
//            n(w,m) = fma_pdb_res_chain_to_rosetta(w,m);
//         }
//      }
//
//
//      return
//      end

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_mutation_arrays(mutations_all_res,mutations_all_aa,
///                                      n_mut_structs,n_mut_per_struct,n)
///
/// @brief
///
/// @detailed
///
/// @param[out]   rma_mut_all_res - out
///         rma_mut_all_aa   out
///         rma_n_mut_structs   out
///         rma_n_mut_per_struct   out
///         rma_pdb_res_chain_to_rosetta   out
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

//$$$      retrieve_mutation_arrays(rma_mut_all_res,rma_mut_all_aa,
//$$$                               rma_n_mut_structs,rma_n_mut_per_struct,
//$$$                               rma_pdb_res_chain_to_rosetta)
//$$$
//$$$      include 'param.h'
//$$$      include 'analyze_interface_ddg.h'
//$$$
//$$$
//$$$      int rma_mut_all_res(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//$$$                            // mutated positions for all structures
//$$$      int rma_mut_all_aa(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//$$$                            // mutated amino acids for all structures
//$$$      int rma_n_mut_structs;     // total number of mutated structures
//$$$      int rma_n_mut_per_struct(MAX_MUT_STRUCTS);
//$$$                            // number of mutations per structure
//$$$      int rma_pdb_res_chain_to_rosetta(MAX_RES(),MAX_CHAINNUMS);
//$$$
//$$$      int alsp_mut_all_res(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//$$$                            // mutated positions for all structures
//$$$      int alsp_mut_all_aa(MAX_MUT_STRUCTS,MAX_MUT_PER_STRUCT);
//$$$                            // mutated amino acids for all structures
//$$$      int alsp_n_mut_structs;     // total number of mutated structures
//$$$      int alsp_n_mut_per_struct(MAX_MUT_STRUCTS);
//$$$                            // number of mutations per structure
//$$$      int n(MAX_RES(),MAX_CHAINNUMS),d,s,w,m;
//$$$
//$$$      common / mutation_arrays_for_ddGbind / alsp_mut_all_res,
//$$$     *  alsp_mut_all_aa,alsp_n_mut_structs,alsp_n_mut_per_struct,n
//$$$
//$$$
//$$$      rma_n_mut_structs = alsp_n_mut_structs;
//$$$
//$$$      for ( int d = 1; d <= MAX_MUT_STRUCTS; ++d ) {
//$$$         rma_n_mut_per_struct(d) = alsp_n_mut_per_struct(d);
//$$$         for ( int s = 1; s <= MAX_MUT_PER_STRUCT; ++s ) {
//$$$            rma_mut_all_res(d,s) = alsp_mut_all_res(d,s);
//$$$            rma_mut_all_aa(d,s) = alsp_mut_all_aa(d,s);
//$$$         }
//$$$      }
//$$$
//$$$      for ( int w = 1; w <= MAX_RES(); ++w ) {
//$$$         for ( int m = 1; m <= MAX_CHAINNUMS; ++m ) {
//$$$            rma_pdb_res_chain_to_rosetta(w,m) = n(w,m);
//$$$         }
//$$$      }
//$$$
//$$$      return
//$$$      end

////////////////////////////////////////////////////////////////////////////////
/// @begin get_alter_spec_flag
///
/// @brief returns design and alter_spec flags
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///     alter_spec
///
/// @remarks
///
/// @references
///
/// @authors dws 10/8/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_alter_spec_flag()
{
	return design::alter_spec;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin design_from_mut_list()
///
/// @brief reads in a list of mutations and for each set of mutations
/// @brief redesigns the neighboring residues to accomodate the mutation
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
design_from_mut_list()
{
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace design;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pdb;

//bk local variables
	FArray1D_int res_orig( MAX_RES()() );
	FArray3D_float full_coord_orig( 3, MAX_ATOM()(), MAX_RES()() );
	FArray2D_int mutation_aa( max_mut_per_sim, max_sim );
	FArray2D_int mutation_seqpos( max_mut_per_sim, max_sim );
	FArray1D_int nmut( max_sim );
	int nsim;
	FArray1D_bool all_contacts( MAX_RES()() );
	FArray1D_bool contacts( MAX_RES()() );
	bool include_current,make_output_file;
	int mutaa,mut_pos;
	float startE,mutE,redesignE,temp;
	std::string wtaa3, mutaa3;
//	FArray2D_int pdb_res_chain( MAX_RES()(), max_num_chains );
	int const max_atom = MAX_ATOM();

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;

	full_atom = true;
	score_set_try_rotamers(false);
	 // don't use rotamer trials when the scorefxn is called

//bk initialize arrays
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		res_orig(seqpos)=res(seqpos);
		for ( int d = 1; d <= 3; ++d ) {
			for ( int atm = 1; atm <= max_atom; ++atm ) {
				full_coord_orig(d,atm,seqpos) = full_coord(d,atm,seqpos);
			}
		}
	}

//ds call function to get list of mutations (incl. # of mutated structures)

	get_redesign_mutation_list(mutation_seqpos,mutation_aa,nsim, nmut);

	user_x.clear();
	user_x.open( "mutation_energies" );

//bk loop through simulations, nsim
	for ( int sim = 1; sim <= nsim; ++sim ) {

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			all_contacts(seqpos) = false;
		}

//bk identify residues that contact the residues to be mutated
		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			identify_contacts(mutation_seqpos(mut,sim),full_coord,res,res_variant,
			 total_residue,contacts);

			std::cout << "mut_pos" << SS( mutation_seqpos(mut,sim) ) << std::endl;
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( contacts(seqpos) && ( res(seqpos) != aa_cys ) ) {
					all_contacts(seqpos) = true;
				}
			}
		}

//bk it is important for the section below that all_contacts is not
//bk true for the mutation positions
		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			all_contacts(mutation_seqpos(mut,sim)) = false;
		}

//bk  determine wild type energy, let all residues to be redesigned repack
		include_current = true;
		make_output_file = false;

		fullatom_nonideal_initialized_pose_from_misc( pose );

		PackerTask Task( pose );
		Task.set_task("packrot", false, all_contacts, include_current);
		Task.setup_residues_to_vary();

		pack_rotamers(pose,Task);
		pose.copy_to_misc();

//bk set fragment size to be whole protein so that score function counts
//bk everything
		score_set_new_pose();
		temp = score12();
		startE = get_fullatom_totalE();

//bk  determine energy with just the mutations
		for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
			if ( is_protein(aa) || is_nonnatural(aa) ) {
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					design_matrix(aa,seqpos) = false;
				}
			}
		}

		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			mutaa = mutation_aa(mut,sim);
			mut_pos = mutation_seqpos(mut,sim);
			design_matrix(mutaa,mut_pos) = true;
		}

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( all_contacts(seqpos) ) {
				design_matrix(res(seqpos),seqpos) = true;
			}
		}

		fullatom_nonideal_initialized_pose_from_misc( pose );

		PackerTask Task2( pose );
		Task2.set_task("design", make_output_file, all_contacts, include_current);
		Task2.setup_residues_to_vary();

		pack_rotamers(pose,Task2);
		pose.copy_to_misc();

//bk set fragment size to be whole protein so that score function counts
//bk everything
		score_set_new_pose();
		temp = score12();
		mutE = get_fullatom_totalE();


//bk  determine energy with mutations plus compensating mutations
		for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
			if ( is_protein(aa) || is_nonnatural(aa) ) {
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					design_matrix(aa,seqpos) = false;
				}
			}
		}

		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			mutaa = mutation_aa(mut,sim);
			mut_pos = mutation_seqpos(mut,sim);
			design_matrix(mutaa,mut_pos) = true;
		}

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( all_contacts(seqpos) ) {
				for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
					if ( is_protein(aa) || is_nonnatural(aa) ) {
						if ( aa != aa_cys ) {
							design_matrix(aa,seqpos) = true;
						}
					}
				}
			}
		}

		fullatom_nonideal_initialized_pose_from_misc( pose );

		PackerTask Task3( pose );
		Task3.set_task("design", make_output_file, interface_residue, include_current);
		Task3.setup_residues_to_vary();

		pack_rotamers(pose,Task3);
		pose.copy_to_misc();

//bk set fragment size to be whole protein so that score function counts
//bk everything
		score_set_new_pose();
		temp = score12();
		redesignE = get_fullatom_totalE();

//bk write out results
		user_x << "  " << std::endl;
		user_x << "------------------------------------------" << std::endl;
		user_x << "Mutation set " << SS( sim ) << std::endl;
		user_x << "The forced mutations are:" << std::endl;
		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			int const seqpos = mutation_seqpos(mut,sim);
			name_from_num(res_orig(seqpos),wtaa3);
			name_from_num(mutation_aa(mut,sim),mutaa3);
			user_x << ' ' << wtaa3 << I( 4, pdb_res_num(seqpos) ) << ' ' <<
			 A( 2, res_chain(seqpos) ) << ' ' << mutaa3 << std::endl;
		}

		user_x << ' ' << std::endl;
		user_x << "The redesigned sequence contains the following mutations:" << std::endl;

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( res(seqpos) != res_orig(seqpos) ) {
				name_from_num(res_orig(seqpos),wtaa3);
				name_from_num(res(seqpos),mutaa3);
				user_x << ' ' << wtaa3 << I( 4, pdb_res_num(seqpos) ) << ' ' <<
				 A( 2, res_chain(seqpos) ) << ' ' << mutaa3 << std::endl;
			}
		}

		user_x << "  " << std::endl;
		user_x << "WT score,   Mut score,  Redesign score" << std::endl;
		user_x << SS( startE ) << SS( mutE ) << SS( redesignE ) << std::endl;

//bk reset sequence coordinate array
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			res(seqpos) = res_orig(seqpos);
			for ( int d = 1; d <= 3; ++d ) {
				for ( int atm = 1; atm <= max_atom; ++atm ) {
					full_coord(d,atm,seqpos) = full_coord_orig(d,atm,seqpos);
				}
			}
		}

	} // different mutant sets

	user_x.close();
	user_x.clear();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin   design_from_cluster()
///     code for help you to do the rational design
///
/// @detailed
/// 1. identify the minimum mutations which can stable the protein
/// 2. provide mutiple sequence solutions for a design run
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Lin Jiang
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
design_from_cluster()
{
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace design;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pdb;
	using namespace protein_maps;
	using namespace int_fullatom_energies;
	using namespace analyze_interface_ddg_ns;

	//ds because this function uses int_fullatom_energies arrays,
	//it must first ensure these arrays are dimensioned.
	initialize_MAX_MUT_STRUCTS();

// Based on the resfile, the repack and design regions are defined.
// For design region, do all possible designs with the allowed
// mutation number, and then output the energy difference between
// the designed structure and the repacked strarting structure.
//
//local variables
	FArray1D_int res_orig( MAX_RES()() );
	FArray1D_int res_variant_orig( MAX_RES()() );
	FArray3D_float full_coord_orig( 3, MAX_ATOM()(), MAX_RES()() );
	FArray2D_int mutation_aa( max_mut_per_sim, max_sim );
	FArray2D_int mutation_seqpos( max_mut_per_sim, max_sim );
	FArray1D_int nmut( max_sim );
	int n_cluster;
	FArray1D_bool all_contacts( MAX_RES()() );
	FArray1D_bool repack_contacts( MAX_RES()() ),design_contacts( MAX_RES()() );
	bool include_current,make_output_file;
	float temp;
  FArray1D_double pow_tmp(MAX_RES()() );
	char wtaa, mutaa;
	std::vector< int > permute_array;
	FArray1D_int pos_index_old( MAX_RES()(), 0 ),pos_index( MAX_RES()(), 0 );
	//---	FArray1D_int design_array( MAX_RES()() ), design_map( MAX_RES()() );
	FArray1D_int design_array( MAX_RES()() ), design_position( MAX_RES()() );
	int design_num=0,cluster_num=0;
	std::string mutname,mutname_in,filename;
	bool mutfile_exist;
	std::ofstream energy_out;
	std::string fullname, outname;

	DesignMap design_map( MAX_RES(), misc::res );

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	PackerTask Task( pose );

	FArray1D_int neighbors_copy( MAX_RES()() );
	FArray2D_bool neighborlist_copy( MAX_RES()(), MAX_RES()() );

	bool is_good_cluster;
//	FArray2D_int pdb_res_chain( MAX_RES()(), max_num_chains );

	bool min_bb,min_sc;
	int const lj_ramp_cycles = { 2 };
	float const min_tol = { 0.001 };

	// make checkpoint file
	std::string cp_file;
	bool checkit = false, skipit = false ;
	int exist=0;
	FArray1D_int cpts( MAX_RES()() ), last_cpts( MAX_RES()() ), last_store_cpts( MAX_RES()() );
	FILE * ofile;

	//output structure
	FArray1D_int cluster_pos( MAX_RES()() );
	FArray1D_char cluster_wtaa( MAX_RES()() );
	FArray1D_char cluster_mutaa( MAX_RES()() );


//lin pass the global viriable
	min_sc = true;
	min_bb = minimize_bb;

	use_design_matrix = true;
	full_atom = true;
	score_set_try_rotamers(false);	 // don't use rotamer trials when the scorefxn is called
	include_current = true;
	make_output_file = false;

//  define the complex
	int startres, endres;
	int p1_start, p1_end, p2_start, p2_end;
	if( n_partner == 3 ) {//monomeric protein
		p1_start   = 1;
		p1_end     = total_residue+1;
		p2_start   = p1_end+1;
		p2_end     = total_residue;

	}else{//protein complex
		get_complex_boundaries(p1_start, p1_end, p2_start, p2_end);
	}

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		allow_repack( seqpos )=true;
	}

//read the resfile
	if ( resfile != "none" ) {
		Task.set_allow_repack(allow_repack);
		Task.set_designmap(design_map);
		Task.load_resfile();
		design_map = Task.get_designmap();

	} else {
		std::cout << "Warning: no resfile, design all positon .... \n" << std::endl;
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( is_protein(res(seqpos)) || is_nonnatural(res(seqpos)) ) {
				for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
					design_map.set(seqpos,aa);
				}
			} else {
				design_map.fix_completely(seqpos);
			}
		}
	}

	if ( mutenergy_out == "mut" ) {
		filename = "mutation_energies";
	} else {
		filename = mutenergy_out;
	}
	mutname = pdb_out_path+filename;
	cp_file = pdb_out_path+filename+".last_point";

	mutfile_exist=false;
	std::string line,tag;
	std::istringstream line_stream;

	std::ifstream data_x( mutname.c_str() );
	if ( data_x ) {
		mutfile_exist=true;
	}
	data_x.close();
	data_x.clear();

	std::ifstream data_y;
	if ( fix_mut ) {
		if ( mutenergy_in == "mut" ) {
			filename = "mutation_energies";
		} else {
			filename = mutenergy_in;
		}
		mutname_in = start_path + filename;
		data_y.open( mutname_in.c_str() );
		if ( !data_y ) {
			std::cout<<"WARNING: no mutfile when use FIX_MUT flag, stop "<<std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	filename = output_file;

	if ( minimize_wt_struct ) {
//  minimize WT structure
		std::cout << "Minimizing the input pdb structures ... " << std::endl;
//define the allow_change region
		reset_move_map();//for minimization
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( design_map.repack_residue(seqpos) ) {
				all_contacts(seqpos) = true;
				allow_insert(seqpos) = true;
				allow_repack(seqpos) = true;
				maps_set_new_rotamer(seqpos);
			} else {
				all_contacts(seqpos) = false;
				allow_insert(seqpos) = false;
				allow_repack(seqpos) = false;
			}
		}
		score_set_new_pose();
		temp = score12();
		monte_carlo_reset();

		float const lj_increment = ( 1.0 - 0.02 ) / 8;
		for ( int k = 0; k <= lj_ramp_cycles; ++k ) {
			score_set_lj_rep_weight(0.02+k*lj_increment);
			minimize_allow_insert_trial("dfpmin",false,true,0.0001);
			monte_carlo_reset();
		}
	}

	//   save original structure and sequence in *orig arrays (analyze_interface_ddg.h)
	copy_xyz_aan_aav_phi_psi(full_coord,xyz_orig,phi,phi_orig,psi,
	 psi_orig,res,aan_orig,res_variant,aav_orig,residue1,residue1_orig,residue3,residue3_orig,
	 Eposition,nres_orig,1,total_residue);

	for ( int which_partner = n_partner; which_partner <= 3; ++which_partner ) {

		set_partner_specific_params(which_partner,startres,endres,p1_start,p1_end,
		 p2_start,p2_end,nres_orig);


		copy_xyz_aan_aav_phi_psi(xyz_orig,full_coord,phi_orig,phi,psi_orig,
			 psi,aan_orig,res,aav_orig,res_variant,residue1_orig,residue1,
			 residue3_orig,residue3,Eposition,total_residue,startres,endres);

		// set fragment size to be whole protein so that score function counts
		// everything
		score_set_new_pose();
		temp = score12();
		calc_sasa_pack_score(total_residue,res,res_variant,full_coord);
		//code from glb calculate hbonds & sasa
		//setup
		set_ds_outpdbonly_flag(true);
		reset_decoy_scores();
		decoystats_reset_output_lines();

		copy_hbenergies();
		std::list < unsatisfied_buried_polar > uns_list;
		std::list < unsatisfied_buried_group > group_uns_list;
		find_unsatisfied_hbonds(uns_list, group_uns_list);
		report_unsatisfied_hbonds("",uns_list,group_uns_list);

		if ( minimize_wt_struct && which_partner == 3 ) {
			outname=pdb_out_path+filename+"_minwt.pdb";
			dump_fullatom_pdb(outname);
			return;
		}

		save_scores(which_partner,total_residue);
		save_wt_scores(which_partner);
	}
	//store the native score
	compile_binding_energy(1); // counter for list

	if ( !mutfile_exist || !check_point ) { // file is new so write out headers
		energy_out.open( mutname.c_str() );

		energy_out << "# " << std::endl;
		energy_out << "********************************************************" << std::endl;
		energy_out << "ENERGY FOR WT STRUCTURE \"" << start_file <<
			"\" is " << dgbind_resenergy(1) << std::endl;
		energy_out << "********************************************************" << std::endl;
		energy_out << "# " << std::endl;
		energy_out << "CONTRIBUTIONS TO THE ENERGY FOR THE WT STRUCUTRE: " << std::endl;
		energy_out << "# " << std::endl;
		energy_out << "-----------------------------------------------" <<
			"--------------------------------------------------------" <<
			"--------------------------" << std::endl;
		energy_out << "            Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
			"   Ehbnd   Epair    Eref       Eh2o  Eh2ohb Eh2osol    Eres   SASA_pack " <<
			"pdb name" << std::endl;
		energy_out << "-----------------------------------------------" <<
			"--------------------------------------------------------" <<
			"--------------------------" << std::endl;
		energy_out << "DG_BIND   " << ' ' <<
			F( 7, 1, dgbind_atrenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_repenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_solenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_probenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_dunenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_intraresenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_hbenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_pair_energy(1) ) << ' ' <<
			F( 7, 1, dgbind_unfenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2oenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2ohbenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2osolenergy(1) ) << ' ' <<
			F( 7, 2, dgbind_resenergy(1) ) << ' ' <<
			F( 9, 3, D_sasa_pack(1) ) << ' ' <<
			A( 20, start_file ) << std::endl;
		energy_out << "-----------------------------------------------" <<
			"---------------------------------------------------------" <<
			"---------------------------" << std::endl;

		energy_out << "# " << std::endl;
		energy_out << "******************************************************" << std::endl;
		energy_out << "CHANGES IN ENERGY FOR MUTANT STRUCTURES: MUT-WT" << std::endl;
		energy_out << "******************************************************" << std::endl;
		energy_out << "# " << std::endl;
		energy_out << "            Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
			"   Ehbnd   Epair    Eref  Eh2o  Eh2ohb Sasa_pack Eres  Ereduce   Enew N_MUT";
		for ( int i = 0; i < cluster_size; ++i ) energy_out << " ROS# PDB# CH WT M NB" ;
		energy_out << std::endl;
		energy_out.close();
		energy_out.clear();
	}

// backup initialize arrays
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		res_orig(seqpos)=res(seqpos);
		res_variant_orig(seqpos)=res_variant(seqpos);
		for ( int d = 1; d <= 3; ++d ) {
			for ( int atm = 1; atm <= MAX_ATOM()(); ++atm ) {
				full_coord_orig(d,atm,seqpos) = full_coord(d,atm,seqpos);
			}
		}
	}

// determine how many neighbors each residue has and make neighbor list
	make_neighbor_info(res,total_residue,full_coord,neighborlist_copy,neighbors_copy);

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		all_contacts(seqpos) = false;
		design_contacts(seqpos) = false;
		repack_contacts(seqpos) = false;
	}

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( design_map.repack_residue(seqpos) ) {
			all_contacts(seqpos) = true;
			for ( int j = 1; j <= MAX_AA(); ++j ) {
				if ( design_map.get(seqpos,j) && j != res_orig(seqpos) ) {
					design_contacts(seqpos) = true;
				}
			}
		}
	}

	if ( !output_structures ) { // permute
		//initialize design array
		design_num = 0;
		for ( int pos2 = 1; pos2 <= total_residue; ++pos2 ) {
			if ( design_contacts(pos2) ) {
				design_array[design_num] = pos2;
				if ( debug_output ) std::cout << "DEBUG: design_array [" << design_num << "]=" << pos2 << std::endl;
				++design_num;
			}
		}
		if ( debug_output ) std::cout << "DEBUG: design_num " << design_num << std::endl;

		//pass the cluster size
		if ( use_cluster_size ) {
			n_cluster = cluster_size;
		} else {
			n_cluster=design_num;
		}

		//initialize permate array
		for ( int i = 0; i < design_num; ++i ) {
			if ( i < n_cluster ) permute_array.push_back(1);
			else permute_array.push_back(2);
		}
	}

	// main cycle for cluster design
	do {

		// reset sequence coordinate array
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			res(seqpos) = res_orig(seqpos);
			res_variant(seqpos) = res_variant_orig(seqpos);
			for ( int d = 1; d <= 3; ++d ) {
				for ( int atm = 1; atm <= MAX_ATOM(); ++atm ) {
					full_coord(d,atm,seqpos) = full_coord_orig(d,atm,seqpos);
				}
			}
		}

		// reset design repack array
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			design_contacts(seqpos) = false;
			repack_contacts(seqpos) = false;
			all_contacts(seqpos) = false;
		}

		if ( fix_mut ) {//read from the mutfile
			if ( debug_output ) std::cout << "Read from the mutfile" << mutname_in.c_str() << std::endl;
			if ( !getline( data_y, line ) ) {
				data_y.close();
				data_y.clear();
				std::cout<<"Finish reading the mutfile"<<std::endl;
			} //quit
			line_stream.clear();
			line_stream.str ( line );
			line_stream.seekg( std::ios_base::beg );
			line_stream >> tag ;
			if ( line_stream.fail() ) {
				line_stream.clear();
				break;
			}
			//std::cout<<tag<<std::endl;
			if ( tag=="DG_CALC" ) {
				int aa,pos1,pos2;
				char pdbchain;
				line_stream.clear();
				line_stream.str ( line );
				line_stream.seekg( std::ios_base::beg );
				line_stream >> bite( 7, tag ) >> skip(122) >> bite(4,n_cluster);
				//std::cout<<"DEBUG: line |"<<line<<"|"<<std::endl;
				for ( int j = 0; j < n_cluster ; ++j ) {
					line_stream >> skip(1) >> bite( 4, pos1 ) >> skip(1) >>
					 bite(4, pos2 ) >> skip(2) >> bite(1, pdbchain) >> skip(1) >>
					 bite(1,wtaa) >> skip(1) >> bite(1,mutaa) >> skip(4);
					num_from_res1(mutaa,aa);
					cluster_wtaa[j]=wtaa;
					cluster_mutaa[j]=mutaa;
					if ( use_pdb_numbering ) {
						cluster_pos[j]=pos2;
						for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
							if ( res_chain(seqpos) == pdbchain && pdb_res_num(seqpos) == pos2 ) {
								for ( int i  = 1; i  <= MAX_AA(); ++i  ) design_map.disable(seqpos,i);
								design_map.set(seqpos,aa);
								design_contacts(seqpos) = true;
							}
						}
					} else {
						cluster_pos[j]=pos1;
						for ( int i  = 1; i  <= MAX_AA(); ++i  ) design_map.disable(pos1,i);
						design_map.set(pos1,aa);
						design_contacts(pos1) = true;
					}
					if ( debug_output ) std::cout << "DEBUG: pos |" << cluster_pos[j] << "| wtaa |" << wtaa << "| mutaa |" << mutaa << ' ' << std::endl;
				}
			} else {
				continue;
			}
		} else { // permute

			if ( debug_output ) std::cout << "DEBUG0: total_residue " << total_residue << std::endl;
			for ( int i = 0; i < design_num; ++i ) {
				if ( permute_array[i] == 1 ) {
					design_contacts(design_array[i]) = true;
					if ( debug_output ) std::cout << "DEBUG: design_contacts " << design_array[i] << std::endl;
				}
			}
			if ( debug_output ) std::cout << "DEBUG: design_num " << design_num << std::endl;
			if ( debug_output ) std::cout << "DEBUG: design_contact ";
			for ( int seqpos = 1, i = 0; seqpos <= total_residue; ++seqpos ) {
				if ( design_contacts( seqpos ) ) {
					if ( debug_output ) std::cout << seqpos << " ";
					last_cpts[i] = cpts[i];
					cpts[i] = seqpos;
					++i;
				}
			}
			if ( debug_output ) std::cout << std::endl;
			//lin check point
			if ( check_point ) {
				//specify the code
				checkit = false;
				for ( int i = 0; i < last_cpts_size; ++i ) {
					if ( last_cpts[i] != cpts[i] ) {
						checkit = true;
						break;
					}
				}
				if ( debug_output ) std::cout << "Debug: check point.. " << checkit << std::endl;
				if ( ! checkit ) {
					skipit = false;
					for ( int i = 0; i < last_cpts_size; ++i ) {
						if ( debug_output ) std::cout << i << " " << cpts[i] << " " << last_store_cpts[i] << std::endl;
						if ( last_store_cpts[i] != cpts[i] ) {
							skipit = true;
							break;
						}
					}
					if ( skipit ) continue; //skip

				} else {
					if ( ( ofile = fopen(cp_file.c_str(),"r") ) == NULL ) {
						for ( int i = 0; i < last_cpts_size; ++i ) {
							last_store_cpts[i] = 0;
						}
					} else {
						for ( int i = 0; i < last_cpts_size; ++i ) {
							fscanf( ofile, "%d ", &last_store_cpts[i] );
					}
						fclose( ofile );
					}

					exist = 1;
					for ( int i = 0; i < last_cpts_size; ++i ) {
						if ( debug_output ) std::cout << i << " " << cpts[i] << " " << last_store_cpts[i] << std::endl;
						if ( cpts[i] > last_store_cpts[i] ) {
							exist = 0;
							break;
						} else if ( cpts[i] < last_store_cpts[i] ) {
							exist = 1;
							break;
						}
					}

					if ( exist == 0 ) {
						if ( ( ofile=fopen(cp_file.c_str(),"w") ) == NULL ) {
							fprintf(stderr,"Cannot Open Write Output File %s \n",cp_file.c_str());
							utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
						}

						for ( int i = 0; i < last_cpts_size; ++i ) {
							fprintf(ofile, "%d ", cpts[i]);
							last_store_cpts[i] = cpts[i];
						}
						fprintf(ofile,"\n");
						fclose(ofile);
					} else {
						for ( int i = 0; i < last_cpts_size; ++i ) {
							last_store_cpts[i] = 0;
						}
						continue;
					}
				}
				if ( debug_output ) std::cout << "Debug: check point continue.." << std::endl;
			} // checkpoint end
		}

		find_cluster_from_design_regions(full_coord,res,res_variant,total_residue,
		 n_cluster,design_contacts,is_good_cluster);
		find_repack_regions_from_design(full_coord,res,res_variant,total_residue,
		 design_contacts,repack_contacts,is_good_cluster);

		if ( debug_output ) {
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( repack_contacts(seqpos) ) std::cout << "DEBUG0: repack_contacts " << seqpos << " " << design_map.repack_residue(seqpos) << std::endl;
			}
			std::cout << "DEBUGtag: " << is_good_cluster << " " << n_cluster << std::endl;
		}

		if ( is_good_cluster ) {

//  determine energy with mutations plus compensating mutations

			// reset sequence coordinate array
			reset_move_map();//for minimization
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				res(seqpos) = res_orig(seqpos);
				res_variant(seqpos)=res_variant_orig(seqpos);
				for ( int d = 1; d <= 3; ++d ) {
					for ( int atm = 1; atm <= MAX_ATOM(); ++atm ) {
						full_coord(d,atm,seqpos) = full_coord_orig(d,atm,seqpos);
					}
				}
			}

			// reset design, repack, minimize array
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				all_contacts(seqpos) = false;
				allow_repack(seqpos) = false;
				allow_insert(seqpos) = false;
				for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
					design_matrix(aa,seqpos) = false;
				}
			}

			//  repack the WT structure
			std::cout << "Repack the WT structure region on seq " ;

			int i=0;
			char tmp_aa=' ';
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( design_contacts(seqpos) ) {
					design_position[i]=seqpos;
					++i;
					std::cout << SS( seqpos ) << "[D]" ;
					for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
						design_matrix(aa,seqpos) = design_map.get(seqpos,aa);
						if(design_map.get(seqpos,aa)) {
							res1_from_num(aa,tmp_aa);
							std::cout<<tmp_aa;
						}
					}
					all_contacts(seqpos) = true;
					allow_insert(seqpos) = true;
					allow_repack(seqpos) = true;
					maps_set_new_rotamer(seqpos);
				} else if ( repack_contacts(seqpos) && design_map.repack_residue(seqpos) ) {
					std::cout << SS( seqpos ) << "[R]" ;
					design_matrix(res(seqpos),seqpos) = true;
					all_contacts(seqpos) = true;
					allow_insert(seqpos) = true;
					allow_repack(seqpos) = true;
					maps_set_new_rotamer(seqpos);
				}
			}
			std::cout << std::endl;
			// continue;
			cluster_num=i;
			if ( cluster_num!=n_cluster && !design_neighbors ) std::cout << "ERROR: i!=n_cluster " << i << " " << n_cluster << std::endl;

			fullatom_nonideal_initialized_pose_from_misc( pose );
			PackerTask Task2( pose );
			Task2.set_task("packrot", make_output_file, all_contacts, include_current);
			Task2.setup_residues_to_vary();

			pack_rotamers(pose,Task2);
			pose.copy_to_misc();

			if ( minimize_structures ) {
				// timer begin
				if ( use_timer ) reset_timer();
				std::cout<<"Minimizing the region..."<<std::endl;
				score_set_new_pose();
				temp = score12();
				monte_carlo_reset();

				float const lj_increment = ( 1.0 - 0.02 ) / lj_ramp_cycles;
				for ( int k = 0; k <= lj_ramp_cycles; ++k ) {
					score_set_lj_rep_weight(0.02+k*lj_increment);
					minimize_allow_insert_trial("dfpmin",min_bb,min_sc,min_tol);
					monte_carlo_reset();
					if ( use_timer ) {
						float decoy_time;
						double const start_minutes = retrieve_timer_start();
						if ( start_minutes < 0.0 ) { // Uninitialized
							decoy_time = 0.0;
						} else {
							decoy_time = minutes() - start_minutes;
						}
						std::cout << "[TIMER] minimizing is taking " << F( 5, 3, decoy_time ) <<" minutes" << std::endl;
						reset_timer();
					}
				}
			}


			//   save original structure and sequence in *orig arrays (analyze_interface_ddg.h)
			copy_xyz_aan_aav_phi_psi(full_coord,xyz_orig,phi,phi_orig,psi,
															 psi_orig,res,aan_orig,res_variant,aav_orig,residue1,residue1_orig,residue3,residue3_orig,
															 Eposition,nres_orig,1,total_residue);

			for ( int which_partner = n_partner; which_partner <= 3; ++which_partner ) {

				set_partner_specific_params(which_partner,startres,endres,p1_start,p1_end,
																		p2_start,p2_end,nres_orig);


				copy_xyz_aan_aav_phi_psi(xyz_orig,full_coord,phi_orig,phi,psi_orig,
																 psi,aan_orig,res,aav_orig,res_variant,residue1_orig,residue1,
																 residue3_orig,residue3,Eposition,total_residue,startres,endres);

				score_set_new_pose();
				temp = score12();
				calc_sasa_pack_score(total_residue, res, res_variant, full_coord);
				if ( output_structures && which_partner == 3 ) {
					std::string tag;
					for ( int j = 0; j < cluster_num; ++j ) {
						std::ostringstream a_stream;
						a_stream<< cluster_wtaa[j] << cluster_pos[j];
						tag += "_" + a_stream.str();
					}
					outname = pdb_out_path + filename + tag + "_repack.pdb";
					dump_fullatom_pdb(outname);
				}

				// permute : output the score
				save_scores(which_partner,total_residue);
				save_wt_scores(which_partner);
			}
			compile_binding_energy(1); // counter for list

			//  design the WT structure
			std::cout << "Design the above region..." << std::endl ;
			if ( int( std::pow( static_cast< double >( n_mutatpos ), cluster_num ) ) < 0 || debug_output ) {
				std::cout << "DEBUG: n_mutatpos=" << n_mutatpos
				 << " cluster_num=" << cluster_num << " the combinations is "
				 << std::pow( static_cast< double >( n_mutatpos ), cluster_num ) << std::endl;
			}
			for ( int j = 0; j < cluster_num; ++j ) {
				pow_tmp[j]=std::pow( static_cast< double >( n_mutatpos ), j );
			}
			// allow to try several mutants on each positon (n_mutatpos)
			for ( int i = 0, ie = int( std::pow( static_cast< double >( n_mutatpos ), cluster_num ) ); i < ie; ++i ) {

				if ( debug_output ) std::cout << "DEBUG: i=" << i << " ";
				// change the mutation
				for ( int j = 0; j < cluster_num; ++j ) {
					pos_index_old[j] = pos_index[j];
					pos_index[j] = int ( i / pow_tmp[j] ) % n_mutatpos;
				}
				int seqpos;
				for ( int j = 0; j < cluster_num; ++j ) {
					if ( debug_output ) std::cout << " pos_index[" << j << "]=" << pos_index[j] << " pos_index_old[" << j << "]=" << pos_index_old[j];
					if ( pos_index[j] != pos_index_old[j] ) {
						//---						seqpos=design_map[j];
						seqpos=design_position[j];
						if ( debug_output ) std::cout << " design_position: "  <<  design_position[j];
						if ( pos_index[j] == 0 ) {
							for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
								design_matrix(aa,seqpos) = design_map.get(seqpos,aa);
							}
						} else {
							design_matrix(res(seqpos),seqpos) = false;
						}
					}
				}
				if ( debug_output ) std::cout << std::endl;

				// reset sequence coordinate array
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					res(seqpos) = res_orig(seqpos);
					res_variant(seqpos)=res_variant_orig(seqpos);
					for ( int d = 1; d <= 3; ++d ) {
						for ( int atm = 1; atm <= MAX_ATOM(); ++atm ) {
							full_coord(d,atm,seqpos) = full_coord_orig(d,atm,seqpos);
						}
					}
				}

				fullatom_nonideal_initialized_pose_from_misc( pose );
				PackerTask Task3( pose );
				Task3.set_task("design", make_output_file, all_contacts, include_current);
				Task3.setup_residues_to_vary();

				pack_rotamers(pose, Task3);
				pose.copy_to_misc();

				if ( minimize_structures ) {
					std::cout << "Minimizing the region..." << std::endl;
					score_set_new_pose();
					temp = score12();
					monte_carlo_reset();

					float const lj_increment = ( 1.0 - 0.02 ) / lj_ramp_cycles;
					for ( int k = 0; k <= lj_ramp_cycles; ++k ) {
						score_set_lj_rep_weight(0.02+k*lj_increment);
						minimize_allow_insert_trial("dfpmin",min_bb,min_sc,min_tol);
						monte_carlo_reset();
					}
				}

				//   save original structure and sequence in *orig arrays (analyze_interface_ddg.h)
				copy_xyz_aan_aav_phi_psi(full_coord,xyz_orig,phi,phi_orig,psi,
															 psi_orig,res,aan_orig,res_variant,aav_orig,residue1,residue1_orig,residue3,residue3_orig,
															 Eposition,nres_orig,1,total_residue);

				for ( int which_partner = n_partner; which_partner <= 3; ++which_partner ) {

					set_partner_specific_params(which_partner,startres,endres,p1_start,p1_end,
																		p2_start,p2_end,nres_orig);


					copy_xyz_aan_aav_phi_psi(xyz_orig,full_coord,phi_orig,phi,psi_orig,
																 psi,aan_orig,res,aav_orig,res_variant,residue1_orig,residue1,
																 residue3_orig,residue3,Eposition,total_residue,startres,endres);

					// set fragment size to be whole protein so that score function counts
					// everything
					score_set_new_pose();
					temp = score12();
					calc_sasa_pack_score(total_residue,res,res_variant,full_coord);
					if ( output_structures && which_partner == 3 ) {
						std::string tag;
						for ( int j = 0; j < cluster_num; ++j ) {
							std::ostringstream a_stream;
							a_stream << cluster_wtaa[j] << cluster_pos[j] << cluster_mutaa[j];
							tag += "_" + a_stream.str() ;
						}
						outname = pdb_out_path + filename + tag + "_design.pdb";
						dump_fullatom_pdb(outname);
					}

					//permute : output the score
					save_scores(which_partner,total_residue);
				}
				compile_binding_energy(i+2); // counter for list

				float res_reduce,res_lin;
				res_reduce=dgbind_atrenergy(i+2)+dgbind_solenergy(i+2)+dgbind_hbenergy(i+2)-(dgbind_atrenergy(1)+dgbind_solenergy(1)+dgbind_hbenergy(1));
				res_lin=0.74*(dgbind_atrenergy(i+2)-dgbind_atrenergy(1))+0.57*(dgbind_solenergy(i+2)-dgbind_solenergy(1))+0.9*(dgbind_hbenergy(i+2)-dgbind_hbenergy(1))+0.08*(dgbind_repenergy(i+2)-dgbind_repenergy(1));
				int n_mut=0;
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					if ( res(seqpos) != res_orig(seqpos) ) ++n_mut;
				}
				if ( debug_output ) std::cout << "DEBUG:  n_mut" << n_mut << std::endl;

				energy_out.open( mutname.c_str(), std::ios_base::app );
				energy_out << "DG_CALC  " <<
					F( 7, 3, dgbind_atrenergy(i+2) - dgbind_atrenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_repenergy(i+2) - dgbind_repenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_solenergy(i+2) - dgbind_solenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_probenergy(i+2) - dgbind_probenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_dunenergy(i+2) - dgbind_dunenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_intraresenergy(i+2) - dgbind_intraresenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_hbenergy(i+2) - dgbind_hbenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_pair_energy(i+2) - dgbind_pair_energy(1) ) << ' ' <<
					F( 7, 3, dgbind_unfenergy(i+2) - dgbind_unfenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_h2oenergy(i+2) - dgbind_h2oenergy(1) ) << ' ' <<
					F( 7, 3, dgbind_h2ohbenergy(i+2) - dgbind_h2ohbenergy(1) ) << ' ' <<
					F( 7, 3, D_sasa_pack(i+2) - D_sasa_pack(1) ) << ' ' <<
					F( 7, 3, dgbind_resenergy(i+2) - dgbind_resenergy(1) ) << ' ' <<
					F( 7, 3, res_reduce ) << ' ' << F( 7, 3, res_lin ) << ' ' <<
					I( 4, cluster_num );

				for ( int j = 0; j < cluster_num; ++j ) {
					int seqpos = design_position[j];
					res1_from_num(res_orig(seqpos),wtaa);
					res1_from_num(res(seqpos),mutaa);
					energy_out << " " << I( 4, seqpos ) << " " <<
						I( 4, pdb_res_num(seqpos) ) << "  " <<
						res_chain(seqpos) << " " << wtaa << " " << mutaa << "  " <<
						I( 2, neighbors_copy(seqpos) );
				}
				energy_out << std::endl;
				energy_out.close();
				energy_out.clear();
			}
		}
	} while ( fix_mut || std::next_permutation( permute_array.begin(), permute_array.end() ) );

	if ( fix_mut ) {
		data_y.close();
		data_y.clear();
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin identify contacts
///
/// @brief identifies residues that contact a given residue
///
/// @detailed
/// @detailed
///
/// @param[in]    in - seqpos1, sequence position of interest
/// @param[in]    in - coord, coordinates of the protein
/// @param[in]    in - sequence, sequence of the protein
/// @param[in]    in - aav, amino acid variant at each position
/// @param[in]    in - nres, number of residues in the protein
/// @param[out]    out - contacts, list of residues that contact seqpos1
///
/// @return
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
identify_contacts(
	int const seqpos1,
	FArray3Da_float coord,
	FArray1Da_int sequence,
	FArray1Da_int aav,
	int const nres,
	FArray1Da_bool contacts
)
{
	using namespace aaproperties_pack;
	using namespace param;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	sequence.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	contacts.dimension( MAX_RES() );

	contacts = false;

	int const aa1 = sequence(seqpos1);
	for ( int seqpos2 = 1, atm1e = nheavyatoms( aa1, aav(seqpos1) );
	 seqpos2 <= nres; ++seqpos2 ) {
		if ( seqpos1 != seqpos2 ) {
			int const aa2 = sequence(seqpos2);
			for ( int atm1 = 5, atm2e = nheavyatoms( aa2, aav(seqpos2) );
			 atm1 <= atm1e; ++atm1 ) {
				for ( int atm2 = 5; atm2 <= atm2e; ++atm2 ) {
					float dis;
					distance_bk( coord(1,atm1,seqpos1), coord(1,atm2,seqpos2), dis );
					if ( dis < 5.5 ) {
						contacts(seqpos2) = true;
					}
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_cluster_from_design_regions
///
/// @brief check the input regions are really cluster which is spatially close
///
/// @detailed
/// only consider the designs which is spatially close. i,j,k are the three distance values: i  The default is 5.0 8.0 12.0.
///
/// @return is_good_cluster
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_cluster_from_design_regions(
	FArray3Da_float coord,
	FArray1Da_int sequence,
	FArray1Da_int aav,
	int const nres,
	int const n_cluster,
	FArray1Da_bool design_region,
	bool & is_good_cluster
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param;
	using namespace analyze_interface_ddg_ns;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	sequence.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	design_region.dimension( MAX_RES() );

	FArray1D_int design_array( MAX_RES()() );
	FArray1D_bool is_cluster( MAX_RES()() );
	int design_num;
	bool is_neighbor;
	int aa1,aa2,pos1,pos2;

	float cb_dis,ca_dis;
	std::vector< std::pair< float,int> > v_dis;

	static FArray1D_int const use_atom( MAX_AA(), atn_use_atom_initializer );

	design_num = 0;
	for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
		if ( design_region(seqpos) ) {
			design_array[design_num] = seqpos;
			++design_num;
		}
	}

	if ( design_neighbors ) {
		for ( int i = 0; i < design_num; ++i ) {
			int seqpos = design_array[i];
			for ( int pos2 = 1; pos2 <= nres; ++pos2 ) {
				are_they_neighbors(sequence(seqpos),sequence(pos2),coord(1,1,seqpos),coord(1,1,pos2),design_dis_cut1,design_dis_cut2,ca_dis,cb_dis,is_neighbor);
				if ( is_neighbor ) {
					design_region(pos2) = true;
					if ( debug_output ) std::cout << "Debug: design_region pos1=" << seqpos
					 << " pos2=" << pos2 << " " << design_dis_cut1 << " " << design_dis_cut2
					 << " " << ca_dis << " " << cb_dis << " " << is_neighbor << std::endl;
				}
			}
		}
	}

	if ( design_num == 0 ) {
		is_good_cluster = false;
		return;
	} else if ( n_cluster == 1 || fix_mut || !use_real_cluster_only || design_neighbors ) {
		is_good_cluster = true;
		return;
	}

	for ( int i = 0; i < design_num; ++i ) is_cluster[i] = false;

	//is a good cluster?
	for ( int i = 0; i < design_num; ++i )
		for ( int j = i; j < design_num; ++j ) {
			pos1 = design_array[i];
			aa1 = sequence(pos1);
			pos2 = design_array[j];
			aa2 = sequence(pos2);
			distance_bk(coord(1,use_atom(aa1),pos1),coord(1,use_atom(aa2),pos2),cb_dis);
			if ( cb_dis > cluster_dis_cut3 ) {
				is_good_cluster = false;
				return;
			}
		}

	for ( int i = 0; i < design_num; ++i ) {
		if ( is_cluster[i] ) continue;
		pos1 = design_array[i];
		aa1 = sequence(pos1);
			for ( int j = 0; j < design_num; ++j ) {
				if ( j == i ) continue;
				pos2 = design_array[j];
				aa2 = sequence(pos2);
				is_neighbor = false;
				are_they_neighbors(aa1,aa2,coord(1,1,pos1),coord(1,1,pos2),cluster_dis_cut1,cluster_dis_cut2,ca_dis,cb_dis,is_neighbor);
				if ( is_neighbor ) {
					if ( i == 0 ) {
						is_cluster[i] = true;
						is_cluster[j] = true;
					} else {
						if ( is_cluster[j] ) is_cluster[i] = true;
					}
				}
				if ( debug_output ) std::cout << "DEBUG: pos1 " << pos1 << " and pos2 " << pos2 << " is_neighbor: " << is_neighbor << " ca_dis: " << ca_dis << " cb_dis: " << cb_dis << std::endl;
			}
	}
	for ( int i = 0; i < design_num; ++i ) {
		if ( !is_cluster[i] ) {
			is_good_cluster = false;
			return;
		}
	}
	is_good_cluster = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_cluster_from_design_regions
///
/// @brief define repacked region based on designed region
///
/// @detailed
/// if repack neighbors during design. The i j are the two distance parameters to define the repacked regions around the designed regions. The default is 5.0 8.0. And see the above.
///
/// @return is_good_cluster
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_repack_regions_from_design(
	FArray3Da_float coord,
	FArray1Da_int sequence,
	FArray1Da_int aav,
	int const nres,
	FArray1Da_bool design_region,
	FArray1Da_bool repack_region,
	bool & is_good_cluster
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param;
	using namespace analyze_interface_ddg_ns;

	coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	sequence.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	design_region.dimension( MAX_RES() );
	repack_region.dimension( MAX_RES() );

	FArray1D_int design_array( MAX_RES()() );
	int design_num;

	bool is_neighbor;
	float cb_dis,ca_dis;
	int aa1,aa2;

	design_num = 0;
	for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
		repack_region(seqpos) = false;
		if ( design_region(seqpos) ) {
			design_array[design_num] = seqpos;
			++design_num;
		}
	}
	if ( !is_good_cluster || !repack_neighbors ) return;

	//define the repack region
	int seqpos1;
	for ( int seqpos2 = 1; seqpos2 <= nres; ++seqpos2 ) {
		aa2 = sequence(seqpos2);
		for ( int i = 0; i < design_num; ++i ) {
			seqpos1 = design_array[i];
			aa1 = sequence(seqpos1);

			are_they_neighbors(aa1,aa2,coord(1,1,seqpos1),coord(1,1,seqpos2),repack_dis_cut1,repack_dis_cut2,ca_dis,cb_dis,is_neighbor);
			if ( is_neighbor ) {
				repack_region(seqpos2) = true ;
				if ( debug_output ) std::cout << "DEBUG: repack_region  seqpos1: " << seqpos1 << " seqpos2: " << seqpos2 << " ca_dis: " << ca_dis << " cb_dis: " << cb_dis << std::endl;
				break;
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_redesign_mutation_list
///
/// @brief
/// 1. read-in complete list of mutations (either single-chain or interface) for
///    a single pdb file from stability_mutlist
/// 2. converts information in the redesign_mutlist (pdb numbers and chains)
///    into ROSETTA numbering
///
/// @detailed
///
/// @return
///
/// @global_read
/// file named redesign_mutlist in design.h
///
/// @param[in,out] - mutations_seqpos - input/output - residues numbers of all mutations
///                            in all structures
/// @param[in,out] - mutation_aa      - input/output - amino acid types of all mutations
///                            in all structures
/// @param[in,out] - nsim             - input/output - number of mutated strcutures
/// @param[in,out] - n_mut            - input/output - number of mutatations per structure
/// @param[in]   pdb_res_chain    - conversion of pdb residue
///                            number and chain id to rosetta number
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Deanne Sammond 2/6/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_redesign_mutation_list(
	FArray2Da_int mutation_seqpos,
	FArray2Da_int mutation_aa,
	int & nsim,
	FArray1Da_int nmut
)
{
	using namespace files_paths;
	using namespace design;
	using namespace misc;
	using namespace param;

	mutation_seqpos.dimension( max_mut_per_sim, max_sim );
	mutation_aa.dimension( max_mut_per_sim, max_sim );
	nmut.dimension( max_sim );

//bk local variables
	std::map<std::pair<int,int>,int> pdb_res_chain_to_rosetta;
	int rosetta_mut_seqpos, num_pdb_chain, pdb_mut_seqpos;
	char pdb_wt_aa, pdb_mut_aa;
	char pdb_chain;
	int num_wt_aa, num_mut_aa;
	std::string startchar;


	if ( design_mutlist != "none" ) {
		data_x.open( design_mutlist );
	  std::cout << "reading mutlist: " << data_x.filename() << std::endl;
	} else {
		std::cout << "did not find the design mutation list, stop!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//ds get the value for pdb_res_chain - convert the pdb sequence number and
//ds chain id to a rosetta sequence number
	map_all_pdb_pos_to_rosetta(pdb_res_chain_to_rosetta);

//ds find start of design_mutation list
L10:
	data_x >> bite( startchar ) >> skip;
	if ( data_x.eof() ) {
		goto L20;
	} else if ( data_x.fail() ) {
		data_x.clear();
		data_x >> skip;
		goto L10;
	}
	if ( startchar != "START" ) goto L10;

//ds get number of mutated structures in the file
	data_x >> bite( 5, nsim ) >> skip;
	if ( data_x.eof() ) goto L20;
	std::cout << "found number of mutated structures: " << SS( nsim ) << std::endl;
	if ( nsim > max_sim ) {
		std::cout << "max_sim EXCEEDED" << std::endl;
		std::cout << "change max_sim in design.cc to at least " << SS( nsim ) << std::endl;
		std::cout << "stop..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//ds loop through all mutated structures
	for ( int sim = 1; sim <= nsim; ++sim ) {
		data_x >> bite( 3, nmut(sim) ) >> skip;
		if ( data_x.eof() ) {
			goto L20;
		} else if ( data_x.fail() ) {
			goto L30;
		}
		for ( int mut = 1, mute = nmut(sim); mut <= mute; ++mut ) {
			data_x >> skip( 9 ) >>
			 bite( 4, pdb_mut_seqpos ) >> skip( 2 ) >>
			 bite( pdb_chain ) >> skip( 2 ) >>
			 bite( pdb_wt_aa ) >> skip( 2 ) >>
			 bite( pdb_mut_aa ) >> skip;
			if ( data_x.eof() ) {
				goto L20;
			} else if ( data_x.fail() ) {
				goto L30;
			}
			num_from_res1(pdb_wt_aa,num_wt_aa);
			num_from_res1(pdb_mut_aa,num_mut_aa);
			convert_chain_char_to_chain_num(pdb_chain,num_pdb_chain);
			rosetta_mut_seqpos =
				pdb_res_chain_to_rosetta[ std::pair<int,int>(pdb_mut_seqpos,num_pdb_chain) ];
			if ( safety ) {
				if ( num_wt_aa != res(rosetta_mut_seqpos) ) {
					std::cout << std::endl;
					std::cout << "WT SEQUENCE IN MUTLIST DOES NOT MATCH PDB FILE:" << std::endl;
					std::cout << "PDB residue number " << pdb_mut_seqpos << " is aa" <<
					 res(rosetta_mut_seqpos) << " but in mutfile it is aa " <<
					 num_wt_aa << std::endl;
					std::cout << "STOP" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
			mutation_seqpos(mut,sim) = rosetta_mut_seqpos;
			mutation_aa(mut,sim) = num_mut_aa;
		} // all mutations in one structure
	} // all structures

	data_x.close();
	data_x.clear();
	return;

L20:
	data_x.close();
	data_x.clear();
	return;
L30:
	std::cout << "ERROR READING mutlist" << std::endl;
	data_x.close();
	data_x.clear();
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ddG_bind
///
/// @brief
/// Calculate delta delta G of binding: dG mutant complex - dG WT
///
/// @detailed
/// The routine is passed a list of mutations in the mutant complex. Before
/// calculating energies it repacks the mutant residues and the neighbors of
/// the mutated positions.  The same residues are repacked in the WT structure
/// to make the comparison 'fair'.
///
/// @return
///
/// @global_read
///   reads in protein information from the namespace misc
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Brian Kuhlman
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_ddG_bind_mut(
	int & nmut,
	FArray1DB_int & mut_pos,
	FArray1DB_int & mut_aa
)
{
	using namespace param;
	using namespace misc;
	using namespace design;

	// variables to hold protein info
	FArray3D_float full_coord_orig( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_orig( MAX_RES()() );
	FArray1D_float psi_orig( MAX_RES()() );
	FArray1D_int res_orig( MAX_RES()() );
	FArray1D_int res_variant_orig( MAX_RES()() );
	FArray1D_char residue1_orig( MAX_RES(), ' ' );
	FArray1D_string residue3_orig( MAX_RES(), std::string( 3, ' ' ) );

	int total_residue_orig;

	FArray1D_int neighbors( MAX_RES()() );
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );
	FArray1D_float dG( 2 );
	FArray1D_bool mut_residue( MAX_RES()(), false);
	FArray1D_bool repack_residue( MAX_RES()(), false);
	FArray1D_float energy( 3 );  // for energies of bound and unbound chains

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	// flags for pack_rotamers
	FArray1D_bool allow_repack( MAX_RES()(), false );  // dummy variable in this instance
	//bool make_output_file = false;
	bool include_current = true;    //use side chains with native coord
	std::string packmode( "design" );
	use_design_matrix = true;

	//bk save original protein
	copy_xyz_aan_aav_phi_psi(full_coord,full_coord_orig,phi,phi_orig,psi,
													 psi_orig,res,res_orig,res_variant,res_variant_orig,residue1,residue1_orig,
													 residue3,residue3_orig,Eposition,total_residue_orig,1,total_residue);

	//bk pick out neighbor residues that will be repacked
	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);
	for ( int mut = 1; mut <= nmut; ++mut ) {
		mut_residue(mut_pos(mut)) = true;
	}
	for ( int mut = 1; mut <= nmut; ++mut ) {
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if (neighborlist(mut_pos(mut),seqpos) && !mut_residue(seqpos)) {
				repack_residue(seqpos) = true;
			}
		}
	}

	for ( int wc = 1; wc <= 2; ++wc ) {   // loop through WT and mutant complex
		for ( int wp = 1; wp <= 3; ++wp ) {  // evaluate bound and unbound chains

			//bk set residue boundaries for the desired chain(s)
			int start_res;
			int end_res;
			int offset;
			if (wp == 1) {      // complex
				start_res = 1;
				end_res = total_residue_orig;
				offset = 0;
			} else if (wp == 2) {       // unbound partner 1
				start_res = 1;
				end_res = domain_end(1);
				offset = 0;
			} else {                        // unbound partner 2
				start_res = domain_end(1) + 1;
				end_res = total_residue_orig;
				offset = start_res - 1;
			}

			//bk fill protein arrays with the current chain(s)
			copy_xyz_aan_aav_phi_psi(full_coord_orig,full_coord,phi_orig,phi,psi_orig,
															 psi,res_orig,res,res_variant_orig,res_variant,residue1_orig,residue1,
															 residue3_orig,residue3,Eposition,total_residue,start_res,end_res);


			//bk map mutant positions and repack positions to the current chain(s)
			int nmut_curr = 0;
			FArray1D_int mut_pos_curr( MAX_RES()(), 0 );
			FArray1D_int mut_aa_curr( MAX_RES()(), 0 );
			FArray1D_bool repack_residue_curr( MAX_RES()(), false );
			for ( int mut = 1; mut <= nmut; ++mut ) {
				int mpos = mut_pos(mut);
				if ( mpos >= start_res && mpos <= end_res ) {
					++nmut_curr;
					mut_pos_curr(nmut_curr) = mpos - offset;
					mut_aa_curr(nmut_curr) = mut_aa(mut);
				}
			}
			for ( int seqpos = start_res; seqpos <= end_res; ++seqpos ) {
				repack_residue_curr(seqpos - offset) = repack_residue(seqpos);
			}

			//bk set which residues to repack by filling global design_matrix
			design_matrix = false;     //initialize

			//bk set design matrix for mutation positions
			for ( int mut = 1; mut <= nmut_curr; ++mut ) {
				int mpos = mut_pos_curr(mut);
				int maa = mut_aa_curr(mut);
				if ( wc == 1 ) {                                         // WT, don't mutate
					design_matrix(res(mpos),mpos) = true;
				} else {                                                 // mutate
					design_matrix(maa,mpos) = true;
				}
			}

			//bk set design matrix to repack mutation neighbors
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if (repack_residue_curr(seqpos)) {
					design_matrix(res(seqpos),seqpos)=true;
				}
			}

			//bk repack/mutate the current chain(s)
			fullatom_nonideal_initialized_pose_from_misc( pose );

			//yl, Create PackerTask and setup values before pass into pack_rotamers
			PackerTask Task( pose );
			Task.set_task(packmode, false, allow_repack, include_current);
			Task.setup_residues_to_vary();

			pack_rotamers( pose,Task );
			pose.copy_to_misc();

			// score, save energies
			score_set_new_pose();
			energy(wp) = score12();

		} // loop through partners

		dG(wc) = energy(1) - energy(2) - energy(3);  // binding energy for complex

	} // loop through complexes

	//bk return protein to original coordinates/sequence
	copy_xyz_aan_aav_phi_psi(full_coord_orig,full_coord,phi_orig,phi,psi_orig,
													 psi,res_orig,res,res_variant_orig,res_variant,residue1_orig,residue1,
													 residue3_orig,residue3,Eposition,total_residue,1,total_residue_orig);


	return dG(2)-dG(1);  // binding energy of mutant minus wt

}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_pmut_to_scan
///
/// @brief
/// used by altered_specificity, sets which point
/// mutants to scan
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Brian Kuhlman
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_pmut_to_scan(
	std::vector< int > & scan_pmut
)
{
	using namespace param;
	using namespace design;
	using namespace files_paths;
	using namespace param_aa;
	using namespace pdb;

	std::string pmut_file;

	scan_pmut.clear();

	//bk read in file that specifies residues to screen
	stringafteroption( "pmut", "none", pmut_file );

	if ( pmut_file != "none" ) {

		data_x.open( pmut_file );

		if ( !data_x ) {
			std::cout << "Open failed for file: " << data_x.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			std::map<std::pair<int,int>,int> pdb_res_chain_to_rosetta;
			map_all_pdb_pos_to_rosetta(pdb_res_chain_to_rosetta);
			int pdb_pos; char pdb_chain; int num_pdb_chain; int ros_pos;
			while ( !data_x.eof() ) {
				data_x >> pdb_pos >> pdb_chain;
				if ( data_x.good() ) {
					convert_chain_char_to_chain_num(pdb_chain,num_pdb_chain);
					ros_pos =
						pdb_res_chain_to_rosetta[ std::pair<int,int>(pdb_pos,num_pdb_chain) ];
					scan_pmut.push_back(ros_pos);
				}
			}
		}
		data_x.close();
		data_x.clear();

	} else {

		//bk identify interface residues
		//bk fills arrays in namespace design
		interface_residues();
		std::cout << "in scan_pmut #intres " << n_interface_residues << std::endl;
		for ( int i = 1; i <= n_interface_residues; ++i ) {
			int seqpos = interface_residue_list(i);
			scan_pmut.push_back(seqpos);
		}

	}

	//bk write to screen point mutants that will be considered
	std::cout << " " << std::endl;
	std::cout << "Residue positions where point mutants will be forced by alter_spec"
						<< std::endl;
	std::cout << "pdb_pos " << "pdb_chain " << "rosetta_pos"
						<< std::endl;
	char pdb_chain;
	for ( std::vector< int >::size_type imut = 0;
				imut != scan_pmut.size(); ++imut ) {
		int ros_pos = scan_pmut[imut];
		int pdb_pos = pdb_res_num(ros_pos);
		pdb_chain = res_chain(ros_pos);
		std::cout << I( 5, pdb_pos ) << "     ";
		std::cout << pdb_chain << "      ";
		std::cout << I( 5, ros_pos ) << " " << std::endl;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_fixlist
///
/// @brief
/// used by altered_specificity, sets which point
/// mutants to scan
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Brian Kuhlman
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_fixlist(
	FArray1DB_bool & fix_res
)
{
	using namespace param;
	using namespace misc;
	using namespace design;
	using namespace files_paths;
	using namespace param_aa;
	using namespace pdb;
	std::string fix_file;

	//bk read in file that specifies residues to fix
	stringafteroption( "fix", "none", fix_file );

	if ( fix_file != "none" ) {

		data_x.open( fix_file );

		if ( !data_x ) {
			std::cout << "Open failed for file: " << data_x.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {

			std::cout << "Reading in residues that are not allowed to change" << std::endl;
			std::cout << "pdb_pos " << "pdb_chain " << "rosetta_pos" << std::endl;

			std::map<std::pair<int,int>,int> pdb_res_chain_to_rosetta;
			map_all_pdb_pos_to_rosetta(pdb_res_chain_to_rosetta);

			int pdb_pos; char pdb_chain; int num_pdb_chain; int ros_pos;
			while ( !data_x.eof() ) {
				data_x >> pdb_pos >> pdb_chain;
				if ( data_x.good() ) {
					convert_chain_char_to_chain_num(pdb_chain,num_pdb_chain);
					ros_pos =
						pdb_res_chain_to_rosetta[ std::pair<int,int>(pdb_pos,num_pdb_chain) ];
					fix_res(ros_pos) = true;
					std::cout << I(5, pdb_pos) << "     ";
					std::cout << pdb_chain << "      ";
					std::cout << I(5, ros_pos) << " " << std::endl;
				}
			}
		}

		data_x.close();
		data_x.clear();
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin apply_dock_perturbation
///
/// @brief
/// used by design_dock_pert, apply docking rigid body movement in design
/// and check the backbone clash between two partners
///
/// @detailed
/// this function does currently only work properly for ligand,
/// since to move ligand change only 1body energy terms
/// this function does currently only work properly for 1body energy terms.
/// if you want to uese it for protein-protein you need to ensure that
/// also the two body energy terms are recomputed, therefore we return for now
///
/// @return
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Lin Jiang, Kaufmann
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
apply_dock_perturbation(
	int total_residue,
  FArray3D_float Eposition
)
{
  using namespace docking;
  using namespace design;
  using namespace files_paths;

//mj return if it is not ligand
  if ( !get_ligand_flag() ) return;
  if ( !design_dock_pert ) return;

//lin     local
  FArray1D_float translation( 3 );
  FArray1D_float rotation( 3 );
//	bool file_exists;
  float bb_vdw_score;
  float const bb_vdwE_MAX = {2.0};

//lin apply the docking translation and rotation matrix
//lin to full_coord based on the best_full_coord
	if( get_ligand_flexible_flag() ){
		//kwk here we recover all the precompute ligand conformations
		for( std::vector<Ligand *>::iterator cur_ligand=
			ligand::ligand_ptr_vector.begin();cur_ligand!=
			ligand::ligand_ptr_vector.end(); cur_ligand++){

			if( (*cur_ligand)->ligand_conformations_base.empty() ){
				generate_base_ligand_conformations();
			}
			(*cur_ligand)->ligand_conformations.clear();
			(*cur_ligand)->ligand_conformations=
				(*cur_ligand)->ligand_conformations_base;
		}
	}

	do {

		//mj make random small perturbation
		choose_rigid_body(translation,rotation,normal_perturbation,
			parallel_perturbation,rotational_perturbation);
		apply_rigid_body(translation,rotation);

		if( get_ligand_flexible_flag() ){
			size_t lig_conf=static_cast<size_t>(random_range(0,
				(*ligand::ligand_one).ligand_conformations.size()));
//		std::cout << "kwk help lig_conf " << lig_conf << std::endl;
			(*ligand::ligand_one).change_to_ligand_conformation(lig_conf);
		}

		//mj check the clash between ligand and backbone
    ligand_bb_vdw(bb_vdw_score,(*ligand::ligand_one));

    //mj std::cout << "bb_vdw_score: " << bb_vdw_score << std::endl;
	} while (bb_vdw_score > bb_vdwE_MAX);
	//mj	} while ( false);

// lin these should be good lines for non ligand small perturbation
//    {
//      //no clash check between the backbones of both partners
//      //will be added soon
//      choose_rigid_body(translation,rotation,normal_perturbation,
//  		      parallel_perturbation,rotational_perturbation);
//      apply_rigid_body(translation,rotation);
//    }

	if( get_ligand_flexible_flag() ){
		std::multimap< float, ligand::LigandInternalCoord > conformations;
		for( std::vector<Ligand *>::iterator cur_ligand=
			ligand::ligand_ptr_vector.begin(); cur_ligand!=
			ligand::ligand_ptr_vector.end(); cur_ligand++){

			conformations.clear();
			float lig_bb_vdw=1000.0;
			//kwk evaluate and store ligand backbone van der Waals scores
			//kwk for each ligand rotamer
			if( (*cur_ligand)->ligand_conformations.size() == 0 ){
				(*cur_ligand)->ligand_conformations.push_back(
					get_ligand_internal_conformation(**cur_ligand));
			}
			for( size_t lig_rot=0;
				lig_rot<(*cur_ligand)->ligand_conformations.size();
				lig_rot++){

				(*cur_ligand)->change_to_ligand_conformation(lig_rot);
				ligand_bb_fa_vdw( lig_bb_vdw, (**cur_ligand));
				std::cout << "ligand bb clash " << lig_bb_vdw << std::endl;
				conformations.insert(std::pair<float, ligand::LigandInternalCoord>(lig_bb_vdw,
					get_ligand_internal_conformation(**cur_ligand)));
			}
			//kwk if the best conformation in greater than 10 store just
			//kwk the best ligand rotamer
			//kwk else store all ligand rotamers up to 10
			(*cur_ligand)->ligand_conformations.clear();
			std::cout << "lowest bb clash " << (*conformations.begin()).first << std::endl;
			if( (*conformations.begin()).first > 10){
				(*cur_ligand)->ligand_conformations.push_back(
					(*conformations.begin()).second
				);
			}else{
				std::multimap< float, ligand::LigandInternalCoord >::const_iterator
					last_conf=conformations.upper_bound(10);
//				std::cout << "kwk got here add multiple conformations" << std::endl;
				for(std::multimap< float, ligand::LigandInternalCoord >::iterator
					c_conf=conformations.begin(); c_conf!=last_conf; c_conf++){
	  				(*cur_ligand)->ligand_conformations.push_back((*c_conf).second);
//	  				std::cout << "push back conf" << std::endl;
				}
			}
		}
	}


//set up for docking movement-----centroid of partner 1 & 2
chain_centroid_CA(Eposition,1,total_residue,part_centroid(1,1));

// lin these should be good lines for non ligand small perturbation
//  	} else if ( dna_interface ) {
//  	//	chain_centroid_CA(Eposition,part_begin(1),part_end(1),part_centroid(1,1));
//  	//	chain_centroid_DNA(centroid,part_begin(2),part_end(2),part_centroid(1,2));
//  	} else {
//  		for ( int i = 1; i <= n_monomers; ++i ) {
//  			chain_centroid_CA(Eposition,part_begin(i),part_end(i),part_centroid(1,i));
//  		}
//  	}

  docking_derive_coord_frame();
  docking_output_geometry( std::cout );
}

void
minimize_allow_insert_trial(
	std::string const & min_type,
	bool vary_phipsi,
	bool vary_chi,
	float min_tol
)
{
	using namespace design;
	using namespace misc;

	bool gfrag( true );
	if ( !vary_phipsi && !vary_chi ) {
		std::cout << "Warning: allow no bb and sc minimizing when do minimizing" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	score_set_try_rotamers(false);
	minimize_exclude_sstype(false, false);
	minimize_set_vary_chi(vary_chi);
	minimize_set_vary_phipsi(vary_phipsi);
	minimize_set_vary_omega(false);
	minimize_reset_exclude_list();
	monte_carlo_accept_best();
	monte_carlo_set_simannealing( true );
	score_set_new_pose();
	mc_global_track::mc_score::score = score12();
	monte_carlo_reset();
	minimize_set_tolerance(min_tol);
	minimize(min_type,"minimize",score12,1,total_residue,gfrag);
	monte_carlo_reset();
}
