// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 01:39:36 -0500 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "dipolar.h"
#include "constraints.h"
#include "dipolar_ns.h"
#include "dock_structure.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "runlevel.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
/// @begin get_dipolar_exist
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_dipolar_exist()
{
	return dipolar::dipolar_exist;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dipolar_set_verbose
///
/// @brief
///car use to turn out detailed output for next call to eval_dipolar
///car reset to false after every call to eval_dipolar
///
/// @detailed
///
/// @param  flag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dipolar_set_verbose( bool flag )
{
	dipolar_vflag::verbose = flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dipolar_get_verbose
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
dipolar_get_verbose()
{
	return dipolar_vflag::verbose;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dipolar_set_verbose_unit
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dipolar_set_verbose_unit( std::ostream & iunit )
{
	dipolar::dunit_p = &iunit;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fixed_dist_initializer
///
/// @brief
///
/// @detailed
///
/// @param  fixed_dist - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fixed_dist_initializer( FArray1D_float & fixed_dist )
{
	int i = 0;
	fixed_dist( ++i ) = 1.01;
	fixed_dist( ++i ) = 1.08;
	fixed_dist( ++i ) = 1.52325877;
	fixed_dist( ++i ) = 1.32874878;
	fixed_dist( ++i ) = 2.032764;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_dipolar
///
/// @brief
///car reads in dipolar constraint file and stores data in arrays
///
/// @detailed
///car file format:
///car line1: version     (not yet in use)
///car line2: parameters  (not yet in use)
///car line3: comment
///car line4: # of lines to read   (<= physical number of lines)
///car line5 -> beginning in 1st column
///car    tag      res1     atom1    res2    atom2      J
///car     a1  2x   i4   1x  a4   ix  a4  1x  4x    1x f10.2
///car
///car
///car if the first character of a line is '#', the line is read, but
///car    the data is ignored.
///car otherwise, the first column of each line contains a tag identifying
///car    the set to which the measurement belongs. (ie different experimental
///car    conditions with different alignment tensor) tags may be any
///car    single character other than '#'
///car
///car Residues and atoms are stored internally with res1 <= res2,
///car   regardless of input order, in addition, the list is sorted in
///car   increasing value of res1
///car Only five types of dipolar couplings are assumed to be possible
///car       1H-1H
///car       1H-15N
///car       1H-13C
///car      13C-15N
///car      13C-13C
///car If more are added, need to correct function get_invDcnst accordingly
///car Also, only bb atoms (including H) are included at present.
///****  Atom numbers start
///****   with the backbone atoms as follows
///*      1  2  3 4  5
///*      NH CA CB C CO .....
///car   0  HN         // all protons should be <= 0
///car  -1  HA  or HA1
///car  -2  HA2        // glycine
///
/// @param  filename - [in] - filename with dipolar constraints
/// @param  total_residue - [in] - length of protein
/// @param  res - [in] - sequence of protein
/// @param  exist - [out] - do rdc constraints exist
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car note: all coupling constants assumed to be collected under
///car the same conditions, such that the same order tensor can be applied
///car to all couplings
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_dipolar(
	std::string const & filename, // filename with dipolar constraints
	int const total_residue,
	FArray1DB_int & res_type,
	bool & exist
)
{
	using namespace align_tensor;
	using namespace dipolar;

//car internal
	std::string comment;
	int res1,res2;
	std::string atom1, atom2;
	char flag;
	FArray1D_char setmap( MAXDIPSETS, ' ' ); // maps characters to set numbers

//car data: fixed bond lengths in Angstroms
	static FArray1D_float const fixed_dist( 5, fixed_dist_initializer );
	 // fixed pair distances for different types

//car H-X bond lengths from get_hxyz
//car bb bond lengths from refold or template residue
//car Clore et al. (1998)JMR 133:216 uses
//car  HCA 1.08; HN 1.02;  N-C' 1.341; CA-C' 1.525

	utility::io::izstream iunit( filename );

	if ( !iunit ) {
		std::cout << "File: " << iunit.filename() << " not found" << std::endl;
		std::cout << "Dipolar constraints will not be used" << std::endl;
		Ndipolar = 0;
		Ndipolar_sets = 0;
		exist = false;
		iunit.close();
		iunit.clear();
		return;
	}

	exist = true;
	dipolar_exist = true;
	int nlines;
	std::cout << " dipolar constraints: " << filename << std::endl;
	iunit >> bite( comment ) >> skip;
	std::cout << space( 4 ) << comment << std::endl;
	iunit >> bite( comment ) >> skip;
	std::cout << space( 4 ) << comment << std::endl;
	iunit >> bite( comment ) >> skip;
	std::cout << space( 4 ) << comment << std::endl;
	iunit >> bite( 5, nlines ) >> skip;

	Ndipolar_sets = 1;
	Ndipolar = 1;
	for ( int i = 1; i <= nlines; ++i ) {
		iunit >> bite( flag ) >> skip( 2 ) >>
		 bite( 4, res1 ) >> skip( 1 ) >>
		 bite( 4, atom1 ) >> skip( 1 ) >>
		 bite( 4, res2 ) >> skip( 1 ) >>
		 bite( 4, atom2 ) >> skip( 1 ) >>
		 bite( 10, Jdipolar(Ndipolar) ) >> skip;
		if ( iunit.eof() ) goto L500;
		if ( flag != '#' ) {
			if (res1 > total_residue || res1 < 1 ) {
				std::cout << " Skipping constraint: invalid residue number: " << res1 << std::endl;
			} else if (res2 > total_residue || res2 < 1) {
				std::cout << " Skipping constraint: invalid residue number: " << res2 << std::endl;
			} else if ( res1 <= res2 ) {
				pairDipolar(1,Ndipolar) = res1;
				pairDipolar(3,Ndipolar) = res2;
				pairDipolar(2,Ndipolar) = classical_constraints::functions::type2index( atom1, res_type(res1) );
				pairDipolar(4,Ndipolar) = classical_constraints::functions::type2index( atom2, res_type(res2) );
			} else {
				pairDipolar(1,Ndipolar) = res2;
				pairDipolar(3,Ndipolar) = res1;
				pairDipolar(2,Ndipolar) = classical_constraints::functions::type2index( atom2, res_type(res2) );
				pairDipolar(4,Ndipolar) = classical_constraints::functions::type2index( atom1, res_type(res1) );
			}
			if ( Ndipolar == 1 ) setmap(1) = flag;
			for ( int j = 1; j <= Ndipolar_sets; ++j ) {
				if ( flag == setmap(j) ) {
					pairDipolar(5,Ndipolar) = j;
					goto L400;
				}
			}
			++Ndipolar_sets;
			setmap(Ndipolar_sets) = flag;
			pairDipolar(5,Ndipolar) = Ndipolar_sets;
L400:		// jump here if set exists
			++Ndipolar;
		}
//		std::cout << SS( i );
//		for ( int j = 1; j <= 5; ++j ) {
//			std::cout << SS( pairDipolar(j,i) );
//		} std::cout << SS( Jdipolar(i) ) << std::endl;
	}
L500:
	iunit.close();
	iunit.clear();
	--Ndipolar;

	if ( Ndipolar > MAXDIPOLAR ) {
		std::cout << "ERROR in read_dipolar:Ndipolar exceeds MAXDIPOLAR." << std::endl;
		std::cout << "Ndipolar = " << Ndipolar << " MAXDIPOLAR = " << MAXDIPOLAR <<
		 std::endl;
		std::cout << "Increase MAXDIPOLAR in dipolar.cc" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	sortdipolar(Ndipolar,pairDipolar,Jdipolar);

//car store invDcnst values & distances for bonded atoms
//car conventions:  if distance can't be precalculated, Ddist < 0
//car  and the value of invDmax must be calculated in eval_dipolar

	for ( int i = 1; i <= Ndipolar; ++i ) {
		invDcnst(i) = get_invDcnst(pairDipolar(1,i));
		int type = dipolartype(pairDipolar(1,i));
		if ( type == 0 ) {
			std::cout << "NOTICE: unrecognized dipolar type in dipolar " <<
			 "constraints file" << std::endl;
			std::cout << "residue 1: " << pairDipolar(1,i) << " atom 1: " <<
			 pairDipolar(2,i) << " residue 2: " << pairDipolar(3,i) << " atom 2: " <<
			 pairDipolar(4,i) << " data set: " << pairDipolar(4,i) << std::endl;
			std::cout << "This constraint will not be used" << std::endl;
		} else if ( type <= 5 ) {
			Ddist(i) = fixed_dist(type);
			invDmax(i) = invDcnst(i) * cube( fixed_dist(type) );
			Jdipolar(i) *= invDmax(i);
		} else {
			Ddist(i) = 0.0;
			invDmax(i) = invDcnst(i);
			Jdipolar(i) *= invDcnst(i);
		}
	}

//$$$	for ( int i = 1; i <= Ndipolar; ++i ) {
//$$$		std::cout << SS( i );
//$$$		for ( int j = 1; j <= 5; ++j ) {
//$$$			std::cout << SS( pairDipolar(j,i) );
//$$$		} std::cout << SS( Jdipolar(i)/invDmax(i) ) << std::endl;
//$$$	}

	std::cout << "number of dipolar constraints: " << Ndipolar << std::endl;
	std::cout << "number of dipolar data sets: " << Ndipolar_sets << std::endl;

//car initialize values of tensor and reject flags:

	x = 0.0;
	reject = true;

//mj compute projection angle restraints

	translate_rdc_to_projection();
	std::cout << "number of projection angle restrictions: " << Nprojection <<
	 std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dipolartype
///
/// @brief
///
/// @detailed
///
/// @param  pairDipolar - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
dipolartype( FArray1Da_int pairDipolar )
{
	pairDipolar.dimension( star );

	int dipolartype; // Return value

//car function differs for rosetta numbering scheme

//car internal
	int low,high;

	dipolartype = 0; // undefined
	if ( pairDipolar(1) == pairDipolar(3) ) {       // same residue
		low = std::min(pairDipolar(2),pairDipolar(4));
		high = std::max(pairDipolar(2),pairDipolar(4));
		if ( low == 0 && high == 1 ) dipolartype = 1; // HN-N
		if ( low == -1 && high == 2 ) dipolartype = 2; // HA-CA
		if ( low == 2 && high == 4 ) dipolartype = 3; // CA-C'
		if ( low == -1 && high == 0 ) dipolartype = 6; // HN-HA
	} else if ( pairDipolar(1) == pairDipolar(3)-1 ) { // adjacent residue
		low = pairDipolar(4);
		high = pairDipolar(2); // C' must be of i-1
		if ( low == 1 && high == 4 ) dipolartype = 4; // C'(i-1)-N(i)
		if ( low == 0 && high == 4 ) dipolartype = 5; // C'(i-1)-HN(i)
	}
	if ( dipolartype > 0 ) return dipolartype;

	std::cout << "undefined dipolar type" << std::endl;
	std::cout << "residue1 " << pairDipolar(1) << " atom1 " << pairDipolar(2) <<
	 " residue2 " << pairDipolar(3) << " atom2 " << pairDipolar(4) <<
	 " data set " << pairDipolar(5) << std::endl;

	return dipolartype;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_invDcnst
///
/// @brief
///
/// @detailed
///car Dmax = -gamma(n)*gamma(m)*h/(2*pi**2*r**3)         Losonczi et al.
///car      = -mu0*gamma(n)*gamma(m)*h/(16*pi**3*r**3)   Fischer et al.
///car where gamma is gyromagnetic ratio of nucleus
///car  gamma(1H)     2.6752e8 1/(Ts)   T = Ns/(Cm)
///car  gamma(15N)   -2.712e7 1/(Ts)
///car  gamma(13C)    6.728e7 1/(Ts)
///car  h             6.62608e-34 Js    J = Nm
///car  uo            4*pi*1e-7 Ns**2/C**2
///car
///car from Fischer eq, units come out in Hz (1/s)
///car Dmax = -(1e-7)(6.62608e-34)(gamma(m))(gamma(n)/[4(pi**2)(1e-30)(r**3)]
///car where 1e-30 converts r3 from cubic meters to cubic angstroms
///car invDmax = invDcnst*(r**3) with r in Angstroms
///car reduced dipolar coupling = Dobs*invDmax = Jdipolar*r**3*invDcnst
///car with r**3 in Angstroms
///
///car dipolar types           Dmax   These are values from ORDERTEN
///car         1H-1H        -240200.0        from Prestegard lab
///car         1H-15N         24350.0  // note paper has a factor of 2 error
///car         1H-13C        -60400.0    // Losonczi et al. JMR 1999 138:334
///car         15N-13C         6125.0    // eq 1, 2 in denom should be squared
///car         13C-13C       -15200.0
///
///car my calculations
///car HH -120118.4     difference 2x
///car HN   12177.1
///car HC  -30209.2
///car NC    3062.48
///car CC   -7597.47
///
///car and inverses:
///car  HH -0.0000083251
///car  HN   0.0000821215
///car  HC  -0.0000331025
///car  NC    0.000326533
///car  CC   -0.000131623
///
///car note:  all atoms number >2 are C
///car            atom 1 is N
///car            atoms <1 are H
///car same for rosetta and non-rosetta scheme
///
///
/// @param  pairDipolar - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
double
get_invDcnst( FArray1Da_int pairDipolar )
{
	pairDipolar.dimension( star );

	// atom numbers, assumed ordered // note assumptions in possible pairs
	int low  = std::min( pairDipolar(2), pairDipolar(4) );
	int high = std::max( pairDipolar(2), pairDipolar(4) );

	if ( high <= 0 ) {         // H-H
		return -0.0000083251;
	}
	if ( low <= 0 ) {          // H-X
		if ( high == 1 ) {         // HN
			return 0.0000821215;
		} else {                        // HC  (HX=HC if X!=N or H)
			return -0.0000331025;
		}
	} else if ( low == 1 ) {      // N-C  (NX=NC if X!=H)
		return 0.000326533;
	}
	return -0.000131623; // C-C  (X1-X2=CC if X1!=H or N)
}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_dipolar
///
/// @brief
/// calculates the total chi squared between predicted and calculated
/// dipolar coupling constants:
///
/// @detailed
///score = sum  [sum(Jdipolar_obs**2-Jdipolar_calc**2)/Ndata] *Ndata
///first sum over all data sets, second sum over all data within a set
///(scores for individual sets are normalized by # of constraints)
///(total score is { multiplied by total # of constraints)
///
///Input data include the first and last residues over which the score
///is to be calculated. These should correspond to the numbering
///scheme in the input dipolar constraints file. Note that dipolar
///couplings involving the HN of residue first are not included in the
///score because the orientation of the HN bond vector is not defined
///unless the phi of residue first is defined.
///
///Input xyz contains backbone coordinates for residues first
///through last. Coordinates are assumed to dimensioned for five atoms
///per residue.
///
///Input dipolar coupling data are obtained via the namespace in
///dipolar.h
///
///Only five types of dipolar couplings are assumed to be possible
///          1H-1H
///          1H-15N
///          1H-13C
///         13C-15N
///         13C-13C
///If more are added, need to adjust function Dmax accordingly
///
/// @param  xyz - [in] - coordinates
/// @param  first - [in] - first residue to eval score over
/// @param  last - [in] - last residue to eval score over
/// @param res_type - [in] - sequence of xyz
/// @param  score - [out] - score
///
/// @global_read
///
/// @global_write
///
/// @references
///Losonzi et al. (1999) JMR 138:334
///Fischer et al. (1999) Biochemistry 38:9013
///Clore et al., (1998) JMR 133:216
///Bothner-By in Encyclopedia of NMR (Grant & Harris ed) pp2932 Wiley 1995
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_dipolar(
	FArray3DB_float const & xyz, // coordinates for first to last residues only
	int  first,
	int last,
	FArray1DB_int & res_type,
	float & score
)
{
	using namespace align_tensor;
	using namespace dipolar;

	bool verbose;
	int nrow; // rows of matrix A = number of dipolars in frag
	FArray1D_int ndata( MAXDIPSETS ); // number of rows with real data
	double setscore;

	FArray2D_double A( MAXDIPOLAR, ORDERSIZE );
	 // coeff matrix for SVD; see  Losonzi et al
	FArray1D_double b( MAXDIPOLAR );
	 // reduced dipolar couplings (Jdipolar/Jmax)
	FArray1D_int map( MAXDIPOLAR );
	 // Jdipolar(map(i)) = coupling constant of row(i)
	// index to access complete arrays

	//(Syy,Szz,Sxy,Sxz,Syz)

	// rotates from mol. frame to PAF

//car ----------------function body ----------------------------------------------------------------

//car constraints assumed to be stored by read_dipolar such that
//car pairDipolar(1,i) <= pairDipolar(3,i)    and
//car pairDipolar(1,i) <= pairDipolar(1,i+1)

	verbose = dipolar_get_verbose();

	score = 0.0;

	if ( !dipolar_exist ) return;

	double score_sum = 0.0;

	if ( verbose ) dunit() << "residue, Jobs, Jcalc, residual**2" << std::endl;
	for ( int iset = 1; iset <= Ndipolar_sets; ++iset ) {
		if ( verbose ) dunit() << "dipolar constraint set " << iset << std::endl;
		assemble_datamatrix(xyz,first,last,iset,A,b,nrow,map);
		ndata(iset) = nrow;
		calc_ordermatrix(nrow,A,b,x(1,iset),reject(iset));
		if ( reject(iset) ) {
			score = 1.0e9;
			return;
		}
		calc_orderparam(x(1,iset),paf_rot(1,1,iset),Azz(iset),eta(iset));
		calc_dipscore(res_type,A,x(1,iset),b,map,ndata(iset),Azz(iset),setscore);
		if ( verbose ) {
			dunit() << "Azz" << SS( Azz(iset) ) << std::endl;
			dunit() << "eta" << SS( eta(iset) ) << std::endl;
			dunit() << 'x';
			for ( int i = 1; i <= 5; ++i ) {
				dunit() << SS( x(i,iset) );
			} dunit() << std::endl;
		}
		score_sum += setscore;
	}

	score_sum *= Ndipolar; //(last-first)+1
	score = score_sum;

	if ( verbose ) dunit() << "total dipolar score: " << SS( score ) << std::endl;
	dipolar_set_verbose(false);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_orderparam
///
/// @brief
///
/// @detailed
///
/// @param  x - [in/out]? -
/// @param  vec - [in/out]? - eigenvectors
/// @param  Azz - [in/out]? -
/// @param  eta - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_orderparam(
	FArray1Da_double x,
	FArray2Da_double vec, // eigenvectors
	double & Azz,
	double & eta
)
{
	using numeric::xyzMatrix_double;
	using numeric::xyzVector_double;

	x.dimension( star );
	vec.dimension( 3, 3 );

//car internal
	FArray1D_int sort( 3 ); // sorted index to val, val(sort(1)) = largest abs val.
	double temp1, temp2;


	// Assemble order matrix
	xyzMatrix_double const S( xyzMatrix_double::rows(
		-x(1)-x(2), x(3), x(4),
		      x(3), x(1), x(5),
		      x(4), x(5), x(2)
	) );

	xyzVector_double val; // Eigenvalues
	xyzMatrix_double xyz_vec; // Eigenvectors

	// Find eigenvalues, eigenvectors of symmetric matrix S
   val = eigenvector_jacobi( S, 1E-9, xyz_vec );

// sort eigenvalues
	sort(1) = 1;
	sort(2) = 2;
	sort(3) = 3;
	if ( std::abs(val(1)) < std::abs(val(2)) ) {     // largest absolute value
		sort(2) = 1;
		sort(1) = 2;
	}
	if ( std::abs(val(sort(2))) < std::abs(val(3)) ) {
		sort(3) = sort(2);
		sort(2) = 3;
		if ( std::abs(val(sort(1))) < std::abs(val(3)) ) {
			sort(2) = sort(1);
			sort(1) = 3;
		}
	}


	Azz = val(sort(1));
	eta = (2.0/3.0) * std::abs(val(sort(2))-val(sort(3))/Azz);

// sort eigen values      // largest to smallest : Azz,Ayy,Axx
	temp1 = val(sort(1));
	temp2 = val(sort(2));
	val(3) = val(sort(3));
	val(2) = temp2;
	val(1) = temp1;

// sort eigen vectors
	for ( int i = 1; i <= 3; ++i ) {
		temp1 = xyz_vec(i,sort(3));
		temp2 = xyz_vec(i,sort(2));
		vec(i,3) = xyz_vec(i,sort(1));
		vec(i,1) = temp1;
		vec(i,2) = temp2;
	}

	Azz = val(1);
	eta = (2.0/3.0) * (val(3)-val(2))/val(1);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_dipscore
///
/// @brief
///
/// @detailed
///
/// @param  res_type - [in] sequence being scored
/// @param  A - [in] -
/// @param  x - [in] -
/// @param  b - [in] -
/// @param  map - [in] -
/// @param  ndata - [in] -
/// @param  Azz - [out] - principal component of order tensor
/// @param  score - [out] - dipolar score
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_dipscore(
	FArray1DB_int & res_type,
	FArray2DB_double const & A,
	FArray1Da_double x,
	FArray1DB_double const & b,
	FArray1DB_int const & map,
	int ndata,
	double Azz,
	double & score
)
{
	using namespace dipolar;

	assert( ( ndata != 0 ) );
	assert( ( Azz != 0.0 ) );

	x.dimension( ORDERSIZE );

	bool const verbose = dipolar_get_verbose();

	score = 0.0;
	for ( int i = 1; i <= ndata; ++i ) {
		double Jcalc = 0.0;
		for ( int j = 1; j <= ORDERSIZE; ++j ) {
			Jcalc += A(i,j) * x(j);
		}
		score += ( b(i) - Jcalc ) * ( b(i) - Jcalc );
		if ( verbose ) dunit() << I( 3, pairDipolar(1,map(i)) ) << ' ' <<
		 F( 7, 2, b(i)/invDmax(map(i)) ) << ' ' <<
		 F( 7, 2, Jcalc/invDmax(map(i)) ) << ' ' <<
		 F( 7, 2, square( b(i) / invDmax(map(i)) - Jcalc / invDmax(map(i)) ) ) << ' ' <<
		 I( 3, pairDipolar(1,map(i)) ) << ' ' <<
		 fmt::A( 4, classical_constraints::functions::index2type(pairDipolar(2,map(i)), res_type(pairDipolar(1,map(i))))) << ' ' <<
		 I( 3, pairDipolar(3,map(i)) ) << ' ' <<
		 fmt::A( 4, classical_constraints::functions::index2type(pairDipolar(4,map(i)), res_type(pairDipolar(3,map(i))))) << std::endl;
	}
	score /= ( ndata * ( Azz * Azz ) ); // normalize by #data, principal order param.
}

////////////////////////////////////////////////////////////////////////////////
/// @begin assemble_datamatrix
///
/// @brief
///
/// @detailed
///
/// @param  xyz - [in/out]? -
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  iset - [in/out]? -
/// @param  A - [in/out]? -
/// @param  b - [in/out]? -
/// @param  nrow - [in/out]? -
/// @param  map - [in/out]? - Jdipolar(map(i)) = coupling constant of row(i)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
assemble_datamatrix(
	FArray3DB_float const & xyz,
	int first,
	int last,
	int iset,
	FArray2DB_double & A,
	FArray1DB_double & b,
	int & nrow,
	FArray1DB_int & map // Jdipolar(map(i)) = coupling constant of row(i)
)
{
	using namespace dipolar;

//car internal
	int i;
	float temp;
	FArray1D_float m( 3 );
	FArray1D_float n( 3 );
	FArray1D_float umn( 3 ); // unit vector between coupled atoms m,n
	float r; // distance between atoms m,n

	nrow = 0;
	for ( i = 1; i <= Ndipolar; ++i ) {
		if ( pairDipolar(1,i) > last ) return;
		if ( pairDipolar(5,i) != iset ) goto L200; // wrong set
		if ( pairDipolar(1,i) < first ) goto L200;
		if ( pairDipolar(3,i) < first || pairDipolar(3,i) > last ) goto L200;
		if ( ( pairDipolar(1,i) == first && pairDipolar(2,i) == 0 ) ||
		 ( pairDipolar(3,i) == first && pairDipolar(4,i) == 0 ) ) goto L200;
		 // HN of first not determined

		get_atomxyz(xyz,pairDipolar(1,i)-first+1,pairDipolar(2,i),m);
		if ( m(1) == 0.0 && m(2) == 0.0 && m(3) == 0.0 ) goto L200;
		get_atomxyz(xyz,pairDipolar(3,i)-first+1,pairDipolar(4,i),n);
		if ( n(1) == 0.0 && n(2) == 0.0 && n(3) == 0.0 ) goto L200;

		++nrow; // acceptable constraint
		map(nrow) = i;


//car calculate direction cosines:
//car molecular coordinate system is 1 0 0
//car                                0 1 0
//car                                0 0 1
//car theta = angle between bond vector and molecular axis.
//car cos(thetax) = dotprod(unm,ux)  ;unm = unitvec along bond, ux = unitvec along x
//car cos(thetax) = umn(x)     i.e. x component of unm

		umn(1) = m(1) - n(1); // get vector mn
		umn(2) = m(2) - n(2);
		umn(3) = m(3) - n(3);
		if ( dipolartype(pairDipolar(1,i)) <= 5 ) {
			r = Ddist(i);
			b(nrow) = Jdipolar(i); // fixed distance
		} else {                   // get dist, calc invDmax
			temp = umn(1)*umn(1) + umn(2)*umn(2) + umn(3)*umn(3);
			r = std::sqrt(temp);
			invDmax(i) = invDcnst(i)*temp*r; // don't need this?-diagnostics
			b(nrow) = Jdipolar(i)*temp*r; // variable distance
		}

		temp = 1.0f/r;
		umn(1) *= temp; // normalize (= cosines)
		umn(2) *= temp;
		umn(3) *= temp;

		temp = umn(1)*umn(1); // assemble A
		A(nrow,1) = umn(2)*umn(2)-temp;
		A(nrow,2) = umn(3)*umn(3)-temp;
		A(nrow,3) = 2.0*umn(1)*umn(2);
		A(nrow,4) = 2.0*umn(1)*umn(3);
		A(nrow,5) = 2.0*umn(2)*umn(3);
L200:;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_atomxyz
///
/// @brief
///
/// @detailed
///
/// @param  xyz - [in/out]? - coordinates
/// @param  res - [in/out]? - residue (ie third index of xyz)
/// @param  atom - [in/out]? -
/// @param  atomxyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_atomxyz(
	FArray3DB_float const & xyz, // coordinates
	int res, // residue (ie third index of xyz)
	int atom,
	FArray1Da_float atomxyz
)
{
	atomxyz.dimension( 3 );

//car differs for rosetta scheme
//car returns coordinates for a specified atom, either copied from the
//car position array, or for protons, calculated from the backbone
//car coordinates using a template coordinate


//****  Atom numbers start
//****   with the backbone atoms as follows
//*      1  2  3 4  5
//*      NH CA CB C CO  .....

//car   0  HN     // all protons should be <= 0
//car  -1  HA


	if ( atom >= 1 ) {       // backbone atom
		atomxyz(1) = xyz(1,atom,res);
		atomxyz(2) = xyz(2,atom,res);
		atomxyz(3) = xyz(3,atom,res);
		return;
	} else if ( atom == 0 ) {
		if ( res == 1 ) {
			std::cout << "WARNING: Error in eval_dipolar!" << std::endl;
			std::cout << " Attempt to calculate coupling constant for " <<
			 "HN of N-terminal residue." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		compute_hxyz(atom,xyz(1,4,res-1),xyz(1,1,res),xyz(1,2,res),atomxyz);
		 // C' is atom 4 in rosetta scheme
		return;
	} else if ( atom == -1 || atom == -2 ) {     // HA,HA1,HA2
		compute_hxyz(atom,xyz(1,1,res),xyz(1,2,res),xyz(1,4,res),atomxyz);
		 // C' is atom 4 in rosetta scheme
		return;
	}

	std::cout << "WARNING: unrecognized atom type in dipolar restraints" << std::endl;
	std::cout << "atom number " << atom << " is not defined in get_atomxyz" << std::endl;
}



void
Hpos_initializer( FArray2D_double & Hpos )
{
	FArray1Da_double Hpos1d( Hpos );
	int i = 0;
	Hpos1d( ++i ) = -0.230853993; // -2=HA2
	Hpos1d( ++i ) = 0.488168418;
	Hpos1d( ++i ) = 0.935306426;
	Hpos1d( ++i ) = -0.230853993; // -1=HA or HA1
	Hpos1d( ++i ) = 0.488168418;
	Hpos1d( ++i ) = -0.935306426;
	Hpos1d( ++i ) = -0.477136907; // 0=HN
	Hpos1d( ++i ) = 0.890191189;
	Hpos1d( ++i ) = 0.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_hxyz
///
/// @brief
///
/// @detailed
///
/// @param  type - [in/out]? - proton type 0=HN,-1=HA,-2=glyHA2
/// @param  atom1 - [in/out]? - coordinates of atom N-term to proton-bound atom
/// @param  atom2 - [in/out]? - coordinates of proton-bound atom
/// @param  atom3 - [in/out]? - coordinates of atom C-term to proton-bound atom
/// @param  hxyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compute_hxyz(
	int type, // proton type 0=HN,-1=HA,-2=glyHA2
	FArray1Da_float atom1, // coordinates of atom N-term to proton-bound atom
	FArray1Da_float atom2, // coordinates of proton-bound atom
	FArray1Da_float atom3, // coordinates of atom C-term to proton-bound atom
	FArray1Da_float hxyz
)
{
	atom1.dimension( 3 );
	atom2.dimension( 3 );
	atom3.dimension( 3 );
	hxyz.dimension( 3 );

//car differs for rosetta conventions cross->cros

//car returns hxyz, the coordinates of a proton of specified type.
//car types:
//car   0  HN         // all protons should be <= 0
//car  -1  HA  or HA1
//car  -2  HA2        // glycine
//car The coordinates of three atoms are required to place the proton
//car atom2=atom to which proton is bound
//car atom1=N-terminally adjacent to atom2, (or closer to backbone)
//car atom3-C-terminally adjacent to atom2, (or further from backbone)

//car internal
	FArray2D_float axes( 3, 3 );

//car parameters
	static FArray2D_double const Hpos( 3, DRange( -2, 0 ), Hpos_initializer );
	 // xyz measured from atom2 using coordinate sys where
	// atom2->atom1 is colinear w/ x axis
	// atom3 lies in xy plane and
	// atom3->atom2 has a +y component

//car  -0.481861031,  0.899004954.  0.0 // HN when bondlength = 1.02 clore
//car  -0.477136907,  0.890191189,  0.0 // HN when bondlength = 1.01 cems
//car  -0.472412784,  0.881377423,  0.0 // HN when bondlength = 1.00 bk
//car  -0.491966549,  0.882082136,  0.0 // 1.01A from N, bisecting the CNCA angle
	// w/refold template, 1.01 A

	for ( int i = 1; i <= 3; ++i ) {
		axes(i,1) = atom3(i)-atom2(i);
		axes(i,2) = atom2(i)-atom1(i);
	}
	cros(axes(1,1),axes(1,2),axes(1,3)); // generate rotation matrix
	cros(axes(1,3),axes(1,1),axes(1,2)); // rosetta function cros
	unitvec(axes(1,1),axes(1,1));
	unitvec(axes(1,3),axes(1,3));
	unitvec(axes(1,2),axes(1,2));

	for ( int i = 1; i <= 3; ++i ) { // template coord x rotation matrix + base coord
		hxyz(i) = atom2(i) + Hpos(1,type) * axes(i,1) + Hpos(2,type) * axes(i,2) +
		 Hpos(3,type) * axes(i,3);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_ordermatrix
///
/// @brief
///
/// @detailed
///
/// @param  nrow - [in/out]? -
/// @param  A - [in/out]? -
/// @param  b - [in/out]? -
/// @param  x - [in/out]? -
/// @param  reject - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_ordermatrix(
	int & nrow,
	FArray2DB_double const & A,
	FArray1DB_double const & b,
	FArray1Da_double x,
	bool & reject
)
{
	using namespace dipolar;

	x.dimension( ORDERSIZE );

// returns the order matrix x and if invalid, sets reject true
// nrow gets increased if the matrix needs to be padded
// no other values are altered

//car parameters
	double const factor = { 1e-6 }; // cutoff factor for singular values in svd

//car internal
	FArray2D_double U( MAXDIPOLAR, ORDERSIZE );
	FArray1D_double w( ORDERSIZE ); // singular values
	FArray2D_double v( ORDERSIZE, ORDERSIZE );
	double wmin, wmax; // min and max values of w
	int sing; // number of singular values in w
	double Sxx;
	int ndata;

	ndata = nrow;
	for ( int i = 1; i <= nrow; ++i ) { // copy A
		U(i,1) = A(i,1); // copy into U
		U(i,2) = A(i,2);
		U(i,3) = A(i,3);
		U(i,4) = A(i,4);
		U(i,5) = A(i,5);
	}

	while ( nrow < ORDERSIZE ) { // not enoughs rows, fill to square w/zeros
		++nrow;
		U(nrow,1) = 0.0;
		U(nrow,2) = 0.0;
		U(nrow,3) = 0.0;
		U(nrow,4) = 0.0;
		U(nrow,5) = 0.0;
	}
	svdcmp(U,nrow,ORDERSIZE,w,v);

//car check w for singular values prior to inversion
	wmax = 0.0;
	for ( int j = 1; j <= ORDERSIZE; ++j ) {
		if ( w(j) > wmax ) wmax = w(j);
	}
	wmin = wmax * factor;
	sing = 0;
	for ( int j = 1; j <= ORDERSIZE; ++j ) {
		if ( w(j) < wmin ) {
			w(j) = 0.0;
			++sing;
		}
	}
	if ( sing > std::abs(ndata-ORDERSIZE) )
	 std::cout << "SVD yielded a matrix singular above expectation " <<
	 "in get_ordermatrix" << std::endl;

//car find solution for exact dipolar values:
	svbksb(U,w,v,nrow,ORDERSIZE,b,x);

//car x components: (Syy,Szz,Sxy,Sxz,Syz)
//car check for acceptable values

	reject = false;
	Sxx = -x(1) - x(2);
	if ( Sxx < -0.5 || Sxx > 1.0 ) reject = true; // Sxx
	if ( x(1) < -0.5 || x(1) > 1.0 ) reject = true; // Syy
	if ( x(2) < -0.5 || x(2) > 1.0 ) reject = true; // Szz
	if ( x(3) < -0.75 || x(3) > 0.75 ) reject = true; // Sxy
	if ( x(4) < -0.75 || x(4) > 0.75 ) reject = true; // Sxz
	if ( x(5) < -0.75 || x(5) > 0.75 ) reject = true; // Syz

	if ( reject ) {
		std::cout << "order matrix not physically meaningful" << std::endl;
//		try with errors on dipolar values? map error?
//		score = 0.0;
		return;
	}

}


//------------------------------------------------------------------------------
//  functions from numerical recipes

////////////////////////////////////////////////////////////////////////////////
/// @begin svdcmp
///
/// @brief
///
/// @detailed
///car Given a matrix A, with logical dimensions MxN and physical dimensions
///car MPxNP, this function computes its singular value decomposition, A=U.W.Vt
///car The matrix U replaces A on output.  The diagonal matrix of singular
///car values (W) is output as a vector W, The matrix V (not the transpose Vt)
///car is output as V.  M must be greater than or equal to N.  If it is smaller,
///car then A should be filled up to square with zero rows.
///
/// @param  a - [in/out]? -
/// @param  m - [in/out]? -
/// @param  n - [in/out]? -
/// @param  mp - [in/out]? -
/// @param  np - [in/out]? -
/// @param  w - [in/out]? -
/// @param  v - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
svdcmp(
	FArray2DB_double & a,
	int m,
	int n,
	FArray1DB_double & w,
	FArray2DB_double & v
)
{

//U    USES pythag
	int i,its,j,jj,k,l,nm;
	FArray1D_double rv1( n );
	double anorm, c, f, g, h, s, scale, x, y, z;
	g = 0.0;
	scale = 0.0;
	anorm = 0.0;
	l = 0;
	nm = 0;
	for ( i = 1; i <= n; ++i ) {
		l = i+1;
		rv1(i) = scale*g;
		g = 0.0;
		s = 0.0;
		scale = 0.0;
		if ( i <= m ) {
			for ( k = i; k <= m; ++k ) {
				scale += std::abs(a(k,i));
			}
			if ( scale != 0.0 ) {
				for ( k = i; k <= m; ++k ) {
					a(k,i) /= scale;
					s += a(k,i)*a(k,i);
				}
				f = a(i,i);
				g = -sign(std::sqrt(s),f);
				h = f*g-s;
				a(i,i) = f-g;
				for ( j = l; j <= n; ++j ) {
					s = 0.0;
					for ( k = i; k <= m; ++k ) {
						s += a(k,i)*a(k,j);
					}
					f = s/h;
					for ( k = i; k <= m; ++k ) {
						a(k,j) += f*a(k,i);
					}
				}
				for ( k = i; k <= m; ++k ) {
					a(k,i) *= scale;
				}
			}
		}
		w(i) = scale *g;
		g = 0.0;
		s = 0.0;
		scale = 0.0;
		if ( (i <= m) && (i != n) ) {
			for ( k = l; k <= n; ++k ) {
				scale += std::abs(a(i,k));
			}
			if ( scale != 0.0 ) {
				for ( k = l; k <= n; ++k ) {
					a(i,k) /= scale;
					s += a(i,k)*a(i,k);
				}
				f = a(i,l);
				g = -sign(std::sqrt(s),f);
				h = f*g-s;
				a(i,l) = f-g;
				for ( k = l; k <= n; ++k ) {
					rv1(k) = a(i,k)/h;
				}
				for ( j = l; j <= m; ++j ) {
					s = 0.0;
					for ( k = l; k <= n; ++k ) {
						s += a(j,k)*a(i,k);
					}
					for ( k = l; k <= n; ++k ) {
						a(j,k) += s*rv1(k);
					}
				}
				for ( k = l; k <= n; ++k ) {
					a(i,k) *= scale;
				}
			}
		}
		anorm = std::max(anorm,(std::abs(w(i))+std::abs(rv1(i))));
	}
	for ( i = n; i >= 1; --i ) {
		if ( i < n ) {
			if ( g != 0.0 ) {
				for ( j = l; j <= n; ++j ) {
					v(j,i) = (a(i,j)/a(i,l))/g;
				}
				for ( j = l; j <= n; ++j ) {
					s = 0.0;
					for ( k = l; k <= n; ++k ) {
						s += a(i,k)*v(k,j);
					}
					for ( k = l; k <= n; ++k ) {
						v(k,j) += s*v(k,i);
					}
				}
			}
			for ( j = l; j <= n; ++j ) {
				v(i,j) = 0.0;
				v(j,i) = 0.0;
			}
		}
		v(i,i) = 1.0;
		g = rv1(i);
		l = i;
	}
	for ( i = std::min(m,n); i >= 1; --i ) {
		l = i+1;
		g = w(i);
		for ( j = l; j <= n; ++j ) {
			a(i,j) = 0.0;
		}
		if ( g != 0.0 ) {
			g = 1.0/g;
			for ( j = l; j <= n; ++j ) {
				s = 0.0;
				for ( k = l; k <= m; ++k ) {
					s += a(k,i)*a(k,j);
				}
				f = (s/a(i,i))*g;
				for ( k = i; k <= m; ++k ) {
					a(k,j) += f*a(k,i);
				}
			}
			for ( j = i; j <= m; ++j ) {
				a(j,i) *= g;
			}
		} else {
			for ( j = i; j <= m; ++j ) {
				a(j,i) = 0.0;
			}
		}
		a(i,i) += 1.0;
	}
	for ( k = n; k >= 1; --k ) {
		for ( its = 1; its <= 30; ++its ) {
			for ( l = k; l >= 1; --l ) {
				nm = l-1;
				if ( (std::abs(rv1(l))+anorm) == anorm ) goto L2;
				if ( (std::abs(w(nm))+anorm) == anorm ) goto L1;
			}
L1:
			c = 0.0;
			s = 1.0;
			for ( i = l; i <= k; ++i ) {
				f = s*rv1(i);
				rv1(i) *= c;
				if ( (std::abs(f)+anorm) == anorm ) goto L2;
				g = w(i);
				h = pythag(f,g);
				w(i) = h;
				h = 1.0/h;
				c = (g*h);
				s = -(f*h);
				for ( j = 1; j <= m; ++j ) {
					y = a(j,nm);
					z = a(j,i);
					a(j,nm) = (y*c)+(z*s);
					a(j,i) = -(y*s)+(z*c);
				}
			}
L2:
			z = w(k);
			if ( l == k ) {
				if ( z < 0.0 ) {
					w(k) = -z;
					for ( j = 1; j <= n; ++j ) {
						v(j,k) = -v(j,k);
					}
				}
				goto L3;
			}
			if ( its == 30) error_stop( "no convergence in svdcmp" );
			x = w(l);
			nm = k-1;
			y = w(nm);
			g = rv1(nm);
			h = rv1(k);
			f = ((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y);
			g = pythag(f,1.0);
			f = ((x-z)*(x+z)+h*((y/(f+sign(g,f)))-h))/x;
			c = 1.0;
			s = 1.0;
			for ( j = l; j <= nm; ++j ) {
				i = j+1;
				g = rv1(i);
				y = w(i);
				h = s*g;
				g *= c;
				z = pythag(f,h);
				rv1(j) = z;
				c = f/z;
				s = h/z;
				f = (x*c)+(g*s);
				g = -(x*s)+(g*c);
				h = y*s;
				y *= c;
				for ( jj = 1; jj <= n; ++jj ) {
					x = v(jj,j);
					z = v(jj,i);
					v(jj,j) = (x*c)+(z*s);
					v(jj,i) = -(x*s)+(z*c);
				}
				z = pythag(f,h);
				w(j) = z;
				if ( z != 0.0 ) {
					z = 1.0/z;
					c = f*z;
					s = h*z;
				}
				f = (c*g)+(s*y);
				x = -(s*g)+(c*y);
				for ( jj = 1; jj <= m; ++jj ) {
					y = a(jj,j);
					z = a(jj,i);
					a(jj,j) = (y*c)+(z*s);
					a(jj,i) = -(y*s)+(z*c);
				}
			}
			rv1(l) = 0.0;
			rv1(k) = f;
			w(k) = x;
		}
L3:;
	}
}
//------------------------------------------------------------------------------

double
pythag(
	double a,
	double b
)
{
	double pythag; // Return value

	double absa = std::abs(a);
	double absb = std::abs(b);
	if ( absa > absb ) {
		double const ratio = absb/absa;
		pythag = absa * std::sqrt( 1.0 + ( ratio * ratio ) );
	} else {
		if ( absb == 0.0 ) {
			pythag = 0.0;
		} else {
			double const ratio = absa/absb;
			pythag = absb * std::sqrt( 1.0 + ( ratio * ratio ) );
		}
	}
	return pythag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin svbksb
///
/// @brief
///
/// @detailed
///car solves A.x=b for a vector X where A is specified by the arrays U,W,V
///car as returned by SVDCMP. M and N are the logical dimensions of A. MP
///car and NP are the physical dimensions of A.  B is the input right-hand
///car side.  X is the output solution vector.  No input quantities are
///car destroyed.
///
/// @param  u - [in/out]? -
/// @param  w - [in/out]? -
/// @param  v - [in/out]? -
/// @param  m - [in/out]? -
/// @param  n - [in/out]? -
/// @param  mp - [in/out]? -
/// @param  np - [in/out]? -
/// @param  b - [in/out]? -
/// @param  x - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
svbksb(
	FArray2DB_double const & u,
	FArray1DB_double const & w,
	FArray2DB_double const & v,
	int m,
	int n,
	FArray1DB_double const & b,
	FArray1DB_double & x
)
{


	FArray1D_double tmp( n );
	double s;
	for ( int j = 1; j <= n; ++j ) {
		s = 0.0;
		if ( w(j) != 0.0 ) {
			for ( int i = 1; i <= m; ++i ) {
				s += u(i,j) * b(i);
			}
			s /= w(j);
		}
		tmp(j) = s;
	}
	for ( int j = 1; j <= n; ++j ) {
		s = 0.0;
		for ( int jj = 1; jj <= n; ++jj ) {
			s += v(j,jj) * tmp(jj);
		}
		x(j) = s;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin sortdipolar
///
/// @brief
///car heapsort algorithm from numerical recipes
///car modified here from hpsort to use multidimensional ra and also sort rb
///
/// @detailed
///
/// @param  n - [in/out]? -
/// @param  ra - [in/out]? -
/// @param  rb - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
sortdipolar(
	int n,
	FArray2DB_int & ra,
	FArray1DB_double & rb
)
{

	int const size = { 5 }; // first dim of ra

	int const key = { 1 }; // key to sort on

	int i,ir,j,l,k;
	FArray1D_int rra( size );
	double rrb;

	if ( n < 2 ) return;
	l = n/2+1;
	ir = n;
L10:
	if ( l > 1 ) {
		--l;
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,l);
		}
		rrb = rb(l);
	} else {
		for ( k = 1; k <= size; ++k ) {
			rra(k) = ra(k,ir);
		}
		rrb = rb(ir);
		for ( k = 1; k <= size; ++k ) {
			ra(k,ir) = ra(k,1);
		}
		rb(ir) = rb(1);
		--ir;
		if ( ir == 1 ) {
			for ( k = 1; k <= size; ++k ) {
				ra(k,1) = rra(k);
			}
			rb(1) = rrb;
			return;
		}
	}
	i = l;
	j = l+l;
L20:
	if ( j <= ir ) {
		if ( j < ir ) {
			if ( ra(key,j) < ra(key,j+1) ) ++j;
		}
		if ( rra(key) < ra(key,j) ) {
			for ( k = 1; k <= size; ++k ) {
				ra(k,i) = ra(k,j);
			}
			rb(i) = rb(j);
			i = j;
			j += j;
		} else {
			j = ir+1;
		}
		goto L20;
	}
	for ( k = 1; k <= size; ++k ) {
		ra(k,i) = rra(k);
	}
	rb(i) = rrb;
	goto L10;
}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//
// TRANSLATION OF DIPOLAR COUPLINGS INTO PROJECTION ANGLE RESTRAINT
//
// Jens Meiler (mj) 03/14/2002 {J.Biom.NMR 16, 245-252, 2000}
//
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//     scoring of projection
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_projection
///
/// @brief
///
/// @detailed
///
/// @param  proj_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_projection(
	float & proj_score,
	FArray3Dp_float const & Eposition
)
{
	using namespace dipolar;
	using namespace numeric::constants::f;

//mj return if no data
	proj_score = 0.0;

	if ( !dipolar_exist ) return;

//mj loop over all projection angles

	FArray2D_float atom_coord( 3, 4 );
	float alpha, alpha_diff;
	float const four_over_pi_sq = 4.0f / ( pi * pi );

	for ( int i = 1; i <= Nprojection; ++i ) {

//mj get coordinates of the four atoms

		get_atomxyz(Eposition,pairProjection(1,i),pairProjection(2,i),
		 atom_coord(1,1));
		get_atomxyz(Eposition,pairProjection(3,i),pairProjection(4,i),
		 atom_coord(1,2));
		get_atomxyz(Eposition,pairProjection(5,i),pairProjection(6,i),
		 atom_coord(1,3));
		get_atomxyz(Eposition,pairProjection(7,i),pairProjection(8,i),
		 atom_coord(1,4));

//mj compute projection angle

		angle_bk(atom_coord(1,1),atom_coord(1,2),atom_coord(1,3),atom_coord(1,4),
		 alpha);

//mj compute alpha_diff

		if ( alpha < anglProjection(1,i) ) {
			alpha_diff = anglProjection(1,i)-alpha;
		} else if ( alpha > anglProjection(4,i) ) {
			alpha_diff = alpha-anglProjection(4,i);
		} else if ( alpha < anglProjection(3,i) && alpha > anglProjection(2,i) ) {
			alpha_diff = std::min(anglProjection(3,i)-alpha,alpha-anglProjection(2,i));
		} else {
			alpha_diff = 0.0;
		}

//mj increase score by normalized deviation

		proj_score += ( alpha_diff * alpha_diff ) * four_over_pi_sq;
	}

//mj normalize score to be proportional to the number of used dipolar couplings

}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//     translation of dipolar couplings into projection angles
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin translate_rdc_to_projection
///
/// @brief
///     translation of dipolar couplings into projection angles
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
translate_rdc_to_projection()
{
	using namespace dipolar;
	using namespace runlevel_ns;

//mj local variables

	FArray1D< FArray1D_double > eigenvalues( Ndipolar_sets );

//mj loop over all alignments to determine projection array size

	int count = 0;
	for ( int i = 1; i <= Ndipolar_sets; ++i ) {

		FArray1D_double & evals( eigenvalues( i ) ); // Shorthand reference
		evals.dimension( 3 );

//mj derive tensor eigenvalues from distribution plot

		get_eigenvalues( evals, i );

//mj if eigenvalues were successfully determined

		if ( evals( 2 ) != 0.0 ) {

//mj loop over all pairs of dipolar couplings

			for ( int j = 1; j < Ndipolar; ++j ) {

//mj consider only if defined distance and correct set

				if ( pairDipolar(5,j) == i && dipolartype(pairDipolar(1,j)) <= 5 ) {

					for ( int k = j + 1; k <= Ndipolar; ++k ) {

//mj consider only if defined distance and correct set

						if ( pairDipolar(5,k) == i && dipolartype(pairDipolar(1,k)) <= 5 ) {

//mj increase count to point on next set of projection angle restriction

							++count;
						}
					}
				}
			}
		}
	}

//mj set total number of projection angles

	Nprojection = count; // Sets projection array sizes: Allocates arrays

	// Now fill projection arrays
	count = 0;
	for ( int i = 1; i <= Ndipolar_sets; ++i ) {

		FArray1D_double & evals( eigenvalues( i ) ); // Shorthand reference

//mj if eigenvalues were successfully determined

		if ( evals( 2 ) != 0.0 ) {

			double const xax = ( 2.0 * evals(3) - evals(2) - evals(1) ) / 6.0;
			double const rho = 2.0 * ( evals(1) - evals(2) ) /
			 ( 2.0 * evals(3) - evals(2) - evals(1) );
			if ( runlevel > standard ) {
				std::cout << "evals set" << SS( i ) << " << " <<
				 SS( evals(1) ) << SS( evals(2) ) << SS( evals(3) ) << std::endl;
				std::cout << "xax:" << SS( xax ) << "rho:" << SS( rho ) << std::endl;
			}

//mj loop over all pairs of dipolar couplings

			for ( int j = 1; j < Ndipolar; ++j ) {

//mj consider only if defined distance and correct set

				if ( pairDipolar(5,j) == i && dipolartype(pairDipolar(1,j)) <= 5 ) {

					for ( int k = j + 1; k <= Ndipolar; ++k ) {

//mj consider only if defined distance and correct set

						if ( pairDipolar(5,k) == i && dipolartype(pairDipolar(1,k)) <= 5 ) {

//mj increase count to point on next set of projection angle restriction

							++count;

//mj compute projection angle restrictions

							get_proj_ang_restriction(xax,rho,Jdipolar(j),Jdipolar(k),
							 anglProjection(1,count));

							if ( runlevel > gush ) std::cout << count <<
							 anglProjection(1,count) << anglProjection(2,count) <<
							 anglProjection(3,count) << anglProjection(4,count) << std::endl;

//mj copy atom data
							for ( int l = 1; l <= 4; ++l ) {
								pairProjection(l  ,count) = pairDipolar(l,j);
								pairProjection(l+4,count) = pairDipolar(l,k);
							}
						}
					}
				}
			}
		}
	}

}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//     derive eigenvalues from dipolar coupling histogram
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin get_eigenvalues
///
/// @brief
///     derive eigenvalues from dipolar coupling histogram
///
/// @detailed
///
/// @param  eigenvalues - [in/out]? -
/// @param  set - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_eigenvalues(
	FArray1DB_double & eigenvalues,
	int const set
)
{
//mj local variables

	int barcount;
	FArray1D_double histogram( 25 );
	double minimum, maximum, trace;

//mj get coupling distribution, maximum and minimum

	get_coupling_distribution(barcount,minimum,maximum,histogram,set);

//mj return if no dipolar couplings were used

	if ( barcount == 0 ) return;
	eigenvalues = 0.0;

//mj find highest peak in histogram

	int best = 1;
	eigenvalues(3) = maximum;
	eigenvalues(2) = minimum;
	for ( int i = 2; i <= barcount; ++i ) {
		if ( histogram(i) > histogram(best) ) best = i;
	}
	eigenvalues(1) = maximum*(2*best+1)/barcount-maximum;
	trace = ( eigenvalues(1) + eigenvalues(2) + eigenvalues(3) ) / 3;
	for ( int i = 1; i <= 3; ++i ) {
		eigenvalues(i) -= trace;
	}

//mj make sure that largest absolute value is in eigenvalues(3)

	if ( std::abs(eigenvalues(3)) < std::abs(eigenvalues(2)) ) {
		trace          = eigenvalues(2);
		eigenvalues(2) = eigenvalues(3);
		eigenvalues(3) = trace;
	}

}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//     computes coupling distribution and writes it into histogram
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin get_coupling_distribution
///
/// @brief
///     computes coupling distribution and writes it into histogram
///
/// @detailed
///
/// @param  barcount - [in/out]? -
/// @param  minimum - [in/out]? -
/// @param  maximum - [in/out]? -
/// @param  histogram - [in/out]? -
/// @param  set - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_coupling_distribution(
	int & barcount,
	double & minimum,
	double & maximum,
	FArray1DB_double & histogram,
	int const set
)
{
	using namespace dipolar;

//mj local variables

	int i,count,pos;
	double abs_maximum;

//mj get coupling distribution, maximum and minimum

	count = 0;
	minimum = 0;
	maximum = 0;
	barcount = 0;
	histogram = 0.0;

//mj loop over all dipolar couplings => minimum,maximum

	for ( i = 1; i <= Ndipolar; ++i ) {

//mj consider only if defined distance and correct set

		if ( pairDipolar(5,i) == set && dipolartype(pairDipolar(1,i)) <= 5 ) {

//mj update minimum, maximum and count
			if ( count == 0 || Jdipolar(i) < minimum ) minimum = Jdipolar(i);
			if ( count == 0 || Jdipolar(i) > maximum ) maximum = Jdipolar(i);
			++count;
		}
	}

//mj check whether data exist, else return

	if ( count == 0 ) return; // no data

//mj define number of bars

	abs_maximum = std::max(std::abs(minimum),std::abs(maximum));
	barcount = count/10;
	if ( barcount < 3 ) barcount = 3;
	if ( barcount > 12 ) barcount = 12;
	barcount = 2*barcount+1;

//mj loop over all dipolar couplings => histogram

	for ( i = 1; i <= Ndipolar; ++i ) {

//mj consider only if defined distance and correct set

		if ( pairDipolar(5,i) == set && dipolartype(pairDipolar(1,i)) <= 5 ) {

//mj update histogram

			pos = static_cast< int >(0.5*barcount*(Jdipolar(i)/abs_maximum+1.0))+1;
			pos = std::max(1,std::min(25,pos));
			histogram(pos) += 1.0;
		}
	}

}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//     derive eigenvalues from dipolar coupling histogram
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin get_proj_ang_restriction
///
/// @brief
///     derive eigenvalues from dipolar coupling histogram
///
/// @detailed
///
/// @param  xax - [in/out]? -
/// @param  rho - [in/out]? -
/// @param  Jdipolar_A - [in/out]? -
/// @param  Jdipolar_B - [in/out]? -
/// @param  angle_restriction - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_proj_ang_restriction(
	double const xax,
	double const rho,
	double const Jdipolar_A,
	double const Jdipolar_B,
	FArray1Da_float angle_restriction
)
{
	using namespace numeric::constants::f;
	using numeric::sin_cos_range;

	angle_restriction.dimension( 4 );

//mj local variables

	int i1, i2;
	bool reverse;
	float phi1, phi2, f1, f2, alphaplus, alphaminus;
	float phi1min, phi2min, phi1max, phi2max;
	float coupling1, coupling2, temp1, temp2;

//mj set start values

	reverse = false;
	phi1min = 0;
	phi2min = 0;
	phi1max = pi;
	phi2max = pi;

	angle_restriction(1) = pi;
	angle_restriction(2) = 0.0;
	angle_restriction(3) = pi;
	angle_restriction(4) = 0.0;

//mj ensure that couplings are within the theoretical possible range

	if ( xax > 0 ) {
		coupling1 = std::max(xax*(-1.0-rho*3.0/2.0),std::min(2.0*xax,Jdipolar_A));
		coupling2 = std::max(xax*(-1.0-rho*3.0/2.0),std::min(2.0*xax,Jdipolar_B));
	} else {
		coupling1 = std::min(xax*(-1.0-rho*3.0/2.0),std::max(2.0*xax,Jdipolar_A));
		coupling2 = std::min(xax*(-1.0-rho*3.0/2.0),std::max(2.0*xax,Jdipolar_B));
	}

//mj find limits for phi1 and phi2
//mj polar coordinates (theta,phi) of vectors between coupled nuclei

	temp1 = (coupling1/xax+1.0)*2.0/3.0/rho;
	if ( std::abs(temp1) <= 1 ) phi1min =    0.5*std::acos(sin_cos_range(temp1));
	if ( std::abs(temp1) <= 1 ) phi1max = pi-0.5*std::acos(sin_cos_range(temp1));
	temp2 = (coupling2/xax+1.0)*2.0/3.0/rho;
	if ( std::abs(temp2) <= 1 ) phi2min =    0.5*std::acos(sin_cos_range(temp2));
	if ( std::abs(temp2) <= 1 ) phi2max = pi-0.5*std::acos(sin_cos_range(temp2));

//mj	std::cout << SS( Jdipolar_A ) << SS( coupling1 ) << SS( temp1 ) <<
//mj	 SS( phi1min ) << SS( phi1max ) << std::endl;
//mj	std::cout << SS( Jdipolar_B ) << SS( coupling2 ) << SS( temp2 ) <<
//mj	 SS( phi2min ) << SS( phi2max ) << std::endl;

	if ( phi1min != 0.0 || phi2min != 0.0 ) reverse = true;

//mj loop over possible extreme points to find most restricting boders
//mj of possible ranges; phi1 and ph2 are polar angles from (theta,phi)
//mj of the two vectors connecting the interacting nuclei; alpha is the
//mj projection angle between the two angles

	for ( i1 = 1; i1 <= 3; ++i1 ) {
		phi1 = phi1min+((phi1max-phi1min)/2.0*(-1.0+i1));
		for ( i2 = 1; i2 <= 3; ++i2 ) {
			phi2 = phi2min+((phi2max-phi2min)/2.0*(-1.0+i2));

//mj 	std::cout << SS( phi1min ) << SS( phi1 ) << SS( phi1max ) <<
//mj 	 SS( phi2min ) << SS( phi2 ) << SS( phi2max ) << std::endl;

			f1 = (coupling1/xax-2.0)/(rho*std::cos(2.0*phi1)-2.0)*2.0/3.0;
			f1 = std::min(1.0f,std::max(0.0f,f1));
			f2 = (coupling2/xax-2.0)/(rho*std::cos(2.0*phi2)-2.0)*2.0/3.0;
			f2 = std::min(1.0f,std::max(0.0f,f2));

			temp1 = std::sqrt(f1*f2);
			temp2 = std::sqrt((1.0-f1)*(1.0-f2));

//mj 	std::cout << SS( f1 ) << SS( f2 ) << SS( temp1 ) << SS( temp2 ) << std::endl;

			alphaplus = std::acos(sin_cos_range(temp1*std::cos(phi1-phi2)+temp2));
			alphaminus = std::acos(sin_cos_range(temp1*std::cos(phi1-phi2)-temp2));

			if ( alphaplus < angle_restriction(1) ) angle_restriction(1) = alphaplus;
			if ( reverse ) {
				if ( alphaminus > angle_restriction(2) ) angle_restriction(2) = alphaminus;
			} else {
				if ( alphaminus < angle_restriction(3) ) angle_restriction(3) = alphaminus;
			}

			alphaplus = std::acos(sin_cos_range(temp1*std::cos(phi1+phi2)+temp2));
			alphaminus = std::acos(sin_cos_range(temp1*std::cos(phi1+phi2)-temp2));

			if ( reverse ) {
				if ( alphaplus < angle_restriction(3) ) angle_restriction(3) = alphaplus;
			} else {
				if ( alphaplus > angle_restriction(2) ) angle_restriction(2) = alphaplus;
			}
			if ( alphaminus > angle_restriction(4) ) angle_restriction(4) = alphaminus;
		}
	}

//mj make sure that angle_restriction(2) <= angle_restriction(3)

	if ( angle_restriction(2) > angle_restriction(3) ) {
		angle_restriction(2) = pi_over_2;
		angle_restriction(3) = pi_over_2;
	}

}
