// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 11985 $
//  $Date: 2007-01-09 02:58:35 -0800 (Tue, 09 Jan 2007) $
//  $Author: johnk $


// Rosetta Headers
#include "disulfides.h"
#include "after_opts.h"
#include "cenlist.h"
#include "constraints.h"
#include "current_pose.h"
#include "design.h"
#include "files_paths.h"
#include "fullatom_energies.h"
#include "input_pdb.h"
#include "misc.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "runlevel.h"
#include "score_ns.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>

namespace disulfides {

///////////////
/// Globals ///
///////////////

/// command line options for disulfides
namespace options {
	bool find_disulf = false; // cys-cys pairing not known,
		// find & score max # of disulfide bonds
	bool fix_disulf = false; // cys-cys pairing is known, score those disulfs
	bool use_disulf_logfile = false; // output physical parameters of disulfides to a user-defined filename
	bool norepack_disulf = false; // should packer or rotamer trials alter disulfs?
	bool minimize_disulf_chi = false;
}

/// Parameters for counting disulfides
namespace constants {
	int const max_cysteines = { 50 };

	int const max_disulfides = { 25 };

	float const disulf_cendist_cutoff = { 49.0 }; // angstroms^2 it was 49

	float const max_native_disulf_cendist = { 20.0 };
	 // max cendist seen in native disulf (approx.20 ang^2)
	//bs      parameter (max_native_disulf_cendist=49.0) // used for astrid

	float const min_native_disulf_cendist = { 2.0 };
	 // min cendist seen in native disulf (approx.2 ang^2)

	float const disulf_cbdist_cutoff = { 49.0 }; // angstroms^2

	float const min_native_disulf_cbdist = { 10.0 };
	 // min cbdist seen in native disulf (approx.10 ang^2)

	float const max_native_disulf_cbdist = { 22.0 };
	 // max cbdist seen in native disulf (approx.20 ang^2)

	float const disulf_ssdist_cutoff = { 3.0 }; // angstroms

	int const disulf_min_sequence_separation = { 3 }; // residues

	//bs ** Note: physical parameters of disulfide bonds are given
	//bs in function score_this_disulfide_fa in disulfides.cc
}


/////////////
/// Local ///
/////////////

/// Inner Namespaces
namespace cb_score_3gauss_fitparams {
	float y_0 = { 5.00 };
	float A_1 = { -3.7 };
	float x0_1 = { 12.445 };
	float xwidth_1 = { 1.66 };
	float A_2 = { -6.1 };
	float x0_2 = { 15.327 };
	float xwidth_2 = { 3.105 };
	float A_3 = { -0.3 };
	float x0_3 = { 14.0 };
	float xwidth_3 = { 0.5 };
}

////////////////
/// Boundary ///
////////////////
// yab:
// functions here act as an interface between the classic monolithic
// Rosetta world and the new object world until misc removal is
// complete and Rosetta's computational pipeline is detangled.
namespace BOUNDARY {

Disulfides * disulfides_object_; // singleton object that Boundary functions operate on

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_disulfide_filter
///
/// @brief  setup filtering of output structures by disulfide patterns
///
/// @detailed
///
/// @global_read
///     fix_disulf, find_disulf  disulfides.h
/// @global_write
///     use_filter    files_paths.h
/// @remarks
///     currently, there is no way to evaluate disulfides without also
///     filtering (except to use -output_all)
///     this should probably be a separate command option??
///
/// @references
///
/// @authors
///     car 9/3/2003
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_disulfide_filter()
{
	using namespace files_paths;

	if ( get_disulf_flag() ) {
		use_filter(disulfide_type) = truefalseoption("disulf_filter");
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_flag
///
/// @brief
///     returns T if a disulf mode is active (if fix_disulf or find_disulf is T)
/// @detailed
///
/// @return
///
/// @global_read
///     fix_disulf,find_disulf
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_disulf_flag()
{
	using namespace disulfides::options;

	return ( fix_disulf || find_disulf );
}

bool
get_minimize_disulf_chi()
{
	return disulfides::options::minimize_disulf_chi;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulfide_options
///
/// @brief
///     checks for command-line disulfide options for evaluating disulfides
/// @detailed
///
/// @global_read
///
/// @global_write
///     block: disulf_options
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_disulfide_options_and_init_module()
{
	// yab:
	// This routine is the entry point for the disulfides module.
	// At present it's only called once from options.cc.
	// It has been modified to initialize the single instance
	// disulfides_object_ in the BOUNDARY and should therefore
	// be the only reference to misc.
	using namespace design;
	using namespace files_paths;
	using namespace disulfides::options;
	using namespace misc; // only used to init disulfides_object_

	static bool init = false;
	if (init) {
		return;
	}

	disulfides_object_ = new Disulfides(&Eposition, &centroid, &full_coord, &phi, &psi, &res, &total_residue);

//car set default values of parameters
	active_rotamer_options.excys = false;

//car check for disulfide option flags
	if ( truefalseoption("find_disulf") ) find_disulf = true;
	if ( truefalseoption("fix_disulf") ) {
		fix_disulf = true;
		stringafteroption( "fix_disulf", "-", disulf_connectivity_file );
		if ( disulf_connectivity_file[1] == '-' ) {
			std::cout << "ERROR:: after -fix_disulf, must specify a filename" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	if ( !find_disulf && !fix_disulf ) return;

//bs check for inconsistent options
	if ( find_disulf && fix_disulf ) {
		std::cout << "ERROR:: both disulfide options selected" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//bs set excys to true if either disulf option selected
//bs this will give extra cys rotamers for disulfides
	active_rotamer_options.excys = true;
	std::cout << "Using extra cysteine rotamers...excys=True" << std::endl;

//car check for additional disulfide specific options

//cb no repack
	if ( truefalseoption("norepack_disulf") ) norepack_disulf = true;

	if ( truefalseoption("minimize_disulf_chi") ) minimize_disulf_chi = true;

//bs disulf logfile
	if ( truefalseoption("use_disulf_logfile") ) {
		use_disulf_logfile = true;
		stringafteroption( "use_disulf_logfile", "-", disulf_logfile );
		if ( disulf_logfile[1] == '-' ) {
			std::cout << "Error:: after -use_disulf_logfile, must specify filename" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

//bs read cys-cys pairs from a file if desired
	if ( fix_disulf ) disulfides_object_->read_disulf_connectivity(input_fa);

	init = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_norepack_disulf
///
/// @brief
///     T if command-line-option norepack_disulf is T (and if in a disulf mode)
/// @detailed
///
/// @return
///
/// @global_read
///     fix_disulf,find_disulf,norepack_disulf
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_norepack_disulf()
{
	using namespace disulfides::options;

	bool get_norepack_disulf = false;
	if ( fix_disulf || find_disulf ) get_norepack_disulf = norepack_disulf;
	return get_norepack_disulf;
}

// mirror method
void
evaluate_disulfides_centroid() {
	disulfides_object_->evaluate_disulfides_centroid();
}

// mirror method
void
output_disulf_stats_to_log(
	std::string & filename,
	std::string const & tag,
	bool const fullatom
)
{
	disulfides_object_->output_disulf_stats_to_log(filename, tag, fullatom);
}

// mirror method
void
identify_cys()
{
	disulfides_object_->identify_cys();
}

// mirror method
bool
cys_pair_in_disulf(
	int res1,
	int res2
)
{
	return disulfides_object_->cys_pair_in_disulf(res1, res2);
}

// mirror method
void
disulf_apply_filter(
	bool & accepted,
	std::string & tag
)
{
	disulfides_object_->disulf_apply_filter(accepted, tag);
}

// mirror method
bool
cys_res_in_disulf( int resnum )
{
	return disulfides_object_->cys_res_in_disulf(resnum);
}

// mirror method
float
getTotWtdScThisDisulf_pack(
	int const & cys1,
	FArray2Da_float cys1coord, // fullatom coords for cys1
	int const & cys2,
	FArray2Da_float cys2coord // fullatom coords for cys2
)
{
	return disulfides_object_->getTotWtdScThisDisulf_pack(cys1, cys1coord, cys2, cys2coord);
}

// mirror method
void
disulf_get_tot_wtd_cent_score( float & disulf_cent_score /* total weighted centroid score for entire protein */
)
{
	disulfides_object_->disulf_get_tot_wtd_cent_score(disulf_cent_score);
}

// mirror method
void
disulf_eval_fa_scores( float & disulf_fa_score /* total weighted fullatom score for entire protein */
)
{
	disulfides_object_->disulf_eval_fa_scores(disulf_fa_score);
}

// mirror method
bool
AtomVdwDisulfNativeClash(
	int atom1_in,
	int atom2_in
)
{
	return disulfides_object_->AtomVdwDisulfNativeClash(atom1_in, atom2_in);
}

// mirror method
void
eval_disulf_dist_deriv(
	int res1,
	int atom1,
	int res2,
	int atom2,
	bool fullatom,
	float & deriv
)
{
	disulfides_object_->eval_disulf_dist_deriv(res1, atom1, res2, atom2, fullatom, deriv);
}

// mirror method
void
eval_disulf_fullatom_energy()
{
	disulfides_object_->eval_disulf_fullatom_energy();
}

// mirror method
int
get_n_disulf_cent_natcut() {
	return disulfides_object_->get_n_disulf_cent_natcut();
}

// mirror method
int
get_n_disulf_centroid() {
	return disulfides_object_->get_n_disulf_centroid();
}

// mirror method
int
get_n_disulf_fa() {
	return disulfides_object_->get_n_disulf_fa();
}

// mirror method
int
get_n_known_disulfides() {
	return disulfides_object_->get_n_known_disulfides();
}

// mirror method
int
get_cys(
	int pos
)
{
	return disulfides_object_->get_cys(pos);
}

// mirror method
int
get_disulf_partner_a(
	int pos
)
{
	return disulfides_object_->get_disulf_partner_a(pos);
}

// mirror method
int
get_disulf_partner_b(
	int pos
)
{
	return disulfides_object_->get_disulf_partner_b(pos);
}

// mirror method
float
get_fullatom_disulf_dist_sc() {
	return disulfides_object_->get_fullatom_disulf_dist_sc();
}

// mirror method
float
get_fullatom_disulf_csang_sc() {
	return disulfides_object_->get_fullatom_disulf_csang_sc();
}

// mirror method
float
get_fullatom_disulf_dihedral_sc() {
	return disulfides_object_->get_fullatom_disulf_dihedral_sc();
}

// mirror method
float
get_fa_disulf_ca_dihed_sc() {
	return disulfides_object_->get_fa_disulf_ca_dihed_sc();
}

}


/***************************/
/* Disulfides class begins */
/***************************/

////////////////////////////
/// Creation/Destruction ///
////////////////////////////

/// Constructor
Disulfides::Disulfides(
	FArray3Dp_float const * Eposition,
	FArray2D_float const * centroid,
	FArray3D_float const * full_coord,
	FArray1D_float const * phi,
	FArray1D_float const * psi,
	FArray1D_int const * res,
	int const * total_residue
)
{
	Eposition_ = Eposition;
	centroid_ = centroid;
	phi_ = phi;
	psi_ = psi;
	full_coord_ = full_coord;
	res_ = res;
	total_residue_ = total_residue;

	initialize();
}

Disulfides::Disulfides() {}

/// Destructor
Disulfides::~Disulfides() {}


//////////////////////
/// Public: Methods ///
//////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_disulfides_centroid
///
/// @brief
///     gets disulfide connectivity based on centroid criteria
/// @detailed
///     causes update in cendist array for cys-cys pairs
///     gets disulf connectivity based on cendist
///     scores the decoy using centroid-level disulf scores
/// @global_read
///
/// @global_write
///     blocks: disulf_connectivity,count_cys_disulf
/// @remarks
///
/// @references
///
/// @authors
///    bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::evaluate_disulfides_centroid()
{
	using namespace disulfides::options;

//bs local
	bool fullatom;

	n_disulfides_centroid = 0;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;
	identify_cys(); //necessary when the misc.h arrays are changed during a run (e.g. interface mode)
	if ( num_cysteines == 0 ) return;
	 // num_cys=0 when clean_pdb calls pack, don't try to count disul

	fullatom = false;
	if ( !pose_flag() && !score_check_current_pose() ) centroid_place();//necessary when the misc.h arrays are changed during a run (e.g. interface mode)
	update_cyscys_cendist();

	if ( fix_disulf ) {
		count_disulfides(fullatom);
	} else if ( find_disulf ) {
		find_and_count_disulfides(fullatom);
	}

	score_disulf_cendist();
	score_disulf_cbdist();
	score_disulf_bb_dih();
	score_disulf_cacbcb();
	score_disulf_cacbcbca();
//bs	score_cadist_minus_cendist();

}


//bs****************************************************************************
//cc Counts the number of cysteines puts that in the global num_cysteines.
//cc Records which residues are cysteines in the global cys.

////////////////////////////////////////////////////////////////////////////////
/// @begin identify_cys
///
/// @brief
///     counts the cys in a protein
/// @detailed
///
/// @global_read
///     integers: total_residue
///     arrays: res
/// @global_write
///     blocks: count_cys_disulf
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::identify_cys()
{
//cc     res is a 1D array with the amino acid identity in alphabetical
//cc     order.  cysteine is residue 2

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;

	FArray1D_int const & res = *res_; // yab: misc removal
	int const & total_residue = *total_residue_; // yab: misc removal

	num_cysteines = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( res(i) == 2 ) {
			++num_cysteines;
			cys(num_cysteines) = i;
//bs           std::cout << "res # " << i << " is cys" << std::endl;
		}
	}

	if ( num_cysteines > disulfides::constants::max_cysteines ) {
		std::cout << "*****ERROR TOO MANY CYSTEINES COUNTED*****" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}


//------------------------------------------------------------------------------
//bs Checks the global disulfide_pairs to see if res1-res2 is listed as a disulfide.
//bs
//bs Note: even if a pair is listed, it may not satisfy the fa criteria.
//bs If fix_disulf = true, then being listed means
//bs that the pair should be a disulfide.
//bs If find_disulf = true, then being listed means
//bs that the pair satisify the cendist criteria.

////////////////////////////////////////////////////////////////////////////////
/// @begin cys_pair_in_disulf
///
/// @brief
///     returns T/F for res1-res2 is listed as a disulfide bond
/// @detailed
///bs Checks the global disulfide_pairs to see if res1-res2 is listed as a disulfide.
///bs
///bs Note: even if a pair is listed, it may not satisfy the fa criteria.
///bs If fix_disulf = true, then being listed means
///bs that the pair should be a disulfide.
///bs If find_disulf = true, then being listed means
///bs that the pair satisify the cendist criteria.
///
/// @param[in]   res1 - in -
/// @param[in]   res2 - in -
///
/// @return
///
/// @global_read
///     blocks: count_cys_disulf,disulf_connectivity
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::cys_pair_in_disulf(
	int res1,
	int res2
)
{
	FArray1D_int const & res = *res_; // yab: misc removal;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return false;
	if ( res(res1) != 2 || res(res2) != 2 ) return false; // both aren't cys;
	if ( num_cysteines < 2 ) return false; // can't have a disulf;
	if ( res1 == res2 ) return false; // can't have a disulf;
//$$$	if ( n_disulfides_centroid == 0 ) return false;
//no disulf (even if fix_disulf is true)

//bs return true if cys_1,cys_2 are a disulf pair
//bs note: disulf_partner arrays are ordered in increasing seq position
//bs but we must allow for the possbility that res1 > res2

//jjh initializing
	int n_disulfides = 0;

	if ( disulfides::options::fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( disulfides::options::find_disulf ) {
		n_disulfides = n_disulfides_centroid;
		if ( n_disulfides == 0 ) return false;
	}

	for ( int ii = 1; ii <= n_disulfides; ++ii ) {
		if ( ( res1 == cys(disulf_partner_a(ii)) ) &&
		 ( res2 == cys(disulf_partner_b(ii)) ) ) {
			return true;
		} else if ( ( res2 == cys(disulf_partner_a(ii)) ) &&
		 /* allow for the possibility that res1 > res2 */
		 ( res1 == cys(disulf_partner_b(ii)) ) ) {
			return true;
		}
	}
	return false;
}

//------------------------------------------------------------------------------
//bs Checks the global disulfide_pairs to see if resnum is a cysteine in a disulfide.
//bs
//bs If fix_disulf = true, then being listed means
//bs that the pair should be a disulfide.
//bs If find_disulf = true, then being listed means
//bs that the pair satisify the cendist criteria.

////////////////////////////////////////////////////////////////////////////////
/// @begin cys_res_in_disulf
///
/// @brief
///     no longer used, this will be deleted ?????
/// @detailed
///
/// @param  resnum - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::cys_res_in_disulf( int resnum )
{
	FArray1D_int const & res = *res_; // yab: misc removal;

	bool cys_res_in_disulf = false; // Return value // default is no.

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return cys_res_in_disulf;
	if ( res(resnum) != 2 ) return cys_res_in_disulf; // not a cys
	if ( num_cysteines < 2 ) return cys_res_in_disulf; // can't have a disulf
	if ( n_disulfides_centroid == 0 ) return cys_res_in_disulf;
	// no disulf (even if fix_disulf is true)

//bs if resnum is in a disulfide, return true.

//$$$    for ( ii = 1; ii <= num_cysteines; ++ii ) {
//$$$           if ( cys_pair_in_disulf(resnum,cys(ii)) ) {
//$$$              cys_res_in_disulf = true;
//$$$              return cys_res_in_disulf;
//$$$           }
//$$$    }

//jjh initializing
	int n_disulfides = 0;

	if ( disulfides::options::fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( disulfides::options::find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	for ( int ii = 1; ii <= n_disulfides; ++ii ) {
		if ( resnum == cys(disulf_partner_a(ii)) ) {
			cys_res_in_disulf = true;
			return cys_res_in_disulf;
		} else if ( resnum == cys(disulf_partner_b(ii)) ) {
			cys_res_in_disulf = true;
			return cys_res_in_disulf;
		}
	}

	return cys_res_in_disulf;
}


//------------------------------------------------------------------------------

void
Disulfides::disulf_apply_filter(
	bool & accepted,
	std::string & tag
)
{
//bs accepted and tag have already been set to
//bs default or by other filters. So only change them
//bs if fail the disulf filter.

	if ( !disulf_filter() ) {
		accepted = false;
		std::cout << "num disulfides centroid too low" << std::endl;
		tag = "disulf_fail";
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_use_disulf_logfile
///
/// @brief
///     T if command-line-option use_disulf_logfile is T
/// @detailed
///
/// @return
///
/// @global_read
///     bool: use_disulf_logfile
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::get_use_disulf_logfile()
{
	using namespace disulfides::options;

	return use_disulf_logfile;
}


//bs****************************************************************************
//bs returns total weighted fullatom disulfide score using packer weights
//bs (declared in param_pack.h, values in pack_db.cc)
//bs Only called from function get_energies in pack.cc

////////////////////////////////////////////////////////////////////////////////
/// @begin getTotWtdScThisDisulf_pack
///
/// @brief
///bs returns total weighted fullatom disulfide score using packer weights
/// @detailed
///bs returns total weighted fullatom disulfide score using packer weights
///bs (declared in param_pack.h, values in pack_db.cc)
///bs Only called from function get_energies in pack.cc
/// @param[in]   cys1 - in - residue number for cys 1
/// @param[in]   cys1coord - in - fa coords cys 1
/// @param[in]   cys2 - in - resnum cys 2
/// @param[in]   cys2coord - in - fa coords cys 2
///
/// @return
///
/// @global_read
/// disulfide packer weights in pack_db.cc
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::getTotWtdScThisDisulf_pack(
  int const & /* cys1 */, // unused
	FArray2Da_float cys1coord, // fullatom coords for cys1
	int const & /* cys2 */, // unused
	FArray2Da_float cys2coord // fullatom coords for cys2
)
{
	using namespace param;
	using namespace param_pack;

	cys1coord.dimension( 3, MAX_ATOM() );
	cys2coord.dimension( 3, MAX_ATOM() );

	float getTotWtdScThisDisulf_pack; // Return value

//bs local
	float dist_sc_this_disulf,csang_sc_this_disulf; // fa disulf scores
	float dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf; // fa disulf scores
	float cbs_sc_this_disulf;
	bool truefalse_fa_disulf;

	// get fullatom score for this disulfide */
	score_this_disulfide_fa(cys1coord,cys2coord,dist_sc_this_disulf,
	 csang_sc_this_disulf,dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf,
	 truefalse_fa_disulf);

//bs get the Cb-S score
	get_cbs_sc_this_disulf(cys1coord,cys2coord,cbs_sc_this_disulf);

	getTotWtdScThisDisulf_pack = dist_sc_this_disulf*Wdisulf_dist_sc +
	 csang_sc_this_disulf*Wdisulf_csang_sc +
	 dihedral_sc_this_disulf*Wdisulf_dihedral_sc +
	 ca_dihedral_sc_this_disulf*Wdisulf_ca_dihedral_sc +
	 cbs_sc_this_disulf*Wdisulf_cbs_dist_sc;


//$$$cbs debugging
//$$$  std::cout << "rot1res,rot2res = " << rot1res << ' ' << rot2res << std::endl;
//$$$  std::cout << "e2b_indexno = " << e2b_indexno << std::endl;
//$$$
//$$$  if ( std::abs(tot_weighted_sc_this_disulf) > 0.1 ) {
//$$$     std::cout << "+++++++++++++++++++++++++++++++++++++++" << std::endl;
//$$$     std::cout << "in get_energies, res1,res2 = " << rot1res << ' ' << rot2res << std::endl;
//$$$     std::cout << "rot1,rot2 = " << rot1 << ' ' << rot2 << std::endl;
//$$$     std::cout << "scores: " << SS( dist_sc_this_disulf ) <<
//$$$         csang_sc_this_disulf <<
//$$$         dihedral_sc_this_disulf << std::endl;
//$$$     std::cout << "wts: " << SS( Wdisulf_dist_sc ) << SS( Wdisulf_csang_sc ) <<
//$$$         SS( Wdisulf_dihedral_sc ) << std::endl;
//$$$     std::cout << "tot_weighted_sc_this_disulf = " <<
//$$$         SS( tot_weighted_sc_this_disulf ) << std::endl;
//$$$     std::cout << "before add disulf sc, e2b = " <<
//$$$         SS( energy2b(e2b_indexno) ) << std::endl;
//$$$  }
//$$$cbs debugging

	return getTotWtdScThisDisulf_pack;
}


//------------------------------------------------------------------------------
//bs returns total weighted centroid score for entire protein
//bs called from scorefxn()


////////////////////////////////////////////////////////////////////////////////
/// @begin disulf_get_tot_wtd_cent_score
///
/// @brief
///
/// @detailed
///bs returns total weighted centroid score for entire protein
///bs called from scorefxn()
///
/// @param[out]   disulf_cent_score - out -
///
/// @global_read
///     block: scoring_disulfide_weights (score_ns.h)
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::disulf_get_tot_wtd_cent_score( float & disulf_cent_score
/* total weighted centroid score for entire protein */ )
{
	using namespace scorefxns;
	using namespace scores;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) {
		disulf_cent_score = 0;
		return;
	}

	disulf_cent_score =
		disulf_cendist_score * disulf_cendist_weight +
		disulf_bb_dih_score * disulf_bb_dih_weight +
		disulf_cbdist_score * disulf_cbdist_weight +
		disulf_cacbcb_score * disulf_cacbcb_weight +
		disulf_cacbcbca_score * disulf_cacbcbca_weight+
		get_disulf_Ncen_sc() * disulf_Ncen_weight +
		get_disulf_Ncb_sc() * disulf_Ncb_weight;

}

//------------------------------------------------------------------------------
//bs This is an optional function to allow more weighting for fullatom disulf scores.
//bs Returns total weighted fullatom disulf score for entire protein,
//bs using weights set in various scores (e.g. score12) -- not use the packer weights
//bs Intended to be called from scorefxn()
//bs **Not currently used** -- currently the disulf fa score is computed by fullatom_energy and
//bs enters the total score in scorefxn via fa_pair).
//bs If this function is called from scorefxn it will add an additional contribution to scorefxn.
//bs If this function is used, must put weights into the appropriate score, as in the following:

//$$$      disulfide_distance_weight = 1.0;
//$$$      disulfide_csangles_weight = 1.0;
//$$$      disulfide_dihedral_weight = 1.0;
//$$$      disulfide_ca_dihedral_weight = 0.0;

//bs additionally, insert the text below into scorefxn

//$$$cbs disulfides x-tra fullatom contribution
//$$$         disulf_eval_fa_scores(disulf_fa_score);
//$$$         scorefxn += disulf_fa_score;

//bs 12_11_02 added disulf_Ncen_score for use with fullatom relaxation. Idea is that decoys
//bs input to fullatom relax have all centroid disulf made, and we do not want to let them
//bs be lost. So disulf_Ncen_sc has a very large weight, to preven the loss of any centroid
//bs disulfides. The actual score will be 0 if all are retained, but +large if any lost.
//bs this score will only work if fix_disulf, else it will be 0.
//bs then I added disulf_Nfa_score which works analogously using fa criteria.

////////////////////////////////////////////////////////////////////////////////
/// @begin disulf_eval_fa_scores
///
/// @brief
///
/// @detailed
///bs returns total weighted disulf fullatom score
///bs uses weights from scorefxns, not from pack.
/// @param  disulf_fa_score - [in/out]? -
///
/// @global_read
/// block: scoring_disulfide_weights (score_ns.h)
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::disulf_eval_fa_scores( float & disulf_fa_score
 /* total weighted fullatom score for entire protein */ )
{
	using namespace scorefxns;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) {
		disulf_fa_score = 0;
		return;
	}

	disulf_fa_score = get_fullatom_disulf_dist_sc() * disulfide_distance_weight +
	 get_fullatom_disulf_csang_sc() * disulfide_csangles_weight +
	 get_fullatom_disulf_dihedral_sc() * disulfide_dihedral_weight +
	 get_fa_disulf_ca_dihed_sc() * disulfide_ca_dihedral_weight +
	 get_fa_disulf_cbs_dist_sc() * disulfide_cbs_dist_weight +
	 get_disulf_Ncen_sc() * disulf_Ncen_weight +
	 get_disulf_Nfa_sc()*disulf_Nfa_weight;
}


//------------------------------------------------------------------------------
//bs called by count_pair, used to turn off LJ between atoms across disulfide bond
//bs if atom_vdw(atom1,atom2) is larger than the minimum distance seen in native disulfide bonds

////////////////////////////////////////////////////////////////////////////////
/// @begin AtomVdwDisulfNativeClash
///
/// @brief
///
/// @detailed
///bs called by count_pair, used to turn off LJ between atoms across disulfide bond
///bs if atom_vdw(atom1,atom2) is larger than the minimum distance seen in native disulfide bonds
///
/// @param[in]   atom1_in - in - // fullatom numbering scheme, but centroid=6.
/// @param[in]   atom2_in - in - // fullatom numbering scheme, but centroid=6.
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::AtomVdwDisulfNativeClash(
	int atom1_in,
	int atom2_in
)
{
	bool AtomVdwDisulfNativeClash; // Return value

//bs local
	// position numbering scheme, and cent=5+aa.
	static FArray1D_int const reorder_cys( 7, reorder_cys_initializer );

//bs minimum distances^2 (ang^2) taken from xray pdbs with resolution 2.1 ang or better
	static float const disulf_nat_min_ca_ca = { 14.0 };
	 // minimum ca-ca dist^2 across native disulfide bond
	static float const disulf_nat_min_ca_cent = { 7.0 };
	 // minimum ca-centroid dist^2 across native disulfide bond
	static float const disulf_nat_min_cb_cb = { 11.0 };
	 // minimum cb-cb dist^2 across native disulfide bond
	static float const disulf_nat_min_cb_cent = { 5.5 };
	 // minimum cb-centroid dist^2 across native disulfide bond
	static float const disulf_nat_min_cent_cent = { 2.5 };
	 // minimum cendist (centroid-centroid dist^2  across native disulfide bond

//bs note atom1_in,atom2_in are in fullatom numbering scheme, except the centroid is 6.
//bs =>N=1,Ca=2,C=3,O=4,Cb=5, cent=6.
//bs we must do checks on atom_vdw, but that uses the position numbering scheme, and the centroid is 5+aa.
//bs =>N=1,Ca=2,Cb=3,C=4,O=5, cent=5+aa...so for cysteine (aa=2), cent=7
//bs to deal with this conflict of numbering schemes, let's convert atom1,atom2 into the position scheme.

//bs convert cys from fullatom w/cent=6 to position w/cent=5+aa (cent part only works for cys////

	int const atom1 = reorder_cys(atom1_in);
	int const atom2 = reorder_cys(atom2_in);
	if ( atom1 == 6 || atom2 == 6 ) {
		std::cout << "ERROR inAtomVdwDisulfNativeClash" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	AtomVdwDisulfNativeClash = false;

	if ( atom1 == 2 && atom2 == 2 ) {    // ca-ca
		if ( vdw::atom_vdw(atom1,atom2) > disulf_nat_min_ca_ca ) {
			AtomVdwDisulfNativeClash = true;
		}
	} else if ( (atom1 == 2 && atom2 == 7) || (atom1 == 7 && atom2 == 2) ) {    // ca-cent
		if ( vdw::atom_vdw(atom1,atom2) > disulf_nat_min_ca_cent ) {
			AtomVdwDisulfNativeClash = true;
		}
	} else if ( atom1 == 3 && atom2 == 3 ) {    // cb-cb
		if ( vdw::atom_vdw(atom1,atom2) > disulf_nat_min_cb_cb ) {
			AtomVdwDisulfNativeClash = true;
		}
	} else if ( (atom1 == 3 && atom2 == 7) || (atom1 == 7 && atom2 == 3) ) {    // cb-cent
		if ( vdw::atom_vdw(atom1,atom2) > disulf_nat_min_cb_cent ) {
			AtomVdwDisulfNativeClash = true;
		}
	} else if ( atom1 == 7 && atom2 == 7 ) {    // cen-cen
		if ( vdw::atom_vdw(atom1,atom2) > disulf_nat_min_cent_cent ) {
			AtomVdwDisulfNativeClash = true;
		}
	}

//$$$	std::cout << "in filter " << atom1_in << ' ' << atom2_in << ' ' <<
//$$$	 atom1 << ' ' << atom2 << ' ' <<
//$$$	 L( AtomVdwDisulfNativeClash ) << std::endl;
	return AtomVdwDisulfNativeClash;
}


//------------------------------------------------------------------------------
void
Disulfides::eval_disulf_dist_deriv(
	int res1,
	int atom1,
	int res2,
	int atom2,
	bool fullatom,
	float & deriv
)
{
//bs local
	int disulf_pair_i;

//bs local relevant to constraints
	int sequence_sep;
	int noe_stage;


//bs please note the atom numbering scheme is from fullcoord (i.e. sulfur = 6)
//bs except if fullatom = F, then centroid will be 6
//bs 1=n,2=ca,3=c,4=o,5=cb, 6=centroid(if fullatom F) or 6=sulfur(if fullatom T)

	deriv = 0.0;
	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;
	if ( !cys_pair_in_disulf(res1,res2) ) return;

//bs deal with constraints...return with 0 if contact order too longrange.
	noe_stage = 9999; // default is not return with 0.
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();
	sequence_sep = std::abs(res2-res1);
	if ( noe_stage < sequence_sep ) return;

	disulf_pair_i = which_disulf_pair(res1,res2);

//bs if fullatom, only do sulf-sulf and cb-cb deriv at the moment.
	if ( fullatom ) {
		if ( atom1 == 6 && atom2 == 6 ) {
			deriv = get_disulf_ssdist_deriv(res1,atom1,res2,atom2,disulf_pair_i,
			 fullatom);
		} else if ( atom1 == 5 && atom2 == 5 ) {
			deriv = get_disulf_cbdist_deriv(res1,atom1,res2,atom2,disulf_pair_i,
			 fullatom);
		}
//bs centroid mode, only doing cb-cb and cen-cen
	} else {
		if ( atom1 == 5 && atom2 == 5 ) {
			deriv = get_disulf_cbdist_deriv(res1,atom1,res2,atom2,disulf_pair_i,
			 fullatom);
		} else if ( atom1 == 6 && atom2 == 6 ) {
			deriv = get_disulf_cendist_deriv(res1,atom1,res2,atom2,disulf_pair_i,
			 fullatom);
		}
	}

//$$$	if ( deriv != 0.0 && atom1 == 5 && atom2 == 5 )
//$$$	 std::cout << "atom1,atom2,deriv = " << atom1 << ' ' << atom2 << SS( deriv ) << std::endl;
//bs	if ( (atom1 == 5 && atom2 == 5) || (atom1 == 6 && atom2 == 6) ) {
//bs		std::cout << "atom1,atom2,deriv = " << atom1 << ' ' << atom2 << SS( deriv ) << std::endl;
//bs	}

}

//------------------------------------------------------------------------------

//bs This function is only called from fullatom_energy().

//bs Updates the global fullatom disulfide scores (disulfides.h). Also, evaluates
//bs the disulfide contributions to the 2 body energy and pair-pair terms
//bs in function fullatom_energy(), using wts in param_pack.

void
Disulfides::eval_disulf_fullatom_energy()
{
	// disulf wts for pack & fullatom_energy declared here

	using namespace fullatom_energies;
	using namespace param_pack;

//bs local
	int cys1,cys2;
	int n_disulfides;
	float tot_weighted_sc_this_disulf;
	 // total weighted score using packer weights (not scorefxn weights)
	float dist_sc_this_disulf,csang_sc_this_disulf;
	float dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf;
	float cbs_sc_this_disulf;
	bool truefalse_fa_disulf;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal

//bs Initialize fullatom disulfide scores
	set_fa_disulfide_scores_to_zero();

//bs count disulfides that satisfy centroid criteria.
//bs After this point, disulfide connectivity is set,
//bs so fullatom disulfide scoring below
//bs will score these cys pairs only.

	evaluate_disulfides_centroid();

//jjh initializing
	n_disulfides = 0;

	if ( disulfides::options::fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( disulfides::options::find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	for ( int ii = 1; ii <= n_disulfides; ++ii ) {
		cys1 = cys(disulf_partner_a(ii));
		cys2 = cys(disulf_partner_b(ii));

//bs get fa disulfide scores
		score_this_disulfide_fa(full_coord(1,1,cys1),full_coord(1,1,cys2),
		dist_sc_this_disulf,csang_sc_this_disulf,dihedral_sc_this_disulf,
		ca_dihedral_sc_this_disulf,truefalse_fa_disulf);

//bs get the Cb-S score

		get_cbs_sc_this_disulf(full_coord(1,1,cys1),full_coord(1,1,cys2),
		cbs_sc_this_disulf);

//bs eval the total weighted disulf fullatom score
		tot_weighted_sc_this_disulf = dist_sc_this_disulf*Wdisulf_dist_sc+
		csang_sc_this_disulf*Wdisulf_csang_sc+dihedral_sc_this_disulf*
		Wdisulf_dihedral_sc+ca_dihedral_sc_this_disulf*Wdisulf_ca_dihedral_sc+
		cbs_sc_this_disulf*Wdisulf_cbs_dist_sc;


//bs add weighted disulf scores into pair_energy and pair_pair arrays.

		pair_energy(cys1) += 0.5*tot_weighted_sc_this_disulf;
		pair_energy(cys2) += 0.5*tot_weighted_sc_this_disulf;

		pair_pair(cys1,cys2) += tot_weighted_sc_this_disulf;
		pair_pair(cys2,cys1) += tot_weighted_sc_this_disulf;

//bs update global fullatom disulf scores (disulfides.h)

		update_disulf_fa_scores(truefalse_fa_disulf,dist_sc_this_disulf,
		csang_sc_this_disulf,dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf,
		cbs_sc_this_disulf);

//$$$ this is used to debug the derivs, other things
//$$$		std::cout << "disulf_fa_sc " <<
//$$$		 SS( dist_sc_this_disulf * Wdisulf_dist_sc * fa_pair_weight ) <<
//$$$		 SS( csang_sc_this_disulf * Wdisulf_csang_sc * fa_pair_weight ) <<
//$$$		 SS( dihedral_sc_this_disulf * Wdisulf_dihedral_sc * fa_pair_weight ) <<
//$$$		 std::endl;

	}
}

int
Disulfides::get_cys(
	int pos
)
{
	return cys(pos);
}

int
Disulfides::get_disulf_partner_a(
	int pos
)
{
	return disulf_partner_a(pos);
}

int
Disulfides::get_disulf_partner_b(
	int pos
)
{
	return disulf_partner_b(pos);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_n_disulf_cent_natcut
///
/// @brief
///     returns value of global n_disulf_cent_natcut
/// @detailed
///     n_disulf_cent_natcut is n disulfides with cendist < native cutoff cendist
/// @return
///
/// @global_read
///     n_disulf_cent_natcut
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
Disulfides::get_n_disulf_cent_natcut()
{
	return n_disulf_cent_natcut;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_n_disulf_centroid
///
/// @brief
///     returns value of global n_disulfides_centroid
/// @detailed
///
/// @return
///
/// @global_read
///     n_disulfides_centroid
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
Disulfides::get_n_disulf_centroid()
{
	return n_disulfides_centroid;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_n_disulf_fa
///
/// @brief
///     returns value of global n_disulfides_fa
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
Disulfides::get_n_disulf_fa()
{
	return n_disulfides_fa;
}

int
Disulfides::get_n_known_disulfides() {
	return n_known_disulfides;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_disulf_dist_sc
///
/// @brief
///     returns value of global disulfide_distance_score
/// @detailed
///     this is the score based on ss-distance
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_fullatom_disulf_dist_sc()
{
	return disulfide_distance_score;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_disulf_csang_sc
///
/// @brief
///     returns value of global disulfide_csangles_score
/// @detailed
///     this score is based on bond angles cb-s-s
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_fullatom_disulf_csang_sc()
{
	return disulfide_csangles_score;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_disulf_dihedral_sc
///
/// @brief
///     returns value of global disulfide_dihedral_score
/// @detailed
///     this score is based on the dihedral cb-s-s-cb
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_fullatom_disulf_dihedral_sc()
{
	return disulfide_dihedral_score;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_fa_disulf_ca_dihed_sc
///
/// @brief
///     returns value of global disulfide_ca_dihedral_score
/// @detailed
///     this score is based on dihedral ca-cb-s-s
///     this is like a chi2 for a cysteine involved in a disulfide
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_fa_disulf_ca_dihed_sc()
{
	return disulfide_ca_dihedral_score;
}

//bs****************************************************************************
//bs

////////////////////////////////////////////////////////////////////////////////
/// @begin read_disulf_connectivity
///
/// @brief
///bs If fix_disulf = true, reads the cys-cys disulfide pairs from a file
/// @detailed
///bs Reads the cys-cys pairs from a file
///bs Format for the file is i2,1x,i2
///bs Records the pairs in globals: disulf_partner_a and disulf_partner_b
///bs Also sets the value of global: n_known_disulfides
///bs outputs message to disulf_logfile if that is used.
/// @param[in]   fullatom - in - bool fa or not
///
/// @global_read
///     fix_disulf,use_disulf_logfile
/// @global_write
///     disulf_partner_a and disulf_partner_b
///     n_known_disulfides,n_disulfides_centroid,n_disulfides_fa
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::read_disulf_connectivity( bool const fullatom )
{
//bs Reads the cys-cys pairs from a file
//bs Format for the file is i2,1x,i2
//bs Records the pairs in globals: disulf_partner_a and disulf_partner_b
//bs Also sets the value of global: n_known_disulfides

	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace files_paths;

//$$$    float dist_between_sulfurs;             // dist between cys sulfurs
//$$$    int cys_1,cys_2;         // residue number of cysteine
//$$$    FArray1D_int n_neighbors(max_cysteines); // number of other cys within cutoff distance
//       int testvar;                // dummy variable

	utility::io::izstream disulf_x;

	disulf_x.open( disulf_connectivity_file );
	if ( !disulf_x ) {
		std::cout << "ERROR:: " << disulf_x.filename() <<
		 " does not exist ... exiting   !" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//bs read in disulf_connectivity file, setting the elements of the disulf_partner_a and _b arrays.

	n_known_disulfides = 0;
	if ( use_disulf_logfile ) {
		open_disulf_logfile(); // this is for output
		disulf_log_x << "***********************************************************" << std::endl;
		disulf_log_x << "**Using pre-defined disulf connectivity**" << std::endl;
		disulf_log_x << "Reading in cys-cys pairs from file..." << std::endl;
		disulf_log_x << "disulfide #    cys # to cys #" << std::endl;
	}
L10:
	while ( disulf_x ) {
		disulf_x >> bite( 2, disulf_partner_a(n_known_disulfides+1) ) >>
		 skip( 1 ) >> bite( 2, disulf_partner_b(n_known_disulfides+1) ) >> skip;
		if ( disulf_x.eof() ) {
			goto L50;
		} else if ( disulf_x.fail() ) {
			disulf_x.clear();
			disulf_x >> skip;
			goto L10;
		}
		++n_known_disulfides;

		if ( use_disulf_logfile ) disulf_log_x << n_known_disulfides << "       " <<
		 disulf_partner_a(n_known_disulfides) << "    " <<
		  disulf_partner_b(n_known_disulfides) << std::endl;

		if ( n_known_disulfides > max_disulfides ) {
			std::cout << "Error::number of cys-cys pairs in " <<
			 disulf_connectivity_file <<
			 " exceeds max_disulfides (" << max_disulfides << ')' << std::endl;
			std::cout << "exiting..." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}

L50:
	disulf_x.close();
	disulf_x.clear();

	if ( use_disulf_logfile ) disulf_log_x << "read in " << n_known_disulfides <<
	 " disulfides" << std::endl;

//$$$    testvar = 2*n_known_disulfides
//$$$    if ( num_cysteines < testvar ) {
//$$$       std::cout << "ERROR...EXITING:: " <<
//$$$				 "num_cysteines too small for that many disulf" << std::endl;
//$$$           std::cout << "n_cys,n_known_disulf = " <<
//$$$					  num_cysteines << ' ' << n_known_disulfides << std::endl;
//$$$       utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
//$$$    }
	if ( use_disulf_logfile ) disulf_log_x <<
	 "***********************************************************" << std::endl;

//bs set elements of the global 2d disulfide connectivity array: disulfide_pairs

	for ( int ii = 1; ii <= max_cysteines; ++ii ) {
		for ( int jj = 1; jj <= max_cysteines; ++jj ) {
			disulfide_pairs(ii,jj) = 0;
		}
	}

	for ( int ii = 1; ii <= n_known_disulfides; ++ii ) {
		disulfide_pairs(disulf_partner_a(ii),disulf_partner_b(ii)) = 1;
		disulfide_pairs(disulf_partner_b(ii),disulf_partner_a(ii)) = 1;
	}

//bs set the globals for # of disulfides

	if ( fullatom ) {
		n_disulfides_fa = n_known_disulfides;
		n_disulfides_centroid = n_known_disulfides;
		 // it could be larger than n_dis_fa, but ignore that.
	} else {
		n_disulfides_fa = 0;
		n_disulfides_centroid = n_known_disulfides;
	}
//$$$cbs look for clusters...
//$$$cbs if any cys that is in a disulfide has more than one neighbor, then we have a cluster.
//$$$    for ( int ii = 1; ii <= n_known_disulfides; ++ii ) {
//$$$       cys_cluster_tag(ii) = 0
//$$$       if ( n_neighbors(disulf_partner_a(ii)) > 1 || n_neighbors(disulf_partner_b(ii)) > 1 ) { // found cluster
//$$$          cys_cluster_tag(ii) = 1
//$$$       }
//$$$    }

	disulf_log_x.close();
	disulf_log_x.clear();
}
//bs****************************************************************************

////////////////////////////////////////////////////////////////////////////////
/// @begin output_disulf_stats_to_log
///
/// @brief
///     outputs disulf stats to disulf_logfile
/// @detailed
///
/// @param[in]   filename - in - decoyname (either pdb or silentfiletag)
/// @param[out]   tag - in - standard decoy tag (e.g output_decoy,input,native, etc.)
/// @param[in]   fullatom - in - bool for fullatom or not
///
/// @global_read
///     logicals: use_disulf_logfile
///     blocks: count_cys_disulf,disulf_connectivity
///     arrays: Eposition,cendist,full_coord
///     reals: MAX_ATOM
/// @global_write
///
/// @remarks
///     This actually recomputes all distances/scores. If I would store the distances
///     it wouldn't be necessary to recompute. which is better?
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::output_disulf_stats_to_log(
	std::string & filename,
	std::string const & tag,
	bool const fullatom
)
{
	using namespace cenlist_ns;
	using namespace disulfides::options;
	using namespace files_paths;
	using namespace param;
	using namespace runlevel_ns;

	FArray3Dp_float const & Eposition = *Eposition_; // yab misc removal
	FArray3D_float const & full_coord = *full_coord_; // yab misc removal
	FArray1D_float const & phi = *phi_; // yab misc removal
	FArray1D_float const & psi = *psi_; // yab misc removal

//bs local for disulfide diagnostics

	int ii;
	int n_disulfides;
	int cys_1,cys_2; // residue number of cysteine
	FArray2D_float cys1coord( 3, 5 );
	FArray2D_float cys2coord( 3, 5 );
	FArray2D_float cys1coord_fa( 3, MAX_ATOM()() );
	FArray2D_float cys2coord_fa( 3, MAX_ATOM()() );

//bs backbone diagnostics
	float chi_1; // cys rotamer: dihedral angle, N1-calpha1-cbeta1-s1
	float chi_2; // cys rotamer: dihedral angle, N2-calpha2-cbeta2-s2
	float phi_1,psi_1,phi_2,psi_2; // phi,psi angles for two cys in disulf
	float centroids_dist; // dist^2 between centroids (ang^2)
	float calphas_dist; // dist^2 between calphas (ang^2)
	float cbetas_dist;
//$$$    float cacent_dist_ab;  // calpha_1 - centroid_2 dist^2
//$$$    float cacent_dist_ba;  // calpha_2 - centroid_1 dist^2
//$$$    float cbcent_dist_ab;  // cb_1 - centroid_2 dist^2
//$$$    float cbcent_dist_ba;  // cb_2 - centroid_1 dist^2
	float cacb_dist_ab; // calpha_1 - cb_2 dist^2
	float cacb_dist_ba; // calpha_2 - cb_1 dist^2
	float disulf_cacbcbca_dih;
	float disulf_cacbcb_ang_1,disulf_cacbcb_ang_2;
	float disulf_bb_dih; // backbone dihedral, N_1-Calpha_1-Calpha_2-N_2

//bs sidechain diagnostics
	float dist_between_sulfurs; // dist between cys sulfurs
	float cs_bond_angle_1,cs_bond_angle_2; // carbon-sulfur bond angs
	float disulf_dihedral_angle; // dihedral (torsion) angle, cbeta1-s1-s2-cbeta2
	float disulf_ca_dihedral_angle_1;
	 // dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	float disulf_ca_dihedral_angle_2;
	 // dihedral (torsion) angle, calpha2-cbeta2-s2-s1
	float dist_cbs_1,dist_cbs_2; // Cb-S distance across disulfide

//bs fullatom disulfide scores
	float dist_sc_this_disulf;
	float csang_sc_this_disulf;
	float dihedral_sc_this_disulf;
	float ca_dihedral_sc_this_disulf;
	float cbs_dist_sc_this_disulf;

	bool truefalse_fa_disulf;

//bs output header only the first time this function is called.
	static bool output_fa_header = { true };

//bs don't output if not want disulf logfile
	if ( !get_use_disulf_logfile() ) return;

//bs don't output for no_pdbfile_fail
	if ( has_prefix( filename, "no_pdbfile_fail" ) ) return;

//bs silent mode.

	if ( runlevel == silent ) filename = "silent***************";


//bs send stuff to disulf logfile => iunit = disulf_log_x

	std::ofstream & iunit( disulf_log_x );
	open_disulf_logfile();

	if ( output_fa_header ) {
		output_fa_header = false;
		iunit << "----------------------------------" << std::endl;
		iunit << "Disulfide params & scores" << std::endl;
		iunit << "----------------------------------" << std::endl;


		iunit << "filename" << space( 15 ) << "Ndis" <<
		 "  cys_a" << "  cys_b" << "  cys_i" << "  cys_j" << "  " << "cendist";
		iunit << "   phi_a" << "   psi_a" << "   phi_b" << "   psi_b";
		iunit << "  bb_dih" << "  ca_dist" << "  cb_dist" <<
		 "  cacb_ab" << "  cacb_ba";
		iunit << "  cacbcb_1" << "  cacbcb_2" << "  cacbcbca";

		if ( fullatom ) {
			iunit << "  ss_dist" << "  cs_ang1" << "  cs_ang2" <<
			 "  dihedral";
			iunit << "  ca_dih1" << "  ca_dih2";
			iunit << "   chi_1" << "   chi_2";
			iunit << "  dist_sc" << "  csang_sc";
			iunit << "  dihed_sc" << "  cadih_sc";
			iunit << "  cbs_dist1" << "  cbs_dist2";
			iunit << "  cbs_dist_sc";
		}

		iunit << "  description" << std::endl;

	} else {
		output_fa_header = false;
	}

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) n_disulfides = n_known_disulfides;
	if ( find_disulf ) n_disulfides = n_disulfides_centroid;

	for ( ii = 1; ii <= n_disulfides; ++ii ) {

		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));

//bs now get parameters of disulfide and score it.

		centroids_dist = cendist(cys_1,cys_2);

//bs cys1coord needs to have the full_coord atom ordering (N,Ca,C,O,Cb)
//bs but Eposition has diff ordering...(N,Ca,Cb,C,O)...so convert
		for ( int k = 1; k <= 3; ++k ) {
			cys1coord(k,1) = Eposition(k,1,cys_1);
			cys2coord(k,1) = Eposition(k,1,cys_2);
			cys1coord(k,2) = Eposition(k,2,cys_1);
			cys2coord(k,2) = Eposition(k,2,cys_2);
			cys1coord(k,3) = Eposition(k,4,cys_1);
			cys2coord(k,3) = Eposition(k,4,cys_2);
			cys1coord(k,4) = Eposition(k,5,cys_1);
			cys2coord(k,4) = Eposition(k,5,cys_2);
			cys1coord(k,5) = Eposition(k,3,cys_1);
			cys2coord(k,5) = Eposition(k,3,cys_2);
		}

//bs get the dist^2 between calphas
//bs get the dist^2 between cbetas
		calphas_dist = 0.0;
		cbetas_dist = 0.0;
//$$$     cacent_dist_ab = 0.0;
//$$$     cacent_dist_ba = 0.0;
//$$$     cbcent_dist_ab = 0.0;
//$$$     cbcent_dist_ba = 0.0;
		cacb_dist_ab = 0.0;
		cacb_dist_ba = 0.0;

		for ( int k = 1; k <= 3; ++k ) {
			float const cys1_k2 = cys1coord(k,2);
			float const cys2_k2 = cys2coord(k,2);
			float const cys1_k5 = cys1coord(k,5);
			float const cys2_k5 = cys2coord(k,5);
			calphas_dist += ( cys2_k2 - cys1_k2 ) * ( cys2_k2 - cys1_k2 );
			cbetas_dist += ( cys2_k5 - cys1_k5 ) * ( cys2_k5 - cys1_k5 );
//$$$			float const cen_k1 = centroid(k,cys_1);
//$$$			float const cen_k2 = centroid(k,cys_2);
//$$$			cacent_dist_ab += ( cys1_k2 - cen_k2 ) * ( cys1_k2 - cen_k2 );
//$$$			cacent_dist_ba += ( cys2_k2 - cen_k1 ) * ( cys2_k2 - cen_k1 );
//$$$			cbcent_dist_ab += ( cys1_k5 - cen_k2 ) * ( cys1_k5 - cen_k2 );
//$$$			cbcent_dist_ba += ( cys2_k5 - cen_k1 ) * ( cys2_k5 - cen_k1 );
			cacb_dist_ab += ( cys1_k2 - cys2_k5 ) * ( cys1_k2 - cys2_k5 );
			cacb_dist_ba += ( cys2_k2 - cys1_k5 ) * ( cys2_k2 - cys1_k5 );
		}


//bs get the disulf backbone dihedral

		get_disulf_bb_dih(cys1coord,cys2coord,disulf_bb_dih);

		get_disulf_cacbcbca_dih(cys1coord,cys2coord,disulf_cacbcbca_dih);

		get_disulf_cacbcb_angles(cys1coord,cys2coord,disulf_cacbcb_ang_1,
		 disulf_cacbcb_ang_2);


//bs get phi,psi
		phi_1 = phi(cys_1);
		psi_1 = psi(cys_1);
		phi_2 = phi(cys_2);
		psi_2 = psi(cys_2);

//bs get fullatom disulfide params

		if ( fullatom ) {
			for ( int j = 1, max_atom = MAX_ATOM(); j <= max_atom; ++j ) {
			 // get coords of cys residues in fullatom mode
				for ( int k = 1; k <= 3; ++k ) {
					cys1coord_fa(k,j) = full_coord(k,j,cys_1);
					cys2coord_fa(k,j) = full_coord(k,j,cys_2);
				}
			}
			get_disulfide_params(cys1coord_fa,cys2coord_fa,dist_between_sulfurs,
			 cs_bond_angle_1,cs_bond_angle_2,disulf_dihedral_angle,
			 disulf_ca_dihedral_angle_1,disulf_ca_dihedral_angle_2);

			get_disulf_cbs_dist(cys1coord_fa,cys2coord_fa,dist_cbs_1,dist_cbs_2);


			score_this_disulfide_fa(cys1coord_fa,cys2coord_fa,dist_sc_this_disulf,
			 csang_sc_this_disulf,dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf,
			 truefalse_fa_disulf);
			get_disulf_chis(cys1coord_fa,cys2coord_fa, /* rotamer angles */ chi_1,
			 chi_2);

			get_cbs_sc_this_disulf(cys1coord_fa,cys2coord_fa,cbs_dist_sc_this_disulf);

		}


//bs        if ( centroids_dist <= max_native_disulf_cendist ) {
		if ( centroids_dist <= 100 ) {
			iunit << A( 21, filename ) << "   " <<
			 I( 3, ii ) << "    " << I( 3, cys_1 ) << "    " <<
			 I( 3, cys_2 ) << "    " << I( 3, disulf_partner_a(ii) ) << "    " <<
			 I( 3, disulf_partner_b(ii) ) << "  " << F( 7, 2, centroids_dist );
			iunit << ' ' << F( 7, 2, phi_1 ) << ' ' <<
			 F( 7, 2, psi_1 ) << ' ' << F( 7, 2, phi_2 ) << ' ' << F( 7, 2, psi_2 );
			iunit << ' ' << F( 7, 2, disulf_bb_dih ) << ' ' <<
			 F( 8, 2, calphas_dist ) << ' ' << F( 8, 2, cbetas_dist ) << ' ' <<
			 F( 8, 2, cacb_dist_ab ) << ' ' << F( 8, 2, cacb_dist_ba );
			iunit << "  " << F( 8, 2, disulf_cacbcb_ang_1 ) << "  " <<
			 F( 8, 2, disulf_cacbcb_ang_2 ) << "  " <<
			 F( 8, 2, disulf_cacbcbca_dih ) << "  ";

			if ( fullatom ) {
				iunit << ' ' << F( 8, 2, dist_between_sulfurs ) << "  " <<
				 F( 7, 2, cs_bond_angle_1 ) << "  " <<
				 F( 7, 2, cs_bond_angle_2 ) << "   " <<
				 F( 7, 2, disulf_dihedral_angle );
				iunit << "  " << F( 7, 2, disulf_ca_dihedral_angle_1 ) << "  " <<
				 F( 7, 2, disulf_ca_dihedral_angle_2 );
				iunit << ' ' << F( 7, 2, chi_1 ) << ' ' << F( 7, 2, chi_2 );
				iunit << "  " << F( 7, 2, dist_sc_this_disulf ) << "  " <<
				 F( 8, 2, csang_sc_this_disulf );
				iunit << "  " << F( 8, 2, dihedral_sc_this_disulf ) << "  " <<
				 F( 8, 2, ca_dihedral_sc_this_disulf );
				iunit << "  " << F( 9, 2, dist_cbs_1 ) << "  " << F( 9, 2, dist_cbs_2 );
				iunit << "  " << F( 11, 2, cbs_dist_sc_this_disulf );

			}

			iunit << "  " << tag << std::endl;
		}

	}


//$$$     iunit << "***************************************************" << std::endl;
//$$$         iunit << "** Total fullatom disulfide scores/weights **" << std::endl;
//$$$     iunit << "disulfide_distance_weight = " << disulfide_distance_weight << std::endl;
//$$$     iunit << "disulfide_csangles_weight = " << disulfide_csangles_weight << std::endl;
//$$$     iunit << "disulfide_dihedral_weight = " << disulfide_dihedral_weight << std::endl;
//$$$     iunit << "disulfide_ca_dihedral_weight = " << disulfide_ca_dihedral_weight << std::endl;
//$$$     iunit << "fa_pair_weight = " << fa_pair_weight << std::endl;
//$$$         iunit << std::endl;
//$$$         iunit << "disulfide_distance_score = " << get_fullatom_disulf_dist_sc() << std::endl;
//$$$         iunit << "disulfide_cs_angles_score = " << get_fullatom_disulf_csang_sc() << std::endl;
//$$$         iunit << "disulfide_dihedral_score = " << get_fullatom_disulf_dihedral_sc() << std::endl;
//$$$         iunit << "disulfide_ca_dihedral_score = " <<
//$$$          get_fa_disulf_ca_dihed_sc() << std::endl;
//$$$     iunit << "***************************************************" << std::endl;
//$$$     iunit << "total weighted fa disulf score = " <<
//$$$      get_fullatom_disulf_dist_sc()*(disulfide_distance_weight+fa_pair_weight)+
//$$$      get_fullatom_disulf_csang_sc()*(disulfide_csangles_weight+fa_pair_weight)+
//$$$      get_fullatom_disulf_dihedral_sc()*(disulfide_dihedral_weight+fa_pair_weight) << std::endl;
//$$$     iunit << "***************************************************" << std::endl;
//$$$     iunit << std::endl;
//$$$         iunit << "** Total centroid disulfide scores/weights **" << std::endl;
//$$$
//$$$     iunit << "disulf_cendist_weight = " << disulf_cendist_weight << std::endl;
//$$$     iunit << "disulf_cbdist_weight = " << disulf_cbdist_weight << std::endl;
//$$$     iunit << "disulf_bb_dih_weight = " << disulf_bb_dih_weight << std::endl;
//$$$     iunit << "disulf_ca_minus_cen_weight = " << disulf_ca_minus_cen_weight << std::endl;
//$$$         iunit << std::endl;
//$$$     iunit << "disulf_cendist_score = " << disulf_cendist_score << std::endl;
//$$$     iunit << "disulf_cbdist_score = " << disulf_cbdist_score << std::endl;
//$$$     iunit << "disulf_bb_dih_score = " << disulf_bb_dih_score << std::endl;
//$$$     iunit << "disulf_ca_minus_cen_score = " << disulf_ca_minus_cen_score << std::endl;
//$$$     iunit << "***************************************************" << std::endl;
//$$$     iunit << "total weighted disulf centroid score = " <<
//$$$      disulf_cendist_score*disulf_cendist_weight+
//$$$      disulf_cbdist_score*disulf_cbdist_weight+disulf_bb_dih_score*
//$$$      disulf_bb_dih_weight+disulf_ca_minus_cen_score*
//$$$      disulf_ca_minus_cen_weight << std::endl;
//$$$     iunit << "***************************************************" << std::endl;



//$$$    iunit << "astrid_line:   " <<
//$$$     A( 21, filename ) << "   " << I( 3, num_cysteines ) << "   " <<
//$$$			 I( 3, n_disulf_cent_natcut ) << std::endl;

	disulf_log_x.close();
	disulf_log_x.clear();
}

//bs****************************************************************************
//bs write out chi angle to logfile for all cys residues

////////////////////////////////////////////////////////////////////////////////
/// @begin output_cys_chi_to_logfile
///
/// @brief
///     writes out chi for each cys, to disulf_logfile
/// @detailed
///
/// @global_read
///     bool: use_disulf_logfile, block: count_cys_disulf
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::output_cys_chi_to_logfile()
{
	using namespace disulfides::options;
	using namespace files_paths;

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal;

//bs local
	FArray1D_float nitro( 3 );
	FArray1D_float calpha( 3 );
	FArray1D_float cbeta( 3 );
	FArray1D_float sulfur( 3 );
//$$$    int cys_1,cys_2;
	float chi;
//$$$        chi_1,chi_2
//$$$    float dist_between_sulfurs;

	if ( use_disulf_logfile ) {
		open_disulf_logfile();

		for ( int i = 1; i <= num_cysteines; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				nitro(j) = full_coord(j,1,cys(i));
				 // atom #1 is N, cys(i) is res # of ith cysteine
				calpha(j) = full_coord(j,2,cys(i)); // atom #2 is calpha
				cbeta(j) = full_coord(j,5,cys(i));
				sulfur(j) = full_coord(j,6,cys(i));
			}
			chi = dihedral(nitro,calpha,cbeta,sulfur);
			disulf_log_x << "allcys_chi " << SS( chi ) <<
			 " cys_res" << SS( cys(i) ) << std::endl;
		}
		disulf_log_x.close();
		disulf_log_x.clear();
	}
}

///////////////////////
/// Private: Methods ///
///////////////////////

//bs****************************************************************************
//bs initialize all disulfide-related globals to 0
//bs includes scores and bookkeeping items.
//bs called in initialize_start

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_disulfide_globals
///
/// @brief
///     initialize all disulfide-related globals to 0.
/// @detailed
///
/// @global_read
///     logicals: fix_disulf,find_disulf
///     blocks: count_cys_disulf,disulf_connectivity
/// @global_write
///     blocks: count_cys_disulf,disulf_connectivity
///     blocks: score_disulf_centroid (score_ns.h)
///     blocks: fullatom_disulfide_scores (disulfides.h)
/// @remarks
///     Why is this necessary?
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::initialize_disulfide_globals()
{
	using namespace disulfides::options;
	using namespace disulfides::constants;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;

	if ( find_disulf ) {

//bs do we really need this? used currently to avoid prob. clean_pdb->pack
//bs		num_cysteines = 0;

		for ( int ii = 1; ii <= max_cysteines; ++ii ) {
			cys(ii) = 0;
			for ( int jj = 1; jj <= max_cysteines; ++jj ) {
				disulfide_pairs(ii,jj) = 0;
			}
		}

		for ( int ii = 1; ii <= max_disulfides; ++ii ) {
			disulf_partner_a(ii) = 0;
			disulf_partner_b(ii) = 0;
			track_disulf_cen_cb(1,ii) = 0; // cen check
			track_disulf_cen_cb(2,ii) = 0; // cb check
		}

		set_cent_disulf_scores_to_zero();
		set_fa_disulfide_scores_to_zero();
	} else if ( fix_disulf ) {

		set_cent_disulf_scores_to_zero();
		set_fa_disulfide_scores_to_zero();
	}

}


//bs****************************************************************************
//bs Counts disulfide bonds in structures with known cys-cys pairs.
//bs Only used if fix_disulf = true
//bs A disulfide is counted if distance < cutoff.
//bs If fullatom = true, uses sulfur-sulfur distance criteria,
//bs and resets value of the global n_disulfides_fa.
//bs If fullatom = false, uses (centroid-centroid distance)^2 criteria, and
//bs resets value of the global n_disulfides_centroid.

////////////////////////////////////////////////////////////////////////////////
/// @begin count_disulfides
///
/// @brief
///     Counts disulfide bonds in structures with known cys-cys pairs.
/// @detailed
///bs Only used if fix_disulf = true
///bs A disulfide is counted if distance < cutoff.
///bs If fullatom = true, uses sulfur-sulfur distance criteria,
///bs and resets value of the global n_disulfides_fa.
///bs If fullatom = false, uses (centroid-centroid distance)^2 criteria, and
///bs resets value of the global n_disulfides_centroid.
///
/// @param[in]   fullatom - in - bool fa or not
///
/// @global_read
///     bool: fix_disulf
///     float: disulf_cendist_cutoff,disulf_ssdist_cutoff
///     block: count_cys_disulf
///     arrays: cendist,full_coord
/// @global_write
///     integer: n_disulfides_centroid,n_disulfides_fa
///     block: disulf_connectivity
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::count_disulfides( bool fullatom )
{
//bs this function uses the global disulf_partner_a and _b arrays, these
//bs are ordered lists of the cysteines in disulfide pairs.

	using namespace cenlist_ns;
	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace runlevel_ns;

// local
	int cys_1,cys_2;
	int n_disulfides;
	float distance_separation;
	float disulf_distance_cutoff;

//bs make sure we should be here.

	if ( !fix_disulf ) {
		std::cout << "Error, called count_disulfides but fix_disulf = F" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( num_cysteines < 2 ) {      // can't have a disulfide
		n_disulfides_fa = 0;
		n_disulfides_centroid = 0;
		if ( runlevel > chat ) std::cout << "less than 2 cys => no disulf" << std::endl;
		return;
	}

//bs initialize

	if ( fullatom ) {
		disulf_distance_cutoff = disulf_ssdist_cutoff;
	} else {
		disulf_distance_cutoff = disulf_cendist_cutoff;
	}

	n_disulfides = 0;

//bs measure the distance between cys pairs in the known disulfides.

	for ( int ii = 1; ii <= n_known_disulfides; ++ii ) {
		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));

		if ( fullatom ) {
			get_disulf_dist_use_globl_crds(cys_1,cys_2,distance_separation);
		} else {
			distance_separation = cendist(cys_1,cys_2);
		}

//$$$		std::cout << "disulf_ii,cendist = " << ii <<
//$$$		 SS( distance_separation ) << std::endl;

		if ( distance_separation < disulf_distance_cutoff ) {
			++n_disulfides;
		}
	}

//bs set the appropriate global variable.

	if ( fullatom ) {
		n_disulfides_fa = n_disulfides;
	} else {
		n_disulfides_centroid = n_disulfides;
	}

}


//bs****************************************************************************
//bs Picks disulfides pairs based on sulfur-sulfur distance (fullatom = true)
//bs or based on centroid-centroid distance (fullatom = false)
//bs Used when find_disulf = true
//bs Resets value of globals: disulfide_pairs,disulf_partner_a and _b, and
//bs n_disulfides_fa or n_disulfides_centroid.

////////////////////////////////////////////////////////////////////////////////
/// @begin find_and_count_disulfides
///
/// @brief
///     Establish disulf connectivity when find_disulf = true
/// @detailed
///bs Picks disulfides pairs based on sulfur-sulfur distance (fullatom = true)
///bs or based on centroid-centroid distance (fullatom = false)
///bs Used when find_disulf = true
///bs Resets value of globals: disulfide_pairs,disulf_partner_a and _b, and
///bs n_disulfides_fa or n_disulfides_centroid.
///
/// @param[in]   fullatom - in - bool fa or not
///
/// @global_read
///     bool: fix_disulf
///     float: disulf_cendist_cutoff,disulf_ssdist_cutoff
///     block: count_cys_disulf
///     arrays: cendist,full_coord
/// @global_write
///     integer: n_disulfides_centroid,n_disulfides_fa
///     block: disulf_connectivity
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::find_and_count_disulfides( bool fullatom )
{
	using namespace disulfides::constants;
	using namespace runlevel_ns;

//bs local
	FArray1D_int n_neighbors( max_cysteines );
	 // number of other cys within cutoff distance, gets zeroed in algorithm.
	FArray2D_float neighbor_distances( max_cysteines, max_cysteines, 0.0f );
	 // sulfur-sulfur distances to neighbors within cutoff
//bs globals set
//bs disulfide_pairs(max_cysteines,max_cysteines) // 2-d array of disulf connectivity, =1 if paired, else = 0.
//bs disulf_partner_a,disulf_partner_b            // 1-d arrays of cys partners,
//bs                                              // gives cys index (1 for 1st cys in protein, 2 for 2nd, etc.)

//bs	int ii,jj;

//bs make sure we should be here.

	if ( !disulfides::options::find_disulf ) {
		std::cout << "Error, called find_and_count_disulfides but find_disulf = F" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//bs initialize
//bs     identify_cys();

	if ( num_cysteines < 2 ) {        // can't have a disulfide
		n_disulfides_fa = 0;
		n_disulfides_centroid = 0;
		if ( runlevel > chat ) std::cout << "less than 2 cys => no disulf" << std::endl;
		return;
	}

//bs Presum cys-cys pairs not known ahead of time.
//bs First get all cys-cys pairs that satisfy distance definition
//bs of a disulfide bond (neighbors).

	get_cys_neighbors(n_neighbors,neighbor_distances,fullatom);
//$$$cbs debugging
//$$$    for ( ii = 1; ii <= num_cysteines; ++ii ) {
//$$$       std::cout << "ii = " << ii << " n_neighbors = " << n_neighbors(ii) << std::endl;
//$$$       for ( jj = 1; jj <= num_cysteines; ++jj ) {
//$$$          if ( neighbor_distances(ii,jj) > 0 )
//$$$           std::cout << "jj = " << jj << " neighb_dist = " << neighbor_distances(ii,jj) << std::endl;
//$$$       }
//$$$    }
//$$$    std::cout << "calling pick_disulf_from_neighbors" << std::endl;
//$$$cbs end of debugging

//bs now pick which cys neighbors should be disulfides...try to get max
//bs possible but don't allow clusters.

	pick_disulf_from_neighbors(n_neighbors,neighbor_distances,fullatom);

//bs create global 1-d ordered lists of cys pairs from global 2-d array of disulfide connectivity

	put_disulfides_in_ordered_lists();

}

//bs****************************************************************************
//bs update cendist global array for all cys-cys pairs.

////////////////////////////////////////////////////////////////////////////////
/// @begin update_cyscys_cendist
///
/// @brief
///     updates the cendist array for cys-cys pairs, based on centroid array
/// @detailed
///
/// @global_read
///     array: centroid and  block: count_cys_disulf
/// @global_write
///     array: cendist
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::update_cyscys_cendist()
{
	using namespace cenlist_ns;
	using namespace scores;

	FArray2D_float const & centroid = *centroid_; // yab: misc removal
	int const & total_residue = *total_residue_; // yab: misc removal

	for ( int ii = 1; ii <= num_cysteines; ++ii ) {
		for ( int jj = ii+1; jj <= num_cysteines; ++jj ) {

			int cys_1 = cys(ii);
			int cys_2 = cys(jj);

			double tempdist = 0.0;
			for ( int kk = 1; kk <= 3; ++kk ) {
			 // loop over x,y,z, add up cent-cent distance squared
				double const cen_dif = centroid(kk,cys_1) - centroid(kk,cys_2);
				tempdist += cen_dif * cen_dif;
			}

			if ( tempdist == 0.0 ) {
				std::cout << "error in update_cyscys_cendist" << std::endl;
				std::cout << "tempdist = 0 between cys_1,cys_2: " <<
				 SS( cys_1 ) << SS( cys_2 ) << std::endl;
				std::cout << "cendist_score = " << SS( disulf_cendist_score ) << std::endl;
				std::cout << "bb_dih_score = " << SS( disulf_bb_dih_score ) << std::endl;
				std::cout << "cbdist_score = " << SS( disulf_cbdist_score ) << std::endl;
				std::cout << "cacbcb_score = " << SS( disulf_cacbcb_score ) << std::endl;
				std::cout << "cacbcbca_score = " << SS( disulf_cacbcbca_score ) << std::endl;

				for ( int kk = 1; kk <= total_residue; ++kk ) {
					std::cout << "residue,centroid x,y,z " << kk <<
					 SS( centroid(1,kk) ) <<
					 SS( centroid(2,kk) ) <<
					 SS( centroid(3,kk) ) << std::endl;
				}
				std::cout << "stopping" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			cendist(cys_1,cys_2) = tempdist;
			cendist(cys_2,cys_1) = tempdist;

		}
	}

}


//bs****************************************************************************
//bs Records disulfide cys-cys pairs in two ordered lists.
//bs Input (a global) is the 2-d array of disulf connectivity (i,j) = 1 if yes, 0 if no
//bs Output are the global arrays disulf_partner_a and disulf_partner_b.
//bs e.g. if disulf_partner_a(i) = 2 and disulf_partner_b(i) = 4
//bs then the ith disulfide bond has the second cys paired with the fourth.

////////////////////////////////////////////////////////////////////////////////
/// @begin put_disulfides_in_ordered_lists
///
/// @brief
///bs Records disulfide cys-cys pairs in two ordered lists.
/// @detailed
///bs Input (a global) is the 2-d array of disulf connectivity (i,j) = 1 if yes, 0 if no
///bs Output are the global arrays disulf_partner_a and disulf_partner_b.
///bs e.g. if disulf_partner_a(i) = 2 and disulf_partner_b(i) = 4
///bs then the ith disulfide bond has the second cys paired with the fourth.
/// @global_read
///     integer: num_cysteines; array: disulfide_pairs
/// @global_write
///     disulfide_partner_a,disulfide_partner_b
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::put_disulfides_in_ordered_lists()
{

//bs global input
//bs  disulfide_pairs(max_cysteines,max_cysteines) // 2-d array of disulf connectivity

//bs reset disulfide connectivity lists to 0.

	disulf_partner_a = 0;
	disulf_partner_b = 0;

//bs record the disulfide connectivity in two ordered lists --
//bs the global arrays disulf_partner_a and _b

	int disulf_counter = 0;
	for ( int ii = 1; ii <= num_cysteines; ++ii ) {
		for ( int jj = ii+1; jj <= num_cysteines; ++jj ) {
			if ( disulfide_pairs(ii,jj) == 1 ) {
				++disulf_counter;
				if ( ii < jj ) {                   // ordered lists
					disulf_partner_a(disulf_counter) = ii;
					disulf_partner_b(disulf_counter) = jj;
				} else {
					disulf_partner_a(disulf_counter) = jj;
					disulf_partner_b(disulf_counter) = ii;
				}
			}
		}
	}

}


//bs****************************************************************************
//bs returns the chi angle for two cys residues in a disulfide

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_chis
///
/// @brief
///     returns the chi angles for two cys residues in a disulfide
/// @detailed
///
/// @param[in]   coord1 - in - fa coords for cys 1
/// @param[in]   coord2 - in - fa coords for cys 2
/// @param[out]   chi_1 - out - chi angle for cys 1
/// @param[out]   chi_2 - out - chi angle for cys 2
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///     not currently used.
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_chis(
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	float & chi_1,
	float & chi_2
)
{
	using namespace param;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

//bs local
	FArray1D_float nitro( 3 );
	FArray1D_float calpha( 3 );
	FArray1D_float cbeta( 3 );
	FArray1D_float sulfur( 3 );
	int j;

	for ( j = 1; j <= 3; ++j ) {
		nitro(j) = coord1(j,1); // atom #1 is N
		calpha(j) = coord1(j,2); // atom #2 is calpha
		cbeta(j) = coord1(j,5);
		sulfur(j) = coord1(j,6);
	}
	chi_1 = dihedral(nitro,calpha,cbeta,sulfur);
	for ( j = 1; j <= 3; ++j ) {
		nitro(j) = coord2(j,1); // atom #1 is N
		calpha(j) = coord2(j,2); // atom #2 is calpha
		cbeta(j) = coord2(j,5);
		sulfur(j) = coord2(j,6);
	}
	chi_2 = dihedral(nitro,calpha,cbeta,sulfur);
}


//bs****************************************************************************
//bs  Finds all cys-cys neighbors within distance cutoff.
//bs  For fa, cutoff is on sulfur-sulfur dist, for centroid is on cendist.
//bs  Ignores cys neighbors with sequence separation < minimum.

////////////////////////////////////////////////////////////////////////////////
/// @begin get_cys_neighbors
///
/// @brief
///bs  Finds all cys-cys neighbors within distance cutoff.
/// @detailed
///bs  Finds all cys-cys neighbors within distance cutoff.
///bs  For fa, cutoff is on sulfur-sulfur dist, for centroid is on cendist.
///bs  Ignores cys neighbors with sequence separation < minimum.
///
/// @param[out]   n_neighbors - out - n neighb within cutoff for each cys
/// @param[out]   neighbor_distances - out - distance to each neighbor for each cys
/// @param[in]   fullatom - in - bool fa or not
///
/// @global_read
///     num_cysteines
///     disulf_cendist_cutoff,disulf_ssdist_cutoff
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_cys_neighbors(
	FArray1Da_int n_neighbors, // number of other cys within cutoff distance
	FArray2Da_float neighbor_distances, // distances to neighbors within cutoff
	bool fullatom
)
{
	using namespace cenlist_ns;
	using namespace disulfides::constants;

	n_neighbors.dimension( max_cysteines );
	neighbor_distances.dimension( max_cysteines, max_cysteines );

//bs initialize

	float disulf_distance_cutoff;
	if ( fullatom ) {
		disulf_distance_cutoff = disulf_ssdist_cutoff;
	} else {
		disulf_distance_cutoff = disulf_cendist_cutoff;
	}

	for ( int ii = 1; ii <= num_cysteines; ++ii ) {
		n_neighbors(ii) = 0;
		neighbor_distances(ii,ii) = 0.0;
		for ( int jj = ii+1; jj <= num_cysteines; ++jj ) {
			neighbor_distances(ii,jj) = 0.0;
			neighbor_distances(jj,ii) = 0.0;
		}
	}

//bs find cys neighbors within distance cutoff
	for ( int ii = 1; ii <= num_cysteines; ++ii ) {

		for ( int jj = ii+1; jj <= num_cysteines; ++jj ) {
			int cys_1 = cys(ii);
			int cys_2 = cys(jj);
			int sequence_separation = std::abs(cys_1-cys_2);
			float distance_separation;
			 // spatial separation of cysteines (either cendist or ssdist)

			if ( fullatom ) {
				get_disulf_dist_use_globl_crds(cys_1,cys_2,distance_separation);
				if ( distance_separation == 0 ) {
					std::cout << "get_cys_neighbors, dist between cys_1 and cys_2 is 0" << std::endl;
					std::cout << "cys_1,cys_2 = " << cys_1 << ' ' << cys_2 << std::endl;
					std::cout << "problem with fullcoord,stopping" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			} else {
				distance_separation = cendist(cys_1,cys_2);
//$$$				std::cout << "get_cys_neib,cys_1,cys_2,dist_sep " <<
//$$$				 cys_1 << ' ' << cys_2 << SS( distance_separation ) << std::endl;
				if ( distance_separation == 0 ) {
					std::cout << "get_cys_neighbors, dist between cys_1 and cys_2 is 0" << std::endl;
					std::cout << "cys_1,cys_2 = " << cys_1 << ' ' << cys_2 << std::endl;
					std::cout << "problem with cendist,stopping" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}

			if ( distance_separation < disulf_distance_cutoff ) {
//bs not allow disulfides to form between cys that are very near along chain.
				if ( sequence_separation >= disulf_min_sequence_separation ) {
					++n_neighbors(ii);
					++n_neighbors(jj);
					neighbor_distances(ii,jj) = distance_separation;
					neighbor_distances(jj,ii) = distance_separation;
				}
			}

		}
	}

}


//bs****************************************************************************
//bs Picks which cys-cys neighbors make disulfide bonds.
//bs If > 1 neighbor, take the closest one in space.
//bs (If fa, you could ask questions about the geometry to pick which one to take, but
//bs we don't do that right now.)

////////////////////////////////////////////////////////////////////////////////
/// @begin pick_disulf_from_neighbors
///
/// @brief
///bs Picks which cys-cys neighbors make disulfide bonds.
///
/// @detailed
///bs If > 1 neighbor, take the closest one in space.
///bs (If fa, you could ask questions about the geometry to pick which one to take, but
///bs we don't do that right now.)
///
/// @param[in,out]  n_neighbors - input/output - n neighb for each cys
/// @param[in]   neighbor_distances - in - neighb distances for each cys
/// @param[in]   fullatom - in - bool fa or not
///
/// @global_read
///     num_cysteines
/// @global_write
///     array: disulfide_pairs
///     integer: n_disulfides_centroid,n_disulfides_fa
/// @remarks
///     at the moment this is never called with fullatom = true
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::pick_disulf_from_neighbors(
	FArray1Da_int n_neighbors,
	 // number of other cys within cutoff distance,gets zeroed in algorithm.
	FArray2Da_float neighbor_distances, // distances to neighbors within cutoff
	bool fullatom
)
{
	using namespace disulfides::constants;

	n_neighbors.dimension( max_cysteines );
	neighbor_distances.dimension( max_cysteines, max_cysteines );

//bs local
	int ii,jj,kk;
//bs	int cys_1,cys_2;         // residue number of cysteine
	int min_neighbors;
	float min_neighbor_distance; // distance to nearest neighbor
	float disulf_distance_cutoff;
	int neighbors_to_check,closest_neighbor;
	int found; // dummy variable
//bs	int disulf_counter      // dummy variable
	int testvar; // dummy

//bs output
	int n_disulfides;

//bs global output
//bs	disulfide_pairs(max_cysteines,max_cysteines);   // 2-d array of disulf connectivity

//bs initialize
	closest_neighbor = 0;

	if ( fullatom ) {
		disulf_distance_cutoff = disulf_ssdist_cutoff;
	} else {
		disulf_distance_cutoff = disulf_cendist_cutoff;
	}

	n_disulfides = 0;

	for ( ii = 1; ii <= num_cysteines; ++ii ) {
		for ( jj = 1; jj <= num_cysteines; ++jj ) {
			disulfide_pairs(ii,jj) = 0;
		}
	}

//bs Try to make as many disulfides as possible from n_neighbors.
//bs Start with those pairs that each have only 1 neighbor.

//bs	std::cout << "***in pick_disulf_from_neighbors, num_cys = " <<
//bs	 num_cysteines << std::endl;

	min_neighbors = 1;
L50:
	found = 0;

	for ( ii = 1; ii <= num_cysteines; ++ii ) {
//bs		std::cout << "cys # " << ii << " n_neib = " << n_neighbors(ii) <<
//bs		 " min_neib = " << min_neighbors << std::endl;
		if ( n_neighbors(ii) == min_neighbors ) {
			found = 1;
			neighbors_to_check = n_neighbors(ii);
			n_neighbors(ii) = 0;
			 // if not make a disulf this pass, won't be in a disulfide
//bs			std::cout << "just set n_neib = 0 for cys # " << ii << std::endl;

//bs Find the closest neighbor that is not already in a disulfide.
//bs			std::cout << "neighbors_to_check = " << neighbors_to_check << std::endl;

//bs			for ( neighbors_checked = 1; neighbors_checked <= neighbors_to_check;
//bs			 ++neighbors_checked ) { // loop over neighbors
//bs				std::cout << "neighbors_checked = " << neighbors_checked << std::endl;

			min_neighbor_distance = disulf_distance_cutoff+1;
			 // unimportant -- just a large number
			for ( jj = 1; jj <= num_cysteines; ++jj ) { // find closest neighbor in space
//bs				std::cout << "neighbor_dist(" << ii << ',' << jj << ") = " <<
//bs				 neighbor_distances(ii,jj) << std::endl;
				if ( neighbor_distances(ii,jj) > 0 &&
				 neighbor_distances(ii,jj) < min_neighbor_distance ) {
					min_neighbor_distance = neighbor_distances(ii,jj);
					closest_neighbor = jj;
				}
			}
//$$$			std::cout << "closest_neighbor = " << closest_neighbor << std::endl;
//$$$			std::cout << "min_neib dist = " << min_neighbor_distance << std::endl;

			if ( closest_neighbor == 0 ) {
				std::cout << "ERROR: closest_neighbor = 0" << std::endl;
				std::cout << "cys # " << ii << std::endl;
				std::cout << "stopping" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			if ( n_neighbors(closest_neighbor) > 0 ) {
			 // the closest neighbor is available =>got a disulfide
				n_neighbors(closest_neighbor) = 0;
				 // make closest_neighbor unavailable for other disulf
				for ( kk = 1; kk <= num_cysteines; ++kk ) {
					neighbor_distances(kk,closest_neighbor) = 0;
					 // set to 0 so will not be closest next time
					neighbor_distances(closest_neighbor,kk) = 0;
					 // set to 0 so will not be closest next time
				}
				++n_disulfides;
				disulfide_pairs(ii,closest_neighbor) = 1;
				disulfide_pairs(closest_neighbor,ii) = 1;
//$$$				std::cout << "got a disulf between " << ii << " and " <<
//$$$				 closest_neighbor << std::endl;
				goto L50;
			} else if ( n_neighbors(closest_neighbor) == 0 ) {
			 // closest neighbor already disulfided =>find another
				for ( kk = 1; kk <= num_cysteines; ++kk ) {
					neighbor_distances(kk,closest_neighbor) = 0;
					 // set to 0 so will not be closest next time
					neighbor_distances(closest_neighbor,kk) = 0;
					 // set to 0 so will not be closest next time
				}
//$$$				std::cout << "closest already disulfided" << std::endl;
			} else {
				std::cout << "ERROR negative n_neighbors" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

//bs           }

		}
	}

	if ( found == 0 ) { // no cys had min_neighbors
//$$$		std::cout << "found = 0, increasing min_neighbors to " <<
//$$$		 min_neighbors+1 << std::endl;
		++min_neighbors;
	}

	for ( ii = 1; ii <= num_cysteines; ++ii ) {
		if ( n_neighbors(ii) >= min_neighbors ) {
//$$$			std::cout << "going to 50, cysteine # " << ii << std::endl;
			goto L50;
		}
	}

//bs check for errors.

	if ( n_disulfides > max_disulfides ) {
		std::cout << "Error::number of disulfides exceeds max_disulfides (" <<
		 max_disulfides << ")" << std::endl;
		std::cout << "stopping..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	testvar = 2*n_disulfides;
	if ( num_cysteines < testvar ) {
		std::cout << "ERROR...STOPPING:: " <<
		 "num_cysteines too small for that many disulf" << std::endl;
		std::cout << "n_cys,n_disulf = " << num_cysteines << ' ' << n_disulfides <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//bs set the value of the appropriate global.

	if ( fullatom ) {
		n_disulfides_fa = n_disulfides;
	} else {
		n_disulfides_centroid = n_disulfides;
	}

}

//------------------------------------------------------------------------------
//bs Returns the index no. for this cys,cys pair
//bs in the global disulf_partner_a, disulf_partner_b arrays
//bs Returns 0 if not a disulfide.

int
Disulfides::which_disulf_pair(
	int res1,
	int res2
)
{

	int which_disulf_pair = 0; // Return value

//jjh initializing
	int n_disulfides = 0;

	if ( disulfides::options::fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( disulfides::options::find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	for ( int ii = 1; ii <= n_disulfides; ++ii ) {
		if ( ( res1 == cys(disulf_partner_a(ii)) ) &&
		 ( res2 == cys(disulf_partner_b(ii)) ) ) {
			which_disulf_pair = ii;
			return which_disulf_pair;
		} else if ( ( res2 == cys(disulf_partner_a(ii)) ) &&
		 /* allow for the possibility that res1 > res2 */
		 ( res1 == cys(disulf_partner_b(ii)) ) ) {
			which_disulf_pair = ii;
			return which_disulf_pair;
		}
	}

	return which_disulf_pair;
}


//------------------------------------------------------------------------------
//bs Disulfide filter.
//bs
//bs if fix_disulf, then all known disulfides must have cendist < native max
//bs if find_disulf and num_cys is even, then num_cys/2 disulfides must have
//bs cendist < native max
//bs if find_disulf and num_cys is odd, then (num_cys-1)/2.

//bs 4/15/03 change for larger proteins
//bs if total_res > 140, require 1 fewer disulfide made.
//bs if total_res > 140 and res > 3, require 2 fewer disulfides.

////////////////////////////////////////////////////////////////////////////////
/// @begin disulf_filter
///
/// @brief
///     returns T if decoy passes the filter, F otherwise
/// @detailed
///bs if fix_disulf, then all known disulfides must have cendist < native max
///bs if find_disulf and num_cys is even, then num_cys/2 disulfides must have
///bs cendist < native max
///bs if find_disulf and num_cys is odd, then (num_cys-1)/2.
///bs in any case, not require more than 3 disulfides.
/// @return
///
/// @global_read
///     logicals:fix_disulf,find_disulf
///     block: count_cys_disulf
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::disulf_filter()
{
	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace files_paths;

	bool disulf_filter = true; // Return value

	if ( !use_filter(disulfide_type) ) return disulf_filter;

//bs can't have a disulf if <2 cys
	if ( num_cysteines < 2 ) {
//mj        disulf_filter = false    // this would cause every protein with less than two CYS to fail the filter
		return disulf_filter;
	}

//bs decide n_disulf_required for different cases:

	int n_disulf_required = 0;

	if ( disulfides::options::fix_disulf ) {
		n_disulf_required = n_known_disulfides; // require all that are read in.
	} else if ( disulfides::options::find_disulf ) {
		if ( even_integer(num_cysteines) ) {
			n_disulf_required = num_cysteines/2;
		} else {
			n_disulf_required = (num_cysteines-1)/2;
		}
	}

	if ( n_disulf_required > 4 ) {
		n_disulf_required = 3;
	} else if ( n_disulf_required == 3 || n_disulf_required == 4 ) {
		n_disulf_required = 2;
	} else {
		n_disulf_required = 1;
	}


//bs reject the decoy if not enough disulfides
	if ( n_disulf_cent_natcut < n_disulf_required ) {
		disulf_filter = false;
	}

	return disulf_filter;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_disulf_cendist
///
/// @brief
///     scores a decoy based on cendist's of disulfides
/// @detailed
///bs (1) counts the number of disulf with cendist below the native cutoff
///bs     (the global n_disulf_cent_natcut)
///bs (2) computes a score based on the centroid-centroid distances between
///bs cysteines in disulf bonds.
///bs     (the global disulf_cendist_score)
///
/// @global_read
///     blocks: count_cys_disulf,disulf_connectivity
/// @global_write
///     block: count_cys_disulf, float: disulf_cendist_score
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_disulf_cendist()
{

	using namespace cenlist_ns;
	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace scores;

	int ii;
	int cys_1,cys_2; // residue #s for cysteines
	int n_disulfides;
	float tempdist;
//jjh commenting out since these variables are not referenced
//       float slope,intercept;  // score = slope*cendist + intercept
//$$$    int const nbins = 45;
//$$$    float const bin_size = 0.5; // in ang^2, first bin is 0-0.5
	int const nbins = { 300 }; // # of bins in cendist native histogram
	float const bin_size = { 0.125 };
	 // bin width for cendist native histogram // in ang^2, first bin is 0-0.125
	float const max_bin_val = { nbins*bin_size };
//jjh more unreferenced variables
//   FArray1D_float cendist_native_histogram(nbins);
//   FArray1D_float cendist_foldcst_score(nbins),cendist_foldcst_score_sm(nbins);
//   FArray1D_float cendist_cnrlx_score(nbins);
	static FArray1D_float const cendist_score_interp300( nbins,
	 cendist_score_interp300_initializer );
	FArray1D_float current_score( nbins );
	int bin;

//bs relevant to constraints
	int sequence_sep;
	int noe_stage;
	bool constraints_exist;

//bs here is the normalized histogram of cendist for natives
//bs tell how many natives and where from...
//bs the bins were every 0.5 ang^2.

//$$$    data cendist_native_histogram
//$$$     #       /0,0,0,0,0.0197,
//$$$     #       0.0709,0.157,0.327,0.642,1.0,
//$$$     #       0.839,0.772,0.740,0.697,0.614,
//$$$     #       0.614,0.547,0.591,0.469,0.437,
//$$$     #       0.516,0.335,0.284,0.284,0.232,
//$$$     #       0.146,0.118,0.0906,0.102,0.102,
//$$$     #       0.059,0.075,0.051,0.071,0.039,
//$$$     #       0.047,0.016,0.032,0.004,0.016,
//$$$     #       0.008,0.028,0.016,0.012,0.0/
//$$$
//$$$        data cendist_foldcst_score
//$$$     #        /5.000,   5.000,   5.000,   5.000,   5.000,
//$$$     #          2.089,  -0.120,  -0.805,  -1.294,  -1.848,
//$$$     #         -1.598,  -1.552,  -1.307,  -1.259,  -0.893,
//$$$     #         -1.002,  -0.723,  -0.801,  -0.487,  -0.517,
//$$$     #         -0.423,   0.067,   0.215,   0.283,   0.454,
//$$$     #          1.008,   1.268,   2.029,   1.522,   1.800,
//$$$     #          2.892,   1.886,   1.651,   1.779,   2.206,
//$$$     #          2.482,   3.630,   3.787,   5.000,   5.000,
//$$$     #          5.000,   5.000,   5.000,   5.000,   5.000/
//$$$
//$$$        data cendist_foldcst_score_sm
//$$$     #         /5.000,   5.000,   5.000,   5.000,   5.000,
//$$$     #          2.089,  -0.120,  -0.805,  -1.294,  -1.732,
//$$$     #         -1.619,  -1.500,  -1.375,  -1.244,  -1.107,
//$$$     #         -0.965,  -0.816,  -0.661,  -0.500,  -0.333,
//$$$     #         -0.160,   0.019,   0.204,   0.395,   0.592,
//$$$     #          0.795,   1.004,   1.220,   1.441,   1.668,
//$$$     #          1.901,   2.140,   2.385,   2.636,   2.893,
//$$$     #          3.157,   3.426,   3.701,   5.000,   5.000,
//$$$     #          5.000,   5.000,   5.000,   5.000,   5.000/
//$$$
//$$$        data cendist_cnrlx_score
//$$$     #         /100.000, 100.000, 100.000, 100.000, 100.000,
//$$$     #          1.593,  -0.738,  -1.287,  -1.520,  -0.987,
//$$$     #         -1.250,  -1.292,  -0.947,  -0.762,  -0.676,
//$$$     #         -0.590,  -0.606,  -0.291,  -0.416,  -0.592,
//$$$     #          0.104,   0.051,   0.128,   0.364,   0.479,
//$$$     #          0.851,   1.107,   1.724,   1.425,   1.751,
//$$$     #          2.614,   1.771,   1.348,   1.762,   1.925,
//$$$     #          2.425,   3.368,   3.678, 100.000, 100.000,
//$$$     #        100.000, 100.000, 100.000, 100.000, 100.000/


	noe_stage = 9999;
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	if ( constraints_exist ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();

//bs setup track_disulf for the cen row
	for ( ii = 1; ii <= max_disulfides; ++ii ) {
		track_disulf_cen_cb(1,ii) = 0;
	}

//bs decide which cendist score to use.
//bs use the native histogram if you are doing final score of a structure for output

	for ( ii = 1; ii <= nbins; ++ii ) {
		current_score(ii) = cendist_score_interp300(ii);
		 // score is -ln(P(natives)/P(decoys))
	}

//bs now loop over the disulfides and get a total score
	disulf_cendist_score = 0.0;
	n_disulf_cent_natcut = 0;

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	if ( n_disulfides == 0 ) return;

	for ( ii = 1; ii <= n_disulfides; ++ii ) {
		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));
		tempdist = cendist(cys_1,cys_2);
		sequence_sep = std::abs(cys_2-cys_1);

		if ( tempdist <= 0.0 ) {
			std::cout << "error score_disulf_cendist, tempdist = " << tempdist <<
			 std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

//bs skip to next iteration of the loop if sequence_sep too large.
//bs this can only occur if running with constraints.
		if ( noe_stage >= sequence_sep ) {
			if ( tempdist > max_bin_val ) {
				disulf_cendist_score += current_score(nbins);
			} else if ( tempdist <= max_bin_val ) {
				bin = static_cast< int >(tempdist/bin_size+1.);
				disulf_cendist_score += current_score(bin);
				if ( tempdist >= min_native_disulf_cendist &&
				 tempdist < max_native_disulf_cendist ) {
					++n_disulf_cent_natcut;
					 // counting # of disulf w/cendist within native bounds
					track_disulf_cen_cb(1,ii) = 1;
					 // record that this particular disulf satisfies cen criteria
				}
			}
		}    // noe_stage check
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_disulf_cbdist
///
/// @brief
///     scores a decoy based on cbdist of disulfides
/// @detailed
///bs (1) counts the number of disulf with cbdist between 10 and 20
///
///bs (2) computes a score based on the cb-cb distances between
///bs cysteines in disulf bonds.
///bs     (the global disulf_cbdist_score)
///
/// @global_read
///     blocks: count_cys_disulf,disulf_connectivityc
/// @global_write
///     block: count_cys_disulf, float: disulf_cbdist_score
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_disulf_cbdist()
{

	using namespace cb_score_3gauss_fitparams;
	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace scores;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

	int ii,k;
	int cys_1,cys_2; // residue #s for cysteines
	int n_disulfides;
	float cbdist;
	int const nbins = { 50 }; // # of bins in cbdist native histogram
//	float const bin_size = { 0.5 };
	 // bin width for cbdist native histogram // in ang^2, first bin is 0-0.5
	static FArray1D_float const cbdist_native_histogram( nbins,
	 cbdist_native_histogram_initializer );
	static FArray1D_float const cbdist_foldcst_score( nbins,
	 cbdist_foldcst_score_initializer );
	static FArray1D_float const cbdist_cnrlx_score( nbins,
	 cbdist_cnrlx_score_initializer );

//jjh currently not referenced
//	FArray1D_float current_score( nbins );
//	int bin;

//bs relevant to constraints
	int sequence_sep;
	int noe_stage;
	bool constraints_exist;

//bs here is the normalized histogram of (cbetas dist)^2 for natives
//bs histo of xray natives with resolution =< 2.1 angstroms
//bs the bins were every 0.5 ang^2.

//bs Triple gauss fit...

	noe_stage = 9999; // default so all disulf will be scored
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	if ( constraints_exist ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();

//bs setup track_disulf for the cb row
	for ( ii = 1; ii <= max_disulfides; ++ii ) {
		track_disulf_cen_cb(2,ii) = 0; // is this one within bounds?
		cb_scale_factor(ii) = 0.0;
	}

//bs now loop over the disulfides and get a total score

	disulf_cbdist_score = 0.0;
	n_disulf_cb_natcut = 0;

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	if ( n_disulfides == 0 ) return;

	for ( ii = 1; ii <= n_disulfides; ++ii ) {
		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));
		sequence_sep = std::abs(cys_2-cys_1);

//bs skip to next iteration of the loop if sequence_sep too large.
//bs can only happen if running with constraints.

		if ( noe_stage >= sequence_sep ) {

//bs get the dist^2 between cbetas

			cbdist = 0.0;
			for ( k = 1; k <= 3; ++k ) {
				float const Edif = Eposition(k,3,cys_2) - Eposition(k,3,cys_1);
				cbdist += Edif * Edif;
			}

			if ( cbdist <= 0.0 ) {
				std::cout << "error score_disulf_cbdist, cbdist = " << cbdist << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

//bs scale factor used to scale the angular scores
//bs linear in the cbdist between 22 and 400
//bs =1 at 22 ang^2 and down to 10 ang^2
//bs =0 at 400 ang^2 and above
			if ( cbdist < max_native_disulf_cbdist && cbdist > min_native_disulf_cbdist ) {
				cb_scale_factor(ii) = 1.0;
			} else if ( cbdist < min_native_disulf_cbdist || cbdist >= 400. ) {
				cb_scale_factor(ii) = 0.0;
			} else {
				cb_scale_factor(ii) = 1.0582 - 0.0026455*cbdist;
			}

//$$$           if ( cbdist > max_native_disulf_cbdist ) {
//$$$               disulf_cbdist_score += current_score(nbins);
//$$$
//$$$               if ( cys_1 == 13 && cys_2 == 37 ) {
//$$$                  std::cout <<
//$$$                   "cb " << I( 3, cys_1 ) << ' ' << I( 3, cys_2 ) << ' ' <<
//$$$									  F( 8, 5, cbdist ) << ' ' << F( 8, 5, current_score(nbins) ) << std::endl;
//$$$                  for ( k = 1; k <= 3; ++k ) {
//$$$                     std::cout << "13  37 " << k << ' ' << cys1coord(k,5) << ' ' << cys2coord(k,5) << std::endl;
//$$$                  }
//$$$               }
//$$$
//$$$           } else if ( cbdist <= max_native_disulf_cbdist ) {
//$$$               bin = static_cast< int >(cbdist/bin_size+1.)
//$$$               disulf_cbdist_score += current_score(bin)
//$$$               if ( cys_1 == 13 && cys_2 == 37 ) {
//$$$                  std::cout <<
//$$$                   "cb " << I( 3, cys_1 ) << ' ' << I( 3, cys_2 ) << ' ' <<
//$$$									  F( 8, 5, cbdist ) << ' ' << F( 8, 5, current_score(nbins) ) << std::endl;
//$$$                  for ( k = 1; k <= 3; ++k ) {
//$$$                     std::cout << "13  37 " << k << ' ' << cys1coord(k,5) << ' ' << cys2coord(k,5) << std::endl;
//$$$                  }
//$$$               }

			disulf_cbdist_score += triple_gauss(cbdist,y_0,A_1,A_2,A_3,x0_1,x0_2,x0_3,
			 xwidth_1,xwidth_2,xwidth_3);

			if ( cbdist >= min_native_disulf_cbdist ) {
//bs // counting # of disulf w/cbdist within native bounds
				++n_disulf_cb_natcut;
				track_disulf_cen_cb(2,ii) = 1;
			}
//$$$  }

		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_disulf_cacbcb
///
/// @brief
///     scores a decoy based on cacbcb "bond" angle of disulfides
/// @detailed
///bs Computes a score based on the ca-cb-cb angles across
///bs disulf bonds (the global disulf_cacbcb_score).
///
/// @global_read
///     blocks: count_cys_disulf,disulf_connectivityc
/// @global_write
///     float: disulf_cacbcb_score
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_disulf_cacbcb()
{

	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace scores;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

	int ii,k;
	int cys_1,cys_2; // residue #s for cysteines
	int n_disulfides;
	FArray2D_float cys1coord( 3, 5 );
	FArray2D_float cys2coord( 3, 5 );
	float disulf_cacbcb_ang_1,disulf_cacbcb_ang_2;
	int const nbins = { 100 }; // # of bins in cacbcb histograms
	float const bin_size = { 3.6 };
	 // bin width for cacbcb histograms // in ang^2, first bin is -180 to -176.4
	static FArray1D_float const cacbcb_native_histogram( nbins,
	 cacbcb_native_histogram_initializer );
	static FArray1D_float const cacbcb_cnrlx_score( nbins,
	 cacbcb_cnrlx_score_initializer );
	FArray1D_float current_score( nbins );
	int bin_1,bin_2;

//bs relevant to constraints
	int sequence_sep;
	int noe_stage;
	bool constraints_exist;

	noe_stage = 9999; // default so all disulf will be scored
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	if ( constraints_exist ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();

//bs decide which cbdist score to use.

	for ( ii = 1; ii <= nbins; ++ii ) {
		current_score(ii) = cacbcb_cnrlx_score(ii);
		 // score is -ln(P(natives)/P(decoys))
	}


//bs now loop over the disulfides and get a total score

	disulf_cacbcb_score = 0.0;

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	if ( n_disulfides == 0 ) return;

	for ( ii = 1; ii <= n_disulfides; ++ii ) {
//bs		if ( track_disulf_cen_cb(1,ii) == 1 &&
//bs		 track_disulf_cen_cb(2,ii) == 1 ) {
		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));
		sequence_sep = std::abs(cys_2-cys_1);

//bs skip to next iteration of the loop if sequence_sep too large.
//bs this can only occur if running with constraints.

		if ( noe_stage >= sequence_sep ) {

//bs get the cbeta coords for both cys...
//bs cys1coord needs to have the full_coord atom ordering (N,Ca,C,O,Cb)

			for ( k = 1; k <= 3; ++k ) {
				cys1coord(k,1) = Eposition(k,1,cys_1);
				cys2coord(k,1) = Eposition(k,1,cys_2);
				cys1coord(k,2) = Eposition(k,2,cys_1);
				cys2coord(k,2) = Eposition(k,2,cys_2);
				cys1coord(k,3) = Eposition(k,4,cys_1);
				cys2coord(k,3) = Eposition(k,4,cys_2);
				cys1coord(k,4) = Eposition(k,5,cys_1);
				cys2coord(k,4) = Eposition(k,5,cys_2);
				cys1coord(k,5) = Eposition(k,3,cys_1);
				cys2coord(k,5) = Eposition(k,3,cys_2);
			}

			get_disulf_cacbcb_angles(cys1coord,cys2coord,disulf_cacbcb_ang_1,
			disulf_cacbcb_ang_2);

//bs gross error check
			if ( std::abs(disulf_cacbcb_ang_1) > 180. || std::abs(disulf_cacbcb_ang_2) > 180. ) {
				std::cout << "error score_disulf_cacbcb cacbcb_1,cacbcb_2 = " <<
				 disulf_cacbcb_ang_1 << disulf_cacbcb_ang_2 << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

//bs safeguard against rounding errors that lead to
//bs subscript out of range.
			if ( disulf_cacbcb_ang_1 > 179.9999 ) disulf_cacbcb_ang_1 = -180.;
			if ( disulf_cacbcb_ang_2 > 179.9999 ) disulf_cacbcb_ang_2 = -180.;


//bs -180 maps to bin 1, +179.9 maps to bin 100
			bin_1 = static_cast< int >((180.+disulf_cacbcb_ang_1)/bin_size+1.);
			bin_2 = static_cast< int >((180.+disulf_cacbcb_ang_2)/bin_size+1.);

//bs scale the score of each individual disulfide by a factor that depends
//bs on cbdist.
			disulf_cacbcb_score += (current_score(bin_1) + current_score(bin_2))*0.5*
			cb_scale_factor(ii);

//bs           }   // track_disulf check
		}
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_disulf_cacbcbca
///
/// @brief
///     scores a decoy based on cacbcbca dihedral angle of disulfides
/// @detailed
///bs Computes a score based on the ca-cb-cb angles across
///bs disulf bonds (the global disulf_cacbcbca_score).
///
/// @global_read
///     blocks: count_cys_disulf,disulf_connectivity
/// @global_write
///     float: disulf_cacbcbca_score
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_disulf_cacbcbca()
{

	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace scores;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

	int cys_1,cys_2; // residue #s for cysteines
	int n_disulfides;
	FArray2D_float cys1coord( 3, 5 );
	FArray2D_float cys2coord( 3, 5 );
	float disulf_cacbcbca_dih;
	int const nbins = { 100 }; // # of bins in cacbcbca histograms
	float const bin_size = { 3.6 };
	 // bin width for cacbcbca histograms // in ang^2, first bin is -180 to -176.4
	static FArray1D_float const cacbcbca_native_histogram_sm( nbins,
	 cacbcbca_native_histogram_sm_initializer );
	static FArray1D_float const cacbcbca_native_histogram_unsm( nbins,
	 cacbcbca_native_histogram_unsm_initializer );
	static FArray1D_float const cacbcbca_foldcst_score_sm( nbins,
	 cacbcbca_foldcst_score_sm_initializer );
	static FArray1D_float const cacbcbca_cnrlx_score( nbins,
	 cacbcbca_cnrlx_score_initializer );
	static FArray1D_float const cacbcbca_cnrlx_score_sm( nbins,
	 cacbcbca_cnrlx_score_sm_initializer );
	FArray1D_float current_score( nbins );
	int bin;


//bs relevant to constraints
	int sequence_sep;
	int noe_stage;
	bool constraints_exist;

	noe_stage = 9999; // default so all disulf will be scored
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	if ( constraints_exist ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();

//bs decide which score to use.

	for ( int ii = 1; ii <= nbins; ++ii ) {
		current_score(ii) = cacbcbca_foldcst_score_sm(ii);
		 // score is -ln(P(natives)/P(decoys))
	}

//bs now loop over the disulfides and get a total score

	disulf_cacbcbca_score = 0.0;

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	if ( n_disulfides == 0 ) return;


	for ( int ii = 1; ii <= n_disulfides; ++ii ) {
//bs		if ( track_disulf_cen_cb(1,ii) == 1 &&
//bs		 track_disulf_cen_cb(2,ii) == 1 ) {
		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));
		sequence_sep = std::abs(cys_2-cys_1);

//bs skip to next iteration of the loop if sequence_sep too large.
//bs this can only occur if running with constraints.

		if ( noe_stage >= sequence_sep ) {

//bs get the backbone coords for both cys...
//bs cys1coord needs to have the full_coord atom ordering (N,Ca,C,O,Cb)


			for ( int k = 1; k <= 3; ++k ) {
				cys1coord(k,1) = Eposition(k,1,cys_1);
				cys2coord(k,1) = Eposition(k,1,cys_2);
				cys1coord(k,2) = Eposition(k,2,cys_1);
				cys2coord(k,2) = Eposition(k,2,cys_2);
				cys1coord(k,3) = Eposition(k,4,cys_1);
				cys2coord(k,3) = Eposition(k,4,cys_2);
				cys1coord(k,4) = Eposition(k,5,cys_1);
				cys2coord(k,4) = Eposition(k,5,cys_2);
				cys1coord(k,5) = Eposition(k,3,cys_1);
				cys2coord(k,5) = Eposition(k,3,cys_2);
			}

			get_disulf_cacbcbca_dih(cys1coord,cys2coord,disulf_cacbcbca_dih);

//bs gross error check
			if ( std::abs(disulf_cacbcbca_dih) > 180 ) {
				std::cout << "error score_disulf_cacbcbca, cacbcbca = " <<
				 disulf_cacbcbca_dih << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

//bs safeguard against rounding errors that lead to
//bs subscript out of range.
			if ( disulf_cacbcbca_dih > 179.9999 ) disulf_cacbcbca_dih = -180.0;


//bs -180 maps to bin 1, +179.9 maps to bin 100
			bin = static_cast< int >((disulf_cacbcbca_dih+180.)/bin_size+1.);
			disulf_cacbcbca_score += current_score(bin)*cb_scale_factor(ii);

//bs           }  // track_disulf check
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_disulf_bb_dih
///
/// @brief
///     scores a protein based on a backbone dihedral angle
///     between 2 cys in each disulfide: N1-CA1-CA2-CPrime2
/// @detailed
///bs score all disulfides in a protein based on a backbone dihedral angle
///bs between 2 cys: N1-CA1-CA2-CPrime2
///bs sets value of global disulf_bb_dih_score
///
/// @global_read
///     arrays: Eposition
///     blocks: count_cys_disulf
/// @global_write
///     disulf_bb_dih_score
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_disulf_bb_dih()
{
	using namespace disulfides::options;
	using namespace disulfides::constants;
	using namespace scores;

	FArray3Dp_float const & Eposition = *Eposition_;

//bs local
	int ii,k;
	int cys_1,cys_2; // residue # of cysteines
	int n_disulfides;
	FArray2D_float cys1coord( 3, 5 );
	FArray2D_float cys2coord( 3, 5 );
	float disulf_bb_dih; // the backbone dihedral angle N1-CA1-CA2-CPrime2

//jjh not referenced
//	float this_bb_dih_sc;      // bb_dih score for one disulf bond


	int const nbins = { 100 }; // # of bins in bbdih histograms
	float const bin_size = { 3.6 };
	 // bin width for bbdih histograms // in ang^2, first bin is 0 to 3.6
	static FArray1D_float const bb_dih_cnrlx_score( nbins,
	 bb_dih_cnrlx_score_initializer );
	static FArray1D_float const bb_dih_cnrlx_score_sm( nbins,
	 bb_dih_cnrlx_score_sm_initializer );
	static FArray1D_float const bb_dih_foldcst_score_sm( nbins,
	 bb_dih_foldcst_score_sm_initializer );
	static FArray1D_float const bb_dih_native_histogram_sm( nbins,
	 bb_dih_native_histogram_sm_initializer );
	static FArray1D_float const bb_dih_native_histogram_unsm( nbins,
	 bb_dih_native_histogram_unsm_initializer );
	FArray1D_float current_score( nbins );
	int bin;

//jjh not referenced
//        float x_temp;

//bs relevant to constraints
	int sequence_sep;
	int noe_stage;
	bool constraints_exist;

//bs native histogram of bb_dih (smoothed and unsmoothed)

//bs histogram of cenrlx score...this is -ln(Pnatives/Pdecoys)
//bs this is a smoothed version of the score...


	noe_stage = 9999; // default so all disulf will be scored
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	if ( constraints_exist ) noe_stage = classical_constraints::BOUNDARY::get_max_seqSep();

//bs decide which score to use.

	for ( ii = 1; ii <= nbins; ++ii ) {
		current_score(ii) = bb_dih_foldcst_score_sm(ii);
		 // score is -ln(P(natives)/P(decoys))
	}

//bs now loop over the disulfides and get a total score

	disulf_bb_dih_score = 0.0;

//jjh initializing
	n_disulfides = 0;

	if ( fix_disulf ) {
		n_disulfides = n_known_disulfides;
	} else if ( find_disulf ) {
		n_disulfides = n_disulfides_centroid;
	}

	if ( n_disulfides == 0 ) return;


	for ( ii = 1; ii <= n_disulfides; ++ii ) {
//bs		if ( track_disulf_cen_cb(1,ii) == 1 &&
//bs		 track_disulf_cen_cb(2,ii) == 1 ) {

		cys_1 = cys(disulf_partner_a(ii));
		cys_2 = cys(disulf_partner_b(ii));
		sequence_sep = std::abs(cys_2-cys_1);

//bs skip to next iteration of the loop if sequence_sep too large.
//bs this can only occur if running with constraints.
		if ( noe_stage >= sequence_sep ) {


//bs get coords of backbone atoms for cys residues.
//bs cys1coord needs to have the full_coord atom ordering (N,Ca,C,O,Cb)
//bs but Eposition has diff ordering...(N,Ca,Cb,C,O)...so convert
			for ( k = 1; k <= 3; ++k ) {
				cys1coord(k,1) = Eposition(k,1,cys_1);
				cys2coord(k,1) = Eposition(k,1,cys_2);
				cys1coord(k,2) = Eposition(k,2,cys_1);
				cys2coord(k,2) = Eposition(k,2,cys_2);
				cys1coord(k,3) = Eposition(k,4,cys_1);
				cys2coord(k,3) = Eposition(k,4,cys_2);
				cys1coord(k,4) = Eposition(k,5,cys_1);
				cys2coord(k,4) = Eposition(k,5,cys_2);
				cys1coord(k,5) = Eposition(k,3,cys_1);
				cys2coord(k,5) = Eposition(k,3,cys_2);
			}

//bs get the disulf backbone dihedral
//bs this will be returned in degrees between 0 and 360 (not -180to+180)!!
			get_disulf_bb_dih(cys1coord,cys2coord,disulf_bb_dih);


			if ( disulf_bb_dih < 0. || disulf_bb_dih > 360. ) {
				std::cout << "error score_disulf_bb_dih, bb_dih = " <<
				 disulf_bb_dih << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

//bs safeguard against rounding errors that lead to
//bs subscript out of range.

			if ( disulf_bb_dih > 359.9999 ) disulf_bb_dih = 0.0;

			bin = static_cast< int >(disulf_bb_dih/bin_size+1.);

			if ( bin > 100 ) {
				std::cout << "ERROR: disulf_bb_dih = " <<
				 F( 15, 8, disulf_bb_dih ) << std::endl;
			}

			disulf_bb_dih_score += current_score(bin)*cb_scale_factor(ii);


//bs           }                // track_disulf check
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin score_this_disulfide_fa
///
/// @brief
///bs Scores a single cys-cys pair as a disulfide bond in a fullatom structures.
///
/// @detailed
///bs Assumes that disulf_partner_a and disulf_partner_b are pre-set.
///bs Also assumes that res1,res2 are cys and listed as disulf.
///bs If fix_disulf, returns a score for any of n_known_disulfides.
///bs If find_disulf, returns a score for any of n_disulfides_centroid.
///
/// @param[in]   coord1 - in - fa coords of cys 1
/// @param[in]   coord2 - in - fa coords of cys 2
/// @param[out]   distance_score_this_disulfide - out -
/// @param[out]   csangles_score_this_disulfide - out -
/// @param[out]   dihedral_score_this_disulfide - out -
/// @param[out]   ca_dihedral_sc_this_disulf - out -
/// @param[out]   truefalse_fa_disulf - out - bool is this a disulfide?
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///     All the parameters for fullatom disulfide score are listed in this function.
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::score_this_disulfide_fa(
	FArray2Da_float coord1, // xyz coords of cys 1
	FArray2Da_float coord2, // xyz coords of cys 2
	float & distance_score_this_disulfide,
	float & csangles_score_this_disulfide,
	float & dihedral_score_this_disulfide,
	float & ca_dihedral_sc_this_disulf,
	bool & truefalse_fa_disulf
)
{
	using namespace disulfides::constants;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

//bs Rationale of disulfide fullatom scores:
//bs All scores lie between 0 and -1.
//bs All scores are gaussians, with center and width similar to that found in natives.
//bs In the case of c-s-s bond angle score (called csangles), there are two gaussians,
//bs one for positive angle and one for negative.
//bs csangles and dihedral scores are scaled by distance_score for that disulfide.
//bs (so the angular terms contribute only if the s-s distance is close to the optimum)


//bs input
//bs	int res1,res2     // residue numbers for two cys residues

//bs local

	float ssdist; // dist between cys sulfurs
	float csang_1,csang_2; // Cb-S-S bond angles
//jjh not referenced
//	float delta_csang_1,delta_csang_2;
//	float delta_dih_ang;
//	float delta_ss_dist;
	float delta_cadih_ang_1,delta_cadih_ang_2;
	float dihed; // dihedral (torsion) angle, cbeta1-s1-s2-cbeta2
	float disulf_ca_dihedral_angle_1;
	 // dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	float disulf_ca_dihedral_angle_2;
	 // dihedral (torsion) angle, calpha2-cbeta2-s2-s1
	float scale_factor;
	 // scales the angular and dihedral scores based on s-s distance

//bs int n_disulfides;

//bs 2-dimensional score...csang vs ssdist

//jjh data initializations for these (above) variables has been
//jjh moved down to the other DATA statements

//jjh not referenced
//	double csang_ssdist_2Dscore_1,csang_ssdist_2Dscore_2;
//	double csang_ssdist_2Dscore;

//bs nbins and bin_size for scores that are -ln()
	float const ssdist_bin_size = { 0.005 };
	 // for interp300 this is 0.005, others is 0.02
	int const nbins_ssdist = { 300 };
	 // for interp300 this is 300, others it is 100.
	float const csang_bin_size = { 1.0 };
	int const nbins_csang = { 100 };
	float const dihed_bin_size = { 3.6 };
	int const nbins_dihed = { 100 };
	int bin,bin_1,bin_2;

//bs scores that are -ln()
//jjh not referenced
//	FArray1D_float farlx_ssdist_score( nbins_ssdist );
//	FArray1D_float farlx_ssdist_score_2( nbins_ssdist );
	static FArray1D_float const farlx_csang_score( nbins_csang,
	 farlx_csang_score_initializer );
	static FArray1D_float const farlx_csang_score_2( nbins_csang,
	 farlx_csang_score_2_initializer );
	static FArray1D_float const farlx_dihed_score( nbins_dihed,
	 farlx_dihed_score_initializer );
	static FArray1D_float const farlx_dihed_score_2( nbins_dihed,
	 farlx_dihed_score_2_initializer );
	static FArray1D_float const farlx_ssdist_sc_interp300( nbins_ssdist,
	 farlx_ssdist_sc_interp300_initializer );

//bs min and max ranges covered by the -ln() scores (e.g. ssdist score values hard-wired only for 1.5 < ssdist < 3.0)
	float const csang_min_range = { 50.0 };
	float const csang_max_range = { 150.0 };
	float const dihed_min_range = { -180.0 };
	//notused-jjg//float const dihed_max_range = { 180.0 };
	float const ssdist_min_range = { 1.5 };
	float const ssdist_max_range = { ssdist_min_range + nbins_ssdist*
	 ssdist_bin_size };


//-----------------------------------------
// physical parameters of disulfide bonds
//-----------------------------------------

//jjg--commented these out to avoid compiler warnings about unused variables
//5/21/04
//
//	float const ideal_ss_dist_in_disulfide = { 2.02 };
//	 // mean sulfur-sulfur distance in natives // angstroms
//
//	float const disulf_ss_dist_stdev = { 0.35 };
//	 // standard dev. of s-s dist in natives // degrees 0.35
//
//	float const ideal_cs_angle_in_disulfide = { 103.4 };
//	 // mean cbeta-sulfur-sulfur angle in natives // degrees
//
//	float const disulf_cs_angle_stdev = { 5.0 };
//	 // standard dev. of cbeta-s-s angle in natives // degrees  2.6
//
//	float const mean_pos_dihedral_in_disulf = { 87.9 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	float const disulf_pos_dihedral_stdev = { 21.8 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	float const mean_pos_dihedral_gauss1 = { 87.4 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	float const stdev_pos_dihedral_gauss1 = { 20.9 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	float const mean_pos_dihedral_gauss2 = { 95.6 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	float const stdev_pos_dihedral_gauss2 = { 3.0 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	float const mean_neg_dihedral_in_disulf = { -86.2 };
//	 // mean negative cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	float const disulf_neg_dihedral_stdev = { 11.1 };
//	 // standard dev. of neg. dihedral angle in natives // degrees
//
//	float const ideal_ca_dihedral_in_disulf = { 74.0 };
//	 // ideal calpha-cbeta-sulfur-sulfur dihedral (abs val) // degrees

//bs this function is for ssdist score 8/26/03.
//bs it starts at ssdist = 1.5 and ends at ssdist = 3.0 and it has 300 points.



//bs-----------------------------------------------------------
//bs  globals used from disulfides.h
//bs
//bs  cys(max_cysteines)
//bs  n_disulfides_fa
//bs  disulf_partner_a,disulf_partner_b
//bs-----------------------------------------------------------

	distance_score_this_disulfide = 0;
	csangles_score_this_disulfide = 0;
	dihedral_score_this_disulfide = 0;
	ca_dihedral_sc_this_disulf = 0;
	truefalse_fa_disulf = false;

//bs If reach this point, res1 and res2 ARE listed as a disulfide.
//bs Now get parameters of disulfide and score it.

	get_disulfide_params(coord1,coord2,ssdist,csang_1,csang_2,dihed,
	disulf_ca_dihedral_angle_1,disulf_ca_dihedral_angle_2);

//bs if farlx, then use -ln() scores

//bs ssdist
	if ( ssdist >= ssdist_max_range ) {
		bin = nbins_ssdist;
	} else if ( ssdist <= ssdist_min_range ) {
		bin = 1;
	} else {
		bin = static_cast< int >((ssdist-ssdist_min_range)/ssdist_bin_size+1.);
	}
	distance_score_this_disulfide = farlx_ssdist_sc_interp300(bin);

//$$$cbs scale factor used to scale the angular scores
//$$$cbs linear in the ssdist between 3 and 6 ang
//$$$cbs =1 for 1.5 < ssdist < 3
//$$$cbs =0 at ssdist >= 6 and ssdist <= 1.5
//$$$       if ( ssdist <= 3.0 && ssdist > 1.5 ) {
//$$$          scale_factor = 1.0;
//$$$       } else if ( ssdist <= 1.5 || ssdist >= 6.0 ) {
//$$$          scale_factor = 0.0;
//$$$       } else {
//$$$          scale_factor = 2.0 - 0.333*ssdist;
//$$$       }
//bs make the scale factor = 1.0 all the time.
	scale_factor = 1.0;

//bs csang
	if ( csang_1 > 180. || csang_2 > 180. ) {
		std::cout << "Error csang > 180" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( csang_1 < csang_min_range ) {
		bin_1 = 1;
	} else if ( csang_1 > csang_max_range ) {
		bin_1 = nbins_csang;
	} else {
		bin_1 = static_cast< int >((csang_1-csang_min_range)/csang_bin_size+1.);
	}

	if ( csang_2 < csang_min_range ) {
		bin_2 = 1;
	} else if ( csang_2 > csang_max_range ) {
		bin_2 = nbins_csang;
	} else {
		bin_2 = static_cast< int >((csang_2-csang_min_range)/csang_bin_size+1.);
	}

	csangles_score_this_disulfide = (farlx_csang_score_2(bin_1)*0.5+
	 farlx_csang_score_2(bin_2)*0.5)*scale_factor;

//bs dihed

//bs safeguard against rounding errors that lead to
//bs subscript out of range.
	if ( dihed > 179.9999 ) dihed = -180.0;

	bin = static_cast< int >((dihed-dihed_min_range)/dihed_bin_size+1.);
	dihedral_score_this_disulfide = farlx_dihed_score_2(bin)*scale_factor;


//bs           std::cout << "ssdist, ssdist_score,scale_factor = " <<
//bs            dist_between_sulfurs << distance_score_this_disulfide <<
//bs            scale_factor << std::endl;
//bs           std::cout << "csang_1, csang_2 csang_score" << SS( cs_bond_angle_1 ) <<
//bs            SS( cs_bond_angle_2 ) << SS( csangles_score_this_disulfide ) << std::endl;
//bs           std::cout << "dihed, dihed_score" << SS( disulf_dihedral_angle ) <<
//bs            SS( dihedral_score_this_disulfide ) << std::endl;

//$$$        if ( score_num == 12 ) {
//$$$
//$$$           csang_ssdist_2Dscore_1 =
//$$$               gauss2D(ssdist,csang_1,z0,A,x0,xwidth,y_0,ywidth,cor)+
//$$$               gauss2D(ssdist,csang_1,z0_2,A_2,x0_2,xwidth_2,
//$$$               y0_2,ywidth_2,cor_2);
//$$$
//$$$          csang_ssdist_2Dscore_2 =
//$$$               gauss2D(ssdist,csang_2,z0,A,x0,xwidth,y_0,ywidth,cor)+
//$$$               gauss2D(ssdist,csang_2,z0_2,A_2,x0_2,xwidth_2,
//$$$               y0_2,ywidth_2,cor_2);
//$$$
//$$$          csang_ssdist_2Dscore = (csang_ssdist_2Dscore_1 +
//$$$               csang_ssdist_2Dscore_2)*0.5;
//$$$
//$$$
//$$$cbs temporary fix. Instead of introducing a new
//$$$cbs global disulf score (csang_ssdist_2Dscore) and have it work in the
//$$$cbs packer also, for now I am just using the old distance and csang scores
//$$$cbs to represent the new score.
//$$$
//$$$           distance_score_this_disulfide = 0.5*csang_ssdist_2Dscore;
//$$$           csangles_score_this_disulfide = distance_score_this_disulfide;
//$$$
//$$$cbs dihed
//$$$
//$$$cbs safeguard against rounding errors that lead to
//$$$cbs subscript out of range.
//$$$           if ( dihed > 179.9999 )
//$$$               dihed = -180.0;
//$$$
//$$$          bin = static_cast< int >((dihed-dihed_angle_min_range)/dihed_bin_size+1.);
//$$$          dihedral_score_this_disulfide =
//$$$               farlx_dihed_score_2(bin);
//$$$
//$$$
//$$$cbs           std::cout << "ssdist, ssdist_score,scale_factor = " << SS( ssdist ) <<
//$$$cbs            SS( distance_score_this_disulfide ) << SS( scale_factor ) << std::endl;
//$$$cbs           std::cout << "csang_1, csang_2 csang_score" << SS( csang_1 ) <<
//$$$cbs            SS( csang_2 ) << SS( csangles_score_this_disulfide ) << std::endl;
//$$$cbs           std::cout << "dihed, dihed_score" << SS( dihed ) <<
//$$$cbs            SS( dihedral_score_this_disulfide ) << std::endl;


//bs ca_dihedral score (this is just the chi_2 angle, rotation about cbeta-sulfur)
//bs chi_2 of cys_1:
	if ( disulf_ca_dihedral_angle_1 > 0.0 ) {       // use the native (+) distribution

		delta_cadih_ang_1 = disulf_ca_dihedral_angle_1-86.; // pos. peak at 86 deg

		float const ang_frac = delta_cadih_ang_1/72.0f;
		ca_dihedral_sc_this_disulf = -std::exp(-(ang_frac*ang_frac));
		 // width of pos peak is ~72 deg
	} else {                               // use the native (-) distribution
		delta_cadih_ang_1 = disulf_ca_dihedral_angle_1-(-74.);
		 // neg. peak at -74 deg

		float const ang_frac = delta_cadih_ang_1/39.0f;
		ca_dihedral_sc_this_disulf = -std::exp(-(ang_frac*ang_frac));
		 // width of neg peak is ~39 deg
	}
//bs chi_2 of cys_2 (add score to that for cys_1)
	if ( disulf_ca_dihedral_angle_2 > 0.0 ) {       // use the native (+) distribution

		delta_cadih_ang_2 = disulf_ca_dihedral_angle_2-86.; // pos. peak at 86 deg

		float const ang_frac = delta_cadih_ang_2/72.0f;
		ca_dihedral_sc_this_disulf -= std::exp(-(ang_frac*ang_frac));
		 // width of pos peak is ~72 deg
	} else {                                   // use the native (-) distribution
		delta_cadih_ang_2 = disulf_ca_dihedral_angle_2-(-74.);
		 // neg. peak at -74 deg

		float const ang_frac = delta_cadih_ang_2/39.0f;
		ca_dihedral_sc_this_disulf -= std::exp(-(ang_frac*ang_frac));
		 // width of neg peak is ~39 deg
	}

	ca_dihedral_sc_this_disulf *= .5 * scale_factor;

//bs Call it a disulfide or not!
	if ( ssdist < disulf_ssdist_cutoff ) {
		truefalse_fa_disulf = true;
	}
}


//******************************************************************************
//******************************************************************************
//******************************************************************************
// alternate functions that allow angular derivatives...
////////////////////////////////////////////////////////////////////////////////
/// @begin score_this_disulfide_fa
///
/// @brief
///bs Scores a single cys-cys pair as a disulfide bond in a fullatom structures.
/// @detailed
///bs Assumes that disulf_partner_a and disulf_partner_b are pre-set.
///bs Also assumes that res1,res2 are cys and listed as disulf.
///bs If fix_disulf, returns a score for any of n_known_disulfides.
///bs If find_disulf, returns a score for any of n_disulfides_centroid.
///
/// @param[in]   coord1 - in - fa coords of cys 1
/// @param[in]   coord2 - in - fa coords of cys 2
/// @param[out]   distance_score_this_disulfide - out -
/// @param[out]   csangles_score_this_disulfide - out -
/// @param[out]   dihedral_score_this_disulfide - out -
/// @param[out]   ca_dihedral_sc_this_disulf - out -
/// @param[out]   truefalse_fa_disulf - out - logical is this a disulfide?
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///     All the parameters for fullatom disulfide score are listed in this function.
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
Disulfides::score_this_disulfide_fa_alt(
	FArray2Da_float coord1, // xyz coords of cys 1
	FArray2Da_float coord2, // xyz coords of cys 2
	float & distance_score_this_disulfide,
	float & csangles_score_this_disulfide,
	float & dihedral_score_this_disulfide,
	float & ca_dihedral_sc_this_disulf,
	bool & truefalse_fa_disulf
)
{
	using namespace disulfides::constants;
	using namespace param;
	using numeric::conversions::radians;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

//bs Rationale of disulfide fullatom scores:
//bs All scores lie between 0 and -1.
//bs All scores are gaussians, with center and width similar to that found in natives.
//bs In the case of c-s-s bond angle score (called csangles), there are two gaussians,
//bs one for positive angle and one for negative.
//bs csangles and dihedral scores are scaled by distance_score for that disulfide.
//bs (so the angular terms contribute only if the s-s distance is close to the optimum)

//bs input
//bs	int res1, res2; // residue numbers for two cys residues

//bs local

	float ssdist; //dist between cys sulfurs
	float csang_1,csang_2; //Cb-S-S bond angles
//jjh not referenced
//	float delta_csang_1,delta_csang_2;
//	float delta_dih_ang;
//	float delta_ss_dist;
	float delta_cadih_ang_1,delta_cadih_ang_2;
	float dihed; //dihedral (torsion) angle, cbeta1-s1-s2-cbeta2
	float disulf_ca_dihedral_angle_1;
	 //dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	float disulf_ca_dihedral_angle_2;
	 //dihedral (torsion) angle, calpha2-cbeta2-s2-s1
	float scale_factor;
	 //scales the angular and dihedral scores based on s-s distance

//bs	int n_disulfides;


//bs nbins and bin_size for scores that are -ln()
	float const ssdist_bin_size = { 0.005 };
	 // for interp300 this is 0.005, others is 0.02
	int const nbins_ssdist = { 300 };
	 // for interp300 this is 300, others it is 100.
	//notused-jjg//float const csang_bin_size = { 1.0 };
	int const nbins_csang = { 100 };
	float const dihed_bin_size = { 3.6 };
	int const nbins_dihed = { 100 };
	int bin; // bin_1,bin_2

//bs scores that are -ln()
//jjh not referenced
//	FArray1D_float farlx_ssdist_score( nbins_ssdist );
//	FArray1D_float farlx_ssdist_score_2( nbins_ssdist );
	static FArray1D_float const farlx_csang_score( nbins_csang,
	 farlx_csang_score_initializer );
	static FArray1D_float const farlx_csang_score_2( nbins_csang,
	 farlx_csang_score_2_initializer );
	static FArray1D_float const farlx_dihed_score( nbins_dihed,
	 farlx_dihed_score_initializer );
	static FArray1D_float const farlx_dihed_score_2( nbins_dihed,
	 farlx_dihed_score_2_initializer );
	static FArray1D_float const farlx_ssdist_sc_interp300( nbins_ssdist,
	 farlx_ssdist_sc_interp300_initializer );

//bs min and max ranges covered by the -ln() scores (e.g. ssdist score values hard-wired only for 1.5<ssdist<3.0)
	//notused-jjg//float const csang_min_range = { 50. };
	//notused-jjg//float const csang_max_range = { 150.0 };
	float const dihed_min_range = { -180. };
	//notused-jjg//float const dihed_max_range = { 180.0 };
	float const ssdist_min_range = { 1.5 };
	float const ssdist_max_range =
	 { ssdist_min_range + nbins_ssdist * ssdist_bin_size };

//     local
	float xS,xS2,E_xS,E_xS2;

//     double_gauss fit to cos(csangle) prob. density
	float const base = { 3. };
	float const amp1 = { -4.4358 };
	float const amp2 = { -1.7374 };
	float const mean1 = { -0.24025 };
	float const mean2 = { -0.31186 };
	float const width1 = { 0.098048 };
	float const width2 = { 0.22224 };


//-----------------------------------------
// physical parameters of disulfide bonds
//-----------------------------------------
//jjg--commented these out to avoid compiler warnings about unused variables
//5/21/04
//
//	float const ideal_ss_dist_in_disulfide = { 2.02 };
//	 // mean sulfur-sulfur distance in natives ! angstroms
//
//	float const disulf_ss_dist_stdev = { 0.35 };
//	 // standard dev. of s-s dist in natives ! degrees 0.35
//
//	float const ideal_cs_angle_in_disulfide = { 103.4 };
//	 // mean cbeta-sulfur-sulfur angle in natives ! degrees
//
//	float const disulf_cs_angle_stdev = { 5.0 };
//	 // standard dev. of cbeta-s-s angle in natives ! degrees  2.6
//
//	float const mean_pos_dihedral_in_disulf = { 87.9 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle ! degrees
//
//	float const disulf_pos_dihedral_stdev = { 21.8 };
//	 // standard dev. of pos. dihedral angle in natives ! degrees
//
//	float const mean_pos_dihedral_gauss1 = { 87.4 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle ! degrees
//
//	float const stdev_pos_dihedral_gauss1 = { 20.9 };
//	 // standard dev. of pos. dihedral angle in natives ! degrees
//
//	float const mean_pos_dihedral_gauss2 = { 95.6 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle ! degrees
//
//	float const stdev_pos_dihedral_gauss2 = { 3.0 };
//	 // standard dev. of pos. dihedral angle in natives ! degrees
//
//	float const mean_neg_dihedral_in_disulf = { -86.2 };
//	 // mean negative cbeta-sulfur-sulfur-cbeta dihedral angle ! degrees
//
//	float const disulf_neg_dihedral_stdev = { 11.1 };
//	 // standard dev. of neg. dihedral angle in natives ! degrees
//
//	float const ideal_ca_dihedral_in_disulf = { 74.0 };
//	 // ideal calpha-cbeta-sulfur-sulfur dihedral (abs val) ! degrees

//$$$        data z0 /10./
//$$$        data A /-15.85/
//$$$        data x0 /2.0227/
//$$$        data xwidth /0.035/
//$$$        data y0 /103.32/
//$$$        data ywidth /4.11488/
//$$$        data cor /0.044559/
//$$$
//$$$        data z0_2 /10.0/
//$$$        data A_2 /-10.0/
//$$$        data x0_2 /2.0227/
//$$$        data xwidth_2 /1.5/
//$$$        data y0_2 /103.32/
//$$$        data ywidth_2 /30.0/
//$$$        data cor_2 /0.044559/

//bs this function is for ssdist score 8/26/03.
//bs it starts at ssdist=1.5 and ends at ssdist = 3.0 and it has 300 points.


//bs-----------------------------------------------------------
//bs  globals used from disulfides.h
//bs
//bs  cys(max_cysteines)
//bs  n_disulfides_fa
//bs  disulf_partner_a,disulf_partner_b
//bs-----------------------------------------------------------

	distance_score_this_disulfide = 0;
	csangles_score_this_disulfide = 0;
	dihedral_score_this_disulfide = 0;
	ca_dihedral_sc_this_disulf = 0;
	truefalse_fa_disulf = false;

//bs If reach this point, res1 and res2 ARE listed as a disulfide.
//bs Now get parameters of disulfide and score it.

	get_disulfide_params(coord1,coord2, ssdist,csang_1,csang_2, dihed,
	 disulf_ca_dihedral_angle_1, disulf_ca_dihedral_angle_2);

//bs if farlx, then use -ln() scores

//bs ssdist
	if ( ssdist >= ssdist_max_range ) {
		bin = nbins_ssdist;
	} else if ( ssdist <= ssdist_min_range ) {
		bin = 1;
	} else {
		bin = static_cast< int >(
		 ( ssdist - ssdist_min_range ) / ssdist_bin_size + 1.0 );
	}
	distance_score_this_disulfide = farlx_ssdist_sc_interp300(bin);

//$$$cbs scale factor used to scale the angular scores
//$$$cbs linear in the ssdist between 3 and 6 ang
//$$$cbs =1 for 1.5<ssdist<3
//$$$cbs =0 at ssdist>=6 and ssdist<=1.5
//$$$	if ( ssdist <= 3.0 && ssdist > 1.5 ) {
//$$$		scale_factor = 1.0;
//$$$	} else if ( ssdist <= 1.5 || ssdist >= 6.0 ) {
//$$$		scale_factor = 0.0;
//$$$	} else {
//$$$		scale_factor = 2.0 - 0.333*ssdist;
//$$$	}
//bs make the scale factor = 1.0 all the time.
	scale_factor = 1.0;

//bs csang
	if ( csang_1 > 180.0 || csang_2 > 180.0 ) {
		std::cout << "Error csang > 180" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//$$$ 12/17/03 comment this out to put in the cos(theta) score
//$$$	if ( csang_1 < csang_min_range ) {
//$$$		bin_1 = 1;
//$$$	} else if ( csang_1 > csang_max_range ) {
//$$$		bin_1 = nbins_csang;
//$$$	} else {
//$$$		bin_1 = static_cast< int >((csang_1-csang_min_range)/csang_bin_size+1.0);
//$$$	}
//$$$
//$$$	if ( csang_2 < csang_min_range ) {
//$$$		bin_2 = 1;
//$$$	} else if ( csang_2 > csang_max_range ) {
//$$$		bin_2 = nbins_csang;
//$$$	} else {
//$$$		bin_2 = static_cast< int >((csang_2-csang_min_range)/csang_bin_size+1.0);
//$$$	}
//$$$
//$$$	csangles_score_this_disulfide = (farlx_csang_score_2(bin_1)*0.5
//$$$	 farlx_csang_score_2(bin_2)*0.5)*scale_factor;

	xS = std::cos( radians( 180 - csang_1 ) );
	xS2 = std::cos( radians( 180 - csang_2 ) );
	E_xS = dbl_gauss(xS,base,amp1,amp2,mean1,mean2,width1,width2);
	E_xS2 = dbl_gauss(xS2,base,amp1,amp2,mean1,mean2,width1,width2);
	csangles_score_this_disulfide = (E_xS+E_xS2)*0.5*scale_factor;

//bs dihed

//bs safeguard against rounding errors that lead to
//bs subscript out of range.
	if ( dihed > 179.9999 ) dihed = -180.0;

	bin = static_cast< int >((dihed-dihed_min_range)/dihed_bin_size+1.0);
	dihedral_score_this_disulfide = farlx_dihed_score_2(bin) * scale_factor;

//bs	std::cout << "ssdist, ssdist_score,scale_factor = " <<
//bs	 SS( dist_between_sulfurs ) << SS( distance_score_this_disulfide ) <<
//bs	 SS( scale_factor ) << std::endl;
//bs	std::cout << "csang_1, csang_2 csang_score" << SS( cs_bond_angle_1 ) <<
//bs	 SS( cs_bond_angle_2 ) << SS( csangles_score_this_disulfide ) << std::endl;
//bs	std::cout << "dihed, dihed_score" << SS( disulf_dihedral_angle ) <<
//bs	 SS( dihedral_score_this_disulfide ) << std::endl;

//$$$	if ( score_num == 12 ) {
//$$$
//$$$		csang_ssdist_2Dscore_1 =
//$$$		 gauss2D(ssdist,csang_1,z0,A,x0,xwidth,y0,ywidth,cor)+
//$$$		 gauss2D(ssdist,csang_1,z0_2,A_2,x0_2,xwidth_2,
//$$$		 y0_2,ywidth_2,cor_2);
//$$$
//$$$		csang_ssdist_2Dscore_2 =
//$$$		 gauss2D(ssdist,csang_2,z0,A,x0,xwidth,y0,ywidth,cor)+
//$$$		 gauss2D(ssdist,csang_2,z0_2,A_2,x0_2,xwidth_2,
//$$$		 y0_2,ywidth_2,cor_2);
//$$$
//$$$		csang_ssdist_2Dscore = ( csang_ssdist_2Dscore_1 +
//$$$		 csang_ssdist_2Dscore_2 ) * 0.5;
//$$$
//$$$
//$$$bs temporary fix. Instead of introducing a new
//$$$bs global disulf score (csang_ssdist_2Dscore) and have it work in the
//$$$bs packer also, for now I am just using the old distance and csang scores
//$$$bs to represent the new score.
//$$$
//$$$	distance_score_this_disulfide = 0.5*csang_ssdist_2Dscore;
//$$$	csangles_score_this_disulfide = distance_score_this_disulfide;
//$$$
//$$$bs dihed
//$$$
//$$$bs safeguard against rounding errors that lead to
//$$$bs subscript out of range.
//$$$	if ( dihed > 179.9999 )
//$$$	 dihed = -180.0;
//$$$
//$$$	bin = static_cast< int >((dihed-dihed_angle_min_range)/dihed_bin_size+1.);
//$$$	dihedral_score_this_disulfide = farlx_dihed_score_2(bin);
//$$$
//$$$
//$$$bs	std::cout << "ssdist, ssdist_score,scale_factor = " << SS( ssdist ) <<
//$$$bs	 SS( distance_score_this_disulfide ) << SS( scale_factor ) << std::endl;
//$$$bs	std::cout << "csang_1, csang_2 csang_score" << SS( csang_1 ) <<
//$$$bs	 SS( csang_2 ) << SS( csangles_score_this_disulfide ) << std::endl;
//$$$bs	std::cout << "dihed, dihed_score" << SS( dihed ) <<
//$$$bs	 SS( dihedral_score_this_disulfide ) << std::endl;


//bs ca_dihedral score (this is just the chi_2 angle, rotation about cbeta-sulfur)
//bs chi_2 of cys_1:
		if ( disulf_ca_dihedral_angle_1 > 0.0 ) {       // use the native (+) distribution

			delta_cadih_ang_1 = disulf_ca_dihedral_angle_1 - 86.0; // pos. peak at 86 deg

			ca_dihedral_sc_this_disulf =
			 -std::exp( -( square( delta_cadih_ang_1 / 72.0 ) ) );
			 // width of pos peak is ~72 deg
		} else {                                       //use the native (-) distribution
			delta_cadih_ang_1 = disulf_ca_dihedral_angle_1 - (-74.0);
			 // neg. peak at -74 deg

			ca_dihedral_sc_this_disulf =
			 -std::exp( -( square( delta_cadih_ang_1 / 39.0 ) ) );
			 // width of neg peak is ~39 deg
		}
//bs chi_2 of cys_2 (add score to that for cys_1)
		if ( disulf_ca_dihedral_angle_2 > 0.0 ) {    // use the native (+) distribution

			delta_cadih_ang_2 = disulf_ca_dihedral_angle_2 - 86.0; // pos. peak at 86 deg

			ca_dihedral_sc_this_disulf -=
			 std::exp( -( square( delta_cadih_ang_2 / 72.0 ) ) ); // width of pos peak is ~72 deg
		} else {                                       //use the native (-) distribution
			delta_cadih_ang_2 = disulf_ca_dihedral_angle_2 - (-74.0);
			 // neg. peak at -74 deg

			ca_dihedral_sc_this_disulf -=
			 std::exp( -( square( delta_cadih_ang_2 / 39.0 ) ) ); // width of neg peak is ~39 deg
		}

		ca_dihedral_sc_this_disulf *= .5 * scale_factor;

//bs Call it a disulfide or not!
		if ( ssdist < disulf_ssdist_cutoff ) {
			truefalse_fa_disulf = true;
		}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_cent_disulf_scores_to_zero
///
/// @brief
///     initializes disulf centroid scores to 0.
/// @detailed
///
/// @global_read
///
/// @global_write
///     block: count_cys_disulf
///     block: score_disulf_centroid (score_ns.h)
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::set_cent_disulf_scores_to_zero()
{
	using namespace scores;

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;

	n_disulfides_centroid = 0;
	n_disulf_cent_natcut = 0;
	n_disulf_cb_natcut = 0;

	disulf_cendist_score = 0;
	disulf_bb_dih_score = 0;
	disulf_cbdist_score = 0;
	disulf_ca_minus_cen_score = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_fa_disulfide_scores_to_zero
///
/// @brief
///     sets disulf fa scores to 0.
/// @detailed
///
/// @global_read
///
/// @global_write
///     block: count_cys_disulf,fullatom_disulfide_scores
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::set_fa_disulfide_scores_to_zero()
{

	if ( !disulfides::BOUNDARY::get_disulf_flag() ) return;

	n_disulfides_fa = 0;
	disulfide_distance_score = 0;
	disulfide_csangles_score = 0;
	disulfide_dihedral_score = 0;
	disulfide_ca_dihedral_score = 0;
	disulfide_cbs_dist_score = 0;
}


//------------------------------------------------------------------------------
//bs Updates fullatom disulfide scores stored in global variables.

////////////////////////////////////////////////////////////////////////////////
/// @begin update_disulf_fa_scores
///
/// @brief
///bs Updates fullatom disulfide scores stored in global variables.
/// @detailed
/// Takes disulf scores for one disulfide bond and adds them to the
/// total disulfide scores for the protein.
/// @param[in]   truefalse_fa_disulf - in -
/// @param[in]   dist_sc_this_disulf - in -
/// @param[in]   csang_sc_this_disulf - in -
/// @param[in]   dihedral_sc_this_disulf - in -
/// @param[in]   ca_dihedral_sc_this_disulf - in -
/// @param[in]   cbs_dist_sc_this_disulf - in -
///
/// @global_read
///
/// @global_write
/// block: fullatom_disulfide_scores
/// integer: n_disulfides_fa
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::update_disulf_fa_scores(
	bool & truefalse_fa_disulf, // did it satisfy fa criteria
	float & dist_sc_this_disulf,
	float & csang_sc_this_disulf,
	float & dihedral_sc_this_disulf,
	float & ca_dihedral_sc_this_disulf,
	float & cbs_dist_sc_this_disulf
)
{
	if ( truefalse_fa_disulf ) {
		++n_disulfides_fa; // counting fa_disulfides
	}

	disulfide_distance_score += dist_sc_this_disulf;
	disulfide_csangles_score += csang_sc_this_disulf;
	disulfide_dihedral_score += dihedral_sc_this_disulf;
	disulfide_ca_dihedral_score += ca_dihedral_sc_this_disulf;
	disulfide_cbs_dist_score += cbs_dist_sc_this_disulf;
}


float
Disulfides::get_disulf_cendist_deriv(
  int /* res1 */, // unused
	int /* atom1 */ , // unused
	int /* res2 */ , // unused
	int /* atom2 */, // unused
	int disulf_pair_i,
	bool /* fullatom */ // unused
)
{
	using namespace cenlist_ns;
	using namespace scorefxns;

	float get_disulf_cendist_deriv; // Return value

	int cys_1,cys_2; // residue #s for cysteines
//$$$      int const nbins = 45;
//$$$      float const bin_size = 0.5; // in ang^2, first bin is 0-0.5
	int const nbins = { 300 }; // # of bins in cendist deriv
	float const bin_size = { 0.125 };
	 // bin width for cendist deriv // in ang^2, first bin is 0-0.125
	float const max_bin_val = { nbins*bin_size };
//jjh not referenced
//    FArray1D_float deriv_cendist_score_sm(nbins); // deriv of cendist_foldcst_score_sm
	static FArray1D_float const deriv_cendist_interp300( nbins,
	 deriv_cendist_interp300_initializer );
	FArray1D_float current_deriv( nbins );
	int bin;
	float tempdist;

//bs this is dE/d(r**2)

//$$$      data deriv_cendist_score_sm
//$$$     #     /   0.000,   0.000,   0.000,   0.000,  -2.911,
//$$$     #       -5.120,  -2.894,  -1.174,  -0.927,  -0.325,
//$$$     #        0.232,   0.244,   0.256,   0.268,   0.280,
//$$$     #        0.292,   0.304,   0.316,   0.328,   0.340,
//$$$     #        0.352,   0.364,   0.376,   0.388,   0.400,
//$$$     #        0.412,   0.424,   0.436,   0.448,   0.460,
//$$$     #        0.472,   0.484,   0.496,   0.508,   0.520,
//$$$     #        0.532,   0.544,   1.574,   1.299,   0.000,
//$$$     #        0.000,   0.000,   0.000,   0.000,   0.000/


	get_disulf_cendist_deriv = 0.0;

	for ( int ii = 1; ii <= nbins; ++ii ) {
		current_deriv(ii) = deriv_cendist_interp300(ii);
	}

	cys_1 = cys(disulf_partner_a(disulf_pair_i));
	cys_2 = cys(disulf_partner_b(disulf_pair_i));
	tempdist = cendist(cys_1,cys_2);

	if ( tempdist <= 0.0 ) {
		std::cout << "error score_disulf_cendist, tempdist = " << tempdist << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( tempdist > max_bin_val ) {
		get_disulf_cendist_deriv = current_deriv(nbins);
	} else if ( tempdist <= max_bin_val ) {
		bin = static_cast< int >(tempdist/bin_size+1.);
		get_disulf_cendist_deriv = current_deriv(bin);
	}

//$$$  std::cout << "i,j,cendist,nowtdrv = " <<
//$$$   I( 3, cys_1 ) << ' ' << I( 3, cys_2 ) << ' ' <<
//$$$	  F( 8, 5, tempdist ) << ' ' << F( 8, 2, get_disulf_cendist_deriv ) << std::endl;

	if ( get_disulf_cendist_deriv == 0.0 ) return get_disulf_cendist_deriv;

//bs dE/dr = dE/d(r**2) * d(r**2)/dr = dE/d(r**2) * 2r

	get_disulf_cendist_deriv *= 2.0 * std::sqrt(tempdist) * disulf_cendist_weight;

	return get_disulf_cendist_deriv;
}


float
Disulfides::get_disulf_cbdist_deriv(
	int /* res1 */,
	int /* atom1 */,
	int /* res2 */,
	int /* atom2 */,
	int disulf_pair_i,
	bool /* fullatom */
)
{
	using namespace cb_score_3gauss_fitparams;
	using namespace scorefxns;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

	float get_disulf_cbdist_deriv; // Return value

	int cys_1,cys_2; // residue #s for cysteines
	int const nbins = { 50 }; // # of bins in cbdist deriv
//	float const bin_size = { 0.5 };
	 // bin width for cbdist deriv // in ang^2, first bin is 0-0.5
	static FArray1D_float const deriv_cbdist_score( nbins,
	 deriv_cbdist_score_initializer ); // deriv of cbdist_foldcst_score_sm
	FArray1D_float current_deriv( nbins );
//jjh not referenced
//	int bin;
	float cbdist;

//bs Triple gauss fit...

//bs this is dE/d(r^2)

	get_disulf_cbdist_deriv = 0.0;

	for ( int ii = 1; ii <= nbins; ++ii ) {
		current_deriv(ii) = deriv_cbdist_score(ii);
	}

	cys_1 = cys(disulf_partner_a(disulf_pair_i));
	cys_2 = cys(disulf_partner_b(disulf_pair_i));

//bs get the dist^2 between cbetas

	cbdist = 0.0;
	for ( int k = 1; k <= 3; ++k ) {
		float const Edif = Eposition(k,3,cys_2) - Eposition(k,3,cys_1);
		cbdist += Edif * Edif;
	}

	if ( cbdist <= 0.0 ) {
		std::cout << "error disulf_deriv_cbdist, cbdist = " << cbdist << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//$$$      if ( cbdist > max_native_disulf_cbdist ) {
//$$$         get_disulf_cbdist_deriv = current_deriv(nbins);
//$$$      } else if ( cbdist <= max_native_disulf_cbdist ) {
//$$$         bin = static_cast< int >(cbdist/bin_size+1.);
//$$$         get_disulf_cbdist_deriv = current_deriv(bin);
//$$$      }


//bs this is dE/d(r**2)

	get_disulf_cbdist_deriv = d_triple_gauss_d_x(cbdist,y_0,A_1,A_2,A_3,x0_1,x0_2,
	 x0_3,xwidth_1,xwidth_2,xwidth_3);


//$$$   std::cout << "i,j,cbdist,nowtdrv = " <<
//$$$    I( 3, cys_1 ) << ' ' << I( 3, cys_2 ) << ' ' << F( 8, 5, cbdist ) << ' ' <<
//$$$		 F( 8, 2, get_disulf_cbdist_deriv ) << std::endl;

	if ( get_disulf_cbdist_deriv == 0.0 ) return get_disulf_cbdist_deriv;

//bs dE/dr = dE/d(r**2) * d(r**2)/dr = dE/d(r**2) * 2r

	get_disulf_cbdist_deriv *= 2 * std::sqrt( cbdist ) * disulf_cbdist_weight;

	return get_disulf_cbdist_deriv;
}


float
Disulfides::get_disulf_ssdist_deriv(
	int /* res1 */,
	int /* atom1 */,
	int /* res2 */,
	int /* atom2 */,
	int disulf_pair_i,
	bool fullatom
)
{
	using namespace param;
	using namespace param_pack;
	using namespace scorefxns;

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal

	float get_disulf_ssdist_deriv; // Return value

	int cys_1,cys_2; // residue #s for cysteines
//jjh not referenced
//      float deriv_ssdist_score;
	float ssdist;
	float cs_bond_angle_1,cs_bond_angle_2;
	float disulf_dihedral_angle,disulf_ca_dihedral_angle_1;
	float disulf_ca_dihedral_angle_2;
	FArray2D_float cys1coord_fa( 3, MAX_ATOM()() );
	FArray2D_float cys2coord_fa( 3, MAX_ATOM()() );
//jjh not referenced
//      float zeta;   // argument of the exp in the 2D gaussian

	int const nbins = { 300 }; // # of bins in ssdist deriv
	float const bin_size = { 0.005 };
	 // bin width for ssdist deriv // in ang, first bin is 1.0-1.005
	float const min_bin_val = { 1.5 };
	float const max_bin_val = { min_bin_val+(nbins-1)*bin_size };
	static FArray1D_float const deriv_ssdist_sc_interp300( nbins,
	 deriv_ssdist_sc_interp300_initializer ); // deriv of ssdist_score_interp300
	FArray1D_float current_deriv( nbins );
	int bin;
//jjh not refrenced
//	float tempdist;

//bs 2D score of csang vs ssdist
//bs param values for 2D score are set in function score_this_disulfide_fa


//bs this function corresponds to the score ssdist_score_interp300
//bs it starts at ssdist = 1.5 and ends at ssdist = 2.995 and it has 300 points.

	get_disulf_ssdist_deriv = 0.0;

	if ( !fullatom ) return get_disulf_ssdist_deriv;

	for ( int ii = 1; ii <= nbins; ++ii ) {
		current_deriv(ii) = deriv_ssdist_sc_interp300(ii);
	}

	cys_1 = cys(disulf_partner_a(disulf_pair_i));
	cys_2 = cys(disulf_partner_b(disulf_pair_i));

//bs get the coords for both cys...
//bs cys1coord needs to have the full_coord atom ordering (N,Ca,C,O,Cb)

	for ( int j = 1, max_atom = MAX_ATOM(); j <= max_atom; ++j ) {
	 // get coords of cys residues in fullatom mode
		for ( int k = 1; k <= 3; ++k ) {
			cys1coord_fa(k,j) = full_coord(k,j,cys_1);
			cys2coord_fa(k,j) = full_coord(k,j,cys_2);
		}
	}
	get_disulfide_params(cys1coord_fa,cys2coord_fa,ssdist,cs_bond_angle_1,
	 cs_bond_angle_2,disulf_dihedral_angle,disulf_ca_dihedral_angle_1,
	 disulf_ca_dihedral_angle_2);

//bs gross error checks

	if ( ssdist <= 0.0 ) {
		std::cout << "error get_disulf_ssdist_deriv, ssdist = " << ssdist << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else if ( std::abs(cs_bond_angle_1) > 180.0 ||
	 std::abs(cs_bond_angle_2) > 180.0 ) {
		std::cout << "error get_disulf_ssdist_deriv csang_1,csang_2 = " <<
		 cs_bond_angle_1 << cs_bond_angle_2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//$$$cbs here is the 2D score and below that is the derivative.
//$$$c$$$           csang_ssdist_2Dscore_1 =
//$$$c$$$            gauss2D(ssdist,csang_1,z0,A,x0,xwidth,y_0,ywidth,cor)+
//$$$c$$$            gauss2D(ssdist,csang_1,z0_2,A_2,x0_2,xwidth_2,
//$$$c$$$            y0_2,ywidth_2,cor_2);
//$$$c$$$
//$$$c$$$           csang_ssdist_2Dscore_2 =
//$$$c$$$            gauss2D(ssdist,csang_2,z0,A,x0,xwidth,y_0,ywidth,cor)+
//$$$c$$$            gauss2D(ssdist,csang_2,z0_2,A_2,x0_2,xwidth_2,
//$$$c$$$            y0_2,ywidth_2,cor_2);
//$$$c$$$
//$$$c$$$           csang_ssdist_2Dscore = (csang_ssdist_2Dscore_1 +
//$$$c$$$            csang_ssdist_2Dscore_2)*0.5;
//$$$
//$$$cbs here is the derivative dz/d(ssdist)...
//$$$
//$$$      get_disulf_ssdist_deriv = 0.5*(
//$$$       d_gauss2D_d_x(ssdist,cs_bond_angle_1,z0,A,x0,xwidth,
//$$$          y_0,ywidth,cor)+
//$$$       d_gauss2D_d_x(ssdist,cs_bond_angle_1,z0_2,A_2,x0_2,xwidth_2,
//$$$          y0_2,ywidth_2,cor_2)+
//$$$       d_gauss2D_d_x(ssdist,cs_bond_angle_2,z0,A,x0,xwidth,
//$$$          y_0,ywidth,cor)+
//$$$       d_gauss2D_d_x(ssdist,cs_bond_angle_2,z0_2,A_2,x0_2,xwidth_2,
//$$$          y0_2,ywidth_2,cor_2));

	if ( ssdist >= max_bin_val ) {
		get_disulf_ssdist_deriv = current_deriv(nbins);
	} else if ( ssdist <= min_bin_val ) {
		get_disulf_ssdist_deriv = current_deriv(1);
	} else {
		bin = static_cast< int >( ( ssdist - min_bin_val ) / bin_size + 1.0 );
		get_disulf_ssdist_deriv = current_deriv(bin);
	}

	if ( get_disulf_ssdist_deriv == 0.0 ) return get_disulf_ssdist_deriv;
	get_disulf_ssdist_deriv *=
	 disulfide_distance_weight + ( Wdisulf_dist_sc * fa_pair_weight );
//bs sum  of the wts is correct b/c disulf ssdist score can contribute
//bs to total score independently in two places
//bs fullatom_energy wts: Wdisulf_dist_sc*fa_pair_weight
//bs score12 xtra contribution (optional call in scorefxns.f):
//bs disulfide_distance_weight
	return get_disulf_ssdist_deriv;
}

//------------------------------------------------------------------------------
void
Disulfides::eval_disulf_ang_deriv(
	int const i,
	int const ii,
	int const j,
	int const jj,
	int const aai,
	int const aaj,
	int const /* aavi */,
	int const /* aavj */,
	bool const fullatom,
	FArray1Da_float F1_disulfE,
	FArray1Da_float F2_disulfE
)
{
	using namespace param_aa;

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal

	F1_disulfE.dimension( 3 );
	F2_disulfE.dimension( 3 );

	F1_disulfE = 0.0;
	F2_disulfE = 0.0;

	if ( ! disulfides::BOUNDARY::get_disulf_flag() ) return;
	if ( aai != aa_cys || aaj != aa_cys ) return;
	if ( ! cys_pair_in_disulf(i,j) ) return;

	if ( ! fullatom ) return;
	 // for the moment we will only do the cs angle (and ssdist) this way

//bs local
	int disulf_pair;
	int cys1, cys2;
	FArray2D_float deriv( 3, 2 ); // components of F1,F2
	FArray1D_float Cbxyz( 3 );
	FArray1D_float Sxyz( 3 );
	FArray1D_float Cb2xyz( 3 );
	FArray1D_float S2xyz( 3 );
	float energy;

	std::cout << "enter eval_ang_deriv " <<
	 SS( i ) << SS( ii ) << SS( j ) << SS( jj ) << std::endl;

	if ( i > j && ii == 5 && jj == 1 ) {
	 // this is the first time called for this disulf -- put in the deriv
		disulf_pair = which_disulf_pair(j,i);
		cys2 = cys(disulf_partner_a(disulf_pair));
		cys1 = cys(disulf_partner_b(disulf_pair));
	} else if ( i < j && ii == 6 ) {
	 // last time called -- puts in (-) deriv b/c cys1,cys2 flipped
		disulf_pair = which_disulf_pair(i,j);
		cys1 = cys(disulf_partner_a(disulf_pair));
		cys2 = cys(disulf_partner_b(disulf_pair));
	} else {
		return;
	}

	for ( int k = 1; k <= 3; ++k ) {
		Cbxyz(k) = full_coord(k,5,cys1);
		Sxyz(k) = full_coord(k,6,cys1);
		Cb2xyz(k) = full_coord(k,5,cys2);
		S2xyz(k) = full_coord(k,6,cys2);
	}

	get_disulf_csang_deriv(Cbxyz,Sxyz,Cb2xyz, S2xyz,energy,deriv);

	for ( int k = 1; k <= 3; ++k ) {
		F1_disulfE(k) = deriv(k,1); //already weighted
		F2_disulfE(k) = deriv(k,2);
	}

	if ( i > j && ii == 5 && jj == 1 ) {
	 // this is the first time called for this disulf -- put in the deriv
		std::cout << A( 11, "first " ) <<
		 ' ' << F( 8, 3, deriv(1,1) ) <<
		 ' ' << F( 8, 3, deriv(2,1) ) <<
		 ' ' << F( 8, 3, deriv(3,1) ) <<
		 ' ' << F( 8, 3, deriv(1,2) ) <<
		 ' ' << F( 8, 3, deriv(2,2) ) <<
		 ' ' << F( 8, 3, deriv(3,2) ) << std::endl;
	} else if ( i < j && ii == 6 ) {
	 // this is the last time called for this disulf -- put in (-) deriv??
		std::cout << A( 11, " last " ) <<
		 ' ' << F( 8, 3, deriv(1,1) ) <<
		 ' ' << F( 8, 3, deriv(2,1) ) <<
		 ' ' << F( 8, 3, deriv(3,1) ) <<
		 ' ' << F( 8, 3, deriv(1,2) ) <<
		 ' ' << F( 8, 3, deriv(2,2) ) <<
		 ' ' << F( 8, 3, deriv(3,2) ) << std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////

void
Disulfides::get_disulf_csang_deriv(
	FArray1Da_float Cbxyz, // cys1 - analog to acceptor base
	FArray1Da_float Sxyz, // cys1 - analog to acceptor
	FArray1Da_float Cb2xyz, // cys2 - analog to donor
	FArray1Da_float S2xyz, // cys2 - analog to H
	float & energy,
	FArray2Da_float deriv // xyz,f1/f2
)
{
//bs Evaluate the derivative of the Cb-S-S angle for disulfide bonds

	Cbxyz.dimension( 3 );
	Sxyz.dimension( 3 );
	Cb2xyz.dimension( 3 );
	S2xyz.dimension( 3 );
	deriv.dimension( 3, 2 );

//  vectors, unit vectors, distances, cross products, etc
//  Cb -> S
	FArray1D_float CbSunit( 3 );
	float CbSdis2;
	float CbSdis;
	float inv_CbSdis;

//  S -> S2
	FArray1D_float SS2unit( 3 );
	FArray1D_float SS2( 3 );
	float SS2dis2;
	float SS2dis;
	float inv_SS2dis;
	FArray1D_float S2xS( 3 );

// S2 -> Cb2
	FArray1D_float S2Cb2unit( 3 );
	float S2Cb2dis2;
	float S2Cb2dis, inv_S2Cb2dis;

//  angle definitions:
//   xS2    cosine of angle at S   (analogous to xD)
//   xS     cosine of angle at S2  (analogous to xH)
	float xS2, xS;

//    energy_r   - distant dependent energy
//    dE_dr      - deriv w/respect to distance
//    energy_xS2   - angle dependent energy (
//    dE_dxS2      - deriv w/respect to std::cos(thetaS2)
//    energy_xS   -angle dependent energy
//    dE_dxS      - deriv w/respect to std::cos(thetaS)
	float energy_r, energy_xS2, energy_xS;
	float dE_dxS, dE_dxS2, dE_dr;

// energies, cross products for derivative increments

	FArray1D_float BS2( 3 );
	FArray1D_float BCb2( 3 );
	FArray1D_float BS2xS( 3 );
	FArray1D_float BCb2xS( 3 );
	float prefactor;

// bounds (temporary)
//	float const MAX_SS2dis2 = { 9.0 };
	float const MAX_SS2dis2 = { 999.0 }; // temp
//	float const MIN_SS2dis2 = { 2.25 }; // 1.5**2
	float const MIN_SS2dis2 = { 0.0 }; // temp
	float const MAX_xS = { 1.0 };
	float const MIN_xS = { -1.0 };

	energy = 3.0; // what is max csang energy???
	deriv = 0.0;

//ar S->S2 unit vector, distance (like A->H)
	SS2(1) = S2xyz(1) - Sxyz(1);
	SS2(2) = S2xyz(2) - Sxyz(2);
	SS2(3) = S2xyz(3) - Sxyz(3);

	SS2dis2 = SS2(1)*SS2(1) + SS2(2)*SS2(2) + SS2(3)*SS2(3);

	if ( SS2dis2 > MAX_SS2dis2 ) return;
	if ( SS2dis2 < MIN_SS2dis2 ) return;
	SS2dis = std::sqrt(SS2dis2);
	inv_SS2dis = 1.0 / SS2dis;
	SS2unit(1) = SS2(1) * inv_SS2dis;
	SS2unit(2) = SS2(2) * inv_SS2dis;
	SS2unit(3) = SS2(3) * inv_SS2dis;

//ar  S2->Cb unit vector, dis2 (like H->D)
	S2Cb2unit(1) = Cb2xyz(1) - S2xyz(1);
	S2Cb2unit(2) = Cb2xyz(2) - S2xyz(2);
	S2Cb2unit(3) = Cb2xyz(3) - S2xyz(3);

	S2Cb2dis2 =
	 S2Cb2unit(1)*S2Cb2unit(1) +
	 S2Cb2unit(2)*S2Cb2unit(2) +
	 S2Cb2unit(3)*S2Cb2unit(3);
	S2Cb2dis = std::sqrt(S2Cb2dis2);
	inv_S2Cb2dis = 1.0 / S2Cb2dis;
	S2Cb2unit(1) *= inv_S2Cb2dis;
	S2Cb2unit(2) *= inv_S2Cb2dis;
	S2Cb2unit(3) *= inv_S2Cb2dis;

//ar  Cb->S unit vector  (like A1->A)

	CbSunit(1) = Sxyz(1) - Cbxyz(1);
	CbSunit(2) = Sxyz(2) - Cbxyz(2);
	CbSunit(3) = Sxyz(3) - Cbxyz(3);

	CbSdis2 =
	 CbSunit(1)*CbSunit(1) +
	 CbSunit(2)*CbSunit(2) +
	 CbSunit(3)*CbSunit(3);
	CbSdis = std::sqrt(CbSdis2);
	inv_CbSdis = 1.0 / CbSdis;
	CbSunit(1) *= inv_CbSdis;
	CbSunit(2) *= inv_CbSdis;
	CbSunit(3) *= inv_CbSdis;

//BW cosines of angle at S,S2
	xS2 =
	 SS2unit(1) * S2Cb2unit(1) +
	 SS2unit(2) * S2Cb2unit(2) +
	 SS2unit(3) * S2Cb2unit(3);

	if ( xS2 < MIN_xS ) return;
	if ( xS2 > MAX_xS ) return;

	xS =
	 CbSunit(1) * SS2unit(1) +
	 CbSunit(2) * SS2unit(2) +
	 CbSunit(3) * SS2unit(3);

	if ( xS < MIN_xS ) return;
	if ( xS > MAX_xS ) return;

//ar compute energy components and derivatives
//bs derivatives are returned weighted, energies un-weighted
	disulf_csang_compute_energy(SS2dis,xS,xS2, energy_r,energy_xS,energy_xS2,
	 dE_dr,dE_dxS,dE_dxS2);

	energy = energy_r + energy_xS + energy_xS2;

//ar distance-dependent gradient component.
//ar see also comments in minimize.f
//ar  dr/dphi  =  Eab x (V-Vb) . (V' - V)/|V-V'|
//db  (first cross product is the displacement of V upon a rotation dphi
//db   around the unit vector Eab, Vb is the coordinates of the second atom
//db   in the bond)

//ar  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / |V-V'|
//ar  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / r
//ar  rearrange...
//ar  dEr/dphi = dEr/dr * [Eab X Vb . (V' - V)  + Eab . (V' x V)] / r

//ar Eab and Eab X Vb are calulated in dfunc_vdw and dependent on the torison
//ar angle with respect to which the derivative is being taken
//ar f1 and f2 are independent of the torsion angle and here we're precomputing
//ar increments to f1 and f2 for each hbond
//ar f1 = dEr/dr * (V'xV) / r
//ar f2 = dEr/dr * (V'-V) / r
//ar here, V' = H
//ar       V  = A
//ar       dE/dr * 1/r = prefactor

//ar Eab and Eab X Vb are calulated in dfunc_vdw, here

	prefactor = inv_SS2dis * dE_dr;

//ar V'x V
	S2xS(1) = S2xyz(2) * Sxyz(3) - S2xyz(3) * Sxyz(2);
	S2xS(2) = S2xyz(3) * Sxyz(1) - S2xyz(1) * Sxyz(3);
	S2xS(3) = S2xyz(1) * Sxyz(2) - S2xyz(2) * Sxyz(1);

// in/decrements to the f1 and f2 of the angle moving donor/acceptor
	deriv(1,1) = prefactor * S2xS(1); //f1
	deriv(2,1) = prefactor * S2xS(2);
	deriv(3,1) = prefactor * S2xS(3);

	deriv(1,2) = prefactor * SS2(1); //f2
	deriv(2,2) = prefactor * SS2(2);
	deriv(3,2) = prefactor * SS2(3);

//ar gradient component for xS2 (theta)
//ar (see comments below for xS gradient)
	BCb2(1) = S2Cb2unit(1) - xS2*SS2unit(1);
	BCb2(2) = S2Cb2unit(2) - xS2*SS2unit(2);
	BCb2(3) = S2Cb2unit(3) - xS2*SS2unit(3);

	prefactor = inv_SS2dis * dE_dxS2;
	BCb2(1) *= prefactor;
	BCb2(2) *= prefactor;
	BCb2(3) *= prefactor;

	BCb2xS(1) = BCb2(2) * Sxyz(3) - BCb2(3) * Sxyz(2);
	BCb2xS(2) = BCb2(3) * Sxyz(1) - BCb2(1) * Sxyz(3);
	BCb2xS(3) = BCb2(1) * Sxyz(2) - BCb2(2) * Sxyz(1);


//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor
	deriv(1,1) += BCb2xS(1); //f1,theta
	deriv(2,1) += BCb2xS(2);
	deriv(3,1) += BCb2xS(3);

	deriv(1,2) += BCb2(1); //f2,theta
	deriv(2,2) += BCb2(2);
	deriv(3,2) += BCb2(3);

//BW gradient component for xS
//bs copied from hbond code, with following correspondence:
//bs A1 -> Cb
//bs A  -> S
//bs H  -> S2
//bs D  -> Cb2
//bs xH -> xS
//bs xD -> xS2
//ar   (from code and paper by Bill Wedemeyer)
//ar  xS = (CbSunit . SS2)/SS2dis
//ar  dxS/dphi = 1/SS2dis * (CbSunit . dSS2/dphi - xS *dSS2dis/dphi)
//ar    (note dCbSunit/dphi = 0)
//ar
//ar  dSS2dis/dphi = SS2unit . dSS2/dphi
//ar
//ar  substituting and rearranging....
//ar  dxS/dphi = 1/SS2dis * dSS2/dphi . (CbSunit - xS*SS2unit)
//ar           =  1/SS2dis * dSS2/dphi . BS2
//ar
//ar note: BS2= (CbSunit - xS*SS2unit) = component of CbSunit that is
//ar       perpendicular to SS2unit
//ar
//ar dSS2/dphi =  Eab x (V-Vb)  . (V' - V)/|V-V'|   from above
//ar dSS2/dphi =  Eab x (H-Vb)  . SS2unit
//ar
//ar dExS/dphi = dExS/dxS * dxS/dphi
//ar           = dExS/dxS * 1/SS2dis * dSS2/dphi . BS2
//ar           = dExS/dxS * 1/SS2dis * (Eab x (H-Vb) .(H-A)/SS2dis) . BS2
//ar
//ar rearrange as with dEr/dr above to get f1 and f2 component

//ar f1 = dE/dxS *(1/SS2dis) * BS2 x H
//ar f2 = dE/dxS *(1/SS2dis) * BS2

	BS2(1) = CbSunit(1) - xS*SS2unit(1);
	BS2(2) = CbSunit(2) - xS*SS2unit(2);
	BS2(3) = CbSunit(3) - xS*SS2unit(3);

	prefactor = inv_SS2dis * dE_dxS;
	BS2(1) *= prefactor;
	BS2(2) *= prefactor;
	BS2(3) *= prefactor;

	BS2xS(1) = BS2(2) * S2xyz(3) - BS2(3) * S2xyz(2);
	BS2xS(2) = BS2(3) * S2xyz(1) - BS2(1) * S2xyz(3);
	BS2xS(3) = BS2(1) * S2xyz(2) - BS2(2) * S2xyz(1);

//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor

	deriv(1,1) += BS2xS(1);
	deriv(2,1) += BS2xS(2);
	deriv(3,1) += BS2xS(3);

	deriv(1,2) += BS2(1);
	deriv(2,2) += BS2(2);
	deriv(3,2) += BS2(3);

}

//------------------------------------------------------------------------------

void
Disulfides::disulf_csang_compute_energy(
	float const SS2dis,
	float const xS,
	float const xS2,
	float & E_r,
	float & E_xS,
	float & E_xS2,
	float & dE_dr,
	float & dE_dxS,
	float & dE_dxS2
)
{
//bs derivatives returned weighted, energies returned un-weighted

	using namespace param_pack;
	using namespace scorefxns;

// double_gauss fit to cos(csangle) prob. density
	float const base = { 3. };
	float const amp1 = { -4.4358 };
	float const amp2 = { -1.7374 };
	float const mean1 = { -0.24025 };
	float const mean2 = { -0.31186 };
	float const width1 = { 0.098048 };
	float const width2 = { 0.22224 };

// note: derivative of dbl_gauss is encoded as a function with same parameters
// (x,base,amp1,amp2,mean1,mean2,width1,width2)

// weights
//bs sum of the wts is correct b/c disulf ssdist score can contribute
//bs to total score independently in two places
//bs fullatom_energy wts: Wdisulf_dist_sc*fa_pair_weight
//bs score12 xtra contribution (optional call in scorefxns.f):
//bs disulfide_distance_weight

//bs For the xS wt there is division by two because the csang score
//bs is the average of the score of the two angles (E_xS+E_xS2)/2.
//bs so the derivative must also be the average (dE_dxS+dE_dxS2)/2.

	float const E_r_wt =
	 Wdisulf_dist_sc*fa_pair_weight + disulfide_distance_weight;
	float const E_xS_wt =
	 ( Wdisulf_csang_sc*fa_pair_weight + disulfide_csangles_weight ) / 2.0;

	E_r = 0.0;
	E_xS = dbl_gauss(xS,base,amp1,amp2,mean1,mean2,width1,width2);
	E_xS2 = dbl_gauss(xS2,base,amp1,amp2,mean1,mean2,width1,width2);

	dE_dr = get_disulf_ssdist_deriv_II(SS2dis);
	dE_dxS = d_double_gauss_d_x(xS,base,amp1,amp2,mean1,mean2,width1,width2);
	dE_dxS2 = d_double_gauss_d_x(xS2,base,amp1,amp2,mean1,mean2,width1,width2);

//	std::cout << "disulf_comp  r " << ' ' << F( 8, 3, SS2dis ) <<
//	 ' ' << F( 8, 3, E_r ) << ' ' << F( 8, 3, dE_dr ) << std::endl;
//	std::cout << "disulf comp xS " << ' ' << F( 9, 4, xS ) <<
//	 ' ' << F( 9, 4, E_xS ) << ' ' << F( 9, 4, dE_dxS ) << std::endl;
//	std::cout << "disulf comp xS2" << ' ' << F( 9, 4, xS2 ) <<
//	 ' ' << F( 9, 4, E_xS2 ) << ' ' << F( 9, 4, dE_dxS2 ) << std::endl;

// put wts on derivatives
	dE_dr *= E_r_wt;
	dE_dxS *= E_xS_wt;
	dE_dxS2 *= E_xS_wt;

// put wts on energies too for debugging
	E_r *= E_r_wt;
	E_xS *= E_xS_wt;
	E_xS2 *= E_xS_wt;

	std::cout << "disulf_comp  r " << ' ' << F( 8, 3, SS2dis ) <<
	 ' ' << F( 8, 3, E_r ) << ' ' << F( 8, 3, dE_dr ) << std::endl;
	std::cout << "disulf comp xS " << ' ' << F( 9, 4, xS ) <<
	 ' ' << F( 9, 4, E_xS ) << ' ' << F( 9, 4, dE_dxS ) << std::endl;
	std::cout << "disulf comp xS2" << ' ' << F( 9, 4, xS2 ) <<
	 ' ' << F( 9, 4, E_xS2 ) << ' ' << F( 9, 4, dE_dxS2 ) << std::endl;

}

//------------------------------------------------------------------------------

float
Disulfides::get_disulf_ssdist_deriv_II( float const ssdist )
{
	int const nbins = { 300 }; // # of bins in ssdist deriv
	float const bin_size = { 0.005 };
	 // bin width for ssdist deriv // in ang, first bin is 1.0-1.005
	float const min_bin_val = { 1.5 };
	float const max_bin_val = { min_bin_val + ( nbins - 1 ) * bin_size };
	static FArray1D_float const deriv_ssdist_sc_interp300( nbins,
	 deriv_ssdist_sc_interp300_initializer ); // deriv of ssdist_score_interp300
	FArray1D_float current_deriv( nbins );

//bs this function corresponds to the score ssdist_score_interp300
//bs it starts at ssdist=1.5 and ends at ssdist = 2.995 and it has 300 points.

	float get_disulf_ssdist_deriv_II = 0.0; // Return value

	for ( int ii = 1; ii <= nbins; ++ii ) {
		current_deriv(ii) = deriv_ssdist_sc_interp300(ii);
	}

//bs gross error checks

	if ( ssdist <= 0.0 ) {
		std::cout << "error get_disulf_ssdist_deriv_II, ssdist = " <<
		 SS( ssdist ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( ssdist >= max_bin_val ) {
		get_disulf_ssdist_deriv_II = current_deriv(nbins);
	} else if ( ssdist <= min_bin_val ) {
		get_disulf_ssdist_deriv_II = current_deriv(1);
	} else {
		int const bin =
		 static_cast< int >( ( ssdist - min_bin_val ) / bin_size + 1.0 );
		get_disulf_ssdist_deriv_II = current_deriv(bin);
	}

	return get_disulf_ssdist_deriv_II;
}

//bs****************************************************************************
//bs given residue num for 2 cys involved in disulfide bond, returns
//bs four quantities:
//bs sulf-sulf dist, 2 carb-sulf bond angles, and a 4-atom dihedral angle.
//bs Angles are returned in degrees.

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulfide_params
///
/// @brief
///     Calculates several geometrical parameters of a fullatom disulfide bond
/// @detailed
///bs given residue num for 2 cys involved in disulfide bond, returns
///bs four quantities:
///bs sulf-sulf dist, 2 carb-sulf bond angles, and a 4-atom dihedral angle.
///bs Angles are returned in degrees.
///
/// @param[in]   coord1 - in - fullatom coords of cys 1
/// @param[in]   coord2 - in - fullatom coords of cys 2
/// @param[out]   dist_between_sulfurs - out - s1-s2 distance
/// @param[out]   cs_bond_angle_1 - out - cb1-s1-s2 bond angle
/// @param[out]   cs_bond_angle_2 - out - cb2-s1-s2 bond angle
/// @param[out]   disulf_dihedral_angle - out - cb1-s1-s2-cb2 dihedral
/// @param[out]   disulf_ca_dihedral_angle_1 - out - ca1-cb1-s1-s2 dihedral
/// @param[out]   disulf_ca_dihedral_angle_2 - out - ca2-cb2-s2-s1 dihedral
///
/// @global_read
///     MAX_ATOM
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulfide_params(
	FArray2Da_float coord1, // coords of cys 1
	FArray2Da_float coord2, // coords of cys 2
	float & dist_between_sulfurs, // dist between cys sulfurs
	float & cs_bond_angle_1,
	float & cs_bond_angle_2,
	float & disulf_dihedral_angle, // dihedral (torsion) angle, cbeta-s-s-cbeta
	float & disulf_ca_dihedral_angle_1,
	 // dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	float & disulf_ca_dihedral_angle_2 // dihedral (torsion) angle, calpha2-cbeta2-s2-s1
)
{
	using namespace param;
	using numeric::conversions::to_degrees;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

//bs     int cys_1,cys_2;        // residue num of cys in disulf

//bs local
	int jj;
	FArray1D_float calpha_1( 3 );
	FArray1D_float cbeta_1( 3 );
	FArray1D_float sulfur_1( 3 );
	FArray1D_float calpha_2( 3 );
	FArray1D_float cbeta_2( 3 );
	FArray1D_float sulfur_2( 3 );

//bs Notes:
//bs   full_coord is a global in misc.h
//cc   full_coord(i,atom_num,res_num) gives coordinates
//cc   (i=1,2,3 for x,y,z) for a particular atom in a particular residue.
//cc   For sulfur in a cys residue, the atom_num is 6.
//$$$Atom numbering for cys residue
//$$$(adapted from a .pdb file)
//$$$
//$$$ATOM    1    N   CYS    21
//$$$ATOM    2    CA  CYS    21
//$$$ATOM    3    C   CYS    21
//$$$ATOM    4    O   CYS    21
//$$$ATOM    5    CB  CYS    21
//$$$ATOM    6    SG  CYS    21
//$$$ATOM    7    H   CYS    21
//$$$ATOM    8    HA  CYS    21
//$$$ATOM    9   1HB  CYS    21
//$$$ATOM    10  2HB  CYS    21
//bs NOTE THAT POSITION ARRAY HAS A DIFFERENT ORDER!!! (SEE README FOR ROSETTA)

//$$$cbs evaluate disulf using coords stored in global array.
//$$$       for ( jj = 1; jj <= 3; ++jj ) {
//$$$          calpha_1(jj) = full_coord(jj,2,cys_1);
//$$$          cbeta_1(jj) = full_coord(jj,5,cys_1);
//$$$          sulfur_1(jj) = full_coord(jj,6,cys_1);
//$$$          calpha_2(jj) = full_coord(jj,2,cys_2);
//$$$          cbeta_2(jj) = full_coord(jj,5,cys_2);
//$$$          sulfur_2(jj) = full_coord(jj,6,cys_2);
//$$$       }

	for ( jj = 1; jj <= 3; ++jj ) {
		calpha_1(jj) = coord1(jj,2);
		cbeta_1(jj) = coord1(jj,5);
		sulfur_1(jj) = coord1(jj,6);
		calpha_2(jj) = coord2(jj,2);
		cbeta_2(jj) = coord2(jj,5);
		sulfur_2(jj) = coord2(jj,6);
	}

	distance_bk(sulfur_1,sulfur_2,dist_between_sulfurs);
	angle_bk(sulfur_1,cbeta_1,sulfur_1,sulfur_2,cs_bond_angle_1);
	angle_bk(sulfur_2,cbeta_2,sulfur_2,sulfur_1,cs_bond_angle_2);
	to_degrees( cs_bond_angle_1 ); // convert
	to_degrees( cs_bond_angle_2 ); // convert
	disulf_dihedral_angle =                              /* already in degrees */
	 dihedral(cbeta_1,sulfur_1,sulfur_2,cbeta_2);
	disulf_ca_dihedral_angle_1 =                         /* already in degrees */
	 dihedral(calpha_1,cbeta_1,sulfur_1,sulfur_2);
	disulf_ca_dihedral_angle_2 =                         /* already in degrees */
	 dihedral(calpha_2,cbeta_2,sulfur_2,sulfur_1);
}


//bs****************************************************************************
//bs Given residue num for 2 cys involved in disulfide bond, returns
//bs the sulf-sulf dist. Uses coords for cys stored in current global
//bs arrays.

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_dist_use_globl_crds
///
/// @brief
///
/// @detailed
///
/// @param[in]   cys_1 - in - residue number for cys 1
/// @param[in]   cys_2 - in - residue number for cys 2
/// @param[out]   dist_between_sulfurs - out - s-s distance in angstroms
///
/// @global_read
///     full_coord
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_dist_use_globl_crds(
	int & cys_1,
	int & cys_2,
	float & dist_between_sulfurs // dist between cys sulfurs
)
{

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal

//bs local
	FArray1D_float sulfur_1( 3 ); // x,y,z coords for the sulfur atoms
	FArray1D_float sulfur_2( 3 ); // involved in disulfide bond.

//bs Notes:
//bs   full_coord is a global in misc.h
//cc   full_coord(i,atom_num,res_num) gives coordinates
//cc   (i=1,2,3 for x,y,z) for a particular atom in a particular residue.
//cc   For sulfur in a cys residue, the atom_num is 6.
//$$$Atom numbering for cys residue
//$$$(adapted from a .pdb file)
//$$$
//$$$ATOM    1    N   CYS    21
//$$$ATOM    2    CA  CYS    21
//$$$ATOM    3    C   CYS    21
//$$$ATOM    4    O   CYS    21
//$$$ATOM    5    CB  CYS    21
//$$$ATOM    6    SG  CYS    21
//$$$ATOM    7    H   CYS    21
//$$$ATOM    8    HA  CYS    21
//$$$ATOM    9   1HB  CYS    21
//$$$ATOM    10  2HB  CYS    21

	for ( int jj = 1; jj <= 3; ++jj ) {
		sulfur_1(jj) = full_coord(jj,6,cys_1);
		sulfur_2(jj) = full_coord(jj,6,cys_2);
	}

	distance_bk(sulfur_1,sulfur_2,dist_between_sulfurs);

}


//bs****************************************************************************
//bs returns backbone dihedral angle for a disulfide bond
//bs N_1-Calpha_1-Calpha_2-Cprime_2
//bs Note: returns values from 0 to 360, not -180 to 180.
//bs (converts the -180 to 0 range to lie between 180 & 360.)

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_bb_dih
///
/// @brief
///     calculates a backbone dihedral angle for a disulfide bond
/// @detailed
///
/// @param[in]   cys1coord - in - fa coords of cys 1
/// @param[in]   cys2coord - in - fa coords of cys 2
/// @param[out]   disulf_bb_dih - out - n1-ca1-ca2-cprime2 dihedral angle
///
/// @global_read
///
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_bb_dih(
	FArray2Da_float cys1coord, // coords of cys 1 from full_coord
	FArray2Da_float cys2coord, // coords of cys 2 from full_coord
	float & disulf_bb_dih // backbone dihedral N_1-Calpha_1-Calpha_2-Cprime_2
)
{
	using namespace param;

	cys1coord.dimension( 3, MAX_POS );
	cys2coord.dimension( 3, MAX_POS );

//bs local
	FArray1D_float nitro_1( 3 );
	FArray1D_float calpha_1( 3 );
	FArray1D_float nitro_2( 3 );
	FArray1D_float calpha_2( 3 );
	FArray1D_float cprime_2( 3 );

	for ( int jj = 1; jj <= 3; ++jj ) {
		nitro_1(jj) = cys1coord(jj,1);
		calpha_1(jj) = cys1coord(jj,2);
		nitro_2(jj) = cys2coord(jj,1);
		calpha_2(jj) = cys2coord(jj,2);
		cprime_2(jj) = cys2coord(jj,3);
	}

	disulf_bb_dih =                               /* already in degrees */
	 dihedral(nitro_1,calpha_1,calpha_2,cprime_2);

//bs shift it to make between 0 and 360 instead of -180 to 180
	if ( disulf_bb_dih < 0.0 ) disulf_bb_dih += 360.0;

}


//bs****************************************************************************
//bs returns backbone dihedral angle for a disulfide bond
//bs Calpha_1-Cbeta1-Calpha_2-Cbeta_2

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_cacbcbca_dih
///
/// @brief
///
/// @detailed
///bs returns backbone dihedral angle for a disulfide bond
///bs Calpha_1-Cbeta1-Calpha_2-Cbeta_2
///
/// @param[in]   cys1coord - in - fa coords of cys 1
/// @param[in]   cys2coord - in - fa coords of cys 2
/// @param[out]   disulf_cacbcbca_dih - out - in degrees
///
/// @global_read
///     MAX_ATOM
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_cacbcbca_dih(
	FArray2Da_float cys1coord, // coords of cys 1 full_coord numbering
	FArray2Da_float cys2coord, // coords of cys 2 full_coord numbering
	float & disulf_cacbcbca_dih
)
{
	using namespace param;

	cys1coord.dimension( 3, MAX_POS );
	cys2coord.dimension( 3, MAX_POS );

//bs local
	FArray1D_float cbeta_1( 3 );
	FArray1D_float calpha_1( 3 );
	FArray1D_float cbeta_2( 3 );
	FArray1D_float calpha_2( 3 );

	for ( int jj = 1; jj <= 3; ++jj ) {
		cbeta_1(jj) = cys1coord(jj,5);
		calpha_1(jj) = cys1coord(jj,2);
		cbeta_2(jj) = cys2coord(jj,5);
		calpha_2(jj) = cys2coord(jj,2);
	}

	disulf_cacbcbca_dih =                               /* already in degrees */
	 dihedral(calpha_1,cbeta_1,calpha_2,cbeta_2);

//$$$cbs shift it to make between 0 and 360 instead of -180 to 180
//$$$    if ( disulf_bb_dih < 0.0 ) disulf_bb_dih += 360.0;

}



//bs****************************************************************************

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_cacbcb_angles
///
/// @brief
///     returns values of bond angles ca1-cb1-cb2 and ca2-cb2-cb1
///     for a disulfide
/// @detailed
///
/// @param[in]   cys1coord - in - fa coords of cys 1
/// @param[in]   cys2coord - in - fa coords of cys 2
/// @param[out]   disulf_cacbcb_ang_1 - out - in degrees
/// @param[out]   disulf_cacbcb_ang_2 - out - in degrees
///
/// @global_read
///     MAX_ATOM
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_cacbcb_angles(
	FArray2Da_float cys1coord, // coords of cys 1 full_coord numbering
	FArray2Da_float cys2coord, // coords of cys 2 full_coord numbering
	float & disulf_cacbcb_ang_1,
	float & disulf_cacbcb_ang_2
)
{
	using namespace param;
	using numeric::conversions::to_degrees;

	cys1coord.dimension( 3, MAX_POS );
	cys2coord.dimension( 3, MAX_POS );

//bs local
	FArray1D_float cbeta_1( 3 );
	FArray1D_float calpha_1( 3 );
	FArray1D_float cbeta_2( 3 );
	FArray1D_float calpha_2( 3 );

	for ( int jj = 1; jj <= 3; ++jj ) {
		cbeta_1(jj) = cys1coord(jj,5);
		calpha_1(jj) = cys1coord(jj,2);
		cbeta_2(jj) = cys2coord(jj,5);
		calpha_2(jj) = cys2coord(jj,2);
	}

	angle_bk(cbeta_1,calpha_1,cbeta_1,cbeta_2,disulf_cacbcb_ang_1);
	angle_bk(cbeta_2,calpha_2,cbeta_2,cbeta_1,disulf_cacbcb_ang_2);

	to_degrees( disulf_cacbcb_ang_1 ); // convert
	to_degrees( disulf_cacbcb_ang_2 ); // convert

}


//bs****************************************************************************

////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_cbs_dist
///
/// @brief
///     calculaes distances cb1-s2 and cb2-s1 in disulfide
/// @detailed
///
/// @param[in]   cys1coord - in - fa coords cys 1
/// @param[in]   cys2coord - in - fa coords cys 2
/// @param[out]   dist_cbs_1 - out -  cb1-s2
/// @param[out]   dist_cbs_2 - out -  cb2-s1
///
/// @global_read
///     MAX_ATOM
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_disulf_cbs_dist(
	FArray2Da_float cys1coord,
	 // coords of cys 1 from full_coord (include sidechain)
	FArray2Da_float cys2coord,
	 // coords of cys 2 from full_coord (include sidechain)
	float & dist_cbs_1,
	float & dist_cbs_2
)
{
	using namespace param;

	cys1coord.dimension( 3, MAX_ATOM() );
	cys2coord.dimension( 3, MAX_ATOM() );

//bs local
	FArray1D_float cbeta_1( 3 );
	FArray1D_float sulfur_1( 3 );
	FArray1D_float cbeta_2( 3 );
	FArray1D_float sulfur_2( 3 );

	for ( int jj = 1; jj <= 3; ++jj ) {
		cbeta_1(jj) = cys1coord(jj,5);
		sulfur_1(jj) = cys1coord(jj,6);
		cbeta_2(jj) = cys2coord(jj,5);
		sulfur_2(jj) = cys2coord(jj,6);
	}

	distance_bk(cbeta_1,sulfur_2,dist_cbs_1);
	distance_bk(cbeta_2,sulfur_1,dist_cbs_2);

}



//bs****************************************************************************
//bs returns the Cb-S distance score for a disulfide bond.

//bs This score is intended to be used in conjuction with count_pair being
//bs false for Cb-S in disulfides.
//bs This score is necessary to get the Cb-S distance correct.
//bs the score is normalized between 0 and -1 (attractive only).
//bs the score is based on a 2 gaussian fit to the (normalized) native histogram of
//bs crystal structures with resolution better than 2.1 angstroms.

//bs the score is under development...commments may be inaccurate.
//bs The score is -ln(P(natives)/P(decoys)). Here P(decoys) was the normalized
//bs histogram of cbs_dist measured with no cbs_dist score.
//bs Instead of directly entering the histogram of -ln(...), which
//bs is a well with minimum at 3.0 ang (ymin = -0.1,ymax=+4 at 2.8&3.4 ang),
//bs I have fit it with 2 gaussians, a y-shift, and an inversion.
//bs (a good fit).

////////////////////////////////////////////////////////////////////////////////
/// @begin get_cbs_sc_this_disulf
///
/// @brief
///bs returns the Cb-S distance score for a disulfide bond.
/// @detailed
///
/// @param[in]   cys1coord - in - fa coords cys 1
/// @param[in]   cys2coord - in - fa coords cys 2
/// @param[out]   cbs_sc_this_disulf - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::get_cbs_sc_this_disulf(
	FArray2Da_float cys1coord,
	 // coords of cys 1 from full_coord (include sidechain)
	FArray2Da_float cys2coord,
	 // coords of cys 2 from full_coord (include sidechain)
	float & cbs_sc_this_disulf
)
{
	using namespace param;

	cys1coord.dimension( 3, MAX_ATOM() );
	cys2coord.dimension( 3, MAX_ATOM() );

//bs local
	float dist_cbs_1; // Cb-S distance (cys1->cys2)
	float dist_cbs_2; // Cb-S distance (cys2->cys1)
	float cbs_sc_1; // cbs score for dist_cbs_1 (0 < cbs_sc_1 < 1)
	float cbs_sc_2; // cbs score for dist_cbs_2 (0 < cbs_sc_2 < 1)

//      float y_shift;     // y-shift of data before gauss fit (together with inversion)

//bs these are the params of a 2 gauss fit to the (normalized) native histogram
	float const base = { 0.0 }; // additive offset in the 2 gauss fit
	float const amp1 = { 0.7448 };
	float const mean1 = { 3.0256 };
	float const sdev1 = { 0.04217 };
	float const amp2 = { 0.2454 };
	float const mean2 = { 3.0298 };
	float const sdev2 = { 0.09273 };
//$$$cbs these are the params of a 2 gauss fit to -ln(Pnat/Pdec)
//$$$cbs this score gave the best improvement in the cbs distribution
//$$$cbs of decoys, but it resulted in poor distributions for
//$$$cbs other parameters (csangles).
//$$$      parameter (base = 0.7453)
//$$$      parameter (amp1 = 2.734)
//$$$      parameter (mean1 = 3.019)
//$$$      parameter (sdev1 = 0.199)
//$$$      parameter (amp2 = 1.501)
//$$$      parameter (mean2 = 3.025)
//$$$      parameter (sdev2 = 0.0593)
//$$$      parameter (y_shift = 4.88452)

	get_disulf_cbs_dist(cys1coord,cys2coord,dist_cbs_1,dist_cbs_2);

	float const dev11 = ( dist_cbs_1 - mean1 ) / sdev1;
	float const dev12 = ( dist_cbs_1 - mean2 ) / sdev2;
	float const dev21 = ( dist_cbs_2 - mean1 ) / sdev1;
	float const dev22 = ( dist_cbs_2 - mean2 ) / sdev2;

//$$$cbs score is 2 gaussian fit to Cb-S (cys1->2) and
//$$$cbs          2 gaussian fit to Cb-S (cys2->1)
//$$$cbs so there are 4 gaussians total.
//$$$      cbs_sc_this_disulf =
//$$$         -( amp1 * std::exp( -( dev11 * dev11 ) ) +
//$$$            amp2 * std::exp( -( dev12 * dev12 ) ) +
//$$$            amp1 * std::exp( -( dev21 * dev21 ) ) +
//$$$            amp2 * std::exp( -( dev22 * dev22 ) ) );

//bs separately score Cb-S (1->2) and Cb-S (2->1)
	cbs_sc_1 = base +
	 amp1 * std::exp( -( dev11 * dev11 ) ) +
	 amp2 * std::exp( -( dev12 * dev12 ) );
	cbs_sc_2 = base +
	 amp1 * std::exp( -( dev21 * dev21 ) ) +
	 amp2 * std::exp( -( dev22 * dev22 ) );

//$$$cbs shift and invert to recover the proper scores
//$$$cbs only used for  the -ln(..) score
//$$$      cbs_sc_1 = -(cbs_sc_1-y_shift);
//$$$      cbs_sc_2 = -(cbs_sc_2-y_shift);


//bs take the average for the total score, and make it (-)
	cbs_sc_this_disulf = -((cbs_sc_1+cbs_sc_2)*.5);

//$$$cbs take the average for the total score
//$$$ used for the -ln(..) score
//$$$      cbs_sc_this_disulf = (cbs_sc_1+cbs_sc_2)*.5;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_disulf_cendist_sc
///
/// @brief
///     returns value of global disulf_cendist_score
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_disulf_cendist_sc()
{
	using namespace scores;

	return disulf_cendist_score;
}


//------------------------------------------------------------------------------

int
Disulfides::get_disulf_Ncen_sc()
{
	using namespace disulfides::options;

	int get_disulf_Ncen_sc = 0; // Return value

	if ( fix_disulf ) {
		get_disulf_Ncen_sc = n_known_disulfides-n_disulf_cent_natcut;
	}

	return get_disulf_Ncen_sc;
}

//------------------------------------------------------------------------------

int
Disulfides::get_disulf_Ncb_sc()
{
	using namespace disulfides::options;

	int get_disulf_Ncb_sc = 0; // Return value

	if ( fix_disulf ) {
		get_disulf_Ncb_sc = n_known_disulfides-n_disulf_cb_natcut;
	}

	return get_disulf_Ncb_sc;
}



//------------------------------------------------------------------------------

int
Disulfides::get_disulf_Nfa_sc()
{
	using namespace disulfides::options;

	int get_disulf_Nfa_sc = 0; // Return value

	if ( fix_disulf ) {
		get_disulf_Nfa_sc = n_known_disulfides-n_disulfides_fa;
	}

	return get_disulf_Nfa_sc;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_fa_disulf_cbs_dist_sc
///
/// @brief
///     returns value of global disulfide_cbs_dist_score
/// @detailed
///     this score based on the distance from cb1-s2
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::get_fa_disulf_cbs_dist_sc()
{
	return disulfide_cbs_dist_score;
}



//bs****************************************************************************
//bs opens logfile and scrolls to the end.

////////////////////////////////////////////////////////////////////////////////
/// @begin open_disulf_logfile
///
/// @brief
///bs opens logfile and scrolls to the end.
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::open_disulf_logfile()
{
	using namespace disulfides::options;
	using namespace files_paths;

//$$$    std::string stat( 3, ' ' );
//$$$    static bool first_time = { true };
//$$$    common /disulf_logfile_init/ first_time
//$$$
//$$$cbs std::cout << "in open_disulf_logfile, first_time = " << first_time << std::endl;
//$$$
//$$$    if ( first_time ) {
//$$$       stat = "new";
//$$$       first_time = false;
//$$$    } else {
//$$$       stat = "old";
//$$$       first_time = false;
//$$$    }

//bs     std::cout << "now, first_time = " << first_time << std::endl;
//bs     std::cout << "stat = " << stat << std::endl;

	if ( use_disulf_logfile ) {

		disulf_log_x.clear();
		disulf_log_x.open( disulf_logfile.c_str(), std::ios_base::out|std::ios_base::app );
//$$$       open( disulf_log_x,disulf_logfile,status=stat,iostat=i)

		if ( !disulf_log_x ) {
			std::cout << "ERROR:: failed to open disulf_logfile" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}
}

///////////////////////////////////////////
/// Private: Methods, Data Initializers ///
///////////////////////////////////////////


void
Disulfides::bb_dih_cnrlx_score_initializer( FArray1D_float & bb_dih_cnrlx_score )
{
	int i = 0;
	bb_dih_cnrlx_score( ++i ) = 0.424;
	bb_dih_cnrlx_score( ++i ) = 0.817;
	bb_dih_cnrlx_score( ++i ) = -0.239;
	bb_dih_cnrlx_score( ++i ) = -0.012;
	bb_dih_cnrlx_score( ++i ) = 0.491;
	bb_dih_cnrlx_score( ++i ) = 0.807;
	bb_dih_cnrlx_score( ++i ) = 0.977;
	bb_dih_cnrlx_score( ++i ) = 0.751;
	bb_dih_cnrlx_score( ++i ) = 0.458;
	bb_dih_cnrlx_score( ++i ) = 0.287;
	bb_dih_cnrlx_score( ++i ) = 0.208;
	bb_dih_cnrlx_score( ++i ) = -0.400;
	bb_dih_cnrlx_score( ++i ) = -0.389;
	bb_dih_cnrlx_score( ++i ) = -0.516;
	bb_dih_cnrlx_score( ++i ) = 0.414;
	bb_dih_cnrlx_score( ++i ) = -0.460;
	bb_dih_cnrlx_score( ++i ) = 0.042;
	bb_dih_cnrlx_score( ++i ) = 0.182;
	bb_dih_cnrlx_score( ++i ) = -0.040;
	bb_dih_cnrlx_score( ++i ) = 0.746;
	bb_dih_cnrlx_score( ++i ) = -0.140;
	bb_dih_cnrlx_score( ++i ) = 0.168;
	bb_dih_cnrlx_score( ++i ) = 1.467;
	bb_dih_cnrlx_score( ++i ) = 1.429;
	bb_dih_cnrlx_score( ++i ) = 0.295;
	bb_dih_cnrlx_score( ++i ) = 1.867;
	bb_dih_cnrlx_score( ++i ) = 1.100;
	bb_dih_cnrlx_score( ++i ) = 1.812;
	bb_dih_cnrlx_score( ++i ) = 1.090;
	bb_dih_cnrlx_score( ++i ) = 1.766;
	bb_dih_cnrlx_score( ++i ) = 1.081;
	bb_dih_cnrlx_score( ++i ) = 1.081;
	bb_dih_cnrlx_score( ++i ) = 1.068;
	bb_dih_cnrlx_score( ++i ) = 0.408;
	bb_dih_cnrlx_score( ++i ) = -0.013;
	bb_dih_cnrlx_score( ++i ) = -0.478;
	bb_dih_cnrlx_score( ++i ) = -0.235;
	bb_dih_cnrlx_score( ++i ) = -0.040;
	bb_dih_cnrlx_score( ++i ) = -0.130;
	bb_dih_cnrlx_score( ++i ) = -0.132;
	bb_dih_cnrlx_score( ++i ) = 0.017;
	bb_dih_cnrlx_score( ++i ) = -0.374;
	bb_dih_cnrlx_score( ++i ) = -0.398;
	bb_dih_cnrlx_score( ++i ) = -1.034;
	bb_dih_cnrlx_score( ++i ) = -0.921;
	bb_dih_cnrlx_score( ++i ) = -0.920;
	bb_dih_cnrlx_score( ++i ) = -1.083;
	bb_dih_cnrlx_score( ++i ) = -0.911;
	bb_dih_cnrlx_score( ++i ) = -1.346;
	bb_dih_cnrlx_score( ++i ) = -0.354;
	bb_dih_cnrlx_score( ++i ) = -0.078;
	bb_dih_cnrlx_score( ++i ) = -0.250;
	bb_dih_cnrlx_score( ++i ) = 0.052;
	bb_dih_cnrlx_score( ++i ) = 0.922;
	bb_dih_cnrlx_score( ++i ) = -0.028;
	bb_dih_cnrlx_score( ++i ) = 0.582;
	bb_dih_cnrlx_score( ++i ) = 0.570;
	bb_dih_cnrlx_score( ++i ) = -0.034;
	bb_dih_cnrlx_score( ++i ) = 0.349;
	bb_dih_cnrlx_score( ++i ) = 0.361;
	bb_dih_cnrlx_score( ++i ) = 1.198;
	bb_dih_cnrlx_score( ++i ) = 1.212;
	bb_dih_cnrlx_score( ++i ) = 1.239;
	bb_dih_cnrlx_score( ++i ) = 0.152;
	bb_dih_cnrlx_score( ++i ) = 0.645;
	bb_dih_cnrlx_score( ++i ) = 2.170;
	bb_dih_cnrlx_score( ++i ) = 1.532;
	bb_dih_cnrlx_score( ++i ) = 1.489;
	bb_dih_cnrlx_score( ++i ) = 1.456;
	bb_dih_cnrlx_score( ++i ) = 1.069;
	bb_dih_cnrlx_score( ++i ) = 1.528;
	bb_dih_cnrlx_score( ++i ) = 2.156;
	bb_dih_cnrlx_score( ++i ) = 1.455;
	bb_dih_cnrlx_score( ++i ) = 1.050;
	bb_dih_cnrlx_score( ++i ) = 0.839;
	bb_dih_cnrlx_score( ++i ) = 0.385;
	bb_dih_cnrlx_score( ++i ) = 0.603;
	bb_dih_cnrlx_score( ++i ) = -0.049;
	bb_dih_cnrlx_score( ++i ) = 0.077;
	bb_dih_cnrlx_score( ++i ) = 0.820;
	bb_dih_cnrlx_score( ++i ) = 0.211;
	bb_dih_cnrlx_score( ++i ) = 0.836;
	bb_dih_cnrlx_score( ++i ) = 0.515;
	bb_dih_cnrlx_score( ++i ) = -0.743;
	bb_dih_cnrlx_score( ++i ) = 0.428;
	bb_dih_cnrlx_score( ++i ) = -0.202;
	bb_dih_cnrlx_score( ++i ) = -0.606;
	bb_dih_cnrlx_score( ++i ) = -0.536;
	bb_dih_cnrlx_score( ++i ) = 0.053;
	bb_dih_cnrlx_score( ++i ) = -0.246;
	bb_dih_cnrlx_score( ++i ) = -0.078;
	bb_dih_cnrlx_score( ++i ) = -0.876;
	bb_dih_cnrlx_score( ++i ) = -0.446;
	bb_dih_cnrlx_score( ++i ) = -0.681;
	bb_dih_cnrlx_score( ++i ) = 0.071;
	bb_dih_cnrlx_score( ++i ) = -0.078;
	bb_dih_cnrlx_score( ++i ) = -0.467;
	bb_dih_cnrlx_score( ++i ) = -0.223;
	bb_dih_cnrlx_score( ++i ) = 0.272;
	bb_dih_cnrlx_score( ++i ) = 1.119;
}


void
Disulfides::bb_dih_cnrlx_score_sm_initializer( FArray1D_float & bb_dih_cnrlx_score_sm )
{
	int i = 0;
	bb_dih_cnrlx_score_sm( ++i ) = 0.349;
	bb_dih_cnrlx_score_sm( ++i ) = 0.347;
	bb_dih_cnrlx_score_sm( ++i ) = 0.352;
	bb_dih_cnrlx_score_sm( ++i ) = 0.391;
	bb_dih_cnrlx_score_sm( ++i ) = 0.466;
	bb_dih_cnrlx_score_sm( ++i ) = 0.544;
	bb_dih_cnrlx_score_sm( ++i ) = 0.580;
	bb_dih_cnrlx_score_sm( ++i ) = 0.542;
	bb_dih_cnrlx_score_sm( ++i ) = 0.429;
	bb_dih_cnrlx_score_sm( ++i ) = 0.267;
	bb_dih_cnrlx_score_sm( ++i ) = 0.096;
	bb_dih_cnrlx_score_sm( ++i ) = -0.046;
	bb_dih_cnrlx_score_sm( ++i ) = -0.131;
	bb_dih_cnrlx_score_sm( ++i ) = -0.153;
	bb_dih_cnrlx_score_sm( ++i ) = -0.125;
	bb_dih_cnrlx_score_sm( ++i ) = -0.065;
	bb_dih_cnrlx_score_sm( ++i ) = 0.009;
	bb_dih_cnrlx_score_sm( ++i ) = 0.093;
	bb_dih_cnrlx_score_sm( ++i ) = 0.188;
	bb_dih_cnrlx_score_sm( ++i ) = 0.302;
	bb_dih_cnrlx_score_sm( ++i ) = 0.447;
	bb_dih_cnrlx_score_sm( ++i ) = 0.620;
	bb_dih_cnrlx_score_sm( ++i ) = 0.805;
	bb_dih_cnrlx_score_sm( ++i ) = 0.977;
	bb_dih_cnrlx_score_sm( ++i ) = 1.121;
	bb_dih_cnrlx_score_sm( ++i ) = 1.234;
	bb_dih_cnrlx_score_sm( ++i ) = 1.314;
	bb_dih_cnrlx_score_sm( ++i ) = 1.353;
	bb_dih_cnrlx_score_sm( ++i ) = 1.343;
	bb_dih_cnrlx_score_sm( ++i ) = 1.276;
	bb_dih_cnrlx_score_sm( ++i ) = 1.148;
	bb_dih_cnrlx_score_sm( ++i ) = 0.959;
	bb_dih_cnrlx_score_sm( ++i ) = 0.720;
	bb_dih_cnrlx_score_sm( ++i ) = 0.459;
	bb_dih_cnrlx_score_sm( ++i ) = 0.215;
	bb_dih_cnrlx_score_sm( ++i ) = 0.030;
	bb_dih_cnrlx_score_sm( ++i ) = -0.081;
	bb_dih_cnrlx_score_sm( ++i ) = -0.133;
	bb_dih_cnrlx_score_sm( ++i ) = -0.160;
	bb_dih_cnrlx_score_sm( ++i ) = -0.200;
	bb_dih_cnrlx_score_sm( ++i ) = -0.279;
	bb_dih_cnrlx_score_sm( ++i ) = -0.401;
	bb_dih_cnrlx_score_sm( ++i ) = -0.549;
	bb_dih_cnrlx_score_sm( ++i ) = -0.697;
	bb_dih_cnrlx_score_sm( ++i ) = -0.818;
	bb_dih_cnrlx_score_sm( ++i ) = -0.890;
	bb_dih_cnrlx_score_sm( ++i ) = -0.902;
	bb_dih_cnrlx_score_sm( ++i ) = -0.844;
	bb_dih_cnrlx_score_sm( ++i ) = -0.715;
	bb_dih_cnrlx_score_sm( ++i ) = -0.530;
	bb_dih_cnrlx_score_sm( ++i ) = -0.314;
	bb_dih_cnrlx_score_sm( ++i ) = -0.099;
	bb_dih_cnrlx_score_sm( ++i ) = 0.086;
	bb_dih_cnrlx_score_sm( ++i ) = 0.224;
	bb_dih_cnrlx_score_sm( ++i ) = 0.310;
	bb_dih_cnrlx_score_sm( ++i ) = 0.354;
	bb_dih_cnrlx_score_sm( ++i ) = 0.382;
	bb_dih_cnrlx_score_sm( ++i ) = 0.429;
	bb_dih_cnrlx_score_sm( ++i ) = 0.516;
	bb_dih_cnrlx_score_sm( ++i ) = 0.638;
	bb_dih_cnrlx_score_sm( ++i ) = 0.764;
	bb_dih_cnrlx_score_sm( ++i ) = 0.866;
	bb_dih_cnrlx_score_sm( ++i ) = 0.946;
	bb_dih_cnrlx_score_sm( ++i ) = 1.033;
	bb_dih_cnrlx_score_sm( ++i ) = 1.146;
	bb_dih_cnrlx_score_sm( ++i ) = 1.271;
	bb_dih_cnrlx_score_sm( ++i ) = 1.375;
	bb_dih_cnrlx_score_sm( ++i ) = 1.437;
	bb_dih_cnrlx_score_sm( ++i ) = 1.465;
	bb_dih_cnrlx_score_sm( ++i ) = 1.476;
	bb_dih_cnrlx_score_sm( ++i ) = 1.466;
	bb_dih_cnrlx_score_sm( ++i ) = 1.411;
	bb_dih_cnrlx_score_sm( ++i ) = 1.286;
	bb_dih_cnrlx_score_sm( ++i ) = 1.097;
	bb_dih_cnrlx_score_sm( ++i ) = 0.875;
	bb_dih_cnrlx_score_sm( ++i ) = 0.667;
	bb_dih_cnrlx_score_sm( ++i ) = 0.508;
	bb_dih_cnrlx_score_sm( ++i ) = 0.415;
	bb_dih_cnrlx_score_sm( ++i ) = 0.383;
	bb_dih_cnrlx_score_sm( ++i ) = 0.381;
	bb_dih_cnrlx_score_sm( ++i ) = 0.368;
	bb_dih_cnrlx_score_sm( ++i ) = 0.313;
	bb_dih_cnrlx_score_sm( ++i ) = 0.211;
	bb_dih_cnrlx_score_sm( ++i ) = 0.079;
	bb_dih_cnrlx_score_sm( ++i ) = -0.053;
	bb_dih_cnrlx_score_sm( ++i ) = -0.162;
	bb_dih_cnrlx_score_sm( ++i ) = -0.234;
	bb_dih_cnrlx_score_sm( ++i ) = -0.273;
	bb_dih_cnrlx_score_sm( ++i ) = -0.295;
	bb_dih_cnrlx_score_sm( ++i ) = -0.321;
	bb_dih_cnrlx_score_sm( ++i ) = -0.357;
	bb_dih_cnrlx_score_sm( ++i ) = -0.387;
	bb_dih_cnrlx_score_sm( ++i ) = -0.389;
	bb_dih_cnrlx_score_sm( ++i ) = -0.353;
	bb_dih_cnrlx_score_sm( ++i ) = -0.285;
	bb_dih_cnrlx_score_sm( ++i ) = -0.194;
	bb_dih_cnrlx_score_sm( ++i ) = -0.085;
	bb_dih_cnrlx_score_sm( ++i ) = 0.034;
	bb_dih_cnrlx_score_sm( ++i ) = 0.131;
	bb_dih_cnrlx_score_sm( ++i ) = 0.170;
}


void
Disulfides::bb_dih_foldcst_score_sm_initializer( FArray1D_float & bb_dih_foldcst_score_sm )
{
	int i = 0;
	bb_dih_foldcst_score_sm( ++i ) = 0.451;
	bb_dih_foldcst_score_sm( ++i ) = 0.425;
	bb_dih_foldcst_score_sm( ++i ) = 0.388;
	bb_dih_foldcst_score_sm( ++i ) = 0.407;
	bb_dih_foldcst_score_sm( ++i ) = 0.492;
	bb_dih_foldcst_score_sm( ++i ) = 0.591;
	bb_dih_foldcst_score_sm( ++i ) = 0.634;
	bb_dih_foldcst_score_sm( ++i ) = 0.587;
	bb_dih_foldcst_score_sm( ++i ) = 0.457;
	bb_dih_foldcst_score_sm( ++i ) = 0.278;
	bb_dih_foldcst_score_sm( ++i ) = 0.092;
	bb_dih_foldcst_score_sm( ++i ) = -0.060;
	bb_dih_foldcst_score_sm( ++i ) = -0.146;
	bb_dih_foldcst_score_sm( ++i ) = -0.160;
	bb_dih_foldcst_score_sm( ++i ) = -0.121;
	bb_dih_foldcst_score_sm( ++i ) = -0.055;
	bb_dih_foldcst_score_sm( ++i ) = 0.024;
	bb_dih_foldcst_score_sm( ++i ) = 0.110;
	bb_dih_foldcst_score_sm( ++i ) = 0.201;
	bb_dih_foldcst_score_sm( ++i ) = 0.305;
	bb_dih_foldcst_score_sm( ++i ) = 0.444;
	bb_dih_foldcst_score_sm( ++i ) = 0.627;
	bb_dih_foldcst_score_sm( ++i ) = 0.831;
	bb_dih_foldcst_score_sm( ++i ) = 1.019;
	bb_dih_foldcst_score_sm( ++i ) = 1.175;
	bb_dih_foldcst_score_sm( ++i ) = 1.302;
	bb_dih_foldcst_score_sm( ++i ) = 1.399;
	bb_dih_foldcst_score_sm( ++i ) = 1.452;
	bb_dih_foldcst_score_sm( ++i ) = 1.450;
	bb_dih_foldcst_score_sm( ++i ) = 1.385;
	bb_dih_foldcst_score_sm( ++i ) = 1.255;
	bb_dih_foldcst_score_sm( ++i ) = 1.057;
	bb_dih_foldcst_score_sm( ++i ) = 0.794;
	bb_dih_foldcst_score_sm( ++i ) = 0.494;
	bb_dih_foldcst_score_sm( ++i ) = 0.211;
	bb_dih_foldcst_score_sm( ++i ) = 0.004;
	bb_dih_foldcst_score_sm( ++i ) = -0.102;
	bb_dih_foldcst_score_sm( ++i ) = -0.133;
	bb_dih_foldcst_score_sm( ++i ) = -0.140;
	bb_dih_foldcst_score_sm( ++i ) = -0.177;
	bb_dih_foldcst_score_sm( ++i ) = -0.272;
	bb_dih_foldcst_score_sm( ++i ) = -0.432;
	bb_dih_foldcst_score_sm( ++i ) = -0.637;
	bb_dih_foldcst_score_sm( ++i ) = -0.851;
	bb_dih_foldcst_score_sm( ++i ) = -1.035;
	bb_dih_foldcst_score_sm( ++i ) = -1.154;
	bb_dih_foldcst_score_sm( ++i ) = -1.176;
	bb_dih_foldcst_score_sm( ++i ) = -1.084;
	bb_dih_foldcst_score_sm( ++i ) = -0.882;
	bb_dih_foldcst_score_sm( ++i ) = -0.607;
	bb_dih_foldcst_score_sm( ++i ) = -0.314;
	bb_dih_foldcst_score_sm( ++i ) = -0.051;
	bb_dih_foldcst_score_sm( ++i ) = 0.154;
	bb_dih_foldcst_score_sm( ++i ) = 0.289;
	bb_dih_foldcst_score_sm( ++i ) = 0.352;
	bb_dih_foldcst_score_sm( ++i ) = 0.362;
	bb_dih_foldcst_score_sm( ++i ) = 0.359;
	bb_dih_foldcst_score_sm( ++i ) = 0.391;
	bb_dih_foldcst_score_sm( ++i ) = 0.495;
	bb_dih_foldcst_score_sm( ++i ) = 0.655;
	bb_dih_foldcst_score_sm( ++i ) = 0.814;
	bb_dih_foldcst_score_sm( ++i ) = 0.916;
	bb_dih_foldcst_score_sm( ++i ) = 0.965;
	bb_dih_foldcst_score_sm( ++i ) = 1.026;
	bb_dih_foldcst_score_sm( ++i ) = 1.146;
	bb_dih_foldcst_score_sm( ++i ) = 1.297;
	bb_dih_foldcst_score_sm( ++i ) = 1.416;
	bb_dih_foldcst_score_sm( ++i ) = 1.469;
	bb_dih_foldcst_score_sm( ++i ) = 1.485;
	bb_dih_foldcst_score_sm( ++i ) = 1.501;
	bb_dih_foldcst_score_sm( ++i ) = 1.516;
	bb_dih_foldcst_score_sm( ++i ) = 1.481;
	bb_dih_foldcst_score_sm( ++i ) = 1.358;
	bb_dih_foldcst_score_sm( ++i ) = 1.153;
	bb_dih_foldcst_score_sm( ++i ) = 0.912;
	bb_dih_foldcst_score_sm( ++i ) = 0.686;
	bb_dih_foldcst_score_sm( ++i ) = 0.513;
	bb_dih_foldcst_score_sm( ++i ) = 0.414;
	bb_dih_foldcst_score_sm( ++i ) = 0.389;
	bb_dih_foldcst_score_sm( ++i ) = 0.405;
	bb_dih_foldcst_score_sm( ++i ) = 0.404;
	bb_dih_foldcst_score_sm( ++i ) = 0.342;
	bb_dih_foldcst_score_sm( ++i ) = 0.216;
	bb_dih_foldcst_score_sm( ++i ) = 0.060;
	bb_dih_foldcst_score_sm( ++i ) = -0.088;
	bb_dih_foldcst_score_sm( ++i ) = -0.203;
	bb_dih_foldcst_score_sm( ++i ) = -0.276;
	bb_dih_foldcst_score_sm( ++i ) = -0.310;
	bb_dih_foldcst_score_sm( ++i ) = -0.325;
	bb_dih_foldcst_score_sm( ++i ) = -0.348;
	bb_dih_foldcst_score_sm( ++i ) = -0.389;
	bb_dih_foldcst_score_sm( ++i ) = -0.422;
	bb_dih_foldcst_score_sm( ++i ) = -0.414;
	bb_dih_foldcst_score_sm( ++i ) = -0.359;
	bb_dih_foldcst_score_sm( ++i ) = -0.277;
	bb_dih_foldcst_score_sm( ++i ) = -0.185;
	bb_dih_foldcst_score_sm( ++i ) = -0.070;
	bb_dih_foldcst_score_sm( ++i ) = 0.085;
	bb_dih_foldcst_score_sm( ++i ) = 0.261;
	bb_dih_foldcst_score_sm( ++i ) = 0.400;
}


void
Disulfides::bb_dih_native_histogram_sm_initializer( FArray1D_float & bb_dih_native_histogram_sm )
{
	int i = 0;
	bb_dih_native_histogram_sm( ++i ) = 0.126;
	bb_dih_native_histogram_sm( ++i ) = 0.141;
	bb_dih_native_histogram_sm( ++i ) = 0.166;
	bb_dih_native_histogram_sm( ++i ) = 0.167;
	bb_dih_native_histogram_sm( ++i ) = 0.141;
	bb_dih_native_histogram_sm( ++i ) = 0.114;
	bb_dih_native_histogram_sm( ++i ) = 0.109;
	bb_dih_native_histogram_sm( ++i ) = 0.125;
	bb_dih_native_histogram_sm( ++i ) = 0.154;
	bb_dih_native_histogram_sm( ++i ) = 0.196;
	bb_dih_native_histogram_sm( ++i ) = 0.258;
	bb_dih_native_histogram_sm( ++i ) = 0.328;
	bb_dih_native_histogram_sm( ++i ) = 0.364;
	bb_dih_native_histogram_sm( ++i ) = 0.349;
	bb_dih_native_histogram_sm( ++i ) = 0.320;
	bb_dih_native_histogram_sm( ++i ) = 0.298;
	bb_dih_native_histogram_sm( ++i ) = 0.273;
	bb_dih_native_histogram_sm( ++i ) = 0.239;
	bb_dih_native_histogram_sm( ++i ) = 0.215;
	bb_dih_native_histogram_sm( ++i ) = 0.206;
	bb_dih_native_histogram_sm( ++i ) = 0.197;
	bb_dih_native_histogram_sm( ++i ) = 0.166;
	bb_dih_native_histogram_sm( ++i ) = 0.125;
	bb_dih_native_histogram_sm( ++i ) = 0.103;
	bb_dih_native_histogram_sm( ++i ) = 0.097;
	bb_dih_native_histogram_sm( ++i ) = 0.085;
	bb_dih_native_histogram_sm( ++i ) = 0.070;
	bb_dih_native_histogram_sm( ++i ) = 0.062;
	bb_dih_native_histogram_sm( ++i ) = 0.061;
	bb_dih_native_histogram_sm( ++i ) = 0.064;
	bb_dih_native_histogram_sm( ++i ) = 0.071;
	bb_dih_native_histogram_sm( ++i ) = 0.087;
	bb_dih_native_histogram_sm( ++i ) = 0.120;
	bb_dih_native_histogram_sm( ++i ) = 0.179;
	bb_dih_native_histogram_sm( ++i ) = 0.249;
	bb_dih_native_histogram_sm( ++i ) = 0.295;
	bb_dih_native_histogram_sm( ++i ) = 0.301;
	bb_dih_native_histogram_sm( ++i ) = 0.287;
	bb_dih_native_histogram_sm( ++i ) = 0.277;
	bb_dih_native_histogram_sm( ++i ) = 0.286;
	bb_dih_native_histogram_sm( ++i ) = 0.321;
	bb_dih_native_histogram_sm( ++i ) = 0.395;
	bb_dih_native_histogram_sm( ++i ) = 0.509;
	bb_dih_native_histogram_sm( ++i ) = 0.648;
	bb_dih_native_histogram_sm( ++i ) = 0.775;
	bb_dih_native_histogram_sm( ++i ) = 0.855;
	bb_dih_native_histogram_sm( ++i ) = 0.871;
	bb_dih_native_histogram_sm( ++i ) = 0.817;
	bb_dih_native_histogram_sm( ++i ) = 0.684;
	bb_dih_native_histogram_sm( ++i ) = 0.495;
	bb_dih_native_histogram_sm( ++i ) = 0.329;
	bb_dih_native_histogram_sm( ++i ) = 0.236;
	bb_dih_native_histogram_sm( ++i ) = 0.188;
	bb_dih_native_histogram_sm( ++i ) = 0.160;
	bb_dih_native_histogram_sm( ++i ) = 0.147;
	bb_dih_native_histogram_sm( ++i ) = 0.146;
	bb_dih_native_histogram_sm( ++i ) = 0.153;
	bb_dih_native_histogram_sm( ++i ) = 0.158;
	bb_dih_native_histogram_sm( ++i ) = 0.148;
	bb_dih_native_histogram_sm( ++i ) = 0.120;
	bb_dih_native_histogram_sm( ++i ) = 0.090;
	bb_dih_native_histogram_sm( ++i ) = 0.079;
	bb_dih_native_histogram_sm( ++i ) = 0.088;
	bb_dih_native_histogram_sm( ++i ) = 0.097;
	bb_dih_native_histogram_sm( ++i ) = 0.084;
	bb_dih_native_histogram_sm( ++i ) = 0.059;
	bb_dih_native_histogram_sm( ++i ) = 0.043;
	bb_dih_native_histogram_sm( ++i ) = 0.041;
	bb_dih_native_histogram_sm( ++i ) = 0.044;
	bb_dih_native_histogram_sm( ++i ) = 0.044;
	bb_dih_native_histogram_sm( ++i ) = 0.040;
	bb_dih_native_histogram_sm( ++i ) = 0.038;
	bb_dih_native_histogram_sm( ++i ) = 0.045;
	bb_dih_native_histogram_sm( ++i ) = 0.062;
	bb_dih_native_histogram_sm( ++i ) = 0.084;
	bb_dih_native_histogram_sm( ++i ) = 0.107;
	bb_dih_native_histogram_sm( ++i ) = 0.131;
	bb_dih_native_histogram_sm( ++i ) = 0.146;
	bb_dih_native_histogram_sm( ++i ) = 0.143;
	bb_dih_native_histogram_sm( ++i ) = 0.129;
	bb_dih_native_histogram_sm( ++i ) = 0.121;
	bb_dih_native_histogram_sm( ++i ) = 0.133;
	bb_dih_native_histogram_sm( ++i ) = 0.170;
	bb_dih_native_histogram_sm( ++i ) = 0.206;
	bb_dih_native_histogram_sm( ++i ) = 0.224;
	bb_dih_native_histogram_sm( ++i ) = 0.243;
	bb_dih_native_histogram_sm( ++i ) = 0.270;
	bb_dih_native_histogram_sm( ++i ) = 0.279;
	bb_dih_native_histogram_sm( ++i ) = 0.265;
	bb_dih_native_histogram_sm( ++i ) = 0.257;
	bb_dih_native_histogram_sm( ++i ) = 0.277;
	bb_dih_native_histogram_sm( ++i ) = 0.306;
	bb_dih_native_histogram_sm( ++i ) = 0.311;
	bb_dih_native_histogram_sm( ++i ) = 0.281;
	bb_dih_native_histogram_sm( ++i ) = 0.244;
	bb_dih_native_histogram_sm( ++i ) = 0.227;
	bb_dih_native_histogram_sm( ++i ) = 0.222;
	bb_dih_native_histogram_sm( ++i ) = 0.201;
	bb_dih_native_histogram_sm( ++i ) = 0.163;
	bb_dih_native_histogram_sm( ++i ) = 0.144;
}


void
Disulfides::bb_dih_native_histogram_unsm_initializer( FArray1D_float & bb_dih_native_histogram_unsm )
{
	int i = 0;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.240;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.100;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.180;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.380;
	bb_dih_native_histogram_unsm( ++i ) = 0.440;
	bb_dih_native_histogram_unsm( ++i ) = 0.400;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.440;
	bb_dih_native_histogram_unsm( ++i ) = 0.240;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.260;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.280;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.180;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.260;
	bb_dih_native_histogram_unsm( ++i ) = 0.380;
	bb_dih_native_histogram_unsm( ++i ) = 0.300;
	bb_dih_native_histogram_unsm( ++i ) = 0.260;
	bb_dih_native_histogram_unsm( ++i ) = 0.280;
	bb_dih_native_histogram_unsm( ++i ) = 0.280;
	bb_dih_native_histogram_unsm( ++i ) = 0.240;
	bb_dih_native_histogram_unsm( ++i ) = 0.440;
	bb_dih_native_histogram_unsm( ++i ) = 0.380;
	bb_dih_native_histogram_unsm( ++i ) = 0.740;
	bb_dih_native_histogram_unsm( ++i ) = 0.780;
	bb_dih_native_histogram_unsm( ++i ) = 0.900;
	bb_dih_native_histogram_unsm( ++i ) = 1.000;
	bb_dih_native_histogram_unsm( ++i ) = 0.700;
	bb_dih_native_histogram_unsm( ++i ) = 1.000;
	bb_dih_native_histogram_unsm( ++i ) = 0.320;
	bb_dih_native_histogram_unsm( ++i ) = 0.220;
	bb_dih_native_histogram_unsm( ++i ) = 0.260;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.220;
	bb_dih_native_histogram_unsm( ++i ) = 0.140;
	bb_dih_native_histogram_unsm( ++i ) = 0.140;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.100;
	bb_dih_native_histogram_unsm( ++i ) = 0.020;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.020;
	bb_dih_native_histogram_unsm( ++i ) = 0.040;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.100;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.080;
	bb_dih_native_histogram_unsm( ++i ) = 0.100;
	bb_dih_native_histogram_unsm( ++i ) = 0.380;
	bb_dih_native_histogram_unsm( ++i ) = 0.120;
	bb_dih_native_histogram_unsm( ++i ) = 0.220;
	bb_dih_native_histogram_unsm( ++i ) = 0.320;
	bb_dih_native_histogram_unsm( ++i ) = 0.320;
	bb_dih_native_histogram_unsm( ++i ) = 0.240;
	bb_dih_native_histogram_unsm( ++i ) = 0.240;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.420;
	bb_dih_native_histogram_unsm( ++i ) = 0.280;
	bb_dih_native_histogram_unsm( ++i ) = 0.360;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.200;
	bb_dih_native_histogram_unsm( ++i ) = 0.280;
	bb_dih_native_histogram_unsm( ++i ) = 0.220;
	bb_dih_native_histogram_unsm( ++i ) = 0.160;
	bb_dih_native_histogram_unsm( ++i ) = 0.060;
}


void
Disulfides::cacbcb_native_histogram_initializer( FArray1D_float & cacbcb_native_histogram )
{
	int i = 0;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.000;
	cacbcb_native_histogram( ++i ) = 0.027;
	cacbcb_native_histogram( ++i ) = 0.147;
	cacbcb_native_histogram( ++i ) = 0.407;
	cacbcb_native_histogram( ++i ) = 0.313;
	cacbcb_native_histogram( ++i ) = 0.280;
	cacbcb_native_histogram( ++i ) = 0.227;
	cacbcb_native_histogram( ++i ) = 0.327;
	cacbcb_native_histogram( ++i ) = 0.287;
	cacbcb_native_histogram( ++i ) = 0.260;
	cacbcb_native_histogram( ++i ) = 0.273;
	cacbcb_native_histogram( ++i ) = 0.447;
	cacbcb_native_histogram( ++i ) = 0.580;
	cacbcb_native_histogram( ++i ) = 0.807;
	cacbcb_native_histogram( ++i ) = 1.000;
	cacbcb_native_histogram( ++i ) = 0.900;
	cacbcb_native_histogram( ++i ) = 0.893;
	cacbcb_native_histogram( ++i ) = 0.880;
	cacbcb_native_histogram( ++i ) = 0.760;
	cacbcb_native_histogram( ++i ) = 0.640;
	cacbcb_native_histogram( ++i ) = 0.647;
	cacbcb_native_histogram( ++i ) = 0.607;
	cacbcb_native_histogram( ++i ) = 0.347;
	cacbcb_native_histogram( ++i ) = 0.380;
	cacbcb_native_histogram( ++i ) = 0.373;
	cacbcb_native_histogram( ++i ) = 0.373;
	cacbcb_native_histogram( ++i ) = 0.513;
	cacbcb_native_histogram( ++i ) = 0.473;
	cacbcb_native_histogram( ++i ) = 0.487;
	cacbcb_native_histogram( ++i ) = 0.453;
	cacbcb_native_histogram( ++i ) = 0.267;
	cacbcb_native_histogram( ++i ) = 0.127;
	cacbcb_native_histogram( ++i ) = 0.067;
	cacbcb_native_histogram( ++i ) = 0.007;
}


void
Disulfides::cacbcb_cnrlx_score_initializer( FArray1D_float & cacbcb_cnrlx_score )
{
	int i = 0;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 100.000;
	cacbcb_cnrlx_score( ++i ) = 1.820;
	cacbcb_cnrlx_score( ++i ) = 0.303;
	cacbcb_cnrlx_score( ++i ) = -0.611;
	cacbcb_cnrlx_score( ++i ) = -0.145;
	cacbcb_cnrlx_score( ++i ) = 0.069;
	cacbcb_cnrlx_score( ++i ) = 0.403;
	cacbcb_cnrlx_score( ++i ) = 0.143;
	cacbcb_cnrlx_score( ++i ) = 0.382;
	cacbcb_cnrlx_score( ++i ) = 0.576;
	cacbcb_cnrlx_score( ++i ) = 0.607;
	cacbcb_cnrlx_score( ++i ) = 0.186;
	cacbcb_cnrlx_score( ++i ) = -0.049;
	cacbcb_cnrlx_score( ++i ) = -0.371;
	cacbcb_cnrlx_score( ++i ) = -0.540;
	cacbcb_cnrlx_score( ++i ) = -0.432;
	cacbcb_cnrlx_score( ++i ) = -0.410;
	cacbcb_cnrlx_score( ++i ) = -0.373;
	cacbcb_cnrlx_score( ++i ) = -0.200;
	cacbcb_cnrlx_score( ++i ) = -0.023;
	cacbcb_cnrlx_score( ++i ) = -0.092;
	cacbcb_cnrlx_score( ++i ) = -0.054;
	cacbcb_cnrlx_score( ++i ) = 0.533;
	cacbcb_cnrlx_score( ++i ) = 0.364;
	cacbcb_cnrlx_score( ++i ) = 0.320;
	cacbcb_cnrlx_score( ++i ) = 0.351;
	cacbcb_cnrlx_score( ++i ) = -0.105;
	cacbcb_cnrlx_score( ++i ) = -0.186;
	cacbcb_cnrlx_score( ++i ) = -0.314;
	cacbcb_cnrlx_score( ++i ) = -0.432;
	cacbcb_cnrlx_score( ++i ) = -0.164;
	cacbcb_cnrlx_score( ++i ) = 0.216;
	cacbcb_cnrlx_score( ++i ) = 0.409;
	cacbcb_cnrlx_score( ++i ) = 1.615;
}




void
Disulfides::cacbcbca_native_histogram_sm_initializer( FArray1D_float & cacbcbca_native_histogram_sm )
{
	int i = 0;
	cacbcbca_native_histogram_sm( ++i ) = 0.287;
	cacbcbca_native_histogram_sm( ++i ) = 0.286;
	cacbcbca_native_histogram_sm( ++i ) = 0.286;
	cacbcbca_native_histogram_sm( ++i ) = 0.292;
	cacbcbca_native_histogram_sm( ++i ) = 0.306;
	cacbcbca_native_histogram_sm( ++i ) = 0.325;
	cacbcbca_native_histogram_sm( ++i ) = 0.339;
	cacbcbca_native_histogram_sm( ++i ) = 0.339;
	cacbcbca_native_histogram_sm( ++i ) = 0.321;
	cacbcbca_native_histogram_sm( ++i ) = 0.292;
	cacbcbca_native_histogram_sm( ++i ) = 0.260;
	cacbcbca_native_histogram_sm( ++i ) = 0.230;
	cacbcbca_native_histogram_sm( ++i ) = 0.202;
	cacbcbca_native_histogram_sm( ++i ) = 0.176;
	cacbcbca_native_histogram_sm( ++i ) = 0.152;
	cacbcbca_native_histogram_sm( ++i ) = 0.132;
	cacbcbca_native_histogram_sm( ++i ) = 0.116;
	cacbcbca_native_histogram_sm( ++i ) = 0.106;
	cacbcbca_native_histogram_sm( ++i ) = 0.103;
	cacbcbca_native_histogram_sm( ++i ) = 0.107;
	cacbcbca_native_histogram_sm( ++i ) = 0.119;
	cacbcbca_native_histogram_sm( ++i ) = 0.137;
	cacbcbca_native_histogram_sm( ++i ) = 0.162;
	cacbcbca_native_histogram_sm( ++i ) = 0.191;
	cacbcbca_native_histogram_sm( ++i ) = 0.225;
	cacbcbca_native_histogram_sm( ++i ) = 0.266;
	cacbcbca_native_histogram_sm( ++i ) = 0.314;
	cacbcbca_native_histogram_sm( ++i ) = 0.362;
	cacbcbca_native_histogram_sm( ++i ) = 0.400;
	cacbcbca_native_histogram_sm( ++i ) = 0.422;
	cacbcbca_native_histogram_sm( ++i ) = 0.428;
	cacbcbca_native_histogram_sm( ++i ) = 0.427;
	cacbcbca_native_histogram_sm( ++i ) = 0.430;
	cacbcbca_native_histogram_sm( ++i ) = 0.443;
	cacbcbca_native_histogram_sm( ++i ) = 0.469;
	cacbcbca_native_histogram_sm( ++i ) = 0.506;
	cacbcbca_native_histogram_sm( ++i ) = 0.555;
	cacbcbca_native_histogram_sm( ++i ) = 0.615;
	cacbcbca_native_histogram_sm( ++i ) = 0.679;
	cacbcbca_native_histogram_sm( ++i ) = 0.733;
	cacbcbca_native_histogram_sm( ++i ) = 0.759;
	cacbcbca_native_histogram_sm( ++i ) = 0.744;
	cacbcbca_native_histogram_sm( ++i ) = 0.691;
	cacbcbca_native_histogram_sm( ++i ) = 0.615;
	cacbcbca_native_histogram_sm( ++i ) = 0.540;
	cacbcbca_native_histogram_sm( ++i ) = 0.482;
	cacbcbca_native_histogram_sm( ++i ) = 0.447;
	cacbcbca_native_histogram_sm( ++i ) = 0.425;
	cacbcbca_native_histogram_sm( ++i ) = 0.400;
	cacbcbca_native_histogram_sm( ++i ) = 0.362;
	cacbcbca_native_histogram_sm( ++i ) = 0.310;
	cacbcbca_native_histogram_sm( ++i ) = 0.255;
	cacbcbca_native_histogram_sm( ++i ) = 0.206;
	cacbcbca_native_histogram_sm( ++i ) = 0.167;
	cacbcbca_native_histogram_sm( ++i ) = 0.139;
	cacbcbca_native_histogram_sm( ++i ) = 0.119;
	cacbcbca_native_histogram_sm( ++i ) = 0.104;
	cacbcbca_native_histogram_sm( ++i ) = 0.092;
	cacbcbca_native_histogram_sm( ++i ) = 0.083;
	cacbcbca_native_histogram_sm( ++i ) = 0.079;
	cacbcbca_native_histogram_sm( ++i ) = 0.080;
	cacbcbca_native_histogram_sm( ++i ) = 0.089;
	cacbcbca_native_histogram_sm( ++i ) = 0.107;
	cacbcbca_native_histogram_sm( ++i ) = 0.130;
	cacbcbca_native_histogram_sm( ++i ) = 0.154;
	cacbcbca_native_histogram_sm( ++i ) = 0.174;
	cacbcbca_native_histogram_sm( ++i ) = 0.188;
	cacbcbca_native_histogram_sm( ++i ) = 0.198;
	cacbcbca_native_histogram_sm( ++i ) = 0.204;
	cacbcbca_native_histogram_sm( ++i ) = 0.205;
	cacbcbca_native_histogram_sm( ++i ) = 0.201;
	cacbcbca_native_histogram_sm( ++i ) = 0.192;
	cacbcbca_native_histogram_sm( ++i ) = 0.182;
	cacbcbca_native_histogram_sm( ++i ) = 0.170;
	cacbcbca_native_histogram_sm( ++i ) = 0.159;
	cacbcbca_native_histogram_sm( ++i ) = 0.150;
	cacbcbca_native_histogram_sm( ++i ) = 0.144;
	cacbcbca_native_histogram_sm( ++i ) = 0.138;
	cacbcbca_native_histogram_sm( ++i ) = 0.132;
	cacbcbca_native_histogram_sm( ++i ) = 0.125;
	cacbcbca_native_histogram_sm( ++i ) = 0.120;
	cacbcbca_native_histogram_sm( ++i ) = 0.123;
	cacbcbca_native_histogram_sm( ++i ) = 0.136;
	cacbcbca_native_histogram_sm( ++i ) = 0.161;
	cacbcbca_native_histogram_sm( ++i ) = 0.190;
	cacbcbca_native_histogram_sm( ++i ) = 0.217;
	cacbcbca_native_histogram_sm( ++i ) = 0.238;
	cacbcbca_native_histogram_sm( ++i ) = 0.251;
	cacbcbca_native_histogram_sm( ++i ) = 0.258;
	cacbcbca_native_histogram_sm( ++i ) = 0.261;
	cacbcbca_native_histogram_sm( ++i ) = 0.260;
	cacbcbca_native_histogram_sm( ++i ) = 0.252;
	cacbcbca_native_histogram_sm( ++i ) = 0.238;
	cacbcbca_native_histogram_sm( ++i ) = 0.218;
	cacbcbca_native_histogram_sm( ++i ) = 0.196;
	cacbcbca_native_histogram_sm( ++i ) = 0.178;
	cacbcbca_native_histogram_sm( ++i ) = 0.168;
	cacbcbca_native_histogram_sm( ++i ) = 0.168;
	cacbcbca_native_histogram_sm( ++i ) = 0.172;
	cacbcbca_native_histogram_sm( ++i ) = 0.175;
}


void
Disulfides::cacbcbca_native_histogram_unsm_initializer( FArray1D_float & cacbcbca_native_histogram_unsm )
{
	int i = 0;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.390;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.366;
	cacbcbca_native_histogram_unsm( ++i ) = 0.659;
	cacbcbca_native_histogram_unsm( ++i ) = 0.220;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.220;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.146;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.049;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.073;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.366;
	cacbcbca_native_histogram_unsm( ++i ) = 0.463;
	cacbcbca_native_histogram_unsm( ++i ) = 0.561;
	cacbcbca_native_histogram_unsm( ++i ) = 0.488;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.317;
	cacbcbca_native_histogram_unsm( ++i ) = 0.585;
	cacbcbca_native_histogram_unsm( ++i ) = 0.366;
	cacbcbca_native_histogram_unsm( ++i ) = 0.463;
	cacbcbca_native_histogram_unsm( ++i ) = 0.585;
	cacbcbca_native_histogram_unsm( ++i ) = 0.537;
	cacbcbca_native_histogram_unsm( ++i ) = 0.634;
	cacbcbca_native_histogram_unsm( ++i ) = 0.805;
	cacbcbca_native_histogram_unsm( ++i ) = 0.927;
	cacbcbca_native_histogram_unsm( ++i ) = 1.000;
	cacbcbca_native_histogram_unsm( ++i ) = 0.659;
	cacbcbca_native_histogram_unsm( ++i ) = 0.610;
	cacbcbca_native_histogram_unsm( ++i ) = 0.488;
	cacbcbca_native_histogram_unsm( ++i ) = 0.366;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.561;
	cacbcbca_native_histogram_unsm( ++i ) = 0.585;
	cacbcbca_native_histogram_unsm( ++i ) = 0.341;
	cacbcbca_native_histogram_unsm( ++i ) = 0.317;
	cacbcbca_native_histogram_unsm( ++i ) = 0.171;
	cacbcbca_native_histogram_unsm( ++i ) = 0.220;
	cacbcbca_native_histogram_unsm( ++i ) = 0.146;
	cacbcbca_native_histogram_unsm( ++i ) = 0.049;
	cacbcbca_native_histogram_unsm( ++i ) = 0.171;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.024;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.049;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.000;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.073;
	cacbcbca_native_histogram_unsm( ++i ) = 0.220;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.073;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.171;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.171;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.171;
	cacbcbca_native_histogram_unsm( ++i ) = 0.146;
	cacbcbca_native_histogram_unsm( ++i ) = 0.146;
	cacbcbca_native_histogram_unsm( ++i ) = 0.073;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.341;
	cacbcbca_native_histogram_unsm( ++i ) = 0.244;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
	cacbcbca_native_histogram_unsm( ++i ) = 0.341;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.293;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.220;
	cacbcbca_native_histogram_unsm( ++i ) = 0.146;
	cacbcbca_native_histogram_unsm( ++i ) = 0.098;
	cacbcbca_native_histogram_unsm( ++i ) = 0.122;
	cacbcbca_native_histogram_unsm( ++i ) = 0.195;
	cacbcbca_native_histogram_unsm( ++i ) = 0.268;
}


void
Disulfides::cacbcbca_foldcst_score_sm_initializer( FArray1D_float & cacbcbca_foldcst_score_sm )
{
	int i = 0;
	cacbcbca_foldcst_score_sm( ++i ) = 0.033;
	cacbcbca_foldcst_score_sm( ++i ) = -0.042;
	cacbcbca_foldcst_score_sm( ++i ) = -0.073;
	cacbcbca_foldcst_score_sm( ++i ) = -0.091;
	cacbcbca_foldcst_score_sm( ++i ) = -0.131;
	cacbcbca_foldcst_score_sm( ++i ) = -0.199;
	cacbcbca_foldcst_score_sm( ++i ) = -0.260;
	cacbcbca_foldcst_score_sm( ++i ) = -0.268;
	cacbcbca_foldcst_score_sm( ++i ) = -0.210;
	cacbcbca_foldcst_score_sm( ++i ) = -0.111;
	cacbcbca_foldcst_score_sm( ++i ) = 0.002;
	cacbcbca_foldcst_score_sm( ++i ) = 0.124;
	cacbcbca_foldcst_score_sm( ++i ) = 0.264;
	cacbcbca_foldcst_score_sm( ++i ) = 0.420;
	cacbcbca_foldcst_score_sm( ++i ) = 0.577;
	cacbcbca_foldcst_score_sm( ++i ) = 0.724;
	cacbcbca_foldcst_score_sm( ++i ) = 0.850;
	cacbcbca_foldcst_score_sm( ++i ) = 0.937;
	cacbcbca_foldcst_score_sm( ++i ) = 0.967;
	cacbcbca_foldcst_score_sm( ++i ) = 0.933;
	cacbcbca_foldcst_score_sm( ++i ) = 0.841;
	cacbcbca_foldcst_score_sm( ++i ) = 0.709;
	cacbcbca_foldcst_score_sm( ++i ) = 0.561;
	cacbcbca_foldcst_score_sm( ++i ) = 0.412;
	cacbcbca_foldcst_score_sm( ++i ) = 0.260;
	cacbcbca_foldcst_score_sm( ++i ) = 0.092;
	cacbcbca_foldcst_score_sm( ++i ) = -0.095;
	cacbcbca_foldcst_score_sm( ++i ) = -0.278;
	cacbcbca_foldcst_score_sm( ++i ) = -0.418;
	cacbcbca_foldcst_score_sm( ++i ) = -0.487;
	cacbcbca_foldcst_score_sm( ++i ) = -0.489;
	cacbcbca_foldcst_score_sm( ++i ) = -0.463;
	cacbcbca_foldcst_score_sm( ++i ) = -0.450;
	cacbcbca_foldcst_score_sm( ++i ) = -0.469;
	cacbcbca_foldcst_score_sm( ++i ) = -0.517;
	cacbcbca_foldcst_score_sm( ++i ) = -0.587;
	cacbcbca_foldcst_score_sm( ++i ) = -0.675;
	cacbcbca_foldcst_score_sm( ++i ) = -0.778;
	cacbcbca_foldcst_score_sm( ++i ) = -0.885;
	cacbcbca_foldcst_score_sm( ++i ) = -0.976;
	cacbcbca_foldcst_score_sm( ++i ) = -1.020;
	cacbcbca_foldcst_score_sm( ++i ) = -0.990;
	cacbcbca_foldcst_score_sm( ++i ) = -0.884;
	cacbcbca_foldcst_score_sm( ++i ) = -0.728;
	cacbcbca_foldcst_score_sm( ++i ) = -0.567;
	cacbcbca_foldcst_score_sm( ++i ) = -0.447;
	cacbcbca_foldcst_score_sm( ++i ) = -0.383;
	cacbcbca_foldcst_score_sm( ++i ) = -0.348;
	cacbcbca_foldcst_score_sm( ++i ) = -0.286;
	cacbcbca_foldcst_score_sm( ++i ) = -0.155;
	cacbcbca_foldcst_score_sm( ++i ) = 0.044;
	cacbcbca_foldcst_score_sm( ++i ) = 0.281;
	cacbcbca_foldcst_score_sm( ++i ) = 0.524;
	cacbcbca_foldcst_score_sm( ++i ) = 0.745;
	cacbcbca_foldcst_score_sm( ++i ) = 0.932;
	cacbcbca_foldcst_score_sm( ++i ) = 1.090;
	cacbcbca_foldcst_score_sm( ++i ) = 1.220;
	cacbcbca_foldcst_score_sm( ++i ) = 1.306;
	cacbcbca_foldcst_score_sm( ++i ) = 1.322;
	cacbcbca_foldcst_score_sm( ++i ) = 1.262;
	cacbcbca_foldcst_score_sm( ++i ) = 1.151;
	cacbcbca_foldcst_score_sm( ++i ) = 1.024;
	cacbcbca_foldcst_score_sm( ++i ) = 0.900;
	cacbcbca_foldcst_score_sm( ++i ) = 0.787;
	cacbcbca_foldcst_score_sm( ++i ) = 0.693;
	cacbcbca_foldcst_score_sm( ++i ) = 0.618;
	cacbcbca_foldcst_score_sm( ++i ) = 0.547;
	cacbcbca_foldcst_score_sm( ++i ) = 0.467;
	cacbcbca_foldcst_score_sm( ++i ) = 0.391;
	cacbcbca_foldcst_score_sm( ++i ) = 0.349;
	cacbcbca_foldcst_score_sm( ++i ) = 0.356;
	cacbcbca_foldcst_score_sm( ++i ) = 0.400;
	cacbcbca_foldcst_score_sm( ++i ) = 0.460;
	cacbcbca_foldcst_score_sm( ++i ) = 0.525;
	cacbcbca_foldcst_score_sm( ++i ) = 0.590;
	cacbcbca_foldcst_score_sm( ++i ) = 0.645;
	cacbcbca_foldcst_score_sm( ++i ) = 0.682;
	cacbcbca_foldcst_score_sm( ++i ) = 0.719;
	cacbcbca_foldcst_score_sm( ++i ) = 0.776;
	cacbcbca_foldcst_score_sm( ++i ) = 0.852;
	cacbcbca_foldcst_score_sm( ++i ) = 0.916;
	cacbcbca_foldcst_score_sm( ++i ) = 0.923;
	cacbcbca_foldcst_score_sm( ++i ) = 0.845;
	cacbcbca_foldcst_score_sm( ++i ) = 0.686;
	cacbcbca_foldcst_score_sm( ++i ) = 0.487;
	cacbcbca_foldcst_score_sm( ++i ) = 0.304;
	cacbcbca_foldcst_score_sm( ++i ) = 0.174;
	cacbcbca_foldcst_score_sm( ++i ) = 0.096;
	cacbcbca_foldcst_score_sm( ++i ) = 0.046;
	cacbcbca_foldcst_score_sm( ++i ) = 0.011;
	cacbcbca_foldcst_score_sm( ++i ) = -0.005;
	cacbcbca_foldcst_score_sm( ++i ) = 0.013;
	cacbcbca_foldcst_score_sm( ++i ) = 0.077;
	cacbcbca_foldcst_score_sm( ++i ) = 0.191;
	cacbcbca_foldcst_score_sm( ++i ) = 0.337;
	cacbcbca_foldcst_score_sm( ++i ) = 0.464;
	cacbcbca_foldcst_score_sm( ++i ) = 0.515;
	cacbcbca_foldcst_score_sm( ++i ) = 0.459;
	cacbcbca_foldcst_score_sm( ++i ) = 0.322;
	cacbcbca_foldcst_score_sm( ++i ) = 0.162;
}


void
Disulfides::cacbcbca_cnrlx_score_initializer( FArray1D_float & cacbcbca_cnrlx_score )
{
	int i = 0;
	cacbcbca_cnrlx_score( ++i ) = 0.192;
	cacbcbca_cnrlx_score( ++i ) = -0.443;
	cacbcbca_cnrlx_score( ++i ) = -0.021;
	cacbcbca_cnrlx_score( ++i ) = -0.037;
	cacbcbca_cnrlx_score( ++i ) = 0.089;
	cacbcbca_cnrlx_score( ++i ) = -0.084;
	cacbcbca_cnrlx_score( ++i ) = -0.324;
	cacbcbca_cnrlx_score( ++i ) = -1.045;
	cacbcbca_cnrlx_score( ++i ) = 0.019;
	cacbcbca_cnrlx_score( ++i ) = 0.162;
	cacbcbca_cnrlx_score( ++i ) = -0.172;
	cacbcbca_cnrlx_score( ++i ) = 0.152;
	cacbcbca_cnrlx_score( ++i ) = -0.112;
	cacbcbca_cnrlx_score( ++i ) = 0.725;
	cacbcbca_cnrlx_score( ++i ) = 0.706;
	cacbcbca_cnrlx_score( ++i ) = 0.530;
	cacbcbca_cnrlx_score( ++i ) = 0.695;
	cacbcbca_cnrlx_score( ++i ) = 1.620;
	cacbcbca_cnrlx_score( ++i ) = 0.690;
	cacbcbca_cnrlx_score( ++i ) = 1.162;
	cacbcbca_cnrlx_score( ++i ) = 0.953;
	cacbcbca_cnrlx_score( ++i ) = 0.703;
	cacbcbca_cnrlx_score( ++i ) = 0.726;
	cacbcbca_cnrlx_score( ++i ) = -0.148;
	cacbcbca_cnrlx_score( ++i ) = 0.748;
	cacbcbca_cnrlx_score( ++i ) = 0.141;
	cacbcbca_cnrlx_score( ++i ) = -0.142;
	cacbcbca_cnrlx_score( ++i ) = -0.330;
	cacbcbca_cnrlx_score( ++i ) = -0.568;
	cacbcbca_cnrlx_score( ++i ) = -0.795;
	cacbcbca_cnrlx_score( ++i ) = -0.680;
	cacbcbca_cnrlx_score( ++i ) = -0.144;
	cacbcbca_cnrlx_score( ++i ) = -0.191;
	cacbcbca_cnrlx_score( ++i ) = -0.822;
	cacbcbca_cnrlx_score( ++i ) = -0.272;
	cacbcbca_cnrlx_score( ++i ) = -0.517;
	cacbcbca_cnrlx_score( ++i ) = -0.771;
	cacbcbca_cnrlx_score( ++i ) = -0.709;
	cacbcbca_cnrlx_score( ++i ) = -0.815;
	cacbcbca_cnrlx_score( ++i ) = -1.065;
	cacbcbca_cnrlx_score( ++i ) = -1.229;
	cacbcbca_cnrlx_score( ++i ) = -1.296;
	cacbcbca_cnrlx_score( ++i ) = -0.846;
	cacbcbca_cnrlx_score( ++i ) = -0.821;
	cacbcbca_cnrlx_score( ++i ) = -0.507;
	cacbcbca_cnrlx_score( ++i ) = -0.287;
	cacbcbca_cnrlx_score( ++i ) = 0.142;
	cacbcbca_cnrlx_score( ++i ) = -0.692;
	cacbcbca_cnrlx_score( ++i ) = -0.701;
	cacbcbca_cnrlx_score( ++i ) = -0.159;
	cacbcbca_cnrlx_score( ++i ) = -0.087;
	cacbcbca_cnrlx_score( ++i ) = 0.526;
	cacbcbca_cnrlx_score( ++i ) = 0.309;
	cacbcbca_cnrlx_score( ++i ) = 0.673;
	cacbcbca_cnrlx_score( ++i ) = 1.760;
	cacbcbca_cnrlx_score( ++i ) = 0.505;
	cacbcbca_cnrlx_score( ++i ) = 0.832;
	cacbcbca_cnrlx_score( ++i ) = 2.447;
	cacbcbca_cnrlx_score( ++i ) = 0.880;
	cacbcbca_cnrlx_score( ++i ) = 1.679;
	cacbcbca_cnrlx_score( ++i ) = 1.030;
	cacbcbca_cnrlx_score( ++i ) = 0.900;
	cacbcbca_cnrlx_score( ++i ) = 0.795;
	cacbcbca_cnrlx_score( ++i ) = 1.279;
	cacbcbca_cnrlx_score( ++i ) = 0.225;
	cacbcbca_cnrlx_score( ++i ) = 0.159;
	cacbcbca_cnrlx_score( ++i ) = 1.513;
	cacbcbca_cnrlx_score( ++i ) = 0.037;
	cacbcbca_cnrlx_score( ++i ) = 0.437;
	cacbcbca_cnrlx_score( ++i ) = 0.085;
	cacbcbca_cnrlx_score( ++i ) = 0.056;
	cacbcbca_cnrlx_score( ++i ) = 0.835;
	cacbcbca_cnrlx_score( ++i ) = 0.460;
	cacbcbca_cnrlx_score( ++i ) = 0.105;
	cacbcbca_cnrlx_score( ++i ) = 0.810;
	cacbcbca_cnrlx_score( ++i ) = 0.772;
	cacbcbca_cnrlx_score( ++i ) = 0.854;
	cacbcbca_cnrlx_score( ++i ) = 0.432;
	cacbcbca_cnrlx_score( ++i ) = 0.650;
	cacbcbca_cnrlx_score( ++i ) = 0.793;
	cacbcbca_cnrlx_score( ++i ) = 1.271;
	cacbcbca_cnrlx_score( ++i ) = 1.008;
	cacbcbca_cnrlx_score( ++i ) = 0.973;
	cacbcbca_cnrlx_score( ++i ) = 0.990;
	cacbcbca_cnrlx_score( ++i ) = 0.561;
	cacbcbca_cnrlx_score( ++i ) = -0.304;
	cacbcbca_cnrlx_score( ++i ) = 0.061;
	cacbcbca_cnrlx_score( ++i ) = 0.390;
	cacbcbca_cnrlx_score( ++i ) = 0.011;
	cacbcbca_cnrlx_score( ++i ) = -0.274;
	cacbcbca_cnrlx_score( ++i ) = 0.218;
	cacbcbca_cnrlx_score( ++i ) = -0.168;
	cacbcbca_cnrlx_score( ++i ) = -0.127;
	cacbcbca_cnrlx_score( ++i ) = 0.216;
	cacbcbca_cnrlx_score( ++i ) = 0.156;
	cacbcbca_cnrlx_score( ++i ) = 0.536;
	cacbcbca_cnrlx_score( ++i ) = 1.194;
	cacbcbca_cnrlx_score( ++i ) = 0.628;
	cacbcbca_cnrlx_score( ++i ) = 0.217;
	cacbcbca_cnrlx_score( ++i ) = -0.094;
}


void
Disulfides::cacbcbca_cnrlx_score_sm_initializer( FArray1D_float & cacbcbca_cnrlx_score_sm )
{
	int i = 0;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.153;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.142;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.101;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.047;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.056;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.183;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.354;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.392;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.246;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.076;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.006;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.075;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.231;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.438;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.597;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.723;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.882;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.016;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.041;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.988;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.891;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.731;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.537;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.391;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.293;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.144;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.086;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.330;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.518;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.591;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.522;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.414;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.396;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.450;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.502;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.565;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.659;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.765;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.886;
	cacbcbca_cnrlx_score_sm( ++i ) = -1.025;
	cacbcbca_cnrlx_score_sm( ++i ) = -1.117;
	cacbcbca_cnrlx_score_sm( ++i ) = -1.089;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.944;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.739;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.515;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.342;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.314;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.399;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.410;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.243;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.022;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.281;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.540;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.804;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.980;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.076;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.236;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.414;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.439;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.311;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.142;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.004;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.906;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.778;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.638;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.600;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.602;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.496;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.328;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.255;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.319;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.419;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.458;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.495;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.602;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.699;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.700;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.667;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.717;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.861;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.993;
	cacbcbca_cnrlx_score_sm( ++i ) = 1.027;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.952;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.756;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.455;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.195;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.104;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.097;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.049;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.012;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.037;
	cacbcbca_cnrlx_score_sm( ++i ) = -0.034;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.011;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.131;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.332;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.564;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.683;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.574;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.343;
	cacbcbca_cnrlx_score_sm( ++i ) = 0.227;
}


void
Disulfides::cbdist_native_histogram_initializer( FArray1D_float & cbdist_native_histogram )
{
	int i = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0.0059;
	cbdist_native_histogram( ++i ) = 0.0059;
	cbdist_native_histogram( ++i ) = 0.0882;
	cbdist_native_histogram( ++i ) = 0.235;
	cbdist_native_histogram( ++i ) = 0.494;
	cbdist_native_histogram( ++i ) = 0.782;
	cbdist_native_histogram( ++i ) = 0.988;
	cbdist_native_histogram( ++i ) = 1.0;
	cbdist_native_histogram( ++i ) = 0.682;
	cbdist_native_histogram( ++i ) = 0.647;
	cbdist_native_histogram( ++i ) = 0.529;
	cbdist_native_histogram( ++i ) = 0.371;
	cbdist_native_histogram( ++i ) = 0.229;
	cbdist_native_histogram( ++i ) = 0.1;
	cbdist_native_histogram( ++i ) = 0.0412;
	cbdist_native_histogram( ++i ) = 0.0235;
	cbdist_native_histogram( ++i ) = 0.0176;
	cbdist_native_histogram( ++i ) = 0.0235;
	cbdist_native_histogram( ++i ) = 0.0;
	cbdist_native_histogram( ++i ) = 0.0294;
	cbdist_native_histogram( ++i ) = 0.0176;
	cbdist_native_histogram( ++i ) = 0.0;
	cbdist_native_histogram( ++i ) = 0.0059;
	cbdist_native_histogram( ++i ) = 0.0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
	cbdist_native_histogram( ++i ) = 0;
}


void
Disulfides::cbdist_foldcst_score_initializer( FArray1D_float & cbdist_foldcst_score )
{
	int i = 0;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 2.886;
	cbdist_foldcst_score( ++i ) = 2.906;
	cbdist_foldcst_score( ++i ) = 0.224;
	cbdist_foldcst_score( ++i ) = -0.701;
	cbdist_foldcst_score( ++i ) = -1.351;
	cbdist_foldcst_score( ++i ) = -1.764;
	cbdist_foldcst_score( ++i ) = -1.906;
	cbdist_foldcst_score( ++i ) = -1.824;
	cbdist_foldcst_score( ++i ) = -1.467;
	cbdist_foldcst_score( ++i ) = -1.351;
	cbdist_foldcst_score( ++i ) = -1.103;
	cbdist_foldcst_score( ++i ) = -0.649;
	cbdist_foldcst_score( ++i ) = -0.064;
	cbdist_foldcst_score( ++i ) = 0.722;
	cbdist_foldcst_score( ++i ) = 1.656;
	cbdist_foldcst_score( ++i ) = 2.230;
	cbdist_foldcst_score( ++i ) = 2.563;
	cbdist_foldcst_score( ++i ) = 2.655;
	cbdist_foldcst_score( ++i ) = 2.750;
	cbdist_foldcst_score( ++i ) = 2.845;
	cbdist_foldcst_score( ++i ) = 2.944;
	cbdist_foldcst_score( ++i ) = 3.795;
	cbdist_foldcst_score( ++i ) = 4.655;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
	cbdist_foldcst_score( ++i ) = 5.000;
}


void
Disulfides::cbdist_cnrlx_score_initializer( FArray1D_float & cbdist_cnrlx_score )
{
	int i = 0;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 2.726;
	cbdist_cnrlx_score( ++i ) = 2.564;
	cbdist_cnrlx_score( ++i ) = -0.084;
	cbdist_cnrlx_score( ++i ) = -1.029;
	cbdist_cnrlx_score( ++i ) = -1.368;
	cbdist_cnrlx_score( ++i ) = -1.332;
	cbdist_cnrlx_score( ++i ) = -0.838;
	cbdist_cnrlx_score( ++i ) = -0.592;
	cbdist_cnrlx_score( ++i ) = -1.029;
	cbdist_cnrlx_score( ++i ) = -0.873;
	cbdist_cnrlx_score( ++i ) = -0.659;
	cbdist_cnrlx_score( ++i ) = -0.372;
	cbdist_cnrlx_score( ++i ) = -0.005;
	cbdist_cnrlx_score( ++i ) = 0.567;
	cbdist_cnrlx_score( ++i ) = 1.546;
	cbdist_cnrlx_score( ++i ) = 2.089;
	cbdist_cnrlx_score( ++i ) = 2.547;
	cbdist_cnrlx_score( ++i ) = 2.447;
	cbdist_cnrlx_score( ++i ) = 2.600;
	cbdist_cnrlx_score( ++i ) = 2.763;
	cbdist_cnrlx_score( ++i ) = 3.097;
	cbdist_cnrlx_score( ++i ) = 2.800;
	cbdist_cnrlx_score( ++i ) = 2.644;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
	cbdist_cnrlx_score( ++i ) = 100.000;
}


void
Disulfides::cendist_score_interp300_initializer( FArray1D_float & cendist_score_interp300 )
{
	int i = 0;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.050;
	cendist_score_interp300( ++i ) = 5.095;
	cendist_score_interp300( ++i ) = 5.131;
	cendist_score_interp300( ++i ) = 5.152;
	cendist_score_interp300( ++i ) = 5.155;
	cendist_score_interp300( ++i ) = 5.133;
	cendist_score_interp300( ++i ) = 5.083;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 4.880;
	cendist_score_interp300( ++i ) = 4.725;
	cendist_score_interp300( ++i ) = 4.539;
	cendist_score_interp300( ++i ) = 4.324;
	cendist_score_interp300( ++i ) = 4.084;
	cendist_score_interp300( ++i ) = 3.821;
	cendist_score_interp300( ++i ) = 3.540;
	cendist_score_interp300( ++i ) = 3.244;
	cendist_score_interp300( ++i ) = 2.934;
	cendist_score_interp300( ++i ) = 2.616;
	cendist_score_interp300( ++i ) = 2.291;
	cendist_score_interp300( ++i ) = 1.964;
	cendist_score_interp300( ++i ) = 1.637;
	cendist_score_interp300( ++i ) = 1.314;
	cendist_score_interp300( ++i ) = 0.997;
	cendist_score_interp300( ++i ) = 0.691;
	cendist_score_interp300( ++i ) = 0.397;
	cendist_score_interp300( ++i ) = 0.120;
	cendist_score_interp300( ++i ) = -0.138;
	cendist_score_interp300( ++i ) = -0.372;
	cendist_score_interp300( ++i ) = -0.582;
	cendist_score_interp300( ++i ) = -0.767;
	cendist_score_interp300( ++i ) = -0.928;
	cendist_score_interp300( ++i ) = -1.068;
	cendist_score_interp300( ++i ) = -1.187;
	cendist_score_interp300( ++i ) = -1.286;
	cendist_score_interp300( ++i ) = -1.367;
	cendist_score_interp300( ++i ) = -1.432;
	cendist_score_interp300( ++i ) = -1.481;
	cendist_score_interp300( ++i ) = -1.515;
	cendist_score_interp300( ++i ) = -1.537;
	cendist_score_interp300( ++i ) = -1.547;
	cendist_score_interp300( ++i ) = -1.547;
	cendist_score_interp300( ++i ) = -1.538;
	cendist_score_interp300( ++i ) = -1.522;
	cendist_score_interp300( ++i ) = -1.499;
	cendist_score_interp300( ++i ) = -1.471;
	cendist_score_interp300( ++i ) = -1.439;
	cendist_score_interp300( ++i ) = -1.405;
	cendist_score_interp300( ++i ) = -1.369;
	cendist_score_interp300( ++i ) = -1.334;
	cendist_score_interp300( ++i ) = -1.299;
	cendist_score_interp300( ++i ) = -1.264;
	cendist_score_interp300( ++i ) = -1.229;
	cendist_score_interp300( ++i ) = -1.194;
	cendist_score_interp300( ++i ) = -1.159;
	cendist_score_interp300( ++i ) = -1.124;
	cendist_score_interp300( ++i ) = -1.089;
	cendist_score_interp300( ++i ) = -1.054;
	cendist_score_interp300( ++i ) = -1.019;
	cendist_score_interp300( ++i ) = -0.984;
	cendist_score_interp300( ++i ) = -0.949;
	cendist_score_interp300( ++i ) = -0.913;
	cendist_score_interp300( ++i ) = -0.877;
	cendist_score_interp300( ++i ) = -0.841;
	cendist_score_interp300( ++i ) = -0.804;
	cendist_score_interp300( ++i ) = -0.768;
	cendist_score_interp300( ++i ) = -0.730;
	cendist_score_interp300( ++i ) = -0.693;
	cendist_score_interp300( ++i ) = -0.655;
	cendist_score_interp300( ++i ) = -0.616;
	cendist_score_interp300( ++i ) = -0.577;
	cendist_score_interp300( ++i ) = -0.537;
	cendist_score_interp300( ++i ) = -0.497;
	cendist_score_interp300( ++i ) = -0.457;
	cendist_score_interp300( ++i ) = -0.416;
	cendist_score_interp300( ++i ) = -0.374;
	cendist_score_interp300( ++i ) = -0.332;
	cendist_score_interp300( ++i ) = -0.290;
	cendist_score_interp300( ++i ) = -0.247;
	cendist_score_interp300( ++i ) = -0.203;
	cendist_score_interp300( ++i ) = -0.159;
	cendist_score_interp300( ++i ) = -0.115;
	cendist_score_interp300( ++i ) = -0.070;
	cendist_score_interp300( ++i ) = -0.024;
	cendist_score_interp300( ++i ) = 0.022;
	cendist_score_interp300( ++i ) = 0.068;
	cendist_score_interp300( ++i ) = 0.115;
	cendist_score_interp300( ++i ) = 0.162;
	cendist_score_interp300( ++i ) = 0.210;
	cendist_score_interp300( ++i ) = 0.259;
	cendist_score_interp300( ++i ) = 0.308;
	cendist_score_interp300( ++i ) = 0.357;
	cendist_score_interp300( ++i ) = 0.407;
	cendist_score_interp300( ++i ) = 0.457;
	cendist_score_interp300( ++i ) = 0.507;
	cendist_score_interp300( ++i ) = 0.557;
	cendist_score_interp300( ++i ) = 0.608;
	cendist_score_interp300( ++i ) = 0.659;
	cendist_score_interp300( ++i ) = 0.711;
	cendist_score_interp300( ++i ) = 0.762;
	cendist_score_interp300( ++i ) = 0.814;
	cendist_score_interp300( ++i ) = 0.865;
	cendist_score_interp300( ++i ) = 0.917;
	cendist_score_interp300( ++i ) = 0.968;
	cendist_score_interp300( ++i ) = 1.020;
	cendist_score_interp300( ++i ) = 1.072;
	cendist_score_interp300( ++i ) = 1.123;
	cendist_score_interp300( ++i ) = 1.174;
	cendist_score_interp300( ++i ) = 1.226;
	cendist_score_interp300( ++i ) = 1.277;
	cendist_score_interp300( ++i ) = 1.327;
	cendist_score_interp300( ++i ) = 1.378;
	cendist_score_interp300( ++i ) = 1.430;
	cendist_score_interp300( ++i ) = 1.481;
	cendist_score_interp300( ++i ) = 1.533;
	cendist_score_interp300( ++i ) = 1.586;
	cendist_score_interp300( ++i ) = 1.639;
	cendist_score_interp300( ++i ) = 1.694;
	cendist_score_interp300( ++i ) = 1.749;
	cendist_score_interp300( ++i ) = 1.806;
	cendist_score_interp300( ++i ) = 1.864;
	cendist_score_interp300( ++i ) = 1.923;
	cendist_score_interp300( ++i ) = 1.984;
	cendist_score_interp300( ++i ) = 2.047;
	cendist_score_interp300( ++i ) = 2.111;
	cendist_score_interp300( ++i ) = 2.178;
	cendist_score_interp300( ++i ) = 2.247;
	cendist_score_interp300( ++i ) = 2.318;
	cendist_score_interp300( ++i ) = 2.391;
	cendist_score_interp300( ++i ) = 2.467;
	cendist_score_interp300( ++i ) = 2.545;
	cendist_score_interp300( ++i ) = 2.626;
	cendist_score_interp300( ++i ) = 2.708;
	cendist_score_interp300( ++i ) = 2.791;
	cendist_score_interp300( ++i ) = 2.876;
	cendist_score_interp300( ++i ) = 2.962;
	cendist_score_interp300( ++i ) = 3.049;
	cendist_score_interp300( ++i ) = 3.136;
	cendist_score_interp300( ++i ) = 3.224;
	cendist_score_interp300( ++i ) = 3.312;
	cendist_score_interp300( ++i ) = 3.399;
	cendist_score_interp300( ++i ) = 3.486;
	cendist_score_interp300( ++i ) = 3.571;
	cendist_score_interp300( ++i ) = 3.656;
	cendist_score_interp300( ++i ) = 3.740;
	cendist_score_interp300( ++i ) = 3.822;
	cendist_score_interp300( ++i ) = 3.902;
	cendist_score_interp300( ++i ) = 3.981;
	cendist_score_interp300( ++i ) = 4.057;
	cendist_score_interp300( ++i ) = 4.130;
	cendist_score_interp300( ++i ) = 4.201;
	cendist_score_interp300( ++i ) = 4.269;
	cendist_score_interp300( ++i ) = 4.334;
	cendist_score_interp300( ++i ) = 4.397;
	cendist_score_interp300( ++i ) = 4.457;
	cendist_score_interp300( ++i ) = 4.515;
	cendist_score_interp300( ++i ) = 4.569;
	cendist_score_interp300( ++i ) = 4.621;
	cendist_score_interp300( ++i ) = 4.670;
	cendist_score_interp300( ++i ) = 4.716;
	cendist_score_interp300( ++i ) = 4.760;
	cendist_score_interp300( ++i ) = 4.800;
	cendist_score_interp300( ++i ) = 4.838;
	cendist_score_interp300( ++i ) = 4.872;
	cendist_score_interp300( ++i ) = 4.904;
	cendist_score_interp300( ++i ) = 4.932;
	cendist_score_interp300( ++i ) = 4.958;
	cendist_score_interp300( ++i ) = 4.981;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.016;
	cendist_score_interp300( ++i ) = 5.030;
	cendist_score_interp300( ++i ) = 5.041;
	cendist_score_interp300( ++i ) = 5.049;
	cendist_score_interp300( ++i ) = 5.055;
	cendist_score_interp300( ++i ) = 5.059;
	cendist_score_interp300( ++i ) = 5.061;
	cendist_score_interp300( ++i ) = 5.061;
	cendist_score_interp300( ++i ) = 5.059;
	cendist_score_interp300( ++i ) = 5.057;
	cendist_score_interp300( ++i ) = 5.053;
	cendist_score_interp300( ++i ) = 5.048;
	cendist_score_interp300( ++i ) = 5.043;
	cendist_score_interp300( ++i ) = 5.037;
	cendist_score_interp300( ++i ) = 5.030;
	cendist_score_interp300( ++i ) = 5.024;
	cendist_score_interp300( ++i ) = 5.017;
	cendist_score_interp300( ++i ) = 5.011;
	cendist_score_interp300( ++i ) = 5.005;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 4.996;
	cendist_score_interp300( ++i ) = 4.992;
	cendist_score_interp300( ++i ) = 4.989;
	cendist_score_interp300( ++i ) = 4.987;
	cendist_score_interp300( ++i ) = 4.985;
	cendist_score_interp300( ++i ) = 4.984;
	cendist_score_interp300( ++i ) = 4.984;
	cendist_score_interp300( ++i ) = 4.984;
	cendist_score_interp300( ++i ) = 4.984;
	cendist_score_interp300( ++i ) = 4.985;
	cendist_score_interp300( ++i ) = 4.986;
	cendist_score_interp300( ++i ) = 4.987;
	cendist_score_interp300( ++i ) = 4.989;
	cendist_score_interp300( ++i ) = 4.990;
	cendist_score_interp300( ++i ) = 4.992;
	cendist_score_interp300( ++i ) = 4.994;
	cendist_score_interp300( ++i ) = 4.995;
	cendist_score_interp300( ++i ) = 4.997;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.001;
	cendist_score_interp300( ++i ) = 5.002;
	cendist_score_interp300( ++i ) = 5.003;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.004;
	cendist_score_interp300( ++i ) = 5.003;
	cendist_score_interp300( ++i ) = 5.003;
	cendist_score_interp300( ++i ) = 5.003;
	cendist_score_interp300( ++i ) = 5.002;
	cendist_score_interp300( ++i ) = 5.002;
	cendist_score_interp300( ++i ) = 5.001;
	cendist_score_interp300( ++i ) = 5.001;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 4.999;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
	cendist_score_interp300( ++i ) = 5.000;
}


//------------------------------------------------------------------------------


void
Disulfides::deriv_cendist_interp300_initializer( FArray1D_float & deriv_cendist_interp300 )
{
	int i = 0;
	deriv_cendist_interp300( ++i ) = 0.400;
	deriv_cendist_interp300( ++i ) = 0.381;
	deriv_cendist_interp300( ++i ) = 0.323;
	deriv_cendist_interp300( ++i ) = 0.228;
	deriv_cendist_interp300( ++i ) = 0.095;
	deriv_cendist_interp300( ++i ) = -0.076;
	deriv_cendist_interp300( ++i ) = -0.285;
	deriv_cendist_interp300( ++i ) = -0.533;
	deriv_cendist_interp300( ++i ) = -0.813;
	deriv_cendist_interp300( ++i ) = -1.099;
	deriv_cendist_interp300( ++i ) = -1.365;
	deriv_cendist_interp300( ++i ) = -1.606;
	deriv_cendist_interp300( ++i ) = -1.821;
	deriv_cendist_interp300( ++i ) = -2.010;
	deriv_cendist_interp300( ++i ) = -2.173;
	deriv_cendist_interp300( ++i ) = -2.311;
	deriv_cendist_interp300( ++i ) = -2.424;
	deriv_cendist_interp300( ++i ) = -2.510;
	deriv_cendist_interp300( ++i ) = -2.572;
	deriv_cendist_interp300( ++i ) = -2.607;
	deriv_cendist_interp300( ++i ) = -2.617;
	deriv_cendist_interp300( ++i ) = -2.601;
	deriv_cendist_interp300( ++i ) = -2.560;
	deriv_cendist_interp300( ++i ) = -2.493;
	deriv_cendist_interp300( ++i ) = -2.401;
	deriv_cendist_interp300( ++i ) = -2.282;
	deriv_cendist_interp300( ++i ) = -2.139;
	deriv_cendist_interp300( ++i ) = -1.969;
	deriv_cendist_interp300( ++i ) = -1.777;
	deriv_cendist_interp300( ++i ) = -1.578;
	deriv_cendist_interp300( ++i ) = -1.386;
	deriv_cendist_interp300( ++i ) = -1.204;
	deriv_cendist_interp300( ++i ) = -1.033;
	deriv_cendist_interp300( ++i ) = -0.872;
	deriv_cendist_interp300( ++i ) = -0.722;
	deriv_cendist_interp300( ++i ) = -0.582;
	deriv_cendist_interp300( ++i ) = -0.453;
	deriv_cendist_interp300( ++i ) = -0.334;
	deriv_cendist_interp300( ++i ) = -0.226;
	deriv_cendist_interp300( ++i ) = -0.128;
	deriv_cendist_interp300( ++i ) = -0.041;
	deriv_cendist_interp300( ++i ) = 0.036;
	deriv_cendist_interp300( ++i ) = 0.103;
	deriv_cendist_interp300( ++i ) = 0.159;
	deriv_cendist_interp300( ++i ) = 0.204;
	deriv_cendist_interp300( ++i ) = 0.239;
	deriv_cendist_interp300( ++i ) = 0.264;
	deriv_cendist_interp300( ++i ) = 0.278;
	deriv_cendist_interp300( ++i ) = 0.282;
	deriv_cendist_interp300( ++i ) = 0.282;
	deriv_cendist_interp300( ++i ) = 0.281;
	deriv_cendist_interp300( ++i ) = 0.280;
	deriv_cendist_interp300( ++i ) = 0.279;
	deriv_cendist_interp300( ++i ) = 0.279;
	deriv_cendist_interp300( ++i ) = 0.279;
	deriv_cendist_interp300( ++i ) = 0.279;
	deriv_cendist_interp300( ++i ) = 0.280;
	deriv_cendist_interp300( ++i ) = 0.280;
	deriv_cendist_interp300( ++i ) = 0.281;
	deriv_cendist_interp300( ++i ) = 0.283;
	deriv_cendist_interp300( ++i ) = 0.284;
	deriv_cendist_interp300( ++i ) = 0.286;
	deriv_cendist_interp300( ++i ) = 0.288;
	deriv_cendist_interp300( ++i ) = 0.291;
	deriv_cendist_interp300( ++i ) = 0.293;
	deriv_cendist_interp300( ++i ) = 0.296;
	deriv_cendist_interp300( ++i ) = 0.299;
	deriv_cendist_interp300( ++i ) = 0.303;
	deriv_cendist_interp300( ++i ) = 0.307;
	deriv_cendist_interp300( ++i ) = 0.311;
	deriv_cendist_interp300( ++i ) = 0.315;
	deriv_cendist_interp300( ++i ) = 0.319;
	deriv_cendist_interp300( ++i ) = 0.322;
	deriv_cendist_interp300( ++i ) = 0.326;
	deriv_cendist_interp300( ++i ) = 0.330;
	deriv_cendist_interp300( ++i ) = 0.334;
	deriv_cendist_interp300( ++i ) = 0.338;
	deriv_cendist_interp300( ++i ) = 0.342;
	deriv_cendist_interp300( ++i ) = 0.346;
	deriv_cendist_interp300( ++i ) = 0.350;
	deriv_cendist_interp300( ++i ) = 0.354;
	deriv_cendist_interp300( ++i ) = 0.358;
	deriv_cendist_interp300( ++i ) = 0.362;
	deriv_cendist_interp300( ++i ) = 0.366;
	deriv_cendist_interp300( ++i ) = 0.370;
	deriv_cendist_interp300( ++i ) = 0.373;
	deriv_cendist_interp300( ++i ) = 0.377;
	deriv_cendist_interp300( ++i ) = 0.381;
	deriv_cendist_interp300( ++i ) = 0.385;
	deriv_cendist_interp300( ++i ) = 0.389;
	deriv_cendist_interp300( ++i ) = 0.393;
	deriv_cendist_interp300( ++i ) = 0.396;
	deriv_cendist_interp300( ++i ) = 0.399;
	deriv_cendist_interp300( ++i ) = 0.401;
	deriv_cendist_interp300( ++i ) = 0.404;
	deriv_cendist_interp300( ++i ) = 0.406;
	deriv_cendist_interp300( ++i ) = 0.408;
	deriv_cendist_interp300( ++i ) = 0.409;
	deriv_cendist_interp300( ++i ) = 0.411;
	deriv_cendist_interp300( ++i ) = 0.412;
	deriv_cendist_interp300( ++i ) = 0.412;
	deriv_cendist_interp300( ++i ) = 0.413;
	deriv_cendist_interp300( ++i ) = 0.413;
	deriv_cendist_interp300( ++i ) = 0.413;
	deriv_cendist_interp300( ++i ) = 0.413;
	deriv_cendist_interp300( ++i ) = 0.412;
	deriv_cendist_interp300( ++i ) = 0.411;
	deriv_cendist_interp300( ++i ) = 0.410;
	deriv_cendist_interp300( ++i ) = 0.409;
	deriv_cendist_interp300( ++i ) = 0.408;
	deriv_cendist_interp300( ++i ) = 0.408;
	deriv_cendist_interp300( ++i ) = 0.409;
	deriv_cendist_interp300( ++i ) = 0.411;
	deriv_cendist_interp300( ++i ) = 0.414;
	deriv_cendist_interp300( ++i ) = 0.419;
	deriv_cendist_interp300( ++i ) = 0.424;
	deriv_cendist_interp300( ++i ) = 0.431;
	deriv_cendist_interp300( ++i ) = 0.439;
	deriv_cendist_interp300( ++i ) = 0.448;
	deriv_cendist_interp300( ++i ) = 0.458;
	deriv_cendist_interp300( ++i ) = 0.469;
	deriv_cendist_interp300( ++i ) = 0.481;
	deriv_cendist_interp300( ++i ) = 0.495;
	deriv_cendist_interp300( ++i ) = 0.509;
	deriv_cendist_interp300( ++i ) = 0.525;
	deriv_cendist_interp300( ++i ) = 0.541;
	deriv_cendist_interp300( ++i ) = 0.559;
	deriv_cendist_interp300( ++i ) = 0.578;
	deriv_cendist_interp300( ++i ) = 0.598;
	deriv_cendist_interp300( ++i ) = 0.617;
	deriv_cendist_interp300( ++i ) = 0.635;
	deriv_cendist_interp300( ++i ) = 0.650;
	deriv_cendist_interp300( ++i ) = 0.663;
	deriv_cendist_interp300( ++i ) = 0.675;
	deriv_cendist_interp300( ++i ) = 0.684;
	deriv_cendist_interp300( ++i ) = 0.691;
	deriv_cendist_interp300( ++i ) = 0.696;
	deriv_cendist_interp300( ++i ) = 0.699;
	deriv_cendist_interp300( ++i ) = 0.700;
	deriv_cendist_interp300( ++i ) = 0.699;
	deriv_cendist_interp300( ++i ) = 0.696;
	deriv_cendist_interp300( ++i ) = 0.691;
	deriv_cendist_interp300( ++i ) = 0.684;
	deriv_cendist_interp300( ++i ) = 0.674;
	deriv_cendist_interp300( ++i ) = 0.663;
	deriv_cendist_interp300( ++i ) = 0.650;
	deriv_cendist_interp300( ++i ) = 0.634;
	deriv_cendist_interp300( ++i ) = 0.617;
	deriv_cendist_interp300( ++i ) = 0.597;
	deriv_cendist_interp300( ++i ) = 0.577;
	deriv_cendist_interp300( ++i ) = 0.556;
	deriv_cendist_interp300( ++i ) = 0.535;
	deriv_cendist_interp300( ++i ) = 0.513;
	deriv_cendist_interp300( ++i ) = 0.492;
	deriv_cendist_interp300( ++i ) = 0.470;
	deriv_cendist_interp300( ++i ) = 0.448;
	deriv_cendist_interp300( ++i ) = 0.426;
	deriv_cendist_interp300( ++i ) = 0.403;
	deriv_cendist_interp300( ++i ) = 0.381;
	deriv_cendist_interp300( ++i ) = 0.358;
	deriv_cendist_interp300( ++i ) = 0.335;
	deriv_cendist_interp300( ++i ) = 0.312;
	deriv_cendist_interp300( ++i ) = 0.288;
	deriv_cendist_interp300( ++i ) = 0.265;
	deriv_cendist_interp300( ++i ) = 0.241;
	deriv_cendist_interp300( ++i ) = 0.217;
	deriv_cendist_interp300( ++i ) = 0.192;
	deriv_cendist_interp300( ++i ) = 0.168;
	deriv_cendist_interp300( ++i ) = 0.143;
	deriv_cendist_interp300( ++i ) = 0.119;
	deriv_cendist_interp300( ++i ) = 0.097;
	deriv_cendist_interp300( ++i ) = 0.076;
	deriv_cendist_interp300( ++i ) = 0.057;
	deriv_cendist_interp300( ++i ) = 0.039;
	deriv_cendist_interp300( ++i ) = 0.023;
	deriv_cendist_interp300( ++i ) = 0.008;
	deriv_cendist_interp300( ++i ) = -0.005;
	deriv_cendist_interp300( ++i ) = -0.016;
	deriv_cendist_interp300( ++i ) = -0.026;
	deriv_cendist_interp300( ++i ) = -0.034;
	deriv_cendist_interp300( ++i ) = -0.041;
	deriv_cendist_interp300( ++i ) = -0.046;
	deriv_cendist_interp300( ++i ) = -0.050;
	deriv_cendist_interp300( ++i ) = -0.052;
	deriv_cendist_interp300( ++i ) = -0.052;
	deriv_cendist_interp300( ++i ) = -0.051;
	deriv_cendist_interp300( ++i ) = -0.048;
	deriv_cendist_interp300( ++i ) = -0.044;
	deriv_cendist_interp300( ++i ) = -0.038;
	deriv_cendist_interp300( ++i ) = -0.032;
	deriv_cendist_interp300( ++i ) = -0.026;
	deriv_cendist_interp300( ++i ) = -0.020;
	deriv_cendist_interp300( ++i ) = -0.015;
	deriv_cendist_interp300( ++i ) = -0.010;
	deriv_cendist_interp300( ++i ) = -0.006;
	deriv_cendist_interp300( ++i ) = -0.002;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.004;
	deriv_cendist_interp300( ++i ) = 0.007;
	deriv_cendist_interp300( ++i ) = 0.009;
	deriv_cendist_interp300( ++i ) = 0.011;
	deriv_cendist_interp300( ++i ) = 0.012;
	deriv_cendist_interp300( ++i ) = 0.013;
	deriv_cendist_interp300( ++i ) = 0.014;
	deriv_cendist_interp300( ++i ) = 0.014;
	deriv_cendist_interp300( ++i ) = 0.014;
	deriv_cendist_interp300( ++i ) = 0.013;
	deriv_cendist_interp300( ++i ) = 0.012;
	deriv_cendist_interp300( ++i ) = 0.010;
	deriv_cendist_interp300( ++i ) = 0.009;
	deriv_cendist_interp300( ++i ) = 0.007;
	deriv_cendist_interp300( ++i ) = 0.005;
	deriv_cendist_interp300( ++i ) = 0.004;
	deriv_cendist_interp300( ++i ) = 0.003;
	deriv_cendist_interp300( ++i ) = 0.002;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.001;
	deriv_cendist_interp300( ++i ) = -0.002;
	deriv_cendist_interp300( ++i ) = -0.002;
	deriv_cendist_interp300( ++i ) = -0.003;
	deriv_cendist_interp300( ++i ) = -0.003;
	deriv_cendist_interp300( ++i ) = -0.004;
	deriv_cendist_interp300( ++i ) = -0.004;
	deriv_cendist_interp300( ++i ) = -0.004;
	deriv_cendist_interp300( ++i ) = -0.004;
	deriv_cendist_interp300( ++i ) = -0.003;
	deriv_cendist_interp300( ++i ) = -0.003;
	deriv_cendist_interp300( ++i ) = -0.003;
	deriv_cendist_interp300( ++i ) = -0.002;
	deriv_cendist_interp300( ++i ) = -0.002;
	deriv_cendist_interp300( ++i ) = -0.001;
	deriv_cendist_interp300( ++i ) = -0.001;
	deriv_cendist_interp300( ++i ) = -0.001;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.001;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = 0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
	deriv_cendist_interp300( ++i ) = -0.000;
}



//------------------------------------------------------------------------------


void
Disulfides::deriv_cbdist_score_initializer( FArray1D_float & deriv_cbdist_score )
{
	int i = 0;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = -2.114;
	deriv_cbdist_score( ++i ) = -2.094;
	deriv_cbdist_score( ++i ) = -2.662;
	deriv_cbdist_score( ++i ) = -3.607;
	deriv_cbdist_score( ++i ) = -1.575;
	deriv_cbdist_score( ++i ) = -1.063;
	deriv_cbdist_score( ++i ) = -0.554;
	deriv_cbdist_score( ++i ) = -0.060;
	deriv_cbdist_score( ++i ) = 0.439;
	deriv_cbdist_score( ++i ) = 0.473;
	deriv_cbdist_score( ++i ) = 0.364;
	deriv_cbdist_score( ++i ) = 0.701;
	deriv_cbdist_score( ++i ) = 1.038;
	deriv_cbdist_score( ++i ) = 1.371;
	deriv_cbdist_score( ++i ) = 1.721;
	deriv_cbdist_score( ++i ) = 1.507;
	deriv_cbdist_score( ++i ) = 0.907;
	deriv_cbdist_score( ++i ) = 0.425;
	deriv_cbdist_score( ++i ) = 0.187;
	deriv_cbdist_score( ++i ) = 0.190;
	deriv_cbdist_score( ++i ) = 0.194;
	deriv_cbdist_score( ++i ) = 0.950;
	deriv_cbdist_score( ++i ) = 1.711;
	deriv_cbdist_score( ++i ) = 1.205;
	deriv_cbdist_score( ++i ) = 0.345;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
	deriv_cbdist_score( ++i ) = 0.000;
}


//------------------------------------------------------------------------------


////////////////////////////////////////////////////////////////////////////////
// This is used to initialize two instances of the array
////////////////////////////////////////////////////////////////////////////////
void
Disulfides::deriv_ssdist_sc_interp300_initializer( FArray1D_float & deriv_ssdist_sc_interp300 )
{
	int i = 0;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.001;
	deriv_ssdist_sc_interp300( ++i ) = -0.002;
	deriv_ssdist_sc_interp300( ++i ) = -0.002;
	deriv_ssdist_sc_interp300( ++i ) = -0.003;
	deriv_ssdist_sc_interp300( ++i ) = -0.003;
	deriv_ssdist_sc_interp300( ++i ) = -0.004;
	deriv_ssdist_sc_interp300( ++i ) = -0.004;
	deriv_ssdist_sc_interp300( ++i ) = -0.005;
	deriv_ssdist_sc_interp300( ++i ) = -0.006;
	deriv_ssdist_sc_interp300( ++i ) = -0.008;
	deriv_ssdist_sc_interp300( ++i ) = -0.009;
	deriv_ssdist_sc_interp300( ++i ) = -0.011;
	deriv_ssdist_sc_interp300( ++i ) = -0.013;
	deriv_ssdist_sc_interp300( ++i ) = -0.015;
	deriv_ssdist_sc_interp300( ++i ) = -0.018;
	deriv_ssdist_sc_interp300( ++i ) = -0.021;
	deriv_ssdist_sc_interp300( ++i ) = -0.025;
	deriv_ssdist_sc_interp300( ++i ) = -0.029;
	deriv_ssdist_sc_interp300( ++i ) = -0.034;
	deriv_ssdist_sc_interp300( ++i ) = -0.039;
	deriv_ssdist_sc_interp300( ++i ) = -0.045;
	deriv_ssdist_sc_interp300( ++i ) = -0.053;
	deriv_ssdist_sc_interp300( ++i ) = -0.061;
	deriv_ssdist_sc_interp300( ++i ) = -0.070;
	deriv_ssdist_sc_interp300( ++i ) = -0.080;
	deriv_ssdist_sc_interp300( ++i ) = -0.092;
	deriv_ssdist_sc_interp300( ++i ) = -0.105;
	deriv_ssdist_sc_interp300( ++i ) = -0.119;
	deriv_ssdist_sc_interp300( ++i ) = -0.135;
	deriv_ssdist_sc_interp300( ++i ) = -0.154;
	deriv_ssdist_sc_interp300( ++i ) = -0.174;
	deriv_ssdist_sc_interp300( ++i ) = -0.196;
	deriv_ssdist_sc_interp300( ++i ) = -0.221;
	deriv_ssdist_sc_interp300( ++i ) = -0.248;
	deriv_ssdist_sc_interp300( ++i ) = -0.277;
	deriv_ssdist_sc_interp300( ++i ) = -0.310;
	deriv_ssdist_sc_interp300( ++i ) = -0.346;
	deriv_ssdist_sc_interp300( ++i ) = -0.385;
	deriv_ssdist_sc_interp300( ++i ) = -0.428;
	deriv_ssdist_sc_interp300( ++i ) = -0.475;
	deriv_ssdist_sc_interp300( ++i ) = -0.525;
	deriv_ssdist_sc_interp300( ++i ) = -0.581;
	deriv_ssdist_sc_interp300( ++i ) = -0.642;
	deriv_ssdist_sc_interp300( ++i ) = -0.708;
	deriv_ssdist_sc_interp300( ++i ) = -0.780;
	deriv_ssdist_sc_interp300( ++i ) = -0.860;
	deriv_ssdist_sc_interp300( ++i ) = -0.949;
	deriv_ssdist_sc_interp300( ++i ) = -1.047;
	deriv_ssdist_sc_interp300( ++i ) = -1.157;
	deriv_ssdist_sc_interp300( ++i ) = -1.281;
	deriv_ssdist_sc_interp300( ++i ) = -1.422;
	deriv_ssdist_sc_interp300( ++i ) = -1.584;
	deriv_ssdist_sc_interp300( ++i ) = -1.772;
	deriv_ssdist_sc_interp300( ++i ) = -1.991;
	deriv_ssdist_sc_interp300( ++i ) = -2.248;
	deriv_ssdist_sc_interp300( ++i ) = -2.551;
	deriv_ssdist_sc_interp300( ++i ) = -2.911;
	deriv_ssdist_sc_interp300( ++i ) = -3.338;
	deriv_ssdist_sc_interp300( ++i ) = -3.847;
	deriv_ssdist_sc_interp300( ++i ) = -4.452;
	deriv_ssdist_sc_interp300( ++i ) = -5.170;
	deriv_ssdist_sc_interp300( ++i ) = -6.020;
	deriv_ssdist_sc_interp300( ++i ) = -7.021;
	deriv_ssdist_sc_interp300( ++i ) = -8.193;
	deriv_ssdist_sc_interp300( ++i ) = -9.558;
	deriv_ssdist_sc_interp300( ++i ) = -11.136;
	deriv_ssdist_sc_interp300( ++i ) = -12.946;
	deriv_ssdist_sc_interp300( ++i ) = -15.003;
	deriv_ssdist_sc_interp300( ++i ) = -17.320;
	deriv_ssdist_sc_interp300( ++i ) = -19.902;
	deriv_ssdist_sc_interp300( ++i ) = -22.751;
	deriv_ssdist_sc_interp300( ++i ) = -25.857;
	deriv_ssdist_sc_interp300( ++i ) = -29.201;
	deriv_ssdist_sc_interp300( ++i ) = -32.753;
	deriv_ssdist_sc_interp300( ++i ) = -36.471;
	deriv_ssdist_sc_interp300( ++i ) = -40.298;
	deriv_ssdist_sc_interp300( ++i ) = -44.167;
	deriv_ssdist_sc_interp300( ++i ) = -47.997;
	deriv_ssdist_sc_interp300( ++i ) = -51.694;
	deriv_ssdist_sc_interp300( ++i ) = -55.157;
	deriv_ssdist_sc_interp300( ++i ) = -58.276;
	deriv_ssdist_sc_interp300( ++i ) = -60.938;
	deriv_ssdist_sc_interp300( ++i ) = -63.028;
	deriv_ssdist_sc_interp300( ++i ) = -64.438;
	deriv_ssdist_sc_interp300( ++i ) = -65.066;
	deriv_ssdist_sc_interp300( ++i ) = -64.826;
	deriv_ssdist_sc_interp300( ++i ) = -63.648;
	deriv_ssdist_sc_interp300( ++i ) = -61.484;
	deriv_ssdist_sc_interp300( ++i ) = -58.313;
	deriv_ssdist_sc_interp300( ++i ) = -54.141;
	deriv_ssdist_sc_interp300( ++i ) = -49.005;
	deriv_ssdist_sc_interp300( ++i ) = -42.971;
	deriv_ssdist_sc_interp300( ++i ) = -36.134;
	deriv_ssdist_sc_interp300( ++i ) = -28.617;
	deriv_ssdist_sc_interp300( ++i ) = -20.565;
	deriv_ssdist_sc_interp300( ++i ) = -12.142;
	deriv_ssdist_sc_interp300( ++i ) = -3.524;
	deriv_ssdist_sc_interp300( ++i ) = 5.106;
	deriv_ssdist_sc_interp300( ++i ) = 13.563;
	deriv_ssdist_sc_interp300( ++i ) = 21.671;
	deriv_ssdist_sc_interp300( ++i ) = 29.266;
	deriv_ssdist_sc_interp300( ++i ) = 36.202;
	deriv_ssdist_sc_interp300( ++i ) = 42.356;
	deriv_ssdist_sc_interp300( ++i ) = 47.633;
	deriv_ssdist_sc_interp300( ++i ) = 51.965;
	deriv_ssdist_sc_interp300( ++i ) = 55.315;
	deriv_ssdist_sc_interp300( ++i ) = 57.676;
	deriv_ssdist_sc_interp300( ++i ) = 59.068;
	deriv_ssdist_sc_interp300( ++i ) = 59.537;
	deriv_ssdist_sc_interp300( ++i ) = 59.152;
	deriv_ssdist_sc_interp300( ++i ) = 57.999;
	deriv_ssdist_sc_interp300( ++i ) = 56.177;
	deriv_ssdist_sc_interp300( ++i ) = 53.795;
	deriv_ssdist_sc_interp300( ++i ) = 50.964;
	deriv_ssdist_sc_interp300( ++i ) = 47.797;
	deriv_ssdist_sc_interp300( ++i ) = 44.402;
	deriv_ssdist_sc_interp300( ++i ) = 40.880;
	deriv_ssdist_sc_interp300( ++i ) = 37.321;
	deriv_ssdist_sc_interp300( ++i ) = 33.807;
	deriv_ssdist_sc_interp300( ++i ) = 30.403;
	deriv_ssdist_sc_interp300( ++i ) = 27.164;
	deriv_ssdist_sc_interp300( ++i ) = 24.130;
	deriv_ssdist_sc_interp300( ++i ) = 21.332;
	deriv_ssdist_sc_interp300( ++i ) = 18.786;
	deriv_ssdist_sc_interp300( ++i ) = 16.501;
	deriv_ssdist_sc_interp300( ++i ) = 14.475;
	deriv_ssdist_sc_interp300( ++i ) = 12.701;
	deriv_ssdist_sc_interp300( ++i ) = 11.166;
	deriv_ssdist_sc_interp300( ++i ) = 9.853;
	deriv_ssdist_sc_interp300( ++i ) = 8.743;
	deriv_ssdist_sc_interp300( ++i ) = 7.814;
	deriv_ssdist_sc_interp300( ++i ) = 7.045;
	deriv_ssdist_sc_interp300( ++i ) = 6.415;
	deriv_ssdist_sc_interp300( ++i ) = 5.905;
	deriv_ssdist_sc_interp300( ++i ) = 5.494;
	deriv_ssdist_sc_interp300( ++i ) = 5.167;
	deriv_ssdist_sc_interp300( ++i ) = 4.907;
	deriv_ssdist_sc_interp300( ++i ) = 4.702;
	deriv_ssdist_sc_interp300( ++i ) = 4.540;
	deriv_ssdist_sc_interp300( ++i ) = 4.411;
	deriv_ssdist_sc_interp300( ++i ) = 4.306;
	deriv_ssdist_sc_interp300( ++i ) = 4.220;
	deriv_ssdist_sc_interp300( ++i ) = 4.146;
	deriv_ssdist_sc_interp300( ++i ) = 4.080;
	deriv_ssdist_sc_interp300( ++i ) = 4.019;
	deriv_ssdist_sc_interp300( ++i ) = 3.961;
	deriv_ssdist_sc_interp300( ++i ) = 3.902;
	deriv_ssdist_sc_interp300( ++i ) = 3.843;
	deriv_ssdist_sc_interp300( ++i ) = 3.782;
	deriv_ssdist_sc_interp300( ++i ) = 3.718;
	deriv_ssdist_sc_interp300( ++i ) = 3.651;
	deriv_ssdist_sc_interp300( ++i ) = 3.580;
	deriv_ssdist_sc_interp300( ++i ) = 3.507;
	deriv_ssdist_sc_interp300( ++i ) = 3.430;
	deriv_ssdist_sc_interp300( ++i ) = 3.351;
	deriv_ssdist_sc_interp300( ++i ) = 3.269;
	deriv_ssdist_sc_interp300( ++i ) = 3.185;
	deriv_ssdist_sc_interp300( ++i ) = 3.098;
	deriv_ssdist_sc_interp300( ++i ) = 3.010;
	deriv_ssdist_sc_interp300( ++i ) = 2.920;
	deriv_ssdist_sc_interp300( ++i ) = 2.830;
	deriv_ssdist_sc_interp300( ++i ) = 2.739;
	deriv_ssdist_sc_interp300( ++i ) = 2.647;
	deriv_ssdist_sc_interp300( ++i ) = 2.555;
	deriv_ssdist_sc_interp300( ++i ) = 2.463;
	deriv_ssdist_sc_interp300( ++i ) = 2.372;
	deriv_ssdist_sc_interp300( ++i ) = 2.281;
	deriv_ssdist_sc_interp300( ++i ) = 2.192;
	deriv_ssdist_sc_interp300( ++i ) = 2.103;
	deriv_ssdist_sc_interp300( ++i ) = 2.016;
	deriv_ssdist_sc_interp300( ++i ) = 1.930;
	deriv_ssdist_sc_interp300( ++i ) = 1.845;
	deriv_ssdist_sc_interp300( ++i ) = 1.763;
	deriv_ssdist_sc_interp300( ++i ) = 1.682;
	deriv_ssdist_sc_interp300( ++i ) = 1.603;
	deriv_ssdist_sc_interp300( ++i ) = 1.527;
	deriv_ssdist_sc_interp300( ++i ) = 1.452;
	deriv_ssdist_sc_interp300( ++i ) = 1.380;
	deriv_ssdist_sc_interp300( ++i ) = 1.310;
	deriv_ssdist_sc_interp300( ++i ) = 1.242;
	deriv_ssdist_sc_interp300( ++i ) = 1.177;
	deriv_ssdist_sc_interp300( ++i ) = 1.114;
	deriv_ssdist_sc_interp300( ++i ) = 1.053;
	deriv_ssdist_sc_interp300( ++i ) = 0.995;
	deriv_ssdist_sc_interp300( ++i ) = 0.939;
	deriv_ssdist_sc_interp300( ++i ) = 0.885;
	deriv_ssdist_sc_interp300( ++i ) = 0.834;
	deriv_ssdist_sc_interp300( ++i ) = 0.785;
	deriv_ssdist_sc_interp300( ++i ) = 0.738;
	deriv_ssdist_sc_interp300( ++i ) = 0.693;
	deriv_ssdist_sc_interp300( ++i ) = 0.651;
	deriv_ssdist_sc_interp300( ++i ) = 0.610;
	deriv_ssdist_sc_interp300( ++i ) = 0.572;
	deriv_ssdist_sc_interp300( ++i ) = 0.535;
	deriv_ssdist_sc_interp300( ++i ) = 0.501;
	deriv_ssdist_sc_interp300( ++i ) = 0.468;
	deriv_ssdist_sc_interp300( ++i ) = 0.437;
	deriv_ssdist_sc_interp300( ++i ) = 0.408;
	deriv_ssdist_sc_interp300( ++i ) = 0.380;
	deriv_ssdist_sc_interp300( ++i ) = 0.354;
	deriv_ssdist_sc_interp300( ++i ) = 0.330;
	deriv_ssdist_sc_interp300( ++i ) = 0.307;
	deriv_ssdist_sc_interp300( ++i ) = 0.285;
	deriv_ssdist_sc_interp300( ++i ) = 0.265;
	deriv_ssdist_sc_interp300( ++i ) = 0.245;
	deriv_ssdist_sc_interp300( ++i ) = 0.228;
	deriv_ssdist_sc_interp300( ++i ) = 0.211;
	deriv_ssdist_sc_interp300( ++i ) = 0.195;
	deriv_ssdist_sc_interp300( ++i ) = 0.180;
	deriv_ssdist_sc_interp300( ++i ) = 0.167;
	deriv_ssdist_sc_interp300( ++i ) = 0.154;
	deriv_ssdist_sc_interp300( ++i ) = 0.142;
	deriv_ssdist_sc_interp300( ++i ) = 0.131;
	deriv_ssdist_sc_interp300( ++i ) = 0.121;
	deriv_ssdist_sc_interp300( ++i ) = 0.111;
	deriv_ssdist_sc_interp300( ++i ) = 0.102;
	deriv_ssdist_sc_interp300( ++i ) = 0.094;
	deriv_ssdist_sc_interp300( ++i ) = 0.087;
	deriv_ssdist_sc_interp300( ++i ) = 0.080;
	deriv_ssdist_sc_interp300( ++i ) = 0.073;
	deriv_ssdist_sc_interp300( ++i ) = 0.067;
	deriv_ssdist_sc_interp300( ++i ) = 0.061;
	deriv_ssdist_sc_interp300( ++i ) = 0.056;
	deriv_ssdist_sc_interp300( ++i ) = 0.051;
	deriv_ssdist_sc_interp300( ++i ) = 0.047;
	deriv_ssdist_sc_interp300( ++i ) = 0.043;
	deriv_ssdist_sc_interp300( ++i ) = 0.039;
	deriv_ssdist_sc_interp300( ++i ) = 0.036;
	deriv_ssdist_sc_interp300( ++i ) = 0.033;
	deriv_ssdist_sc_interp300( ++i ) = 0.030;
	deriv_ssdist_sc_interp300( ++i ) = 0.027;
	deriv_ssdist_sc_interp300( ++i ) = 0.025;
	deriv_ssdist_sc_interp300( ++i ) = 0.023;
	deriv_ssdist_sc_interp300( ++i ) = 0.020;
	deriv_ssdist_sc_interp300( ++i ) = 0.019;
	deriv_ssdist_sc_interp300( ++i ) = 0.017;
	deriv_ssdist_sc_interp300( ++i ) = 0.015;
	deriv_ssdist_sc_interp300( ++i ) = 0.014;
	deriv_ssdist_sc_interp300( ++i ) = 0.013;
	deriv_ssdist_sc_interp300( ++i ) = 0.011;
	deriv_ssdist_sc_interp300( ++i ) = 0.010;
	deriv_ssdist_sc_interp300( ++i ) = 0.009;
	deriv_ssdist_sc_interp300( ++i ) = 0.008;
	deriv_ssdist_sc_interp300( ++i ) = 0.008;
	deriv_ssdist_sc_interp300( ++i ) = 0.007;
	deriv_ssdist_sc_interp300( ++i ) = 0.006;
	deriv_ssdist_sc_interp300( ++i ) = 0.006;
	deriv_ssdist_sc_interp300( ++i ) = 0.005;
	deriv_ssdist_sc_interp300( ++i ) = 0.005;
	deriv_ssdist_sc_interp300( ++i ) = 0.004;
	deriv_ssdist_sc_interp300( ++i ) = 0.004;
	deriv_ssdist_sc_interp300( ++i ) = 0.003;
	deriv_ssdist_sc_interp300( ++i ) = 0.003;
	deriv_ssdist_sc_interp300( ++i ) = 0.003;
	deriv_ssdist_sc_interp300( ++i ) = 0.002;
	deriv_ssdist_sc_interp300( ++i ) = 0.002;
	deriv_ssdist_sc_interp300( ++i ) = 0.002;
	deriv_ssdist_sc_interp300( ++i ) = 0.002;
	deriv_ssdist_sc_interp300( ++i ) = 0.002;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.001;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
	deriv_ssdist_sc_interp300( ++i ) = 0.000;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin farlx_csang_score_initializer
///
/// @brief
/// This is used to initialize two instances of the array
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::farlx_csang_score_initializer( FArray1D_float & farlx_csang_score )
{
	int i = 0;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 2.999;
	farlx_csang_score( ++i ) = 2.999;
	farlx_csang_score( ++i ) = 2.997;
	farlx_csang_score( ++i ) = 2.994;
	farlx_csang_score( ++i ) = 2.988;
	farlx_csang_score( ++i ) = 2.978;
	farlx_csang_score( ++i ) = 2.958;
	farlx_csang_score( ++i ) = 2.925;
	farlx_csang_score( ++i ) = 2.871;
	farlx_csang_score( ++i ) = 2.785;
	farlx_csang_score( ++i ) = 2.654;
	farlx_csang_score( ++i ) = 2.464;
	farlx_csang_score( ++i ) = 2.199;
	farlx_csang_score( ++i ) = 1.846;
	farlx_csang_score( ++i ) = 1.397;
	farlx_csang_score( ++i ) = 0.853;
	farlx_csang_score( ++i ) = 0.227;
	farlx_csang_score( ++i ) = -0.453;
	farlx_csang_score( ++i ) = -1.145;
	farlx_csang_score( ++i ) = -1.797;
	farlx_csang_score( ++i ) = -2.353;
	farlx_csang_score( ++i ) = -2.758;
	farlx_csang_score( ++i ) = -2.973;
	farlx_csang_score( ++i ) = -2.973;
	farlx_csang_score( ++i ) = -2.758;
	farlx_csang_score( ++i ) = -2.353;
	farlx_csang_score( ++i ) = -1.797;
	farlx_csang_score( ++i ) = -1.145;
	farlx_csang_score( ++i ) = -0.453;
	farlx_csang_score( ++i ) = 0.227;
	farlx_csang_score( ++i ) = 0.853;
	farlx_csang_score( ++i ) = 1.397;
	farlx_csang_score( ++i ) = 1.846;
	farlx_csang_score( ++i ) = 2.199;
	farlx_csang_score( ++i ) = 2.464;
	farlx_csang_score( ++i ) = 2.654;
	farlx_csang_score( ++i ) = 2.785;
	farlx_csang_score( ++i ) = 2.871;
	farlx_csang_score( ++i ) = 2.925;
	farlx_csang_score( ++i ) = 2.958;
	farlx_csang_score( ++i ) = 2.978;
	farlx_csang_score( ++i ) = 2.988;
	farlx_csang_score( ++i ) = 2.994;
	farlx_csang_score( ++i ) = 2.997;
	farlx_csang_score( ++i ) = 2.999;
	farlx_csang_score( ++i ) = 2.999;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
	farlx_csang_score( ++i ) = 3.000;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin farlx_csang_score_2_initializer
///
/// @brief
/// This is used to initialize two instances of the array
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::farlx_csang_score_2_initializer( FArray1D_float & farlx_csang_score_2 )
{
	int i = 0;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 9.999;
	farlx_csang_score_2( ++i ) = 9.999;
	farlx_csang_score_2( ++i ) = 9.998;
	farlx_csang_score_2( ++i ) = 9.997;
	farlx_csang_score_2( ++i ) = 9.995;
	farlx_csang_score_2( ++i ) = 9.992;
	farlx_csang_score_2( ++i ) = 9.987;
	farlx_csang_score_2( ++i ) = 9.980;
	farlx_csang_score_2( ++i ) = 9.969;
	farlx_csang_score_2( ++i ) = 9.954;
	farlx_csang_score_2( ++i ) = 9.932;
	farlx_csang_score_2( ++i ) = 9.901;
	farlx_csang_score_2( ++i ) = 9.858;
	farlx_csang_score_2( ++i ) = 9.799;
	farlx_csang_score_2( ++i ) = 9.719;
	farlx_csang_score_2( ++i ) = 9.614;
	farlx_csang_score_2( ++i ) = 9.475;
	farlx_csang_score_2( ++i ) = 9.298;
	farlx_csang_score_2( ++i ) = 9.073;
	farlx_csang_score_2( ++i ) = 8.793;
	farlx_csang_score_2( ++i ) = 8.450;
	farlx_csang_score_2( ++i ) = 8.037;
	farlx_csang_score_2( ++i ) = 7.549;
	farlx_csang_score_2( ++i ) = 6.981;
	farlx_csang_score_2( ++i ) = 6.333;
	farlx_csang_score_2( ++i ) = 5.608;
	farlx_csang_score_2( ++i ) = 4.811;
	farlx_csang_score_2( ++i ) = 3.954;
	farlx_csang_score_2( ++i ) = 3.054;
	farlx_csang_score_2( ++i ) = 2.129;
	farlx_csang_score_2( ++i ) = 1.204;
	farlx_csang_score_2( ++i ) = 0.306;
	farlx_csang_score_2( ++i ) = -0.537;
	farlx_csang_score_2( ++i ) = -1.295;
	farlx_csang_score_2( ++i ) = -1.940;
	farlx_csang_score_2( ++i ) = -2.448;
	farlx_csang_score_2( ++i ) = -2.798;
	farlx_csang_score_2( ++i ) = -2.977;
	farlx_csang_score_2( ++i ) = -2.977;
	farlx_csang_score_2( ++i ) = -2.798;
	farlx_csang_score_2( ++i ) = -2.448;
	farlx_csang_score_2( ++i ) = -1.940;
	farlx_csang_score_2( ++i ) = -1.295;
	farlx_csang_score_2( ++i ) = -0.537;
	farlx_csang_score_2( ++i ) = 0.306;
	farlx_csang_score_2( ++i ) = 1.204;
	farlx_csang_score_2( ++i ) = 2.129;
	farlx_csang_score_2( ++i ) = 3.054;
	farlx_csang_score_2( ++i ) = 3.954;
	farlx_csang_score_2( ++i ) = 4.811;
	farlx_csang_score_2( ++i ) = 5.608;
	farlx_csang_score_2( ++i ) = 6.333;
	farlx_csang_score_2( ++i ) = 6.981;
	farlx_csang_score_2( ++i ) = 7.549;
	farlx_csang_score_2( ++i ) = 8.037;
	farlx_csang_score_2( ++i ) = 8.450;
	farlx_csang_score_2( ++i ) = 8.793;
	farlx_csang_score_2( ++i ) = 9.073;
	farlx_csang_score_2( ++i ) = 9.298;
	farlx_csang_score_2( ++i ) = 9.475;
	farlx_csang_score_2( ++i ) = 9.614;
	farlx_csang_score_2( ++i ) = 9.719;
	farlx_csang_score_2( ++i ) = 9.799;
	farlx_csang_score_2( ++i ) = 9.858;
	farlx_csang_score_2( ++i ) = 9.901;
	farlx_csang_score_2( ++i ) = 9.932;
	farlx_csang_score_2( ++i ) = 9.954;
	farlx_csang_score_2( ++i ) = 9.969;
	farlx_csang_score_2( ++i ) = 9.980;
	farlx_csang_score_2( ++i ) = 9.987;
	farlx_csang_score_2( ++i ) = 9.992;
	farlx_csang_score_2( ++i ) = 9.995;
	farlx_csang_score_2( ++i ) = 9.997;
	farlx_csang_score_2( ++i ) = 9.998;
	farlx_csang_score_2( ++i ) = 9.999;
	farlx_csang_score_2( ++i ) = 9.999;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
	farlx_csang_score_2( ++i ) = 10.000;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin farlx_dihed_score_initializer
///
/// @brief
/// This is used to initialize two instances of the array
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::farlx_dihed_score_initializer( FArray1D_float & farlx_dihed_score )
{
	int i = 0;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 2.999;
	farlx_dihed_score( ++i ) = 2.997;
	farlx_dihed_score( ++i ) = 2.992;
	farlx_dihed_score( ++i ) = 2.981;
	farlx_dihed_score( ++i ) = 2.958;
	farlx_dihed_score( ++i ) = 2.915;
	farlx_dihed_score( ++i ) = 2.836;
	farlx_dihed_score( ++i ) = 2.701;
	farlx_dihed_score( ++i ) = 2.486;
	farlx_dihed_score( ++i ) = 2.166;
	farlx_dihed_score( ++i ) = 1.722;
	farlx_dihed_score( ++i ) = 1.153;
	farlx_dihed_score( ++i ) = 0.480;
	farlx_dihed_score( ++i ) = -0.244;
	farlx_dihed_score( ++i ) = -0.942;
	farlx_dihed_score( ++i ) = -1.521;
	farlx_dihed_score( ++i ) = -1.893;
	farlx_dihed_score( ++i ) = -1.997;
	farlx_dihed_score( ++i ) = -1.817;
	farlx_dihed_score( ++i ) = -1.383;
	farlx_dihed_score( ++i ) = -0.763;
	farlx_dihed_score( ++i ) = -0.049;
	farlx_dihed_score( ++i ) = 0.668;
	farlx_dihed_score( ++i ) = 1.317;
	farlx_dihed_score( ++i ) = 1.854;
	farlx_dihed_score( ++i ) = 2.263;
	farlx_dihed_score( ++i ) = 2.553;
	farlx_dihed_score( ++i ) = 2.744;
	farlx_dihed_score( ++i ) = 2.862;
	farlx_dihed_score( ++i ) = 2.929;
	farlx_dihed_score( ++i ) = 2.966;
	farlx_dihed_score( ++i ) = 2.985;
	farlx_dihed_score( ++i ) = 2.993;
	farlx_dihed_score( ++i ) = 2.997;
	farlx_dihed_score( ++i ) = 2.999;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 3.000;
	farlx_dihed_score( ++i ) = 2.999;
	farlx_dihed_score( ++i ) = 2.998;
	farlx_dihed_score( ++i ) = 2.996;
	farlx_dihed_score( ++i ) = 2.993;
	farlx_dihed_score( ++i ) = 2.987;
	farlx_dihed_score( ++i ) = 2.976;
	farlx_dihed_score( ++i ) = 2.958;
	farlx_dihed_score( ++i ) = 2.929;
	farlx_dihed_score( ++i ) = 2.884;
	farlx_dihed_score( ++i ) = 2.816;
	farlx_dihed_score( ++i ) = 2.717;
	farlx_dihed_score( ++i ) = 2.578;
	farlx_dihed_score( ++i ) = 2.390;
	farlx_dihed_score( ++i ) = 2.144;
	farlx_dihed_score( ++i ) = 1.836;
	farlx_dihed_score( ++i ) = 1.464;
	farlx_dihed_score( ++i ) = 1.034;
	farlx_dihed_score( ++i ) = 0.560;
	farlx_dihed_score( ++i ) = 0.064;
	farlx_dihed_score( ++i ) = -0.427;
	farlx_dihed_score( ++i ) = -0.879;
	farlx_dihed_score( ++i ) = -1.257;
	farlx_dihed_score( ++i ) = -1.532;
	farlx_dihed_score( ++i ) = -1.679;
	farlx_dihed_score( ++i ) = -1.685;
	farlx_dihed_score( ++i ) = -1.549;
	farlx_dihed_score( ++i ) = -1.283;
	farlx_dihed_score( ++i ) = -0.912;
	farlx_dihed_score( ++i ) = -0.464;
	farlx_dihed_score( ++i ) = 0.025;
	farlx_dihed_score( ++i ) = 0.522;
	farlx_dihed_score( ++i ) = 0.998;
	farlx_dihed_score( ++i ) = 1.432;
	farlx_dihed_score( ++i ) = 1.809;
	farlx_dihed_score( ++i ) = 2.122;
	farlx_dihed_score( ++i ) = 2.373;
	farlx_dihed_score( ++i ) = 2.565;
	farlx_dihed_score( ++i ) = 2.708;
	farlx_dihed_score( ++i ) = 2.810;
	farlx_dihed_score( ++i ) = 2.880;
	farlx_dihed_score( ++i ) = 2.926;
	farlx_dihed_score( ++i ) = 2.956;
	farlx_dihed_score( ++i ) = 2.975;
	farlx_dihed_score( ++i ) = 2.986;
	farlx_dihed_score( ++i ) = 2.992;
	farlx_dihed_score( ++i ) = 2.996;
	farlx_dihed_score( ++i ) = 2.998;
	farlx_dihed_score( ++i ) = 2.999;
	farlx_dihed_score( ++i ) = 3.000;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin farlx_dihed_score_2_initializer
///
/// @brief
/// This is used to initialize two instances of the array
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::farlx_dihed_score_2_initializer( FArray1D_float & farlx_dihed_score_2 )
{
	int i = 0;
	farlx_dihed_score_2( ++i ) = 9.960;
	farlx_dihed_score_2( ++i ) = 9.938;
	farlx_dihed_score_2( ++i ) = 9.906;
	farlx_dihed_score_2( ++i ) = 9.861;
	farlx_dihed_score_2( ++i ) = 9.796;
	farlx_dihed_score_2( ++i ) = 9.706;
	farlx_dihed_score_2( ++i ) = 9.584;
	farlx_dihed_score_2( ++i ) = 9.422;
	farlx_dihed_score_2( ++i ) = 9.209;
	farlx_dihed_score_2( ++i ) = 8.937;
	farlx_dihed_score_2( ++i ) = 8.595;
	farlx_dihed_score_2( ++i ) = 8.174;
	farlx_dihed_score_2( ++i ) = 7.668;
	farlx_dihed_score_2( ++i ) = 7.072;
	farlx_dihed_score_2( ++i ) = 6.385;
	farlx_dihed_score_2( ++i ) = 5.612;
	farlx_dihed_score_2( ++i ) = 4.764;
	farlx_dihed_score_2( ++i ) = 3.858;
	farlx_dihed_score_2( ++i ) = 2.917;
	farlx_dihed_score_2( ++i ) = 1.970;
	farlx_dihed_score_2( ++i ) = 1.049;
	farlx_dihed_score_2( ++i ) = 0.192;
	farlx_dihed_score_2( ++i ) = -0.565;
	farlx_dihed_score_2( ++i ) = -1.189;
	farlx_dihed_score_2( ++i ) = -1.649;
	farlx_dihed_score_2( ++i ) = -1.924;
	farlx_dihed_score_2( ++i ) = -1.998;
	farlx_dihed_score_2( ++i ) = -1.869;
	farlx_dihed_score_2( ++i ) = -1.543;
	farlx_dihed_score_2( ++i ) = -1.037;
	farlx_dihed_score_2( ++i ) = -0.374;
	farlx_dihed_score_2( ++i ) = 0.414;
	farlx_dihed_score_2( ++i ) = 1.291;
	farlx_dihed_score_2( ++i ) = 2.222;
	farlx_dihed_score_2( ++i ) = 3.170;
	farlx_dihed_score_2( ++i ) = 4.104;
	farlx_dihed_score_2( ++i ) = 4.995;
	farlx_dihed_score_2( ++i ) = 5.823;
	farlx_dihed_score_2( ++i ) = 6.571;
	farlx_dihed_score_2( ++i ) = 7.231;
	farlx_dihed_score_2( ++i ) = 7.798;
	farlx_dihed_score_2( ++i ) = 8.275;
	farlx_dihed_score_2( ++i ) = 8.664;
	farlx_dihed_score_2( ++i ) = 8.973;
	farlx_dihed_score_2( ++i ) = 9.210;
	farlx_dihed_score_2( ++i ) = 9.383;
	farlx_dihed_score_2( ++i ) = 9.501;
	farlx_dihed_score_2( ++i ) = 9.569;
	farlx_dihed_score_2( ++i ) = 9.593;
	farlx_dihed_score_2( ++i ) = 9.576;
	farlx_dihed_score_2( ++i ) = 9.521;
	farlx_dihed_score_2( ++i ) = 9.427;
	farlx_dihed_score_2( ++i ) = 9.293;
	farlx_dihed_score_2( ++i ) = 9.117;
	farlx_dihed_score_2( ++i ) = 8.895;
	farlx_dihed_score_2( ++i ) = 8.625;
	farlx_dihed_score_2( ++i ) = 8.302;
	farlx_dihed_score_2( ++i ) = 7.924;
	farlx_dihed_score_2( ++i ) = 7.487;
	farlx_dihed_score_2( ++i ) = 6.991;
	farlx_dihed_score_2( ++i ) = 6.437;
	farlx_dihed_score_2( ++i ) = 5.828;
	farlx_dihed_score_2( ++i ) = 5.168;
	farlx_dihed_score_2( ++i ) = 4.468;
	farlx_dihed_score_2( ++i ) = 3.736;
	farlx_dihed_score_2( ++i ) = 2.987;
	farlx_dihed_score_2( ++i ) = 2.237;
	farlx_dihed_score_2( ++i ) = 1.502;
	farlx_dihed_score_2( ++i ) = 0.802;
	farlx_dihed_score_2( ++i ) = 0.155;
	farlx_dihed_score_2( ++i ) = -0.419;
	farlx_dihed_score_2( ++i ) = -0.904;
	farlx_dihed_score_2( ++i ) = -1.283;
	farlx_dihed_score_2( ++i ) = -1.545;
	farlx_dihed_score_2( ++i ) = -1.681;
	farlx_dihed_score_2( ++i ) = -1.686;
	farlx_dihed_score_2( ++i ) = -1.560;
	farlx_dihed_score_2( ++i ) = -1.308;
	farlx_dihed_score_2( ++i ) = -0.938;
	farlx_dihed_score_2( ++i ) = -0.461;
	farlx_dihed_score_2( ++i ) = 0.107;
	farlx_dihed_score_2( ++i ) = 0.748;
	farlx_dihed_score_2( ++i ) = 1.445;
	farlx_dihed_score_2( ++i ) = 2.178;
	farlx_dihed_score_2( ++i ) = 2.928;
	farlx_dihed_score_2( ++i ) = 3.678;
	farlx_dihed_score_2( ++i ) = 4.411;
	farlx_dihed_score_2( ++i ) = 5.115;
	farlx_dihed_score_2( ++i ) = 5.778;
	farlx_dihed_score_2( ++i ) = 6.391;
	farlx_dihed_score_2( ++i ) = 6.950;
	farlx_dihed_score_2( ++i ) = 7.452;
	farlx_dihed_score_2( ++i ) = 7.894;
	farlx_dihed_score_2( ++i ) = 8.279;
	farlx_dihed_score_2( ++i ) = 8.610;
	farlx_dihed_score_2( ++i ) = 8.890;
	farlx_dihed_score_2( ++i ) = 9.123;
	farlx_dihed_score_2( ++i ) = 9.315;
	farlx_dihed_score_2( ++i ) = 9.471;
	farlx_dihed_score_2( ++i ) = 9.596;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin farlx_ssdist_sc_interp300_initializer
///
/// @brief
/// This is used to initialize two instances of the array
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Disulfides::farlx_ssdist_sc_interp300_initializer( FArray1D_float & farlx_ssdist_sc_interp300 )
{
	int i = 0;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.006;
	farlx_ssdist_sc_interp300( ++i ) = 3.006;
	farlx_ssdist_sc_interp300( ++i ) = 3.005;
	farlx_ssdist_sc_interp300( ++i ) = 3.004;
	farlx_ssdist_sc_interp300( ++i ) = 3.003;
	farlx_ssdist_sc_interp300( ++i ) = 3.003;
	farlx_ssdist_sc_interp300( ++i ) = 3.001;
	farlx_ssdist_sc_interp300( ++i ) = 3.000;
	farlx_ssdist_sc_interp300( ++i ) = 2.999;
	farlx_ssdist_sc_interp300( ++i ) = 2.998;
	farlx_ssdist_sc_interp300( ++i ) = 2.996;
	farlx_ssdist_sc_interp300( ++i ) = 2.994;
	farlx_ssdist_sc_interp300( ++i ) = 2.992;
	farlx_ssdist_sc_interp300( ++i ) = 2.990;
	farlx_ssdist_sc_interp300( ++i ) = 2.987;
	farlx_ssdist_sc_interp300( ++i ) = 2.985;
	farlx_ssdist_sc_interp300( ++i ) = 2.981;
	farlx_ssdist_sc_interp300( ++i ) = 2.978;
	farlx_ssdist_sc_interp300( ++i ) = 2.974;
	farlx_ssdist_sc_interp300( ++i ) = 2.970;
	farlx_ssdist_sc_interp300( ++i ) = 2.966;
	farlx_ssdist_sc_interp300( ++i ) = 2.961;
	farlx_ssdist_sc_interp300( ++i ) = 2.955;
	farlx_ssdist_sc_interp300( ++i ) = 2.949;
	farlx_ssdist_sc_interp300( ++i ) = 2.943;
	farlx_ssdist_sc_interp300( ++i ) = 2.935;
	farlx_ssdist_sc_interp300( ++i ) = 2.927;
	farlx_ssdist_sc_interp300( ++i ) = 2.917;
	farlx_ssdist_sc_interp300( ++i ) = 2.907;
	farlx_ssdist_sc_interp300( ++i ) = 2.895;
	farlx_ssdist_sc_interp300( ++i ) = 2.881;
	farlx_ssdist_sc_interp300( ++i ) = 2.866;
	farlx_ssdist_sc_interp300( ++i ) = 2.848;
	farlx_ssdist_sc_interp300( ++i ) = 2.827;
	farlx_ssdist_sc_interp300( ++i ) = 2.803;
	farlx_ssdist_sc_interp300( ++i ) = 2.776;
	farlx_ssdist_sc_interp300( ++i ) = 2.743;
	farlx_ssdist_sc_interp300( ++i ) = 2.705;
	farlx_ssdist_sc_interp300( ++i ) = 2.661;
	farlx_ssdist_sc_interp300( ++i ) = 2.610;
	farlx_ssdist_sc_interp300( ++i ) = 2.550;
	farlx_ssdist_sc_interp300( ++i ) = 2.480;
	farlx_ssdist_sc_interp300( ++i ) = 2.400;
	farlx_ssdist_sc_interp300( ++i ) = 2.307;
	farlx_ssdist_sc_interp300( ++i ) = 2.201;
	farlx_ssdist_sc_interp300( ++i ) = 2.080;
	farlx_ssdist_sc_interp300( ++i ) = 1.942;
	farlx_ssdist_sc_interp300( ++i ) = 1.788;
	farlx_ssdist_sc_interp300( ++i ) = 1.615;
	farlx_ssdist_sc_interp300( ++i ) = 1.423;
	farlx_ssdist_sc_interp300( ++i ) = 1.212;
	farlx_ssdist_sc_interp300( ++i ) = 0.981;
	farlx_ssdist_sc_interp300( ++i ) = 0.732;
	farlx_ssdist_sc_interp300( ++i ) = 0.464;
	farlx_ssdist_sc_interp300( ++i ) = 0.180;
	farlx_ssdist_sc_interp300( ++i ) = -0.118;
	farlx_ssdist_sc_interp300( ++i ) = -0.429;
	farlx_ssdist_sc_interp300( ++i ) = -0.749;
	farlx_ssdist_sc_interp300( ++i ) = -1.074;
	farlx_ssdist_sc_interp300( ++i ) = -1.399;
	farlx_ssdist_sc_interp300( ++i ) = -1.722;
	farlx_ssdist_sc_interp300( ++i ) = -2.036;
	farlx_ssdist_sc_interp300( ++i ) = -2.337;
	farlx_ssdist_sc_interp300( ++i ) = -2.619;
	farlx_ssdist_sc_interp300( ++i ) = -2.878;
	farlx_ssdist_sc_interp300( ++i ) = -3.109;
	farlx_ssdist_sc_interp300( ++i ) = -3.308;
	farlx_ssdist_sc_interp300( ++i ) = -3.470;
	farlx_ssdist_sc_interp300( ++i ) = -3.594;
	farlx_ssdist_sc_interp300( ++i ) = -3.676;
	farlx_ssdist_sc_interp300( ++i ) = -3.715;
	farlx_ssdist_sc_interp300( ++i ) = -3.711;
	farlx_ssdist_sc_interp300( ++i ) = -3.664;
	farlx_ssdist_sc_interp300( ++i ) = -3.576;
	farlx_ssdist_sc_interp300( ++i ) = -3.448;
	farlx_ssdist_sc_interp300( ++i ) = -3.283;
	farlx_ssdist_sc_interp300( ++i ) = -3.086;
	farlx_ssdist_sc_interp300( ++i ) = -2.859;
	farlx_ssdist_sc_interp300( ++i ) = -2.609;
	farlx_ssdist_sc_interp300( ++i ) = -2.340;
	farlx_ssdist_sc_interp300( ++i ) = -2.056;
	farlx_ssdist_sc_interp300( ++i ) = -1.763;
	farlx_ssdist_sc_interp300( ++i ) = -1.465;
	farlx_ssdist_sc_interp300( ++i ) = -1.168;
	farlx_ssdist_sc_interp300( ++i ) = -0.874;
	farlx_ssdist_sc_interp300( ++i ) = -0.588;
	farlx_ssdist_sc_interp300( ++i ) = -0.312;
	farlx_ssdist_sc_interp300( ++i ) = -0.050;
	farlx_ssdist_sc_interp300( ++i ) = 0.198;
	farlx_ssdist_sc_interp300( ++i ) = 0.428;
	farlx_ssdist_sc_interp300( ++i ) = 0.642;
	farlx_ssdist_sc_interp300( ++i ) = 0.837;
	farlx_ssdist_sc_interp300( ++i ) = 1.015;
	farlx_ssdist_sc_interp300( ++i ) = 1.175;
	farlx_ssdist_sc_interp300( ++i ) = 1.319;
	farlx_ssdist_sc_interp300( ++i ) = 1.447;
	farlx_ssdist_sc_interp300( ++i ) = 1.560;
	farlx_ssdist_sc_interp300( ++i ) = 1.660;
	farlx_ssdist_sc_interp300( ++i ) = 1.748;
	farlx_ssdist_sc_interp300( ++i ) = 1.825;
	farlx_ssdist_sc_interp300( ++i ) = 1.893;
	farlx_ssdist_sc_interp300( ++i ) = 1.952;
	farlx_ssdist_sc_interp300( ++i ) = 2.004;
	farlx_ssdist_sc_interp300( ++i ) = 2.051;
	farlx_ssdist_sc_interp300( ++i ) = 2.092;
	farlx_ssdist_sc_interp300( ++i ) = 2.129;
	farlx_ssdist_sc_interp300( ++i ) = 2.162;
	farlx_ssdist_sc_interp300( ++i ) = 2.193;
	farlx_ssdist_sc_interp300( ++i ) = 2.221;
	farlx_ssdist_sc_interp300( ++i ) = 2.248;
	farlx_ssdist_sc_interp300( ++i ) = 2.273;
	farlx_ssdist_sc_interp300( ++i ) = 2.297;
	farlx_ssdist_sc_interp300( ++i ) = 2.320;
	farlx_ssdist_sc_interp300( ++i ) = 2.342;
	farlx_ssdist_sc_interp300( ++i ) = 2.364;
	farlx_ssdist_sc_interp300( ++i ) = 2.385;
	farlx_ssdist_sc_interp300( ++i ) = 2.406;
	farlx_ssdist_sc_interp300( ++i ) = 2.427;
	farlx_ssdist_sc_interp300( ++i ) = 2.447;
	farlx_ssdist_sc_interp300( ++i ) = 2.467;
	farlx_ssdist_sc_interp300( ++i ) = 2.487;
	farlx_ssdist_sc_interp300( ++i ) = 2.506;
	farlx_ssdist_sc_interp300( ++i ) = 2.525;
	farlx_ssdist_sc_interp300( ++i ) = 2.544;
	farlx_ssdist_sc_interp300( ++i ) = 2.562;
	farlx_ssdist_sc_interp300( ++i ) = 2.580;
	farlx_ssdist_sc_interp300( ++i ) = 2.598;
	farlx_ssdist_sc_interp300( ++i ) = 2.615;
	farlx_ssdist_sc_interp300( ++i ) = 2.632;
	farlx_ssdist_sc_interp300( ++i ) = 2.649;
	farlx_ssdist_sc_interp300( ++i ) = 2.665;
	farlx_ssdist_sc_interp300( ++i ) = 2.681;
	farlx_ssdist_sc_interp300( ++i ) = 2.696;
	farlx_ssdist_sc_interp300( ++i ) = 2.711;
	farlx_ssdist_sc_interp300( ++i ) = 2.725;
	farlx_ssdist_sc_interp300( ++i ) = 2.739;
	farlx_ssdist_sc_interp300( ++i ) = 2.753;
	farlx_ssdist_sc_interp300( ++i ) = 2.766;
	farlx_ssdist_sc_interp300( ++i ) = 2.778;
	farlx_ssdist_sc_interp300( ++i ) = 2.790;
	farlx_ssdist_sc_interp300( ++i ) = 2.802;
	farlx_ssdist_sc_interp300( ++i ) = 2.813;
	farlx_ssdist_sc_interp300( ++i ) = 2.824;
	farlx_ssdist_sc_interp300( ++i ) = 2.834;
	farlx_ssdist_sc_interp300( ++i ) = 2.844;
	farlx_ssdist_sc_interp300( ++i ) = 2.853;
	farlx_ssdist_sc_interp300( ++i ) = 2.862;
	farlx_ssdist_sc_interp300( ++i ) = 2.871;
	farlx_ssdist_sc_interp300( ++i ) = 2.879;
	farlx_ssdist_sc_interp300( ++i ) = 2.887;
	farlx_ssdist_sc_interp300( ++i ) = 2.895;
	farlx_ssdist_sc_interp300( ++i ) = 2.902;
	farlx_ssdist_sc_interp300( ++i ) = 2.908;
	farlx_ssdist_sc_interp300( ++i ) = 2.915;
	farlx_ssdist_sc_interp300( ++i ) = 2.921;
	farlx_ssdist_sc_interp300( ++i ) = 2.926;
	farlx_ssdist_sc_interp300( ++i ) = 2.932;
	farlx_ssdist_sc_interp300( ++i ) = 2.937;
	farlx_ssdist_sc_interp300( ++i ) = 2.942;
	farlx_ssdist_sc_interp300( ++i ) = 2.946;
	farlx_ssdist_sc_interp300( ++i ) = 2.951;
	farlx_ssdist_sc_interp300( ++i ) = 2.955;
	farlx_ssdist_sc_interp300( ++i ) = 2.959;
	farlx_ssdist_sc_interp300( ++i ) = 2.962;
	farlx_ssdist_sc_interp300( ++i ) = 2.965;
	farlx_ssdist_sc_interp300( ++i ) = 2.969;
	farlx_ssdist_sc_interp300( ++i ) = 2.972;
	farlx_ssdist_sc_interp300( ++i ) = 2.974;
	farlx_ssdist_sc_interp300( ++i ) = 2.977;
	farlx_ssdist_sc_interp300( ++i ) = 2.979;
	farlx_ssdist_sc_interp300( ++i ) = 2.982;
	farlx_ssdist_sc_interp300( ++i ) = 2.984;
	farlx_ssdist_sc_interp300( ++i ) = 2.986;
	farlx_ssdist_sc_interp300( ++i ) = 2.988;
	farlx_ssdist_sc_interp300( ++i ) = 2.989;
	farlx_ssdist_sc_interp300( ++i ) = 2.991;
	farlx_ssdist_sc_interp300( ++i ) = 2.992;
	farlx_ssdist_sc_interp300( ++i ) = 2.994;
	farlx_ssdist_sc_interp300( ++i ) = 2.995;
	farlx_ssdist_sc_interp300( ++i ) = 2.996;
	farlx_ssdist_sc_interp300( ++i ) = 2.997;
	farlx_ssdist_sc_interp300( ++i ) = 2.998;
	farlx_ssdist_sc_interp300( ++i ) = 2.999;
	farlx_ssdist_sc_interp300( ++i ) = 3.000;
	farlx_ssdist_sc_interp300( ++i ) = 3.001;
	farlx_ssdist_sc_interp300( ++i ) = 3.002;
	farlx_ssdist_sc_interp300( ++i ) = 3.002;
	farlx_ssdist_sc_interp300( ++i ) = 3.003;
	farlx_ssdist_sc_interp300( ++i ) = 3.003;
	farlx_ssdist_sc_interp300( ++i ) = 3.004;
	farlx_ssdist_sc_interp300( ++i ) = 3.005;
	farlx_ssdist_sc_interp300( ++i ) = 3.005;
	farlx_ssdist_sc_interp300( ++i ) = 3.005;
	farlx_ssdist_sc_interp300( ++i ) = 3.006;
	farlx_ssdist_sc_interp300( ++i ) = 3.006;
	farlx_ssdist_sc_interp300( ++i ) = 3.006;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.007;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.008;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.009;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
	farlx_ssdist_sc_interp300( ++i ) = 3.010;
}


void
Disulfides::reorder_cys_initializer( FArray1D_int & reorder_cys )
{
	int i = 0;
	reorder_cys( ++i ) = 1;
	reorder_cys( ++i ) = 2;
	reorder_cys( ++i ) = 4;
	reorder_cys( ++i ) = 5;
	reorder_cys( ++i ) = 3;
	reorder_cys( ++i ) = 7;
	reorder_cys( ++i ) = 6;
}




////////////////////////////////////////////////////////////////////////////////
/// @begin gauss
///
/// @brief
///     returns value of Gauss function at x-val given params
///
/// @detailed
///
/// @param[in]   x_val - in - x-val where Gauss is evaluated
/// @param[in]   base - in - y-value offset
/// @param[in]   amp - in - y-amplitude
/// @param[in]   mean - in - mean of Gauss
/// @param[in]   width - in - width of Gauss
///
/// @return
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::gauss(
	float const x, // x value at which gaussian will be evaluated
	float const base,
	float const amp,
	float const mean,
	float const width
)
{
	float const ratio = ( x - mean ) / width;
	return base + amp * std::exp( -( ratio * ratio ) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dbl_gauss
///
/// @brief
///     returns the value for 2 superimposed gaussians at x=x_val
/// @detailed
///     param explanations same as in function gauss above
/// @param[in]   x - in - value at which gaussian will be evaluated
/// @param[in]   base - in -
/// @param[in]   amp1 - in -
/// @param[in]   amp2 - in -
/// @param[in]   mean1 - in -
/// @param[in]   mean2 - in -
/// @param[in]   width1 - in -
/// @param[in]   width2 - in -
///
/// @return
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
Disulfides::dbl_gauss(
	float const x, // x value at which gaussian will be evaluated
	float const base,
	float const amp1,
	float const amp2,
	float const mean1,
	float const mean2,
	float const width1,
	float const width2
)
{
	float const ratio1 = ( x - mean1 ) / width1;
	float const ratio2 = ( x - mean2 ) / width2;
	return base +
	 amp1 * std::exp( -( ratio1 * ratio1 ) ) +
	 amp2 * std::exp( -( ratio2 * ratio2 ) );
}

//bs****************************************************************************
//bs returns the value for 3 superimposed gaussians

float
Disulfides::triple_gauss(
	float const x, // x value at which gaussian will be evaluated
	float const base,
	float const amp1,
	float const amp2,
	float const amp3,
	float const mean1,
	float const mean2,
	float const mean3,
	float const width1,
	float const width2,
	float const width3
)
{
	float const ratio1 = ( x - mean1 ) / width1;
	float const ratio2 = ( x - mean2 ) / width2;
	float const ratio3 = ( x - mean3 ) / width3;
	return base +
	 amp1 * std::exp( -( ratio1 * ratio1 ) ) +
	 amp2 * std::exp( -( ratio2 * ratio2 ) ) +
	 amp3 * std::exp( -( ratio3 * ratio3 ) );
}
//bs****************************************************************************
//bs returns d(triple_gauss)/dx

float
Disulfides::d_triple_gauss_d_x(
	float const x, // x value at which gaussian will be evaluated
	float const /* base */, // unused
	float const amp1,
	float const amp2,
	float const amp3,
	float const mean1,
	float const mean2,
	float const mean3,
	float const width1,
	float const width2,
	float const width3
)
{
	float const ratio1 = ( x - mean1 ) / width1;
	float const ratio2 = ( x - mean2 ) / width2;
	float const ratio3 = ( x - mean3 ) / width3;
	return -2 * (
	 ((x-mean1)/( width1 * width1 ))*amp1*std::exp(-( ratio1 * ratio1 )) +
	 ((x-mean2)/( width2 * width2 ))*amp2*std::exp(-( ratio2 * ratio2 )) +
	 ((x-mean3)/( width3 * width3 ))*amp3*std::exp(-( ratio3 * ratio3 )));
}

//bs****************************************************************************
//bs Gaussian peak in two-dimensions

float
Disulfides::gauss2D(
	float const x,
	float const y,
	float const z0, // offset = value of 2D gauss at (x,y) = (0,0)
	float const A, // amplitude
	float const x0,
	float const xwidth,
	float const y_0,
	float const ywidth,
	float const cor // cross-correlation term
)
{
	if ( xwidth == 0.0 || ywidth == 0.0 ) {
		std::cout << "xwidth or ywidth = 0" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	float const xratio = ( x - x0 ) / xwidth;
	float const yratio = ( y - y_0 ) / ywidth;
	float const gauss2D = z0 + A * std::exp(-1.0f/(2.0f*(1.0f-( cor * cor )))*
	 ( ( xratio * xratio ) + ( yratio * yratio ) -
	 2.0f*cor*(x-x0)*(y-y_0)/( xwidth * ywidth ) ) );

//$$$ std::cout << "gauss" << F( 11, 3, x ) << ' ' <<
//$$$	 F( 11, 3, y ) << ' ' << F( 11, 3, gauss2D ) << std::endl;

	return gauss2D;
}
//bs****************************************************************************

//bs Derivative of gauss2D w.r.t x-dimension.

float
Disulfides::d_gauss2D_d_x(
	float const x,
	float const y,
	float const z0, // offset = value of 2D gauss at (x,y) = (0,0)
	float const A, // amplitude
	float const x0,
	float const xwidth,
	float const y_0,
	float const ywidth,
	float const cor // cross-correlation term
)
{
//$$$	float const arg = -1.0/(2.0*(1.0-( cor * cor )))*
//$$$	 ( square( (x-x0)/xwidth ) + square( (y-y_0)/ywidth )
//$$$	 - 2.0*cor*(x-x0)*(y-y_0)/(xwidth*ywidth) );

	float const z = gauss2D(x,y,z0,A,x0,xwidth,y_0,ywidth,cor);
	float const d_arg_d_x = -1.0f/(1.0f-( cor * cor ))*
	 ( (x-x0)/( xwidth * xwidth ) - cor*(y-y_0)/( xwidth * ywidth ) );
	 // d(arg)/d(x), where arg = argument of the exp() term in gauss2D

	float const d_gauss2D_d_x = ( z - z0 ) * d_arg_d_x; // Return value

//bs   std::cout << "deriv" << F( 6, 1, x ) << ' ' <<
//bs	  F( 6, 1, y ) << ' ' << F( 6, 1, z ) << ' ' << F( 6, 1, z0 ) << ' ' <<
//bs	  F( 6, 1, d_arg_d_x ) << ' ' << F( 6, 1, d_gauss2D_d_x ) << std::endl;
	return d_gauss2D_d_x;
}



//bs****************************************************************************
//bs returns d(double_gauss)/dx

float
Disulfides::d_double_gauss_d_x(
	float const x, // x value at which gaussian will be evaluated
	float const /* base */,
	float const amp1,
	float const amp2,
	float const mean1,
	float const mean2,
	float const width1,
	float const width2
)
{
	return -2 * (
	 ( ( x - mean1 ) / square( width1 ) ) * amp1 *
	 std::exp( -( square( ( x - mean1 ) / width1 ) ) ) +
	 ( ( x - mean2 ) / square( width2 ) ) * amp2 *
	 std::exp( -( square( ( x - mean2 ) / width2 ) ) ) );
}


//------------------------------------------------------------------------------

//bs returns true if an input integer is even (or 0), false if odd
//bs gives nonsense results if the input is a real number!!!

////////////////////////////////////////////////////////////////////////////////
/// @begin even_integer
///
/// @brief
///
/// @detailed
///bs returns true if an input integer is even (or 0), false if odd
///bs gives nonsense results if the input is a real number!!!
///
/// @param[in]   num - in - must be integer
///
/// @return
///
/// @global_read
///     none
/// @global_write
///     none
/// @remarks
///
/// @references
///
/// @authors
///     bs
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Disulfides::even_integer( int & num )
{
	bool even_integer = false; // Return value

	float testnum = num/2.0 - static_cast< int >(num/2.0);
//bs	std::cout << "num,testnum = " << num << ' ' << testnum << std::endl;
//bs	std::cout << "num/2.0, int(num/2.0)" << SS( num/2.0 ) << ' ' <<
//bs	 static_cast< int >(num/2.0) << std::endl;
	if ( testnum == 0.0 ) even_integer = true;
//bs     std::cout << "even_integer = " << even_integer << std::endl;
	return even_integer;
}


//////////////////////////////////////////////
/// Private: Methods, Class Initialization ///
//////////////////////////////////////////////

void
Disulfides::initialize()
{
	using namespace disulfides::constants;

	// connectivity
	disulfide_pairs = FArray2D_int(max_cysteines, max_cysteines);
	disulf_partner_a = FArray1D_int(max_disulfides);
	disulf_partner_b = FArray1D_int(max_disulfides);

	// counts
	cys = FArray1D_int(max_cysteines);
	track_disulf_cen_cb = FArray2D_int(2, max_disulfides);
	cb_scale_factor = FArray1D_float(max_disulfides);

	// The following call to evaluate_disulfides_centroid()
	// ensure that cysteines are found correctly during initialization.
	// It does a lot of extra unnecessary stuff, but keep it in for
	// now until we can be sure that extracting calls to just identify/find/count
	// disulfides works properly.
	evaluate_disulfides_centroid();
}



//------------------------------------------------------------------------------
//bs returns the total weighted additions to the scorefxn total score due to disulifides
//bs actually, doesn't account for zeroing out the pair_log, LJ, solv terms.

//$$$  float get_fa_disulf_tot_weighted_sc()
//$$$	 {
//$$$    using namespace param;
//$$$    using namespace score;
//$$$    using namespace disulfides;
//$$$
//$$$    float get_fa_disulf_tot_weighted_sc =
//$$$     disulfide_distance_score*
//$$$     (disulfide_distance_weight+fa_pair_weight)+
//$$$     disulfide_csangles_score*
//$$$     (disulfide_csangles_weight+fa_pair_weight)+
//$$$     disulfide_dihedral_score*
//$$$     (disulfide_dihedral_weight+fa_pair_weight);
//$$$
//$$$    return get_fa_disulf_tot_weighted_sc;
//$$$  }


//------------------------------------------------------------------------------
//$$$cbs returns total weighted fullatom disulfide score for one disulf bond
//$$$cbs (same quantity returned by get_tot_weighted_sc_this_disulf_in_pack)
//$$$cbs but also update the global fullatom disulfide scores
//$$$cbs This function is called by fullatom_energy.
//$$$
//$$$      void getTotWtdScThisDisulf_UpdGlbls(
//$$$                       cys1,cys1coord,cys2,cys2coord,
//$$$                       tot_weighted_sc_this_disulf)
//$$$
//$$$
//$$$      implicit none
//$$$      include 'param.h'
//$$$      include 'param_pack.h'       // disulf wts for pack & fullatom_energy declared here
//$$$cbs      include 'disulfides.h'  // global disulf fa scores
//$$$
//$$$cbs input
//$$$      int cys1,cys2                                  // residue numbers of 2 cys residues
//$$$      float cys1coord(3,MAX_ATOM()),cys2coord(3,MAX_ATOM())   // fullatom coords of 2 cys
//$$$
//$$$cbs output
//$$$      float tot_weighted_sc_this_disulf     // total weighted score using packer weights (not scorefxn weights)
//$$$
//$$$cbs local
//$$$      float dist_sc_this_disulf,csang_sc_this_disulf
//$$$      float dihedral_sc_this_disulf,ca_dihedral_sc_this_disulf
//$$$      float cbs_sc_this_disulf
//$$$      bool truefalse_fa_disulf
//$$$
//$$$
//$$$                  score_this_disulfide_fa(cys1coord,cys2coord,
//$$$                      dist_sc_this_disulf,
//$$$                      csang_sc_this_disulf,
//$$$                      dihedral_sc_this_disulf,
//$$$                      ca_dihedral_sc_this_disulf,
//$$$                      truefalse_fa_disulf);
//$$$
//$$$cbs get the Cb-S score
//$$$
//$$$                  get_cbs_sc_this_disulf(cys1coord,cys2coord,
//$$$                      cbs_sc_this_disulf);
//$$$
//$$$c$$$cbs debug
//$$$c$$$                  if ( cys1 == 29 && cys2 == 50 )
//$$$c$$$                   std::cout << "getTotWtd 29-50 " <<
//$$$c$$$                   F( 8, 1, dist_sc_this_disulf ) << ' ' << F( 8, 1, csang_sc_this_disulf ) << ' ' <<
//$$$c$$$                   F( 8, 1, dihedral_sc_this_disulf ) << std::endl;
//$$$
//$$$                  tot_weighted_sc_this_disulf =
//$$$                      dist_sc_this_disulf*Wdisulf_dist_sc+
//$$$                      csang_sc_this_disulf*Wdisulf_csang_sc+
//$$$                      dihedral_sc_this_disulf*Wdisulf_dihedral_sc+
//$$$                      ca_dihedral_sc_this_disulf*
//$$$                      Wdisulf_ca_dihedral_sc+
//$$$                      cbs_sc_this_disulf*Wdisulf_cbs_dist_sc;
//$$$
//$$$cbs update global fullatom disulf scores (disulfides.h)
//$$$
//$$$                  update_disulf_fa_scores(truefalse_fa_disulf,
//$$$                      dist_sc_this_disulf,
//$$$                      csang_sc_this_disulf,
//$$$                      dihedral_sc_this_disulf,
//$$$                      ca_dihedral_sc_this_disulf,
//$$$                      cbs_sc_this_disulf);
//$$$
//$$$        }

}

