// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10024 $
//  $Date: 2006-08-28 13:54:52 -0700 (Mon, 28 Aug 2006) $
//  $Author: ashworth $


// Rosetta Headers
#include "dna_am_min.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "are_they_neighbors.h"
#include "atom_is_backbone.h"
#include "design.h"
#include "DesignMap.h"
#include "dna.h"
#include "dna_classes.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "geometric_solvation.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "input_pdb.h"
#include "minimize.h"
#include "misc.h"
#include "orient_rms.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdb.h"
#include "read_paths.h"
#include "refold.h"
#include "random_numbers.h"
#include "rotamer_functions.h"
#include "score.h"
#include "runlevel.h"
#include "template_pack.h"
#include "util_basic.h"
#include "util_vector.h"
#include "water.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <vector>

// NOTE: these deprecated methods are being reimplemented using Pose kinematics by Phil Bradley.  Currently and in the future, no functions here are called from outside of this file.  It is best that the methods be extracted and rewritten elsewhere as necessary.

// Alex Morozov 11/29/04 morozov@edsb.rockefeller.edu
// Here are some explanations of how the new knowledge-based potential and the protein-dna
// interface minimizer are set up. Most changes are confined to dna.cc, dna.h, dna_ns.cc, dna_ns.h
// in Rosetta.
// =====================
// 1. DNA scoring
//   The quadratic knowledge-based potential is derived exactly as in W.K.Olson et al., PNAS 95, 11163-11168, 1998.
// The potential is a quadratic function of 6 effective geometric parameters representing basepairing and neighboring
// basepair stacking. Please consult the paper and refs in it for all further details.
// The parameterization is done outside Rosetta on a set of protein-dna complexes named New_DNA_Set provided by
// Jim Havranek (~100 complexes). The protocol is the same as in W.K.Olson et al. except the terminal basesteps are
// not excluded (some structures have multiple termini, ie 2 shorter strands paired with a longer one -
// how do you treat the "internal" nick, plus we exclude all data more that 3 stddev away anyway).
// Another change is that a similar potential was created for basepairs (W.K.Olson et al. only parameterized the basestep
// potential). The basepair potential is necessary for DNA conformational change (ie minimization), to keep reasonable basepair
// geometries. Following W.K.Olson et al., all basepairs and basesteps for which at least one of the 6 geometric parameters
// is outside 3 stddev are omitted (this has to be done interatively but converges in 2-3 steps).
// DNA scoring relies on the identification of basesteps and basepairs by Rosetta, which is done by global
// sequence complementarity for 2 strands (followed by a distance check), and by distance checks for 3 or more
// strands. Non-standard base pairs are not accepted; there is no RNA support in basepairing; the final list of basesteps and
// basepairs is printed out for manual inspection. "Overstretched" basepairs with a lot of deformation are
// skipped, the idea being that the DNA potential parameterized using much more "normal" data (with outliers excluded, see
// above) is unable to make any predictions in such extreme cases. The skipping is hardwired with some cutoffs, and thus basepairs
// and basesteps should always be manually expected in each case (but seem to be reasonable for my datasets).
// Rosetta output PDBs are modified as follows for protein-dna complexes (no change occurs in output if no dna was found):
// 2 extra scores are output: dna_bs is the basestep score, and dna_bp is the basepair score; there are corresponding
// extra columns in the residue breakdown; there are tables of effective DNA parameters (twist etc. and propeller etc.)
// and of dna dihedral angles (alpha etc.; 7 per base).
// =====================
// 2. Changes to resfile: in order to facilitate dna mutations, a new instruction was added to the design resfile:
// "PIKBA  X" (X=A,C,G,T) allows base mutations into X (note that unlike PIKAA, multiple substitutions are not
// allowed). Since one should mutate basepairs rather than individual bases, a
// -force_bp_muts T/F flag was added
// to enforce proper mutations of all bases complementary to user-provided bases to mutate. If this flag is false, the user
// is responsible for proper basepairing of all mutations specified through the resfile.
// NOTE: bases flagged with VARYB will NOT be basepaired under -force_bp_muts flag (ie this flag only affects
// PIKBA instructions).
// =====================
// 3. Protein-DNA interface minimization
//    The protein-dna interface minimization is done for several reasons: to get rid of clashes at the protein-dna interface;
// to introduce dna conformational change into Rosetta; to build better homology models via minimization of the
// region around newly substituted sidechains; to provide rigid body motion of the protein pieces (ie recognition
// helices) that are in contact with dna. We use powell for minimization since we do not know the derivatives of
// W.Olson's potential with respect to dihedral angles (and minimization is done in torsional space to keep bond lengths and
// bond angles fixed - standard procedure in Rosetta).
// The all atom dihedral->xyz map is provided by high level refold_prot/refold_dna
// functions built using mostly Carol's refold.cc stuff. WARNING: some of Carol's stuff is numerically unstable in the limit
// of many refolds (>2000-3000)!! I think I fixed the problem in this particular case, but if you see weird atomic drift
// (usually accompanied by minimization failure), let me know. A single powell move is then a change in one of the dihedral
// angles allowed to relax. The list of angles is defined by the list of "active" residues which are: anything labelled by
// PIKBA/VARYB in the resfile (these labels are valid for dna bases only!) plus their aa neighbors. Note that labeling aa's
// with NATAA,PIKAA etc. will NOT flag them for the minimizer, just for rotamer repack! Given a list of "active" residues,
// a list of angles is constructed based on one of the following modes: protein - 'none', 'all' - phi/psi/chi, 'phipsi', 'chi';
// dna - 'none', 'all' - alpha through xi (along phosphate backbone) + chi, 'bb' - alpha through xi only, 'chi'. Thus, minimization
// in ("chi","chi") mode will move only aa sidechains and dna bases, but both protein and dna backbones will stay fixed. Likewise,
// ("none","none") mode will cause minimization to be skipped, etc. The modes are input to map_interface_2_minimize in pack.cc, and are
// not currently set via any global flags. The minimizer setup functions also add aa's and bases with "protein-dna bumps" (as defined
// by LJrep) to the list of "active" residues. The protein-dna bump check operates under the same modes passed from map_interface_2_minimize.
//    Rotamer repack can precede or follow minimization (its order is defined by where the high level functions
// are in pack_rotamers, pack.cc), but I have not played with that yet (AM 11/04).
// Scoring happens in the xyz space (function compute_energy) and currently involves dna basestep + basepair potentials
// as well as usual Rosetta scores (LJrep, LJatr, LKsol, HBsc_sc etc.; GB is not implemented since one'd have to recompute all Born radii
// at each step). This scoring excludes interactions within the fixed region (upstream of the first residue to minimize), and thus is not
// equal to the final output in the PDB file (but the res-res level functions for energy computations are of course the same as those
// used by other people).


//////////////////////////////////////////////////////////////////////////////
/// @begin map_interface_2_minimize
///
/// @brief
/// This function sets res2minim & ang2minim residue and dihedral angle
/// selectors given info about protein-dna binding interface and user input
/// (specifically, which dna bases have been labeled with VARYB/PIKBA flags).
/// The minimization selectors are then used as input by many minimizer-related
/// routines.
/// @detailed
/// Dihedral angles are set to active according to one of the following modes:
/// I. protein: 'none'; 'all' - phi/psi/chi; 'phipsi'; 'chi'
/// II. dna: 'none'; 'all' - alpha through xi (along phosphate backbone) + chi; 'bb' - alpha through xi only; 'chi'
/// @param   res1 - [in/out]? -
/// @param   res2 - [in/out]? -
/// @param   xyz1 - [in/out]? -
/// @param   xyz2 - [in/out]? -
/// @param   neighbor - [in/out]? -
/// @param   dis2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
map_interface_2_minimize(
	int const nres,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	std::string const & prot_mode,
	std::string const & dna_mode,
	DesignMap & design_map
)
{
	using namespace dna_variables;
	using namespace aaproperties_pack;
	using namespace pdb;
	using namespace param;
	using namespace param_aa;

	aan.dimension( MAX_RES() );

	// Init.arrays:
	res2minim = false;
	ang2minim = false;

	// Check the modes:
	if ( prot_mode != "none" && prot_mode != "all" && prot_mode != "phipsi" && prot_mode != "chi" ) {
		std::cerr << "ERROR: unknown prot_mode: " << prot_mode << " in map_interface_2_minimize!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( dna_mode != "none" && dna_mode != "all" && dna_mode != "bb" && dna_mode != "chi" ) {
		std::cerr << "ERROR: unknown dna_mode: " << dna_mode << " in map_interface_2_minimize!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( prot_mode == "none" && dna_mode == "none" ) {
		std::cerr << "WARNING: no minimization of protein and dna requested!\n";
		return;
	}

	std::cout << "*************************************\n";
	std::cout << "*** Minimizer mode: prot=" << prot_mode << "; dna=" << dna_mode << std::endl;
	std::cout << "*************************************\n";

	// Set fixed_prot_bb_refold according to prot_mode:
	if ( prot_mode == "none" || prot_mode == "chi" ) {
		fixed_prot_bb_refold = true;
	} else {
		fixed_prot_bb_refold = false;
	}

	// Find which residues to vary:
	int base_cnt = 0;
	int aa_cnt = 0;
	//ja using std nomenclature: res1=protein, res2=dna
	for ( int res2 = 1; res2 <= nres; ++res2 ) {
		if ( !is_DNA(aan(res2)) ) continue; // want to look only at dna bases
		if ( dna_mode != "none" && !res2minim(res2) ) {
			++base_cnt;
			res2minim(res2) = true;
		}
		//ja find out which amino acids to minimize
		for ( int res1 = 1; res1 <= nres; ++res1 ) {
			if ( design_map.repack_residue(res1) ) {
				if ( prot_mode != "none" && !res2minim(res1) ) {
					++aa_cnt;
					res2minim(res1) = true;
				}
			}
		}
	}

	std::cout << "Found " << base_cnt << " dna base(s) and " << aa_cnt << " amino acid(s) to minimize ..\n";
	if ( base_cnt==0 && dna_mode != "none" ) { // found no dna bases to vary, even though we wanted to
		std::cerr << "WARNING: failed to find dna bases to minimize in dna_mode " << dna_mode << "!" << std::endl;
	}
	if ( base_cnt > 0 && aa_cnt == 0 && prot_mode != "none" ) { // found dna bases to vary, but no aa's, even though we wanted to
		std::cerr << "WARNING: failed to find neighboring aa's in prot_mode " << prot_mode
							<< " - dna_neighbor array may not have been set!!" << std::endl;
	}

	//Detect prot-dna bumps and add them to the minimizer:
	protdna_bump_detector( nres, aan, xyz, prot_mode, dna_mode );

	// Print out a final list of residues to be minimized:
	if (dna_debug == 0) {
		std::cout << "======== List of residues to be minimized ========\n";
		for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
			int aa = aan(seqpos);
			if ( !res2minim(seqpos) ) continue;
			std::cout << aa_name3(aa) << "-" << pdb_res_num(seqpos) << " (" << res_chain(seqpos) << ")\n";
		}
		std::cout << "==================================================\n";
	}

	// Find which angles to vary (mode-dependent); remove residues from res2minim if in current mode there are no variable angles
	int base_ang_cnt = 0;
	int aa_ang_cnt = 0;
	for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
		int aa = aan(seqpos);
		if ( !res2minim(seqpos) ) continue;
		int cur_base_ang_cnt = 0;
		int cur_aa_ang_cnt = 0;
		if ( is_protein(aa) || is_nonnatural(aa) ) {
			if ( prot_mode == "all" ) {
				for ( int j = 1, je = nchi(aa,1)+2; j <= je; ++j ) {
					ang2minim(j,seqpos) = true;
					++cur_aa_ang_cnt;
				}
			} else if ( prot_mode == "phipsi" ) {
				for ( int j = nchi(aa,1)+1, je = nchi(aa,1)+2; j <= je; ++j ) {
					ang2minim(j,seqpos) = true;
					++cur_aa_ang_cnt;
				}
			} else if ( prot_mode == "chi" ) {
				for ( int j = 1, je = nchi(aa,1); j <= je; ++j ) {
					ang2minim(j,seqpos) = true;
					++cur_aa_ang_cnt;
				}
			}
		} else { // dna
			if ( dna_mode == "all" ) {
				for ( int j = 1; j <= 7; ++j ) {
					ang2minim(j,seqpos) = true;
					++cur_base_ang_cnt;
				}
			} else if ( dna_mode == "bb" ) {
				for ( int j = 1; j <= 6; ++j ) {
					ang2minim(j,seqpos) = true;
					++cur_base_ang_cnt;
				}
			} else if ( dna_mode == "chi" ) {
				ang2minim(7,seqpos) = true;
				++cur_base_ang_cnt;
			}
		}
		if (cur_aa_ang_cnt==0 && cur_base_ang_cnt==0) { // no angles to vary!
			res2minim(seqpos)  = false;
		}
		aa_ang_cnt += cur_aa_ang_cnt;
		base_ang_cnt += cur_base_ang_cnt;
	}

	// Exclude ill-defined angles from the minimizer; angles below should stay
	// at their default values:
	// dna: alpha=0.0 if 1st in chain, epsilon/xi=0.0 if last in chain.
	// protein: phi=-90 if 1st in chain, psi=130 if last in chain.
	int domain_start,domain_finish;
	for ( int k = 1; k <= misc::total_domains; ++k ) {
		if (k==1) {
			domain_start = 1;
			domain_finish = misc::domain_end(k);
		} else {
			domain_start = misc::domain_end(k-1) + 1;
			domain_finish = misc::domain_end(k);
		}
		//debug
		//cout << "domain_start = " << domain_start << std::endl;
		//cout << "domain_finish = " << domain_finish << std::endl;
		if (IsDnaDomain(domain_finish,aan) ) { // exclude ill-defined dna dihedral angles from minimization
			if ( ang2minim(1,domain_start) ) { // alpha
				ang2minim(1,domain_start) = false;
				--base_ang_cnt;
			}
			/*
			if ( ang2minim(4,domain_finish) ) { // delta: gives O3* which is the last atom in the dna chain and thus is free to rotate w/out affecting
                                          // the intra-dna energy (and probably hardly affecting prot-dna energies in many cases)
				ang2minim(4,domain_finish) = false;
				--base_ang_cnt;
			}
			*/
			if ( ang2minim(5,domain_finish) ) { // epsilon
				ang2minim(5,domain_finish) = false;
				--base_ang_cnt;
			}
			if ( ang2minim(6,domain_finish) ) { // xi
				ang2minim(6,domain_finish) = false;
				--base_ang_cnt;
			}
		} else { // exclude ill-defined protein dihedral angles from minimization
			int aa1 = aan(domain_start);
			int aa2 = aan(domain_finish);
			if ( is_protein(aa1) || is_nonnatural(aa1) ) {
				if ( ang2minim(nchi(aa1,1)+1,domain_start) ) { // phi
					ang2minim(nchi(aa1,1)+1,domain_start) = false;
					--aa_ang_cnt;
				}
			}
			if ( is_protein(aa2) || is_nonnatural(aa2) ) {
				if ( ang2minim(nchi(aa2,1)+2,domain_finish) ) { // psi
					ang2minim(nchi(aa2,1)+2,domain_finish) = false;
					--aa_ang_cnt;
				}
			}
		}
	}

	std::cout << "Found " << base_ang_cnt << " dna base angles and " << aa_ang_cnt << " aa angles to minimize ..\n";

}

//////////////////////////////////////////////////////////////////////////////
/// @begin load_dihedral_angles
///
/// @brief
/// for DNA - load (and print if dna_debug) backbone dihedral angles
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - dna_dihedrals(7, MAX_RES) in dna_variables namespace gets set from xyz
/// in  - everything else
/// @global_read
///
/// @global_write
///
/// @remarks
/// NOTE: all dna dihedrals are in degrees (as returned by dihedral(...) )
/// NOTE2: all dna dihedral angles and protein sidechain chi angles are computed here,
/// but protein backbone angles (phi,psi) computed by bk are used, because there is no
/// angle range issue which sometimes makes symmetric sidechain flip with bk's chi angle
/// assignments (this is not an issue with idealized rotamers of course).
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
load_dihedral_angles(
	int const nres,
	FArray1Da_int aan,
	FArray3Da_float xyz
)
{
	using namespace dna_variables;
	using namespace files_paths;
	using namespace misc;
	using namespace design;
	using namespace param;
	using namespace param_aa;
	using namespace pdb;
	using namespace aaproperties_pack;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	//Init. the dna_dihedrals array:
	dna_dihedrals = empty_value;

	// Init. the protein sidechain dihedrals array (cannot use bk
	// chiarray because it conforms to Dunbrack dih_angle
	// range conventions and thus sometimes flips symmetric sidechains
	// such as TYR & PHE: OK in idealized rotamers, not OK in
	// minimization of native sidechains).
	prot_chi_dihedrals = -1; // nonsensical value

	// Set dihedral angles for each dna chain separately:
	int domain_start = -1;
	int domain_finish = -1;
	//int domain_cnt = 0;

	for ( int k = 1; k <= total_domains; ++k ) {
		if (k==1) { // first domain is dna domain
			domain_start = 1;
			domain_finish = domain_end(k);
		} else {
			domain_start = domain_end(k-1) + 1;
			domain_finish = domain_end(k);
		}

		if (IsDnaDomain(domain_end(k),aan) ) { // set dihedral angles for this dna domain
			//debug
			//std::cout << "Domain " << domain_end(k) << " is dna!\n";
			//++domain_cnt;
			if (k==1) { // first domain is dna domain
				domain_start = 1;
				domain_finish = domain_end(k);
			} else {
				domain_start = domain_end(k-1) + 1;
				domain_finish = domain_end(k);
			}
			for ( int i = domain_start; i <= domain_finish; ++i ) {
				int aa = aan(i);
				if ( i == domain_start ) { // first 1 dihedral in each chain undefined
					dna_dihedrals(1,i) = 0.0; //alpha: O3*_{i-1} P O5* C5* [9 1 4 5]
				} else {
					//dna_dihedrals(1,i) = dihedral(xyz(1,o3star,i-1),xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i) ); //alpha: O3*_{i-1} P O5* C5* [9 1 4 5]
					dihedral_bk(xyz(1,o3star,i-1),xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i),dna_dihedrals(1,i) );
				}
				//dna_dihedrals(2,i) = dihedral(xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i) ); //beta: P O5* C5* C4* [1 4 5 6]
				dihedral_bk(xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i),dna_dihedrals(2,i) );
				//dna_dihedrals(3,i) = dihedral(xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i),xyz(1,c3star,i) ); //gamma: O5* C5* C4* C3* [4 5 6 8]
				dihedral_bk(xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i),xyz(1,c3star,i),dna_dihedrals(3,i) );
				//dna_dihedrals(4,i) = dihedral(xyz(1,c5star,i),xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i) ); //delta: C5* C4* C3* O3* [5 6 8 9]
				dihedral_bk(xyz(1,c5star,i),xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i),dna_dihedrals(4,i) );
				if ( i == domain_finish ) { // last 2 dihedrals in each chain undefined
					dna_dihedrals(5,i) = 0.0; //epsilon: C4* C3* O3* P_{i+1} [6 8 9 1]
					dna_dihedrals(6,i) = 0.0; //xi: C3* O3* P_{i+1} O5*_{i+1} [8 9 1 4]
				} else {
					//dna_dihedrals(5,i) = dihedral(xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1) ); //epsilon: C4* C3* O3* P_{i+1} [6 8 9 1]
					dihedral_bk(xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1),dna_dihedrals(5,i) );
					//dna_dihedrals(6,i) = dihedral(xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1),xyz(1,o5star,i+1) ); //xi: C3* O3* P_{i+1} O5*_{i+1} [8 9 1 4]
					dihedral_bk(xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1),xyz(1,o5star,i+1),dna_dihedrals(6,i) );
				}
				if (aa_name3(aa)=="  A") {
					//dna_dihedrals(7,i) = dihedral(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9A,i),xyz(1,c4A,i) ); //chi: O4* C1* N9 C4 for A
					dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9A,i),xyz(1,c4A,i),dna_dihedrals(7,i) );
				} else if (aa_name3(aa)=="  G") {
					//dna_dihedrals(7,i) = dihedral(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9G,i),xyz(1,c4G,i) ); //chi: O4* C1* N9 C4 for G
					dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9G,i),xyz(1,c4G,i),dna_dihedrals(7,i) );
				} else if (aa_name3(aa)=="  C") {
					//dna_dihedrals(7,i) = dihedral(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1C,i),xyz(1,c2,i) ); //chi: O4* C1* N1 C2 for C
					dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1C,i),xyz(1,c2,i),dna_dihedrals(7,i) );
				} else if (aa_name3(aa)=="  T") {
					//dna_dihedrals(7,i) = dihedral(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1T,i),xyz(1,c2,i) ); //chi: O4* C1* N1 C2 for T
					dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1T,i),xyz(1,c2,i),dna_dihedrals(7,i) );
				} else {
					std::cerr << "Unknown basepair name: ==" << aa_name3(aa) << "==!" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				if (dna_debug==1) { // print dihedrals if debugging
					std::cout << "res:" << i << ": alpha=" << dna_dihedrals(1,i) << " beta=" << dna_dihedrals(2,i) << " gamma=" << dna_dihedrals(3,i)
										<< " delta=" << dna_dihedrals(4,i) << " epsilon=" << dna_dihedrals(5,i) << " xi=" << dna_dihedrals(6,i) << " chi=" << dna_dihedrals(7,i) << std::endl;
				}
			}
		} else {
			for ( int seqpos = domain_start; seqpos <= domain_finish; ++seqpos ) {
				if ( is_DNA(aan(seqpos)) ) continue; // skip RNA & other non-protein/non-DNA domains
				int aa = aan(seqpos);
				int const aav = 1; // aav disabled
				for ( int chino = 1; chino <= MAX_CHI; ++chino ) {
					if ( nchi(aa,aav) >= chino ) {     // if has chino, measure it
						dihedral_bk(xyz(1,chi_atoms(1,chino,aa,aav),seqpos),
												xyz(1,chi_atoms(2,chino,aa,aav),seqpos),
												xyz(1,chi_atoms(3,chino,aa,aav),seqpos),
												xyz(1,chi_atoms(4,chino,aa,aav),seqpos),
												prot_chi_dihedrals(chino,seqpos) );
					}
				}
			}
		}
	}
	//if (domain_cnt != 2 ) {
	//	std::cerr << "PDB file must contain 2 dna strands!\n";
	//	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	//}

	if (dna_debug==1) {
		for ( int res1 = 1; res1 <= nres; ++res1 ) {
			int aa = aan( res1 );
			if ( is_NA(aa) ) {
				for ( int atom1 = 1, atom1e = natoms( aa, 1 ); atom1 <= atom1e; ++atom1 ) {
					std::cout << "res:" << res1 << "===" << atom_name( atom1, aa, 1 )
					 << "===" << nchi( aa, 1 ) << "===" << aa_name3( aa )
					 << "===" << res_chain( res1 ) << std::endl;
				}
			}
		}

		std::cout << "Number of residues:" << nres << std::endl;
		for ( int k = 1; k <= total_domains; ++k ) {
			std::cout << "Domain_end = " << domain_end(k) << " for chain " << res_chain(domain_end(k)) << std::endl;
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin refold_dna
///
/// @brief
/// for DNA - compute xyz given old xyz + new DNA dihedral angles (ie refold DNA)
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// in - begin_frag - start of refold region in the 5'-3' direction
/// in - end_frag - end of refold region in the 5'-3' direction
/// in -  xyz
/// out - refolded_xyz
///
/// @global_read
/// dna_variables stuff
/// @global_write
///
/// @remarks
/// note the cute use of atom pseudonames for increased readability
/// [begin_frag,end_frag] interval for refold is end-point inclusive
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_dna(
	int begin_frag,
	int end_frag,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	FArray3Da_float refolded_xyz
)
{
	using namespace param;
	using namespace param_aa;
	using namespace dna_variables;
	using namespace aaproperties_pack;
	using namespace misc;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	refolded_xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	int domain_start,domain_finish;
	FArray2D_float Mgl(4,4);

	//Check input parameters:
	if ( end_frag < begin_frag ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is not allowed in refold_dna!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//Out-of-range checks:
	if ((begin_frag<1 || begin_frag>MAX_RES()) ||
			(end_frag<1 || end_frag>MAX_RES()) ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is out of range in refold_dna!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//Prot-dna checks:
	if ( !is_DNA(aan(begin_frag)) || !is_DNA(aan(end_frag)) ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is not dna only in refold_dna!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Work only with dna domains; copy protein domains over:
	for ( int k = 1; k <= total_domains; ++k ) {
		if (k==1) { // first domain is dna domain
			domain_start = 1;
			domain_finish = domain_end(k);
		} else {
			domain_start = domain_end(k-1) + 1;
			domain_finish = domain_end(k);
		}
		//debug
		//cout << "domain_start = " << domain_start << std::endl;
		//cout << "domain_finish = " << domain_finish << std::endl;
		if (IsDnaDomain(domain_end(k),aan) ) { // reset xyz for this dna domain
			if (domain_finish < begin_frag || domain_start > end_frag) { // this whole domain simply needs to be copied (we do not work with this chain)
				for ( int res = domain_start; res <= domain_finish; ++res ) {
					int aa = aan(res);
					copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
				}
			} else if (domain_start <= begin_frag && domain_finish >= end_frag) { // fragment is inside this domain, copy->refold->rotate
				// copy all residues that are not going to move
				for ( int res=domain_start; res<begin_frag; ++res) {
					int aa = aan(res);
					copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
				}
				// copy the 4 'stub' atoms:
				copyXYZ(o1p,o1p,xyz(1,1,begin_frag),refolded_xyz(1,1,begin_frag) );
				copyXYZ(o2p,o2p,xyz(1,1,begin_frag),refolded_xyz(1,1,begin_frag) );
				copyXYZ(p,p,xyz(1,1,begin_frag),refolded_xyz(1,1,begin_frag) );
				copyXYZ(o5star,o5star,xyz(1,1,begin_frag),refolded_xyz(1,1,begin_frag) );
				// build region from dihedral angles: assume that O1P,P,O5* of the 1st residue exist and build from there
				for ( int res = begin_frag; res <= end_frag; ++res ) {
					int aa = aan(res);
					if (res==domain_start) { // c5star of the 1st base in chain cannot be built (alpha undef), just copy it
						copyXYZ(c5star,c5star,xyz(1,1,begin_frag),refolded_xyz(1,1,begin_frag) );
					} else {
						build_new_atom_dihedral (xyz(1,o3star,res-1), xyz(1,p,res), xyz(1,o5star,res), xyz(1,c5star,res),
																		 refolded_xyz(1,o3star,res-1), refolded_xyz(1,p,res), refolded_xyz(1,o5star,res), refolded_xyz(1,c5star,res),
																		 dna_dihedrals(1,res) ); // alpha
					}
					build_new_atom_dihedral (xyz(1,p,res), xyz(1,o5star,res), xyz(1,c5star,res), xyz(1,c4star,res),
																	 refolded_xyz(1,p,res), refolded_xyz(1,o5star,res), refolded_xyz(1,c5star,res), refolded_xyz(1,c4star,res),
																	 dna_dihedrals(2,res) ); // beta
					build_new_atom_dihedral (xyz(1,o5star,res), xyz(1,c5star,res), xyz(1,c4star,res), xyz(1,c3star,res),
																	 refolded_xyz(1,o5star,res), refolded_xyz(1,c5star,res), refolded_xyz(1,c4star,res), refolded_xyz(1,c3star,res),
																	 dna_dihedrals(3,res) ); // gamma
					build_new_atom_dihedral (xyz(1,c5star,res), xyz(1,c4star,res), xyz(1,c3star,res), xyz(1,o3star,res),
																	 refolded_xyz(1,c5star,res), refolded_xyz(1,c4star,res), refolded_xyz(1,c3star,res), refolded_xyz(1,o3star,res),
																	 dna_dihedrals(4,res) ); // delta

					if (res!=domain_finish) { // epsilon and xi undefined for last residue; no atoms to build
						build_new_atom_dihedral (xyz(1,c4star,res), xyz(1,c3star,res), xyz(1,o3star,res), xyz(1,p,res+1),
																		 refolded_xyz(1,c4star,res), refolded_xyz(1,c3star,res), refolded_xyz(1,o3star,res), refolded_xyz(1,p,res+1),
																		 dna_dihedrals(5,res) ); // eps
						build_new_atom_dihedral (xyz(1,c3star,res), xyz(1,o3star,res), xyz(1,p,res+1), xyz(1,o5star,res+1),
																		 refolded_xyz(1,c3star,res), refolded_xyz(1,o3star,res), refolded_xyz(1,p,res+1), refolded_xyz(1,o5star,res+1),
																		 dna_dihedrals(6,res) ); // xi
						// Rebuild O1P_{i+1} and O2P_{i+1}:
						get_GL_matrix(xyz(1,o3star,res), xyz(1,p,res+1), xyz(1,o5star,res+1),
													refolded_xyz(1,o3star,res), refolded_xyz(1,p,res+1), refolded_xyz(1,o5star,res+1), Mgl);
						GL_rot(Mgl, xyz(1,o1p,res+1), refolded_xyz(1,o1p,res+1) );
						GL_rot(Mgl, xyz(1,o2p,res+1), refolded_xyz(1,o2p,res+1) );
					}
					// Rebuild the sugar ring of residue i:
					get_GL_matrix(xyz(1,c5star,res), xyz(1,c4star,res), xyz(1,c3star,res),
												refolded_xyz(1,c5star,res), refolded_xyz(1,c4star,res), refolded_xyz(1,c3star,res), Mgl);
					for ( int atom = 1, atome = natoms( aa, 1 ); atom <= atome; ++atom ) {
						if ( atom_name( atom, aa, 1 )[3] == '*'
						 && ( atom != c4star && atom != c3star && atom != o3star && atom != o5star && atom != c5star) ) {
							GL_rot( Mgl, xyz( 1, atom, res ), refolded_xyz( 1, atom, res ) );
						}
					}
					// Rebuild the dna sidechain:
					int a1(0),a2(0);
					if (aa_name3(aa)=="  A") {
						a1 = n9A;
						a2 = c4A;
					} else if (aa_name3(aa)=="  G") {
						a1 = n9G;
						a2 = c4G;
					} else if (aa_name3(aa)=="  C") {
						a1 = n1C;
						a2 = c2;
					} else if (aa_name3(aa)=="  T") {
						a1 = n1T;
						a2 = c2;
					} else {
						std::cerr << "Unknown basepair name: ==" << aa_name3(aa) << "==!" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					GL_rot (Mgl, xyz(1,a1,res), refolded_xyz(1,a1,res) ); // Rebuild N1 (C,T) or N9 (A,G)
					build_new_atom_dihedral (xyz(1,o4star,res), xyz(1,c1star,res), xyz(1,a1,res), xyz(1,a2,res),
																	 refolded_xyz(1,o4star,res), refolded_xyz(1,c1star,res), refolded_xyz(1,a1,res), refolded_xyz(1,a2,res),
																	 dna_dihedrals(7,res) ); // chi

					get_GL_matrix(xyz(1,c1star,res), xyz(1,a1,res), xyz(1,a2,res),
												refolded_xyz(1,c1star,res), refolded_xyz(1,a1,res), refolded_xyz(1,a2,res), Mgl);
					// Rebuild the "sidechain": note that the methyl group in T (C5M + [1,2,3]H5M) is rebuilt rigidly, even though the hydrogens are free to rotate.
					// Something to fix in the future (maybe). am 11/04
					for ( int atom = 1, atome = natoms(aa,1); atom <= atome; ++atom ) {
						if ( ( atom_name(atom,aa,1)[3] == ' ' || atom_name(atom,aa,1)[3] == 'M' )
						 && (atom!=p && atom!=a1 && atom!=a2) ) {
							GL_rot(Mgl, xyz(1,atom,res), refolded_xyz(1,atom,res) );
						}
					}
					/*
					//insert
					FArray2D_float mat( 3, 3 );
					FArray1D_float vec( 3 );
					lineup_bk(xyz(1,c1star,res), xyz(1,a1,res), refolded_xyz(1,c1star,res), refolded_xyz(1,a1,res),
										mat, vec);
					for ( int atom = 1, atome = natoms(aa,1); atom <= atome; ++atom ) {
						if ( ( atom_name(atom,aa,1)[3] == ' ' || atom_name(atom,aa,1)[3] == 'M' ) && (atom!=p && atom!=a1 && atom!=a2) ) {
							for ( int j=1; j<=3; ++j )
								refolded_xyz(j,atom,res) = xyz(j,atom,res);
							move_bk( refolded_xyz(1,atom,res), mat, vec );
						}
					}
					align_bk(refolded_xyz(1,c1star,res), refolded_xyz(1,a1,res), xyz(1,a2,res), refolded_xyz(1,a2,res),
									 mat, vec);
					for ( int atom = 1, atome = natoms(aa,1); atom <= atome; ++atom ) {
						if ( ( atom_name(atom,aa,1)[3] == ' ' || atom_name(atom,aa,1)[3] == 'M' ) && (atom!=p && atom!=a1 && atom!=a2) ) {
							move_bk( refolded_xyz(1,atom,res), mat, vec );
						}
					}
					//insert
					*/
				}
				// rotate the rest of the region as a rigid body; note that the 'stub' already exists at this point
				if ( end_frag < domain_end(k) ) {

					get_GL_matrix(xyz(1,o1p,end_frag+1), xyz(1,p,end_frag+1), xyz(1,o5star,end_frag+1),
												refolded_xyz(1,o1p,end_frag+1), refolded_xyz(1,p,end_frag+1), refolded_xyz(1,o5star,end_frag+1), Mgl);
					for ( int res = end_frag + 1, res_end = domain_end(k); res <= res_end; ++res ) {
						int aa = aan(res);
						for ( int atom = 1, atome = natoms( aa, 1 ); atom <= atome; ++atom ) {
							if ( res == end_frag+1 && ( atom == o1p || atom == p || atom == o5star) ) continue; // do not rebuild stub atoms
							GL_rot( Mgl, xyz( 1, atom, res ), refolded_xyz( 1, atom, res ) );
						}
					}
					/*
					//insert
					FArray2D_float mat( 3, 3 );
					FArray1D_float vec( 3 );
					lineup_bk(xyz(1,o1p,end_frag+1), xyz(1,p,end_frag+1), refolded_xyz(1,o1p,end_frag+1), refolded_xyz(1,p,end_frag+1),
										mat, vec);
					for ( int res = end_frag + 1, res_end = domain_end(k); res <= res_end; ++res ) {
						int aa = aan(res);
						for ( int atom = 1, atome = natoms(aa,1); atom <= atome; ++atom ) {
							if (res==end_frag+1 && (atom==o1p || atom==p || atom==o5star) ) continue; // do not rebuild stub atoms
							for ( int j = 1; j <= 3; ++j )
								refolded_xyz(j,atom,res) = xyz(j,atom,res);
							move_bk( refolded_xyz(1,atom,res), mat, vec );
						}
					}
					align_bk(refolded_xyz(1,o1p,end_frag+1), refolded_xyz(1,p,end_frag+1), xyz(1,o5star,end_frag+1), refolded_xyz(1,o5star,end_frag+1),
									 mat, vec);
					for ( int res = end_frag + 1, res_end = domain_end(k); res <= res_end; ++res ) {
						int aa = aan(res);
						for ( int atom = 1, atome = natoms(aa,1); atom <= atome; ++atom ) {
							if (res==end_frag+1 && (atom==o1p || atom==p || atom==o5star) ) continue; // do not rebuild stub atoms
							move_bk( refolded_xyz(1,atom,res), mat, vec );
						}
					}
					//insert
					*/
				}
			} else { // croak if the user-given fragment spans multiple domains
				std::cerr << "Chain-spanning fragments are not allowed!\n";
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		} else { // just copy protein domains
			for ( int res = domain_start; res <= domain_finish; ++res ) {
				int aa = aan(res);
				copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin refold_prot
///
/// @brief
/// for protein-DNA complexes (but will work on proteins in general) - compute xyz given old xyz + new protein dihedral angles
///
/// @detailed
///
/// @param	- [in/out]? -
/// in - begin_frag - start of refold region in the n2c direction
/// in - end_frag - end of refold region in the n2c direction
/// in -  xyz
/// out - refolded_xyz
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// NOTE: Carol's routines (ie refold_update_coords) not used since there are no centroids, and no idealization
/// is wanted here (even though it's mostly gone from her routines as well - am 10/04)
/// Also, note the cute use of atom pseudonames for increased readability
/// [begin_frag,end_frag] interval for refold is end-point inclusive
/// WARNING: omega angles are considered fixed at their native values (in part because bk does not compute them)
/// @references
/// NOTE2 (AM 12/04): fixed_prot_bb_refold flag added to cope with
/// numerical instabilities in Carol refold routines,
/// which produce sizable numerical errors when refold_prot is invoked
/// too many times. Unfortunately, there seems to be no easy fix for
/// the refold routines themselves (acc. to Carol). Under this flag,
/// backbone atoms are copied over rather than refolded.
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_prot(
	int begin_frag,
	int end_frag,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	FArray3Da_float refolded_xyz
)
{
	using namespace param;
	using namespace param_aa;
	using namespace dna_variables;
	using namespace aaproperties_pack;
	using namespace misc;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	refolded_xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	int domain_start,domain_finish;
	FArray2D_float Mgl(4,4);

	//Check input parameters:
	if ( end_frag < begin_frag ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is not allowed in refold_prot!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//Out-of-range checks:
	if ((begin_frag<1 || begin_frag>MAX_RES()) ||
			(end_frag<1 || end_frag>MAX_RES()) ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is out of range in refold_prot!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//Prot-dna checks:
	if ( is_NA(aan(begin_frag)) || is_NA(aan(end_frag)) ) {
		std::cerr << "Refold region: (" << begin_frag << "," << end_frag << ") is not protein only in refold_prot!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Work only with protein domains; copy dna domains over:
	for ( int k = 1; k <= total_domains; ++k ) {
		if (k==1) { // first domain is dna domain
			domain_start = 1;
			domain_finish = domain_end(k);
		} else {
			domain_start = domain_end(k-1) + 1;
			domain_finish = domain_end(k);
		}
		//debug
		//cout << "domain_start = " << domain_start << std::endl;
		//cout << "domain_finish = " << domain_finish << std::endl;
		if (!IsDnaDomain(domain_end(k),aan) ) { // reset xyz for this protein domain
			if (domain_finish < begin_frag || domain_start > end_frag) { // this whole domain simply needs to be copied (we do not work with this chain)
				for ( int res = domain_start; res <= domain_finish; ++res ) {
					int aa = aan(res);
					copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
				}
			} else if (domain_start <= begin_frag && domain_finish >= end_frag) { // fragment is inside this domain, copy->refold->rotate
				// copy all residues that are not going to move
				for ( int res=domain_start; res<begin_frag; ++res) {
					int aa = aan(res);
					copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
				}
				// build region from (phi,psi) dihedral angles (omega is fixed):
				for ( int res = begin_frag; res <= end_frag; ++res ) {
					int aa = aan(res);
					int nh = HNpos(aa,1);  // HN atom number
					if (res==begin_frag) {
						// copy the 3 'stub' atoms (N_i,HN_i,CA_i) which are determined by i-1 dihedral angles:
						copyXYZ(1,2,xyz(1,1,res),refolded_xyz(1,1,res) ); // N_i, CA_i
						if ( nh > 0 ) {
							copyXYZ(nh,nh,xyz(1,1,res),refolded_xyz(1,1,res) ); // HN_i
						}
						if (res==domain_start) { // begin_frag = domain_start
							copyXYZ(3,3,xyz(1,1,res),refolded_xyz(1,1,res) ); // C of the 1st base in chain cannot be built (phi_i undef), just copy it
						} else {
							if ( fixed_prot_bb_refold ) {
								copyXYZ(3,3,xyz(1,1,res),refolded_xyz(1,1,res) ); // C_i
							} else {
								build_new_atom_dihedral (xyz(1,3,res-1), xyz(1,1,res),
									xyz(1,2,res), xyz(1,3,res),
									refolded_xyz(1,3,res-1), refolded_xyz(1,1,res),
									refolded_xyz(1,2,res), refolded_xyz(1,3,res),
									template_pack::phi(res) ); // C_i [phi_i]
							}
						}
					} else {
						if ( fixed_prot_bb_refold ) {
							copyXYZ(1,3,xyz(1,1,res),refolded_xyz(1,1,res) ); // N_i,CA_i,C_i
							if ( nh > 0 ) {
								copyXYZ(nh,nh,xyz(1,1,res),refolded_xyz(1,1,res) ); // HN_i
							}
						} else {
							build_new_atom_dihedral (xyz(1,1,res-1), xyz(1,2,res-1), xyz(1,3,res-1), xyz(1,1,res),
																		 refolded_xyz(1,1,res-1), refolded_xyz(1,2,res-1), refolded_xyz(1,3,res-1), refolded_xyz(1,1,res),
																		 template_pack::psi(res-1) ); // N_i [psi_i-1]
							build_new_atom_fixed (xyz(1,2,res-1), xyz(1,3,res-1), xyz(1,1,res), xyz(1,2,res),
																	refolded_xyz(1,2,res-1), refolded_xyz(1,3,res-1), refolded_xyz(1,1,res), refolded_xyz(1,2,res) ); // CA_i [omega_i-1]
							build_new_atom_dihedral (xyz(1,3,res-1), xyz(1,1,res), xyz(1,2,res), xyz(1,3,res),
																		 refolded_xyz(1,3,res-1), refolded_xyz(1,1,res), refolded_xyz(1,2,res), refolded_xyz(1,3,res),
																		 template_pack::phi(res) ); // C_i [phi_i]
							build_new_atom_fixed (xyz(1,1,res), xyz(1,2,res), xyz(1,3,res), xyz(1,nh,res),
																	refolded_xyz(1,1,res), refolded_xyz(1,2,res), refolded_xyz(1,3,res), refolded_xyz(1,nh,res) ); // HN_i
						}
					}
					if ( fixed_prot_bb_refold ) {
						copyXYZ(4,4,xyz(1,1,res),refolded_xyz(1,1,res) ); // O_i
					} else {
						build_new_atom_fixed (xyz(1,1,res), xyz(1,2,res), xyz(1,3,res), xyz(1,4,res),
							refolded_xyz(1,1,res), refolded_xyz(1,2,res),
							refolded_xyz(1,3,res), refolded_xyz(1,4,res) ); // O_i
					}

					// Rebuild the sidechain:
					FArray2D_float rotcoord( 3, MAX_ATOM()() );
					rotcoord = -1;
					get_rot_coord(refolded_xyz,aa,1,dna_variables::prot_chi_dihedrals(1,res),
						rotcoord,res,xyz(1,1,res) ); // last arg is used to pass in native rather than ideal rotcoords
					copyXYZ(1,natoms(aa,1),rotcoord,refolded_xyz(1,1,res) );
				}
				// rotate the rest of the region as a rigid body; last refolded residue is used as a 'stub'
				if ( end_frag < domain_end(k) ) {
					get_GL_matrix(xyz(1,1,end_frag), xyz(1,2,end_frag), xyz(1,3,end_frag),
												refolded_xyz(1,1,end_frag), refolded_xyz(1,2,end_frag), refolded_xyz(1,3,end_frag), Mgl);
					for ( int res = end_frag + 1, res_end = domain_end(k); res <= res_end; ++res ) {
						int aa = aan(res);
						int nh = HNpos(aa,1);  // HN atom number
						if ( fixed_prot_bb_refold ) {
							copyXYZ(1,4,xyz(1,1,res),refolded_xyz(1,1,res) ); // N_i,CA_i,C_i,O_i
							if ( nh > 0 ) {
								copyXYZ(nh,nh,xyz(1,1,res),refolded_xyz(1,1,res) ); // HN_i
							}
							for ( int atom = 5, atome = natoms( aa, 1 ); atom <= atome; ++atom ) {
								if (atom==nh) continue; // skip HN_i
								GL_rot( Mgl, xyz(1,atom,res), refolded_xyz(1,atom,res) ); // rotate all atoms
							}
						} else {
							for ( int atom = 1, atome = natoms( aa, 1 ); atom <= atome; ++atom ) {
								GL_rot( Mgl, xyz( 1, atom, res ), refolded_xyz( 1, atom, res ) ); // rotate all atoms
							}
						}
					}
				}
			} else { // croak if the user-given fragment spans multiple domains
				std::cerr << "Chain-spanning fragments are not allowed!\n";
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		} else { // just copy dna domains
			for ( int res = domain_start; res <= domain_finish; ++res ) {
				int aa = aan(res);
				copyXYZ(1,natoms(aa,1),xyz(1,1,res),refolded_xyz(1,1,res) );
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_new_atom_dihedral
///
/// @brief
/// general use - builds a dihedral atom given 4 old atoms (for bond lengths etc.) + 3 new atoms + dihedral angle
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - a4_new, coordinates of the newly placed dihedral atom
///
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
/// perhaps should belong elsewhere -  not DNA specific!
/// Essentially, this function is a wrapper around refold_coord_sys/build_atom, taking old/new atomic coords
/// as input and building a new dihedral atom.
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_new_atom_dihedral(
	FArray1Da_float a1_old,
	FArray1Da_float a2_old,
	FArray1Da_float a3_old,
	FArray1Da_float a4_old,
	FArray1Da_float a1_new,
	FArray1Da_float a2_new,
	FArray1Da_float a3_new,
	FArray1Da_float a4_new, // out
	float dih_angle  // in degrees
)
{
	using namespace numeric::constants::f;
//	using numeric::conversions::radians;

	a1_old.dimension(3);
	a2_old.dimension(3);
	a3_old.dimension(3);
	a4_old.dimension(3);
	a1_new.dimension(3);
	a2_new.dimension(3);
	a3_new.dimension(3);
	a4_new.dimension(3);

	// local
	FArray2D_float M(3,3); // rotation matrix
	FArray1D_float X(3),b(3); // 2 vectors defining the coord_system matrix (ie b=O3*_{i-1}->P; X=P->O5* for O3*_{i-1}-P-O5*-C5*)
	FArray1D_float junk(3);

	//subvec(a2_new, a3_new, X);
	//subvec(a1_new, a2_new, b);
	subvec(a3_new, a2_new, X); // Carol's instructions regarding build_atom are WRONG: this is the right order of args for the 1->2->3->4 chain of atoms,
	subvec(a2_new, a1_new, b); // with the dihedral angle around 2->3 computed via dihedral from input_pdb.cc

	//refold_coord_sys(X, b, M(1,1), M(1,2), M(1,3) );
	refold_coord_sys(b, X, M(1,1), M(1,2), M(1,3) ); // Carol's instructions regarding build_atom are WRONG: b (used to find y-axis)
                                                   // must be 1st arg, and X (used to find x-axis) must be 2nd arg

	float D = vec_dist(a3_old, a4_old);
	//float ang = pi - radians( vec_angle(a2_old, a3_old, a4_old) ); // for some reason, Carol's vec_angle has a small numerical error which
	                                                              // grows with multiple refolds. So use bk's instead.
	float ang = 0.0;
	angle_bk(a3_old, a2_old, a3_old, a4_old, ang);
	ang = pi - ang;
	build_atom(M,a3_new,cos(ang),sin(ang),dih_angle,D,a4_new,junk);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_new_atom_fixed
///
/// @brief
/// general use - builds a non-dihedral atom given 4 old atoms (for bond lengths etc.) + 3 new atoms
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - a4_new, coordinates of the newly placed non-dihedral atom
///
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
/// perhaps should belong elsewhere -  not DNA specific!
/// Essentially, this function is a wrapper around get_GL_matrix/GL_rot, taking old/new atomic coords
/// as input and building a new atom.
/// NOTE: do not use if the same rotation matrix can be reused multiple times!!!
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_new_atom_fixed(
	FArray1Da_float a1_old,
	FArray1Da_float a2_old,
	FArray1Da_float a3_old,
	FArray1Da_float a4_old,
	FArray1Da_float a1_new,
	FArray1Da_float a2_new,
	FArray1Da_float a3_new,
	FArray1Da_float a4_new // out
)
{

	a1_old.dimension(3);
	a2_old.dimension(3);
	a3_old.dimension(3);
	a4_old.dimension(3);
	a1_new.dimension(3);
	a2_new.dimension(3);
	a3_new.dimension(3);
	a4_new.dimension(3);

	// local
	FArray2D_float Mgl(4,4);
	get_GL_matrix(a1_old, a2_old, a3_old, a1_new, a2_new, a3_new, Mgl);
	GL_rot(Mgl, a4_old, a4_new);

}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_bp_triad
///
/// @brief
/// for DNA - compute triad for a given basepair
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - M(4,4) with the triad: the output rotation is the (M(1,1),M(1,2),M(1,3)), the offset vector is M(1,4)
/// in  - basepair coords and indices
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
///
/// @references
/// Lu, Hassan, Hunter JMB 273 (1997), 668-680.
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_bp_triad(
	FArray2Da_float base1, // xyz coords of base 1
	FArray2Da_float base2, // xyz coords of base 2
	int aa1,
	int aa2,
	int res1,
	int res2,
	FArray2Da_float M
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;

	base1.dimension( 3, MAX_ATOM() );
	base2.dimension( 3, MAX_ATOM() );
	M.dimension( 4, 4 );

	M = -1;

	// Y-axis is C6->C8 or C8->C6 (always from strand II to strand I, defined by the order of input)
	int a1(0),a2(0); // atom numbers used in the triad calcs
	int ind1(0),ind2(0); // base identity (order: ACGT)
	if (aa_name3(aa1)=="  A" && aa_name3(aa2)=="  T") {
		a1 = c8A;
		a2 = c6T;
		ind1 = 1;
		ind2 = 4;
	} else if (aa_name3(aa1)=="  T" && aa_name3(aa2)=="  A") {
		a1 = c6T;
		a2 = c8A;
		ind1 = 4;
		ind2 = 1;
	} else if (aa_name3(aa1)=="  C" && aa_name3(aa2)=="  G") {
		a1 = c6C;
		a2 = c8G;
		ind1 = 2;
		ind2 = 3;
	} else if (aa_name3(aa1)=="  G" && aa_name3(aa2)=="  C") {
		a1 = c8G;
		a2 = c6C;
		ind1 = 3;
		ind2 = 2;
	} else {
		std::cerr << aa_name3(aa1) << " pairs not with " << aa_name3(aa2) << ", tovarisch!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Start with superimposing the ideal base onto the experimental base (as is done in 3DNA):
	FArray2D_float tr1( 3, MAX_ATOM()() );
	copyXYZ(1,MAX_ATOM(),triad_icoor(1,1,ind1),tr1);
	superimpose_bases(base1, triad_icoor(1,1,ind1), tr1); // tr1, tr2 are ideal coords superimposed onto experimental bases
	FArray2D_float tr2( 3, MAX_ATOM()() );
	copyXYZ(1,MAX_ATOM(),triad_icoor(1,1,ind2),tr2);
	superimpose_bases(base2, triad_icoor(1,1,ind2), tr2);
	// Y-axis
	FArray1D_float v1(3);
	vector_midpoint(tr1(1,a1),tr2(1,a2),M(1,4) );
	subvec(tr1(1,a1),M(1,4),v1);
	unitvec(v1,M(1,2) );
	//vector_normalize(v1);
	//vector_copy(v1,M(1,2));

	// Z-axis is perpendicular to the least squares plane of the basepair:
	// Find the least squares plane through all triad_icoor (reference) atoms of the *basepair*
	FArray1D_float n(3),CM(3);
  lsf_plane_normal(tr1,tr2,res1,res2,triad_icoor_natoms(ind1)+triad_icoor_natoms(ind2),CM,n);
	vector_project_normal(M(1,2),n); // project n normal to y
	vector_normalize(n);
	vector_copy(n,M(1,3) );

	// X-axis is just a cross product of Y and Z:
	cros(M(1,2),M(1,3),M(1,1));

}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_triad
///
/// @brief
/// for DNA - compute triad for a given base
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - M(4,4) with the triad: the output rotation is the (M(1,1),M(1,2),M(1,3)), the offset vector is M(1,4)
/// in  - basepair coord and index, strand index (1 or 2; should be chosen once and for all)
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
///
/// @references
/// Lu, Hassan, Hunter JMB 273 (1997), 668-680.
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_triad(
	FArray2Da_float base, // xyz coords of the base
	int aa,
	int res,
	int strand,
	FArray2Da_float M
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;

	base.dimension( 3, MAX_ATOM() );
	M.dimension( 4, 4 );

	if ( strand!=1 && strand!=2 ) {
		std::cerr << "Illegal strand index: " << strand << "!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	M = -1;

	// Y-axis is N1<->C4 (A,G) or N3<->C6 (C,T) (always from strand II to strand I, defined by the variable 'strand')
	int a1(0),a2(0); // atom numbers used in the triad calcs
	int ind(0); // base identity (order: ACGT)
	if (aa_name3(aa)=="  A") {
		if (strand==1) {
			a1 = n1;
			a2 = c4A;
		} else {
			a1 = c4A;
			a2 = n1;
		}
		ind = 1;
	} else if (aa_name3(aa)=="  G") {
		if (strand==1) {
			a1 = n1;
			a2 = c4G;
		} else {
			a1 = c4G;
			a2 = n1;
		}
		ind = 3;
	} else if (aa_name3(aa)=="  C") {
		if (strand==1) {
			a1 = n3C;
			a2 = c6C;
		} else {
			a1 = c6C;
			a2 = n3C;
		}
		ind = 2;
	} else if (aa_name3(aa)=="  T") {
		if (strand==1) {
			a1 = n3T;
			a2 = c6T;
		} else {
			a1 = c6T;
			a2 = n3T;
		}
		ind = 4;
	} else {
		std::cerr << aa_name3(aa) << " is unknown to me, tovarisch!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Start with superimposing the ideal base onto the experimental base (as is done in 3DNA):
	FArray2D_float tr( 3, MAX_ATOM()() );
	copyXYZ(1,MAX_ATOM(),triad_icoor(1,1,ind),tr);
	superimpose_bases(base, triad_icoor(1,1,ind), tr); // tr is ideal coords superimposed onto experimental base

	// Y-axis
	FArray1D_float v1(3);
	vector_midpoint(tr(1,a1),tr(1,a2),M(1,4) );
	subvec(tr(1,a2),M(1,4),v1);
	unitvec(v1,M(1,2) );
	//vector_normalize(v1);
	//vector_copy(v1,M(1,2));

	// Z-axis is perpendicular to the least squares plane of the basepair:
	// Find the least squares plane through all triad_icoor (reference) atoms of the *basepair*
	FArray1D_float n(3),CM(3);
  lsf_plane_normal(tr,res,triad_icoor_natoms(ind),CM,n); // NOTE: triad_icoor bases are already planar, so this step is NOT necessary
                                                     // (but done here anyway for consistency and generality)
	vector_project_normal(M(1,2),n); // project n normal to y
	vector_normalize(n);
	vector_copy(n,M(1,3) );

	// X-axis is just a cross product of Y and Z:
	cros(M(1,2),M(1,3),M(1,1));

}

//////////////////////////////////////////////////////////////////////////////
/// @begin apply_rotation
///
/// @brief
/// Rotate vector v around axis R by theta radians.
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - pos (rotated vector)
/// in  - v (original vector), R (rotation axis), theta
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
/// perhaps should belong elsewhere -  not DNA specific!
/// @references
/// adapted from PTK (originally written by Eric Alm I think)
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_rotation(
	FArray1Da_float v,
	FArray1Da_float R,
	FArray1Da_float pos,
	double theta
)
{
	v.dimension(3);
	R.dimension(3);
	pos.dimension(3);

	FArray1D_float A1(3);
	FArray1D_float A2(3);
	FArray1D_float A3(3);

	double cost,sint,d,Rx2,Ry2,Rz2;

	cost=std::cos(theta);
	sint=-std::sin(theta);
	//  This matrix was originally computed by Ed Thayer, and double-checked
	//  by Eric Alm.  It is the product of a basis transformation into a basis in which
	//  R is one of the coordinate axes, a simple rotation around R, and
	//  a basis transformation back again.
	Rx2 = R(1)*R(1);  Ry2 = R(2)*R(2);  Rz2 = R(3)*R(3);
	d = Rx2+Ry2;
	A1(1) = Rx2+(Ry2+Rx2*Rz2)*cost/d;
	A1(2) = R(1)*R(2)*(1-cost)+R(3)*sint;
	A1(3) = R(1)*R(3)*(1-cost)-R(2)*sint;
	A2(1) = R(1)*R(2)*(1-cost)-R(3)*sint;
	A2(2) = Ry2+(Rx2+Ry2*Rz2)*cost/d;
	A2(3) = R(2)*R(3)*(1-cost)+R(1)*sint;
	A3(1) = R(1)*R(3)*(1-cost)+R(2)*sint;
	A3(2) = R(2)*R(3)*(1-cost)-R(1)*sint;
	A3(3) = Rz2+d*cost;

	//  This is just an obscure way of writing the product of matrix A
	//  with vector v.
	//  pos is the output vector.
	pos(1) = dotprod(v,A1);
	pos(2) = dotprod(v,A2);
	pos(3) = dotprod(v,A3);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_basestep_prms
///
/// @brief
/// Compute 6 base-step parameters given xyz coords of 4 bases (ie of 2 base-pairs)
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - params
/// in  -
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
/// WARNING: no check for being in a base-pair is done inside this function!
/// @references
/// Lu, Hassan, Hunter JMB 273 (1997), 668-680.
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_basestep_prms(
	FArray2Da_float base1, int ind1, int aa1, // base1 & base2 = base-pair
	FArray2Da_float base2, int ind2, int aa2,
	FArray2Da_float base3, int ind3, int aa3, // base3 & base4 = base-pair
	FArray2Da_float base4, int ind4, int aa4,
	FArray1Da_float params
)
{
	using namespace param;
//	using numeric::conversions::radians;

	base1.dimension( 3, MAX_ATOM() );
	base2.dimension( 3, MAX_ATOM() );
	base3.dimension( 3, MAX_ATOM() );
	base4.dimension( 3, MAX_ATOM() );
	params.dimension( 6 ); // [twist roll tilt shift slide rise]

	FArray2D_float M1(4,4),M2(4,4);
	build_bp_triad(base1,base2,aa1,aa2,ind1,ind2,M1);
	build_bp_triad(base3,base4,aa3,aa4,ind3,ind4,M2);
	// Good place to check if both z-axes point in the same direction:
	if (dotprod(M1(1,3),M2(1,3) ) < 0.0) {
		if ( dna_variables::base_flip_allowed ) {
			dna_variables::flip_occured(ind1) = true;
			dna_variables::flip_occured(ind2) = true;
			dna_variables::flip_occured(ind3) = true;
			dna_variables::flip_occured(ind4) = true;
			std::cerr << "Marking bases " << pdb::pdb_res_num(ind1) << "(" << ind1 << ")/" << pdb::pdb_res_num(ind2) << "(" << ind2
								<< ") and " << pdb::pdb_res_num(ind3) << "(" << ind3 << ")/" << pdb::pdb_res_num(ind4) << "(" << ind4 << ") as flipped!\n";
		} else {
			std::cerr << "Error: z_{i} and z_{j} point in opposite directions in get_basestep_prms for bases "
								<< ind1 << "/" << ind2 << " and " << ind3 << "/" << ind4 << "!\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		dna_variables::flip_occured(ind1) = false;
		dna_variables::flip_occured(ind2) = false;
		dna_variables::flip_occured(ind3) = false;
		dna_variables::flip_occured(ind4) = false;
	}
	// z_{i} and z_{j} are already translated to the origin: compute angle between them
	static FArray1D_float const zero( 3, 0.0f );
	//float Gamma = std::abs( radians( vec_angle( M1(1,3), zero, M2(1,3) ) ) );
	float Gamma = 0.0;
	angle_bk(zero,M1(1,3),zero,M2(1,3),Gamma);
	Gamma = std::abs(Gamma);
	// Get the Roll-Tilt axis:
	FArray1D_float rt(3);
	cros( M1(1,3), M2(1,3), rt );
	vector_normalize(rt);
	// Rotate around rt so that xy planes (and z-axes) of both triads become parallel:
	FArray1D_float xi(3),yi(3),zi(3);
	FArray1D_float xj(3),yj(3),zj(3);
	apply_rotation(M1(1,1), rt, xi, Gamma/2.0);
	apply_rotation(M1(1,2), rt, yi, Gamma/2.0);
	apply_rotation(M1(1,3), rt, zi, Gamma/2.0);
	apply_rotation(M2(1,1), rt, xj, -Gamma/2.0);
	apply_rotation(M2(1,2), rt, yj, -Gamma/2.0);
	apply_rotation(M2(1,3), rt, zj, -Gamma/2.0);
	// Check orthogonality after rotation:
	if (dna_variables::dna_debug==1) {
		float overlap_i = std::abs(dotprod(xi,yi)) + std::abs(dotprod(xi,zi)) + std::abs(dotprod(yi,zi));
		float overlap_j = std::abs(dotprod(xj,yj)) + std::abs(dotprod(xj,zj)) + std::abs(dotprod(yj,zj));
		float const eps = 0.01;
		if (overlap_i > eps || overlap_j > eps) {
			std::cerr << "Orthogonality lost after rotation in get_basestep_prms!\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	// Find the mid-step triad (MST):
	FArray1D_float xMST(3),yMST(3),zMST(3);
	FArray1D_float MST_orig(3);
	// a) MST origin:
	vector_midpoint(M1(1,4),M2(1,4),MST_orig);
	// b) MST unit vectors:
	vector_sum(xi,xj,xMST);
	vector_normalize(xMST);
	vector_sum(yi,yj,yMST);
	vector_normalize(yMST);
	vector_copy(zi,zMST);
	//vector_sum(zi,zj,zMST); // zi and zj must coincide!
	//vector_normalize(zMST);
	// Finally, basestep parameters:
	FArray1D_float tmp(3);
	// Angles:
	//params(1) = std::abs( radians( vec_angle(yi,zero,yj) ) ); // twist
	angle_bk(zero,yi,zero,yj,params(1) );
	params(1) = std::abs(params(1));
	cros(yi,yj,tmp);
	if (dotprod(tmp,zMST) < 0.0) params(1) *= (-1);
	//float phi = std::abs( radians( vec_angle(rt,zero,yMST) ) );
	float phi = 0.0;
	angle_bk(zero,rt,zero,yMST,phi);
	phi = std::abs(phi);
	cros(rt,yMST,tmp);
	if (dotprod(tmp,zMST) < 0.0) phi *= (-1);
	params(2) = Gamma*std::cos(phi);
	params(3) = Gamma*std::sin(phi);
	// Displacements:
	subvec(M2(1,4),M1(1,4),tmp);
	params(4) = dotprod(tmp,xMST);
	params(5) = dotprod(tmp,yMST);
	params(6) = dotprod(tmp,zMST);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin load_basestep_prms
///
/// @brief
/// This function computes and stores all basestep parameters.
///
/// @detailed
///
/// @param	- [in/out]? -
/// out -
/// in  - begin, end, aan, xyz
/// @global_read
/// none
/// @global_write
/// basestep_prms
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
load_basestep_prms(
	FArray1Da_int aan,
	FArray3Da_float xyz,
	int const begin,
	int const end
)
{
	using namespace dna_variables;
	using namespace param;
	using numeric::conversions::to_degrees;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	// Order of storage: params for the (i,i+1) step are stored in the ith bin.
	// Params are computed for all basesteps that are: a) basepaired; b) adjacent in sequence
	for ( int i1 = begin; i1 <= end; ++i1 ) {
		if (basestep_nums(i1) != -1) { // there is a basestep for this basepair, compute params
			int ind1 = i1;
			int ind2 = basepairs[i1].rvspos();
			int ind3 = basestep_nums(i1);
			int ind4 = basepairs[ind3].rvspos();
			assert(ind2 != -1); // check basepairing, just in case
			assert(ind4 != -1);
			get_basestep_prms(xyz(1,1,ind1), ind1, aan(ind1), // base1 & base2 = base-pair
												xyz(1,1,ind2), ind2, aan(ind2),
												xyz(1,1,ind3), ind3, aan(ind3), // base3 & base4 = base-pair
												xyz(1,1,ind4), ind4, aan(ind4),
												basestep_prms(1,i1) );
			for ( int j = 1; j <= 3; ++j ) { // convert angles into degrees
				to_degrees( basestep_prms(j,i1) );
			}
			if (dna_debug==1) {
				std::cout << "res:" << i1 << ": twist=" << basestep_prms(1,i1) << " roll=" << basestep_prms(2,i1)
				 << " tilt=" << basestep_prms(3,i1) << " shift=" << basestep_prms(4,i1)
				 << " slide=" << basestep_prms(5,i1) << " rise=" << basestep_prms(6,i1) << std::endl;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_basepair_prms
///
/// @brief
/// Compute 6 base-pair parameters given xyz coords of 2 bases
///
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - params
/// in  -
/// @global_read
/// none
/// @global_write
/// none
/// @remarks
/// WARNING: no check for being in a base-pair is done inside this function!
/// @references
/// Lu, Hassan, Hunter JMB 273 (1997), 668-680.
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_basepair_prms(
	FArray2Da_float base1, int aa1, int ind1, // base1 & base2 = base-pair
	FArray2Da_float base2, int aa2, int ind2,
	FArray1Da_float params
)
{
	using namespace param;
//	using numeric::conversions::radians;

	base1.dimension( 3, MAX_ATOM() );
	base2.dimension( 3, MAX_ATOM() );
	params.dimension( 6 ); // [propeller buckle opening shear stretch stagger]
	FArray2D_float M1(4,4), M2(4,4);

 // note that base1 is assumed to be on strand I
 // and base2 is assumed to be on strand II

	build_triad( base1, aa1, ind1, 1, M1 );
	build_triad( base2, aa2, ind2, 2, M2 );

	// flips
	if ( dotprod(M1(1,3),M2(1,3) ) < 0.0 ) {
		if ( dna_variables::base_flip_allowed ) {
			dna_variables::flip_occured(ind1) = true;
			dna_variables::flip_occured(ind2) = true;

			std::cerr << "Marking bases " << pdb::pdb_res_num(ind1) << "(" << ind1 <<
			 ")/"	<< pdb::pdb_res_num(ind2) << "(" << ind2 << ") as flipped!\n";

		} else {
			std::cerr << "Error: z_{1} and z_{2} point in opposite directions " <<
			 "in get_basepair_prms for bases " << ind1 << "/" << ind2 << "!\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		dna_variables::flip_occured(ind1) = false;
		dna_variables::flip_occured(ind2) = false;
	}

	// y_{1} and y_{2} are already translated to the origin
	// compute angle between them
	static FArray1D_float const zero( 3, 0.0f );
	//float gamma = std::abs( radians( vec_angle( M1(1,2), zero,M2(1,2) ) ) );
	float gamma = 0.0;
	angle_bk( zero, M1(1,2), zero,M2(1,2), gamma );
	gamma = std::abs(gamma);
	// Get the Buckle-Opening axis:
	FArray1D_float bo(3);
	cros(M2(1,2),M1(1,2),bo);
	vector_normalize(bo);
	// Rotate around bo so that xy planes (and z-axes) of both triads become parallel:
	FArray1D_float xi(3),yi(3),zi(3);
	FArray1D_float xj(3),yj(3),zj(3);
	apply_rotation(M2(1,1), bo, xi, gamma/2.0);
	apply_rotation(M2(1,2), bo, yi, gamma/2.0);
	apply_rotation(M2(1,3), bo, zi, gamma/2.0);
	apply_rotation(M1(1,1), bo, xj, -gamma/2.0);
	apply_rotation(M1(1,2), bo, yj, -gamma/2.0);
	apply_rotation(M1(1,3), bo, zj, -gamma/2.0);

	// Check orthogonality after rotation:
	if ( dna_variables::dna_debug == 1 ) {

		float overlap_i = std::abs(dotprod(xi,yi)) +
		                  std::abs(dotprod(xi,zi)) +
											std::abs(dotprod(yi,zi));

		float overlap_j = std::abs(dotprod(xj,yj)) +
		                  std::abs(dotprod(xj,zj)) +
											std::abs(dotprod(yj,zj));
		float const eps = 0.01;

		if (overlap_i > eps || overlap_j > eps) {
			std::cerr << "Orthogonality lost after rotation in get_basepair_prms!\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	// Find the mid-base triad (MBT):
	FArray1D_float xMBT(3),yMBT(3),zMBT(3);
	FArray1D_float MBT_orig(3);
	// a) MBT origin:
	vector_midpoint(M1(1,4),M2(1,4),MBT_orig);
	// b) MST unit vectors:
	vector_sum(xi,xj,xMBT);
	vector_normalize(xMBT);
	vector_sum(zi,zj,zMBT);
	vector_normalize(zMBT);
	vector_copy(yi,yMBT);
	//vector_sum(yi,yj,yMBT); // yi and yj must coincide!
	//vector_normalize(yMBT);
	// Finally, basepair parameters:
	FArray1D_float tmp(3);
	// Angles:
	//params(1) = std::abs( radians( vec_angle(xi,zero,xj) ) ); // propeller
	angle_bk( zero, xi, zero ,xj, params(1) );
	params(1) = std::abs(params(1) );
	cros(xi,xj,tmp);
	if (dotprod(tmp,yMBT) < 0.0) params(1) *= (-1);
	//float phi = std::abs( radians( vec_angle(bo,zero,xMBT) ) );
	float phi = 0.0;
	angle_bk(zero,bo,zero,xMBT,phi);
	phi = std::abs(phi);
	cros(bo,xMBT,tmp);
	if (dotprod(tmp,yMBT) < 0.0) phi *= (-1);
	params(2) = gamma*std::cos(phi);
	params(3) = gamma*std::sin(phi);
	// Displacements:
	subvec(M1(1,4),M2(1,4),tmp);
	params(4) = dotprod(tmp,xMBT);
	params(5) = dotprod(tmp,yMBT);
	params(6) = dotprod(tmp,zMBT);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin load_basepair_prms
///
/// @brief
/// This function computes and stores all basestep parameters.
///
/// @detailed
///
/// @param	- [in/out]? -
/// out -
/// in  - nres, aan, xyz
/// @global_read
/// none
/// @global_write
/// basepair_prms
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
load_basepair_prms(
	FArray1D_int const & aan,
	FArray3D_float const & xyz,
	int const begin,
	int const end
)
{
	using namespace dna_variables;
	using namespace param;
	using numeric::conversions::to_degrees;

	// Order of storage: params for the ith basepair are stored in the ith bin.
	for ( int i1 = begin; i1 <= end; ++i1 ) {
		int ind1 = i1;
		int ind2 = basepairs[i1].rvspos();

		// base1 & base2 = base-pair
		get_basepair_prms( xyz(1,1,ind1), aan(ind1), ind1,
											 xyz(1,1,ind2), aan(ind2), ind2,
											 basepair_prms(1,i1) );

		for ( int j = 1; j <= 3; ++j ) { // convert angles into degrees
			to_degrees( basepair_prms(j,i1) );
		}
		if ( dna_debug == 1 ) {
			std::cout << "res:" << i1 << ": propeller=" << basepair_prms(1,i1) <<
			 " buckle=" << basepair_prms(2,i1) << " opening=" <<
			 basepair_prms(3,i1) << " shear=" << basepair_prms(4,i1) <<
			 " stretch=" << basepair_prms(5,i1) << " stagger=" <<
			 basepair_prms(6,i1) << std::endl;
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_dna_chains
///
/// @brief
/// for DNA - identifies DNA chains, finds basepairs and basesteps for subsequent calculations of
/// effective DNA parameters
/// @detailed
/// This is a wrapper function invoking other functions for actual calculations. We use global
/// seq. complementarity for 2 stranded DNA, N3/N1 distance checks + basepair complementarity in other cases.
/// @param	- [in/out]? -
/// in  - aan,xyz
/// @global_read
///
/// @global_write
/// dna_variables
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_dna_chains(
	int const nres,
	FArray1D_int const & aan,
	FArray3D_float const & xyz
)
{
	using namespace dna_variables;
	using namespace param;

	assert(nres>0);

	float const MAX_BP_CUTOFF = 3.6; // max allowed distance between N3/N1 atoms in a basepair
	float const MAX_BS_CUTOFF = 5.3; // max allowed distance between N3/N1 atoms in a basestep

	std::cout << "Looking for DNA basesteps and basepairs ...\n";

	init_triad_coords(); // ideal coords for placing basepair and basestep triads
	LoadDNAParams(); // read in all parameters for the effective DNA potential
	//debug
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	// Init. arrays:
	basestep_nums = -1;
	basestep_nums_inv = -1;

	// Initially, disallow base flips and set the array of flip flags to false:
	base_flip_allowed = false;
	flip_occured = false;

	int dna_domains = CountDnaDomains(aan,xyz);
	//int mult_domains = PairDnaDomains(nres,aan,xyz); // which domains are paired
	std::cerr << "This PDB file contains " << dna_domains << " dna chains ..\n";

	if (dna_domains==1) {
		std::cerr << "One stranded DNA is not allowed in this mode - cannot pair one strand!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else	if (dna_domains==2) { // use global pairing by sequence (more robust) + check N3/N1 distances
		find_dna_chains_seq(nres,aan,xyz,MAX_BP_CUTOFF);
	} else { // pairing by structure (less robust) + check local sequence pairing
		find_dna_chains_local( aan,xyz,MAX_BP_CUTOFF );
	}
	find_basesteps(nres,aan,xyz,MAX_BS_CUTOFF);
	set_chain_dir(nres,xyz);
	if (dna_debug==0) {
		print_basesteps(nres,aan);
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin CountDnaDomains
///
/// @brief
/// for DNA - counts DNA chains in a PDB file
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - domain_cnt,chain_dir
/// in  - aan,xyz
/// @global_read
///
/// @global_write
/// dna_variables
/// @remarks
/// NOTE: *obsolete* also sets the global 5'-3' vector for strand I (defined simply as the 1st DNA chain in PDB)
/// This is used later to determine global direction of normals to bases and basepairs
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
CountDnaDomains(
	FArray1Da_int aan,
	FArray3Da_float xyz
)
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	int domain_start = -1;
	int domain_finish = -1;
	int domain_cnt = 0;

	// Identify and count DNA domains:
	for ( int k = 1; k <= total_domains; ++k ) {
		//debug
		//std::cout << "Domain " << k << " out of " << total_domains << ":\n";
		//std::cout << pdb::res_chain(domain_end(k)) << " " << domain_end(k) << std::endl;
		if (IsDnaDomain(domain_end(k),aan) ) {
			++domain_cnt;
			if (k==1) { // first domain is dna domain
				domain_start = 1;
				domain_finish = domain_end(k);
			} else {
				domain_start = domain_end(k-1) + 1;
				domain_finish = domain_end(k);
			}
			// Save the 5'-3' vector for strand I (using C1* atoms): <= obsolete am 11/04
			//if (domain_cnt==1)
			//	subvec(xyz(1,c1star,domain_finish),xyz(1,c1star,domain_start),chain_dir);
		}
	}

	return domain_cnt;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_chain_dir
///
/// @brief
/// for DNA - sets directional vectors (using C1* atoms of adjacent bases)
/// which determine signs of lsf plane normals for all basesteps & basepairs
///
/// @detailed
/// motivation: use local (i->i+1 base) rather than global (1st->last base)
/// directional vectors since a single global vector is ambiguous in cases like
/// IHF, where DNA makes a 180deg turn.  In other words, trace strand I (and
/// III etc. depending on already identified basesteps/basepairs) in the 5'-3'
/// direction using a set of C1*{i}->C1*{i+1} vectors.
///
/// @remarks
/// NOTE: "active" base - base i, for which the basestep prms are recorded in
/// the [i,i+1] step "passive" base - base i+1 in such a step
///
/// @authors
/// Alex Morozov
///
////////////////////////////////////////////////////////////////////////////////
void
set_chain_dir(
	int const nres,
	FArray3D_float const & xyz
)
{
	using namespace dna_variables;
	using namespace param;

	//Init.array:
	chain_dir = -1;

	int chain_dir_cnt = 0; // count directional vectors
	for ( int pos = 1; pos <= nres; ++pos ) {
		if ( basestep_nums(pos) != -1 ) { // this base is "active" in a basestep

			// set chain_dir along "leading" strand:
			subvec( xyz(1,c1star, basestep_nums(pos)), xyz(1,c1star,pos),
			 chain_dir(1,pos) );

				// set SAME chain_dir along "trailing" strand:
				subvec( xyz(1,c1star,basestep_nums(pos)), xyz(1,c1star,pos),
				 chain_dir(1,basepairs[pos].rvspos()) );
				++chain_dir_cnt;

		// this base is "passive" in a basestep, or there is no basestep with it
		// "passive" case, should be most likely
		} else {
			if ( basestep_nums_inv(pos) != -1 ) {

				if ( chain_dir(1,basestep_nums_inv(pos)) != -1 &&
						 chain_dir(2,basestep_nums_inv(pos)) != -1 &&
						 chain_dir(3,basestep_nums_inv(pos)) != -1 ) {
					for ( int j = 1; j <= 3; ++j ) {
						chain_dir(j,pos) = chain_dir(j,basestep_nums_inv(pos));
						chain_dir(j,basepairs[pos].rvspos()) =
							chain_dir(j,basestep_nums_inv(pos));
					}
					++chain_dir_cnt;

				} else {
					std::cerr << "Base " << pos <<
					 " is involved in a basestep with base " << basestep_nums_inv(pos) <<
					 ", but chain_dir is NOT set for base " << basestep_nums_inv(pos) <<
					 " at this point!" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}

			// rare case of an isolated basepair
			// just find closest valid chain_dir using base seq.
			} else {
				int k = pos;
				while ( k >= 1 ) {
					if ( chain_dir(1,k) != -1 &&
						   chain_dir(2,k) != -1 &&
							 chain_dir(3,k) != -1 ) {

						for ( int j = 1; j <= 3; ++j ) {
							chain_dir(j,pos) = chain_dir(j,k);
							chain_dir(j,basepairs[pos].rvspos()) = chain_dir(j,k);
						}

						++chain_dir_cnt;
						break;
					}
					--k;
				}
				// Nothing found in the whole set:
				if ( chain_dir(1,pos) == -1 &&
					   chain_dir(2,pos) == -1 &&
						 chain_dir(3,pos) == -1 ) {

					std::cerr << "NO suitable chain_dir found for the " << pos << "/" <<
					 basepairs[pos].rvspos() << " basepair!\nExiting ..\n";
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}
	// Check the results:
	if ( chain_dir_cnt == 0 ) {
		std::cerr << "No chain_dir vectors set in set_chain_dir!\n";
		std::cerr << "Most likely this function was invoked before finding " <<
		 "basesteps/basepairs!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else if ( dna_debug == 0 ) {
		std::cout << "Created " << chain_dir_cnt << " dna chain_dir vectors ..\n";
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_dna_chains_seq
///
/// @brief
///
/// @remarks
/// WARNING: only works with double stranded dna, use find_dna_chains_local
/// with other cases
///
/// @authors
/// Alex Morozov
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

void
find_dna_chains_seq(
	int const nres,
	FArray1D_int const & aan,
	FArray3D_float const & xyz,
	float const bp_cutoff
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;
	using namespace misc;
	using namespace pdb;

	std::vector< std::string > resIDs_I;
	std::vector< std::string > resIDs_II;
	std::vector< int > resnums_I;
	std::vector< int > resnums_II;

	if (dna_debug==1) {
		std::cout << "Doing basepairing by global sequence match ...\n";
		std::cout << nres << std::endl;
	}

	// Identify DNA domains and save the sequence:
	int domain_start = -1;
	int domain_finish = -1;
	int domain_cnt = 0;
	for ( int k = 1; k <= total_domains; ++k ) {
		if (IsDnaDomain(domain_end(k),aan) ) {
			++domain_cnt;
			if (k==1) { // first domain is dna domain
				domain_start = 1;
				domain_finish = domain_end(k);
			} else {
				domain_start = domain_end(k-1) + 1;
				domain_finish = domain_end(k);
			}
			for ( int i = domain_start; i <= domain_finish; ++i ) {
				int aa = aan(i);
				if (domain_cnt==1) {
					resIDs_I.push_back(aa_name3(aa));
					resnums_I.push_back(i);
					//debug
					//cout << "Strand I:" << aa_name3(aa) << std::endl;
				} else if (domain_cnt==2) {
					resIDs_II.push_back(aa_name3(aa));
					resnums_II.push_back(i);
					//debug
					//cout << "Strand II:" << aa_name3(aa) << std::endl;
				} else {
					std::cerr << "find_dna_chains_seq requires that PDB file contain 2 dna strands!\n";
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	// Find paired DNA strands by scoring sequences:
	int max_score = int(empty_value);
	int max_offset = int(empty_value);
	int cur_end;
	for ( int offset = -6; offset <= 6; ++offset ) { // NOTE: we only slide by 6 bp in each dir, so single-stranded overhangs must be shorter!
		int cur_score = 0;
		cur_end = resIDs_II.size() - 1;
		for ( int i = offset, s = resIDs_I.size(); i < s; ++i ) {
			if ( i < 0 ) {
				--cur_end;
				continue;
			}
			if ( cur_end < 0 ) break;
			if (
			 ( resIDs_I[i]=="  A" && resIDs_II[cur_end]=="  T" ) ||
			 ( resIDs_I[i]=="  T" && resIDs_II[cur_end]=="  A" ) ||
			 ( resIDs_I[i]=="  C" && resIDs_II[cur_end]=="  G" ) ||
			 ( resIDs_I[i]=="  G" && resIDs_II[cur_end]=="  C" ) ) { // this is a putative basepair
				++cur_score;
				//debug
				//cout << resIDs_I[i] << " pairs with " << resIDs_II[cur_end] << std::endl;
			}
			--cur_end;
		}
		if ( cur_score > max_score ) {
			max_score = cur_score;
			max_offset = offset;
		}
	}

	// Check distances between certain atoms in putative basepairs,
	// remove if atoms are too far
	int i1, i2, aa1, aa2, a1(0), a2(0);
	for ( std::vector< DnaPosInfo >::const_iterator bp( basepairs.begin() );
	      bp != basepairs.end(); ++bp ) {

		i1 = bp->fwdpos();
		i2 = bp->rvspos();
		std::cout << i1 << " maps to " << i2 << std::endl;

		aa1 = aan(i1);
		aa2 = aan(i2);
		if      ( aa1 == na_ade ||
			        aa1 == na_gua ) a1 = n1;
		else if ( aa1 == na_cyt ) a1 = n3C;
		else if ( aa1 == na_thy ) a1 = n3T;
		if      ( aa2 == na_ade ||
			        aa2 == na_gua ) a2 = n1;
		else if ( aa2 == na_cyt ) a2 = n3C;
		else if ( aa2 == na_thy ) a2 = n3T;

		FArray1D_float dist(3);
		subvec(xyz(1,a1,i1), xyz(1,a2,i2), dist);
		float l = vector_norm(dist);
		if (l > bp_cutoff) {
			std::cerr << "Distance " << l << " Ang between bases " << aa_name3(aa1) <<
			 "-" << pdb_res_num(i1) << " (" << res_chain(i1) << ") and " <<
			 aa_name3(aa2) << "-" << pdb_res_num(i2) << " (" << res_chain(i2) <<
			 ") is bigger than max cutoff of " << bp_cutoff <<
			 " Ang!\nDisabling basepair.\n";
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_dna_chains_local
///
/// @brief
/// for DNA - pairs up all bases regardless of DNA chain using structural proximity only
/// (with sequence complementarity checked afterwards and non-canonical basepairs removed).
///
/// @detailed
/// Purpose: create a basepair finder that's more flexible than find_dna_chains_seq,
/// allowing for: a) breaks in the chain; b) 3 and 4 stranded dna.
/// @param	- [in/out]? -
/// in  - aan,xyz
/// @global_read
///
/// @global_write
/// dna_variables
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_dna_chains_local(
	FArray1Da_int aan,
	FArray3Da_float xyz,
	float const bp_cutoff
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;
	using namespace misc;
	using namespace pdb;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	if (dna_debug==1) {
		std::cout << "Doing basepairing by local distance constraints ...\n";
	}

	// Find putative basepairs by checking distances between N3/N1 atoms; confirm by sequence:
	int ds1, ds2;
	int df1, df2;
	for ( int k1 = 1; k1 <= total_domains; ++k1 ) {
		if (!IsDnaDomain(domain_end(k1),aan) ) continue; // skip non-DNA domains
		if (k1==1) { // 1st domain in list is dna domain
			ds1 = 1;
			df1 = domain_end(k1);
		} else {
			ds1 = domain_end(k1-1) + 1;
			df1 = domain_end(k1);
		}
		for ( int k2 = 1; k2 <= total_domains; ++k2 ) {
			if (k2==k1) continue; // skip self => means no single chain double-stranded dna allowed
			if (!IsDnaDomain(domain_end(k2),aan) ) continue; // skip non-DNA domains
			if (k2==1) { // 1st domain in list is dna domain
				ds2 = 1;
				df2 = domain_end(k2);
			} else {
				ds2 = domain_end(k2-1) + 1;
				df2 = domain_end(k2);
			}
			//////////////////
			for ( int i1 = ds1; i1 <= df1; ++i1 ) {
				int aa1 = aan(i1);
				float min_l = 1000.0;
				int resnum_min = -1;
				for ( int i2 = ds2; i2 <= df2; ++i2 ) {
					if ( i2 == i1 ) continue; // skip self
					int aa2 = aan(i2);
					// Set atom names which we will use in structural proximity checks:
					int a1(0),a2(0);
					if (aa_name3(aa1)=="  A" || aa_name3(aa1)=="  G") {
						a1 = n1;
					} else if (aa_name3(aa1)=="  C") {
						a1 = n3C;
					} else if (aa_name3(aa1)=="  T") {
						a1 = n3T;
					} else {
						std::cerr << "Residue name " << aa_name3(aa1) << " is not DNA!\n";
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					if (aa_name3(aa2)=="  A" || aa_name3(aa2)=="  G") {
						a2 = n1;
					} else if (aa_name3(aa2)=="  C") {
						a2 = n3C;
					} else if (aa_name3(aa2)=="  T") {
						a2 = n3T;
					} else {
						std::cerr << "Residue name " << aa_name3(aa2) << " is not DNA!\n";
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					FArray1D_float dist(3);
					subvec(xyz(1,a1,i1), xyz(1,a2,i2), dist);
					float l = vector_norm(dist);
					if ( l < min_l ) {
						min_l = l;
						resnum_min = i2;
					}
				}
				if ( min_l < bp_cutoff && resnum_min != -1 ) {
					if ( (aa_name3(aa1)=="  A" && aa_name3(aan(resnum_min))=="  T") ||
					 (aa_name3(aa1)=="  T" && aa_name3(aan(resnum_min))=="  A") ||
					 (aa_name3(aa1)=="  C" && aa_name3(aan(resnum_min))=="  G") ||
					 (aa_name3(aa1)=="  G" && aa_name3(aan(resnum_min))=="  C") ) {
						// this is a basepair confirmed by seq
						if (dna_debug==1) {
							std::cout << "Bases " << aa_name3(aa1) << "-" <<
							 pdb_res_num(i1) << " (" << res_chain(i1) << ") and "
							 << aa_name3(aan(resnum_min)) << "-" << pdb_res_num(resnum_min) <<
							 " (" << res_chain(resnum_min) << ") with N3/N1 cutoff of " <<
							 min_l << " Ang form a standard basepair!\n";
						}
					} else { // croak and remove if the putative basepair is non-canonical
						std::cerr << "Bases " << aa_name3(aa1) << "-" << pdb_res_num(i1) <<
						 " (" << res_chain(i1) << ") and " << aa_name3(aan(resnum_min)) <<
						 "-" << pdb_res_num(resnum_min) << " (" << res_chain(resnum_min) <<
						 ") with N3/N1 cutoff of " << min_l <<
						 " Ang form a non-canonical basepair!\nRemoving ...\n";
					}
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_basesteps
///
/// @brief
/// for DNA - identifies basesteps given basepairs
///
/// @detailed
/// Purpose: a basestep finder that identifies 2 valid basepairs as a basestep
/// a) 2nd basepair is next in residue sequence; b) they are also close in space
/// @param	- [in/out]? -
/// out - basestep_nums in dna_variables
/// @global_read
///
/// @global_write
/// dna_variables
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

void
find_basesteps(
	int const nres,
	FArray1D_int const & aan,
	FArray3D_float const & xyz,
	float const bs_cutoff
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;
	using namespace pdb;

	int comp, pnxt, cnxt;
	FArray1D_int aa( 4, 0 );
	FArray1D_int atom( 4, 0 );

	for ( int pos = 1; pos <= nres - 1; ++pos ) {

		comp = basepairs[pos].rvspos();

		std::cout << pos << " maps to " << comp << std::endl;

		pnxt = pos+1; // next basepair

		cnxt = basepairs[pnxt].rvspos();

		aa(1) = aan(pos);
		aa(2) = aan(comp);
		aa(3) = aan(pnxt);
		aa(4) = aan(cnxt);

		// Set atom names which we will use in structural proximity checks
		// (pos->pnxt and comp->cnxt)):
		for ( int j = 1; j <= 4; ++j ) {
			if      ( aa(j) == na_ade ||
				        aa(j) == na_gua ) atom(j) = n1;
			else if ( aa(j) == na_cyt ) atom(j) = n3C;
			else if ( aa(j) == na_thy ) atom(j) = n3T;
		}

		// check for breaks and confirm using N1/N3 cutoffs:
		FArray1D_float dist1( 3 ), dist2( 3 );

		subvec( xyz(1,atom(1),pos), xyz(1,atom(3),pnxt), dist1 );
		subvec( xyz(1,atom(2),comp), xyz(1,atom(4),cnxt), dist2 );
		float l1 = vector_norm(dist1);
		float l2 = vector_norm(dist2);

		if ( l1 < bs_cutoff && l2 < bs_cutoff ) { // basestep close enough
			basestep_nums(pos) = pnxt;
			basestep_nums_inv(pnxt) = pos;

			if ( dna_debug == 1 ) {
				std::cout << "Bases " << aa_name3(aa(1)) << "-" << pdb_res_num(pos) <<
				 " (" << res_chain(pos) << ") and " << aa_name3(aa(3)) << "-" <<
				 pdb_res_num(pnxt) << " (" << res_chain(pnxt) <<
				 ") with N3/N1 cutoff of " << l1 << " Ang form a basestep!" <<
				 std::endl;
			}

		} else {
			std::cout << "WARNING: base " << aa_name3(aa(1)) << "-" <<
			 pdb_res_num(pos) << " (" << res_chain(pos) <<
			 ") has no basestep because one or both of the N3/N1 distances: (" <<
			 l1 << "," << l2 << ") is greater than the max allowed cutoff: " <<
			 bs_cutoff << std::endl;
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin print_basesteps
///
/// @brief
/// for DNA - diagnostic routine for printing info about valid basepairs and basesteps
///
/// @detailed
///
/// @param	- [in/out]? -
/// out - none
/// @global_read
///
/// @global_write
/// none
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
print_basesteps(
	int const nres,
	FArray1D_int const & aan
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace param;
	using namespace pdb;

	std::cout << "\n=================== Final basesteps: =================\n";
	for ( int k = 1; k <= nres; ++k ) {
		int ind1 = k;
		int ind2 = basestep_nums(k);
		if ( ind2 != -1 ) {
			std::cout << "Bases " << aa_name3(aan(ind1)) << "-" <<
			 pdb_res_num(ind1) << " (" << res_chain(ind1) << ") and " <<
			 aa_name3(aan(ind2)) << "-" << pdb_res_num(ind2) << " (" <<
			 res_chain(ind2) << ") form a basestep\n";
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin superimpose_bases
///
/// @brief
/// general - uses existing routines in orient_rms to superimpose ideal base coords onto experimental base coords.
/// @detailed
///
/// @param	- [in/out]? -
/// in  - base1 and base2 coords; natoms to fit
/// out - normal_i (CM for the set of points); normal_f (coords of the normal in the frame centered at CM)
/// @global_read
///
/// @global_write
///
/// @remarks
/// Only a subset of atoms (not marked with 'none') are used in the fitting (for DNA, this is base heavy atoms [+ C1']).
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void superimpose_bases(
	FArray2Da_float exp_base,   // in
	FArray2Da_float ideal_base, // in
	FArray2Da_float ideal_base_superimposed // out
)
{
	using namespace dna_variables;
	using namespace param;

	exp_base.dimension( 3, MAX_ATOM() );
	ideal_base.dimension( 3, MAX_ATOM() );
	ideal_base_superimposed.dimension( 3, MAX_ATOM() );

	// First, compress coord arrays into consecutive arrays:
	int active_atoms = 0;
	for ( int atom = 1, max_atom = MAX_ATOM(); atom <= max_atom; ++atom ) {
		if ( ideal_base(1,atom)==-1 && ideal_base(2,atom)==-1 && ideal_base(3,atom)==-1 ) continue;
		++active_atoms;
	}
	FArray2D_double XX(3,active_atoms);
	FArray2D_float	XX2(3,active_atoms);
	FArray2D_double YY(3,active_atoms);
	FArray1D_double WW(active_atoms); // weights for weighted RMSD, here should be all 1.0
	FArray1D_int ind(active_atoms); // indices of active atoms in the orig.array
	int counter = 1;
	for ( int atom = 1, max_atom = MAX_ATOM(); atom <= max_atom; ++atom ) {
		if ( ideal_base(1,atom)==-1 && ideal_base(2,atom)==-1 && ideal_base(3,atom)==-1 ) continue;
		XX(1,counter) = ideal_base(1,atom);
		XX(2,counter) = ideal_base(2,atom);
		XX(3,counter) = ideal_base(3,atom);
		XX2(1,counter) = ideal_base(1,atom); // make a copy of the ideal base coords since XX is transformed inside findUU_trans
		XX2(2,counter) = ideal_base(2,atom);
		XX2(3,counter) = ideal_base(3,atom);
		YY(1,counter) = exp_base(1,atom);
		YY(2,counter) = exp_base(2,atom);
		YY(3,counter) = exp_base(3,atom);
		ind(counter)  = atom;
		WW(counter)   = 1.0;
		++counter;
	}

	// Second, find the rotation matrix and offsets for superposition:
	// Storage for output rotation and offsets:
	FArray2D_double UU(3,3);
	FArray1D_double xx_offset(3);
	FArray1D_double yy_offset(3);
	double sigma3 = 0.0;
	// Main function calls:
	findUU_trans(XX,YY,WW,active_atoms,UU,sigma3,xx_offset,yy_offset); // find rotation and offsets
	UU_rotate(XX2,active_atoms,xx_offset,yy_offset,UU); // apply them to the ideal base
	if (dna_debug==1) {
		float rms_out;
		calc_rms_fast(rms_out,XX,YY,WW,active_atoms,sigma3);
		std::cout << "RMSD (exp => ideal) = " << rms_out << std::endl;
	}

	// Third, insert the rotated base coords back into the full array:
	for ( int atom = 1, max_atom = MAX_ATOM(); atom <= max_atom; ++atom ) {
		for ( int i = 1; i <= 3; ++i ) {
			ideal_base_superimposed(i,atom) = empty_value;
		}
	}
	for ( int atom = 1; atom <= active_atoms; ++atom ) {
		ideal_base_superimposed(1,ind(atom)) = XX2(1,atom);
		ideal_base_superimposed(2,ind(atom)) = XX2(2,atom);
		ideal_base_superimposed(3,ind(atom)) = XX2(3,atom);
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin lsf_plane_normal
///
/// @brief
/// general - returns normal to the least squares fit plane for a set of points in 3D space
/// @detailed
///
/// @param	- [in/out]? -
/// in  - base1 and base2 coords; natoms to fit
/// out - normal_i (CM for the set of points); normal_f (coords of the normal in the frame centered at CM)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
/// V.Schomaker et al. Acta Cryst. (1959) 12, 600-604
/// D.M.Blow Acta Cryst. (1960) 13, 168
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
lsf_plane_normal(
	FArray2Da_float base1,       // in
	FArray2Da_float base2,       // in
	int ind1,                    // in
	int ind2,                    // in
	int const natoms2fit,        // in
	FArray1Da_float normal_i,    // out
	FArray1Da_float normal_f     // out
)
{
	using namespace dna_variables;
	using namespace param;

	base1.dimension( 3, MAX_ATOM() );
	base2.dimension( 3, MAX_ATOM() );
	normal_i.dimension( 3 );
	normal_f.dimension( 3 );

	// Create a coord array with basepair atoms:
	//debug
	//std::cout << "COORDS to fit:\n";
	FArray2D_float coords2fit(3,natoms2fit);
	int counter = 1;
	for ( int atom = 1, max_atom = MAX_ATOM(); atom <= max_atom; ++atom ) {
		if (base1(1,atom)== empty_value && base1(2,atom)== empty_value && base1(3,atom)== empty_value ) continue;
		for ( int i = 1; i <= 3; ++i ) {
			coords2fit(i,counter) = base1(i,atom);
		}
		//debug
		//vector_write_line(std::cout,coords2fit(1,counter));
		++counter;
	}
	for ( int atom = 1, max_atom = MAX_ATOM(); atom <= max_atom; ++atom ) {
		if (base2(1,atom)== empty_value && base2(2,atom)== empty_value && base2(3,atom)== empty_value ) continue;
		for ( int i = 1; i <= 3; ++i ) {
			coords2fit(i,counter) = base2(i,atom);
		}
		//debug
		//vector_write_line(std::cout,coords2fit(1,counter));
		++counter;
	}

	// counter should have overshot by one
	--counter;
	assert(counter==natoms2fit);
	//debug
	//std::cout << "=================\n";

	// Translate all atoms to the CM (which is where the unit normal to plane starts):
	normal_i = 0.0;
	for ( int a = 1; a <= natoms2fit; ++a ) {
		for ( int i = 1; i <= 3; ++i ) {
			normal_i(i) += coords2fit(i,a);
		}
	}
	normal_i /= natoms2fit;
	// debug
	//std::cout << "CM:\n";
	//vector_write_line(std::cout,normal_i);

	for ( int a = 1; a <= natoms2fit; ++a ) {
		for ( int i = 1; i <= 3; ++i ) {
			coords2fit(i,a) -= normal_i(i);
		}
	}
	// Create a matrix which provides coeffs for the cubic equation. Note that A(i,j) = A(j,i).
	FArray2D_float A( 3, 3, 0.0f );
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = i; j <= 3; ++j ) {
			for ( int a = 1; a <= natoms2fit; ++a ) {
				A(i,j) += coords2fit(i,a) * coords2fit(j,a);
			}
		}
	}
	// Compute cubic eqn. coeffs:
	float a = A(1,1) + A(2,2) + A(3,3); // alpha
	float b = A(1,3)*A(1,3) + A(1,2)*A(1,2) + A(2,3)*A(2,3) - A(2,2)*A(3,3) - A(1,1)*A(3,3) - A(1,1)*A(2,2); // beta
	float g = A(1,1)*A(2,2)*A(3,3) + 2*A(1,2)*A(2,3)*A(1,3) - A(1,1)*A(2,3)*A(2,3) - A(2,2)*A(1,3)*A(1,3) - A(3,3)*A(1,2)*A(1,2); // gamma
	// The solution to the cubic eqn. is obtained through neglecting the lambda**3 term (lambda is small for nearly planar atom sets):
	// WARNING: bound to fail for REALLY nonplanar atom sets!
	float lambda = (-b - std::sqrt(b*b - 4*a*g) )/(2*a);
	// debug
	//std::cout << "lambda = " << lambda << std::endl;
	//std::cout << "D = " << (b*b - 4*a*g) << std::endl;
	// Finally, compute the eigenvector corresponding to the least squares plane:
	normal_f(1) = (A(2,2)-lambda)*A(1,3) - A(1,2)*A(2,3);
	normal_f(2) = (A(1,1)-lambda)*A(2,3) - A(1,2)*A(1,3);
	normal_f(3) = A(1,2)*A(1,2) - (A(2,2)-lambda)*(A(1,1)-lambda);
	vector_normalize(normal_f);
	// Check consistence of chain_dir indices (basepairs should have identical chain_dir vectors):
	if ( chain_dir(1,ind1)!=chain_dir(1,ind2) ||
	 chain_dir(2,ind1)!=chain_dir(2,ind2) ||
	 chain_dir(3,ind1)!=chain_dir(3,ind2) ) {
		std::cerr << "Basepair " << ind1 << "/" << ind2 << " does not have the same chain_dir vector in lsf_plane_normal!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	// Flip a sign to conform to the convention:
	if (dotprod(chain_dir(1,ind1),normal_f) < 0.0) {
		normal_f *= -1;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin lsf_plane_normal
///
/// @brief
/// general - overloaded lsf_plane_normal to fit the lsf plane to one base (rather than two)
/// @detailed
///
/// @param	- [in/out]? -
/// in  - base coords; natoms to fit
/// out - normal_i (CM for the set of points); normal_f (coords of the normal in the frame centered at CM)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
/// V.Schomaker et al. Acta Cryst. (1959) 12, 600-604
/// D.M.Blow Acta Cryst. (1960) 13, 168
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
lsf_plane_normal(
	FArray2Da_float base,       // in
	int ind,                    // in
	int const natoms2fit,       // in
	FArray1Da_float normal_i,   // out
	FArray1Da_float normal_f    // out
)
{
	using namespace dna_variables;
	using namespace param;

	FArray2D_float dummy_base( 3, MAX_ATOM()(), static_cast< float >( empty_value ) );
	lsf_plane_normal(base,dummy_base,ind,ind,natoms2fit,normal_i,normal_f);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin test_lsf_plane_normal
///
/// @brief
/// general - tests lsf_plane_normal using the example from D.M.Blow Acta Cryst. (1960) 13, 168
/// The answer for the normal to lsf plane is: (0.9931, 0.0346, -0.1119)
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
test_lsf_plane_normal()
{
	using namespace dna_variables;
	using namespace param;

	int natoms2fit = 6; // 3 in base1 and 3 in base2, arbitrarily
	FArray1D_float ni(3),nf(3);
	FArray2D_float base1( 3, MAX_ATOM()(), static_cast< float >( empty_value ) );
	FArray2D_float base2( 3, MAX_ATOM()(), static_cast< float >( empty_value ) );

	////////
	base1(1,1) = 0.1860;
	base1(2,1) = 4.9741;
	base1(3,1) = 4.4804;
	base1(1,2) = 0.0697;
	base1(2,2) = 6.1712;
	base1(3,2) = 3.8151;
	base1(1,3) = 0.0994;
	base1(2,3) = 7.3637;
	base1(3,3) = 4.5212;
	////////
	base2(1,1) = 0.2618;
	base2(2,1) = 7.3200;
	base2(3,1) = 5.9058;
	base2(1,2) = 0.3883;
	base2(2,2) = 6.1238;
	base2(3,2) = 6.6023;
	base2(1,3) = 0.3302;
	base2(2,3) = 4.9323;
	base2(3,3) = 5.8446;

	lsf_plane_normal(base1,base2,1,1,natoms2fit,ni,nf); // note dummy ind1,ind2 base numbers - choose valid chain_dir vectors based on actual PDB
                                                      // (ie choose the valid numbers by hand for this test) !!
	std::cout << "Normal: ";
	vector_write_line(std::cout,nf);
	std::cout << "CM: ";
	vector_write_line(std::cout,ni);
	//debug
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin test_refold
///
/// @brief test refold_dna & refold_prot routines
///
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// NOTE: identifies and prints atoms whose coordinates have changed significantly
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
test_refold(
	int const begin,
	int const end,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	FArray3Da_float refolded_xyz
)
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	refolded_xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	//float const EPS = 2e-3;
	float const EPS = 1e-2;

	assert( begin <= end );

	for ( int res = begin; res <= end; ++res ) {
		int aa = aan(res);
		for ( int atom = 1, atome = natoms( aa, 1 ); atom <= atome; ++atom ) {
			FArray1D_float v(3);
			std::cout << aa_name3(aa) << "-" << res << "(" << atom_name(atom,aa,1) << ") before refold: ";
			vector_write_line( std::cout, xyz( 1, atom, res ) );
			std::cout << aa_name3(aa) << "-" << res << "(" << atom_name(atom,aa,1) << ") after refold: ";
			vector_write_line( std::cout, refolded_xyz( 1, atom, res ) );
			subvec( xyz(1,atom,res), refolded_xyz( 1, atom, res ), v );
			float norm_v = vector_norm(v);
			if ( norm_v > EPS ) {
				std::cout << "Atom " << aa_name3(aa) << "-" << res << "(" << atom_name(atom,aa,1) << ") diverged after refold!\n";
			}
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin test_build_new_atom_dihedral
///
/// @brief
///
/// @detailed
/// simple example on which to test atom xyz building routines
/// @param	- [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
test_build_new_atom_dihedral()
{
	FArray1D_float a(3),b(3),c(3),d(3),dnew(3);
	a(1) = 0; a(2) = 0; a(3) = -1;
	b(1) = 0; b(2) = 0; b(3) = 0;
	c(1) = 1; c(2) = 0; c(3) = 0;
	d(1) = 1; d(2) = 0; d(3) = 1;
	float dih = 180.0;
	build_new_atom_dihedral(a,b,c,d,a,b,c,dnew,dih);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_dna_scores
///
/// @brief
/// for DNA scores - init. DNA energy arrays
///
/// @authors
/// Alex Morozov
///
////////////////////////////////////////////////////////////////////////////////
void
init_dna_scores(
	int const nres,
	FArray1D_int const & aan,
	FArray3D_float const & xyz
)
{
	using namespace dna_variables;
	using namespace param_pack;

	dna_bp_score = 0.0; // init.arrays
	dna_bs_score = 0.0;

	// Init. minimizer energies to large (nonsensical) values:
	total_minim_energy = 1e+15;
	chiral_deviation = 0.0;

	// Load geometric parameters:
	load_basestep_prms( aan, xyz, 1, nres );
	load_basepair_prms( aan, xyz, 1, nres );
	// Score DNA (residue energies set inside score_dna_xx):
	dna_bs_total = pack_wts.Wdna_bs() * score_dna_bs( aan, 1, nres );
	dna_bp_total = pack_wts.Wdna_bp() * score_dna_bp( aan, 1, nres );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_dna_bs
///
/// @brief
/// for DNA - compute effective DNA basestep energies for a given structure
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
/// dna_variables
/// @global_write
/// 1) Basesteps and basepairs must be identified and mapped out
/// 2) Effective DNA basestep params (twist etc.) must be computed and stored
/// 3) Input params for the DNA basestep energy (means, stddevs and force constants) must be loaded
///
/// @references
/// Olson W.K. et al PNAS 1998, 95, 11163-11168.
/// @authors
/// Alex Morozov
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
score_dna_bs(
	FArray1D_int const & aan,
	int const begin,
	int const end
)
{
	using namespace dna_variables;
	using namespace param;
	using namespace param_aa;

	float dna_score = 0.0;
	float const FLIPPED_DNA_SCORE = 1000.0; // big penalty if base flip occured
	float const MAX_SCORE_DEV = 100.0; // max allowed deviation from average beyond which the score is capped
	std::string l1,l2;

	// Score all basepairs using: a) basestep params; b) params for effective DNA potential
	for ( int i1 = begin; i1 <= end; ++i1 ) {
		dna_bs_score(i1) = 0.0;
		if (basestep_nums(i1) != -1 ) { // there is a basestep for this basepair, read params
			int aa1 = aan(i1);
			int aa2 = aan(basestep_nums(i1) );
			//Check for the flip; penalize if found:
			if (flip_occured(i1) || flip_occured(basestep_nums(i1) ) ) {
				dna_bs_score(i1) = FLIPPED_DNA_SCORE;
			} else {
				//Make a label for the basestep type lookup:
				l1 = aa_name3(aa1).substr(2,1);
				l2 = aa_name3(aa2).substr(2,1);
				int ind = get_basestep_index(l1,l2);
				for ( int i = 1; i <= 6; ++i ) {
					float delta_i = basestep_prms(i,i1) - means_bs(ind,i);
					if (delta_i > MAX_SCORE_DEV*stddevs_bs(ind,i) )
						delta_i = MAX_SCORE_DEV*stddevs_bs(ind,i); // reset outliers to max allowed values
					if (delta_i < -MAX_SCORE_DEV*stddevs_bs(ind,i) )
						delta_i = -MAX_SCORE_DEV*stddevs_bs(ind,i); // (beyond which the potential is unreliable)
					for ( int j = 1; j <= 6; ++j ) {
						float delta_j = basestep_prms(j,i1) - means_bs(ind,j);
						if (delta_j > MAX_SCORE_DEV*stddevs_bs(ind,j) )
							delta_j = MAX_SCORE_DEV*stddevs_bs(ind,j); // reset outliers to max allowed values
						if (delta_j < -MAX_SCORE_DEV*stddevs_bs(ind,j) )
							delta_j = -MAX_SCORE_DEV*stddevs_bs(ind,j); // (beyond which the potential is unreliable)

						dna_bs_score(i1) += Fij(ind,i,j)*delta_i*delta_j;
					}
				}
			}
			dna_score += dna_bs_score(i1);
		}
	}
	if (dna_debug==1) {
		std::cout << "Current DNA basestep score is: " << dna_score << std::endl;
	}
	return dna_score;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_dna_bp
///
/// @brief
/// for DNA - compute effective DNA basepair energies for a given structure
///
/// @remarks
/// WARNING: this function relies on 3 kinds of global dna_variables:
/// 1) Basesteps and basepairs must be identified and mapped out
/// 2) Effective DNA basepair params (propeller etc.) must be computed and
///    stored
/// 3) Input params for the DNA basepair energy (means, stddevs and force
///    constants) must be loaded
///
/// @references
/// Olson W.K. et al PNAS 1998, 95, 11163-11168.
///
/// @authors
/// Alex Morozov
///
////////////////////////////////////////////////////////////////////////////////
float
score_dna_bp(
	FArray1D_int const & aan,
	int const begin,
	int const end
)
{
	using namespace dna_variables;
	using namespace param;
	using namespace param_aa;

	float dna_score = 0.0;
	float const FLIPPED_DNA_SCORE = 1000.0; // big penalty if base flip occured
	// max allowed deviation from average beyond which the score is capped
	float const MAX_SCORE_DEV = 100.0;

	// Score all basepairs using:
	// a) basepair params; b) params for effective DNA basepair potential
	for ( int i1 = begin; i1 <= end; ++i1 ) {

		dna_bp_score(i1) = 0.0;
		int aa = aan(i1);

		//Check for the flip; penalize if found:
		if ( flip_occured(i1) || flip_occured(basepairs[i1].rvspos() ) ) {
			dna_bp_score(i1) = FLIPPED_DNA_SCORE;

		} else {
			//Make a label for the basestep type lookup:
			// 3rd char of 3-letter code
			int ind = get_basepair_index( aa_name3(aa)[2] );

			for ( int i = 1; i <= 6; ++i ) {

				float delta_i = basepair_prms(i,i1) - means_bp(ind,i);

				if ( delta_i > MAX_SCORE_DEV*stddevs_bp(ind,i) )
					// reset outliers to max allowed values
					delta_i = MAX_SCORE_DEV*stddevs_bp(ind,i);

				if ( delta_i < -MAX_SCORE_DEV*stddevs_bp(ind,i) )
					// (beyond which the potential is unreliable)
					delta_i = -MAX_SCORE_DEV*stddevs_bp(ind,i);

				for ( int j = 1; j <= 6; ++j ) {

					float delta_j = basepair_prms(j,i1) - means_bp(ind,j);

					if ( delta_j > MAX_SCORE_DEV*stddevs_bp(ind,j) )
						// reset outliers to max allowed values
						delta_j = MAX_SCORE_DEV*stddevs_bp(ind,j);

					if ( delta_j < -MAX_SCORE_DEV*stddevs_bp(ind,j) )
						// (beyond which the potential is unreliable)
						delta_j = -MAX_SCORE_DEV*stddevs_bp(ind,j);

					dna_bp_score(i1) += Gij(ind,i,j)*delta_i*delta_j;
				}
			}
		}
		dna_score += dna_bp_score(i1);
	}
	if ( dna_debug == 1 ) {
		std::cout << "Current DNA basepair score is: " << dna_score << std::endl;
	}
	return dna_score;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_dna_interface
///
/// @brief
/// for prot-DNA - main function for running powell minimizer on protein-DNA
/// interfaces
///
/// @detailed
/// This function minimizes conformations of dna bases, protein mainchain,
/// protein sidechains (all together or in any combination) in the protein
/// (bk terms, gb etc.) + dna_empirical potential.
///
/// @remarks
/// NOTE: since minimizer uses func() with fixed args to evaluate energies,
/// everything has to happen through global variables.
///
/// @authors
/// Alex Morozov
///
////////////////////////////////////////////////////////////////////////////////
void
minimize_dna_interface()
{
	using namespace misc;
	using namespace param;
	using namespace param_pack;
	using namespace dna_variables;

	load_dihedral_angles( total_residue, res, full_coord );
	init_dna_scores( total_residue, res, full_coord );

	map_clusters( true ); // save cluster_map in a dna_variables array; find residue from which to start minimization

	// Allow base flips during minimization (but they will be heavily penalized in dna_bp/dna_bs scoring):
	base_flip_allowed = true;
	//base_flip_allowed = false;
	flip_occured = false;

	nangles = count_angles();
	FArray1D_float active_angles( nangles ); // statically allocated -> dynamic does not work!
	active_angles = empty_value; // initialize array
	pack_angles(active_angles,nangles); // NOTE: both dna and aa chi-angles have to be computed prior to invoking
	                                    //       this function!!!

	//Nothing to do if no angles to minimize:
	if (nangles==0) {
		std::cerr << "No angles to minimize - prot-dna interface minimization skipped!\n";
		return;
	}

	bool gfrag = true;
	int iter;
	float fret = 0.0; // energy returned by minimizer

	//debug
	//std::cout << "Before func_dna:\n";
	//print_dihedral("dna_dih",88,88);

	// Evaluate initial energy:
	float startE = func_dna(active_angles,gfrag); // gfrag is dummy in func_dna call
	if ( dna_debug==0 ) {
		std::cout << "start_energy: " << SS( startE ) << std::endl;
	}

	//debug
	//std::cout << "After func_dna:\n";
	//print_dihedral("dna_dih",88,88);
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	//std::cout << "Energy evaluated again: \n" << func_dna(active_angles,gfrag) << std::endl;
	//print_dihedral("chi",72,78);
	//std::cout << "Before minimize:\n";
	//print_dihedral("dna_dih",88,106);
	/*
	for ( int res1 = 1; res1 <= nres; ++res1 ) {
		if (res1 != 88) continue;
		int aa = aan(res1);
		for ( int a = 1, ae = aaproperties_pack::natoms(aa,1); a <= ae; ++a ) {
			std::cout << SS(param_aa::aa_name3(aa) ) << "-" << pdb::pdb_res_num(res1) << " [" << pdb::res_chain(res1) << "]: xyz = ";
			for ( int j = 1; j <= 3; ++j ) {
				std::cout << F(7, 2, xyz(j,a,res1)) << "  ";
			}
			std::cout << std::endl;
		}
	}
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	*/
	// Minimize energy:
	minimize_set_func(8); // set func(gfrag,active_angles) to our energy function
	powell(gfrag,active_angles,nangles,0.1,iter,fret);
	if ( gfrag ) { // minim. success
		//unpack_angles(active_angles,nangles); // update global angle arrays after minimization
		if ( dna_debug==0 ) {
			std::cout << "end_energy: " << SS( fret ) << " (iter = " << SS( iter ) << ")" << std::endl;
		}
		if ( (startE-fret) < -1.0 ) { // this should never happen with powell
			std::cerr << "WARNING!!! " << std::endl;
			std::cerr << "Energy increased in protein-dna powell minimization !!!" << std::endl;
		}

		//debug
		//print_dihedral("dna_dih",88,88);
		//print_dihedral("phipsi",72,78);
		//std::cout << "After minimize:\n";
		//print_dihedral("dna_dih",88,106);
		/*
		for ( int res1 = 1; res1 <= nres; ++res1 ) {
			if (res1 != 88) continue;
			int aa = aan(res1);
			for ( int a = 1, ae = aaproperties_pack::natoms(aa,1); a <= ae; ++a ) {
				std::cout << SS(param_aa::aa_name3(aa) ) << "-" << pdb::pdb_res_num(res1) << " [" << pdb::res_chain(res1) << "]: xyz = ";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << F(7, 2, xyz(j,a,res1)) << "  ";
				}
				std::cout << std::endl;
			}
		}
		*/
	} else {
		std::cerr << "gfrag is false!!!" << std::endl;
		std::cerr << "Final energy: " << SS( fret ) << std::endl;
		std::cerr << "Protein-dna minimization failed!!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( dna_debug==1 ) {
		std::cout << "============= Final energy check: =============" << std::endl; // rebuild coords and evaluate energy once more
		std::cout << "Final energy (evaluated once more): " << SS( func_dna(active_angles,gfrag) ) << std::endl;
	}
	// Restore full DNA scores after minimizer run, by summing up all residue energies:
	dna_bs_total = 0.0;
	dna_bp_total = 0.0;
	for ( int res = 1; res <= total_residue; ++res ) {
		dna_bs_total += dna_bs_score(res);
		dna_bp_total += dna_bp_score(res);
	}
	dna_bs_total *= pack_wts.Wdna_bs(); // apply weights
	dna_bp_total *= pack_wts.Wdna_bp();
	//debug
	//std::cout << "DNA_final: bs_score=" << dna_bs_total << "\n";
	//std::cout << "DNA_final: bp_score=" << dna_bp_total << "\n";

	// Disallow base flips after minimization:
	base_flip_allowed = false;
	flip_occured = false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin count_angles
///
/// @brief
/// for prot-DNA minimizer - count the number of angles involved
/// @detailed
///
/// @param	- [in/out]? -
/// active_angles - out (packed angle array)
/// nangles - out (its dimension)
/// @global_read
/// template_pack and dna_variables, especially res2minim and ang2minim which show which residues and angles are
/// "active"
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
/// Jim Havranek
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
count_angles()
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	int count_ang = 0;

	if ((MAX_CHI+2) > 7) { // someone reset MAX_CHI from the current value of 4
		std::cerr << "MAX_CHI is reset to " << MAX_CHI << " causing ang2minim array overflow!!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Find actual array dimension:
	for ( int i = 1; i <= total_residue; ++i ) {
		if (!res2minim(i) ) continue;
		int aa = res(i);
		if ( is_protein(aa) || is_nonnatural(aa) ) { // protein aa
			for ( int j = 1, je = nchi(aa,1) + 2; j <= je; ++j ) // note that aav is not supported (whoever needs it can do it himself)
				if (ang2minim(j,i) ) ++count_ang;
		} else { // dna & rna
			for ( int j = 1; j <= 7; ++j )
				if (ang2minim(j,i) ) ++count_ang;
		}
	}
	//debug
	//std::cout << "Found " << count_ang << " angles to minimize ..\n";
	if (count_ang > MAX_ANGLES) {
		std::cerr << "Requested number of active dihedral angles: " << count_ang
							<< " exceeds hardwired maximum of " << MAX_ANGLES << " in pack_angles!\n";
		std::cerr << "Exiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return count_ang;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin pack_angles
///
/// @brief
/// for prot-DNA minimizer - pack phi/psi/chi/dna_dihedrals (in any combination) into a single angle array
/// @detailed
///
/// @param	- [in/out]? -
/// active_angles - out (packed angle array)
/// nangles - out (its dimension)
/// @global_read
/// template_pack and dna_variables, especially res2minim and ang2minim which show which residues and angles are
/// "active"
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pack_angles(
	FArray1Da_float active_angles,
	int & nangles
)
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	active_angles.dimension( nangles );

	if ((MAX_CHI+2) > 7) { // someone reset MAX_CHI from the current value of 4
		std::cerr << "MAX_CHI is reset to " << MAX_CHI << " causing ang2minim array overflow!!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Fill the array:
	int count_angles = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !res2minim(i) ) continue;
		int aa = res(i);
		if ( is_protein(aa) || is_nonnatural(aa) ) { // protein aa
			for ( int j = 1, je = nchi(aa,1)+2; j <= je; ++j ) { // note that aav is not supported (whoever needs it can do it himself)
				if ( ang2minim(j,i) ) {
					if ( j <= nchi(aa,1) ) {
						active_angles(++count_angles) = prot_chi_dihedrals(j,i);
					} else if ( j == nchi(aa,1)+1 ) {
						active_angles(++count_angles) = template_pack::phi(i);
					} else if ( j == nchi(aa,1)+2 ) {
						active_angles(++count_angles) = template_pack::psi(i);
					}
				}
			}
		} else { // dna & rna
			for ( int j = 1; j <= 7; ++j ) {
				if ( ang2minim(j,i) ) {
					active_angles(++count_angles) = dna_dihedrals(j,i);
				}
			}
		}
	}

	if ( count_angles != nangles ) {
		std::cerr << "Expected " << nangles << ", but counted " << count_angles << " in pack_angles!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//debug
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin unpack_angles
///
/// @brief
/// for prot-DNA minimizer - opposite of pack_angles: unpack single angle array
/// into phi/psi/chi/dna_dihedrals global arrays
/// @detailed
///
/// @param	- [in/out]? -
/// active_angles - in (packed angle array)
/// nangles - in (its dimension)
/// @global_read
/// template_pack and dna_variables, especially res2minim and ang2minim which show which residues and angles are
/// "active"
/// @global_write
/// phi,psi,prot_chi_dihedrals,dna_dihedrals
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
unpack_angles(
	FArray1Da_float active_angles,
	int const nangles
)
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using std::abs;

	active_angles.dimension( nangles );
	FArray1D_bool changed_angles( nangles );
	changed_angles = false;
	float const signif_deviation = 1e-7;

	if (nangles > MAX_ANGLES) {
		std::cerr << "Requested number of active dihedral angles: " << nangles
							<< " exceeds hardwired maximum of " << MAX_ANGLES << " in unpack_angles!\n";
		std::cerr << "Exiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ((MAX_CHI+2) > 7) { // someone reset MAX_CHI from the current value of 4
		std::cerr << "MAX_CHI is reset to " << MAX_CHI << " causing ang2minim array overflow!!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Restrict all angles changed by the minimizer to a reasonable range:
	for ( int angle = 1; angle <= nangles; ++angle ) {
		active_angles(angle) = periodic_range(active_angles(angle),360.);
		//debug
		//if (angle==28) {
		//	std::cout << "angle_28 set to: " << active_angles(angle) << std::endl;
		//}
	}

	chiral_deviation = 0.0; // global variable containing \sum_i |chi(old)_i - chi(new)_i|

	int count_angles = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !res2minim(i) ) continue;
		int aa = res(i);
		if ( is_protein(aa) || is_nonnatural(aa) ) { // protein aa
			for ( int j = 1, je = nchi(aa,1) + 2; j <= je; ++j ) { // note that aav is not supported (whoever needs it can do it himself)
				if ( ang2minim(j,i) ) {
					if ( j <= nchi(aa,1) ) { // chi
						++count_angles;
						if (std::abs(prot_chi_dihedrals(j,i) - active_angles(count_angles) ) > signif_deviation) {
							changed_angles(count_angles) = true;
							chiral_deviation += std::abs(prot_chi_dihedrals(j,i) - active_angles(count_angles) );
							//debug
							//std::cout << "CHANGED: protein sidechain angle " << j << " in res=" << i << " (ang=" << count_angles << ")" << std::endl;
						}
						prot_chi_dihedrals(j,i) = active_angles(count_angles);
					} else if ( j == nchi(aa,1) + 1 ) { // phi
						++count_angles;
						if (std::abs(template_pack::phi(i) - active_angles(count_angles) ) > signif_deviation) {
							changed_angles(count_angles) = true;
							chiral_deviation += std::abs(template_pack::phi(i) - active_angles(count_angles) );
							//debug
							//std::cout << "CHANGED: protein phi angle in res=" << i << " (ang=" << count_angles << ")" << std::endl;
						}
						template_pack::phi(i) = active_angles(count_angles);
					} else if ( j == nchi(aa,1) + 2 ) { // psi
						++count_angles;
						if (std::abs(template_pack::psi(i) - active_angles(count_angles) ) > signif_deviation) {
							changed_angles(count_angles) = true;
							chiral_deviation += std::abs(template_pack::psi(i) - active_angles(count_angles) );
							//debug
							//std::cout << "CHANGED: protein psi angle in res=" << i << " (ang=" << count_angles << ")" << std::endl;
						}
						template_pack::psi(i) = active_angles(count_angles);
					}
				}
			}
		} else { // dna & rna
			for ( int j = 1; j <= 7; ++j ) {
				if (ang2minim(j,i) ) {
					++count_angles;
					//debug
					//float devia = std::abs(dna_dihedrals(j,i)-active_angles(count_angles) );
					//if (devia > 1.0) {
					//	std::cerr << "Large deviation (" << devia << ") detected for angle " << j << " in base " << i << std::endl;
					//}
					//if ( i == 107 ) {
					//	std::cout << "base " << aa_name3(aan(i)) << "-" << i << ", angle " << j << ": before="
					//						<< dna_dihedrals(j,i) << ", after=" << active_angles(count_angles) << std::endl;
					//}
					if (std::abs(dna_dihedrals(j,i) - active_angles(count_angles) ) > signif_deviation) {
						changed_angles(count_angles) = true;
						chiral_deviation += std::abs(dna_dihedrals(j,i) - active_angles(count_angles) );
						//debug
						//std::cout << "CHANGED: dna angle " << j << " in res=" << i << " (ang=" << count_angles << ")" << std::endl;
					}
					dna_dihedrals(j,i) = active_angles(count_angles);
				}
			}
		}
	}

	if ( count_angles != nangles ) {
		std::cerr << "Expected " << nangles << ", but counted " << count_angles << " in unpack_angles!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Reset cluster map:
	reset_clusters(changed_angles,nangles);

	// Enforce ill-defined angles to stay fixed at their default values:
	// DNA: alpha=0.0 if 1st in chain, epsilon/xi=0.0 if last in chain.
	// protein: phi=-90 if 1st in chain, psi=130 if last in chain.
	int domain_start,domain_finish;
	for ( int k = 1; k <= misc::total_domains; ++k ) {
		if (k==1) {
			domain_start = 1;
			domain_finish = misc::domain_end(k);
		} else {
			domain_start = misc::domain_end(k-1) + 1;
			domain_finish = misc::domain_end(k);
		}
		//debug
		//cout << "domain_start = " << domain_start << std::endl;
		//cout << "domain_finish = " << domain_finish << std::endl;
		if (IsDnaDomain(domain_finish,res) ) { // reset ill-defined dihedral angles to their defaults
			dna_dihedrals(1,domain_start) = 0.0; // alpha
			dna_dihedrals(5,domain_finish) = 0.0; // epsilon
			dna_dihedrals(6,domain_finish) = 0.0; // xi
		} else {
			if ( is_protein( res(domain_start) ) || is_nonnatural( res(domain_start) ) )
				template_pack::phi(domain_start) = -90.0;
			if ( is_protein( res(domain_finish) ) || is_nonnatural( res(domain_finish) ) )
				template_pack::psi(domain_finish) = 130.0;
		}
	}
	//debug
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_clusters
///
/// @brief
/// resets res2minim & ang2minim according to what angles have changed in a single
/// minimizer step.
/// @detailed
/// many minimizers change only one or a few angles at each step, thus it makes a lot
/// of sense to remap clusters before invoking the refold_prot/refold_dna routines. Reducing
/// the number of clusters will result in big computational savings (as each refold is done once
/// per cluster) and moreover will help control numerical errors in xyz which inevitably appear if
/// refold_prot/refold_dna are used too many times.
/// @param	- [in/out]? -
/// changed_angles array
/// @global_read
///
/// "active"
/// @global_write
/// res2minim, ang2minim
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_clusters(
	FArray1Da_bool changed_angles,
	int const nangles
)
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	changed_angles.dimension( nangles );

	if ( nangles > MAX_ANGLES ) {
		std::cerr << "Requested number of active dihedral angles: " << nangles
							<< " exceeds hardwired maximum of " << MAX_ANGLES << " in reset_clusters!\n";
		std::cerr << "Exiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Save the old res2minim & ang2minim:
	FArray1D_bool const res2minim_saved( res2minim );
	FArray2D_bool const ang2minim_saved( ang2minim );

	// Reset ang2minim to false for unchanged angles:
	int count_angles = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !res2minim(i) ) continue;
		int aa = res(i);
		if ( is_protein(aa) || is_nonnatural(aa) ) { // protein aa
			for ( int j = 1, je = nchi(aa,1)+2; j <= je; ++j ) { // note that aav is not supported (whoever needs it can do it himself)
				if ( ang2minim(j,i) && !changed_angles(++count_angles) ) {
					ang2minim(j,i) = false;
				}
			}
		} else { // dna & rna
			for ( int j = 1; j <= 7; ++j ) {
				if ( ang2minim(j,i) && !changed_angles(++count_angles) ) {
					ang2minim(j,i) = false;
				}
			}
		}
	}

	if ( count_angles != nangles ) {
		std::cerr << "Expected " << nangles << ", but counted " << count_angles << " in reset_clusters!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Reset res2minim to false if all ang2minim are false for this residue:
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !res2minim(i) ) continue;
		int aa = res(i);
		bool angle_flag = false;
		if ( is_protein(aa) || is_nonnatural(aa) ) { // protein aa
			for ( int j = 1; j <= nchi(aa,1)+2; ++j ) { // note that aav is not supported (whoever needs it can do it himself)
				if ( ang2minim(j,i) ) {
					angle_flag = true;
					break;
				}
			}
		} else { // dna & rna
			for ( int j = 1; j <= 7; ++j ) {
				if ( ang2minim(j,i) ) {
					angle_flag = true;
					break;
				}
			}
		}
		res2minim(i) = angle_flag;
	}

	// Re-map clusters:
	map_clusters(false);

	// Restore res2minim & ang2minim for the next round:
	res2minim = res2minim_saved;
	ang2minim = ang2minim_saved;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin rebuild_coords
///
/// @brief
/// for prot-DNA minimizer - produce new xyz coords given a set of
/// dihedral angles (phi/psi/chi/dna_dihedrals in any combination)
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
/// cluster_map (+ dihedral angles are read in refold_* functions)
/// @global_write
/// xyz
/// @remarks
/// WARNING: cluster_map must be set before invoking this function!
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
rebuild_coords()
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param_aa;
	using namespace param;

	FArray3D_float xyz2( 3, MAX_ATOM()(), MAX_RES()() );
	//int cluster_cnt = 0;
	for ( int k = 1; k <= total_residue; ++k ) {
		if (cluster_map(k) != -1 ) {
			xyz2 = empty_value;
			if ( is_NA(res(k)) && is_NA(res(cluster_map(k))) ) {
				refold_dna(k,cluster_map(k),res,full_coord,xyz2); // will only refold/move dna domains
			} else if ( !is_NA(res(k)) && !is_NA(res(cluster_map(k))) ) {
				refold_prot(k,cluster_map(k),res,full_coord,xyz2); // will only refold/move prot domains
			} else {
				std::cerr << "Cluster [" << k << "," << cluster_map(k) << "] is part protein, part dna!!\nExiting ..\n";
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			full_coord = xyz2;

		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin map_clusters
///
/// @brief
/// for prot-DNA minimizer - create a map of clusters (or fragments) to refold
/// given a list of residues to be minimized.
/// @detailed
/// Adjacent residues in the same chain are marked as being in one cluster.
/// @param	- [in/out]? -
/// init_run: true/false - does not reset dna_cluster_present & minimize_start if false
///
/// @global_read
/// res2minim
/// @global_write
/// minimize_start, cluster_map
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_clusters(
	bool init_run
)
{
	using namespace dna_variables;
	using namespace pdb;
	using namespace misc;

	std::vector< int > cluster_start;
	std::vector< int > cluster_finish;

	cluster_map = -1; // Init. cluster_map

	if ( init_run ) {
		dna_cluster_present = false; // flag showing if there are any clusters of dna bases
		minimize_start = 100000; // big positive number
	}

	// Find residues adjacent in seq. (ie clusters)
	int ds,df;
	for ( int k = 1; k <= total_domains; ++k ) {
		if (k==1) {
			ds = 1;
			df = domain_end(k);
		} else {
			ds = domain_end(k-1) + 1;
			df = domain_end(k);
		}
		// Energy computations have to include all dna bases because of basepairing.
		// WARNING: it is *very* dangerous to include a base into the minimizer but not its pair,
    // or to move a base without computing its basepairing energy.
		if ( init_run ) {
			if (IsDnaDomain(df,res) && ds<minimize_start) {
				minimize_start = ds;
			}
		}
		for ( int i = ds; i <= df; ++i ) {
			if ( i == ds ) {
				if ( res2minim(i) ) {
					cluster_start.push_back(i);
				}
			} else {
				if ( res2minim(i) ) { // this is the beginning of cluster
					if ( !res2minim(i-1) ) cluster_start.push_back(i);
					if ( i == df ) cluster_finish.push_back(i);
				}
				if ( !res2minim(i) && res2minim(i-1) ) { // this is the end of cluster
					cluster_finish.push_back(i-1);
				}
			}
		}
	}
	assert( cluster_start.size() == cluster_finish.size() );
	if ( cluster_start.empty() ) { // this means we are in minimizer with nothing to minimize!
		if ( init_run ) {
			std::cerr << "Warning: no clusters found in minimizer setup!\n";
		}
		return;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( dna_debug==0 && init_run ) {
		std::cout << "Mapped out " << cluster_start.size() << " cluster(s) to minimize ..\n";
	}
	// Save cluster info in global space
	if ( init_run ) {
		if (cluster_start[0] < minimize_start)
			minimize_start = cluster_start[0]; // residue number from which minimization starts
		if (dna_debug==0) {
			std::cout << "Minimization start: resno=" << pdb_res_num(minimize_start) << " (" << res_chain(minimize_start) << ")\n";
		}
	}

	for ( std::vector< int >::size_type i = 0, s = cluster_start.size(); i < s; ++i ) {
		cluster_map(cluster_start[i]) = cluster_finish[i];

		// Set the dna_cluster_present flag which will turn on DNA scoring in the minimizer
		// energy function. NOTE: any DNA base (paired or not) should trigger this flag, since
		// refold of even an isolated base will affect xyz of all downstream bases in the same
		// strand (eg refold of unpaired T97C in 1aay will move the rest of the strand C around).
		// This of course will affect dna scoring as most of the moving bases will be paired with
		// a complementary strand.
		if ( init_run ) {
			if ( IsDnaDomain(cluster_start[i],res) ) { // there is a cluster of DNA bases
				dna_cluster_present = true;
			}
		}
    if ( dna_debug==0 && init_run ) {
    //if ( dna_debug==0 ) {
      std::cout << "Cluster " << i+1 << ": resno=" << pdb_res_num(cluster_start[i]) << "-" << pdb_res_num(cluster_finish[i])
                << " (" << res_chain(cluster_start[i]) << ")\n";
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin compute_energy
///
/// @brief
/// for prot-DNA minimizer - function to compute structure energy given
/// a global xyz array.
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
/// yes
/// @global_write
/// @remarks
/// WARNING: gborn is not used here (for now).
/// NOTE: this function uses neighborlist computed once *before* minimization, so no
/// large-scale motion is expected!!
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
compute_energy()
{
	using namespace misc;
	using namespace dna_variables;
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace template_pack;

	float tmp1, tmp2;

	float energy = 0.0; // energy of this structure
	//float cur_dna_bs = 0.0;
	//float cur_dna_bp = 0.0;
	/*
	//debug
	std::cout << "Inside compute_energy:\n";
	for ( int res1 = 1; res1 <= nres; ++res1 ) {
		if (res1 != 88) continue;
		int aa = aan(res1);
		for ( int a = 1, ae = aaproperties_pack::natoms(aa,1); a <= ae; ++a ) {
			std::cout << SS(param_aa::aa_name3(aa) ) << "-" << pdb::pdb_res_num(res1) << " [" << pdb::res_chain(res1) << "]: xyz = ";
			for ( int j = 1; j <= 3; ++j ) {
				std::cout << F(7, 2, xyz(j,a,res1)) << "  ";
			}
			std::cout << std::endl;
		}
	}
	*/

	//ja factoring
	energy += dna_geom_energy();

	//Given new xyz coords, determine the coordinates for the action center of each sidechain
	for ( int seqpos = minimize_start; seqpos <= total_residue; ++seqpos ) {
		int aa = res(seqpos);
		if ( is_protein(aa) || is_nonnatural(aa) ) {
			put_wcentroid(full_coord(1,1,seqpos),actcoord(1,seqpos),aa);
		}
		//debug
		//for ( int j = 1, je = aaproperties_pack::natoms(aa,1); j <= je; ++j ) {
		//	std::cout << "r=" << I(4,seqpos) << ",a=" << I(4,j) << " (" << SS( aaproperties_pack::atom_name(j,aa,1) ) << "): ";
		//	vector_write_line(std::cout,xyz(1,j,seqpos));
		//}
	}

	// Recompute all energies after each move; do not recompute if both residues have not moved
	for ( int res1 = 1, max_atom = MAX_ATOM(); res1 <= total_residue; ++res1 ) {
		int aa1 = res(res1);
		FArray2D_float xyz_111( 3, MAX_ATOM()() );
		for ( int i = 1; i <= 3; ++i )
			for ( int j = 1; j <= max_atom; ++j )
				xyz_111(i,j) = full_coord(i,j,res1);
		FArray1D_float actcoord_1( 3 );
		for ( int i = 1; i <= 3; ++i )
			actcoord_1 = actcoord(i,res1);
		//float & xyz_111( xyz(1,1,res1) );
		//float & actcoord_1( actcoord(1,res1) );

		//if ( res1 >= minimize_start && ( is_protein(aa1) || is_nonnatural(aa1) ) ) // recompute centroid for this residue if aa
		//	put_wcentroid(xyz(1,1,res1),actcoord(1,res1),aa1 );

		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
			int aa2 = res(res2);
			float & xyz_112( full_coord(1,1,res2) );
			float & actcoord_2( actcoord(1,res2) );

			if (res1<minimize_start && res2<minimize_start) continue; // skip fixed residues
			if (res1>res2) continue; // do not recompute res-res interactions twice
			if ( is_DNA(aa1) && is_DNA(aa2) ) continue; // dna-dna interactions are not computed by the energy functions anyway

			//if ( res2!=res1 && res2>=minimize_start && ( is_protein(aa1) || is_nonnatural(aa1) ) ) // recompute centroid for this residue if aa
			//	put_wcentroid(xyz_112,actcoord_2,aa2 );

			if (res1==res2) { // intraresidue energy
				float Paa_phipsiE = 0.0;
				float intra_atrE = 0.0, intra_repE = 0.0;

				//dun_trial(res) = neg_Wdun_log_rperc; // no dunbrack here
				if (pack_wts.Wone() != 0.0) {
					get_Paa_ppE(template_pack::phi(res1),template_pack::psi(res1),aa1,Paa_phipsiE,tmp1,tmp2); // -lnP(aa|phi,psi)
					energy += pack_wts.Wone() * Paa_phipsiE; // -lnP term
				}
				if (pack_wts.Wintra() != 0.0) {
					get_intra_resE(aa1,1,full_coord(1,1,res1),intra_atrE,intra_repE); // intrares rep and attr LJ; summed up inside this function
					energy += pack_wts.Wintra() * (intra_repE + intra_atrE); // intrares rep and attr LJ
				}

				if ( geometric_sol::geometric_sol_flag ) {
					energy += res_res_geometric_sol(res1,res1,aa1,1,aa1,1,full_coord(1,1,res1),full_coord(1,1,res1),true,true,true,true);
				}

				//debug
				//std::cout << "res1 [" << res1 << "] : Paa_phipsiE = " << Paa_phipsiE << " [Wone=" << pack_wts.Wone << "]\n";
				//std::cout << "res1 [" << res1 << "] : intra_repE = " << intra_repE << " [Wintra=" << pack_wts.Wintra << "]\n";
				//std::cout << "res1 [" << res1 << "] : intra_atrE = " << intra_atrE << " [Wintra=" << pack_wts.Wintra << "]\n";
			} else { // res-res interaction
				float solE=0.0,atrE=0.0,repE=0.0,elecE=0.0,pairE=0.0,plane_totalE=0.0; // bk pairwise energies
				float schbE=0.0,srbbhbE=0.0,lrbbhbE=0.0,sc_bbhbE=0.0;

				if ( neighborlist(res2,res1) ) { // NOTE: neighborlist is not changed by minimization!! (we do not want to bring residues
					                               // in and out of minimizer)
					                               // NOTE2: neighborlist(res1,res2) = neighborlist(res2,res1)
					get_res_resE(aa1,aa2,1,1,xyz_111,xyz_112,actcoord_1,actcoord_2,
											 res1,res2,solE,atrE,repE,pairE,plane_totalE,elecE); // get pairwise energies

					get_hbE(true, aa1,aa2,1,1,res1,res2,neighbors(res1),
									neighbors(res2),full_coord(1,1,res1),xyz_112,schbE,srbbhbE,lrbbhbE,sc_bbhbE);

					if ( geometric_sol::geometric_sol_flag ) {
						solE += res_res_geometric_sol(res1,res2,aa1,1,aa2,1,full_coord(1,1,res1),xyz_112,true,true,true,true);
						solE += res_res_geometric_sol(res2,res1,aa2,1,aa1,1,xyz_112,full_coord(1,1,res1),true,true,true,true);
					}

					energy += pack_wts.Watr()*atrE + pack_wts.Wrep()*repE + pack_wts.Wsol()*solE + pack_wts.Whbond_sc()*schbE + // two-body terms
						pack_wts.Whbond_bb()*(lrbbhbE + srbbhbE) + pack_wts.Whbond_bb_sc()*sc_bbhbE;
					/*
					//debug
					//if ((is_protein(aa1) && !is_protein(aa2)) || (!is_protein(aa1) && is_protein(aa2)) ) {
					if ( is_protein(aa1) && !is_protein(aa2) ) {
						std::cout << "neighbors: " << neighborlist(res2,res1) << "," << neighborlist(res1,res2) << std::endl;
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: atrE = " << atrE << " [Watr=" << pack_wts.Watr() << "]\n";
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: repE = " << repE << " [Wrep=" << pack_wts.Wrep() << "]\n";
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: solE = " << solE << " [Wsol=" << pack_wts.Wsol() << "]\n";
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: sc_hbE = " << schbE << " [Whbond_sc=" << pack_wts.Whbond_sc() << "]\n";
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: bb_hbE = " << (lrbbhbE + srbbhbE) << " [Whbond_bb=" << pack_wts.Whbond_bb() << "]\n";
						std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: scbb_hbE = " << sc_bbhbE << " [Whbond_bb_sc=" << pack_wts.Whbond_bb_sc() << "]\n";
					}
					*/
				}
			}
		}
	}
	//debug
	//std::cout << "energy = " << energy << std::endl;
	//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	return energy;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dna_geom_energy
///
/// @brief
/// DNA intramolecular score
/// (the "misc" version)
///
/// @authors
/// Morozov
/// ashworth (refactor)
///
////////////////////////////////////////////////////////////////////////////////

float dna_geom_energy()
{
	using namespace dna_variables; // dna_bs_total, dna_bp_total
	using namespace misc; // globals
	using namespace param_pack; // Wdna_bs, Wdna_bp

	if ( !design::dna_interface ) return 0.0;

	float energy( 0.0 );

	if ( dna_cluster_present ) {

		if ( pack_wts.Wdna_bs() != 0.0 ) {
			// Recompute geom.parameters:
			load_basestep_prms( res, full_coord, 1, total_residue );
			// Re-score DNA (residue energies set inside score_dna_xx):
			dna_bs_total = pack_wts.Wdna_bs() * score_dna_bs( res , 1, total_residue );
			energy += dna_bs_total;
		}

		if ( pack_wts.Wdna_bp() != 0.0 ) {
			// Recompute geom.parameters:
			load_basepair_prms( res, full_coord, 1, total_residue );
			// Re-score DNA (residue energies set inside score_dna_xx):
			dna_bp_total = pack_wts.Wdna_bp() * score_dna_bp( res, 1, total_residue );
			energy += dna_bp_total;
		}
	}

	return energy;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin protdna_bump_detector
///
/// @brief
/// prot-DNA minimizer setup (extra option) - function that identifies unusually high prot-dna repulsive energies (bumps) and adds bumping amino acids & bases to the lists passed into the minimizer.
///
/// @authors
/// Alex Morozov
/////////////////////////////////////////////////////////////////////////////////
void
protdna_bump_detector(
	int const nres,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	std::string const & prot_mode,
	std::string const & dna_mode
)
{
	using namespace dna_variables;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace template_pack;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	float const MAX_REPENERGY = 10.0; // repE larger than this will trigger minimization
	int base_cnt = 0;
	int aa_cnt = 0;

	// Check the modes:
	if ( prot_mode != "none" && prot_mode != "all" && prot_mode != "phipsi" && prot_mode != "chi" ) {
		std::cerr << "ERROR: unknown prot_mode: " << prot_mode << " in protdna_bump_detector!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( dna_mode != "none" && dna_mode != "all" && dna_mode != "bb" && dna_mode != "chi" ) {
		std::cerr << "ERROR: unknown dna_mode: " << dna_mode << " in protdna_bump_detector!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( prot_mode == "none" && dna_mode == "none" ) {
		std::cerr << "WARNING: none/none mode in protdna_bump_detector - bump check skipped!\n";
		return;
	}

	//Given (possibly new) xyz coords, determine the coords for the action center of each sidechain
	for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
	int aa = aan(seqpos);
	if ( is_protein(aa) || is_nonnatural(aa) ) {
		put_wcentroid(xyz(1,1,seqpos),actcoord(1,seqpos),aa);
		}
	}

	// Compute all res-res bk (rep,atr etc.) energies & find clashes:
	for ( int res1 = 1, max_atom = MAX_ATOM(); res1 <= nres; ++res1 ) {
		int aa1 = aan(res1);
		FArray2D_float xyz_111( 3, MAX_ATOM()() );
		for ( int i = 1; i <= 3; ++i )
			for ( int j = 1; j <= max_atom; ++j )
				xyz_111(i,j) = xyz(i,j,res1);
		FArray1D_float actcoord_1( 3 );
		for ( int i = 1; i <= 3; ++i )
			actcoord_1 = actcoord(i,res1);
		//float & xyz_111( xyz(1,1,res1) );
		//float & actcoord_1( actcoord(1,res1) );

		//if ( is_protein(aa1)|| is_nonnatural(aa1) ) // recompute centroid for this residue if aa
		//	put_wcentroid(xyz(1,1,res1),actcoord(1,res1),aa1 );

		for ( int res2 = 1; res2 <= nres; ++res2 ) {
			int aa2 = aan(res2);
			float & xyz_112( xyz(1,1,res2) );
			float & actcoord_2( actcoord(1,res2) );

			float cur_resresE = 0.0; // interaction energy between res1 & res2

			if (res1>=res2) continue; // do not recompute res-res interactions twice; no intrares energy in bumpcheck
			//if (res1==res2) continue; // no intrares energy in bumpcheck
			if ( ( is_DNA(aa1) && is_DNA(aa2) ) ||
			 ( !is_DNA(aa1) && !is_DNA(aa2) ) ) continue; // compute prot-dna interactions only

			//if (res2!=res1 && is_protein(aa2) ) // recompute centroid for this residue if aa
			//	put_wcentroid(xyz_112,actcoord_2,aa2 );

			float solE=0.0,atrE=0.0,repE=0.0,elecE=0.0,pairE=0.0,plane_totalE=0.0; // bk pairwise energies

			if ( neighborlist(res2,res1) ) { // neighborlist(res1,res2) = neighborlist(res2,res1)
				get_res_resE(aa1,aa2,1,1,xyz_111,xyz_112,actcoord_1,actcoord_2,
										 res1,res2,solE,atrE,repE,pairE,plane_totalE,elecE); // get pairwise energies

				cur_resresE = pack_wts.Watr()*atrE + pack_wts.Wrep()*repE + pack_wts.Wsol()*solE; // two-body terms
				if (cur_resresE > MAX_REPENERGY) {
					std::cout << "Residues " << SS(aa_name3(aa1) ) << "-" << pdb::pdb_res_num(res1) << " [" << pdb::res_chain(res1) << "] and "
										<< SS(aa_name3(aa2) ) << "-" << pdb::pdb_res_num(res2) << " [" << pdb::res_chain(res2) << "] have bkE = "
										<< cur_resresE << "! Adding to minimizer (mode dependent!) ..\n";
					if ( !res2minim(res1) ) {
						if ( ( is_protein( aa1 ) || is_nonnatural(aa1) ) && prot_mode != "none" ) {
							res2minim(res1) = true;
							++aa_cnt;
						} else if ( is_NA( aa1 ) && dna_mode != "none" ) {
							res2minim(res1) = true;
							++base_cnt;
							// Basepairs have to be minimized together:
							if ( !res2minim( basepairs[res1].rvspos() ) ) {
								res2minim( basepairs[res1].rvspos() ) = true;
								++base_cnt;
							}
						}
					}
					if ( !res2minim(res2) ) {
						if ( ( is_protein( aa2 ) || is_nonnatural( aa2 ) ) && prot_mode != "none" ) {
							res2minim(res2) = true;
							++aa_cnt;
						} else if ( is_DNA( aa2 ) && dna_mode != "none" ) {
							res2minim(res2) = true;
							++base_cnt;
							// Basepairs have to be minimized together:
							if ( !res2minim(basepairs[res2].rvspos() ) ) {
								res2minim(basepairs[res2].rvspos() ) = true;
								++base_cnt;
							}
						}
					}
				}
				/*
				//debug
				std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: atrE = " << atrE << " [Watr=" << pack_wts.Watr() << "]\n";
				std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: repE = " << repE << " [Wrep=" << pack_wts.Wrep() << "]\n";
				std::cout << "res1/res2 [" << res1 << "/" << res2 << "]: solE = " << solE << " [Wsol=" << pack_wts.Wsol() << "]\n";
				*/

			}
		}
	}
	std::cout << "Bump check added " << base_cnt << " dna base(s) and " << aa_cnt << " amino acid(s) to minimize ..\n";

}

//////////////////////////////////////////////////////////////////////////////
/// @begin print_dihedral
///
/// @brief
/// aux.function - prints phi/psi/chi stored in template_pack + dna dihedrals stored in dna_variables
/// a global xyz array.
/// @detailed
///
/// @param	- [in/out]? -
///
/// @global_read
/// yes, from template_pack & dna_variables
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
print_dihedral(
	std::string const & mode,
	int const begin,
	int const end
)
{
	using namespace dna_variables;
	using namespace misc;
	using namespace pdb;
	using namespace param_aa;
	using namespace aaproperties_pack;

	//Check user-provided [begin,end]:
	if (begin<1 || begin>total_residue || end<begin || end<1 || end>total_residue) {
		std::cerr << "[" << begin << "," << end << "] printing range is illegal!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if (mode=="chi") {
		std::cout << "********** Protein chi angles **********\n";
		for ( int res1 = begin; res1 <= end; ++res1 ) {
			int aa = res(res1);
			if ( is_protein(aa) || is_nonnatural(aa) ) {
				std::cout << SS(aa_name3(aa) ) << "-" << pdb_res_num(res1) << " [" << res_chain(res1) << "]: chi = ";
				for ( int ang = 1, ange = nchi(aa,1); ang <= ange; ++ang ) {
					std::cout << F(7, 2, prot_chi_dihedrals(ang,res1)) << "  ";
				}
				std::cout << std::endl;
			}
		}
		std::cout << "****************************************\n";
	} else if ( mode == "phipsi" ) {
		std::cout << "********** Protein phi/psi angles **********\n";
		for ( int res1 = begin; res1 <= end; ++res1 ) {
			int aa = res(res1);
			if ( is_protein(aa) || is_nonnatural(aa) ) {
				std::cout << SS(aa_name3(aa) ) << "-" << pdb_res_num(res1) << " [" << res_chain(res1) << "]: ";
				std::cout << "phi=" << F(7, 2, template_pack::phi(res1)) << "  psi=" << F(7, 2, template_pack::psi(res1)) << std::endl;
			}
		}
		std::cout << "********************************************\n";
	} else if (mode=="dna_dih") {
		std::cout << "********** DNA dihedral angles *********\n";
		for ( int res1 = begin; res1 <= end; ++res1 ) {
			int aa = res(res1);
			if ( is_DNA(aa) ) {
				std::cout << SS(aa_name3(aa) ) << "-" << pdb_res_num(res1) << " [" << res_chain(res1) << "]: ";
				std::cout << "alpha=" << F(7, 2, dna_dihedrals(1,res1)) << " beta=" << F(7, 2, dna_dihedrals(2,res1))
									<< " gamma=" << F(7, 2, dna_dihedrals(3,res1)) << " delta=" << F(7, 2, dna_dihedrals(4,res1))
									<< " epsilon=" << F(7, 2, dna_dihedrals(5,res1)) << " xi=" << F(7, 2, dna_dihedrals(6,res1))
									<< " chi=" << F(7, 2, dna_dihedrals(7,res1)) << std::endl;
			}
		}
		std::cout << "****************************************\n";
	} else {
		std::cerr << "Unknown mode in print_dihedral: " << mode << "!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

