// -*- mode:c++;tab-width:1;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15655 $
//  $Date: 2007-06-26 10:16:38 -0700 (Tue, 26 Jun 2007) $
//  $Author: ashworth $

#ifndef INCLUDED_dna_classes
#define INCLUDED_dna_classes

#include <ObjexxFCL/FArray1Ds.fwd.hh>
#include <utility/basic_sys_util.hh> // utility::exit

#include <list>
#include <vector>
#include <string>
#include <iostream> // std::cout default argument

////////////////////////////////////////////////////////////////////////////////
// Small-scale classes for DNA modes
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// options struct, for reading DNA-design-related command line options
class DnaOptions {
	public:

		// bool options
		bool verbose;
		bool design_by_base;
		bool no_design;
		bool dna_minimize;
		bool dna_scan;
		bool revert;
		bool all_interface_stats;
		bool decoystats;
		bool ddG;
		bool smm_exhaustive;

		// bool options with parameters
		bool user_seq; std::string userseq;
		bool probe_specificity, normalize_Boltzmann; int num_packs;
		bool smm; float ms_offset;
		bool dna_bb_moves; int outer_runs;

		// parameters
		float dz_cutoff;
		int runindex_offset;
		float Boltz_temp;

		DnaOptions();
		~DnaOptions(){}
};

////////////////////////////////////////////////////////////////////////////////
//ja The following three classes (ResInfo, DnaPosInfo, DnaSeqInfo) are for keeping track of residue information in a more descriptive way than using generic containers

////////////////////////////////////////////////////////////////////////////////
// basic information about a residue (protein or DNA)
class ResInfo {
	public:
		int pos;
		int type;
		bool minimize;
		bool design;

		ResInfo():
			pos(0), type(0), minimize(false), design(false) {}
		ResInfo( int const p ):
			pos(p), type(0), minimize(false), design(false) {}
		ResInfo( int const p, int const t ):
			pos(p), type(t), minimize(false), design(false) {}
		ResInfo( int const p, int const t, bool const d ):
			pos(p), type(t), minimize(true), design(d) {}

		bool operator == ( const ResInfo & o ) const {
			return pos == o.pos && type == o.type;
		}
};

////////////////////////////////////////////////////////////////////////////////
// paired aggregate of DNA ResInfos ('rvs' is unused if unpaired)
// Nucleic acid chains are often basepaired.  Currently it is easier to store information about them as paired (or unpaired) positional units, than to model independent strands
class DnaPosInfo {
	private:
		ResInfo fwd;
		ResInfo rvs;
		bool paired_;

	public:
		DnaPosInfo(){}
		// unpaired
		DnaPosInfo( int posf, int typef ): fwd(posf,typef), paired_(false) {}
		// paired
		DnaPosInfo( int posf, int typef, int posr, int typer ):
		            fwd(posf,typef), rvs(posr,typer), paired_(true) {}

		void fail() { utility::exit( EXIT_FAILURE, __FILE__, __LINE__ ); }

		// unpaired
		void set( int posf, int typef ) {
			if ( paired_ ) fail(); fwd.pos = posf; fwd.type = typef;
		}
		void set_type( int typef ) {
			if ( paired_ ) fail(); fwd.type = typef;
		}

		// paired
		void set( int posf, int typef, int posr, int typer ) {
			if ( !paired_ ) fail();
			fwd.pos = posf; fwd.type = typef; rvs.pos = posr; rvs.type = typer;
		}
		void set_type( int typef, int typer ) {
			if ( !paired_ ) fail();
			fwd.type = typef; rvs.type = typer;
		}

		void set_design( bool const val ) {
			fwd.design = val; if ( paired_ ) rvs.design = val;
		}
		void set_minimize( bool const val ) {
			fwd.minimize = val; if ( paired_ ) rvs.minimize = val;
		}

		int fwdpos() const { return fwd.pos; }
		int fwdtype() const { return fwd.type; }
		int rvspos() const { return rvs.pos; }
		int rvstype() const { return rvs.type; }
		bool paired() const { return paired_; }
		bool design() const { return ( fwd.design || rvs.design ); }
		bool minimize() const { return ( fwd.minimize || rvs.minimize ); }

};

////////////////////////////////////////////////////////////////////////////////
// information about a DNA sequence
class DnaSeqInfo {
	public:
		std::vector< DnaPosInfo > dna;

		DnaSeqInfo(){};
		DnaSeqInfo( std::string const seqstr );

		// rosetta seqpos lookup
		DnaPosInfo & operator[] ( int const pos ); // NOT the dna vector index!!
		DnaPosInfo const & operator[] ( int const pos ) const;

		void push_back( DnaPosInfo bp ) { dna.push_back( bp ); }
		void update_from_res( ObjexxFCL::FArray1DB_int const & res );
		void clear() { dna.clear(); }

		// all below are const

		unsigned size() const { return dna.size(); }
		bool empty() const { return dna.size() == 0; }
		bool contains( int const pos ) const;
		bool contains( int const pos, int const na ) const;
		std::string const str() const;
		void print( std::ostream & out = std::cout ) const;

		// refer subsequences of those DnaPosInfo's for which 'design' is true
		std::vector<int> const design_positions() const;
		DnaSeqInfo design_set() const;
		DnaSeqInfo target_set( std::string const seqstr ) const;
		std::list< DnaSeqInfo >
		single_mutant_list( bool const verbose = false ) const;

		ObjexxFCL::FArray1D_int res_from_seq(
			ObjexxFCL::FArray1DB_int const & res
		) const;

		std::vector< DnaPosInfo >::const_iterator begin() const
			{ return dna.begin(); }

		std::vector< DnaPosInfo >::const_iterator end() const
			{ return dna.end(); }
};

////////////////////////////////////////////////////////////////////////////////
// pairs a string and a float, but is more descriptive than std::pair
// could be templated?
class NamedFloat {
	public:
		std::string name;
		float val;
		NamedFloat(): name(""), val(0.) {}
		NamedFloat( std::string namein, float valin ): name(namein), val(valin) {}
};

////////////////////////////////////////////////////////////////////////////////
// DnaNeighbor classes for storing interface vicinity and contact information
// this could be combined with ResInfo?
class DnaNeighbor {
	public:
	bool close, contact;
	DnaNeighbor(){ close = false; contact = false; }
	~DnaNeighbor(){}
};

////////////////////////////////////////////////////////////////////////////////
// Rosetta sequence indices start at 1, must shift for vector indices
class DnaNeighbors {
	private:
		std::vector< std::vector< DnaNeighbor > > neighbors;
	public:
		DnaNeighbors(){}
		~DnaNeighbors(){}
		DnaNeighbors( int const nres ) { dimension( nres ); }
		void dimension( int const nres );
		void clear() { neighbors.clear(); }
		bool empty() const { return neighbors.size() == 0; }

		DnaNeighbor & operator() ( int const pos1, int const pos2 )
			{ return neighbors[pos1-1][pos2-1]; }

		DnaNeighbor const & operator() ( int const pos1, int const pos2 ) const
			{ return neighbors[pos1-1][pos2-1]; }
};

#endif
