// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15540 $
//  $Date: 2007-06-19 18:37:31 -0400 (Tue, 19 Jun 2007) $
//  $Author: chu $

// Code to use known dna-protein structural motifs for design

// Rosetta Headers
#include "aa_name_conversion.h"
#include "after_opts.h"
#include "dna_motifs.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "dna_motifs_ns.h"
#include "dock_structure.h"
#include "files_paths.h"
#include "namespace_fullatom_flag.h"
#include "maps.h"
#include "misc.h"
#include "orient_rms.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "random_numbers.h"
#include "recover.h"
#include "RotamerSet.h"
#include "rotamer_functions.h"
#include "template_pack.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "score.h"
#include "pose.h"

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/IndexRange.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <fstream>
#include <iostream>
#include <string>
#include <vector>
#include <functional>
#include <algorithm>

float const deg2rad = { 3.14159265359/180.0 };

using namespace DNA_Motifs;

// Storage for motif info
std::vector< Motif * > DNA_MotifVector;

//*****************************************************
// Begin internal prototypes
//*****************************************************

// I don't want to include <vector> and <string> in dna_motifs.h,
// and these are purely internal - no other files call them

void ReadDNAMotifFile( std::string motif_file );

void AlignRegionBySingleResidueThruCb(
	FArray2Da_float fixed_coord,
	FArray3Da_float mobile_coords,
	int closest_pos,
	RegionPair &MoveRegion );

void
FindClosestPositionInRegion(
	FArray2Da_float rot_coord,
	FArray3Da_float region_coords,
	int &closest_pos,
	RegionPair &Region );

void
CopyCoordsResVariants(
	FArray3Da_float store_coords,
	FArray1Da_int store_res,
	FArray1Da_int store_res_variant,
	FArray3Da_float full_coord,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	int nres );

float GetRegionBumpEnergy(
	FArray3Da_float full_coord,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	int total_residue,
	RegionPair &SearchInfo,
	std::vector< RegionPair > &OmitPairVector );

// A functor for querying 'found' status in standard library
// algorithms

class RotFoundCheck : public std::unary_function< JJH_Rotamer, bool > {
		bool	_value;
	public:
		explicit RotFoundCheck( const bool _v_init ) : _value( _v_init ) {}
		bool operator() ( const JJH_Rotamer &rot ) const { return rot.found == _value; }
};

bool CheckForBump( JJH_Rotamer rot ) { return rot.found; }

//*****************************************************
// End internal prototypes
//*****************************************************

DNAMotifRotamerLibrary
GetDNAMotifRotamerLibrary(
	RegionPair & SearchInfo,
	RegionPairLibrary & ,
	RegionPairLibrary &DesignBaseVector,
	pose_ns::Pose const & pose )
{

	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using namespace fullatom_flag;
	using namespace DNA_Motifs;
	using namespace files_paths;

	full_atom = true;

	// Vector of Omit Regions
//	std::vector< RegionPair > OmitPairVector;

	// Vector of Bases to design against
//	std::vector< RegionPair > DesignBaseVector;

	// Search Region
//	RegionPair SearchInfo;

	// Vector for storing rotamer information
	unsigned int init_rot_size = 150000;
	std::vector< JJH_Rotamer > RotVector;
	RotVector.reserve( init_rot_size );


	// Read in the information you need to set up the
	// whole deal:
	//
	// First, a list of motifs to use,
	// Second, the bases you want to contact
	// Third, protein regions to omit from the bb search (loops)
	// Fourth, the protein region to use for rigid body search
	//
	// For now, the motifs are in a file called "motif_file" and the
	// other info is in a file called "motifs_region"

	// **********************************************
	// *** Begin Input of DNA Motif Information ***
	// **********************************************

	// Get the motif library

	ReadDNAMotifFile( "motif_file" );

	// Get the regions (search, dna target, omit ) from file

//	ReadDNARegionsFile( "motif_regions", SearchInfo, OmitPairVector,
//		DesignBaseVector );

	// **********************************************
	// *** End Input of DNA Motif Information ***
	// **********************************************

	// **********************************************
	// **********************************************
	// **********************************************

	// *** Use Alex Morozov's DNA base pair identifier

//	find_dna_chains( pose.total_residue(), pose.res(), pose.full_coord() );

	// *** Begin Construction of Inverse Rotamers ***

	// Setup the arrays in misc from the arrays in the pose
	dna_motifs_setup_template_pack( pose );

	// Assign each motif their id
	for ( int l = 0, el = DNA_MotifVector.size(); l < el; ++l ) {
		DNA_MotifVector[l]->id = l;
	}

	// Loop over bases of interest

	// Loop through base region vector, then through each region
	for ( int i = 0, e = DesignBaseVector.size(); i < e; ++i ) {
		for ( int spos = DesignBaseVector[i].start(), epos = DesignBaseVector[i].end();
					spos <= epos; ++spos ) {
			// Get the na_type for this base
			int this_type = pose.res( spos );
			if ( is_protein( this_type ) || is_nonnatural( this_type ) ) {
				std::cout << "Warning:  trying to use dna motif on amino acid" << std::endl;
				continue;
			}

			// For each, loop over motifs
			for ( int l = 0, el = DNA_MotifVector.size(); l < el; ++l ) {
				Motif * this_mot = DNA_MotifVector[l];

				// Check to see if the Motif should be applied
				if ( !this_mot->ApplyMotif( pose, this_type, spos ) ) {
					continue;
				}

				// Ok, here we are. Place the rotamers for this motif
				dna_motifs_place_rotamers( pose, spos, SearchInfo, RotVector, this_mot );
			}
		}
	}

	if ( RotVector.size() > init_rot_size ) {
		std::cout << "Warning! using " << RotVector.size() << " rotamers " << std::endl;
		std::cout << "but initial allocation was " << init_rot_size << std::endl;
		std::cout << "To avoid costly reallocation, increase init_rot_size in dna_motifs.cc " << std::endl;
	}

	std::cout << "Using " << RotVector.size() << " rotamers " << std::endl;


	// ***  End Construction of Inverse Rotamers  ***

	return RotVector;

}

void dna_motifs_setup_template_pack(
	pose_ns::Pose const & pose
)
{
	using namespace param;

	pose.copy_to_misc();

	for ( int i = 1; i <= pose.total_residue(); ++i ) {
	  template_pack::phi(i) = pose.phi(i);
	  template_pack::psi(i) = pose.psi(i);
	}

}

void dna_motifs_place_rotamers(
	pose_ns::Pose const & pose,
	int spos,
	DNA_Motifs::RegionPair & search_region,
	std::vector< JJH_Rotamer > & rot_vector,
	DNA_Motifs::Motif * motif_ptr )
{

	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace design;

	FArray3D_float const & fcoord ( pose.full_coord() );
	int const nres( pose.total_residue() );

// Use the middle-ish point of the search region for
// the build point - defines which bin of the bbdep
// library to use

	int use_pos = ( search_region.start() + search_region.end() ) / 2;

// Get the amino acid to build from the motif

	int this_aa = motif_ptr->aa_type();

// Use variant 1 - structures don't know where the protons are

	int this_aav = 1;

// Instantiate a RotamerSet
	RotamerSet rotamer_set;

	int nrotaa = 0;
	bool include_current = false;

// Disable extra rotamers
// jk Note: why isn't this actually being disabled, as the comment suggests??
	RotamerOptions saved_rotamer_options = active_rotamer_options;

// Get all the rotamers, skipping the bump check

	std::string const mode( "dnamotif" );

	PackerTask Task( pose );
	for( int ipos = 1, epos = MAX_RES() ; ipos <= epos ; ++ipos ) {
		Task.get_designmap().set( ipos, pose.res(ipos) );
	}

	rotamer_set.get_rotamers_seqpos_aa_aav( pose.total_residue(), pose.res(), pose.res_variant(),
    pose.full_coord(),this_aa, this_aav, use_pos, nrotaa, include_current, mode, Task );

// Reinstate extra rotamer settings
	active_rotamer_options = saved_rotamer_options;

// Place the rotamers relative to the base according to the
// motif-specific recipe

  FArray2D_float tmpactcoord( 3, 1 );
	FArray1D_int store_base( rotamer_set.nrotamers(), -1 );
	FArray1D_int store_base2( rotamer_set.nrotamers(), -1 );

	for ( int loop = 1, eloop = rotamer_set.nrotamers(); loop <= eloop; ++loop) {

		for ( int i = 1; i <= 3; ++i ) {
			tmpactcoord(i, 1) = rotamer_set.get_rotactcoord( i, loop );
		}

		int extra_build_pos(-1); ///jjh For motifs that touch two bases
		motif_ptr->PlaceRotamer( pose, this_aa, this_aav, spos,
			rotamer_set.get_rotcoord(loop), tmpactcoord( 1,1 ), extra_build_pos  );
		store_base( loop ) = spos;
		store_base2( loop ) = extra_build_pos;
	}

// Perform my own bump check, omitting the search region and
// the omit regions

	FArray1D_float dummyactcoord1( 3 );
	FArray1D_float dummyactcoord2( 3 );
	for ( int i = 1; i <= 3; ++i ) {
		dummyactcoord1(i) = 1.0;
		dummyactcoord2(i) = 3.0;
	}

	float atrE, repE, elecE, solvE, planeE, pairE;

	FArray1D_bool bump_remove( rotamer_set.nrotamers(), false);
	for ( int loop = 1, eloop = rotamer_set.nrotamers(); loop <= eloop; ++loop) {
		// Check bumps with DNA only
		float bump_total = 0;
		float bump_energy = 0;

		int close_pos = -1;
		float best_dist = 9999.0;
		float test_dist = 0.0;
		for ( int bres = 1; bres < nres; ++bres ) {
			distance_bk(rotamer_set.get_rotcoord(loop),fcoord(1,1,bres),test_dist);
			if ( test_dist < best_dist ) {
				close_pos = bres;
				best_dist = test_dist;
			}
		}

//		std::cout << "Closest pos is " << close_pos << std::endl;

		for ( int bres = 1; bres < nres; ++bres ) {

			int baa = pose.res( bres );

			if ( is_protein( baa ) || is_nonnatural( baa ) ) continue;

			// Careful - make sure it is the NA sidechain and the search
			// region's backbone
			get_sc_bbE(pose.res(bres),pose.res_variant(bres),this_aa,this_aav,
				fcoord(1,1,bres),
				rotamer_set.get_rotcoord(loop), bres,close_pos,solvE,atrE,repE,elecE);
			bump_energy = pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

			get_sc_scE(pose.res(bres),pose.res_variant(bres),fcoord(1,1,bres),
					this_aa,this_aav,rotamer_set.get_rotcoord(loop),
					dummyactcoord1,dummyactcoord2,bres,close_pos,
					solvE,atrE,repE,pairE,planeE,elecE);

			bump_energy += pack_wts.Watr()*atrE + pack_wts.Wrep()*repE;


			bump_total += bump_energy;
		}

		if ( bump_total > 2.0 ) bump_remove( loop ) = true;

	}

// Take everything that passes and put them in the RotVector


	for ( int loop = 1, eloop = rotamer_set.nrotamers(); loop <= eloop; ++loop ) {

		// Don't store rotamers that bump
		if ( bump_remove( loop ) ) continue;

		// Make a new JJH_Rotamer object from the existing Rotamer
		//		JJH_Rotamer this_rot = rotamer_set.get_rotamer( loop );
		JJH_Rotamer this_rot( rotamer_set.get_rotamer( loop ) );

		this_rot.motif_id = motif_ptr->id;
		this_rot.motif_pos = spos;
		this_rot.base_pos = store_base( loop );
		this_rot.base_pos2 = store_base2( loop );

		// Store it in the vector
		rot_vector.push_back( this_rot );

	}


}


void MoveRegionAwayFromPoint(
	DNA_Motifs::RegionPair & move_region,
	FArray1Da_float point,
	FArray3Da_float xyz,
	FArray1Da_int aa,
	FArray1Da_int aav
)
{
	using namespace param;
	using namespace aaproperties_pack;

	point.dimension( 3 );
	aa.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	FArray1D_float reg_ref( 3 );

	// Use the N atom - doesn't matter which, really
	reg_ref( 1 ) = xyz( 1, 1, move_region.start() );
	reg_ref( 2 ) = xyz( 2, 1, move_region.start() );
	reg_ref( 3 ) = xyz( 3, 1, move_region.start() );

	// Subtract off repel point
	reg_ref -= point;

	// Double space
	reg_ref *= 2.0;

	// Now add this to all atoms

	for ( int ires = move_region.start(), eres = move_region.end();
				ires <= eres; ++ires ) {
		for ( int atm = 1, eatm = natoms( aa( ires), aav( ires ) );
					atm <= eatm; ++atm ) {
			xyz( 1, atm, ires ) += reg_ref( 1 );
			xyz( 2, atm, ires ) += reg_ref( 2 );
			xyz( 3, atm, ires ) += reg_ref( 3 );
		}
	}

}

void MoveRegionRandomRigid(
	DNA_Motifs::RegionPair & move_region,
	FArray3Da_float xyz,
	FArray1Da_int aa,
	FArray1Da_int aav
)
{
	using namespace param;
	using namespace aaproperties_pack;
	using numeric::conversions::radians;

	aa.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	FArray1D_float origin_ref( 3 );

	// Pick a pivot point to move to origin

	// For now use rough COM

	for ( int k = 1; k <= 3; ++k ) {
		origin_ref( k ) = 0.5*( xyz( k, 1, move_region.start() ) +
		 xyz( k, 1, move_region.end() ) );
	}


	// Center on origin

	for ( int ires = move_region.start(), eres = move_region.end(); ires <= eres; ++ires ) {
		for ( int atm = 1, eatm = natoms( aa( ires), aav( ires ) ); atm <= eatm; ++atm ) {
			xyz( 1, atm, ires ) -= origin_ref( 1 );
			xyz( 2, atm, ires ) -= origin_ref( 2 );
			xyz( 3, atm, ires ) -= origin_ref( 3 );
		}
	}

	// Get random rigid body move

	FArray1D_float trans( 3 );
	FArray1D_float angle( 3 );
	FArray2D_float matrix( 3, 3 );

	float trans_range = 1.0;
	float angle_range = radians( 10.0 );

	trans( 1 ) = ( 2.0*trans_range * ran3() - trans_range );
	trans( 2 ) = ( 2.0*trans_range * ran3() - trans_range );
	trans( 3 ) = ( 2.0*trans_range * ran3() - trans_range );
	angle( 1 ) = ( 2.0*angle_range * ran3() - angle_range );
	angle( 2 ) = ( 2.0*angle_range * ran3() - angle_range );
	angle( 3 ) = ( 2.0*angle_range * ran3() - angle_range );

	Matrix_From_Euler_angles( angle, matrix );

	// Apply the rotations

	for ( int ires = move_region.start(), eres = move_region.end(); ires <= eres; ++ires ) {
		for ( int atm = 1, eatm = natoms( aa( ires), aav( ires ) ); atm <= eatm; ++atm ) {
			move_bk( xyz( 1, atm, ires ), matrix, trans );
		}
	}

	// Apply the translation and move back to origin

	for ( int ires = move_region.start(), eres = move_region.end(); ires <= eres; ++ires ) {
		for ( int atm = 1, eatm = natoms( aa( ires), aav( ires ) ); atm <= eatm; ++atm ) {
			xyz( 1, atm, ires ) += origin_ref( 1 );
			xyz( 2, atm, ires ) += origin_ref( 2 );
			xyz( 3, atm, ires ) += origin_ref( 3 );
		}
	}

	// Move back from origin

}

int CountInverseRotamers(
	float cutoff,
	FArray3Da_float xyz,
	DNA_Motifs::RegionPair & search_region,
	std::vector< JJH_Rotamer > & rot_vector
)
{
	using namespace param;

	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );

	int count = 0;

	for ( int irot = 0, erot = rot_vector.size(); irot < erot; ++irot )
		rot_vector[irot].found = false;

	// Loop search region
	for ( int ires = search_region.start(), eres = search_region.end();
				ires <= eres; ++ires ) {
		// For each, loop inverse rotamer vector
		for ( int irot = 0, erot = rot_vector.size(); irot < erot; ++irot ) {

			// Calculate rmsd

			float this_rmsd = 0.0;
			// Use rmsd for first three atms ( N, CA, C )
			for ( int atm = 1, eatm = 3; atm <= eatm; ++atm ) {
				for ( int k =  1; k <= 3; ++k ) {
					float diff = xyz( k, atm, ires ) - rot_vector[irot].get_rotcoord( k, atm );
					this_rmsd += diff*diff;
				}
			}

			this_rmsd /= 9.0;
			this_rmsd = std::sqrt( this_rmsd );

			// Increment if passes
			if ( this_rmsd < cutoff ) {
				rot_vector[irot].found = true;
				++count;
			}
		}
	}

return count;
}

int
FillInverseRotamers(
	FArray3Da_float xyz,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	std::vector< JJH_Rotamer > & rot_vector
)
{
	using namespace param;

	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	res.dimension( MAX_RES() );
	res_variant.dimension( MAX_RES() );

	int count = 0;

	// Loop all rotamers
	for ( int irot = 0, erot = rot_vector.size();
				irot < erot; ++irot ) {

		if ( !(rot_vector[irot].found) ) continue;

		// Fill in
		++count;

		res( count ) = rot_vector[irot].aa();
		res_variant( count ) = 1;
		for ( int atm = 1, eatm = MAX_ATOM(); atm <= eatm; ++atm ) {
			for ( int k = 1; k <= 3; ++k ) {
				xyz( k, atm, count ) = rot_vector[irot].get_rotcoord( k, atm );
			}
		}
	}

	std::cout << "Filled in " << count << std::endl;

	return count;
}

int
CheckInverseRotamers(
	FArray3Da_float xyz,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	std::vector< JJH_Rotamer > & rot_vector
)
{
	using namespace param;

	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	res.dimension( MAX_RES() );
	res_variant.dimension( MAX_RES() );

	int count = 0;

	// Loop all rotamers
	for ( int irot = 0, erot = rot_vector.size();
				irot < erot; ++irot ) {

		if ( !(rot_vector[irot].found) ) continue;

		// Fill in
		++count;

	}

	//std::cout << "Filled in " << count << std::endl;

	return count;
}

void
res_bump_check(
	int & aa,
	int & aav,
	int & res,
	FArray2Da_float coord,
	FArray3Da_float xyz,
	FArray1Da_int aan,
	FArray1Da_int aa_variant,
	FArray1Da_bool omit_res,
	int & nres,
	float & bumpenergy
)
{
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	coord.dimension( 3, MAX_ATOM() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	aan.dimension( MAX_RES() );
	aa_variant.dimension( MAX_RES() );
	omit_res.dimension( MAX_RES() );

//bk local variables
	int bbres,i;
	FArray1D_float dummyactcoord1( 3 );
	FArray1D_float dummyactcoord2( 3 );
	float solvE, atrE, repE, elecE, pairE, planeE, dis;
	int bbaa;

	bumpenergy = 0.0;
	for ( bbres = 1; bbres <= nres; ++bbres ) { // loop through backbone positions

		if ( omit_res(bbres) ) continue;
		if ( res == bbres ) continue;

		distance_bk(coord(1,1),xyz(1,2,bbres),dis);
		if ( dis > 18 ) continue;

		for ( i = 1; i <= 3; ++i ) {
			dummyactcoord1(i) = 1.0;
			dummyactcoord2(i) = 3.0;
		}

		bbaa = aan( bbres );

		// Only the search region interacts - with other
		// protein residues, it interacts with the bb.  With nucleic acids,
		// it interacts with all of the residue - but, we have for better or
		// worse tagged the entire n.a. residues as sidechain, so we can use
		// that for the interaction.
		if ( is_protein( bbaa ) || is_nonnatural( bbaa ) ) {
			get_bb_bbE_P( bbres, res, aan(bbres), aa, aa_variant(bbres), aav,
				xyz(1,1,bbres),coord, repE,atrE,solvE,elecE);
			bumpenergy += pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

			get_sc_bbE( aa, aav, aan(bbres), aa_variant(bbres), coord, xyz(1,1,bbres),
				res, bbres, solvE,atrE,repE,elecE);
			bumpenergy += pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

		} else {
			// Careful - make sure it is the NA sidechain and the search
			// region's backbone
			get_sc_bbE(aan(bbres),aa_variant(bbres),aa,aav,xyz(1,1,bbres),
				coord, bbres,res,solvE,atrE,repE,elecE);
			bumpenergy += pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

			get_sc_scE(aan(bbres),aa_variant(bbres),xyz(1,1,bbres),aa,aav,
				coord, dummyactcoord1, dummyactcoord2, bbres,res,solvE,atrE,repE,
				pairE, planeE, elecE);
			bumpenergy += pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

		}

	}               // end looping through backbone positions

}

void
score_dna_motif_inv_rotamers(
	DNA_Motifs::DNAMotifRotamerLibrary & inv_rot_lib,
	DNA_Motifs::RegionPairLibrary & target_regions
)
{
	using namespace misc;
	using namespace param;
	using namespace template_pack;
	using namespace param_aa;
	using namespace param_pack;
	using namespace hbonds;

///jjh The purpose of this function is to calculate and store
///jjh an energy for each inverse rotamer that consists of its
///jjh energetic interaction with the target DNA region as well
///jjh as its intrinsic energy.  The eventual use is to determine
///jjh which inv. rotamers are to be preferred energetically when
///jjh optimizing backbone position.

//jjh local variables
	int bbres,i;
	FArray1D_float dummyactcoord1( 3 );
	FArray1D_float dummyactcoord2( 3 );
	float solvE, atrE, repE, elecE, pairE, planeE, dis;
	float srbbhbE,lrbbhbE,schbE,sc_bbhbE;

	///jjh Neighbor info is needed for scoring
	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);


	for ( int irot = 0, erot = inv_rot_lib.size();
				irot < erot; ++irot ) {

		int thisaa = inv_rot_lib[irot].aa();
		int thisaav = inv_rot_lib[irot].aav();
		int thisres = inv_rot_lib[irot].seqpos();

		float rot_energy = 0.0;

		///jjh First get instrinsics

		///jjh Reference energy
//		rot_energy += -pack_wts.Waa(thisaa);

		///jjh Phi/Psi (one) energy
//		float tempE, tmp1, tmp2;
//		get_Paa_ppE( phi(thisres), psi(thisres), thisaa, tempE, tmp1, tmp2 );
//		rot_energy += pack_wts.Wone()*tempE;

		///jjh Dunbrack score
//		float dunE = -std::log( inv_rot_lib[irot].rperc );
//		rot_energy += pack_wts.Wdun()*dunE;

		for ( int rp = 0, end_rp = target_regions.size() ; rp <= end_rp; ++rp ) {
			int start_res = target_regions[rp].start();
			int end_res = target_regions[rp].end();
			for ( bbres = start_res; bbres <= end_res; ++bbres ) {
				// loop through target DNA regions positions

				distance_bk(inv_rot_lib[irot].get_rotcoord()(1,2),full_coord(1,2,bbres),dis);
				if ( dis > 18 ) continue;

				for ( i = 1; i <= 3; ++i ) {
					dummyactcoord1(i) = 1.0;
					dummyactcoord2(i) = 3.0;
				}

				int bbaa = res( bbres );
				int bbaav = res_variant( bbres );

				///jjh Ok, we want the inverse rotamer side chain's interactions
				///jjh with the nucleic acid (which is all side chain) via
				///jjh the Lennard-Jones, hydrogen bond, and Generalized Born
				///jjh potentials.

				get_sc_scE(bbaa,bbaav,full_coord(1,1,bbres),thisaa,thisaav,
					inv_rot_lib[irot].get_rotcoord(), dummyactcoord1, dummyactcoord2,
					bbres,thisres,solvE,atrE,repE, pairE, planeE, elecE);

				rot_energy += pack_wts.Watr()*atrE+pack_wts.Wrep()*repE;

				float & xyz_11s( full_coord(1,1,bbres) );

				get_hbE(true,thisaa,bbaa,thisaav,bbaav,thisres,bbres,neighbors(thisres),
					neighbors(bbres),inv_rot_lib[irot].get_rotcoord(),xyz_11s,schbE,srbbhbE,
					lrbbhbE,sc_bbhbE);

				rot_energy +=	pack_wts.Whbond_sc() * schbE + pack_wts.Whbond_bb() * ( srbbhbE + lrbbhbE ) +
											pack_wts.Whbond_bb_sc() * sc_bbhbE;

			}               // end looping through backbone positions
		}               // end looping through target region pairs

		inv_rot_lib[irot].energy = rot_energy;
//		std::cout << "Energy for irot " << irot << " is " << rot_energy << std::endl;

	} // end of loop through inverse rotamer library

}

void
find_closest_to_dna_motif_inv_rotamers(
	pose_ns::Pose & pose,
	DNA_Motifs::DNAMotifRotamerLibrary & inv_rot_lib,
	DNA_Motifs::RegionPair & target_region
)
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

//jjh local variables
	int bbres;

	float remove_cutoff = 4.0 * DNA_Motifs::Motifs_close_enough;
	FArray3D_float const & fcoord ( pose.full_coord() );

	for ( int irot = 0, erot = inv_rot_lib.size();
				irot < erot; ++irot ) {

		int thisaa = inv_rot_lib[irot].aa();
//		int thisaav = inv_rot_lib[irot].aav();

		inv_rot_lib[irot].found = false;

		float closest_rmsd( 9999.0 );
		int closest_pos( -1 );

			int start_res = target_region.start();
			int end_res = target_region.end();
			for ( bbres = start_res; bbres <= end_res; ++bbres ) {
				// loop through target DNA regions positions

				int bbaa = pose.res( bbres );
//				int bbaav = pose.res_variant( bbres );

				float rmsd = 0.0;
				float tmp_rmsd;

				int num_atoms = 0;
				distance2_bk( inv_rot_lib[irot].get_rotcoord()(1,1), fcoord(1,1,bbres), tmp_rmsd );
				rmsd += tmp_rmsd;
				num_atoms++;

				distance2_bk( inv_rot_lib[irot].get_rotcoord()(1,2), fcoord(1,2,bbres), tmp_rmsd );
				rmsd += tmp_rmsd;
				num_atoms++;

				distance2_bk( inv_rot_lib[irot].get_rotcoord()(1,3), fcoord(1,3,bbres), tmp_rmsd );
				rmsd += tmp_rmsd;
				num_atoms++;

				int iat1 = HApos( thisaa, 1 );
				int iat2 = HApos( bbaa, 1 );

				distance2_bk( inv_rot_lib[irot].get_rotcoord()(1,iat1), fcoord(1,iat2,bbres), tmp_rmsd );
				rmsd += tmp_rmsd;
				num_atoms++;

				rmsd /= ( float( num_atoms ) );
				rmsd = std::sqrt( rmsd );

				if( rmsd < closest_rmsd ) {
					closest_rmsd = rmsd;
					closest_pos = bbres;
				}


			}               // end looping through backbone positions
		inv_rot_lib[irot].set_seqpos(closest_pos);
//		std::cout << "Rotamer " << irot << " closest rmsd is " << closest_rmsd << " with " << closest_pos << std::endl;
		if( closest_rmsd <= remove_cutoff ) {
			inv_rot_lib[irot].found = true;
		}
	} // end of loop through inverse rotamer library

///jjh Sort by the closest position

sort( inv_rot_lib.begin(), inv_rot_lib.end() );

///jjh delete those not found
inv_rot_lib.erase(
  remove_if( inv_rot_lib.begin(), inv_rot_lib.end(), RotFoundCheck( false ) ),
  inv_rot_lib.end() );

std::cout << "Rotamers trimmed to include only those " << remove_cutoff << " Ang from native BB" << std::endl;
std::cout << "Using " << inv_rot_lib.size() << " rotamers " << std::endl;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin align_rotamer
///
/// @brief  move the coordinates and sequence for a new rotamer onto the
///					backbone
///
/// @authors Jim Havranek
/// @last_modified May 3, 2005
////////////////////////////////////////////////////////////////////////////////
void
align_rotamer(
  int const seqpos,
  int const aa,
  int const aav,
  FArray2Da_float coord
)
{
  using namespace param;
  using namespace misc;
  using namespace aaproperties_pack;

  coord.dimension( 3, MAX_ATOM() );

// Get the transformation to move the rotamer onto the backbone

	FArray2D_double rc( 3, 4, 0.0 );
	FArray2D_double hc( 3, 4, 0.0 );
	FArray1D_double wghts( 4, 1.0 );
	FArray2D_double uu( 3, 3, 0.0 );
	FArray1D_double xx( 3, 0.0 );
	FArray1D_double yy( 3, 0.0 );
	double sigma3;

	int num_align = 0;

	for( int ci = 1 ; ci <= 3 ; ci++) {
		if( ci == 4 ) continue;
		num_align++;
		for( int cj = 1 ; cj <= 3 ; cj++) {
			hc( cj, num_align ) = full_coord( cj, ci, seqpos );
			rc( cj, num_align ) = coord( cj, ci );
		}
	}

	int iat1 = HApos( res(seqpos), res_variant(seqpos) );
	int iat2 = HApos( aa, aav );
	num_align++;
	for( int cj = 1 ; cj <= 3 ; cj++) {
		hc( cj, num_align ) = full_coord( cj, iat1, seqpos );
		rc( cj, num_align ) = coord( cj, iat2 );
	}


	findUU_trans( rc, hc, wghts, num_align, uu, sigma3, xx, yy );

// Align the rotamer to the backbone
	UU_rotate( coord( 1, 1 ), MAX_ATOM(), xx, yy, uu );

// Copy it all over

  res(seqpos) = aa;
  res_variant(seqpos) = aav;

  for ( int i = 1; i <= 3; ++i ) {
    for ( int atm = 1; atm <= natoms(aa,aav); ++atm) {
      full_coord(i,atm,seqpos) = coord(i,atm);
    }
  }

	///jjh Force Epos to reflect full_coord
	for ( int i = 1; i <= 3; ++i ) {
		Eposition(i,2,seqpos) = full_coord(i,2,seqpos);
	}

  update_sequence();
  maps_set_new_rotamer(seqpos);
	score12();
}

void
ReadDNAMotifFile( std::string motif_file )
{

	// Open the motif file

	std::ifstream motifs_in;
	motifs_in.open( motif_file.c_str() );
	if( !motifs_in ) {
		std::cout << "Fatal error:  no motif_file" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Parse it for data
	std::string key_in;
	while( motifs_in >> key_in ) {
		if( key_in == "single" ) {
			SingleMotif *ptr = new SingleMotif( motifs_in );
			// This memory is never freed, but its doesn't need to be
			DNA_MotifVector.push_back( ptr );
		} else if( key_in == "stacked" ) {
			StackedMotif *ptr = new StackedMotif( motifs_in );
			DNA_MotifVector.push_back( ptr );
		} else if( key_in == "diagonal" ) {
			DiagonalMotif *ptr = new DiagonalMotif( motifs_in );
			DNA_MotifVector.push_back( ptr );
		} else if( key_in == "double" ) {
		// TO BE CODED ONCE I HAVE DOUBLE MOTIFS
		}
	}

	std::cout << "Found " << DNA_MotifVector.size() << " Motifs" << std::endl;

	// Close it

	motifs_in.close();
	motifs_in.clear();

	return;
}

int
find_inv_rot_overlaps(
  DNA_Motifs::DNAMotifRotamerLibrary & inv_rot_lib,
  DNA_Motifs::RegionPair & loop_region )
{
	using namespace param;
	using namespace misc;

	FArray1D_float best_match( MAX_RES()(), 9999.0 );
	FArray1D_int best_match_id( MAX_RES()(), -1 );

	for( int irot = 0, erot = inv_rot_lib.size() ; irot < erot ; ++irot ) {
		inv_rot_lib[irot].set_seqpos(0);
		inv_rot_lib[irot].found = false;
		for( int ires = loop_region.start(), eres = loop_region.end() ;
					ires <= eres ; ++ires ) {

//			std::cout << "Checking rotamer " << irot << " with " << ires << std::endl;
			float rmsd;
				if( do_bb_atoms_overlap( full_coord( 1, 1, ires), res( ires ),
						inv_rot_lib[irot].get_rotcoord(), inv_rot_lib[irot].aa(), rmsd ) ) {

				float test_rmsd = std::sqrt( rmsd );

				float test_energy = inv_rot_lib[irot].energy * ( 1.0 - test_rmsd ) *
																											( 1.0 - test_rmsd );

				if( test_energy > best_match( ires ) ) continue;

				best_match( ires ) = test_energy;
				if( best_match_id( ires ) >= 0 ) {
					inv_rot_lib[ best_match_id( ires ) ].found = false;
				}
				best_match_id( ires ) = irot;

				inv_rot_lib[irot].set_seqpos(ires);
				inv_rot_lib[irot].found = true;
				inv_rot_lib[irot].rmsd = test_rmsd;
			}
		}
	}

	int hit_count = 0;
	for( int irot = 0, erot = inv_rot_lib.size() ; irot < erot ; ++irot ) {
		if( !inv_rot_lib[irot].found ) continue;
				++hit_count;

        std::cout << "Rotamer " << irot << " match at protein position " <<
            inv_rot_lib[irot].seqpos() << " aa " <<
            param_aa::aa_name3( inv_rot_lib[irot].aa() ) << " touches " <<
            inv_rot_lib[irot].base_pos;
        if( inv_rot_lib[irot].base_pos2 > 0 ) {
          std::cout << " and " << inv_rot_lib[irot].base_pos2;
        }
        std::cout << " with rmsd " << inv_rot_lib[irot].rmsd;
        std::cout << std::endl;

	}

	return hit_count;

}

void
mutate_region_to_aa(
	pose_ns::Pose & pose,
  DNA_Motifs::RegionPair & loop_region,
	int aa_code,
	std::vector<int> & keep_the_same )
{
	using namespace param;

	FArray3D_float const & fcoord ( pose.full_coord() );

	for ( int ires = loop_region.start(), eres = loop_region.end()
					; ires <= eres; ++ires ) {

		if( find( keep_the_same.begin(), keep_the_same.end(), pose.res( ires ) ) !=
				keep_the_same.end() ) continue;

		std::cout << "Mutating position " << ires << std::endl;

		FArray2D_float temp_copy( 3, MAX_ATOM()() );
		FArray1D_float temp_rchi( MAX_CHI, 0.0 );

		get_rot_coord( fcoord, aa_code, 1, temp_rchi, temp_copy, ires );

		pose.copy_sidechain( ires, aa_code, 1, temp_copy );

	}
	return;
}

void
mutate_vector_to_aa(
	pose_ns::Pose & pose,
  std::vector<int> & positions,
	int aa_code )
{
	using namespace param;

	FArray3D_float const & fcoord ( pose.full_coord() );

	for ( int ivec = 0, evec = positions.size()
					; ivec < evec; ++ivec ) {

		int ires = positions[ivec];

		std::cout << "Mutating position " << ires << std::endl;

		FArray2D_float temp_copy( 3, MAX_ATOM()() );
		FArray1D_float temp_rchi( MAX_CHI, 0.0 );

		get_rot_coord( fcoord, aa_code, 1, temp_rchi, temp_copy, ires );

		pose.copy_sidechain( ires, aa_code, 1, temp_copy );

	}
	return;
}

void
store_loop_region_coords(
	DNA_Motifs::RegionPair & loop_region,
	FArray3Da_float stored_coords )
{
	using namespace param;
	using namespace misc;

	SRange R( loop_region.start(), loop_region.end() );

	stored_coords.dimension( 3, MAX_ATOM(), R );

	for ( int ires = loop_region.start(), eres = loop_region.end(),
					max_atom = MAX_ATOM(); ires <= eres; ++ires ) {
		for ( int ia = 1; ia <= max_atom; ++ia ) {
			for ( int ic = 1; ic <= 3; ++ic ) {
				stored_coords( ic, ia, ires ) = full_coord( ic, ia, ires );
			}
		}
	}
	return;
}


void
recover_loop_region_coords(
	DNA_Motifs::RegionPair & loop_region,
	FArray3Da_float stored_coords )
{
	using namespace param;
	using namespace misc;

	SRange R( loop_region.start(), loop_region.end() );

	stored_coords.dimension( 3, MAX_ATOM(), R );

	for ( int ires = loop_region.start(), eres = loop_region.end(),
					max_atom = MAX_ATOM(); ires <= eres; ++ires ) {
		for ( int ia = 1; ia <= max_atom; ++ia ) {
			for ( int ic = 1; ic <= 3; ++ic ) {
				full_coord( ic, ia, ires ) = stored_coords( ic, ia, ires );
			}
		}
	}
	return;
}


bool
do_bb_atoms_overlap(
  FArray2Da_float xyz1,
	int aa1,
  FArray2Da_float xyz2,
	int aa2,
	float & rmsd )
{
	using namespace param;
	using namespace aaproperties_pack;

	xyz1.dimension( 3, MAX_ATOM() );
	xyz2.dimension( 3, MAX_ATOM() );

	rmsd = 0.0;
	float tmp_rmsd;

	int num_atoms = 0;
	distance2_bk( xyz1(1,1), xyz2(1,1), tmp_rmsd );
	rmsd += tmp_rmsd;
	num_atoms++;

	distance2_bk( xyz1(1,2), xyz2(1,2), tmp_rmsd );
	rmsd += tmp_rmsd;
	num_atoms++;

	distance2_bk( xyz1(1,3), xyz2(1,3), tmp_rmsd );
	rmsd += tmp_rmsd;
	num_atoms++;

	int iat1 = HApos( aa1, 1 );
	int iat2 = HApos( aa2, 1 );

	distance2_bk( xyz1(1,iat1), xyz2(1,iat2), tmp_rmsd );
	rmsd += tmp_rmsd;
	num_atoms++;

	rmsd /= float( num_atoms );

	float close_enough = DNA_Motifs::Motifs_close_enough;

	float close_check = close_enough*close_enough;

//	std::cout << " rmsd squared is " << rmsd << std::endl;

	if( rmsd <= close_check ) return true;

	return false;

}


void
ReadDNARegionsFile( std::string regions_file,
	RegionPair &SearchInfo,
	std::vector< RegionPair > &OmitPairVector,
	std::vector< RegionPair > &DesignBaseVector,
	std::vector< int > &shave_positions )
{
	std::ifstream motifs_in;
	std::string key_in;

	shave_positions.clear();

	float fdummy;
	realafteroption("motifs_close_enough", 1.2, fdummy);
	DNA_Motifs::Motifs_close_enough = fdummy;

	motifs_in.open( regions_file.c_str() );
	if( !motifs_in ) {
		std::cout << "Fatal error:  no motif_file" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Parse it for data

	while( motifs_in >> key_in ) {
		if( key_in == "search" ) {
			SearchInfo = RegionPair( motifs_in );
		} else if( key_in == "omit" ) {
			RegionPair SingleOmit( motifs_in );
			OmitPairVector.push_back( SingleOmit );
		} else if( key_in == "bases" ) {
			RegionPair SingleBase( motifs_in );
			DesignBaseVector.push_back( SingleBase );
		} else if( key_in == "shave" ) {
			int shave_pos( 0 );
			motifs_in >> shave_pos;
			shave_positions.push_back( shave_pos );
		}
}

	std::cout << "Search region is from " << SearchInfo.start() << " to " <<
								SearchInfo.end() << std::endl;
	std::cout << "Found " << OmitPairVector.size() << " Omit Regions" << std::endl;
	std::cout << "Found " << DesignBaseVector.size() << " Base Regions" << std::endl;

	// Close it

	motifs_in.close();
	motifs_in.clear();

	return;
}

void
reset_dna_match_info() {
	using namespace DNA_Motifs;
	dna_match_regions.clear();
	return;
}

void
set_dna_match_library( DNA_Motifs::DNAMotifRotamerLibrary * inv_rot_lib ) {
	using namespace DNA_Motifs;
	dna_match_library = inv_rot_lib;
	return;
}

void
add_dna_match_region( DNA_Motifs::RegionPair & loop_region ) {
	using namespace DNA_Motifs;
	dna_match_regions.push_back( loop_region );
	return;
}

float
get_dna_match_score() {

  using namespace param;
  using namespace misc;
	using namespace DNA_Motifs;

	float ret_score = 0.0;

	if( dna_match_library == NULL ) {
		std::cout << "WARNING:  Attempt to score dna match without library" <<
			std::endl;
		return 0.0;
	}
	DNAMotifRotamerLibrary & inv_rot_lib = (*dna_match_library);

	FArray1D_float best_match( MAX_RES()(), 9999.0 );
	FArray1D_int best_match_id( MAX_RES()(), -1 );

  for( int irot = 0, erot = inv_rot_lib.size() ; irot < erot ; ++irot ) {
    inv_rot_lib[irot].set_seqpos(0);
    inv_rot_lib[irot].found = false;
		for( int ireg = 0, ereg = dna_match_regions.size() ; ireg < ereg ; ++ireg ) {
    	for( int ires = dna_match_regions[ireg].start(),
								eres = dna_match_regions[ireg].end() ; ires <= eres ; ++ires ) {

//   	   std::cout << "Checking rotamer " << irot << " with " << ires << std::endl;
				float rmsd;
				if( do_bb_atoms_overlap( full_coord( 1, 1, ires), res( ires ),
						inv_rot_lib[irot].get_rotcoord(), inv_rot_lib[irot].aa(), rmsd ) ) {

					float test_rmsd = std::sqrt( rmsd );
					float test_energy = inv_rot_lib[irot].energy * ( 1.0 - test_rmsd ) * (1.0 - test_rmsd );
					if( test_energy > best_match( ires ) ) continue;
					best_match( ires ) = test_energy;
					if( best_match_id( ires ) >= 0 ) {
						inv_rot_lib[ best_match_id( ires ) ].found = false;
					}
					best_match_id( ires ) = irot;

					inv_rot_lib[irot].set_seqpos(ires);
					inv_rot_lib[irot].found = true;
					inv_rot_lib[irot].rmsd  = test_rmsd;
//					ret_score += inv_rot_lib[irot].energy * ( 1.0 - rmsd ) * (1.0 - rmsd );
//					std::cout << "DNA match rmsd " << rmsd << std::endl;
				}
			}
    }
  }

	int hits = 0;

	///jjh only sum up the best
  for( int irot = 0, erot = inv_rot_lib.size() ; irot < erot ; ++irot ) {
		if( inv_rot_lib[irot].found ) {
					hits++;
					ret_score += inv_rot_lib[irot].energy * ( 1.0 - inv_rot_lib[irot].rmsd ) *
					( 1.0 - inv_rot_lib[irot].rmsd );
		}
	}

//	std::cout << "There are " << hits << " hits" << std::endl;
	num_matches = hits;

	if( ret_score != 0.0 ) {
//		std::cout << "DNA Match score = " << ret_score << std::endl;
	}
 	return ret_score;
}

float
get_motif_tether_score() {

  using namespace param;
  using namespace misc;
	using namespace DNA_Motifs;

	float ret_score = 0.0;
	float dis2;

	int emot = motif_set.size();
	if( emot == 0 ) return ret_score;

	for( int imot = 0 ; imot < emot ; ++imot ){
		int pos = motif_set[imot].res_num;
		for( int atm = 1 ; atm <= 3; ++atm ) {
			int this_atm = motif_set[imot].atom_num(atm);
			distance2_bk(full_coord( 1, this_atm, pos ),
									motif_set[imot].coords( 1, atm ), dis2 );
			ret_score += dis2;
		}
	}

//	static int mcounter(0);
//	if( !(mcounter++ % 10) )
//		std::cout << "Motif tether score is " << ret_score << std::endl;

 	return ret_score;
}


void
clear_motif_set()
{
using namespace DNA_Motifs;

motif_set.clear();

}

void
add_motif_to_set(
	int motif_id,
	int res_pos,
	FArray2Da_float coord
)
{

using namespace param;
using namespace misc;
using namespace DNA_Motifs;

coord.dimension( 3, MAX_ATOM() );

MotifTether new_mt;

new_mt.res_num = res_pos;

new_mt.atom_num(1) = DNA_MotifVector[motif_id]->aa_atm1;
new_mt.atom_num(2) = DNA_MotifVector[motif_id]->aa_atm2;
new_mt.atom_num(3) = DNA_MotifVector[motif_id]->aa_atm3;

for( int i = 1 ; i <= 3 ; ++i ) {
	for( int j = 1 ; j <= 3 ; ++j ) {
		new_mt.coords( j, i ) = coord( j, new_mt.atom_num(i) );
	}
}

std::cout << "Adding motif constraint " << new_mt.coords( 1,1) << " " << new_mt.coords( 2, 1) << " " <<
	new_mt.coords( 3,1 ) << std::endl;
std::cout << "Adding motif constraint " << new_mt.coords( 1,2) << " " << new_mt.coords( 2, 2) << " " <<
	new_mt.coords( 3,2 ) << std::endl;
std::cout << "Adding motif constraint " << new_mt.coords( 1,3) << " " << new_mt.coords( 2, 3) << " " <<
	new_mt.coords( 3,3 ) << std::endl;

motif_set.push_back( new_mt );

}

bool
cond_fetch_motif_set_coords(
	int res_pos,
	FArray1Da_int atoms,
	FArray2Da_float coord
)
{

using namespace param;
using namespace misc;
using namespace DNA_Motifs;

atoms.dimension( 3 );
coord.dimension( 3, 3 );

if( motif_set.size() == 0 ) return false;

for( int i = 0, ei = motif_set.size() ; i < ei; ++i ) {
	if( motif_set[i].res_num == res_pos ) {
		for( int a = 1 ; a <= 3 ; ++a ) {
			atoms(a) = motif_set[i].atom_num(a);
			for( int k = 1 ; k <= 3 ; ++k ) {
				coord(k, a) = motif_set[i].coords(k,a);
			}
		}
		return true;
	}
}

return false;

}

void
FetchAtomsByMotif(
	int m_id,
	int & atm1,
	int & atm2,
	int & atm3
) {

		atm1 = DNA_MotifVector[m_id]->aa_atm1;
		atm2 = DNA_MotifVector[m_id]->aa_atm2;
		atm3 = DNA_MotifVector[m_id]->aa_atm3;

		return;

	}

void
set_motif_sidechain_cst(
	cst_set_ns::Cst_set & cst_set,
	JJH_Rotamer & inv_rot,
	int bb_pos
)
{

	std::cout << "Setting motif cst at position " << bb_pos << std::endl;

	int aa1(-1);
	int aa2(-1);
	int aa3(-1);
	int m_id = inv_rot.motif_id;

	FetchAtomsByMotif( m_id, aa1, aa2, aa3 );

	kin::Atom_id const atm1( aa1, bb_pos );
	kin::Atom_id const atm2( aa2, bb_pos );
	kin::Atom_id const atm3( aa3, bb_pos );

	numeric::xyzVector_float xyz1( &(inv_rot.get_rotcoord()( 1, aa1 ) ) );
	numeric::xyzVector_float xyz2( &(inv_rot.get_rotcoord()( 1, aa2 ) ) );
	numeric::xyzVector_float xyz3( &(inv_rot.get_rotcoord()( 1, aa3 ) ) );

//	std::cout << "motif coords 1 " << xyz1.x() << " " << xyz1.y() << " " << xyz1.z() << std::endl;

	cst_set.add_coordinate_constraint( atm1, xyz1 );
	cst_set.add_coordinate_constraint( atm2, xyz2 );
	cst_set.add_coordinate_constraint( atm3, xyz3 );

//	std::cout << "Using atoms " << aa1 << " " << aa2 << " " << aa3 << std::endl;

	return;

}

int
set_vector_motif_sidechain_cst(
	cst_set_ns::Cst_set & cst_set,
	DNA_Motifs::DNAMotifRotamerLibrary & inv_rot_lib,
	FArray1Da_int  bb_pos_array,
	int nres
)
{
	bb_pos_array.dimension( nres );

	int nmotifs( 0 );
	for( int chkset = 1, echk = nres ;
			chkset <= echk ; ++chkset ) {
		if( bb_pos_array(chkset) < 0 ) continue;
		int this_mot = bb_pos_array(chkset);
		nmotifs++;
		set_motif_sidechain_cst( cst_set, inv_rot_lib[this_mot], chkset );
	}

	return nmotifs;
}

