// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 01:39:36 -0500 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "dock_pivot.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "cenlist.h"
#include "diagnostics_rosetta.h"
#include "dock_pivot_ns.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_movement.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "fullatom.h"
#include "input_pdb.h"
#include "loops.h"
#include "maps.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h"
#include "pack_fwd.h"
#include "PackerTask.h"
#include "param.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_fwd.h"
#include "current_pose.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "runlevel.h"
#include "score_ns.h"
#include "status.h"
#include "termini.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


// dock_pivot.cc: dock_pivot functions for performing pivot
// moves in conjunction with docking
//
// Summary:  dock_pivot.cc was created by Lian Guo in January 2004.  It was designed
// to pivot selected residues from the C-terminal helix of hpr kinase, since this helix
// moves when it docks to hpr (CAPRI target 1).  Up to five residues are allowed to
// fluctuate their phi and psi angles in the process of pivoting.
//
// Lian Guo January 2004
// revised April 2004
//
//
//
// adjust_pivot_MC_step
// perform_pivot_MC
// main_pivot_residue_trial
// initialize_pivot_residues
// load_pivot_residue_file
// validate_pivot_residues
// show_pivot_pert (output/debug)
// copy_fullcoord_to_position
// torsion_from_fullcoord
// pivot_refold
// pivot_refold_coordinates
// get_pivot_rmsd
// global_optimize_pivot
// optimize_pivot
// pivot_move
// pivot_repack
// detect_pivot_surrounding
// docking_initial_helix




//////////////////////////////////////////////////////////////////////////////
/// @begin get_dock_pivot_flag
///
/// @brief
/// the get_dock_pivot_flag indicates that a docking mode run is also doing dock_pivot moves
/// This is used in multiple statements that should only be active if dock_pivot is set to true,
/// but for which dock_pivot.h should not be included.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_dock_pivot_flag()
{
	using namespace dock_pivot;

	return using_pivot_residues;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin adjust_pivot_MC_step
///
/// @brief adjust pivot_magnitude based on pivot MC move
///  success rate, if performing dock_pivot moves
///
/// @detailed
///  constructed to allow rigid_body_MC_cycle_adaptive (see dock_structure.cc)
///  to do pivot moves
///
///  Inputs:  pivot_magnitude, pivot_success_rate
///  Outputs:  none, but pivot_magnitude may change
///
/// @param[in]   pivot_magnitude - in
/// @param[in]   pivot_success_rate - in
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors Mike Daily
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
adjust_pivot_MC_step(
	float & pivot_magnitude,
	float const pivot_success_rate
)
{
	using namespace dock_pivot;

	if ( ! using_pivot_residues ) return;

	if ( pivot_success_rate < 0.5 ) {
		pivot_magnitude *= 0.9;
	} else {
		pivot_magnitude *= 1.1;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin perform_pivot_MC
///
/// @brief
///
/// @detailed
///  Inputs:  see parameter list
///  Outputs:  none, but pivot_accepts changes
///  functions:  call main_pivot_residue_trial if dock_pivot
///  is turned on (using_pivot_residues
///  constructed to allow rigid_body_MC_cycle (see dock_structure.cc)
///  to do pivot moves
///
/// @param  scorefxn - [in/out]?
/// @param  cycle_number - [in/out]?
/// @param  repack_flag - [in/out]?
/// @param  pivot_magnitude - [in/out]?
/// @param  pivot_accepts - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors Mike Daily
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
perform_pivot_MC(
	Scoring_Function scorefxn,
	int const cycle_number,
	bool const repack_flag,
	float const pivot_magnitude,
	int & pivot_accepts
)
{
	using namespace dock_pivot;

	if ( ! using_pivot_residues ) {
		return;
	}
	main_pivot_residue_trial(scorefxn,cycle_number,repack_flag,pivot_magnitude);
	if ( get_monte_carlo_accept() >= 1 ) {
		++pivot_accepts;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin main_pivot_residue_trial
///
/// @brief
///
/// @detailed
///
/// @param  scoring_function - [in/out]? - which score to use
/// @param  cycle_number   [in/out]? - for labelling output
/// @param  repack_flag - [in/out]? - repack sidechains after the move?
/// @param  pivot_magnitude - [in/out]? - magnitude of phi-psi pivot
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_pivot_residue_trial(
	Scoring_Function scoring_function, // which score to use
	int const cycle_number, // for labelling output
	bool const repack_flag, // repack sidechains after the move?
	float const pivot_magnitude // magnitude of phi-psi pivot
)
{
	using namespace dock_pivot;
	using namespace docking;
	using namespace misc;
	using namespace param;
	using namespace scores;

	FArray1D_float cal_phi( MAX_RES()() );
	FArray1D_float cal_psi( MAX_RES()() );
	FArray1D_float cal_omega( MAX_RES()() );
	FArray1D_float phi2( MAX_RES()() );
	FArray1D_float psi2( MAX_RES()() );
	FArray1D_float omega2( MAX_RES()() );

	static int currRes = { 0 };

// -- Optionally repack sidechains --
	if ( repack_flag ) {
		docking_repack(true); // true means only repack interface
		std::cout << "interface side chain repacked ... interface !!!" << std::endl;
	}

	torsion_from_position(total_residue,position,cal_phi,cal_psi,cal_omega);

	for ( int k = 1; k <= max_pivot_residue_count; ++k ) {
		currRes = pivot_residue_list(k);
		if ( currRes > 0 ) {
			phi(currRes) = cal_phi(currRes); //!! important
			psi(currRes) = cal_psi(currRes); //!! consistence condition
		}
	}

// -- Make phi-psi move --

	global_optimize_pivot(pivot_magnitude);

	get_pivot_rmsd(pivot_residue_list(1),domain_end(1));
	 // careful only one direction

	torsion_from_position(total_residue,position,phi2,psi2,omega2);

	for ( int k = 1; k <= max_pivot_residue_count; ++k ) {
		currRes = pivot_residue_list(k);
	}

// -- Optionally repack sidechains --
	if ( repack_flag ) {
		pivot_repack(true);
		std::cout << "pivot surrounding side chain repacked ... pivot !!!" << std::endl;
	}

	mc_global_track::mc_score::score = scoring_function();

	save_status_info( "pivot_residue", 1, 0 );
	monte_carlo(cycle_number);
	increment_trial_counters(cycle_number);

//	if ( get_monte_carlo_accept() >= 1 )
//	 std::cout << "               pivot-perturb accept!" << std::endl;
	torsion_from_position(total_residue,position,cal_phi,cal_psi,cal_omega);

	for ( int k = 1; k <= max_pivot_residue_count; ++k ) {
		currRes = pivot_residue_list(k);
		if ( currRes > 0 ) {
			phi(currRes) = cal_phi(currRes); //!! important
			psi(currRes) = cal_psi(currRes); //!! consistence condition
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin initialize_pivot_residues
///
/// @brief
///  Initialize important variables necessary for pivot residues
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_pivot_residues()
{
	using namespace dock_pivot;

	if ( ! using_pivot_residues ) return;
	using_pivot_residue_file = false;

	load_pivot_residue_file(using_pivot_residue_file);

	if ( using_pivot_residue_file ) {
		std::cout << "Pivot file found, perturbing selected residues" << std::endl;
	} else { // don't think this option works yet!
		std::cout << "No pivot file, perturbing random residues" << std::endl;
	}

	refold_set_direction(1);

	pivot_moves = 1;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin load_pivot_residue_file
///
/// @brief
///  Load the pivot residue if it exists.  Set flag that
///  pivot residues are loaded if the operation succeeds.
///
/// @detailed
///  The pivot file is expected to contain integers, one
///  per line.  Each integer represents the residue number
///  of a residue that will be allowed at each docking run.
///
/// @param  isSuccess - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
load_pivot_residue_file( bool & isSuccess )
{
	using namespace dock_pivot;

	std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
	std::cout << "Attempting to load pivot residue file: " <<
	 pivot_residue_filename << std::endl;

	utility::io::izstream pivot_stream( pivot_residue_filename );

	if ( !pivot_stream ) {
		std::cout << "[Error]: Could not open pivot-residue file: " <<
		 pivot_stream.filename() << std::endl;
		std::cout << "Not using pivot residues." << std::endl;
		isSuccess = false;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int res_num_input;
	char res_chain_input;
	for ( int i = 1; i <= max_pivot_residue_count; ++i ) {

		pivot_stream >> res_num_input >> res_chain_input >> skip;

		if ( !pivot_stream ) {
			pivot_residue_list_size = i-1;
			goto L200;
		}

// convert the res+chain id to rosetta id.
		res_num_from_pdb_res_num_chain( pivot_residue_list(i), res_num_input,
		 res_chain_input );

		if ( pivot_residue_list(i) != -1 ) {
			std::cout << "Using pivot residue: " <<
			 SS( res_num_input ) << SS( res_chain_input ) << std::endl;
		} else {
			std::cout << "Can't use pivot: " <<
			 SS( res_num_input ) << SS( res_chain_input ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}
	pivot_residue_list_size = max_pivot_residue_count;

	pivot_stream >> res_num_input >> skip;
	if ( !pivot_stream.eof() ) {
		std::cout << "[Error]: Too many pivot residues, only loading " <<
		 SS( max_pivot_residue_count ) << std::endl;
	}

L200:
	pivot_stream.close();
	pivot_stream.clear();
	pivot_residues_loaded = true;

	validate_pivot_residues(isSuccess);
	std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin validate_pivot_residues
///
/// @brief
///
/// @detailed
///
/// @param  isSuccess - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  Make sure there is at least one pivot loaded
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
validate_pivot_residues( bool & isSuccess )
{
	using namespace dock_pivot;

	isSuccess = false;
// Check if we loaded any pivots
	std::cout << "pivot list: ";
	for ( int i = 1; i <= max_pivot_residue_count; ++i ) {
		std::cout << SS( pivot_residue_list(i) );
	} std::cout << std::endl;
	for ( int i = 1; i <= max_pivot_residue_count; ++i ) {
		if ( pivot_residue_list(i) >= 0 ) {
			isSuccess = true;
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin show_pivot_pert
///
/// @brief
///  A debugging routing that writes the perturbation of
///  pivot residue in the form of delta Phi and Psi
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
show_pivot_pert()
{
	using namespace dock_pivot;
	using namespace misc;
	using namespace native;
	using namespace param;
	using namespace scores;

	using namespace mc_global_track::mc_score; // yab: misc removal

	FArray1D_float cal_phi( MAX_RES()() );
	FArray1D_float cal_psi( MAX_RES()() );
	FArray1D_float cal_omega( MAX_RES()() );
	FArray1D_float phi2( MAX_RES()() );
	FArray1D_float psi2( MAX_RES()() );
	FArray1D_float omega2( MAX_RES()() );

	int currRes = pivot_residue_list(1);

	torsion_from_fullcoord(MAX_ATOM(),total_residue,full_coord,cal_phi,cal_psi,
	 cal_omega);
	torsion_from_position(total_residue,position,phi2,psi2,omega2);

	std::cout << "**************************** " << SS( currRes ) <<
	 " *****************************" << std::endl;
	for ( int k = currRes-1, ke = currRes+2; k <= ke; ++k ) {
		std::cout << I( 4, k ) << " phi(n,a,c,f)|psi(n,a,c,f): " <<
		 F( 9, 3, native_phi(k) ) <<
		 F( 9, 3, phi(k) ) <<
		 F( 9, 3, phi2(k) ) <<
		 F( 9, 3, cal_phi(k) ) << "  ||" <<
		 F( 9, 3, native_psi(k) ) <<
		 F( 9, 3, psi(k) ) <<
		 F( 9, 3, psi2(k) ) <<
		 F( 9, 3, cal_psi(k) ) << std::endl;
	}

	std::cout <<
	 "    sc/bs/ls: " <<
	 F( 9, 2, score ) <<
	 F( 9, 2, best_score ) <<
	 F( 9, 2, low_score ) << std::endl;
	std::cout <<
	 "    vdw    rama    pair     hb_lrbb  hb_srbb    d_env   d_cont   score" << std::endl;
	std::cout <<
	 F( 8, 2, vdw_score ) <<
	 F( 8, 2, ramachandran_score ) <<
	 F( 8, 2, pair_score ) <<
	 F( 8, 2, hb_lrbb_score ) <<
	 F( 8, 2, hb_srbb_score ) <<
	 F( 8, 2, docking_env_score ) <<
	 F( 8, 2, docking_contact_score ) <<
	 F( 8, 2, score ) << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dist
///
/// @brief
///
/// @detailed
///
/// @param  mres - [in/out]?
/// @param  natom - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
dist(
	int const mres,
	int const natom
)
{
	using namespace misc;

	float dist_sq = 0.0;
	if ( natom == 1 ) {
		for ( int i = 1; i <= 3; ++i ) {
			dist_sq += square( full_coord(i,1,mres) - full_coord(i,3,mres-1) );
		}
	} else if ( natom == 2 ) {
		for ( int i = 1; i <= 3; ++i ) {
			dist_sq += square( full_coord(i,2,mres) - full_coord(i,1,mres) );
		}
	} else if ( natom == 3 ) {
		for ( int i = 1; i <= 3; ++i ) {
			dist_sq += square( full_coord(i,3,mres) - full_coord(i,2,mres) );
		}
	}

	return std::sqrt( dist_sq );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin copy_fullcoord_to_position
///
/// @brief
///
/// @detailed
///
/// @param  position - [in/out]?
/// @param  fullcoord - [in/out]?
/// @param  maxatom - [in/out]?
/// @param  nres - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_fullcoord_to_position(
	FArray3Da_float position,
	FArray3Da_float fullcoord,
	int const maxatom,
	int const nres
)
{
	position.dimension( 3, 5, nres );
	fullcoord.dimension( 3, maxatom, nres );

	for ( int i = 1; i <= nres; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			position(j,1,i) = fullcoord(j,1,i); // N
			position(j,2,i) = fullcoord(j,2,i); // CA
			position(j,4,i) = fullcoord(j,3,i); // C -- note atom order change
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin torsion_from_fullcoord
///
/// @brief
///
/// @detailed
///
/// @param  maxatom - [in/out]?
/// @param  mres - [in/out]?
/// @param  coord - [in/out]?
/// @param  phi - [in/out]?
/// @param  psi - [in/out]?
/// @param  omega - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
torsion_from_fullcoord(
	int const maxatom,
	int const mres,
	FArray3Da_float coord,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega
)
{
	using namespace param;

	coord.dimension( 3, maxatom, mres );
	phi.dimension( mres );
	psi.dimension( mres );
	omega.dimension( mres );

	FArray3D_float position( 3, 5, MAX_RES()() );

	copy_fullcoord_to_position(position,coord,maxatom,mres);
	torsion_from_position(mres,position,phi,psi,omega);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin pivot_refold
///
/// @brief
///
/// @detailed
///
/// @param  pivot_residue - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pivot_refold( int & pivot_residue )
{
	using namespace dock_pivot;
	using namespace misc;
	using namespace param;

// local
	FArray1D_bool fixresidue( MAX_RES()() );

	int const begin_res = pivot_residue;
	int const end_res = pivot_residue;

// idiot hunt:
	if ( begin_res > end_res ) {
		std::cout << "WARNING!! refold called with begin_res>end_res" << std::endl;
		std::cout << "          retrieving best pose..." << std::endl;
		retrieve_best_pose();
	}

	bool const fullatom = get_fullatom_flag();

	int seg1 = identify_segment(1);
	int seg2 = identify_segment(total_residue);

// fold seg1->seg2
	retrieve_allow_insert(fixresidue,total_residue);

//d	std::cout << I( 5, seg1 ) << I( 5, seg2 ) << std::endl;

	int seg_begin, seg_end, dir;
	for ( int i = seg1; i <= seg2; ++i ) {
		retrieve_segment_ends(i,seg_begin,seg_end);
		refold_select_dir(i,pivot_residue,pivot_residue,dir);

		pivot_refold_coordinates(dir,pivot_residue,seg_begin,seg_end,fullatom,res,
		 best_position,best_centroid,best_full_coord,phi,psi,position,centroid,
		 full_coord,best_phi,best_psi,res_variant);

		maps_set_new_phipsi(pivot_residue,pivot_residue,seg_begin,seg_end, dir);

//d		std::cout << I( 5, fold_begin ) << I( 5, fold_end ) << I( 5, seg_begin ) <<
//d		 I( 5, seg_end ) << I( 5, frag_begin ) << I( 5, frag_end ) << std::endl;
//d		std::cout << "refold is called ..." << std::endl;
//d		show_pivot_pert();

//		if ( trigger1 ) {
//			std::cout << "hpr residue 47 position after helix perturb: "
//			for ( int j = 1; j <= 3; ++j ) {
//				std::cout << F( 8, 3, Eposition(j,2,77) );
//			} std::cout << std::endl;
//		}

//d		show_pivot_pert();

	}


	if ( get_fullatom_flag() ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( std::abs(Eposition(1,2,i) - full_coord(1,2,i)) > .001 ) {
				std::cout << "WARNING!!! position and full_coord disagree!" << std::endl;
				std::cout << SS( i ) << SS( Eposition(1,2,i) ) <<
				 SS( full_coord(1,2,i) ) << std::endl;
				std::cout << SS( i ) << SS( Ebest_position(1,2,i) ) <<
				 SS( best_full_coord(1,2,i ) ) << std::endl;
			}
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin pivot_refold_coordinates
///
/// @brief
///
/// @detailed
/// This should be incorporated into regular refold_coordinates
/// I suspect there is special treatment here to retain native bond
/// lengths
/// and angles--Chu has also addressed this.  So, Chu's solution should be used
/// to remove this function and include it in the regular refold_coordinates
///
/// @param  refold_dir - [in/out]? - refold direction
/// @param  pivot_residue - [in/out]? - residue which torsion angles change
/// @param  seg_begin - [in/out]? -
/// @param  seg_end - [in/out]? -
/// @param  fullatom - [in/out]? - should full_coord be updated?
/// @param  res - [in/out]? - sequence
/// @param  best_position - [in/out]? - source coordinates
/// @param  best_centroid - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  position - [in/out]? - updated coords
/// @param  centroid - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  best_phi - [in/out]? -
/// @param  best_psi - [in/out]? -
/// @param  res_variant - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pivot_refold_coordinates(
	int const refold_dir, // refold direction
	int const pivot_residue, // residue which torsion angles change
	int const seg_begin,
	int const seg_end,
	bool const fullatom, // should full_coord be updated?
	FArray1Da_int res, // sequence
	FArray3Da_float best_position, // source coordinates
	FArray2Da_float best_centroid,
	FArray3Da_float best_full_coord, // source rotamers
	FArray1Da_float phi, // angles to use for folding
	FArray1Da_float psi,
	FArray3Da_float position, // updated coords
	FArray2Da_float centroid,
	FArray3Da_float full_coord,
	FArray1Da_float best_phi,
	FArray1Da_float best_psi,
	FArray1Da_int res_variant
)
{
	using namespace aaproperties_pack;
	using namespace native;
	using namespace param;
	using numeric::conversions::radians;
	using namespace termini_ns;

	res.dimension( seg_end );
	best_position.dimension( 3, MAX_POS, seg_end );
	best_centroid.dimension( 3, seg_end );
	best_full_coord.dimension( 3, MAX_ATOM(), seg_end );
	phi.dimension( seg_end );
	psi.dimension( seg_end );
	position.dimension( 3, MAX_POS, seg_end );
	centroid.dimension( 3, seg_end );
	full_coord.dimension( 3, MAX_ATOM(), seg_end );
	best_phi.dimension( seg_end );
	best_psi.dimension( seg_end );
	res_variant.dimension( seg_end );

	int const n2c = { 1 };
//	int const c2n = { -1 };

// local
	double phi8, psi8;
	FArray2D_double mat( 3, 3 );
	FArray1D_double off( 3 );
	FArray1D_double xn( 3 );
	FArray1D_double xca( 3 );
	FArray1D_double xc( 3 );
	FArray1D_double xo( 3 );

	if ( pivot_residue < seg_begin || pivot_residue > seg_end ) {
		copy_coordinates(seg_begin,seg_end,fullatom,best_position,position,
		 best_centroid,centroid, best_full_coord,full_coord);
		return;
	}

	if ( refold_dir == n2c ) { // transform fold_end+1 to seg_end

		phi8 = radians( static_cast< double >( phi(pivot_residue) - best_phi(pivot_residue) ) );
		for ( int n = 1; n <= 3; ++n ) {
			xn(n) = position(n,1,pivot_residue);
			xca(n) = position(n,2,pivot_residue);
		}
		getrot(xn,xca,phi8,mat,off);

		transform_coordinates(pivot_residue,seg_end,fullatom,best_position,
		 position,best_centroid,centroid,best_full_coord,full_coord,mat,off);

		psi8 = radians( static_cast< double >( psi(pivot_residue) - best_psi(pivot_residue) ) );
		for ( int n = 1; n <= 3; ++n ) {
			xc(n) = position(n,4,pivot_residue);
			xo(n) = position(n,5,pivot_residue);
		}
		getrot(xca,xc,psi8,mat,off);

		transform_coordinates(pivot_residue+1,seg_end,fullatom,position,position,
		 centroid,centroid,                 /* from position to position */
		 full_coord,full_coord,mat,off);
		mover(xo,mat,off); // maybe problem if fullatom
		for ( int n = 1; n <= 3; ++n ) {
			position(n,5,pivot_residue) = xo(n);
		}                  // move Oxygen

//------------------------------------------------------------------------------

	} else {                      // transform seg_begin to fold_begin-1
		// notice the phi psi pertb sequence changes

		psi8 = radians( static_cast< double >( psi(pivot_residue) - best_psi(pivot_residue) ) );
		for ( int n = 1; n <= 3; ++n ) {
			xca(n) = position(n,2,pivot_residue);
			xc(n) = position(n,4,pivot_residue);
			xo(n) = position(n,5,pivot_residue);
		}
		getrot(xca,xc,-psi8,mat,off);

		transform_coordinates(seg_begin,pivot_residue,fullatom,best_position,
		 position,best_centroid,centroid,best_full_coord,full_coord,mat,off);
		for ( int n = 1; n <= 3; ++n ) {
			position(n,5,pivot_residue) = xo(n);
		}                  // restore Oxygen

		phi8 = radians( static_cast< double >( phi(pivot_residue) - best_phi(pivot_residue) ) );
		for ( int n = 1; n <= 3; ++n ) {
			xn(n) = position(n,1,pivot_residue);
			xca(n) = position(n,2,pivot_residue); // doesn't change actually
		}
		getrot(xn,xca,-phi8,mat,off);

		transform_coordinates(seg_begin,pivot_residue-1,fullatom,   // notice -1
		 position,position,centroid,centroid,full_coord,full_coord,mat,off);

	}

// generate fullatom coordinates for the region folded from angles
	if ( fullatom ) {
		build_fullcoord(position,pivot_residue,pivot_residue,res,res_variant,
		 is_N_terminus,is_C_terminus,phi(pivot_residue),best_full_coord,full_coord);
//		build_fullcoord(position(1,1,pivot_residue),1,MAX_ATOM(),
//		 res(pivot_residue),natoms,best_full_coord(1,1,pivot_residue),
//		 full_coord(1,1,pivot_residue));
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_pivot_rmsd
///
/// @brief
///
/// @detailed
///
/// @param  seg_begin - [in/out]?
/// @param  seg_end - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pivot_rmsd(
	int const seg_begin,
	int const seg_end
)
{
	using namespace dock_pivot;
	using namespace misc;
	using namespace native;

// rmsd averaged over all residues influenced by pivot perturbation
	float tsd = 0.0;
	for ( int i = seg_begin; i <= seg_end; ++i ) {
		tsd +=
		 square( Eposition(1,2,i) - native_ca(1,i) ) +
		 square( Eposition(2,2,i) - native_ca(2,i) ) +
		 square( Eposition(3,2,i) - native_ca(3,i) );
	}
	pivot_rmsd = std::sqrt( tsd / ( seg_end - seg_begin + 1 ) );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin global_optimize_pivot
///
/// @brief
///
/// @detailed
///
/// @param  magnitude - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
global_optimize_pivot( float const magnitude )
{
	using namespace dock_pivot;

	pivot_min_fret = 100.0; // set initial value a large number
	optimize_pivot(magnitude);
	pivot_move(pivot_min_list); // restore low_func structure
}

//////////////////////////////////////////////////////////////////////////////
/// @begin optimize_pivot
///
/// @brief
///
/// @detailed
///
/// @param  magnitude - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
optimize_pivot( float const magnitude )
{
	using namespace dock_pivot;
	using namespace misc;
	using namespace native;

	bool gfrag( true );
	FArray1D_float p( max_pivot_residue_count*2 );
	float fret;
	float phi_tmp,psi_tmp;
	int iter,currRes;

	minimize_set_func(7);
	float ftol = 1e-4;
	int n = 0;

	float big_angle = magnitude;
	float small_angle = big_angle/2.0;

	for ( int i = 1; i <= max_pivot_residue_count; ++i ) {
		currRes = pivot_residue_list(i);
		if ( currRes > 0 ) {

			if ( mod(pivot_moves,8) == 0 ) {
				phi_tmp = best_phi(currRes) - small_angle + ran3()*big_angle;
				 // here *best_torsion was the
				psi_tmp = best_psi(currRes) - small_angle + ran3()*big_angle;
				 // center of pivot perturb
//           Keep angles between -180 -> 180
				if ( phi_tmp > 180.0 ) {
					phi_tmp -= 360.0;
				} else if ( phi_tmp < -180.0 ) {
					phi_tmp += 360.0;
				}
				if ( psi_tmp > 180.0 ) {
					psi_tmp -= 360.0;
				} else if ( psi_tmp < -180.0 ) {
					psi_tmp += 360.0;
				}
			} else {
				phi_tmp = best_phi(currRes); // here *best_torsion was the
				psi_tmp = best_psi(currRes); // center of pivot perturb
			}

//			phi_tmp = native_phi(currRes); // for optimizing consistence test
//			psi_tmp = native_psi(currRes); // for optimizing consistence test

			n += 2;
			p(2*i-1) = phi_tmp;
			p(2*i)   = psi_tmp;

		}
	}         // loop over pivot_residue_list

	++pivot_moves;

//	std::cout << "optimizing dimensions:" << I( 2, n ) << std::endl;
	powell(gfrag,p,n,ftol,iter,fret);

	pivot_move(p); // restore low_func structure, exist also in upper sub
	get_pivot_rmsd(pivot_residue_list(1),domain_end(1));

	if ( fret < pivot_min_fret ) {
		pivot_min_fret = fret;
		for ( int i = 1; i <= n; ++i ) {
			pivot_min_list(i) = p(i);
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin func_pivot
///
/// @brief
///
/// @detailed
///
/// @param  TA - [in/out]?
/// @param  gfrag - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func_pivot(
	FArray1Da_float TA
)
{
	using namespace dock_pivot;
	using namespace misc;
//	using namespace param;

	TA.dimension( pivot_residue_list_size*2 );

	pivot_move(TA);

// FArray1D_float phi2( MAX_RES()() ), psi2( MAX_RES()() ), omega2( MAX_RES()() );
//	torsion_from_position(total_residue,position,phi2,psi2,omega2);
//	std::cout << "---desired torsion angles:" <<
//	 F( 8, 2, TA(1) ) << F( 8, 2, TA(2) ) << F( 8, 2, TA(3) ) <<
//	 F( 8, 2, TA(4) ) << F( 8, 2, TA(5) ) << F( 8, 2, TA(6) ) << std::endl;
//	std::cout << "---real    torsion angles:" <<
//	 F( 8, 2, phi(317) ) << F( 8, 2, psi(317) ) << F( 8, 2, phi(318) ) <<
//	 F( 8, 2, psi(318) ) << F( 8, 2, phi(319) ) << F( 8, 2, psi(319) ) <<
//	 F( 8, 3, pivot_rmsd ) << std::endl; // debug...

// pivot_repack(true); // unnecessary since pivot moves are in centroid mode

	return score4dp();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin pivot_move
///
/// @brief
///
/// @detailed
///
/// @param  TA - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pivot_move( FArray1Da_float TA )
{
	using namespace dock_pivot;
	using namespace misc;
	using namespace param;

	TA.dimension( pivot_residue_list_size*2 );

	FArray2D_float bak_position( 3, MAX_POS * MAX_RES()() );
	FArray2D_float bak_centroid( 3, MAX_RES()() );
	FArray3D_float bak_full_coord( 3, MAX_ATOM()(), MAX_RES()() );
	int currRes;

	int res_end = domain_end(1);
	copy_coordinates(1,res_end,false,best_position,bak_position,best_centroid,
	 bak_centroid,best_full_coord,bak_full_coord);
	for ( int i = 1; i <= pivot_residue_list_size; ++i ) {
		currRes = pivot_residue_list(i);
		assert( ( currRes > 0 ) );
		phi(currRes) = TA(2*i-1);
		psi(currRes) = TA(2*i);
		refold_set_direction(1);
		pivot_refold(currRes);

		copy_coordinates(1,res_end,false,position,best_position,
		 centroid, best_centroid,full_coord,best_full_coord);
		// have to do so in pivot_refold best_position move
	}         // loop over pivot_residue_list

	copy_coordinates(1,res_end,false,bak_position,best_position,bak_centroid,
	 best_centroid,bak_full_coord,best_full_coord);
	 // recover the best position, etc
}

//////////////////////////////////////////////////////////////////////////////
/// @begin pivot_repack
///
/// @brief
///
/// @detailed
///
/// @param  rotamers_exist - [in/out]? - passed to pack
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pivot_repack( bool const rotamers_exist /* passed to pack */ )
{
	using namespace dock_pivot;
	using namespace docking;
	using namespace misc;
	using namespace param;
	using namespace pdbstatistics_pack;
	using namespace runlevel_ns;

	if ( ! using_pivot_residues ) return;
	if ( ! get_fullatom_flag() ) {
		std::cout << "How can I do repacking not in FULLATOM mode ?!" << std::endl;
		return;
	} else {
		std::cout << "pivot surrounding residues repacking is ongoing ..." << std::endl;
	}

	FArray1D_bool allow_repack( MAX_RES()() );
	int startres,endres;
	FArray2D_int unbound_rot( MAX_CHI, MAX_RES()() );
	FArray2D_float unbound_chi( MAX_CHI, MAX_RES()() );

	//yl flags for pose_from_misc
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const coords_init( true );
	pose_ns::Pose pose;
	pose_from_misc(pose,fullatom, ideal_pose, coords_init);

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );

	//yl create local pose and prepare the Epositions for misc
	//for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ){
	//	for ( int k=1; k<= 3; ++k ) {
	//		Eposition(k,1,seqpos) = full_coord(k,1,seqpos);
	//		Eposition(k,2,seqpos) = full_coord(k,2,seqpos);
	//		Eposition(k,4,seqpos) = full_coord(k,3,seqpos);
	//		Eposition(k,5,seqpos) = full_coord(k,4,seqpos);
	//		Eposition(k,3,seqpos) = full_coord(k,5,seqpos);
	//	}
	//}


	if ( ! norepack1 ) {
		startres = 1;
		if ( ! norepack2 ) {
			endres = total_residue;
		} else {
			endres = part_end(1);
		}
	} else {
		if ( ! norepack2 ) {
			startres = part_begin(2);
			endres = part_end(2);
		} else {
			std::cout << "something is wrong in pivot_repack, nothing to repack" << std::endl;
			startres = 1;
			endres = 0;
		}
	}

// 1-pack all pivot surrounding residues within repack limits

	detect_pivot_surrouding();
	int n = 0;
	for ( int i = startres; i <= endres; ++i ) {
		allow_repack(i) = pvt_res8(i);
		if ( pvt_res8(i) ) ++n;
	}
	for ( int i = 1; i < startres; ++i ) { // not between startres and endres
		allow_repack(i) = false;
	}
	for ( int i = endres+1; i <= total_residue; ++i ) {
	 // not between startres and endres
		allow_repack(i) = false;
	}

	std::cout << "--pivot_repack: " << I( 2, startres ) << "--" <<
	 I( 4, endres ) << " total" << I( 4, n ) << " pivot surrounding residues" <<
	 L( 1, !norepack1 ) << L( 1, !norepack2 ) << std::endl;

	if ( runlevel >= verbose ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( allow_repack(i) ) {
				write_res( std::cout, i );
				std::cout << std::endl;
			}
		}
	}

// eval_norepack_disulf(allow_repack); // disulfides are special

	std::string const packmode( "packrot" );
	//bool make_output_file = false;
	if ( unboundrot ) retrieve_unbound_rotamers(unbound_rot,unbound_chi);

	Task.set_task(packmode, false, allow_repack, rotamers_exist, unboundrot, unbound_rot, unbound_chi);
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task );
	pose.copy_to_misc();

	save_status_info("pivot_repack",1,total_residue);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin detect_pivot_surrouding
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
detect_pivot_surrouding()
{
	using namespace cenlist_ns;
	using namespace dock_pivot;
	using namespace misc;

	int const pvt_begin = pivot_residue_list(1);
	 // it occurs the same in the call of get_pivot_rmsd()
	int const pvt_end = domain_end(1); // similiar in function pivot_refold

	std::cout << "--pls check: pvt_begin/end" <<
	 I( 4, pvt_begin ) << I( 4, pvt_end ) << std::endl;

	for ( int i = 1; i <= total_residue; ++i ) {
		pvt_res8(i) = false;
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = pvt_begin; j <= pvt_end; ++j ) {
			if ( ( cendist(i,j) < 64.0 ) && ( !pvt_res8(i) ) ) {
				pvt_res8(i) = true;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begina docking_pivot_perturb_segment
///
/// @brief
///
/// @detailed
///
/// @param  magnitude - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_pivot_perturb_segment( float const magnitude )
{
	using namespace misc;
	using namespace dock_pivot;

	if ( ! using_pivot_residues ) return;

	int currRes;
	FArray1D_float p( max_pivot_residue_count*2 );
	float big_angle, small_angle, phi_tmp, psi_tmp;

	big_angle = magnitude;
	small_angle = big_angle/2.0;

	for ( int i = 1; i <= max_pivot_residue_count; ++i ) {
		currRes = pivot_residue_list(i);
		if ( currRes > 0 ) {

			phi_tmp = best_phi(currRes) - small_angle + ran3()*big_angle;
			 // here *best_torsion was the
			psi_tmp = best_psi(currRes) - small_angle + ran3()*big_angle;
			 // center of pivot perturb for bound_pert
			 // replaced by start_torsion for unbound_pert
			// Keep angles between -180 -> 180
			if ( phi_tmp > 180.0 ) {
				phi_tmp -= 360.0;
			} else if ( phi_tmp < -180.0 ) {
				phi_tmp += 360.0;
			}
			if ( psi_tmp > 180.0 ) {
				psi_tmp -= 360.0;
			} else if ( psi_tmp < -180.0 ) {
				psi_tmp += 360.0;
			}

			p(2*i-1) = phi_tmp;
			p(2*i) = psi_tmp;
		}
	}

	pivot_move(p);
	mc_global_track::mc_score::score = score4d();
	save_status_info("pivot_perturb_segment",0,0);
	monte_carlo_reset();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin docking_pivot_store_cen_scores
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_pivot_store_cen_scores()
{
	using namespace dock_pivot;
	using namespace scores;

	if ( ! using_pivot_residues ) return;
	low_resolution_sc = mc_global_track::mc_score::score;
	low_resolution_vdw = vdw_score;
	low_resolution_dcontact = docking_contact_score;
}
