// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  $Revision: 19901 $
//  $Date: 2008-01-24 20:37:36 -0500 (Thu, 24 Jan 2008) $
//  $Author: aroop $


// Rosetta Headers
#include "docking_constraints.h"
#include "cenlist.h"
#include "design.h"
#include "design_structure.h"
#include "diagnostics_rosetta.h"
#include "dock_fab.h"
#include "dock_structure.h"
#include "docking.h"
#include "docking_db.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "interface.h"
#include "misc.h"
#include "param.h"
#include "pdb.h"
#include "runlevel.h"
#include "score_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


//     docking_constraints.cc: functions to put biochemical constraints
//     into the docking protocol

//     1-residue-residue distance constraints
//     2-interface site constraints

//     functions related to antibody-fragment docking
//     aug-sep 2001
//     jeff gray




//------------------------------------------------------------------------------
//     DISTANCE CONSTRAINTS =======================================




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_read_dist_constraint
///
/// @brief read distance constraints from the PDB1.dst file
///
/// @detailed
///     distance constraint file format:  (filter only)
///
///     10.0       distance (Angstroms)
///     23 H       residue 1
///     24 H       residue 2
///
///
/// @global_read docking_flag
///
/// @global_write interface.h - dist_data block
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_read_dist_constraint()
{
	using namespace files_paths;
	using namespace interface;
	using namespace runlevel_ns;

//     yokels
	std::string filename;

	int dist_pdb_res_num;
	char dist_chain;
	float dist_constraint_distance;

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;

//     open the file

	filename = constraints_path + protein_name_prefix + protein_name + ".dst";

	utility::io::izstream cnstr_zx( filename );

	if ( !cnstr_zx ) {
		std::cout << "File: " << cnstr_zx.filename() << " not found" << std::endl;
		std::cout << "Running without dist constraints" << std::endl;
		dist_constraint_exists = false;
		cnstr_zx.close();
		cnstr_zx.clear();
		return;
	}

	dist_constraint_exists = true;
	if ( runlevel >= standard ) std::cout <<
	 "loading docking distance constraints from " << filename << std::endl;

	cnstr_zx >> dist_constraint_distance >> skip;

	dist_constraint_distance_sq =
	 dist_constraint_distance * dist_constraint_distance;

	for ( int k = 1; k <= 2; ++k ) {
		cnstr_zx >> dist_pdb_res_num >> dist_chain >> skip;
		res_num_from_pdb_res_num_chain(dist_constraint_residues(k),dist_pdb_res_num,
		 dist_chain);
		if ( dist_constraint_residues(k) == -1) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	cnstr_zx.close();
	cnstr_zx.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_print_dist_constraint
///
/// @brief print the distance constraints to the screen
///
/// @detailed
///
/// @global_read interface.h dist_data block
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_print_dist_constraint()
{
	using namespace interface;
	using namespace misc;
	using namespace pdb;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	if ( !dist_constraint_exists ) return;

	std::cout << "\nDocking distance filter constraint:\n" << std::endl;

	std::cout << "Require " << std::sqrt( dist_constraint_distance_sq )
	 << " A maximum between residues:" << std::endl;

	int r = dist_constraint_residues(1);
	std::cout << space( 5 ) << residue3(r)
	 << I( 4, pdb_res_num(r) ) << res_chain(r)
	 << " (Rosetta" << I( 4, r ) << ')' << std::endl;

	std::string const buffer( right_string_of( pdb_res_num(r), 4 ) + res_chain(r) + ',' );

	r = dist_constraint_residues(2);
	std::cout << space( 5 ) << residue3(r)
	 << I( 4, pdb_res_num(r) ) << res_chain(r)
	 << " (Rosetta" << I( 4, r ) << ')' << std::endl;

	//buffer += right_string_of( pdb_res_num(r), 4 ) + res_chain(r); //Objexx:FMD Not used in write

	std::cout << "      so select " << buffer.substr( 0, 2*6-8 ) << "\n" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_dist_constr_filter
///
/// @brief
///     return true if the current structure meets the constraint
///
/// @detailed
///
/// @return
///
/// @global_read interface.h dist_data block
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
docking_dist_constr_filter()
{
	using namespace cenlist_ns;
	using namespace interface;

	bool docking_dist_constr_filter; // Return value

//     jan 2003 revised
	if ( dist_constraint_exists ) {

		docking_dist_constr_filter =
		 (cendist(dist_constraint_residues(1),dist_constraint_residues(2)) <
		 dist_constraint_distance_sq );

	} else {

		docking_dist_constr_filter = true;

	}

	return docking_dist_constr_filter;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_dist_constr_bonus
///
/// @brief give a bonus score for meeting the distance constraint
///
/// @detailed
///     give a bonus of 100 points if the dist_constraint is met.  if not,
///     ramp the score down with the cendist -- this creates a nice
///     gradient for the monte carlo function to follow jjg 9/12/1
///
/// @return
///
/// @global_read interface.h dist_data block
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_dist_constr_bonus()
{
	using namespace cenlist_ns;
	using namespace interface;

	float docking_dist_constr_bonus = 0.0; // Return value

	float extrad;

	if ( !dist_constraint_exists ) return docking_dist_constr_bonus;

	extrad = cendist(dist_constraint_residues(1),dist_constraint_residues(2)) -
	 dist_constraint_distance_sq;

	docking_dist_constr_bonus = -100.0 + extrad;

	if ( docking_dist_constr_bonus < -100.0 ) docking_dist_constr_bonus = -100.0;

	return docking_dist_constr_bonus;
}





//------------------------------------------------------------------------------

//     SITE CONSTRAINTS ===========================================

//     aug-sep 2001 jjg

//

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_read_site_constraints
///
/// @brief read the docking site constraints
///
/// @detailed this allows us to forbid or require certain residues at the interface
///
///     constraint file format:
///
///     2          number of sites
///     -1000      score bonus
///     23 H       residue list... (H is the chain ID)
///     24 H
///     25 H
///     0  H       stopper
///     100        next bonus...
///     12 L
///     0 L
///     filter -10 score at which to not reject decoys
///
/// @global_read docking_flag
///
/// @global_write interface.h site_data block
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_read_site_constraints()
{
	using namespace files_paths;
	using namespace interface;
	using namespace runlevel_ns;

//     yokels
	std::string filename;

	int site_pdb_res_num;
	char site_chain;
	std::string filter_flag;
	float filter_value;

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;

// open the file

	filename = constraints_path + protein_name_prefix + protein_name + '.' + cst_ext;

	utility::io::izstream cnstr_zx( filename );

	if ( !cnstr_zx ) {
		std::cout << "File: " << cnstr_zx.filename() << " not found" << std::endl;
		std::cout << "Running without site constraints" << std::endl;
		site_constraints_exist = false;
		cnstr_zx.close();
		cnstr_zx.clear();
		return;
	}

	site_constraints_exist = true;
	if ( runlevel >= standard ) std::cout <<
	 "loading docking site constraints from " << filename << std::endl;

	cnstr_zx >> N_site_constraints >> skip;

	if ( N_site_constraints > max_site_constraints ) {
		std::cout << "N_site_constraints = " << N_site_constraints <<
		 "is larger than max_site_constraints = " << max_site_constraints << std::endl;
		error_stop("Aborting docking_constraints.cc");
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( runlevel >= standard ) std::cout << "loading " << N_site_constraints <<
	 " docking site constraints" << std::endl;

	for ( int site = 1; site <= N_site_constraints; ++site ) {

		cnstr_zx >> site_constraint_bonus(site) >> skip;

		site_constraint_residues(site).clear();

		cnstr_zx >> site_pdb_res_num >> site_chain >> skip;
//ora  make constraints possible for files without chain
		if ( site_chain == '_' ) site_chain = ' ';
		while ( site_pdb_res_num != 0 ) {
			int res_num_out;
			res_num_from_pdb_res_num_chain(res_num_out,site_pdb_res_num,site_chain);
			if ( res_num_out == -1 ) {
				std::cout << "bad residue selection " << site_pdb_res_num << ' ' <<
				 site_chain << " stopping" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			site_constraint_residues(site).push_back(res_num_out);

			cnstr_zx >> site_pdb_res_num >> site_chain >> skip;
//ora make constraints possible for files without chain: if chain _ -> change to " "
			if ( site_chain == '_' ) site_chain = ' ';
		}
	}

//     use a filter?
	cnstr_zx >> filter_flag >> filter_value >> skip;
	if ( filter_flag == "filter" ) {
		site_constraint_filter_score = filter_value;
	} else {
		site_constraint_filter_score = 9999.;
	}

	cnstr_zx.close();
	cnstr_zx.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin res_num_from_pdb_res_num_chain
///
/// @brief
///     get the rosetta sequential residue number from the pdb residue
///     number and chain id
///
/// @detailed
///
/// @param[out]   res_num_out - out - sequential rosetta residue number
/// @param[in]   pdb_res_num_in - in - residue number in pdb file
/// @param[in]   pdb_chain_in - in - chain in the pdb file
///
/// @global_read res_chain, pdb_res_num in misc.h
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
res_num_from_pdb_res_num_chain(
	int & res_num_out,
	int const pdb_res_num_in,
	char const pdb_chain_in
)
{
	using namespace misc;
	using namespace pdb;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( res_chain(i) == pdb_chain_in && pdb_res_num(i) == pdb_res_num_in ) {
			res_num_out = i;
			return;
		}
	}

	std::cout << "WARNING: pdb residue not found " << pdb_res_num_in << ' ' <<
	 pdb_chain_in << std::endl;
	res_num_out = -1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin res_num_from_pdb_res_num_chain
///
/// @brief
///			Overloaded version of above function
///			uses the pdb information not from misc, but from a pose that is passed in
///     get the rosetta sequential residue number from the pdb residue
///     number and chain id
///
/// @detailed
///
/// @param[out]   res_num_out - out - sequential rosetta residue number
/// @param[in]	 pdb_info - in - pdb information for a specific pose
/// @param[in]   pdb_res_num_in - in - residue number in pdb file
/// @param[in]   pdb_chain_in - in - chain in the pdb file
/// @param[in]   size - in - size of area to search
///
/// @global_read res_chain, pdb_res_num in misc.h
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/// Monica Berrondo June 27 2006
/////////////////////////////////////////////////////////////////////////////////
void
res_num_from_pdb_res_num_chain(
	int & res_num_out,
	Pdb_info pdb_info,
	int const pdb_res_num_in,
	char const pdb_chain_in,
	int const size
)
{
	using namespace misc;
	using namespace pdb;

	for ( int i = 1; i <= size; ++i ) {
		if ( pdb_info.res_chain(i) == pdb_chain_in && pdb_info.pdb_res_num(i) == pdb_res_num_in ) {
			res_num_out = i;
			return;
		}
	}

	std::cout << "WARNING: pdb residue not found " << pdb_res_num_in << ' ' <<
	 pdb_chain_in << std::endl;
	res_num_out = -1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_print_site_constraints
///
/// @brief print the distance constraints to the screen
///
/// @detailed
///
/// @global_read interface.h site_data block
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_print_site_constraints()
{
	using namespace interface;
	using namespace misc;
	using namespace pdb;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) return;
	if ( !site_constraints_exist ) return;

	std::cout << "\nDocking constraints:\n" << std::endl;

	std::string buffer;

	for ( int i = 1; i <= N_site_constraints; ++i ) {
		buffer.clear();
		std::cout << "Bonus of " << site_constraint_bonus(i) << " for residues:" << std::endl;
		for ( int j = 1; j <= int( site_constraint_residues(i).size() ); ++j ) {
			int r = site_constraint_residues(i)[j];
			std::cout << space( 5 ) << residue3(r) <<
				I( 4, pdb_res_num(r) ) << res_chain(r) << " (Rosetta" <<
				I( 4, r ) << ')' << std::endl;
			buffer += right_string_of( pdb_res_num(r), 4 ) + res_chain(r) + ',';
		}

		if ( int( site_constraint_residues(i).size() ) > 0 ) {
			buffer.erase( buffer.length() - 1 ); // Erase last comma character.
			std::cout << "      so select " << buffer << std::endl;
 		}
		std::cout << std::endl;
	}

	std::cout << "site constraint filter set to " << site_constraint_filter_score
						<< "\n" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_evaluate_site_score
///
/// @brief score to measure whether interface site constraint is met
///
/// @detailed
///     The docking site constraint ensures that the interface contains
///     residues that are known to be included at the site.
///
///     Each 'site' is a vector of residues and a 'site_bonus' is the
///     score rewarded if *any* of the residues in that site are present
///     at the interface.  This score could be positive to penalize
///     residues that should not be at the interface, or negative to
///     reward known biological sites.
///
///     Be sure to call docking_evaluate_all_scores() first, so that
///     int_res is set.
///
///     JJG 8/1/1
///
/// @global_read interface.h dist_data block
///
/// @global_write docking_site_constraint_score
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_evaluate_site_score()
{
	using namespace interface;
	using namespace scores;

	docking_site_constraint_score = 0.0;

	if ( site_constraints_exist ) docking_evaluate_site_matchup(
	site_constraint_residues,
	site_constraint_bonus,docking_site_constraint_score);

	docking_site_constraint_score += docking_dist_constr_bonus();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_evaluate_site_matchup
///
/// @brief matches site_residues (desired interface)
///       with int_res (current interface)
///
/// @detailed
///     This function matches the site_residues with those in int_res.  It
///     can be used to evaluate the site_constraint score or to compare
///     the current site with the native site.
///
///     Each 'site' is a vector of residues and a 'site_bonus' is the
///     score rewarded if *any* of the residues in that site are present
///     at the interface.  This score could be positive to penalize
///     residues that should not be at the interface, or negative to
///     reward known biological sites.
///
///     Be sure to call docking_evaluate_all_scores() first, so that
///     int_res is set.
///
///     JJG 8/1/1
///
/// @param[in]   site_residues - in
/// @param[in]   site_bonus - in
/// @param[out]   score - out
///
/// @global_read docking.h - docking info
///             interface.h - int_res and site_residues
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_evaluate_site_matchup(
	FArray1Da< utility::vector1_int > site_residues,
	FArray1Da_float site_bonus,
	float & score
)
{
	using namespace interface;

	site_residues.dimension( max_site_constraints );
	site_bonus.dimension( max_site_constraints );

	score = 0.0;
	for ( int site = 1; site <= N_site_constraints; ++site ) {
		bool site_found = false;
		for ( int j = 1; j <= int( site_residues(site).size() ); ++j ) {
			if ( int_res(site_residues(site)[j]) ) {
				site_found = true;
				break;
			}
		}
		if ( site_found ) score += site_bonus(site);
	}

}





////////////////////////////////////////////////////////////////////////////////
/// @begin docking_site_constr_filter
///
/// @brief return true if structure passes the site constraints
///
/// @detailed
///
/// @return  bool
///
/// @global_read docking_site_constraint_score and site_constraint_filter_score
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
docking_site_constr_filter()
{
	using namespace interface;
	using namespace scores;

	bool docking_site_constr_filter; // Return value

	float filter_score;

	if ( site_constraints_exist ) {
		filter_score = site_constraint_filter_score;

//     if we're using a distance constraint, that gives a 100 point
//     bonus to the site_constraint_score that we have to compensate
		if ( dist_constraint_exists ) filter_score -= 100.0;

		std::cout << "Checking site constraints. Current score = " << docking_site_constraint_score << ".  Filter level = " << filter_score << std::endl;

		docking_site_constr_filter =
		 ( docking_site_constraint_score < filter_score );

	} else {
		docking_site_constr_filter = true;
	}

	return docking_site_constr_filter;
}




//------------------------------------------------------------------------------
//     FAB/ALIGNMENT CONSTRAINTS ===============================================


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fab_read_sites
///
/// @brief read alignment profile from PDB1.fab file
///
/// @detailed originally written to match antibody CDR regions at the interface,
///      these functions can be used in general to create a profile of the
///      likelihood of each residue being at the interface.
///
///     To identify the complementarity-determining region, we read in an
///     alignment to other antibodies in complex:
///
///     Fab constraint file format:
///
///1dqjL           divltqspatlsvtpgdsvslscrasq---sisnn------lhwyqqkshesprllikya
///                ..........................3...2333331..23.1.11.....1.12..33.
///1dqjL           s----qsis-gipsrfsgs--gsgtdftlsinsvetedfgmyfcqqsn--swp-----yt
///                2....3123.1.111......1..............21.......13313333...1.31
///1dqjL           fgggtkleik-radaaptvsifppsseqltsggasvvcflnnfypkdinvkwkidgserq
///                ............................................................
///1dqjH           evqlqesgpslvkp-sqtlsltcsvtgdsvts--dy-wswirkfpgnkleymgyis-ysg
///                23........................133233.3333.3...........2..3133331
///1dqjH           styyhpslksrisitrdtsknqyylqlnsvttedtatyycas-------wgg--------
///                3231112.11..1............................3.3333333333331....
///1dqjH           --------dvwgagttvtvssakttapsvyplapvcgdttgssvtlgclvkgyfpepvtl
///                .....22232..................................................
///1dqjH           twnsgslssgvhtfpavlqs-dlytlsssvtvtsstwpsqsitcnvahpasstkvdkki-
///                ............................................................
///
///
///     the numbers indicate the observations of a particular residue
///     being at the interface
///
///     more recently, the codes BTNF are used for bonus, true, neutral, false.
///     see score values for each of these in the code.
///
/// @global_read protein name
///
/// @global_write dock_fab.h
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_fab_read_sites()
{
	using namespace dock_fab;
	using namespace docking;
	using namespace files_paths;
	using namespace misc;
	using namespace pdb;
	using namespace runlevel_ns;

	if ( !multi_chain || design::dna_interface || antibody_modeler ) return;
	if ( !( fab1 || fab2 ) ) return;

	std::string filename = constraints_path + protein_name_prefix + protein_name + ".fab";

	utility::io::izstream cnstr_zx( filename );

	if ( !cnstr_zx ) {
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::cout << "ERROR: FAB ALIGNMENT FILE NOT FOUND" << std::endl;
		std::cout << "Searched for: " << cnstr_zx.filename() << std::endl;
		std::cout << "Exiting..." << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( runlevel >= standard ) std::cout
	 << "loading docking fab constraints from " << cnstr_zx.filename() << std::endl;

	// set fab residue limits
	int const fab_start_res = ( fab1 ? part_begin(1) : part_begin(2) );
	int const fab_end_res = ( fab1 ? part_end(1) : part_end(2) );

	// create the arrays to read into
	FArray1D_char fab_seq( DRange( fab_start_res, fab_end_res ) );
	FArray1D_char fab_hits( DRange( fab_start_res, fab_end_res ) );

	// read the alignment
	int fab_res = fab_start_res - 1;
	std::string fab_seq_line, fab_hits_line;
	while ( cnstr_zx ) {
		cnstr_zx >> skip( 16 ) >> bite( 60, fab_seq_line ) >> skip;
		if ( cnstr_zx ) {
			cnstr_zx >> skip( 16 ) >> bite( 60, fab_hits_line ) >> skip;
		}
		if ( cnstr_zx ) {
			std::cout << fab_seq_line << std::endl;
			std::cout << fab_hits_line << std::endl;
			for ( std::string::size_type l = 0, e = fab_seq_line.length(); l < e; ++l ) {
				if ( ! is_any_of( fab_seq_line[ l ], " -" ) ) {
					fab_seq( ++fab_res ) = uppercased( fab_seq_line[ l ] );
					fab_hits( fab_res ) = fab_hits_line[ l ];
				}
			}
		}
	}
	cnstr_zx.close();
	cnstr_zx.clear();

	// check number of alignment residues
	if ( fab_res == fab_start_res - 1 ) {
		std::cout << "no fab alignment found" << std::endl;
		std::cout << fab_seq_line << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( fab_res != fab_end_res ) {
		std::cout << "not enough residues in alignment: " << fab_res
		 << " should be " << fab_end_res << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// check the alignment & assign scores
	fab_score.dimension( DRange( fab_start_res, fab_end_res ) ); // Size fab_score
	for ( int i = fab_start_res; i <= fab_end_res; ++i ) {

		if ( fab_seq( i ) != residue1( i ) ) { // Misaligned
			std::cout << "Fab data misaligned for residue " << i << ':' << std::endl;
			std::cout << fab_seq( i ) << " in .fab file" << std::endl;
			std::cout << residue1( i ) << " in pdb res " << pdb_res_num( i ) << std::endl;
			write_res( std::cout, i );
			std::cout << std::endl;
			write_sequence( 0 );
			std::cout << "fab sequence: ";
			for ( int m = i, me = std::min( i + 3, fab_end_res ); m <= me ; ++m ) std::cout << fab_seq( m );
			std::cout << "..." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

//		std::cout << i << std::endl;
//		print_docking_interface_pack();
		if ( fab_hits( i ) == '.' || fab_hits( i ) == 'F' ) {
			fab_score( i ) = fab_bad; // don't make these contacts!
		} else if ( fab_hits( i ) == '1' || fab_hits( i ) == 'N' ) {
			fab_score( i ) = fab_neutral; // no penalty for rare contacts
		} else if ( fab_hits( i ) == '2' ) {
			fab_score( i ) = fab_ok; // slight bonus here
		} else if ( ( fab_hits( i ) >= '3' && fab_hits( i ) <= '9' ) || fab_hits( i ) == 'T' ) {
			fab_score( i ) = fab_good; // we want these contacts
		} else if ( fab_hits( i ) == 'B' ) {
			fab_score( i ) = fab_bonus; // Bonus!  for capri2 cdr3
		} else {
			std::cout << "alignment code " << fab_hits( i ) << " unknown " << i << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

//		write_res( std::cout, i );
//		std::cout << fab_seq( i ) << ' ' << i << ' ' << residue1( i ) << ' ' << i
//		 << SS( fab_score( i ) ) << std::endl;

	}

	docking_fab_print_sites();

	// given a successful alignment, we also want to load the fab-specific ENV scores
	docking_load_fab_env_table();

}




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fab_print_sites
///
/// @brief
///     print out the residues identified as likely to be at the interface
///     or almost-likely
///
/// @detailed
///
/// @global_read dock_fab.h
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_fab_print_sites()
{
	using namespace dock_fab;

	std::cout << "\nFab sites:\n" << std::endl;

	for ( int i = fab_score.l(), e = fab_score.u(); i <= e; ++i ) {
		if ( fab_score(i) <= fab_good ) {
			write_res( std::cout, i );
			std::cout << F( 6, 2, fab_score(i) ) << std::endl;
		}
	}

	std::cout << "\nClose-to Fab sites:\n" << std::endl;

	for ( int i = fab_score.l(), e = fab_score.u(); i <= e; ++i ) {
		if ( fab_score(i) != fab_bad && fab_score(i) > fab_good ) {
			write_res( std::cout, i );
			std::cout << F( 6, 2, fab_score(i) ) << std::endl;
		}
	}

	std::cout << std::endl;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin docking_calc_fab_score
///
/// @brief evaluate the docking fab/alignment score
///
/// @detailed
///     This function scores the docking fab score, or antibody-specific
///     score.  It also modifies the contact score appropriately.  The
///     contact score should already be calculated by
///     docking_calc_env_contact_scores.  (8/22/01 JJG)
///
/// @global_read dock_fab.h
///
/// @global_write docking_fab_score
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_calc_fab_score()
{
	using namespace dock_fab;
	using namespace interface;
	using namespace scores;

	docking_fab_score = 0.0;

	if ( !(fab1 || fab2) ) return;

	int fab_interface_residues = 0;

	for ( int i = fab_score.l(), e = fab_score.u(); i <= e; ++i ) {
		if ( int_res(i) ) {
			++fab_interface_residues;
			docking_fab_score += fab_score(i);
		}
	}
}



////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fab_filter
///
/// @brief return true for structures that pass the fab filter.
///
/// @detailed
///     Good means no forbidden fab residues make contact across the interface.
///     8/22/01 JJG
///
///     Let's require two bad contacts before we ding something completely
///     8/27
///
/// @return  bool
///
/// @global_read dock_fab, protein position
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
docking_fab_filter()
{
	using namespace dock_fab;
	using namespace interface;

	bool docking_fab_filter = true; // Return value

	if ( !(fab1 || fab2) ) return docking_fab_filter;

	int bad_residues = 0;

	for ( int i = fab_score.l(), e = fab_score.u(); i <= e; ++i ) {
		if ( int_res(i) ) {
			if ( fab_score(i) > fab_neutral ) {
				++bad_residues;
				if ( bad_residues >= 2 ) {
					docking_fab_filter = false;
					return docking_fab_filter;
				}
			}
		}
	}

	return docking_fab_filter;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_load_fab_env_table
///
/// @brief change the data for the env score for antibodies
///
/// @detailed Ora compiled antibody-specific env scores in 2001 or 2002
///
/// @global_read log_docking_env_fab
///
/// @global_write log_docking_env
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 10/10/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_load_fab_env_table()
{
//     replace the ENV score table with statistics particular to antibody
//     interfaces

	using namespace docking_db;
//KMa phospho_ser MAX_AA_PLUS=21
	std::cout << "Loading antibody-specific ENV scores\n" << std::endl;
	for ( int i = 1; i <= param::MAX_AA_PLUS(); ++i ) {
		for ( int j = 1; j <= 4; ++j ) {
			log_docking_env(j,i) = 0.33f * log_docking_env_fab(j,i);
		}
	}
}
