// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

// Rosetta Headers
#include "etable_manager.h"
#include "etable_class.h"
#include "etable.h"
#include "fullatom_energy.h"
#include "fullatom_setup.h"
#include "misc.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "score.h"
#include "weights_manager.h"

// ObjexxFCL Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <iostream>

namespace etable_manager_ns {
	EtableManager Etable_db;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin EtableManager::EtableManager
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @remarks
///
/// @refrences
///
/// @authors John Karanicolas
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
EtableManager::EtableManager()
{

	Etable empty_Etable;
	EtableMap_[ ETAB_STANDARD] = empty_Etable;
	EtableMap_[ ETAB_SOFT_REP] = empty_Etable;
	EtableMap_[ ETAB_SMALL_RADII] = empty_Etable;
	EtableMap_[ ETAB_AW] = empty_Etable;
	EtableMap_[ ETAB_BW] = empty_Etable;

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin setCurrentEtable
///
/// @brief
///
///   Note: returns true if the Etable has changed, false otherwise
///
/// @detailed
///
/// @global_read
///
/// @remarks
///
/// @refrences
///
/// @authors John Karanicolas
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool setCurrentEtable( etable_id const & etab_id )
{

	using namespace etable_manager_ns;
	using namespace pdbstatistics_pack;

	Etable * pStartingEtable = pCurrentEtable;

	EtableMap_Iter p = Etable_db.find( etab_id );
	if ( p == Etable_db.end() ) {
		std::cout << "ERROR in setCurrentEtable()!" << std::endl;
		std::cout << "Could not find desired Etable id " << etab_id << std::endl;
		std::cout << "ABORTING..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	pCurrentEtable = &(p->second);

	if ( pStartingEtable == pCurrentEtable ) return false;

	if ( etab_id == ETAB_SOFT_REP ) {
		etable::fa_lj_soft_initializer(etable::fa_lj_radius);
	} else {
		etable::fa_lj_radius_initializer(etable::fa_lj_radius);
	}

	etable::fa_lk_dgfree_initializer( etable::fa_lk_dgfree );
	if ( ( etab_id == ETAB_AW ) || ( etab_id == ETAB_BW ) ) {
		// Note: hard-coded atom-type here!!
		etable::fa_lk_dgfree( 8 ) =   -9.0; // N in His side-chain,*** too many buried His***
	}

	if ( etab_id == ETAB_STANDARD ) {
		bool const setup_already_done = pCurrentEtable->done_setup();
		pCurrentEtable->setup_Etable(true, 0.6);
		if ( ! setup_already_done ) {
			pCurrentEtable->read_Etable();
			pCurrentEtable->write_Etable();
		}
	} else if ( etab_id == ETAB_SOFT_REP ) {
		pCurrentEtable->setup_Etable(false, 0.91);
	} else if ( etab_id == ETAB_SMALL_RADII ) {
		pCurrentEtable->setup_Etable(false, 0.6, 0.95);
	} else if ( etab_id == ETAB_AW ) {
		pCurrentEtable->setup_Etable(false, 0.85);
	} else if ( etab_id == ETAB_BW ) {
		pCurrentEtable->setup_Etable(false,0.85);
	} else {
		std::cout << "ERROR in setCurrentEtable()!" << std::endl;
		std::cout << "Unsupported Etable id " << etab_id << std::endl;
		std::cout << "ABORTING..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin disable_packing_etables
///
/// @brief turn soft_rep / use_aw / use_bw energy tables off, restore standard
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk Aug. 24/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
disable_packing_etables(
												int total_residue,
												FArray1D_int const & res,
												FArray1D_int const & res_variant,
												FArray3D_float const & full_coord,
												bool const suppress_output
)
{

	if ( ! param_pack::packer_logical::use_packing_etables_always ) {
		bool const rescore_needed = setCurrentEtable( ETAB_STANDARD );
		if ( rescore_needed ) {
			RetrieveWeightsToCurrent( PW_STANDARD );
			apply_packer_weights( true, suppress_output );
			score_enable_rotamer_trials( false );
			score_set_new_pose();
			fullatom_energy_full(res, res_variant, full_coord, total_residue);
			score_enable_rotamer_trials( true );
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin enable_packing_etables
///
/// @brief turn on soft_rep / use_aw / use_bw energy etables
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk Aug. 24/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
enable_packing_etables(
	int total_residue,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord,
	bool const suppress_output
)
{

	//using namespace misc;

	bool rescore_needed = false;
	if ( param_pack::packer_logical::soft_rep ) {
		rescore_needed = setCurrentEtable( ETAB_SOFT_REP );
	} else if ( param_pack::packer_logical::small_radii ) {
		rescore_needed = setCurrentEtable( ETAB_SMALL_RADII );
	} else if ( param_pack::packer_logical::use_aw ) {
		rescore_needed = setCurrentEtable( ETAB_AW );
	} else if ( param_pack::packer_logical::use_bw ) {
		rescore_needed = setCurrentEtable( ETAB_BW );
	}

	if ( rescore_needed ) {
		RetrieveWeightsToCurrent( PW_STANDARD );
		apply_packer_weights( false, suppress_output );
		score_enable_rotamer_trials( false );
		score_set_new_pose();
		fullatom_energy_full(res, res_variant, full_coord, total_residue);
		score_enable_rotamer_trials( true );
	}

}


