// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 12404 $
//  $Date: 2007-02-01 15:47:11 -0800 (Thu, 01 Feb 2007) $
//  $Author: bblum $

// Rosetta Headers
#include "featurizer_classes.h"

// ObjexxFCL Headers

// C++ Headers

namespace featurizer_ns {

	// stream inserter for features
	std::ostream & operator <<( std::ostream & out, Pairing_feature const & f )
	{
		// now internal whitespace in this guy:
		std::ostringstream os;
		os << f.pos1 << ',' << f.pos2 << ',' << f.o() << ",ex:" << f.extent[0] << ',' << f.extent[1] << ',' << f.extent[2] << ',' << f.extent[3];
		out << std::right << std::setw( 9 ) << os.str();
		return out;
	}

	Pairing_feature::Pairing_feature ( int const i, int const j, int const o, int const p ):
		Beta_feature( i, j, o, p )
	{
		initialize_shifts(o);
	}

	Pairing_feature::Pairing_feature ( int const i, int const j, int const maxi, int const o, int const p ):
		Beta_feature( i, j, o, p ),
		maxpos(maxi)
	{
		initialize_shifts(o);
	}

	Pairing_feature::~Pairing_feature() {
	}

	void Pairing_feature::initialize_shifts(int orientation) {
		DIFF1[0] = -1; DIFF1[1] = -1; DIFF1[2] = 1; DIFF1[3] = 1;
		DIFF2[0] = -1; DIFF2[1] = 1; DIFF2[2] = 1; DIFF2[3] = -1;
		if(orientation == PARALLEL) {
			extent[MM] = extent[PP] = 2;
			extent[MP] = extent[PM] = 1;
		} else if(orientation == ANTIPARALLEL) {
			extent[MM] = extent[PP] = 1;
			extent[MP] = extent[PM] = 2;
		}
	}

	// The lowest register shift index (can be changed, other stuff will
	// change smartly
	int Pairing_feature::shiftmin() {
		int dir;
		if(orientation == PARALLEL)
			dir = PM;
		else
			dir = PP;
		return -extent[(dir+2)%4];
	}

	// The highest register shift index
	int Pairing_feature::shiftmax() {
		int dir;
		if(orientation == PARALLEL)
			dir = PM;
		else
			dir = PP;
		return extent[dir];
	}

	bool Pairing_feature::raw_contains_square(const int i1, const int i2) {
		for(int dir=0; dir <4; dir++)
			if(!satisfies_constraint(i1,i2,dir))
				return false;
		return true;
	}

	bool Pairing_feature::contains_square(const int i1, const int i2) {
		return raw_contains_square(i1, i2) && legal_square(i1,i2);
	}

	bool Pairing_feature::satisfies_constraint(const int i1, const int i2, const int dir) {
		int off1 = i1 - pos1, off2 = i2 - pos2;
		if(DIFF1[dir]*off1 + DIFF2[dir]*off2 <= extent[dir])
			return true;
		else
			return false;
	}

	// Return the first dimension of the starting corner for the dir side.
	// The starting corner is the one between dir and dir-1 (so it's on the
	// clockwise side).
	void Pairing_feature::get_start_corner(int &i1, int &i2, int dir) {
		get_start_corner(i1, i2, dir, extent[dir], extent[(dir+3)%4]);
		while(!legal_square(i1,i2) && raw_contains_square(i1,i2))
			dir_move(i1,i2, dir);
	}

	// Return the first dimension of the starting corner given arbitrary
	// extensions for the side in the specified dimension and the previous
	// dimension.
	void Pairing_feature::get_start_corner(int &i1, int &i2, int dir, int dirextent, int backextent) {
		i1=pos1 + (dirextent * DIFF1[dir] +
				backextent * DIFF1[(dir+3)%4] ) / 2 ;
		i1 -= DIFF1[(dir+1)%4]; // a little cushion
		i2 = matching_coord2(i1, dir, dirextent);
		while(!satisfies_constraint(i1,i2, (dir+3)%4))
			dir_move(i1, i2, dir);
	}

	void Pairing_feature::dir_move(int &i1, int &i2, int dir) {
		i1 += DIFF1[(dir+1)%4];
		i2 += DIFF2[(dir+1)%4];
	}

	// Returns the 2nd coordinate matching the given 1st coordinate for
	// a point displaced by extent along direction dir.
	int Pairing_feature::matching_coord2(int offset1, int dir, int extent) {
		return pos2 + DIFF2[dir] * (extent - (offset1-pos1)*DIFF1[dir]);
	}

	void Pairing_feature::get_shift_start_corner(int &i1, int &i2, int s) {
		int dir;
		if(orientation == PARALLEL)
			dir = PM;
		else
			dir = PP;
		get_start_corner(i1, i2, dir, s, extent[(dir+3)%4]);
		while(!legal_square(i1,i2) && raw_contains_square(i1,i2))
			shift_move(i1,i2);
	}

	void Pairing_feature::shift_move(int &i1, int &i2) {
		int dir;
		if(orientation == PARALLEL)
			dir = PM;
		else
			dir = PP;
		dir_move(i1,i2,dir);
	}

	bool Pairing_feature::legal_square(int i1, int i2) {
		if(i1 <= 0 || i1 > maxpos || i2 <= 0 || i2 > maxpos)
			return false;
		if((i1 - i2 >= 0 && pos1 - pos2 < 0) || (i1 - i2 <= 0 && pos1 - pos2 > 0))
			return false;
		return true;
	}

	bool Pairing_feature::legal() {
		int i1, i2;
		for(int dir=0; dir < 4; dir++) {
			get_start_corner(i1,i2,dir);
			if(!raw_contains_square(i1,i2))
				return false;
		}
		return true;
	}

	bool Pairing_feature::grow(int dir) {
		extent[dir]++;
		if(!legal()) {
			extent[dir]--;
			return false;
		}
		return true;
	}

	bool Pairing_feature::shrink(int dir) {
		if(extent[dir] > 0) {
			extent[dir]--;
			return true;
		}
		return false;
	}

	float Pairing_feature::match(Pairing_feature &other) {
		float total = 0.0;
		if(other.orientation != orientation)
			return total;
		int m = extent[0];
		for(int i=1; i < 4; i++)
			if(extent[i] > m)
				m = extent[i];
		for(int i1=pos1-m; i1 <= pos1+m; i1++)
			for(int i2=pos2-m; i2 <= pos2+m; i2++)
				if(other.contains_square(i1, i2) && contains_square(i1,i2))
					total += 1.0;
		return total;
	}

	void Pairing_feature::merge(const Pairing_feature &other) {
		for(int i = 0; i < 4; i++) {
			int other_ext = (other.pos1-pos1)*DIFF1[i]+(other.pos2-pos2)*DIFF2[i]+other.extent[i];
			while(other_ext > extent[i])
				extent[i]++;
		}
	}

	void Pairing_feature::outputEdges(std::ostream &out) {
		int i1, i2;
		for(int d=0; d < 4; d++) {
			for(get_start_corner(i1, i2, d);
					contains_square(i1,i2);
					dir_move(i1,i2,d))
				if(antiparallel())
					out << i1 << ' ' << i2 << '\n';
				else
					out << i2 << ' ' << i1 << '\n';
		}
	}

	int Pairing_feature::center1() {
		int dir;
		if(parallel())
			dir = PM;
		else
			dir = PP;
		return pos1 + DIFF1[(dir+2)%4] * (extent[(dir+2)%4]+1)/2;
	}

	int Pairing_feature::center2() {
		int dir;
		if(parallel())
			dir = PM;
		else
			dir = PP;
		return pos2 + DIFF2[(dir+2)%4] * extent[(dir+2)%4]/2;
	}

	void Pairing_feature::recenter(const int i1, const int i2) {
		assert(raw_contains_square(i1, i2));
		for(int dir=0; dir < 4; dir++) {
			extent[dir] -= (i1-pos1) * DIFF1[dir] + (i2-pos2) * DIFF2[dir];
		}
		pos1 = i1;
		pos2 = i2;
	}
}
