// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15185 $
//  $Date: 2007-05-25 19:47:34 -0400 (Fri, 25 May 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "fold_tree.h"
#include "random_numbers.h" // ran3

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <iostream>
#include <list>

// Utility Headers
#include <utility/basic_sys_util.hh>

namespace pose_ns {
	/////////////////////////////////////////////////////////////////////////////
	// these two should be inverses:
	std::ostream & operator <<( std::ostream & os, const Edge & e ) {
		os << " EDGE " << e.start << ' ' << e.stop << ' ' << e.label << ' ';
		return os;
	}
	/////////////////////////////////////////////////////////////////////////////
	std::istream & operator >>( std::istream & is, Edge & e ) {
		std::string tag;
		is >> tag;
		if ( tag != "EDGE" ) {
			is.setstate( std::ios_base::failbit );
			return is;
		}
		is >> e.start >> e.stop >> e.label;
		return is;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool operator< ( Edge const & a, Edge const & b ) {
		return ( a.start <  b.start ||
						 a.start == b.start && a.stop <  b.stop ||
						 a.start == b.start && a.stop == b.stop && a.label < b.label );
	}

	/////////////////////////////////////////////////////////////////////////////
	bool operator== ( Edge const & a, Edge const & b ) {
		return ( a.start == b.start && a.stop == b.stop && a.label == b.label );
	}

	/////////////////////////////////////////////////////////////////////////////
	bool operator!= ( Edge const & a, Edge const & b ) {
		return ( a.start != b.start || a.stop != b.stop || a.label != b.label );
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	operator==(
		Fold_tree const & a,
		Fold_tree const & b
	)
	{
		if ( a.edge_list.size() != b.edge_list.size() ) return false;
		for ( Fold_tree::const_iterator
						edge_a( a.edge_list.begin() ), a_end( a.edge_list.end() ),
						edge_b( b.edge_list.begin() );
					edge_a != a_end; ++edge_a, ++edge_b ) {
			if ( *edge_a != *edge_b ) return false;
		}
		return true;
	}


	/////////////////////////////////////////////////////////////////////////////
	void Fold_tree::add_edge( int const start, int const stop, int const label )
	{
		if ( start == stop ) {
			//std::cout << "no self edges in fold_trees: start == stop == " <<
			//start << std::endl;
			return;
		}
		new_topology = true; // book-keeping
		edge_list.push_back( Edge( start,stop,label) );
	}

	/////////////////////////////////////////////////////////////////////////////
	void Fold_tree::delete_edge( Fold_tree::iterator edge )
	{
		new_topology = true; // book-keeping
		if ( edge < edge_list.begin() || edge >= edge_list.end() ) {
			std::cout << "Fold_tree::delete_edge(...) edge not contained in edge_list" <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		edge_list.erase( edge );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Fold_tree::delete_edge(
		Edge const & edge
	)
	{
		delete_unordered_edge( edge.start, edge.stop, edge.label );
	}

	/////////////////////////////////////////////////////////////////////////////
	void Fold_tree::delete_unordered_edge(
		int const start,
		int const stop,
		int const label
	)
	{
		new_topology = true;
		bool found(false);
		for ( Fold_tree::iterator it=edge_list.begin(), it_end=edge_list.end();
				 it != it_end; ++it ) {
			if ( it->label == label &&
					 ( ( it->start == start && it->stop == stop ) ||
						 ( it->start == stop  && it->stop == start ) ) ) {
				edge_list.erase( it );
				found = true;
				break;
			}
		}
		if ( !found ) {
			std::cout << "Fold_tree::delete_unordered_edge(...) edge not in tree: " <<
				' ' << start << ' ' << stop << ' ' << label << std::endl;
			std::cout << *this;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	//////////////////////////////////////////////////////////////////
	// assumes that the segment is completely contained in a single edge
	// of the tree
	void
	Fold_tree::delete_segment(
		int const seg_begin,
		int const seg_end
	)
	{
		int const n2c(1), c2n(-1);
		std::cout << "Fold_tree::delete_segment: " << seg_begin << ' ' <<
			seg_end << std::endl;
		int const size( seg_end - seg_begin + 1 );

		FArray1D_int mapping( nres, -1 );
		for ( int i=1; i<= nres; ++i ) {
			if ( i < seg_begin ) {
				mapping(i) = i;
			} else if ( i > seg_end ) {
				mapping(i) = i - size;
			}
		}

		std::vector< Edge > new_edge_list;

		for ( iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			int pos1( mapping( it->start ) );
			int pos2( mapping( it->stop  ) );
			int const dir( it->start < it->stop ? n2c :c2n );
			if ( pos1 == -1 || pos2 == -1 ) assert( !it->is_jump() );

			if ( pos1 == -1 ) {
				assert( dir == n2c && it->start == seg_begin && it->stop > seg_end ||
								dir == c2n && it->start == seg_end   && it->stop < seg_begin );
				if ( dir == n2c ) {
					pos1 = seg_begin; // n2c
				} else {
					pos1 = seg_begin - 1; // c2n
				}
			} else if ( pos2 == -1 ) {
				assert( dir == c2n && it->stop == seg_begin && it->start > seg_end ||
								dir == n2c && it->stop == seg_end   && it->start < seg_begin );
				if ( dir == c2n ) {
					pos2 = seg_begin; // c2n
				} else {
					pos2 = seg_begin - 1; // n2c
				}
			}
			if ( pos1 != pos2 ) {
				new_edge_list.push_back( Edge( pos1, pos2, it->label ) );
				//std::cout << "remap edge: " << *it << ' ' << pos1 << ' ' << pos2 <<
				//std::endl;
			}
		}

		new_topology = true;
		edge_list = new_edge_list;
	}

	//////////////////////////////////////////////////////////////////
	// should this set new_topology TRUE???
	//
	void Fold_tree::update_edge_label(
		int const start,
		int const stop,
		int const old_label,
		int const new_label
	)
	{
		bool found(false);
		for ( Fold_tree::iterator it=edge_list.begin(), it_end=edge_list.end();
				 it != it_end; ++it ) {
			if ( it->label == old_label &&
					 ( ( it->start == start && it->stop == stop ) ||
						 ( it->start == stop  && it->stop == start ) ) ) {
				it->label = new_label;
				found = true;
				break;
			}
		}
		if ( !found ) {
			std::cout << "Fold_tree::update_edge_label(...) edge not in tree: " <<
				' ' << start << ' ' << stop << ' ' << old_label << std::endl;
			std::cout << *this;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}


	///////////////////////////////////////////////////////
	// after deleting a jump, there may be vertices of the
	// tree which are neither jumps nor cutpoints
	//
	// delete them!
	//
	void Fold_tree::delete_extra_vertices()
	{
		// want to use is_jump_point, is_cutpoint
		// so ensure that this data is up-to-date
		check_topology();

		//
		while ( true ) {
			int kill_vertex( 0 );
			for ( iterator it=edge_list.begin(),it_end = edge_list.end();
						it != it_end && !kill_vertex; ++it ) {
				// are either of these vertices extraneous ( neither jump-point nor cutpoint)
				if ( !is_jump_point( it->start ) &&
						 !is_cutpoint  ( it->start ) &&
						 !is_cutpoint  ( it->start-1 ) ) kill_vertex = it->start;
				if ( !is_jump_point( it->stop ) &&
						 !is_cutpoint  ( it->stop ) &&
						 !is_cutpoint  ( it->stop-1  ) ) kill_vertex = it->stop;
				if ( kill_vertex ) break;
			}
			if ( !kill_vertex ) break;
			std::cout << "delete vertex: " << kill_vertex << std::endl;
			int nbounds(0);
			FArray1D_int bounds(2,0);
			for ( iterator it=edge_list.begin(),it_end = edge_list.end();
						it != it_end && nbounds<2; ++it ) {
				if ( it->start == kill_vertex ) {
					nbounds++;
					bounds( nbounds ) = it->stop;
				} else if ( it->stop == kill_vertex ) {
					nbounds++;
					bounds( nbounds ) = it->start;
				}
			}
			std::cout << "bounds: " << bounds(1) << ' ' << bounds(2) << '\n' << *this;
			delete_unordered_edge( bounds(1), kill_vertex, PEPTIDE );
			delete_unordered_edge( bounds(2), kill_vertex, PEPTIDE );
			add_edge( bounds(1), bounds(2), PEPTIDE );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	int
	Fold_tree::downstream_jump_residue(
		int const jump_number
	) const
	{
		assert( jump_number >= 1 && jump_number <= num_jump );
		for ( const_iterator it = edge_list.begin(),
						it_end = edge_list.end(); it != it_end; ++it ) {
			if ( it->label == jump_number ) return it->stop;
		}
		return 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Fold_tree::upstream_jump_residue(
		int const jump_number
	) const
	{
		assert( jump_number >= 1 && jump_number <= num_jump );
		for ( const_iterator it = edge_list.begin(),
						it_end = edge_list.end(); it != it_end; ++it ) {
			if ( it->label == jump_number ) return it->start;
		}
		return 0;
	}

	///////////////////////////////////////////////////////
	// will this be too slow? creates a new edge_list object,
	// then at the end copies it into edge_list;
	bool Fold_tree::reorder( int const start_residue )
	{
		if ( new_topology ) update_nres(); // need nres

		// static FArray ==> only re-allocate upon changes of nres
		static FArray1D_bool linked;
		if ( nres != int( linked.size1() ) ) {
			linked.dimension( nres );
		}

		if ( ! allow_reorder ) return false;

		Edge_list new_edge_list;

		// keep track of which residues have been added to the new list
		linked = false;

		linked( start_residue) = true;

		bool new_member (true);

		while ( new_member ) {
			new_member = false;
			for ( const_iterator it = edge_list.begin(),
							it_end = edge_list.end(); it != it_end; ++it) {
				int const start ( it->start);
				int const stop ( it->stop);
				int const label ( it->label);

				if ( linked( start ) && !linked( stop ) ) {
					new_edge_list.push_back( Edge( start, stop, label ) );
					linked( stop ) = true;
					new_member = true;
				} else if ( linked( stop ) && !linked( start ) ) {
					new_edge_list.push_back( Edge( stop, start, label ) );
					linked( start ) = true;
					new_member = true;
				}
			}
		}// while ( new_member )

		if ( new_edge_list.size() != edge_list.size() ) {
			std::cout << "Fold_tree::reorder() failed, new/old edge_list size mismatch\n";
			std::cout << edge_list.size() << ' ' << new_edge_list.size() << std::endl;
			std::cout << *this;
			return false;
		}

		// slow: clear edge_list then copy from new_edge_list...
		edge_list = new_edge_list;
		new_order = true;
		return true; // success
	} // Fold_tree::reorder(...)


	///////////////////////////////////////////////////////////////////////
	// ensure that stored data depending on the topology are up-to-date
	// modifications by add_edge, delete_edge, reorder, etcetc
	// are indicated by setting new_topology and/or new_order to true

	// make a simple, 1->total_residue tree
	void Fold_tree::simple_tree( int const nres_in ) {
		new_topology = true; // ensure that derived data are re-calculated
		edge_list.clear();
		add_edge(1, nres_in, PEPTIDE); // from 1->total_residue
	}


	/////////////////////////////////////////////////////////////////////////////
	// add a new jump to an existing fold tree
	//
	// returns the jump_number
	//
	int
	Fold_tree::new_jump(
		int const jump_pos1,
		int const jump_pos2,
		int const cutpoint
	)
	{
		check_topology(); //ensure jump_point etc is up to date

		// copy jump_point, cutpoint arrays
		int const njump( num_jump + 1 );
		FArray2D_int jpoint( 2, njump );
		FArray1D_int cuts( njump );

		for ( int i=1; i<= njump-1; ++i ) {
			jpoint(1,i) = jump_point(1,i);
			jpoint(2,i) = jump_point(2,i);
			cuts(i) = fold_tree_cutpoint(i);
		}
		jpoint(1,njump) = std::min( jump_pos1, jump_pos2 );
		jpoint(2,njump) = std::max( jump_pos1, jump_pos2 );
		cuts(njump) = cutpoint;

		int const root( edge_list.begin()->start );
		tree_from_jumps_and_cuts( nres, njump, jpoint, cuts );

		reorder( root );
		new_topology = true;
		return njump;
	}


	/////////////////////////////////////////////////////////////////////////////
	// this assumes that we can make a tree, ie that the number of cuts
	// is equal to the number of jumps
	bool Fold_tree::tree_from_jumps_and_cuts(
		int const nres_in,
		int const num_jump_in,
		FArray2D_int const & jump_point_in,
		FArray1D_int const & cuts,
		bool const verbose /* = true */
	)
	{
		// tells routines that need derived data to re-update,eg connected() needs nres
		new_topology = true;

		if ( num_jump_in == 0 ) {
			// make a simple tree. this could also have been done by simple_tree()
			edge_list.clear();
			add_edge( 1, nres_in, PEPTIDE );
			return true; // success
		}

		///jjh Report jumps for reporting purposes
		if (verbose){
			for ( int i = 1, iend = num_jump_in ; i <= iend ; ++i ) {
				std::cout << "Jump #" << i << " from " << jump_point_in( 1, i ) << " to " <<
					jump_point_in( 2, i ) << std::endl;
			}
		}
		// make a list of the unique jump_points in increasing order:
		// so we can construct the peptide edges
		typedef std::list< int > Int_list;
		Int_list vertex_list;
		vertex_list.push_back( 1 );
		FArray1D_bool is_cut( nres_in, false ); // keep track of cuts
		for ( int i = 1; i <= num_jump_in; ++i ) {
			for ( int j = 1; j <= 2; ++j ) {
				int const pos ( jump_point_in(j,i) );
				assert( pos >= 1 && pos <= nres_in );
				vertex_list.push_back( pos );
			}
			assert( jump_point_in(1,i) < jump_point_in(2,i) );
			int const cutpoint( cuts(i) );
			assert( cutpoint >= 1 && cutpoint < nres_in );
			is_cut( cutpoint ) = true;
			vertex_list.push_back( cutpoint );
			vertex_list.push_back( cutpoint+1 );
		}
		vertex_list.push_back( nres_in );

		vertex_list.sort();
		vertex_list.unique(); // something like that...

		// start building the tree, add peptide edges
		edge_list.clear();
		for ( Int_list::iterator it = vertex_list.begin(),
						it_end = vertex_list.end(); it != it_end; ++it ) {
			Int_list::iterator it_next (it);
			++it_next;
			if ( it_next == it_end ) break;

			int const start ( *it );
			int const stop ( *it_next );
			assert( start >= 1 && start < stop && stop <= nres_in );
			if ( !is_cut(start) ) {
				add_edge( start, stop, PEPTIDE );
			} else {
				assert( stop == start + 1 );
			}
		}

		//
		//assert( edge_list[0].start == 1 &&
		// edge_list[ edge_list.size() -1 ].stop == nres_in );

		// now add the edges corresponding to jumps
		for ( int i=1; i<= num_jump_in; ++i ) {
			add_edge( jump_point_in(1,i), jump_point_in(2,i), i );
		}
		reorder(1);
		assert( check_fold_tree() ); // tree should be valid now
		return true;
	}

	//////////////////////////////////////////////////////////////////////////////
	// pick a set of cutpoints
	// note that during the process of building out tree, we cant use
	// any of the derived data, since conceptually the derived data comes
	// after the tree
	bool Fold_tree::random_tree_from_jump_points(
																							 int const nres_in,
																							 int const num_jump_in,
																							 FArray2D_int const & jump_point_in,
																							 FArray1D_float const & cut_bias)
	{
		std::vector< int > obligate_cut_points;
		return random_tree_from_jump_points( nres_in, num_jump_in, jump_point_in, obligate_cut_points, cut_bias );
	}

	bool Fold_tree::random_tree_from_jump_points(
		int const nres_in,
		int const num_jump_in,
		FArray2D_int const & jump_point_in,
		std::vector< int > const & obligate_cut_points,
		FArray1D_float const & cut_bias
		)
	{
		// tells routines that need derived data to re-update,eg connected() needs nres
		new_topology = true;

		if ( num_jump_in == 0 ) {
			// make a simple tree. this could also have been done by simple_tree()
			edge_list.clear();
			add_edge( 1, nres_in, PEPTIDE );
			return true; // success
		}

		// this array is passed in to the edge-cutting routines
		FArray1D_float cut_bias_sum( DRange(0,nres_in) );
		cut_bias_sum(0) = 0.0;
		for ( int i=1; i<= nres_in; ++i ) {
			cut_bias_sum(i) = cut_bias_sum( i-1 ) + cut_bias(i);
		}


		// make a list of the unique jump_points in increasing order:
		// so we can construct the peptide edges
		typedef std::list< int > Int_list;
		Int_list jump_list;
		for ( int i = 1; i <= num_jump_in; ++i ) {
			for ( int j = 1; j <= 2; ++j ) {
				jump_list.push_back( jump_point_in(j,i) );
			}
		}

		jump_list.sort();
		jump_list.unique(); // something like that...

		// start building the tree, add peptide edges
		edge_list.clear();

		//Add beginning edge.
		int const jump_stop( *jump_list.begin() );
		if (jump_stop > 1) add_edge( 1, jump_stop, PEPTIDE );

		//Add intervening segments.
		for ( Int_list::iterator it = jump_list.begin(),
						it_end = jump_list.end(); it != it_end; ++it ) {
			Int_list::iterator it_next (it);
			++it_next;
			if ( it_next == it_end ) break;

			int const start ( *it );
			int const stop ( *it_next );
			int const label ( -2 );

			assert( start >= 1 && start < stop && stop <= nres_in );
			add_edge( start, stop, label );
		}

		//Add final edge.
		int const jump_start( edge_list[ edge_list.size() - 1].stop );
		if (jump_start < nres_in) add_edge(  jump_start, nres_in, PEPTIDE );

		assert( edge_list[0].start == 1 &&
						 edge_list[ edge_list.size() -1 ].stop == nres_in );

		// now add the edges corresponding to jumps
		for ( int i=1; i<= num_jump_in; ++i ) {
			add_edge( jump_point_in(1,i), jump_point_in(2,i), i );
		}

		//Add cuts that the user has given ... there could be none.
		int const num_user_cut_points = obligate_cut_points.size();
		for ( int i = 0; i < num_user_cut_points; i++ ) {
			int const cut_point = obligate_cut_points[i];
			bool const success = cut_edge( cut_point );

			if (!success) {
				// this is a problem!
				std::cout << "Problem with user-defined cutpoint: " << cut_point << std::endl;
				std::cerr << "Problem with user-defined cutpoint: " << cut_point << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

		}

		// keep cutting edges until we get a tree
		bool is_a_tree ( false );
		while ( ! is_a_tree ) {
			update_edge_labels();
			is_a_tree = true;
			for ( iterator it = edge_list.begin(), it_end = edge_list.end(); it != it_end; ++it) {
				if ( it->label == -2 ) {
					is_a_tree = false;
					break;
				}
			}

			if ( ! is_a_tree ) {
				if ( ! cut_random_edge( cut_bias_sum, nres_in ) ) {
					// failed to cut
					std::cout << "failure in Fold_tree::choose_random_cuts(...)" << std::endl;
					return false; // signal failure
				}
			}
		} // while ( ! is_a_tree )

		reorder(1);
		assert( check_fold_tree() ); // tree should be valid now
		return true;
	} // random_tree_from_jump_points(...)

	/////////////////////////////////////////////////////////////////////////////
	// slow: just for convenience routines
	std::vector< int >
	Fold_tree::cutpoints() const
	{
		check_topology();
		std::vector< int > cuts;
		for( int i=1; i<= num_fold_tree_cutpoint; ++i ) {
			cuts.push_back( fold_tree_cutpoint(i) );
		}
		return cuts;
	}

	/////////////////////////////////////////////////////////////////////////////

	// private:
	//------------------------------------------------------------------------------
	// this routine assigns labels to the edges of a graph based
	// on whether or not those edges are separating -- ie whether
	// they can be cut without disconnecting the graph.
	// we know we have a tree when all the edges are separating
	//
	// edge labels:
	// +N means that the edge corresponds to jump #N (=> uncuttable)
	// -2 means cuttable
	// -1 means separating == PEPTIDE
	// 0 means cut
	//
	// we assume that the only possible change in edge labelling that we
	// need to make is from a -2 to a -1
	// ie, the -1's are still correct
	// also, there shouldn't be any 0's before this is called
	// 0's are for communicating between this function and
	// the logical function connected_graph(g)

	void Fold_tree::update_edge_labels() {
		assert( PEPTIDE != -2 );
		for ( iterator it = edge_list.begin(), it_end = edge_list.end(); it != it_end; ++it ) {
			if ( it->label == -2 ) { // labelled as not separating
				it->label = 0;
				if ( ! connected() ) {
					it->label = PEPTIDE; // now it's separating
				} else {
					it->label = -2; // still not separating
				}
			} else if ( it->label == 0 ) { // debug
				std::cout << "zero edge label in graph:" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	}

	///////////////////////////////////////////////////////////
	bool Fold_tree::cut_edge( int const cut_point ) {
		for ( iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			if ( it ->label < 0 &&
					 ( ( it->start <= cut_point && it->stop  >= cut_point+1 ) ||
						 ( it->stop  <= cut_point && it->start >= cut_point+1 ) ) ) {
				if ( it->label == PEPTIDE ) {
					// you cant cut at this cutpoint
					break;
				} else {
					assert( it->label == -2 );

					int const start( std::min( it->start, it->stop) );
					int const stop ( std::max( it->start, it->stop) );
					delete_edge( it );
					if ( start  < cut_point ) add_edge( start, cut_point, PEPTIDE); // separating
					if ( cut_point+1 < stop ) add_edge( cut_point+1, stop, PEPTIDE); // separating
					return true; // successfully cut the tree
				}
			}
		} // loop over edge_list
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	// private: returns true if success, false if failure
	// operates on the edge_list
	// doesnt assume any derived data is up-to-date
	bool Fold_tree::cut_random_edge( FArray1D_float const & cut_bias_sum, int const nres_in ) {
		int tries(0);

		while ( tries < 100000 ) {
			++tries;

			int const cut_point ( pick_loopy_cutpoint( nres_in, cut_bias_sum ) );
			bool success = cut_edge( cut_point );
			if (success) return true;

		} // keep trying

		return false; // too many tries
	}

	/////////////////////////////////////////////////////////////////////////////
	///
	// after we delete a jump, we may want to re-number the others
	// of course this will invalidate the jump_transform array of
	// any pose with this fold_tree, so be sure to call jumps_from_positions
	// or something
	void Fold_tree::renumber_jumps() {
		int counter(0);
		new_topology = true;
		for ( iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			if ( it->is_jump() ) {
				++counter;
				std::cout << "renumber jumps:: from,to " << it->label << ' ' <<
					counter << std::endl;
				it->label = counter;
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// returns true if fold_tree is connected
	// doesnt assume that the fold_tree is in valid folding order, or even a tree
	bool Fold_tree::connected() const {
		static FArray1D_bool linked;
		if ( new_topology ) update_nres();
		if ( int( linked.size1() ) != nres ) linked.dimension( nres );

		if ( edge_list.size() <1 ) return true;

		linked = false;

		// grow out a single connected component from the start vertex:
		const_iterator const it_begin ( edge_list.begin() );
		const_iterator const it_end ( edge_list.end() );

		// mark start vertex as linked:
		linked( it_begin -> start ) = true;

		bool new_member ( true );

		while ( new_member ) {     // keep adding new members
			new_member = false;
			for ( const_iterator it = it_begin; it != it_end; ++it ) {
				if ( it->label == 0 ) continue;
				if ( linked( it-> start ) && ! linked( it->stop) ) {
					linked(it->stop) = true;
					new_member = true;
				} else if ( linked( it->stop ) && ! linked( it->start ) ) {
					linked( it->start ) = true;
					new_member = true;
				}
			}
		}

		for ( const_iterator it = it_begin; it != it_end; ++it ) {
			if ( ! linked( it->start ) || ! linked( it->stop ) ) {
				// it's disconnected!
				return false;
			}
		}
		return true;
	} // Fold_tree::connected()


	/////////////////////////////////////////////////////////////////////////////
	void
	Fold_tree::partition_by_jump(
		int const jump_number,
		FArray1D_bool & partner1
	) const
	{
		check_topology(); // update derived data if necessary

		assert( jump_number <= num_jump );
		assert( int(partner1.size1()) >= nres );

		// find n-terminal jump vertex
		int const pos1( jump_point(1,jump_number ) );

		// mark start vertex as linked:
		partner1( pos1 ) = true;

		bool new_member ( true );

		// get pointers to the beginning and end of the edge_list
		// const_iterator is a typedef in fold_tree.h:
		//
		// typedef std::vector< Edge > Edge_list;
		// typedef Edge_list::iterator iterator;
		// typedef Edge_list::const_iterator const_iterator;

		const_iterator it_begin( edge_list.begin() );
		const_iterator it_end  ( edge_list.end() );

		while ( new_member ) {     // keep adding new members
			new_member = false;
			for ( const_iterator it = it_begin; it != it_end; ++it ) {
				if ( it->label == jump_number ) continue; // skip jump

				int const start( std::min( it->start, it->stop ) );
				int const stop ( std::max( it->start, it->stop ) );
				if ( partner1( start ) && !partner1( stop ) ||
						 partner1( stop ) && !partner1( start ) ) {
					new_member = true;
					if ( it->is_jump() ) {
						// just the vertices
						partner1( start ) = true;
						partner1( stop ) = true;
					} else {
						// all the residues
						for ( int i=start; i<= stop; ++i ) {
							partner1( i ) = true;
						}
					}
				}
			}
		}
	}
	////////////////////////////////////////////////////////////////////////////
	int
	Fold_tree::cutpoint_by_jump(
		int const jump_number
	) const
	{
		FArray1D_bool partner1( nres, false );
		partition_by_jump( jump_number, partner1 );
		int i = 1;
		while ( i < nres && partner1(i) == partner1(i+1) ) i++;
		if ( i == nres ) {
			std::cout << " Fold_tree::cutpoint_by_jump error: "
								<< "can not find the cutpoint!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			return i;
		} else {
			return i;
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	// re-orders for efficient folding: ///////////////////////////////////
	// assumes that begin_res --> end_res are contained in a single
	// segment (or are the entire protein)
	// no bb-move, uses only the first jump with jump_moved == true

	void Fold_tree::refold_reorder( int const begin_res, int const end_res,
																	FArray1D_bool const & jump_moved ) {
		static FArray1D_bool is_vertex;

		if ( ! allow_reorder ) {
			std::cout << "no reordering of the tree allowed\n";
			return;
		}
		// did everything move?
		if ( begin_res <= 1 && end_res >= nres ) return;

		// make sure the tree-topology-dependent derived_data arrays are up to date
		check_topology();

		// setup the is_vertex FArray
		if ( int( is_vertex.size1() ) != nres ) is_vertex.dimension( nres );
		is_vertex = false;
		for ( const_iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			is_vertex ( it->start ) = true;
			is_vertex ( it->stop  ) = true;
		}

		int fix_pos ( 1 );

		if ( begin_res <= end_res && begin_res >= 1 && end_res <= nres ) {
			// fragment insertion
			int best ( 0 );
			fix_pos = 1;

			if ( ! is_cutpoint( begin_res-1 ) ) {
				int const n_fixed ( edge_count( begin_res ) );
				if ( n_fixed > best ) {
					best = n_fixed;
					fix_pos = begin_res-1;
				}
			}

			if ( ! is_cutpoint( end_res ) ) {
				int const c_fixed ( nres - edge_count( end_res + 1) );
				if ( c_fixed > best ) {
					best = c_fixed;
					fix_pos = end_res + 1;
				}
			}

			assert( 1 <= fix_pos && fix_pos <= nres );

			// how to test this stuff?
			for ( int i = 1; i<= num_jump; ++i ) {
				for ( int j=1; j<= 2; ++j ) {
					int const pos ( jump_point(j,i) );
					if ( begin_res <= pos && pos <= end_res ) {
						int const fixed
							( j==1 ? nres - jump_edge_count( i ) : jump_edge_count( i ) );
						if ( fixed > best ) {
							best = fixed;
							// fix the other side:
							fix_pos = jump_point( j==1 ? 2 : 1, i );
						}
					}
				}
			}

			// may have to adjust fix_pos to be a vertex of the fold_tree:
			while ( ! is_vertex( fix_pos ) ) {
				if ( fix_pos < begin_res ) {
					--fix_pos;
				} else {
					assert( fix_pos > end_res );
					++fix_pos;
				}
			}
		} else {
			// look for a jump that has moved
			for ( int i = 1; i<= num_jump; ++i ) {
				if ( jump_moved( i ) ) {
					if ( jump_edge_count ( i ) > nres/2 ) {
						// more stuff on the n-terminal side
						fix_pos = jump_point(1,i);
					} else {
						fix_pos = jump_point(2,i);
					}
					break; // just pick the first moved jump
				}
			}
		} // peptide or jump move?

		assert( is_vertex( fix_pos ) );

		reorder( fix_pos );
	} // Fold_tree::refold_reorder(...)


	//////////////////////////////////////////////////////////////////////////
	/// edge_count(i): delete the edge (i-1,i), how many residues are in the
	///   component of the graph containing i-1?
	///   Note: edge_count(cutpoint+1) doesn't really make sense
	///         currently set to 0 but routines should avoid looking at this
	///         value (see eg refold_reorder(...) )
	///
	/// jump_edge_count(i): delete jump_number i. How many residues are in
	///   the component of the graph containing the jump point on the
	///   N-terminal side of the jump.
	///
	/// edge

	void Fold_tree::setup_edge_counts() const {
		// redimension?
		if ( int ( edge_count.size1() ) != nres ) {
			edge_count.dimension( nres );
		}
		if ( int( jump_edge_count.size1() ) != num_jump ) {
			jump_edge_count.dimension( num_jump );
		}

		min_edge_count = nres;
		edge_count = 0; // 0's are kept at positions: {cutpoint+1}
		jump_edge_count = -1;

		const_iterator const it_begin ( edge_list.begin() );
		const_iterator const it_end   ( edge_list.end()   );
		FArray1D_bool linked(nres);

		for ( const_iterator it = it_begin; it != it_end; ++it ) {
			linked = false;
			int const begin_res ( std::min( it->start, it->stop) );
			int link_count (0);
			linked( begin_res ) = true;

			// find all the residues linked to begin_res, when we arent allowed
			// to traverse the edge *it
			bool new_member = true;
			while ( new_member ) {
				new_member = false;

				for ( const_iterator it2 = it_begin; it2 != it_end; ++it2 ) {
					if ( it2 == it ) continue;
					int const start ( std::min( it2->start, it2->stop ) );
					int const stop  ( std::max( it2->start, it2->stop ) );
					int new_link(0);
					if ( linked(start) && !linked(stop) ) {
						new_link = stop;
					} else if ( linked(stop) && !linked(start) ) {
						new_link = start;
					}
					if ( new_link > 0 ) {
						new_member = true;
						linked(new_link) = true;
						// how many new residues does this edge add?
						link_count +=
							( it2->is_jump() ) ? 1 : stop - start;
					}
				}
			} // while ( new_member )

			if ( it->is_jump() ) {
				// jump
				int const jump_number ( it->label );
				jump_edge_count( jump_number ) = link_count + 1;
				min_edge_count = std::min( min_edge_count,std::max( jump_edge_count( jump_number ),
																									nres - jump_edge_count( jump_number ) ) );
			} else {
				// peptide edge
				int const end_res  ( std::max(it->start, it->stop) );

				for ( int i= begin_res+1; i<= end_res; ++i ) {
					edge_count(i) = link_count + i - begin_res;
					min_edge_count = std::min( min_edge_count, std::max( edge_count(i), nres - edge_count(i) ) );
				}
			}
		}

		for ( int i=1; i<= num_jump; ++i ) assert( jump_edge_count( i ) >= 1 );

	} // Fold_tree::setup_edge_counts(...)


	///////////////////////////////////////////////////////////////////////
	void Fold_tree::update_cutpoints() const {
		// re-dimension?
		int const fold_tree_cutpoint_size1 ( 2*num_jump);
		if ( int ( fold_tree_cutpoint.size1() ) != fold_tree_cutpoint_size1 ) {
			fold_tree_cutpoint.dimension( fold_tree_cutpoint_size1 );
		}
		if ( int ( is_cutpoint.size1() ) != nres+1 ) {
			is_cutpoint.dimension( DRange(0,nres) );
		}
		if( int( cutpoint_map.size1() ) != nres ) {
			cutpoint_map.dimension( nres );
		}
		// first: is_cutpoint
		is_cutpoint = true;

		// loop through the peptide edges, each implies a range of NON-cutpoints:
		for ( const_iterator it = edge_list.begin(),
						it_end = edge_list.end(); it != it_end; ++it ) {
			if ( ! it->is_jump() ) {
				for ( int j = std::min( it->start, it->stop ),
								j_end = std::max( it->start, it->stop ); j < j_end; ++j ) {
					is_cutpoint(j) = false;
				}
			}
		}
		assert( is_cutpoint( 0 ) && is_cutpoint( nres ) );

		// count the cutpoints. note that 0,total_residue dont count as cutpoints
		num_fold_tree_cutpoint = 0;
		cutpoint_map = 0;

		for ( int i = 1; i < nres; ++i ) {
			if ( is_cutpoint(i) ) {
				++num_fold_tree_cutpoint;
				if ( num_fold_tree_cutpoint > fold_tree_cutpoint_size1 ) {
					std::cout << "Fold_tree::update_cutpoints: too many cutpoints: " <<
						num_fold_tree_cutpoint << ' ' << num_jump << std::endl;
					std::cout << *this;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				fold_tree_cutpoint(num_fold_tree_cutpoint) = i;
				cutpoint_map(i) = num_fold_tree_cutpoint;
			}
		}
	}

	////////////////////////////////////////////////////////////////
	void Fold_tree::update_nres() const {
		int tmp_nres (0);
		for ( const_iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			tmp_nres = std::max( tmp_nres, std::max( it->start, it->stop) );
		}
		if ( tmp_nres != nres ) {
			//std::cout << "Fold_tree::update_nres: nres has changed from: " << nres
			//					<< " to: " << tmp_nres << std::endl;
			nres = tmp_nres;
		}
	}

	////////////////////////////////////////////////////////////////
	void Fold_tree::update_num_jump() const {
		int tmp_num_jump (0);
		int biggest_label (0); // for debugging
		for ( const_iterator it = edge_list.begin(), it_end = edge_list.end();
		 it != it_end; ++it ) {
			if ( it->is_jump() ) {
				++tmp_num_jump;
				biggest_label = std::max( biggest_label, it->label );
			}
		}

		if ( biggest_label != tmp_num_jump ) {
			std::cout << "problem with the fold_tree: biggest_label != num_jump " <<
				biggest_label << ' ' << tmp_num_jump << std::endl;
			std::cout << *this;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if ( tmp_num_jump != num_jump ) {
			//std::cout << "Fold_tree::update_num_jump: num_jump has changed from: "
			//					<< num_jump << " to: " << tmp_num_jump << std::endl;
			num_jump = tmp_num_jump;
		}
	} // Fold_tree::update_num_jump


	/////////////////////////////////////////////////////////////
	// fills is_jump_point, jump_point
	//
	void Fold_tree::update_jump_points() const {
		// re-dimension?
		if ( int (is_jump_point.size1()) != nres ) {
			is_jump_point.dimension( nres );
		}
		if ( int ( jump_point.size2() ) != num_jump ) {
			jump_point.dimension( 2, num_jump );
		}

		is_jump_point = false; //array init
		for ( const_iterator it = edge_list.begin(), it_end = edge_list.end();
					it != it_end; ++it ) {
			if ( it->is_jump() ) {
				int const jump_number ( it->label );
				assert( jump_number <= num_jump );
				is_jump_point( it->start ) = true;
				is_jump_point( it->stop ) = true;
				jump_point(1,jump_number) = std::min( it->start, it->stop);
				jump_point(2,jump_number) = std::max( it->start, it->stop);
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	// 03/23/05 -- make single-line I/O
	std::ostream & operator <<( std::ostream & os, Fold_tree const & t )
	{
		os << "FOLD_TREE ";
		for ( Fold_tree::const_iterator it = t.begin(), it_end = t.end();
					it != it_end; ++it ) {
			os << *it;
		}
		os << '\n';
		return os;
	}


	/////////////////////////////////////////////////////////////////////////////
	std::istream & operator >>( std::istream & is, Fold_tree & t )
	{
		t.new_topology = true;
		t.edge_list.clear();

		std::string tag;
		is >> tag;
		if ( !is.fail() && tag == "FOLD_TREE" ) {
			while ( !is.fail() ) {
				Edge e;
				is >> e;
				if ( is.fail() ) break;
				t.edge_list.push_back( e );
			}
			is.clear();
		}

		if ( t.edge_list.size() == 0 ) {
			is.setstate( std::ios_base::failbit );
			std::cout << "no fold_tree info in this stream" << std::endl;
		} else {
			if ( ! t.check_fold_tree() ) {
				std::cout << "bad fold_tree, reordering" << std::endl;
				t.reorder( t.edge_list.begin()->start );
				if ( ! t.check_fold_tree() ) {
					std::cout << "bad fold_tree still bad" << std::endl;
					std::cout << t;
				}
			}
		}
		return is;
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	bool Fold_tree::check_fold_tree() const
	{
		if ( edge_list.size() <= 0 ) return false;
		static FArray1D_bool seen;
		if ( new_topology ) update_nres(); // largest vertex
		if ( int( seen.size1() ) != nres ) seen.dimension( nres );

		seen = false;
		const_iterator it ( edge_list.begin() );
		seen( it->start ) = true;
		for ( const_iterator it_end = edge_list.end(); it != it_end; ++it ) {
			int const start( it->start );
			int const stop ( it->stop );
			if ( ! seen( start ) || ( start != stop && seen( stop ) ) ) {
				std::cout << "bad fold tree!\n" << *this << std::endl;
				return false;
			}
			if ( start == stop ) continue;
			if ( it->is_jump() ) {
				seen( stop ) = true;
			} else {
				int const dir( start < stop ? 1 : -1 );
				for ( int i=start + dir; i!= stop + dir; i+= dir ) {
					if ( seen( i ) ) {
						std::cout << "bad fold tree2!\n" << *this << std::endl;
						return false;
					}
					// for debugging purposes, do not uncomment unless you want it to
					// print out very often!
					//std::cout << "i=" << i << std::endl;
					seen( i ) = true;
				}
			}
		}
		for ( int i=1; i<= nres; ++i ) {
			if ( !seen(i) ) {
				std::cout << "bad fold tree3!\n" << *this << std::endl;
				return false;
			}
		}
		return true;
	} // check_fold_tree()

	/////////////////////////////////////////////////////////////////////////////
	void
	Fold_tree::set_jump_atoms(
		int const jump_number,
		int const upstream_atomno,
		int const downstream_atomno,
		bool const build_as_jump // = true
		)
	{
		//std::cout << "check_order(sja): " << check_order() << std::endl;
		jump_atoms[ jump_number ] =
			std::make_pair( upstream_atomno, // total ridiculous hack:
											( build_as_jump ? 1 : -1 ) * downstream_atomno );
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Fold_tree::upstream_atomno(
		int const jump_number
	) const
	{
		//std::cout << "check_order: " << check_order() << std::endl;
		if ( jump_atoms.count( jump_number ) ) {
			return jump_atoms[ jump_number ].first;
		}
		return 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Fold_tree::downstream_atomno(
		int const jump_number
	) const
	{
		//std::cout << "check_order: " << check_order() << std::endl;
		//check_order();
		if ( jump_atoms.count( jump_number ) ) {
			return jump_atoms[ jump_number ].second;
		}
		return 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////




	/////////////////////////////////////////////////////////////////////////////
	/////////////////////// HELPER //////////////
	// temporarily putting this here:

	int
	pick_loopy_cutpoint(
		int const nres,
		FArray1D_float const & cut_bias_sum
	)
	{

		float r = ran3() * cut_bias_sum( nres );

		int cutpoint( 0 );

		for ( int i = 1; i <= nres; ++i ) {
			if ( r > cut_bias_sum(i-1) && r <= cut_bias_sum(i) ) {
				cutpoint = i;
			}
		}

		if ( cutpoint == 0 ) {
			std::cout << "pick loopy cutpoint = 0! setting = 1" << std::endl;
			cutpoint = 1;
		}

		return cutpoint;
	}


} // namespace Pose_ns

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
