// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15155 $
//  $Date: 2007-05-25 02:18:26 -0400 (Fri, 25 May 2007) $
//  $Author: rhiju $

#ifndef INCLUDED_fold_tree
#define INCLUDED_fold_tree

// Rosetta Headers

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// C++ Headers
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iosfwd>
#include <map>
#include <vector>

/////////////////////////////////////////////////////////////////////////////80
namespace pose_ns {

	///////////////////////////////////////////////////////////////////////////80
	// single edge of the fold_tree
	//
	class Edge {
	public:
		int start;
		int stop;
		int label;

		Edge():
			start(0),
			stop(0),
			label(0)
		{}

		Edge( int const start_in, int const stop_in, int const label_in):
			start (start_in),
			stop (stop_in),
			label (label_in)
		{}

		inline bool is_jump() const { return label > 0;}
		inline int dir0() const {
			return ( is_jump() ? 0 : ( start < stop ? 1 : -1 ) ); // magic numbers
		}

		// stream I/O ////////////////////////
		// these two should be inverses!
		friend std::istream & operator >>(std::istream & is, Edge & e);
		friend std::ostream & operator <<(std::ostream & os, const Edge & e);
		friend bool operator <( Edge const & a, Edge const & b );
		friend bool operator ==( Edge const & a, Edge const & b );
		friend bool operator !=( Edge const & a, Edge const & b );
	};


	////////////////////////////////////////////////////////////////////////////////80
	// the fold_tree class:
	//
	// note that all the derived data is "mutable", so that we can
	// update them as needed on the fly inside "const" member functions
	// this emphasizes that the core data is just the edge_list


	class Fold_tree {
	public:
		typedef std::vector< Edge > Edge_list;
		typedef Edge_list::iterator iterator;
		typedef Edge_list::const_iterator const_iterator;

		static int const PEPTIDE = -1; // must be negative, see Edge::is_jump()

	private:
		// list of edges. vector for fast traversal, but re-ordering, deleting are slow.
		Edge_list edge_list;

		// book-keeping, so we know when to update derived data
		mutable bool new_topology;
		mutable bool new_order;

		// do we allow re-ordering of the tree?
		mutable bool allow_reorder;

		// some derived data ////////////////////////////////////////
		// you dont need to worry about setting any derived data, they are all
		// calculated as needed from the fold_tree.
		// accessed by get_XXX where XXX is the data name
		// note that these are MUTABLE so they can be synced with the
		// edge_list on the fly inside "const" access methods
		//
		mutable int nres; // just the largest vertex in edge_list
		mutable int num_jump; // number of edges in edge_list with label>0
		mutable int num_fold_tree_cutpoint; // == num_jump (if a connected tree)
		mutable FArray2D_int jump_point; // num_jump
		mutable FArray1D_bool is_jump_point; // nres
		mutable FArray1D_int fold_tree_cutpoint; // num_fold_tree_cutpoint
		mutable FArray1D_int cutpoint_map; // nres
		mutable FArray1D_bool is_cutpoint; // nres
		mutable FArray1D_int edge_count; // nres
		mutable int min_edge_count;
		mutable FArray1D_int jump_edge_count; // num_jump
		mutable std::map< int, std::pair< int, int > > jump_atoms;

		///////////////////////////// internal book-keeping /////////////
		// maintain things that depend on both topology and/or order
		inline bool check_topology() const;
		inline bool check_order() const;
		// private methods for updating derived data
		void update_nres() const;
		void update_num_jump() const;
		void update_jump_points() const;
		void update_cutpoints() const;
		void setup_edge_counts() const;

		// private edge_list modifiers
		void update_edge_labels();
		bool cut_edge( int const cut_point );
		bool cut_random_edge( FArray1D_float const & cut_bias_sum, int const nres_in );

	public:

		// constructor
		Fold_tree():
			new_topology (true), // new_topology == true ==> new_order == true
			allow_reorder (true),
			nres(0),
			num_jump(0)
		{}

		// operator=
		// this version doesnt copy any of the derived data!
		// will this be too slow? it will trigger re-calculating of everything
		// every time we reject a move....
		Fold_tree & operator =( Fold_tree const & src ) {
			edge_list = src.edge_list; jump_atoms = src.jump_atoms;
			new_topology = true; return *this;}

		// non-modifying //////////////////////////////////////////////////////
		inline int size() const { return edge_list.size(); }
		inline const_iterator begin() const { return edge_list.begin(); }
		inline const_iterator end() const { return edge_list.end(); }
		inline iterator begin() { return edge_list.begin(); }
		inline iterator end() { return edge_list.end(); }

		// modifiers //////////////////////////////////////////////////////////
		// add,delete edges:
		void add_edge( int const start, int const stop, int const label );
		void delete_edge( iterator edge );
		void delete_edge( Edge const & edge );
		void delete_unordered_edge( int const start, int const stop, int const label );
		void update_edge_label( int const start, int const stop,
														int const old_label, int const new_label );
		void clear() {edge_list.clear(); new_topology=true;}
		void renumber_jumps();
		void delete_extra_vertices();
		void delete_segment( int const seg_begin, int const seg_end );

		int new_jump( int const jump_pos1, int const jump_pos2,
									int const cutpoint );

		// ways to build an entire tree:
		// ** simple_tree makes a 1-edge tree: (1,total_residue,PEPTIDE)
		// ** random_tree_from_jumps builds a tree from a list of jump_points and a
		//    bias array telling us where we should prefer to cut: P(cut at i ) ~ cut_bias(i)
		//    returns false if choosing fails (incompatible jumps? cycles?....)
		void simple_tree( int const nres_in ); // makes a 1-->total_residue tree
		bool random_tree_from_jump_points( int const nres_in, int const num_jump,
																			 FArray2D_int const & jump_point,
																			 FArray1D_float const & cut_bias );
		bool random_tree_from_jump_points( int const nres_in, int const num_jump,
																			 FArray2D_int const & jump_point,
																			 std::vector< int > const & obligate_cut_points,
																			 FArray1D_float const & cut_bias );
		bool tree_from_jumps_and_cuts( int const nres_in, int const num_jump,
																	 FArray2D_int const & jump_point,
																	 FArray1D_int const & cuts,
																	 bool const verbose = true );

		// reorder(): reorder the tree to start at residue start_residue. returns false if
		// no re-ordering allowed!
		bool reorder( int const start_residue );
		void refold_reorder( int const begin_res, int const end_res,
												 FArray1D_bool const & jump_moved );


		// check status ////////////////////////////////////////////////////////
		bool check_fold_tree() const; // foldable?
		bool connected() const; // connected
		bool get_allow_reorder() const { return allow_reorder; }
		void set_allow_reorder( bool const setting ) const { allow_reorder = setting; }
		int upstream_jump_residue( int const jump_number ) const;
		int downstream_jump_residue( int const jump_number ) const;

		void
		partition_by_jump(
			int const jump_number,
			FArray1D_bool & partner1
		) const ;

		int
		cutpoint_by_jump(
			int const jump_number
		) const;

		// these routines are for storing extra information about the jumps
		// you can specify which atoms should be the up/downstream atoms for
		// each jump. Note that this assumes that the topology and order of the
		// fold_tree are fixed, at least until this data is needed.
		//
		// the data are cleared upon re-ordering/wiring of the tree.
		int
		upstream_atomno( int const jump_number ) const;

		int
		downstream_atomno( int const jump_number ) const;

		void
		set_jump_atoms(
			int const jump_number,
			int const upstream_atomno,
			int const downstream_atomno,
			bool const build_as_jump = true
		);


		// stream I/O //////////////////////////////////////////////////////////
		// these two should be inverses!
		// depend on I/O for the class: Edge
		friend std::istream & operator >>(std::istream & is, Fold_tree & t);
		friend std::ostream & operator <<(std::ostream & os, Fold_tree const & t);

		///////////////////////////// derived data //////////////////////
		// routines for retrieving derived data, typically fast
		inline int get_nres() const;
		inline int get_num_jump() const;
		inline FArray2D_int const & get_jump_point() const;
		inline FArray1D_bool const & get_is_jump_point() const;
		inline FArray1D_int const & get_fold_tree_cutpoint( int & ncutpoint ) const;
		inline int get_num_fold_tree_cutpoint() const;
		inline FArray1D_bool const & get_is_cutpoint() const;
		inline FArray1D_int const & get_cutpoint_map() const;
		inline FArray1D_int const & get_jump_edge_count() const;
		inline int count_fixed_residues( int const begin_res, int const size,
																		 int & min_edge_count_out ) const;

		// more routines for getting derived data
		// slow: creates vector each time, just for convenience
		std::vector< int >
		cutpoints() const;


		friend
		bool
		operator==(
			Fold_tree const & a,
			Fold_tree const & b
		);

	}; // end class definition

	////////////////////////////////////////////////////////////////////////////////80
	// these two routines handle the updating of data that depends on
	// tree topology and/or tree order
	// any routine that depends on the stored derived data (eg any of the access
	// methods ) should call check_topology() or check_order() at the beginning
	//
	// also, any method function that changes the tree topology or order should set
	// the private data members new_topology and/or new_topology to true
	//

	inline bool Fold_tree::check_topology() const {
		//std::cout << "check_topology(): " << new_topology << std::endl;
		if ( !new_topology ) return false;
		new_topology = false;
		new_order = true;

		// insert updating calls here: ///////////////////////////
		update_nres();
		update_num_jump();
		update_jump_points();
		update_cutpoints();
		setup_edge_counts();
		//////////////////////////////////////////////////////////
		return true;
	}

	// returns true if order has changed
	inline bool Fold_tree::check_order() const {
		check_topology();
		if ( !new_order ) return false;
		new_order = false;

		// insert updating calls here: ///////////////////////////
		//////////////////////////////////////////////////////////
		// dont want these to stick around after reordering or rewiring the tree:
		// undoing this for the time being, since operator= sets new_topology=true
		//jump_atoms.clear();
		//////////////////////////////////////////////////////////
		return true;
	}


	///////////////////////////////////////////////////////////////////////////80
	// routines for retrieving the derived data
	// will call check_topology and/or check_order first

	inline int Fold_tree::get_nres() const
	{
		check_topology();
		return nres;
	}
	inline int Fold_tree::get_num_jump() const
	{
		check_topology();
		return num_jump;
	}
	inline FArray1D_bool const & Fold_tree::get_is_jump_point() const
	{
		check_topology();
		return is_jump_point;
	}
	inline FArray2D_int const & Fold_tree::get_jump_point() const
	{
		check_topology();
		return jump_point;
	}
	inline FArray1D_int const & Fold_tree::get_fold_tree_cutpoint( int & ncutpoint ) const
	{
		check_topology();
		ncutpoint = num_fold_tree_cutpoint;
		return fold_tree_cutpoint;
	}
	inline int Fold_tree::get_num_fold_tree_cutpoint() const
	{
		check_topology();
		return num_fold_tree_cutpoint;
	}
	inline FArray1D_int const & Fold_tree::get_cutpoint_map() const
	{
		check_topology();
		return cutpoint_map;
	}
	inline FArray1D_bool const & Fold_tree::get_is_cutpoint() const
	{
		check_topology();
		return is_cutpoint;
	}
	inline FArray1D_int const & Fold_tree::get_jump_edge_count() const
	{
		check_topology();
		return jump_edge_count;
	}

	// this assumes no fragment insertions across chainbreaks which is
	// guaranteed by the settings of the insert_size map
	//
	// borrowed some code from refold_reorder
	// returns the size of the largest single fixed region after
	// a fragment is inserted from begin_res to begin_res+size-1
	inline int Fold_tree::count_fixed_residues( int const begin_res, int const size,
																							int & min_edge_count_out ) const {
		check_topology();
		// pass out the value for min_edge_count
		// this is a measure of the magnitude of the largest single-residue or single-jump
		// move we could possibly make. ie, its the number of fixed residues for the
		// move with the smallest number of fixed residues
		min_edge_count_out = min_edge_count;

		int const end_res ( begin_res + size - 1);
		assert( begin_res >= 1 && end_res <= nres );

		int best = 0;
		if ( ! is_cutpoint( begin_res-1 ) ) {
			int const n_fixed ( edge_count( begin_res ) );
			if ( n_fixed > best ) {
				best = n_fixed;
			}
		}

		if ( ! is_cutpoint( end_res ) ) {
			int const c_fixed ( nres - edge_count( end_res + 1) );
			if ( c_fixed > best ) {
				best = c_fixed;
			}
		}

		// how to test this stuff?
		for ( int i = 1; i<= num_jump; ++i ) {
			for ( int j=1; j<= 2; ++j ) {
				int const pos ( jump_point(j,i) );
				if ( begin_res <= pos && pos <= end_res ) {
					int const fixed
						( j==1 ? nres - jump_edge_count( i ) : jump_edge_count( i ) );
					if ( fixed > best ) {
						best = fixed;
					}
				}
			}
		}
		return best;
	}

	// HELPER FUNCTIONS:
	int pick_loopy_cutpoint( int const nres, FArray1D_float const & cut_bias_sum );

}
////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80

#endif
