// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20659 $
//  $Date: 2008-02-27 00:48:58 -0500 (Wed, 27 Feb 2008) $
//  $Author: bblum $


// Rosetta Headers
#include "force_barcode.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "barcode_classes.h"
#include "barcode_stats.h"
#include "barcode_stats_classes.h"
#include "cenlist.h"
#include "dssp.h"
#include "refold.h" // barcode_moves
#include "maps_ns.h"
#include "make_pdb.h"//dump_pdb
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "nblist.h"
#include "pack.h"
#include "param.h" // MAX_CHI!
#include "param_pack.h" // read-only access to Wdun for apply_chi_tether
#include "pose_disulfides.h" //reads constraints into namespace pose_disulf
#include "pose_rna_jumping.h" //for definition of Pairing_RNA class.
#include "ramachandran.h"
#include "random_numbers.h"
#include "recover.h"
#include "runlevel.h"
#include "score.h"
#include "smallmove.h"
#include "status.h"
#include "tether_ns.h" // for bb-tethering
#include "taboo_search.h"
#include "wobble.h" // barcode_moves

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ocstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <iostream>
#include <utility>
#include <vector>

// get the classes
using namespace barcode_classes;

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
// this file has routines for applying barcode constraints,
// as well as initialization routines and the barcode_ns namespace
// which contains the common variable cst_list = the collection
// of possible barcode constraints for this Rosetta run.

////////////////////////////////////////////////////// local namespace:
// initialize:
namespace barcode_ns {
	feature_set cst_list; // calls default constructor
	bool use_barcode = {false};
	bool increment_barcode = {false};
}

namespace barcode_sspair {
	std::map< constraint, float > score_sum;
	float min_score( -999.9 );
	bool init( false );
}

namespace barcode_energy {
	using namespace param;

	bool need_DSSP = {false};
	bool ignore_beta_pair_barcode_in_score = {false};

	FArray2D_float barcode_contact_energy    ( MAX_RES(), MAX_RES(), 0.0 );
	FArray2D_float barcode_contact_distance  ( MAX_RES(), MAX_RES(), 0.0 );
	FArray4D_float barcode_beta_pair_energy  ( MAX_RES(), MAX_RES(), 2, 2, 0.0 );
	FArray2D_float barcode_big_bin_energy    ( MAX_RES(), BigBin::NUM_STATES, 0.0 );
	FArray2D_float barcode_ss_energy         ( MAX_RES(), Secstruct::NUM_STATES, 0.0 );
	FArray2D_float barcode_dssp_energy       ( MAX_RES(), DSSP::NUM_STATES, 0.0 );
	FArray2D_float barcode_bblum_energy      ( MAX_RES(), BBlum::NUM_STATES, 0.0 );
	int const MAX_NEIGHBORS( 30 );
	FArray2D_float barcode_env_energy   ( MAX_RES(), MAX_NEIGHBORS, 0.0 );


}

namespace BigBin {
	int const NUM_STATES = 5;
	FArray1D_char StateChar( NUM_STATES, BigBinStateCharInitializer );
}

namespace Secstruct {
	int const NUM_STATES = 3;
	FArray1D_char StateChar( NUM_STATES, SecStructCharInitializer );
}

namespace DSSP {
	int const NUM_STATES = 8;
	FArray1D_char StateChar( NUM_STATES, DSSPCharInitializer );
}

namespace BBlum {
	int const NUM_STATES = 6;
	FArray1D_char StateChar( NUM_STATES, BBlumCharInitializer );
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
// returns a const reference to the global feature_set object
// stored in barcode_ns
//
// if we write scoring routines as methods of feature_set, then
// it won't be so hard to switch to multiple instances of
// feature_set objects
//
barcode_classes::feature_set const &
barcode_feature_set()
{
	using namespace barcode_ns;
	if ( !use_barcode ) {
		std::cout << "barcode_feature_set() called with use_barcode == FALSE" <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return cst_list;
}

/////////////////////////////////////////////////////////////////////////
// called from initialize_start

//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_initialize_start
///
/// @brief resets counters for each new starting structure; initializes
///  barcode stuff the first time it's called.
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
barcode_initialize_start( int const nres )
{
	static bool init = { false };

	using namespace barcode_ns;
	using namespace runlevel_ns;

	if ( init && ! use_barcode ) return;

	if ( ! init ) {
		init = true;

		// should get these from the command line
		if ( !truefalseoption("barcode_mode") ) {
			use_barcode = false;
			return;
		}
		use_barcode = true;

		// get the mode
		int mode;
		intafteroption("barcode_mode",0,mode);

		// !!! this is the point at which we need to jump to another
		// function to do the fragment file based initialization of
		// cst_list !!!

		if ( truefalseoption("increment_barcode") ) {
			increment_barcode = true;
		}

		// changing this to support reading from a file as well as from fragments

		// <PM>

		// original:
// 		if( truefalseoption("barcode_from_fragments") ) {
// 			//std::cout << "PAUL: analyzing fragments" << std::endl;
// 			cst_list.analyze_fragments(nres,mode);

// 		}
// 		else {
// 			// get the filename
// 			std::string filename;
// 			stringafteroption( "barcode_file", "junk.txt", filename );

// 			// read the constraints
// 			cst_list.read_file( filename, nres, mode);

// 		}

		cst_list.clear(); // need to clear prior to read_file &
											// analyze_fragments, which for compatibility
											// don't do this themselves

		if( truefalseoption("barcode_file") ) {
			// get the filename
			std::string filename;
			stringafteroption("barcode_file","junk.txt",filename);
			// read the constraints
			bool verbose = (runlevel >= standard);  //Don't be verbose in "silent" mode (runlevel = -4!).
			cst_list.read_file(filename,nres,mode,verbose);
		}

		if( truefalseoption("barcode_from_fragments") ) {
			cst_list.analyze_fragments(nres,mode);
		}

		// </PM>

		// should we force rotamers inside get_rotamers?
		cst_list.set_force_rotamer( truefalseoption("barcode_force_rotamer") );


	}

	cst_list.reset();
}



/////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_initialize_decoy
///
/// @brief called from main_rosetta each time through the nstruct loop
///  increments counters, updates bb-tether arrays if necessary
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
barcode_initialize_decoy()
{
	using namespace barcode_ns;
	using namespace runlevel_ns;
	using namespace tether; // bummer

	if ( !use_barcode ) return;

	cst_list.increment_counters();

	cst_list.fill_bb_tether_arrays( tether_angle_res_weight, phi_tether, psi_tether );

	cst_list.fill_barcode_energy_arrays();

	cst_list.update_pairing_list_RNA();

	if (runlevel > standard) cst_list.show_constraints( utility::io::oc::cout );
}


/////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_score
///
/// @brief applies backbone phi-psi bin and secstruct tether scores
///  called from scorefxn() if barcode_weight is non-zero
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
barcode_score(
	int nres_in,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega,
	FArray1Da_char secstruct
)
{
	using namespace barcode_ns;

	if ( ! use_barcode ) return 0.0; // barcode_ns

	//assert(nres_in == cst_list.nres);

	phi.dimension(nres_in);
	psi.dimension(nres_in);
	omega.dimension(nres_in);
	secstruct.dimension(nres_in);

	float score_sum = 0.0;

	for ( constraint_iterator it = cst_list.begin(), it_end = cst_list.end();
				it != it_end; ++it ) {
		int const residue = it->residue;
		if ( it->bb_big_bin_constraint() ||
				 it->bb_small_bin_constraint() ||
				 it->bb_cluster_constraint() ||
				 it->ss_constraint() ) {

			score_sum += it->bb_score( phi(residue), psi(residue),
																 omega(residue), secstruct(residue) );
		}
	}
	return score_sum;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_score_residue
///
/// @brief applies backbone phi-psi bin and secstruct tether scores
///  called from scorefxn() if barcode_weight is non-zero.
///  scoring is on a per residue basis.  useful for resorting fragments
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
barcode_score_residue(
	int pos,
	float phi,
	float psi,
	float omega,
	char secstruct
)
{
	using namespace barcode_ns;

	if ( ! use_barcode ) return 0.0; // barcode_ns

	//assert(nres_in == cst_list.nres);

	float score_sum = 0.0;

	for ( constraint_iterator it = cst_list.begin(), it_end = cst_list.end();
				it != it_end; ++it ) {
		int const residue = it->residue;
		if (residue == pos) {
			if ( it->bb_big_bin_constraint() ||
					 it->bb_small_bin_constraint() ||
					 it->bb_cluster_constraint() ||
					 it->ss_constraint() ) {

				score_sum += it->bb_score( phi, psi,
																 omega, secstruct );

			}
		}
	}
	return score_sum;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin max_barcode_score_residue
///
/// @brief returns the max possible barcode score achievable at this residue,
///  useful as a baseline in fragment picking.  
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
max_barcode_score_residue(
	int pos
)
{
	using namespace barcode_ns;

	if ( ! use_barcode ) return 0.0; // barcode_ns

	//assert(nres_in == cst_list.nres);

	float score_sum = 0.0;

	for ( constraint_iterator it = cst_list.begin(), it_end = cst_list.end();
				it != it_end; ++it )
		if ( it->residue == pos && it->weight > 0.0 && 
				   (it->bb_big_bin_constraint() ||
					 it->bb_small_bin_constraint() ||
					 it->bb_cluster_constraint() ||
					 it->ss_constraint() ) )
			score_sum += it->weight;
	return score_sum;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin min_barcode_score_residue
///
/// @brief returns the min possible barcode score achievable at this residue,
///  useful as a baseline in fragment picking.  
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
min_barcode_score_residue(
	int pos
)
{
	using namespace barcode_ns;

	if ( ! use_barcode ) return 0.0; // barcode_ns

	//assert(nres_in == cst_list.nres);

	float score_sum = 0.0;

	for ( constraint_iterator it = cst_list.begin(), it_end = cst_list.end();
				it != it_end; ++it )
		if ( it->residue == pos && it->weight < 0.0 && 
				   (it->bb_big_bin_constraint() ||
					 it->bb_small_bin_constraint() ||
					 it->bb_cluster_constraint() ||
					 it->ss_constraint() ) )
			score_sum += it->weight;
	return score_sum;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Simple energy bonuses for seeing the features.
//     rhiju, May 2007.
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
int
get_big_bin_state( char const bb_torsion ){
	int i( 1 );
	for (i = 1; i <= BigBin::NUM_STATES; i++)
		if (bb_torsion == BigBin::StateChar(i)) break;

	return i;
}

////////////////////////////////////////////////////////////////////////////////
int
get_big_bin_state( float const phi, float const psi, float const omega ) {
	char const bb_torsion =  torsion2big_bin( phi, psi, omega );
	return get_big_bin_state( bb_torsion );
}

////////////////////////////////////////////////////////////////////////////////
int
get_secstruct_state( char const ss ) {
	int i( 1 );
	for (i = 1; i <= Secstruct::NUM_STATES; i++)
		if (ss == Secstruct::StateChar(i)) break;

	return i;
}
////////////////////////////////////////////////////////////////////////////////
int get_dssp_state( char const ss ) {

	int i( 1 );
	for (i = 1; i <= DSSP::NUM_STATES; i++)
		if (ss == DSSP::StateChar(i)) break;

	//Note that loop is a blank in DSSP, but I want to always turn it into 'L'.
	if (i > DSSP::NUM_STATES) i = DSSP::L;

	return i;
}
////////////////////////////////////////////////////////////////////////////////
int get_bblum_state( char const ss ) {

	int i( 1 );
	for (i = 1; i <= BBlum::NUM_STATES; i++)
		if (ss == BBlum::StateChar(i)) break;

	return i;
}

////////////////////////////////////////////////////////////////////////////////
float eval_barcode_big_bin_energy(){
	using namespace barcode_energy;
	float score = 0.0;

	for ( int ii=1; ii<= misc::total_residue; ++ii ) {
		int const current_big_bin_state =
			get_big_bin_state( misc::phi(ii), misc::psi(ii), misc::omega(ii) );
		score += barcode_big_bin_energy( ii, current_big_bin_state );
		//		std::cout << current_big_bin_state;
	}

	//	std::cout << std::endl;

	//	std::cout << "BIG_BIN_SCORE " << score << std::endl;

	return score;
}
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_ss_energy(){
	using namespace barcode_energy;
	float score = 0.0;

	for ( int ii=1; ii<= misc::total_residue; ++ii ) {

		int const secstruct_state = get_secstruct_state( misc::secstruct(ii) );

		//		std::cout << secstruct_state;

		score += barcode_ss_energy( ii, secstruct_state );
	}

	//	std::cout << std::endl;

	//	std::cout << "SS_SCORE " << score << std::endl;

	return score;
}
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_dssp_energy( dssp_ns::DSSP & dssp ) {
	using namespace barcode_energy;
	float score = 0.0;

	if (!need_DSSP) return score;

	FArray1D_char dssp_secstruct( misc::total_residue );
	dssp.dssp( dssp_secstruct );

	for ( int ii=1; ii<= misc::total_residue; ++ii ) {
		int const current_dssp_state = get_dssp_state( dssp_secstruct(ii) );
		score += barcode_dssp_energy( ii, current_dssp_state );
		//		std::cout << current_dssp_state;
	}

	//	std::cout << std::endl;

	//	std::cout << "DSSP_SCORE " << score << std::endl;

	return score;
}
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_bblum_energy( dssp_ns::DSSP & dssp ) {
	using namespace barcode_energy;
	float score = 0.0;

	if (!need_DSSP) return score;

	FArray1D_char bblum_secstruct( misc::total_residue );
	dssp.dssp_featurizer( bblum_secstruct );

	for ( int ii=1; ii<= misc::total_residue; ++ii ) {
		int const current_bblum_state = get_bblum_state( bblum_secstruct(ii) );
		score += barcode_bblum_energy( ii, current_bblum_state );
		//		std::cout << current_bblum_state;
	}

	//	std::cout << std::endl;

	//	std::cout << "BBLUM_SCORE " << score << std::endl;

	return score;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_contact_energy(){
	using namespace barcode_energy;
	using namespace cenlist_ns;

	float score = 0.0;

	for ( int j = 1; j <= misc::total_residue; ++j ) {
		for ( int i = 1, l = cendist.index(i,j); i <= misc::total_residue; ++i, ++l ) {
			if (cendist[ l ] <= barcode_contact_distance(i,j) ) {
				score += barcode_contact_energy(i,j);
			}
		}
	}

	//	std::cout << "CONTACT_SCORE " << score << std::endl;

	return score;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_beta_pair_energy( dssp_ns::DSSP & dssp ) {
	using namespace barcode_energy;
	using namespace dssp_ns;

	float score = 0.0;

	if (!need_DSSP) return score;

	std::vector< BetaPair > beta_pairs( dssp.strand_pairing_set()->get_beta_pairs() );

	int const num_beta_pairs = static_cast <int> ( beta_pairs.size() );

	for ( int n = 0; n < num_beta_pairs; n++) {
		BetaPair beta_pair = beta_pairs[n];
		score += barcode_beta_pair_energy( beta_pair.res1, beta_pair.res2, beta_pair.orientation, beta_pair.pleating );
	}

	//	std::cout << "BETA_PAIR_SCORE " << score << std::endl;

	return score;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float eval_barcode_env_energy(){
	using namespace barcode_energy;
	using namespace cenlist_ns;
	float score = 0.0;

	for ( int ii=1; ii<= misc::total_residue; ++ii ) {
			int num_neighbors = static_cast<int>( fcen10( ii ) );
			num_neighbors = std::max( num_neighbors, 1);
			num_neighbors = std::min( num_neighbors, MAX_NEIGHBORS);
			score += barcode_env_energy( ii, num_neighbors );
	}

	//	std::cout << "ENV_SCORE " << score << std::endl;

	return score;
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float
barcode_energy_score()
{
	using namespace barcode_ns;
	using namespace barcode_energy;
	using namespace dssp_ns;

	if ( ! use_barcode ) return 0.0; // barcode_ns

	float score_sum = 0.0;

	//Any DSSP computation required?
	dssp_ns::DSSP dssp;
	if (need_DSSP) dssp.compute();

	//To do this reasonably efficiently, instead of cycling through constraints
	// (there might tons of them!), go through residues.
	score_sum += eval_barcode_big_bin_energy();
	score_sum += eval_barcode_ss_energy();
	score_sum += eval_barcode_dssp_energy( dssp );
	score_sum += eval_barcode_bblum_energy( dssp );

	// Contact features
	score_sum += eval_barcode_contact_energy();

	// Beta pairing features.
	if(!ignore_beta_pair_barcode_in_score)
		score_sum += eval_barcode_beta_pair_energy( dssp );

	// Neighbors features
	score_sum += eval_barcode_env_energy();

	return score_sum;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin apply_chi_tether
///
/// @brief scores chi and rot tether. called from get_rotamer_probability(...)
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// note that the max penalty applicable through this routine is limited
/// by min_rot_prob in get_rotamer_probability
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
apply_chi_tether(
	int const seqpos,
	int const aa,
	int const aa_variant,
	FArray1Da_float chi,
	FArray1Da_int rot
)

  //this is called from fullatom_energy where the chi angles are available
{
	using namespace barcode_ns;

	// the weights in the barcode file are the actual penalties applied
	// to a mismatch in the case of sc_bin and rotamer constraints
	// that's why we need Wdun, below

	// for a chi constraint, the squared chi-angle deviation is multiplied
	// by the weight and divided by 800(!) to get the actual score penalty

	// note that depending on the setting of min_rot_prob in the
	// dunbrack energy routine, the maximal score penalty will be capped
	// ( at around 10 for 1e-8, 20 for 1e-16, etc)


	if ( !use_barcode || cst_list.size() == 0 ) return 1.0;

	float const WDUN ( param_pack::pack_wts.Wdun() );
	float const CHI_CONSTRAINT_FACTOR ( 1.0 / 800.0 );

	chi.dimension( param::MAX_CHI );
	rot.dimension( param::MAX_CHI );

	float score = 0.0;
	int const nchi( aaproperties_pack::nchi(aa,aa_variant) );

	for ( constraint_iterator  it = cst_list.sc_begin( seqpos ),
					it_end = cst_list.sc_end( seqpos);
				it != it_end; ++it ) {
		assert ( it->sc_constraint() && it->residue == seqpos );
		const int chi_number = it->chi_number();

		if ( it->chi_constraint() ) {
			if ( chi_number <= nchi ) {
				float const chi_diff
					( subtract_chi_angles( chi(chi_number), it->float_value,
																 aa, chi_number ) );
				score += it->weight * CHI_CONSTRAINT_FACTOR * chi_diff * chi_diff;
			}
		} else if ( it->rot_constraint() ) {
			if ( !it->rot_match( rot, nchi ) ) {
				score += it->weight;
			}
		} else if ( it->sc_bin_constraint() ) {
			if ( !it->sc_bin_match( chi, nchi ) ) {
				score += it->weight;
			}
		} else {
			std::cout << "unrecognized sidechain tag" << std::endl;
		}
	}

	const	float tether_factor = exp( -score/WDUN );

	return tether_factor;
}


/////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_get_rot_cst_exists
///
/// @brief called from get_rotamers_seqpos_aa_aav
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
barcode_get_rot_cst_exists( int const seqpos )
{
	using namespace barcode_ns;

	return ( use_barcode && cst_list.size() > 0 &&
					 cst_list.get_force_rotamer() &&
					 ( cst_list.sc_begin( seqpos ) != cst_list.sc_end( seqpos ) ) );
}


//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_allow_rotamer
///
/// @brief called from get_rotamers_seqpos_aa_aav
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
barcode_allow_rotamer(
	const int seqpos,
	const int aa,
	const int aa_variant,
	FArray1Da_float chi,
	FArray1Da_int rot
)

  //this is called from get_rotamers_seqpos_aa_aav
{
	using namespace barcode_ns;

	const float MAX_CHI_DEV( 20.0 );

	if ( !use_barcode || cst_list.size() == 0 ||
			 !cst_list.get_force_rotamer() ) return true;

	chi.dimension( param::MAX_CHI );
	rot.dimension( param::MAX_CHI );

	// the logic here is: must match at least one of each type of sc constraint
	// present

	std::map< std::string, int > matches, mismatches;
	const int nchi( aaproperties_pack::nchi(aa,aa_variant) );
	for ( constraint_iterator  it = cst_list.sc_begin( seqpos ),
					it_end = cst_list.sc_end( seqpos);
				it != it_end; ++it ) {
		assert ( it->sc_constraint() && it->residue == seqpos );

		if ( it->chi_constraint() ) {
			const int chi_number = it->chi_number();
			const float chi_dev( std::abs( subtract_chi_angles( chi(chi_number),
																													it->float_value,
																													aa, chi_number ) ) );
			if ( chi_dev < MAX_CHI_DEV || nchi < chi_number ) {
				++matches["CHI"];
			} else {
				++mismatches["CHI"];
			}
		} else if ( it->rot_constraint() ) {
			if ( it->rot_match( rot, nchi ) ) {
				++matches["ROT"];
			} else {
				++mismatches["ROT"];
			}
		} else if ( it->sc_bin_constraint() ) {
			if ( it->sc_bin_match( chi, nchi ) ) {
				++matches["SC_BIN"];
			} else {
				++mismatches["SC_BIN"];
			}
		} else {
			std::cout << "unrecognized sidechain tag" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	bool allow( true );
	//std::cout << "barcode_allow_rotamer:: " << seqpos << ' ' <<
	//	matches.size() << ' ' << mismatches.size() << std::endl;
	for ( std::map< std::string, int >::const_iterator it=
				 mismatches.begin(), it_end = mismatches.end(); it!=it_end; ++it ) {
		const std::string & tag( it->first );
		assert( it->second > 0 );
		if ( !matches.count(tag) ) {
			allow = false;
			break;
		}
	}
	return allow;
}


///////////////////////////////////////////////////////////////////////////////
// called from make_pdb

//////////////////////////////////////////////////////////////////////////////
/// @begin output_barcode_info
///
/// @brief show info about current barcode constraints
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_barcode_info( utility::io::orstream & out )
{
	if ( !barcode_ns::use_barcode ) return;
	barcode_ns::cst_list.show_constraints( out );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_exist
///
/// @brief check if barcode constraints exist
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
barcode_exist()
{
	using namespace barcode_ns;

	return use_barcode;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_closest_barcode
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

std::map< std::string, int >
get_closest_barcode()
{
	return barcode_ns::cst_list.get_closest_barcode();
}


std::map< std::string, int >
get_closest_barcode(
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	return barcode_ns::cst_list.get_closest_barcode(phi,psi);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin flavor_move
///
/// @brief change phipsi array according to the flavor angles
///
/// @detailed
///         feature_id: id of feature to move
///         num_out:actual number of RESIDUES moved
///         list: position of residues moved
///         first: first residue moved
///         last: last residue moved
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void flavor_move(
	int feature_id,
	int flavor_id,
	std::vector< int > & moved_residues,
	int & num_out,
	int & first,
	int & last
)
{
	using namespace barcode_ns;
	using namespace misc;
	using namespace protein_maps;

	// para for rama score check
	float temp = { 0.50 };

//------------------------------------------------------------------------------

	// initialize
	num_out = 0;
	first = 0;
	last = 0;
	moved_residues.clear();

	// find the feature
	barcode_classes::feature this_feature;
	pick_a_feature( feature_id, this_feature );

	// find the flavor
	std::vector< flavor >::iterator this_flavor = this_feature.begin();
	++this_flavor; //first flavor is a dummy

	int gg = 0;
	for ( this_flavor = this_flavor;
				this_flavor != this_feature.end();
				++this_flavor ) {
		if ( gg == flavor_id ) break;
		++gg;
	}
	if ( this_flavor == this_feature.end() ) {
			std::cout << "can't find flavor in select random flavor" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// go through each constraint
	std::vector< constraint >::iterator this_cst;
	for ( this_cst = this_flavor->constraints.begin();
			this_cst != this_flavor->constraints.end();
			++this_cst ) {
		if ( this_cst->bb_cluster_constraint() ) {// only work on bb_cluster

			int j = this_cst->residue;
			if ( !allow_insert( j ) ) continue; // next cst

			if ( contains ( moved_residues, j ) ) {
				std::cout << "ERROR: Residue " << j << " appeared in a feature twice!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			int nfail = 0;
			bool good_phipsi_found = false;
			float phi_tmp, psi_tmp;

			while ( !good_phipsi_found && nfail <= 1000) {

				// pad center of the flavor upto the threshhold value
				float padding = ran3() * (2 * this_cst->float_value3) - this_cst->float_value3;
				phi_tmp = this_cst->float_value  + padding;
				psi_tmp = this_cst->float_value1 + padding;

				// rama score chck
				float tmp1, tmp2, old_rama_score, new_rama_score;
				eval_rama_score_residue(res(j),phi(j),psi(j),secstruct(j),old_rama_score,
					tmp1,tmp2);
				eval_rama_score_residue(res(j),phi_tmp,psi_tmp,secstruct(j),new_rama_score,
					tmp1,tmp2);
				if ( new_rama_score > old_rama_score ) {
					float boltz_factor = (old_rama_score-new_rama_score)/temp;
					float probability = std::exp( std::max( -40.0f, boltz_factor ) );
					float crap = ran3(); // bug if  ran3 call in if statment
					if ( crap < probability ) {
						good_phipsi_found = true;
					}
				}
				++nfail;
			}

			if ( nfail > 1000 ) { continue; } // next cst

			phi(j) = phi_tmp;
			psi(j) = psi_tmp;
			name(j) = "-FM-";
			++num_out;// one more moved residue
			moved_residues.push_back( j ); // keep track of moved residues
		} //end if loop
	} //end for cst

	if ( static_cast< int > ( moved_residues.size() ) > 0 ) {
		first = moved_residues.at(0);
		last = moved_residues.at(0);
		for ( int j = 1; j < num_out; ++j ) {
			first = std::min(first,moved_residues.at(j));
			last = std::max(last,moved_residues.at(j));
		}
	}

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin pick_a_feature
///
/// @brief pick a feature from the feature list
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pick_a_feature(
	int num,
	barcode_classes::feature & found_feature
)
{
	using namespace barcode_ns;

	//output
	std::map< std::string, barcode_classes::feature >::iterator this_feature;

	int counter = 0;

	for ( this_feature = cst_list.feature_begin();
					this_feature != cst_list.feature_end();
					++this_feature ) {
		if ( counter == num ){
			found_feature = this_feature->second;
			return;
		}
		++counter;
	}

	std::cout << "no feature found in pick_a_feature "
			<< num << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin barcode_move
///
/// @brief
///
/// @detailed
///car update global position array as well as phi,psi,omega
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  total_begin - [in/out]? -
/// @param  total_end - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// bqian - only tested with nmoves = 1
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
barcode_moves(
	int const nmoves,
	FArray1Da_int move_list,
	int & length,
	int const nwobble,
	int const max_wobble_gap,
	int & begin,
	int & end
)
{
	using namespace misc;
	using namespace param;
	using namespace barcode_ns;

	std::vector< int > all_moved_residues;
	length = 0;
	begin = 0;
	end = 0;

	std::map< std::string, barcode_classes::feature >::iterator this_feature;

	int num_features = 0;
	int num_movable_features = 0;
	for ( this_feature = cst_list.feature_begin();
					this_feature != cst_list.feature_end();
					++this_feature ) {
		if ( has_movable_res( this_feature->second ) ) ++num_movable_features;
		++num_features;
	}

	// can only move upto the number of movable features
	int real_nmove = std::min( nmoves, num_movable_features );

	if ( real_nmove < 1 ) {
		if ( num_movable_features > 1 ) {
			real_nmove = 1;
		} else {
			std::cout << "no movable positions in barcode_moves" << std::endl;
			return;
		}
	}


	for ( int num = 0; num < real_nmove; ++num ) {

		int ntrials = 0;

		// find a feature which has at least one movable residue
		int feature_id = static_cast< int >( ran3()*num_features );
		barcode_classes::feature rand_feature;
		pick_a_feature( feature_id, rand_feature );
		std::vector< flavor >::iterator this_flavor = rand_feature.begin();

		while ( !has_movable_res( rand_feature ) ||
				(++this_flavor)->constraints.begin()->residue <= 7 ||
				(++this_flavor)->constraints.begin()->residue >= total_residue-7) {
			feature_id = static_cast< int >( ran3()*num_features );
			pick_a_feature( feature_id, rand_feature );
			this_flavor = rand_feature.begin();
			++ntrials;
			if (ntrials > 1000) {
				std::cout << "WARNING: No movable features in barcode_move!" << std::endl;
				return;
			}
		}

		int num_flavors = static_cast< int >( rand_feature.size() - 1 );
		int flavor_id = static_cast< int >( ran3() * num_flavors );

		std::vector< int > moved_residues;

		flavor_move_wobble(
			feature_id, flavor_id, nwobble, max_wobble_gap, moved_residues );

		// add the moved_residues into all_moved_residues
		for ( int i = 0; i < int( moved_residues.size() ); ++i ) {
			if ( ! contains( all_moved_residues, moved_residues.at(i) )) {
				all_moved_residues.push_back( moved_residues.at(i) );
			}
		}

	}

	length = int ( all_moved_residues.size() );

	if (length > 0) {
		move_list.dimension( length );
		for ( int j = 0; j < length; ++j ) {
			move_list(j+1) = all_moved_residues.at(j);
		}
		begin = move_list( 1 );
		end = move_list( 1 );
		for ( int j = 2; j <= length; ++j ) {
			begin = std::min(begin,move_list(j));
			end = std::max(end,move_list(j));
		}
	}

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin flavor_move_wobble
///
/// @brief wobble move after a flavor_move
///
/// @detailed
///         std::string nat_code: native barcode
///         moved_residues: position of residues moved
///         first: first residue moved
///         last: last residue moved
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
flavor_move_wobble(
	int const feature_id,
	int const flavor_id,
	int const nwobble,
	int const max_wobble_gap,
	std::vector< int > & moved_residues
)
{
	// initialize
	moved_residues.clear();

//db  I tested this with nwobble=7,  wobble_gap=0, and max_wobble_gap=0
	int wobble_gap;
	if ( max_wobble_gap > 0 ) {
		wobble_gap = static_cast< int >( max_wobble_gap * ran3() ) + 1;
	} else {
		wobble_gap = max_wobble_gap;
	}

	int length = 0;
	int begin = 0;
	int end = 0;

	flavor_move( feature_id, flavor_id, moved_residues, length, begin, end );
	if ( length == 0 ) return; // flavor move failed

	int dir; //refold direction
	refold_get_dir(dir);
	int total_begin, total_end;
	if ( dir == 1 ) {
		total_begin = begin;
		total_end = end + nwobble + wobble_gap;
	} else {
		total_begin = begin - ( nwobble + wobble_gap );
		total_end = end;
	}

	float rama_score, wobble_cost;
	bool gfrag( true ); // if true, powell minimized sucessfully
	if (total_begin < 1) return; //warning sign!
	add_fast_wobble(begin,end-begin+1,nwobble,wobble_gap,rama_score,wobble_cost,
		gfrag,total_begin,total_end);
	if ( !gfrag ) return;// jump out of the loop

//		if ( rama_score > 7.0 || wobble_cost > 30.0 ) {
//			retrieve_best_pose();
//			moved_features = last_moved_features;
//			++nfail;
//			if ( nfail > 200) {
//				break;
//			}
//			goto L170; // try again
//		}

		// add the moved residues in wobble move
	for ( int i = total_begin; i <= total_end; ++i ) {
		if ( !contains( moved_residues, i) )
			moved_residues.push_back(i);
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin make_nat_code_decoy
///
/// @brief make a phipsi array according to the native barcode
///
/// @detailed
///         std::string nat_code: native barcode
///         moved_residues: position of residues moved
///         first: first residue moved
///         last: last residue moved
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
make_nat_code_decoy(
	int & first,
	int & last,
	std::vector< int > & moved_residues
)
{
	using namespace barcode_ns;
	using namespace misc;
	using namespace protein_maps;

//	float temp = { 0.50 };

	int num_out = 0;
//------------------------------------------------------------------------------
//
	if ( taboo_map_ns::nat_code == "") {
			std::cout << "ERROR: must define nat_code in command line" <<
					" before using make_nat_code_decoy! " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::vector< int > native_code;
	std::stringstream line_stream( taboo_map_ns::nat_code );
	int code;
	line_stream >> code;
	while ( !line_stream.fail() ) {
		native_code.push_back( code );
		line_stream >> code;
	}


	int hh = 0;
	std::map< std::string, barcode_classes::feature >::iterator this_feature;
	int num_features = 0;
	for ( this_feature = cst_list.feature_begin();
					this_feature != cst_list.feature_end();
					++this_feature ) {
		// pick the native flavor
		std::vector< flavor >::iterator this_flavor = this_feature->second.begin();
		++this_flavor; //first flavor is a dummy

		int gg = 0;
		for ( this_flavor = this_flavor;
					this_flavor != this_feature->second.end();
					++this_flavor ) {
			if ( gg == native_code.at( num_features ) ) break;
			++gg;
		}

		std::cout << "this falvor nat code:" << SS(gg) << std::endl;

		if ( this_flavor == this_feature->second.end() ) {
				std::cout << "ERROR: can't find native flavor!" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// go through each constraint
		std::vector< constraint >::iterator this_cst;
		for ( this_cst = this_flavor->constraints.begin();
				this_cst != this_flavor->constraints.end();
				++this_cst ) {
			if ( this_cst->bb_cluster_constraint() ) {// only work on bb_cluster
				float phi_tmp = this_cst->float_value;
				float psi_tmp = this_cst->float_value1;

				int j = this_cst->residue;

				// rama score chck
//				float tmp1, tmp2, old_rama_score, new_rama_score;
//				eval_rama_score_residue(res(j),phi(j),psi(j),secstruct(j),old_rama_score,
//						tmp1,tmp2);
//				eval_rama_score_residue(res(j),phi_tmp,psi_tmp,secstruct(j),new_rama_score,
//						tmp1,tmp2);
//				if ( new_rama_score > old_rama_score ) {
//					float boltz_factor = (old_rama_score-new_rama_score)/temp;
//					float probability = std::exp(std::max(-40.0f,boltz_factor) );
//					float crap = ran3(); // bug if  ran3 call in if statment
//					if ( crap >= probability ) {
//						continue;
//					}
//				}

				phi(j) = phi_tmp;
				psi(j) = psi_tmp;
				name(j) = "-SM-";

		int nwobble = 3;
		int wobble_gap = 0;
		int begin = j+1;
		int end = j+1;
		refold( begin, end );
		int dir; //refold direction
		refold_get_dir(dir);
		int total_begin, total_end;
		if ( dir == 1 ) {
			total_begin = begin;
			total_end = end + nwobble + wobble_gap;
		} else {
			total_begin = begin - ( nwobble + wobble_gap );
			total_end = end;
		}
		float rama_score, wobble_cost;
		bool gfrag( true ); // if true, powell minimized sucessfully
		if (total_begin < 1) return; //warning sign!
		add_fast_wobble(begin,end-begin+1,nwobble,wobble_gap,rama_score,wobble_cost,
			gfrag,total_begin,total_end);
		refold( 1, total_residue );

		std::stringstream binj, binhh;
		binj << j;
		binhh << hh;
		std::string file( "wobble" + binj.str() + "_" + binhh.str() + ".pdb" );
		dump_pdb( file );
		++hh;

				++num_out;// one more moved residue
				moved_residues.push_back( j ); // keep track of moved residues
			} //end if loop
		} //end for cst

		++num_features;

  }

//	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	first = moved_residues.at(0);
	last = moved_residues.at(0);
	for ( int j = 1; j < num_out; ++j ) {
		first = std::min(first,moved_residues.at(j));
		last = std::max(last,moved_residues.at(j));
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin barcode_iterative_trial
///
/// @brief try all the flavor and features in the barcode cst list
///
/// @detailed
///         num_in: intended number of FEATURES to move
///         num_out:actual number of RESIDUES moved
///         list: position of residues moved
///         first: first residue moved
///         last: last residue moved
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void
barcode_iterative_trial(
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace barcode_ns;
	using namespace misc;
	using namespace protein_maps;


//	float temperature = 0.8;
	std::string const move_type( "barciter" );

	std::map< std::string, barcode_classes::feature >::iterator this_feature;

	bool gfrag = true;

	// save the starting pose
//	save_barcode_pose();

	// go through all the features
	int num_movable_features = 0;
	int feature_id = -1;
	for ( this_feature = cst_list.feature_begin();
					this_feature != cst_list.feature_end();
					++this_feature ) {
		++feature_id;
		if ( !has_movable_res( this_feature->second ) ) continue;// skip unmovable feature
		++num_movable_features;

// go through all the flavors
//		std::vector< flavor >::iterator low_score_flavor;
//		float lower_score = misc::best_score;// + std::abs(0.1 * misc::best_score);
//		bool move_found = false;

		std::vector< flavor >::iterator this_flavor = this_feature->second.begin();
		++this_flavor; //first flavor is a dummy
		int flavor_id = -1;
		for ( this_flavor = this_flavor;
					this_flavor != this_feature->second.end();
					++this_flavor ) {
			++flavor_id;

			std::vector< int > moved_residues;
//			flavor_move_wobble(
//				feature_id, flavor_id, nwobble, max_wobble_gap, moved_residues );
			flavor_move_wobble(
				feature_id, flavor_id, 7, 0, moved_residues );

			int length, begin, end;
			length = int ( moved_residues.size() );

			begin = moved_residues.at(0);
			end = moved_residues.at(0);
			for ( int j = 1; j < length; ++j ) {
				begin = std::min( begin, moved_residues.at(j) );
				end = std::max( end, moved_residues.at(j) );
			}
			refold (begin, end);

			minimize_exclude_frag(begin, end-begin+1);
			gfrag = true;
			minimize( type, move_type, score_fxn, begin, end, gfrag );
			if ( !gfrag )	{
				resetphipsi();
				continue; // try next flavor
			}
			save_status_info( move_type, begin, begin-end+1 );
			// monte_carlo accept
			monte_carlo( cycle_number );

/*
			float const boltz_factor = ( lower_score - misc::score ) / temperature;
			float const probability = std::exp(std::min(40.0f,std::max(-40.0f,boltz_factor)) );
			float const crap = ran3(); // bug if call to ran3 in "if" statement

			int monte_carlo_accept = 2;
			if ( crap < probability || misc::score < lower_score ) { // accept
//			if ( misc::score < lower_score ) {
std::cout << "real low score: " << misc::score << std::endl;
				if ( misc::score < lower_score ) {
					monte_carlo_accept_low();
					monte_carlo_accept = 3;
				} else {
					monte_carlo_accept = 1;
				}
				monte_carlo_accept_best();
				lower_score = misc::score;
				if ( misc::rms_err < misc::rms_min ) misc::rms_min = misc::rms_err;
				increment_trial_counters( cycle_number );
				output_status_file( monte_carlo_accept, temperature );
				reset_status_info();
				save_status_info(move_type,0,1);
			} else {
				increment_trial_counters(cycle_number);
				retrieve_best_nblist();
				retrieve_best_pose();
				reset_status_info();
			}
*/

		} //end for flavor loop

	} // end for feature loop

}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_increment_barcode
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_increment_barcode()
{
	using namespace barcode_ns;
	return increment_barcode;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin increment_barcode( FArray1D_float phi, FArray1D_float psi )
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void
add_barcode(
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	barcode_ns::cst_list.add_new_bbcluster_flavor( phi, psi );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void
dump_barcode()
{
	barcode_ns::cst_list.dump_barcode_file();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_feature_residues
///
/// @brief: get residues in the feature list
///
/// @detailed
/// input num determines how many residues will be returned
/// if num > total_feature_residues, then all will be returned.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::vector< int >
get_feature_residues( int num )
{

	using namespace barcode_ns;

	if ( ! barcode_exist() ) {
		std::cout << "ERROR:: Required barcode file not found!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::vector< int > barcode_res;

	std::map< std::string, barcode_classes::feature >::iterator this_feature;

	// go through all the features
	int num_features = 0;
	for ( this_feature = cst_list.feature_begin();
					this_feature != cst_list.feature_end();
					++this_feature ) {
//		if ( !has_movable_res( this_feature->second ) ) continue;
		++num_features;

		// go through all the flavors

		std::vector< flavor >::iterator this_flavor = this_feature->second.begin();
		++this_flavor; //first flavor is a dummy
		for ( this_flavor = this_flavor;
					this_flavor != this_feature->second.end();
					++this_flavor ) {

			std::vector< constraint >::iterator this_cst;
			for ( this_cst = this_flavor->constraints.begin();
				this_cst != this_flavor->constraints.end();
				++this_cst ) {

				if ( !contains( barcode_res, this_cst->residue ))
					barcode_res.push_back( this_cst->residue );

			}
		}
	}

	std::vector< int > pick_res;

	while ( int( pick_res.size() ) < num && int( pick_res.size() ) < num_features ) {
		int j = static_cast< int >(ran3()* num_features);
		if ( !contains( pick_res, barcode_res.at(j) ) ) {
			pick_res.push_back( barcode_res.at(j) );
		}
	}

	return pick_res;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin has_movable_res
///
/// @brief check if the feature has movable residues
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
has_movable_res( barcode_classes::feature f )
{
	using namespace protein_maps;

	bool movable_cst = false;

	std::vector< flavor >::iterator this_flavor = f.begin();
	++this_flavor; //first flavor is a dummy

	for ( this_flavor = this_flavor; this_flavor != f.end();
						++this_flavor ) {
		std::vector< constraint >::iterator this_cst;
		for ( this_cst = this_flavor->constraints.begin();
				this_cst != this_flavor->constraints.end();
				++this_cst ) {
			if ( this_cst->bb_cluster_constraint() ) {
				if ( allow_insert( this_cst->residue ) ) {
					movable_cst = true; // found a movable cst
					return movable_cst;
				}
			}
		}
	}

	return movable_cst;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief find if a member exist in a int vector
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
contains( std::vector< int > const & v, int i )
{

	for ( int kk = 0; kk < int( v.size() ); ++kk ) {
		if ( i == v.at(kk) ) return true;
	}

	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin barcode_accept
///
/// @brief: things to do after monte_carlo accept
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: bqian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void
barcode_accept()
{
	using namespace misc;

	if ( !barcode_exist() ) return;

	if ( get_increment_barcode() ) {
		add_barcode( misc::phi, misc::psi );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin target_small_wobble_move
///
/// @brief
///
/// @detailed
///db make  ~10 degree  changes in the torsion angles of a residue not in
///db regular secondary structure, and wobble adjacent residues to
///db reduce perturbation.  this is more conservative than a traditional
///db fragment based wobble move
///car update global position array as well as phi,psi,omega
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  total_begin - [in/out]? -
/// @param  total_end - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
target_small_wobble_move(
	int const nmoves,
	FArray1DB_int & move_list,
	int & length,
	int const nwobble,
	int const max_wobble_gap,
	int & begin,
	int & end
)
{
	using namespace param;
	using namespace misc;

	length = 0;
	begin = 0;
	end = 0;
//db  I tested this with nwobble=7,  wobble_gap=0, and max_wobble_gap=0
//db  I think this should be very effective for loop modeling, but logic
//db  not currently in place

	std::vector< int > feature_res;
	int ntrial = 0;
	bool has_end = true;
	while ( has_end && ntrial < 500 ) {
		feature_res = get_feature_residues( nmoves );
		has_end = false;
		for ( int i = 0; i < int( feature_res.size() ); ++i ) {
			if ( feature_res.at(i) <= 7 || feature_res.at(i) >= total_residue-7 ) {
				has_end = true;
				break;
			}
		}
	}
	if ( ntrial >= 500 ) return; // failed

	int wobble_gap;
	if ( max_wobble_gap > 0 ) {
		wobble_gap = static_cast< int >( max_wobble_gap * ran3() ) + 1;
		 // or look for close L?
	} else {
		wobble_gap = max_wobble_gap;
	}

	length = 0;
	ntrial = 0;
	while ( length == 0 && ntrial < 500 ) {
		target_small_moves_wob(feature_res,move_list,length,begin,end);
		++ntrial;
	}
	if ( length ==0 ) return;
	refold(begin,end);

	int dir; //refold direction
	refold_get_dir(dir);
	int total_begin, total_end;
	if ( dir == 1 ) {
		total_begin = begin;
		total_end = end + nwobble + wobble_gap;
	} else {
		total_begin = begin - ( nwobble + wobble_gap );
		total_end = end;
	}

	float rama_score, wobble_cost;
	bool gfrag( true ); // if true, powell minimized sucessfully
	if (total_begin < 1) return; //warning sign!
	add_fast_wobble(begin,end-begin+1,nwobble,wobble_gap,rama_score,wobble_cost,
	 gfrag,total_begin,total_end);
// 	std::cout << "wbl" << SS( total_begin ) << SS( total_end ) <<
// 		SS( wobble_cost ) << SS( nwobble ) << SS( begin ) << SS( end ) << std::endl;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin update_active_cst_list
///
/// @brief: update the active cst_list based on a residue vector
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
loop_update_active_cst_list (
	int const & loop_begin,
	int const & loop_end,
	std::vector< int > const & free_res,
	int const & loop_begin_extend,
	int const & loop_end_extend
)
{

	using namespace barcode_ns;

	if ( ! barcode_exist() ) {
		return;
	}


	cst_list.loop_update_active_constraints(
		loop_begin, loop_end, free_res, loop_begin_extend, loop_end_extend );

}

//////////////////////////////////////////////////////////////////////////////
/// @begin update_active_cst_list
///
/// @brief: update the active cst_list based on a residue vector
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
reset_barcode_list ()
{

	using namespace barcode_ns;

	if ( ! barcode_exist() ) {
		return;
	}

	cst_list.update_active_constraints();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_barcode_status
///
/// @brief set barcode constraints exist
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_barcode_status( bool yes_no )
{
	using namespace barcode_ns;

	use_barcode  = yes_no;

}
///////////////////////////////////////////////////////////////////////////////
// filter based on what's currently in misc
//
// note call to update_cendist -- may not be necessary and could
// be slow? actually this happens inside get_misc_pairings now...
//
// returns TRUE if structure is accepted
//

bool
topology_filter()
{
	using namespace barcode_stats_ns;

	static bool init( false );
	static bool use_filter( false );
	if (!init ) {
		init = true;
		use_filter = truefalseoption("topology_filter");
	}

	if ( !use_filter || !barcode_ns::use_barcode ) return true;

	// get a list of the pairings in this decoy
	std::vector< Beta_feature > pairings;
 	get_misc_pairings( pairings ); // this calls update_cendist

	// loop over feature constraints, see if all are matched
	bool all_matched( true );
	int nbad_features(0), nmatched_bad_features(0);
	for ( constraint_iterator cst = barcode_ns::cst_list.begin(),
					cst_end = barcode_ns::cst_list.end(); cst != cst_end; ++cst ) {
		if ( cst->sspair_constraint() && std::abs( cst->weight ) < 0.011 ) {
			// assume low-weight constraints are feature descriptors
			bool matched( false );
			for ( std::vector< Beta_feature >::const_iterator p=pairings.begin(),
							p_end = pairings.end(); p != p_end; ++p ) {
				float const theta ( p->antiparallel() ? 175.0 : 5.0 );
				float const score
					( cst->sspair_constraint_score( p->pos1, p->pos2, theta ) );
				if ( std::abs( score ) > 0.5 * std::abs( cst->weight ) ) {
					matched = true;
					break;
				}
			}

			if ( cst->weight > 0.0 ) {
				// penalty case
				++nbad_features;
				if ( matched ) ++nmatched_bad_features;
			} else {
				// forced pairing
				if ( !matched ) {
					std::cout << "Unmatched sspair constraint: " << *cst << std::endl;
					all_matched = false;
				}
			}
		} else if ( cst->contact_constraint() && std::abs( cst->weight ) < 0.011 ) {
			float const score
				( cst->contact_score( misc::total_residue, cenlist_ns::cendist ) );
			if ( std::abs( score ) < 0.5 * std::abs( cst->weight ) ) {
				std::cout << "Unmatched contact constraint: " << *cst << std::endl;
				all_matched = false;
			}
		} // is it filter constraint
	} // loop over active constraints


	if ( nbad_features && nmatched_bad_features == nbad_features ) {
		std::cout << "Matched bad topology!! Failing topology filter!" <<
			std::endl;
		all_matched = false;
	}

	return all_matched;
}

///////////////////////////////////////////////////////////////////////////////
void
reset_barcode_sspair_limits(
)
{
	if ( !barcode_sspair::init ) {
		barcode_sspair::init = true;
		realafteroption("barcode_sspair_limit",-999.9,barcode_sspair::min_score );
	}

	barcode_sspair::score_sum.clear();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float
barcode_sspair_constraint_score(
	int const pos1,
	int const pos2,
	float const theta,
	float ss_score
)
{
	using namespace barcode_sspair;
	using namespace barcode_ns;

	assert( barcode_sspair::init );

	float total_score(0.0);
	// will this be too slow? called inside sspair score
	// for each dimer pair... well -- actually only for dimer_pairs
	// with favorable dimer scores... probably not too many then

	for ( constraint_const_iterator it = cst_list.begin(),
					it_end = cst_list.end(); it != it_end; ++it ) {
		if ( it->sspair_constraint() ) {
			float const weight( it->sspair_constraint_score( pos1, pos2, theta ) );
			if ( weight < -0.01 ) {
				// bonus constraint, matched

				// increment the total ss-score for this constraint and store value
				float const sum( score_sum[*it] += ss_score );
				assert( std::abs( sum - score_sum[*it] ) < 0.001 ); // sanity

				float const over( sum - barcode_sspair::min_score );
				if ( over < 0 ) {
					float const new_ss_score( std::min( 0.0f, ss_score - over ) );
					total_score += -1 * weight * new_ss_score;
				} else {
					total_score += -1 * weight * ss_score;
				}
			} else {
				// penalty constraint or unmatched bonus constraint
				total_score += -1 * weight * ss_score;
			}
		}
	}
	return total_score;
}

//RVERNON: Used to read barcode constraints into the pose_disulf namespace, not used by any mainstream disulfide modes, just my experimental personal mode.
void
barcode_read_disulf_constraints( )
{
	using namespace pose_disulf;
	using namespace barcode_ns;
	//assert( barcode_vdw???:init);

	found_disulf = true;
	num_jump = 0;

	for ( constraint_const_iterator it = cst_list.begin(),
					it_end = cst_list.end(); it != it_end; ++it ) {
		int const res1 = it->residue;
		int const res2 = it->int_value;
		num_jump += 1;
		jump_point(1,num_jump) = res1;
		jump_point(2,num_jump) = res2;
	}
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

bool
get_output_flavor()
{
	static bool init = {false};
	static bool output_flavor;

	if (!init) {
		output_flavor = truefalseoption("output_flavor");
		init = true;
	}

	return output_flavor;
}

void
barcode_flavor_output( std::ostream & iunit )
{
	using namespace barcode_ns;
	iunit << A(60, cst_list.concise_flavor_output() );
}



///////////////////////////////////////////////////////////////////////////////

void
BigBinStateCharInitializer( FArray1D_char & StateChar){
	StateChar( BigBin::A ) = 'A';
	StateChar( BigBin::B ) = 'B';
	StateChar( BigBin::G ) = 'G';
	StateChar( BigBin::E ) = 'E';
	StateChar( BigBin::O ) = 'O';
}

void
SecStructCharInitializer( FArray1D_char & StateChar){
	StateChar( Secstruct::E ) = 'E';
	StateChar( Secstruct::H ) = 'H';
	StateChar( Secstruct::L ) = 'L';
}

void
DSSPCharInitializer( FArray1D_char & StateChar){
	StateChar( DSSP::B ) = 'B';
	StateChar( DSSP::E ) = 'E';
	StateChar( DSSP::G ) = 'G';
	StateChar( DSSP::H ) = 'H';
	StateChar( DSSP::L ) = 'L';
	StateChar( DSSP::I ) = 'I';
	StateChar( DSSP::S ) = 'S';
	StateChar( DSSP::T ) = 'T';
}

void
BBlumCharInitializer( FArray1D_char & StateChar){
	StateChar( BBlum::b ) = 'b';
	StateChar( BBlum::B ) = 'B';
	StateChar( BBlum::e ) = 'e';
	StateChar( BBlum::E ) = 'E';
	StateChar( BBlum::H ) = 'H';
	StateChar( BBlum::L ) = 'L';
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
