// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15154 $
//  $Date: 2007-05-25 02:00:33 -0400 (Fri, 25 May 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "fragments.h"
#include "after_opts.h"
#include "angles.h"
#include "chuck.h"
#include "constraints.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "fragments_ns.h"
#include "gunn.h"
#include "maps.h"
#include "maps_ns.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "runlevel.h"
#include "ssblocks.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>


////////////////////////////////////////////////////////////////////////////////
/// @begin choose_frag_set_top_N_frags
///
/// @brief sets global top_N_frags which limits the number of fragments from
///  the fragment list which are used
///
/// @detailed
///
/// @param[in]   nn - in -  desired setting
///
/// @global_read
///
/// @global_write
///   top_N_frags  static
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_frag_set_top_N_frags( int nn )
{
	using namespace param;
	using namespace fragments::choose_frag_parm;

//     method for setting top_N_frags which limits the fragment list to
//     say the top 25 out of a list of 200.

	top_N_frags = nn;
	if ( nn > MAX_NEIGH() ) {
		std::cout << "fragments.cc: request for more than " << MAX_NEIGH() <<
		 " fragments" << std::endl;
		std::cout << "setting top_N_frags to MAX_NEIGH " << MAX_NEIGH() << std::endl;
		top_N_frags = MAX_NEIGH();
	}
}
///////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/// @begin choose_frag_set_check_ss
///
/// @brief sets check_ss which controls helix length check in choose_fragment
///
/// @detailed
///
/// @param[in]   check_ss  - in -  desired setting
///
/// @global_read
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors db 03/04/2005
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_frag_set_check_ss( bool on_off )
{
	using namespace param;
	using namespace fragments::choose_frag_parm;
  check_ss = on_off;
}


/////////////////////////////////////////////////////////////////////////////////
void
choose_frag_get_check_ss( bool & check_ss )
{
	check_ss = fragments::choose_frag_parm::check_ss;
}


/// @begin choose_frag_get_top_N_frags
///
/// @brief retrieves value global parameter top_N_frags, the number of fragments in the frag list to use
///
/// @detailed
///
/// @param[out]   nn - out - current value of top_N_frags
///
/// @global_read
///  top_N_frags  static
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_frag_get_top_N_frags( int & nn )
{
	nn = fragments::choose_frag_parm::top_N_frags;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_fragment
///
/// @brief randomly selects a fragment of desired size and inserts it into the phi/psi/omega arrays
///
/// @detailed
///
/// @param  size - [in/out]? - desired fragment size for insertion
/// @param[out]   begin - out - first residue of insertion window
///
/// @global_read
///   total_residue misc.h
///   insert_map   maps_ns.h
///   total_size   maps_ns.h
///   neighbors   block  fragments_ns.h
///   ss_type            fragments_ns.h
///   align_name         fragments_ns.h
///
/// @global_write
///  frag_pointer - block static; ss_blocks bookeeping
///  phi   misc.h
///  psi   misc.h
///  omega misc.h
///  secstruct misc.h
///  name  misc.h
/// @remarks
///  enforces secondary structure rules: strands of length 2, helix of length 4
///  in jumping mode, the size returned may be smaller than the desired size
///    this is done so that we can make insertions into small segments in the
///    insert_map
///
/// @references
///
/// @authors  car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_fragment(
	int & size,
	int & begin // location frag was inserted
)
{
	using namespace fragments::frag_pointer;
	using namespace fragments;
	using namespace misc;
	using namespace protein_maps;

//car local
	int nn_num;
	int helix_len;
	int strand_len;
	int top_N_frags;
	bool check_ss;
	float r;
	float end_dist;
	float const end_bias = 60.0f;

//------------------------------------------------

	bool jumping = false; //temporary
	int size_in = size;            // store input value
	int nfail = 0;

	int size_bin = get_index_by_frag_size(size);
	choose_frag_get_check_ss(check_ss);
	choose_frag_get_top_N_frags(top_N_frags);
L30:

	if ( nfail > 1000 ) {
		std::cout << "couldn't find fragment to insert!!" << std::endl;
		return;
	}

	size = size_in; // in jumping_mode, size may change, so reset it

//choose an insertion point, and map it onto 1->total_residue
	begin =
	 insert_map( static_cast< int >( ran3() * ( total_insert - size + 1 ) ) + 1 );
	end_dist = std::abs( begin - ( total_residue / 2 ) );
	r = ran3();
	if ( total_insert == total_residue && ! jumping &&
	 r > std::exp( -( end_dist / end_bias ) ) ) {
//         std::cout << "too close to end in choose fragment" << std::endl;
		goto L30;
		//} else if ( jumping ) {
		//if ( ! jmp_end_bias_check(begin,size) ) goto L30;
		//jmp_adjust_frag_size(begin,size); // shorten size if necessary to fit insert_map
	}
	if ( align_depth(begin,size_bin) == 0 ) {
//         std::cout << "no frags of size " << size << std::endl;
		goto L30;
	}
	for ( int i = begin; i < begin + size; ++i ) {
		if ( !allow_insert(i) ) {
//            std::cout << "fixed torsion angles: residue " << i << std::endl;
			goto L30;
		}
	}


// pick random fragment from the list, up to the specified top_N_frags depth
	if ( get_ssblock_state() || get_random_frag_state() ) {
		int const jnn_num = static_cast< int >( ran3() *
		 std::min(align_depth(begin,size_bin),block_depth(begin,size_bin)) ) + 1;
		nn_num = block_frag_pointer(jnn_num,begin,size_bin);
	} else {
		nn_num = static_cast< int >( ran3() *
		 std::min(top_N_frags,align_depth(begin,size_bin)) ) + 1;
	}

// insert into phi/psi arrays
	for ( int i = 0; i < size; ++i ) {
		if ( align_phi(begin,nn_num,i,size_bin) == 0.0 ||
		 align_psi(begin,nn_num,i,size_bin) == 0.0 ) {
			resetphipsi();
			goto L30;
		}
		phi(i+begin) = align_phi(begin,nn_num,i,size_bin);
		psi(i+begin) = align_psi(begin,nn_num,i,size_bin);
		omega(i+begin) = align_omega(begin,nn_num,i,size_bin);
		secstruct(i+begin) = ss_type(begin,nn_num,i,size_bin);
		name(i+begin) = align_name(begin,nn_num,size_bin);
		if (get_jitter_frag_state()) {
				phi(i+begin)=jitter_frag(phi(i+begin),secstruct(i+begin));
				psi(i+begin)=jitter_frag(psi(i+begin),secstruct(i+begin));
		}
	}

	if ( same_neighbor() == 1 ) goto L30;


//     DONT ALLOW HELICES OF LESS THAN 3 OR STRANDS OF LESS THAN 2

  if (check_ss) {
	helix_len = 0;
	strand_len = 0;
	for ( int i = 1; i < total_residue; ++i ) {
		if ( secstruct(i) == 'H' ) ++helix_len;
		if ( secstruct(i) == 'E' ) ++strand_len;
		if ( secstruct(i) != secstruct(i+1) ) {
			if ( helix_len != 0 && helix_len < 3 ) {
//				std::cout << "helix length violation in choose_frag" <<
//				 SS( helix_len ) << SS( strand_len ) << SS( i ) << std::endl;
//				for ( int j = 1; j <= i; ++j ) {
//					std::cout << secstruct(j);
//				} std::cout << std::endl;
//				std::cout << SS( begin ) << SS( nn_num ) << std::endl;
				resetphipsi();
				++nfail;
				goto L30;
			}
			if ( strand_len != 0 && strand_len < 2 ) {
				resetphipsi();
//				std::cout << "strand length violation in choose_frag" << std::endl;
//				std::cout << "in choose,strand" << SS( helix_len ) <<
//				 SS( strand_len ) << std::endl;
				++nfail;
				goto L30;
			}
			helix_len = 0;
			strand_len = 0;
		}
	}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin same_neighbor
///
/// @brief returns 0 if phi,psi differs from best_phi,best_psi at at least one  position; else returns 1
///
/// @detailed
///
/// @return
///
/// @global_read
///  phi            misc.h
///  psi            misc.h
///  best_phi       misc.h
///  best_psi       misc.h
///  total_residue  misc.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
same_neighbor()
{
	using namespace misc;

	int same_neighbor = 1; // Return value

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( phi(i) != best_phi(i) ) same_neighbor = 0;
		if ( psi(i) != best_psi(i) ) same_neighbor = 0;
	}

	return same_neighbor;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin same_frag
///
/// @brief returns true if phi,psi differs from best_phi,best_psi at at least one
///car position in specified range; else returns false
///
/// @detailed
///
/// @param[in]   frag_begin - in - beginning of region to compare
/// @param[in]   frag_end - in -   end of region to compare
///
/// @return
///
/// @global_read
///  phi            misc.h
///  psi            misc.h
///  best_phi       misc.h
///  best_psi       misc.h
///  total_residue  misc.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
same_frag(
	int & frag_begin,
	int & frag_end
)
{
	using namespace misc;

	bool same_frag = true; // Return value

	for ( int i = frag_begin; i <= frag_end; ++i ) {
		if ( phi(i) != best_phi(i) ) same_frag = false;
		if ( psi(i) != best_psi(i) ) same_frag = false;
	}

	return same_frag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_index_by_frag_size
///
/// @brief  returns the index into fragment arrays for fragments of the specified size
///
/// @detailed
///
/// @param[in]   want_size - in - fragment size for which index is desired
///
/// @return get_index_by_frag_size
///
/// @global_read
///  n_frag_files files_paths.h
///  frag_sizes   files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
get_index_by_frag_size( int const want_size )
{
	using namespace files_paths;
	using namespace param;

	// Active fragment libraries
	int const a_frag_sizes =
	 ( n_frag_sizes().zvalue() > 0 ? n_frag_sizes() : n_frag_files().zvalue() );

	for ( int i = 1; i <= a_frag_sizes; ++i ) {
		if ( frag_sizes(i) == want_size ) {
			return i;
		}
	}

	std::cout << " requested frag size not available in get_index_by_frag_size()" << std::endl;
	std::cout << " you asked for " << want_size << std::endl;
	std::cout << " available ";
	for ( int i = 1; i <= a_frag_sizes; ++i ) {
		std::cout << SS( frag_sizes(i) );
	} std::cout << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	return 0; // Avoids compiler warning
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_fragments
///
/// @brief read fragment data from fragment files
///
/// @detailed
///
/// @global_read param.h, misc.h, files_paths.h
///     protein_name, protein_chain, name_length, fragments_path,
///     frag_sizes()
///
/// @global_write fragment data blocks in fragments_ns.h
///
/// @remarks also identifies SS-blocks and checks for homologs
///
/// @references
///
/// @authors Jeff Gray 3/12/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_fragments()
{
	using namespace files_paths;
	using namespace fragments;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

	std::string::size_type locate;
	std::string const search( NAME_LENGTH + 1, '*' );
	std::string filename;

	assert( ( total_residue > 0 ) ); // Should have been set already

	// See if 3mer library present
	int bin3 = 0;
	for ( int i = 1, ie = n_frag_files(); i <= ie; ++i ) {
		if ( frag_sizes(i) == 3 ) bin3 = i;
	}

	// Set fragment-related array dimensions: Arrays allocated
	n_frag_sizes() = ( bin3 > 0 ? n_frag_files()() + 1 : n_frag_files()() );
	int max_frag_length = 0;
	for ( int i = 1, ie = n_frag_files(); i <= ie; ++i ) {
		max_frag_length = std::max( max_frag_length, frag_sizes( i ) );
	}
	param::MAX_LEN() = max_frag_length;

	for ( int i = 1, ie = n_frag_files(); i <= ie; ++i ) {

		if ( ( locate = frag_file_name(i).find( search ) ) == std::string::npos ) {
			// Hmm, if reading in fragments twice, the frag_file_name may already have ***** replaced with protein name.
			if ( ( locate = frag_file_name(i).find( protein_name)) == std::string::npos){
				std::cout << "ERROR: improper fragment filename: "
									<< frag_file_name(i) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		frag_file_name(i).replace( locate, NAME_LENGTH, protein_name );

		frag_file_name(i)[ locate + NAME_LENGTH ] = protein_chain;
		filename = fragments_path_1 + frags_name_prefix + frag_file_name(i);

		read_fragments_one_file( filename, frag_sizes(i), align_phi(1,1,0,i), align_psi(1,1,0,i),
		 align_omega(1,1,0,i), align_depth(1,i), MAX_NEIGH(), frag_sizes(i), MAX_RES(),
		 total_residue, align_name(1,1,i), ss_type(1,1,0,i), align_chain(1,1,i),
		 align_res_id(1,1,0,i), align_resseq(1,1,i) );

		fragments_nres = misc::total_residue;

		std::cout << "Total Residue " << total_residue << std::endl;
		std::cout << "frag size: " << frag_sizes(i) << "    frags/residue: " <<
		 align_depth(1,i) << std::endl;

	}

	// Generate 1mer library from 3mer library if present
	if ( bin3 > 0 ) { // 3mer library present
		std::cout << "generating 1mer library from 3mer library" << std::endl;
		int const bin1 = n_frag_sizes();
		frag_sizes( bin1 ) = 1;
		for ( int i = 1; i <= total_residue; ++i ) {

			int pos3 = 1; // keep central res of 3mer
			int begin3 = i - 1;
			if ( i == 1 ) {
				begin3 = i; // first res of first frag
				pos3 = 0;
			} else if ( i == total_residue ) {
				begin3 = i - 2; // last res of last frag
				pos3 = 2;
			}
			align_depth( i, bin1 ) = align_depth( begin3, bin3 );
			for ( int j = 1, je = align_depth( i, bin1 ); j <= je; ++j ) {
				align_phi(i,j,0,bin1) = align_phi( begin3, j, pos3, bin3 );
				align_psi(i,j,0,bin1) = align_psi( begin3, j, pos3, bin3 );
				align_omega(i,j,0,bin1) = align_omega( begin3, j, pos3, bin3 );
				ss_type(i,j,0,bin1) = ss_type( begin3, j, pos3, bin3 );
				align_name(i,j,bin1) = align_name( begin3, j, bin3 );
			}
		}
	}

	identify_ssblocks();
	homolog_check();
}

///////////////////////////////////////////////////////////////////////////////
//
// doesnt depend on misc being initialized
//

void
read_fragments(
	int const nres
)
{
	int const save_total_residue( misc::total_residue);
	misc::total_residue = nres;

	if ( nres > param::MAX_RES()() ) {
		std::cout << "WARNING:: read_fragments: MAX_RES increase from " <<
			param::MAX_RES()() << " to " << nres <<
			"\nThis will kill old data in MAX_RES sized arrays" << std::endl;
		param::MAX_RES_assign_res( nres );
	}

	read_fragments();
	misc::total_residue = save_total_residue;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_fragments_one_file
///
/// @brief Read fragment details from a fragment file
///
/// @detailed Fragment details include phi,psi,omega,SS
///
/// @param[in]   filename - in - fragment file name
/// @param[in]   size  - in - length of fragments (e.g. 3 or 9 residues)
/// @param[out]   a_phi      - out - phi angle of fragments
/// @param[out]   a_psi      - out - psi angle of fragments
/// @param[out]   a_omega    - out - omega angle of fragments
/// @param[out]   a_frag_list_depth - out - number of fragments at each position
/// @param[in]   a_max_neigh - in - array size limit of loop depth
/// @param[in]   a_max_len   - in - array size limit for length of fragments
/// @param[in]   a_max_res   - in - array size limit fornumber of residues
/// @param[in]   a_total_residue - in - number of residues in target protein
/// @param[out]   a_name     - out - pdb names of fragment sources
/// @param[out]   a_ss_type  - out - secondary structure of fragment sources
/// @param[out]   a_chain    - out - chains of fragment sources
/// @param[out]   a_res_id   - out - residue types of fragment sources
/// @param[out]   a_resseq   - out - residue numbers of fragment sources
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 3/12/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_fragments_one_file(
	std::string filename,
	int const size,
	FArray3Da_float a_phi,
	FArray3Da_float a_psi,
	FArray3Da_float a_omega,
	FArray1Da_int a_frag_list_depth,
	int const a_max_neigh,
	int const a_max_len,
	int const a_max_res,
	int const a_total_residue,
	FArray2Da_string a_name,
	FArray3Da_char a_ss_type,
	FArray2Da_char a_chain,
	FArray3Da_char a_res_id,
	FArray2Da_int a_resseq // for new vall
)
{
	a_phi.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_psi.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_omega.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_frag_list_depth.dimension( a_max_res );
	a_name.dimension( a_max_res, a_max_neigh );
	a_ss_type.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_chain.dimension( a_max_res, a_max_neigh );
	a_res_id.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_resseq.dimension( a_max_res, a_max_neigh );

// a_max_neigh, a_max_len, and a_max_res must be physical size of arrays!

//------------------------------------------------------------------------------
//  THIS FUNCTION WILL READ THE NEAREST NEIGHBOR FILES
//------------------------------------------------------------------------------

	std::string frag_format( "new" );
	int depth; //, resseq  for old vall
//	char res_id, chain;
//	std::string resseq( 5, ' ' );       // for new vall
//	int resseq;        // for new vall
//-------------------------------------------------

	a_phi = 0.0f;
	a_psi = 0.0f;
	a_omega = 0.0f;

	int const npositions = a_total_residue - size + 1;

// dek create izstream (fragment files can be gzipped)
	utility::io::izstream funit( filename );

	if ( !funit ) {
		std::cout << " DAMN: read_fragments can't locate  " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::cout << "fragment file: " << funit.filename() << std::endl;

L100:
	int i = 0;
	for ( i = 1; i <= npositions; ++i ) {
		funit >> skip( 43 ) >> bite( 10, depth ) >> skip;
		if ( funit.eof() ) break;
		int const clippped_depth = std::min( depth, a_max_neigh );
		a_frag_list_depth(i) = clippped_depth;
		funit >> skip;
		for ( int j = 1; j <= clippped_depth; ++j ) {
			for ( int k = 0; k < size; ++k ) {
				if ( frag_format == "new" ) { // new format
					funit >> skip( 1 ) >>
					 bite( 4, a_name(i,j) ) >> skip( 1 ) >>
					 bite( a_chain(i,j) ) >> skip( 1 ) >>
					 bite( 5, a_resseq(i,j) ) >> skip( 1 ) >>
					 bite( a_res_id(i,j,k) ) >> skip( 1 ) >>
					 bite( a_ss_type(i,j,k) ) >>
					 bite( 9, a_phi(i,j,k) ) >>
					 bite( 9, a_psi(i,j,k) ) >>
					 bite( 9, a_omega(i,j,k) ) >> skip;
					if ( ! is_any_of( a_ss_type(i,j,k), "LHE" ) ) {
						if ( ( i == 1 ) && ( j == 1 ) ) { // Try as old format
							frag_format = "old";
							funit.seek_beg();

							goto L100;
						} else { // Bad format
							std::cout << "Format error in frag file. Position: "
							 << SS(i) << " Fragment: " << SS(j) << " Residue: "
							 << SS(k+1) << std::endl;
							utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
						}
					}
				} else { // old format
					funit >> skip( 1 ) >>
					 bite( 4, a_name(i,j) ) >> skip( 1 ) >>
					 bite( a_chain(i,j) ) >> skip( 1 ) >>
					 bite( 4, a_resseq(i,j) ) >> skip( 1 ) >>
					 bite( a_res_id(i,j,k) ) >> skip( 1 ) >>
					 bite( a_ss_type(i,j,k) ) >>
					 bite( 9, a_phi(i,j,k) ) >>
					 bite( 9, a_psi(i,j,k) ) >>
					 bite( 9, a_omega(i,j,k) ) >> skip;
				}
				if ( ! is_any_of( a_ss_type(i,j,k), "LHE" ) ) {
					std::cout << "unable to determine frag file format" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}               // end k
			funit >> skip;
		}                  // end j

// normally a_max_neigh > depth, but just in case we keep reading and
// discarding excess lines past max read depth a_max_neigh

		if ( a_max_neigh < depth ) {
			for ( int j = a_max_neigh+1; j <= depth; ++j ) {
			 // note nothing happens if depth < a_max_neig
				for ( int k = 0; k < size; ++k ) {
					funit >> skip; // dummy read
				}
				funit >> skip; // dummy read

			}               // end j
		}                  // depth check

	}                     // end i

//car end of file, check length)
	if ( i != a_total_residue - size + 2 && !dna_variables::enable_dna ) {
		std::cout << "length mismatch in frag file" << std::endl;
		std::cout << "total_residue: " << a_total_residue
					 << " length " << i + size - 2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	funit.close();
	funit.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin homolog_check
///
/// @brief
///
/// @detailed This function looks for homologs in the fragments by checking for
/// long stretches of sequence in fragments that all come from the same
/// parent protein
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
homolog_check()
{
	using namespace fragments;
	using namespace misc;

	if ( !truefalseoption("check_homs") ) return;

	std::cout << "checking for homologs in fragments..." << std::endl;

	int const count_strigency = { 20 };
	int const max_hits = { 50 };
	FArray1D_string hit_name( max_hits, std::string( 4, ' ' ) );
	std::string donator_name;
	FArray1D_char hit_chain( max_hits, ' ' );
	char donator_chain;
	int const h_size = 9;
	FArray1D_int hit_register( max_hits, 0 );
	int the_register;
	FArray1D_int hit_count( max_hits );
	int hits = 0;

	int size_bin = get_index_by_frag_size( h_size );
	for ( int h_position = 1, h_position_end = ( total_residue - h_size ) + 1;
	 h_position <= h_position_end; ++h_position ) {
		for ( int h_fragment = 1; h_fragment <= 200; ++h_fragment ) {
			int register_count = 1;
			donator_name = align_name( h_position, h_fragment, size_bin );
			donator_chain = align_chain( h_position, h_fragment, size_bin );
			the_register = align_resseq( h_position, h_fragment, size_bin ) - h_position;
			int i = 0;
			bool not_found = true;
			while ( ( i < hits ) && ( not_found ) ) {
				++i;
				if ( donator_name == hit_name(i) ) {
					if ( donator_chain == hit_chain(i) ) {
						if ( the_register == hit_register(i) ) {
							not_found = false;
						}
					}
				}

			}
//            std::cout << "progress " << h_position << ' ' << h_fragment << ' ' <<
//             donator_name << ' ' << donator_chain << ' ' << the_register << std::endl;
			if ( not_found ) {
				for ( int nextposition = h_position + 1, nextposition_end = ( total_residue - h_size ) + 1;
				 nextposition <= nextposition_end; ++nextposition ) {
					for ( int nextfragment = 1; nextfragment <= 200; ++nextfragment ) {
						if ( donator_name == align_name( nextposition, nextfragment, size_bin ) ) {
							if ( donator_chain == align_chain( nextposition, nextfragment, size_bin ) ) {
								if ( the_register == align_resseq( nextposition, nextfragment, size_bin ) -
								 nextposition ) {
									++register_count;
								}
							}
						}
					}
				}
				if ( register_count > count_strigency ) {
//                  std::cout << "Possible structural homolog in fragments!" << std::endl;
					std::cout << "hit:   " << donator_name << donator_chain <<
					 "   " << I( 6, register_count ) << ' ' <<
					 "in register fragments." << std::endl;
					++hits;
					if ( hits > max_hits ) {
						std::cout << "more than " << max_hits << " homologs detected!" << std::endl;
						std::cout << "increase max_hits in homolog_check.cc." << std::endl;
						goto L101;
					}
					hit_name(hits) = donator_name;
					hit_chain(hits) = donator_chain;
					hit_register(hits) = the_register;
					hit_count(hits) = register_count;
				}
			}
		}
	}
L101:
	if ( hits > 0 ) {
		std::cout << "WARNING:: Structural homologs likely present in fragments!" << std::endl;
//  	int i = 0;
//         while ( i < hits ) {
//            ++i;
//            std::cout << hit_name(i) << hit_chain(i) << std::endl;
//         }
		std::cout << "STOPPING" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

//    if ( register_count > 10 ) {
//       std::cout << "frags " << h_position << ' ' << h_fragment << ' ' <<
//        donator_name << ' ' <<
//        align_chain(h_position,h_fragment,size_bin) << ' ' <<
//        align_resseq(h_position,h_fragment,size_bin) << ' ' <<
//        the_register << ' ' << register_count << std::endl;
//    }

////////////////////////////////////////////////////////////////////////////////
/// @begin precompute_frag_movement
///
/// @brief
///  fold each fragment and compute parameters (currently for chuck & gunn)
///
/// @detailed
///
/// @param  length - [in] -
/// @param  size_bin - [in] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
precompute_frag_movement(
	int const length,
	int const size_bin,
	int const total_residue
)
{
	using namespace fragments;
	using namespace param;
	using namespace param_aa;

//car self init
	if ( frag_precompute::initialized(size_bin) ) return; // already initialized

	FArray3D_float pos(3,MAX_POS,length+1);
	FArray1D_float phi(length);
	FArray1D_float psi(length);
	FArray1D_float omega(length);

	// declararations for old refold
	//car  'b' arrays left uninitialized as they will not be used
	FArray3D_float bpos( 3, MAX_POS, length );
	FArray2D_float bcen( 3, length );
	FArray2D_float cen( 3, length );
	FArray3D_float bfull( 3, MAX_ATOM()(), length );
	FArray3D_float full( 3, MAX_ATOM()(), length );
	FArray1D_int res( length, aa_ala ); // set arbitrarily to ala
	FArray1D_int resv( length, 1 );     // set arbitrarily to 1

	std::cout << "pre-computing chuck/gunn move set for frag length " << length <<
	 std::endl;
	if ( length > MAX_LEN() ) {
		std::cout << "precompute_frag_movement: Houston there's a problem with the " <<
		 "frag lengths " << length << ' ' << MAX_LEN() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	for ( int i = 1; i <= total_residue-length+1; ++i ) {
		for ( int j = 1, je = align_depth(i,size_bin); j <= je; ++j ) {
			// MAX_NEIGH for size and position
			for ( int k = 0; k < length; ++k ) {
				phi(k+1)   = align_phi(i,j,k,size_bin);
				psi(k+1)   = align_psi(i,j,k,size_bin);
				omega(k+1) = align_omega(i,j,k,size_bin);
			}

			backbone_fold(length,phi,psi,omega,pos);

			precompute_gunn(length,i,j,size_bin,pos);

			precompute_chuck(i,j,size_bin, pos(1,2,1));
		}                  // j,  next frag  at same resisue position
	}   // i ,  next residue insertion window

//  set init flag for this size
	frag_precompute::initialized(size_bin) = true;

}


// I think this can be made cleaner by having choose_fragment  always use block_frag_pointer and having it initialized to simple ascending sequence?

void
pick_random_frags( int frag3_pool_size, int frag3_final, int frag9_pool_size, int frag9_final )
{
	using namespace fragments::frag_pointer;
	using namespace misc;
	using namespace fragments;
	int size, pool_size, final_size;
	for ( int frag_bin = 1; frag_bin <= 2; ++frag_bin ) {
		if ( frag_bin == 1 ) {
			pool_size = frag3_pool_size;
			final_size = frag3_final;
			size = 3;
		} else {
			pool_size = frag9_pool_size;
			final_size = frag9_final;
			size = 9;
		}

		int frag_num;
		for ( int i = 1; i <= total_residue-size+1; ++i ) {
			int found = 0;
			std::vector< int > frag_list; // is this automatically initialized?
			while ( found <= final_size ) {
				frag_num=static_cast< int >( ran3()*pool_size + 1 );
				if ( std::find(frag_list.begin(),frag_list.end(),frag_num)==frag_list.end() ) {
					++found;
					frag_list.push_back(frag_num);
					block_frag_pointer(found,i,frag_bin)=frag_num;
				}
			}
			block_depth(i,frag_bin)=found;
		}
	}

}

bool
get_random_frag_state()
{
	static bool random_frag_state = { false };
	static bool init = { false };

	if ( !init ) {
		random_frag_state = truefalseoption("random_frag");
		init = true;
	}
	return random_frag_state;
}

float
jitter_frag(float angle_val, char const ss)
{
	static bool init = {false};
	static float jitter_val={2.0};
  static std::string jitter_type ("flat") ;
	if (!init) {
		realafteroption("jitter_amount",2.0,jitter_val);
		stringafteroption("jitter_variation","flat",jitter_type);
		init = true;
	}
	if (jitter_type == "gauss"){
		if (ss == 'H'){
			angle_val= angle_val +  0.05 *gaussian()  * jitter_val;
		}else if (ss=='E') {
	  	angle_val= angle_val +  0.2 *gaussian()  * jitter_val;
		} else {
		  angle_val= angle_val +  1.0 *gaussian()  * jitter_val;
		}

	} else {
		if (ss == 'L') 	angle_val= angle_val + (ran3()-0.5)  * jitter_val;
	}
	return(angle_val);
}

bool
get_jitter_frag_state()
{
	static bool jitter_frag_state= { false };
	static bool init = { false };

	if ( !init ) {
		jitter_frag_state= truefalseoption("jitter_frag");
		init = true;
	}
	return jitter_frag_state;
}

/////////////////////////////////////////////////////////
/// here starts fragments w/constraints related stuff ///
/////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_fragment_cst
///
/// @brief
///
/// @detailed
///
/// @param  tolerance - [in/out]? -
/// @param  size - [in/out]? -
/// @param  frag_begin - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_fragment_cst(
	float tolerance,
	int size,
	int & frag_begin
)
{
	using namespace fragments::chuck;
	using namespace fragments;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

//car local
	FArray1D_int insert_map( MAX_RES()() );
	int total_insert;
	int size_bin;
	int ncst;
	FArray1D_int cst_list( classical_constraints::constants::MAX_CONSTRAINTS );
	int top_N_frags;
	float dist2;
	float cst_score, check_score;
	int nn_num,final_nn_num,max_nn_num; // ptrs into frag list

	FArray2D_float R( 3, 3 );
	FArray1D_float L( 3 );

	size_bin = get_index_by_frag_size(size);
	precompute_frag_movement(size,size_bin,total_residue); // make sure it's initialized
	retrieve_insertmap(insert_map,total_insert);
	int map_site = static_cast< int >( ran3() * ( total_insert - size ) ) + 1;

//car check that it's really an insertable position
L400:
	for ( int i = 1; i <= size; ++i ) {
		if ( insert_map(map_site+i-1) != insert_map(map_site+i)-1 ) {
			map_site += i;
			goto L400;
		}
	}
	int start_map_site = map_site;
	int counter = 0;
	int counter2 = 0;

L300:
	frag_begin = insert_map(map_site);
	++counter;
	choose_frag_get_top_N_frags(top_N_frags);

//car make a list of relevant constraints:
	ncst = 0;
	int np_pairs = classical_constraints::BOUNDARY::np_pairs();
	for ( int i = 1; i <= np_pairs; ++i ) {
		if ( classical_constraints::BOUNDARY::pairSeqSep(i) < size ) goto L100;
		if ( std::min(classical_constraints::BOUNDARY::constraintPair(i,1), classical_constraints::BOUNDARY::constraintPair(i,2)) > frag_begin ) goto L100;
		if ( std::max(classical_constraints::BOUNDARY::constraintPair(i,1), classical_constraints::BOUNDARY::constraintPair(i,2)) < frag_begin+size-1 )
		 goto L100;
		++ncst;
		cst_list(ncst) = i;
L100:;
	}
//      std::cout << "ncst " << ncst << std::endl;
	if ( ncst == 0 ) {
		nn_num = static_cast< int >(ran3()*25) + 1;
		 // pick a random frag from the top 25
		goto L200;
	}


	{     //scope
	retrieve_best_pose();

	FArray2D_float ZtC( 3, 3 );
	FArray1D_float O2( 3 );
	FArray2D_float A2( 3, 3 );
	FArray1D_float U2( 3 );

	compute_frag_move(Eposition(1,1,frag_begin),phi(frag_begin),
	 Eposition(1,1,frag_begin+size),phi(frag_begin+size),A2,U2,ZtC,O2);

//car evaluate constraints:
	FArray1D_float xyz1( 3 ), xyz2( 3 );
	cst_score = 0.0;
	for ( int i = 1; i <= ncst; ++i ) {
		classical_constraints::BOUNDARY::get_noe_coords(cst_list(i),xyz1,xyz2);
		classical_constraints::BOUNDARY::noe_dist2(xyz1,xyz2,dist2);
		cst_score += std::max(dist2 - classical_constraints::BOUNDARY::pairRadius2(cst_list(i)),0.0f);
	}
//	std::cout << "cst_score" << SS( cst_score ) << SS( dist2 ) <<
//	 SS( pairRadius2(1) ) << std::endl;

//car evaluate frags, starting from random list position
	max_nn_num = std::min(top_N_frags,align_depth(frag_begin,size_bin));
	nn_num = static_cast< int >(ran3()*max_nn_num) + 1;
	++counter2;
	final_nn_num = nn_num - 1;
	if ( final_nn_num == 0 ) final_nn_num = max_nn_num;
//      std::cout << "nn_num " << nn_num << ' ' << final_nn_num << std::endl;
	while ( nn_num != final_nn_num ) {
		precompute_fragmove_noedist(frag_rot(1,1,nn_num,frag_begin,size_bin),
		 frag_off(1,nn_num,frag_begin,size_bin),A2,U2,ZtC,O2,R,L);
		check_score = 0.0;
		for ( int i = 1; i <= ncst; ++i ) {
			FArray1D_float xyz1( 3 ), xyz2( 3 );
			classical_constraints::BOUNDARY::get_noe_coords(cst_list(i),xyz1,xyz2);
			frag_move_noe_dist2(xyz1,xyz2,R,L,dist2);
			check_score += std::max(dist2 - classical_constraints::BOUNDARY::pairRadius2(cst_list(i)),0.0f);
		}
//		std::cout << "scores " << nn_num << ' ' << cst_score << ' ' <<
//		 check_score << std::endl;
		if ( check_score <= cst_score*tolerance ) {
			for ( int i = 0; i < size; ++i ) { // different than what's in structure
				if ( align_name(frag_begin,nn_num,size_bin) != name(i+frag_begin) )
				 goto L200;
			}
		}
		++nn_num;
		++counter2;
		if ( nn_num > max_nn_num ) nn_num = 1;
	}

//car at this point, tried entire frag list at frag_begin
	++map_site;
	if ( map_site > total_insert - size ) map_site = 1;

//car check that the next size positions in insert_map are contiguous:
L500:
	for ( int i = 1; i <= size; ++i ) {
		if ( insert_map(map_site+i-1) != insert_map(map_site+i)-1 ) {
			map_site += i;
			goto L500;
		}
	}

	if ( map_site == start_map_site ) {
		std::cout << "unable to find any cst_frags" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	goto L300; // try next insertion point

	} //end scope

L200:
	if ( counter > 10 && runlevel >= inform ) std::cout << "#sites: " << counter <<
	 " #frags: " << counter2 << std::endl;
	insert_frag(size,size_bin,frag_begin,nn_num);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin precompute_fragmove_noedist
///
/// @brief
///
/// @detailed
///car given point V, how do you determine where it will be upon replacement
///car of one frag with another?
///
///car ZtC and O2 are rotation and offset to move stub at frag_begin to the
///car std orientation
///car once in the std orientation, A2 and U2 'unmake' the current frag
///car insertion (ie transform the stub at frag_end+1 to that at frag_begin
///car in the std coord system, A1 and U1 'insert' the new frag
///car so the following transformations need to be applied:
///car V' = ZtC(V-O2)           move to std coord sys
///car V'' = A2(V'-U2)          take out the  current frag
///car V''' = A1^transpose * V'' + U1     insert the new frag
///car V'''' = ZtC^transpose * V''' + O2
///
///car The code below computes the net rotation (R) and offset (L) to be
///car applied:
///
///car R = ZtC^t * A1^t * A2 * ZtC
///car L = -ZtC^t * A1^t * A2 * ZtC * O2 - ZtC^t * A1^t * A2 *U2
///car     +  ZtC^t * U1 + O2
///
/// @param  A1 - [in/out]? -
/// @param  U1 - [in/out]? -
/// @param  A2 - [in/out]? -
/// @param  U2 - [in/out]? -
/// @param  ZtC - [in/out]? -
/// @param  O2 - [in/out]? -
/// @param  R - [in/out]? -
/// @param  L - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
precompute_fragmove_noedist(
	FArray2Da_float A1,
	FArray1Da_float U1,
	FArray2Da_float A2,
	FArray1Da_float U2,
	FArray2Da_float ZtC,
	FArray1Da_float O2,
	FArray2Da_float R,
	FArray1Da_float L
)
{
	A1.dimension( 3, 3 );
	U1.dimension( 3 );
	A2.dimension( 3, 3 );
	U2.dimension( 3 );
	ZtC.dimension( 3, 3 );
	O2.dimension( 3 );
	R.dimension( 3, 3 );
	L.dimension( 3 );


	FArray2D_float S( 3, 3 );
	FArray2D_float T( 3, 3 );
	FArray1D_float M( 3 );
	FArray1D_float N( 3 );

	mat_multiply3(A2,ZtC,R);
	transpose3(ZtC);
	mat_multiply_transpose3(A1,R,T); // T = A1^transpose * A2 *ZtC
	mat_multiply3(ZtC,T,R); // R=ZtC^trans*A1^trans * A2 *ZtC

	mat_multiply_transpose3(A1,A2,T); // T = A1^transpose * A2
	mat_multiply3(ZtC,T,S); // S=ZtC^trans*A1^trans *A2

	rotate(R,O2,L); // L = ZtC^trans*A1^trans * A2 *ZtC *O2
	rotate(S,U2,M); // M = ZtC^trans*A1^trans *A2 *U2
	rotate(ZtC,U1,N); // N = ZtC^trans * U1

	L(1) = -L(1)-M(1)+N(1)+O2(1);
	L(2) = -L(2)-M(2)+N(2)+O2(2);
	L(3) = -L(3)-M(3)+N(3)+O2(3);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin frag_move_noe_dist2
///
/// @brief
///
/// @detailed
///
/// @param  xyz1 - [in] -
/// @param  xyz2 - [in] -
/// @param  R - [in/out]? -
/// @param  L - [in/out]? -
/// @param  dis2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
frag_move_noe_dist2(
	FArray1Da_float xyz1,
	FArray1Da_float xyz2_in,
	FArray2Da_float R,
	FArray1Da_float L,
	float & dis2
)
{
	xyz1.dimension( 3 );
	xyz2_in.dimension( 3 );

	R.dimension( 3, 3 );
	L.dimension( 3 );

	FArray1D_float tmp( 3 );

//	std::cout << "xyz1";
//	for ( int k = 1; k <= 3; ++k ) {
//		std::cout << SS( xyz1(k) );
//	} std::cout << std::endl;
//	std::cout << "xyz2_in";
//	for ( int k = 1; k <= 3; ++k ) {
//		std::cout << SS( xyz2_in(k) );
//	} std::cout << std::endl;

	rotate(R,xyz2_in,tmp);

	FArray1D_float xyz2( 3 );
	xyz2(1) = tmp(1) + L(1);
	xyz2(2) = tmp(2) + L(2);
	xyz2(3) = tmp(3) + L(3);

//	std::cout << "xyz2 orig frame";
//	for ( int k = 1; k <= 3; ++k ) {
//		std::cout << SS( xyz2(k) );
//	} std::cout << std::endl;

	float const dif1 = xyz1(1) - xyz2(1);
	float const dif2 = xyz1(2) - xyz2(2);
	float const dif3 = xyz1(3) - xyz2(3);
	dis2 = ( dif1 * dif1 ) + ( dif2 * dif2 ) + ( dif3 * dif3 );

}

/////////////////////////////////////////////////////////////////////////////////////

void
copy_fragments_one_size(
												int const source_res,
												int const dest_res,
												int const numres ,
												FArray3Da_float a_phi,
												FArray3Da_float a_psi,
												FArray3Da_float a_omega,
												FArray1Da_int a_frag_list_depth,
												int const a_max_neigh,
												int const a_max_len,
												int const a_max_res,
												FArray2Da_string a_name,
												FArray3Da_char a_ss_type,
												FArray2Da_char a_chain,
												FArray3Da_char a_res_id,
												FArray2Da_int a_resseq // for new vall
												 )
{
	a_phi.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_psi.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_omega.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_frag_list_depth.dimension( a_max_res );
	a_name.dimension( a_max_res, a_max_neigh );
	a_ss_type.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_chain.dimension( a_max_res, a_max_neigh );
	a_res_id.dimension( a_max_res, a_max_neigh, SRange( 0, a_max_len ) );
	a_resseq.dimension( a_max_res, a_max_neigh );

	for ( int i = 1; i <= numres; ++i ) {
		int const source_i = source_res + i - 1;
		int const dest_i   = dest_res   + i - 1;

		a_frag_list_depth( dest_i ) = a_frag_list_depth( source_i );

		for ( int j = 1; j <= a_frag_list_depth( source_i ) ; ++j ) {

			for ( int k = 0; k < a_max_len; ++k ) {

				a_phi  ( dest_i, j, k ) = a_phi  ( source_i, j, k );
				a_psi  ( dest_i, j, k ) = a_psi  ( source_i, j, k );
				a_omega( dest_i, j, k ) = a_omega( source_i, j, k );

				a_res_id ( dest_i, j, k ) = a_res_id ( source_i, j, k );
				a_ss_type( dest_i, j, k ) = a_ss_type( source_i, j, k );

			}

			a_name  ( dest_i, j ) = a_name  ( source_i, j );
			a_chain ( dest_i, j ) = a_chain ( source_i, j );
			a_resseq( dest_i, j ) = a_resseq( source_i, j );


		}
	}

}

/////////////////////////////////////////////////////////////////////////////////////
void
copy_fragments( int const source_res, int const dest_res, int const numres ){
	using namespace fragments;
	using namespace files_paths;
	using namespace param;

	for ( int i = 1; i <= n_frag_sizes(); i++ ) {

		copy_fragments_one_size( source_res, dest_res, numres,
														 align_phi(1,1,0,i), align_psi(1,1,0,i),
														 align_omega(1,1,0,i), align_depth(1,i), MAX_NEIGH(), frag_sizes(i), MAX_RES(),
														 align_name(1,1,i), ss_type(1,1,0,i), align_chain(1,1,i),
														 align_res_id(1,1,0,i), align_resseq(1,1,i) );

	}

}

