// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20460 $
//  $Date: 2008-02-18 23:02:35 -0500 (Mon, 18 Feb 2008) $
//  $Author: possu $


// Rosetta Headers
#include "fragments_pose.h"
#include "after_opts.h"
#include "angles.h"
#include "barcode_stats.h"
#include "chuck.h"
#include "current_pose.h"
#include "files_paths.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "gunn.h"
#include "jumping_ns.h"
#include "pose.h"
#include "maps_ns.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "random_numbers.h"
#include "read_paths.h"
#include "recover.h"
#include "refold.h"
#include "ssblocks.h"
#include "vall_data.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>


//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>


///////////////////////////////////////////////////////////////////////////////
void
choose_fragment_pose(
	pose_ns::Pose & pose,
	int const size_in,
	bool const do_ss_check, // = false
	bool const do_end_bias_check // = false
)
{
	int size( size_in ), begin;
	choose_fragment_pose( pose, size, begin, do_ss_check, do_end_bias_check );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_fragment_pose
///
/// @brief randomly selects a fragment of desired size and inserts it into the
/// phi/psi/omega arrays of a pose.
///
/// @detailed
///
///	@param pose  - the pose to be modified
///	@param size  - size of the fragment to be inserted
///	@param begin - location for fragment to be inserted
///	@param do_ss_check - check secondary structure to avoid inserting tiny helices or strands
///	@param do_end_bias_check - stochastically avoid inserting fragments into the end of the protein
///
/// @global_read
///  fragments fragment_ns.h
///  fragments::frag_pointer fragments_ns.h
///
/// @global_write
/// @remarks
///
/// @references
///  see choose_fragment method in fragments.cc for old, non-pose method method.
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
choose_fragment_pose(
	pose_ns::Pose & pose,
	int & size,
	int & begin, // location frag was inserted
	bool const do_ss_check,
	bool const do_end_bias_check
)
{
	using namespace fragments::frag_pointer;
	using namespace fragments;
	//using namespace misc;
	//using namespace protein_maps;

	//---------------------------------------------------------------
	// local array, static for speed; needed for SS-length check:
	static FArray1D_char secstruct;

	// retrieve pose info:
	int const total_residue ( pose.total_residue() );
	if ( total_residue > int( secstruct.size1() ) ) {
		// need to re-dimension secstruct
		secstruct.dimension( total_residue );
	}

	// get the pose secstruct (const refs)
	FArray1D_char const & pose_secstruct( pose.secstruct() );

	// retrieve the insert map, total_insert, insert_size from pose
	int total_insert;
	FArray1D_int const & insert_map
		( pose.get_insert_map( total_insert ) );
	FArray1D_int const & insert_size
		( pose.get_insert_size() );

	if ( total_insert <= 0 ) {
		// debug:
		std::cout << "choose_fragment_pose:: total_insert <= 0 " << total_insert <<
			std::endl;
		begin = 0;
		size = 0;
		return;
	}

	// how deep should we look in the frags?
	int top_N_frags;
	choose_frag_get_top_N_frags(top_N_frags);

	// get index of fragment set: size should match one of the frag set sizes
	int const size_bin ( get_index_by_frag_size(size) );

	// store input value
	int const init_size  ( size < total_insert ? size : total_insert );

	int tries(0);

 L20: // escape if we've modified secstruct
	// copy pose secstruct to local array:
	for ( int i=1; i<= total_residue; ++i ) {
		secstruct(i) = pose_secstruct(i);
	}

 L30: // escape for bad choice of begin,size ////////////////////////////
	++tries;
	if (tries%1000 == 0 ) {
		std::cout << tries << " tries in choose_fragment_pose" << std::endl;
		assert( false );
	}


	size = init_size; // in jumping_mode, size may change, so reset it

	// choose an insertion point, and map it onto 1->total_residue
	// PB 09-29-05 change from total_insert-size+1 to total_insert
	begin =
		insert_map( static_cast< int >( ran3() * total_insert ) + 1 );

	// trim back as necessary: insert_size stores the length of the longest
	// allowed insert that begins at begin
	size = std::min( insert_size( begin ), size );
	assert( size > 0 ); //debug

	// check how many residues we are moving:
	if ( do_end_bias_check ) {
		if (! end_bias_check ( pose.fold_tree(), begin, size) )goto L30;
	}

	if ( align_depth(begin,size_bin) == 0 ) {
		// not enough frags at this pos
		goto L30;
	}

	// pick random fragment from the list, up to the specified top_N_frags depth
	int nn_num;
	if ( get_ssblock_state() ) {
		int const jnn_num = static_cast< int >
			( ran3() * std::min(align_depth(begin,size_bin),block_depth(begin,size_bin)) ) + 1;
		nn_num = block_frag_pointer(jnn_num,begin,size_bin);
	} else if ( use_weighted_frags() ) {
		// calculate a weighted nn_num based on fragment weights.
		nn_num = static_cast< int >
			( ran3() * std::min(top_N_frags, align_depth(begin,size_bin)) ) + 1;
	} else {
		nn_num = static_cast< int >
			( ran3() * std::min(top_N_frags,align_depth(begin,size_bin)) ) + 1;
	}

	// insert frag SS into local secstruct array for SS-length check
	for ( int i = 0; i < size; ++i ) {
		if ( align_phi(begin,nn_num,i,size_bin) == 0.0 ||
				 align_psi(begin,nn_num,i,size_bin) == 0.0 ) {
			goto L20; // resets secstruct
		}
		secstruct(i+begin) = ss_type(begin,nn_num,i,size_bin);
	}

	// how often does this happen, I wonder?
	//if ( same_neighbor() == 1 ) goto L20;

	//     DONT ALLOW HELICES OF LESS THAN 3 OR STRANDS OF LESS THAN 2
	if ( do_ss_check ) {
		int helix_len = 0;
		int strand_len = 0;
		for ( int i = 1; i < total_residue; ++i ) {
			if ( secstruct(i) == 'H' ) ++helix_len;
			if ( secstruct(i) == 'E' ) ++strand_len;
			if ( secstruct(i) != secstruct(i+1) ) {
				if ( helix_len != 0 && helix_len < 3 ) {
					goto L20; //resets secstruct
				}
				if ( strand_len != 0 && strand_len < 2 ) {
					goto L20; //resets secstruct
				}
				helix_len = 0;
				strand_len = 0;
			}
		}
	}

	// insert into pose arrays once we are happy:
	for ( int i = 0; i < size; ++i ) {
		pose.set_phi       (i+begin, align_phi   (begin,nn_num,i,size_bin) );
		pose.set_psi       (i+begin, align_psi   (begin,nn_num,i,size_bin) );
		pose.set_omega     (i+begin, align_omega (begin,nn_num,i,size_bin) );
		pose.set_secstruct (i+begin, ss_type     (begin,nn_num,i,size_bin) );
		pose.set_name      (i+begin, align_name  (begin,nn_num,size_bin) );
	}
}


///////////////////////////////////////////////////////////////////////////////
bool
end_bias_check(
  pose_ns::Fold_tree const & fold_tree,
	int const begin,
	int const size
	)
{
	// min_fixed_residues is the number of residues fixed by the
	// single-residue or single-jump move that moves the most residues
	//
	// its possible that fixed_residues is less than min_fixed_residues
	// if we are making a multi-residue move. In this case we will automatically
	// accept. If on the other hand we are just wiggling a tail somewhere then
	// fixed residues will be large and ran3() might be less than the factor
	// below
	int min_fixed_residues;
	int const fixed_residues
		( fold_tree.count_fixed_residues( begin, size, min_fixed_residues ) );

	return ( ran3() <= std::exp( float ( min_fixed_residues - fixed_residues ) /
															 jumping::jumping_end_bias ) );

	// OLD WAY:
	//  	return ( ran3() <=
	//  	 std::exp( ( residues_moved - max_edge_count ) / jumping_end_bias ) );



	return true;
}

///////////////////////////////////////////////////////////////////////////////
// returns true on success
bool
read_vall(
	const std::string & filename,
	Vall::Vall_data & vall_data
)
{

	utility::io::izstream & data ( open_data_file(filename) );
	//std::ifstream data ( filename.c_str() );
	if ( !data ) {
		std::cout << "cant open file: " << filename << std::endl;
		return false;
	}

	//////////////////////////////////////////////
	// this file parsing should be very fast since
	// the vall is enormous
	bool const new_format( filename.find( "trimmed" ) != std::string::npos );

	if ( new_format ) {
		char line[250];
		float phi,psi,omega;
		char seq,ss;
		while ( data ) {
			data.getline( line, 250 );
			if ( data.eof() ) break;

			std::sscanf( line   , "%1c", &seq);
			std::sscanf( line+ 1, "%1c", &ss);


			std::sscanf( line+ 2, "%9f", &phi);
			std::sscanf( line+11, "%9f", &psi);
			std::sscanf( line+20, "%9f", &omega);

			if ( !vall_data.add_line( seq, ss, phi, psi, omega ) ) {
				// out of space
				break;
			}
		}
	} else {
		char line[250];
		float phi,psi,omega;
		int resseq;
		char pdb[3],chain,seq,ss;

		while ( data ) {
			data.getline( line, 250 );
			if ( data.eof() ) break;

			std::sscanf( line,     "%4s",    pdb);
			std::sscanf( line+4 ,  "%1c", &chain);
			std::sscanf( line+10 , "%5d", &resseq);

			std::sscanf( line+6 , "%1c", &seq);
			std::sscanf( line+8 , "%1c", &ss);

			std::sscanf( line+52, "%9f", &phi);
			std::sscanf( line+61, "%9f", &psi);
			std::sscanf( line+70, "%9f", &omega);

			if ( !vall_data.add_line2( pdb, chain, resseq, seq, ss, phi, psi, omega ) ) {
				// out of space
				break;
			}
		}
	}
	data.close();
	return true;
}

///////////////////////////////////////////////////////////////////////////////
//
void
get_frags_by_ss(
	std::string const & target_ss,
	int const begin,
	int const end,
	int const frag_size,
	int const nfrags,
	bool const exclude_gly,
	bool const exclude_pro,
	bool const exclude_cys_peptides,
	int const size_bin
)
{
	int const nres( target_ss.size() );

	/////////////
	// check args
	if ( nfrags > param::MAX_NEIGH() ) {
		std::cout << "no space for that many frags!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( frag_size > param::MAX_LEN() ) {
		std::cout << "no space frags that size!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( nres > param::MAX_RES() ) {
		std::cout << "get_frags_by_ss: increase MAX_RES before filling frag " <<
			"arrays" <<	std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//////////////////////////////////
	// fill in fragment data ( UGLY ):
	files_paths::frag_sizes(size_bin) = frag_size;
	fragments::fragments_nres = std::max( fragments::fragments_nres, nres );


	// params
	const float mismatch_penalty(1.0); // for ss mismatch

	// static data
	static bool init( false );
	static Vall::Vall_data vall;
	static FArray1D_int heap( param::MAX_NEIGH() + 2 );
	static FArray1D_float coheap( param::MAX_NEIGH() + 2 );

	// initialize: read vall
	if ( !init ) {
		init = true;
		// read vall
		std::string filename;
		stringafteroption( "vall",
			"vall.dat.pc60_2002-12-27_v01.2", filename );
		read_vall( filename, vall);
	}

	//
	const int vall_size( vall.size() );


	// from vall
	const FArray1D_float & vall_phi  ( vall.get_phi  () );
	const FArray1D_float & vall_psi  ( vall.get_psi  () );
	const FArray1D_float & vall_omega( vall.get_omega() );

	const FArray1D_char & vall_sequence ( vall.get_sequence () );
	const FArray1D_char & vall_secstruct( vall.get_secstruct() );

	// pick fragments for any window that overlaps the end, and within protein
	int const end_frag( std::min( end, int(target_ss.size()) - frag_size + 1 ));

	for ( int frag_begin = begin; frag_begin <= end_frag; ++frag_begin ) {
		// reset heaps
		heap_init( heap, coheap, nfrags );

		// get sequence ss for this window
		std::string const frag_ss = target_ss.substr(frag_begin-1,frag_size);

		for ( int vall_pos=1; vall_pos <= vall_size - frag_size+1; ++vall_pos ) {
			// score this position
			bool bad_frag( false );
			float score(0.0); // bigger is worse
			for ( int k=0; k< frag_size; ++k ) {
				const float phi  ( vall_phi       ( vall_pos+k ) );
				const float psi  ( vall_psi       ( vall_pos+k ) );
				const float omega( vall_omega     ( vall_pos+k ) );
				const char  seq  ( vall_sequence  ( vall_pos+k ) );
				const char  ss   ( vall_secstruct ( vall_pos+k ) );
				if ( ( std::abs( phi ) < 0.01 ) ||
						 ( std::abs( psi ) + std::abs( omega ) < 0.01 ) ||
						 ( seq == 'G' && exclude_gly ) ||
						 ( seq == 'P' && exclude_pro ) ||
						 ( std::abs( omega ) < 90.0 && exclude_cys_peptides ) ) {
					bad_frag = true;
					break;
				}
				if ( ss != frag_ss[k] ) {
					score += mismatch_penalty;
				}
			}
			if ( bad_frag ) continue;

			// randomly reorder frags with the same score
			score += mismatch_penalty * 0.1 * ran3();

			// insert into heap
			bool err;
			heap_insert( heap, coheap, vall_pos, -score, err );
		}

		// now extract top nfrags matches, copy into frag array
		// fragments come out of the heap from worst to best
		int exact_matches(0);
		float worst_score(999);
		for ( int nn= nfrags; nn >=1; --nn ) {
			bool err;
			int vall_pos;
			float score;
			heap_extract( heap, coheap, vall_pos, score, err);
			assert( !err );

			if ( score >= -0.1 * mismatch_penalty ) ++exact_matches;
			if ( nn == nfrags ) worst_score = -score;

			// copy into the frag arrays
			for ( int k=0; k< frag_size; ++k ) {
				fragments::align_phi
					( frag_begin, nn, k, size_bin ) = vall_phi       ( vall_pos+k );
				fragments::align_psi
					( frag_begin, nn, k, size_bin ) = vall_psi       ( vall_pos+k );
				fragments::align_omega
					( frag_begin, nn, k, size_bin ) = vall_omega     ( vall_pos+k );
				fragments::ss_type
					( frag_begin, nn, k, size_bin ) = vall_secstruct ( vall_pos+k );

// 				std::cout << frag_begin << ' ' << nn << ' ' << k << ' ' <<
// 					size_bin << ' ' <<
// 					fragments::align_phi  ( frag_begin, nn, k, size_bin ) << ' ' <<
// 					fragments::align_psi  ( frag_begin, nn, k, size_bin ) << ' ' <<
// 					fragments::align_omega( frag_begin, nn, k, size_bin ) << std::endl;
			}
		} // nn

		std::cout << "ss-frags: " << frag_begin <<
			" frag_ss: " << frag_ss << " exact_matches: " << exact_matches <<
			" worst_score: " << worst_score << std::endl;

		fragments::align_depth( frag_begin, size_bin ) = nfrags;

		//if ( exact_matches > min_frags ) {
		//	fragments::align_depth( frag_begin, size_bin ) = exact_matches;
		//} else {
		//	fragments::align_depth( frag_begin, size_bin ) = min_frags;
		//}

	} // frag_begin
}


///////////////////////////////////////////////////////////////////////////////
void
read_fragments_simple(
	std::string const & frag_name, // eg 1b72A
	int const frag_nres
)
{
	if ( !param::MAX_RES().initialized() || frag_nres > param::MAX_RES()() ) {
		std::cout << "WARNING:: increasing MAX_RES to " << frag_nres << std::endl;
		param::MAX_RES_assign_res( frag_nres );
	}

	if ( truefalseoption("frag_dir") ) {
		files_paths::fragments_path_1 = stringafteroption("frag_dir")+"/";
	}

	std::string save_name ( files_paths::protein_name );
	char save_chain ( files_paths::protein_chain );
	int const save_nres ( misc::total_residue);

	files_paths::protein_name = frag_name.substr(0,4);
	files_paths::protein_chain = frag_name[4];
	misc::total_residue = frag_nres;

	read_fragments();

	files_paths::protein_name = save_name;
	files_paths::protein_chain = save_chain;
	misc::total_residue = save_nres;
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
void
get_vall_frags(
	std::string const & target_seq,
	std::string const & target_ss,
	std::string const & target_bb,
	float const seq_weight,
	float const ss_weight,
	float const bb_weight,
	int const begin,
	int const end,
	int const frag_size,
	int const nfrags,
	bool const exclude_gly,
	bool const exclude_pro,
	bool const exclude_cys_peptides,
	int const size_bin
)
{
	int const nres( target_ss.size() );
	bool const score_bb( std::abs( bb_weight ) > 1e-3 );

	if ( !param::MAX_LEN().initialized() ) {
		param::MAX_LEN() = frag_size;
	}

	if ( !param::n_frag_sizes().initialized() ) {
		param::n_frag_sizes() = size_bin;
	}


	/////////////
	// check args
	if ( nfrags > param::MAX_NEIGH()() ) {
		std::cout << "no space for that many frags!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( frag_size > param::MAX_LEN()() ) {
		std::cout << "no space frags that size!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( nres > param::MAX_RES()() ) {
		std::cout << "get_frags_by_ss: increase MAX_RES before filling frag " <<
			"arrays" <<	std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( size_bin > param::n_frag_sizes()() ) {
		std::cout << "WARNING!!!!!! increasing n_frag_sizes!!!\nThis will " <<
			" delete all your old fragments! " << size_bin << ' ' <<
			param::n_frag_sizes() << std::endl;
		param::n_frag_sizes() = size_bin;
	}

// 	if ( frag_size > param::MAX_LEN()() ) {
// 		std::cout << "WARNING!!!!!! increasing param::MAX_LEN!!!\nThis will " <<
// 			" delete all your old fragments! " << frag_size << ' ' <<
// 			param::MAX_LEN() << std::endl;
// 		param::MAX_LEN() = frag_size;
// 	}

	// ensure no bogus frags are used
	// this is a little tricky, eg in fibril case where we are only picking
	// frags for windows contained in the first monomer, but MAX_RES and
	// nres will be larger than the start of the final fragment window
	// (unless size == 1 )
	//
	for ( int i=begin; i<= end; ++i ) {
		fragments::align_depth( i, size_bin ) = 0;
	}

	//////////////////////////////////
	// fill in fragment data ( UGLY ):
	files_paths::frag_sizes(size_bin) = frag_size;
	fragments::fragments_nres = std::max( fragments::fragments_nres, nres );

	// static data
	static bool init( false );
	static Vall::Vall_data vall;
	static FArray1D_int heap( param::MAX_NEIGH() + 2 );
	static FArray1D_float coheap( param::MAX_NEIGH() + 2 );

	// initialize: read vall
	if ( !init ) {
		init = true;
		// read vall
		std::string filename;
		stringafteroption( "vall",
			"vall.dat.pc60_2002-12-27_v01.2", filename );
		read_vall( filename, vall);
	}

	//
	const int vall_size( vall.size() );


	// from vall
	const FArray1D_float & vall_phi  ( vall.get_phi  () );
	const FArray1D_float & vall_psi  ( vall.get_psi  () );
	const FArray1D_float & vall_omega( vall.get_omega() );

	const FArray1D_string & vall_pdb   ( vall.get_pdb () );
	const FArray1D_char   & vall_chain ( vall.get_chain() );
	const FArray1D_int    & vall_resseq( vall.get_resseq() );

	const FArray1D_char & vall_sequence ( vall.get_sequence () );
	const FArray1D_char & vall_secstruct( vall.get_secstruct() );

	// pick fragments for any window that overlaps the end, and within protein
	int const end_frag( std::min( end, int(target_ss.size()) - frag_size + 1 ));

	for ( int frag_begin = begin; frag_begin <= end_frag; ++frag_begin ) {
		// reset heaps
		heap_init( heap, coheap, nfrags );

		// get sequence ss for this window
		std::string const frag_ss  = target_ss .substr(frag_begin-1,frag_size);
		std::string const frag_seq = target_seq.substr(frag_begin-1,frag_size);
		std::string const frag_bb  = target_bb .substr(frag_begin-1,frag_size);

		for ( int vall_pos=1; vall_pos <= vall_size - frag_size+1; ++vall_pos ) {
			// score this position
			bool bad_frag( false );
			float score(0.0); // bigger is worse
			for ( int k=0; k< frag_size; ++k ) {
				const float phi  ( vall_phi       ( vall_pos+k ) );
				const float psi  ( vall_psi       ( vall_pos+k ) );
				const float omega( vall_omega     ( vall_pos+k ) );
				const char  seq  ( vall_sequence  ( vall_pos+k ) );
				const char  ss   ( vall_secstruct ( vall_pos+k ) );
				if ( ( std::abs( phi ) < 0.01 ) ||
						 ( std::abs( psi ) + std::abs( omega ) < 0.01 ) ||
						 ( seq == 'G' && exclude_gly && seq != frag_seq[k] ) ||
						 ( seq == 'P' && exclude_pro && seq != frag_seq[k] ) ||
						 ( std::abs( omega ) < 90.0 && exclude_cys_peptides ) ) {
					bad_frag = true;
					break;
				}
				if ( ss != frag_ss[k] && frag_ss[k] != 'D' ) {
					score += ss_weight;
				}
				if ( seq != frag_seq[k] ) {
					score += seq_weight;
				}
				if ( score_bb &&  torsion2big_bin( phi, psi, omega )  != frag_bb[k] ) {
					score += bb_weight;
				}
			}

			if ( bad_frag ) continue;

			// randomly reorder frags with the same score
			score += std::min( ss_weight, seq_weight ) * 0.1 * ran3();

			// insert into heap, with negative score! since bigger==better for heaps
			bool err;
			heap_insert( heap, coheap, vall_pos, -score, err );
		}

		// now extract top nfrags matches, copy into frag array
		// fragments come out of the heap from worst to best
		int exact_matches(0);
		float worst_score(999), best_score(999);
		std::map< char, int > rsd_count;
		for ( int nn= nfrags; nn >=1; --nn ) {
			bool err;
			int vall_pos;
			float score;
			heap_extract( heap, coheap, vall_pos, score, err);
			assert( !err );

			if ( score >= -0.1 * std::min( ss_weight, seq_weight )) ++exact_matches;

			if ( nn == nfrags ) worst_score = -score;
			else if ( nn == 1 ) best_score = -score;

			fragments::align_name
				( frag_begin, nn, size_bin ) = vall_pdb  ( vall_pos );
			fragments::align_chain
				( frag_begin, nn, size_bin ) = vall_chain( vall_pos );

			fragments::align_resseq
				( frag_begin, nn, size_bin ) = vall_resseq( vall_pos );

			// copy into the frag arrays
			for ( int k=0; k< frag_size; ++k ) {
				fragments::align_phi
					( frag_begin, nn, k, size_bin ) = vall_phi       ( vall_pos+k );
				fragments::align_psi
					( frag_begin, nn, k, size_bin ) = vall_psi       ( vall_pos+k );
				fragments::align_omega
					( frag_begin, nn, k, size_bin ) = vall_omega     ( vall_pos+k );
				fragments::ss_type
					( frag_begin, nn, k, size_bin ) = vall_secstruct ( vall_pos+k );
				fragments::align_res_id
					( frag_begin, nn, k, size_bin ) = vall_sequence  ( vall_pos+k );


				if ( k==0 ) ++rsd_count[ vall_sequence( vall_pos ) ];
			}
		} // nn

		std::cout << "ss-frags: " << frag_begin <<
			" frag_ss: " << frag_ss <<
			" frag_seq: " << frag_seq <<
			" seq-match: " << rsd_count[ target_seq[ frag_begin - 1 ] ] <<
			" exact_matches: " << exact_matches <<
			" best_score: " << best_score <<
			" worst_score: " << worst_score << std::endl;

		if ( false ) {
			// show seq stats:
			std::cout << "target_seq: " << target_seq[ frag_begin-1 ];
			for ( std::map< char, int >::const_iterator it=rsd_count.begin();
						it != rsd_count.end(); ++it ) {
				std::cout << ' ' << it->first << it->second;
			}
			std::cout << std::endl;
		}

		fragments::align_depth( frag_begin, size_bin ) = nfrags;

		//if ( exact_matches > min_frags ) {
		//	fragments::align_depth( frag_begin, size_bin ) = exact_matches;
		//} else {
		//	fragments::align_depth( frag_begin, size_bin ) = min_frags;
		//}

	} // frag_begin
}


///////////////////////////////////////////////////////////////////////////////
void
dump_frags(
					 std::string const & filename,
					 int const size,
					 int const first_window,
					 int const last_window
					 )
{
	using namespace fragments;
	utility::io::ozstream out( filename );

	int const size_bin ( get_index_by_frag_size(size) );

	for ( int pos= first_window; pos<= last_window; ++pos ) {
		int const depth( align_depth( pos, size_bin ) );
		out << "BEGIN pos= " << pos << " depth= " << depth << '\n';
		for ( int i=1; i<= depth; ++i ) {
			out << I(4,i);
			for ( int j=0; j< size; ++j ) {
				out << ' ' << ss_type( pos, i, j, size_bin ) <<
					' ' << F(9,3,align_phi  ( pos, i, j, size_bin )) <<
					' ' << F(9,3,align_psi  ( pos, i, j, size_bin )) <<
					' ' << F(9,3,align_omega( pos, i, j, size_bin ));
			}
			out << '\n';
		}
	}
	out.close();
}

///////////////////////////////////////////////////////////////////////////////
void
read_frags_new(
							 std::string const & filename,
							 int const size,
							 int const size_bin
							 )
{
	using namespace fragments;
	utility::io::izstream data( filename );

	if ( !param::MAX_LEN().initialized() ) {
		param::MAX_LEN() = size;
	}

	if ( !param::n_frag_sizes().initialized() ) {
		param::n_frag_sizes() = size_bin;
	}


	/////////////
	// check args

	if ( size > param::MAX_LEN()() ) {
		std::cout << "no space frags that size!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( size_bin > param::n_frag_sizes()() ) {
		std::cout << "WARNING!!!!!! increasing n_frag_sizes!!!\nThis will " <<
			" delete all your old fragments! " << size_bin << ' ' <<
			param::n_frag_sizes() << std::endl;
		param::n_frag_sizes() = size_bin;
	}

	//////////////////////////////////
	// fill in fragment data ( UGLY ):
	files_paths::frag_sizes(size_bin) = size;


	//int const size_bin ( get_index_by_frag_size(size) );

	FArray3Dp_float phi  ( align_phi  (1,1,0,size_bin), param::MAX_RES(),
												 param::MAX_NEIGH(), SRange( 0, size ) );
	FArray3Dp_float psi  ( align_psi  (1,1,0,size_bin), param::MAX_RES(),
												 param::MAX_NEIGH(), SRange( 0, size ) );
	FArray3Dp_float omega( align_omega(1,1,0,size_bin), param::MAX_RES(),
												 param::MAX_NEIGH(), SRange( 0, size ) );

	std::string line,tag1,tag2,tag3;
	int last_pos(0);
	while ( getline( data, line ) ) {
		std::istringstream is( line );
		int pos, depth;
		//out << "BEGIN pos= " << pos << " depth= " << depth << '\n';
		is >> tag1 >> tag2 >> pos >> tag3 >> depth;
		if ( is.fail() ) {
			std::cout << "format error: " << line << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( depth > param::MAX_NEIGH()() ) {
			std::cout << "no space for that many frags!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( pos > param::MAX_RES()() ) {
			std::cout << "get_frags_by_ss: increase MAX_RES before filling frag " <<
				"arrays" <<	std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		last_pos = std::max( last_pos,pos);
		align_depth( pos, size_bin ) = depth;

		for ( int i=1; i<= depth; ++i ) {
			getline( data, line );
			std::istringstream l( line );
			//out << I(4,i);
			int itmp;
			l >> itmp;
			if ( i != itmp ) {
				std::cout << "format error2: " << line << std::endl;
				std::exit( EXIT_FAILURE );
			}
			for ( int j=0; j< size; ++j ) {
// 				out << ' ' << ss_type( pos, i, j, size_bin ) <<
// 					' ' << F(9,3,align_phi  ( pos, i, j, size_bin )) <<
// 					' ' << F(9,3,align_psi  ( pos, i, j, size_bin )) <<
// 					' ' << F(9,3,align_omega( pos, i, j, size_bin ));
				l >> ss_type( pos, i, j, size_bin ) >>
					phi  ( pos, i, j ) >>
					psi  ( pos, i, j ) >>
					omega( pos, i, j );
			}
		}
	}
	data.close();

	fragments::fragments_nres = std::max( fragments::fragments_nres,
																				last_pos+size-1 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin use_weighted_frags
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
use_weighted_frags()
{
	static bool use_weighted_frags = { false };
	static bool init = { false };

	if ( !init ) {
		use_weighted_frags = truefalseoption("use_weighted_frags");
		init = true;
	}
	return use_weighted_frags;
}

void
read_fragment_weights()
{
	std::string weight_file = stringafteroption("frag_weight_file");
	std::ifstream wdata( weight_file.c_str() );

	if ( !wdata ) {
		std::cout << "Error opening file in read_fragment_weights!"
							<< " (filename = " << weight_file << std::endl;
		std::exit( EXIT_FAILURE );
	}

	using namespace misc;
	using namespace files_paths;

	int position, frag_index, size_bin, size;
	float weight;
	FArray3D_float fragment_weights( total_residue, max_frags, param::n_frag_sizes() );
	std::string line;
	while ( std::getline( wdata, line ) ) {
		std::istringstream line_stream (line);
		line_stream >> position >> frag_index >> weight >> size >> skip;
		size_bin = get_index_by_frag_size(size);
		fragment_weights( position, frag_index, size_bin ) = weight;
	}

	wdata.close();
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
