// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13703 $
//  $Date: 2007-03-23 22:18:34 -0500 (Fri, 23 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "fullatom.h"
#include "are_they_neighbors.h"
#include "after_opts.h"
#include "angles.h"
#include "assemble_domains.h"
#include "namespace_assemble_options.h"
#include "constraints.h"
#include "decoystats.h"
#include "decoy_features.h"
#include "design.h"
#include "docking.h"
#include "docking_movement.h"
#include "files_paths.h"
#include "fullatom_setup.h"
#include "maps.h"
#include "maps_ns.h"
#include "misc.h"
#include "monte_carlo.h"
#include "namespace_fullatom_flag.h"
#include "nblist.h"
#include "pack.h"
#include "PackerTask.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "prof.h"
#include "pose.h"
#include "pose_io.h"
#include "current_pose.h"
#include "recover.h"
#include "rotamer_trials.h"
#include "score.h"
#include "start.h"
#include "status.h"
#include "termini.h"
#include "water_ns.h"
#include "ligand.h"
#include "make_pdb.h"
// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cassert>
#include <cstdlib>
#include <iostream>

// Namespaces
namespace fullatom_packall {
	bool pack_all = { true };
}

//Utility Headers
#include <utility/basic_sys_util.hh>

//////////////////////////////////////////////////////////////////////////////
/// @begin get_packing_options
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_packing_options()
{
	using namespace design;
	using namespace exchi_flags;
	using namespace files_paths;
	using namespace water;

	active_rotamer_options.read_rotamer_flags();

//bk this flag turns on repulsion forces between like charged amino acids by
//bk changing the probabilites in the pair term (see fullatom_setup)
	use_electrostatic_repulsion = truefalseoption("use_electrostatic_repulsion");

	explicit_h2o = truefalseoption("explicit_h2o");
	solvate = truefalseoption("solvate");
	if (solvate) explicit_h2o = true;

 	if( truefalseoption("pH") ) {
 		realafteroption("pH",7.4,pH_ns::pH);
 		get_pH_packing_options();
 	}
	try_both_his_tautomers = truefalseoption("try_both_his_tautomers");
	if (use_conformer) try_both_his_tautomers = true;

///jjh Option to include hydrated (major groove only) DNA bases
	hydrate_dna = truefalseoption("hydrate_dna");
	ex_dna_waters = truefalseoption("ex_dna_waters");

	minimize_rot = truefalseoption("minimize_rot");
	use_design_matrix = false;

//lin  read native water
	read_hetero_h2o = truefalseoption("read_hetero_h2o");
	if ( read_hetero_h2o ) use_hetero_h2o = true;

// turn off the flag Wint_score_only and Wint_repack_only
	if ( truefalseoption("Wint_score_only") ||
			 truefalseoption("Wint_repack_only") ) {
		std::cout << "the flag of Wint_score_only and Wint_repack_only is no longer used" << std::endl;
	}

//lin  get ligand flag
  get_ligand_flag();

}


///////////////////////////////////////////////////////////////////////////////
// used by pose routines, not generally safe -- doesnt do crazy setup
//
void
set_fullatom_flag_simple(
	bool const setting
)
{
	using namespace fullatom_flag;
	if ( setting == full_atom ) return; // no change
	if ( setting ) initialize_fullatom();

	full_atom = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_fullatom_flag
///
/// @brief switches rosetta from centroid mode to fullatom mode or the reverse
///
/// @detailed initializes coordinates and scoring functions for fullatom operation
///
/// @param[in]   yes_no - in -  turn fullatom on?
///
/// @global_read
///    input_fa            files_paths.h
///    require_start       files_paths.h
///    start_full_coord    block start.h
///    current_pose        misc.h
///    full_atom           static
///
/// @global_write
///    full_coord          misc.h
///    full_atom           static
///
/// @remarks
///car WARNING:: a myriad of indirect global reads and writes
///car
///car option dependencies:
///car input_fa && require_start:
///car    true   use input rotamers  (no checking for sequence compatiblity)
///car    false  repack low backbone
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_fullatom_flag( bool yes_no )
{
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace misc;
	using namespace start;
	using namespace termini_ns;

	if ( yes_no == full_atom ) return; // no change

	if ( pose_flag() ) {
		std::cout << "set_fullatom_flag() and pose_flag() -- why am I here?" <<
			std::endl;
		// just set the variable
		if ( yes_no ) initialize_fullatom();
		full_atom = yes_no;
		return;
	}

	retrieve_low_pose();

	full_atom = yes_no;

	if ( full_atom ) {
		std::cout << "initializing full atom coordinates" << std::endl;
		if ( input_fa ) {
//car put the starting rotamers on the current backbone
			build_fullcoord(Eposition,1,total_residue,res,res_variant,
			is_N_terminus,is_C_terminus,phi(1),start_fullcoord,full_coord);
		} else {
			main_repack(false); // no rotamers exist yet
		}
	}
	setup_allatom_list();
	score_set_new_pose();
	score_set_default_function(full_atom);
	score_enable_rotamer_trials(false); // can't call until mc-reset
	mc_global_track::mc_score::score = scorefxn();
	score_enable_rotamer_trials(true);
	save_status_info("fullatom_flag",0,0);
	monte_carlo_reset();

	// re-save the decoystats / decoyfeatures native with full atoms coors,
	// if we're just turning full-atom on now
	if ( full_atom ) {
		decoystats_store_native();
		decoy_features_ns::decoy_features_store_native();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_flag
///
/// @brief returns fullatom state
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///   full_atom  static
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_fullatom_flag()
{
	using namespace fullatom_flag;

	return full_atom;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_repack
///
/// @brief  set global array describing what can be repacked
///
/// @detailed
///
/// @param[in]   setting - in - desired value of allow_repack array
/// @param[in]   total_residue - in - logical size of setting
///
/// @global_read
///
/// @global_write
///   allow_repack  maps_ns.h
///
/// @remarks
///
/// @references
///
/// @authors  car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_repack(
	FArray1DB_bool const & setting,
	int total_residue
)
{
	using namespace protein_maps;

	for ( int i = 1; i <= total_residue; ++i ) {
		allow_repack(i) = setting(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_allow_repack
///
/// @brief retrieve global array describing what can be repacked
///
/// @detailed
///
/// @param[out]   allow - out - current value of allow_repack
/// @param[in]   total_residue - in - logical size of allow
///
/// @global_read
///   allow_repack  maps_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_allow_repack(
	FArray1DB_bool & allow,
	int total_residue
)
{
	using namespace protein_maps;

	for ( int i = 1; i <= total_residue; ++i ) {
		allow(i) = allow_repack(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_set_packall
///
/// @brief set flag for auto-determination of allow_repack global array
///
/// @detailed
///
/// @param[in]   setting - in - desired setting of pack_all flag
///
/// @global_read
///    pack_all    static
///
/// @global_write
///
/// @remarks
///car    pack_all   false   only variable regions and regions they contact
///car                         are repacked
///car               true    current settings of allow_repack used
///
///car  design and docking do not depend on this flag because they do
///car  not utilize main_repack
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fullatom_set_packall( bool setting )
{
	using namespace fullatom_packall;

	pack_all = setting;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_get_packall
///
/// @brief  retrieve current setting of pack_all flag
///
/// @detailed
///
/// @return
///
/// @global_read
///    pack_all    static
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
fullatom_get_packall()
{
	using namespace fullatom_packall;

	return pack_all;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin select_interface
///
/// @brief identify residues to repack for auto-determination of packing interface
///
/// @detailed
/// selects residues that have moved relative to nearby residues
/// (ie pair_moved=true for neighbors)
///
///
/// @param[out]   allow_repack - out -  should residue be repacked?
///
/// @global_read
///    total_residue misc.h
///    res           misc.h
///    full_coord    misc.h
///    paircutoff    aaproperties_pack.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
select_interface( FArray1DB_bool & allow_repack )
{
	using namespace misc;

//car local
	FArray2DB_bool const & pair_moved( retrieve_pair_moved() );
	float dis2;
  bool neighbors;

  allow_repack = false;
	for ( int i = 1; i <= total_residue; ++i ) {
//car repack residues that have moved relatve to their neighbors
    for ( int j = i+1; j <= total_residue; ++j ) {
			if ( pair_moved(i,j) ) {
        are_they_neighbors(res(i),res(j),full_coord(1,1,i),
            full_coord(1,1,j),dis2,neighbors);
				if ( neighbors ) {
				   allow_repack(i) = true;
				   allow_repack(j) = true;
        }
			}
		}  //jres
	}                     // i res

}
////////////////////////////////////////////////////////////////////////////////
/// @begin main_repack_trial
///
/// @brief repacks current_pose and accepts according to monte_carlo criterion
///
/// @detailed
///
/// @param[in]   scoring_function - in - external
/// @param[in]   cycle_number - in - passed to monte_carlo
///
/// @global_read
///
/// @global_write
///   score     misc.h
///
/// @remarks
///   indirectly reads and modifies many globals: best & current pose, scores
///   internal scoring data
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_repack_trial(
	Scoring_Function scoring_function,
	int cycle_number,
	bool do_rtmin
)
{
	retrieve_best_pose();
	main_repack(true);
	if ( do_rtmin ) {
		bool save_try_rotamers = score_get_try_rotamers();
		bool save_minimize_rot = get_minimize_rot_flag();
		score_set_try_rotamers( true );
		score_set_minimize_rot( true );
		mc_global_track::mc_score::score = scoring_function();
		score_set_minimize_rot( save_minimize_rot );
		score_set_try_rotamers( save_try_rotamers );
		save_status_info("repack_RTMIN",0,0);
	} else {
		mc_global_track::mc_score::score = scoring_function();
		save_status_info("repack",0,0);
	}
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_repack
///
/// @brief repacks current_pose
///
/// @detailed
///
/// @param[in]   rotamers_exist - in - should rotamers currently in the full_coord
///             array be included in the repacking algorithm?
///
/// @global_read
///
/// @global_write
///   score     misc.h
///
/// @remarks
///car if rotamers_exist = true then the HN,N,CA,HA,C,O,CB atoms are
///car assumed to already exist on backbone (as well as sidechains)
///car if rotamers_exist = false then these atoms are copied from position
///
///  modify only the current pose, need to call scorefxn to update scoring data
///
///  behavior depends on pack_all global, allow_repack global
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified chu 12/17/2004
/////////////////////////////////////////////////////////////////////////////////
void
main_repack( bool const rotamers_exist )
{
	using namespace misc;
	using namespace param;
	using namespace assemble_options;

	if ( !get_fullatom_flag() ) return;

	PROF_START( prof::MAIN_REPACK );

	if ( get_docking_flag() ) {
		docking_repack(rotamers_exist);
		PROF_STOP( prof::MAIN_REPACK );
		return;
	}

	if ( !rotamers_exist ) initialize_fullcoord_array(Eposition,full_coord,
	 total_residue,res,res_variant);

	// determine which residues to pack
	FArray1D_bool allow_repack( MAX_RES()() );
	retrieve_allow_repack(allow_repack,total_residue);

// bk 3/21/05 removed global overide of allow_repack
// the benchmarks did not change, the change was made to allow fix_natsc to
// work
//
// chu 03/29/2005 -- more comments on bk's change:
// there is a potential risk of removing the global overide of allow_repack
// because if rotamer_exist is false, backbone in fullcoord will be copied from
// Eposition and if fullcoord has not been intitialized, ie. input_fa is F, or
// if Eposition does not agree fullcood, ie. after centroid refolding, this may
// result in a 'mixed' fullcoord with no/outdated sidechains. I think this is
// why Carol orginally want to overide the global allow_repack array since this
// will ensure building  all sidechains on the new backbone completely anyway.
// therefore, with bk's changes, there will be a problem when the following 3
// conditions are all met:
// 1. rotamer_exist  is False
// 2. fullcoord not intialized or outdated with Eposition
// 3. allow_repack is TRUE only for a subset of residues
// It seems to be very unlikely to run into such a situation (I have not found
// any cases in Rosetta currently), but for safety developers should double
// check this when calling main_repack in the future.
//
//car overide global allow_repack array
//	if ( !rotamers_exist ) {
//		for ( int i = 1; i <= total_residue; ++i ) {
//			allow_repack(i) = true; // repack all
//		}
//  }

	if (rotamers_exist && !fullatom_get_packall() ) {
		select_interface(allow_repack);
	}

	if ( get_assemble_flag()) {
		if (relax_repack_flag) {
			assemble_detect_interf_res();
			assemble_update_allow_repack();
		}
	}

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );
	Task.set_task("packrot", false, allow_repack, rotamers_exist );
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task );
	pose.copy_to_misc();

	if ( get_minimize_rot_flag() )
	 set_allow_rotamer_trials(allow_repack,total_residue);

	PROF_STOP( prof::MAIN_REPACK );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_score_position
///
/// @brief scores the current_pose using fullatom scoring when fullatom flag is false
/// (ie when not in fullatom mode)
///
/// @detailed
///
/// @param[in] repack,          // repack the sidechains before scoring?
/// @param{in] include_current // include current rotamers?
///
/// @global_read
///    current_pose block misc.h
///
/// @global_write
///
///
/// @remarks
///car used to score backbones with complete sidechains without switching
///car rosetta to fullatom mode, for example in initialize to score native
///car and starting structures
///car NOTE:: uses the current settings for scorefxn (may or may not be fullatom)
///
///  repack and include_current are relevant only if fullcoord_exist is true
///
///  indirectly writes to all scoring globals
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fullatom_score_position(
	bool const repack,          // repack the sidechains before scoring?
	bool const include_current  // include current rotamers?
)
{
	using namespace fullatom_flag;

//car local
	bool save_fullatom_state = full_atom;

	full_atom = true;

//generate fullcoord with sidechains if requested
	if (repack) fullatom_pack_position(include_current);

//car score:
	score_enable_rotamer_trials(false);
	score_set_new_pose();
	score_set_default_function( true /*output_fa*/);
	mc_global_track::mc_score::score = scorefxn();
	score_enable_rotamer_trials(true);
	classical_constraints::BOUNDARY::output_constraints_status( -2, std::cout );
	calc_docking_interf_energy( get_flexbb_docking_flag() );
	full_atom = save_fullatom_state;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_pack_position
///
/// @brief packs the current_pose and generates a complete fullcoord array
/// when fullatom_flag is false (ie not in fullatom mode)
///
/// @detailed
///
/// @param[in] rotamers_exist - in -should current rotamers be included?
///
/// @global_read
///    current_pose block misc.h
///
/// @global_write
///
///
/// @remarks
///car used to generate fullatom coordintes for backbones with complete sidechains
///car without switching rosetta to fullatom mode, for example in initialize to
///car generate fullatom native and starting structures to score
///
///car if rotamers_exist is true, it is assumed that the fullcoord array is initialized
///car and usable.
///
///
/// @references
///
/// @authors car 9/11/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
fullatom_pack_position(
	bool const rotamers_exist  // include current rotamers?
)
{
	using namespace fullatom_flag;

//car local
	bool save_fullatom_state = full_atom;

	full_atom = true;
	main_repack(rotamers_exist);
	full_atom = save_fullatom_state;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_position_to_fullcoord
///
/// @brief syncs position and full_coord backbone coordinates
///
/// @detailed copies N,C,CA,O coordinates from position to fullcoord
///
/// @param[in]   position - in - backbone coordinates, source
/// @param[out]   fullcoord - out - fullatom coordinates, destination
/// @param[in]   nres - in - logical size of arrays, number of residues to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   changes atom order: position N,CA,CB,C,O  fullcoord N,CA,C,O ...
///   does not copy CB coordinates
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_position_to_fullcoord(
	FArray3DB_float const & pos,
	FArray3DB_float & fullcrd,
	int nres
)
{
	using namespace misc;
	using namespace param_aa;

	assert( ( pos.size1() == 3 ) );
	assert( ( fullcrd.size1() == 3 ) );

	for ( int i = 1; i <= nres; ++i ) {
		if ( is_protein(res(i)) || is_nonnatural(res(i)) ) {
			for ( int j = 1, lf = fullcrd.index(j,1,i), lp = pos.index(j,1,i);
			 j <= 3; ++j, ++lf, ++lp ) {
				fullcrd[ lf   ] = pos[ lp    ]; // fullcrd(j,1,i) = pos(j,1,i) // N
				fullcrd[ lf+3 ] = pos[ lp+3  ]; // fullcrd(j,2,i) = pos(j,2,i) // CA
				fullcrd[ lf+6 ] = pos[ lp+9  ]; // fullcrd(j,3,i) = pos(j,4,i) // C -- note atom order change
				fullcrd[ lf+9 ] = pos[ lp+12 ]; // fullcrd(j,4,i) = pos(j,5,i) // O
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_fullcoord_array
///
/// @brief generate a fullcoord array ready for side chain packing
///
/// @detailed copies all atoms from position to fullcoord, calcuates HA and HN coords
///
/// @param[in]   position - in - source of backbone coordinates
/// @param[out]   fullcoord - out - initialize coordinate array
/// @param[in]   nres - in - logical size of arrays, number of residues
/// @param[in]   aan - in - sequence
/// @param[in]   aa_variant - in - sequence variants
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car changes atom order: position N,CA,CB,C,O  fullcoord N,CA,C,O,CB, ...
///
///car NOTE:: the packer updates coordinates for CB, HA, HN but
///car in cases where existing rotamers are not given to the packer
///car coordinates for CB,HA,HN must still be provided
///car
///car NOTE:: HN protons are placed using coordinates C,N,CA with NO check for
///car chain breaks; HN protons will be place incorrectly at chain breaks!!
///car rosetta allows chain breaks at C-N bonds when refolding proteins
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_fullcoord_array(
	FArray3DB_float const & position,
	FArray3DB_float & fullcoord,
	int nres,
	FArray1DB_int const & aan,
	FArray1DB_int const & aa_variant
)
{
	using namespace param_aa;
	using namespace termini_ns;

	copy_position_to_fullcoord(position,fullcoord,nres);


	for ( int i = 1; i <= nres; ++i ) {
		if ( aan(i) != aa_gly && ( is_protein(aan(i)) || is_nonnatural(aan(i)) ) ) {
			for ( int j = 1, lf = fullcoord.index(j,5,i), lp = position.index(j,3,i); j <= 3; ++j, ++lf, ++lp ) {
				// CB -- note atom order change
				fullcoord[ lf ] = position[ lp ]; // fullcoord(j,5,i) = position(j,3,i)
			}
		}
	}

	if ( ! quiettruefalseoption( "no_clean" ) ) { //SGM Option leaves HA and HN alone for testing
		put_cah(fullcoord,1,nres,aan,aa_variant); // HA
		put_nh(fullcoord,1,nres,aan,aa_variant,is_N_terminus,is_C_terminus,0.0); // HN
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_fullatom_checkpoint_status
///
/// @brief prepare rosetta for fullatom mode start from a checkpoint pdb
///
/// @detailed initializes coordinates and scoring functions for fullatom operation
///
/// @param[in]   yes_no - in -  turn fullatom on?
///
/// @global_read
///    start_full_coord    block start.h
///    current_pose        misc.h
///
/// @global_write
///    full_coord          misc.h
///    full_atom           static
///
/// @remarks
///
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_checkpoint_status(
	bool const & fullatom
)
{
	if ( pose_flag() ) return;

	setup_allatom_list();
	score_set_new_pose();
	score_set_default_function( fullatom );
	score_enable_rotamer_trials(false); // can't call until mc-reset
	mc_global_track::mc_score::score = scorefxn();
	score_enable_rotamer_trials(true);
	save_status_info("checkpt_flag",0,0);
	monte_carlo_reset();
}

