// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18939 $
//  $Date: 2007-12-06 15:28:23 -0500 (Thu, 06 Dec 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "fullatom_energy.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h"
#include "bond_angle.h"
#include "count_pair.h"
#include "constraints.h"
#include "current_pose.h"
#include "design.h"
#include "disulfides.h"
#include "docking.h"
#include "dunbrack_pack.h"
#include "Dunbrack4D.h"
#include "force_barcode.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_setup.h"
#include "files_paths.h"
#include "gb_elec.h"
#include "gb_elec_ns.h"
#include "geometric_solvation.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "kin_min.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "enzyme.h"
#include "enzyme_ns.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "nblist.h"
#include "nblist_ns.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "pairenergy.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pdbstatistics_pack.h"
#include "planes.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pose.h"
#include "pose_io.h"
#include "prof.h"
#include "relax_structure.h"
#include "rotamer_functions.h"
#include "runlevel.h"
#include "scale_res_energy.h"
#include "score.h"
#include "template_pack.h"
#include "util_interpolate.h"
#include "water.h"
#include "water_ns.h"

#ifdef GL_GRAPHICS
#include "gl_graphics.h"
#endif

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray.io.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>

namespace fullatom_energy_debug {
	bool debug( false );
	bool in_fullatom_energy( false );
}

namespace do_not_use {
	float junk_float;
	FArray1D_float junk_oneDf( param::MAX_CHI );
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//  INTERFACE
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------


//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_totalE
///
/// @brief
///bk fullatomE      :total full atom energy,no backbone-backbone hbondenergies
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_fullatom_totalE()
{
	return fullatom_energies::fullatomE;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sum_fullatom_energies
///
/// @brief
///car output weighted total scores
///
/// @detailed
///
/// @param  nres - [in/out]? -
/// @param  attractiveE - [in/out]? - LJ attractive energy
/// @param  repulsiveE - [in/out]? - LJ repulsive energy
/// @param  solvationE - [in/out]? - Lazaridis Karplus solvation energy
/// @param  referenceE - [in/out]? - aa reference energies
/// @param  protonationE - [in/out]? - Protonation Potential for amino acid
/// @param  dunbrackE - [in/out]? - P(rot,phi,psi)
/// @param  Paa_ppE - [in/out]? - P(aa,phi,psi)
/// @param  pairprobE - [in/out]? - pair term
/// @param  planeprobE - [in/out]? - plane term
/// @param  h2oE - [in/out]? -
/// @param  h2ohbE - [in/out]? -
/// @param  h2o_solE - [in/out]? -
/// @param  srbb_hbondE - [in/out]? - sidechain backbone hydrogen bond energy
/// @param  lrbb_hbondE - [in/out]? -
/// @param  sc_bb_hbondE - [in/out]? - sidechain backbone hydrogen bond energy
/// @param  intra_resE - [in/out]? - intra residue energy (repulsive only)
/// @param  gb_elecE - [in/out]? - Generalized Born energy
/// @param  cstE - [in/out]? - Pose constraint energy
/// @param  bond_angleE - [in/out]? - bond angle energy (C-alpha only)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
sum_fullatom_energies(
	int const nres,
	float & attractiveE,
	float & repulsiveE,
	float & solvationE,
	float & referenceE,
	float & protonationE,
	float & dunbrackE,
	float & Paa_ppE,
	float & pairprobE,
	float & planeprobE,
	float & h2oE,
	float & h2ohbE,
	float & h2o_solE,
	float & srbb_hbondE,
	float & lrbb_hbondE,
	float & sc_bb_hbondE,
	float & intra_resE,
	float & gb_elecE,
  float & elecE,
	float & cstE,
	float & bond_angleE
)
{

	using namespace fullatom_energies;

//car local
	float tot_hbondE = 0.0;

	solvationE = 0.0;
	attractiveE = 0.0;
	repulsiveE = 0.0;
	referenceE = 0.0;
	protonationE = 0.0;
	dunbrackE = 0.0;
	Paa_ppE = 0.0;
	pairprobE = 0.0;
	planeprobE = 0.0;
	h2oE = 0.0;
	h2ohbE = 0.0;
	h2o_solE = 0.0;
	intra_resE = 0.0;
	gb_elecE = 0.0;
  elecE    = 0.0;

	cstE = 0.0;
	bond_angleE = 0.0;

	if ( !score_check_current_pose() || !score_get_current_pose().symmetric()){
		for ( int res = 1; res <= nres; ++res ) {
			solvationE += solenergy(res);
			attractiveE += atrenergy(res);
			repulsiveE += repenergy(res);
			referenceE += unfenergy(res);
			protonationE += protonation_energy(res);
			dunbrackE += dunenergy(res);
			Paa_ppE += probenergy(res);
			pairprobE += pair_energy(res);
			planeprobE += plane_energy(res);
			tot_hbondE += hbenergy(res);
			h2oE += h2oenergy(res);
			h2ohbE += h2ohbenergy(res);
			h2o_solE += wsolenergy(res);
			intra_resE += intraresenergy(res);
			gb_elecE += gb_elecenergy(res);
      elecE    += elecenergy(res);

			cstE += cstenergy(res);
			bond_angleE += bondangleenergy(res);
		}
//bk   subtract backbone-backbone hydrogen bonding energy from tot_hbondE
//bk   in order to get the total sidechain-sidechain/sidechain-backbone
//bk   hydrogen bond energy
//car  hbenergy_bb was saved in namespace with last evaluation
		srbb_hbondE = hbenergy_srbb;
		lrbb_hbondE = hbenergy_lrbb;
		sc_bb_hbondE = tot_hbondE-(hbenergy_srbb+hbenergy_lrbb);
	} else {
		srbb_hbondE = 0.0;
		lrbb_hbondE = 0.0;
		sc_bb_hbondE = 0.0;
		pose_ns::Symmetry_info const & s
			( score_get_current_pose().symmetry_info() );
		for ( int res = 1; res <= nres; ++res ) {
			if ( !s.chi_independent( res ) ) continue;
			float const factor( 1.0f + s.chi_clones(res).size() );
			solvationE += factor * solenergy(res);
			attractiveE += factor * atrenergy(res);
			repulsiveE += factor * repenergy(res);
			referenceE += factor * unfenergy(res);
			protonationE += factor * protonation_energy(res);
			dunbrackE += factor * dunenergy(res);
			Paa_ppE += factor * probenergy(res);
			pairprobE += factor * pair_energy(res);
			planeprobE += factor * plane_energy(res);
			tot_hbondE += factor * hbenergy(res);
			h2oE += factor * h2oenergy(res);
			h2ohbE += factor * h2ohbenergy(res);
			h2o_solE += factor * wsolenergy(res);
			intra_resE += factor * intraresenergy(res);
			gb_elecE += factor * gb_elecenergy(res);
      elecE    += factor * elecenergy(res);

		  cstE += factor * cstenergy(res);
			srbb_hbondE  += factor * hbenergy_sr_bb(res);
			lrbb_hbondE  += factor * hbenergy_lr_bb(res);
			sc_bb_hbondE += factor * ( hbenergy_sc_sc(res) + hbenergy_sc_bb(res) );
		}
	}


}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_energy_full
///
/// @brief
///car a simple wrapper for fullatom_energy to do a complete full energy
///car calculation on the current structure.  The fullatom_energy arrays
///car are updated, but the rosetta scores ARE NOT!!
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fullatom_energy_full(
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord,
	int total_residue
)
{
	maps_set_new_pose();
	fullatom_energy(res,res_variant,full_coord,total_residue,false);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_fullatom_scores
///
/// @brief
///car check that partial energy calculations are correct
///car call immediately after calling fullatom_energy with begin and
///car size not set to 1 and total_residue
///
/// @detailed
///
/// @param  use_subset_energy - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_fullatom_scores( bool const use_subset_energy )
{

	using namespace scores;

	float tmp_fasolv,tmp_faatr,tmp_farep,tmp_faref,tmp_protonation,tmp_fadun,tmp_faprob1b,
	 tmp_fapair,tmp_faplane,tmp_fah2o,tmp_fah2ohb,tmp_fah2osol,tmp_intrares;
	float tmp_srbbhb,tmp_lrbbhb,tmp_schb,tmp_gb_elec,tmp_cst,tmp_bondangle;
  float tmp_elec;

	score_set_new_pose();

  //// yab: misc removal
	pose_ns::Pose pose;
	pose_from_misc(pose, true, false, true);
	fullatom_energy(pose.res(), pose.res_variant(), pose.full_coord(), pose.total_residue_for_scoring(), use_subset_energy);
	int total_residue = pose.total_residue();
	//// yab: misc removal

	sum_fullatom_energies(total_residue,tmp_faatr,tmp_farep,tmp_fasolv,tmp_faref,
	 tmp_protonation,tmp_fadun,tmp_faprob1b,tmp_fapair,tmp_faplane,tmp_fah2o,tmp_fah2ohb,
	 tmp_fah2osol,tmp_srbbhb,tmp_lrbbhb,tmp_schb,tmp_intrares,tmp_gb_elec,tmp_elec,tmp_cst,tmp_bondangle);

	if (
	 tmp_faatr != fa_atr_score ||
	 tmp_farep != fa_rep_score ||
	 tmp_fasolv != fa_solv_score ||
	 tmp_fapair != fa_pair_score ||
	 tmp_faplane != fa_plane_score ||
	 tmp_fadun != fa_dun_score ||
	 tmp_faprob1b != fa_prob1b_score ||
	 tmp_fah2o != fa_h2o_score ||
	 tmp_fah2ohb != fa_h2o_hb_score ||
	 tmp_fah2osol != fa_h2o_solv_score ||
	 tmp_faref != fa_ref_score ||
	 tmp_protonation != fa_pH_score ||
	 tmp_gb_elec != fa_gb_elec_score ||
	 tmp_srbbhb != hb_srbb_score ||
	 tmp_lrbbhb != hb_lrbb_score ||
	 tmp_schb != hb_sc_score ||
	 tmp_intrares != fa_intrares_score ||
   tmp_elec != fa_elec_score ||
	 tmp_cst != pc_score ||
	 tmp_bondangle != bond_angle_score) {
		std::cout << "atr" << SS( tmp_faatr - fa_atr_score ) << std::endl;
		std::cout << "rep" << SS( tmp_farep - fa_rep_score ) << std::endl;
		std::cout << "solv" << SS( tmp_fasolv - fa_solv_score ) << std::endl;
		std::cout << "pair" << SS( tmp_fapair - fa_pair_score ) << std::endl;
		std::cout << "plane" << SS( tmp_faplane - fa_plane_score ) << std::endl;
		std::cout << "dun" << SS( tmp_fadun - fa_dun_score ) << std::endl;
		std::cout << "prob1b" << SS( tmp_faprob1b - fa_prob1b_score ) << std::endl;
		std::cout << "ref" << SS( tmp_faref - fa_ref_score ) << std::endl;
		std::cout << "h2o" << SS( tmp_fah2o - fa_h2o_score ) << std::endl;
		std::cout << "h2ohb" << SS( tmp_fah2ohb - fa_h2o_hb_score ) << std::endl;
		std::cout << "h2osolv" << SS( tmp_fah2osol - fa_h2o_solv_score ) << std::endl;
		std::cout << "hbsrbb" << SS( tmp_srbbhb - hb_srbb_score ) << std::endl;
		std::cout << "hblrbb" << SS( tmp_lrbbhb - hb_lrbb_score ) << std::endl;
		std::cout << "hbside" << SS( tmp_schb - hb_sc_score ) << std::endl;
		std::cout << "intrares" << SS( tmp_intrares - fa_intrares_score) << std::endl;
    std::cout << "elec" << SS( tmp_elec - fa_elec_score ) << std::endl;
		std::cout << "cst" << SS( tmp_cst - pc_score ) << std::endl;
		std::cout << "bondangle" << SS( tmp_bondangle - bond_angle_score ) << std::endl;
		if ( get_pH_packing_flag() ) {
			std::cout << "fapH" << SS( tmp_protonation - fa_pH_score ) << std::endl;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin compare_fullatom_energy
///
/// @brief
///car compare best and current energy arrays following a full energy calculation
///
/// @detailed
///car report residue pairs that differ (in sol,atr,rep) that are not
///car expected to based on the values of frag_begin and frag_size
///car for such residue pairs, write out differences in atom  distances
///car in the fullcoord and best_fullcoord arrays
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compare_fullatom_energy()
{
	using namespace aaproperties_pack;
	using namespace fullatom_energies;
	using namespace misc;
	using namespace param;

	float delta,dis1,dis2;

	FArray2DB_bool const & pair_moved_global( retrieve_pair_moved() );

	//car make local copy
	FArray2D_bool const  pair_moved( pair_moved_global );

	debug_move_maps(true);
	fullatom_energy(res, res_variant, full_coord, total_residue, false);

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = i+1; j <= total_residue; ++j ) {
			delta = atr_pair(i,j) - best_atr_pair(i,j);
			if ( ( std::abs(delta) > 0.01 ) && ( ! pair_moved(i,j) ) ) {
				std::cout << "ATR" << ' ' << i << ' ' << j << SS( delta ) << std::endl;
				for ( int ii = 1, iie = natoms( res(i), res_variant(i) ),
				 jje = natoms( res(j), res_variant(j) ); ii <= iie; ++ii ) {
					for ( int jj = 1; jj <= jje; ++jj ) {
						distance_bk( full_coord(1,ii,i), full_coord(1,jj,j), dis1 );
						distance_bk( best_full_coord(1,ii,i), best_full_coord(1,jj,j), dis2 );
						if ( std::abs(dis1-dis2) > 0.0 )
							std::cout << "res/at res/at dis best" << SS( i ) << SS( ii )
										 << SS( j ) << SS( jj ) << SS( dis1 ) << SS( dis2 ) << std::endl;
					}
				}
			}
		}
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = i+1; j <= total_residue; ++j ) {
			delta = rep_pair(i,j) - best_rep_pair(i,j);
			if ( ( std::abs(delta) > 0.01 ) && ( ! pair_moved(i,j) ) ) {
				std::cout << "REP" << ' ' << i << ' ' << j << SS( delta ) << std::endl;
				for ( int ii = 1, iie = natoms( res(i), res_variant(i) ),
				 jje = natoms( res(j),res_variant(j) ); ii <= iie; ++ii ) {
					for ( int jj = 1; jj <= jje; ++jj ) {
						distance_bk( full_coord(1,ii,i), full_coord(1,jj,j), dis1 );
						distance_bk( best_full_coord(1,ii,i), best_full_coord(1,jj,j), dis2 );
						if ( std::abs(dis1-dis2) > 0.0 )
							std::cout << "res/at res/at dis best" << SS( i ) << SS( ii )
										 << SS( j ) << SS( jj ) << SS( dis1 ) << SS( dis2 ) << std::endl;
					}
				}
			}
		}
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = i+1; j <= total_residue; ++j ) {
			delta = sol_pair(i,j) - best_sol_pair(i,j);
			if ( ( std::abs(delta) > 0.01 ) && ( ! pair_moved(i,j) ) ) {
				std::cout << "SOL" << ' ' << i << ' ' << j << SS( delta ) << std::endl;
				for ( int ii = 1, iie = natoms( res(i), res_variant(i) ),
				 jje = natoms( res(j), res_variant(j) ); ii <= iie; ++ii ) {
					for ( int jj = 1; jj <= jje; ++jj ) {
						distance_bk( full_coord(1,ii,i), full_coord(1,jj,j), dis1 );
						distance_bk( best_full_coord(1,ii,i), best_full_coord(1,jj,j), dis2 );
						if ( std::abs(dis1-dis2) > 0.0 )
							std::cout << "res/at res/at dis best" << SS( i ) << SS( ii )
										 << SS( j ) << SS( jj ) << SS( dis1 ) << SS( dis2 ) << std::endl;
					}
				}
			}
		}
	}

	debug_move_maps(false);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//  END INTERFACE
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------



////////////////////////////////////////////////////////////////////////////////
/// @begin zero_atom_Earrays
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
zero_atom_Earrays()
{
	using namespace fullatom_energies;
	using namespace param;

	solE_atm = 0.0;
	atrE_atm = 0.0;
	repE_atm = 0.0;
	n_atm = 0;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin fill_Earrays
///
/// @brief
///
/// @detailed
///
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  solE - [in/out]? -
/// @param  atrE - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
fill_Earrays(
	int res1,
  int res2,
  float solE,
	float atrE,
	float repE,
  float elecE
)
{
	using namespace fullatom_energies;
	using namespace param_pack;

	// Linear array indices: Assumes arrays have same dimensions
	int const l12 = atr_pair.index(res1,res2);
	int const l21 = atr_pair.index(res2,res1);


	float temp = 0.5f * pack_wts.Watr() * atrE;
	atrenergy(res1) += temp;
	atrenergy(res2) += temp;
	temp += temp;
	atr_pair[ l12 ] += temp; // atr_pair(res1,res2)
	atr_pair[ l21 ] += temp; // atr_pair(res2,res1)

	temp = 0.5f * pack_wts.Wrep() * repE;
	repenergy(res1) += temp;
	repenergy(res2) += temp;
	temp += temp;
	rep_pair[ l12 ] += temp; // rep_pair(res1,res2)
	rep_pair[ l21 ] += temp; // rep_pair(res2,res1)

	temp = 0.5f * pack_wts.Wsol() * solE;
	solenergy(res1) += temp;
	solenergy(res2) += temp;
	temp += temp;
	sol_pair[ l12 ] += temp; // sol_pair(res1,res2)
	sol_pair[ l21 ] += temp; // sol_pair(res2,res1)

  temp = 0.5f * pack_wts.Wele() * elecE;
  elecenergy(res1) += temp;
  elecenergy(res2) += temp;
  temp += temp;
  elec_pair[ l12 ] += temp;
  elec_pair[ l21 ] += temp;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fill_repE_array
///
/// @brief
///
/// @detailed
///
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
fill_repE_array(
	int res1,
	int res2,
	float repE,
  float elecE
)
{
	using namespace fullatom_energies;
	using namespace param_pack;
  using namespace param_aa;

	float Wrep_repE = pack_wts.Wrep() * repE;
	rep_pair(res1,res2) += Wrep_repE;
	rep_pair(res2,res1) += Wrep_repE;
	Wrep_repE *= 0.5f;
	repenergy(res1) += Wrep_repE;
	repenergy(res2) += Wrep_repE;

  float W_elecE = pack_wts.Wele() * elecE;
  float W_elecE2 = W_elecE * 0.5f;
  elec_pair(res1,res2) += W_elecE;
  elec_pair(res2,res1) += W_elecE;
  elecenergy(res1) += W_elecE2;
  elecenergy(res2) += W_elecE2;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fill_solE_array
///
/// @brief
///
/// @detailed
///
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fill_solE_array(
	int res1,
	int res2,
	float solE
)
{
	using namespace fullatom_energies;
	using namespace param_pack;

	float Wsol_solE = pack_wts.Wsol() * solE;
	sol_pair(res1,res2) += Wsol_solE;
	sol_pair(res2,res1) += Wsol_solE;
	Wsol_solE *= 0.5f;
	solenergy(res1) += Wsol_solE;
	solenergy(res2) += Wsol_solE;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fill_atom_Earrays
///
/// @brief
///
/// @detailed
///
/// @param  type1 - [in/out]? -
/// @param  type2 - [in/out]? -
/// @param  sE1 - [in/out]? -
/// @param  sE2 - [in/out]? -
/// @param  atrE - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
fill_atom_Earrays(
	int type1,
	int type2,
	float sE1,
	float sE2,
	float atrE,
	float repE
)
{
	using namespace fullatom_energies;
	using namespace param_pack;

	if ( geometric_sol::geometric_sol_flag ) {
	  std::cout << "fill_atom_Earrays is not supported with geometric_solvation" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	solE_atm(type1) += pack_wts.Wsol() * sE1;
	solE_atm(type2) += pack_wts.Wsol() * sE2;
	float temp = 0.5f * pack_wts.Watr() * atrE;
	atrE_atm(type1) += temp;
	atrE_atm(type2) += temp;
	temp = 0.5f * pack_wts.Wrep() * repE;
	repE_atm(type1) += temp;
	repE_atm(type2) += temp;
	++n_atm(type1);
	++n_atm(type2);
}


// returns 1 if charged, 2 if aromatic phe.
// Helpful for detecting pairs of aro-aro or aro-charged atoms. If sum of is_ar_charged for the two atoms is >=3, then the pair is aro-aro or aro-charged
// if sum is >= 2 then the two atoms are aro-aro, aro-charged, or charged-charged
inline int
is_aro_charged(
int aa,
int attype,
int atnum
)
{
  using namespace param_aa;

  if( ( aa == aa_phe || aa == aa_trp || aa == aa_tyr || param_aa::is_ligand(aa) ) && ( attype == 6 || attype == 7 || attype == 24 ) ) // aroC, Ntrp, aroH
    return 2;
  if( ( aa == aa_asp || param_aa::is_ligand(aa) ) && ( attype == 4 || attype == 2 || attype == 15 || atnum == 11 || atnum == 12 ) || // CB, CG, HB, OOC
      ( aa == aa_glu || param_aa::is_ligand(aa) ) && ( attype == 4 || attype == 2 || attype == 15 || atnum == 14 || atnum == 15 ) || // CG, CD, HG, OOC
      ( aa == aa_arg || param_aa::is_ligand(aa)) && ( attype == 22 || attype == 6 || attype == 11 || atnum == 7 || atnum == 23 || atnum == 24 ) || // CD,HD,NE,NH,CZ,HH
     ( aa == aa_lys || param_aa::is_ligand(aa) ) && ( attype == 22 || attype == 10 || atnum == 8 || atnum == 21 || atnum == 22 ) ) // CE,NZ,HE,HZ
    return 1;

	//rhiju RNA. Yay! Magic numbers!
  if( is_RNA( aa )) {
		//		if ( ( ( aa == na_ura ) && ( atnum >= 13 && atnum <= 20) )  || // uracil base heavy atoms
		//				 ( ( aa == na_ura ) && ( atnum >= 28) )  || // uracil base hydrogens
		//				 ( ( aa == na_rad ) && ( atnum >= 13 && atnum <= 23) )  || // adenosine base heavy atoms
		//				 ( ( aa == na_rad ) && ( atnum >= 30) )  || // adenosine base hydrogens
		//				 ( ( aa == na_rcy ) && ( atnum >= 13 && atnum <= 20) )  || // cytosine base heavy atoms
		//				 ( ( aa == na_rcy ) && ( atnum >= 28) )  || // cytosine base hydrogens
		//				 ( ( aa == na_rgu ) && ( atnum >= 13 && atnum <= 23) )  || // guanosine base heavy atoms
		//				 ( ( aa == na_rgu ) && ( atnum >= 31) ) ) // guanosine base hydrogens
		//			return 2;
		if (  atnum >= 2 && atnum <=3 ) return 1; //O1P and O2P?
	}

  return 0;
}

/////////////////////////////////////////////////////
// SJF Check the types of atoms in a given pair of atoms
// There are three preset categorizations below for different
// interaction types, which can be changed by commenting out
// two out of the three return statements.
//////////////////////
bool
aro_charged_pair(
int aa1,
int attype1,
int atnum1,
int aa2,
int attype2,
int atnum2
)
{
  if ( get_simple_elec() ) {
    int res1 = is_aro_charged( aa1, attype1, atnum1 );
    int res2 = is_aro_charged( aa2, attype2, atnum2 );

    return( res1 >= 1 && res2 >= 1 ); // chrg-chrg, chrg-aro, aro-aro
//  return ((res1 == 2 && res2 >= 1) || (res1 >= 1 && res2 == 2)); // chrg-aro, aro-aro
//  return ( res1 == 2 && res2 == 2 ); // aro-aro
  }
  else return ( false );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_energy
///
/// @brief
///bk what is the fullatom score for a protein
///
/// @detailed
///
/// @param  res_inout - [in/out]? - amino acid at each residue position
/// @param  aav_inout - [in/out]? - amino acid variant at each residue position
/// @param  xyz_inout - [in/out]? - positions of atoms (full_coord,atom number,residue number)
/// @param  nres_inout - [in/out]? - number of residues
/// @param  subset - [in/out]? - only calculate energy between residues that have
///       changed relative positions
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
fullatom_energy(
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord,
	int total_residue,
	bool subset
)
{
	using namespace aaproperties_pack;
	using namespace fullatom_energies;
	using namespace design;
	using namespace gb_elec;
	using namespace ligand;
	using namespace enzyme;
	using namespace nblist;
	using namespace param;
	using namespace param_pack;
	using namespace pdbstatistics_pack;
	using namespace pH_ns;
	using namespace runlevel_ns;
	using namespace scale_res_energy_ns;
	using namespace template_pack;
	using namespace water;
	using namespace enable_ligaa_ns;

	initialize_fullatom();
	if ( !get_fullatom_flag() ) return;

//bk local variables
	int aa1,aa2,pos2,type1,aav1,aav2;
	FArray1D_float rperc( total_residue );
	float solE,atrE,repE,elecE,pairE,virE,h2oE,Paa_ppE,sE1,sE2,d2,plane_totalE;
	float hbenergy_scmc,hbenergy_sc,hbE;
	float cstE;
	float intra_atrE,intra_repE;
	float cp_weight;
	float energy_add;
  float chrg1, chrg2; // SJF the pair's atomic charges
	//Objexx: static arrays for speed
	static FArray1D_float tmp3( MAX_CHI );
	float tmp1, tmp2;
	float tmp4, tmp5; // dummy variables for get_Paa_ppE
//mj local ligand variables
	static FArray2D_float f1( 3, MAX_ATOM() );
	static FArray2D_float f2( 3, MAX_ATOM() );
//KMa local variables //AA ligand energies
	float latr, lrep, lsol, lgb, lh2o,lhb, lpair;

	PROF_START( prof::FULLATOM_ENERGY );
	fullatom_energy_debug::in_fullatom_energy = true;

//bk update & retrieve pair_moved, which indicates which pairs of atoms have
//bk moved relative position since the last accept - this is based on
//bk frag_begin and frag_end, not actual distances
	FArray2DB_bool const & pair_moved( retrieve_pair_moved() );
	FArray1DB_bool const & res_moved( retrieve_res_moved() );

	// chi angles
	ang_from_pdb_wsc(total_residue,res,res_variant,full_coord,template_pack::phi,template_pack::psi,chiarray,rotarray);

//bk determine how many neighbors each residue has, eventually get from rosetta
	PROF_START( prof::F_E_NB_INFO );
	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);
	PROF_STOP( prof::F_E_NB_INFO );

//lin determine how many neighbors each hetero water has
	if ( use_hetero_h2o ) count_native_water_neighbors(res,total_residue,full_coord,N_h2oatom,
	 h2o_position,hetero_water_neighbors);

//lin determine how many neighbors each explicit water has
	if ( explicit_h2o ) count_explicit_water_neighbors(res,res_variant,total_residue,full_coord,
	 explicit_water_neighbors);

//lin determine how many waters are close to each heavy atom
//  make_close_water_info(res,res_variant,total_residue,full_coord,nh2o_close_to_atm);

//bk retrieve rotamer preferences
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		get_rotamer_probability(res(seqpos),res_variant(seqpos),template_pack::phi(seqpos),
		 template_pack::psi(seqpos),seqpos,total_residue,chiarray(1,seqpos),rotarray(1,seqpos),
		 rperc(seqpos),tmp1,tmp2,tmp3);
		if ( fullatom_energy_debug::debug )
			std::cout << "rotprob: " << seqpos << ' ' << res(seqpos) << ' ' <<
				res_variant(seqpos) << ' ' << template_pack::phi(seqpos) << ' ' <<
				template_pack::psi(seqpos) << ' ' << rperc(seqpos) << ' ' <<
				chiarray(1,seqpos) << ' ' <<
				chiarray(2,seqpos) << ' ' <<
				chiarray(3,seqpos) << ' ' <<
				chiarray(4,seqpos) << ' ' <<
				rotarray(1,seqpos) << ' ' <<
				rotarray(2,seqpos) << ' ' <<
				rotarray(3,seqpos) << ' ' <<
				rotarray(4,seqpos) << std::endl;
	}
//bk determine the coordinates for the action center of each sidechain
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		put_wcentroid(full_coord(1,1,seqpos),actcoord(1,seqpos),res(seqpos));
	}

//mj reset satisfied h-bond donors and acceptors
	if ( get_ligand_flag() ){
		for( std::vector< Ligand * >::iterator c_ligand=ligand::ligand_ptr_vector.begin();
			c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++){

			reset_satisfied_ha_hbond(**c_ligand);

		}
	}

//bk loop through sidechains, calculate energies. Do not recalculate
//bk energies between two residues that did not move or one body energies
//bk for residues that did not move

 	// Zero-initialize arrays
	atrenergy = 0.0;
	repenergy = 0.0;
	solenergy = 0.0;
	hbenergy = 0.0;
	hbenergy_sc_sc = 0.0; // pb for symm
	hbenergy_sc_bb = 0.0; // pb for symm
	hbenergy_sr_bb = 0.0; // pb for symm
	hbenergy_lr_bb = 0.0; // pb for symm
	pair_energy = 0.0;
	plane_energy = 0.0;
	probenergy = 0.0;
	dunenergy = 0.0;
	unfenergy = 0.0;
	protonation_energy = 0.0;
	h2oenergy = 0.0;
	h2ohbenergy = 0.0;
	wsolenergy = 0.0;
	intraresenergy = 0.0;
	gb_elecenergy = 0.0;
  elecenergy = 0.0;
	cstenergy = 0.0;
	//lin ligaa array
	if( cst_set_ns::debug_output ) {
		ligaa_atrE = 0.0;
		ligaa_repE = 0.0;
		ligaa_solE = 0.0;
		ligaa_hbE = 0.0;
	}

	//KMa aa-ligand(surface) excluding aa-aa energies -KM- 2006/1
			latrenergy		= 0.0;
			lrepenergy		= 0.0;
			lsolenergy		= 0.0;
			lgb_elecenergy	= 0.0;
			lh2oenergy		= 0.0;
			lhbenergy		= 0.0;
			lpair_energy	= 0.0;
			lresenergy		= 0.0;

//bk calculate energy terms that only depend on phi,psi,rot and aa
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		aa1 = res(seqpos);
		aav1 = res_variant(seqpos);
		float const & xyz_11s( full_coord(1,1,seqpos) );
		if ( res_moved(seqpos) ) {
			get_Paa_ppE(template_pack::phi(seqpos),template_pack::psi(seqpos),aa1,Paa_ppE,tmp4,tmp5);
			probenergy(seqpos) = pack_wts.Wone() * Paa_ppE;
			dunenergy(seqpos) = pack_wts.Wdun() * ( -std::log(rperc(seqpos)) );
			unfenergy(seqpos) = pack_wts.Waa(aa1);
			if ( get_pH_packing_flag() ) {
			protonation_energy(seqpos) = dGprotonation_table(aa1,aav1);
			}
			h2oenergy(seqpos) = pack_wts.Wh2o_intra() * nh2o(aa1,aav1);
			get_intra_resE(aa1,aav1,xyz_11s,intra_atrE,intra_repE);
			intraresenergy(seqpos) = pack_wts.Wintra() * intra_repE;
			bondangleenergy(seqpos) = scorefxns::bond_angle_weight ?
			                          get_bond_angleE(aa1, aav1, xyz_11s) : 0;
		} else if ( !subset ) {
			probenergy(seqpos) = best_probenergy(seqpos);
			dunenergy(seqpos) = best_dunenergy(seqpos);
			unfenergy(seqpos) = best_unfenergy(seqpos);
			protonation_energy(seqpos) = best_protonation_energy(seqpos);
			h2oenergy(seqpos) = pack_wts.Wh2o_intra() * nh2o(aa1,aav1);
			intraresenergy(seqpos) = best_intraresenergy(seqpos);
			bondangleenergy(seqpos) = best_bondangleenergy(seqpos);
		}

//lin water-free energy for attached atoms
//      under construction now!!!
//      get_h2o_dGfreeE(aa1,aav1,seqpos,wsolenergy(seqpos));

//lin calculate the lj repulsive energy between hetero water and protein atoms
		if ( use_hetero_h2o ) {
			get_sc_hetero_h2oE(aa1,aav1,full_coord(1,1,seqpos),h2oE);
			h2oenergy(seqpos) += pack_wts.Wh2o_lj() * h2oE;
		}

	}


//bk Zero-initialize energies between pairs of residues
	atr_pair = 0.0;
	sol_pair = 0.0;
	rep_pair = 0.0;
	pair_pair = 0.0;
	plane_plane = 0.0;
	hbond_pair = 0.0;
	h2o_pair = 0.0;
	h2o_hb_pair = 0.0;
	cst_pair = 0.0;
	gb_elec_pair = 0.0;
  elec_pair = 0.0;

//bk retrieve saved energies
//bk saved energies are not used for the pair term and the hydrogen bonding
//bk term because they are not completely pairwise additive.  Sidechain
//bk backbone hbonds depend on the absence of a backbone-backbone hbond and
//bk the pairterm depends on the number of neighbors
	PROF_START( prof::F_E_FA_CACHE );

	if ( !subset ) {
		bool const copy_gb( gen_born && !get_update_born_radii() );
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
			for ( int res1 = 1, l = best_atr_pair.index(res1,res2),
							l_pm = pair_moved.index(res1,res2); res1 <= res2;
			 ++res1, ++l, ++l_pm ) {
				// Assumes arrays have same dimensions: [l] == (res1,res2)
				// actually - pair_moved might have different dimensions
				if ( !pair_moved[ l_pm ] ) {
					energy_add = 0.5f * best_atr_pair[ l ];
					atrenergy(res1) += energy_add;
					atrenergy(res2) += energy_add;

					energy_add = 0.5f * best_rep_pair[ l ];
					repenergy(res1) += energy_add;
					repenergy(res2) += energy_add;

					energy_add = 0.5f * best_sol_pair[ l ];
					solenergy(res1) += energy_add;
					solenergy(res2) += energy_add;

// 					energy_add = 0.5f * best_h2o_pair[ l ];
// 					h2oenergy(res1) += energy_add;
// 					h2oenergy(res2) += energy_add;

// 					energy_add = 0.5f * best_h2o_hb_pair[ l ];
// 					h2ohbenergy(res1) += energy_add;
// 					h2ohbenergy(res2) += energy_add;

					energy_add = 0.5f * best_cst_pair[ l ];
					cstenergy(res1) += energy_add;
					cstenergy(res2) += energy_add;

          energy_add = 0.5f * best_elec_pair[ l ];
          elecenergy(res1) += energy_add;
          elecenergy(res2) += energy_add;

					if ( copy_gb ) {
						energy_add = 0.5f * best_gb_elec_pair[ l ];
						gb_elecenergy(res1) += energy_add;
						gb_elecenergy(res2) += energy_add;
					}
				}
			} // res1
		} // res2

		for ( int res1 = 1; res1 <= total_residue; ++res1 ) { // Lower triangle & res1=res2
			for ( int res2 = res1, l = atr_pair.index(res2,res1),
							l_pm = pair_moved.index(res2,res1); res2 <= total_residue;
			 ++res2, ++l, ++l_pm ) {
				// Assumes arrays have same dimensions: [l] == (res2,res1)
				if ( !pair_moved[ l_pm ] ) {
					atr_pair[ l ] = best_atr_pair[ l ];
					rep_pair[ l ] = best_rep_pair[ l ];
					sol_pair[ l ] = best_sol_pair[ l ];
// 					h2o_pair[ l ] = best_h2o_pair[ l ];
// 					h2o_hb_pair[ l ] = best_h2o_hb_pair[ l ];
					cst_pair[ l ] = best_cst_pair[ l ];

          elec_pair[ l ] = best_elec_pair[ l ];
					if ( copy_gb ) gb_elec_pair[ l ] = best_gb_elec_pair[ l ];
				}
			} // res2
		} // res1

		// Do the upper triangles separately for efficient col-major access
		for ( int res2 = 2; res2 <= total_residue; ++res2 ) { // Upper triangle
			for ( int res1 = 1, l = atr_pair.index(res1,res2),
							l_pm = pair_moved.index(res1,res2); res1 < res2;
			 ++res1, ++l, ++l_pm ) {
				// Assumes arrays have same dimensions: [l] == (res1,res2)
				if ( !pair_moved[ l_pm ] ) {
					atr_pair[ l ] = best_atr_pair[ l ];
					rep_pair[ l ] = best_rep_pair[ l ];
					sol_pair[ l ] = best_sol_pair[ l ];
// 					h2o_pair[ l ] = best_h2o_pair[ l ];
// 					h2o_hb_pair[ l ] = best_h2o_hb_pair[ l ];
					cst_pair[ l ] = best_cst_pair[ l ];
          elec_pair[ l ] = best_elec_pair[ l ];
					if ( copy_gb ) gb_elec_pair[ l ] = best_gb_elec_pair[ l ];
				}
			} // res1
		} // res2
	}
	PROF_STOP( prof::F_E_FA_CACHE );

//bk zero atom_Earrays if they are being calculated
	if ( natrot ) {
		zero_atom_Earrays();
	}

	// Set score_variant boolean flags outside loops for efficiency
	bool const loc_lj_variant = ( get_score_variant() == "loc_lj" );
	bool const lk_sol_variant = ( get_score_variant() == "lk_sol" );

	PROF_START( prof::F_E_PAIR_ENERGY );

//bk calculate atom-atom energies
	if ( !get_use_nblist() ) {     // not using atom-atom nb list

		int const xyz_size1 = full_coord.size1();
		for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
			aa1 = res(res1);
			aav1 = res_variant(res1);
			int const atm1e = nheavyatoms(aa1,aav1);
			int const l1s = fullatom_type.index(1,aa1,aav1);
			float const & xyz_111( full_coord(1,1,res1) );
			for ( int res2 = res1; res2 <= total_residue; ++res2 ) {
				if ( !pair_moved(res1,res2) ) goto L20;
				if ( !neighborlist(res1,res2) ) goto L20;
				{ // Scope
				aa2 = res(res2);
				aav2 = res_variant(res2);
				int const atm2e = nheavyatoms(aa2,aav2);
				int const l2s = fullatom_type.index(1,aa2,aav2);
				int const lfull_coords = full_coord.index(1,1,res2);
				float const & xyz_112( full_coord(1,1,res2) );
				bool const res_diff_gt4 = ( res2 - res1 > 4 ); // res2 > res1 here
				bool const scale_energy_matrix_12 = scale_energy_matrix(res1,res2);
				bool const scale_rep_energy_matrix_12 = scale_rep_energy_matrix(res1,res2);
				for ( int atm1 = 1, l1 = l1s; atm1 <= atm1e; ++atm1, ++l1 ) {

					type1 = fullatom_type[ l1 ]; // fullatom_type(atm1,aa1,aav1)
          chrg1 = atomic_charge[ l1 ];

//cmd This is for turning Rosetta-built 5' phosphates into ignored virtual
// atoms. Causes a 15% hit in performance time.
//				type1 = dna_interface ? sub_missing_atomtypes(atm1,aa1,aav1,
//          res1) : fullatom_type[ l1 ];

					int l11 = full_coord.index(1,atm1,res1);
					float const & xyz_1a11( full_coord[ l11 ] ); // full_coord(1,atm1,res1)
					float const xyz_2a11 = full_coord[ ++l11 ]; // full_coord(2,atm1,res1)
					float const xyz_3a11 = full_coord[ ++l11 ]; // full_coord(3,atm1,res1)
					for ( int atm2 = 1, l2 = l2s, lfull_coord = lfull_coords;
					 atm2 <= atm2e; ++atm2, ++l2, lfull_coord+=xyz_size1 ) {
						int type2 = fullatom_type[ l2 ]; // fullatom_type(atm2,aa2,aav2)
						if ( count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,
						 cp_weight) ) {
              chrg2 = atomic_charge[ l2 ];
//cmd This is for turning Rosetta-built 5' phosphates into ignored virtual
// atoms. Causes a 15% hit in performance time.
//						type2 = dna_interface ? sub_missing_atomtypes(atm2,aa2,
//              aav2,res2) : fullatom_type[ l2 ];

              bool aro_charged = aro_charged_pair( aa1, type1, atm1, aa2, type2, atm2 );
              pairenergy(xyz_1a11,full_coord[ lfull_coord ],type1,type2,
                     chrg1, chrg2,sE1,sE2,atrE, repE, elecE, d2); // [lfull_coord] == (1,atm2,res2)

              if ( !aro_charged )
                elecE = 0.0;
							//glb if scale_res_energy_flag then SCALE repE,sE1,sE2 based on res1 and res2
//Objexx: Next four lines replaced by if-block below for speed
//							atrE = scale_res_energy(res1,res2,atrE); //glb
//							repE = scale_res_rep_energy(res1,res2,repE); //glb
//							sE1 = scale_res_energy(res1,res2,sE1); //glb
//							sE2 = scale_res_energy(res1,res2,sE2); //glb
							if ( get_scale_res_energy_flag() ) { //Objexx
								if ( scale_energy_matrix_12 ) {
									atrE *= scale_res_energy_weight; //glb
									sE1 *= scale_res_energy_weight; //glb
									sE2 *= scale_res_energy_weight; //glb
								}
								if ( scale_rep_energy_matrix_12 ) {
									repE *= scale_res_rep_energy_weight; //glb
								}
							}

							atrE *= cp_weight;
							repE *= cp_weight;
              elecE *= cp_weight;
							if ( loc_lj_variant ) {
								if ( res_diff_gt4 ) atrE *= 3.0f;
								solE = ( ( sE1 + sE2 ) * cp_weight ) * 0.6f;
							} else if ( lk_sol_variant ) {
								solE = 0.0;
							} else {
								solE = ( sE1 + sE2 ) * cp_weight;
							}
//elecE = 0.0;
							fill_Earrays(res1,res2,solE,atrE,repE,elecE );
							if( cst_set_ns::debug_output ) {
								fill_ligaa_pairEarrays(atm1,aa1,solE,atrE,repE);
								fill_ligaa_pairEarrays(atm2,aa2,solE,atrE,repE);
							}
						} else {
							d2 =
							 square( xyz_1a11 - full_coord[ lfull_coord ] ) +
							 square( xyz_2a11 - full_coord[ lfull_coord+1 ] ) +
							 square( xyz_3a11 - full_coord[ lfull_coord+2 ] );
						}
//bk        evaluate hydrogens attached to carbons
// SJF the electrostatic potential cutoff is at 5.5Ang, so I've changed the cutoff below to 7.5^2, because
// d2 is measured between heavy atoms. This lead to a severe performance hit, so it's only
// evaluated when the simple_elec switch is turned on and the atoms are aro_charged
            bool aro_charged = aro_charged_pair( aa1, type1, atm1, aa2, type2, atm2 );
            if ( d2 < hydrogen_interaction_cutoff  || // SJF
                 ( get_simple_elec() && aro_charged && d2 < hydrogen_interaction_cutoff_long ) ) {
							repE = 0.0;
              elecE = 0.0;
		  				fast_pairenergy_attached_h(res1,res2,atm1,atm2,aa1,aa2,aav1,aav2,
							 xyz_111,xyz_112,repE,elecE);
              if (!aro_charged ) elecE = 0.0;
 							fill_repE_array(res1,res2,( d2 >= hydrogen_interaction_cutoff ? 0.0 : repE ),elecE );

							if( cst_set_ns::debug_output ) {
								fill_ligaa_repEarray(atm1,aa1,repE);
								fill_ligaa_repEarray(atm2,aa2,repE);
							}
						}

//            if ( natrot ) {
//              fill_atom_Earrays(type1,type2,sE1,sE2,atrE,repE);
//            }

					} // atm2
				} // atm1
				} // Scope
L20:;
			} // res2
		} // res1

	} else if ( !score_check_current_pose()
							|| score_get_current_pose().atom_tree() == 0 ) {
    // using atom-atom neighbor list
		for ( int pos1 = 1; pos1 <= total_atoms; ++pos1 ) {
			int atm1 = global_atm_num2res_atm_num_map(pos1);
			if ( atm1 > 0 ) {
				int res1 = global_atm_num2res_num_map(pos1);
				aa1 = res(res1);
				aav1 = res_variant(res1);

				type1 = fullatom_type(atm1,aa1,aav1);
        chrg1 = atomic_charge(atm1,aa1,aav1);

//cmd This is for turning Rosetta-built 5' phosphates into ignored virtual
// atoms. Causes a 15% hit in performance time.
//      type1 = dna_interface ? sub_missing_atomtypes(atm1,aa1,
//        aav1,res1) : fullatom_type(atm1,aa1,aav1);

				float const & xyz_1a11( full_coord(1,atm1,res1) );

				for ( int j = nb_start(pos1), lnb = nb.index(j,pos1), je = nb_end(pos1);
				 j <= je; ++j, ++lnb ) {
					pos2 = nb[ lnb ]; // nb(j,pos1)
					int atm2 = global_atm_num2res_atm_num_map(pos2);
					if ( atm2 <= 0 ) goto L50;
//bk do not calculate energies for atoms that don't change relative position
					{ // Scope
					int res2 = global_atm_num2res_num_map(pos2);
					if ( !pair_moved(res1,res2) ) goto L50;
					aa2 = res(res2);
					aav2 = res_variant(res2);
					if ( count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,
					 cp_weight) ) {

						int type2 = fullatom_type(atm2,aa2,aav2);
            chrg2 = atomic_charge(atm2,aa2,aav2);
//cmd This is for turning Rosetta-built 5' phosphates into ignored virtual
// atoms. Causes a 15% hit in performance time.
//          type2 = dna_interface ? sub_missing_atomtypes(atm2,aa2,
//            aav2,res2) : fullatom_type(atm2,aa2,aav2);


            bool aro_charged = aro_charged_pair( aa1, type1, atm1, aa2, type2, atm2 );
            pairenergy(xyz_1a11,full_coord(1,atm2,res2), type1, type2, chrg1, chrg2,sE1,sE2,
             atrE,repE,elecE, d2);
           if ( !aro_charged )
             elecE = 0.0;


						//glb if scale_res_energy_flag then SCALE repE,sE1,sE2 based on res1 and res2
//Objexx: Next four lines replaced by if-block below for speed
//						atrE = scale_res_energy(res1,res2,atrE); //glb
//						repE = scale_res_rep_energy(res1,res2,repE); //glb
//						sE1 = scale_res_energy(res1,res2,sE1); //glb
//						sE2 = scale_res_energy(res1,res2,sE2); //glb
						if ( get_scale_res_energy_flag() ) { //Objexx
							if ( scale_energy_matrix(res1,res2) ) {
								atrE *= scale_res_energy_weight; //glb
								sE1 *= scale_res_energy_weight; //glb
								sE2 *= scale_res_energy_weight; //glb
							}
							if ( scale_rep_energy_matrix(res1,res2) ) {
								repE *= scale_res_rep_energy_weight; //glb
							}
						}

						atrE *= cp_weight;
						repE *= cp_weight;
            elecE *= cp_weight;
						if ( loc_lj_variant ) {
							if ( std::abs( res1 - res2 ) > 4 ) atrE *= 3.0f;  // JSS magic numbers are evil
							solE = ( ( sE1 + sE2 ) * cp_weight ) * 0.6f;
						} else if ( lk_sol_variant ) {
							solE = 0.0;
						} else {
							solE = ( sE1 + sE2 ) * cp_weight;
						}
//elecE = 0.0;
						fill_Earrays(res1, res2,solE,atrE,repE,elecE );
					}
					} // Scope
L50:;
				} // loop through neighbors for each entry in the atom list
			}
		} // loop through atom list
	} else {
		// if scoring a pose and the current pose has an atom_tree
		// use the new Min_map nblist
		nblist_atompair_energy( res, res_variant, full_coord );

	} // use nb_list or not

	PROF_STOP( prof::F_E_PAIR_ENERGY );

// compute constraints energies
	PROF_START( prof::CST_ENERGY );
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) {
		for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
			aa1 = res(res1);
			aav1 = res_variant(res1);
			for ( int res2 = res1; res2 <= total_residue; ++res2 ) {
				if ( pair_moved(res1,res2) ) {
					aa2 = res(res2);
					aav2 = res_variant(res2);
					cstE = classical_constraints::BOUNDARY::get_res_res_cstE(res1,res2,aa1,aav1,aa2,aav2,full_coord(1,1,res1),full_coord(1,1,res2));
					fill_cstE_array(res1,res2,cstE);
				}
			}
		}
	}
	PROF_STOP( prof::CST_ENERGY );

#ifdef GL_GRAPHICS
	gl_graphics_reset_bonds( pair_moved, get_update_born_radii());
#endif

//bk compute hydrogen bonding energies
	PROF_START( prof::F_E_HBOND );
	fill_hbond_arrays(true, res,res_variant,full_coord,total_residue,neighborlist,neighbors,
                    hbonds::hbderiv_NONE,hbenergy_srbb,hbenergy_lrbb,hbenergy_scmc,hbenergy_sc);
	PROF_STOP( prof::F_E_HBOND );

	// jk geometric solvation term
	if ( geometric_sol::geometric_sol_flag ) {
		fill_geo_sol_arrays(true,res,res_variant,full_coord,total_residue,neighborlist);
	}

// ****************Start GB electrostatics******************************
// jjh

	if ( gen_born ) {
		PROF_START( prof::F_E_GB );
		bool const update_radii( get_update_born_radii() );

		if ( update_radii ) {
			PROF_START( prof::F_E_GB_RADII );
			gb_get_all_born_radii();
			PROF_STOP( prof::F_E_GB_RADII );
		}

		PROF_START( prof::F_E_GB_ENERGY );

		for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
			aa1 = res(res1);
			aav1 = res_variant(res1);
			for ( int res2 = res1; res2 <= total_residue; ++res2 ) {

				if ( !update_radii && !pair_moved( res1, res2 ) ) {
					continue;
				}

				aa2 = res(res2);
				aav2 = res_variant(res2);

				float const tempgbE = gb_get_res_res_elecE(
													res1, aa1, aav1,
													full_coord(1,1,res1),
													born_radius(1,res1),
													res2, aa2, aav2,
													full_coord(1,1,res2),
													born_radius(1,res2)
													);
				gb_elecenergy(res1) += 0.5 * pack_wts.Wgb_elec() * tempgbE;
				gb_elecenergy(res2) += 0.5 * pack_wts.Wgb_elec() * tempgbE;
				gb_elec_pair(res1, res2) = pack_wts.Wgb_elec() * tempgbE;
				gb_elec_pair(res2, res1) = pack_wts.Wgb_elec() * tempgbE;
			}
		}
		PROF_STOP( prof::F_E_GB_ENERGY );
		PROF_STOP( prof::F_E_GB );
	}

// ************** End GB electrostatics ******************************


//lin compute water energies
	PROF_START( prof::F_E_H2O );
	fill_h2ototalE_arrays(res,res_variant,full_coord,total_residue,neighborlist);
	PROF_STOP( prof::F_E_H2O );

// #ifdef GL_GRAPHICS
// 	gl_graphics_save_bonds();
// #endif

//bk compute pairterm energies, since pair term is not completely pair additive
//bk (it depends on the number of neighbors a residue has), don't use any saved
//bk pairterm energies.
	PROF_START( prof::F_E_PAIR );
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		aa1 = res(res1);
		aav1 = res_variant(res1);
		float & actcoord_11( actcoord(1,res1) );
		float const & xyz_111( full_coord(1,1,res1) );
		for ( int res2 = res1+1; res2 <= total_residue; ++res2 ) {
			if ( neighborlist(res1,res2) ) {
				aa2 = res(res2);


				if ( std::abs(pack_wts.Wpair()) > 1E-20 ) { // Get pairterm score
					aav2 = res_variant(res2);
					get_pairtermE(aa1,aa2,res1,res2,actcoord_11,actcoord(1,res2),pairE,tmp1);
					float Wpair_pairE = pack_wts.Wpair() * pairE;
					pair_pair(res1,res2) = Wpair_pairE;
					pair_pair(res2,res1) = Wpair_pairE;
					Wpair_pairE *= 0.5f;
					pair_energy(res1) += Wpair_pairE;
					pair_energy(res2) += Wpair_pairE;
				} else {
					pair_pair(res1,res2) = pair_pair(res2,res1) = 0.0;
				}

				if ( std::abs(pack_wts.Wplane_total()) > 1E-20 ) { //km get plane orientation score
					aav2 = res_variant(res2);
					find_plane_orientation(aa1,aav1,aa2,aav2,xyz_111,full_coord(1,1,res2),
					 plane_totalE);
//					std::cout << "from find_plane_orientation: plane_totalE" <<
//					 SS( plane_totalE ) << std::endl;
					float Wplane_totalE = pack_wts.Wplane_total() * plane_totalE;
					plane_plane(res1,res2) = Wplane_totalE;
					plane_plane(res2,res1) = Wplane_totalE;
					Wplane_totalE *= 0.5f;
					plane_energy(res1) += Wplane_totalE;
					plane_energy(res2) += Wplane_totalE;
				} else {
					plane_plane(res1,res2) = plane_plane(res2,res1) = 0.0;
				}

			} // neighborlist
		} // res2
	} // res1
	PROF_STOP( prof::F_E_PAIR );

//bs Disulfide contributions to the 2 body energy and pair-pair term.
	if ( disulfides::BOUNDARY::get_disulf_flag() ) disulfides::BOUNDARY::eval_disulf_fullatom_energy();

//mj add energies from protein - ligand interactions
	if ( get_ligand_flag() ) { // start check full atom ligand flag

//mj recompute interface
		bool new_ligand_interface;
		FArray2D_float het_atom_coord;
    FArray1D_float het_atom_charge;
    FArray1D_int het_atom_type;
	for( std::vector< Ligand * >::iterator c_ligand=ligand::ligand_ptr_vector.begin();
		c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++){
		if ( get_recompute_ligand_interface(**c_ligand) ) {
			new_ligand_interface = true;
			detect_ligand_interface(**c_ligand, false, total_residue, res, full_coord);
		} else {
			new_ligand_interface = false;
		}

//mj set energies to be zero
		(*c_ligand)->lig_repE = 0.0;
		(*c_ligand)->lig_atrE = 0.0;
		(*c_ligand)->lig_solE = 0.0;
		(*c_ligand)->lig_couE = 0.0;
		(*c_ligand)->lig_virE = 0.0;
		(*c_ligand)->lig_h2oE = 0.0;
		(*c_ligand)->lig_hbE = 0.0;
		(*c_ligand)->lig_sumE = 0.0;
		(*c_ligand)->lig_int_repE = 0.0;
		(*c_ligand)->lig_int_atrE = 0.0;
		(*c_ligand)->lig_int_solvE = 0.0;
		(*c_ligand)->lig_int_hbE = 0.0;
		(*c_ligand)->lig_int_coulombE = 0.0;
		(*c_ligand)->lig_int_dihedE = 0.0;
		if( get_ligand_flexible_flag() ) {(*c_ligand)->get_ligand_internal_energy(false);}

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if ( res_moved(seqpos) || new_ligand_interface ) {
				aa1 = res(seqpos);
				aav1 = res_variant(seqpos);
				float const & xyz_11s( full_coord(1,1,seqpos) );

//mj if not complete ligand interface was recomputed and amino acid moved, recompute interface of this aa
				if ( !new_ligand_interface ) detect_ligand_interface_res(seqpos,**c_ligand, false, res, full_coord);

//mj compute virtual energies
				get_sc_ha_virtualE(seqpos,aa1,aav1,xyz_11s,virE,true,true,false,**c_ligand);

//mj compute vdw atr, vdw rep
				get_sc_haE(seqpos,aa1,aav1,xyz_11s,solE,atrE,repE,true,true,**c_ligand);

//mj compute h bond energies
				get_sc_ha_hbondE(seqpos,aa1,aav1,xyz_11s,true,true,**c_ligand,hbE); // no derivatives

//mj compute electrostatics
				if ( !gen_born )
					get_sc_ha_coulombE(seqpos,aa1,aav1,xyz_11s,*c_ligand,pairE,true,true);
				else {
//jjh get the gb interactions between the residues and the ligand
					(*c_ligand)->get_FArray2D_of_coordinates((*c_ligand)->atom_vector, het_atom_coord);
					(*c_ligand)->get_FArray1D_of_charge((*c_ligand)->atom_vector, het_atom_charge);
					(*c_ligand)->get_FArray1D_of_atom_type((*c_ligand)->atom_vector, het_atom_type);
					pairE = gb_get_ligand_elecE(natoms( aa1, aav1 ), MAX_ATOM(),
																			xyz_11s, born_radius( 1, seqpos ),
																			atomic_charge( 1, aa1, aav1 ),
																			fullatom_type( 1, aa1, aav1),
																			(*c_ligand)->atom_vector.size(), HETERO_ATOM_MAX(),
																			het_atom_coord( 1, 1 ),
																			(*c_ligand)->hetero_born_radius( 1 ),
																			het_atom_charge( 1 ),
																			het_atom_type( 1 ),
																			false
																			);
				}
//jjh Get the gbligand-ligand interaction
				if ( gen_born ) {
					FArray2D_float het_atom_coord;
					FArray1D_float het_atom_charge;
					FArray1D_int het_atom_type;
					(*c_ligand)->get_FArray2D_of_coordinates((*c_ligand)->atom_vector,het_atom_coord);
					(*c_ligand)->get_FArray1D_of_charge((*c_ligand)->atom_vector, het_atom_charge);
					(*c_ligand)->get_FArray1D_of_atom_type((*c_ligand)->atom_vector, het_atom_type);
					float const tempgbE = gb_get_ligand_elecE((*c_ligand)->atom_vector.size(), HETERO_ATOM_MAX(),
																										het_atom_coord( 1, 1 ),
																										(*c_ligand)->hetero_born_radius( 1 ),
																										het_atom_charge( 1 ),
																										het_atom_type( 1 ),
																										(*c_ligand)->atom_vector.size(), HETERO_ATOM_MAX(),
																										het_atom_coord( 1, 1 ),
																										(*c_ligand)->hetero_born_radius( 1 ),
																										het_atom_charge( 1 ),
																										het_atom_type( 1 ),
																										true );
					(*c_ligand)->lig_couE += pack_wts.Wlig_cou() * tempgbE;
				}

//mj compute h2o energies
				get_sc_ha_h2oE(seqpos,aa1,aav1,xyz_11s,h2oE,**c_ligand);

//KMa aa-ligand(surface) excluding aa-aa energies  2006/1
			latr  = pack_wts.Wlig_atr() * atrE;
			lrep  = pack_wts.Wlig_rep() * repE;
			lsol  = pack_wts.Wlig_sol() * solE;
			lgb   = pack_wts.Wlig_cou() * pairE;
			lh2o  = pack_wts.Wlig_h2o() * h2oE;
			lhb   = pack_wts.Wlig_hb()  * hbE;
			lpair = pack_wts.Wlig_vir() * virE;

			latrenergy( seqpos ) = latr;
			lrepenergy( seqpos ) = lrep;
			lsolenergy( seqpos ) = lsol;
			lgb_elecenergy( seqpos ) = lgb ;
			lh2oenergy( seqpos ) = lh2o;
			lhbenergy( seqpos ) = lhb;
			lpair_energy( seqpos ) = lpair;
			lresenergy( seqpos ) = latr + lrep + lsol + lgb + lh2o + lhb + lpair;

//mj save energies
			atrenergy	 (seqpos)	+= latr;
			repenergy	 (seqpos)	+= lrep;
			solenergy	 (seqpos)	+= lsol;
			gb_elecenergy(seqpos)	+= lgb;
			h2oenergy	 (seqpos)	+= lh2o;
			hbenergy	 (seqpos)	+= lhb;
			pair_energy  (seqpos)	+= lpair;

//mj sum energies in ligand field
				(*c_ligand)->lig_atrE += latr;
				(*c_ligand)->lig_repE += lrep;
				(*c_ligand)->lig_solE += lsol;
				(*c_ligand)->lig_couE += lgb;
				(*c_ligand)->lig_h2oE += lh2o;
				(*c_ligand)->lig_hbE  += lhb;
				(*c_ligand)->lig_virE += lpair;

//mj print energies
//        if ( runlevel >= verbose ) std::cout << "ligand energies at AA " <<
//         SS( seqpos ) << " : " <<
//         SS( pack_wts.Wlig_sol() * solE ) << ' ' <<
//         SS( pack_wts.Wlig_atr() * atrE ) << ' ' <<
//         SS( pack_wts.Wlig_rep() * repE ) << ' ' <<
//         SS( pack_wts.Wlig_hb() * hbE ) << ' ' <<
//         SS( pack_wts.Wlig_cou() * pairE ) << ' ' <<
//         SS( pack_wts.Wlig_vir() * virE ) << ' ' <<
//         SS( pack_wts.Wlig_h2o() * h2oE ) << std::endl;
			}
		}
		(*c_ligand)->lig_sumE = (*c_ligand)->lig_repE + (*c_ligand)->lig_atrE +
							(*c_ligand)->lig_solE + (*c_ligand)->lig_couE +
							(*c_ligand)->lig_h2oE + (*c_ligand)->lig_hbE +
							(*c_ligand)->lig_virE +
							Wlig_int_rep*(*c_ligand)->lig_int_repE +
							Wlig_int_atr*(*c_ligand)->lig_int_atrE +
							Wlig_int_solv*(*c_ligand)->lig_int_solvE +
							Wlig_int_cou*(*c_ligand)->lig_int_coulombE +
							Wlig_int_dihed*(*c_ligand)->lig_int_dihedE +
							Wlig_int_hb*(*c_ligand)->lig_int_hbE;
		}// end of for loop c_ligand
	} // end check full atom ligand flag

//mj end protein - ligand interactions

//bk add up total energies and put into global variables
	fullatomE = 0.0;
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		resenergy(seqpos) =
		 atrenergy(seqpos) + repenergy(seqpos) + solenergy(seqpos) +
		 dunenergy(seqpos) + hbenergy(seqpos) + probenergy(seqpos) +
		 pair_energy(seqpos) + h2oenergy(seqpos) + h2ohbenergy(seqpos) +
		 wsolenergy(seqpos) - unfenergy(seqpos) + protonation_energy(seqpos) +
		 intraresenergy(seqpos) + gb_elecenergy(seqpos) + cstenergy(seqpos) +
		 plane_energy(seqpos) + bondangleenergy(seqpos);

     if ( get_simple_elec() )
       resenergy(seqpos) += elecenergy(seqpos);

		fullatomE += resenergy(seqpos);
	}

	if ( score_check_current_pose() ) {
		pose_ns::Pose & pose( score_get_current_pose() );

		// sum over independent positions
		if ( pose.symmetric() ) {
			pose_ns::Symmetry_info const & s( pose.symmetry_info() );
			fullatomE = 0.0;
			for ( int i=1; i<= total_residue; ++i ) {
				if ( s.chi_independent(i) ) {
					fullatomE += ( s.chi_clones(i).size() + 1 ) * resenergy( i );
				}
			}
		}

		// set resenergy
		pose_ns::Score_state dont_care;
		FArray1D_float & resE( pose.set_1D_score( pose_ns::RESENERGY, dont_care ));
		for ( int seqpos=1; seqpos <= total_residue; ++seqpos ) {
			resE(seqpos) = resenergy(seqpos);
		}
	}

	fullatom_energy_debug::in_fullatom_energy = false;
	PROF_STOP( prof::FULLATOM_ENERGY );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin copy_fullatom_energies_to_best
///
/// @brief
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_fullatom_energies_to_best( int total_residue )
{
	using namespace fullatom_energies;

	if ( !get_fullatom_flag() ) return;

	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		best_dunenergy(res1) = dunenergy(res1);
		best_unfenergy(res1) = unfenergy(res1);
		best_protonation_energy(res1) = protonation_energy(res1);
		best_probenergy(res1) = probenergy(res1);
		best_atrenergy(res1) = atrenergy(res1);
		best_repenergy(res1) = repenergy(res1);
		best_solenergy(res1) = solenergy(res1);
		best_intraresenergy(res1) = intraresenergy(res1);
		best_bondangleenergy(res1) = bondangleenergy(res1);
		best_resenergy(res1) = resenergy(res1);
    best_elecenergy(res1) = elecenergy(res1);
	}
	for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
		for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
			best_atr_pair(res1,res2) = atr_pair(res1,res2);
			best_rep_pair(res1,res2) = rep_pair(res1,res2);
			best_sol_pair(res1,res2) = sol_pair(res1,res2);
			best_plane_plane(res1,res2) = plane_plane(res1,res2);
			best_h2o_pair(res1,res2) = h2o_pair(res1,res2);
			best_h2o_hb_pair(res1,res2) = h2o_hb_pair(res1,res2);
			best_cst_pair(res1,res2) = cst_pair(res1,res2);
			best_gb_elec_pair(res1,res2) = gb_elec_pair(res1,res2);
      best_elec_pair(res1,res2) = elec_pair(res1,res2);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_rotamer_probability
///
/// @brief
///bk uses dunbrack probabilities and standard deviations to calculate
///bk the probability of a rotamer
///
/// @detailed
/// ctsa - added bilinear interpolation over phi,psi for all probabilities and
///   derivative of the interpolation w.r.t. phi and psi
///
///
/// @param  aa - [in/out]? -
/// @param  res_variant - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  seqpos - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  rot - [in/out]? -
/// @param  rot_prob - [in/out]? -
/// @param  drperc_dphi - [in/out]? -
/// @param  drperc_dpsi - [in/out]? -
/// @param  drperc_dchi - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_rotamer_probability(
	int aa,
	int res_variant,
	float phi,
	float psi,
	int seqpos,
	int total_residue,
	FArray1Da_float chi,
	FArray1Da_int rot,
	float & rot_prob,
	float & drperc_dphi,
	float & drperc_dpsi,
	FArray1D_float & drperc_dchi
)
{
	using namespace aaproperties_pack;
	using namespace dunbrack_pack;
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;
	using namespace files_paths; // for use_conformer t/f switch

	static bool have_initialised = false;
	static bool dun_sd_correction = false;
	if(!have_initialised){
		have_initialised = true;
		dun_sd_correction = truefalseoption("dun_sd_correction");
	}

//jjh escape for nucleic acids, which have no rotamers / rotamer info
	if ( !is_protein(aa) && !is_nonnatural(aa) ) {
		drperc_dphi = 0.0;
		drperc_dpsi = 0.0;
		rot_prob = 1.0;
		return;
	}

	chi.dimension( MAX_CHI );
	rot.dimension( MAX_CHI );

	// jk default values for chain breaks / terminii
	if ( std::abs(phi) < 0.0001 ) {
		phi = template_pack::neutral_phi;
	}
	if ( std::abs(psi) < 0.0001 ) {
		psi = template_pack::neutral_psi;
	}

// constants
	float const binrange = { 10.0 };
	int const nbins = { 36 };
	float const min_rot_prob = { 1.e-8 };
//  	float const min_rot_prob = { 1.e-16 };

	// the setting of min_rot_prob controls
	// how much of a chi- and rot- tether effect
	// we can apply through this routine
	// 1.e-8 is the current setting, this gives
	// about 10 units of score. 1e-16 seems to
	// have no ill-effects, yields instead around 20 units.

// local
	int phibin,phibin_next,psibin,psibin_next;
	float p0;
	float dp0_dphi,dp0_dpsi;
	float chi_expect,chi_sd,chi_sd2,chi_diff;
	float phi_err,psi_err;
	float lnp,drperc_dphi_tmp,drperc_dpsi_tmp;
	FArray1D_float p( MAX_CHI );
	FArray1D_float dexpect_dphi( MAX_CHI );
	FArray1D_float dexpect_dpsi( MAX_CHI );
	FArray1D_float dsd_dphi( MAX_CHI );
	FArray1D_float dsd_dpsi( MAX_CHI );
	FArray1D_float dp_dchi( MAX_CHI );
	FArray1D_float dp_dexpect( MAX_CHI );
	FArray1D_float dp_dsd( MAX_CHI );

	bool const debug( false && fullatom_energy_debug::debug );
	get_phi_interpolate_bin(phi,psi,phibin,psibin,phibin_next,psibin_next,phi_err,
	 psi_err,seqpos,total_residue);

	//std::cout << phibin << " " << phibin_next << " : "  << psibin << " " << psibin_next << std::endl;

//bk backbone dependent probability for chi1 and chi2
	bool treat_as_angles = false;

	int packed_rotno = get_aa_rotno_to_packedrotno(aa,res_variant,rot);
	if ( debug ) std:: cout << "packed_rotno: " << packed_rotno << std::endl;
	if ( bias_this_rot_to_native(aa,seqpos,rot) ) {
		if ( debug ) std::cout << "bias-rot" << std::endl;
		interpolate_bilinear(phibin,phibin_next,phi_err,psibin,psibin_next,psi_err,
												 highest_dun_prob(1,1,aa),nbins,nbins,binrange,
												 treat_as_angles,p0,dp0_dphi,dp0_dpsi);
	} else {
		interpolate_bilinear(phibin,phibin_next,phi_err,psibin,psibin_next,psi_err,
												 dun_prob(1,1,packed_rotno),nbins,nbins,binrange,
												 treat_as_angles,p0,dp0_dphi,dp0_dpsi);
		//	std::cout << "interp out: " << p0 << std::endl;
	}

	/// now apply the tether constraints
	/// tether_factor == 1.0 if no constraints
	float const tether_factor = apply_chi_tether(seqpos,aa,res_variant,chi,rot);
	//std::cout << "tether_factor: " << tether_factor << std::endl;
	p0 *= tether_factor;
	dp0_dphi *= tether_factor;
	dp0_dpsi *= tether_factor;


	// ctsa - check validity of result
	if ( p0 < 0. ) { //chu: there are 0.0 prob rotamers in library
		std::cout << "ABORT:: invalid dunbrack rotamer base prob:" << std::endl;
		std::cout << "  possible invalid aa-rotamer combination" << std::endl;
		std::cout << "aa,rot_id,p0: " << aa << ' ' << rot << ' ' << p0 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//bk penalize for [all] chi angles that are not optimal
//bk (according to dunbrack). This is done by assuming a gaussian probability
//bk for the chi angles

	// ctsa - interpolate chi and sd for each chi angle over phi,psi values
	int const nchi_aa = nchi(aa,res_variant);
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		p(i) = 1.0;

		dexpect_dphi(i) = 0.;
		dexpect_dpsi(i) = 0.;

		dsd_dphi(i) = 0.;
		dsd_dpsi(i) = 0.;

		dp_dchi(i) = 0.;
		dp_dexpect(i) = 0.;
		dp_dsd(i) = 0.;

		if ( ( i <= nchi_aa ) && ( seqpos != 1 ) && ( seqpos != total_residue )
		 && ( phi != -90.0 ) && ( psi != 130.0 )//cmd escape for default phi psi values of chain breaks
		 && ( ! is_chi_proton_rotamer(aa,res_variant,i) ) ) { // escape for proton rotamers
		 //std::cout << "escape?" << std::endl;

			treat_as_angles = true;

			//Objexx:SGM Replaced call below since dun_chi is no longer sliceable
			interpolate_bilinear_by_value(
				dun_chi( phibin, psibin, packed_rotno, i ),
				dun_chi( phibin_next, psibin, packed_rotno, i ),
				dun_chi( phibin, psibin_next, packed_rotno, i ),
				dun_chi( phibin_next, psibin_next, packed_rotno, i ),
				phi_err, psi_err, binrange, treat_as_angles, chi_expect,
				dexpect_dphi(i), dexpect_dpsi(i)
			);
//			interpolate_bilinear( phibin, phibin_next, phi_err, psibin, psibin_next,
//			 psi_err, dun_chi(1,1,packed_rotno,i), nbins, nbins, binrange,
//			 treat_as_angles, chi_expect, dexpect_dphi(i), dexpect_dpsi(i) );

			chi_diff = subtract_chi_angles(chi(i),chi_expect,aa,i);

			// if ( i == 2 && ( aa == aa_phe || aa == aa_tyr ) && std::abs(chi_diff) > 90. ) {
// 				std::cout << "ABORT: symmetry failure" << std::endl;
// 				std::cout << "aa,chino,chi_expect,chi(chi_no) " <<
// 				 aa << ' ' << i << SS( chi_expect ) << SS( chi(i) ) << std::endl;
// 				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			//}

			treat_as_angles = false;

			//Objexx:SGM Replaced call below since dun_sd is no longer sliceable
			interpolate_bilinear_by_value(
				dun_sd( phibin, psibin, packed_rotno, i ),
				dun_sd( phibin_next, psibin, packed_rotno, i ),
				dun_sd( phibin, psibin_next, packed_rotno, i ),
				dun_sd( phibin_next, psibin_next, packed_rotno, i ),
				phi_err, psi_err, binrange, treat_as_angles, chi_sd,
				dsd_dphi(i), dsd_dpsi(i)
			);
	//		std::cout << "chi_sd at chi " << i << ":"  << chi_sd << std::endl;
//			interpolate_bilinear( phibin, phibin_next, phi_err, psibin, psibin_next,
//			 psi_err, dun_sd(1,1,packed_rotno,i), nbins, nbins, binrange,
//			 treat_as_angles, chi_sd, dsd_dphi(i), dsd_dpsi(i) );

			// ctsa - check validity of result
			if ( chi_sd < 0.0 ) { //mj from chu: there are 0.0 prob rotamers in library
				std::cout << "ABORT:: invalid dunbrack rotamer sd:" << std::endl;
				std::cout << "  possible invalid aa-rotamer combination" << std::endl;
				std::cout << "aa,rot_id: " << aa << ' ' << rot << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}


			// This is a temporary way to correct for the oversharpening of the chi_deviations
			// in rosetta decoys. Ultimately they should be corrected in the dunbrack data file
			// but for now this is easier to manipulate.
			if(dun_sd_correction){
				if( i == 1 ){   // only correct chi1
					// correct standard deviations
					if( aa == param_aa::aa_arg) chi_sd /= 0.4001;   //ARG
					if( aa == param_aa::aa_asn) chi_sd /= 0.4501;   //ASN
					if( aa == param_aa::aa_asp) chi_sd /= 0.4001;   //ASP
					if( aa == param_aa::aa_cys) chi_sd /= 0.4201;   //CYS
					if( aa == param_aa::aa_gln) chi_sd /= 0.3501;   //GLN
					if( aa == param_aa::aa_glu) chi_sd /= 0.3501;   //GLU
					if( aa == param_aa::aa_his) chi_sd /= 0.5000;   //HIS
					if( aa == param_aa::aa_ile) chi_sd /= 0.5001;   //ILE
					if( aa == param_aa::aa_leu) chi_sd /= 0.5200;   //LEU
					if( aa == param_aa::aa_lys) chi_sd /= 0.4001;   //LYS
					if( aa == param_aa::aa_met) chi_sd /= 0.4501;   //MET
					if( aa == param_aa::aa_phe) chi_sd /= 0.7001;   //PHE
					if( aa == param_aa::aa_ser) chi_sd /= 0.4001;   //SER
					if( aa == param_aa::aa_thr) chi_sd /= 0.5001;   //THR
					if( aa == param_aa::aa_trp) chi_sd /= 0.7001;   //TRP
					if( aa == param_aa::aa_tyr) chi_sd /= 0.7001;   //TYR
					if( aa == param_aa::aa_val) chi_sd /= 0.5000;   //VAL
				}
			}


			chi_sd2 = chi_sd * chi_sd;
			lnp = -((chi_diff*chi_diff)/(2*chi_sd2));
			p(i) = std::exp(lnp);

			dp_dchi(i) = -p(i)*chi_diff/chi_sd2;
			dp_dexpect(i) = -dp_dchi(i);
			dp_dsd(i) = p(i)*(-2)*lnp/chi_sd;
			if ( debug ) std::cout << "p" << i << " : " << p(i) << std::endl;
		}
	}

	// dunbrack energy
	//float const p_product;
	//if (use_conformer) {
	//	 p_product = 1;
	//}
	//else {
	float const	 p_product = p(1) * p(2) * p(3) * p(4);
//	}
	rot_prob = p0 * p_product;
	//std::cout << "rot_prob: " << rot_prob << std::endl;
	if ( debug ) std::cout << "rot_prob: " << p0 << ' ' << p_product <<
								 std::endl;

	// ctsa - dunbrack energy chi derivatives
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		drperc_dchi(i) = p0;
		for ( int j = 1; j <= MAX_CHI; ++j ) {
			if ( j == i ) {
				drperc_dchi(i) *= dp_dchi(j);
			} else {
				drperc_dchi(i) *= p(j);
			}
		}
	}

	// ctsa - dunbrack energy phi/psi derivatives
	drperc_dphi = dp0_dphi * p_product;
	drperc_dpsi = dp0_dpsi * p_product;
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		drperc_dphi_tmp = p0;
		drperc_dpsi_tmp = p0;
		for ( int j = 1; j <= MAX_CHI; ++j ) {
			if ( i == j ) {
				drperc_dphi_tmp *= dp_dexpect(i)*dexpect_dphi(i) +
				 dp_dsd(i)*dsd_dphi(i);
				drperc_dpsi_tmp *= dp_dexpect(i)*dexpect_dpsi(i) +
				 dp_dsd(i)*dsd_dpsi(i);
			} else {
				drperc_dphi_tmp *= p(j);
				drperc_dpsi_tmp *= p(j);
			}
		}
		drperc_dphi += drperc_dphi_tmp;
		drperc_dpsi += drperc_dpsi_tmp;
	}

	// ctsa - check for nan's
	if ( drperc_dphi != drperc_dphi ) {
		std::cout << "ABORT: invalid drperc_dphi value in get_rotamer_probability" << std::endl;
		std::cout << "aa,aav,phi,psi,seqpos,total_residue: " << SS( aa ) <<
		 SS( res_variant ) << SS( phi ) << SS( psi ) << SS( seqpos ) << SS( total_residue ) <<
		 std::endl;
		std::cout << "chi1234: " << SS( chi(1) ) << SS( chi(2) ) << SS( chi(3) ) <<
		 SS( chi(4) ) << std::endl;
		std::cout << "rot1234: " << SS( rot(1) ) << SS( rot(2) ) << SS( rot(3) ) <<
		 SS( rot(4) ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( drperc_dpsi != drperc_dpsi ) {
		std::cout << "ABORT: invalid drperc_dpsi value in get_rotamer_probability" << std::endl;
		std::cout << "aa,aav,phi,psi,seqpos,total_residue: " << SS( aa ) <<
		 SS( res_variant ) << SS( phi ) << SS( psi ) << SS( seqpos ) << SS( total_residue ) <<
		 std::endl;
		std::cout << "chi1234: " << SS( chi(1) ) << SS( chi(2) ) << SS( chi(3) ) <<
		 SS( chi(4) ) << std::endl;
		std::cout << "rot1234: " << SS( rot(1) ) << SS( rot(2) ) << SS( rot(3) ) <<
		 SS( rot(4) ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= MAX_CHI; ++i ) {
		if ( drperc_dchi(i) != drperc_dchi(i) ) {
			std::cout << "ABORT: invalid drperc_dchi value in get_rotamer_probability" << std::endl;
			std::cout << "i,drperc_dchi(i), " << i << SS( drperc_dchi(i) ) << std::endl;
			std::cout << "aa,aav,phi,psi,seqpos,total_residue: " << SS( aa ) <<
			 SS( res_variant ) << SS( phi ) << SS( psi ) << SS( seqpos ) << SS( total_residue ) <<
			 std::endl;
			std::cout << "chi1234: " << SS( chi(1) ) << SS( chi(2) ) << SS( chi(3) ) <<
			 SS( chi(4) ) << std::endl;
			std::cout << "rot1234: " << SS( rot(1) ) << SS( rot(2) ) << SS( rot(3) ) <<
			 SS( rot(4) ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
// ctsa

//bk debug
	if ( rot_prob <= min_rot_prob ) {
		if ( runlevel > inform ) {
			std::cout << "low_prob_rot: pos/aa/r1/2/3/4 prob,all/chi1/2/3/4" << std::endl;
			std::cout << "low_prob_rot: " << I( 4, seqpos ) << ' ' <<
			 I( 2, aa ) << ' ' << I( 1, rot(1) ) << ' ' << I( 1, rot(2) ) << ' ' <<
			 I( 1, rot(3) ) << ' ' << I( 1, rot(4) ) << ' ' << ' ' <<
			 E( 7, 2, rot_prob) << ' ' << E( 7, 2, p(1) ) << ' ' <<
			 E( 7, 2, p(2) ) << ' ' << E( 7, 2, p(3) ) << ' ' <<
			 E( 7, 2, p(4) ) << "\n" << std::endl;
		}
		rot_prob = min_rot_prob;
	}


}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_dunbrack_derivs
///
/// @brief
///   Get the derivative of the dunbrack energy with respect to
/// the backbone phi or psi angle, and chi1,chi2. ctsa
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  seqpos - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  rot - [in/out]? -
/// @param  dE_dphi - [in/out]? -
/// @param  dE_dpsi - [in/out]? -
/// @param  dE_dchi - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_dunbrack_derivs(
	int & aa,
	int & aav,
	float & phi,
	float & psi,
	int & seqpos,
	int & total_residue,
	FArray1Da_float chi,
	FArray1Da_int rot,
	float & dE_dphi,
	float & dE_dpsi,
	FArray1Da_float dE_dchi
)
{
	using namespace param;

	chi.dimension( MAX_CHI );
	rot.dimension( MAX_CHI );
	dE_dchi.dimension( MAX_CHI );


	// local
	FArray1D_float dp_dchi( MAX_CHI, 0.0 );
	float invp, dp_dphi(0.0), dp_dpsi(0.0);

	// get deriv w.r.t. phi/psi/chi
	float p_dun = 0.0;

	// get P_dunbrack(phi,psi) in all cases
	get_rotamer_probability(aa,aav,phi,psi,seqpos,total_residue,chi,rot,p_dun,dp_dphi,
	 dp_dpsi,dp_dchi);

  if ( std::abs(p_dun) > 1E-20 ) {
		invp = -1.0/p_dun;
	} else {
		invp = 0.0;
	}

	dE_dphi = invp * dp_dphi;
	dE_dpsi = invp * dp_dpsi;
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		dE_dchi(i) = invp * dp_dchi(i);
	}

}

//////////////////////////////////////////////////////////////////////
//vats   fill constraint energies
//
//////////////////////////////////////////////////////////////////////

void
 fill_cstE_array(
    int res1,
    int res2,
    float & energy
 )
{
 using namespace fullatom_energies;

 float temp = 0.5*energy;
 cstenergy(res1) += temp;
 cstenergy(res2) += temp;

 cst_pair(res1,res2) += energy;
 if ( res1 != res2 ) cst_pair(res2,res1) += energy;
 return;

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// called by fullatom_energy
//
// communicates the score via fill_Earrays
//

void
nblist_atompair_energy(
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord
)

{
	using namespace kin;
  using namespace aaproperties_pack;
	Minimizer_map const & min_map( retrieve_min_map() );
	int total_nbrs(0);

	// Set score_variant boolean flags outside loops for efficiency
	bool const loc_lj_variant = ( get_score_variant() == "loc_lj" );
	bool const lk_sol_variant = ( get_score_variant() == "lk_sol" );

	for ( std::vector< Atom_node* >::const_iterator
					atom1_iter=min_map.atoms_begin(), atom1_end=min_map.atoms_end();
				atom1_iter!=atom1_end; ++atom1_iter ) {
		Atom_node * atom1( *atom1_iter );
		int const atm1( atom1->atomno() );
		int const res1( atom1->rsd() );
		int const aa1 = res(res1);
		int const aav1 = res_variant(res1);
		int const type1 = aaproperties_pack::fullatom_type(atm1,aa1,aav1);
    const float chrg1 = atomic_charge(atm1,aa1,aav1);
		float const & xyz_1a11( full_coord(1,atm1,res1) );
		std::vector< Atom_node* >::const_iterator atom2_begin;
		std::vector< Atom_node* >::const_iterator atom2_end;
		min_map.get_nbr_bounds( atom1, atom2_begin, atom2_end );
		for ( std::vector< Atom_node* >::const_iterator atom2_iter = atom2_begin;
					atom2_iter != atom2_end; ++atom2_iter ) {
			int const atm2( (*atom2_iter)->atomno() );
			int const res2( (*atom2_iter)->rsd() );
			if ( res2 < res1 || ( res2 == res1 && atm2 <= atm1 ) ) continue;
			++total_nbrs;
			{ // Scope
				//assert( pair_moved( res1, res2 ) );
				int const aa2 = res(res2);
				int const aav2 = res_variant(res2);
        float const chrg2 = atomic_charge(atm2, aa2, aav2);
			  float cp_weight;
				if ( count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,
												cp_weight) ) {
					int const type2 = aaproperties_pack::fullatom_type(atm2,aa2,aav2);
					float solE(0.0), sE1(0.0), sE2(0.0), atrE(0.0), repE(0.0), d2(0.0), elecE(0.0);

          bool aro_charged = aro_charged_pair( aa1, type1, atm1, aa2, type2, atm2 );
          pairenergy(xyz_1a11,full_coord(1,atm2,res2),type1,type2,chrg1,chrg2, sE1,sE2,atrE,repE,elecE, d2);
					//if( repE > 1.0 ) std::cout<<" Bad clash between: "<<atm1<<" "<<aa1<<" "<<res1<<" "<<atm2<<" "<<aa2<<" "<<res2<<std::endl;

          if ( !aro_charged )
            elecE = 0.0;
					if ( get_scale_res_energy_flag() ) { //Objexx
						using namespace scale_res_energy_ns;
						if ( scale_energy_matrix(res1,res2) ) {
							atrE *= scale_res_energy_weight; //glb
							sE1 *= scale_res_energy_weight; //glb
							sE2 *= scale_res_energy_weight; //glb
						}
						if ( scale_rep_energy_matrix(res1,res2) ) {
							repE *= scale_res_rep_energy_weight; //glb
						}
					}

					atrE *= cp_weight;
					repE *= cp_weight;
          elecE *= cp_weight;
					if ( loc_lj_variant ) {
						if ( std::abs( res1 - res2 ) > 4 ) atrE *= 3.0f;
						solE = ( ( sE1 + sE2 ) * cp_weight ) * 0.6f;
					} else if ( lk_sol_variant ) {
						solE = 0.0;
					} else {
						solE = ( sE1 + sE2 ) * cp_weight;
					}
//elecE = 0.0;
					fill_Earrays(res1, res2, solE,atrE,repE, elecE );
				}
			} // Scope
		} // loop through neighbors for each entry in the atom list
	} // loop through atom list
}


///////////////////////////////////////////////////////////////////////////////
// Fullatom_energy() doesn't save atom by atom energy contributions,
// just residue by residue. If we want to visualize energy at atomic resolution
// need to calculate it.
void
evaluate_atom_energy( FArray2D_float & atom_energy )
{

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param_aa;
	using namespace pdbstatistics_pack;
	using namespace nblist;
	using namespace template_pack;
	using namespace param_pack;

	atom_energy = 0.0; //dimensions should be MAX_ATOM(), total_residue!

	//
	//Copied from fullatom_energy()
	// stripped out some silly stuff, commented lines, etc.
	// Apportioned atom energy based on weights of 1.0, might not be
	// better to accept a weight map.
	//

	int type1,aa1,aav1,aa2,aav2;
	float cp_weight,chrg1,chrg2,sE1,sE2,atrE,repE,elecE,d2;

	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

	int const xyz_size1 = full_coord.size1();
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		aa1 = res(res1);
		aav1 = res_variant(res1);
		int const atm1e = natoms(aa1,aav1);
		int const l1s = fullatom_type.index(1,aa1,aav1);
		for ( int res2 = res1; res2 <= total_residue; ++res2 ) {
			if ( !neighborlist(res1,res2) ) continue;
			aa2 = res(res2);
			aav2 = res_variant(res2);
			int const atm2e = natoms(aa2,aav2);
			int const l2s = fullatom_type.index(1,aa2,aav2);
			int const lfull_coords = full_coord.index(1,1,res2);
			for ( int atm1 = 1, l1 = l1s; atm1 <= atm1e; ++atm1, ++l1 ) {

				type1 = fullatom_type[ l1 ]; // fullatom_type(atm1,aa1,aav1)
				chrg1 = atomic_charge[ l1 ];

				int l11 = full_coord.index(1,atm1,res1);
				float const & xyz_1a11( full_coord[ l11 ] ); // full_coord(1,atm1,res1)
				for ( int atm2 = 1, l2 = l2s, lfull_coord = lfull_coords;
							atm2 <= atm2e; ++atm2, ++l2, lfull_coord+=xyz_size1 ) {
					int type2 = fullatom_type[ l2 ]; // fullatom_type(atm2,aa2,aav2)
					if ( count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,
													cp_weight) ) {
						chrg2 = atomic_charge[ l2 ];

						bool aro_charged = aro_charged_pair( aa1, type1, atm1, aa2, type2, atm2 );
						pairenergy(xyz_1a11,full_coord[ lfull_coord ],type1,type2,
											 chrg1, chrg2,sE1,sE2,atrE, repE, elecE, d2); // [lfull_coord] == (1,atm2,res2)

						if ( !aro_charged )
							elecE = 0.0;

						atrE *= cp_weight;
						repE *= cp_weight;
						elecE *= cp_weight;
						sE1 *= cp_weight;
						sE2 *= cp_weight;

						atrE *= pack_wts.Watr();
						repE *= pack_wts.Wrep();
						sE1  *= pack_wts.Wsol();
						sE2  *= pack_wts.Wsol();
						elecE *= pack_wts.Wele();

						if ( fullatom_type( atm1, aa1, aav1) <= 21 && fullatom_type(atm2, aa2, aav2) <= 21 ) {
							// no hydrogens involved
							atom_energy( atm1, res1 ) += 0.5f * atrE + 0.5f * repE + 0.5f * elecE + sE1;
							atom_energy( atm2, res2 ) += 0.5f * atrE + 0.5f * repE + 0.5f * elecE + sE2;
							//							atom_energy( atm1, res1 ) += 0.5f * elecE;
							//							atom_energy( atm2, res2 ) += 0.5f * elecE;
						} else if (d2 < hydrogen_interaction_cutoff) {
							atom_energy( atm1, res1 ) += 0.5f * repE;
							atom_energy( atm2, res2 ) += 0.5f * repE;
						}

						//						if (repE > 0.0) std::cout << "HEY " << aa_name1( aa1 )  <<  res1 << " " << atom_name( atm1, aa1, aav1 ) <<
						//															" " << aa_name1( aa2 )  <<  res2 << " " << atom_name( atm2, aa2, aav2 ) <<"   : " << repE <<   std::endl;

					}

				} // atm2
			} // atm1
		} // res2
	} // res1

	float hbenergy_srbb,hbenergy_lrbb,hbenergy_scmc,hbenergy_sc;
	fill_hbond_arrays(	true, res,res_variant,full_coord,total_residue,neighborlist,neighbors,
											hbonds::hbderiv_NONE,hbenergy_srbb,hbenergy_lrbb,hbenergy_scmc,hbenergy_sc,
											atom_energy );


}


///////////////////////////////////////////////////////////////////////////////
