// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16119 $
//  $Date: 2007-07-19 13:20:22 -0400 (Thu, 19 Jul 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "fullatom_sasa.h"
#include "aaproperties_pack.h"
#include "decoystats.h"
#include "dna_ns.h" //for quicker rna calculation
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom_extra_props.h"
#include "fullatom_sasa_ns.h"
#include "misc.h"
#include "pose.h"
#include "read_paths.h"
#include "param.h"
#include "score_name.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ubyte.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


// this lookup table is used in sasa computation (also in void.cc)
short const bit_count[] = { // lookup table for number of 1 bits in a ubyte
	0,1,1,2,1,2,2,3, 1,2,2,3,2,3,3,4,   1,2,2,3,2,3,3,4, 2,3,3,4,3,4,4,5,  // 0x 1x
	1,2,2,3,2,3,3,4, 2,3,3,4,3,4,4,5,   2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,  // 2x 3x
	1,2,2,3,2,3,3,4, 2,3,3,4,3,4,4,5,   2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,  // 4x 5x
	2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,   3,4,4,5,4,5,5,6, 4,5,5,6,5,6,6,7,  // 6x 7x
	1,2,2,3,2,3,3,4, 2,3,3,4,3,4,4,5,   2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,  // 8x 9x
	2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,   3,4,4,5,4,5,5,6, 4,5,5,6,5,6,6,7,  // Ax Bx
	2,3,3,4,3,4,4,5, 3,4,4,5,4,5,5,6,   3,4,4,5,4,5,5,6, 4,5,5,6,5,6,6,7,  // Cx Dx
	3,4,4,5,4,5,5,6, 4,5,5,6,5,6,6,7,   4,5,5,6,5,6,6,7, 5,6,6,7,6,7,7,8,  // Ex Fx
};

////////////////////////////////////////////////////////////////////////////////
/// @begin input_sasa_dats
///
/// @brief
///cj    Reads in SASA-angles.dat  SASA-masks.dat
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
input_sasa_dats()
{
	using namespace fullatom_sasa;

	FArray1D_short tmp( nbytes );
	static bool init = { false };

	if ( init ) return;
	init = true;

//cj    inputting the masks they are 21 ubytes long, 162x100 (see header)
//cj    expects file to be complete
	utility::io::izstream & masks_stream( open_data_file( "SASA-masks.dat" ) );

	for ( int i = 1; i <= nolp*nori; ++i ) {
		for ( int j = 1; j <= nbytes; ++j ) {
			masks_stream >> tmp(j);
		} masks_stream >> skip;

		for ( int j = 1; j <= nbytes; ++j ) {
			masks(j,i) = static_cast< ubyte>(tmp(j));
		}
	}
	masks_stream.close();
	masks_stream.clear();

//cj    inputting the angle lookup for the mask, need to add a 1 to each number
	utility::io::izstream & angles_stream( open_data_file( "SASA-angles.dat" ) );

//cj    2D array is aphi by theta
	for ( int i = 1; i <= nphi; ++i ) {
		for ( int j = 1; j <= ntheta; ++j ) {
			angles_stream >> angles(i,j);
		} angles_stream >> skip;
//cj       for ( j = 1; j <= ntheta; ++j ) {
//cj          ++angles(i,j);
//cj       }
	}
	angles_stream.close();
	angles_stream.clear();

	std::cout << "========================================" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_overlap
///
/// @brief
///
/// @detailed
///cj    getting overlap from a to b (or i to j, see below)
///cj    this returns the degree of overlap between two atoms
///cj    adapted from erics code in area.c GetD2
///cj    returns value from 1 to 100
///cj    This calculation is based on the law of cosines,
///cj    see LeGrand and Merz, Journal of Computational
///cj    Chemistry 14(3):349-52 (1993).
///cj    Note that equation (4) is wrong, the denominator
///cj    should be 2r r   instead of 2r r
///cj                i iq              i q
///
/// @param  a - [in/out]? -
/// @param  ra - [in/out]? -
/// @param  b - [in/out]? -
/// @param  rb - [in/out]? -
/// @param  dist - [in/out]? -
/// @param  olp - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_overlap(
	FArray1DB_float const & a,
	float & ra,
	FArray1DB_float const & b,
	float & rb,
	float & dist,
	int & olp
)
{
	float epsilon,costh;

//cj    min distance cutoff
	epsilon = 0.01;

	if ( dist < epsilon ) {
//cj    atoms too close, causes round off error
//cj    use this cutoff
		if ( ra < rb ) {
			olp = 100;
		} else {
			olp = 1;
		}
	} else if ( rb+dist <= ra ) {
//cj    If atom a completely engulfs atom b, consider a to have
//cj    no overlap due to atom b.
		olp = 1;
	} else if ( rb+dist <= ra ) {
//cj    If atom a is completely engulfed by atom b, then turn it
//cj    completely off (i.e. d2 = 99).
		olp = 100;
	} else {
//cj    Otherwise, compute the amount of overlap using the law of cosines.
//cj    "costh" is the angle of the cone of intersection that atom b
//cj    imposes on atom a.  "ra" is the radius of atom a, and "rb" is
//cj    the radius of atom b.  "sqrtd" is the actual distance between
//cj    the a and b centers, while "dist" is the square of this distance.
		costh = (ra*ra+dist*dist-rb*rb)/(2*ra*dist);
		olp = static_cast< int >((1.0f-costh)*50)+1;
		if ( olp > 100 ) {
			olp = 100;
		} else if ( olp < 0 ) {
//cj       We already hopefully accounted for this possibility by requiring that
//cj       dist < epsilon, but in case not we don't want a potential bug to go
//cj       unnoticed.
			std::cout << "problem in calculating overlap between:" << std::endl;
			std::cout << "a  " <<
			 F( 7, 3, a(1) ) << ' ' << F( 7, 3, a(2) ) << ' ' << F( 7, 3, a(3) ) <<
			 std::endl;
			std::cout << "b  " <<
			 F( 7, 3, b(1) ) << ' ' << F( 7, 3, b(2) ) << ' ' << F( 7, 3, b(3) ) <<
			 std::endl;
			std::cout << "ra=" << SS( ra ) << std::endl;
			std::cout << "rb=" << SS( rb ) << std::endl;
			std::cout << "dist=" << SS( dist ) << std::endl;
			std::cout << "costh=" << SS( costh ) << std::endl;
			std::cout << "Teminiating calculation" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_orientation
///
/// @brief
///cj    gets the orientation of a to b (i to j, see below)
///cj    does this by calculating two angles, aphi and theta
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  aphi - [in/out]? -
/// @param  theta - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_orientation(
	FArray1DB_float const & a,
	FArray1DB_float const & b,
	int & aphi,
	int & theta,
	float dist
)
{
	using namespace fullatom_sasa;
	using namespace numeric::constants::f;
	using numeric::sin_cos_range;

	//apl allocate this once only
	static FArray1D_float diff( 3 );

//cj    figure the difference between a and b
	//apl - You've already computed the distance! reuse it.
	diff(1) = (a(1)-b(1) ) / dist;
	diff(2) = (a(2)-b(2) ) / dist;
	diff(3) = (a(3)-b(3) ) / dist;

//cj    now figure out polar values of aphi and theta
//cj    Normalize the difference
//cj    first get the length of the vector
	//vector_normalize(diff);

//cj    figuring aphi

	float p = std::acos( sin_cos_range( diff(3) ) );

	p *= nphi / pi_2;
	aphi = static_cast< int >( p );
	++aphi; // for fortran goes from 1 to n
	if ( aphi > nphi ) aphi = 1;

//cj    figuring theta
	float t = std::atan2(diff(2),diff(1));
	t *= ntheta / pi_2;
	theta = static_cast< int >( t );
	++theta; // for fortran goes from 1 to n
	if ( theta < 1 ) {
		theta += ntheta;
	} else if ( theta > ntheta ) {
		theta = 1;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_2way_orientation
///
/// @brief
///cj    gets the orientation of a to b (i to j, see below)
///cj    does this by calculating two angles, aphi and theta
///
/// @detailed
///
/////////////////////////////////////////////////////////////////////////////////
void
get_2way_orientation(
	FArray1DB_float const & a,
	FArray1DB_float const & b,
	int & aphi_a2b,
	int & theta_a2b,
	int & aphi_b2a,
	int & theta_b2a,
	float dist
)
{
	using namespace fullatom_sasa;
	using namespace numeric::constants::f;
	using numeric::sin_cos_range;

	//apl allocate this once only
	static FArray1D_float diff( 3 );

	//cj    figure the difference between a and b
	//apl - You've already computed the distance! reuse it.
	diff(1) = (a(1)-b(1) ) / dist;
	diff(2) = (a(2)-b(2) ) / dist;
	diff(3) = (a(3)-b(3) ) / dist;

	//diff(4) = (b(1)-a(1) ) / dist;
	//diff(5) = (b(2)-a(2) ) / dist;
	//diff(6) = (b(3)-a(3) ) / dist;

	//cj    now figure out polar values of aphi and theta
	//cj    Normalize the difference
	//cj    first get the length of the vector
	//vector_normalize(diff);

	//figuring aphi_a2b
	float p_a2b = std::acos( sin_cos_range( diff(3) ) );
	p_a2b *= nphi / pi_2;
	float p_b2a = nphi / 2 - p_a2b;

	aphi_a2b = static_cast< int >( p_a2b );
	aphi_b2a = static_cast< int >( p_b2a );

	++aphi_a2b; // for fortran goes from 1 to n
	++aphi_b2a; // for fortran goes from 1 to n

	if ( aphi_a2b > nphi ) aphi_a2b = 1;
	if ( aphi_b2a > nphi ) aphi_b2a = 1;

	//figuring theta_a2b
	float t_a2b = std::atan2(diff(2),diff(1));
	t_a2b *= ntheta / pi_2;

	theta_a2b = static_cast< int >( t_a2b );
	++theta_a2b; // for fortran goes from 1 to n
	if ( theta_a2b < 1 ) {
		theta_a2b += ntheta;
	} else if ( theta_a2b > ntheta ) {
		theta_a2b = 1;
	}

	//figuring theta_a2b
	float t_b2a = ntheta / 2.0f + t_a2b;
	if (t_b2a > ntheta / 2.0f ) t_b2a -= ntheta;


	theta_b2a = static_cast< int >( t_b2a );
	++theta_b2a; // for fortran goes from 1 to n
	if ( theta_b2a < 1 ) {
		theta_b2a += ntheta;
	} else if ( theta_b2a > ntheta ) {
		theta_b2a = 1;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin calc_aprox_sasa
///
/// @brief
///cj    function to calculate the aproximate solvent accessible surface area
///cj    changed :: only do for those necessary
///
/// @detailed
///jg   compared a straight-up sasa calculation (no asp weightings)
///jg   to results from naccess; we agree within 2%, which is probably
///jg   fine considering the differing vdw radii.
///jg   also, fixed atom scope bug.  (jjg 2/5/2)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///cj    NOTE :: UNCOMMENTED NOT USE 2D FULLCOORD ARRAY (NEED TO CHANGE)
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_aprox_sasa()
{
	using namespace aaproperties_pack;
	using namespace fullatom_ex_props;
	using namespace fullatom_sasa;
	using namespace misc;
	using namespace param;
	using namespace numeric::constants::f;

//cj  scaling factor for gsolt (original weighting used this scaling factor)
	static float const gsolt_orig_weight = { 0.0008f };

	static bool init_d = { false };
	static FArray1D_float ic( 3 );
	static FArray1D_float jc( 3 );
	static FArray2D_float rsq_min( MAX_ATOMTYPES(), MAX_ATOMTYPES() );

//jo	bool phobei,phobej;
//jo	static FArray2D_bool phobic( MAX_ATOM(), total_residue ); // hydrophobicity lookup
	FArray3D_ubyte atm_masks( nbytes, MAX_ATOM()(), total_residue, ubyte( 0 ) );
	 // full mask for atoms
//jo	FArray2D_ubyte hpb_masks( nbytes, MAX_ATOM()(), total_residue ); // hydrophophic mask for atoms
//jo	FArray2D_ubyte pol_masks( nbytes, MAX_ATOM()(), total_residue ); // polar mask for atoms
	int in,it,jn,jt,ctr,olp,aphi,theta,point,masknum;
	float dist,rij,irad,jrad,fraction,total_sa,expose;
//jo	std::string aname;

//jg	float sasa; // solvent-accessible surface area (no weightings)
	float dist_sq; // square of the distance between atoms

//cj set all score counters to zero - all defined in fullatom_sasa.h
//jo	hpbur = 0.0;                 // total hydrophobic buried
//jo	hpexp = 0.0;                 // total hydrophobic exposed
//jo	pobur = 0.0;                 // total hydrophillic (polar) buried
//jo	poexp = 0.0;                 // total hydrophillic (polar) exposed
//jo	sdhpbur = 0.0;               // sidechain hydrophobic buried
//jo	sdhpexp = 0.0;               // sidechain hydrophobic exposed
//jo	sdpobur = 0.0;               // sidechain hydrophillic buried
//jo	sdpoexp = 0.0;               // sidechain hydrophillic exposed
//jo	mnhpbur = 0.0;               // mainchain hydrophobic buried
//jo	mnhpexp = 0.0;               // mainchain hydrophobic exposed
//jo	mnpobur = 0.0;               // mainchain hydrophillic buried
//jo	mnpoexp = 0.0;               // mainchain hydrophillic exposed

//jg	sasa = 0.0;                  // plain old surface area
	gsolt = 0.0;
	sasa_gsolt = 0.0;
	sasa_polar = 0.0;
	sasa_apolar = 0.0;

// dG solvation total
//jo	gsolnp = 0.0;                // dG solvation non polar
//jo	gsolpo = 0.0;                // dG solvation polar
//jo	genvt = 0.0;                 // dG environment total
//jo	genvnp = 0.0;                // dG environment non polar
//jo	genvpo = 0.0;                // dG environment polar
//jo	npca = 0.0;                  // total non polar contact area
//jo	poca = 0.0;                  // total polar contact area
//jo	npnp = 0.0;                  // non polar non polar contact area
//jo	nppo = 0.0;                  // non polar polar contact area
//jo	ponp = 0.0;                  // polar non polar contact area
//jo	popo = 0.0;                  // polar polar contact area
//jo	genvpof = 0.0;               // dG environment polar (only those no hbonded)
//jo	ponpf = 0.0;                 // nonhbonded polar atom's nonpolar buried SASA

//jo	npohb = 0;                   // number of polars hydrogen bonded
//jo	nponh = 0;                   // number of polars not hydrogen bonded

//jo	ponpfbur = 0.0;              // buried nonhbonded polars nonpolar SASA
//jo	nponhbur = 0;                // number of buried nonhbonded polars


//cj----initialize all the sasa data arrays (done once only)
	if ( !init_d ) {
		init_d = true;
		input_sasa_dats();
		calc_min_rsq(rsq_min);

//$$$cj    determine hydrophobicity - don't really use this at the moment
//$$$         for ( int ir = 1; ir <= total_residue; ++ir ) {
//$$$            in = res(ir);
//$$$            for ( int ia = 1, iae = natoms(in); ia <= iae; ++ia ) {
//$$$               aname = atom_name(ia,in);
//$$$               phobic(ia,ir) = false;
//$$$               if ( aname[1] == 'C' ) phobic(ia,ir) = true;
//$$$            }
//$$$         }

	}


//cj----now do calculations: get the atm_masks by looping over all_atoms x all_atoms
//cj    only do for heavy atoms (skip Hydrogens)
//cj    std::cout << "ir " << ir << " ia " << ia << " in " << in << " natoms " <<
//cj		 natoms(in) << ' ' << atom_name(ia,in) << SS( residue3(ir) ) << std::endl;
	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		int const res_variant_ir = res_variant(ir);
		for ( int ia = 1, li = full_coord.index(1,ia,ir),
		 lfi = fullatom_type.index(ia,in,res_variant_ir),
		 iae = natoms(in,res_variant_ir); ia <= iae; ++ia, ++lfi ) {

			// store the coordinates and atom type, since we will reuse this a bunch
			for ( int m = 1; m <= 3; ++m, ++li ) {
				ic(m) = full_coord[ li ]; // full_coord(m,ia,ir);
			}

			it = fullatom_type[ lfi ]; // fullatom_type(ia,in,res_variant_ir);
			irad = aradii(it) + 1.4f;

//cj          could do this more efficiently if found neighbors first???
//jg         loop over jr > ir, to cover all pair-wise interactions
			for ( int jr = ir; jr <= total_residue; ++jr ) {
				jn = res(jr);
				int const res_variant_jr = res_variant(jr);
				for ( int ja = 1, lj = full_coord.index(1,ja,jr),
				 lfj = fullatom_type.index(ja,jn,res_variant_jr),
				 jae = natoms(jn,res_variant_jr); ja <= jae; ++ja, ++lfj ) {

					for ( int m = 1; m <= 3; ++m , ++lj ) {
						jc(m) = full_coord[ lj ]; // full_coord(m,ja,jr);
					}

					jt = fullatom_type[ lfj ]; // fullatom_type(ja,jn,res_variant_jr);

//cj                getting squared distance between atoms
//jg               (faster than taking sqrt)
					dist_sq = distance_squared(ic,jc);

					if ( dist_sq <= rsq_min(it,jt) ) {

//car ignore all zero distances, either i,j are same atom or it's
//car an invalid pair for some reason...
						if ( dist_sq <= 0.0 ) goto L100;

						jrad = aradii(jt) + 1.4f;
						rij = irad + jrad;

						dist = std::sqrt(dist_sq);

						// account for j overlapping i:

						get_overlap(ic,irad,jc,jrad,dist,olp);
						get_orientation(ic,jc,aphi,theta, dist);

						point = angles(aphi,theta);
						masknum = point * 100 + olp;

						for ( int bb = 1, l = atm_masks.index(bb,ia,ir),
						 m = masks.index(bb,masknum); bb <= nbytes; ++bb, ++l, ++m ) {
							atm_masks[ l ] = bit_or( atm_masks[ l ], masks[ m ] );
//							atm_masks(bb,ia,ir) = bit_or( atm_masks(bb,ia,ir), masks(bb,masknum) );
						}

						// account for i overlapping j:

						get_overlap(jc,jrad,ic,irad,dist,olp);
						get_orientation(jc,ic,aphi,theta, dist);

						point = angles(aphi,theta);
						masknum = point * 100 + olp;

						for ( int bb = 1, l = atm_masks.index(bb,ja,jr),
						 m = masks.index(bb,masknum); bb <= nbytes; ++bb, ++l, ++m ) {
							atm_masks[ l ] = bit_or( atm_masks[ l ], masks[ m ] );
//							atm_masks(bb,ja,jr) = bit_or( atm_masks(bb,ja,jr), masks(bb,masknum) );
						}

//cj        For Patrice's solvation
//          phobej = phobic(ja,jr);

//jo        if ( phobej ) {
//jo          for ( int bb = 1; bb <= nbytes; ++bb ) {
//jo             hpb_masks(bb,ia,ir) = bit_or( hpb_masks(bb,ia,ir), masks(bb,masknum) );
//jo          }
//jo       } else {
//jo          for ( int bb = 1; bb <= nbytes; ++bb ) {
//jo             pol_masks(bb,ia,ir) = bit_or( pol_masks(bb,ia,ir), masks(bb,masknum) );
//jo          }
//jo       }

					} else if ( dist_sq > 289 ) { // 17A^2 = 10A + 2 (1.4 + 2.1)
						goto L110; // this residue is too far away! go to next residue j
					}     // dij_sq <= rij_sq
L100:;
				}        // ja
L110:;
			}           // jr
		}              // ia
	}                 // ir


//-----calculate the gsolt from the atm_masks
	float const four_pi = 4.0f * pi;
	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		int const res_variant_ir = res_variant(ir);
		for ( int ia = 1, lfi = fullatom_type.index(ia,in,res_variant_ir),
		 iae = natoms(in,res_variant_ir); ia <= iae; ++ia, ++lfi ) {

			it = fullatom_type[ lfi ]; // fullatom_type(ia,in,res_variant_ir);
			irad = aradii(it) + 1.4f;

//cj       to get SASA:
//cj       - count the number of 1's
//cj       - figure fraction that they are
//cj       - multiply by 4*pi*r_sqared
			ctr = 0;
			for ( int bb = 1, l = atm_masks.index(bb,ia,ir); bb <= nbytes; ++bb, ++l ) {
				ctr += bit_count[atm_masks[l]];
				//for ( short k = 0; k <= 7; ++k ) {
			    //if ( bit_test( static_cast< short >( atm_masks[ l ] ), k ) ) ++ctr;  // Don't compute what you can look up.
					 // atm_masks(bb,ia,ir)
				//}
			}

			fraction = static_cast< float >( ctr ) / maskbits;
			total_sa = four_pi * ( irad * irad );
//jo         bur = fraction*total_sa;
//jo         a_sa_bur(i) = bur;
			expose = ( 1.0f - fraction ) * total_sa;
//jo         a_sa_exp(i) = expose;

//cj          now categorize
//cj          figure if phobic or mainchain
//            phobei = phobic(ia,ir);
//jo         ismn = mainchain(i);

//jo         if ( phobei ) {
//jo            hpbur += bur;
//jo            hpexp += expose;
//jo            if ( ismn ) {
//jo               mnhpbur += bur;
//jo               mnhpexp += expose;
//jo            } else {
//jo               sdhpbur += bur;
//jo               sdhpexp += expose;
//jo            }
//jo         } else {
//jo            pobur += bur;
//jo            poexp += expose;
//jo            if ( ismn ) {
//jo               mnpobur += bur;
//jo               mnpoexp += expose;
//jo            } else {
//jo               sdpobur += bur;
//jo               sdpoexp += expose;
//jo            }
//jo         }


//cj          for Patrice's solvation
//cj          for hydrophobic buried surface area
//jo         ctr = 0;
//jo         for ( int bb = 1; bb <= nbytes; ++bb ) {
//jo            for ( short k = 0; k <= 7; ++k ) {
//jo               if ( bit_test( static_cast< short >( hpb_masks(bb,ia,ir) ), k ) ) ++ctr;
//jo            }
//jo         }
//jo         fraction = float(ctr)/float(maskbits);
//jo         npsa = fraction*total_sa;

//cj          for polar buried surface area
//jo         ctr = 0;
//jo         for ( int j = 1; j <= nbytes; ++j ) {
//jo            for ( short k = 0; k <= 7; ++k ) {
//jo               if ( bit_test( static_cast< short >( pol_masks(j,ia,ir) ), k ) ) ++ctr;
//jo            }
//jo         }
//jo         fraction = float(ctr)/float(maskbits);
//jo         posa = fraction*total_sa;

//jg            sasa += expose;

//jo         only using solvation and polar contact with non-polar surface area
			gsolt += asp(it) * expose;
			sasa_gsolt +=  expose;
			if ( atomtype_sasapolarity(it) == POLAR ) {
				sasa_polar += expose;
			} else if ( atomtype_sasapolarity(it) == APOLAR ) {
				sasa_apolar += expose;
			}

//jo         genvt -= (asp*npsa);
//jo         npca += npsa;
//jo         poca += posa;
//            if ( phobei ) {
//jo            gsolnp += (asp*expose);
//jo            genvnp -= (asp*npsa);
//jo            npnp += npsa;
//jo            nppo += posa;
//            } else {
//jo            gsolpo += (asp*expose);
//jo            genvpo -= (asp*npsa);
//jo            ponp += npsa;
//jo            popo += posa;
//jo            if ( !hbonded(i) ) {
//jo               genvpof -= (asp*npsa);
//jo               ponpf += npsa;
//jo               ++nponh;
//jo               if ( expose < 0.05 ) {
//jo                  ponpfbur += npsa;
//jo                  ++nponhbur;
//jo               }
//jo               std::cout << I( 3, i ) << ' ' <<
//jo							  A( 3, atmnam(i) ) << ' ' << I( 3, resnum(i) ) << ' ' <<
//jo							  A( 3, resnam(i) ) << ' ' << F( 11, 3, npsa ) << std::endl;
//jo            } else {
//jo               ++npohb;
//jo            }
//            }
		}                  // ia
	}                     // ir

//cj    std::cout << "####### gsolt " << gsolt << std::endl;

//jg      std::cout << "####### sasa " << sasa << std::endl;
//	std::cout << "sasa_gsolt " << sasa_gsolt << std::endl;
	gsolt *= gsolt_orig_weight; // scaling factor
}

bool is_rna_ring( int const ia ){
	if (ia != rna_variables::c1star &&
			ia != rna_variables::c2star &&
			ia != rna_variables::c3star &&
			ia != rna_variables::c4star &&
			ia != rna_variables::c5star) return false;
	return true;
}


//////////////////////////////////////////////////////////////////////////////
/// @ begin calc_sasa
/// @ breif returns sasa of misc
//////////////////////////////////////////////////////////////////////////////
float
calc_sasa()
{
	using namespace param;

	FArray2D_float atom_sasa( MAX_ATOM(), MAX_RES(), 0.0f );
	FArray1D_float rsd_sasa( MAX_RES(), 0.0f );
	float const water_radius = 1.4f;
	bool const heavyatoms_only = false;
	bool const self_chain_only = false;

	calc_per_atom_sasa( atom_sasa, rsd_sasa, water_radius, heavyatoms_only, self_chain_only);

	float sum = 0;
	for (int ii = 1; ii <= MAX_RES(); ++ii) sum += rsd_sasa( ii );

	return sum;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_per_atom_sasa
///
/// @brief
///pb   basically a stripped down copy of the above for diagnostics
///     slower -- no rsq_min stuff
///
/// @detailed
///
/// @param atom_sasa - [in/out]? -
/// @param rsd_sasa - [in/out]? -
/// @param[in]  change_radius -
/// @param[in]  heavyatoms_only - if true, ignore hydrogens in computing SASA
/// @param[in]  self_chain_only - if true, ignore SASA occluded by
///			       chains other than the one harboring the current atom
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calc_per_atom_sasa(
	FArray2Da_float atom_sasa,
	FArray1Da_float rsd_sasa,
	float const probe_radius, // input
	bool const heavyatoms_only, // jk if true, ignore hydrogens and waters in computing SASA
	bool const self_chain_only, // jk if true, ignore SASA occluded by
	bool const rna_ring // rhiju
			// chains other than the one harboring the current atom
			// (ie. compute SASA in the "unbound" state)
)
{
	using namespace aaproperties_pack;
	using namespace docking;
	using namespace fullatom_ex_props;
	using namespace fullatom_sasa;
	using namespace misc;
	using namespace param;
	using namespace numeric::constants::f;

	atom_sasa.dimension( MAX_ATOM(), MAX_RES() );
	rsd_sasa.dimension( MAX_RES() );

	// jk initialize to -1 for "not computed"
	atom_sasa = -1.0;

// local:
	FArray3D_ubyte atm_masks( nbytes, MAX_ATOM()(), MAX_RES()() );
	 // full mask for atoms

	int in,it,jn,jt,ctr,olp,aphi,theta,point,masknum;
	FArray1D_float ic( 3 );
	FArray1D_float jc( 3 );
	float dist,rij,irad,jrad,fraction,total_sa,expose;

	float const big_distance = 10.0 + 2 * ( probe_radius + 2.1 );
	 // same logic as above for escaping if two residues are too far apart

	bool const USE_BIG_POLAR_H = get_use_big_polar_h();

	// jk ensure we have a legit multi_chain system if we want self_chain_only
	if ( self_chain_only && ( ! files_paths::multi_chain || part_begin(1)==0) ) {
		std::cout << "Error! calc_per_atom_sasa for self_chain requested, ";
		std::cout << "but multi_chain not properly set up" << std::endl;
		if ( ! files_paths::multi_chain ) {
			std::cout << "The multi_chain variable is false" << std::endl;
		} else {
			std::cout << "part_begin(1) is zero" << std::endl;
		}
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	input_sasa_dats(); //returns if already done

//-----initialize atm_masks to zero
	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		int iae;
		if ( heavyatoms_only ) {
			iae = nheavyatoms(in,res_variant(ir));
		} else {
			iae = natoms(in,res_variant(ir));
		}
		for ( int ia = 1; ia <= iae; ++ia ) {
//cj   first zero mask for all atoms
			for ( int bb = 1; bb <= nbytes; ++bb ) {
				atm_masks(bb,ia,ir) = 0;
			}
		}                  // ia
	}                     // ir

//cj----now do calculations: get the atm_masks by looping over all_atoms x all_atoms
	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		int iae;
		if ( heavyatoms_only ) {
			iae = nheavyatoms(in,res_variant(ir));
		} else {
			iae = natoms(in,res_variant(ir));
		}

		int jres_last = total_residue;
		if ( self_chain_only ) {
			// jk if ir is on chain 1, stop jr at residue part_end(1)
			// jk note: the fact that jr >= ir takes care of case of ir on chain 2
			if ( ir <= part_end(1) ) jres_last = part_end(1);
		}

		for ( int ia = 1; ia <= iae; ++ia ) {

			if (rna_ring && !is_rna_ring( ia )) continue;

			// store the coordinates and atom type, since we will reuse this a bunch
			for ( int m = 1; m <= 3; ++m ) {
				ic(m) = full_coord(m,ia,ir);
			}

			it = fullatom_type(ia,in,res_variant(ir));

			irad = aradii(it) + probe_radius;
			if ( ( it == 22 || it == 25 ) && USE_BIG_POLAR_H )
				irad = 1.08 + probe_radius;

			int jr_first = ir;
			if (rna_ring) jr_first = 1;
			for ( int jr = jr_first; jr <= jres_last; ++jr ) {
				jn = res(jr);

				int jae;
				if ( heavyatoms_only ) {
					jae = nheavyatoms(jn,res_variant(jr));
				} else {
					jae = natoms(jn,res_variant(jr));
				}
				for ( int ja = 1; ja <= jae; ++ja ) {
					for ( int m = 1; m <= 3; ++m ) {
						jc(m) = full_coord(m,ja,jr);
					}

					jt = fullatom_type(ja,jn,res_variant(jr));

					jrad = aradii(jt) + probe_radius;
					if ( ( jt == 22 || jt == 25 ) && USE_BIG_POLAR_H )
						jrad = 1.08 + probe_radius;

					dist = std::sqrt( vec_dist2(ic,jc) );

					if ( dist <= irad + jrad ) {

						if ( dist <= 0.0 ) goto L200;
						rij = irad + jrad;

						// account for j overlapping i:
						// jk Note: compute the water SASA, but DON'T allow the water
						// jk to contribute to the burial of non-water atoms
						if ( jt != 26 ) {
							get_overlap(ic,irad,jc,jrad,dist,olp);
							get_orientation(ic,jc,aphi,theta, dist);
							point = angles(aphi,theta);
							masknum = point * 100 + olp;
							for ( int bb = 1, l = atm_masks.index(bb,ia,ir),
											m = masks.index(bb,masknum); bb <= nbytes; ++bb, ++l, ++m ) {
								atm_masks[ l ] = bit_or( atm_masks[ l ], masks[ m ] );
//	  	  					atm_masks(bb,ia,ir) = bit_or(atm_masks(bb,ia,ir),masks(bb,masknum));
							}
						}

						if (rna_ring) continue;
						// account for i overlapping j:
						// jk Note: compute the water SASA, but DON'T allow the water
						// jk to contribute to the burial of non-water atoms
						if ( it != 26 ) {
							get_overlap(jc,jrad,ic,irad,dist,olp);
							get_orientation(jc,ic,aphi,theta, dist);
							point = angles(aphi,theta);
							masknum = point * 100 + olp;
							for ( int bb = 1, l = atm_masks.index(bb,ja,jr),
											m = masks.index(bb,masknum); bb <= nbytes; ++bb, ++l, ++m ) {
								atm_masks[ l ] = bit_or( atm_masks[ l ], masks[ m ] );
//							    atm_masks(bb,ja,jr) = bit_or(atm_masks(bb,ja,jr),masks(bb,masknum));
							}
						}

					} else if ( dist > big_distance ) { // = 10A + 2 (probe_radius + 2.1)
						goto L210; // this residue is too far away! go to next residue j
					}         // dij_sq <= rij_sq

L200:;
				}            // ja
L210:;
			}               // jr
		}                  // ia
	}                     // ir


//-----calculate the residue and atom sasa
	float const four_pi = 4.0f * pi;
	for ( int ir = 1; ir <= total_residue; ++ir ) {
		in = res(ir);
		int const res_variant_ir = res_variant(ir);
		int iae;
		if ( heavyatoms_only ) {
			iae = nheavyatoms(in,res_variant_ir);
		} else {
			iae = natoms(in,res_variant_ir);
		}
		rsd_sasa(ir) = 0.0;
		for ( int ia = 1, lfi = fullatom_type.index(ia,in,res_variant_ir);
			ia <= iae; ++ia, ++lfi ) {

			it = fullatom_type[ lfi ]; // fullatom_type(ia,in,res_variant_ir);

			atom_sasa(ia,ir) = 0.0;

			irad = aradii(it) + probe_radius;

//cj       to get SASA:
//cj       - count the number of 1's
//cj       - figure fraction that they are
//cj       - multiply by 4*pi*r_sqared
			ctr = 0;
			for ( int bb = 1, l = atm_masks.index(bb,ia,ir); bb <= nbytes; ++bb, ++l ) {
				ctr += bit_count[atm_masks[ l ]]; // atm_masks(bb,ia,ir)
			}

			fraction = static_cast< float >( ctr ) / maskbits;
			total_sa = four_pi * ( irad * irad );
			expose = ( 1.0f - fraction ) * total_sa;
			atom_sasa(ia,ir) = expose;
			// jk Water SASA doesn't count toward the residue's SASA
			if ( it != 26 ) rsd_sasa(ir) += expose;

		}                  // ia
	}                     // ir
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ex_gsolt
///
/// @brief
///cj    simple function that returns gsolt
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_ex_gsolt()
{
	using namespace fullatom_sasa;

	return gsolt;
}


////////////////////////////////////////////////////////////////////////////////
///vatsan    To compute the solvent accessible surface area
////////////////////////////////////////////////////////////////////////////////


float
get_ex_sasa_gsolt()
{
	using namespace fullatom_sasa;
	return sasa_gsolt;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_min_rsq
///
/// @brief
///
/// @detailed
///     this function pre-tabulates the minimun squared-distances between
///     atoms of all types.   this should speed up the calculation
///     tremendously, by allowing us not to take square roots in the main
///     loop of calc_approx_sasa when not necessary.  jjg 2/4/2
///
/// @param  rsq_min - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_min_rsq( FArray2D_float & rsq_min )
{
	using namespace fullatom_ex_props;
	using namespace param;

	for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
		float const aradii_i_28 = aradii(i) + 2.8;
		for ( int j = i; j <= e; ++j ) {
			float const aradii_sum = aradii_i_28 + aradii(j);
			rsq_min(j,i) = rsq_min(i,j) = aradii_sum * aradii_sum;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////
void
calc_gsolt( pose_ns::Pose & pose ){

	using namespace pose_ns;

	pose.copy_to_misc();

	calc_aprox_sasa();
	pose.set_0D_score( GSOLT, get_ex_gsolt() );
	pose.set_0D_score( SASA,  get_ex_sasa_gsolt() );

}
