// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 17813 $
//  $Date: 2007-10-16 16:49:26 -0400 (Tue, 16 Oct 2007) $
//  $Author: johnk $


// Rosetta Headers
#include "fullatom_setup.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "design.h"
#include "design_structure.h"
#include "dunbrack_pack.h"
#include "Dunbrack4D.h"
#include "Dunbrack5D.h"
#include "etable.h"
#include "etable_manager.h"
#include "files_paths.h"
#include "gb_elec.h"
#include "hbonds_ns.h"
#include "ligand.h"
#include "namespace_fa_lj_rep_slope_setting.h"
#include "namespace_fullatom_init.h"
#include "namespace_lj_etable_settings.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pKa_mode.h"
#include "planes.h"
#include "RotamerOptions.h"
#include "read_paths.h"
#include "water_ns.h"
#include "enzyme.h"
#include "weights_manager.h"
//KMa phospho_ser
#include "add_pser.h"

// ObjexxFCL Headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <list>
#include <sstream>
#include <map>
#include <iterator>




namespace fullatom_setup_ns {
	float mod_hhrep_height   = 1.0;
	float mod_hhrep_width    = 1.0;
	float mod_hhrep_center   = 2.8;
	float mod_hhrep_exponent = 4.0;
	float smooth_etable_ljweight = 1.16;
	float smooth_etable_solvweight = 1.5;
}



//////////////////////////////////////////////////////////////////////////////
/// @begin set_fa_lj_rep_slope
///
/// @brief
///
/// @detailed
///
/// @param setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_fa_lj_rep_slope( std::string const & setting )
{
	fa_lj_rep_slope_setting::fa_lj_rep_slope = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_fullatom
///
/// @brief initializes fullatom data structures
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_fullatom()
{
	using namespace design;
	using namespace fa_lj_rep_slope_setting;
	using namespace param_pack;

	if ( fullatom_init::init ) return;
	std::cout << "initializing full atom functions" << std::endl;

	//jjh Set up the packer weights database
	InitializePackerWeightsDB();

	//lin reset packer weight set from file
	if( false && truefalseoption("packer_wtsfile") ) {
		//lin input: design_matrix define the designed sequence profile
		//lin input: weight set take the standard packer weights
		//lin input:            or take from the file if the file exist
		PackerWeights design_weight=RetrieveWeights( PW_STANDARD ); //default the PW_STANDARD
		design_weight.read_weight_file(stringafteroption("packer_wtsfile"));
		param_pack::pack_wts = design_weight;
		//param_pack::pack_wts.print();
	}

	if ( truefalseoption( "dna_weights" ) ) {
		std::cout << "applying dna_weights" << std::endl;
		param_pack::packer_logical::dna_weights = true;
	}

	//mj set up soft radii if soft_rep option was used
	if ( truefalseoption( "soft_rep" ) ) {
		std::cout << "using soft potentials as requested by the -soft_rep flag" << std::endl;
		param_pack::packer_logical::soft_rep = true;
	}

	//jjh Check for soft_rep with design weights
	if ( truefalseoption( "soft_rep_design" ) ) {
		std::cout << "using soft potentials as requested by the -soft_rep_design flag" << std::endl;
		param_pack::packer_logical::soft_rep_design = true;
		param_pack::packer_logical::soft_rep = true;
	}

	// xh set up soft repulsion and weights for all fold proteins, if use_aw was used
	if ( truefalseoption( "use_aw") ) {
    std::cout << "using soft potentials for all proteins as requested by the -use_aw flag" << std::endl;
		param_pack::packer_logical::use_aw = true;
  }

	// xh set up soft repulsion and weights for beta proteins, if use_bw was used
	if ( truefalseoption( "use_bw") ) {
    std::cout << "using soft potentials for beta proteins as requested by the -use_bw flag" << std::endl;
		param_pack::packer_logical::use_bw = true;
  }

	//jjh Check for use_packing_etables_always flag
	if ( truefalseoption( "use_packing_etables_always" ) ) {
		std::cout << "using soft_rep everywhere (scoring, minimizations, output PDBs, etc.)" << std::endl;
		param_pack::packer_logical::use_packing_etables_always = true;
	}

	//sheffler these weights are coupled to smooth etables
	if (truefalseoption("smooth_etable")){
		std::cout << "using smooth etable weights" << std::endl;
		param_pack::packer_logical::smooth_etable = true;
	}
	//sheffler these weights are coupled to smooth etables
	if (truefalseoption("reduce_templates")){
		std::cout << "using new templates from reduce" << std::endl;
		param_pack::packer_logical::reduce_templates = true;
	}

	//mj set up soft radii if gen_born option was used
	if ( truefalseoption( "gen_born" ) ) {
		std::cout << "using generalized born radii as requested by the -gen_born flag" << std::endl;
		param_pack::packer_logical::gen_born = true;
		//rh Undocumented gen_born option that should be command line accessible
		if ( truefalseoption( "use_simple_elec" ) ) {
			std::cout << "using simple electrostatics by the -use_simple_elec flag" << std::endl;
			gb_set_use_simple_electrostatics(true);
		}
		//rh Turn off intra-residue gen_born electrostatics
		if ( truefalseoption( "no_intra_res" ) ) {
			std::cout << "not calculating intra residue generalized born electrostatics by the -no_intra_res flag" << std::endl;
			gb_set_use_no_intra_res(true);
		}

	}

	if ( get_pH_packing_flag() ) calc_dGprotonation_table();

	//bk small_radii option
	if ( truefalseoption( "small_radii" ) ) {
		std::cout << "using smaller radii as requested by the -small_radii flag" << std::endl;
		param_pack::packer_logical::small_radii = true;
	}

	//mj rest of initialization
	stringafteroption( "fa_lj_rep_slope", fa_lj_rep_slope, fa_lj_rep_slope );

	if ( ! param_pack::packer_logical::use_packing_etables_always ) {
		setCurrentEtable( ETAB_STANDARD );
	} else if ( param_pack::packer_logical::soft_rep ) {
		setCurrentEtable( ETAB_SOFT_REP );
	} else if ( param_pack::packer_logical::small_radii ) {
		setCurrentEtable( ETAB_SMALL_RADII );
	} else if ( param_pack::packer_logical::use_aw ) {
		setCurrentEtable( ETAB_AW );
	} else if ( param_pack::packer_logical::use_bw ) {
		setCurrentEtable( ETAB_BW );
	}

	if (files_paths::skip_dun) {
		std::cout << "WARNING! WARNING! Skipping bbdep, avgE, table readin." << std::endl;
	} else {
		setup_packer();
	}

	fullatom_init::init = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_dun
///
/// @brief read in dunbrack's rotamer set and setup rotamer data structures
///
/// @detailed
///
/// reads in the dunbrack backbone dependent rotamer set and creates a
/// variety of data structures to store and access the rotamer data,
/// currently using not-yet-published updated rotamer calculation method
/// from dunbrack. This new method primarily adds some ideas from the
/// richardson's penultimate paper to the original method described in
/// Dunbrack & Cohen 97'. Here is Dunbrack's description of the changes
/// made since the 97' version:
///
///          """ A couple of points:
///
///          1) these have been processed the way Jane Richardson does hers --
///          no B-factors over 30, removing sidechains with clashes from the
///          data set, building hydrogens and flipping amides and histidines
///          if necessary.
///
///          2) Asn and His now have 360 degrees rotation about chi2. Asn has
///          6 rotamers for chi2 to represent the space more accurately; same
///          for Gln chi3. [ctsa - note that "same" here refers to the 360
///          degrees, not the 6 rotamers -- gln chi3 has 4 rotamers now and
///          his chi2 has 3]
///
///          3) the way I estimate the variance is completely different. There
///          is now a prior distribution on the variance, instead of using a
///          point value, so even when there isn't much data I have reasonable
///          estimates.  """
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// original function written by bk, updated to new dunbrack rotamer
/// set 11-2002 with significant changes to the chi3,4 treatment
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_dun()
{
	using namespace aaproperties_pack;
	using namespace dunbrack_pack;
	using namespace param;
	using namespace param_aa;

	//Objexx:SGM Set number of phi & psi angle bins here to prevent array allocation unless needed
	dunbrack_pack::nbins() = 36;

	// ctsa - build map from aa,rot1234 to rotamer lookup index (packed_rotno)
	int packed_rotno = 0;
	int packed_rotno_max = 0;
	int aa_base_packed_rotno;
	std::list< int > dun_prob_1; // Probability==1 list
	for ( int aan = 1; aan <= MAX_AA(); ++aan ) {
		//dr this will be ok for nnaa with less than 4 chi as theses will be in the DRL
		//dr otherwise we will have to do somthing different
		if ( is_protein(aan) || is_nonnatural(aan) ) {
			aa_base_packed_rotno = packed_rotno;
			for ( int i0 = 0; i0 <= rot_per_chi; ++i0 ) {
				for ( int i1 = 0; i1 <= rot_per_chi; ++i1 ) {
					for ( int i2 = 0; i2 <= rot_per_chi; ++i2 ) {
						for ( int i3 = 0; i3 <= rot_per_chi; ++i3 ) {
							packed_rotno = aa_base_packed_rotno + 1 + i0 +(rot_per_chi+1)*(i1 + (rot_per_chi+1)*(i2 + (rot_per_chi+1)*i3));
							aa_rotno_to_packedrotno(aan,i0,i1,i2,i3) = packed_rotno;
							packed_rotno_max = std::max( packed_rotno_max, packed_rotno );
							if ( nchi(aan,1) <= 0 ) goto L230;
							if ( nchi(aan,1) <= 1 ) goto L225;
							if ( nchi(aan,1) <= 2 ) goto L220;
							if ( nchi(aan,1) <= 3 ) goto L215;
						}
					L215:; // chi3
					}
					if ( aan == aa_gln ) {
						for ( int i2 = (rot_per_chi+1); i2 <= max_rot_per_chi3; ++i2 ) {
							packed_rotno = aa_base_packed_rotno +1 + i0 + (rot_per_chi+1)*(i1+(rot_per_chi+1)*i2);
							aa_rotno_to_packedrotno(aan,i0,i1,i2,0) = packed_rotno;
							packed_rotno_max = std::max( packed_rotno_max, packed_rotno );
						}
					}
				L220:; // chi2
				}
				if ( aan == aa_asn ) {
					for ( int i1 = (rot_per_chi+1); i1 <= max_rot_per_chi2; ++i1 ) {
						packed_rotno = aa_base_packed_rotno +1 + i0 + (rot_per_chi+1)*(i1);
						aa_rotno_to_packedrotno(aan,i0,i1,0,0) = packed_rotno;
						packed_rotno_max = std::max( packed_rotno_max, packed_rotno );
					}
				}
			L225:; // chi1
			}
		L230:; // chi0
			if ( aan == aa_gly || aan == aa_ala ) {
				dun_prob_1.push_back( packed_rotno ); // Add to list of prob==1 entries
				for ( int i1 = 1; i1 <= 36; ++i1 ) {
					for ( int i0 = 1; i0 <= 36; ++i0 ) {
						highest_dun_prob(i0,i1,aan) = 1.0;
					}
				}
			}
		} // end if stament for nonnaturals and naturals
	} // end for loop looping through MAX_AA()

	//Objexx:SGM Allocates and reinitializes dunbrack_pack arrays
	dunbrack_pack::max_packed_rotno() = packed_rotno_max;

	// Assign prob==1 entries in dun_prob now that it has been allocated
	for ( std::list< int >::const_iterator i = dun_prob_1.begin(),
	 ie = dun_prob_1.end(); i != ie; ++i ) {
		int const packed_rotno_1 = *i;
		for ( int i1 = 1; i1 <= 36; ++i1 ) {
			for ( int i0 = 1; i0 <= 36; ++i0 ) {
				dun_prob(i0,i1,packed_rotno_1) = 1.0;
			}
		}
	}
	dun_prob_1.clear(); // Don't need it anymore

	int countrot = 0;
	int prevphi = 0;
	int prevpsi = 0;
	float perctot = 0.0;
	int aan = -1;

//KMa phospho_ser 2006-01

	//  read in bbdep rotamer data
	std::string  bbdep_rot_file;

	if ( add_pser () 	)
	{
		std::cout << "read rotamer data for 20 genuine amino-acids plus SEP (phospho-serine)"  << std::endl;
		 bbdep_rot_file = "bb21sdep06.Jan.sortlib";

	}
	else 	bbdep_rot_file =  "bbdep02.May.sortlib" ;

	utility::io::izstream & iunit( open_data_file( bbdep_rot_file ) );


	char line[110]; //S&S std::string line;
	int phi, psi;
	float perc;
	char three[4];
	short rotno;
	FArray1D_byte tmp_rotno( MAX_CHI );
	FArray1D_float tmp_chi( MAX_CHI );
	FArray1D_float tmp_sd( MAX_CHI );
	int scan_count;
	while ( iunit ) {
		iunit.getline( line, 110 ); //S&S getline( iunit, line );
		if ( iunit.eof() ) {
			goto L425;
		} else if ( iunit.fail() ) {
			goto L400;
		}
		std::sscanf( line, "%3s", three );

		 // Naughty cast but efficient
		std::sscanf( line+5, "%4d", &phi );
		if ( phi > 175 ) goto L299;
		 // ctsa - skip phi,psi == 180 b/c this is a copy of -180
		std::sscanf( line+10, "%4d", &psi );
		if ( psi > 175 ) goto L299;
		 // ctsa - skip phi,psi == 180 b/c this is a copy of -180
		std::sscanf( line+24, "%1hd", &rotno ); tmp_rotno(1) = rotno;
		std::sscanf( line+26, "%1hd", &rotno ); tmp_rotno(2) = rotno;
		std::sscanf( line+28, "%1hd", &rotno ); tmp_rotno(3) = rotno;
		std::sscanf( line+30, "%1hd", &rotno ); tmp_rotno(4) = rotno;
		std::sscanf( line+33, "%8f", &perc );
		std::sscanf( line+43, "%6f", &tmp_chi(1) );
		std::sscanf( line+51, "%6f", &tmp_chi(2) );
		std::sscanf( line+59, "%6f", &tmp_chi(3) );
		std::sscanf( line+67, "%6f", &tmp_chi(4) );
		std::sscanf( line+77, "%6f", &tmp_sd(1) );
		std::sscanf( line+85, "%6f", &tmp_sd(2) );
		std::sscanf( line+93, "%6f", &tmp_sd(3) );
		scan_count = std::sscanf( line+101, "%6f", &tmp_sd(4) );

		if ( scan_count == EOF ) goto L400; // Read problem

// This is the Slick & Slow (S&S) stream-based method that is too slow for large
// files like this one, at least under the GCC 3.3.1 stream implementation.
// It should be retried on future releases and target compilers because there is
// no reason it cannot be competitive with good optimization and inlining.
// If this is used the <cstdio> can be removed and the 2 //S&S-commented
// statements above used in place of the lines they trail.
//
//		{ // Scope
//			std::istringstream line_stream( line );
//			line_stream >> bite( three ) >> skip( 2 ) >>
//			 bite( 4, phi ) >> skip( 1 ) >> bite( 4, psi );
//
//			// ctsa - skip phi,psi == 180 b/c this is a copy of -180
//			if ( psi > 175 || phi > 175 ) goto L299;
//
//			line_stream >> skip( 10 ) >>
//			 bite( 1, tmp_rotno(1) ) >> skip( 1 ) >>
//			 bite( 1, tmp_rotno(2) ) >> skip( 1 ) >>
//			 bite( 1, tmp_rotno(3) ) >> skip( 1 ) >>
//			 bite( 1, tmp_rotno(4) ) >> skip( 2 ) >>
//			 bite( 8, perc ) >> skip( 2 ) >>
//			 bite( 6, tmp_chi(1) ) >> skip( 2 ) >>
//			 bite( 6, tmp_chi(2) ) >> skip( 2 ) >>
//			 bite( 6, tmp_chi(3) ) >> skip( 2 ) >>
//			 bite( 6, tmp_chi(4) ) >> skip( 4 ) >>
//			 bite( 6, tmp_sd(1) ) >> skip( 2 ) >>
//			 bite( 6, tmp_sd(2) ) >> skip( 2 ) >>
//			 bite( 6, tmp_sd(3) ) >> skip( 2 ) >>
//			 bite( 6, tmp_sd(4) );
//			if ( line_stream.eof() ) {
//				goto L400;
//			} else if ( line_stream.fail() ) {
//				goto L400;
//			}
//		}

		phi = ((phi+190)/10);
		psi = ((psi+190)/10);

		if ( prevphi != phi || prevpsi != psi ) {
			num_from_name( three, aan );
			countrot = 1;
			prevphi = phi;
			prevpsi = psi;
			perctot = 0.0;
		}

		// ctsa -there's still only one chi for proline, but the new
		//  rotamer library marks proline chi2 with a (constant)
		//  value of 1 instead of 0. This hacks it back to 0.
		if ( aan == aa_pro ) {
			tmp_rotno(2) = 0;
		}

		packed_rotno = aa_rotno_to_packedrotno(aan,tmp_rotno(1),tmp_rotno(2),
		 tmp_rotno(3),tmp_rotno(4));
		{ //Objexx:SGM Set space-saving Dunbrack4/5D array values
			int const nchi_aa = nchi(aan,1);
			dun_chi.set( phi, psi, packed_rotno, nchi_aa, tmp_chi );
			dun_sd.set( phi, psi, packed_rotno, nchi_aa, tmp_sd );
			dun_group_rotno_to_rotno.set( phi, psi, aan, countrot, tmp_rotno );
		}

//Objexx:SGM Replaced by above
//		for ( int i = 1; i <= 4; ++i ) {
//			dun_chi(phi,psi,packed_rotno,i) = tmp_chi(i);
//			dun_sd(phi,psi,packed_rotno,i) = tmp_sd(i);
//			dun_group_rotno_to_rotno(phi,psi,aan,countrot,i) = tmp_rotno(i);
//		}

		dun_prob(phi,psi,packed_rotno) = perc;

		if ( countrot == 1 ) {
			highest_dun_prob(phi,psi,aan) = perc;
		}

		perctot += perc;
		++countrot;

L299:; // skip line in bbdep rotamer file
	}

L400:
	std::cout << "STOP:: error reading bbdep rotamer data" << iunit.filename() << std::endl;
	iunit.close();
	iunit.clear();
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

L425:
	iunit.close();
	iunit.clear();

	// Remove excess capacity from arrays
	dun_chi.shrink();
	dun_sd.shrink();
}



////////////////////////////////////////////////////////////////////////////////
/// @begin setup_exrot_db
///
/// @brief read and parse exdb file
///
/// @detailed
///
/// if the -exdb option is used, then strained rotamer types are read
/// in from a text file and stored in the structure: total_chi_set_db
///
/// @global_read knick-knacks
///
/// @global_write
///
/// dunbrack_pack.h: total_chi_set_db
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_exrot_db()
{

	//  ctsa - read in extra rotamer set from file

	using namespace aaproperties_pack;
	using namespace design;
	using namespace dunbrack_pack;
	using namespace files_paths;
	using namespace param;

	if ( !active_rotamer_options.exrotset_from_db ) return;

	// local
	int tmp_int;
	int aa;
	FArray1D_float tmp_chi( MAX_CHI );
	std::string line;

  utility::io::izstream exrot_file_x;

	exrot_file_x.open( exrot_file );
	if ( !exrot_file_x ) {
		std::cout << "STOP: unable to find extra rot file: " <<
		 exrot_file_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	while ( true ) {
		exrot_file_x >> bite( line ) >> skip;
		if ( exrot_file_x.eof() ) {
			goto L100;
		} else if ( exrot_file_x.fail() ) {
			goto L105;
		}

		// skip comment lines
		if ( line[0] == '#' ) goto L115;

		{ // Scope
			std::istringstream line_stream( line );
			line_stream >> bite( 3, tmp_int ) >>
			 skip( 1 ) >> bite( 2, aa ) >>
			 skip( 1 ) >> bite( 5, tmp_chi(1) ) >>
			 skip( 1 ) >> bite( 5, tmp_chi(2) ) >>
			 skip( 1 ) >> bite( 5, tmp_chi(3) ) >>
			 skip( 1 ) >> bite( 5, tmp_chi(4) );
			if ( line_stream.eof() ) {
				goto L100;
			} else if ( line_stream.fail() ) {
				goto L105;
			}
		}

		for ( int i = 1; i <= MAX_CHI; ++i ) {
			// assume variant 1
			if ( i > nchi(aa,1) ) {
				if ( tmp_chi(i) != 0. ) {
					std::cout << "ABORT: invalid extra rot database file" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}

		(total_chi_set_db[aa]).push_back(tmp_chi);

L115:; // skip comment lines
	}

L105:
	exrot_file_x.close();
	exrot_file_x.clear();
	std::cout << "ABORT: invalid extra rotamer database" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

L100:
	exrot_file_x.close();
	exrot_file_x.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_packer
///
/// @brief setup fullatom-mode datastructures
///
/// @detailed
///
/// read and setup dunbrack rotamer data, residue pair data, general
/// residue statistics and precalculate some potentials which are
/// stored in fast lookup tables.
///
/// @global_read
///
/// @global_write
///
/// pdbstatistics_pack.h: paircorr
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_packer()
{
//bk setup variables for bkpacker
	using namespace files_paths;
	using namespace design;
	using namespace param;
	using namespace param_aa;
	using namespace pdbstatistics_pack;

	int aa;
	int rnbin;
	int aa1,aa2,e1,e2,r12bin,nb;
	float pairprob,probability;

	setup_dun();
	setup_bbind_dun();
	read_plane_table();

	active_rotamer_options.setup_exflags();

	setup_exrot_db();

//bk read in statistics on the amino acids
	utility::io::izstream & pdb_stream( open_data_file( "pdbpairstats_fine" ) );

	for ( int i = 1; i <= 20*20*2*2*5; ++i ) {
		pdb_stream >> bite( 2, aa1 ) >> skip( 1 ) >>
		 bite( 2, aa2 ) >> skip( 1 ) >>
		 bite( 2, e1 ) >> skip( 1 ) >>
		 bite( 2, e2 ) >> skip( 1 ) >>
		 bite( 2, r12bin ) >> skip( 1 ) >>
		 bite( 12, pairprob ) >> skip;
		paircorr( aa1, aa2, e1, e2, r12bin ) = pairprob;
//KMa phospho_ser 2006-01
		if ( add_pser () 	)
		{
		 if(aa1==aa_ser) paircorr( aa_sep, aa2, e1, e2, r12bin ) = pairprob;
		 if(aa2==aa_ser) paircorr( aa1, aa_sep, e1, e2, r12bin ) = pairprob;
		 if(aa1==aa_ser && aa2==aa_ser) paircorr( aa_sep, aa_sep, e1, e2, r12bin ) = pairprob;
		}

	}
	pdb_stream.close();
	pdb_stream.clear();

//bk one interesting property of the pair statistics are that they favor unlike charges being
//bk near each other, but only very close range interactions between like charges are
//bk disfavored.  This is presumably because like charged residues frequently come togethor to
//bk bind charged ligands.  When performing protein design in the absence of extra ligands, it is
//bk therefore probably more correct to penalize like charges being near each other.
//bk The following section makes it unfavorable to have like charged residues near each other
//bk if the -use_electrostic_repulsion flag is turned on.  The penalty is set to be roughly equal
//bk but opposite to the favorable energy given to unlike charges.

	if ( use_electrostatic_repulsion ) {
		for ( int e1 = 1; e1 <= 2; ++e1 ) {
			for ( int e2 = 1; e2 <= 2; ++e2 ) {
				paircorr(aa_asp,aa_asp,e1,e2,1) = 0.3;         // 3-4.5 angstroms
				paircorr(aa_asp,aa_asp,e1,e2,2) = 0.5;         // 4.5-6 angstroms
				paircorr(aa_asp,aa_asp,e1,e2,3) = 0.75;        // 6-8 angstroms

				paircorr(aa_asp,aa_glu,e1,e2,1) = 0.3;
				paircorr(aa_asp,aa_glu,e1,e2,2) = 0.5;
				paircorr(aa_asp,aa_glu,e1,e2,3) = 0.75;

				paircorr(aa_glu,aa_asp,e1,e2,1) = 0.3;
				paircorr(aa_glu,aa_asp,e1,e2,2) = 0.5;
				paircorr(aa_glu,aa_asp,e1,e2,3) = 0.75;

				paircorr(aa_glu,aa_glu,e1,e2,1) = 0.3;
				paircorr(aa_glu,aa_glu,e1,e2,2) = 0.5;
				paircorr(aa_glu,aa_glu,e1,e2,3) = 0.75;

				paircorr(aa_lys,aa_lys,e1,e2,1) = 0.3;
				paircorr(aa_lys,aa_lys,e1,e2,2) = 0.5;
				paircorr(aa_lys,aa_lys,e1,e2,3) = 0.75;

				paircorr(aa_arg,aa_arg,e1,e2,1) = 0.3;
				paircorr(aa_arg,aa_arg,e1,e2,2) = 0.5;
				paircorr(aa_arg,aa_arg,e1,e2,3) = 0.75;

				paircorr(aa_arg,aa_lys,e1,e2,1) = 0.3;
				paircorr(aa_arg,aa_lys,e1,e2,2) = 0.5;
				paircorr(aa_arg,aa_lys,e1,e2,3) = 0.75;

				paircorr(aa_lys,aa_arg,e1,e2,1) = 0.3;
				paircorr(aa_lys,aa_arg,e1,e2,2) = 0.5;
				paircorr(aa_lys,aa_arg,e1,e2,3) = 0.75;
			}
		}
	}

//bk read in the intrinsic probabilities for the amino acids
	utility::io::izstream & Paa_stream( open_data_file( "Paa" ) );

	for ( int i = 1; i <= 20; ++i ) {
		Paa_stream >> skip( 4 ) >> bite( 9, Paa(i) ) >> skip;
	}
	Paa_stream.close();
	Paa_stream.clear();
//KMa phospho_ser 2006-01
	if ( add_pser () )
		Paa(aa_sep)=Paa(aa_ser) ;

//bk read in P(aa|n) where n is number of neighbors
	utility::io::izstream & Paa_n_stream( open_data_file( "Paa_n" ) );
	for ( int i = 1; i <= MAX_AUTH_AA*max_neighbor_bins; ++i ) {
		Paa_n_stream >> bite( 2, aa ) >> skip( 1 ) >>
		 bite( 2, rnbin ) >> skip( 1 ) >>
		 bite( 9, probability ) >> skip;
		Paa_n(aa,rnbin) = probability;
	}
	Paa_n_stream.close();
	Paa_n_stream.clear();

	//KMa phospho_ser 2006-01
	if ( add_pser () )
		for ( int i = 1; i <= max_neighbor_bins; ++i )
			Paa_n (aa_sep,i) = Paa_n (aa_ser,i) ;

//bk read in Paa_pp
	utility::io::izstream & Paa_pp_stream( open_data_file( "Paa_pp" ) );

	for ( int i = 1; i <= max_phipsi_bins; ++i ) {
		for ( int j = 1; j <= max_phipsi_bins; ++j ) {
			for ( int k = 1; k <= MAX_AUTH_AA; ++k ) {
				Paa_pp_stream >> skip( 30 ) >> bite( 7, Paa_pp(i,j,k) ) >> skip;
				//bk if probability equals zero, set it to a small nonzero number
				if ( Paa_pp(i,j,k) <= 0.001 ) {
					Paa_pp(i,j,k) = 0.001;
				}
				//KMa phospho_ser 2006-01
				if ( add_pser () )
					if (k==aa_ser)
						Paa_pp(i,j,aa_sep)=Paa_pp(i,j,aa_ser);
			}
		}
	}
	Paa_pp_stream.close();
	Paa_pp_stream.clear();

//bk read in average energies of different amino acids
	utility::io::izstream & avgE_stream( open_data_file( "avgE_from_pdb" ) );
	avgE_stream >> skip;
	while ( avgE_stream ) {
		avgE_stream >>
			bite( 2, aa ) >> skip( 1 ) >>
			bite( 2, nb ) >> skip( 1 );
		if ( avgE_stream ) {
			avgE_stream >>
				bite( 6, atr_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, rep_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, tlj_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, sol_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, pair_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, one_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, hb_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, dun_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, intrares_avg(aa,nb) ) >> skip( 1 ) >>
				bite( 6, tot_avg(aa,nb) ) >> skip;
		}
	}
	avgE_stream.close();
	avgE_stream.clear();

//bk the avgE table starts at nb = 5, set nb 1-4 equal to nb = 5
	for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
		for ( int nb = 1; nb <= 4; ++nb ) {
			atr_avg(aa,nb) = atr_avg(aa,5);
			rep_avg(aa,nb) = rep_avg(aa,5);
			tlj_avg(aa,nb) = tlj_avg(aa,5);
			sol_avg(aa,nb) = sol_avg(aa,5);
			pair_avg(aa,nb) = pair_avg(aa,5);
			one_avg(aa,nb) = one_avg(aa,5);
			hb_avg(aa,nb) = hb_avg(aa,5);
			dun_avg(aa,nb) = dun_avg(aa,5);
			intrares_avg(aa,nb) = intrares_avg(aa,5);
			tot_avg(aa,nb) = tot_avg(aa,5);
		}
	}



//KMa phospho_ser 2006-01
	if ( add_pser () )
	for ( int nb = 1; nb <= 30; ++nb )
	{
			atr_avg		(aa_sep,nb) = atr_avg		(aa_ser,nb);
			rep_avg		(aa_sep,nb) = rep_avg		(aa_ser,nb);
			tlj_avg		(aa_sep,nb) = tlj_avg		(aa_ser,nb);
			sol_avg		(aa_sep,nb) = sol_avg		(aa_ser,nb);
			pair_avg	(aa_sep,nb) = pair_avg		(aa_ser,nb);
			one_avg		(aa_sep,nb) = one_avg		(aa_ser,nb);
			hb_avg		(aa_sep,nb) = hb_avg		(aa_ser,nb);
			dun_avg		(aa_sep,nb) = dun_avg		(aa_ser,nb);
			intrares_avg(aa_sep,nb) = intrares_avg	(aa_ser,nb);
			tot_avg		(aa_sep,nb) = tot_avg		(aa_ser,nb);
	}

	if (use_conformer) {
		confLibHandler.readBDALib(confLibChoice);
	}
}

// ctsa --  methods for generating lj vdw and lk solv tables
//   mostly coded using functions and speed-ups originally
//     written by Brian K. and Bill W.




//////////////////////////////////////////////////////////////////////////////
/// @begin setup_intra_cp_table
///
/// @brief
///chu   set up the table indicating whether two atoms within the same residue
///     should be counted; if counted, are they separated by only three bonds or
///     more than three bonds.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
setup_intra_cp_table()
{

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	//     initialize the table
	intra_cp_table = 0.0f;
//	for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
//		for ( int aav = 1, aave = MAX_AA_VARIANTS(); aav <= aave; ++aav ) {
//			for ( int atom1 = 1, atome = natoms(aa,aav); atom1 <= atome; ++atom1 ) {
//				for ( int atom2 = 1; atom2 <= atome; ++atom2 ) {
//					intra_cp_table(atom1,atom2,aa,aav) = 0.0;
//				}
//			}
//		}
//	}


	for ( int aa = 1; aa <= MAX_AA(); ++aa ) {
		if ( is_protein(aa) || is_nonnatural(aa) ) {
			for ( int aav = 1, aave = nvar(aa); aav <= aave; ++aav ) {
				for ( int atom1 = 1, atome = natoms(aa,aav); atom1 <= atome; ++atom1 ) {
					for ( int atom2 = atom1 + 1; atom2 <= atome; ++atom2 ) {
						intra_cp_table( atom2, atom1, aa, aav ) = intra_cp_table( atom1, atom2, aa, aav )
							= intra_res_count_pair( aa, aav, atom1, atom2 );
					}
				}
			}
		}
	}
//$$$  debug
//$$$      for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
//$$$         int triangle = 0;
//$$$         int square = 0;
//$$$         for ( int atom1 = 1, atome = natoms(aa,aav); atom1 <= atome; ++atom1 ) {
//$$$            for ( int atom2 = atom1 + 1; atom2 <= atome; ++atom2 ) {
//$$$               if ( intra_cp_table(atom1,atom2,aa,aav) == 0.4 ) ++triangle;
//$$$               if ( intra_cp_table(atom1,atom2,aa,aav) == 1.0 ) ++square;
//$$$               std::cout << "residue:" << SS( aa ) << "atom1:" << SS( atom1 ) <<
//$$$							   "atom2:" << SS( atom2 ) << "count:" <<
//$$$                SS( intra_cp_table(atom1,atom2,aa,aav) ) << std::endl;
//$$$            }
//$$$            std::cout << " " << std::endl;
//$$$         }
//$$$         std::cout << "triangle:" << SS( triangle ) <<
//$$$          "square:" << SS( square ) << std::endl;
//$$$      }

}

//////////////////////////////////////////////////////////////////////////////
/// @begin one_bond_away
///
/// @brief
///
/// @detailed
///
/// @param  atom1 - [in/out]?
/// @param  atom2 - [in/out]?
/// @param  aa - [in/out]?
/// @param  aav - [in/out]?
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     not applied to atoms on the aromatic ring
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
one_bond_away(
	int atom1,
	int atom2,
	int aa,
	int aav
)
{

	using namespace aaproperties_pack;

	int base1 = atom_base(atom1,aa,aav);
	int base2 = atom_base(atom2,aa,aav);

	if ( atom1 == base2 || atom2 == base1 ) { // one is the base of the other
		return true;
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin two_bond_away
///
/// @brief
///
/// @detailed
///
/// @param  atom1 - [in/out]? -
/// @param  atom2 - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     not applied to atoms on the aromatic ring
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
two_bond_away(
	int atom1,
	int atom2,
	int aa,
	int aav
)
{

	using namespace aaproperties_pack;

	int base1 = atom_base(atom1,aa,aav);
	int base2 = atom_base(atom2,aa,aav);

	if ( one_bond_away(atom1,base2,aa,aav) || one_bond_away(atom2,base1,aa,aav) ) {
		return true;
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin three_bond_away
///
/// @brief
///
/// @detailed
///
/// @param  atom1 - [in/out]? -
/// @param  atom2 - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     not applied to atoms on the aromatic ring
///
/// @references
///
/// @authors
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
bool
three_bond_away(
	int atom1,
	int atom2,
	int aa,
	int aav
)
{
//     not applied to atoms on the aromatic ring

	using namespace aaproperties_pack;

	int base1 = atom_base(atom1,aa,aav);
	int base2 = atom_base(atom2,aa,aav);

	if ( two_bond_away(atom1,base2,aa,aav) ) {
		return true;
	}

	if ( two_bond_away(atom2,base1,aa,aav) ) {
		return true;
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///chu   this function decides whether the pairwise interaction between two atoms
///     within the same residue should be counted or not. If yes, further
///     down-weight the three-bond-away atom-pair interaction. 02/17/2003
///
/// @detailed
///
/// @param  aa - [in/out?] -
/// @param  aav - [in/out?] -
/// @param  atom1 - [in/out?] -
/// @param  atom2 - [in/out?] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
intra_res_count_pair(
	int const aa,
	int const aav,
	int const atom1,
	int const atom2
)
{

	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace design;

	float intra_res_count_pair = 0.0; // Return value

	static float weight = { 0.0 };

	if ( atom1 == atom2 ) return intra_res_count_pair;

	if ( aav != 1 && ( aa != aa_his || !try_both_his_tautomers ) )
		return intra_res_count_pair; // currently don't consider explicit water

	if ( ( aav > 2 ) && ( aa == aa_his ) && ( try_both_his_tautomers ) )
// jk consider only variants 1 and 2 of His if explicit water + try_both_his_tautomers
		return intra_res_count_pair;

//     no pairs for Gly,Ala,Pro
	if ( aa == aa_gly || aa == aa_ala || aa == aa_pro ) return intra_res_count_pair;
//     no pairs between backbone atoms,HN,HA and CB(atoms above the 1st chi)
	if ( (!chi_required(1,atom1,aa,aav)) && (!chi_required(1,atom2,aa,aav)) )
	 return intra_res_count_pair;

	int chi = nchi(aa,aav); // number of chi angles
//     without water variant, they should have one less chi angles
	if ( aa == aa_ser || aa == aa_thr || aa == aa_tyr ) --chi;
//     no pairs between atoms below the last chi angle
	if ( ( chi_required(chi,atom1,aa,aav) || ( atom1 == chi_atoms(2,chi,aa,aav) ) ||
	 ( atom1 == chi_atoms(3,chi,aa,aav) ) ) && ( chi_required(chi,atom2,aa,aav) ||
	 ( atom2 == chi_atoms(2,chi,aa,aav) ) || ( atom2 == chi_atoms(3,chi,aa,aav) ) ) )
	 return intra_res_count_pair;

	int base1 = atom_base(atom1,aa,aav);
	int base2 = atom_base(atom2,aa,aav);
//     no pairs for hydrogens with uncertain positions
	if ( fullatom_type(atom1,aa,aav) >= 21 ) {
		if ( fullatom_type(base1,aa,aav) == 5 ) return intra_res_count_pair;
		// atom1 is H(CH3);
		if ( fullatom_type(base1,aa,aav) == 13 ) return intra_res_count_pair;
		// atom1 is H(OH);
		if ( fullatom_type(base1,aa,aav) == 16 ) return intra_res_count_pair;
		// atom1 is H(SH);
		if ( fullatom_type(base1,aa,aav) >= 9 && fullatom_type(base1,aa,aav) <= 11 ) {
			if ( atom1 == 17 ) goto L10; // HE in Arg
			return intra_res_count_pair;
		}                  // atom1 is H(NH2)
	}
L10:
	if ( fullatom_type(atom2,aa,aav) >= 21 ) {
		if ( fullatom_type(base2,aa,aav) == 5 ) return intra_res_count_pair;
		// atom2 is H(CH3);
		if ( fullatom_type(base2,aa,aav) == 13 ) return intra_res_count_pair;
		// atom2 is H(OH);
		if ( fullatom_type(base2,aa,aav) == 16 ) return intra_res_count_pair;
		// atom2 is H(SH);
		if ( fullatom_type(base2,aa,aav) >= 9 && fullatom_type(base2,aa,aav) <= 11 ) {
			if ( atom2 == 17 ) goto L20; // HE in Arg
			return intra_res_count_pair;
		}                  // atom2 is H(NH2)
	}
L20:
//     no pairs between atoms within two bonds
	if ( one_bond_away(atom1,atom2,aa,aav) ) return intra_res_count_pair;
	if ( two_bond_away(atom1,atom2,aa,aav) ) return intra_res_count_pair;
//     down weight atom pairs three bonds away
	if ( three_bond_away(atom1,atom2,aa,aav) ) {
		intra_res_count_pair = weight;
		return intra_res_count_pair;
	}
//     finally atom pairs more than three bonds away
	intra_res_count_pair = 1.0;

	return intra_res_count_pair;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin apply_packer_weights
///
/// @brief
///     function to set packer weights for diffrent mode
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_packer_weights( bool const standard_weights, bool const suppress_output )
{
	using namespace param_pack;
	using namespace design;

	if ( weightfile::use_weightfile && !dna_interface ) {
		// jk read weights from a file, if desired
		ReadWeights( weightfile::weight_fname );

	} else if ( dna_interface || param_pack::packer_logical::dna_weights ) {
		//cmd  set fullatom weights and proton radii for dna_interface mode

		// AM NOTE: PW_STANDARD (default packer) weights should be applied instead
		// of PW_DNA, and hydrogen resizing turned OFF for specificity prediction
		// calcs and protein-dna interface minimization. Otherwise the predictive
		// power of hb_sc, fa_atr and other fullatom scores in ddG calcs is
		// considerably decreased!!

		// cmd smaller proton radii: weights optimized for sequence recovery at DNA
		// cmd interfaces using smaller radii for protons
		using namespace etable;
		fa_lj_radius(22) = 0.100; // polar H
		fa_lj_radius(23) = 0.700; // nonpolar H
		fa_lj_radius(24) = 0.700; // aromatic H
		fa_lj_radius(25) = 0.100; // backbone HN

		if ( ! suppress_output ) {
			std::cout << "!!! dna weights for packer applied !!!\n";
			std::cout << "!!! modified Hydrogen radii used !!!\n";
			if ( gen_born || ! standard_weights ) {
				std::cout << "!!! WARNING: no weights yet available for packer with " <<
					"-soft_rep flag. -gen_born is standard !!!\n !!! WARNING: for dna => " <<
					" both flags are therefore ignored in dna mode. !!!\n";
			}
		}

		if ( weightfile::use_weightfile ) ReadWeights( weightfile::weight_fname );
		else if ( gen_born ) RetrieveWeightsToCurrent( PW_DNA );
		else RetrieveWeightsToCurrent( PW_DNA_noGB );
		return;

	} else if ( smooth_etable ) {
		//sheffler smooth etable weights applied here
		RetrieveWeightsToCurrent( PW_SMOOTH_ETABLE );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -smooth_etable flag !!!" << std::endl;
		}

	} else if ( get_ligand_flag() ) {
		//mj   set fullatom weights for ligand mode
		get_ligand_packer_weights( ! standard_weights );

	} else if ( pKa_mode::get_pKa_flag() ) {
		//rh   set fullatom weights for pKa mode

		RetrieveWeightsToCurrent( PW_PKA );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -pH and the -pKa flag !!!" << std::endl;
		}

	} else if ( soft_rep_design && ! standard_weights ) {
		///jjh set fullatom weights for soft_rep_design flag
		RetrieveWeightsToCurrent( PW_SOFT_REP_DESIGN );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -soft_rep_design flag !!!" << std::endl;
		}

	} else if ( gen_born && ! standard_weights ) {
		//mj   set fullatom weights for gen_born and soft_rep flags
		RetrieveWeightsToCurrent( PW_SOFT_REP_GEN_BORN );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -gen_born flag and the -soft_rep flag !!!" << std::endl;
		}

	} else if ( ! standard_weights ) {
		//mj   set fullatom weights for soft_rep flags
		RetrieveWeightsToCurrent( PW_SOFT_REP );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -soft_rep flag !!!" << std::endl;
		}

	} else if ( gen_born ) {
		//mj   set fullatom weights for gen_born flags
		RetrieveWeightsToCurrent( PW_GEN_BORN );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -gen_born flag !!!" << std::endl;
		}

	} else if ( use_aw && ! standard_weights ) {
		//xh  set fullatom weights for use_aw flags
		RetrieveWeightsToCurrent( PW_ALLFOLD_SOFT_REP );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -use_aw flag !!!" << std::endl;
		}

	} else if ( use_bw && ! standard_weights ) {
		//xh  set fullatom weights for use_bw flags
		RetrieveWeightsToCurrent( PW_BETA_SOFT_REP );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -use_bw flag !!!" << std::endl;
		}

	} else if ( small_radii && ! standard_weights ) {
		//bk smaller radii
		RetrieveWeightsToCurrent( PW_SMALL_RADII );
		if ( ! suppress_output ) {
			std::cout << "!!! weights for packer modified according to the -small_radii flag !!!" << std::endl;
		}
	}

	// jk change water weights if solvate flag is used
	if ( design::solvate ) {
		if ( ! suppress_output ) {
			std::cout << "applying solvate flag" << std::endl;
		}
		param_pack::pack_wts.set_Wh2o_lj(0.80); // weight for lj repulsive energy between water
		param_pack::pack_wts.set_Wh2o_hb(0.0); // turn off water-mediated hbonds
		param_pack::pack_wts.set_Wh2o_intra(-0.40); // change the sign of the water bonus
		// ie. make it favorable to add waters
	}

	return;

}


