// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 17358 $
//  $Date: 2007-09-24 23:08:09 -0400 (Mon, 24 Sep 2007) $
//  $Author: harrison $


// Rosetta Headers
#include "gb_elec.h"
#include "aaproperties_pack.h"
#include "DesignMap.h"
#include "etable.h"
#include "fullatom_energy.h"
#include "gb_elec_ns.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "misc.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
//#include "pdbstatistics_pack.h"
#include "pKa_mode.h"
#include "RotamerSet.h"
#include "template_pack.h"
#include "util_vector.h"

#ifdef GL_GRAPHICS
#include "gl_graphics.h"
#include "param_pack.h"
#include "water_ns.h"
#endif

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <map> //PBH


////////////////////////////////////////////////////////////////////////////////
/// @begin gb_clear_born_radii()
///
/// @brief zeroes out the Born radii array for a residue
///
/// @detailed
///
/// @param  gb_rad_array - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/16/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gb_clear_born_radii(
	FArray1Da_float gb_rad_array,
	int const aa,
	int const aav
)
{
	using namespace aaproperties_pack;
	using namespace param;

	gb_rad_array.dimension( MAX_ATOM() );

	for ( int atm = 1, atme = natoms(aa,aav); atm <= atme; ++atm ) {
		gb_rad_array(atm) = 0.0;
	}
}

///////////////////////////////////////////////////////////////////////////////
void
gb_set_use_simple_electrostatics(
	bool const setting
)
{
	if ( setting ) setup_simple_electrostatics();
	gb_elec::use_simple_elec = setting;
}

///////////////////////////////////////////////////////////////////////////////
void
gb_set_use_no_intra_res(
	bool const setting
)
{
	gb_elec::use_no_intra_res = setting;
}

///////////////////////////////////////////////////////////////////////////////
bool
get_update_born_radii()
{
	using namespace gb_elec;
	return update_radii && !use_simple_elec;
}

///////////////////////////////////////////////////////////////////////////////
void
set_update_born_radii( bool const setting )
{
//	std::cout << "setting update_born_radii to " << setting << std::endl;
	gb_elec::update_radii = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin gb_res_res_burial()
///
/// @brief Calculates the burial of one residue by another residue
///
/// @detailed
///
/// @param  brad1 - [in/out]? -
/// @param  coord1 - [in/out]? -
/// @param  natoms1 - [in/out]? -
/// @param  fa_type1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  natoms2 - [in/out]? -
/// @param  fa_type2 - [in/out]? -
/// @param  same_res - [in/out]? -
/// @param  override - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/16/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gb_res_res_burial(
	FArray1Da_float brad1,
	FArray2Da_float coord1,
	int const natoms1,
	int const dim_atoms1,
	FArray1Da_int fa_type1,
	FArray2Da_float coord2,
	int const natoms2,
	int const dim_atoms2,
	FArray1Da_int fa_type2,
	bool const same_res,
	int const override
)
{
	using namespace gb_elec;

	brad1.dimension( dim_atoms1 );
	coord1.dimension( 3, dim_atoms1 );
	coord2.dimension( 3, dim_atoms2 );
	fa_type1.dimension( dim_atoms1 );
	fa_type2.dimension( dim_atoms2 );

//jjh local
	FArray1D_float atm_radii1( dim_atoms1 );
	FArray1D_float atm_radii2( dim_atoms2 );

//jjh Fill the radii array - later will allow for overrides
	for ( int atm1 = 1, atm1e = natoms1; atm1 <= atm1e; ++atm1 ) {
		atm_radii1(atm1) = gb_radius(fa_type1(atm1));
	}
	for ( int atm2 = 1, atm2e = natoms2; atm2 <= atm2e; ++atm2 ) {
		if (atm2 == override) {
//jjh 0.8 is the default scale factor
			atm_radii2(atm2) = 0.8 * ( gb_dummy_rad - ParamS );
		} else {
			atm_radii2(atm2) = gb_scale(fa_type2(atm2)) *
								(gb_radius(fa_type2(atm2)) - ParamS );
		}
	}

	for ( int atm1 = 1, atm1e = natoms1; atm1 <= atm1e; ++atm1 ) {
		float const rad1 = atm_radii1(atm1);
		float const rwork1 = rad1 - ParamS;
		float const inv_r1 = 1.0/rwork1;
		for ( int atm2 = 1, atm1e = natoms2; atm2 <= atm1e; ++atm2 ) {
			float const rwork2 = atm_radii2(atm2);
			float dis2;
			distance2_bk(coord1(1,atm1), coord2(1, atm2), dis2);
			float const dis = std::sqrt(dis2);
			float const inv_dis = 1.0/(dis+1.0e-6);
			float const rwork2_sq = rwork2*rwork2;

			if ( same_res && (atm1 == atm2) ) continue;

			if ( dis > (3.5 * rwork2) ) {
				float const inv_dis2 = inv_dis * inv_dis;
				float const tmpsd = rwork2_sq * inv_dis2;
				float const dumbo = Param_TA+tmpsd*(Param_TB+
											tmpsd*(Param_TC+tmpsd*(Param_TD+tmpsd*Param_TDD)));
				brad1(atm1) -= rwork2_sq*rwork2*inv_dis2*inv_dis2*dumbo;
			} else if ( dis > (rwork1 + rwork2)) {
				brad1(atm1) -= (0.5*(rwork2/(dis2 - rwork2_sq) +
												0.5*inv_dis*std::log((dis-rwork2)/(dis+rwork2))));
			} else if ( dis > std::abs( rwork1 - rwork2 ) ) {
				float const theta = 0.5*inv_r1*inv_dis*(dis2+rwork1*rwork1 - rwork2_sq);
				float const U12 = 1.0/(dis+rwork2);
				brad1(atm1) -= 0.25*(inv_r1 * (2.0-theta) - U12 +
															inv_dis*std::log(rwork1*U12));
			} else if ( rwork1 < rwork2 ) {
				brad1(atm1) -= 0.5*(rwork2/(dis2-rwork2_sq) + 2.0*inv_r1 +
												0.5*inv_dis*std::log( (rwork2-dis)/(rwork2+dis)));
			}
			///jjh Store this as 1/2 * the inverse of the Born radius
//			brad1( atm1 ) = 0.5 / brad1( atm );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin gb_get_all_born_radii()
///
/// @brief Gets Born radii for the current fullcoord set
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/16/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
gb_get_all_born_radii()
{
	using namespace param;
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace misc;
	using namespace template_pack;
	using namespace ligand;

	if ( use_simple_elec ) return;

//jjh zero out arrays
	born_radius = 0.0;

//jjh zero out ligand born radii arrays
	if ( get_ligand_flag() ) {
		ligand::ligand_one->hetero_born_radius = 0.0;
	}

//jjh get the residue-residue burial
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
//      if ( neighborlist(res1,res2) || ( res1 == res2 ) ) {
					bool same_res;
						if (res1 == res2) {
							same_res = true;
						} else {
							same_res = false;
						}
			gb_res_res_burial( born_radius(1,res1),
													full_coord(1,1,res1),
													natoms(res(res1),res_variant(res1)),MAX_ATOM(),
													fullatom_type(1,res(res1),res_variant(res1)),
													full_coord(1,1,res2),
													natoms(res(res2),res_variant(res2)), MAX_ATOM(),
													fullatom_type(1,res(res2),res_variant(res2)),
													same_res,
													-1 );
//      }
		}

// get the residue-ligand, then ligand-residue burial

		if ( get_ligand_flag() ) {
			bool same_res = false;
			FArray2D_float het_atom_coord;
			FArray1D_int het_atom_type;
			ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
			ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
// First the burial of the residue by the ligand
			gb_res_res_burial( born_radius(1,res1),
													full_coord(1,1,res1),
													natoms(res(res1),res_variant(res1)),MAX_ATOM(),
													fullatom_type(1,res(res1),res_variant(res1)),
													het_atom_coord(1,1),
													ligand::ligand_one->atom_vector.size(),
													HETERO_ATOM_MAX(),
													het_atom_type(1),
													same_res,
													-1 );

// Now the burial of the ligand by the residue
			gb_res_res_burial( ligand::ligand_one->hetero_born_radius(1),
													het_atom_coord(1,1),
													ligand::ligand_one->atom_vector.size(),
													HETERO_ATOM_MAX(),
													het_atom_type(1),
													full_coord(1,1,res1),
													natoms(res(res1),res_variant(res1)), MAX_ATOM(),
													fullatom_type(1,res(res1),res_variant(res1)),
													same_res,
													-1 );
		}
	}

// Finally, get ligand-ligand burial
	if (get_ligand_flag()) {
		bool same_res = true;
		FArray2D_float het_atom_coord;
		FArray1D_int het_atom_type;
		ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
		ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
		gb_res_res_burial( ligand::ligand_one->hetero_born_radius(1),
							het_atom_coord(1,1),
							ligand::ligand_one->atom_vector.size(),
							HETERO_ATOM_MAX(),
							het_atom_type(1),
							het_atom_coord(1,1),
							ligand::ligand_one->atom_vector.size(),
							HETERO_ATOM_MAX(),
							het_atom_type(1),
							same_res,
							-1 );
	}


//jjh convert to Born radius

// First residues
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int atm = 1, atme = natoms(res(res1), res_variant(res1));
		 atm <= atme; ++atm ) {
			float const radius = gb_radius(fullatom_type(atm, res(res1),
																										res_variant(res1)));
			float const factor = radius - ParamS;
			float const br_save = born_radius(atm, res1);
			float const integral = (-1.0)*factor*br_save;
			float const inv_brad = (1.0/factor) -
														std::tanh( (ParamD - ParamB*integral +
														ParamG*integral*integral)*integral)/radius;

			born_radius(atm, res1) = 1.0/inv_brad;

//      std::cout << "Born radius for " << SS( res1 ) << ' ' << SS( atm ) <<
//       " is " << SS( born_radius(atm,res1) ) <<
//       " from " << SS( br_save ) << std::endl;
		}
	}

// Now, ligand atoms
	if ( get_ligand_flag()) {

		for ( size_t atm = 1; atm <= ligand::ligand_one->atom_vector.size(); ++atm ) {
			float const radius = gb_radius(ligand::ligand_one->atom_vector[atm-1]->get_ros_atom_type());
			float const factor = radius - ParamS;
			float const br_save = ligand::ligand_one->hetero_born_radius(atm);
			float const integral = (-1.0)*factor*br_save;
			float const inv_brad = (1.0/factor) -
														std::tanh( (ParamD - ParamB*integral +
														ParamG*integral*integral)*integral)/radius;

			ligand::ligand_one->hetero_born_radius(atm) = 1.0/inv_brad;

//      std::cout << "Born radius for " << SS( res1 ) << ' ' << SS( atm ) <<
//       " is " << SS( born_radius(atm,res1) ) <<
//       " from " << SS( br_save ) << std::endl;
		}


	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin gb_get_all_born_radii_dummy()
///
/// @brief Gets Born radii for the current fullcoord set
///
/// @detailed
///
/// @param dummy_pos [in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/16/2004
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////


void
gb_get_all_born_radii_dummy( int dummy_pos )
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace misc;

	if ( use_simple_elec ) return;

//jjh zero out arrays
	born_rad = 0.0;

//jjh get the residue-residue burial
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
//    if ( res1 == dummy_pos ) continue;
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
					bool same_res;
						if (res1 == res2) {
							same_res = true;
						} else {
							same_res = false;
						}

			if (res2 != res1 && dummy_pos == 0) {
				gb_res_res_burial( born_rad(1,res1),
											full_coord(1,1,res1),
											natoms(res(res1),res_variant(res1)),MAX_ATOM(),
											fullatom_type(1,res(res1),res_variant(res1)),
											place_holder_xyz(1,1,res2),
											natoms(aa_gly,1), MAX_ATOM(),
											fullatom_type(1,aa_gly,1),
											same_res,
											7 );
			}
			else {
				gb_res_res_burial( born_rad(1,res1),
											full_coord(1,1,res1),
											natoms(res(res1),res_variant(res1)),MAX_ATOM(),
											fullatom_type(1,res(res1),res_variant(res1)),
											full_coord(1,1,res2),
											natoms(res(res2),res_variant(res2)), MAX_ATOM(),
											fullatom_type(1,res(res2),res_variant(res2)),
											same_res,
											-1 );
			}
		}
	}

//jjh convert to Born radius

// First residues
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
//    if (res1 == dummy_pos) continue;
		for ( int atm = 1, atme = natoms(res(res1), res_variant(res1));
		 atm <= atme; ++atm ) {
			float const radius = gb_radius(fullatom_type(atm, res(res1),
																										res_variant(res1)));
			float const factor = radius - ParamS;
			float const br_save = born_rad(atm, res1);
			float const integral = (-1.0)*factor*br_save;
			float const inv_brad = (1.0/factor) -
														std::tanh( (ParamD - ParamB*integral +
														ParamG*integral*integral)*integral)/radius;

			born_rad(atm, res1) = 1.0/inv_brad;

//      std::cout << "Born radius for " << SS( res1 ) << ' ' << SS( atm ) <<
//       " is " << SS( born_rad(atm,res1) ) <<
//       " from " << SS( br_save ) << std::endl;
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin gb_build_placeholders()
///
/// @brief builds a place-holding side chain on repacked/designed sites
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  res - [in/out]? -
/// @param  res_variant - [in/out]? -
/// @param  build_this- [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/18/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gb_build_placeholders(
	int const total_residue,
	FArray3Da_float full_coord,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	//---	FArray1Da_bool build_this
	const DesignMap & design_map
)
{
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace param;
	using namespace param_aa;

	if ( use_simple_elec ) return;

	full_coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	res.dimension( MAX_RES() );
	res_variant.dimension( MAX_RES() );
	//---	build_this.dimension( MAX_RES() );



//jjh locals
	FArray2D_float work_coords( 3 , MAX_ATOM()() );
	FArray2D_float ideal_coords( 3 , MAX_ATOM()() );
	FArray2D_float mat( 3, 3 );
	FArray1D_float vec( 3 );
	FArray1D_float normal( 3 );

// These coordinates are taken from GLY
	ideal_coords( 1 , 1 ) = 0.0000; // Atom N, x coord
	ideal_coords( 2 , 1 ) =10.7220; // Atom N, y coord
	ideal_coords( 3 , 1 ) =19.0990; // Atom N, z coord
	ideal_coords( 1 , 2 ) = 0.0000; // Atom CA, x coord
	ideal_coords( 2 , 2 ) =10.6280; // Atom CA, y coord
	ideal_coords( 3 , 2 ) =20.5540; // Atom CA, z coord
	ideal_coords( 1 , 3 ) = 0.0000; // Atom C, x coord
	ideal_coords( 2 , 3 ) =12.0090; // Atom C, y coord
	ideal_coords( 3 , 3 ) =21.1960; // Atom C, z coord
	ideal_coords( 1 , 4 ) =-0.0030; // Atom O, x coord
	ideal_coords( 2 , 4 ) =13.0390; // Atom O, y coord
	ideal_coords( 3 , 4 ) =20.5210; // Atom O, z coord
	ideal_coords( 1 , 5 ) = 0.0000; // Atom H, x coord
	ideal_coords( 2 , 5 ) =11.6400; // Atom H, y coord
	ideal_coords( 3 , 5 ) =18.6790; // Atom H, z coord
	ideal_coords( 1 , 6 ) =-0.8900; // Atom 2HA, x coord
	ideal_coords( 2 , 6 ) =10.0870; // Atom 2HA, y coord
	ideal_coords( 3 , 6 ) =20.8760; // Atom 2HA, z coord
	ideal_coords( 1 , 7 ) = 0.8890; // Dummy Atom, x coord
	ideal_coords( 2 , 7 ) =10.0870; // Dummy Atom, y coord
	ideal_coords( 3 , 7 ) =20.8760; // Dummy Atom, z coord

// put the dummy at the correct dist

	normal(1) = ideal_coords( 1, 7) - ideal_coords(1, 2);
	normal(2) = ideal_coords( 2, 7) - ideal_coords(2, 2);
	normal(3) = ideal_coords( 3, 7) - ideal_coords(3, 2);
	float vmag = std::sqrt(normal(1)*normal(1) + normal(2)*normal(2) +
												 normal(3)*normal(3));
	normal(1) /= vmag;
	normal(2) /= vmag;
	normal(3) /= vmag;

	ideal_coords( 1, 7) = ideal_coords(1, 2) + gb_dummy_dist * normal(1);
	ideal_coords( 2, 7) = ideal_coords(2, 2) + gb_dummy_dist * normal(2);
	ideal_coords( 3, 7) = ideal_coords(3, 2) + gb_dummy_dist * normal(3);

	for ( int bbres = 1; bbres <= total_residue; ++bbres ) {
		//---		if ( build_this(bbres) ) {
		if ( design_map.repack_residue(bbres) ) {
			// Get a fresh copy every time
			work_coords = ideal_coords;

			// Align the N, CA, and C
			lineup_bk(work_coords(1,1),work_coords(1,2),
								full_coord(1, 1, bbres),full_coord(1, 2, bbres),mat,vec);

			for ( int i = 1; i <= 7; ++i ) {
				move_bk(work_coords(1,i),mat,vec);
			}

			align_bk(work_coords(1,1),work_coords(1,2),work_coords(1,3),
							 full_coord(1, 3, bbres), mat(1,1), vec(1));

			for ( int i = 3; i <= 7; ++i ) {
				move_bk(work_coords(1,i),mat,vec);
			}

			for ( int atm = 1; atm <= 7; ++atm ) {
				for ( int i = 1; i <= 3; ++i ) {
					place_holder_xyz(i,atm,bbres) = work_coords(i,atm);
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin gb_get_template_born_radii()
///
/// @brief Gets the Born radii for template atoms for the packer, using
/// a place-holder sidechain for unknown (designed or repacked) sites.
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  born_radius - [in/out]? -
/// @param  res - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  protamers - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/19/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gb_get_template_born_radii(
	int const total_residue,
	FArray3Da_float full_coord,
	FArray2Da_float born_radius,
	FArray1Da_int res,
	FArray1Da_int aav,
	//---	FArray1Da_bool dummy_check
	const DesignMap & design_map
)
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace ligand;

	if ( use_simple_elec ) return;

	full_coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	born_radius.dimension( MAX_ATOM(), MAX_RES() );
	res.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );
	//--	dummy_check.dimension( MAX_RES() );

//jjh First clear all values
	born_radius = 0.0;

//jjh zero out ligand born radii arrays
	if ( get_ligand_flag() ) {
		ligand::ligand_one->hetero_born_radius = 0.0;
	}

//jjh When s site has a true value for dummy_check, we use the
//jjh place-holder coordinates to calculate the born radii,
//jjh but it is still stored in the array in the common block
//jjh defined in template_pack.h.

	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {

			bool same_res = ( res1 == res2 );

// Don't get Born radii for placeholders
			if ( design_map.repack_residue(res1) ) continue;


// switch depending on whether burial is from a real residue
// or a placeholder
			if ( design_map.repack_residue(res2) ) {
				gb_res_res_burial( born_radius(1,res1),
											full_coord(1,1,res1),
											natoms(res(res1),aav(res1)),MAX_ATOM(),
											fullatom_type(1,res(res1),aav(res1)),
											place_holder_xyz(1,1,res2),
											natoms(aa_gly,1), MAX_ATOM(),
											fullatom_type(1,aa_gly,1),
											same_res,
											7 );
			} else {
				gb_res_res_burial( born_radius(1,res1),
											full_coord(1,1,res1),
											natoms(res(res1),aav(res1)),MAX_ATOM(),
											fullatom_type(1,res(res1),aav(res1)),
											full_coord(1,1,res2),
											natoms(res(res2),aav(res2)), MAX_ATOM(),
											fullatom_type(1,res(res2),aav(res2)),
											same_res,
											-1 );
			}
		}

//jjh Now the ligand stuff

		if ( get_ligand_flag() ) {
			bool same_res = false;
// First the burial of the residue by the ligand
			if ( !design_map.repack_residue(res1) ) {

//jjh A known, non-placeholder residue

//jjh First the burial of the residue by the ligand
			FArray2D_float het_atom_coord;
			FArray1D_int het_atom_type;
			ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
			ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
				gb_res_res_burial( born_radius(1,res1), full_coord(1,1,res1),
					natoms(res(res1),aav(res1)),MAX_ATOM(),
					fullatom_type(1,res(res1),aav(res1)),
					het_atom_coord(1,1), ligand::ligand_one->atom_vector.size(),
					HETERO_ATOM_MAX(), het_atom_type(1), same_res, -1 );

//jjh Now the burial of the ligand by the residue
				gb_res_res_burial( ligand::ligand_one->hetero_born_radius(1),
					het_atom_coord(1,1), ligand::ligand_one->atom_vector.size(),
					HETERO_ATOM_MAX(), het_atom_type(1), full_coord(1,1,res1),
					natoms(res(res1),aav(res1)), MAX_ATOM(),
					fullatom_type(1,res(res1),aav(res1)), same_res, -1 );
			} else {

//jjh Just a placeholder, don't need to calculate its burial

//jjh Just the effect of the placeholder on the ligand
				FArray2D_float het_atom_coord;
				FArray1D_int het_atom_type;
				ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
				ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
				gb_res_res_burial( ligand::ligand_one->hetero_born_radius(1),
					het_atom_coord(1,1), ligand::ligand_one->atom_vector.size(),
					HETERO_ATOM_MAX(), het_atom_type(1),
					place_holder_xyz(1,1,res1),
					natoms(aa_gly,1), MAX_ATOM(),
					fullatom_type(1,aa_gly,1), same_res, 7 );
			}
    }
	}

//jjh Finally, ligand-ligand burial
	if (get_ligand_flag()) {
		bool same_res = true;
		FArray2D_float het_atom_coord;
		FArray1D_int het_atom_type;
		ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
		ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
		gb_res_res_burial( ligand::ligand_one->hetero_born_radius(1),
			het_atom_coord(1,1), ligand::ligand_one->atom_vector.size(), HETERO_ATOM_MAX(),
			het_atom_type(1), het_atom_coord(1,1), ligand::ligand_one->atom_vector.size(),
			HETERO_ATOM_MAX(), het_atom_type(1), same_res, -1 );
	}


//jjh Finally, convert from burial to Born radius

	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {

		if ( design_map.repack_residue(res1) ) continue;

		for ( int atm = 1, atme = natoms(res(res1),aav(res1)); atm <= atme; ++atm ) {
			float const radius = gb_radius(fullatom_type(atm, res(res1),aav(res1)));
			float const factor = radius - ParamS;
			float const br_save = born_radius(atm, res1);
			float const integral = (-1.0)*factor*br_save;
			float const inv_brad = (1.0/factor)  -
			 std::tanh( (ParamD - ParamB*integral + ParamG*integral*integral)*integral)/radius;

			born_radius(atm, res1) = 1.0/inv_brad;

//      std::cout << "TEMP Born " << SS( res1 ) << " " << SS( atm ) << " " << born_radius(atm, res1) << std::endl;

		}
	}

// Now, ligand atoms
	if ( get_ligand_flag()) {
		for ( size_t atm = 1; atm <= ligand::ligand_one->atom_vector.size(); ++atm ) {
			float const radius = gb_radius(ligand::ligand_one->atom_vector[atm-1]->get_ros_atom_type());
			float const factor = radius - ParamS;
			float const br_save = ligand::ligand_one->hetero_born_radius(atm);
			float const integral = (-1.0)*factor*br_save;
			float const inv_brad = (1.0/factor) -
					std::tanh( (ParamD - ParamB*integral +
					ParamG*integral*integral)*integral)/radius;

			ligand::ligand_one->hetero_born_radius(atm) = 1.0/inv_brad;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin gb_get_rotamers_born_radii()
///
/// @brief
/// Gets the Born radii for rotamer atoms for the packer, using
/// a place-holder sidechain for unknown (designed or repacked) sites.
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  res - [in/out]? -
/// @param  aav - [in/out]? -
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/19/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gb_get_rotamers_born_radii(
	int const total_residue,
	FArray3Da_float full_coord,
	FArray1Da_int res,
	FArray1Da_int aav,
	RotamerSet & rotamer_set,
	const DesignMap & design_map
)
{
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace param;
	using namespace param_aa;

	if ( use_simple_elec ) return;

	full_coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	res.dimension( MAX_RES() );
	aav.dimension( MAX_RES() );

//jjh inputs

//jjh locals
	int aarot, aavrot, rotres;
	float tmp;

//jjh First clear all values
	for ( int irot = 1; irot <= rotamer_set.nrotamers(); ++irot ) {
		for ( int atm = 1, atme = natoms(rotamer_set.report_aa(irot),rotamer_set.report_aav(irot));
		 atm <= atme; ++atm ) {
			rotamer_set.set_rot_born_radius(atm, irot) = 0.0;
		}
	}

//jjh When s site has a true value for use_dummy, we use the
//jjh place-holder coordinates to calculate the born radii,
//jjh but it is still stored in the array in the common block
//jjh defined in template_pack.h.

	for ( int irot = 1; irot <= rotamer_set.nrotamers(); ++irot ) {
		aarot = rotamer_set.report_aa(irot);
		aavrot = rotamer_set.report_aav(irot);
		rotres = rotamer_set.report_seqpos(irot);
		for ( int ires = 1; ires <= total_residue; ++ires ) {
			if ( ires == rotres ) {
//jjh get the rotamer's self burial
				bool same_res = true;
				gb_res_res_burial(
				rotamer_set.get_rot_born_radius(irot), rotamer_set.get_rotcoord(irot),
				natoms(aarot, aavrot), MAX_ATOM(), fullatom_type(1, aarot, aavrot),
				rotamer_set.get_rotcoord(irot), natoms(aarot, aavrot), MAX_ATOM(),
				fullatom_type(1, aarot, aavrot),
				same_res, -1);
				//---} else if ( use_dummy(ires) ) {
			}	else if ( design_map.repack_residue(ires) ) {
//jjh this template position is a dummy place-holder
				bool same_res = false;
				gb_res_res_burial(rotamer_set.get_rot_born_radius(irot), rotamer_set.get_rotcoord(irot),
				natoms(aarot, aavrot), MAX_ATOM(), fullatom_type(1, aarot, aavrot),
				place_holder_xyz(1,1,ires), natoms(aa_gly, 1), MAX_ATOM(),
				fullatom_type(1, aa_gly, 1),
				same_res, 7);
			} else {
//jjh not a place-holder
				bool same_res = false;
				gb_res_res_burial(rotamer_set.get_rot_born_radius(irot), rotamer_set.get_rotcoord(irot),
				natoms(aarot, aavrot), MAX_ATOM(), fullatom_type(1, aarot, aavrot),
				full_coord(1,1,ires), natoms(res(ires), aav(ires)), MAX_ATOM(),
				fullatom_type(1, res(ires), aav(ires)),
				same_res, -1);
			}
		}
	}

//jjh Finally, convert from burial to Born radius

//jjh First clear all values
	for ( int irot = 1; irot <= rotamer_set.nrotamers(); ++irot ) {
		int rotaa = rotamer_set.report_aa(irot);
		int rotaav = rotamer_set.report_aav(irot);
		for ( int atm = 1, atme = natoms(rotaa, rotaav);
			atm <= atme; ++atm ) {

			float radius = gb_radius(fullatom_type(atm,rotaa,rotaav));
			float factor = radius-ParamS;
//      float br_save = rotamer_set.get_rot_born_radius(atm,irot);
			tmp = (-1.0)*factor*rotamer_set.get_rot_born_radius(atm,irot);

			float newtmp =
				1.0/factor - tanh( (ParamD - ParamB*tmp + ParamG*tmp*tmp)*tmp)/radius;

			rotamer_set.set_rot_born_radius(atm, irot) = 1.0/newtmp;

		}
	}
}


//////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/// @begin gb_get_res_res_elecE()
///
/// @brief Gets the Generalized Born electrostatic energy between two
///      residues.
///
/// @detailed
///
/// @param  seqpos1 - [in/out]? -
/// @param  aa1 - [in/out]? -
/// @param  aav1 - [in/out]? -
/// @param  coord1 - [in/out]? -
/// @param  brad1 - [in/out]? -
/// @param  seqpos2 - [in/out]? -
/// @param  aa2 - [in/out]? -
/// @param  aav2 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  brad2 - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/19/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////
float
gb_get_res_res_elecE(
	int const seqpos1,
	int const aa1,
	int const aav1,
	FArray2Da_float coord1,
	FArray1Da_float brad1,
	int const seqpos2,
	int const aa2,
	int const aav2,
	FArray2Da_float coord2,
	FArray1Da_float brad2
)
{
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace param;
	using namespace param_aa;

	if ( use_simple_elec ) {
		return gb_get_res_res_simple_elecE( seqpos1, aa1, aav1, coord1,
																				seqpos2, aa2, aav2, coord2 );
	}

	coord1.dimension( 3, MAX_ATOM() );
	brad1.dimension( MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );
	brad2.dimension( MAX_ATOM() );

	if ( is_NA( aa1 ) && is_NA( aa2 ) ) return 0.0;

	int natoms1 = natoms(aa1, aav1);
	int natoms2 = natoms(aa2, aav2);

	float const inv_Ep = 1.0/Ep;
	float const tau = (1.0/Ep - 1.0/Ew);
	bool const same_res = (seqpos1 == seqpos2);

	FArray2D_int bonds_between( MAX_ATOM()(), MAX_ATOM()() );

	// the default value is 4 or more bonds apart
	bonds_between = 4;

	if (same_res) {
		for ( int atm1 = 1; atm1 <= natoms1; ++atm1 )
			for ( int dih = 1, dihe = ndihe_neighbors(atm1, aa1, aav1); dih <= dihe; ++dih ) {
				int atm2 = dihe_neighbor(dih, atm1, aa1, aav1);
				bonds_between(atm1, atm2) = 3;
				bonds_between(atm2, atm1) = 3;
			}

		for ( int atm1 = 1; atm1 <= natoms1; ++atm1 )
			for ( int ang = 1, ange = nangle_neighbors(atm1, aa1, aav1); ang <= ange; ++ang ) {
				int atm2 = angle_neighbor(ang, atm1, aa1, aav1);
				bonds_between(atm1, atm2) = 2;
				bonds_between(atm2, atm1) = 2;
			}

		for ( int atm1 = 1; atm1 <= natoms1; ++atm1 ) {
			for ( int bnd = 1, bnde = nbonded_neighbors(atm1, aa1, aav1); bnd <= bnde; ++bnd ) {
				int atm2 = bonded_neighbor(bnd, atm1, aa1, aav1);
				bonds_between(atm1, atm2) = 1;
				bonds_between(atm2, atm1) = 1;
			}
			bonds_between(atm1, atm1) = 0;
		}
	} else if (seqpos1 == (seqpos2 + 1)) {
/// *********** BEGIN seqpos2---seqpos1 ***********
		if ( ( is_protein(aa1) || is_nonnatural(aa1) ) && ( is_protein(aa2) || is_nonnatural(aa2) ) ) {
// Set up the bonds and angles added for a peptide bond
				bonds_between(1, 3) = 1;  // N-C bond
				bonds_between(1, 2) = 2;  // N1-C2-CA2 angle
				bonds_between(1, 4) = 2;  // N1-C2-O2 angle
				bonds_between(2, 3) = 2;  // C2-N1-CA1 angle
				if (aa1 == aa_pro) {
					bonds_between(7, 3) = 2;  // C2-N1-CD for pro
				} else {
					bonds_between(HNpos(aa1, aav1), 3) = 2; // C2-N1-HN for other
				}
// Now the dihedrals - some numbers are independent of the residues
				bonds_between(3, 3) = 3;                // C-X-X-C
				bonds_between(2, 2) = 3;                // CA-X-X-CA
				bonds_between(2, 4) = 3;                // CA-X-X-O
				bonds_between(HApos(aa2, aav2), 3) = 3; // HA-X-X-C
				bonds_between(1, 1) = 3;                // N-X-X-N
				bonds_between(1, HApos(aa2, aav2)) = 3; // N-X-X-HA

// Now the annoying residue-dependent stuff
//             defaults
				int down_off_N = HNpos(aa1, aav1);
				int down_off_CA = 5;
				int up_off_CA = 5;

				if (aa1 == aa_pro)
					down_off_N = 7;
				else if (aa1 == aa_gly)
					down_off_CA = 6;

				if (aa2 == aa_gly)
					up_off_CA = 6;

// Make the assignments
				bonds_between(down_off_CA,         3) = 3;
				bonds_between(down_off_N ,         2) = 3;
				bonds_between(down_off_N ,         4) = 3;
				bonds_between(1         , up_off_CA) = 3;

// Extras if aa1 is a proline
				if (aa1 == aa_pro) {
					bonds_between(6, 3) = 3;
					bonds_between(8, 3) = 3;
					bonds_between(9, 3) = 3;
				}
			}

// ***********  END seqpos2---seqpos1 ***********
	} else if (seqpos1 == (seqpos2 - 1)) {
/// *********** BEGIN seqpos1---seqpos2 ***********

		if ( ( is_protein(aa1) || is_nonnatural(aa1) ) && ( is_protein(aa2) || is_nonnatural(aa2) ) ) {
// Set up the bonds and angles added for a peptide bond
				bonds_between(3, 1) = 1;  // N-C bond
				bonds_between(2, 1) = 2;  // N1-C2-CA2 angle
				bonds_between(4, 1) = 2;  // N1-C2-O2 angle
				bonds_between(3, 2) = 2;  // C2-N1-CA1 angle
				if (aa2 == aa_pro) {
					bonds_between(3, 7) = 2;  // C2-N1-CD for pro
				} else {
					bonds_between(3, HNpos(aa2, aav2)) = 2; // C2-N1-HN for other
				}
// Now the dihedrals - some numbers are independent of the residues
				bonds_between(3, 3) = 3;                // C-X-X-C
				bonds_between(2, 2) = 3;                // CA-X-X-CA
				bonds_between(4, 2) = 3;                // CA-X-X-O
				bonds_between(3, HApos(aa2, aav2)) = 3; // HA-X-X-C
				bonds_between(1, 1) = 3;                // N-X-X-N
				bonds_between(HApos(aa1, aav1), 1) = 3; // N-X-X-HA

// Now the annoying residue-dependent stuff
//             defaults
				int down_off_N = HNpos(aa2, aav2);
				int down_off_CA = 5;
				int up_off_CA = 5;

				if (aa2 == aa_pro)
					down_off_N = 7;
				else if (aa2 == aa_gly)
					down_off_CA = 6;

				if (aa1 == aa_gly)
					up_off_CA = 6;

// Make the assignments
				bonds_between(3       , down_off_CA) = 3;
				bonds_between(2       , down_off_N) = 3;
				bonds_between(4       , down_off_N) = 3;
				bonds_between(up_off_CA,           1) = 3;

// Extras if aa2 is a proline
				if (aa2 == aa_pro) {
					bonds_between(3, 6) = 3;
					bonds_between(3, 8) = 3;
					bonds_between(3, 9) = 3;
				}
			}
// ***********  END seqpos1---seqpos2 ***********
	}

	float elecE = 0.0;
	for ( int atm1 = 1; atm1 <= natoms1; ++atm1 ) {
		float const q1 = atomic_charge(atm1, aa1, aav1);
		for ( int atm2 = 1; atm2 <= natoms2; ++atm2 ) {
			float const q2 = atomic_charge(atm2, aa2, aav2);

			float dis2;
			distance2_bk(coord1(1, atm1), coord2(1, atm2), dis2);
			float const exparg = (-dis2)/(4.0 * brad1(atm1) * brad2(atm2));
			float const denom = std::sqrt(dis2 +
														brad1(atm1)*brad2(atm2)*std::exp(exparg));
			float this_intx;
			if (same_res) {
				//rh Remove intra residue electrostatics in pKa mode
				//rh Intra residue electrostatics bias equilibrium towards charged state
				//rh For pKa mode ignore intra-residue electrostatics such that at the IpKa, Eproto = Edeproto
				if ( gb_elec::use_no_intra_res ) {
					this_intx = 0.0;
				} else {
					// tau = (1.0/Ep - 1.0/Ew)    Ep = 4.0  Ew=80.0
					this_intx = (-166.0)*tau*q1*q2/denom;
				}
			} else {
				this_intx = (-332.0)*tau*q1*q2/denom;
			}
			elecE += this_intx;
//      total_gb += this_intx;

			this_intx = 0.0;
			if (bonds_between(atm1, atm2) >= 3) {
				float const dis = std::sqrt(dis2);
				float const r1 = gb_radius(fullatom_type(atm1, aa1, aav1));
				float const r2 = gb_radius(fullatom_type(atm2, aa2, aav2));
				this_intx = 166.0*gb_shell_intxn(inv_Ep*q1, r1, q2, r2, dis);
				if (!same_res) this_intx *= 2.0;
				//rh Remove intra residue electrostatics in pKa mode
				if (same_res && gb_elec::use_no_intra_res ) this_intx *= 0.0;
			}

			elecE += this_intx;
//      total_coul += this_intx;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	//// this loop just for graphics:
	/////////////////////////////////////////////////////////////////////////////
#ifdef GL_GRAPHICS
	float const bondE_threshold( gl_graphics_bond_energy_threshold() );

	// show bonds for graphics
	if ( inside_fullatom_energy() && seqpos1 != seqpos2 &&
			 !( is_protein(aa1) && is_protein(aa2) ) &&
			 ( std::abs( param_pack::pack_wts.Wgb_elec()*elecE ) > bondE_threshold ) ) {
		static FArray2D_float gbE( param::MAX_ATOM(), param::MAX_ATOM() );
		gbE = 0.0;
		for ( int atm1 = 1; atm1 <= natoms1; ++atm1 ) {
			int const atm1_base( graphics::intxn_atm(atm1,aa1,aav1) );
			float const q1 = atomic_charge(atm1, aa1, aav1);
			for ( int atm2 = 1; atm2 <= natoms2; ++atm2 ) {
				int const atm2_base( graphics::intxn_atm(atm2,aa2,aav2) );
				float const q2 = atomic_charge(atm2, aa2, aav2);

				float dis2;
				distance2_bk(coord1(1, atm1), coord2(1, atm2), dis2);
				float const exparg = (-dis2)/(4.0 * brad1(atm1) * brad2(atm2));
				float const denom = std::sqrt(dis2 +
																			brad1(atm1)*brad2(atm2)*std::exp(exparg));
				float this_intx, total_intx(0.0);
				if (same_res) {
					// tau = (1.0/Ep - 1.0/Ew)    Ep = 4.0  Ew=80.0
					this_intx = (-166.0)*tau*q1*q2/denom;
				} else {
					this_intx = (-332.0)*tau*q1*q2/denom;
				}
				total_intx += this_intx;
				//      total_gb += this_intx;

				this_intx = 0.0;
				if (bonds_between(atm1, atm2) >= 3) {
					float const dis = std::sqrt(dis2);
					float const r1 = gb_radius(fullatom_type(atm1, aa1, aav1));
					float const r2 = gb_radius(fullatom_type(atm2, aa2, aav2));
					this_intx = 166.0*gb_shell_intxn(inv_Ep*q1, r1, q2, r2, dis);
					if (!same_res) this_intx *= 2.0;
				}

				total_intx += this_intx;
				//      total_coul += this_intx;
				gbE( atm1_base, atm2_base ) += total_intx;
			}
		}
		for ( int i=1; i<= natoms1; ++i ) {
			for ( int j=1; j<= natoms2; ++j ) {
				gbE(i,j) *= param_pack::pack_wts.Wgb_elec();
				if ( std::abs( gbE( i,j) ) > bondE_threshold ) {
					gl_graphics_store_bond( seqpos1, i, seqpos2, j, gbE(i,j),
																	graphics::GB_BOND );
				}
			}
		}
	} // res-resE > bondE_threshold

	/////////////////////////////////////////////////////////////////////////////
	//// end show bonds for graphics ////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
#endif


	return elecE;
}

///////////////////////////////////////////////////////////////////////////////
float
gb_shell_intxn_deriv(
	float const qai,
	float const rai,
	float const qbi,
	float const rbi,
	float const dist
)
{

	if (dist >= (rai+rbi))
		return (-1.0 * qai * qbi / ( dist * dist ) );


// Make sure rb is larger than ra

	float qa;
	float ra;
	float qb;
	float rb;

	if (rai > rbi) {
		qa = qbi;
		ra = rbi;
		qb = qai;
		rb = rai;
	} else {
		qa = qai;
		ra = rai;
		qb = qbi;
		rb = rbi;
	}

	if (((ra+rb) > dist) && (dist > (rb - ra))) {
		float dfout;
		float fout;
		if (ra == rb) {
			fout = 0.5 * (1.0+0.5*dist/ra);
			dfout = 0.25/ra;
		} else {
			fout = 0.5*(1.0 + 0.5*(ra*ra - rb*rb)/(ra*dist) + 0.5*dist/ra);
			dfout = 0.25*(rb*rb - ra*ra)/(ra*dist*dist) + 0.25/ra;
		}
		float dfin = -1.0*dfout;
		return qa*qb*(dfin/rb -
									2.0*fout/( (dist+ra+rb)*(dist+ra+rb) ) +
									2.0*dfout/( dist + ra + rb ) );
	} else {
		///jjh already fully buried
		return 0.0;
	}
}

///////////////////////////////////////////////////////////////////////////////

bool
count_pair_gb(
	const int atm1,
	const int pos1,
	const int aa1,
	const int aav1,
	const int atm2,
	const int pos2,
	const int aa2,
	const int aav2
)
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;

	if ( is_NA(aa1) && is_NA(aa2) ) return false;

	if( pos1 == pos2 ) {

		if( atm1 == atm2 ) return false;

		for ( int bnd = 1, bnde = nbonded_neighbors(atm1, aa1, aav1); bnd <= bnde;
					++bnd ) {
			if( atm2 == bonded_neighbor(bnd, atm1, aa1, aav1)) return false;
		}

		for ( int ang = 1, ange = nangle_neighbors(atm1, aa1, aav1); ang <= ange;
					++ang ) {
			if( atm2 == angle_neighbor(ang, atm1, aa1, aav1)) return false;
		}

		return true;

	} else if ( pos1 == ( pos2 - 1 ) ) {
		if ( is_protein(aa1) && is_protein(aa2) ) {
			if( atm1 == 3 && atm2 == 1 ) return false;
			if( atm1 == 2 && atm2 == 1 ) return false;
			if( atm1 == 4 && atm2 == 1 ) return false;
			if( atm1 == 3 && atm2 == 2 ) return false;

			if (aa2 == aa_pro) {
				// C2-N1-CD for pro
				if( atm1 == 3 && atm2 == 7 ) return false;
			} else {
				// C2-N1-HN for other
				if( atm1 == 3 && atm2 == HNpos( aa2, aav2) ) return false;
			}
		} else if ( is_protein(aa1) || is_protein(aa2) ) {
			return true;
		} else {
			return false; // no protein at all
		}
	} else if ( pos1 == ( pos2 + 1 ) ) {

		if ( is_protein(aa1) && is_protein(aa2) ) {
				if( atm2 == 3 && atm1 == 1 ) return false;
				if( atm2 == 2 && atm1 == 1 ) return false;
				if( atm2 == 4 && atm1 == 1 ) return false;
				if( atm2 == 3 && atm1 == 2 ) return false;

				if (aa1 == aa_pro) {
					// C2-N1-CD for pro
					if( atm2 == 3 && atm1 == 7 ) return false;
				} else {
					// C2-N1-HN for other
					if( atm2 == 3 && atm1 == HNpos( aa1, aav1) ) return false;
				}
		} else if ( is_protein(aa1) || is_protein(aa2) ) {
			return true;
		} else {
			return false; // no protein at all
		}

	} else {
		return true;
	}
	return true;
}

//////////////////////////////////////////////////////////////////////////
/// @begin gb_get_ligand_elecE()
///
/// @brief Gets the Generalized Born electrostatic energy between a
///      residue and a ligand, or a ligand and itself.  A lot like
///      gb_get_res_res_elecE(), but without processing of bonded
///      information.
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 4/19/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////

float
gb_get_ligand_elecE(
	int const natoms1,
	int const dim_atoms1,
	FArray2Da_float coord1,
	FArray1Da_float brad1,
	FArray1Da_float charge1,
	FArray1Da_int fa_type1,
	int const natoms2,
	int const dim_atoms2,
	FArray2Da_float coord2,
	FArray1Da_float brad2,
	FArray1Da_float charge2,
	FArray1Da_int fa_type2,
	bool const same_res
)
{
	using namespace aaproperties_pack;
	using namespace gb_elec;
	using namespace param;
	using namespace param_aa;

	coord1.dimension( 3, dim_atoms1 );
	brad1.dimension( dim_atoms1 );
	charge1.dimension( dim_atoms1 );
	fa_type1.dimension( dim_atoms1 );

	coord2.dimension( 3, dim_atoms2 );
	brad2.dimension( dim_atoms2 );
	charge2.dimension( dim_atoms2 );
	fa_type2.dimension( dim_atoms2 );
	float const inv_Ep = 1.0/Ep;
	float const tau = (1.0/Ep - 1.0/Ew);

	float elecE = 0.0;
	for ( int atm1 = 1; atm1 <= natoms1; ++atm1 ) {
		float const q1 = charge1(atm1);
		for ( int atm2 = 1; atm2 <= natoms2; ++atm2 ) {
			float const q2 = charge2(atm2);

			float dis2;
			distance2_bk(coord1(1, atm1), coord2(1, atm2), dis2);
			float const exparg = (-dis2)/(4.0 * brad1(atm1) * brad2(atm2));
			float const denom = std::sqrt(dis2 +
														brad1(atm1)*brad2(atm2)*std::exp(exparg));
			float this_intx;
			if (same_res) {
				this_intx = (-166.0)*tau*q1*q2/denom;
			} else {
				this_intx = (-332.0)*tau*q1*q2/denom;
			}

			elecE += this_intx;

			//mjdebug
			//			std::cout << "===> 1 " << SS(this_intx) << SS(dis2) << SS(atm1) << SS(atm2)
			//				<< SS( brad1(atm1)) << SS( brad2(atm2)) << SS(q1) << SS(q2) << exparg << denom << std::endl;
			//mjend

			this_intx = 0.0;
			float const dis = std::sqrt(dis2);
			float const r1 = gb_radius(fa_type1(atm1));
			float const r2 = gb_radius(fa_type2(atm2));
			this_intx = 166.0*gb_shell_intxn(inv_Ep*q1, r1, q2, r2, dis);

			if (!same_res) this_intx *= 2.0;

			elecE += this_intx;

			//mjdebug
			//std::cout << "===> 2 " << SS(this_intx) << SS(dis2) << SS(atm1) << SS(atm2)
			//				<< SS( brad1(atm1)) << SS( brad2(atm2)) << SS(r1) << SS(r2) << std::endl;
			//mjend
		}
	}

	//mjdebug
	//std::cout << "===> 3 " << SS(elecE) << std::endl;
	//mjend


	return elecE;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin gb_shell_intxn()
///
/// @brief Calculates the interaction energy of two shells of charge.
///      Doesn't blow up as shells pass through each other
///
/// @detailed
///
/// @param  qai - [in/out]? -
/// @param  rai - [in/out]? -
/// @param  qbi - [in/out]? -
/// @param  rbi - [in/out]? -
/// @param  dist - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jjh 5/17/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
gb_shell_intxn(
	float const qai,
	float const rai,
	float const qbi,
	float const rbi,
	float const dist
)
{

	if (dist >= (rai+rbi))
		return (qai * qbi /dist);

// Make sure rb is larger than ra

	float qa;
	float ra;
	float qb;
	float rb;

	if (rai > rbi) {
		qa = qbi;
		ra = rbi;
		qb = qai;
		rb = rai;
	} else {
		qa = qai;
		ra = rai;
		qb = qbi;
		rb = rbi;
	}

	if (((ra+rb) > dist) && (dist > (rb - ra))) {
		float fout;
		if (ra == rb) {
			fout = 0.5 * (1.0+0.5*dist/ra);
		} else {
			fout = 0.5*(1.0 + 0.5*(ra*ra - rb*rb)/(ra*dist) + 0.5*dist/ra);
		}
		float fin = 1.0 - fout;
		return qa*qb*(fin/rb + fout*2.0/(dist+ra+rb));
	} else {
		return qa*qb/rb;
	}
}

///////////////////////////////////////////////////////////////////////////////
void
setup_simple_electrostatics()
{
	using namespace etable;
	using namespace pdbstatistics_pack;
	using namespace param;
	using namespace aaproperties_pack;
	using namespace gb_elec;

	std::cout << "Setting up the simple electrostatics gb hijack " <<
		"calculation" << std::endl;

	//////////////////////////////////////////////////////////////
	// hack the atomic charge array by zeroing out aliphatic and
	// aromatic hydrogens, pulling their charge down onto the base
	// atoms
	//

	simple_charge.dimension( MAX_ATOM()(), MAX_AA()(), MAX_AA_VARIANTS()());
	simple_charge = 0.0; // initialize to 0.0

	for ( int aa=1; aa<= MAX_AA()(); ++aa ) {
		for ( int aav=1; aav<= nvar(aa); ++aav ) {
			for ( int j=1; j<= natoms(aa,aav); ++j ) {
				int const t( fullatom_type(j,aa,aav) );
				if ( t == 23 || t== 24 ) {
					// apolar H or aro H
					int const base( atom_base(j,aa,aav) );
					{
						int const base_type( fullatom_type(base,aa,aav));
						assert( base_type < 22 || base_type > 26 );
					}
					simple_charge( base, aa, aav ) += atomic_charge( j, aa, aav );
				} else {
					simple_charge( j,aa,aav) += atomic_charge( j, aa, aav );
				}
			}
		}
	}

	////////////////////////
	//// map charges to bins
	charge_bin.dimension( MAX_ATOM()(), MAX_AA()(), MAX_AA_VARIANTS()() );
	int nbins(0);
	std::map< int, float > bin2charge;
	std::map< float, int > charge2bin;

	for ( int aa=1; aa<= MAX_AA()(); ++aa ) {
		for ( int aav=1; aav<= nvar(aa); ++aav ) {
			for ( int j=1; j<= natoms(aa,aav); ++j ) {
				if ( std::abs( simple_charge(j,aa,aav)) < 1e-2 ) {
					// want exactly zero!
					simple_charge( j, aa, aav ) = 0.0;
					charge_bin( j,aa,aav ) = 0;
				} else {
					float const q( simple_charge(j,aa,aav));

					std::cout << "simple_charge: " << j << ' ' << aa << ' ' << aav <<
						' ' << q << std::endl;

					if ( charge2bin.count( q ) ) {
						charge_bin(j,aa,aav) = charge2bin[q];
					} else {
						++nbins;
						charge2bin[q] = nbins;
						bin2charge[nbins] = q;
						charge_bin(j,aa,aav) = nbins;
					}
				}
			}
		}
	}


	std::cout << "Found " << nbins << " total charge bins" << std::endl;

	//////////////////////////////
	//// fill in the etable values

	simple_elecE_table.dimension( etable_disbins, nbins, nbins );

	float const dis2_step = 1.0 / fa_bins_per_A2;

 	float const max_dis2( pdbstatistics_pack::fa_max_dis2 );
	float const min_dis2( 1.5 * 1.5 );

	float const die( 10.0 );

	for ( int bin1=1; bin1<= nbins; ++bin1 ) {
		float const q1( bin2charge[ bin1] );
		for ( int bin2=1; bin2<= nbins; ++bin2 ) {
			float const q2( bin2charge[ bin2] );
			for ( int disbin=1; disbin <= etable_disbins; ++disbin ) {

				//
				float dis2 = ( disbin-1 ) * dis2_step;

				// calculate electrostatics value
				float elecE(0.0);
				if ( dis2 <= max_dis2 ) {
					if ( dis2 < min_dis2 ) dis2 = min_dis2;
					elecE = 322.0637 * q1 * q2 * ( 1 / dis2 - 1 / max_dis2 ) / die;
				}
				simple_elecE_table( disbin, bin1, bin2 ) = elecE;
			}
		}
	}

	}

///////////////////////////////////////////////////////////////////////////////
// a complete misnomer since this is not gb anymore.

float
gb_get_res_res_simple_elecE(
	int const seqpos1,
	int const aa1,
	int const aav1,
	FArray2Da_float coord1,
	int const seqpos2,
	int const aa2,
	int const aav2,
	FArray2Da_float coord2
)
{
	coord1.dimension( 3, param::MAX_ATOM()() );
	coord2.dimension( 3, param::MAX_ATOM()() );

	// currently using shifted potential w/o smoothing
	//static float const max_dis2( 10000.0 );
 	static float const max_dis2( pdbstatistics_pack::fa_max_dis2 );
	static float const min_dis2( 1.5 * 1.5 );

	static float const C0( 322.0637 );
	static float const die( 10.0 );
	static float const C1( C0 / die );
	static float const C2( C1 / max_dis2 );

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace gb_elec;

	int const natoms1( natoms(aa1,aav1) );
	int const natoms2( natoms(aa2,aav2) );

	float elecE( 0.0 );

	for ( int atom1=1; atom1<= natoms1; ++atom1 ) {
		float const q1( simple_charge(atom1,aa1,aav1) );
		if ( q1 == 0.0 ) continue;
		else assert( std::abs(q1)>1e-2 ); // sanity

		assert( fullatom_type(atom1,aa1,aav1) != 23 && // nonpolar H
						fullatom_type(atom1,aa1,aav1) != 24 ); // aromatic H

		int c1_l( coord1.index(1,atom1) );
		float const x1( coord1[   c1_l ] );
		float const y1( coord1[ ++c1_l ] );
		float const z1( coord1[ ++c1_l ] );

		int const atom2_begin( seqpos1 == seqpos2 ? atom1+1 : 1 );

		for ( int atom2=atom2_begin, c2_l=coord2.index(1,atom2),
						ac_l=simple_charge.index(atom2,aa2,aav2); atom2<= natoms2;
					++atom2, c2_l += 3, ++ac_l ){

			float const q2( simple_charge[ ac_l ] );
			assert( q2 == simple_charge(atom2,aa2,aav2) );
			if ( q2 == 0.0 ) continue;
			else assert( std::abs(q2)>1e-2 ); // sanity

			assert( x1 == coord1(1,atom1) && coord2[ c2_l   ] == coord2(1,atom2) &&
							y1 == coord1(2,atom1) && coord2[ c2_l+1 ] == coord2(2,atom2) &&
							z1 == coord1(3,atom1) && coord2[ c2_l+2 ] == coord2(3,atom2) );

			float d2
				( square ( x1 - coord2[ c2_l   ] ) +
					square ( y1 - coord2[ c2_l+1 ] ) +
					square ( z1 - coord2[ c2_l+2 ] ) );

			if ( d2 > max_dis2 ) continue;
			else if ( d2 < min_dis2 ) d2 = min_dis2;

			// note: C2 = C1 / max_dis2
			assert( std::abs( C1 / max_dis2 - C2 ) < 1e-2 );

			elecE += q1 * q2 * ( C1 / d2 - C2 );
		}
	}
	return elecE;

}

///////////////////////////////////////////////////////////////////////////////
// a complete misnomer since this is not gb anymore.

float
gb_get_res_res_simple_elecE_fast(
	int const seqpos1,
	int const aa1,
	int const aav1,
	FArray2Da_float coord1,
	int const seqpos2,
	int const aa2,
	int const aav2,
	FArray2Da_float coord2
)
{
	// currently using shifted potential w/o smoothing
 	static float const max_dis2( pdbstatistics_pack::safe_max_dis2 );

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace pdbstatistics_pack;
	using namespace gb_elec;

	int const natoms1( natoms(aa1,aav1) );
	int const natoms2( natoms(aa2,aav2) );

	float elecE( 0.0 );

	// reuse
	int bin2, disbin1, l1, l2;
	float frac, d2, d2_bin, e1, e2;

	int const table_size1( simple_elecE_table.size1() );
	int const table_size2( simple_elecE_table.size2() );

	for ( int atom1=1; atom1<= natoms1; ++atom1 ) {
		int const bin1( charge_bin(atom1,aa1,aav1) );
		if ( bin1 == 0 ) continue;

		int const bin1_offset( table_size1 * table_size2 * (bin1-1) );

		int c1_l( coord1.index(1,atom1) );
		float const x1( coord1[   c1_l ] );
		float const y1( coord1[ ++c1_l ] );
		float const z1( coord1[ ++c1_l ] );

		int const atom2_begin( seqpos1 == seqpos2 ? atom1+1 : 1 );

		for ( int atom2=atom2_begin, c2_l=coord2.index(1,atom2),
						cb_l=charge_bin.index(atom2,aa2,aav2); atom2<= natoms2;
					++atom2, c2_l += 3, ++cb_l ){

			bin2 = charge_bin[ cb_l ];
			assert( bin2 == charge_bin(atom2,aa2,aav2) );
			if ( bin2 == 0 ) continue;

			assert( x1 == coord1(1,atom1) && coord2[ c2_l   ] == coord2(1,atom2) &&
							y1 == coord1(2,atom1) && coord2[ c2_l+1 ] == coord2(2,atom2) &&
							z1 == coord1(3,atom1) && coord2[ c2_l+2 ] == coord2(3,atom2) );

			d2 = ( square ( x1 - coord2[ c2_l   ] ) +
						 square ( y1 - coord2[ c2_l+1 ] ) +
						 square ( z1 - coord2[ c2_l+2 ] ) );

			if ( d2 > max_dis2 ) continue;

			d2_bin = fa_bins_per_A2 * d2;
			disbin1 = static_cast< int >( d2_bin ) /*+ 1*/;
			frac = d2_bin - ( disbin1 /*- 1*/ );

			l1 = bin1_offset + table_size1 * (bin2-1) + disbin1;
			l2 = l1+1;

			e1 = simple_elecE_table[ l1 ];
			e2 = simple_elecE_table[ l2 ];

			assert( e1 == simple_elecE_table( disbin1+1, bin2, bin1) &&
							e2 == simple_elecE_table( disbin1+2, bin2, bin1) );

			elecE  += ( e1 + frac * ( e2 - e1 ) );
		}
	}
	return elecE;

}

//////////////////////////////////////////////////////////////////////////
//
// WARNING:: by analogy with eval_dE_dR returns dE_dR / r
//

float
gb_simple_elec_eval_dE_dR_over_r(
	int const seqpos1,
	int const atom1,
	int const aa1,
	int const aav1,
	FArray1Da_float coord1,
	int const seqpos2,
	int const atom2,
	int const aa2,
	int const aav2,
	FArray1Da_float coord2,
	bool const update_f1f2,
	FArray1Da_float f1,
	FArray1Da_float f2
)
{
	if ( !gb_elec::use_simple_elec ) {
		std::cout << "wrong deriv fxn!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	if ( seqpos1 == seqpos2 && atom1 == atom2 ) return 0.0;

	using gb_elec::simple_charge;

	static float const C0( 322.0637 );
	static float const die( 10.0 );
	static float const dEfac( -2.0 * C0 / die );
 	static float const max_dis2( pdbstatistics_pack::fa_max_dis2 );
	static float const min_dis2( 1.5 * 1.5 );

	// get atomic charges
	float const q1( simple_charge( atom1, aa1, aav1 ) );
	float const q2( simple_charge( atom2, aa2, aav2 ) );
	// return if either neutral
	if ( q1 == 0.0 || q2 == 0.0 ) return 0.0;

	coord1.dimension(3);
	coord2.dimension(3);

	float dis2(0.0);

	if ( update_f1f2 ) {
		f1.dimension(3);
		f2.dimension(3);
		f2(1) = coord1(1) - coord2(1);
		f2(2) = coord1(2) - coord2(2);
		f2(3) = coord1(3) - coord2(3);
		dis2 = ( f2(1) * f2(1) ) + ( f2(2) * f2(2) ) + ( f2(3) * f2(3) );
		cros(coord1,coord2,f1);
	} else {
		dis2 =
			square( coord1(1) - coord2(1) ) +
			square( coord1(2) - coord2(2) ) +
			square( coord1(3) - coord2(3) );
	}

	if ( dis2 > max_dis2 ) return 0.0;
	else if ( dis2 < min_dis2 ) return 0.0; // flat in this region

	return dEfac * q1 * q2 / ( dis2 * dis2 );
}

///////////////////////////////////////////////////////////////////////////////
