// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14781 $
//  $Date: 2007-05-08 13:35:56 -0400 (Tue, 08 May 2007) $
//  $Author: jiangl $

#include "gl_graphics.h"

#ifdef GL_GRAPHICS
#include "after_opts.h"
#include "atom_is_backbone.h"
#include "aaproperties_pack.h"
#include "counters.h"
#include "enzyme.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "initialize.h"
#include "make_pdb.h"
#include "misc.h"
#include "monte_carlo.h"
#include "namespace_low_pose.h"
#include "namespace_trajectory.h"
#include "native.h"
#include "param.h"
#include "param_aa.h"
#include "protein_graphics.h"
#include "read_aaproperties.h"
#include "score_ns.h"

#ifdef MAC
#include <GLUT/glut.h>
#endif

#ifndef MAC
#include "GL/glut.h"
#endif


#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>

#include <cstdlib>
#include <iostream>
#include <pthread.h>

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//// internal functions:
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

namespace graphics { // forward declaration
	class FAS;
	class Frame;
	protein_graphics::GraphicsState
	  current_gs( protein_graphics::SHOW_CARTOON,
								protein_graphics::SHOW_WIREFRAME,
								protein_graphics::SHOW_LIG_SPACEFILL,
								protein_graphics::RESIDUE_CPK_COLOR,
								protein_graphics::SHOW_ALL_TRIALS );

}

void
show_scores(
						graphics::FAS const & scores,
						std::string const & name
);

void
draw_score_plot(
	 std::vector<float> & score_vecter,
	 bool virtical,
	 float & score_min,
	 float & score_max
);

void
draw_box_boundary( xyzVector_float const & line_color );

void
frame_display(
							graphics::Frame const & f,
							std::string const & name
);

void
low_accept(
	graphics::Frame const & f,
	bool const copy_coords = true
);

void
log_display();

void
position_window_init();

void
text_window_init();

void
trajectory_window_init();

void
monte_carlo_windows_init();

void
listen_to_rosetta( void ) ;

void
position_display( void ) ;

void
text_window_display();

void
trajectory_display( void ) ;

void
best_display( void ) ;

void
low_display( void ) ;

void
FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	FArray1DB_char const & secstruct,
	FArray2DB_float const & occ,
	bool const fullatom
);

void
FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	FArray1DB_char const & secstruct,
	FArray2DB_float const & occ,
	bool const fullatom,
	std::vector< graphics::Bond > const & hbonds,
	int const anchor_pos = -1
);

void
FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	bool const fullatom,
	std::vector< graphics::Bond > const & hbonds,
	int const anchor_pos
);

void
ShowScores( float score );

void
proteinViewReshape( int, int );

void
glVertex3fxyz( numeric::xyzVector_float  coord);

void
drawsequence();

void
broadcast_okay_to_start();

void
writeString( std::string const & text_string );

void
init_rainbow_color( int const nres );

void
init_sequencedependent_color(
	int const nres,
	FArray1DB_int const & res
);

void
clear_residue_color();

void
append_residue_color( numeric::xyzVector_float thecolor);

void
processMouse(int button, int state, int x, int y);

void
processMouseActiveMotion(int x, int y);

void
keyPressFunc(unsigned char key, int x, int y);

void
textWindowKeyPressFunc(unsigned char key, int x, int y);

void
textWindowSpecialKeyPressFunc( int key, int, int );

void
specialKeyPressFunc(int key, int x, int y);

void
redrawallwindows();

void
redraw_text_window();

void
get_score_rms_from_trajectory(
		std::vector< graphics::T_point > const & data,
		std::vector< float > & score,
		std::vector< float > & rms,
		protein_graphics::GraphicsState const & gs);

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//// graphics namespace:
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

namespace graphics{
  bool gl_graphics = { true };
  bool worker_done = { false };
  bool wait_on_the_graphics = { false };
  bool pause_after_graphics  = { false };
  bool new_drawing  = { true };
  pthread_mutex_t position_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_cond_t position_cond = PTHREAD_COND_INITIALIZER;
  pthread_mutex_t start_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_cond_t start_cond = PTHREAD_COND_INITIALIZER;
  pthread_mutex_t trajectory_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t log_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t current_frame_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t best_frame_mut = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t low_frame_mut = PTHREAD_MUTEX_INITIALIZER;

	// windows
  int pos_window(999), best_window(999), low_window(999), trajectory_window(999), text_window(999);
  float pixelsPerAngstrom = { 10 };
  int window_size = 30;
	int specialKey = 0;
  bool click;
  bool clicked_button;
  int click_x;
  int click_y;
	bool backgroundblack;
  bool draw_centroids = { false };
	bool rainbow_color = { true };
	//bool gray_backbone = { true }; // PBHACK
	bool peptide_planes( false );

	// pose_graphics stuff
	bool pose_graphics( false );
	bool new_pose( false );
	bool new_best_pose( false );
	bool new_low_pose( false );
	bool new_log( false );
	//int nres_pose( 0 );
	//int nres_best_pose( 0 );
	//int nres_low_pose( 0 );
	//int default_anchor_pos( 0 );
	float bondE_threshold( 0.2 );
	FArray3D_int intxn_atm( param::MAX_ATOM(), param::MAX_AA(),
													param::MAX_AA_VARIANTS() );
	std::vector< std::string > logs;
	std::string title;
	std::string command_string;
	int current_command(0);
	std::vector< std::string > commands;
	std::vector< std::string > param_tags;

	// user configurable params
	std::map< std::string, float > f_param;
	std::map< std::string, std::string > s_param;

	// score trajectory data
	enum T_type {
		RESET,
		TRIAL,
		NEW_SCORING_FUNCTION,
		RECOVER_LOW,
		NOMC_TRIAL
	};

	class T_point {
	public:
		float score;
		float best_score;
		float low_score;
		float rms_err;
		std::string move_type;
		int accept_type;
		T_type type;

		bool best_accept() const { return score - best_score < 1e-3; }
		bool low_accept() const { return score - low_score < 1e-3; }
		bool mc_accept() const { return type != NOMC_TRIAL ; }

		T_point(){}
		T_point(
			float const s,
			float const bs,
			float const ls,
			float const d,
			std::string mt,
			int const at,
			T_type const t
			):
			score(s),
			best_score(bs),
			low_score(ls),
			rms_err(d),
			move_type(mt),
			accept_type(at),
			type(t)
		{}

		T_point(
			float const s,
			float const bs,
			float const ls,
			float const d,
			T_type const t
			):
			score(s),
			best_score(bs),
			low_score(ls),
			rms_err(d),
			move_type(),
			accept_type(),
			type(t)
		{}
	}; // class T_point

	std::vector< T_point > trajectory;
	bool show_all_trials( false );
	bool label_low_accepts( true );

	class Bond {
	public:
		int seqpos1;
		int atm1;
		int seqpos2;
		int atm2;
		float bondE;
		Bond_type type;
		Bond(
					int const s1,
					int const a1,
					int const s2,
					int const a2,
					float const E,
					Bond_type const t
					):
			seqpos1(s1),
			atm1(a1),
			seqpos2(s2),
			atm2(a2),
			bondE(E),
			type(t)
		{}
	};

	std::vector< Bond > tmp_bond_vector;

	// fullatom score object
	class FAS {
	public:
		float score;
		float bk_tot;
		float fa_atr;
		float fa_rep;
		float fa_sol;
		float fa_dun;
		float fa_h2o;
		float fa_h2o_hb;
		float fa_gb_elec;
		float hb_srbb;
		float hb_lrbb;
		float hb_sc;
		std::vector< Bond > bonds;
		FArray1D_float dunenergy;
		FArray1D_float resenergy;

		void
		setup(
			float const _score,
			std::vector< Bond > const & _bonds
		)
		{
			score = _score;
			bk_tot = fullatom_energies::totals::fullatomE;
			fa_atr     = scores::fa_energies::fa_atr_score;
			fa_rep     = scores::fa_energies::fa_rep_score;
			fa_sol     = scores::fa_energies::fa_solv_score;
			fa_dun     = scores::fa_energies::fa_dun_score;
			fa_h2o     = scores::fa_energies::fa_h2o_score;
			fa_h2o_hb  = scores::fa_energies::fa_h2o_hb_score;
			fa_gb_elec = scores::fa_energies::fa_gb_elec_score;
			hb_srbb = scores::hbond_scores::hb_srbb_score;
			hb_lrbb = scores::hbond_scores::hb_lrbb_score;
			hb_sc   = scores::hbond_scores::hb_sc_score;
			bonds = _bonds;
			// 1d energies
			int const nres( misc::total_residue );
			if ( int(dunenergy.size1()) != nres ) dunenergy.dimension(nres);
			if ( int(resenergy.size1()) != nres ) resenergy.dimension(nres);
			for ( int i=1; i<= nres; ++i ) {
				dunenergy(i) = fullatom_energies::dunenergy(i);
				resenergy(i) = fullatom_energies::resenergy(i);
			}
		}
	}; // FAS

	FAS best_scores;

	/////////////////////////////////////////////////////////////////////////////
	class Frame {
	public:
		int nres;
		bool fullatom;
		int anchor_pos;
		FArray3DB_float const * xyz;
		FArray1D_int    const * res;
		FArray1D_int    const * resv;
		FArray2D_float const * occ;
		FArray1D_char  const * secstruct;
		FAS scores;
		bool owner;

		// make a new frame from an old frame
		// we dont start out owning the coordinate data
		Frame( Frame const & src ):
			nres( src.nres ),
			fullatom( src.fullatom ),
			anchor_pos( src.anchor_pos ),
			xyz ( src.xyz  ),
			res ( src.res  ),
			resv( src.resv ),
			occ( src.occ ),
			secstruct( src.secstruct ),
			scores( src.scores ),
			owner( false )
		{}

		// claim data as your own
		void
		copy()
		{
			assert(!owner);
			owner = true;
			xyz  = new FArray3D_float( *xyz );
			res  = new FArray1D_int  ( *res  );
			resv = new FArray1D_int  ( *resv );
			secstruct = new FArray1D_char  ( *secstruct  );
			occ = new FArray2D_float  ( *occ );
		}

		~Frame() {
			if ( owner ) {
				delete xyz;
				delete res;
				delete resv;
				delete secstruct;
				delete occ;
			}
		}

		Frame():
			owner( false )
		{}

	}; // class Frame

	Frame current_frame;
	Frame best_frame;
	std::vector< Frame > low_accepts;
	int current_low_accept( 0 );
}

namespace residue_color {
  std::vector <float> red;
  std::vector <float> green;
  std::vector <float> blue;

}

///////////////////////////////////////////////////////////////////////////////
inline
float
graphics_f_param(
								 std::string const & tag,
								 float const d
								 )
{
	using graphics::f_param;
	using graphics::param_tags;

	// this could get slow
	if ( std::find( param_tags.begin(), param_tags.end(), tag ) ==
			 param_tags.end() ) {
		param_tags.push_back( tag );
	}

	if ( f_param.count( tag ) ){
		return f_param.find( tag )->second;
	} else {
		return d;
	}
}


///////////////////////////////////////////////////////////////////////////////
inline
std::string const &
graphics_s_param(
								 std::string const & tag,
								 std::string const & d
								 )
{
	using graphics::s_param;
	using graphics::param_tags;

	// this could get slow
	if ( std::find( param_tags.begin(), param_tags.end(), tag ) ==
			 param_tags.end() ) {
		param_tags.push_back( tag );
	}

	if ( s_param.count( tag ) ){
		return s_param.find( tag )->second;
	} else {
		return d;
	}
}


///////////////////////////////////////////////////////////////////////////////

inline
graphics::Frame &
low_frame()
{
	using namespace graphics;
	assert( low_accepts.size() );
	return low_accepts[ low_accepts.size() - 1];
}

///////////////////////////////////////////////////////////////////////////////
void
low_accept(
					 graphics::Frame const & f,
					 bool const copy_coords // = true
					 )
{
	using namespace graphics;
	static bool init( false );
	if ( !init ) {
		// reduce memory overruns
		init = true;
		low_accepts.reserve( 1000 );
	}

	pthread_mutex_lock(&low_frame_mut);
	low_accepts.push_back( f ); // calls copy-constructor
	if ( copy_coords ) low_frame().copy(); // now we own the coords
	if ( current_low_accept >= int(low_accepts.size()) - 2 ) {
		current_low_accept = low_accepts.size() - 1;
		new_low_pose = true;
	}
	pthread_mutex_unlock(&low_frame_mut);
}


///////////////////////////////////////////////////////////////////////////////
void
setup_intxn_atom()
{
	using graphics::intxn_atm;
	using namespace aaproperties_pack;
	using namespace param_aa;
	for ( int aa=1; aa<= param::MAX_AA()(); ++aa ) {
		for ( int aav=1; aav<= nvar(aa); ++aav ) {
			for ( int i=1; i<= natoms(aa,aav); ++i ) {
				std::string const & name( atom_name(i,aa,aav) );
				if ( is_protein( aa ) ) {
					if ( atom_is_backbone(i,aa,aav) ) {
						intxn_atm( i,aa,aav) = 2; // C-alpha
					} else {
						if ( aa == aa_arg ) {
							intxn_atm(i,aa,aav) = LookupByName(aa,aav," CZ " );
						} else if ( aa == aa_lys ) {
							intxn_atm(i,aa,aav) = LookupByName(aa,aav," NZ " );
						} else if ( aa == aa_glu || aa == aa_gln ) {
							intxn_atm(i,aa,aav) = LookupByName(aa,aav," CD " );
						} else if ( aa == aa_asp || aa == aa_asn ) {
							intxn_atm(i,aa,aav) = LookupByName(aa,aav," CG " );
						} else {
							intxn_atm(i,aa,aav) = 5; // c-beta
						}
					}
				} else if ( is_DNA( aa ) ) {
					if ( name[1] == 'P' || name[3] == 'P' || name[3] == '*' ) {
						intxn_atm(i,aa,aav) = LookupByName(aa,aav," O5*" );
					} else {
						intxn_atm(i,aa,aav) = LookupByName(aa,aav," C4 " );
					}
				} else {
					intxn_atm(i,aa,aav) = i;
				}
			}
		}
	}

}

void
gl_graphics_set_title( std::string const & t )
{
	graphics::title = t;
}


float
gl_graphics_bond_energy_threshold()
{
	return graphics::bondE_threshold;
}

void
writeStrokeString(
	const std::string & text_string,
	float const xscale,
	float const yscale
)
{
	char const *temp_cstring = text_string.c_str();
	glPushMatrix();
	glScalef(xscale,yscale,1.0);
	glDisable( GL_BLEND );
	glDisable( GL_DEPTH_TEST );
	int len = (int) strlen(temp_cstring);
	for (int i = 0; i < len; i++) {
		glutStrokeCharacter(GLUT_STROKE_ROMAN, temp_cstring[i]);
	}
	glPopMatrix();
	glEnable( GL_BLEND );
	glEnable( GL_DEPTH_TEST );

}

inline
void
glColor3fxyz(
	numeric::xyzVector_float const & c
)
{
	glColor3f(c(1),c(2),c(3));
}


void*
graphics_thread_wrapper(void* )
{
  int argc = 1;
  char* s = "x";
  char** argv = &s;
  main_glut(argc, argv);
  return NULL;
}

int
main_glut(int argc, char** argv)
{
  using namespace graphics;
  pthread_mutex_lock( &start_mut );
  pthread_cond_wait( &start_cond, &start_mut );
  pthread_mutex_unlock( &start_mut );
  init_rainbow_color( misc::total_residue );
	backgroundblack = true;
  glutInit(&argc, argv);
  glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
	position_window_init();
//   trajectory_window_init();
//   text_window_init();
//   monte_carlo_windows_init();
  glutIdleFunc( listen_to_rosetta );
  glutMainLoop();
  return 0;
}

typedef numeric::xyzVector_float rgbcolor; // This isn't quite right, should make a new struct.

namespace rgbcolors {
	rgbcolor colorred    (1.0, 0.0, 0.0);
	rgbcolor colorblue   (0.0, 0.0, 1.0);
	rgbcolor colorgray   (0.5, 0.5, 0.5);
	rgbcolor colorbrown  (0.8, 0.5, 0.5);
	rgbcolor colorgreen  (0.0, 1.0, 0.0);
	rgbcolor coloryellow (1.0, 1.0, 0.0);
	rgbcolor colormagenta(1.0, 0.0, 1.0);
	rgbcolor colororange (1.0, 0.5, 0.0);
	rgbcolor colordarkblue(0.0, 0.0, 0.5);
}


///////////////////////////////////////////////////////////////////////////////
void
plot_trajectory(
	std::vector< graphics::T_point > const & data
)
{
	using namespace graphics;

	if ( data.size() < 1 ) return;

	/////////////////
	//// setup colors
	static std::vector< rgbcolor > colors;
	static bool init( false );
	if ( !init ) {
		init = true;
		using namespace rgbcolors;
		colors.push_back( colorred );
		colors.push_back( colorblue );
		colors.push_back( colorgray );
		colors.push_back( colorbrown );
		colors.push_back( colorgreen );
		colors.push_back( coloryellow );
		colors.push_back( colormagenta );
		colors.push_back( colororange );
		colors.push_back( colordarkblue );
	}


	//////////////////////////////////////
	//// figure out min/max, number of steps
	//
	// state:
	//  show_all_trials -- should we show all trials or only accepts
	//  user_max_score -- user specified max score
	//  user_min_trial -- user specified when to start showing trials
	//


	int total_steps(0);
	float min_score( data[0].best_score );
	float max_score( data[0].best_score );
	for ( unsigned int i=0; i<data.size(); ++i ) {
		T_point const & p( data[i] );
		if ( !p.best_accept() && !(p.mc_accept()&&show_all_trials) ) continue;
		++total_steps;
		max_score = std::max( p.best_score, max_score );
		min_score = std::min( p.best_score, min_score );
	}


	if ( total_steps > 0 ) {
		////////////////////////
		//// setup gl projection
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		float const xbuffer( total_steps/20 );
		float const ybuffer( (max_score-min_score)/20 );
		gluOrtho2D( 0-xbuffer, total_steps-1+xbuffer,
								min_score-ybuffer, max_score+ybuffer );

		glMatrixMode(GL_MODELVIEW);
		glDisable( GL_DEPTH_TEST );
		glLoadIdentity();



		////////////////////////////
		//// now make the trajectory

		glBegin( GL_LINE_STRIP );

		// different colors for different scorefxns
		int color(0);
		glColor3fxyz( colors[color] );

		std::vector< std::pair< int, int > > low_accept_steps;

		int x(-1);
		for ( unsigned int step = 0; step < data.size(); ++step ) {
			T_point const & p( data[step] );
			if ( !p.best_accept() && !(p.mc_accept()&&show_all_trials) ) continue;

			++x;

			graphics::T_type const t( p.type );

			if ( t == graphics::RECOVER_LOW ){
				glEnd();

				/////////////////////////////////////
				// make a square for each recover low
				float const xscale = .0025*float(total_steps-1);
				float const yscale = .0025*(max_score-min_score);
				float const y( p.best_score );
				glBegin( GL_QUADS );
				glVertex2f( x-xscale, y-yscale );
				glVertex2f( x+xscale, y-yscale );
				glVertex2f( x+xscale, y+yscale );
				glVertex2f( x-xscale, y+yscale );
				glEnd();

				glBegin( GL_LINE_STRIP );
			}

			if ( t == graphics::NEW_SCORING_FUNCTION ) {
				glEnd();
				// update the color
				++color;
				if ( color >= int(colors.size()) ) color = 0;
				glColor3fxyz( colors[color] );
				glBegin( GL_LINE_STRIP );
			}

			// show the move type for low accepts
			if ( t == TRIAL && p.low_accept() && label_low_accepts ) {
				low_accept_steps.push_back( std::make_pair( x, step ) );
			}

			glVertex2f( x, p.best_score );
		}
		glEnd();
		glLoadIdentity();

		////////////////////////
		// label the low-accepts
		if ( low_accept_steps.size() < 500 ) {

		std::map< std::string, int > labels;
		std::map< int, std::string > label_names;

		for ( unsigned int i=0; i< low_accept_steps.size(); ++i ) {
			int const x( low_accept_steps[i].first );
			int const step( low_accept_steps[i].second );
			float const plot_score
				( 0.5 * ( data[ step-1 ].low_score + data[step].low_score ));

			// setup a mapping between integers and the move type
			std::string const move_type( data[step].move_type );
			std::string tag;
			if ( !labels.count( move_type ) ) {
				int const label = labels.size()+1;
				labels.insert( std::make_pair( move_type, label ) );
				label_names.insert( std::make_pair( label, move_type ) );
			}
			tag = string_of( labels[move_type] );

			glLoadIdentity();
			glTranslatef( x-0.5, plot_score, 0 );
 			float const xscale = .00010*float(total_steps-1);
 			float const yscale = .00040*(max_score-min_score);
			writeStrokeString( tag, xscale, yscale );
		}

		// make a key
		for ( int i=1; i<= int(labels.size()); ++i ) {
 			float const xscale = .00010*float(total_steps-1);
 			float const yscale = .00040*(max_score-min_score);
			glLoadIdentity();
			glTranslatef( 0.0, max_score - 120 * i * yscale, 0 );
			writeStrokeString( string_of( i ) + ": " + label_names[ i ],
												 xscale, yscale );
		}

		glEnable( GL_DEPTH_TEST );
		glLoadIdentity();
		}
	}
}

bool
wait_on_graphics() {
  using namespace graphics;
  if (!gl_graphics) return false;

	static int counter(0);
	if ( false && ++counter%100 == 0 )
		std::cout << "wait_on_graphics: " << counter << ' ' <<
			wait_on_the_graphics << std::endl;

	static bool init = {false};
	if (!init) {
		wait_on_the_graphics = truefalseoption("wait_on_graphics");
		pause_after_graphics = truefalseoption("pause_after_graphics");
		init = true;
	}

	if (!wait_on_the_graphics) return false;

  // should this lock come at the beginning of monte carlo?
  pthread_mutex_lock( &position_mut );
  // wait until graphics broadcasts that it is done drawing
  pthread_cond_wait( &position_cond, &position_mut );
  pthread_mutex_unlock( &position_mut );

	//	while (pause_after_graphics) {
		//		std::cout << "Hit return to end.";
		//std::string dummy;
		//std::cin >> dummy;
	//}

	return pause_after_graphics;
}


void
position_window_init() {

  glutInitWindowSize (900, 900);
  glutInitWindowPosition (370, 10);
  glutCreateWindow ( "ROSETTA Searching ..." );
  glutDisplayFunc( position_display );
	//  glClearColor (0.0, 0.0, 0.0, 0.0);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glEnable( GL_DEPTH_TEST );

  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

	glutMouseFunc(processMouse);
	glutMotionFunc(processMouseActiveMotion);
	glutKeyboardFunc(keyPressFunc);
	//glutSpecialFunc(specialKeyPressFunc);

	glLoadIdentity();

  graphics::pos_window = glutGetWindow();
}


///////////////////////////////////////////////////////////////////////////////
void
text_window_init() {
  bool static uninitialized = { true };
  if (uninitialized) {
    uninitialized = false;

  glutInitWindowSize(750, 250);
  glutInitWindowPosition (10,520);
  glutCreateWindow ( "text_window" );
  glutDisplayFunc( text_window_display );
	//  glClearColor (0.0, 0.0, 0.0, 0.0);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho( -graphics::window_size,
					 graphics::window_size,
					 -graphics::window_size,
					 graphics::window_size,
					 -10*graphics::window_size,
					 10*graphics::window_size );
  glEnable( GL_DEPTH_TEST );
  glMatrixMode(GL_MODELVIEW);

  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

	//glutMouseFunc(processMouse);
	//glutMotionFunc(processMouseActiveMotion);
	glutKeyboardFunc(textWindowKeyPressFunc);
	glutSpecialFunc(textWindowSpecialKeyPressFunc);

  graphics::text_window = glutGetWindow();
	}
}


///////////////////////////////////////////////////////////////////////////////
//
void trajectory_key_press_func(unsigned char key, int, int) {
	using namespace graphics;

	switch (key) {
	case 'A':
	case 'a':
		show_all_trials = !show_all_trials;
		break;
	}
	glutSetWindow( trajectory_window );
	trajectory_display();
	glutPostRedisplay();
}

void
trajectory_window_init() {
  bool static uninitialized = { true };
  if (uninitialized) {
    uninitialized = false;

  glutInitWindowSize (250, 250);
  glutInitWindowPosition (520, 10);
  glutCreateWindow ( "trajectory" );
  glutDisplayFunc( trajectory_display );
	//  glClearColor (0.0, 0.0, 0.0, 0.0);

	if ( true ) {
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glOrtho(
			-graphics::window_size, graphics::window_size, -graphics::window_size,
			graphics::window_size, -10*graphics::window_size,
			10*graphics::window_size);
		glEnable( GL_DEPTH_TEST );
		glMatrixMode(GL_MODELVIEW);

		glEnable(GL_LINE_SMOOTH);
		glEnable(GL_POINT_SMOOTH);
		glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

// 		glutMouseFunc(processMouse);
// 		glutMotionFunc(processMouseActiveMotion);
// 		glutKeyboardFunc(keyPressFunc);
	}
	glutKeyboardFunc( trajectory_key_press_func );

  graphics::trajectory_window = glutGetWindow();
	}
}

void
monte_carlo_windows_init() {
  bool static uninitialized = { true };
  if (uninitialized) {
    uninitialized = false;

  glutInitWindowSize (750, 900);
  glutInitWindowPosition (10, 10);
  glutCreateWindow ( "Monte Carlo Best Position" );
  glutDisplayFunc( best_display );
	//  glClearColor (0.0, 0.0, 0.0, 0.0);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-graphics::window_size, graphics::window_size, -graphics::window_size, graphics::window_size, -10*graphics::window_size, 10*graphics::window_size);
  glEnable( GL_DEPTH_TEST );
  glMatrixMode(GL_MODELVIEW);

  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glutMouseFunc(processMouse);
	glutMotionFunc(processMouseActiveMotion);
	glutKeyboardFunc(keyPressFunc);
	//glutSpecialFunc(specialKeyPressFunc);

  graphics::best_window = glutGetWindow();


  glutInitWindowSize (750, 900);
  glutInitWindowPosition (520,295);
  glutCreateWindow ( "Monte Carlo Low Position" );
  glutDisplayFunc( low_display );
	//  glClearColor (0.0, 0.0, 0.0, 0.0);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-graphics::window_size, graphics::window_size, -graphics::window_size, graphics::window_size, -10*graphics::window_size, 10*graphics::window_size);
  glEnable( GL_DEPTH_TEST );

  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

  glMatrixMode(GL_MODELVIEW);

	glutMouseFunc(processMouse);
	glutMotionFunc(processMouseActiveMotion);
	glutKeyboardFunc(keyPressFunc);
	glutSpecialFunc(specialKeyPressFunc);

  graphics::low_window = glutGetWindow();
  }

}

///////////////////////////////////////////////////////////////////////////////
void
listen_to_rosetta( void )
{
  using namespace graphics;

  if ( graphics::worker_done == true ) {
    exit( 0 );
  }
  // make drawing mutually exclusive with changing the position arrays?
  pthread_mutex_lock(&position_mut);
  int accept = get_monte_carlo_accept();
/// return values:
///                3 = accepted:score beat low score and best_score,
///                2 = accepted:score beat best_score,
///                1 = thermally accepted: score worse than best_score
///                0 = not accepted,


	if ( !pose_graphics ) {
		// old style
		glutSetWindow( pos_window );
		glutPostRedisplay();

		if ( accept > 0 ) {
			monte_carlo_windows_init();
			glutSetWindow( best_window );
			glutPostRedisplay();
		}
		if ( accept == 3 ) {
			glutSetWindow( low_window );
			glutPostRedisplay();
		}
	} else {
		// pose-enabled graphics
		if ( new_pose ) {
			new_pose = false;
			glutSetWindow( pos_window );
			glutPostRedisplay();
		}

		if ( new_best_pose ) {
			new_best_pose = false;
			monte_carlo_windows_init();
			glutSetWindow( best_window );
			glutPostRedisplay();

			trajectory_window_init();
			glutSetWindow( graphics::trajectory_window );
			glutPostRedisplay();

		}

		if ( new_low_pose ) {
			new_low_pose = false;
			glutSetWindow( low_window );
			glutPostRedisplay();
		}

		if ( new_log ) {
			redraw_text_window();
		}

	} // pose_graphics?


  pthread_cond_broadcast(&position_cond);
  pthread_mutex_unlock(&position_mut);
}


///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_log(
	std::string const & msg
)
{
	using namespace graphics;
	int const max_logs( static_cast<int>( graphics_f_param( "max_logs",10.0)));

	std::cout << msg << std::endl;
	pthread_mutex_lock(&log_mut);
	logs.push_back(msg);
	if ( int(logs.size()) > max_logs ) {
		logs.erase( logs.begin() );
	}
	pthread_mutex_unlock(&log_mut);
	new_log = true;
}


///////////////////////////////////////////////////////////////////////////////
//
// called by pose refold
//
// note that this does not fill the scores info in current_frame
//
// that's done by save_scores which is called by scorefxn
//
void
gl_graphics_new_pose(
	int const nres,
	bool const fullatom,
	int const anchor_pos
)
{
	using namespace graphics;
	// setup the current frame
	pthread_mutex_lock(&current_frame_mut);
	current_frame.nres = nres;
	current_frame.fullatom = fullatom;
	current_frame.anchor_pos = anchor_pos;
	if ( fullatom ) current_frame.xyz = &misc::full_coord;
	else current_frame.xyz = &misc::Eposition;
	current_frame.res  = &misc::res;
	current_frame.resv = &misc::res_variant;
	current_frame.secstruct = &misc::secstruct;
	current_frame.occ = &pdb::occupancy;

	pthread_mutex_unlock(&current_frame_mut);

	while ( wait_on_graphics() ) {
	}

}

void
gl_graphics_new_pose(){
	using namespace graphics;
	pose_graphics = true;
	new_pose = true;
}

///////////////////////////////////////////////////////////////////////////////
// after an accept
void
update_best_frame(
	graphics::Frame const & f
)
{
	using namespace graphics;
	pthread_mutex_lock(&best_frame_mut);
	best_frame.nres       = f.nres;
	best_frame.fullatom   = f.fullatom;
	best_frame.anchor_pos = f.anchor_pos;
	best_frame.scores     = f.scores;

	// connect pointers -- really only needs to be done once
	// and whenever we switch fullatom<->centroid
	if ( f.fullatom ) best_frame.xyz = &(misc::best_full_coord );
	else best_frame.xyz = &(misc::Ebest_position);
	best_frame.res  = &(misc::best_res);
	best_frame.resv = &(misc::best_res_variant);
	best_frame.secstruct = &(misc::secstruct);
	best_frame.occ = &(pdb::occupancy);

	// signal to redraw
	graphics::new_best_pose = true;
	pthread_mutex_unlock(&best_frame_mut);
}

///////////////////////////////////////////////////////////////////////////////
void
setup_colors(
	int const nres,
	FArray1DB_int const & res
)
{
	using namespace graphics;
	// shouldnt need to do this! -- make per-res function calls inside routines?
	if ( rainbow_color ) {
		init_rainbow_color( nres );
	} else {
		init_sequencedependent_color( nres, res );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_mc_trial(
	float const score,
	float const best_score,
	float const low_score,
	float const rms_err,
	std::string const & move_type,
	int const accept_type
)
{
	using namespace graphics;

	{
		std::string const type( accept_type == 0 ? "reject" :
														( accept_type == 1 ? "thermal-accept" :
															( accept_type == 2 ? "accept" : "low-accept" )));
		GRAPHICS_LOG( "trial: " + move_type + " " + type +
									fixed_string_of( score, 12, 3 ) +
									fixed_string_of( best_score, 12, 3 ) +
									fixed_string_of( low_score, 12, 3 ) );
	}

	// append to trajectory
  pthread_mutex_lock( &trajectory_mut );
	graphics::trajectory.push_back( T_point( score, best_score, low_score,
																					 rms_err, move_type, accept_type, TRIAL ) );
  pthread_mutex_unlock( &trajectory_mut );


	if ( accept_type >= 1 ) {
		// best_pose has changed
		update_best_frame( current_frame );

		if ( accept_type == 3 ){
			//			low_accept( current_frame ); // calls copy-constructor
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_reset_bonds()
{
	using namespace graphics;
	setup_intxn_atom();
	tmp_bond_vector.clear();
}


///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_save_scores(
												float const total_score,
												bool const fullatom
												)
{
	using namespace graphics;
  pthread_mutex_lock( &current_frame_mut );
	if ( fullatom ) current_frame.scores.setup( total_score, tmp_bond_vector );
	else {
		current_frame.scores.score = total_score;
		current_frame.scores.bonds.clear();
	}
	new_pose = true;
  pthread_mutex_unlock( &current_frame_mut );
}


///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_reset_bonds(
												FArray2DB_bool const & pair_moved,
												bool const update_born_radii
												)
{
	using namespace graphics;
	setup_intxn_atom();

	if ( update_born_radii ) {
		// if we are updating the born radii then we have to recalculate all
		// the residue-residue interactions anyhow
		tmp_bond_vector.clear();

	} else {
		// save gb intxns from non-moved pairs
		static std::vector< Bond > save_gb;
		save_gb.clear();
		for ( unsigned i=0; i< tmp_bond_vector.size(); ++i ) {
			Bond const & b( tmp_bond_vector[i] );
			if ( b.type == GB_BOND && !pair_moved( b.seqpos1, b.seqpos2 ) ) {
				save_gb.push_back( b );
			}
		}
		tmp_bond_vector = save_gb;
	}
}


///////////////////////////////////////////////////////////////////////////////
void
gl_graphics_store_bond(
	int const dres,
	int const dhatm,
	int const ares,
	int const aatm,
	float const hbE,
	graphics::Bond_type const type
)
{
	using namespace graphics;
	tmp_bond_vector.push_back( Bond(dres,dhatm,ares,aatm,hbE,type));
}


///////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////
// assumes that we just scored the low structure

void
gl_graphics_mc_new_scorefxn(
	float const best_score,
	float const low_score,
	float const best_rms
)
{
	using namespace graphics;

	// update trajectory
  pthread_mutex_lock( &trajectory_mut );
	graphics::trajectory.push_back( T_point( best_score, best_score, low_score,
																 best_rms, NEW_SCORING_FUNCTION ) );
  pthread_mutex_unlock( &trajectory_mut );

	// best frame scores were already stored by Monte_carlo::set_weight_map

	// update low_accepts
	low_accept( Frame( low_frame() ), // not sure if explicit construction is
							false );              // necessary. shouldnt happen too often

	// we just scored the low structure so copy the info
	{
		assert( current_frame.scores.score = low_score );
		pthread_mutex_lock  (&low_frame_mut);
		low_frame().scores = current_frame.scores;
		pthread_mutex_unlock(&low_frame_mut);
	}
}

///////////////////////////////////////////////////////////////////////////////
//
void
gl_graphics_mc_recover_low(
	float const low_score,
	float const low_rms
)
{
	using namespace graphics;

	// update trajectory
  pthread_mutex_lock( &trajectory_mut );
	graphics::trajectory.push_back( T_point( low_score, low_score, low_score,
																 low_rms, RECOVER_LOW ) );
  pthread_mutex_unlock( &trajectory_mut );


	// update current,best score info
	pthread_mutex_lock  (&current_frame_mut);
	current_frame.scores = low_frame().scores;
	pthread_mutex_unlock(&current_frame_mut);
	update_best_frame( low_frame() );


}

///////////////////////////////////////////////////////////////////////////////
// assumes that we just scored the pose
void
gl_graphics_save_best_scores()
{
	update_best_frame( graphics::current_frame );
}


///////////////////////////////////////////////////////////////////////////////
// assumes that we just scored the pose
void
gl_graphics_mc_reset(
	float const low_score,
	float const low_rms
)
{
	using namespace graphics;
	assert( std::abs( low_score - current_frame.scores.score ) < 1e-2 );

	// reset the trajectory
  pthread_mutex_lock( &trajectory_mut );
	graphics::trajectory.clear();
	graphics::trajectory.push_back( T_point( low_score, low_score, low_score,
																 low_rms, RESET ) );
  pthread_mutex_unlock( &trajectory_mut );

	// reset frames
	update_best_frame( current_frame );
	pthread_mutex_lock  (&low_frame_mut);
	low_accepts.clear();
	pthread_mutex_unlock(&low_frame_mut);
	low_accept( current_frame );
}

///////////////////////////////////////////////////////////////////////////////
void
get_bounds(
					 std::vector< graphics::T_point > const & t,
					 float & mn,
					 float & mx
					 )
{
	if ( !t.size() ) return;
	mn = t[0].score;
	mx = t[0].score;
	for ( int i=1, i_end= t.size(); i< i_end; ++i ) {
		mn = std::min( mn, t[i].score);
		mx = std::max( mx, t[i].score);
	}
}
///////////////////////////////////////////////////////////////////////////////
void
get_bounds(
					 std::vector< float > const & t,
					 float & mn,
					 float & mx
					 )
{
	if ( !t.size() ) return;
	mn = t[t.size()-1];
	mx = t[t.size()-1];
	int cut ( int(t.size()*0.35) );
	float midian;
	for ( int i=t.size()-2, i_end= cut; i> i_end; --i ) {
		mn = std::min( mn, t[i]);
		mx = std::max( mx, t[i]);
	}
}



///////////////////////////////////////////////////////////////////////////////
void
trajectory_display( void ) {
	using namespace graphics;

	//std::cout << "traj-display: " << trajectory.size() << std::endl;
	if ( graphics::trajectory.size() ) {
		glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
		//Set background color
		if (backgroundblack) glClearColor( 0.0, 0.0, 0.0, 1.0); //black
		else  glClearColor( 1.0, 1.0, 1.0, 1.0); // white
		pthread_mutex_lock( &trajectory_mut );
		plot_trajectory( graphics::trajectory );
		pthread_mutex_unlock( &trajectory_mut );
		glLoadIdentity();
		glutSwapBuffers(); // Draw it!
	}
}

///////////////////////////////////////////////////////////////////////////////
void
position_display( void )
{
	graphics::current_gs.Trajectory_state = protein_graphics::SHOW_ALL_TRIALS;

	if ( graphics::pose_graphics ) {
		using namespace graphics;
		// new way
		pthread_mutex_lock  (&current_frame_mut);

		setup_colors( current_frame.nres, *current_frame.res );

		frame_display( current_frame, title );
		pthread_mutex_unlock(&current_frame_mut);

	} else {
		// old way
		using namespace misc::ints;
		using namespace misc::current_pose;
		FArray_Calpha_display( total_residue, Eposition, centroid, res,
													 res_variant, secstruct, pdb::occupancy, false );
		ShowScores( mc_global_track::mc_score::score);
	}
  glutSwapBuffers(); // Draw it!
}

///////////////////////////////////////////////////////////////////////////////
void
best_display( void ) {
	graphics::current_gs.Trajectory_state = protein_graphics::SHOW_BEST;

	if ( graphics::pose_graphics ) {
		using namespace graphics;
		pthread_mutex_lock  (&best_frame_mut);
		// new way
		setup_colors( best_frame.nres, *best_frame.res );

		frame_display( best_frame, title );
		pthread_mutex_unlock(&best_frame_mut);
	} else {
		using namespace misc::best_pose;
		FArray_Calpha_display(  misc::total_residue, Ebest_position, best_centroid,
														best_res, best_res_variant, best_secstruct, pdb::occupancy, false);
		ShowScores( mc_global_track::mc_score::best_score );

	}
  glutSwapBuffers(); // Draw it!
}

///////////////////////////////////////////////////////////////////////////////
void
low_display( void ) {
	graphics::current_gs.Trajectory_state = protein_graphics::SHOW_LOW;

	if ( graphics::pose_graphics ) {
		if ( graphics::low_accepts.size() ) {
		using namespace graphics;
		// new way
		pthread_mutex_lock  (&low_frame_mut);

		setup_colors( best_frame.nres, *best_frame.res );

		std::string const tag( title +"  "+string_of( current_low_accept+1 ) +
													 " / " +string_of( low_accepts.size() ) );
		frame_display( low_accepts[ current_low_accept ], tag );

		pthread_mutex_unlock(&low_frame_mut);
		}
	} else {
		using namespace low_pose;
		FArray_Calpha_display( misc::total_residue, Elow_position, low_centroid,
													 low_res, low_res_variant, low_secstruct, pdb::occupancy, false );
		ShowScores( mc_global_track::mc_score::low_score );
	}

  glutSwapBuffers(); // Draw it!
}


///////////////////////////////////////////////////////////////////////////////
inline
numeric::xyzVector_float
get_atom_color(
							 int const atomno,
							 int const aa,
							 int const aav,
							 int const resnum = 0,
							 int const nres = 0
							 )
{
	using namespace param_aa;

	if (is_RNA(aa)) {
		if (atomno < 12) {
			float red,green,blue;
			get_rsd_color( resnum, aa, nres, red, green, blue);
			return numeric::xyzVector_float(red, green, blue);
		} else {
			if (aa == na_rad){ //yellow
				return numeric::xyzVector_float( 1.0, 1.0, 0.0);
			} else if (aa == na_ura){//red
				return numeric::xyzVector_float( 1.0, 0.0, 0.0);
			} else if (aa == na_rgu){//blue
				return numeric::xyzVector_float( 0.0, 0.0, 1.0);
			} else { //green
				assert( aa == na_rcy);
				return numeric::xyzVector_float( 0.0, 1.0, 0.0);
			}
		}
	}
	int const type( aaproperties_pack::fullatom_type(atomno,aa,aav));

  if( is_ligand(aa) ) {
		if( type < 7 ) {//orange
			return numeric::xyzVector_float( 1.0, 0.5, 0.0);
		}
	}
	if ( type >= 7 && type <= 12 || type == 17 ) { // blue
		return numeric::xyzVector_float( 0.0, 0.0, 1.0);
	} else if ( type >= 13 && type <= 15 || type == 20 ) { // red
		return numeric::xyzVector_float( 1.0, 0.0, 0.0);
	} else if ( type == 16 ) { // yellow
		return numeric::xyzVector_float( 1.0, 1.0, 0.0);
	} else if ( type == 21 ) { // orange
		return numeric::xyzVector_float( 1.0, 0.5, 0.0);
	} else {
		return numeric::xyzVector_float( 0.5, 0.5, 0.5 );
	}
}


///////////////////////////////////////////////////////////////////////////////
inline
numeric::xyzVector_float
get_residue_color(
									int const seqpos
)
{
	using namespace residue_color;
	if ( seqpos < 1 || seqpos > int(red.size()) ) {
		std::cout << "get_residue_color: seqpos " << seqpos << " out of range " <<
			red.size() << std::endl;
		std::exit( EXIT_FAILURE );
	}
	return numeric::xyzVector_float
		( red[ seqpos-1 ], green[ seqpos-1 ], blue[ seqpos-1 ] );
}


///////////////////////////////////////////////////////////////////////////////
void
setup_gray_colors(
									int const nres
									)
{
	using namespace residue_color;
	clear_residue_color();
	for ( int i=1; i<= nres; ++i ) {
		red.push_back( 0.5 );
		green.push_back( 0.5 );
		blue.push_back( 0.5 );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
setup_energy_colors(
										int const nres,
										FArray1D_float const & energy,
										std::string const etag
										)
{
	using namespace residue_color;

	clear_residue_color();

	float const mean( graphics_f_param( etag+"_mean", 0.0 ) );
	float const sd  ( graphics_f_param( etag+"_sd"  , 10.0 ) );

	for ( int i=1; i<= nres; ++i ) {
		float c( std::min( 1.0f, std::max( -1.0f, ( energy(i) - mean )/sd ) ));
		float b,r,g;



		if ( c < 0 ) {
			c *= -1;
			b = 0.5 * ( 1+c );

			r = 0.5 * ( 1-c );
			g = 0.5 * ( 1-c );
		} else {
			r = 0.5 * ( 1+c );

			b = 0.5 * ( 1-c );
			g = 0.5 * ( 1-c );
		}
		red.push_back( r );
		green.push_back( g );
		blue.push_back( b );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
setup_frame_colors(
									 graphics::Frame const & f
									 )
{
	std::string const & color_mode( graphics_s_param( "color", "gray" ) );

	int const nres( f.nres );
	if ( color_mode == "group" ) {
		init_rainbow_color( nres );

	} else if ( color_mode == "shapely" ) {
		init_sequencedependent_color( nres, *f.res );

	} else if ( color_mode == "gray" ) {
		setup_gray_colors( nres );

	} else if ( color_mode == "dun" ) {
		setup_energy_colors( nres, f.scores.dunenergy, color_mode );

	} else if ( color_mode == "res" ) {
		setup_energy_colors( nres, f.scores.resenergy, color_mode );

	} else {
		//std::cout << "undefined color mode: " << color_mode << std::endl;
		graphics::s_param[ "color" ] = "group";
		setup_gray_colors( nres );
	}
}


///////////////////////////////////////////////////////////////////////////////
void
frame_display(
							graphics::Frame const & f,
							std::string const & name
)
{
	using namespace protein_graphics;
	setup_frame_colors( f );
	FArray_Calpha_display(
		  f.nres, *f.xyz, misc::centroid /*unused?*/, *f.res, *f.resv,
			*f.secstruct, *f.occ, f.fullatom, f.scores.bonds, f.anchor_pos );

	show_scores( f.scores, name );

}

///////////////////////////////////////////////////////////////////////////////
void

FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	FArray1DB_char const & secstruct,
	FArray2DB_float const & occ,
	bool const fullatom
)
{
	using namespace protein_graphics;

	std::vector< graphics::Bond > empty;
	FArray_Calpha_display( nres, xyz, centroidxyz, res, res_variant,
												 secstruct, occ, fullatom, empty );
}


///////////////////////////////////////////////////////////////////////////////
void
FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	FArray1DB_char const & secstruct,
	FArray2DB_float const & occ,
	bool const fullatom,
	std::vector< graphics::Bond > const & bonds,
	int const anchor_pos // = -1
)
{
	//lin param
	using namespace graphics;
	using namespace protein_graphics;
	int const box_size ( 900 );
	float const window_ratio ( 0.2 );
	int const margin ( 5 );

	float energy_min, energy_max, rmsd_min, rmsd_max;
	std::vector< float > energy_vector, rmsd_vector;
	xyzVector_float line_color(0.5,0.5,0.5);

	//lin generate box
	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	//Set background color
	if (backgroundblack) glClearColor( 0.0, 0.0, 0.0, 1.0); //black
	else  glClearColor( 1.0, 1.0, 1.0, 1.0); // white

	// draw the boundary of plot box
	glPushMatrix();//save it for later
	glViewport( 0, 0, box_size, box_size );
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glViewport( margin, margin , int(box_size*(1-window_ratio)) - 2*margin, int(box_size * window_ratio)- 2*margin );
	draw_box_boundary(line_color);

	glViewport( margin, margin + int(box_size * window_ratio), int(box_size*(1-window_ratio)) - 2*margin, int(box_size*(1-window_ratio)) - 2*margin );
	draw_box_boundary(line_color);

	glViewport( margin + int(box_size*(1-window_ratio)), margin + int(box_size * window_ratio), int(box_size*window_ratio) - 2*margin, int(box_size*(1-window_ratio)) - 2*margin );
	draw_box_boundary(line_color);

	glViewport( margin + int(box_size*(1-window_ratio)), margin, int(box_size*window_ratio) - 2*margin, int(box_size*window_ratio) - 2*margin );
	draw_box_boundary(line_color);

	// draw in the score plot box
	get_score_rms_from_trajectory( graphics::trajectory, energy_vector, rmsd_vector, current_gs );
	glViewport( margin, margin , int(box_size*(1-window_ratio)) - 2*margin, int(box_size * window_ratio) - 2*margin );
	draw_score_plot( energy_vector, false, energy_min, energy_max );

	//Draw the rmsd plot
	glViewport( margin + int(box_size*(1-window_ratio)), margin + int(box_size * window_ratio), int(box_size*window_ratio) - 2*margin, int(box_size*(1-window_ratio)) - 2*margin );
	draw_score_plot( rmsd_vector, true, rmsd_min, rmsd_max );

	//draw score vs rmsd
	glViewport( margin + int(box_size*(1-window_ratio)), margin, int(box_size*window_ratio) - 2*margin, int(box_size*window_ratio) - 2*margin );
	plot_2D( rmsd_vector, energy_vector,
					 rmsd_min, rmsd_max, energy_min, energy_max ) ;
	glLoadIdentity();

	glPopMatrix();

	//lin draw the structure
	glViewport( margin, margin + int(box_size * window_ratio), int(box_size*(1-window_ratio)) - 2*margin, int(box_size*(1-window_ratio)) - 2*margin );
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glOrtho(-graphics::window_size, graphics::window_size, -graphics::window_size, graphics::window_size, -10*graphics::window_size, 10*graphics::window_size);
	glMatrixMode(GL_MODELVIEW);

	glPushMatrix();//save it for later
	if( graphics::new_drawing ) {
		draw_pose( nres, xyz, misc::Eposition/*unused?*/, res, res_variant, secstruct, occ, graphics::current_gs, fullatom , true );
	} else {
		FArray_Calpha_display( nres, xyz, centroidxyz, res, res_variant, fullatom,
													 bonds, anchor_pos );
	}
	glPopMatrix();
}

///////////////////////////////////////////////////////////////////////////////
void
FArray_Calpha_display(
	int const nres,
	FArray3DB_float const & xyz,
	FArray2DB_float const & centroidxyz,
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	bool const fullatom,
	std::vector< graphics::Bond > const & bonds,
	int const anchor_pos
)
{
  using namespace numeric;
	using namespace graphics;
	using namespace protein_graphics;
	using param_aa::is_protein;

	float const bond_width( 0.2 );
	float const backbone_width( 0.2 ); // old default is 0.5
	float const sidechain_width( 0.1 );

  int center( std::max(1,int(nres/2.0)-3) );
	if ( anchor_pos >= 1 && anchor_pos <= nres ) center = anchor_pos;

	GLfloat currentrotation[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
	xyzVector_float const z
		(currentrotation[2],currentrotation[6],currentrotation[10]);

  if ( nres > 0 ) {
		bool draw_backbone = true;
		if (draw_backbone) {
 			float xwidth = backbone_width;
			//    glLoadIdentity();
			xyzVector_float center_pos( &xyz(1,2,center) );
			xyzVector_float prev1, prev2;
			bool last_bonded = false;
			for ( int i = 1; i<nres; i++ ) {
				if ( !is_protein(res(i)) ||
						 !is_protein(res(i+1) )) {
					last_bonded = false;
					continue;
				}

				xyzVector_float const i_color( get_residue_color(i) );
				xyzVector_float const ip1_color( get_residue_color(std::min(i+1,nres)) );

				xyzVector_float ca_pos1( &xyz(1,2,i) );
				xyzVector_float ca_pos2( &xyz(1,2,i+1) );
				ca_pos1 = ca_pos1 - center_pos;
				ca_pos2 = ca_pos2 - center_pos;
				xyzVector_float bond;
				bond = ca_pos2 - ca_pos1;


				xyzVector_float width( cross( bond, z ));
				if ( width.length_squared() > 0 )  width.normalize();

				width = width * xwidth;
				//float badness = .5;
				//if ( get_native_exists() ) {
				//	badness = torsion_diff( native::native_phi(i), misc::best_phi(i) );
				//	badness /= 180.0;
				//}
				//badness = sqrt(badness);
				//      glColor3f(1, 1-badness, 1-badness);
				glColor3fxyz( i_color );

				if ( i > 1 && last_bonded && !peptide_planes) { // Draws the "elbow"
					glBegin(GL_POLYGON);
					glVertex3fxyz ( prev1 );
					glVertex3fxyz ( ca_pos1 + width );
					glVertex3fxyz ( ca_pos1 - width );
					glVertex3fxyz ( prev2 );
					glEnd();
				}
				last_bonded = false;
				if (bond.length_squared() < 16 ) {
					last_bonded = true;
					if ( !peptide_planes) {
						//	float atom_size = 20;
						glBegin(GL_POLYGON);
						glVertex3fxyz ( ca_pos1 + width );
						glVertex3fxyz ( ca_pos1 - width );
						glColor3fxyz( ip1_color );
						glVertex3fxyz ( ca_pos2 - width );
						glVertex3fxyz ( ca_pos2 + width );
						glEnd();
					} else if ( peptide_planes ) {
						// experiment with peptide plane drawing
						//
						// draw i->i+1 peptide plane
						// CA i
						// OC i
						// CA i+1
						// HN i+1
						xyzVector_float o_pos( &xyz(1, (fullatom ?4:5) ,i  ) );
						o_pos -= center_pos;
						xyzVector_float hn_pos( ca_pos2 + ( ca_pos1 - o_pos ) );

						glColor3f(0.5,0.5,0.5);
						glBegin( GL_POLYGON );
						glVertex3fxyz( ca_pos1 );
						glVertex3fxyz( ca_pos2 );
						glColor3f(1.0,0.0,0.0);
						glVertex3fxyz( o_pos );
						glEnd();

						glColor3f(0.5,0.5,0.5);
						glBegin( GL_POLYGON );
						glVertex3fxyz( ca_pos1 );
						glVertex3fxyz( ca_pos2 );
						glColor3f(0.0,0.0,1.0);
						glVertex3fxyz( hn_pos );
						glEnd();
					}
				}
				prev1 = ca_pos2 + width;
				prev2 = ca_pos2 - width;

				if (draw_centroids) {
					xyzVector_float cen_pos1( &centroidxyz(1,i) );
					cen_pos1 = cen_pos1 - center_pos;
					bond = cen_pos1 - ca_pos1;
					width = cross( bond, z );
					if ( width.length_squared() > 0 )  width.normalize();
					float xwidth2 = xwidth*0.3;
					width = width * xwidth2;

					glBegin(GL_POLYGON);
					glColor3fxyz( i_color );

					glVertex3fxyz ( ca_pos1 + width );
					glVertex3fxyz ( cen_pos1 + width );
					glVertex3fxyz ( cen_pos1 - width );
					glVertex3fxyz ( ca_pos1 - width );
					glEnd();

				}
			}


			//    glLineWidth( 2 );
			for ( int i = 1; i<nres; i++ ) {
				xyzVector_float ca_pos1( &xyz(1,2,i) );
				int j = i<nres ? i+1 : i;
				xyzVector_float ca_pos2( &xyz(1,2,j) );
				ca_pos1 = ca_pos1 - center_pos;
				ca_pos2 = ca_pos2 - center_pos;
				xyzVector_float bond;
				bond = ca_pos2 - ca_pos1;
				//      xyzVector_float z( 0, 0, 1);
				xyzVector_float width( cross( bond, z ));
				if ( width.length_squared() > 0 )  width.normalize();
				//float halo = 0;
				//      float halo_width = xwidth + halo;
				float z_offset = .1;

				width = width * (xwidth/*+halo*/);
				xyzVector_float zsmalloffset = z * z_offset;
				glColor3f(0,0,0);

				if ( i > 1 && last_bonded && !peptide_planes) {
					glBegin(GL_LINES);
					glVertex3fxyz ( prev1 );
					glVertex3fxyz ( ca_pos1 + width );
					glVertex3fxyz ( ca_pos1 - width );
					glVertex3fxyz ( prev2 );
					glEnd();
				}
				last_bonded = false;
				if (bond.length_squared() < 16 && is_protein(res(i)) &&
						is_protein(res(i+1))) {
					last_bonded = true;
					if ( !peptide_planes ) {

						glBegin(GL_LINES);
						glVertex3fxyz ( zsmalloffset + ca_pos1 + width );
						glVertex3fxyz ( zsmalloffset + ca_pos2 + width );
						glVertex3fxyz ( zsmalloffset + ca_pos2 - width );
						glVertex3fxyz ( zsmalloffset + ca_pos1 - width );
						glEnd();
					} else  if ( false ) {
						xyzVector_float o_pos( &xyz(1, (fullatom ?4:5) ,i  ) );
						o_pos -= center_pos;
						xyzVector_float hn_pos( ca_pos2 + ( ca_pos1 - o_pos ) );
						glBegin(GL_LINES);
						glVertex3fxyz( ca_pos1 );
						glVertex3fxyz( o_pos );
						glVertex3fxyz( ca_pos2 );
						glVertex3fxyz( hn_pos );
						glEnd();
					}
				}

				prev1 = ca_pos2 + width;
				prev2 = ca_pos2 - width;

				if (draw_centroids && is_protein(res(i)) ){
					xyzVector_float cen_pos1( &centroidxyz(1,i) );
					cen_pos1 = cen_pos1 - center_pos;
					bond = cen_pos1 - ca_pos1;
					width = cross( bond, z );
					if ( width.length_squared() > 0 )  width.normalize();
					float xwidth2 = xwidth*0.3;
					width = width * xwidth2;

					glBegin(GL_LINES);
					glVertex3fxyz ( zsmalloffset + ca_pos1 + width );
					glVertex3fxyz ( zsmalloffset + cen_pos1 + width );
					glVertex3fxyz ( zsmalloffset + cen_pos1 - width );
					glVertex3fxyz ( zsmalloffset + ca_pos1 - width );
					glEnd();
				}

				if ( fullatom && (!is_protein(res(i))||get_show_sc(i)) ) {
					///////////////////////
					// draw sidechain atoms
					using namespace aaproperties_pack;
					using param_aa::is_NA;

					int const aa ( res( i ) );
					int const aav( res_variant( i ) );

					float const z_offset = .1;
					xyzVector_float zsmalloffset = z * z_offset;

					// loop through each heavy atom
					float xwidth = sidechain_width;
					float ywidth = 0.2; //for ligand
// 					float xwidth = .2;

					std::vector< int > atom1_list, atom2_list;
					int const atom2_begin( is_protein(res(i)) ? 5 : 1 );

					for ( int atom2 = atom2_begin; atom2 <= nheavyatoms(aa, aav);
								++atom2 ) {
						int const atom1 = atom_base(atom2,aa,aav);
						atom1_list.push_back( atom1 );
						atom2_list.push_back( atom2 );
					}
					if ( aa == param_aa::aa_phe ) { /*PHE CZ->CE2 */
						atom1_list.push_back( 10 );
						atom2_list.push_back( 11 );
					} else if ( aa == param_aa::aa_his ) { /*HIS NE2->CE1*/
						atom1_list.push_back( 9 );
						atom2_list.push_back( 10);
					} else if ( aa == param_aa::aa_pro ) { /*PRO CD->N */
						atom1_list.push_back( 1 );
						atom2_list.push_back( 7 );
					} else if ( aa == param_aa::aa_trp ) { /*TRP NE1->CE2*/
						atom1_list.push_back( 9 );
						atom2_list.push_back( 10 );
						/*TRP CH2->CZ3*/
						atom1_list.push_back( 13 );
						atom2_list.push_back( 14 );
					} else if ( aa == param_aa::aa_tyr ) {
						atom1_list.push_back( 10 );
						atom2_list.push_back( 11 );
					} else if ( aa == param_aa::na_ade || aa == param_aa::na_rad ) { //// NOW DO DNA
						static int const n1( LookupByName( aa, 1, " N1 " ) );
						static int const c2( LookupByName( aa, 1, " C2 " ) );
						atom1_list.push_back( n1 );
						atom2_list.push_back( c2 );
						static int const c4( LookupByName( aa, 1, " C4 " ) );
						static int const c5( LookupByName( aa, 1, " C5 " ) );
						atom1_list.push_back( c4 );
						atom2_list.push_back( c5 );
					} else if ( aa == param_aa::na_thy || aa == param_aa::na_ura ) {
						static int const n3( LookupByName( aa, 1, " N3 " ) );
						static int const c4( LookupByName( aa, 1, " C4 " ) );
						atom1_list.push_back( n3 );
						atom2_list.push_back( c4 );
					} else if ( aa == param_aa::na_gua || aa == param_aa::na_rgu ) {
						static int const n1( LookupByName( aa, 1, " N1 " ) );
						static int const c6( LookupByName( aa, 1, " C6 " ) );
						atom1_list.push_back( n1 );
						atom2_list.push_back( c6 );
						static int const c4( LookupByName( aa, 1, " C4 " ) );
						static int const c5( LookupByName( aa, 1, " C5 " ) );
						atom1_list.push_back( c4 );
						atom2_list.push_back( c5 );
					} else if ( aa == param_aa::na_cyt || aa == param_aa::na_rcy ) {
						static int const n3( LookupByName( aa, 1, " N3 " ) );
						static int const c4( LookupByName( aa, 1, " C4 " ) );
						atom1_list.push_back( n3 );
						atom2_list.push_back( c4 );
					}

					bool link_ahead = false;
					if ( is_NA( res(i) ) && is_NA( res(i+1) ) ) {
						// add the connection to the next residue residue
						// these two atom numbers should be aa-independent
						static int const o3( LookupByName( aa, 1, " O3*" ) );
						static int const p ( LookupByName( aa, 1, " P  " ) );
						atom1_list.push_back( o3 );
						atom2_list.push_back( p  );
						link_ahead = true;
					}

					assert( atom1_list.size() == atom2_list.size() );

					//
					std::map< int, xyzVector_float> prev1, prev2;
					for ( unsigned k = 0; k < atom1_list.size(); ++k ) {
						int const atom1 = atom1_list[k];
						int const atom2 = atom2_list[k];

						xyzVector_float atom_pos1( &xyz(1,atom1,i));
						xyzVector_float atom_pos2( &xyz(1,atom2,i));

						if ( link_ahead && k == atom1_list.size()-1) {
							atom_pos2 = &(xyz(1,atom2,i+1));
						}

						// colors
						xyzVector_float
							atom1_color( get_atom_color( atom1,aa,aav,i,nres)),
							atom2_color( get_atom_color( atom2,aa,aav,i,nres));

						atom_pos1 -= center_pos;
						atom_pos2 -= center_pos;

						xyzVector_float bond;
						bond = atom_pos2 - atom_pos1;
						if ( bond.length_squared() > 16.0 ) continue;
						xyzVector_float width( cross( bond, z ));
						if ( width.length_squared() > 0 )  width.normalize();
						if( param_aa::is_ligand(aa) ) {
							width = width * ywidth;
						} else {
							width = width * xwidth;
						}
						glColor3fxyz( atom1_color );
						//if ( atom2 > 5 && ! (aa == param_aa::aa_pro && atom1 == 1)) {
						if ( prev1.count( atom1 ) ) {
							glBegin(GL_POLYGON);
							glVertex3fxyz ( prev1[atom1] );
							glVertex3fxyz ( atom_pos1 + width );
							glVertex3fxyz ( atom_pos1 - width );
							glVertex3fxyz ( prev2[atom1] );
							glEnd();
						}
						glBegin(GL_POLYGON);
						glVertex3fxyz ( atom_pos1 + width );
						glVertex3fxyz ( atom_pos1 - width );
						glColor3fxyz( atom2_color );
						glVertex3fxyz ( atom_pos2 - width );
						glVertex3fxyz ( atom_pos2 + width );
						glEnd();

						if ( false ) {
							// is it OK to do this right here, or later??
							glColor3f(0,0,0);
							glBegin(GL_LINES);
							glVertex3fxyz ( zsmalloffset + atom_pos1 + width );
							glVertex3fxyz ( zsmalloffset + atom_pos2 + width );
							glVertex3fxyz ( zsmalloffset + atom_pos2 - width );
							glVertex3fxyz ( zsmalloffset + atom_pos1 - width );
							glEnd();
						}

						prev1[atom2] = ( atom_pos2 + width );
						prev2[atom2] = ( atom_pos2 - width );
					} // atom1_list == list of bonds


					{ // show waters ////////////////////////////////////////////////////
						for ( int j=1; j<= nh2o( aa, aav ); ++j ) {
							int const atomno( h2opos( j, aa, aav ) );
							xyzVector_float water_pos( &xyz(1,atomno,i));
							water_pos -= center_pos;
							float const radius( 0.4 );
							xyzVector_float
								xoff( radius, 0, 0 ),
								yoff( 0, radius, 0 ),
								zoff( 0, 0, radius );

							glBegin(GL_LINES);
							glColor3f(1.0,0.0,0.0);
							glVertex3fxyz( water_pos - xoff );
							glVertex3fxyz( water_pos + xoff );
							glVertex3fxyz( water_pos - yoff );
							glVertex3fxyz( water_pos + yoff );
							glVertex3fxyz( water_pos - zoff );
							glVertex3fxyz( water_pos + zoff );
							glEnd();
						}
					} // scope -- waters

					glColor3f(0.0,0.0,0.0);
				} // if ( fullatom )

			} // i=1,nres-1 (!)

			// show hbonds
			if ( bonds.size() ) {
				for ( unsigned i=0; i<bonds.size(); ++i ) {
					// draw a line between two atoms
					graphics::Bond const & b( bonds[i] );

					// exclude helical hbonds:
					int const aa1( res(b.seqpos1));
					int const aa2( res(b.seqpos2));
					if ( b.type == HBOND &&
							 b.seqpos1 == b.seqpos2 + 4 &&
							 is_protein(aa1) && is_protein(aa2) &&
							 b.atm1 == 1 /* NHbb */ &&
							 b.atm2 == 4 /* OCbb */ ) continue;
					if( is_protein(aa1) && is_protein(aa2) &&
							!get_show_sc(b.seqpos1) && !get_show_sc(b.seqpos2) ) continue;

					xyzVector_float
						atm1( &xyz(1, b.atm1, b.seqpos1 )) ,
						atm2( &xyz(1, b.atm2, b.seqpos2 ));
					atm1 -= center_pos;
					atm2 -= center_pos;


					xyzVector_float const bond( atm2 - atm1 );
					xyzVector_float width( cross( bond, z ));
					if ( width.length_squared() > 0 )  width.normalize();

					float xwidth;
					if ( true ) {
						// width ~ energy
						float const transparency( 0.5 );
						xwidth = bond_width * std::abs( b.bondE ) / 2.0;
						if ( b.type == HBOND ) {
							glColor4f( 0, 1, 0, transparency ); // green
						} else if ( b.type == WATER_HBOND ) {
							glColor4f( 0, 1.0, 0.5, transparency ); // aqua?
						} else {
							if ( b.bondE < 0.0 ) glColor4f( 1, 1, 0, transparency); // yellow
							else glColor4f( 1, 0, 1, transparency ); // purple
						}
					} else {
						// color ~ energy
						xwidth = 0.1;
						// etcetc
					}
					width = width * xwidth;

					glBegin(GL_POLYGON);
					glVertex3fxyz ( atm1 + 0.25f * bond + width );
					glVertex3fxyz ( atm1 + 0.75f * bond + width );
					glVertex3fxyz ( atm1 + 0.75f * bond - width );
					glVertex3fxyz ( atm1 + 0.25f * bond - width );
					glEnd();

				}
			} // show hbonds
		} // draw_backbone

		// show logs
		//log_display();
  } // nres > 0
}

///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////


void
log_display() {
	using namespace::graphics;

	//Before writing text, transform back to original frame.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix(); //Save for later
	glLoadIdentity();

	if (backgroundblack) glColor3f( 1.0, 1.0, 1.0); //black
	else  glColor3f( 0.0, 0.0, 0.0); // white

	pthread_mutex_lock(&log_mut);
	for ( unsigned i=0; i< logs.size(); ++i ) {

		glRasterPos2f( -graphics::window_size + 1,
									 graphics::window_size - 5*i -1 );
 		writeString( logs[i] );
	}
	pthread_mutex_unlock(&log_mut);

	//rhiju
	glPopMatrix();
}


///////////////////////////////////////////////////////////////////////////////
void
text_window_display(
)
{
	using namespace::graphics;

	float const line_sep( graphics_f_param( "line_sep", 4.0 ) );

  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	if (backgroundblack) glClearColor( 0.0, 0.0, 0.0, 1.0); //black
	else  glClearColor( 1.0, 1.0, 1.0, 1.0); // white

	//Before writing text, transform back to original frame.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix(); //Save for later
	glLoadIdentity();

	if ( backgroundblack ) glColor3f( 1.0, 1.0, 1.0);
	else glColor3f( 0.0, 0.0, 0.0);


	// show the current command string
	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 1 );

	writeString( ">" + command_string );


	// now show the log messages
	pthread_mutex_lock(&log_mut);
	int const nlogs( logs.size() );
	for ( int i=0; i< nlogs; ++i ) {

		glRasterPos2f( -graphics::window_size + 1,
									 -graphics::window_size + 1 + (i+2) * line_sep );
		writeString( logs[ nlogs-1-i ] );
	}
	pthread_mutex_unlock(&log_mut);



	//rhiju
	glPopMatrix();

	//g_display();

	glutSwapBuffers(); // Draw it!
}


///////////////////////////////////////////////////////////////////////////////
void
process_command(
	std::string const & cmd
)
{

	std::string tag;
	std::istringstream l( cmd.c_str() );
	l >> tag;
	if ( tag == "set" ) {
		// set a parameter
		std::string setting;
		l >> tag >> setting;

		if ( !l.fail() ) {
			if ( is_float( setting ) ) {
				graphics::f_param[ tag ] = float_of( setting );
			} else {
				graphics::s_param[ tag ] = setting;
			}
			redrawallwindows();
		} else {
			using graphics::param_tags;
			std::string s( "params for set command:");
			for ( std::vector< std::string >::const_iterator it=
							param_tags.begin(); it != param_tags.end(); ++it ) {
				s += " " + *it;
			}
			gl_graphics_log( s );
			l.clear();
		}

	} else {
		// unable to parse
		l.setstate( std::ios_base::failbit );
	}

	if ( l.fail() ){
		gl_graphics_log( "parse error!" );
	} else {
		graphics::commands.push_back( cmd );
		graphics::current_command = graphics::commands.size();
	}
}

///////////////////////////////////////////////////////////////////////////////
void
textWindowKeyPressFunc(
	unsigned char key,
	int,
	int
)
{
	using namespace graphics;

	switch( key ) {
	case 8:
		// backspace
		int const s( command_string.size() );
		if ( s ) {
			command_string = command_string.substr(0,s-1);
		}
		break;
	case 13:
		// enter
		process_command( command_string );
		command_string = "";
		break;
	default:
		command_string += key;
	}

	redraw_text_window();
}


///////////////////////////////////////////////////////////////////////////////
void
textWindowSpecialKeyPressFunc( int key, int, int ) {
	using namespace graphics;

	bool update_command( false );
	switch ( key ) {

	case GLUT_KEY_UP:
		current_command-=1;
		update_command = true;
		break;

	case GLUT_KEY_DOWN:
		current_command+=1;
		update_command = true;
		break;
	}

	if ( update_command && commands.size() ) {

		current_command = std::min( int(commands.size()),
																std::max( 0, current_command ) );
		if ( current_command == int(commands.size() ) ) {
			command_string = "";
		} else {
			command_string = commands[ current_command ];
		}

		redraw_text_window();
	}
}

///////////////////////////////////////////////////////////////////////////////
void
show_scores(
						graphics::FAS const & scores,
						std::string const & name
)
{
	using namespace graphics;

	//Before writing text, transform back to original frame.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix(); //Save for later
	glLoadIdentity();

	if ( backgroundblack ) glColor3f( 1.0, 1.0, 1.0);
	else glColor3f( 0.0, 0.0, 0.0);

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 1 );

	writeString( name );

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 3 );

	std::string energy_text = "E: " +
		fixed_string_of( scores.score, 8, 2 ) +
		fixed_string_of( scores.fa_atr, 8, 2 ) +
		fixed_string_of( scores.fa_rep, 8, 2 ) +
		fixed_string_of( scores.fa_sol, 8, 2 ) +
		fixed_string_of( scores.fa_dun, 8, 2 ) +
		fixed_string_of( scores.fa_h2o + scores.fa_h2o_hb, 8, 2 ) +
		fixed_string_of( scores.fa_gb_elec, 8, 2 ) +
		fixed_string_of( scores.hb_sc, 8, 2 );

	writeString( energy_text );
	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 5 );

	std::string energy_labels
		( std::string("E: ") +
			"   score"+
			"  fa_atr"+
			"  fa_rep"+
			"  fa_sol"+
			"  fa_dun"+
			"  fa_h2o"+
			" gb_elec"+
			"   hb_sc" );

	writeString( energy_labels );

	//rhiju
	glPopMatrix();
}


///////////////////////////////////////////////////////////////////////////////
void
ShowScores( float local_score){
	using namespace::graphics;

	//Before writing text, transform back to original frame.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix(); //Save for later
	glLoadIdentity();

	if (backgroundblack) glColor3f( 1.0, 1.0, 1.0); //black
	else  glColor3f( 0.0, 0.0, 0.0); // white

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 1 );
	writeString( title );
// 	writeString( counters::move_type );

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 3 );

	std::string energy_text = "Energy:    " + string_of(local_score, 8);
	writeString( energy_text );

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 5 );
	std::string text2 = "Accepted RMSD:   " + string_of(mc_global_track::diagnose::best_rms, 4);
	writeString( text2 );

	glRasterPos2f( -graphics::window_size + 1,
								 -graphics::window_size + 7 );
	std::string text3 = "Low Energy RMSD: "  + string_of(mc_global_track::diagnose::low_rms, 4);
	writeString( text3 );

	//rhiju
	glPopMatrix();
}


///////////////////////////////////////////////////////////////////////////////
void
writeString( std::string const & text_string) {
  char const *temp_cstring = text_string.c_str();
  int len = (int) strlen(temp_cstring);
  for (int i = 0; i < len; i++) {
    //    glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, temp_cstring[i]);
    //    glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, temp_cstring[i]);
    glutBitmapCharacter(GLUT_BITMAP_9_BY_15, temp_cstring[i]);
  }
}


///////////////////////////////////////////////////////////////////////////////
void
glVertex3fxyz( numeric::xyzVector_float  coord ) {
  glVertex3f(coord.x(), coord.y(), coord.z() );
}


///////////////////////////////////////////////////////////////////////////////
void
drawsequence()
{
  float kern = 1.0;
  float ypos = 1 - graphics::window_size;
  float xpos = 1 - graphics::window_size;
  for ( int i = 1; i<misc::total_residue; i++ ) {

    float badness = 0;
    if ( get_native_exists() ) {
      badness = torsion_diff( native::native_phi(i), misc::best_phi(i) );
      badness /= 180.0;
    }
    badness = sqrt(badness);
    glColor3f(1, 1-badness, 1-badness);

    xpos += kern;
    //    glWindowPos2f( xpos, ypos );
    glRasterPos2f( xpos, ypos );
    //    char const *string = misc::residue1(i);
    //  int len = (int) strlen(string);
    //  for (int i = 0; i < len; i++) {
    //    glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, string[i]);
    //    glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, misc::residue1(i));
    glutBitmapCharacter(GLUT_BITMAP_8_BY_13, misc::residue1(i));
  }
}


void
get_rsd_color(
							int const i,
							int const, // aa,
							int const nres,
							float & red,
							float & green,
							float & blue
							)
{
	bool rainbow = true;
	if (rainbow) {
		int muting = 1;
		float my_color = (float) i/(float) nres;
		if (my_color > .5) {
			red = (2*my_color)-1;
			blue = 0;
		}
		if (my_color < .5) {
			blue = (1-2*my_color);
			red = 0;
		}
		if ((my_color > .25) && (my_color < .75)) {
			green = (2*my_color)-.5;
		}
		else {
			green = 0;
		}
		red = my_color;
		blue = 1 - my_color ;
		if (my_color < .5) {
			green = 2*my_color;
		}
		else {
			green = 2-2*my_color;
		}
		float saturation = sqrt(red*red + green*green + blue*blue);
		red = muting*red/saturation;
		green = muting*green/saturation;
		blue = muting*blue/saturation;
	}
}

void
init_rainbow_color(
	int const nres
)
{
  //using namespace misc;
  using namespace residue_color;
  //int nres = total_residue;

	clear_residue_color();

  for ( int i = 1; i<=nres; i++ ) {
    float red,blue,green;
    bool rainbow = true;
    if (rainbow) {
      int muting = 1;
      float my_color = (float) i/(float) nres;
      if (my_color > .5) {
				red = (2*my_color)-1;
				blue = 0;
      }
      if (my_color < .5) {
				blue = (1-2*my_color);
				red = 0;
      }
      if ((my_color > .25) && (my_color < .75)) {
				green = (2*my_color)-.5;
      }
      else {
				green = 0;
      }
      red = my_color;
      blue = 1 - my_color ;
      if (my_color < .5) {
				green = 2*my_color;
      }
      else {
				green = 2-2*my_color;
      }
      float saturation = sqrt(red*red + green*green + blue*blue);
      red = muting*red/saturation;
      green = muting*green/saturation;
      blue = muting*blue/saturation;
      residue_color::red.push_back(red);
      residue_color::green.push_back(green);
      residue_color::blue.push_back(blue);
    }
  }
}

void
init_sequencedependent_color(
	int const nres,
	FArray1DB_int const & res
)
{
  //using namespace misc;
  using namespace residue_color;
	using namespace rgbcolors;
  //int nres = total_residue;

	std::vector <rgbcolor> sequencecolors;
	sequencecolors.push_back( colorgray);    // A
	sequencecolors.push_back( coloryellow);  // C
	sequencecolors.push_back( colorred);     // D
	sequencecolors.push_back( colorred);     // E
	sequencecolors.push_back( colorbrown);   // F
	sequencecolors.push_back( colororange);  // G
	sequencecolors.push_back( colordarkblue);// H
	sequencecolors.push_back( colorgray);    // I
	sequencecolors.push_back( colorblue);    // K
	sequencecolors.push_back( colorgray);    // L
	sequencecolors.push_back( colorgray);    // M
	sequencecolors.push_back( colorgreen);   // N
	sequencecolors.push_back( colormagenta); // P
	sequencecolors.push_back( colorgreen);   // Q
	sequencecolors.push_back( colorblue);    // R
	sequencecolors.push_back( colorgreen);   // S
	sequencecolors.push_back( colorgreen);   // T
	sequencecolors.push_back( colorgray);    // V
	sequencecolors.push_back( colorbrown);   // W
	sequencecolors.push_back( colorbrown);   // Y
	sequencecolors.push_back( colororange);   // LIG1
	sequencecolors.push_back( colororange);   // LIG2

	clear_residue_color();

	rgbcolor thecolor;
  for ( int i = 1; i<=nres; i++ ) {
		int res_i = res(i);
		if ((res_i >= 1) && (res_i <= 22)) {
			thecolor =	sequencecolors[res_i - 1];
			append_residue_color( thecolor);
			//			std::cout << res_i << " " << thecolor[0] << " " <<
			//	thecolor[1] << " " << thecolor[2] << std::endl;
		}
		else {
			std::cout << "Could not allocate color: " << res_i << std::endl;
			append_residue_color( colorgray);
		}
	}
}

void clear_residue_color() {
	using namespace residue_color;
	red.erase  ( red.begin()  , red.end()  );
	blue.erase ( blue.begin() , blue.end() );
	green.erase( green.begin(), green.end());
}

void append_residue_color( rgbcolor thecolor){
	using namespace residue_color;
	residue_color::red.push_back(   thecolor[0]);
	residue_color::green.push_back(  thecolor[1]);
	residue_color::blue.push_back( thecolor[2]);
}

void
open_windows(
	bool const use_pose_graphics // = false
)
{
	graphics::pose_graphics = use_pose_graphics;
  pthread_cond_broadcast( &graphics::start_cond );
}

void
set_worker_done( bool state )
{
  graphics::worker_done = state;
}

void processMouse(int button, int state, int x, int y) {
	using namespace graphics;

  specialKey = glutGetModifiers();

  click_x = x;
  click_y = y;
  if (state == GLUT_DOWN) {
    clicked_button = button;
    if (!click) {
      click = true;
			//      std::cout << "click button: " << button << std::endl;
    }
  }
  if (state == GLUT_UP) {
    clicked_button = -1;
    click = false;
  }
}

// rhiju
// Jack/Phil's rotate with mouse routine. Now with more
// intuitive rotating.
void processMouseActiveMotion(int x, int y) {
	using namespace graphics;

  static int old_x;
  static int old_y;
    if (click) {
    old_x = click_x;
    old_y = click_y;
     click = false;
    }

  float delta_x = old_x - x;
  float delta_y = old_y - y;


  if (specialKey == GLUT_ACTIVE_SHIFT & clicked_button == 0) { // Zoom in/out
    double s = exp( -1.0* (double) delta_y*0.01);
    glScalef(s,s,s);
  }
	else if (specialKey == GLUT_ACTIVE_SHIFT & clicked_button > 0){ //Rotate around z-axis
		// See below for explanation of premultiplication.
    GLfloat currentrotation[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
    glLoadIdentity();
    glRotatef(delta_x,0,0,1);
    glMultMatrixf(currentrotation);
  }
  else if (clicked_button > 0){ //Pan
     GLint viewport[4];
     glGetIntegerv(GL_VIEWPORT,viewport);
		 //glTranslatef( -1.0*delta_x * (_right-_left)/(viewport[2]),
		 //  -1.0*delta_y * (_bottom-_top)/(viewport[3]), 0);
		 //Scale factors map from screen coordinates to molecule coordinates.
		 glMatrixMode(GL_MODELVIEW);
		 double xscale = (graphics::window_size * 2.0)/(viewport[2]);
		 double yscale = (graphics::window_size * 2.0)/(viewport[3]);

		 GLfloat currentrotation[16];
		 glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
		 glLoadIdentity();
     glTranslatef( -delta_x * xscale, delta_y * yscale, 0);
		 glMultMatrixf(currentrotation);
  }
  else { //Rotate the sucker.
		//double axis_z = 0;
		double axis_x = -delta_y;
		double axis_y = -delta_x;
		double userangle = sqrt(delta_x*delta_x + delta_y*delta_y);

		glMatrixMode(GL_MODELVIEW);
		// Standard GLUT rotation is a postmultiplication - rotation around
		// molecule's inertial frame --  and leads to
		// non-intuitive behavior.
		//glRotatef(userangle,axis_x,axis_y,0.0);

		//A premultiplication -- rotates around the axis the user actually sees.
		// A little more complicated to code; unfortunately GLUT doesn't have a one-line
		// function for it.
		GLfloat currentrotation[16];
		glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
		glLoadIdentity();
		glRotatef(userangle,axis_x,axis_y,0.0);
		glMultMatrixf(currentrotation);
	}

	glutPostRedisplay();

	old_x = x;
	old_y = y;
}


// rhiju
// CENTRAL COMMANDS
void
specialKeyPressFunc( int key, int, int ) {
	using namespace graphics;

	bool redraw_low_window( false );
	switch ( key ) {

	case GLUT_KEY_LEFT:
		current_low_accept = std::max( current_low_accept-1, 0 );
		redraw_low_window = true;
		break;

	case GLUT_KEY_RIGHT:
		current_low_accept = std::min( current_low_accept+1,
																	 int(low_accepts.size())-1);
		redraw_low_window = true;
		break;

	}
	if ( redraw_low_window ) {
		glutSetWindow( low_window );
		low_display();
		glutPostRedisplay();
	}
}

void keyPressFunc(unsigned char key, int, int) {
	using namespace protein_graphics;
	using namespace graphics;

	switch (key) {
	case 'b':
	case 'B': // Toggle background color black/white
		backgroundblack = !backgroundblack;
		break;
	case 'x':
	case 'X': // Centroids on/off.
		draw_centroids = !draw_centroids;
		break;
	case 'c':
	case 'C': // Toggle color scheme.
		rainbow_color = !rainbow_color;
		if (rainbow_color)
			init_rainbow_color( pose_graphics ? current_frame.nres :
													misc::total_residue );
		else
			init_sequencedependent_color(
				pose_graphics ? current_frame.nres : misc::total_residue, misc::res );
		//current gs
		current_gs.Color_mode = ColorMode ( current_gs.Color_mode + 1 );
		if ( current_gs.Color_mode > RESIDUE_CPK_COLOR ) current_gs.Color_mode = RAINBOW_COLOR;
		break;
	case 'l':
	case 'L':
		peptide_planes = !peptide_planes;
		//current gs
	  current_gs.LIGdisplay_state = LIGdisplayState ( current_gs.LIGdisplay_state + 1 );
		if ( current_gs.LIGdisplay_state > SHOW_LIG_SPACEFILL ) current_gs.LIGdisplay_state = SHOW_NOLIG;
		break;
	case 'p':
	case 'P': // pause.
		pause_after_graphics = !pause_after_graphics;
		//		std::cout << "Pause after graphics " << pause_after_graphics << std::endl;
		if (pause_after_graphics) wait_on_the_graphics = true;
		else wait_on_the_graphics = false;
		break;
  case 's':
  case 'S':
		current_gs.SCdisplay_state = SCdisplayState ( current_gs.SCdisplay_state + 1 );
		//if ( current_gs.SCdisplay_state > SHOW_WIREFRAME ) current_gs.SCdisplay_state = SHOW_NOSC;
		if ( current_gs.SCdisplay_state > SHOW_SC_SPACEFILL ) current_gs.SCdisplay_state = SHOW_NOSC;
		break;
  case 'a':
  case 'A':
		current_gs.BBdisplay_state = BBdisplayState ( current_gs.BBdisplay_state + 1 );
		//if ( current_gs.BBdisplay_state > SHOW_BACKBONE ) current_gs.BBdisplay_state = SHOW_NOBB;
		if ( current_gs.BBdisplay_state > SHOW_BB_SPACEFILL ) current_gs.BBdisplay_state = SHOW_CARTOON;
		break;
	case 'n':
	case 'N':
		new_drawing = !new_drawing;
		break;
	}

	redrawallwindows();
}

///////////////////////////////////////////////////////////////////////////////
// Need to redraw everything when user has asked for different color
// scheme, centroids, etc. with a key press
void redrawallwindows() {
	using namespace graphics;

	//Redraw all windows:
	glutSetWindow( pos_window );
	position_display();
	glutPostRedisplay();

	if( best_window != 999 ) {
		glutSetWindow( best_window );
		best_display();
		glutPostRedisplay();
	}

	if( low_window != 999 ) {
		glutSetWindow( low_window );
		low_display();
		glutPostRedisplay();
	}

	if( text_window != 999 ) {
		glutSetWindow( text_window );
		text_window_display();
		glutPostRedisplay();
	}

	if( trajectory_window != 999 ) {
		glutSetWindow( trajectory_window );
		trajectory_display();
		glutPostRedisplay();
	}

}

///////////////////////////////////////////////////////////////////////////////
void
redraw_text_window()
{
	text_window_init();
	glutSetWindow( graphics::text_window );
	text_window_display();
	glutPostRedisplay();
}


///////////////////////////////////////////////////////////////////////////////
void
draw_score_plot(
    std::vector<float> & score_vector,
		bool vertical,
		float & score_min,
		float & score_max ) {
//lin show score trajectory
	score_min = 0.0;// = int (misc::low_score - 10 );
	score_max = 0.0;// = int (misc::low_score + 30 );
	get_bounds( score_vector, score_min, score_max );
	float margin ( (score_max-score_min) * 0.1 );
	score_min -= margin;
	score_max += margin;
	//std::cout<<score_min<<" "<<score_max<<std::endl;
	protein_graphics::plot_timeseries( score_vector, vertical, score_min, score_max );
	//glLoadIdentity();
}

void
draw_box_boundary( xyzVector_float const & line_color ) {
	gluOrtho2D( 0, 1, 0, 1 );
	glMatrixMode(GL_MODELVIEW);
	glDisable( GL_DEPTH_TEST );
	glLoadIdentity();
	glColor3fxyz(line_color);
	glTranslatef( 0.0, 0.0, 0.0 );
	glBegin( GL_LINE_STRIP ) ;
	glVertex2f(0,0);
	glVertex2f(0,1);
	glVertex2f(1,1);
	glVertex2f(1,0);
	glVertex2f(0,0);
	glEnd();
	glLoadIdentity();
}

void
get_score_rms_from_trajectory(
		std::vector< graphics::T_point > const & data,
		std::vector< float > & score,
		std::vector< float > & rms,
		protein_graphics::GraphicsState const & gs ) {
	using namespace protein_graphics;
	using namespace graphics;

	score.clear();
	rms.clear();

	for ( unsigned int i=0; i<data.size(); ++i ) {
		T_point const & p( data[i] );
		switch( gs.Trajectory_state ) {
		case SHOW_LOW:
			if( p.low_accept() ) {
				score.push_back(p.score);
				rms.push_back(p.rms_err);
			}
			break;
		case SHOW_BEST:
			if( p.best_accept() ) {
				score.push_back(p.score);
				rms.push_back(p.rms_err);
			}
			break;
		case SHOW_MC_TRIALS:
			if( p.mc_accept() ) {
				score.push_back(p.score);
				rms.push_back(p.rms_err);
			}
			break;
		case SHOW_ALL_TRIALS:
			score.push_back(p.score);
			rms.push_back(p.rms_err);
			break;
		}
	}
	//std::cout<<"trajectory: "<<gs.Trajectory_state<<" "<<score.size() <<" "<<rms.size()<<std::endl;
}

void
gl_graphics_clear_trajectory()
{ graphics::trajectory.clear(); }

void
gl_graphics_append_trajectory( float const score, float const rms ) {

	graphics::trajectory.push_back( graphics::T_point( score, score, score,
																 rms, graphics::NOMC_TRIAL ) );
}



#endif

