// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15154 $
//  $Date: 2007-05-25 02:00:33 -0400 (Fri, 25 May 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "gunn.h"
#include "current_pose.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fragments_pose.h"
#include "loops.h"
#include "maps.h"
#include "misc.h"
#include "param.h"
#include "pose.h"
#include "random_numbers.h"
#include "refold.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/trig.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>


// Namespaces
namespace gunnvars {
	using namespace param;
	FArray3D_float q1( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
	FArray3D_float q2( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
	FArray3D_float q3( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
	FArray3D_float q4( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
	FArray3D_float q5( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
	FArray3D_float q6( MAX_RES(), MAX_NEIGH(), n_frag_sizes() );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin gunn
///
/// @brief
///  calculate Gunn variables for residues j and k in position array
///
/// @detailed
///
/// @param  j - [in/out]? -
/// @param  k - [in/out]? -
/// @param  q1a - [in/out]? -
/// @param  q2a - [in/out]? -
/// @param  q3a - [in/out]? -
/// @param  q4a - [in/out]? -
/// @param  q5a - [in/out]? -
/// @param  q6a - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
gunn(
	int j,
	int k,
	FArray3Da_float Eposition,
	float & q1a,
	float & q2a,
	float & q3a,
	float & q4a,
	float & q5a,
	float & q6a
)
{
	using namespace param;
	using numeric::sin_cos_range;

	Eposition.dimension( 3, MAX_POS, k );

	assert( j <= k );


// local
	FArray1D_float p1( 3 );
	FArray1D_float p2( 3 );
	FArray1D_float p3( 3 );
	FArray1D_float R( 3 );
	FArray1D_float x1( 3 );
	FArray1D_float x2( 3 );
	FArray1D_float y1( 3 );
	FArray1D_float y2( 3 );
	FArray1D_float z1( 3 );
	FArray1D_float z2( 3 );
	FArray1D_float temp( 3 );

//   setup coordinate system around residue j

	p1(1) = Eposition(1,1,j); // N
	p1(2) = Eposition(2,1,j);
	p1(3) = Eposition(3,1,j);

	p2(1) = Eposition(1,2,j); // CA
	p2(2) = Eposition(2,2,j);
	p2(3) = Eposition(3,2,j);

	p3(1) = Eposition(1,4,j); // C
	p3(2) = Eposition(2,4,j);
	p3(3) = Eposition(3,4,j);

	subvec(p1,p2,temp);
	unitvec(temp,x1);

	subvec(p3,p2,temp);
	cros(x1,temp,z1);
	unitvec(z1,z1);

	cros(z1,x1,y1);

//  setup coordinate system around residue k

	p1(1) = Eposition(1,1,k);
	p1(2) = Eposition(2,1,k);
	p1(3) = Eposition(3,1,k);

	p2(1) = Eposition(1,2,k);
	p2(2) = Eposition(2,2,k);
	p2(3) = Eposition(3,2,k);

	p3(1) = Eposition(1,4,k);
	p3(2) = Eposition(2,4,k);
	p3(3) = Eposition(3,4,k);

	subvec(p1,p2,temp);
	unitvec(temp,x2);

	subvec(p3,p2,temp);
	cros(x2,temp,z2);
	unitvec(z2,z2);

	cros(z2,x2,y2);


//  compute inter residue vector

	R(1) = Eposition(1,2,k)-Eposition(1,2,j);
	R(2) = Eposition(2,2,k)-Eposition(2,2,j);
	R(3) = Eposition(3,2,k)-Eposition(3,2,j);


//  compute gunn quantities

	q6a = std::sqrt( dotprod(R,R) );

	unitvec(R,R);
	q1a = dotprod(x1,R);

	q2a = dotprod(x2,R);
	q3a = dotprod(x1,x2) - q1a*q2a;
	q3a /= ( std::sqrt( (1-q1a*q1a) * (1-q2a*q2a) ) + .0001 );
	q3a = std::acos( sin_cos_range( q3a ) );
	q4a = std::acos( sin_cos_range( dotprod(y1,R) / ( std::sqrt(1-q1a*q1a) + .0001 ) ) );
	q5a = std::acos( sin_cos_range( dotprod(y2,R) / ( std::sqrt(1-q2a*q2a) + .0001 ) ) );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin precompute_gunn
///
/// @brief
///car fills the gunn variable arrays
///
/// @detailed
///                   WARNING WARNING WARNING
/// This overwrites the first 'length' positions in the position array
/// and does not restore them. The phi,psi omega arrays are restored!
///
/// @param  length - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
precompute_gunn(
	int length,
	int frag_start,
	int frag_depth,
	int size_bin,
	FArray3Da_float Eposition
)
{
	using namespace gunnvars;
	using namespace param;

	gunn(1,length, Eposition,q1(frag_start,frag_depth,size_bin),
	 q2(frag_start,frag_depth,size_bin),q3(frag_start,frag_depth,size_bin),
	 q4(frag_start,frag_depth,size_bin), q5(frag_start,frag_depth,size_bin),
	 q6(frag_start,frag_depth,size_bin));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin compare_gunn
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? - position being compared
/// @param  length - [in/out]? - actual length
/// @param  size - [in/out]? - frag bin number corresponding to this length
/// @param  cost - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compare_gunn(
	int i,
	int length,
	int size,
	FArray1DB_float & cost
)
{
	using namespace fragments;
	using namespace gunnvars;
	using namespace param;
	using namespace misc; // Eposition-- operate on global array
	using namespace numeric::constants::f;

	float d3,d4,d5;
	float q1p,q2p,q3p,q4p,q5p,q6p,c1,c2,c3,c4;

	if ( ! frag_precompute::initialized(size) ) {
		precompute_frag_movement( length, size, total_residue ); // make sure it's initialized
	}

	c1 = 2.035;
	c2 = 0.346;
	c3 = 5.72;
	c4 = 3.84;

//  compute gunn variables for current protein structure

	gunn(i,i+length-1,Eposition,q1p,q2p,q3p,q4p,q5p,q6p);

//  compute gunn variables for all frags

	int const q_size1 = q1.size1();
	for ( int j = 1, lq = q1.index(i,j,size), je = align_depth(i,size); j <= je;
	 ++j, lq += q_size1 ) { // MAX_NEIGH
		//Objexx: Linear indexing assumes all q arrays have the same dimensions
		//Objexx: q_[ lq ] below is q_(i,j,size) for each array q_

		d3 = std::abs(q3[ lq ]-q3p);
		if ( d3 > pi_over_2 ) d3 = pi - d3; //Objexx:FMD - change to a function to compute the reference angle?
		d4 = std::abs(q4[ lq ]-q4p);
		if ( d4 > pi_over_2 ) d4 = pi - d4;
		d5 = std::abs(q5[ lq ]-q5p);
		if ( d5 > pi_over_2 ) d5 = pi - d5;

		cost(j) = 2.92 +
		 c3 * std::log( 1.0 + ( std::abs(q1[ lq ]-q1p) + std::abs(q2[ lq ]-q2p) ) ) +
		 c2 * std::log( 1.0 + std::abs(q6[ lq ]-q6p) ) +
		 c1 * std::log( 1.0 + d3 ) +
		 c4 * std::log( 1.0 + d4 + d5 );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_fragment_gunn
///
/// @brief
///  THIS FUNCTION WILL CHOOSE A SECONDARY SEQUENCE RANDOMLY FROM THE
///  ALIGNMENT DATA
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  frag_begin - [in/out]? - return value (in maps namespace not the global)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_fragment_gunn(
	int const size,
	float cutoff,
	int & frag_begin // return value (not the global)
)
{
// return values are in maps namespace
// return values: frag_size, frag_begin

	using namespace fragments;
	using namespace misc;
	using namespace param;

	int nn_num,size_bin;
	FArray1D_float cost( MAX_NEIGH()() );
	float costmin;
	FArray1D_int goodfrag( MAX_NEIGH()() );
	int numgood, minfrag;
	int problems;
	float end_bias,end_dist;
	FArray1D_bool allow_insert( MAX_RES()() );
	FArray1D_int insert_map( MAX_RES()() );
	int total_insert;

	end_bias = 60.;
	retrieve_allow_insert(allow_insert,total_residue);
	retrieve_insertmap(insert_map,total_insert);
	size_bin = get_index_by_frag_size(size);

L50:; // return here if can't find a good fragment after 10
	// tries at position frag_begin
	problems = 0;

L55:
	frag_begin = insert_map(static_cast< int >(ran3()*(total_insert-size))+1);

	end_dist = std::abs(frag_begin-total_residue/2);
	if ( !get_loop_flag() && ran3() > std::exp(-(end_dist/end_bias )) ) goto L55;
	if ( align_depth(frag_begin,size_bin) == 0 ) goto L55; // no frags that size
	for ( int i = frag_begin, ie = frag_begin + size - 1; i <= ie; ++i ) {
		if ( !allow_insert(i) ) goto L55; // insertion into fixed region
	}

	compare_gunn(frag_begin,size,size_bin,cost);

L40:; // jump back here to try a new choice at frag_begin

	costmin = 1000;
	numgood = 0;
	minfrag = 0;
	int const je = align_depth( frag_begin, size_bin );
	for ( int j = 1; j <= je; ++j ) { // MAX_NEIGH
		goodfrag(j) = 0;
		if ( cost(j) < 2.95 ) cost(j) = 100; // too similar
	}

	for ( int j = 1; j <= je; ++j ) { // MAX_NEIGH
		if ( cost(j) < costmin ) {
			costmin = cost(j);
			minfrag = j;
		}

		if ( cost(j) < cutoff ) {
			++numgood;
			goodfrag(numgood) = j;
		}
	}

	if ( numgood < 1 ) {
		nn_num = minfrag;
//        std::cout << "no fragments for " << frag_begin << SS( cost(minfrag) ) << std::endl;
		if ( cost(minfrag) > 12. ) goto L50;
	} else {
		nn_num = static_cast< int >(ran3()*numgood)+1;
		nn_num = goodfrag(nn_num);
	}
//      best_cost = cost(nn_num);


//car  March 21,2001
//car  no longer split phi and psi for residues frag_begin and
//car  frag_begin+size-1.  This change results in some error
//car  in the gunn predictors (which don't use phi(frag_begin) or
//car  psi(frag_begin+size-1) but is necessary to prevent residues
//car  from ending up in disallowed regions of the rama table.

//$$$      psi(frag_begin) = align_psi(frag_begin,nn_num,0,size_bin);
//$$$      phi(frag_begin+size-1) =
//$$$       align_phi(frag_begin,nn_num,size-1,size_bin);

	for ( int i = 0; i < size; ++i ) {
//$$$         if ( i > 0 && i < (size-1) ) {
		phi(i+frag_begin) = align_phi(frag_begin,nn_num,i,size_bin);
		psi(i+frag_begin) = align_psi(frag_begin,nn_num,i,size_bin);
//$$$         }
		if ( align_phi(frag_begin,nn_num,i,size_bin) == 0.0 ||
		 align_psi(frag_begin,nn_num,i,size_bin) == 0.0 ) {
//            std::cout << align_phi(frag_begin,nn_num,i,size_bin) <<
//             align_psi(frag_begin,nn_num,i,size_bin) << std::endl;
			goto L30;
		}
		omega(i+frag_begin) = align_omega(frag_begin,nn_num,i,size_bin);
		secstruct(i+frag_begin) = ss_type(frag_begin,nn_num,i,size_bin);
		name(i+frag_begin) = align_name(frag_begin,nn_num,size_bin);
	}

// DONT ALLOW HELICES OF LESS THAN 4 OR STRANDS OF LESS THAN 2
//      helix_len = 0
//      strand_len = 0
//      for ( i = 1; i < total_residue; ++i ) {
//         if ( secstruct(i) == 'H' ) ++helix_len;
//         if ( secstruct(i) == 'E' ) ++strand_len;
//         if ( secstruct(i) != secstruct(i+1) ) {
//            if ( helix_len != 0 && helix_len < 4 ) {
//               resetphipsi();
//               goto L30;
//            }
//            if ( strand_len != 0 && strand_len < 2 ) {
//               resetphipsi();
//               goto L30;
//            }
//            helix_len = 0;
//            strand_len = 0;
//         }
//      }

	return;

L30:

	++problems;
	cost(nn_num) = 1000;
	if ( problems < 10 ) {
		goto L40;
	} else {
		std::cout << " problems with fragments "
		 << frag_begin << ' ' << nn_num << std::endl;
		goto L50;
	}
}

///////////////////////////////////////////////////////////////////////////////
void
choose_fragment_gunn_pose(
	pose_ns::Pose & pose,
	int const size,
	float const cutoff
)
{
	int begin = 0;
	choose_fragment_gunn_pose( pose, size, begin, cutoff );
}

void
choose_fragment_gunn_pose(
	pose_ns::Pose & pose,
	const int size,
	int & frag_begin, // location frag was inserted
	float const cutoff
)
{
	moves_set_current_pose( pose );

	choose_fragment_gunn( size, cutoff, frag_begin );

	for ( int i=frag_begin; i<frag_begin+size; ++i ) {
		pose.set_phi      ( i, misc::phi      ( i ));
		pose.set_psi      ( i, misc::psi      ( i ));
		pose.set_omega    ( i, misc::omega    ( i ));
		pose.set_secstruct( i, misc::secstruct( i ));
	}

	moves_reset_current_pose();

}
