
// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13795 $
//  $Date: 2007-03-27 19:00:45 -0500 (Tue, 27 Mar 2007) $
//  $Author: rvernon $

#ifndef INCLUDED_jump_classes
#define INCLUDED_jump_classes


// Rosetta Headers
#include "kin_stub.h"
#include "pose_param.h"

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
//#include <numeric/xyz.functions.hh>
//#include <numeric/xyzVector.io.hh>
//#include <numeric/xyzMatrix.io.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>

// C++ Headers
#include <iosfwd>


////////////////////////////////////////////////////////////////////////////////80
// first try at making a "pose" object
//
// interface: .set_phi(...) .score() .refold()
//
// book-keeping:
// torsions are accessible through set() functions
// these calls automatically set both the corresponding variable and
// internal book-keeping variables that look like new_<tag>_REFOLD
// where <tag> is "bb","chi","jump",and "tree" (for changes to the fold_tree)
//
//
// .refold() checks these variables to see what torsions have changed and folds
// accordingly. So the job of these vars is to record mismatches between the
// torsions and the coordinates
//
// at the end of .refold() the new_*_REFOLD vars all set to false, and
// corresponding new_*_SCORE variables are set to true. These variables
// record changes in the coordinates so that stored score data can be
// updated properly.
//
// outside routines can get (res_moved, pair_moved) and/or the domain_map
// to figure out which parts of the structure have changed.
//
// score book-keeping: this is probably the trickiest part. The idea is to
// include in the pose all the common variables like cendist, vdw_bumps
// and fullatom energies that are used to make scoring fast, but that
// make book-keeping such a headache. All this stuff is bundled inside
// the "Score_data" obect, defined in score_data.h
// During the course of a score
// evaluation routines can access and/or modify this data, by calls
// to score_data.get_*() or score_data.set_*()
// get_* return const references, set_* return non-const references
// The key assumption is that calls to set_* accessing a given type
// of stored score data will update this data to accurately measure
// the current coordinates. This allows us to know for any type of
// score data what its status is, and this status is returned along
// with the references in calls to get_ or set_

////////////////////////////////////////////////////////////
// thinking more about how to efficiently mesh with Rosetta:
// maybe use the misc arrays to communicate with scoring functions
//
// the main things to take from pose:
//
// ** stored score data like cendist, bumps, atr_pair
// ** allow arrays like allow_bb_move ( --> allow_insert )
// ** book-keeping arrays like domain_map, pair_moved
//
// ** also monte_carlo will take stored data from get_current_MC()
//    which returns the current Monte Carlo book-keeping object
//
// torsion modifications have to occur through calls to pose.set_*
//
// changing the torsions in misc will have no effect on the next refold,
// just might screw up rama,paa,etc evaluations ( checking for this in after_scoring)
//
// so the torsion-modifying routines like small/shear-moves, choose_fragment,
// and unpack_phipsi have to work with the pose.
//
// refold_pose ( Pose & pose ) will be in charge of filling misc arrays



namespace pose_ns
{

	/////////////////////////////////////////////////////////////////////////////
	class RT { // rotation + translation object, used in jumps
	private:
		numeric::xyzMatrix_double rotation; //FArray2D_double rotation; // 3x3
		numeric::xyzVector_double translation; //FArray1D_double translation; // 3


	public: // Types


		// Types to prevent compile failure when std::distance is in scope
		typedef  void  iterator_category;
		typedef  void  difference_type;


	public:
		// initialize with a pair of N,CA,C positions
		RT ():
			rotation(0.0), //rotation(3,3,0.0),
			translation(0.0) //translation(3,0.0)
		{}
		RT ( FArray2Da_float Epos1, FArray2Da_float Epos2 );
		RT ( kin::Stub const & stub1, kin::Stub const & stub2 );
		RT ( const RT & src ):
			rotation( src.rotation ),
			translation( src.translation )
		{}
		void from_stubs( kin::Stub const & stub1, kin::Stub const & stub2 );
		void reverse();
		void reset();
		void identity_transform(); // reset to identity matrix, 0 translation
		void
		set_translation(
			numeric::xyzVector_double const & t
		);

		void
		set_rotation(//chutmp
			numeric::xyzMatrix_double const & r
		);

		numeric::xyzMatrix_double const &
		get_rotation() const;

		numeric::xyzVector_double const &
		get_translation() const;


		void fold_in_rb_deltas( FArray1Da_double rb, const numeric::xyzVector_double & rb_center );

		void make_jump( FArray2Da_float Epos_in, FArray2Da_float Epos_out ) const;
		void make_stub_jump( kin::Stub const & stub1, kin::Stub & stub2 ) const;
		friend std::ostream & operator <<( std::ostream & os, const RT & rt );
		friend std::istream & operator >>( std::istream & is, RT & rt );
		RT & operator =( const RT & src ) {
			rotation = src.rotation;
			translation = src.translation;
			return *this;
		}
		bool ortho_check() const; // check for orthogonality

		friend
		inline
		double
		distance( RT const & a, RT const & b );
	};

	inline
	double
	distance( RT const & a, RT const & b )
	{
		using namespace numeric; //SGM Work-around for GCC 3.x bug
		return
			std::sqrt( distance_squared( a.rotation.col(1), b.rotation.col(1) ) +
								 distance_squared( a.rotation.col(2), b.rotation.col(2) ) +
								 distance_squared( a.rotation.col(3), b.rotation.col(3) ) +
								 distance_squared( a.translation,     b.translation ) );
	}

        void
	jump_distance(  Jump const & a_in, Jump const & b_in,
		       double & dist, double & theta );

	/////////////////////////////////////////////////////////////////////////////
	class Jump {
	private:
		RT rt;
		FArray2D_double rb_delta; // 6x2
		FArray1D< numeric::xyzVector_double > rb_center; // 3x2
	public:
		Jump ():
			rt(),
			rb_delta( 6, 2, 0.0 ),
			rb_center( 2, numeric::xyzVector_double(0.0) )
		{}
		Jump( const RT & src_rt ):
			rt( src_rt ),
			rb_delta ( 6, 2, 0.0 ),
			rb_center( 2, numeric::xyzVector_double(0.0) )
		{}
		Jump ( FArray2Da_float Epos1, FArray2Da_float Epos2 ):
			rt( Epos1, Epos2),
			rb_delta( 6, 2, 0.0 ),
			rb_center( 2, numeric::xyzVector_double(0.0) )
		{}
		Jump ( kin::Stub const & stub1, kin::Stub const & stub2 ):
			rt( stub1, stub2),
			rb_delta( 6, 2, 0.0 ),
			rb_center( 2, numeric::xyzVector_double(0.0) )
		{}
		Jump ( const Jump & src ):
			rt( src.rt ),
			rb_delta( src.rb_delta ),
			rb_center( src.rb_center )
		{}

		Jump &
		operator =( Jump const & src ) {
			rt = src.rt;
			rb_delta = src.rb_delta;
			rb_center = src.rb_center;
			return *this;
		}
		void from_stubs( kin::Stub const & stub1, kin::Stub const & stub2 );
		void
		make_stub_jump(
			int const dir,
			kin::Stub const & stub1,
			kin::Stub & stub2
		) const;
		void
		make_stub_jump( kin::Stub const & stub1, kin::Stub & stub2) const {
			make_stub_jump( pose_param::n2c, stub1, stub2 );}
		bool ortho_check() const;
		bool nonzero_deltas() const;
		void reset();
	  //void random_rot( const float alpha_in );
		void random_trans( const float dist_in );

		void gaussian_move(
			int const dir,
			float const trans_mag,
			float const rot_mag
		);

		void rotation_by_matrix(
			FArray2Da_float Epos,
			numeric::xyzVector_double const & center, //in xyz frame
			numeric::xyzMatrix_double const & matrix //in xyz frame
		);

		void rotation_by_matrix(
			kin::Stub const & stub,
			numeric::xyzVector_double const & center, //in xyz frame
			numeric::xyzMatrix_double const & matrix //in xyz frame
		);

		void rotation_by_axis(
			FArray2Da_float Epos,
			numeric::xyzVector_double const & axis_in, //in xyz frame
			numeric::xyzVector_double const & center, // in xyz frame
			float const alpha
		);

		void translation_along_axis(
			FArray2Da_float Epos,
			numeric::xyzVector_double const & axis,
			float const dist
		);

		void translation_along_axis(
			kin::Stub const & stub,
			numeric::xyzVector_double const & axis,
			float const dist
		);

		void make_jump( const int dir, FArray2Da_float Epos_in,
										FArray2Da_float Epos_out ) const;

		void
		identity_transform(); // reset to identity matrix, 0 translation

		void
		reverse();

		numeric::xyzMatrix_double const &
		get_rotation() const;

		numeric::xyzVector_double const &
		get_translation() const;

		void
		set_rotation( numeric::xyzMatrix_double const & R_in );

		void
		set_translation(
			numeric::xyzVector_double const & t
		);

		// void
// 		rotation_by_axis(
// 			double const phi,
// 			double const theta,
// 			double const alpha
// 		);

		void
		symmetry_jump(
			numeric::xyzVector_double axis,
			numeric::xyzVector_double const & offset,
			int const N
		);

		void fold_in_rb_deltas(); // call after we're done minimizing

		// get/set rb_delta
		// get
		inline FArray1Da_double const get_rb_delta( int const dir ) const;
		inline double get_rb_delta( int const rb_no, int const dir ) const;
		// set
		inline void set_rb_delta( int const rb_no, int const dir, double const value );

		// get/set rb_center
		// get
		inline numeric::xyzVector_double const get_rb_center( int const dir ) const;
		// set
		void set_rb_center(
			const int dir,
			FArray2Da_float jump_Epos,
			numeric::xyzVector_double const & center
		);

		// set
		void set_rb_center(
			const int dir,
			kin::Stub const & stub,
			numeric::xyzVector_double const & center
		);

		// stream output
		friend std::ostream & operator <<( std::ostream & os, const Jump & jump );
		friend std::istream & operator >>( std::istream & is, Jump & jump );
		friend double distance( Jump const & a_in, Jump const & b_in );
	};
	// inline definitions for Jump class

	// access/set deltas
	// get rb_delta
	inline FArray1Da_double const Jump::get_rb_delta( int const dir ) const {
		return rb_delta( 1, (dir==pose_param::n2c) ? 1 : 2 ); }

	inline double Jump::get_rb_delta( int const rb_no, int const dir ) const {
		return rb_delta( rb_no, (dir==pose_param::n2c ? 1 : 2 ) ); }

	// set rb_delta
	inline void Jump::set_rb_delta( const int rb_no, const int dir, const double value ) {
		rb_delta( rb_no, (dir==pose_param::n2c ? 1 : 2 ) ) = value; }

	// access/set centers
	// get rb_center
	inline numeric::xyzVector_double const Jump::get_rb_center( int const dir ) const {
		return rb_center( (dir==pose_param::n2c) ? 1 : 2 ); }

	// set rb_center
//  	inline void Jump::set_rb_center( const int dir, const FArray1D_double & center ) {
//  		const int index ( ( dir == pose_param::n2c ) ? 1 : 2 );
//  		for ( int i=1; i<= 3; ++i ) {
//  			rb_center( i, index ) = center( i );
//  		}
//  	}
}

#endif
