// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18526 $
//  $Date: 2007-11-19 02:09:57 -0500 (Mon, 19 Nov 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "jumping_loops.h"
#include "after_opts.h"
#include "angles.h"
#include "current_pose.h"
#include "dna_ns.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "idealize.h"
#include "jumping_diagnostics.h"
#include "jumping_ns.h"
#include "jumping_minimize.h" // debug
#include "jumping_refold.h"
#include "jumping_util.h"
#include "kin_test.h"
#include "loops.h"
#include "loop_relax.h"
#include "loop_class.h"
#include "make_pdb.h" // debugging
#include "misc.h"
#include "monte_carlo.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_constraints.h"
#include "prof.h"
#include "ramachandran.h"
#include "random_numbers.h"
#include "refold.h"
#include "refold_ns.h"
#include "score.h"
#include "score_ns.h"
#include "util_basic.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>

//DANGER:
using namespace pose_ns;

//////////////////////////////////////////////////////////////////////////////
/// @begin set_jmp_chainbreak_weight
///
/// @brief
///
/// @detailed
///
/// @param[in]  weight
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_jmp_chainbreak_weight( float const weight /* input */ )
{
	using namespace scorefxns;

	std::cout << "set_jmp_chainbreak_weight:: "<<
		"This no longer works. these kinds of external weight-" <<
		"setting routines are too dangerous.\nYou need to create a " <<
		"Score_weight_map object and set the weight inside there" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	std::cout << "setting jmp_chainbreak_weight=" << SS( weight ) << std::endl;
	jmp_chainbreak_weight = weight;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_show_chainbreaks
///
/// @brief
/// shows chainbreaks in the best structure
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
jmp_show_chainbreaks(
	pose_ns::Pose const & pose
)
{
	using namespace jumping;
	//using namespace misc;

	float const dist2_target = { 1.7424 }; // C to N distance, squared

	int cutpoint;
	float dist2;

	Fold_tree const & fold_tree ( pose.fold_tree() );
	int num_fold_tree_cutpoint;
	FArray1D_int const & fold_tree_cutpoint
		( fold_tree.get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );

	FArray3D_float const & Epos( pose.Eposition() );
	for ( int ncut = 1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
		cutpoint = fold_tree_cutpoint(ncut);
		if ( !pose.is_protein( cutpoint ) || !pose.is_protein( cutpoint+1 ) )
			continue;
		dist2 =
		 square( Epos(1,4,cutpoint) - Epos(1,1,cutpoint+1) ) +
		 square( Epos(2,4,cutpoint) - Epos(2,1,cutpoint+1) ) +
		 square( Epos(3,4,cutpoint) - Epos(3,1,cutpoint+1) );
		if ( dist2 > dist2_target + 2.0 ) {
			std::cout << "chain_break: " << I( 4, cutpoint ) << ' ' <<
			 F( 9, 3, std::sqrt(dist2) - std::sqrt(dist2_target) ) << std::endl;
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_get_chainbreaks_by_loops
///
/// @brief
/// shows chainbreaks in the best structure
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_writep
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
jmp_get_chainbreaks_by_loops(
	pose_ns::Pose const & pose,
	pose_ns::Loops const & loops,
	std::vector< float > & chain_breaks
)
{
	using namespace jumping;
	//using namespace misc;

	float const dist2_target = { 1.7424 }; // C to N distance, squared

	int cutpoint;
	float dist2;

	FArray3D_float const & Epos( pose.Eposition() );

	// rebuilt by looprlx
	for( pose_ns::Loops::const_iterator it = loops.begin(), it_end = loops.end();
			 it != it_end; ++it ) {

		cutpoint = it->cut();
		if( cutpoint == 1 || cutpoint == pose.total_residue() ){
			chain_breaks.push_back(0.0);
			continue;
		}
		dist2 =
		 square( Epos(1,4,cutpoint) - Epos(1,1,cutpoint+1) ) +
		 square( Epos(2,4,cutpoint) - Epos(2,1,cutpoint+1) ) +
		 square( Epos(3,4,cutpoint) - Epos(3,1,cutpoint+1) );
		chain_breaks.push_back( std::sqrt(dist2) - std::sqrt(dist2_target) );
	}
}


//////////////////////////////////////////////////////////////////////////////
/// @begin set_jmp_chainbreak_overlap
///
/// @brief
///
/// @detailed
///
/// @param[in]  setting
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_jmp_chainbreak_overlap( int const setting /* input */ )
{
	jumping::jmp_chainbreak_overlap = setting;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_chainbreak_score
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
jmp_chainbreak_score()
{
	//using namespace jumping;
	//using namespace misc;
	using namespace param_aa;
	using namespace rna_variables; //to reduce magic numbers.

	if ( !score_check_current_pose( ) ) {
		std::cout << "jmp_chainbreak_score: Shouldn't be here!" << std::endl;
		return 0.0;
	}

	if ( jumping::darpa ) {
		// THIS IS A TOTAL HACK!!
		return darpa_chainbreak_score( score_get_current_pose() );
	}

	int cutpoint,istart,istop,seqpos;
	float dist2,msd;

	float const dist2_target = { 1.7424 }; // C to N distance, squared

	///////
	// POSE
	Pose & pose ( score_get_current_pose() );

	// cutpoints:
	int num_fold_tree_cutpoint;
	FArray1D_int const & fold_tree_cutpoint
		( pose.fold_tree().get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );
	// overlap coords:
	FArray4D_float const & jmp_overlap_Eposition
		( pose.get_overlap_Eposition( jumping::jmp_chainbreak_overlap ));

	// cut_weight
	FArray1D_float cut_weight( num_fold_tree_cutpoint, 1.0);
	if ( cut_weight.size1() == scorefxns::cut_weight.size1() )
		cut_weight = scorefxns::cut_weight;

	float chainbreak_score( 0.0 ); // will be stored in pose at the end

	if ( num_fold_tree_cutpoint <= 0 ) {
		// pass
	} else if ( jumping::jmp_chainbreak_overlap == 0 ) {
		// distance only; no overlap, linear in delta from target
		if ( get_currently_minimizing() ) {
			std::cout << "DANGER:: 0-overlap chainbreak score does not match the " <<
				"derivative!!!!!!!!!!!!!!\nTalk to Phil about fixing this." <<
				std::endl;
		}
		for ( int ncut = 1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
			cutpoint = fold_tree_cutpoint(ncut);
			if ( pose.is_protein( cutpoint ) && pose.is_protein( cutpoint+1 ) ) {
				dist2 =
					square( misc::Eposition(1,4,cutpoint) - misc::Eposition(1,1,cutpoint+1) ) +
					square( misc::Eposition(2,4,cutpoint) - misc::Eposition(2,1,cutpoint+1) ) +
					square( misc::Eposition(3,4,cutpoint) - misc::Eposition(3,1,cutpoint+1) );

				chainbreak_score += std::sqrt( std::abs( dist2 - dist2_target ) ) * cut_weight(ncut);

			}
			if ( pose.is_RNA( cutpoint ) && pose.is_RNA( cutpoint+1 ) ) {
				// O3* to P.
				numeric::xyzVector_float o3star_vec( & misc::full_coord(1, o3star, cutpoint) );
				numeric::xyzVector_float p_vec     ( & misc::full_coord(1, p     , cutpoint+1) );

				static float const dist2_o3star_p_target = 1.61 * 1.61;
				dist2 = (o3star_vec - p_vec).length_squared();
				chainbreak_score += std::sqrt( std::abs( dist2 - dist2_o3star_p_target ) ) * cut_weight(ncut);

				static bool const stringent_rna_chainbreak = truefalseoption( "stringent_rna_chainbreak" );
				if (stringent_rna_chainbreak) {

					numeric::xyzVector_float c4star_vec( & misc::full_coord(1, c4star, cutpoint) );
					numeric::xyzVector_float c3star_vec( & misc::full_coord(1, c3star, cutpoint) );
					numeric::xyzVector_float o1p_vec   ( & misc::full_coord(1, o1p   , cutpoint+1) );
					numeric::xyzVector_float o2p_vec   ( & misc::full_coord(1, o2p   , cutpoint+1) );
					numeric::xyzVector_float o5star_vec( & misc::full_coord(1, o5star, cutpoint+1) );

					//enforce bond angles
					static float const dist2_c3star_p_target = 2.61 * 2.61;
					dist2 = (c3star_vec - p_vec).length_squared();
					chainbreak_score += 0.5 * std::sqrt( std::abs( dist2 - dist2_c3star_p_target ) ) * cut_weight(ncut);
					//					std::cout << "C3star P " << dist2 << " " << dist2_c3star_p_target << " " << cut_weight(ncut) <<  std::endl;

					static float const dist2_o3star_o5star_target = 2.52 * 2.52;
					dist2 = (o3star_vec - o5star_vec).length_squared();
					chainbreak_score += 0.5 * std::sqrt( std::abs( dist2 - dist2_o3star_o5star_target ) ) * cut_weight(ncut);
					//					std::cout << "O3star O5star " << dist2 << " " << dist2_o3star_o5star_target << " " << cut_weight(ncut) << std::endl;

					static float const dist2_o3star_o1p_target = 2.50 * 2.50;
					dist2 = (o3star_vec - o1p_vec).length_squared();
					chainbreak_score += 0.5 * std::sqrt( std::abs( dist2 - dist2_o3star_o1p_target ) ) * cut_weight(ncut);

					static float const dist2_o3star_o2p_target = 2.55 * 2.55;
					dist2 = (o3star_vec - o2p_vec).length_squared();
					chainbreak_score += 0.5 * std::sqrt( std::abs( dist2 - dist2_o3star_o2p_target ) ) * cut_weight(ncut);


					//enforce epsilon torsion to be trans.
					static float const dist2_c4star_p_target = 3.88 * 3.88;
					dist2 = (c4star_vec - p_vec).length_squared();
					chainbreak_score += 0.25 * std::sqrt( std::abs( dist2 - dist2_c4star_p_target ) ) * cut_weight(ncut);

				}

			}
		}
	} else if ( get_fullatom_flag() || get_pose_loop_flag() || get_ccd_closure_exist() || docking::docking_pose_symm_looprlx ) {
		// fullatom -- quadratic in overlap deviation
		// this matches the derivative ( I hope )

		// chu: this derivatives works in both fullatom and centroid and it is set
		// up this way only because Phil wants the AI derivative in his ab initio
		// folding. Since I am not doing that, we will check a specific flag.

		msd = 0.0;
		float tmp;
		for ( int ncut = 1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
			cutpoint = fold_tree_cutpoint(ncut);
			if ( !pose.is_protein( cutpoint ) || !pose.is_protein( cutpoint+1 ) )
				continue;

			istart = std::max( 1-jumping::jmp_chainbreak_overlap, 1-cutpoint );
			istop  = std::min(   jumping::jmp_chainbreak_overlap, misc::total_residue-cutpoint );

			for ( int i = istart; i <= istop; ++i ) {
				seqpos = cutpoint + i;
				for ( int j = 1; j <= 5; ++j ) {
					if ( j != 3 || misc::res(seqpos) != aa_gly ) { // no gly c-beta
						for ( int k = 1; k <= 3; ++k ) {
							tmp = ( misc::Eposition(k,j,seqpos) - jmp_overlap_Eposition(k,j,i,ncut) );
							msd += tmp*tmp*cut_weight(ncut);
						}         // k
					}            // no gly c-beta
				}               // j
			}                  // i

		}                     // ncut
		chainbreak_score = msd; // note: not really Msd, no dividing by natoms
	} else {
		// AI version -- linear in overlap deviation /////////////////////////////
		if ( get_currently_minimizing() ) {
			std::cout << "DANGER:: AI chainbreak score does not match the " <<
				"derivative!!!!!!!!!!!!!!\nTalk to Phil about fixing this." <<
				std::endl;
		}

		for ( int ncut = 1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
			cutpoint = fold_tree_cutpoint(ncut);
			if ( !pose.is_protein( cutpoint ) || !pose.is_protein( cutpoint+1 ) )
				continue;

			istart = std::max( 1-jumping::jmp_chainbreak_overlap, 1-cutpoint );
			istop  = std::min(   jumping::jmp_chainbreak_overlap, misc::total_residue-cutpoint );

			for ( int i = istart; i <= istop; ++i ) {
				seqpos = cutpoint + i;
				for ( int j = 1; j <= 5; ++j ) {
					if ( j != 3 || misc::res(seqpos) != aa_gly ) { // no gly c-beta
						chainbreak_score += cut_weight(ncut) *
							std::sqrt( square( misc::Eposition(1,j,seqpos) - jmp_overlap_Eposition(1,j,i,ncut) ) +
												 square( misc::Eposition(2,j,seqpos) - jmp_overlap_Eposition(2,j,i,ncut) ) +
												 square( misc::Eposition(3,j,seqpos) - jmp_overlap_Eposition(3,j,i,ncut) ) );
					}
				}               // j
			}                  // i
		}                     // ncut
		chainbreak_score /= 20.0;
	}

	pose.set_0D_score( CHAINBREAK, chainbreak_score );
	return chainbreak_score;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin scored_ccd_loop_closure
///
/// @brief
///
/// @detailed
///
/// @param  loop_begin - [in/out]? -
/// @param  loop_end - [in/out]? -
/// @param  cutpoint - [in/out]? -
/// @param  ii_cycles - [in/out]? -
/// @param[in]  score_fxn -
/// @param[out]  forward_deviation -
/// @param[out]  backward_deviation -
/// @param  score_check - [in/out]? -
/// @param  rama_check - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
scored_ccd_loop_closure(
	int const loop_begin,
	int const loop_end,
	int const cutpoint,
	int const ii_cycles,
	Scoring_Function score_fxn, // input
	float & forward_deviation, // output
	float & backward_deviation, // output
	bool const score_check,
	bool const rama_check
)
{
	using namespace files_paths;
	// using namespace jumping;
	using namespace misc;

// !!!!!!!!!! DOUBLE PRECISION !!!!!!!!!!!!!!!!!!!
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );
	double alpha, dev, tmp_dev, max_angle_delta;

	int direction, start_pos, stop_pos, increment;

	float starting_score, starting_phi, starting_psi,
	 old_rama_score, new_rama_score,
	 tmp1, tmp2, boltz_factor, probability, crap, tmp_score;

	bool const local_debug = { false };
	bool const local_verbose = { false };

	int const n2c = { 1 };
	int const c2n = { -1 };

	double const max_angle_delta_loop = { 10.0 };
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_helix = { 1.0 };

	float const ccd_temperature = { 0.25 };
	 // for accepting moves based on rama score

	double const min_dev = { 0.001 };

	// PB 06/22/04: took out this call to refold:
	// refold(1,total_residue);

	if ( local_verbose ) std::cout << "start-pdb:" << SS( loop_begin ) << SS( loop_end ) <<
	 SS( cutpoint ) << std::endl;
//	dump_phil_pdb();


	if ( score_check ) {
		starting_score = score_fxn();
	} else {
		starting_score = 0.0;
	}


	int ii = 0;
	for ( ii = 1; ii <= ii_cycles; ++ii ) {

// first forward, then backward
		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			if ( repeat == 1 ) {
				direction = n2c;
				start_pos = loop_begin;
				stop_pos = std::min(loop_end,cutpoint);
				increment = 1;
			} else {
				direction = c2n;
				start_pos = loop_end;
				stop_pos = std::max(loop_begin,cutpoint+1);
				increment = -1;
			}

			for ( int pos = start_pos, end_pos = stop_pos * increment; pos * increment <= end_pos;
			 pos += increment ) {
				if ( secstruct(pos) == 'H' ) {
					max_angle_delta = max_angle_delta_helix;
				} else if ( secstruct(pos) == 'E' ) {
					max_angle_delta = max_angle_delta_strand;
				} else {
					max_angle_delta = max_angle_delta_loop;
				}

				starting_phi = phi(pos);
				starting_psi = psi(pos);

				for ( int torsion = 1; torsion <= 2; ++torsion ) { // 1 is phi, 2 is psi

					get_overlap_pos(cutpoint, direction, M);

					if ( direction == n2c ) {
						for ( int i = 1; i <= 3; ++i ) {
							F(i,1) = Eposition(i,1,cutpoint+1); // N
							F(i,2) = Eposition(i,2,cutpoint+1); // CA
							F(i,3) = Eposition(i,4,cutpoint+1); // C
						}
					} else {
						for ( int i = 1; i <= 3; ++i ) {
							F(i,1) = Eposition(i,1,cutpoint); // N
							F(i,2) = Eposition(i,2,cutpoint); // CA
							F(i,3) = Eposition(i,4,cutpoint); // C
						}
					}

					calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);

					if ( local_debug ) {
						if ( local_verbose ) {
							std::cout << "debug dev:" << SS( pos ) << SS( torsion ) <<
							 SS( direction ) << SS( alpha ) << SS( dev ) << std::endl;
						}

						if ( torsion == 1 ) {
							phi(pos) += alpha;
							phi(pos) = periodic_range(phi(pos),360.0);
						} else {
							psi(pos) += alpha;
							psi(pos) = periodic_range(psi(pos),360.0);
						}

//						dump_pdb('debugprefold');

						refold(loop_begin,loop_end);
						if ( direction == n2c ) {
							for ( int i = 1; i <= 3; ++i ) {
								F(i,1) = Eposition(i,1,cutpoint+1); // N
								F(i,2) = Eposition(i,2,cutpoint+1); // CA
								F(i,3) = Eposition(i,4,cutpoint+1); // C
							}
						} else {
							for ( int i = 1; i <= 3; ++i ) {
								F(i,1) = Eposition(i,1,cutpoint); // N
								F(i,2) = Eposition(i,2,cutpoint); // CA
								F(i,3) = Eposition(i,4,cutpoint); // C
							}
						}

//						dump_pdb('debugrefold');

						get_overlap_pos(cutpoint,direction,M);

						tmp_dev = 0.0;
						for ( int i = 1; i <= 3; ++i ) {
							for ( int j = 1; j <= 3; ++j ) {
								tmp_dev += square( M(j,i) - F(j,i) );
							}
						}

						Dassert_equal( std::sqrt(tmp_dev), std::sqrt(dev), 111 );

						if ( torsion == 1 ) {
							phi(pos) -= alpha;
							phi(pos) = periodic_range( phi(pos),360.0);
						} else {
							psi(pos) -= alpha;
							psi(pos) = periodic_range( psi(pos),360.0);
						}
					}

					if ( alpha > max_angle_delta ) alpha = max_angle_delta;
					 // NOTE: max_angle_delta should be positive
					if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

					if ( torsion == 1 ) {
						phi(pos) += alpha;
						phi(pos) = periodic_range( phi(pos),360.0);
					} else {
						psi(pos) += alpha;
						psi(pos) = periodic_range( psi(pos),360.0);
					}

					refold(loop_begin,loop_end);

					get_overlap_pos(cutpoint,direction,M);

					tmp_dev = 0.0;
					for ( int i = 1; i <= 3; ++i ) {
						for ( int j = 1; j <= 3; ++j ) {
							tmp_dev += square( M(j,i) - F(j,i) );
						}
					}

					if ( score_check ) {

						tmp_score = score_fxn();
						if ( tmp_score > starting_score ) { // reject the move
							if ( torsion == 1 ) {
								phi(pos) = starting_phi;
							} else {
								psi(pos) = starting_psi;
							}
							if ( local_verbose ) std::cout << "score reject:" <<
							 SS( tmp_score - starting_score ) << std::endl;
							refold(loop_begin,loop_end);
						} else {
							if ( local_verbose ) std::cout << "score accept:" <<
							 SS( tmp_score - starting_score ) << std::endl;
						}
					} else {
						tmp_score = 0.0;
					}

					if ( local_verbose ) std::cout << "pos: " << I( 3, pos ) <<
					 " alpha: " << fmt::F( 7, 3, alpha ) <<
					 " del_score: " << fmt::F( 7, 3, tmp_score - starting_score ) <<
					 " dev1: " << fmt::F( 13, 9, dev ) <<
					 " dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

				}               // torsion = 1,2

// evaluate the rama score difference:

				if ( rama_check ) {

					eval_rama_score_residue( res(pos),starting_phi,starting_psi,
					 secstruct(pos),old_rama_score,tmp1,tmp2);

					eval_rama_score_residue( res(pos),phi(pos),psi(pos), secstruct(pos),
					 new_rama_score,tmp1,tmp2);

					if ( new_rama_score > old_rama_score ) {
						boltz_factor = (old_rama_score-new_rama_score)/ccd_temperature;
						probability = std::exp(std::max(-40.0f,boltz_factor) );
						crap = ran3(); // bug if  ran3 call in if statment
						if ( crap >= probability ) { // undo change
							phi(pos) = starting_phi;
							psi(pos) = starting_psi;
							if ( local_verbose ) std::cout << "rama reject:" <<
							 SS( probability ) << SS( new_rama_score-old_rama_score ) << std::endl;
							refold(loop_begin,loop_end);
						} else {
							if ( local_verbose ) std::cout << "rama accept:" <<
							 SS( probability ) << SS( new_rama_score-old_rama_score ) << std::endl;
						}
					} else {             // rama score is better
						if ( local_verbose ) std::cout << "rama accept: better-score" <<
						 SS( new_rama_score-old_rama_score ) << std::endl;
					}
				}               // if ( rama_check )

			}                  // pos = start_pos,stop_pos
		}                  // repeat = 1,2   1=n2c; 2=c2n
//		dump_phil_pdb();

		if ( mod(ii,5) == 0 || ii == ii_cycles ) {
// forward_deviation:
			direction = n2c;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = Eposition(i,1,cutpoint+1); // N
				F(i,2) = Eposition(i,2,cutpoint+1); // CA
				F(i,3) = Eposition(i,4,cutpoint+1); // C
			}

			get_overlap_pos(cutpoint,direction,M);

			tmp_dev = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					tmp_dev += square( M(j,i) - F(j,i) );
				}
			}

			forward_deviation = tmp_dev;

// backward_deviation:
			direction = c2n;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = Eposition(i,1,cutpoint); // N
				F(i,2) = Eposition(i,2,cutpoint); // CA
				F(i,3) = Eposition(i,4,cutpoint); // C
			}

			get_overlap_pos(cutpoint,direction,M);

			tmp_dev = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					tmp_dev += square( M(j,i) - F(j,i) );
				}
			}

			backward_deviation = tmp_dev;

			if ( local_verbose ) std::cout << "fbdev:" <<
			 ' ' << I( 5, loop_begin ) <<
			 ' ' << I( 5, loop_end ) <<
			 ' ' << I( 5, cutpoint ) <<
			 ' ' << I( 5, ii ) <<
			 ' ' << fmt::F( 12, 6, forward_deviation ) <<
			 ' ' << fmt::F( 12, 6, backward_deviation ) <<
			 ' ' << start_file << std::endl;

			if ( forward_deviation < min_dev && backward_deviation < min_dev ) {
				if ( local_verbose ) std::cout << "closed: ii=" << SS( ii ) << std::endl;
				goto L123;
			}
		}                  // check deviations

	}                     // ii=1,ii_cycles

L123:; // escape if loop is closed

	if ( ! get_loop_fold_with_dunbrack() ) monte_carlo_reset();
	if ( local_verbose ) std::cout << "end-pdb:" <<
	 ' ' << I( 5, loop_begin ) <<
	 ' ' << I( 5, loop_end ) <<
	 ' ' << I( 5, cutpoint ) <<
	 ' ' << I( 5, ii ) <<
	 ' ' << fmt::F( 12, 6, forward_deviation ) <<
	 ' ' << fmt::F( 12, 6, backward_deviation ) <<
	 ' ' << start_file << std::endl;

//	dump_phil_pdb();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin calculate_ccd_angle
///
/// @brief
///
/// @detailed
///
/// @param   F - [in/out]? - F,M are arrays (3,n)
/// @param   M - [in/out]? - F,M are arrays (3,n)
/// @param   n_atoms - [in/out]? - number of points
/// @param   pos - [in/out]? - the position whose torsion angle is begin varied
/// @param   torsion - [in/out]? - torsion is 1 for phi, 2 for psi
/// @param   direction - [in/out]? - 1 for n2c, -1 for c2n
/// @param   angle - [in/out]? -
/// @param   dev - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// double precision everywhere:: F,M,angle and dev
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calculate_ccd_angle(
	FArray2Da_double F,
	FArray2Da_double M,
	int const n_atoms,
	int const pos,
	int const torsion,
	int const direction,
	double & angle,
	double & dev
)
{
	using numeric::conversions::radians;
	using numeric::conversions::degrees;

	F.dimension( 3, n_atoms );
	M.dimension( 3, n_atoms );

// !!!!!!!!!! DOUBLE PRECISION !!!!!!!!!!!!!!!!!!!
// doubled letters are scalars: aa,bb,cc,dd
	FArray1D_double axis_vector( 3 );
	FArray1D_double r_hat( 3 );
	FArray1D_double s_hat( 3 );
	FArray1D_double AM( 3 );
	FArray1D_double A( 3 );
	FArray1D_double v( 3 );
	FArray1D_double O( 3 );
	FArray1D_double r( 3 );
	FArray1D_double f_vector( 3 );
	double dd, r_length, f_length, alpha;

	int const n2c = { 1 };
	int const c2n = { -1 };

	bool const local_debug = { false };

	get_torsion_axis( pos, torsion, A, axis_vector );

// accumulate the components of the ccd equation
	double aa = 0.0;
	double bb = 0.0;
	double cc = 0.0;

	for ( int i = 1; i <= n_atoms; ++i ) { // 1,2,3: N,CA,C
		// project M onto the axis ==> O
		Dsubvec( M(1,i), A, AM ); // AM = M-A
		dd = Ddotprod( AM, axis_vector );
		for ( int j = 1; j <= 3; ++j ) {
			v(j) = dd * axis_vector(j);
		}
		vector_sum( A, v, O );

		// calculate r
		Dsubvec( M(1,i), O, r );
		r_length = std::sqrt( Ddotprod( r,r));
		// calculate r_hat
		Dunitvec(r,r_hat);

		// calculate s_hat; orientation choice comes here!!
		Dcros(axis_vector, r_hat, s_hat);

		// calculate f_vector
		Dsubvec(F(1,i), O, f_vector);
		f_length = std::sqrt(Ddotprod(f_vector,f_vector));

		aa += r_length * r_length + f_length * f_length;

		bb += 2 * r_length * Ddotprod(f_vector,r_hat);

		cc += 2 * r_length * Ddotprod(f_vector,s_hat);

		if ( local_debug ) {
			Dassert_equal(
			 Ddotprod( M(1,i), axis_vector ),
			 Ddotprod( O, axis_vector ), 1 );

			Dassert_equal(0.0,Ddotprod(r,axis_vector),2);

			// some checks on orthonormality
			Dassert_equal(0.0,Ddotprod(axis_vector,r_hat),3);
			Dassert_equal(0.0,Ddotprod(axis_vector,s_hat),4);
			Dassert_equal(0.0,Ddotprod(r_hat,s_hat),5);

			Dassert_equal(1.0,Ddotprod(axis_vector,axis_vector),6);
			Dassert_equal(1.0,Ddotprod(s_hat,s_hat),7);
			Dassert_equal(1.0,Ddotprod(r_hat,r_hat),8);
		}

	}                     // i=1,n

	alpha = std::atan2( cc, bb );

	dev = aa - bb * std::cos( alpha ) - cc * std::sin( alpha );

	if ( local_debug ) {
		double const one_degree = radians( 1.0 ); // one degree expressed in radians
		if ( ( dev > aa - bb * std::cos( alpha - one_degree ) - cc * std::sin( alpha - one_degree ) )
		 || ( dev > aa - bb * std::cos( alpha + one_degree ) - cc * std::sin( alpha + one_degree ) ) ) {
			std::cout << "ccd problemo!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	if ( direction == n2c ) {
		angle = degrees(alpha);
	} else if ( direction == c2n ) {
		angle = -degrees(alpha);
	} else {
		std::cout << "unrecognized direction:" << SS( direction ) << std::endl;
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin Dassert_equal
///
/// @brief
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  tag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Dassert_equal(
	double const a,
	double const b,
	int const tag
)
{
	double const threshold = { 0.1 };

	if ( std::abs( a - b ) > threshold ) {
		std::cout << "assertion failed" << SS( tag ) << SS( a ) << SS( b ) <<
		 SS( std::abs( a - b ) ) << std::endl;
		jumping_abort();
	}
}


//////////////////////////////////////////////////////////////////////////////
/// @begin get_overlap_pos
///
/// @brief
///
/// @detailed
///
/// @param  pos - [in/out]? -
/// @param  direction - [in/out]? -
/// @param[out]  M -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// when I get a chance to test this, it should replace prev routine
///
/// cutpoint is always the residue before the break
///
/// note double precision on M!!
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_overlap_pos(
	int const cutpoint,
	int const dir,
	FArray2Da_double M // output
)
{
	using namespace misc;
	assert( !refold_check_current_pose() );

	M.dimension( 3, 3 );

	int const n2c = { 1 };
	//int const c2n = { -1 }; // fix stupid compiler warning

	assert( dir == n2c || dir == -1 ); // have to hardcode stupid c2n!

	// get the stub for folding
	int const stub_res ( cutpoint + 1 );
	FArray1D_float c_xyz( 3 ), n_xyz( 3 ), ca_xyz( 3 );
	build_stub( stub_res, dir, c_xyz, n_xyz, ca_xyz );

	// now build backbone at position seqpos:
	int const seqpos ( dir == n2c ? cutpoint+1 : cutpoint );
	FArray2D_float dummy_pos( 3, 5 );
	FArray1D_float dummy_cen( 3 );
	build_backbone(dir,c_xyz,n_xyz,ca_xyz, 1,
								 phi(seqpos), psi(seqpos), omega(seqpos), res(seqpos),
								 dummy_pos, dummy_cen, c_xyz,n_xyz,ca_xyz,
								 seqpos);

	for ( int i = 1; i <= 3; ++i ) {
		M(i,1) = dummy_pos(i,1);
		M(i,2) = dummy_pos(i,2);
		M(i,3) = dummy_pos(i,4);
	}
}
///////////////////////////////////////////////////////////////////////////////
void
get_overlap_pos_pose(
	pose_ns::Pose const & pose,
	FArray3DB_float const & Eposition,// dont get from pose -- dont want a refold
	int const cutpoint,
	int const dir,
	FArray2Da_double M // output
)
{
	//using namespace misc;

	static FArray2D_float overlap_xyz(3,6);

	build_overlap_coords( cutpoint, pose.psi( cutpoint ), pose.omega( cutpoint ),
		pose.phi( cutpoint+1 ), Eposition, overlap_xyz, true, dir );

	int const n2c(1);
	int const j_offset( dir == n2c ? 3 : 0 );
	for ( int k=1; k<=3; ++k ) {
		for ( int j=1; j<= 3; ++j ) {
			M(k,j) = overlap_xyz(k,j+j_offset);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void
get_overlap_pos_pose_old(
	pose_ns::Pose const & pose,
	int const cutpoint,
	int const dir,
	FArray2Da_double M // output
)
{
	using namespace misc;

	refold_set_current_pose( pose );

	M.dimension( 3, 3 );

	int const n2c = { 1 };
	//int const c2n = { -1 }; // fix stupid compiler warning

	assert( dir == n2c || dir == -1 ); // have to hardcode stupid c2n!

	// get the stub for folding
	int const stub_res ( cutpoint + 1 );
	FArray1D_float c_xyz( 3 ), n_xyz( 3 ), ca_xyz( 3 );
	build_stub_pose( pose, stub_res, dir, c_xyz, n_xyz, ca_xyz );

	// now build backbone at position seqpos:
	int const seqpos ( dir == n2c ? cutpoint+1 : cutpoint );
	FArray2D_float dummy_pos( 3, 5 );
	FArray1D_float dummy_cen( 3 );
	build_backbone(dir,c_xyz,n_xyz,ca_xyz, 1,
								 phi(seqpos), psi(seqpos), omega(seqpos), res(seqpos),
								 dummy_pos, dummy_cen, c_xyz,n_xyz,ca_xyz,
								 seqpos);
	refold_reset_current_pose();

	if ( pose.atom_tree() ) { ///////////////////////////////////////
		// HACK! have to modify a little since bond geometry not working
		FArray2D_float Mgl( 4, 4 );
		if ( dir == n2c ) {
			get_GL_matrix( Eposition( 1, 4, seqpos ),
										 Eposition( 1, 1, seqpos ),
										 Eposition( 1, 2, seqpos ),
										 dummy_pos( 1, 4 ),
										 dummy_pos( 1, 1 ),
										 dummy_pos( 1, 2 ),
										 Mgl );
		} else {
			get_GL_matrix( Eposition( 1, 1, seqpos ),
										 Eposition( 1, 4, seqpos ),
										 Eposition( 1, 2, seqpos ),
										 dummy_pos( 1, 1 ),
										 dummy_pos( 1, 4 ),
										 dummy_pos( 1, 2 ),
										 Mgl );
		}
		FArray2D_float m(3,3); // FLOAT not DOUBLE
		GL_rot( Mgl, Eposition( 1, 1, seqpos ), m(1,1) );
		GL_rot( Mgl, Eposition( 1, 2, seqpos ), m(1,2) );
		GL_rot( Mgl, Eposition( 1, 4, seqpos ), m(1,3) );
// 		{ // just debugging
// 			numeric::xyzVector_float m1( &m(1,1) ), m2( &m(1,2) ), m3( &m(1,3) ),
// 				d1( &(dummy_pos(1,1) ) ),
// 				d2( &(dummy_pos(1,2) ) ),
// 				d3( &(dummy_pos(1,4) ) );
// 			float const dev1
// 				( dir == n2c ? distance( m1,d1 ) : distance( m3,d3 ) );
// 			float const dev2
// 				( dir == n2c ?
// 					distance( (m2-m1).normalized(),(d2-d1).normalized()) :
// 					distance( (m2-m3).normalized(),(d2-d3).normalized()) );
// 			std::cout << "debug dev:" << dev1  << ' ' << dev2 << std::endl;
// 			assert( dev1 < 1e-2 && dev2 < 1e-2 );
// 		}

		for ( int i=1; i<= 3; ++i ) {
			for ( int j=1; j<= 3; ++j ) {
				M(j,i) = m(j,i);
			}
		}
	} else {
		for ( int i = 1; i <= 3; ++i ) {
			M(i,1) = dummy_pos(i,1);
			M(i,2) = dummy_pos(i,2);
			M(i,3) = dummy_pos(i,4);
		}
	}
}

//////////////////////////////// routine for use here -- no jumping stuff:
//////////////////////////////////////////////////////////////////////////////
/// @begin get_torsion_axis
///
/// @brief
///
/// @detailed
///
/// @param  pos - [in/out]? -
/// @param  torsion - [in/out]? -
/// @param  A - [in/out]? -
/// @param  axis_vector - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_torsion_axis(
	int const pos,
	int const torsion,
	FArray1Da_double A,
	FArray1Da_double axis_vector
)
{
	using namespace misc;

	A.dimension( 3 );
	axis_vector.dimension( 3 );

	FArray1D_double B( 3 );
	FArray1D_double BA( 3 );

	if ( torsion == 1 ) {    // phi => N to CA
		for ( int i = 1; i <= 3; ++i ) {
			A(i) = Eposition(i,2,pos); // CA
			B(i) = Eposition(i,1,pos); // N
		}
	} else if ( torsion == 2 ) { // psi => CA to C
		for ( int i = 1; i <= 3; ++i ) {
			A(i) = Eposition(i,4,pos); // C
			B(i) = Eposition(i,2,pos); // CA
		}
	} else {
		std::cout << "unrecognized torsion!!" << SS( torsion ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	Dsubvec(A,B,BA);
	Dunitvec(BA,axis_vector);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin test_jumping_loop_closure
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
//  void
//  test_jumping_loop_closure()
//  {
//  	using namespace jumping;
//  	using namespace misc;

//  	int cutpoint,begin_res,end_res,ccd_cycles;

//  	float forward_deviation,backward_deviation;
//  	bool rama_check, score_check, choose_fold_tree_fail;

//  	bool idealize_fail;

//  //$$$	retrieve_allow_insert(allow_insert,total_residue)
//  //$$$	for ( int i = 1; i <= total_residue; ++i ) {
//  //$$$		std::cout << "allow_insert:" << SS( i ) << SS( allow_insert(i) ) << std::endl;
//  //$$$	}

//  	rama_check = !truefalseoption("skip_rama");
//  	score_check = !truefalseoption("skip_score");
//  	intafteroption("ccd_cycles",100,ccd_cycles);

//  	refold(1,total_residue);
//  	set_jmp_chainbreak_overlap(1);
//  	set_jmp_chainbreak_weight(4.0);
//  	score = score4();
//  	monte_carlo_reset();
//  	new_fold_tree(); // ensure that derived data is up-to-date

//  	for ( int i = 1; i <= num_fold_tree_cutpoint; ++i ) {
//  		monte_carlo_reset();
//  		cutpoint = fold_tree_cutpoint(i);

//  		begin_res = cutpoint;
//  		while ( ! is_jump_point(begin_res) ) {
//  			--begin_res;
//  			if ( begin_res < 1 ) {
//  				std::cout << "this really shouldn't happen: begin_res<1" << std::endl;
//  				jumping_abort();
//  			}
//  		}
//  		++begin_res;

//  		end_res = cutpoint+1;
//  		while ( ! is_jump_point(end_res) ) {
//  			++end_res;
//  			if ( end_res > total_residue ) {
//  				std::cout << "this really shouldn't happen: end_res>tr" << std::endl;
//  				jumping_abort();
//  			}
//  		}
//  		--end_res;

//  		if ( false ) {
//  			std::cout << "sscut:" <<
//  			 ' ' << I( 4, begin_res ) <<
//  			 ' ' << I( 4, cutpoint ) <<
//  			 ' ' << I( 4, end_res ) <<
//  			 ' ' << I( 4, end_res-begin_res+1 ) << ' ';
//  			for ( int j = begin_res; j <= end_res; ++j ) {
//  				std::cout << secstruct(j);
//  			} std::cout << std::endl;

//  			std::cout << "sscut:" <<
//  			 ' ' << I( 4, begin_res ) <<
//  			 ' ' << I( 4, cutpoint ) <<
//  			 ' ' << I( 4, end_res ) <<
//  			 ' ' << I( 4, end_res-begin_res+1 ) << ' ';
//  			for ( int j = begin_res; j <= cutpoint; ++j ) {
//  				std::cout << secstruct(j);
//  			} std::cout << std::endl;
//  		}

//  //		if ( end_res - begin_res > 5 ) {
//  		scored_ccd_loop_closure(begin_res,end_res,cutpoint,ccd_cycles,score4,
//  		 forward_deviation,backward_deviation, score_check, rama_check);

//  	}                     // i=1,num_fold_tree_cutpoint

//  	monte_carlo_reset();
//  	set_jumping_flag(false); //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

//  	std::cout << "un-idealized:" << std::endl;
//  //	dump_phil_pdb();

//  	idealize_set_check_chainbreak(false);
//  	idealize(idealize_fail);
//  	idealize_set_check_chainbreak(true);

//  	std::cout << "idealized:" << SS( idealize_fail ) << std::endl;
//  //	dump_phil_pdb();
//  	set_jumping_flag(true); //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//  	num_jump = 0;
//  	choose_fold_tree(choose_fold_tree_fail); // ok with num_jump=0
//  	if ( choose_fold_tree_fail ) jumping_abort();
//  	new_fold_tree();
//  	refold(1,total_residue);
//  	std::cout << "refolded in jumping with no jumps:" << std::endl;
//  //	dump_phil_pdb();

//  	monte_carlo_reset();

//  }

//------------------------------------------------------------------------------
//
// tolerance in Angstroms, forward and backward splice RMS over N,CA,C must
// be less than tolerance for an early return. otherwise goes through the
// loop ii_cycles, each time both forward and backward
//
// returns the number of cycles actually taken

int
fast_ccd_loop_closure(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end,
	int const cutpoint,
	int const ii_cycles,
	float const tolerance,
	bool const rama_check,
	float const max_rama_score_increase,
	float const max_total_delta_helix,
	float const max_total_delta_strand,
	float const max_total_delta_loop,
	float & forward_deviation, // output
	float & backward_deviation, // output
	float & torsion_delta,
	float & rama_delta
)
{
	PROF_START( prof::CCD_CLOSE );
	pose.copy_to_misc();
	assert( misc_in_sync( pose ));

	/////////
	// params
	/////////
	bool const local_debug = { false };
	bool const local_verbose = { false };
	int const n2c = { 1 };
	int const c2n = { -1 };
	// per-cycle max move params
	double const max_angle_delta_helix = { 1.0 };
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	// for rama-checking with Boltzman criterion (should be higher?)
	float const ccd_temperature = { 0.25 };

	///////
	// POSE
	FArray1D_bool allow_bb_move( loop_end, false );
	for ( int i=loop_begin; i<= loop_end; ++i ) {
		allow_bb_move( i ) = pose.get_allow_bb_move( i );
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		int const pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	//////////////////////////////////////
	// save starting torsions, rama scores
	FArray1D_float start_phi(loop_end), start_psi(loop_end),
		start_rama_score(loop_end);
	float tmp1,tmp2;
	for ( int i=loop_begin; i<= loop_end; ++i ) {
		start_phi(i) = misc::phi(i);
		start_psi(i) = misc::psi(i);
		eval_rama_score_residue( misc::res(i), misc::phi(i), misc::psi(i),
										 misc::secstruct(i), start_rama_score(i), tmp1, tmp2 );
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	// for reporting how many cycles we actually took
	int total_cycles = ii_cycles;

	////////////////////////////
	// now start cycling through
	////////////////////////////
	for ( int ii = 1; ii <= ii_cycles; ++ii ) {
		// first forward, then backward
		for ( int repeat = 1; repeat <= 2; ++repeat ) {
			int direction,start_pos,stop_pos,increment;
			if ( repeat == 1 ) {
				direction = n2c;
				start_pos = loop_begin;
				stop_pos = std::min(loop_end,cutpoint);
				increment = 1;
			} else {
				direction = c2n;
				start_pos = loop_end;
				stop_pos = std::max(loop_begin,cutpoint+1);
				increment = -1;
			}

			// get fixed and moving positions for this direction:
			if ( direction == n2c ) {
				for ( int i = 1; i <= 3; ++i ) {
					F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
					F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
					F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
				}
			} else {
				for ( int i = 1; i <= 3; ++i ) {
					F(i,1) = misc::Eposition(i,1,cutpoint); // N
					F(i,2) = misc::Eposition(i,2,cutpoint); // CA
					F(i,3) = misc::Eposition(i,4,cutpoint); // C
				}
			}
			get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M );
			// as we change torsion angles through the loop the moving
			// atoms in M will be transformed along with the downstream segment
			// of the loop, by the routine refold_loop_torsions(...)

			for ( int pos = start_pos; pos*increment <= stop_pos*increment;
						pos += increment ) {

				if ( !allow_bb_move( pos ) ) continue;

				float max_angle_delta, max_total_delta;
				if ( misc::secstruct(pos) == 'H' ) {
					max_angle_delta = max_angle_delta_helix;
					max_total_delta = max_total_delta_helix;
				} else if ( misc::secstruct(pos) == 'E' ) {
					max_angle_delta = max_angle_delta_strand;
					max_total_delta = max_total_delta_strand;
				} else {
					max_angle_delta = max_angle_delta_loop;
					max_total_delta = max_total_delta_loop;
				}

				if ( max_total_delta <= 0.01 ) {
					//std::cout << "cant move this residue " << pos << std::endl;
					continue;
				}

				// save values for rama score eval, and in case we rama-reject
				float const starting_phi ( misc::phi(pos) );
				float const starting_psi ( misc::psi(pos) );

				// save in case we rama-reject!
				// re-use these index ranges later
				int const Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
				int const Epos_l_end   ( Epos_l_begin +
																 3 * param::MAX_POS * (loop_end-loop_begin+1) );
				assert( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
				for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
					save_Eposition[l] = misc::Eposition[l];
				}

				for ( int torsion = 1; torsion <= 2; ++torsion ) { // 1 is phi, 2 is psi

					double alpha,dev;
					calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
					double const alpha_orig( alpha );

					// impose per-move deltas
					if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
					if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

					// check for total movement during closure run:
					float const total_delta
						( ( torsion == 1 ) ? subtract_degree_angles( start_phi(pos),
																	 misc::phi(pos) + alpha )
							                 : subtract_degree_angles( start_psi(pos),
																	 misc::psi(pos) + alpha ) );

					if ( total_delta > max_total_delta ) {
						// this logic is a little tricky: if adding alpha to the previous
						// delta pushes us past 180 from start, then it wont work, so check for
						// that (note that if max_total_delta > 180 we wont even get here ):
						assert( alpha + max_total_delta < 180 );
						if ( alpha > 0 ) {
							alpha -= ( total_delta - max_total_delta + 0.01 );
						} else {
							alpha += ( total_delta - max_total_delta + 0.01 );
						}
					}

					// just rebuilds N,CA,C in Eposition
					if ( local_verbose )
						std::cout << "refold_loop_torsion: " << alpha << ' ' << pos <<
							' ' << torsion << ' ' << direction << ' ' << cutpoint << std::endl;

					refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
															 misc::Eposition, M );

					//dump_pdb("refold_loop_torsion");

					// update torsions in misc (NOT in pose yet, only when we accept!)
					if ( torsion == 1 ) {
						misc::phi(pos) += alpha;
						misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
					} else {
						misc::psi(pos) += alpha;
						misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
					}

					// debugging:
					if ( local_debug ) {
						// this assumes that the tree is not rooted at cutpoint or
						// cutpoint+1

						FArray3D_float tmp_Eposition( 3, param::MAX_POS, loop_end );
						for ( int i = loop_begin; i <= loop_end; ++i ) {
							for ( int j = 1; j <= param::MAX_POS; ++j ) {
								for ( int k = 1; k <= 3; ++k ) {
									tmp_Eposition(k,j,i) = misc::Eposition(k,j,i);
								}
							}
						}
						// copy torsions into pose:
						pose.set_phi( pos, misc::phi(pos) );
						pose.set_psi( pos, misc::psi(pos) );

						// refold the loop
						pose_to_misc( pose );
						//pose.refold();
						//dump_pdb("refold_pose");
						// undo torsion move --
						// the logic for rejecting moves ( see below ) assumes that we
						// didnt update pose torsions
						pose.set_phi( pos, starting_phi);
						pose.set_psi( pos, starting_psi);

						// calculate coordinate dev
						float coord_dev = 0.0;
						for ( int i = loop_begin; i <= loop_end; ++i ) {
							for ( int j = 1; j <= param::MAX_POS; ++j ) {
								if ( j == 1 || j == 2 || j == 4 ) {
									for ( int k = 1; k <= 3; ++k ) {
										coord_dev += std::abs( tmp_Eposition(k,j,i) -
																					 misc::Eposition(k,j,i) );
									}
								}
							}
						}

						if ( coord_dev > 0.1 ) {
							std::cout << "fold_loop_torsion dev::" << SS(coord_dev) <<
								std::endl;
							jumping_abort();
						}

						double tmp_dev = 0.0;
						for ( int i = 1; i <= 3; ++i ) {
							for ( int j = 1; j <= 3; ++j ) {
								tmp_dev += square( M(j,i) - F(j,i) );
							}
						}

						if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
							std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
								tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
						}

						if ( local_verbose )
							std::cout << "pos: " << I( 3, pos ) <<
								" alpha-orig: " << fmt::F( 7, 3, alpha_orig ) <<
								" alpha: " << fmt::F( 7, 3, alpha ) <<
								" dev1: " << fmt::F( 13, 9, dev ) <<
								" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

					} // if ( local_debug )
				}               // torsion = 1,2


				if ( rama_check ) {
					//////////////////////////////////////
					// evaluate the rama score difference:
					float old_rama_score,new_rama_score;
					eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
																	 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

					eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
																	 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

					if ( new_rama_score > old_rama_score ) {
						float const boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
						float const probability ( std::exp(std::max(-40.0f,boltz_factor) ) );
						if ( new_rama_score - start_rama_score( pos ) > max_rama_score_increase ||
								 ran3() >= probability ) {
							// undo the change:
							misc::phi(pos) = starting_phi;
							misc::psi(pos) = starting_psi;
							// copy saved Eposition
							for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
								misc::Eposition[l] = save_Eposition[l];
							}
							continue; // go to next position
						} // rama reject
					} // score got worse
				} // if ( rama_check )

				////////////////////////////////////////////////////
				// if we get here we have accepted the phi/psi move:
				////////////////////////////////////////////////////
				pose.set_phi(pos,misc::phi(pos));
				pose.set_psi(pos,misc::psi(pos));
			} // pos = start_pos,stop_pos
		} // repeat = 1,2   1=n2c; 2=c2n

		if ( mod(ii,5) == 0 || ii == ii_cycles ) {
			// check overlap deviations to see if loop is closed
			// every 5 cycles or on the last one.
			//
			// forward_deviation:
			int direction = n2c;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
			get_overlap_pos_pose( pose, misc::Eposition, cutpoint,direction,M);
			forward_deviation = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					forward_deviation += square( M(j,i) - F(j,i) );
				}
			}
			forward_deviation = sqrt( forward_deviation / 3 );
			// backward_deviation:
			direction = c2n;
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
			get_overlap_pos_pose( pose, misc::Eposition, cutpoint,direction,M);
			backward_deviation = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					backward_deviation += square( M(j,i) - F(j,i) );
				}
			}
			backward_deviation = sqrt( backward_deviation / 3 );
			if ( forward_deviation < tolerance && backward_deviation < tolerance ) {
				if ( local_verbose )
					std::cout << "closed early: ii= " << ii << ' ' << tolerance
										<< std::endl;
				total_cycles = ii;
				break;
			}
		}                  // check deviations
	}                     // ii=1,ii_cycles

	// calculate torsion dev, rama delta
	torsion_delta = 0.0;
	rama_delta = 0.0;

	for ( int i=loop_begin; i<= loop_end; ++i ) {
		torsion_delta += std::abs( subtract_degree_angles( start_phi(i), misc::phi(i) ));
		torsion_delta += std::abs( subtract_degree_angles( start_psi(i), misc::psi(i) ));
		float final_rama_score;
		eval_rama_score_residue( misc::res(i), misc::phi(i), misc::psi(i),
														 misc::secstruct(i), final_rama_score, tmp1, tmp2 );
		rama_delta += ( final_rama_score - start_rama_score(i) );
	}
	torsion_delta /= ( loop_end - loop_begin + 1);
	rama_delta /= ( loop_end - loop_begin + 1);

	PROF_STOP( prof::CCD_CLOSE );
	return total_cycles;
}
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// this routine is used by ccd loop closure
// it doesn't update centroid or all of Eposition
// just N,CA,C and the moving array M

// operates on Epos array that is passed in

void
refold_loop_torsion(
	float const alpha,
	int const pos,
	int const torsion,
	int const dir,
	int const cutpoint,
	FArray3DB_float & Epos,
	FArray2Da_double M)
{
	using numeric::conversions::radians;

	M.dimension(3,3); // the moving atoms at the end of the loop

	int const atom_order[3] = {1,2,4}; // N,CA,C

	assert( ( dir == 1 || dir == -1 ) &&
					 ( torsion   == 1 || torsion   ==  2 ) );

	int fold_end;
	if (dir == 1) {
		fold_end = cutpoint;
	} else {
		fold_end = cutpoint+1;
	}

	// local variables
	FArray1D_double A(3),B(3),vec(3), off(3); // axis atoms; dummy; transform offset
	FArray2D_double mat(3,3); // transform matrix

	// get the rotation matrix, offset
	double const theta = radians( static_cast< double >( -1 * dir * alpha ) );

	int a_atom(0), b_atom(0);
	if ( torsion == 1 ) {    // phi => N to CA
		a_atom = 2; // CA
		b_atom = 1; // N
	} else if ( torsion == 2) {
		a_atom = 4; // C
		b_atom = 2; // CA
	} else {
		std::cout << "undefined torsion type in refold_loop_torsion!!!"
							<< SS(torsion) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int k = 1; k <= 3; ++k ) {
		A(k) = Epos(k,a_atom,pos);
		B(k) = Epos(k,b_atom,pos);
	}

	getrot( A,B,theta,mat,off);

	// transform at pos
	int atom = 0;
	if ( dir == 1 && torsion == 1) {
		atom = 4;
	} else if ( dir == -1 && torsion == 2 ) {
		atom = 1;
	}
	if ( atom > 0 ) {
		for ( int k = 1; k <= 3; ++k ) {
			vec(k) = Epos(k,atom,pos);
		}
		mover( vec, mat, off );
		for ( int k = 1; k <= 3; ++k ) {
			Epos(k,atom,pos) = vec(k);
		}
	}

	// transform N,CA,C from pos+dir to fold_end
	for ( int i = pos + dir; i != fold_end + dir; i += dir ) {
		for ( int j = 0; j < 3; ++j ) {
			atom = atom_order[j]; // C-array!!
			for ( int k = 1; k <= 3; ++k ) {
				vec(k) = Epos(k,atom,i);
			}
			mover( vec, mat, off );
			for ( int k = 1; k <= 3; ++k ) {
				Epos(k,atom,i) = vec(k);
			}
		}
	}

	// move the overlap pos
	mover( M(1,1), mat, off );
	mover( M(1,2), mat, off );
	mover( M(1,3), mat, off );
}

//------------------------------------------------------------------------------
//--------------------------------------------------------------------------------
float
calc_chainbreak(
	FArray3Da_float Epos,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega
)
{
	using namespace param; // MAX_POS
	using namespace refold_ns;

	// dimension the args
	Epos.dimension(3,MAX_POS,2);
	phi.dimension(2);
	psi.dimension(2);
	omega.dimension(2);

	// local vars for refolding
	FArray2D_float X(3,3), M(3,3);

	// build a fake N for position 2 by folding forward:
	FArray1D_float pseudo_N(3);

	subvec(Epos(1,ca,1),Epos(1,n,1),X(1,ph));
	subvec(Epos(1,c,1),Epos(1,ca,1),X(1,ps));

	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom(M,Epos(1,c,1),cT(n,n2c),sT(n,n2c),psi(1),D(n,n2c),
						 pseudo_N,X(1,om));

	// build a fake C for position 1 by folding backward:
	FArray1D_float pseudo_C(3);

	subvec(Epos(1,n,2),Epos(1,ca,2),X(1,ph));
	subvec(Epos(1,ca,2),Epos(1,c,2),X(1,ps));

	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,Epos(1,n,2),cT(c,c2n),sT(c,c2n),phi(2),D(c,c2n),
						 pseudo_C,X(1,om));

	// calc score
	float d1 = sqrt( ( Epos(1,n,2) - pseudo_N(1) ) * ( Epos(1,n,2) - pseudo_N(1) ) +
									 ( Epos(2,n,2) - pseudo_N(2) ) * ( Epos(2,n,2) - pseudo_N(2) ) +
									 ( Epos(3,n,2) - pseudo_N(3) ) * ( Epos(3,n,2) - pseudo_N(3) ) );

	float d2 = sqrt( ( Epos(1,c,1) - pseudo_C(1) ) * ( Epos(1,c,1) - pseudo_C(1) ) +
									 ( Epos(2,c,1) - pseudo_C(2) ) * ( Epos(2,c,1) - pseudo_C(2) ) +
									 ( Epos(3,c,1) - pseudo_C(3) ) * ( Epos(3,c,1) - pseudo_C(3) ) );

	return (d1+d2)/2;
}

/////////////////////////////////////////////////////////////////////////////
void
ccd_moves(
	const int total_moves,
	pose_ns::Pose & pose,
	const int loop_begin,
	const int loop_end,
	const int cutpoint
)
{
	PROF_START( prof::CCD_MOVES );

	// debug:
	pose.copy_to_misc();
	assert( misc_in_sync( pose ) );

	/////////
	// params
	/////////
	bool const local_debug = { false };
	bool const local_verbose = { false };
	int const n2c = { 1 };
	int const c2n = { -1 };
	double const max_angle_delta_helix = { 1.0 }; // per-cycle max move params
	double const max_angle_delta_strand = { 5.0 };
	double const max_angle_delta_loop = { 10.0 };
	float const ccd_temperature = { 0.25 }; // for boltzmann rama-checking

	// which residues are insertable?
	// setup arrays for choosing residues at random weighted more toward loop
	// residues
	int total_insert;
	const FArray1D_int & insert_map ( pose.get_insert_map( total_insert ) );
	float const H_weight ( 0.5 );
	float const E_weight ( 1.0 );
	float const L_weight ( 8.5 );
	float total_weight(0.0);
	FArray1D_float weight_map ( total_insert );
	int loop_total_insert(0), first_insert(0);
	for ( int i=1,pos; i<= total_insert; ++i ) {
		pos = insert_map(i);
		if ( pos >= loop_begin && pos <= loop_end ) {
			++loop_total_insert;
			if ( first_insert == 0 ) {
				first_insert = i;
			}

			char ss ( pose.secstruct( pos ) );
			if ( ss == 'H' ) {
				total_weight += H_weight;
			} else if ( ss == 'E' ) {
				total_weight += E_weight;
			} else {
				assert ( ss == 'L' );
				total_weight += L_weight;
			}
			weight_map(loop_total_insert) = total_weight;
		}
	}
	if ( loop_total_insert <= 0 ) {
		std::cout << "no insertable residues in the loop!" << std::endl;
		return;
	}

	// array for undoing rama-rejected moves
	// static for speed... is this necessary?
	static FArray3D_float save_Eposition;
	if ( int( save_Eposition.size3() ) < loop_end ) {
		const int pad (20);
		save_Eposition.dimension(3, param::MAX_POS ,loop_end+pad);
	}

	// overlap pos arrays (note double precision)
	FArray2D_double M( 3, 3 );
	FArray2D_double F( 3, 3 );

	////////////////////////////
	// now start cycling through
	////////////////////////////

	int ntries(0);
	int nmoves(0);


// 	{ // debugging
// 		// get fixed and moving positions for this direction:
// 		std::cout << "ccd_moves: start fdev,bdev= " << fmt::I(4,cutpoint);
// 		for ( int d=1; d<= 2; ++d ) {
// 			int const direction( d==1 ? n2c : c2n );
// 			float dev(0.0);
// 			if ( direction == n2c ) {
// 				for ( int i = 1; i <= 3; ++i ) {
// 					F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
// 					F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
// 					F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
// 				}
// 			} else {
// 				for ( int i = 1; i <= 3; ++i ) {
// 					F(i,1) = misc::Eposition(i,1,cutpoint); // N
// 					F(i,2) = misc::Eposition(i,2,cutpoint); // CA
// 					F(i,3) = misc::Eposition(i,4,cutpoint); // C
// 				}
// 			}
// 			get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M);
// 			for ( int k=1; k<= 3; ++k ) {
// 				for ( int j=1; j<= 3; ++j ) {
// 					dev += square( F(k,j) - M(k,j) );
// 				}
// 			}
// 			std::cout << fmt::F(9,3,dev);
// 		}
// 		std::cout << std::endl;
// 	}


	while ( nmoves < total_moves ) {
		++ntries;
		if ( ntries > 5*total_moves ) {
// 			std::cout << "ccd_moves:: too many tries: " <<
// 				ntries << " nmoves: " << nmoves << " total_moves: " <<total_moves <<
// 				std::endl;
			break;
		}

		// choose a residue at random:
		const float weight ( ran3() * total_weight );
		int ipos;
		for ( ipos=1; ipos<= loop_total_insert; ++ipos ) {
				if ( weight <= weight_map(ipos) ) break;
			}
		if ( ipos > loop_total_insert ) {
			std::cout << "ccd_moves: bad weight? " << weight << ' ' <<
				total_weight << ' ' << weight_map( loop_total_insert ) << std::endl;
			ipos = loop_total_insert;
		}

		const int pos = insert_map( first_insert + ipos - 1 );
		const int direction ( pos <= cutpoint ? n2c : c2n );

		// get fixed and moving positions for this direction:
		if ( direction == n2c ) {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
				F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
			}
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				F(i,1) = misc::Eposition(i,1,cutpoint); // N
				F(i,2) = misc::Eposition(i,2,cutpoint); // CA
				F(i,3) = misc::Eposition(i,4,cutpoint); // C
			}
		}
		get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M);

// 		{ // debugging
// 			if ( direction == n2c ) {
// 				numeric::xyzVector_float m( M(1,1), M(2,1), M(3,1) ),
// 					e( &(misc::Eposition(1,4,cutpoint)));
// 				std::cout << "n2cdev: " << distance( m,e) << ' ' <<
// 					refold_ns::D(1,1) << std::endl;
// 				assert( std::abs( distance( m,e) - refold_ns::D(1,1))<1e-3);
// 			} else {
// 				numeric::xyzVector_float m( M(1,3), M(2,3), M(3,3) ),
// 					e( &(misc::Eposition(1,1,cutpoint+1)));

// 				std::cout << "c2ndev: " << distance( m,e) << ' ' <<
// 					refold_ns::D(1,1) << std::endl;
// 				assert( std::abs( distance( m,e) - refold_ns::D(1,1))<1e-3);
// 			}
// 		}


		// as we change torsion angles through the loop the moving
		// atoms in M will be transformed along with the downstream segment
		// of the loop, by the routine refold_loop_torsions(...)

		float max_angle_delta;
		if ( misc::secstruct(pos) == 'H' ) {
			max_angle_delta = max_angle_delta_helix;
		} else if ( misc::secstruct(pos) == 'E' ) {
			max_angle_delta = max_angle_delta_strand;
		} else {
			max_angle_delta = max_angle_delta_loop;
		}

		// save values for rama score eval, and in case we rama-reject
		const float starting_phi ( misc::phi(pos) );
		const float starting_psi ( misc::psi(pos) );

		// save in case we rama-reject!
		// re-use these index ranges later
		const int Epos_l_begin ( misc::Eposition.index(1,1,loop_begin) );
		const int Epos_l_end   ( Epos_l_begin +
														 3 * param::MAX_POS * (loop_end-loop_begin+1) );
		assert ( Epos_l_begin == int(save_Eposition.index(1,1,loop_begin)) );
		for ( int l= Epos_l_begin; l<Epos_l_end; ++l ) {
			save_Eposition[l] = misc::Eposition[l];
		}

		for ( int torsion = 1; torsion <= 2; ++torsion ) { // 1 is phi, 2 is psi

			double alpha,dev;
			calculate_ccd_angle(F,M,3,pos,torsion,direction,alpha,dev);
			double const alpha_orig( alpha );

			// impose per-move deltas
			if ( alpha >  max_angle_delta ) alpha =  max_angle_delta;
			if ( alpha < -max_angle_delta ) alpha = -max_angle_delta;

			// just rebuilds N,CA,C in Eposition
			refold_loop_torsion( alpha, pos, torsion, direction, cutpoint,
													 misc::Eposition, M );

			// update torsions in misc (NOT in pose yet, only when we accept!)
			if ( torsion == 1 ) {
				misc::phi(pos) += alpha;
				misc::phi(pos) = periodic_range( misc::phi(pos),360.0);
			} else {
				misc::psi(pos) += alpha;
				misc::psi(pos) = periodic_range( misc::psi(pos),360.0);
			}

			// debugging:
			if ( local_debug ) {
				// this assumes that the tree is not rooted at cutpoint or
				// cutpoint+1

				FArray3D_float tmp_Eposition(3,param::MAX_POS,loop_end);
				for ( int i=loop_begin; i<=loop_end; i++ ) {
					for ( int j=1; j<= param::MAX_POS; j++ ) {
						for ( int k=1; k<= 3; k++ ) {
							tmp_Eposition(k,j,i) = misc::Eposition(k,j,i);
						}
					}
				}
				// copy torsions into pose:
				pose.set_phi( pos, misc::phi(pos));
				pose.set_psi( pos, misc::psi(pos));

				// refold the loop
				//pose.refold();
				pose_to_misc( pose );
				//dump_pdb("refold_pose");
				// undo torsion move --
				// the logic for rejecting moves ( see below ) assumes that we
				// didnt update pose torsions
				pose.set_phi( pos, starting_phi);
				pose.set_psi( pos, starting_psi);

				// calculate coordinate dev
				float dev = 0.0;
				for ( int i=loop_begin; i<=loop_end; i++ ) {
					for ( int j=1; j<= param::MAX_POS; j++ ) {
						if ( j==1 || j==2 || j == 4 ) {
							for ( int k=1; k<= 3; k++ ) {
								dev+= std::abs( tmp_Eposition(k,j,i) - misc::Eposition(k,j,i) );
							}
						}
					}
				}
				std::cout << "DEBUG:: fold_loop_torsion dev= " << dev << std::endl;
				if (dev > 0.1) {
					std::cout << "fold_loop_torsion dev::" << SS(dev) << std::endl;
					jumping_abort();
				}
			} // if ( local_debug )

			if ( local_verbose ) {
				double tmp_dev = 0.0;
				for ( int i = 1; i <= 3; ++i ) {
					for ( int j = 1; j <= 3; ++j ) {
						tmp_dev += square( M(j,i) - F(j,i) );
					}
				}
				std::cout << "pos: " << I( 3, pos ) <<
					" alpha: " << fmt::F( 7, 3, alpha ) <<
					" dev1: " << fmt::F( 13, 9, dev ) <<
					" dev2: " << fmt::F( 13, 9, tmp_dev ) << std::endl;

				if ( alpha == alpha_orig && std::abs( dev - tmp_dev) > 0.1 ) {
					std::cout << "WARNING:: ccd-dev error: " << dev << ' ' <<
						tmp_dev << ' ' << std::abs( dev-tmp_dev) << std::endl;
				}



			}
		}               // torsion = 1,2


		if ( true ) { // always rama-checking
			float tmp1,tmp2;
			//////////////////////////////////////
			// evaluate the rama score difference:
			float old_rama_score,new_rama_score;
			eval_rama_score_residue( misc::res(pos), starting_phi, starting_psi,
															 misc::secstruct(pos), old_rama_score, tmp1, tmp2 );

			eval_rama_score_residue( misc::res(pos), misc::phi(pos), misc::psi(pos),
															 misc::secstruct(pos), new_rama_score, tmp1, tmp2 );

			if ( new_rama_score > old_rama_score ) {
				const float boltz_factor ( (old_rama_score-new_rama_score)/ccd_temperature );
				const float probability ( std::exp(max(-40.0f,boltz_factor) ) );
				if ( ran3() >= probability ) {
					// undo the change:
					misc::phi(pos) = starting_phi;
					misc::psi(pos) = starting_psi;
					// copy saved Eposition
					for ( int l= Epos_l_begin; l< Epos_l_end; ++l ) {
						misc::Eposition[l] = save_Eposition[l];
					}
					continue; // go to next position
				} // rama reject
			} // score got worse
		} // if ( rama_check )

		////////////////////////////////////////////////////
		// if we get here we have accepted the phi/psi move:
		////////////////////////////////////////////////////
		//std::cout << "ccd_move accepted " << pos << std::endl;
		pose.set_phi(pos,misc::phi(pos));
		pose.set_psi(pos,misc::psi(pos));
		++nmoves;
	}
// 	{ // debugging
// 		// get fixed and moving positions for this direction:
// 		std::cout << "ccd_moves:   end fdev,bdev= " << fmt::I(4,cutpoint);
// 		for ( int d=1; d<= 2; ++d ) {
// 			int const direction( d==1 ? n2c : c2n );
// 			float dev(0.0);
// 			if ( direction == n2c ) {
// 				for ( int i = 1; i <= 3; ++i ) {
// 					F(i,1) = misc::Eposition(i,1,cutpoint+1); // N
// 					F(i,2) = misc::Eposition(i,2,cutpoint+1); // CA
// 					F(i,3) = misc::Eposition(i,4,cutpoint+1); // C
// 				}
// 			} else {
// 				for ( int i = 1; i <= 3; ++i ) {
// 					F(i,1) = misc::Eposition(i,1,cutpoint); // N
// 					F(i,2) = misc::Eposition(i,2,cutpoint); // CA
// 					F(i,3) = misc::Eposition(i,4,cutpoint); // C
// 				}
// 			}
// 			get_overlap_pos_pose( pose, misc::Eposition, cutpoint, direction, M);
// 			for ( int k=1; k<= 3; ++k ) {
// 				for ( int j=1; j<= 3; ++j ) {
// 					dev += square( F(k,j) - M(k,j) );
// 				}
// 			}
// 			std::cout << fmt::F(9,3,dev);
// 		}
// 		std::cout << std::endl;
// 	}
	PROF_STOP( prof::CCD_MOVES );
}

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
// A couple more routines that use that fancy CCD closure above
// to create a more guaranteed (?) wobble.
//
// An apology: these aren't totally pose-centric,
//  but they do the job. I might move them to another routine
//  after I write more wobble stuff with the pose.
//
// rhiju 09/29/06
/////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
bool
get_wobble_CCD()
{
	static bool wobble_CCD( false );
	static bool init( false );

	if ( !init ) {
		wobble_CCD = truefalseoption("wobble_CCD");
		init = true;
	}
	return wobble_CCD;
}

/////////////////////////////////////////////////////////////////
// Some local options
/////////////////////////////////////////////////////////////////
int get_ccd_cycles()               {static int   x; intafteroption ( "ccd_cycles", 200, x);return x;}
float get_ccd_tol()                {static float x; realafteroption( "ccd_tol", 0.001, x);return x;}
float get_max_rama_score_increase(){static float x; realafteroption( "max_rama_score_increase", 2.0, x); return x;}
/////////////////////////////////////////////////////////////////

void
close_loop_at_cutpoint(
	int const cutpoint,
	pose_ns::Pose & ccd_pose,
	int splint_begin,
	int splint_end,
	bool & closed_loop,
	float & deviation_score,
	float & rama_delta
)
{
  static int   ccd_cycles = get_ccd_cycles(); // num of cycles of ccd_moves
  static float ccd_tol = get_ccd_tol(); // criterion for a closed loop
  bool  const rama_check = { true };
  static float max_rama_score_increase = get_max_rama_score_increase();
  //  float max_rama_score_increase = { 2.0 }; //
  float const max_total_delta_helix = { 10.0 }; // max overall angle changes for a helical residue
	float const max_total_delta_strand = { 30.0 }; // ... for a residue in strand
  float const max_total_delta_loop = { 75.0 }; // ... for a residue in loop
  // output for ccd_closure
  float forward_deviation, backward_deviation; // actually loop closure msd, both dirs
  float torsion_delta; // actually torsion and rama score changes, averaged by loop_size

  int loop_begin = std::max(splint_begin, 1);
  int loop_end =   std::min(splint_end, misc::total_residue);
  int actual_cycles;

  closed_loop = false;

  actual_cycles = fast_ccd_loop_closure( ccd_pose, loop_begin, loop_end, cutpoint, ccd_cycles,
					 ccd_tol, rama_check, max_rama_score_increase, max_total_delta_helix,
					 max_total_delta_strand, max_total_delta_loop, forward_deviation,
					 backward_deviation, torsion_delta, rama_delta );

	//	std::cout << "Attempted CCD closure at " << cutpoint << ", with residues at " << loop_begin << "-" << loop_end << ".  ccd_cycles " << ccd_cycles << "   forward_deviation " << forward_deviation << std::endl;

  if (forward_deviation <= ccd_tol){
    closed_loop = true;
  }

	deviation_score = (forward_deviation + backward_deviation)/ccd_tol - 2.0; // total guess!!!

}
//////////////////////////////////////////////////////////////////////////////
void
move_segment_after_cutpoint_back( int const cutpoint, int const dir )
{
	using namespace misc;

  int invariant_start, invariant_end;
  int const n2c(1); //, c2n(-1);

	//Move residues back to "best" position.
	// The result is a protein conformation with part of the residues
	// in one conformation perturbed by a fragment insertion, then a chainbreak,
	// then the rest of the protein in the original unperturbed conformation.
  if (dir == n2c){
    invariant_start = cutpoint + 1;
    invariant_end = misc::total_residue;
  } else {
    invariant_start = 1;
    invariant_end = cutpoint;
  }

	copy_coordinates( invariant_start, invariant_end, get_fullatom_flag(), best_position, position, best_centroid, centroid, best_full_coord, full_coord);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Important notes:
// (1) Only works with simple poses!! For now, at least.
// (2) Assumes that a prior routine has propagated the fragment insertion through the
//     fragment insertion region to the wobble region.
///////////////////////////////////////////////////////////////////////////////
void
add_fast_wobble_CCD(
	int & frag_begin,
	int size,
	int nwobble,
	int wobble_gap,
	float & rama_score,
	float & msd_score,
	bool & gfrag,
	int & fold_begin,
	int & fold_end
)
{
	using namespace misc;
	using namespace param;

	int dir;

	gfrag = true;

	//////////////////////////////////////////////
	//	Force nwobble to be at least 2          //
	//////////////////////////////////////////////
	//	if ( nwobble < 1 ) return;
	if ( nwobble < 2) nwobble = 2;

	if ( nwobble + wobble_gap < 0 ) return; // wobble must include frag end
																					// or be outside frag
	//	refold(frag_begin,frag_begin+size-1); // should this be here?
																				// safe, but not efficient
	refold_get_dir(dir);
	int splint_begin,splint_end;
	int cutpoint;
	int splint_length = nwobble;

	if ( dir == 1 ) {
		splint_begin = frag_begin + size + wobble_gap;
		splint_end = splint_begin + splint_length - 1;
		fold_begin = frag_begin;
		fold_end = splint_end;

		cutpoint = splint_begin-1;

	} else {
		splint_end = frag_begin - 1 - wobble_gap;
		splint_begin = splint_end - splint_length + 1;
		fold_begin = splint_begin - 1;
		fold_end = frag_begin + size - 1;

		cutpoint = splint_end;
	}

	// *********************************************************************************
	//  Important assumption: refold has already taken place
	//  to propagate prior fragment insertion through relevant segment!!!
	// *********************************************************************************
	//	refold(fold_begin, fold_end);
	//	refold_get_dir(dir);
	move_segment_after_cutpoint_back( cutpoint, dir);

  bool const ideal_pos( false );
  bool const coords_init( true );
	float rama_delta (0.0), deviation_score (0.0);
	bool closed_loop = false;
	pose_ns::Pose pose;
	pose_from_misc( pose, get_fullatom_flag(), ideal_pos, coords_init );

	close_loop_at_cutpoint( cutpoint, pose, fold_begin, fold_end, closed_loop, deviation_score, rama_delta );
	//	close_loop_at_cutpoint( cutpoint, pose, splint_begin, splint_end, closed_loop, rama_delta );

	// for some reason I don't fully understand this doesn't work or matter.
	// better to leave misc as is.
	//	pose.copy_to_misc();
	// 	easydump("blah2.pdb"); // diagnostic, this is in make_pdb.cc

	refold(fold_begin, fold_end);

	//hey, do i need a scaling to match the output from add_fast_wobble?
	msd_score = deviation_score; // hmm, this is CCD deviation divided by tolerance.
	rama_score = rama_delta;
}

//----------------------------------THE----------------------------------------
//----------------------------------END----------------------------------------
