// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15362 $
//  $Date: 2007-06-07 23:25:06 -0400 (Thu, 07 Jun 2007) $
//  $Author: andre $


// Rosetta Headers
#include "jumping_refold.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "current_pose.h"
#include "fullatom.h"
#include "jumping_ns.h"
#include "jumping_diagnostics.h"
#include "jumping_util.h"
#include "jumping_minimize.h"
#include "pose.h"
#include "jumping_pairings.h"
#include "maps.h"
//#include "make_pdb.h" // debugging only
#include "misc.h"
#include "namespace_best_position_init_common.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "pdb.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "minimize_chi.h"
#include "refold_ns.h"
#include "rotamer_functions.h"
#include "start.h"
#include "symmetric_design.h"
#include "termini.h"
#include "util_vector.h"
#include "util_basic.h"

#include "misc_removal.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstddef>
#include <cstdlib>
#include <iostream>

// Namespaces
namespace track_refold {
	FArray1D_bool folded;
	FArray1D_bool moved;
	FArray1D_bool full_coord_ok;
}

// DANGER DANGER DANGER:: using pose_ns
using namespace pose_ns;


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

void
jmp_refold_tree(
	Pose const & pose,
	bool const full_coord_init,
	FArray1D_bool const & new_bb,
	FArray1D_bool const & new_chi,
	FArray1D_bool const & new_jump
)
{
	bool const verbose ( jumping::jverbose );

	if ( verbose ) {
		std::cout << "jmp_refold_tree::\n";
		std::cout << pose.fold_tree();
	}
	if ( full_coord_init ) assert( pose.fullatom() ); // sanity
	assert( pose.fold_tree().check_fold_tree() );

	// fold_tree pointers:
	Fold_tree const & fold_tree ( pose.fold_tree() );
	Fold_tree::const_iterator it;
	Fold_tree::const_iterator const it_begin ( fold_tree.begin() );
	Fold_tree::const_iterator const it_end ( fold_tree.end() );

	int const total_residue ( pose.total_residue() );

	// setup arrays for monitoring folding
	if ( int( track_refold::folded.size1() ) != total_residue ) {
		track_refold::folded.dimension( total_residue );
		track_refold::moved.dimension( total_residue );
		track_refold::full_coord_ok.dimension( total_residue );
	}
	track_refold::folded = false; // array assignment
	track_refold::moved = false;
	track_refold::full_coord_ok = false;

	// fold the first residue: -----------------------------------------
	// this either copies from best or rebuilds the backone depending on
	// whether the first residue of the first edge of the tree is
	// contained in the fragment

	int const fold_start ( it_begin->start );

	track_refold::moved( fold_start ) = new_bb( fold_start );
	fold_first_residue( full_coord_init, fold_start,
											track_refold::moved(fold_start) );
	track_refold::folded( fold_start ) = true;

	// now fold the edges of the tree in order //////////////////////
	for ( it = it_begin; it != it_end; ++it ) {
		fold_edge( pose, pose.fullatom() && full_coord_init,
							 it->start, it->stop, it->label, new_bb, new_jump );
	}

	//dump_pdb("test_cen");

	if ( pose.fullatom() ) {
		// fills in places that were built from scratch
		// track_refold::full_coord_ok records the positions
		// that have already been built
		jumping_build_fullcoord( pose, full_coord_init,
														 track_refold::full_coord_ok);

		// refold new chi onto misc::full_coord
		FArray2D_float const & chi ( pose.chi() );
		FArray1D_bool update_chi( param::MAX_CHI, true );
		for ( int i=1; i<= total_residue; ++i ) {
			if ( new_chi(i) ) {
				int const aa  ( pose.res         (i));
				int const aav ( pose.res_variant (i));
				refold_new_chi_residue( misc::full_coord(1,1,i), aa, aav,
																chi(1,i), update_chi );
			}
		}
	}

	// debug
	for ( int i = 1; i <= fold_tree.get_nres(); ++i ) {
		assert( track_refold::folded(i) );
	}
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void
jumping_build_fullcoord(
	Pose const & pose,
	bool const full_coord_init,
	FArray1D_bool const & already_done
)
{
	//using namespace misc;

	int const total_residue( pose.total_residue() );

	if ( !pose.fullatom() ) return;

	if ( !full_coord_init ) {
		// this used to be handled by setup_full_coord
		// copy backbone from misc::Eposition to misc::full_coord
		// build NH, HA
		initialize_fullcoord_array( misc::Eposition, misc::full_coord,
			misc::total_residue, misc::res, misc::res_variant );

		// fix nh positions at chainbreaks
		FArray1D_bool const & is_cutpoint ( pose.fold_tree().get_is_cutpoint() );
		for ( int i=2; i<= total_residue; ++i ) {
			if ( is_cutpoint(i-1) ) {
				build_nh_simple( misc::res(i), misc::res_variant(i), misc::phi(i),
												 misc::full_coord(1,1,i) );
			}
		}

		// put icoor sidechains at all positions
		// we may replace these later, but what the heck? why not
		FArray2D_float const & chi ( pose.chi() );
		FArray2D_float rotcoord( 3, param::MAX_ATOM()() );
		for ( int i=1; i<= total_residue; ++i ) {
			int const aa( misc::res(i) );
			int const aav( misc::res_variant(i) );
			get_rot_coord( misc::full_coord, aa, aav, chi(1,i), rotcoord, i );
			assert( rotcoord.index(1,1) == 0 );
			for( int l = 0, offset = misc::full_coord.index(1,1,i),
						 l_end = 3 * param::MAX_ATOM()(); l < l_end; ++l ) {
				misc::full_coord[l+offset] = rotcoord[l];
			}
		}

		// dont need to do the stuff below:
		return;
	}

	// the function put_one_full_coord_nh goes back and redoes NH positions
	// that might be gotten wrong by the current setup -- either the
	// previous full_coord position hadn't been setup yet, or it's at a
	// chainbreak...

	for ( int i=1; i<= total_residue; ++i ) {
		if ( !already_done(i) ) {

			build_fullcoord(misc::Eposition,i,i, misc::res,
					misc::res_variant,termini_ns::is_N_terminus,
					termini_ns::is_C_terminus,misc::phi(i),misc::best_full_coord,
				misc::full_coord);

			put_one_full_coord_nh(i);
		}
	}
}

//------------------------------------------------------------------------------
// fold the first residue: --------------------------------------
// this either copies from best or rebuilds the backone depending on
// whether the first residue of the first edge of the tree is
// contained in the fragment
//
// it will also build the N,CA (i+1) and  C(i-1) stub atoms
// unless there are chain breaks
void
fold_first_residue(
	const bool fullatom,
	int const fold_start,
	bool const res_moved // is fold_start contained in fragment insert?
)
{
	using namespace aaproperties_pack;
	//using namespace jumping;
	//using namespace misc;
	using namespace refold_ns;
	//using namespace start;

// local
	FArray1D_float c_xyz( 3 );
	FArray1D_float n_xyz( 3 );
	FArray1D_float ca_xyz( 3 );

	assert( refold_check_current_pose() );
// 	assert( !use_st_bond() );
	if ( res_moved ) { // dont use the best_position coordinates
		if ( !Epos_init_zero( misc::Ebest_position(1,1,fold_start ) ) ) {
			// translates C-alpha to origin
			build_default_stub( fold_start, c_xyz, n_xyz, ca_xyz );
		} else {
			// try to preserve the location and orientation
			// of the fold_start residue. Specifically: preserves
			// the C-alpha position, the N-CA vector and the N-CA-C plane
			rebuild_default_stub( fold_start, c_xyz, n_xyz, ca_xyz );
		}
		build_backbone(n2c,c_xyz,n_xyz,ca_xyz, 1,
			misc::phi(fold_start), misc::psi(fold_start), misc::omega(fold_start),
			misc::res(fold_start),
			misc::Eposition(1,1,fold_start), misc::centroid(1,fold_start),
			c_xyz,n_xyz,ca_xyz,
			fold_start);

	} else {             // fold_start not contained in fragment
		copyXYZ(1,5,misc::Ebest_position(1,1,fold_start),
						misc::Eposition(1,1,fold_start));
		copyXYZ(1,1,misc::best_centroid(1,fold_start),
						misc::centroid(1,fold_start));
		if ( fullatom ) {
			int const natoms( aaproperties_pack::natoms( misc::res(fold_start),
				misc::res_variant(fold_start)));
			copyXYZ( 1, natoms,
							 misc::best_full_coord(1,1,fold_start),
							 misc::full_coord(1,1,fold_start));
			track_refold::full_coord_ok( fold_start ) = true;
		}
	}

	if ( false ) {
		static int counter(0);

		// debug the rebuild_default_stub
		FArray3D_float tmp_Eposition( misc::Eposition );
		FArray2D_float tmp_centroid( misc::centroid );

		rebuild_default_stub( fold_start, c_xyz, n_xyz, ca_xyz );

		build_backbone(n2c,c_xyz,n_xyz,ca_xyz, 1,
			misc::phi(fold_start), misc::psi(fold_start), misc::omega(fold_start),
			misc::res(fold_start),
			tmp_Eposition(1,1,fold_start), tmp_centroid(1,fold_start),
			c_xyz,n_xyz,ca_xyz,
			fold_start);

		float dev(0.0);
		for ( int j=1; j<= 5; ++j ) {
			if ( j==3 ) continue;
			for ( int k=1; k<=3; ++k ) {
				dev += std::abs( tmp_Eposition  (k,j,fold_start) -
												 misc::Eposition(k,j,fold_start) );
			}
		}
		++counter;
		if ( dev > 1.0e-3 || counter%1000==0 ) {
			std::cout << "[DEBUG] rebuild_default_stub: dev= " << dev << std::endl;
		}
	}
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// this routine builds coords for a single edge of the fold_tree
// it is assumed that coords for estart already exist in Eposition

void
fold_edge(
	Pose const & pose,
	bool const fullatom,
	int const estart,
	int const estop,
	int const elabel,
	FArray1D_bool const & new_bb,
	FArray1D_bool const & new_jump
)
{
	bool const verbose ( jumping::jverbose );

	assert( estart != estop );
	//using namespace aaproperties_pack;
	//using namespace jumping;
	//using namespace refold_ns;
	//using namespace misc;

	// some params:
	int const n2c(1), c2n(-1), n(1), ca(2), c(4);
	bool const ideal_backbone( pose.ideal_backbone() );

	// debug: make sure we've generated coords for start residue
	assert( track_refold::folded(estart) );

	if ( verbose ) std::cout << "fold_edge: " << estart << ' ' << estop << ' ' << elabel << ' ' <<
									 track_refold::moved(estart ) << std::endl;

	if ( elabel > 0 ) {
		////////////// fold a jump //////////////////////////////////////////////
		int const jump_number (elabel);

		if ( ! track_refold::moved(estart) && !new_bb(estop) && !new_jump(jump_number) ) {
			// copy coords
			if ( verbose ) std::cout << "jump:copy\n";
			copy_coordinates(estop,estop,fullatom,
											 misc::best_position,misc::position,
											 misc::best_centroid,misc::centroid,
											 misc::best_full_coord,misc::full_coord);

			track_refold::moved(estop) = false;
			track_refold::folded(estop) = true;
			track_refold::full_coord_ok(estop) = true;
		} else {
			if ( verbose ) std::cout << "jump:fold\n";
			int const dir ( estart < estop ? n2c : c2n );
			const Jump & jump
				( pose.get_jump( jump_number ));

			jump.make_jump( dir, misc::Eposition(1,1,estart), misc::Eposition(1,1,estop) );

			// make_jump takes the bond geometry from estart and copies it to
			// estop. update geometry will restore correct geometry,
			// preserving: C-alpha location, N->C-alpha vector, and N,CA,C plane
			update_geometry( estop );

			track_refold::moved(estop) = true;
			track_refold::folded(estop) = true;
		}
	} else {
		//////////// fold a peptide edge /////////////////////////////////////
		// three stages:
		// 1. copy/transform from best anything between estart and fold_start
		// 2. fold from fold_start to fold_stop
		// 3. transform anything between fold_stop and estop (including estop)

		assert( elabel == -1 );
		if ( estart == estop ) return; // nothing to do

		int const dir ( estart< estop ? n2c : c2n );

		int fold_start ( estart+dir), fold_stop(estop);

		// little tricky: dont want to go beyond 1,total_residue:
		while ( fold_start != estop+dir && ! new_bb( fold_start) ) fold_start += dir;
		while ( fold_stop != estart && ! new_bb( fold_stop) ) fold_stop -= dir;

		if ( new_bb( estart ) ) {
			fold_start = estart+dir; // think about this?
			if ( fold_stop == estart ) fold_stop = estart+dir;
		}

		bool const folding ( fold_start != estop+dir );

		// guarantee Ebest_position OK for stub coords
		if ( folding && fold_stop != estop ) fold_stop += dir;


		if ( folding )
			if ( verbose )
				std::cout << "fold_start: " << fold_start << " fold_stop: " << fold_stop << ' ' <<
					dir << std::endl;

		if ( false ) {
			// extend the folded region by 1 residue on either side,
			// so that any superimposing of stubs will be done with
			// consistent bond angles
			//
			// this isnt the best way to handle this -- would rather
			// have bond-geometry moves update new_bb_refold properly
			//
			// anyhow, not doing much use_current_bond() stuff anyhow...
			// this will probable be replaced with the pose_bonds stuff
			//
			if ( fold_start != estart+dir ) fold_start -= dir;
			if ( fold_stop != estop ) fold_stop += dir;
		}

		if ( folding ) assert( dir * (fold_stop - fold_start) >= 0 );

//  		if ( begin_res > std::max(estop,estart+dir) ||
//  				 end_res   < std::min(estop,estart+dir) || begin_res > end_res) {
//  			folding = false;
//  			fold_start = estop + dir; // past the segment end, so we copy/transform everything
//  			fold_stop = estop; // not used
//  		} else {
//  			folding = true;
//  			if ( dir == n2c ) {
//  				fold_start = std::min(estop,std::max(estart+dir,begin_res));
//  				fold_stop = std::min(estop,end_res); //couldnt be less than estart+1
//  			} else {
//  				fold_start = std::max(estop,std::min(estart+dir,end_res));
//  				fold_stop = std::max(estop,begin_res); //couldnt be greater than estart-1
//  			}
//  		}

		// re-used variables:
		FArray2D_float Mgl( 4, 4 );
		FArray1D_float c_xyz( 3 );
		FArray1D_float n_xyz( 3 );
		FArray1D_float ca_xyz( 3 );


		///// Stage 1: Transform or Copy before folding /////////////////////////////////
		if ( ! folding || fold_start != estart + dir ) {
			assert( folding || fold_start == estop + dir );
			if ( ! track_refold::moved(estart) ) {
				//// copy
				for ( int i = estart+dir; i*dir < fold_start*dir; i += dir ) {
					assert( !new_bb( i ) );
					copyXYZ(1,5,misc::Ebest_position(1,1,i),misc::Eposition(1,1,i));
					copyXYZ(1,1,misc::best_centroid(1,i),misc::centroid(1,i));
					if ( fullatom ) {
						copyXYZ(1, aaproperties_pack::natoms(misc::res(i),misc::res_variant(i)),
										misc::best_full_coord(1,1,i), misc::full_coord(1,1,i));
						track_refold::full_coord_ok(i) = true;
					}
					track_refold::moved(i) = false;
					track_refold::folded(i) = true;
				}
			} else {
				//// transform
				// stub atoms formed by N,CA of stub_res; C of stub_res -1
				int const stub_res ( dir==n2c ? estart+1 : estart );

				build_stub( stub_res, dir, c_xyz, n_xyz, ca_xyz );

				assert( !new_bb( stub_res ) && !new_bb( stub_res-1) );
				get_GL_matrix( misc::Ebest_position(1,c ,stub_res-1), // where we are
											 misc::Ebest_position(1,n ,stub_res),
											 misc::Ebest_position(1,ca,stub_res),
											 c_xyz, n_xyz, ca_xyz, // where we want  to go
											 Mgl);

				// transform best coords by Mgl into current array
				for ( int i = estart+dir;	i*dir < fold_start*dir;	i += dir ) {
					assert( !new_bb( i ) );
					GL_rotate(5,1,Mgl,misc::Ebest_position(1,1,i), misc::Eposition(1,1,i));
					GL_rotate(1,1,Mgl,misc::best_centroid(1,i), misc::centroid(1,i));
					if ( fullatom ) {
						GL_rotate(aaproperties_pack::natoms(misc::res(i),
																								misc::res_variant(i)), 1,Mgl,
											misc::best_full_coord(1,1,i),misc::full_coord(1,1,i));
						track_refold::full_coord_ok(i) = true;
					}
					track_refold::folded(i) = true;
					track_refold::moved(i) = true;
				}
			}               // has estart moved?
		}                  // do we need a copy/transform segment?


		if ( folding ) {
			///////////////////////// Stage 2: folding //////////////////////////////////////
			int const fold_begin ( std::min(fold_start,fold_stop) );
			int const fold_length ( ( fold_stop - fold_start + dir ) * dir );

			assert( fold_length > 0 );
			int const stub_res ( dir==n2c ? fold_start : fold_start + 1);

			build_stub( stub_res, dir, c_xyz, n_xyz, ca_xyz );

			build_backbone(dir,c_xyz,n_xyz,ca_xyz, fold_length,
										 misc::phi(fold_begin), misc::psi(fold_begin), misc::omega(fold_begin),
										 misc::res(fold_begin),
										 misc::Eposition(1,1,fold_begin), misc::centroid(1,fold_begin),
										 c_xyz,n_xyz,ca_xyz,
										 fold_begin);

			if ( dir == c2n && !ideal_backbone ) {
				rebuild_misc_centroids_n2c( fold_begin, fold_length, c_xyz );
			}

			for ( int i = fold_start, iend = fold_stop * dir; i * dir <= iend; i += dir ) {
				track_refold::folded(i) = true;
				track_refold::moved(i) = true;
			}

			if ( fold_stop != estop ) {
				///// stage 3: transform on the other side of the folding segment ///////////////////
				int const stub_res  ( dir==n2c ? fold_stop+1 : fold_stop );

				assert( !new_bb( stub_res ) && !new_bb( stub_res-1) );

				get_GL_matrix( misc::Ebest_position(1,c ,stub_res-1),
											 misc::Ebest_position(1,n ,stub_res),
											 misc::Ebest_position(1,ca,stub_res), // where we are now
											 c_xyz,n_xyz,ca_xyz, // where we want to go
											 Mgl); // output transform matrix

				// rotate
				for ( int i = fold_stop + dir, iend = estop * dir; i * dir <= iend; i += dir ) {
					assert( !new_bb( i ) );
					GL_rotate(5,1,Mgl,misc::Ebest_position(1,1,i), misc::Eposition(1,1,i));
					GL_rotate(1,1,Mgl,misc::best_centroid(1,i), misc::centroid(1,i));
					if ( fullatom ) {
						GL_rotate(aaproperties_pack::natoms(misc::res(i),
																								misc::res_variant(i)), 1,Mgl,
											misc::best_full_coord(1,1,i),misc::full_coord(1,1,i));
						track_refold::full_coord_ok(i) = true;
					}
					track_refold::folded(i) = true;
					track_refold::moved(i) = true;
				}
			}           // is there something left to transform?
		}              // anything to fold
	}                     // are we a jump or a segment?
}


//------------------------------------------------------------------------------
// it assumes that misc::Eposition & bb-torsions are current, but this is
// through build_stub so it could be easily switched to depend
// on pose's coordinates
//
// new: this is only called by pose.get_overlap_Eposition which handles
//      dimensioning

void
jmp_update_overlap_positions(
	pose_ns::Pose const & pose,
	int const chainbreak_overlap,
	FArray4D_float & overlap_Eposition
)
{
	//using namespace jumping;
	//using namespace misc;
	using namespace param;
	using namespace refold_ns;

	FArray1D_float c_xyz( 3 );
	FArray1D_float n_xyz( 3 );
	FArray1D_float ca_xyz( 3 );
	FArray2D_float dummy_cen( 3, MAX_RES()() );

	// get the cutpoint info:
	int num_fold_tree_cutpoint;
	FArray1D_int const & fold_tree_cutpoint
		( pose.fold_tree().get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );

	assert( overlap_Eposition.size4() >= static_cast< std::size_t >( num_fold_tree_cutpoint ) );
	assert( overlap_Eposition.size3() >= static_cast< std::size_t >( 2 * chainbreak_overlap ) );

	refold_set_current_pose( pose );
	for ( int ncut = 1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
		int const cutpoint ( fold_tree_cutpoint(ncut) );
		if ( !pose.is_protein( cutpoint ) || !pose.is_protein( cutpoint+1 ) || pose.pseudo( cutpoint ) )
			continue;

		int overlap;

		// first fold forward:
		overlap = std::min( chainbreak_overlap, misc::total_residue-cutpoint);
		build_stub( cutpoint+1, n2c, c_xyz, n_xyz, ca_xyz );
		build_backbone(n2c,c_xyz,n_xyz,ca_xyz,overlap,
									 misc::phi   ( cutpoint+1 ),
									 misc::psi   ( cutpoint+1 ),
									 misc::omega ( cutpoint+1 ),
									 misc::res   ( cutpoint+1 ),
									 overlap_Eposition(1,1,1,ncut), dummy_cen,
									 c_xyz, n_xyz, ca_xyz,
									 cutpoint+1); // torsion array offset

		// now fold backward
		overlap = std::min( chainbreak_overlap, cutpoint );
		build_stub( cutpoint+1, c2n, c_xyz, n_xyz, ca_xyz );
		build_backbone(c2n,c_xyz,n_xyz,ca_xyz,overlap,
									 misc::phi   ( cutpoint-overlap+1 ),
									 misc::psi   ( cutpoint-overlap+1 ),
									 misc::omega ( cutpoint-overlap+1 ),
									 misc::res   ( cutpoint-overlap+1 ),
									 overlap_Eposition(1,1,1-overlap,ncut), dummy_cen,
									 c_xyz,n_xyz,ca_xyz,
									 cutpoint-overlap+1);
	} // ncut
	refold_reset_current_pose();
}


//////////////////////////////////////////////////////////////////////////////
// called after repacking the pose

// void
// update_misc_full_coord(
// 	Pose const & pose
// )
// {
// 	if ( !pose.fullatom() ) {
// 		std::cout << "update_misc_full_coord:: shouldnt be here\n";
// 		return;
// 	}

// 	int const nres ( pose.total_residue() );
// 	FArray3D_float const & fcoord ( pose.full_coord() );
// 	FArray3D_float & dest_fcoord( misc::full_coord );

// 	assert( nres == misc::total_residue );
// 	assert( fcoord.index(1,1,1) == dest_fcoord.index(1,1,1) &&
// 					fcoord.size1() == dest_fcoord.size1() &&
// 					fcoord.size2() == dest_fcoord.size2() );
// 	for ( int l = fcoord.index(1,1,1),
// 					l_end = fcoord.index(1,1,1) + nres * fcoord.size1() * fcoord.size2();
// 				l < l_end; ++l ) {
// 		dest_fcoord[l] = fcoord[l];
// 	}
// }


//////////////////////////////////////////////////////////////////////////////
void
copy_coordinates_pose(
	bool const fullatom,
	int const nres,
	FArray3DB_float const & src_Epos,
	FArray2DB_float const & src_cen,
	FArray3DB_float const & src_fcoord,
	FArray3DB_float & dest_Epos,
	FArray2DB_float & dest_cen,
	FArray3DB_float & dest_fcoord
)
{
	// copy pose coords into dest
	assert( src_Epos.index(1,1,1) == dest_Epos.index(1,1,1) &&
					src_Epos.size1()      == dest_Epos.size1() &&
					src_Epos.size2()      == dest_Epos.size2() &&
					src_cen.index(1,1)    == dest_cen.index(1,1) &&
					src_cen.size1()       == dest_cen.size1() );

	// Eposition
	for ( int l = src_Epos.index(1,1,1), l_end = src_Epos.index(1,1,1) +
					nres * src_Epos.size1() * src_Epos.size2();
				l < l_end; ++l ) {
		dest_Epos[l] = src_Epos[l];
	}

 	// centroid
	for ( int l = src_cen.index(1,1), l_end = src_cen.index(1,1) +
					nres * src_cen.size1(); l< l_end; ++l ) {
		dest_cen[l] = src_cen[l];
	}

	// full_coord
	if ( fullatom ) {
		assert( src_fcoord.index(1,1,1) == dest_fcoord.index(1,1,1) &&
						src_fcoord.size1()      == dest_fcoord.size1() &&
						src_fcoord.size2()      == dest_fcoord.size2() );
		for ( int l = src_fcoord.index(1,1,1), l_end = src_fcoord.index(1,1,1) +
						nres * src_fcoord.size1() * src_fcoord.size2(); l < l_end; ++l ) {
			dest_fcoord[l] = src_fcoord[l];
		}
	}
}

// @brief
// copy pose information into global namespaces, which is in fact misc.h
// and pdb.h
// @authors Phil Bradley
// @revised Monica Berrondo
void
pose_to_misc(
	pose_ns::Pose const & pose
)
{
	//pose_update_MAX_RES( pose );

	copy_pose_torsions_to_misc( pose );

	FArray3D_float dummy_full_coord;

	copy_coordinates_pose( pose.fullatom(), pose.total_residue(),
		pose.Eposition(), pose.centroid(),
	  ( pose.fullatom() ? pose.full_coord() : dummy_full_coord ),
		misc::Eposition, misc::centroid, misc::full_coord );

	//// yab: misc removal
	// see function definition above, ensure initialization of certain new classes
	// Please DO NOT add anything here without consulting yab, as this
	// functions in this code block are scheduled for removal.
	misc_removal::init_modules_TEMPORARY();
	//// yab: misc removal

}

//
// this routine can trigger a call to update_sequence()

void
copy_pose_torsions_to_misc(
	pose_ns::Pose const & pose
)
{
	pose_update_MAX_RES( pose );

	int const nres( pose.total_residue() );

	set_fullatom_flag_simple( pose.fullatom() );

	bool sequence_mismatch ( misc::total_residue != nres );
	misc::total_residue = nres;
	// copy pose torsions into misc arrays

	for ( int i=1; i<= nres; ++i ) {
		misc::phi(i) = pose.phi(i);
		misc::psi(i) = pose.psi(i);
		misc::omega(i) = pose.omega(i);
		misc::secstruct(i) = pose.secstruct(i);
		misc::name(i) = pose.name(i);
		sequence_mismatch = ( sequence_mismatch || misc::res(i) != pose.res(i) ||
													misc::res_variant(i) != pose.res_variant(i) );
		misc::res(i) = pose.res(i);
		misc::res_variant(i) = pose.res_variant(i);
	}

	if ( sequence_mismatch ) update_sequence( false );

  // fill in PDB information
  pose.pdb_info().pdb_info_to_global();

}

//////////////////////////////////////////////////////////////////////////////
// hack for proof-of-concept refactoring demo
//

bool
misc_in_sync(
	Pose const & pose
)
{
	bool const verbose( false );

	// this DEBUGGING routine checks to see whether the misc torsions
	// are still synced with the passed pose
	// returns TRUE if synced, FALSE if out of sync
	int const nres ( pose.total_residue() );
	if ( misc::total_residue != nres ) {
		std::cout << "total_residue mismatch " << misc::total_residue << ' ' <<
			nres << std::endl;
		return false;
	}

	float const max_coord_dev ( 0.1 );
	float const max_angle_dev ( 0.1 );
//  	float const max_angle_dev ( 1.0 );

	bool sync ( true );
	float max_dev ( 0.0 );

	// check torsions
	for ( int i=1; i<= nres; ++i ) {
		if ( !pose.is_protein(i) ) continue;
		float const dev
			( std::abs( subtract_degree_angles( misc::phi   (i), pose.phi   (i) )) +
				std::abs( subtract_degree_angles( misc::psi   (i), pose.psi   (i) )) +
				std::abs( subtract_degree_angles( misc::omega (i), pose.omega (i) )) );
		max_dev = std::max( dev, max_dev );

		sync = sync &&
			misc::secstruct   (i) == pose.secstruct   (i) &&
			misc::res         (i) == pose.res         (i) &&
			misc::res_variant (i) == pose.res_variant (i);

		if ( verbose || max_dev >= max_angle_dev || !sync ) {
			std::cout << I(4,i) <<
				misc::secstruct(i) << ' ' << pose.secstruct(i) << ' ' <<
				misc::res(i) << ' ' << misc::res_variant(i) << ' ' <<
				pose.res(i)  << ' ' <<  pose.res_variant(i) << ' ' <<
				F(9,3,misc::phi(i)) << F(9,3,misc::psi(i)) << F(9,3,misc::omega(i)) <<
				F(9,3, pose.phi(i)) << F(9,3, pose.psi(i)) << F(9,3, pose.omega(i)) <<
				' ' << dev << ' ' << sync << std::endl;
		}

	}
	sync = sync && ( max_dev < max_angle_dev );
	//std::cout << "misc_in_sync:: torsion_dev: " << max_dev << ' ' << nres << std::endl;

	///////////////
	// check coords

	// Eposition
	max_dev = 0.0;
	FArray3D_float const & Epos ( pose.Eposition() );
	FArray2D_float const & cen ( pose.centroid() );
	assert( Epos.index(1,1,1) == misc::Eposition.index(1,1,1) &&
					 Epos.size1() == misc::Eposition.size1() &&
					 Epos.size2() == misc::Eposition.size2() );
	for ( int l = Epos.index(1,1,1),
					l_end = Epos.index(1,1,1) + nres * Epos.size1() * Epos.size2();
				l < l_end; ++l ) {
		max_dev = std::max( max_dev, std::abs( Epos[l] - misc::Eposition[l] ) );
	}
	if ( verbose || max_coord_dev<max_dev)
		std::cout <<"Eposition: " << max_dev << std::endl;

	// centroid
	assert(cen.index(1,1) == misc::centroid.index(1,1));
	for ( int l=cen.index(1,1), l_end = cen.index(1,1) + cen.size1()*nres;
				l<l_end; ++l ) {
		max_dev = std::max( max_dev, std::abs( cen[l] - misc::centroid[l] ) );
	}
	if ( verbose || max_coord_dev<max_dev )
		std::cout <<"centroid: " << max_dev << std::endl;

	// full_coord
	if ( pose.fullatom() ) {
		using aaproperties_pack::nheavyatoms;
		FArray3D_float const & fcoord( pose.full_coord() );
		assert( fcoord.index(1,1,1) == misc::full_coord.index(1,1,1) &&
						fcoord.size1() == misc::full_coord.size1() &&
						fcoord.size2() == misc::full_coord.size2() );
		for ( int i=1; i<= nres; ++i ) {
			int const aa ( pose.res(i) );
			int const aav( pose.res_variant(i) );
			for ( int j=1, j_end=nheavyatoms(aa,aav); j<= j_end; ++j ) {
				for ( int k=1; k<=3; ++k ) {
					max_dev = std::max( max_dev, std::abs( fcoord(k,j,i) -
																								 misc::full_coord(k,j,i)));
				}
				if ( max_dev > max_coord_dev ) {
					std::cout << "misc_in_sync: full_coord_dev: " << i << ' ' << aa <<
						aav << ' ' << j << ' ' << max_dev << ' ' <<
						aaproperties_pack::atom_name( j,aa,aav) <<
						' ' << param_aa::aa_name3(aa) << ' ' << std::endl;
				}
			} // j
		} // i
	}
	sync = sync && ( max_dev < max_coord_dev );
	if ( !sync ) {
		std::cout << "misc_in_sync:: coord_dev: " << max_dev << ' ' << nres <<
			std::endl;
// 		dump_phil_pdb();
// 		pose.dump_pdb( "pose_dump.pdb" );
	}
	assert(sync );
	return sync;
}


///////////////////////////////////////////////////////////////
void
rebuild_misc_centroids_n2c(
	int const begin,
	int const size,
	FArray1D_float const & c_xyz
)
{
	int const n ( refold_ns::n );
	int const ca( refold_ns::ca );
	int const c ( refold_ns::c );
	for ( int i=begin; i< begin+size; ++i ) {
		if ( i == begin ) {
			build_centroid_n2c( misc::res(i), misc::phi(i), c_xyz,
			misc::Eposition( 1, n, i ), misc::Eposition( 1, ca, i ),
			misc::centroid( 1, i ) );
		} else {
			build_centroid_n2c( misc::res(i), misc::phi(i),
			misc::Eposition( 1, c, i-1 ),
			misc::Eposition( 1, n, i ), misc::Eposition( 1, ca, i ),
			misc::centroid( 1, i ) );
		}
	}
}


//---------------------THE END-------------------------------------------------
