// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

// Rosetta Headers
#include "kin_atom_tree.h"
//#include "input_pdb.h" // dihedral!
#include "jumping_util.h"
#include "kin_util.h"
#include "pose.h"
#include "prof.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h" // update_nblist! should just make method?

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/FArray4D.h>
//#include <ObjexxFCL/formatted.io.h>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>


namespace kin {


  /////////////////////////////////////////////////////////////////////////////
	void
	Atom_tree::clear()
	{
		if ( root_ ) {
			root_->erase();
			delete root_;
			root_ = 0;
		}
		atom_pointer = 0;
	}

  /////////////////////////////////////////////////////////////////////////////
	//
 	void
 	Atom_tree::copy_torsions(
 		Atom_tree const & src
	)
 	{
		PROF_START( prof::ATOM_TREE_COPY_TORSIONS );
 		root_->copy_torsions( *(src.root()) );
		PROF_STOP ( prof::ATOM_TREE_COPY_TORSIONS );
 	}

  /////////////////////////////////////////////////////////////////////////////
	void
	Atom_tree::update_torsions(
		FArray3D_float const & full_coord,
		int const nres,
		FArray1D_int const & res,
		FArray1D_int const & res_variant
	)
	{
		Coords_FArray_const coords( full_coord );

		root_->update_torsions( coords, true /*recursive*/);

		// in case any of the sneaky offsets have changed... only
		// relevant for chi1 if folding c2n or phi/psi when folding
		// backward from the sidechain
		for ( int i=1; i<= nres; ++i ) {
			update_torsion_pointer( i, res(i), res_variant(i), coords );
		}
	}


  /////////////////////////////////////////////////////////////////////////////
	// call this after repacking, rotamer trials, etc
	//
	// DANGER: assumes that backbone atoms are completely fixed
	// also assumes standard backbone -> sidechain folding...
	// new_aa_and_coords is more general in this respect.
	//
	void
	Atom_tree::new_sidechain_coords(
		int const seqpos,
		int const aa,
		int const aav,
		FArray3D_float const & coords_in
	)
	{
		if ( aa == param_aa::aa_gly ) return;
		Coords_FArray_const coords( coords_in );
		int parent_atomno(0), root_atomno(0);

		if ( param_aa::is_protein(aa) ) {
			parent_atomno = 2; // C-alpha
			root_atomno   = 5; // C-beta
		} else if ( param_aa::is_DNA(aa) ) {
			parent_atomno = aaproperties_pack::na_anchor(1,aa,aav);
			root_atomno   = aaproperties_pack::na_anchor(2,aa,aav);
		} else {
			std::cout << "new_sidechain:: unsupported aa: " << aa << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		Atom* parent( atom_pointer( parent_atomno, seqpos ) );
		Atom* root  ( atom_pointer(   root_atomno, seqpos) );
		assert( root != 0 && parent != 0 );

		root->set_torsion_moved( true, true, false ); // for scoring
		parent->update_child_torsions( root, coords ); // will fail if not child
	}



  /////////////////////////////////////////////////////////////////////////////
	// builds a new amino acid to replace the old
	// uses a call to add_kin_atom
	//
	// assumes single connection into seqpos in current tree
	// temporary: assumes single connection out of seqpos-- easy to change
	//
	//
	//
	// allow-move arrays are left at false
	//
	// currently, this will fail if seqpos is the root residue of the
	// tree -- need to fix this

	void
	Atom_tree::new_aa_and_coords(
		int const seqpos,
		int const old_aa,
		int const old_aav,
		int const new_aa,
		int const new_aav,
		FArray3D_float const & coords_in
	)
	{
		using namespace aaproperties_pack;

		// wrapper for FArray
		Coords_FArray_const coords( coords_in );

		// first determine how the existing residue is "wired in" to the tree
		// only handle cases with one incoming connection
		int root_atomno(0);
		Atom* parent_atom(0), *old_root_atom(0);
		std::vector< std::pair< int, Atom* > > outgoing;
		{
			for ( int j=1; j<= natoms(old_aa,old_aav); ++j ) {
				Atom* a( atom_pointer(j,seqpos) );
				if ( a->parent == 0 || a->parent->atom_id.rsd() != seqpos ) {
					assert( !root_atomno ); // only one incoming connexn
					old_root_atom = a;
					root_atomno = j;
					parent_atom = a->parent;
				}
				for ( int k=0; k< a->nchildren(); ++k ) {
					if ( a->child(k)->atom_id.rsd() != seqpos ) {
						outgoing.push_back( std::make_pair( j, a->child(k) ) );
						//anchor_atomno = j;
						//child_atom = a->child(k);
					}
				}
			}
		}

		////////////////////////////////////////////
		// build a new atom tree for this amino acid
		int const n2c(1);
		FArray1D< Atom* > new_atom_ptr( param::MAX_ATOM()() );
		new_atom_ptr = 0;
		int const dir( old_root_atom->is_jump() ? dir_jump : n2c );

		// uses connectivity info in aaproperties_pack to build a tree:
		Atom* new_root_atom
			( add_kin_atom( parent_atom, new_aa, new_aav, seqpos, root_atomno,
											new_atom_ptr, dir ) );

		// ensure full score calculation -- if the backbone isn't changing
		// this will be overkill
		// also, this is prob already set by default inside add_kin_atom
		new_root_atom->set_torsion_moved( true, true, true );


		/////////////////////
		// rewire connections
		if ( parent_atom ) {
			parent_atom->replace_atom( old_root_atom, new_root_atom );
		} else {
			// this residue is the root of the fold_tree
			if ( old_root_atom != root_ ) {
				std::cout << "kin_atom_tree::new_aa_and_coords: parent=0 but not " <<
					"my root! " << old_root_atom->atom_id << ' ' <<
					root_->atom_id << std::endl;
				std::exit( EXIT_FAILURE );
			}
			root_ = new_root_atom;
		}

		for ( unsigned i=0; i< outgoing.size(); ++i ) {
			int const anchor_atomno( outgoing[i].first );
			Atom* const child_atom( outgoing[i].second );
			Atom* const new_anchor_atom( new_atom_ptr( anchor_atomno ) );
			assert( new_anchor_atom != 0 );
			new_anchor_atom->insert_atom( child_atom );
		}

		//// at this point, the new aa is wired into the tree

		// delete old atoms
		for ( int j=1; j<= natoms( old_aa, old_aav ); ++j ) {
			delete atom_pointer(j,seqpos);
			atom_pointer(j,seqpos) = 0;
		}

		// update atom_pointer, recalculate torsions for new atoms and nbrs
		for ( int j=1; j<= natoms( new_aa, new_aav ); ++j ) {
			new_atom_ptr(j)->update_torsions( coords, false /*recursive*/ );
			atom_pointer(j,seqpos) = new_atom_ptr(j);
		}

		//// following calls are unnecessary if backbone of new aa is unmoved

		// update torsions of child atoms:
		for ( unsigned i=0; i< outgoing.size(); ++i ) {
			Atom* const child_atom( outgoing[i].second );
			child_atom->update_torsions( coords, false /* recursive */ );
		}

		// update torsions of root atom's next sibling
		// unless we're the root of the tree
		if ( parent_atom ) {
			Atom* next_sibling( parent_atom->next_child( new_root_atom ) );
			if ( next_sibling ) {
				// unnec unless the position of root_atomno has changed, ie bb moved
				next_sibling->update_torsions( coords, false /* recursive */ );
			}
		}

		// update the torsion_pointer array
		update_torsion_pointer( seqpos, new_aa, new_aav, coords );
	}


  /////////////////////////////////////////////////////////////////////////////

	void
	Atom_tree::setup(
		pose_ns::Fold_tree const & fold_tree,
		FArray3D_float const & full_coord,
		FArray1D_int const & res,
		FArray1D_int const & res_variant
	)
	{
		using aaproperties_pack::natoms;
		using aaproperties_pack::nchi;
		using aaproperties_pack::atom_name;
		//		using namespace::param_torsion;

		int const nres( fold_tree.get_nres() );

		atom_pointer.dimension( param::MAX_ATOM()(), nres );
		torsion_pointer_.dimension( param::MAX_TORSIONS, nres );
		torsion_offset_. dimension( param::MAX_TORSIONS, nres );

		// build the tree
		root_ = build_tree( fold_tree, res, res_variant,
												atom_pointer );
		root_->set_allow_move( RB_TRANSLATION, false );
		root_->set_allow_move( RB_ROTATION, false );

		// debug the tree
		for ( int i=1; i<= nres; ++i ) {
			int const aa( res(i) );
			int const aav( res_variant(i) );
			for ( int j=1; j<= param::MAX_ATOM()(); ++j ) {
				assert( ( j <= natoms(aa,aav) && atom_pointer(j,i) != 0 ) ||
								( j >  natoms(aa,aav) && atom_pointer(j,i) == 0 ) );
				assert( atom_pointer(j,i) == 0 ||
								( atom_pointer(j,i)->atom_id.rsd() == i &&
									atom_pointer(j,i)->atom_id.atomno() == j ) );
			}
		}


		// fill in the torsion angles
		this->update_torsions( full_coord, nres, res, res_variant );

	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Atom_tree::get_rosetta_torsion(
																 int const torsion_number,
																 int const rsd
																 ) const
	{
		using numeric::conversions::degrees;
		float offset;
		Atom const * atom( torsion_pointer( torsion_number, rsd, offset ));
		if ( atom == 0 ) {
			//std::cout << "bad torsion: " << torsion_number << ' ' <<
			//	rsd << std::endl;
			return 0.0;
		}
		return periodic_range( degrees( atom->get_torsion( PHI ) + offset ),
													 360.0 );
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Atom_tree::get_rosetta_torsion_atom(
		int const torsion,
		int const rsd
	) const
	{
		assert( torsion <= param::MAX_TORSIONS );
		float offset;
		return torsion_pointer( torsion, rsd, offset );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Atom_tree::set_rosetta_torsion(
		int const torsion_number,
		int const rsd,
		float const value
	)
	{
		using numeric::conversions::radians;
		using numeric::constants::d::pi;
		float offset;
		bool chainbreak;
		Atom* atom( torsion_pointer( torsion_number, rsd, offset, chainbreak ));
		if ( atom == 0 ) {
			int const nres( torsion_pointer_.size2() );
			if ( torsion_number == 1 && rsd == 1 ||
					 torsion_number == 2 && rsd == nres ||
					 torsion_number == 3 && rsd == nres ) return;
			if ( chainbreak ) {
				//				std::cout << "Atom_tree::set_rosetta_torsion: chainbreak torsion: " <<
				//					torsion_number << ' ' << rsd << std::endl;
				return;
			}
			std::cout << "Atom_tree::set_rosetta_torsion: bad torsion: " <<
				torsion_number << ' ' << rsd << std::endl;
			return;

// It would be nice to put these back in to flag weird stuff.
// 			assert( false );
// 			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		atom->set_torsion( PHI, periodic_range( radians( value ) - offset, 2*pi) );
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom *
	Atom_tree::get_rosetta_torsion_atom(
		int const torsion,
		int const rsd
	)
	{
		assert( torsion <= param::MAX_TORSIONS );
		float offset;
		return torsion_pointer( torsion, rsd, offset );
	}

	/////////////////////////////////////////////////////////////////////////////
	// the sequence has changed at position seqpos
	//
	// so update the torsion_pointer array
	//
	//
	void
	Atom_tree::update_torsion_pointer(
		int const seqpos,
		int const aa,
		int const aav,
		Coords const & coords
	)
	{
		using param_aa::is_protein;
		using param_aa::is_DNA;
		using param_aa::is_RNA;
		int const max_bb_tor( is_protein( aa ) ? 3 :
													( is_DNA( aa ) || is_RNA( aa) ? 6 : 0 ));
		int const max_chi( is_protein( aa ) ? aaproperties_pack::nchi(aa,aav):
											 ( is_DNA( aa ) ? 1 : ( is_RNA( aa ) ? 4 : 0 )));

		for ( int j=1; j<= max_bb_tor+max_chi; ++j ) {
			torsion_pointer_(j,seqpos) = find_torsion_pointer( j, seqpos, aa, aav,
																		 torsion_offset_(j,seqpos), coords );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Atom_tree::torsion_pointer(
		int const torsion,
		int const seqpos,
		float & offset
	) const
	{
		offset = torsion_offset_(torsion,seqpos);
		Atom_id const & id( torsion_pointer_(torsion,seqpos) );
		if ( id == BOGUS_ATOM ) {
			return 0;
		} else {
			return this->atom( id );
		}
	}

	Atom *
	Atom_tree::torsion_pointer(
		int const torsion,
		int const seqpos,
		float & offset
	)
	{
		bool chainbreak;
		return torsion_pointer( torsion, seqpos, offset, chainbreak );
	}

	Atom *
	Atom_tree::torsion_pointer(
		int const torsion,
		int const seqpos,
		float & offset,
		bool & chainbreak
	)
	{
		offset = torsion_offset_(torsion,seqpos);
		chainbreak = false;
		Atom_id const & id( torsion_pointer_(torsion,seqpos) );

		if ( id == CHAINBREAK_BOGUS_ATOM ) {
			chainbreak = true;
			return 0;
		}

		if ( id == BOGUS_ATOM ) {
			return 0;
		} else {
			return this->atom( id );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// could cache the results -- would be faster but we'd have to remember
	// to recalculate things when the tree changes
	Atom_id	Atom_tree::find_torsion_pointer(
		int const j, //torsion,
		int const i, //seqpos,
		int const aa,
		int const aav,
		float & offset,
		kin::Coords const & coords // for calculating offsets!!!
	) const
	{
		using param_aa::is_protein;
		using param_aa::is_DNA;
		using param_aa::is_RNA;
		using aaproperties_pack::nchi;
		//		using param_torsion::total_torsion;
		using param_torsion::phi_torsion;
		using param_torsion::psi_torsion;
		using param_torsion::omega_torsion;
		using param_torsion::chi1_torsion;
		using numeric::constants::d::pi;
		using numeric::conversions::radians;

		int const nres( atom_pointer.size2() );

		if ( !is_protein(aa) && !is_DNA(aa) && !is_RNA(aa) ) {
			return BOGUS_ATOM;
		}

		Atom_id const N( 1, i ), C( 3, i);

		///////////////////////////////////
		// this is important, do not change

		offset = 0.0;


		///////////////////////////////////////////////////////////////////////////
		// setup information about which atoms define this torsion angle:


		int const max_bb_tor( is_protein( aa ) ? 3 :
													( is_DNA( aa ) || is_RNA( aa) ? 6 : 0 ));
		int const max_chi( is_protein( aa ) ? aaproperties_pack::nchi(aa,aav):
											 ( is_DNA( aa ) ? 1 : ( is_RNA( aa ) ? 4 : 0 )));
		int const total_torsion = max_bb_tor + max_chi;

		FArray2D_int torsion_atom( 4, total_torsion );
		FArray2D_int torsion_rsd_offset( 4, total_torsion );
		setup_torsion_atom_aa_aav( aa, aav, torsion_atom, torsion_rsd_offset );

		if ( j > max_bb_tor + max_chi ) {
			std::cout << "WARNING:: attempt to find torsion_pointer for bad torsion "
								<< j << ' ' << aa << ' ' << aav << std::endl;
			return BOGUS_ATOM;
		}

		std::vector< Atom_id > atom_ids;
		std::vector< Atom* > atoms;

		//std::cout << "Torsion: " << i << ' ' << j;
		for ( int k=1; k<=4; ++k ) {
			int const rsd( i + torsion_rsd_offset( k, j ) );
			int const atomno( torsion_atom( k, j ) );
			if ( rsd < 1 || rsd > nres ) break;
			Atom* ptr( atom_pointer( atomno, rsd ) );
			if ( ptr == 0 ) break;
			atoms.push_back( ptr );
			atom_ids.push_back( Atom_id( atomno, rsd ) );

			//				std::cout << ' ' << atom_name( atomno, res(rsd),
			//																			 res_variant(rsd) ) << ' ' << rsd;

		}
		//std::cout << std::endl;

		if ( atoms.size() != 4 ) return BOGUS_ATOM; // no match




		///////////////////////////////////////////////////////////////////////////
		// modify for chain breaks:

		if ( j == phi_torsion && is_protein( aa ) ) {

			// atoms are C(i-1) N(i) Ca(i) C(i)
			assert( atom_ids[0].atomno() == 3 && atom_ids[1].atomno() == 1 );
			if ( atoms[0]->parent != atoms[1] && atoms[1]->parent != atoms[0] ) {
				// chainbreak!
 				//std::cout << "Atom_tree::setup: chainbreak in tree detected: " <<
 				//	j << ' ' << i << std::endl;
				Atom * N_atom( atom_pointer( atom_ids[1].atomno(), atom_ids[1].rsd()));
				Atom const * tor_atom;
				if ( N_atom->is_jump() ) {
					// confirm we are at a cutpoint
					if ( !( N_atom->get_nonjump_atom(1) &&
									!N_atom->get_nonjump_atom(2) ) ) {
						std::cout << "WARNING:: cant find phi but not a chainbreak?" <<
							std::endl;
						return BOGUS_ATOM;
					}
					tor_atom = N_atom->get_nonjump_atom(1);
				} else {
					// confirm that this is a chainbreak
					if ( N_atom->nchildren() != 1 ) {
						std::cout << "WARNING:: cant find phi but not a chainbreak?" <<
							std::endl;
						return BOGUS_ATOM;
					}
					tor_atom = N_atom->child(0);
				}
				std::string const & tor_atom_name
					( aaproperties_pack::atom_name( tor_atom->atom_id.atomno(), aa,
																					aav ) );
				assert( tor_atom_name == " H  " || tor_atom_name == " CD " );
				atom_ids[0] = tor_atom->atom_id;
				atoms[0] = atom_pointer( atom_ids[0].atomno(), atom_ids[0].rsd() );
				offset = pi;
			}
		} else if ( j == psi_torsion && is_protein(aa) ) {
			// atoms are N(i) Ca(i) C(i) N(i+1)
			assert( atom_ids[2].atomno() == 3 && atom_ids[3].atomno() == 1 );
			if ( atoms[2]->parent != atoms[3] && atoms[3]->parent != atoms[2] ){
 				//std::cout << "Atom_tree::setup: chainbreak in tree detected: " <<
 				//	j << ' ' << i << std::endl;
				Atom * C_atom( atom_pointer( atom_ids[2].atomno(), atom_ids[2].rsd()));
				assert( !C_atom->is_jump() );
				if ( C_atom->nchildren() != 1 ) {
					std::cout << "WARNING:: cant find psi but not a chainbreak?" <<
						std::endl;
					return BOGUS_ATOM;
				}
				Atom const * tor_atom = C_atom->child(0);
				atom_ids[3] = tor_atom->atom_id;
				atoms[3] = atom_pointer( atom_ids[3].atomno(), atom_ids[3].rsd() );
				assert( atom_ids[3].atomno() == 4 ); // O
				offset = pi;
			}
		} else if ( j == 1 && is_RNA(aa) ) {
			// atoms are O3*(i-1) P(i) O5*(i) C5*(i)
			assert( atom_ids[0].atomno() == 9 && atom_ids[1].atomno() == 1 );
			if ( atoms[0]->parent != atoms[1] && atoms[1]->parent != atoms[0] ){
				// 				std::cout << "Atom_tree::setup: chainbreak in tree detected: " <<
				// 					j << ' ' << i << std::endl;
				Atom * P_atom( atom_pointer( atom_ids[1].atomno(), atom_ids[1].rsd()));
				assert( !P_atom->is_jump() );
				if ( P_atom->nchildren() != 2 ) {
					std::cout << "WARNING:: cant find alpha but not a chainbreak?" <<
						std::endl;
					return BOGUS_ATOM;
				}
				Atom const * tor_atom = P_atom->child(0);
				atom_ids[0] = tor_atom->atom_id;
				atoms[0] = atom_pointer( atom_ids[0].atomno(), atom_ids[0].rsd() );
				assert( atom_ids[0].atomno() == 2 ); // O1P
				offset = 2*pi / 3.0; //Hey is the sign right?
			}
		}

		// Torsions which really don't exist for a chainbreak: epsilon, zeta for RNA.
		// Also put in omega for protein?
		if ( j == 5 && is_RNA(aa) ) {
			// atoms are C4*(i) C3*(i) O3*(i) P(i+1)
			assert( atom_ids[2].atomno() == 9 && atom_ids[3].atomno() == 1 );
			if ( atoms[2]->parent != atoms[3] && atoms[3]->parent != atoms[2] ){
				return CHAINBREAK_BOGUS_ATOM;
			}
		} else if  ( j == 6 && is_RNA(aa) ) {
			// atoms are C3*(i) O3*(i) P(i+1) O5*(i+1)
			assert( atom_ids[1].atomno() == 9 && atom_ids[2].atomno() == 1 );
			if ( atoms[2]->parent != atoms[1] && atoms[1]->parent != atoms[2] ){
				return CHAINBREAK_BOGUS_ATOM;
			}
		}


		///////////////////////////////////////////////////////////////////////////
		// now try to find a match!

		Atom* const atom0( atoms[0] );
		Atom* const atom3( atoms[3] );

		///////////////////////////////////////////////////////////////////////////
		// now check for backward match

		// first check for a forward match:
		if ( ! atom3->is_jump() ) {
			// check for forward match
			if ( atom3->input_stub_atom1() == atom_ids[2] &&
					 atom3->input_stub_atom2() == atom_ids[1] ) {
				Atom_id const & stub_atom3_id( atom3->input_stub_atom3() );
				if ( stub_atom3_id == atom_ids[0] ) {
					// match!
					return atom_ids[3];

				} else {
					// stub_atom3 is not the same as atoms[0]
					// if they are both connected to stub_atom2 then all we need
					// to do is add a dihedral offset

					Atom const * stub_atom3( atom_pointer( stub_atom3_id.atomno(),
																								 stub_atom3_id.rsd() ) );
					Atom const * stub_atom2( atoms[1] );

					if ( stub_atom2->parent == stub_atom3 &&
							 atom0->parent == stub_atom2 ) {
						// stub_atom3 and atoms[0] are both connected to stub_atom2
						// need to add additional offset:
						float new_offset;
						numeric::dihedral( coords.get_xyz( stub_atom3_id ),
															 coords.get_xyz( atom_ids[2] ),
															 coords.get_xyz( atom_ids[1] ),
															 coords.get_xyz( atom_ids[0] ), new_offset );
						// NOTE: offset is not nec equal to 0 at this point
						offset += radians(new_offset);
						return atom_ids[3];
					}
				}
			}
		}


		if ( ! atom0->is_jump() ) {
			if ( atom0->input_stub_atom1() == atom_ids[1] &&
					 atom0->input_stub_atom2() == atom_ids[2] ) {
				Atom_id const & stub_atom3_id( atom0->input_stub_atom3() );
				if ( stub_atom3_id == atom_ids[3] ) {
					// perfect match!
					return atom_ids[0];
				} else {
					// stub_atom3 is not the same as atoms[3]
					// if they are both connected to stub_atom2 then all we need
					// to do is add a dihedral offset

					Atom const * stub_atom3( atom_pointer( stub_atom3_id.atomno(),
																								 stub_atom3_id.rsd() ) );
					Atom const * stub_atom2( atoms[2] );

					if ( stub_atom2->parent == stub_atom3 &&
							 atom3->parent == stub_atom2 ) {
						// stub_atom3 and atoms[3] are both connected to stub_atom2
						// need to add additional offset:

						float new_offset;
						numeric::dihedral( coords.get_xyz( stub_atom3_id ),
															 coords.get_xyz( atom_ids[1] ),
															 coords.get_xyz( atom_ids[2] ),
															 coords.get_xyz( atom_ids[3] ), new_offset );
						// NOTE: offset is not nec equal to 0 at this point
						offset += radians(new_offset);
						return atom_ids[0];
					}
				}
			}
		}

		// Hey, maybe could put this above with CHAINBREAK_BOGUS_ATOM stuff.
		if ( is_protein(aa) && j != omega_torsion ) {
			std::cout << "WARNING:: missing protein torsion angle: " << j << ' ' <<
				i << std::endl;
		}

// 		// special cases for chain breaks
// 		if ( is_protein(aa ) ) {
// 			if ( j == phi_torsion || j == psi_torsion ) {
// 				// we are at a chainbreak
// 				offset = pi;
// 				Atom const * tor_atom(0);
// 				// find the alternate tor_atom
// 				if ( j == phi_torsion ) {
// 					Atom * N_atom( atom_pointer( 1 /* magic */, i ) );
// 					if ( N_atom->is_jump() ) {
// 						// confirm we are at a cutpoint
// 						assert( N_atom->get_nonjump_atom(1) &&
// 										!N_atom->get_nonjump_atom(2) );
// 						tor_atom = N_atom->get_nonjump_atom(1);
// 						// sanity:
// 						std::string const & tor_atom_name
// 							( aaproperties_pack::atom_name( tor_atom->atom_id.atomno(), aa,
// 																							aav ) );
// 						assert( tor_atom_name == " H  " || tor_atom_name == " CD " );
// 					} else {
// 						// confirm that this is a chainbreak
// 						assert( N_atom->nchildren() == 1 );
// 						tor_atom = N_atom->child(0);
// 					}
// 				} else if ( j == psi_torsion ) {
// 					Atom * C_atom( atom_pointer( 3 /* magic */, i ) );
// 					// confirm that this is a chainbreak
// 					assert( C_atom->nchildren() == 1 );
// 					tor_atom = C_atom->child(0);
// 				}

// 				assert( ( tor_atom->input_stub_atom1() == atom_ids[1] &&
// 									tor_atom->input_stub_atom2() == atom_ids[2] &&
// 									tor_atom->input_stub_atom3() == atom_ids[3] ) ||
// 								( tor_atom->input_stub_atom1() == atom_ids[2] &&
// 									tor_atom->input_stub_atom2() == atom_ids[1] &&
// 									tor_atom->input_stub_atom3() == atom_ids[0] ) );
// 				return tor_atom->atom_id;

// 			} else if ( j != omega_torsion ) {
// 				std::cout << "WARNING:: missing protein torsion angle: " <<
// 					j << ' ' << i << std::endl;
// 			}
// 		}

		return BOGUS_ATOM;
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom*
	Atom_tree::get_jump_atom(
		int const seqpos
	)
	{
		Atom* jump_pointer(0);

		for ( int j=1; j<= param::MAX_ATOM()(); ++j ){
			Atom* atom( atom_pointer( j, seqpos ) );
			if ( atom && atom->is_jump() ) {
				// should be only one jump going into a residue
				assert( jump_pointer == 0 );
				jump_pointer = atom;
			}
		}
		return jump_pointer;
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Atom_tree::get_jump_atom(
		int const seqpos
	) const
	{
		Atom* jump_pointer(0);

		for ( int j=1; j<= param::MAX_ATOM()(); ++j ){
			Atom* atom( atom_pointer( j, seqpos ) );
			if ( atom && atom->is_jump() ) {
				// should be only one jump going into a residue
				assert( jump_pointer == 0 );
				jump_pointer = atom;
			}
		}
		return jump_pointer;
	}


	/////////////////////////////////////////////////////////////////////////////
	// this assumes that we can construct a reasonable starting stub for
	// the root atom ie no 0'd coords

	void
	Atom_tree::refold(
		FArray3D_float & coords
	) const
	{
		Coords_FArray tmp_coords( coords );
		Stub stub( root_->get_stub( tmp_coords ) );

		root_->update_coords( stub, tmp_coords );
	}

	/////////////////////////////////////////////////////////////////////////////
	pose_ns::Jump &
	Atom_tree::jump(
		int const jump_seqpos
	)
	{
		return get_jump_atom( jump_seqpos )->jump();
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Atom_tree::update_domain_map(
		FArray1D_int & domain_map
	) const
	{
		domain_map = -1;

		if ( dynamic_cast<Moving_atom*>( root_ )->torsion_moved ) {
			// signal for full score calculation
			domain_map = 0;
		} else {

			int current_color(1), biggest_color(1);
			PROF_START( prof::DOMAIN_MAP );
			root_->update_domain_map( current_color, biggest_color, domain_map );
			PROF_STOP ( prof::DOMAIN_MAP );
		}


		//
		// we dont want to do this yet!!!!!!!!!
// 		root_->set_torsion_moved( false /* setting */,
// 															true  /* include_fixed */,
// 															true  /* recursive */ );
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Atom_tree::new_score_pose()
	{
		root_->set_torsion_moved( true, true /*include_fixed_torsions*/,
															false /*recursive*/);
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_tree &
	Atom_tree::operator=( Atom_tree const & src )
	{
		// copy tree
		root_ = src.root_->copy( 0 /*parent=0*/ );

		// fill in atom_pointer
		atom_pointer.dimension(src.atom_pointer.size1(), src.atom_pointer.size2());
		atom_pointer = 0;
		root_->update_atom_pointer( atom_pointer );

		// these guys we can just copy
		torsion_pointer_ = src.torsion_pointer_;
		torsion_offset_ = src.torsion_offset_;
		return *this;
	}

} // namespace kin
