// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

// Rosetta Headers
#include "kin_jump_atom.h"
#include "jumping_util.h"
#include "kin_util.h"
#include "pose.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h" // update_nblist! should just make method?

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/FArray4D.h>
//#include <ObjexxFCL/formatted.io.h>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>


namespace kin {
	/////////////////////////////////////////////////////////////////////////////
	void
	Jump_atom::set_allow_move(
		Kin_torsion_type const type,
		bool const setting
	)
	{
		if ( !parent ) {
			rb_translation_fixed = true;
			rb_rotation_fixed = true;
			return;
		}

		std::cout<<"set_jump_atoms: "<<atom_id<<" "<<parent->atom_id<<std::endl;
		if ( type == RB_TRANSLATION ) {
			rb_translation_fixed = !setting;
		} else if ( type == RB_ROTATION ) {
			rb_rotation_fixed = !setting;
		} else if ( type == D || type == THETA || type == PHI || type == ALL ) {
			rb_translation_fixed = !setting;
			rb_rotation_fixed = !setting;
		} else if ( get_rb_number( type ) > 0 ) {
			std::cout << "WARNING: Jump_atom::set_allow_move only accepts two types:"
								<< " RB_TRANSLATION and RB_ROTATION." << std::endl;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Jump_atom::get_allow_move(
		Kin_torsion_type const type
	) const
	{
		if ( type == RB_TRANSLATION ) {
			return !rb_translation_fixed;
		} else if ( type == RB_ROTATION ) {
			return !rb_rotation_fixed;
		} else if ( type == D || type == THETA || type == PHI ) {
			return !torsions_fixed();
		} else if ( type == ALL ) {
			return !torsions_fixed();
		} else {
			std::cout << "WARNING: Jump_atom::get_allow_move only accepts two types:"
								<< " RB_TRANSLATION and RB_ROTATION." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return false;
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Jump_atom::keep_torsion_fixed(
																Kin_torsion_type const //type
	) const
	{
		// keep synced w/previous routine
		return ( parent == 0 );
	}

	/////////////////////////////////////////////////////////////////////////////

	float
	Jump_atom::get_torsion(
		Kin_torsion_type const type
	) const
	{
		int const n2c(1);
		int const rb_no( get_rb_number( type ) );
		if ( rb_no == 0 ) {
			std::cout << "bad torsion type for Jump_atom: " << type << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return jump_.get_rb_delta( rb_no, n2c );
	}

	/////////////////////////////////////////////////////////////////////////////

	void
	Jump_atom::set_torsion(
		Kin_torsion_type const type,
		float const value
	)
	{
		assert( parent );
		int const n2c(1);
		torsion_moved = true;
		int const rb_no( get_rb_number( type ) );
		if ( rb_no == 0 ) {
			std::cout << "bad torsion type for Jump_atom: " << type << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		assert( ! ( rb_translation_fixed && rb_no <= 3 ) &&
						! ( rb_rotation_fixed && rb_no > 3 ) );
		jump_.set_rb_delta( rb_no, n2c, value );
	}

	/////////////////////////////////////////////////////////////////////////////
	pose_ns::Jump const &
	Jump_atom::jump() const
	{
		return jump_;
	}

	/////////////////////////////////////////////////////////////////////////////
	// assume that they are going to change it!
	// not completely safe -- they could hold onto it and change it later,
	// after we've scored once...
	//
	pose_ns::Jump &
	Jump_atom::jump()
	{
		torsion_moved = true;
		return jump_;
	}

	/////////////////////////////////////////////////////////////////////////////
	// this asserts equal topology
	void
	Jump_atom::copy_torsions(
		Atom const & src
	)
	{
		torsion_moved = src.get_torsion_moved();

		jump_ = src.jump();

		rb_translation_fixed = !src.get_allow_move( RB_TRANSLATION );
		rb_rotation_fixed    = !src.get_allow_move( RB_ROTATION );

		if ( atom_id != src.atom_id || int(atoms.size()) != src.nchildren() ) {
			std::cout << "Jump_atom:: copy_torsions: topology_mismatch!" <<
				atom_id  << ' ' << src.atom_id << ' ' << atoms.size() << ' ' <<
				src.nchildren() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		int i(0);
		for ( std::vector< Atom* >::iterator it = atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it, ++i ) {
			(*it)->copy_torsions( *(src.child(i) ) );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump_atom::update_coords(
		Stub & stub, // in fact is const
		Coords & coords
	) const
	{
// 		if ( true ) { // debug
// 			std::cout << "Jump_atom::update_coords: input_stub_dev= " <<
// 				distance( get_input_stub( coords ), stub ) << ' ' << atom_id <<
// 				std::endl;
// 		}
		Stub new_stub;
		jump_.make_stub_jump( stub, new_stub );
		coords.set_xyz( atom_id, new_stub.v );


		for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->update_coords( new_stub, coords );
		}
	}




	/////////////////////////////////////////////////////////////////////////////
	void
	Jump_atom::update_torsions(
		Stub & stub,
		Coords const & coords,
		bool const recursive // = true
	)
	{
		torsion_moved = true; // ensure complete scoring

// 		if ( true ) { // debug
// 			std::cout << "Jump_atom::update_torsions: input_stub_dev= " <<
// 				distance( get_input_stub( coords ), stub ) << ' ' << atom_id <<
// 				std::endl;
// 		}
		Stub new_stub( get_stub( coords ) );

		jump_.from_stubs( stub, new_stub );

		if ( recursive ) {
			for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
							it_end = atoms.end(); it != it_end; ++it ) {
				(*it)->update_torsions( new_stub, coords );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom*
	Jump_atom::copy( Atom* parent_in ) const
	{

		Jump_atom* new_me = new Jump_atom();

		new_me->atom_id = atom_id;
		new_me->parent = parent_in;
		new_me->torsion_moved = torsion_moved;

		// copy torsions
		new_me->jump_ = jump_;

		// allow move:
		new_me->rb_translation_fixed = rb_translation_fixed;
		new_me->rb_rotation_fixed    = rb_rotation_fixed;

		// copy atoms
		for ( std::vector< Atom* >::const_iterator a=atoms.begin(),
						a_end=atoms.end(); a != a_end; ++a ) {
			new_me->atoms.push_back( (*a)->copy( new_me ) );
		}

		return new_me;
	}




	/////////////////////////////////////////////////////////////////////////////
	// this will not modify last_torsion, unlike Atom::setup_min_map
	//
	void
	Jump_atom::setup_min_map(
		Torsion_id & last_torsion, // const!!!
		Minimizer_map & min_map
	) const
	{
		Torsion_id last_torsion_local( last_torsion );

		// root torsions dont move
		if ( parent != 0 && ( !rb_translation_fixed || !rb_rotation_fixed ) ) {
			for ( int k=1; k<=6; ++k ) {
				if ( ( k<=3 && !rb_translation_fixed ) ||
						 ( k>3  && !rb_rotation_fixed ) ) {
					Torsion_id rb_torsion( atom_id, get_rb_type( k ) );
					min_map.add_torsion( rb_torsion, last_torsion_local );
					last_torsion_local = rb_torsion;
				}
			}
		}

		// add me to the min_map
		min_map.add_atom( atom_id, last_torsion_local );

		for ( std::vector< Atom* >::const_iterator it=atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->setup_min_map( last_torsion_local, min_map );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump_atom::get_torsion_axis_and_end_pos(
		Coords const & coords,
		numeric::xyzVector_float & axis,
		numeric::xyzVector_float & end_pos,
		Kin_torsion_type const type
	) const
	{
		Stub input_stub( this->get_input_stub( coords ) );
		int const rb_no( get_rb_number( type ) );
		if ( rb_no <= 3 ) {
			axis = input_stub.M.col( rb_no );
		} else {
			Stub my_stub( this->get_stub( coords ) );

			int const n2c( 1 );
			FArray1Da_double const rb_delta( jump_.get_rb_delta( n2c ) );
			numeric::xyzVector_double const & rb_center( jump_.get_rb_center( n2c ));

			end_pos = my_stub.v + my_stub.M * rb_center;

			if ( type == RB6 ) {
				axis = input_stub.M.col(3);
			} else if ( type == RB5 ) {
				double const theta_z = rb_delta(6);
				axis = ( input_stub.M * Z_rot( theta_z ) ).col(2);
			} else if ( type == RB4 ) {
				double const theta_z = rb_delta(6), theta_y = rb_delta(5);
				axis = ( input_stub.M * Z_rot( theta_z ) * Y_rot( theta_y ) ).col(1);
			} else {
				std::cout << "Bad torsion type for Atom" << type << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	}


} // namespace kin
