// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

// Rosetta Headers
#include "kin_moving_atom.h"
#include "jumping_util.h"
#include "kin_fixed_atom.h"
#include "kin_bonded_atom_FS.h"
#include "kin_jump_atom_FS.h"
#include "kin_util.h"
#include "pose.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h" // update_nblist! should just make method?

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/FArray4D.h>
//#include <ObjexxFCL/formatted.io.h>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>


namespace kin {

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// Moving_atom
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	void
	Moving_atom::set_torsion_moved(
		bool const setting,
		bool const include_fixed_torsions,
		bool const recursive
	)
	{
		torsion_moved = setting;
		if ( !include_fixed_torsions && torsions_fixed() ) {
			torsion_moved = false;
		}
		if ( recursive ) {
			for ( std::vector< Atom* >::iterator it = atoms.begin(),
							it_end = atoms.end(); it != it_end; ++it ) {
				(*it)->set_torsion_moved( setting, include_fixed_torsions, recursive );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::erase()
	{
		for ( std::vector< Atom* >::iterator it = atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->erase();
			delete *it;
		}
		atoms.clear();
	}


	/////////////////////////////////////////////////////////////////////////////
	// this is fairly complicated
	//
	// the topology of the tree gets screwed up
	// dont try changing it after trimming, the stub atoms are hard-coded
	// and wont update properly
	//
	// just for fast refolding/copying of coords
	//
	// note -- this doesn't delete the old tree, in fact we need the old
	// tree for some of the setup

	Atom*
	Moving_atom::trim_fixed_atoms(
		Atom* & parent_in, // may be modified
		bool & good_stub_in // may be modified
	)
	{
		assert( parent && parent_in || !parent && !parent_in );

		// this gets passed during the recursive calls below
		// it will change to new_me if I'm a moving atom
		Atom* current_parent( parent_in );
		Atom* parent_out( parent_in );

		// good_stub determines whether my children will receive
		// a stub that is compatible with the old tree
		bool good_stub_to_children;

		// good_stub_out determines whether my younger siblings will
		// receive such a stub
		bool good_stub_to_siblings( good_stub_in );

		// for appending to atom_list
		Atom* new_me( 0 );

		if ( parent && torsions_fixed() ) {
			// I'm fixed and not the root

			// fixed atoms do not modify the stub in update_torsions/coords
			// so if I'm not a jump, then my siblings will not receive the same
			// stub they used to
			new_me = new Fixed_atom;
			// my children will not get a good stub
			good_stub_to_children = false;
			// nor will my younger siblings:
			if ( !is_jump() ) good_stub_to_siblings = false;

		} else {
			if ( is_jump() || !good_stub_in ) {
				// in this case, I want to preserve the information about
				// stub atoms since we are perturbing the topology of the tree
				//

				// either I'm a jump, or
				// I'm moving but my parent (or previous sibling) is fixed...
				//
				// parent is a pointer to my old parent
				// need to identify atoms that define the stub
				// which is passed in to me by update_torsions/coords
				//

				// these determine the stub that gets passed in during folding

				// for tracking stub atoms
				// we want these to be the same after trimming
				// but since the topology is changing we cant depend on the topology
				// of the tree anymore

				Atom_id const
					in_stub_id0( input_stub_atom0() ),
					in_stub_id1( input_stub_atom1() ),
					in_stub_id2( input_stub_atom2() ),
					in_stub_id3( input_stub_atom3() );

				if ( is_jump() ) {
					new_me = new Jump_atom_FS
						( in_stub_id0, in_stub_id1, in_stub_id2, in_stub_id3,
							stub_atom1(), stub_atom2(), stub_atom3() );
				} else {
					assert( !good_stub_in );
					new_me = new Bonded_atom_FS
						( in_stub_id0, in_stub_id1, in_stub_id2, in_stub_id3,
							keep_torsion_fixed( PHI ), keep_torsion_fixed( THETA ) );
					// we make the remainder of our original parent's
					// children depend on us, if our parent is fixed, we are not a jump,
					// and we are not fixed
					parent_out = new_me;
					good_stub_to_siblings = false;
				}

			} else {
				// my parent is moving also, I'm a normal atom, so everything is cool
				assert( !is_jump() && good_stub_in );
				new_me = new Bonded_atom;

			} // is my parent moving?

			// I'm moving, so pass me into the recursive calls on my children below
			current_parent = new_me;
			good_stub_to_children = true; // my kids will get a decent stub

		} // am I moving?


		// common to all types
		new_me->atom_id = atom_id;
		new_me->parent = parent_in;

		// add to parent's list of atoms
		parent_in->add_atom( new_me );

		for ( std::vector< Atom* >::iterator it = atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			// note that current_parent may change during these calls
			// as can good_stub
			(*it)->trim_fixed_atoms( current_parent, good_stub_to_children );
		}


		// we may potentially make the remainder of our original parents
		// children depend on us, if our parent is fixed, we are not a jump,
		// and we are not fixed
		assert( parent_out == parent_in ||
						( parent_out == new_me && !is_jump() && !torsions_fixed() &&
							!good_stub_in ) );

		parent_in = parent_out;
		good_stub_in = good_stub_to_siblings;
		return new_me;
	}



	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::update_atom_pointer(
		FArray2D< Atom* > & atom_pointer
	)
	{
		atom_pointer( atom_id.atomno(), atom_id.rsd() ) = this;
		for ( std::vector< Atom* >::const_iterator a=atoms.begin(),
						a_end = atoms.end(); a != a_end; ++a ) {
			(*a)->update_atom_pointer( atom_pointer );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// the coordinates of the atom "*child" -- one of my children -- have
	// changed. This routine updates the torsions to reflect this. Useful
	// if we have just repacked or rotamer-trialed, ie sidechain atoms
	// have moved but the backbone is still the same, more efficient than
	// calling update_torsions on the entire tree...
	//
	// note that update_torsions is called recursively on all of *child's
	// children.
	//
	void
	Moving_atom::update_child_torsions(
		Atom* const child,
		Coords const & coords
	)
	{
		Stub my_stub( this->get_stub( coords ) );
		bool found( false );
		for ( std::vector< Atom* >::iterator it = atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			if ( *it == child ) {
				(*it)->update_torsions( my_stub, coords );
				// phi torsion for the atom after child may have changed
				++it;
				if ( it != it_end ) {
					(*it)->update_torsions( my_stub, coords, false /* not recursive */ );
				}
				found = true;
				break;
			} else {
				// just advances the stub as if we had called update_torsions
				(*it)->update_stub( my_stub );
			}
		}
		if ( !found ) {
			std::cout << "update_child_torsions:: child not in atoms list" <<
				atom_id << ' ' << child->atom_id << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	/////////////////////////////////////////////////////////////////////////////

	void
	Moving_atom::add_atom(
		Atom* const atom
	)
	{
		atom->parent = this;
		if ( atom->is_jump() ) {
			atoms.insert( nonjump_atoms_begin(), atom );
		} else {
			atoms.push_back(atom);
		}
	}

	/////////////////////////////////////////////////////////////////////////////

	void
	Moving_atom::delete_atom(
		Atom* const child
	)
	{
		std::vector< Atom* >::iterator iter
			( std::find( atoms.begin(), atoms.end(), child ) );
		if ( iter == atoms.end() ) {
			std::cout << "child not present in atoms list! " <<
				atoms.size() << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		atoms.erase( iter );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::insert_atom(
		Atom* const atom
	)
	{
		atom->parent = this;
		if ( atom->is_jump() ) {
			atoms.insert( atoms.begin(), atom );
		} else {
			atoms.insert( nonjump_atoms_begin(), atom );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Moving_atom::nchildren() const
	{
		return atoms.size();
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Moving_atom::child( int const k ) const
	{
		assert( 0<= k && k< int(atoms.size()) );
		return atoms[k];
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom *
	Moving_atom::child( int const k )
	{
		assert( 0<= k && k< int(atoms.size()) );
		return atoms[k];
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::replace_atom(
		Atom* const old_atom,
		Atom* const new_atom
	)
	{
		assert( ( old_atom->is_jump() && new_atom->is_jump() ) ||
						( !old_atom->is_jump() && !new_atom->is_jump() ) );

		std::vector< Atom* >::iterator iter
			( std::find( atoms.begin(), atoms.end(), old_atom ) );
		if ( iter == atoms.end() ) {
			std::cout << "old_atom not present in atoms list! " <<
				atoms.size() << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		new_atom->parent = this;
		iter = atoms.insert( iter, new_atom );
// 		std::cout << (*iter == new_atom) << ' ' <<
// 			(*iter == old_atom) << std::endl;
		++iter;
		assert( *iter == old_atom );
		atoms.erase( iter );
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Moving_atom::downstream( const Atom* atom1 ) const
	{
		bool found ( false );
		for(int ii=0, ie= nchildren(); ii < ie; ii++ ) {
				if( child(ii) == atom1 ) {
					found = true;
				}	else {
					found = child(ii)->downstream( atom1 );
				}
				if( found ) return true;
		}
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Moving_atom::fixed_downstream( const Atom* atom1, Kin_torsion_type const fixed_tor_id ) const
	{
		bool found ( false );
		//to check only atoms fixed to be moved
		for(int ii=0, ie= nchildren(); ii < ie; ii++ ) {
			//std::cout<<"downstream: "<<child(ii)->atom_id<<" "<<child(ii)->get_allow_move( fixed_tor_id )<<" "<<atom1->atom_id<<std::endl;
			if( !child(ii)->get_allow_move( fixed_tor_id ) ) {
				if( child(ii) == atom1 ) {
					found = true;
				}	else {
					found = child(ii)->fixed_downstream( atom1, fixed_tor_id );
				}
			}
			if( found ) return true;
		}
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	std::vector< Atom* >::const_iterator
	Moving_atom::nonjump_atoms_begin() const
	{
		std::vector< Atom* >::const_iterator iter( atoms.begin() );
		while ( iter != atoms.end() && (*iter)->is_jump() ) ++iter;
		return iter;
	}


	/////////////////////////////////////////////////////////////////////////////
	std::vector< Atom* >::iterator
	Moving_atom::nonjump_atoms_begin()
	{
		std::vector< Atom* >::iterator iter( atoms.begin() );
		while ( iter != atoms.end() && (*iter)->is_jump() ) ++iter;
		return iter;
	}


	/////////////////////////////////////////////////////////////////////////////
	// returns 0 if atom doesnt exist
	Atom const *
	Moving_atom::get_nonjump_atom(
		int const index
	) const
	{
		std::vector< Atom* >::const_iterator iter( nonjump_atoms_begin() );
		iter+=index;
		if ( iter >= atoms.end() ) {
			return 0;
		} else {
			return *iter;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Moving_atom::stub_defined() const
	{
		// have to handle a couple of cases here:

		// note -- in counting dependent atoms, exclude Jump_atom's
		//

		// 1. no dependent atoms --> no way to define new coord sys
		//    on this end. ergo take parent's M and my xyz
		//
		// 2. one dependent atom --> no way to define unique coord
		//    on this end, still take parent's M and my xyz
		//
		// 3. two or more dependent atoms
		//    a) if my first atom has a dependent atom, use
		//       myself, my first atom, and his first atom
		//
		//    b) otherwise, use
		//       myself, my first atom, my second atom
		//

		if ( is_jump() ) {
			Atom const * first = get_nonjump_atom(0);
			if ( first != 0 &&
					 ( first->get_nonjump_atom(0) != 0 || get_nonjump_atom(1) != 0 ) ) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// my stub

	Stub
	Moving_atom::get_stub(
		Coords const & coords
	) const
	{
// 		std::cout <<
// 			"get_stub: me= " << atom_id <<
//  			" stub_atom1= " << stub_atom1() <<
//  			" stub_atom2= " << stub_atom2() <<
//  			" stub_atom3= " << stub_atom3() << std::endl;

		return Stub( coords.get_xyz( atom_id ),
								 coords.get_xyz( stub_atom1() ),
								 coords.get_xyz( stub_atom2() ),
								 coords.get_xyz( stub_atom3() ) );
	}


	/////////////////////////////////////////////////////////////////////////////
	// the stub that is passed to me during folding

	Stub
	Moving_atom::get_input_stub(
		Coords const & coords
	) const
	{
// 		std::cout << "get_input_stub: " << atom_id << ' ' <<
// 			input_stub_atom0() << ' ' <<
// 			input_stub_atom1() << ' ' <<
// 			input_stub_atom2() << ' ' <<
// 			input_stub_atom3() << std::endl;

		return Stub( coords.get_xyz( input_stub_atom0() ),
								 coords.get_xyz( input_stub_atom1() ),
								 coords.get_xyz( input_stub_atom2() ),
								 coords.get_xyz( input_stub_atom3() ) );
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::stub_atom1() const
	{
		if ( is_jump() ) {
			if ( stub_defined() ) {
				return atom_id;
			} else {
				return parent->atom_id;
			}
		} else {
			return atom_id;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::stub_atom2() const
	{
		if ( is_jump() ) {
			if ( stub_defined() ) {
				return get_nonjump_atom(0)->atom_id;
			} else {
				return parent->stub_atom2();
			}
		} else {
			return parent->atom_id;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::stub_atom3() const
	{
		//std::cout << "stub_atom3: " << this << ' ' << parent << std::endl();
		if ( is_jump() ) {
			if ( stub_defined() ) {
				Atom const * first( get_nonjump_atom(0) );
				Atom const * second( first->get_nonjump_atom(0) );
				if ( second != 0 ) {
					return second->atom_id;
				} else {
					return get_nonjump_atom(1)->atom_id;
				}
			} else {
				return parent->stub_atom3();
			}
		} else {
			// not a jump

			if ( parent->is_jump() ) {
				assert( parent->stub_defined() ); // weird behavior otherwise
				Atom_id const & p_stub2( parent->stub_atom2() );
				if ( atom_id == p_stub2 ) {
					// very special case!!
					return parent->stub_atom3();
				} else {
					return p_stub2;
				}
			} else {
				return parent->stub_atom2();
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::input_stub_atom0() const
	{
		if ( !parent ) {
			return atom_id;
		} else {
			return parent->atom_id;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::input_stub_atom1() const
	{
		if ( !parent ) {
			return stub_atom1();
		} else {
			return parent->stub_atom1();
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::input_stub_atom2() const
	{
		if ( !parent ) {
			return stub_atom2();
		} else {
			return parent->stub_atom2();
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom_id const &
	Moving_atom::input_stub_atom3() const
	{
		if ( !parent ) {
			return stub_atom3();
		} else {
			Atom const * sibling( previous_sibling() );
			if ( is_jump() || sibling == 0 || sibling->is_jump() ||
					 ( parent->is_jump() && sibling->atom_id == parent->stub_atom2())) {
				return parent->stub_atom3();
			} else {
				return sibling->atom_id;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Moving_atom::previous_sibling() const
	{
		if ( parent != 0 ) {
			return parent->previous_child( this );
		} else {
			return 0;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom const *
	Moving_atom::previous_child(
		Atom const * child
	) const
	{
		//std::cout << "atoms.size() = " << atoms.size() << std::endl;
		std::vector< Atom* >::const_iterator iter
			( std::find( atoms.begin(), atoms.end(), child ) );
		if ( iter == atoms.end() ) {
			std::cout << "child not present in atoms list! " <<
				atoms.size() << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( iter == atoms.begin() ) {
			return 0;
		} else {
			--iter;
			return *iter;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom *
	Moving_atom::next_child(
		Atom const * child
	)
	{
		//std::cout << "atoms.size() = " << atoms.size() << std::endl;
		std::vector< Atom* >::const_iterator iter
			( std::find( atoms.begin(), atoms.end(), child ) );
		if ( iter == atoms.end() ) {
			std::cout << "child not present in atoms list! " <<
				atoms.size() << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		++iter;
		if ( iter == atoms.end() ) {
			return 0;
		} else {
			return *iter;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::update_domain_map(
		int & current_color,
		int & biggest_color,
		FArray1D_int & domain_map
	) const
	{
		int color( current_color );
		if ( torsion_moved ) {
			biggest_color++;
			if ( !is_jump() ) {
				// propagates through atoms after me in the list
				// that's assuming it's my bond torsion that has changed
				current_color = biggest_color;
			}
			color = biggest_color;
		}

		// update domain_map
		int const current_map( domain_map( atom_id.rsd() ) );
		if ( current_map < 0 ) {
			// unassigned
			domain_map( atom_id.rsd() ) = color;
		} else if ( current_map == 0 || current_map == color ) {
			// leave the same
		} else {
			// color change within a residue
			domain_map( atom_id.rsd() ) = 0;
		}


		for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->update_domain_map( color, biggest_color, domain_map );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Moving_atom::show() const
	{
		std::cout << "ATOM: " << atom_id << std::endl;
		std::cout << "   PARENT: ";
		if ( parent ) {
			std::cout << parent->atom_id << std::endl;
		} else {
			std::cout << " NULL" << std::endl;
		}
		std::cout << "   CHILDREN: ";
		for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			std::cout << (*it)->atom_id << ' ';
		}
		std::cout << std::endl;
		for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->show();
		}
	}


} // namespace kin
