// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#include "lee_richards.h"
#include "murphp_macros.h"


#include <iostream>
#include <math.h>

// #include <Util/Util.hh>
// #include <Util/Macros.hh>
// #include <Util/Directory.hh>

namespace murphp {

  namespace LeeRichardsInfo {

    namespace {

      const char* translate_event_type(const SLE_createArcs::Type& event_type) {

	switch( event_type ) {
	case SLE_createArcs::BEGIN:
	  return "SLE_createArcs::BEGIN";
	case SLE_createArcs::OCCLUDE_HI:
	  return "SLE_createArcs::OCCLUDE_HI";
	case SLE_createArcs::OCCLUDE_LO:
	  return "SLE_createArcs::OCCLUDE_LO";
	case SLE_createArcs::REVEAL_HI:
	  return "SLE_createArcs::REVEAL_HI";
	case SLE_createArcs::REVEAL_LO:
	  return "SLE_createArcs::REVEAL_LO";
	case SLE_createArcs::END:
	  return "SLE_createArcs::END";
	default:
	  return "SLE_createArcs::UNKNOWN";
	} // switch

      } // translate_event_type

    } // namespace

      // ____________________ Atom ____________________

    //     Atom::Atom(const Util::Atom* util_atom_) : util_atom(util_atom_) {
    //     } // Atom::Atom

//     const Point3& Atom::getPosn() const {
//       return util_atom->getPosn();
//     } // Atom::getPosn


//    const Point3& Atom::getPosn() const {
//       return util_atom->getPosn();
//     } // Atom::getPosn

//     const double Atom::getRadius() const {

//       static const double C = 2.2;
//       static const double N = 1.6;
//       static const double O = 1.6;
//       static const double H = 0.7;
//       static const double S = 1.89;
//       static const double UNK = 1.8;

//       static const double W = 1.4;

//       double r = UNK;
//       switch( util_atom->getAtomType()[1] ) {
//       case 'C': r = C; break;
//       case 'N': r = N; break;
//       case 'O': r = O; break;
//       case 'H': r = H; break;
//       case 'S': r = S; break;
//       }

//       return r + W;
//     } // Atom::getRadius

    // ____________________ Arc ____________________

    Arc::Arc(const Circle* circle_, const Arc::Type type_) :
      circle(circle_), type(type_) {
      next_arc = 0;
      boundary = 0;
    } // Arc::Arc

    const double Arc::getArea() const {
      //const double A_zab = -Point3::cross(a,b)[2] / 2.0; // this only works when summing over a closed polygon
      const double A_zab = ( Arc::type == Arc::LO ? 1 : -1 ) * Point3::cross(begin,end)[2] / 2.0;
      const double theta = std::abs(theta_b-theta_a);
      const double r2 = circle->getRadius() * circle->getRadius();
      const double A_chomp = r2 * ( theta - sin(theta) ) / 2.0;
      return  A_zab + A_chomp;
    } // Arc::getArea

    const double Arc::getPerimeter() const {
      const double theta = std::abs(theta_b-theta_a);
      return circle->getRadius() * theta;
    } // Arc::getArea

    void Arc::setNext(Arc* next_arc_, const SLE_createArcs::Type event_type, const SLE_createArcs::Type other_event_type) {
      if( next_arc != 0 ) {
	cout << "setNextArc - uh oh, " << translate_event_type(event_type) << " -> " << translate_event_type(other_event_type) << endl;
      }
      assert( next_arc == 0 );
      next_arc = next_arc_;
    } // setNextArc

    void Arc::setBoundary(const Boundary* b) {
      assert( boundary == 0 );
      boundary = b;
    } // Arc::setBoundary

    void Arc::setPoints(const Point2& a_, const Point2& b_) {
      begin = a_;
      end   = b_;

      const double TWO_PI = 2.0 * M_PI;

      switch( type ) {

      case Arc::HI:

	theta_a = asin( (begin.y - circle->getCenter().y)/circle->getRadius() );
	if( begin.x < circle->getCenter().x ) {
	  theta_a = M_PI - theta_a;
	}

	theta_b = asin( (end.y - circle->getCenter().y)/circle->getRadius() );
	if( end.x < circle->getCenter().x ) {
	  theta_b = M_PI - theta_b;
	}

	break;

      case Arc::LO:

	theta_a = asin( (begin.y - circle->getCenter().y)/circle->getRadius() );
	if( begin.x < circle->getCenter().x ) {
	  theta_a = M_PI - theta_a;
	}
	else {
	  theta_a = TWO_PI + theta_a;
	}

	theta_b = asin( (end.y - circle->getCenter().y)/circle->getRadius() );
	if( end.x < circle->getCenter().x ) {
	  theta_b = M_PI - theta_b;
	}
	else {
	  theta_b = TWO_PI + theta_b;
	}

	//swap(a,b); // don't want to swap here

	break;

      default:
	assert( false );

      } // switch

      setChord();

    } // Arc::setPoints

    void Arc::setChord() {

      // 	const Point2& i = getBegin();
      // 	const Point2& j = getEnd();

      const double dx = end.x - begin.x;
      const double dy = end.y - begin.y;
      assert( dx != 0 );
      chord_m = dy/dx;
      chord_b = begin.y - chord_m * begin.x;

    } // Arc::setChord

    const double Arc::getChord(const double x) const {
      return chord_m*x + chord_b;
    } // Arc::getChord

    //       const Point2& Arc::getBegin() const {
    // 	if( begin == 0 ) {
    // 	  begin = (a < b) ? &a : &b;
    // 	}
    // 	return *begin;
    //       } // Arc::getBegin

    //       const Point2& Arc::getEnd() const {
    // 	if( end == 0 ) {
    // 	  end = (a < b) ? &b : &a;
    // 	}
    // 	return *end;
    //       } // Arc::getEnd

    ostream& operator<<(ostream& out, const Arc& arc) {

      static const int m = 5;
      const int da = static_cast<int>(180.0/M_PI*arc.theta_a);
      const int db = static_cast<int>(180.0/M_PI*arc.theta_b);
      const int x = 100 + static_cast<int>(m*arc.circle->getCenter().x) + static_cast<int>(1*arc.getBoundary()->getSlice()->getIndex());
      const int y = 100 + static_cast<int>(m*arc.circle->getCenter().y) + static_cast<int>(1*arc.getBoundary()->getSlice()->getIndex());
      const int r = static_cast<int>(m*arc.circle->getRadius());

      switch( arc.type ) {
      case Arc::HI:
	out << x << " " << y << " " << r << " " << db << " " << da << " arc stroke";
	break;
      case Arc::LO:
	out << x << " " << y << " " << r << " " << da << " " << db << " arc stroke";
	break;
      }

      return out;
    } // operator<<

      // ____________________ Circle ____________________

    void Circle::clear() {
      pending[0] = pending[1] = false;  nho = 0; nlo = 0; nhr = 0; nlr = 0;

      pending_arc[0] = pending_arc[1] = 0;
    } // Circle::clear

    Circle::Circle(const Atom* atom_, const Point2& center_, const double radius_) : atom(atom_), center(center_), radius(radius_) {
      clear();
    } // Circle::Circle

    void Circle::assertSanity() {
      //     bool either = false;
      //     if( initial_occ + nho - nhr != final_occ ) { //cerr << "HI OCC/REV PROBLEM" << endl; either = true; }
      //     if( initial_occ + nlo - nlr != final_occ ) { //cerr << "LO OCC/REV PROBLEM" << endl; either = true;  }
      //     if( either ) {
      //       cerr << "initial_occ = " << initial_occ << endl;
      //       cerr << "final_occ = " << final_occ << endl;
      //       cerr << "nho = " << nho << endl;
      //       cerr << "nhr = " << nhr << endl;
      //       cerr << "nlo = " << nlo << endl;
      //       cerr << "nlr = " << nlr << endl;
      //     }
    } // Circle::assertSanity

    void Circle::initialize(int hi_occ, int lo_occ, const Point2& p, Circle* other, SLE_createArcs::Type other_type) {
      //initial_occ = _occ;
      occ[0] = hi_occ; // hi and lo occ are not necessarily the same if the beginning point lies on a boundary
      occ[1] = lo_occ;
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( occ[0] == 0 ) {
	pending[0] = true;
	last[0] = p;

	assert( pending_arc[0] == 0 );
	pending_arc[0] = new Arc(this,Arc::HI);

      }
      if( occ[1] == 0 ) {
	pending[1] = true;
	last[1] = p;

	assert( pending_arc[1] == 0 );
	pending_arc[1] = new Arc(this,Arc::LO);

	if( pending_arc[0] ) {
	  pending_arc[1]->setNext(pending_arc[0],SLE_createArcs::BEGIN,SLE_createArcs::BEGIN);
	}
	else {
	  cout << "Circle::initialize - hard case not handled pending arc 1 != 0 but pending arc 0 = 0" << endl;
	}

      }
      //     else {
      //       pending[0] = false;
      //       pending[1] = false;
      //     }
    } // Circle::initialize

    void Circle::occlude_hi(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type) {
      nho++;
      occ[0]++;
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( pending[0] ) {
	pending[0] = false;
	//addArc_hi(last[0], p, vArcs);

	assert(pending_arc[0]);
	pending_arc[0]->setPoints(last[0],p);

	vArcs.push_back(pending_arc[0]);

	if( other_type == SLE_createArcs::REVEAL_HI ||
	    other_type == SLE_createArcs::OCCLUDE_LO ) {
	  // don't nullify pending_arc[0] yet
	}
	else {
	  pending_arc[0] = 0;
	}

      }
    } // Circle::occlude_hi

    void Circle::occlude_lo(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type) {
      nlo++;
      occ[1]++;
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( pending[1] ) {
	pending[1] = false;
	//addArc_lo(last[1], p, vArcs);

	assert(pending_arc[1]);
	pending_arc[1]->setPoints(last[1],p);

	if( other_type == SLE_createArcs::OCCLUDE_HI ) {
	  assert( other->pending_arc[0] );
	  other->pending_arc[0]->setNext(pending_arc[1],SLE_createArcs::OCCLUDE_LO,SLE_createArcs::OCCLUDE_HI);
	  other->pending_arc[0] = 0;
	}

	vArcs.push_back(pending_arc[1]);

	if( other_type == SLE_createArcs::REVEAL_LO ) {
	  // don't nullify pending_arc[1] in this case
	}
	else {
	  pending_arc[1] = 0;
	}

      }
    } // Circle::occlude_lo

    void Circle::reveal_hi(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type) {
      nhr++;
      occ[0]--;
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( occ[0] == 0 ) {
	last[0] = p;
	pending[0] = true;

	assert( pending_arc[0] == 0 );
	pending_arc[0] = new Arc(this,Arc::HI);

	if( other_type == SLE_createArcs::OCCLUDE_HI ) {
	  assert( other->pending_arc[0] );
	  other->pending_arc[0]->setNext(pending_arc[0],SLE_createArcs::REVEAL_HI,SLE_createArcs::OCCLUDE_HI);
	  other->pending_arc[0] = 0;
	}
	else if( other_type == SLE_createArcs::REVEAL_LO ) {
	  assert( other->pending_arc[1] );
	  other->pending_arc[1]->setNext(pending_arc[0],SLE_createArcs::REVEAL_HI,SLE_createArcs::REVEAL_LO);
	  // don't nullify here
	}

      }
    } // Circle::reveal_hi

    void Circle::reveal_lo(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type) {
      nlr++;
      occ[1]--;
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( occ[1] == 0 ) {
	last[1] = p;
	pending[1] = true;

	assert( pending_arc[1] == 0 ) ;
	pending_arc[1] = new Arc(this,Arc::LO);

	if( other_type == SLE_createArcs::OCCLUDE_LO ) {
	  assert( other->pending_arc[1]);
	  pending_arc[1]->setNext(other->pending_arc[1],SLE_createArcs::REVEAL_LO,SLE_createArcs::OCCLUDE_LO);
	  other->pending_arc[1] = 0;
	}

      }
    } // Circle::reveal_lo

    void Circle::finalize(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type) {
      //cerr << "hi occ = " << occ[0] << ", lo occ = " << occ[1] << endl;
      if( pending[0] ) {
	pending[0] = false;

	assert( pending_arc[0] ) ;
	pending_arc[0]->setPoints(last[0],p);
	vArcs.push_back(pending_arc[0]);

	if( pending_arc[1] ) {
	  pending_arc[0]->setNext(pending_arc[1],SLE_createArcs::END,SLE_createArcs::END);
	}
	else {
	  cout << "Circle::finalize - hard case not handled, pending_arc[0] != 0 but pending_arc[1] == 0" << endl;
	}

	pending_arc[0] = 0;



      }
      if( pending[1] ) {
	pending[1] = false;

	assert( pending_arc[1] ) ;
	pending_arc[1]->setPoints(last[1],p);
	vArcs.push_back(pending_arc[1]);
	pending_arc[1] = 0;

      }
      assertSanity();
    } // Circle::finalize

    const bool Circle::intersect(const Circle& a, const Circle& b, Point2& isx0, Point2& isx1) {
      // a is the Circle that is already on the event stack
      // b is the Circle that is just BEGIN-ing
      // p is the first point scanned in b

      Point2 dif = b.getCenter()-a.getCenter();
      double d = dif.size();

      //   d_hi_occ = 0;
      //   d_lo_occ = 0;

      //   if( Point2::dist(p, a.getCenter()) < a.radius ) {
      //     d_hi_occ = 1;
      //     d_lo_occ = 1;
      //   }

      if( d < a.radius + b.radius && // make sure they're close enough to intersect
	  a.radius < d + b.radius &&  // make sure a doesn't totally include b
	  b.radius < d + a.radius ) { // make sure b doesn't totally include a

	// compute the intersection with the law of cosines
	double theta = acos((d*d + a.radius*a.radius - b.radius*b.radius) / (2.0*a.radius*d) );
	double h = a.radius*sin(theta);
	double f = a.radius*sin(M_PI_2 - theta);

	// normalize dif at this point...
	dif = dif/d;
	Point2 mid = a.getCenter() + dif*f;
	Point2 nrm(-dif.y, dif.x); nrm = nrm*h;

	// return the intersecting points
	isx0 = mid + nrm;
	isx1 = mid - nrm;

	//     if( isx0 == p ) {
	//       if( a.getCenter().y > p.y ) d_hi_occ = 1;
	//       else d_lo_occ = 1;
	//     }
	//     if( isx1 == p ) {
	//       if( a.getCenter().y < p.y ) d_lo_occ = 1;
	//       else d_hi_occ = 1;
	//     }

	//cerr << "Found two intersections: " << isx0 << ", " << isx1 << endl;

	return true;
      }

      //   if( b.radius < d + a.radius ) {
      //     d_hi_occ = 1;
      //     d_lo_occ = 1;
      //   }

      return false;
    } // const bool Circle::intersect

    ostream& operator<<(ostream& out, const Circle& c) {
      return out << "Circle:\tcenter " << c.getCenter() << "\tradius " << c.getRadius();
    } // operator<<

      // ____________________ SLE_createArcs ____________________

    SLE_createArcs::SLE_createArcs(SLE_createArcs::Type event_type_, const Point2& p_, Circle* c_, Circle* other_, SLE_createArcs::Type other_event_type_) : event_type(event_type_), other_event_type(other_event_type_), p(p_), c(c_), other(other_) {
    } // SLE_createArcs::SLE_createArcs

    const bool SLE_createArcs::operator<(const SLE_createArcs& other) const {

      if( p.x != other.p.x ) {
	return p.x < other.p.x;
      }
      else if (p.y != other.p.y ){
	return p.y < other.p.y;
      }
      else if( event_type != other.event_type ) {
	return event_type < other.event_type;
      }
      else {
	return this < &other;
      }

    } // SLE_createArcs::operator<

    ostream& operator<<(ostream& out, const SLE_createArcs& e) {
      out << "Event: ";
      switch( e.event_type ) {
      case SLE_createArcs::BEGIN: out << "begin"; break;
      case SLE_createArcs::END: out << "end"; break;
      case SLE_createArcs::OCCLUDE_HI: out << "occlude hi"; break;
      case SLE_createArcs::OCCLUDE_LO: out << "occlude lo"; break;
      case SLE_createArcs::REVEAL_HI: out << "reveal hi"; break;
      case SLE_createArcs::REVEAL_LO: out << "reveal lo"; break;
      } // switch
      out << ", center = " << e.c->getCenter() << ", radius = " << e.c->getRadius() << ", event-point = " << e.p;
      return out;
    } // operator<<

      // ____________________ Boundary ____________________

    Boundary::Boundary(const Arc* arc0_, const Slice* slice_) : arc0(arc0_), slice(slice_), volume(0) {
      const Arc* arc = arc0;
      area = 0;
      perimeter = 0;
      num_arcs = 0;

      do {

	const_cast<Arc*>(arc)->setBoundary(this);

	area += arc->getArea();
	perimeter += arc->getPerimeter();

	arc = arc->getNext();
	assert( arc != 0 );
	++num_arcs; // XXX should really test whether this is a sensible number of arcs

      } while( arc != arc0 );

    } // Boundary::Boundary

    Boundary::~Boundary() {
    } // Boundary::~Boundary

    const int Boundary::getNumArcs() const {
      return num_arcs;
    } // Boundary::getNumArcs

    const double Boundary::getArea() const {
      return area;
    } // Boundary::getArea

    const double Boundary::getPerimeter() const {
      return perimeter;
    } // Boundary::getPerimeter

    const bool Boundary::isConvex() const {
      return area > 0;
    } // Boundary::isConvex

    const bool Boundary::isConcave() const {
      return !isConvex();
    } // Boundary::isConvex

    void Boundary::setVolume(const Volume* v) {
      volume = v;
    } // Boundary::setVolume

    void Boundary::addBoundary(const Boundary* b) {
      sBoundaries.insert(b);
    } // Boundary::addBoundary

    void Boundary::reportArcs(ostream& out) const {
      const Arc* arc = getArc0();

      do {
	out << *arc << "\n";
	arc = arc->getNext();
	assert( arc != 0 );
      } while( arc != getArc0() );

    } // Boundary::reportArcs


    const double Boundary::THRESHOLD_AREA_POS =  0.001;
    const double Boundary::THRESHOLD_AREA_NEG =  -0.001;


    //       const double Boundary::THRESHOLD_AREA_POS =  1;
    //       const double Boundary::THRESHOLD_AREA_NEG =  -1;

    const bool Boundary::isIgnorable() const {
      if( area > 0 ) {
	return area < THRESHOLD_AREA_POS;
      }
      else {
	return area > THRESHOLD_AREA_NEG;
      }
    } // Boundary::isIgnorable

    const bool Boundary::isSame(const Boundary* other) const {
      if( isConvex() ) {
	return other->isConvex();
      }
      else if( isConcave() ) {
	return other->isConcave();
      }
      else {
	assert( false );
      }
    } //  Boundary::isSame

      // ____________________ Slice ____________________

    namespace {

      void swap( Point2& a, Point2& b) {
	Point2 c = a;
	a = b;
	b = c;
      } // void swap

    } // namespace

    Slice::Slice() : prev(0), next(0) {
    } // Slice::Slice

    Slice::~Slice() {
    } // Slice::~Slice

    void Slice::addCircles(const vector<Atom*>& vAtoms, const double z_) {

      z = z_;

      FORVPC(i,Atom,vAtoms) {
	const Atom* a = *i;
	const Point3& p = a->getPosn();

	const double dz = p[2] - getZ();
	const double R  = 2; // this should really reflect the atom radius
	const double radius2 = R*R - dz*dz;

	if( radius2 > 0 ) {
	  Circle* c = new Circle(a,Point2(p[0],p[1]),sqrt(radius2));
	  vCircles.push_back(c);
	}

      } // i

    } // Slice::addCircles

    void Slice::addEvents(Point2& isx0, Point2& isx1, Circle& a, Circle& b) {
      if( isx1.x < isx0.x ) swap(isx0, isx1);

      SLE_createArcs::Type ta0, ta1, tb0, tb1;
      bool ba0 = isx0.y > a.getCenter().y;
      bool ba1 = isx1.y > a.getCenter().y;

      // define messages for Circle a
      if( (ba0 && ba1) ) { // 0 and 1 higher than a
	ta0 = SLE_createArcs::OCCLUDE_HI;
	ta1 =  SLE_createArcs::REVEAL_HI;
      }
      else if( ba0 ) { // 0 higher, 1 lower than a
	ta0 = SLE_createArcs::OCCLUDE_HI;
	ta1 = SLE_createArcs::OCCLUDE_LO;
      }
      else if( ba1 ) { // 0 lower, 1 higher than a
	ta0 = SLE_createArcs::OCCLUDE_LO;
	ta1 = SLE_createArcs::OCCLUDE_HI;
      }
      else { // 0 lower, 1 lower than a
	ta0 = SLE_createArcs::OCCLUDE_LO;
	ta1 =  SLE_createArcs::REVEAL_LO;
      }

      double d = Point2::dist(Point2(b.getCenter().x - b.getRadius(), b.getCenter().y), a.getCenter());
      bool bfina = d < a.getRadius();
      bool bfona = d == a.getRadius();

      bool gt0b = isx0.y > b.getCenter().y;
      bool gt1b = isx1.y > b.getCenter().y;
      bool eq0b = isx0.y == b.getCenter().y;
      bool eq1b = isx1.y == b.getCenter().y;

      if( bfina ) {
	if( gt0b ) {
	  if( gt1b ) {
	    tb0 = SLE_createArcs::REVEAL_HI;
	    tb1 = SLE_createArcs::OCCLUDE_HI;
	  }
	  else if( eq1b ) {
	    tb0 = SLE_createArcs::REVEAL_HI;
	    tb1 = SLE_createArcs::OCCLUDE_HI;
	  }
	  else { // lt1b
	    tb0 = SLE_createArcs::REVEAL_HI;
	    tb1 = SLE_createArcs::REVEAL_LO;
	  }
	}
	else { // lt0b
	  if( gt1b ) {
	    tb0 = SLE_createArcs::REVEAL_LO;
	    tb1 = SLE_createArcs::REVEAL_HI;
	  }
	  else if ( eq1b ) {
	    tb0 = SLE_createArcs::REVEAL_LO;
	    tb1 = SLE_createArcs::OCCLUDE_LO;
	  }
	  else { // lt1b
	    tb0 = SLE_createArcs::REVEAL_LO;
	    tb1 = SLE_createArcs::OCCLUDE_LO;
	  }
	}
      }
      else if( bfona ) {
	double yavg = (isx0.y + isx1.y)/2.0;
	if( gt1b ) {
	  if( a.getCenter().y > yavg ) { // case I
	    tb0 = SLE_createArcs::OCCLUDE_HI;
	    tb1 = SLE_createArcs::REVEAL_HI;
	  }
	  else { // case II
	    tb0 = SLE_createArcs::OCCLUDE_LO;
	    tb1 = SLE_createArcs::OCCLUDE_HI;
	  }
	}
	else { // lt1b
	  if( a.getCenter().y < yavg ) {
	    tb0 = SLE_createArcs::OCCLUDE_LO;
	    tb1 = SLE_createArcs::REVEAL_LO;
	  }
	  else { // case II
	    tb0 = SLE_createArcs::OCCLUDE_HI;
	    tb1 = SLE_createArcs::OCCLUDE_LO;
	  }
	}
      }
      else { // not bfina, not bfona
	if( gt0b ) {
	  if( gt1b ) {
	    tb0 = SLE_createArcs::OCCLUDE_HI;
	    tb1 = SLE_createArcs::REVEAL_HI;
	  }
	  else if( eq1b ) {
	    tb0 = SLE_createArcs::OCCLUDE_HI;
	    tb1 = SLE_createArcs::REVEAL_HI;
	  }
	  else {
	    tb0 = SLE_createArcs::OCCLUDE_HI;
	    tb1 = SLE_createArcs::OCCLUDE_LO;
	  }
	}
	else { // lt0b
	  if( gt1b ) {
	    tb0 = SLE_createArcs::OCCLUDE_LO;
	    tb1 = SLE_createArcs::OCCLUDE_HI;
	  }
	  else if( eq1b ) {
	    tb0 = SLE_createArcs::OCCLUDE_LO;
	    tb1 = SLE_createArcs::REVEAL_LO;
	  }
	  else {
	    tb0 = SLE_createArcs::OCCLUDE_LO;
	    tb1 = SLE_createArcs::REVEAL_LO;
	  }
	}
      }

      event_stack.insert(SLE_createArcs(ta0, isx0, &a, &b, tb0));
      event_stack.insert(SLE_createArcs(ta1, isx1, &a, &b, tb1));
      event_stack.insert(SLE_createArcs(tb0, isx0, &b, &a, ta0));
      event_stack.insert(SLE_createArcs(tb1, isx1, &b, &a, ta1));

    } // void Slice::addEvents

    void Slice::clear() {
      vCircles.clear();
      event_stack.clear();
      vArcs.clear();
      vBoundaries.clear();
    } // Slice::clear

    void Slice::addArcs() {
      //vector<Circle*> vc; // centers of Circles to be drawn, all radii 1, let's say...
      //vc = get_input();

      set<Circle*> circle_stack;

      // initialize the event stack
      event_stack.clear();
      for( vector<Circle*>::iterator i = vCircles.begin(); i != vCircles.end(); ++i ) {
	Circle& c = **i;
	//cerr << "Inserting Circle: " << c.getCenter() << ", " << c.getRadius() << endl;
	event_stack.insert(SLE_createArcs(SLE_createArcs::BEGIN, Point2(c.getCenter().x - c.getRadius(), c.getCenter().y), &c, 0, SLE_createArcs::BEGIN));
	event_stack.insert(SLE_createArcs(SLE_createArcs::END,   Point2(c.getCenter().x + c.getRadius(), c.getCenter().y), &c, 0, SLE_createArcs::END));
      } // i

	// sequentially pop the event stack

      vArcs.clear(); // XXX should really delete here

      while( event_stack.size() != 0 ) {

	int occ;
	SLE_createArcs e = *(event_stack.begin());
	event_stack.erase(event_stack.begin());
	//cerr << e << endl;
	switch( e.event_type ) {

	case SLE_createArcs::BEGIN:
	  // calculate intersections with all Circles on the stack
	  occ = 0;
	  for( set<Circle*>::iterator i = circle_stack.begin(); i != circle_stack.end(); ++i ) {
	    Circle& c = **i;
	    double d = Point2::dist(e.p, c.getCenter());
	    if( d < c.getRadius() ) {
	      occ++;
	    }
	    Point2 isx0, isx1;
	    int d_hi_occ, d_lo_occ;
	    if( Circle::intersect(c, *e.c, isx0, isx1) ) {
	      addEvents(isx0, isx1, c, *e.c);
	    }
	  }
	  circle_stack.insert(e.c);
	  e.c->initialize(occ, occ, e.p, e.other, e.other_event_type);
	  break;

	case SLE_createArcs::END:

	  circle_stack.erase(e.c);
	  e.c->finalize(e.p, vArcs, e.other, e.other_event_type);
	  break;

	case SLE_createArcs::OCCLUDE_HI:
	  e.c->occlude_hi(e.p, vArcs, e.other, e.other_event_type);
	  break;

	case SLE_createArcs::OCCLUDE_LO:
	  e.c->occlude_lo(e.p, vArcs, e.other, e.other_event_type);
	  break;

	case SLE_createArcs::REVEAL_HI:
	  e.c->reveal_hi(e.p, vArcs, e.other, e.other_event_type);
	  break;

	case SLE_createArcs::REVEAL_LO:
	  e.c->reveal_lo(e.p, vArcs, e.other, e.other_event_type);
	  break;

	default:
	  cerr << "Erg::Slice::execute - you fucked up" << endl;
	  assert( false );
	} // switch

      } // while event stack

    } // int Slice::addArcs

    void Slice::addBoundaries() {

      FORVPC(i,Arc,vArcs) {
	if( (*i)->getBoundary() == 0 ) {
	  vBoundaries.push_back(new Boundary(*i,this));
	}
      } // i

    } // Slice::addBoundaries

    void Slice::execute(const vector<Atom*>& vAtoms, const double z_) {
      addCircles(vAtoms,z_);
      addArcs();
      addBoundaries();
      setIndices();
    } // Slice::execute

    const double Slice::getArea() const {
      double area = 0;
      FORVPC(i,Boundary,vBoundaries) {
	area += (*i)->getArea();
      }
      return area;
    } // Slice::getArea

    const double Slice::getPerimeter() const {
      double perimeter = 0;
      FORVPC(i,Boundary,vBoundaries) {
	perimeter += (*i)->getPerimeter();
      }
      return perimeter;
    } // Slice::getPerimeter

    void Slice::reportArcs(ostream& out) const {
      FORVPC(i,Arc,vArcs) {
	out << **i << "\n";
      } // i

    } // Slice::reportArcs

    void Slice::reportBoundaries(ostream& out) const {
      FORVPC(i,Boundary,vBoundaries) {
	const Boundary* b = *i;
	out << "Boundary: " << b->getArc0() << "\t" << b->getNumArcs() << "\t" << b->getArea() << "\t" << b->getPerimeter() << "\n";
      }
    } // Slice::reportArcs

    void Slice::report(ostream& out) const {
      reportBoundaries(out);
    } // Slice::report

    const double Slice::getWidth() const {

      // XXX there are probably more accurate ways of dealing with this

      double width = 0;

      if( getPrev() ) {
	width += getZ() - getPrev()->getZ();
      }
      if( getNext() ) {
	width += getNext()->getZ() - getZ();
      }

      return std::abs(width/2.0);

    } // Slice::getWidth

    void Slice::setIndices() {

      vBoundaries_convex.clear();
      vBoundaries_concave.clear();

      for( int i = 0; i < vBoundaries.size(); ++i ) {
	vBoundaries[i]->setIndex(i);
      } // i

      FORVPC(i,Boundary,vBoundaries) {
	Boundary* b = *i;
	if( b->isConvex() ) {
	  //b->setIndex(vBoundaries_convex.size());
	  vBoundaries_convex.push_back(b);
	}
	else {
	  assert( b->isConcave() );
	  //b->setIndex(vBoundaries_concave.size());
	  vBoundaries_concave.push_back(b);
	}
      } // i


	/*
	  const size_t N_concave = vBoundaries_concave.size();
	  cout << "Slice: N_concave = " << N_concave << endl;
	  for( size_t i = 0 ; i < N_concave; ++i ) {
	  cout << "Slice: concave boundary: " << i << ": @" << vBoundaries_concave[i] << ", n_arcs=" << vBoundaries_concave[i]->getNumArcs() << endl;
	  } // i

	  const size_t N_convex = vBoundaries_convex.size();
	  cout << "Slice: N_convex = " << N_convex << endl;
	  for( size_t i = 0 ; i < N_convex; ++i ) {
	  cout << "Slice: convex boundary: " << i << ": @" << vBoundaries_convex[i] << ", n_arcs=" << vBoundaries_convex[i]->getNumArcs() << endl;
	  } // i
	*/

    } // Slice::setIndices

    const vector<Boundary*>& Slice::getBoundaries() const {
      return vBoundaries;
    } // Slice::getBoundaries_convex

    const vector<const Boundary*>& Slice::getBoundaries_convex() const {
      return vBoundaries_convex;
    } // Slice::getBoundaries_convex

    const vector<const Boundary*>& Slice::getBoundaries_concave() const {
      return vBoundaries_concave;
    } // Slice::getBoundaries_concave

      // 	namespace {
      // 	  // ____________________ LinEqnSolver ____________________

      // 	  class LinEqnSolver {
      // 	  public:
      // 	    double a,b,c,d,e,f;

      // 	    const bool solve(double& x, double& y) const;

      // 	  }; // LinEqnSolver

      // 	  const bool LinEqnSolver::solve(double& x, double& y) const {
      // 	    const double det = a*d - b*c;
      // 	    if( det == 0 ) {
      // 	      cout << "not handled" << endl;
      // 	      return false;
      // 	    }
      // 	    else {
      // 	      x = (d*e - b*f) / det;
      // 	      y = (c*e - a*f) / det;
      // 	      return true;
      // 	    }
      // 	  } // LinEqnSolver::solve

      // 	} // namespace

    // ____________________ SLE_createVolumes ____________________

    SLE_createVolumes::SLE_createVolumes(const SLE_createVolumes::Type type_, const Arc* arc_) : type(type_), arc(arc_) {
    } // SLE_createVolumes::SLE_createVolumes

    const Point2& SLE_createVolumes::getPoint() const {
      switch( type ) {
      case SLE_createVolumes::BEGIN:
	return arc->getBegin();
      case SLE_createVolumes::END:
	return arc->getEnd();
      default:
	assert( false );
      } // switch
    } // SLE_createVolumes::getPoint

    const bool SLE_createVolumes::operator<(const SLE_createVolumes& other) const {

      const Point2& p = getPoint();
      const Point2& q = other.getPoint();

      return
	p < q ||
	p == q && (type < other.type ||
		   type == other.type && arc < other.arc );

    } // SLE_createVolumes::operator<

    // ____________________ CreateVolumes ____________________

    const Slice* CreateVolumes::getOtherSlice(const Arc* arc) {
      if( arc->getBoundary()->getSlice() == s_hi ) {
	return s_lo;
      }
      else if( arc->getBoundary()->getSlice() == s_lo ) {
	return s_hi;
      }
      else {
	assert( false );
      }
    } // CreateVolumes::getOtherSlice

    list<const Arc*>& CreateVolumes::getActiveArcs(const Slice* slice) {
      if( slice == s_hi ) {
	return active_hi;
      }
      else if( slice == s_lo ) {
	return active_lo;
      }
      else {
	assert( false );
      }
    } // CreateVolumes

    void CreateVolumes::addOverlap(const Boundary* a, const Boundary* b) {
      const_cast<Boundary*>(a)->addBoundary(b);
      const_cast<Boundary*>(b)->addBoundary(a);
    } // CreateVolumes::addOverlap

    namespace {

      class LessThanAt {
      private:
	const double x;
      public:
	LessThanAt(const double x_) : x(x_) {}
	const bool operator()(const Arc* a, const Arc* b) const {
	  const double ya = a->getChord(x);
	  const double yb = b->getChord(x);
	  return (ya < yb);
	} // operator()

      }; // class CompareAt

      class CompareAt {
      private:
	const double x;
      public:
	CompareAt(const double x_) : x(x_) {}
	const int operator()(const Arc* a, const Arc* b) const {

	  const double TOL = 1e-5;

	  const double ya = a->getChord(x);
	  const double yb = b->getChord(x);

	  if( std::abs(ya-yb) < TOL ) {
	    return 0;
	  }
	  else {
	    return (ya < yb) ? -1 : 1;
	  }

	} // operator()

      }; // class CompareAt

    } // namespace

    void CreateVolumes::isContained(const Arc* arc) {

      const Slice* other_slice = getOtherSlice(arc);

      const size_t N = other_slice->getBoundaries().size();

      vector<int> v_lt,v_gt,num_arcs;
      v_lt.resize(N);
      v_gt.resize(N);
      num_arcs.resize(N);
      for( int i = 0; i < N; ++i ) {
	v_lt[i] = 0;
	v_gt[i] = 0;
	num_arcs[i] = 0;
      } // i

      const list<const Arc*>& active = getActiveArcs(other_slice);

      CompareAt cmp(sle.getPoint().x);

      for( list<const Arc*>::const_iterator i = active.begin(); i != active.end(); ++i ) {

	const Arc* other_arc = *i;

	++num_arcs[other_arc->getBoundary()->getIndex()];

	switch( cmp(arc,other_arc) ) {

	case -1:
	  ++v_lt[other_arc->getBoundary()->getIndex()];
	  break;

	case 1:
	  ++v_gt[other_arc->getBoundary()->getIndex()];
	  break;

	case 0:
	  addOverlap(arc->getBoundary(),other_arc->getBoundary());
	  ++v_gt[other_arc->getBoundary()->getIndex()];
	  break;

	default:
	  assert( false );
	}

      } // i

      // Step 2 - for each time this arc is gt/lt other arcs an even number of times, it is contained in that boundary

      for( size_t i = 0; i < N; ++i ) {

	assert( (v_lt[i] + v_gt[i]) % 2 == 0 );

	if( v_lt[i] % 2 != 0 ) {
	  // contained
	  //cout << "Contained" << endl;
	  addOverlap(arc->getBoundary(),other_slice->getBoundaries()[i]);
	}

      } // i

    } // CreateVolumes::isContained

    void CreateVolumes::addActive(const Arc* arc) {

      // need to do two things:
      // 1 - test whether this point is contained in any of the other boundaries
      // 2 - test whether this point intersects any other active line segment
      // if either of the above is true, then register that one boundary is contained in the other boundary
      // then finally, need to add this arc to the active arcs for this slice

      isContained(arc);
      //isIntersected(arc); !!! not doing the 2nd part it won't help that much

      list<const Arc*>& active = getActiveArcs(arc->getBoundary()->getSlice());
      active.push_back(arc);


    } // CreateVolumes::addActive

    void CreateVolumes::removeActive(const Arc* arc) {

      // need to do one thing:
      // 1 - test whether this point is contained in any of the other boundaries
      // if the above is true, then register that one boundary is contained in the other boundary
      // then finally, need to remove this arc from the active arcs for this slice

      isContained(arc);

      list<const Arc*>& active = getActiveArcs(arc->getBoundary()->getSlice());
      list<const Arc*>::iterator iter = find(active.begin(),active.end(),arc);
      assert( iter != active.end() );
      active.erase(iter);

    } // CreateVolumes::removeActive

    void CreateVolumes::addEvents(const vector<const Boundary*>& v) {
      FORVPC(i,const Boundary,v) {
	const Boundary* b = *i;
	const Arc* arc = b->getArc0();
	do {
	  event_stack.insert( SLE_createVolumes(SLE_createVolumes::BEGIN,arc));
	  event_stack.insert( SLE_createVolumes(SLE_createVolumes::END,arc));

	  arc = arc->getNext();
	} while ( arc != b->getArc0());

      } // i
    } // CreateVolumes

    void CreateVolumes::execute(const Slice* s_hi_, const Slice* s_lo_ ) {

      s_hi = s_hi_;
      s_lo = s_lo_;

      // Step 0 - clear all data

      event_stack.clear();

      // Step 1 - fill the event stack

      addEvents(s_hi->getBoundaries_concave());
      addEvents(s_hi->getBoundaries_convex());
      addEvents(s_lo->getBoundaries_concave());
      addEvents(s_lo->getBoundaries_convex());

      // Step 2 - pop the event stack until empty

      while( event_stack.size() != 0 ) {

	// pop the stack
	sle = *(event_stack.begin());
	event_stack.erase(event_stack.begin());

	// process
	switch( sle.type ) {

	case SLE_createVolumes::BEGIN:
	  addActive(sle.arc);
	  break;
	case SLE_createVolumes::END:
	  removeActive(sle.arc);
	  break;
	default:
	  assert( false );
	} // switch

      } // while


    } // CreateVolumes::execute

      // ____________________ Volume ____________________

    Volume::Volume(const Boundary* b0) {

      assert( b0 != 0 );

      set<const Boundary*> sBoundaries;
      set<const Boundary*> frontier;

      frontier.insert(b0);

      while( frontier.size() != 0 ) {

	set<const Boundary*> frontier_next;
	FORSPC(i,const Boundary,frontier) {
	  const Boundary* b = *i;

	  assert( !b->isIgnorable() );
	  assert( b->getVolume() == 0 );

	  const_cast<Boundary*>(b)->setVolume(this);
	  sBoundaries.insert(b);

	  const set<const Boundary*>& sBoundaries_b = b->getBoundaries();

	  // 	    if( iter == msOverlaps.end() ) {
	  // 	      cout << "Volume::Volume - WARNING - no overlaps for boundary " << b << endl;
	  // 	    }
	  // 	    else {
	  // 	      cout << "Volume::Volume - VERIFIED " << " there are some overlaps for boundary " << b << endl;
	  // 	    }

	  FORSPC(j,const Boundary,sBoundaries_b) {
	    const Boundary* b_next = *j;

	    assert( b_next->getSlice() != b->getSlice() );
	    //cout << "Volume::Volume - VERIFIED " << REPORT1(b_next) << endl;

	    if( b_next->getVolume() == 0 && !b_next->isIgnorable() && b_next->isSame(b0) ) {
	      frontier_next.insert(b_next);
	    }

	  }  // j

	} // i


	frontier = frontier_next;
      } // while

      FORSPC(i,const Boundary,sBoundaries) {
	const Boundary* b = *i;
	mvBoundaries[b->getSlice()].push_back(b);
      } // i

    } // Volume::Volume

    Volume::~Volume() {
    } // Volume::~Volume

    const double Volume::getSurfaceArea() const {
      double rval = 0;
      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	const Slice* slice = i->first;
	const vector<const Boundary*>& vBoundaries = i->second;

	const double width = slice->getWidth();

	FORVPC(j,const Boundary,vBoundaries) {
	  const Boundary* b = *j;
	  rval += width*b->getPerimeter();
	} // j

      } // i
      return rval;
    } // Volume::getSurfaceArea

    const double Volume::getVolume() const {
      double rval = 0;
      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	const Slice* slice = i->first;
	const vector<const Boundary*>& vBoundaries = i->second;

	const double width = slice->getWidth();

	FORVPC(j,const Boundary,vBoundaries) {
	  const Boundary* b = *j;
	  rval += width*b->getArea();
	} // j
      } // i
      return rval;
    } // Volume::getVolume

    void Volume::reportArcs(ostream& out) const {

      int min_index = INT_MAX;

      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	min_index = min(min_index,i->first->getIndex());
      }


      out << "% vol " << getVolume() << endl;
      out << "% sa " << getSurfaceArea() << endl;

      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	const Slice* slice = i->first;
	const vector<const Boundary*>& vBoundaries = i->second;

	out << "% slice#" << slice->getIndex() << endl;

	const double d = static_cast<double>(1 + slice->getIndex() - min_index) / (mvBoundaries.size()+1); // where do i really get these numbers

	if( getVolume() > 0 ) {
	  out << d << " " << d << " 0 setrgbcolor\n";
	}
	else {
	  out << d << " 0 " << d << " setrgbcolor\n";
	}

	FORVPC(j,const Boundary,vBoundaries) {
	  (*j)->reportArcs(out);
	} // j
      } // i
    } // Volume::report

    const bool Volume::isConcave() const {
      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	const vector<const Boundary*> vBoundaries = i->second;
	FORVPC(j,const Boundary,vBoundaries) {
	  if( (*j)->isConvex() ) {
	    return false;
	  }
	} // j
      } // i
      return true;
    } // Volume::isConcave

    const bool Volume::isConvex() const {
      FORMC(i,const Slice*,vector<const Boundary*>,mvBoundaries) {
	const vector<const Boundary*> vBoundaries = i->second;
	FORVPC(j,const Boundary,vBoundaries) {
	  if( (*j)->isConcave() ) {
	    return false;
	  }
	} // j
      } // i
      return true;
    } // Volume::isConvex

  } // namespace LeeRichardsInfo


    // ____________________ LeeRichards ____________________

//   void LeeRichards::addAtoms() {

//     const double TOL = 1;

//     z_lo = HUGE;
//     z_hi = -HUGE;

//     for( Tube::iterator i = getTube()->begin(); i != getTube()->end(); ++i ) {
//       const Residue* r = *i;
//       for( Residue::const_iterator j = r->begin(); j != r->end(); ++j ) {
// 	const Atom* a = *j;

// 	//if( !a->isHydrogen() ) {

// 	LeeRichardsInfo::Atom* lr_atom = new LeeRichardsInfo::Atom(a);

// 	z_lo = min(z_lo,lr_atom->getPosn()[2] - lr_atom->getRadius() - TOL);
// 	z_hi = max(z_hi,lr_atom->getPosn()[2] + lr_atom->getRadius() + TOL);

// 	vAtoms.push_back(lr_atom);

// 	//}
//       } // j
//     } // i
//   } // LeeRichards::addAtoms

  void LeeRichards::addAtoms(const vector<LeeRichardsInfo::Atom*>& vAtoms_) {

    const double TOL = 1;

    clear();

    z_lo = HUGE;
    z_hi = -HUGE;

    FORVPC(i,LeeRichardsInfo::Atom,vAtoms_) {

      LeeRichardsInfo::Atom* lr_atom = *i;

      vAtoms.push_back(lr_atom);

      z_lo = min(z_lo,lr_atom->getPosn()[2] - lr_atom->getRadius() - TOL);
      z_hi = max(z_hi,lr_atom->getPosn()[2] + lr_atom->getRadius() + TOL);

    } // i

  } // LeeRichards::addAtoms

  void LeeRichards::clear() {
    FORVPC(i,LeeRichardsInfo::Atom,vAtoms) {
      delete *i;
    } // i
      // same for slices & volumes
  } // LeeRichards::clear

//   LeeRichards::LeeRichards(const Tube* tube) : Tubeable(tube) {
//     addAtoms();
//   } // LeeRichards::LeeRichards

  LeeRichards::LeeRichards(const vector<LeeRichardsInfo::Atom*>& vAtoms_) {
    addAtoms(vAtoms_);
  } // LeeRichards::LeeRichards

  LeeRichards::~LeeRichards() {
    clear();
  } // LeeRichards::~LeeRichards

  void LeeRichards::addSlices(const int n ) {
    using namespace LeeRichardsInfo;

    const double dz = ( z_hi - z_lo ) / (n+1);

    assert( vSlices.size() == 0 );
    Slice* prev = 0;

    for( double z = z_lo; z < z_hi; z += dz ) {

      //cout << "addSlices z =" << z << endl;

      Slice* slice = new Slice();

      slice->execute(vAtoms,z);

      slice->setIndex(vSlices.size());
      vSlices.push_back(slice);

      if( prev != 0 ) {
	slice->setPrev(prev);
	prev->setNext(slice);
      }

      prev = slice;

    } // z
  } // LeeRichards::addSlices

  void LeeRichards::addSlices_z(const double z ) {
    using namespace LeeRichardsInfo;

    Slice* prev = 0;
    Slice* slice = new Slice();

    slice->execute(vAtoms,z);

    slice->setIndex(vSlices.size());
    vSlices.push_back(slice);

    if( prev != 0 ) {
      slice->setPrev(prev);
      prev->setNext(slice);
    }

    prev = slice;

  } // LeeRichards::addSlices


  void LeeRichards::addVolumes() {

    using namespace LeeRichardsInfo;
    //using namespace LeeRichardsInfo::OverlapInfo;

    CreateVolumes create_volumes;

    for( size_t k = 1; k < vSlices.size(); ++k ) {

      Slice* s_hi = vSlices[k-1];
      Slice* s_lo = vSlices[k];
      create_volumes.execute(s_hi,s_lo);

    } // k

    FORVPC(i,Slice,vSlices) {
      const Slice* slice = *i;

      const vector<Boundary*> vBoundaries = slice->getBoundaries();
      FORVPC(j,Boundary,vBoundaries) {

	const Boundary* b = *j;
	if( !b->isIgnorable() && b->getVolume() == 0 ) {

	  Volume* v = new Volume(b);
	  vVolumes.push_back(v);

	}
      } // j
    } // i

  } // LeeRichards::addVolumes

  void LeeRichards::execute(const int n) {
    addSlices(n);
    addVolumes();
  } // LeeRicharsd::execute

  void LeeRichards::execute_z(const double z) {
    addSlices_z(z);
    //addVolumes();
  } // LeeRicharsd::execute

  void LeeRichards::reportArcs(ostream& out) const {
    using namespace LeeRichardsInfo;
    //Directory dir(".");

    FORVPC(i,Slice,vSlices) {
      const Slice* slice = *i;

      //const string outfile = dir.getNextFilename("slice",4,"ps");

//       ofstream fout;
//       open(fout,outfile);
//       slice->reportArcs(fout);
//       fout.close();

    } // i

  } // LeeRichards::report

  void LeeRichards::reportBoundaries(ostream& out) const {
    using namespace LeeRichardsInfo;

    FORVPC(i,Slice,vSlices) {
      const Slice* slice = *i;
      out << "==================== slice @ z=" << slice->getZ() << " ====================" << endl;
      slice->reportBoundaries(out);
    } // i

  } // LeeRichards::reportBoundaries

  void LeeRichards::reportSlices(ostream& out) const {
    using namespace LeeRichardsInfo;

    FORVPC(i,Slice,vSlices) {
      const Slice* slice = *i;
      out << "==================== slice @ z=" << slice->getZ() << " ====================" << endl;
      slice->report(out);
    }

  } // LeeRichards::reportSlices

  void LeeRichards::reportVolumes(ostream& out) const {
    using namespace LeeRichardsInfo;

    cout << "vVolumes.size() = " << vVolumes.size() << endl;

    //Directory dir(".");

    FORVPC(i,Volume,vVolumes) {
      const Volume* volume = *i;

      const bool b_convex  = volume->isConvex();
      const bool b_concave = volume->isConcave();

      const double vol = volume->getVolume();
      const double sa  = volume->getSurfaceArea();

      //cout << "Volume - " << REPORT4(b_convex,b_concave,vol,sa) << "\t";
      cout << "Volume - " << "b_convex " << b_convex << "\tb_concave " << b_concave << "\tvol " << vol << "\tsa " << sa << "\t";
      if( b_convex && b_concave ) {
	cout << "BOTH!!!";
      }
      else if( !b_convex && !b_concave ) {
	cout << "NEITHER!!!";
      }
      cout << endl;

      // 	if( vol > 1 || vol < -1 ) {
      // 	  const string outfile = dir.getNextFilename("volume",4,"ps");
      // 	  ofstream fout;
      // 	  open(fout,outfile);
      // 	  volume->reportArcs(fout);
      // 	  fout.close();
      // 	}


    } // i

  } // LeeRichards::reportVolumes

  const double LeeRichards::getPerimeter() const {
    using namespace LeeRichardsInfo;

    double rval = 0;

    FORVPC(i,Slice,vSlices) {
      rval += (*i)->getPerimeter();
    } // i

    return rval;

  } // LeeRichards::getSurfaceArea

} // namespace murphp

