// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20171 $
//  $Date: 2008-02-05 16:25:00 -0500 (Tue, 05 Feb 2008) $
//  $Author: monica $


// Rosetta Headers
#include "make_pdb.h"
#include "aa_name_conversion.h"
#include "after_opts.h"
#include "aaproperties_pack.h"
#include "cenlist.h"
#include "constraints.h"
#include "counters.h"
#include "decoystats.h"
#include "decoystats_classes.h"
#include "decoy_features.h"
#include "design.h"
#include "dipolar.h"
#include "dna.h"
#include "dna_ns.h"
#include "dock_pivot.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_ns.h"
#include "domins_ns.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "force_barcode.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "hbonds_ns.h"
#include "initialize.h"
#include "int_fullatom_energies.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "loops.h"
#include "loop_relax.h"
#include "maxsub.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "native.h"
#include "orient_rms.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_pack.h"
#include "param_aa.h"
#include "pdb.h"
#include "pdbstatistics_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pose_ligand.h"
#include "add_pser.h" // need to know if pser mode  in output_residueE_partII()
#include "ramachandran.h"
#include "read_aaproperties.h"
#include "relax.h"
#include "score_ns.h"
#include "start.h"
#include "taboo_search.h"
#include "template_pack.h"
#include "util_basic.h"
#include "void.h"
#include "void_ns.h"
#include "water.h"
#include "enzyme.h"
#include "analyze_interface_ddg_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <string>
#include <fstream>
#include <vector>


// this namespace provides a uniform way to set the bfac, occ and hetero atoms that
// will be included by make_pdb. this is hackish, but less hackish than the current
// method of having "if ( mymode ) load_my_bactors" type statements in make_pdb for
// everybodys custom mode.
namespace make_pdb_ns {
	using namespace param;
	std::string nonscored_hetero_atoms = ""; // any additional "hetero" atoms
  	// Note: these are not real hetero atoms for modeling purposes, they just show up in output PDBs
	FArray2D_float bfactor( MAX_ATOM(), MAX_RES(), 0. );
	FArray2D_float occupancy( MAX_ATOM(), MAX_RES(), 1. );
}

bool get_remark_option(){
	static bool first = true;
	static bool remark_flag = false;
	if(first){
		first = false;
		remark_flag = truefalseoption("remark_output");
		return remark_flag;
	}
	else{
		return remark_flag;
	}
}


//car general purposes functions for making pdb files
//car   dump_(fullatom)_pdb    :  output coordinates (either the position
//car                             array or the fullcoord array) to a given
//car                             filename; handles file opening and closing
//car                             use for debugging
//car   output_<array_name>_pdb:  write the specifed array in pdb format
//car                             to a given file unit, coordinates only
//car                             open file units are required, not closed
//car   output_fullatom_pdb    :  write any fullatom coordinate array to
//car                             a specified fileunit in pdb format
//car                             coord, sequence etc req'd as arguments
//car  make_pdb                :  output coordinates, scores, angles to open
//car                             file unit, returns closed unit
//car-----------------------------------------------------------------------------------------------
//car   DO NOT OPEN PDB FILES WITH 'open_pdb_decoy' !!!!

////////////////////////////////////////////////////////////////////////////////
/// @begin dump_pdb
///
/// @brief
///car write position array to a file in pdb format
///
/// @detailed
///
/// @param  filename - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dump_pdb( std::string const & filename )
{

	using namespace files_paths;

	bool const save_option = use_pdb_numbering;
	use_pdb_numbering = false;
	std::cout << "dump_pdb: " << filename << std::endl;

	utility::io::ozstream debug_xstream( filename );

	if ( !debug_xstream ) {
		std::cout << "Open failed for file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	output_position_pdb( debug_xstream );
	output_torsion_angles( debug_xstream );
	output_pdb_stats( debug_xstream, false );
//output_dna_params( debug_xstream );
	debug_xstream.close();
	debug_xstream.clear();
	use_pdb_numbering = save_option;
}

void
dump_native_pdb( std::string const & filename )
{

	using namespace files_paths;

	bool const save_option = use_pdb_numbering;
	use_pdb_numbering = false;
	std::cout << "dump_pdb: " << filename << std::endl;

	utility::io::ozstream debug_xstream( filename );

	if ( !debug_xstream ) {
		std::cout << "Open failed for file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	output_native_pdb( debug_xstream );
	output_torsion_angles( debug_xstream );
	output_pdb_stats( debug_xstream, false );
//output_dna_params( debug_xstream );
	debug_xstream.close();
	debug_xstream.clear();
	use_pdb_numbering = save_option;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dump_fullatom_pdb
///
/// @brief
///car write the fullcoord array to a file in pdb format
///
/// @detailed
///
/// @param  filename - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dump_fullatom_pdb( std::string const & filename )
{

	using namespace files_paths;

	std::cout << "dump_fullatom_pdb: " << filename << std::endl;

	utility::io::ozstream debug_xstream( filename );

	if ( !debug_xstream ) {
		std::cout << "Open failed for file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	output_fullcoord_pdb( debug_xstream );
	output_torsion_angles( debug_xstream );
	output_pdb_stats( debug_xstream, true );
//output_dna_params( debug_xstream );
	debug_xstream.close();
	debug_xstream.clear();
}


//////////////////////////////////////////////////////////////////////////////
/// @begin atom_initializer
///
/// @brief
///
/// @detailed
///
/// @param  atom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
atom_initializer( FArray1D_string & atom )
{
	atom( 1 ) = " N  ";
	atom( 2 ) = " CA ";
	atom( 3 ) = " CB ";
	atom( 4 ) = " C  ";
	atom( 5 ) = " O  ";
	atom( 6 ) = " CEN"; // CEN not used
}


//////////////////////////////////////////////////////////////////////////////
/// @begin make_pdb_reorder_initializer
///
/// @brief
///
/// @detailed
///
/// @param  reorder - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
make_pdb_reorder_initializer( FArray1D_int & reorder )
{
	reorder( 1 ) = 1;
	reorder( 2 ) = 2;
	reorder( 3 ) = 4;
	reorder( 4 ) = 5;
	reorder( 5 ) = 3;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_position_pdb
///
/// @brief
///car write out position array in pdb format
///
/// @detailed
///car option dependencies:
///car  use_pdb_numbering  - restore numbering/chains of input pdb file
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_position_pdb( utility::io::orstream & iunit )
{
	using namespace files_paths;
	using namespace misc;
	using namespace native;
	using namespace param;
	using namespace pdb;

	static FArray1D_string const atom( 6, atom_initializer );
	float occ,bfactor;
	int res_number,atom_number;
	char chain_out = ' '; // for output, depends on use_pdb_numbering
	int res_number_out; // ditto
	char insert_let_out = ' '; // ditto
	int atom_number_out = 0; // counter for atom numbers

	static FArray1D_int const reorder( 5, make_pdb_reorder_initializer );
	// array to rearrange backbone atoms

	if ( !use_pdb_numbering ) {
		insert_let_out = ' ';
		chain_out = protein_chain;
		if ( chain_out == '_' ) chain_out = ' ';
	}

	int idomain = 1;
	for ( res_number = 1; res_number <= total_residue; ++res_number ) {
		if ( get_native_exists() ) {
			bfactor = ObjexxFCL::max(
			 torsion_diff(native_phi(res_number),phi(res_number)),
			 torsion_diff(native_psi(res_number),psi(res_number)),
			 torsion_diff(native_omega(res_number),omega(res_number)))/45.0;
		} else {
			bfactor = 0.0;
		}
		loop_set_bfactor(bfactor,res_number);
		looprlx_set_bfactor(bfactor,res_number,1);

		if ( res_number == 1 && psi(res_number) == init_psi &&
		 omega(res_number) == init_omega ) {
			occ = -1.0;
		} else if ( res_number == total_residue && phi(res_number) == init_phi ) {
			occ = -1.0;
		} else if ( phi(res_number) == init_phi && psi(res_number) == init_psi &&
		 omega(res_number) == init_omega ) {
			occ = -1.0;
		} else {
			occ = 1.0;
		}
		looprlx_set_occupancy(occ, res_number,1);

		if ( use_pdb_numbering ) {
			res_number_out = pdb_res_num(res_number);
			insert_let_out = pdb_insert_let(res_number);
			chain_out = res_chain(res_number);
		} else {
			res_number_out = res_number;
		}

		for ( int i = 1; i <= 5; ++i ) { // loop over atoms in each residue
			atom_number = reorder(i);
			++atom_number_out;
			if ( !( residue3(res_number) == "GLY" && atom_number == 3 ) ) {
			 iunit << "ATOM  " << I( 5, atom_number_out ) << ' ' <<
			 A( 4, atom(atom_number) ) << ' ' << residue3(res_number) << ' ' <<
			 chain_out << I( 4, res_number_out ) << insert_let_out << "   " <<
			 F( 8, 3, Eposition(1,atom_number,res_number) ) <<
			 F( 8, 3, Eposition(2,atom_number,res_number) ) <<
			 F( 8, 3, Eposition(3,atom_number,res_number) ) <<
			 F( 6, 2, occ ) << F( 6, 2, bfactor ) << std::endl;
       }
		}                  // atoms

		if (output_centroids){
			++atom_number_out;
			iunit << "ATOM  " << I( 5, atom_number_out ) << ' ' <<
				A( 4, " CEN" ) << ' ' << residue3(res_number) << ' ' <<
				chain_out << I( 4, res_number_out ) << insert_let_out << "   " <<
				F( 8, 3, centroid(1,res_number) ) <<
				F( 8, 3, centroid(2,res_number) ) <<
				F( 8, 3, centroid(3,res_number) ) <<
				F( 6, 2, occ ) << F( 6, 2, bfactor ) << std::endl;
		}

		if ( multi_chain && res_number == domain_end(idomain) ) {
			iunit << "TER" << std::endl;
			++idomain;
		}

	} // residues
	if ( !multi_chain ) iunit << "TER" << std::endl;

//                     23    CA    GLY   L  99 A      xyz  occ  bfactor
//                   atom#  atom   res  ch res# ins  coords
}


void
output_native_pdb( utility::io::orstream & iunit )
{
	using namespace files_paths;
	using namespace misc;
	using namespace native;
	using namespace param;
	using namespace pdb;

	static FArray1D_string const atom( 6, atom_initializer );
	float occ,bfactor;
	int res_number,atom_number;
	char chain_out = ' '; // for output, depends on use_pdb_numbering
	int res_number_out; // ditto
	char insert_let_out = ' '; // ditto
	int atom_number_out = 0; // counter for atom numbers

	static FArray1D_int const reorder( 5, make_pdb_reorder_initializer );
	// array to rearrange backbone atoms

	if ( !use_pdb_numbering ) {
		insert_let_out = ' ';
		chain_out = protein_chain;
		if ( chain_out == '_' ) chain_out = ' ';
	}

	int idomain = 1;
	for ( res_number = 1; res_number <= total_residue; ++res_number ) {
		if ( get_native_exists() ) {
			bfactor = ObjexxFCL::max(
			 torsion_diff(native_phi(res_number),phi(res_number)),
			 torsion_diff(native_psi(res_number),psi(res_number)),
			 torsion_diff(native_omega(res_number),omega(res_number)))/45.0;
		} else {
			bfactor = 0.0;
		}
		loop_set_bfactor(bfactor,res_number);

		if ( res_number == 1 && psi(res_number) == init_psi &&
		 omega(res_number) == init_omega ) {
			occ = -1.0;
		} else if ( res_number == total_residue && phi(res_number) == init_phi ) {
			occ = -1.0;
		} else if ( phi(res_number) == init_phi && psi(res_number) == init_psi &&
		 omega(res_number) == init_omega ) {
			occ = -1.0;
		} else {
			occ = 1.0;
		}

		if ( use_pdb_numbering ) {
			res_number_out = pdb_res_num(res_number);
			insert_let_out = pdb_insert_let(res_number);
			chain_out = res_chain(res_number);
		} else {
			res_number_out = res_number;
		}

		for ( int i = 1; i <= 5; ++i ) { // loop over atoms in each residue
			atom_number = reorder(i);
			++atom_number_out;
			if ( !( residue3(res_number) == "GLY" && atom_number == 3 ) )
			 iunit << "ATOM  " << I( 5, atom_number_out ) << ' ' <<
			 A( 4, atom(atom_number) ) << ' ' << residue3(res_number) << ' ' <<
			 chain_out << I( 4, res_number_out ) << insert_let_out << "   " <<
			 F( 8, 3, native_Eposition(1,atom_number,res_number) ) <<
			 F( 8, 3, native_Eposition(2,atom_number,res_number) ) <<
			 F( 8, 3, native_Eposition(3,atom_number,res_number) ) <<
			 F( 6, 2, occ ) << F( 6, 2, bfactor ) << std::endl;
		}                  // atoms

		if ( multi_chain && res_number == domain_end(idomain) ) {
			iunit << "TER" << std::endl;
			++idomain;
		}

	} // residues
	if ( !multi_chain ) iunit << "TER" << std::endl;

//                     23    CA    GLY   L  99 A      xyz  occ  bfactor
//                   atom#  atom   res  ch res# ins  coords
}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_pdb
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? - file stream to which to write
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car file must be opened before calling
///car write position arrays, scores, and torsion angles to final output pdb file
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_pdb(
	utility::io::ozstream & iunit, // file stream to which to write
	bool fullatom
)
{
	using namespace design;

	if ( fullatom ) output_fullcoord_pdb( iunit );
	else output_position_pdb( iunit );

	for( std::vector< Ligand * >::iterator
   c_ligand=ligand::ligand_ptr_vector.begin();
	 c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++ ){
		make_pdb_hetero( iunit, (*(*c_ligand)) );
	}
	make_pdb_water( iunit );

	output_pdb_stats( iunit, fullatom );
	if ( fullatom )	output_chi( iunit );
	output_torsion_angles( iunit );

//if ( dna_interface ) output_dna_params( iunit );

// SJF cat_pdb_writer will close the buffer
//	iunit.close();
//	iunit.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin make_pdb_orstream
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? - stream to which to write
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// mtyka - this is a clone of the normal make_pdb but this one can write
//          to a string stream
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_pdb_orstream(
	utility::io::orstream & iunit, // stream to which to write
	bool fullatom
)
{
	using namespace design;

	if ( fullatom ) output_fullcoord_pdb( iunit );
	else output_position_pdb( iunit );

	for( std::vector< Ligand * >::iterator
	 c_ligand=ligand::ligand_ptr_vector.begin();
	 c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++ ){
		make_pdb_hetero( iunit, (*(*c_ligand)) );
	}
	make_pdb_water( iunit );

	output_pdb_stats( iunit, fullatom );
	if ( fullatom )	output_chi( iunit );
	output_torsion_angles( iunit );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_torsion_angles
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? - stream to which to write
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_torsion_angles( utility::io::orstream & iunit )
{
	using namespace cenlist_ns;
	using namespace files_paths;
	using namespace misc;
	using namespace pdb;
	using namespace start;
	using namespace native;

	if ( get_docking_flag() && !get_flexbb_docking_flag() ) return;

	float rama,tmp1,tmp2;
	FArray1D_float tmp_phi = misc::phi;
	FArray1D_float tmp_psi = misc::psi;
	FArray1D_float tmp_omega = misc::omega;
	tmp_phi(1) = 0.0f;
	for ( int i = 1; i < total_domains; i++ ) {
		tmp_psi(domain_end(i)) = 0.0f;
		tmp_omega(domain_end(i)) = 0.0f;
		tmp_phi(domain_end(i)+1) = 0.0f;
	}
	tmp_psi(domain_end(total_domains)) = 0.0f;
	tmp_omega(domain_end(total_domains)) = 0.0f;

//car 'complete' is the tag searched for to read in torsion angles
//car from rosetta files and should be the last line before the angles begin
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "complete: res ss phi psi omega frag rama seq dstart_tor";
	if ( get_native_exists() ) {
		iunit << " dnat_tor";
	}
	if ( output_cen_vals ) {
		iunit << " cen6 cen10";
	}
  iunit << std::endl;
	for ( int i = 1; i <= total_residue; ++i ) {
		int ii = i;
		if ( use_pdb_numbering ) ii = pdb_res_num(i);
		eval_rama_score_residue(res(i),tmp_phi(i),tmp_psi(i),secstruct(i),rama,
			tmp1,tmp2);
		if(get_remark_option()) iunit << "REMARK ";
		iunit << I( 4, ii ) << ' ' << secstruct(i) << ' ' << F( 9, 3, tmp_phi(i) )
					<< F( 9, 3, tmp_psi(i) ) << F( 9, 3, tmp_omega(i) ) << ' ' << name(i)
					<< ' ' << F( 9, 2, rama ) << ' ' << residue1(i)
					<< F( 9, 3, std::abs(subtract_degree_angles(tmp_phi(i),start_phi(i)))
						+ std::abs(subtract_degree_angles(tmp_psi(i),start_psi(i)))
						+ std::abs(subtract_degree_angles(tmp_omega(i),start_omega(i))) );
		if ( get_native_exists() ) {
			iunit <<
				F( 9, 3, std::abs(subtract_degree_angles(tmp_phi(i),native_phi(i))) +
					std::abs(subtract_degree_angles(tmp_psi(i),native_psi(i))) +
					std::abs(subtract_degree_angles(tmp_omega(i),native_omega(i))));
		}
		if ( output_cen_vals ) {
			iunit << " " << cen6(i) << " " << cen10(i);
		}
		iunit << std::endl;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_dna_params
///
/// @brief
/// this function outputs effective params for dna basesteps, basepairs, and dihedrals
/// @detailed
///
/// @param  iunit - [in/out]? - stream to which to write
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// Alex Morozov
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_dna_params( utility::io::orstream & iunit )
{
	using namespace dna_variables;
	using namespace misc;
	using namespace param_aa;
	using namespace pdb;
	using namespace files_paths;

	if ( !design::dna_interface ) return; // nothing to do if no dna/rna found

	/* Basestep output */
	// Header
	iunit << "DNA basesteps:\n";
	iunit << A(4,"pos") << "  " << A(9,"twist") << A(9,"roll") << A(9,"tilt") <<
	 A(9,"shift") << A(9,"slide") << A(9,"rise") << std::endl;
	// Params
	for ( int i = 1; i < total_residue; ++i ) {
		if ( !is_DNA(res(i)) ) continue;
		int ii = i;
		if ( use_pdb_numbering ) ii = pdb_res_num(i);
		 // both ith and (i+1)th residues are basepaired, params exist
//		if ( basepair_map[i] && basepair_map[i+1] != -1 ) {
			iunit << I( 4, ii ) << ' ' << residue3(i).substr(2,1) <<
			 residue3(i+1).substr(2,1) << ' ' << F( 9, 3, basestep_prms(1,i) ) <<
			 F( 9, 3, basestep_prms(2,i) ) << F( 9, 3, basestep_prms(3,i) ) <<
			 F( 9, 3, basestep_prms(4,i) ) << F( 9, 3, basestep_prms(5,i) ) <<
			 F( 9, 3, basestep_prms(6,i) ) << std::endl;
//		}
	}
	/* Basepair output */
	// Header
	iunit << "DNA basepairs:\n";
	iunit << A(4,"pos") << " " << A(9,"propeller") << A(9,"buckle") <<
	 A(9,"opening") << A(9,"shear") << A(9,"stretch") << A(9,"stagger") <<
	 std::endl;
	 // Params
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !is_DNA(res(i)) ) continue;
		int ii = i;
		if ( use_pdb_numbering ) ii = pdb_res_num(i);
//		if ( basepair_map[i] ) {
			iunit << I( 4, ii ) << ' ' << residue3(i).substr(2,1) << ' ' <<
			 F( 9, 3, basepair_prms(1,i) ) << F( 9, 3, basepair_prms(2,i) ) <<
			 F( 9, 3, basepair_prms(3,i) ) << F( 9, 3, basepair_prms(4,i) ) <<
			 F( 9, 3, basepair_prms(5,i) ) << F( 9, 3, basepair_prms(6,i) ) <<
			 std::endl;
//		}
	}
	/* DNA dihedral output */
	// Header
	iunit << "DNA dihedrals:\n";
	iunit << A(4,"pos") << " " << A(9,"alpha") << A(9,"beta") << A(9,"gamma") <<
	 A(9,"delta") << A(9,"epsilon") << A(9,"xi") << A(9,"chi") << std::endl;
	// Params
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( !is_DNA(res(i)) ) continue;
		int ii = i;
		if ( use_pdb_numbering ) ii = pdb_res_num(i);
		if ( dna_dihedrals(1,i) != empty_value && dna_dihedrals(2,i) != empty_value && dna_dihedrals(3,i) != empty_value ) {
			iunit << I( 4, ii ) << ' ' << residue3(i).substr(2,1) << ' ' <<
			F( 9, 3, dna_dihedrals(1,i) ) << F( 9, 3, dna_dihedrals(2,i) ) <<
			F( 9, 3, dna_dihedrals(3,i) ) << F( 9, 3, dna_dihedrals(4,i) ) <<
			F( 9, 3, dna_dihedrals(5,i) ) << F( 9, 3, dna_dihedrals(6,i) ) <<
			F( 9, 3, dna_dihedrals(7,i) ) << std::endl;
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_pdb_stats
///
/// @brief
///
/// @detailed
///
/// @param  iunit - [in/out]? - file to which to write
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_pdb_stats(
	utility::io::orstream & iunit,
	bool fullatom
)
{
	using namespace counters;
	using namespace design;
	using namespace dna_variables;
	using namespace docking;
	using namespace domins_ns;
	using namespace files_paths;
	using namespace ligand;
	using namespace misc;
	using namespace native;
	using namespace param_pack;
	using namespace pH_ns;
	using namespace scores;
	using namespace start;
	using namespace scorefxns;

	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal

	int const max_loop = { 1000 };

	float fast_rms;
	float rms_to_start;
	int nalign, lig_contacts, don_sat, don_non, acc_sat, acc_non;
	float maxsub_rms,log_evalue;
	int num_loop;
	FArray1D_float splicerms( max_loop );

	if ( ntrials == 0 ) ntrials = 1;

	if ( get_docking_silent_input_flag() ) {
		iunit << A( 15, "tag_name:")
					<< docking_silent_input_get_current_tag(false) << "\n";
	}
	if ( get_pH_packing_flag() ) {
		iunit << A( 15, "pH:" ) << F( 11, 3, pH ) << '\n'; //std::endl;
	}
	iunit << A( 15, "ntrials:" ) << I( 11, ntrials ) << std::endl;

	iunit << A( 15, "%accepted:" ) <<
	 F( 11, 2, 100*float(naccept)/float(ntrials) ) << std::endl;
	if ( get_native_exists() && !domain_insertion) {
		iunit << A( 15, "rms:" ) << F( 11, 2, rms_err ) << std::endl;
		maxsub_native(Eposition,nalign,maxsub_rms,log_evalue);
		iunit << A( 15, "maxsub:" ) << " nalign: " << I( 4, nalign ) <<
		 "  rms: " << F( 5, 2, maxsub_rms ) <<
		 " log_Evalue: " << F( 7, 2, log_evalue ) << std::endl;
		iunit << A( 15, "%_nat_cont:" ) << F( 11, 2, fraction_native_contacts()) << std::endl;
	}

//km  this works
//std::cout << "** IN OUTPUT_PDB_STATS **" << SS( fullatom ) << std::endl;

	if ( require_start && ( !multi_chain || design::dna_interface || antibody_modeler ) ) {
		fast_rms_x(position,start_position,rms_to_start);
		iunit << A( 15, "rms_to_start:" ) << SS( rms_to_start ) << std::endl;
	}

	if ( get_loop_flag() ) {
		if ( get_native_exists() ) {
			loop_local_rms(0,position,fast_rms);
			iunit << A( 15, "local_rms:" ) << SS( fast_rms ) << std::endl;
		}
		loop_get_all_splicerms(splicerms,num_loop);
		if ( num_loop > max_loop ) {
			std::cout << "increase local max_loop in make_pdb.cc" << std::endl;
			std::cout << "max_loop:  " << max_loop << " num_loop:  " << num_loop <<
			 std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		iunit << A( 15, "splicerms:" );
		for ( int i = 1; i <= num_loop; ++i ) {
			iunit << ' ' << F( 5,2, splicerms(i) );
		}
		iunit << '\n';
	}

	if( get_looprlx_exist() )
		output_loop_chain_breaks( iunit );
	/*
	iunit << A( 15, "score:" ) << F( 11, 2, score ) << '\n';
	iunit << A( 15, "env:" ) << F( 11, 2, env_score ) << '\n';
	iunit << A( 15, "pair:" ) << F( 11, 2, pair_score ) << '\n';
	iunit << A( 15, "vdw:" ) << F( 11, 2, vdw_score ) << '\n';
	iunit << A( 15, "hs:" ) << F( 11, 2, hs_score ) << '\n';
	iunit << A( 15, "ss:" ) << F( 11, 2, ss_score ) << '\n';
	iunit << A( 15, "sheet:" ) << F( 11, 2, sheet_score ) << '\n';
	iunit << A( 15, "r-sigma:" ) << F( 11, 2, rsigma_score ) << '\n';
	iunit << A( 15, "cb:" ) << F( 11, 2, cb_score ) << '\n';
	iunit << A( 15, "rg:" ) << F( 11, 2, rg ) << '\n';
	iunit << A( 15, "co:" ) << F( 11, 2, co ) << '\n';
	iunit << A( 15, "contact_pred:") << F( 11, 2, contact_prediction ) << '\n';
	iunit << A( 15, "rama:" ) << F( 11, 2, ramachandran_score ) << '\n';
	iunit << A( 15, "hb_srbb:" ) << F( 11, 2, hb_srbb_score ) << '\n';
	iunit << A( 15, "hb_lrbb:" ) << F( 11, 2, hb_lrbb_score ) << std::endl;
	*/
	int weight_width = 20;
	iunit << A( 15, "score:" ) << F( 11, 2, score ) << '\n';
	iunit << A( 15, "env:" ) << F( 11, 2, env_score ) << "    "
				<< A( weight_width,"env_weight:  ") << env_weight << '\n';
	iunit << A( 15, "pair:" ) << F( 11, 2, pair_score ) << "    "
				<< A( weight_width,"pair_weight:  ") << pair_weight << '\n';
	iunit << A( 15, "vdw:" ) << F( 11, 2, vdw_score ) << "    "
				<< A( weight_width,"vdw_weight:  ") << vdw_weight << '\n';
	iunit << A( 15, "hs:" ) << F( 11, 2, hs_score ) << "    "
				<< A( weight_width,"hs_weight:  ") << hs_weight << '\n';
	iunit << A( 15, "ss:" ) << F( 11, 2, ss_score ) << "    "
				<< A( weight_width,"ss_weight:  ") << ss_weight << '\n';
	iunit << A( 15, "sheet:" ) << F( 11, 2, sheet_score ) << "    "
				<< A( weight_width,"sheet_weight:  ") << sheet_weight << '\n';
	iunit << A( 15, "r-sigma:" ) << F( 11, 2, rsigma_score ) << "    "
				<< A( weight_width,"r-sigma_weight:  ") << rsigma_weight << '\n';
	iunit << A( 15, "cb:" ) << F( 11, 2, cb_score ) << "    "
				<< A( weight_width,"cb_weight:  ") << cb_weight << '\n';
	iunit << A( 15, "rg:" ) << F( 11, 2, rg ) << "    "
				<< A( weight_width,"rg_weight:  ") << rg_weight << '\n';
	iunit << A( 15, "co:" ) << F( 11, 2, co ) << "    "
				<< A( weight_width,"co_weight:  ") << co_weight << '\n';
	iunit << A( 15, "contact_pred:" ) << F( 11, 2, contact_prediction ) << "    "
				<< A( weight_width,"contact_weight:  ") << contact_prediction_weight << '\n';
	iunit << A( 15, "rama:" ) << F( 11, 2, ramachandran_score ) << "    "
				<< A( weight_width,"rama_weight:  ") << ramachandran_weight << '\n';
	iunit << A( 15, "hb_srbb:" ) << F( 11, 2, hb_srbb_score ) << "    "
				<< A( weight_width,"hb_srbb_weight:  ") << pack_wts.Whb_srbb() << '\n';
	iunit << A( 15, "hb_lrbb:" ) << F( 11, 2, hb_lrbb_score ) << "    "
				<< A( weight_width,"hb_lrbb_weight:  ") << pack_wts.Whb_lrbb() << std::endl;

	// domins chainbreak score
	if ( domain_insertion ) {
		iunit << A( 15, "chbrk:" ) << F( 11, 2, chainbreak_score ) << "    "
					<< A( weight_width,"chbrk_weight:  ") << jmp_chainbreak_weight << '\n';
	}

//km  this works
//	std::cout << "** IN OUTPUT_PDB_STATS2 **" << SS( fullatom ) << std::endl;
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) {
		iunit << A( 15, "pc: " ) << F( 11, 2, pc_score ) << '\n';
		iunit << A( 15, "pc_viol: " ) << F( 11, 2, pc_viol ) << '\n'; //std::endl;
	}
	if ( get_dipolar_exist() ) {
		iunit << A( 15, "dipolar: " ) <<
		 F( 11, 2, dipolar_score ) << '\n';
		iunit << A( 15, "projection: " ) <<
		 F( 11, 2, projection_score ) << '\n'; //std::endl;
	}

	if ( multi_chain && !design::dna_interface && !antibody_modeler ) {
		iunit << A( 30, "docking_env: " ) <<
		 F( 10, 2, docking_env_score ) << '\n';
		iunit << A( 30, "docking_pair: " ) <<
		 F( 10, 2, docking_pair_score ) << '\n';
		iunit << A( 30, "docking_contact: " ) <<
		 F( 10, 2, docking_contact_score ) << '\n';
		iunit << A( 30, "docking_vdw: " ) <<
		 F( 10, 2, docking_vdw_score ) << '\n';
		iunit << A( 30, "docking_site_constraint: " ) <<
		 F( 10, 2, docking_site_constraint_score ) << '\n';
		iunit << A( 30, "docking_fab: " ) <<
		 F( 10, 2, docking_fab_score ) << '\n'; //std::endl;
		if ( docking_fullatom_flag && ( part_begin(1) != 0 ) ) {
			calc_docking_number_of_contacts();
			iunit << A( 30, "ncont: " ) << get_countcont() << '\n'; //std::endl;
		}
	}

	if ( domain_insertion ) {
		iunit << A( 15, "domain_contact: " ) <<
		 F( 10, 2, domain_contact_score ) << '\n';
	}

	if ( fullatom ) {
		if(get_remark_option()){
			iunit << A( 15, "total: " ) << F( 11, 2, get_fullatom_totalE() ) << '\n';
			iunit << A( 15, "LJatr: " ) << F( 11, 2, fa_atr_score ) << '\n';
			iunit << A( 15, "LJrep: " ) << F( 11, 2, fa_rep_score ) << '\n';
			iunit << A( 15, "LKsol: " ) << F( 11, 2, fa_solv_score ) << '\n';
			iunit << A( 15, "E_h2o_sol: " ) << F( 11, 2, fa_h2o_solv_score ) << '\n';
			iunit << A( 15, "E_rot: " ) << F( 11, 2, fa_dun_score ) << '\n';
			iunit << A( 15, "E_intrares: " ) << F( 11, 2, fa_intrares_score ) << '\n';
			iunit << A( 15, "E_pair: " ) << F( 11, 2, fa_pair_score ) << '\n';
			iunit << A( 15, "gb_elec:" ) << F( 11, 2, fa_gb_elec_score ) << '\n';
			iunit << A( 15, "fa_plane:" ) << F( 11, 2, fa_plane_score ) << '\n';
			iunit << A( 15, "E_phipsi:" ) << F( 11, 2, fa_prob1b_score ) << '\n';
			iunit << A( 15, "E_h2o: " ) << F( 11, 2, fa_h2o_score ) << '\n';
			iunit << A( 15, "E_h2o_hb: " ) << F( 11, 2, fa_h2o_hb_score ) << '\n';
			iunit << A( 15, "E_ref: " ) << F( 11, 2, fa_ref_score ) << '\n';
		}
		else{
			iunit << A( 15, "bk_tot:" ) << F( 11, 2, get_fullatom_totalE() ) << '\n';
			iunit << A( 15, "fa_atr:" ) << F( 11, 2, fa_atr_score ) << '\n';
			iunit << A( 15, "fa_rep:" ) << F( 11, 2, fa_rep_score ) << '\n';
			iunit << A( 15, "fa_sol:" ) << F( 11, 2, fa_solv_score ) << '\n';
			iunit << A( 15, "fa_h2o_sol:" ) << F( 11, 2, fa_h2o_solv_score ) << '\n';
			iunit << A( 15, "fa_dun:" ) << F( 11, 2, fa_dun_score ) << '\n';
			iunit << A( 15, "fa_intrares:" ) << F( 11, 2, fa_intrares_score ) << '\n';
			iunit << A( 15, "fa_pair:" ) << F( 11, 2, fa_pair_score ) << '\n';
			iunit << A( 15, "gb_elec:" ) << F( 11, 2, fa_gb_elec_score ) << '\n';
			iunit << A( 15, "fa_plane:" ) << F( 11, 2, fa_plane_score ) << '\n';
			iunit << A( 15, "fa_prob:" ) << F( 11, 2, fa_prob1b_score ) << '\n';
			iunit << A( 15, "fa_h2o:" ) << F( 11, 2, fa_h2o_score ) << '\n';
			iunit << A( 15, "fa_h2o_hb:" ) << F( 11, 2, fa_h2o_hb_score ) << '\n';
			iunit << A( 15, "fa_ref:" ) << F( 11, 2, fa_ref_score ) << '\n';
		}
		if ( get_pH_packing_flag() ) {
			iunit << A( 15, "fa_pH:" )
			 << F( 11, 2, fa_pH_score ) << '\n'; //std::endl;
		}
		iunit << A( 15, "gsolt:" ) << F( 11, 2, get_ex_gsolt() ) << '\n';
		iunit << A( 15, "sasa:" ) << F(11, 2, get_ex_sasa_gsolt() ) << '\n';
		iunit << A( 15, "hb_sc:" ) << F( 11, 2, hb_sc_score ) << std::endl;

    if ( get_simple_elec() )
	    iunit << A( 15, "fa_elec:" ) << F( 11, 2, fa_elec_score ) << std::endl;

		if ( multi_chain && !design::dna_interface && !antibody_modeler ) {
      if ( !get_simple_elec() )
			  iunit << A( 30, "warshel_elec:" ) << F( 10, 2, docking_warshel_elec_score ) << std::endl;

			if ( docking::docking_pose_symmetry ) {
				iunit << A( 30, "symm_mon_score:" ) <<
				F( 10, 2, symm_mon_score ) << std::endl;
			}
		}
		iunit << A( 15, "sasapack:" ) << F( 11, 2, retrieve_sasa_pack_score() ) << std::endl;

		if ( dna_interface ) {
			iunit << A( 15, "dna_bs:" ) << F( 11, 2, dna_bs_total ) << '\n';
			iunit << A( 15, "dna_bp:" ) << F( 11, 2, dna_bp_total ) << std::endl;
		}

//		std::cout << "** IN OUTPUT_PDB_STATS3 **" << SS( fullatom ) << std::endl;

		if ( get_ligand_flag() ) {
			int i_ligand=1;
			for( std::vector< Ligand * >::iterator c_ligand=ligand::ligand_ptr_vector.begin();
			c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++){
			count_satisfied_ha_hbond( don_sat, don_non, acc_sat, acc_non, (**c_ligand));
			calc_number_of_ligand_contacts( lig_contacts,(**c_ligand) );
			iunit << A( 30, "Ligand ") << I(4, i_ligand) << '\n';
			iunit << A( 30, "lig_sum:" ) << F( 10, 2, (*c_ligand)->lig_sumE ) << '\n';
			iunit << A( 30, "lig_rep:" ) << F( 10, 2, (*c_ligand)->lig_repE ) << '\n';
			iunit << A( 30, "lig_atr:" ) << F( 10, 2, (*c_ligand)->lig_atrE ) << '\n';
			iunit << A( 30, "lig_sol:" ) << F( 10, 2, (*c_ligand)->lig_solE ) << '\n';
			iunit << A( 30, "lig_cou:" ) << F( 10, 2, (*c_ligand)->lig_couE ) << '\n';
			iunit << A( 30, "lig_hb :" ) << F( 10, 2, (*c_ligand)->lig_hbE ) << '\n';
			iunit << A( 30, "lig_vir:" ) << F( 10, 2, (*c_ligand)->lig_virE ) << '\n';
			iunit << A( 30, "lig_h2o:" ) << F( 10, 2, (*c_ligand)->lig_h2oE ) << '\n';
			iunit << A( 30, "lig_sas:" ) << F( 10, 2, (*c_ligand)->lig_sasE ) << '\n';
			iunit << A( 30, "lig_int_rep:" ) << F( 10, 2, (*c_ligand)->lig_int_repE ) << '\n';
			iunit << A( 30, "lig_int_atr:" ) << F( 10, 2, (*c_ligand)->lig_int_atrE ) << '\n';
			iunit << A( 30, "lig_int_solv:" ) << F( 10, 2, (*c_ligand)->lig_int_solvE ) << '\n';
			iunit << A( 30, "lig_int_dihed:" ) << F( 10, 2, (*c_ligand)->lig_int_dihedE ) << '\n';
			iunit << A( 30, "lig_int_coulomb:" ) << F( 10, 2, (*c_ligand)->lig_int_coulombE ) << '\n';
			iunit << A( 30, "lig_int_hb:" ) << F( 10, 2, (*c_ligand)->lig_int_hbE ) << '\n';
			iunit << A( 15, "lig_con:" ) << I( 11, lig_contacts ) << '\n';
			iunit << A( 15, "don_sat:" ) << I( 11, don_sat ) << '\n';
			iunit << A( 15, "don_non:" ) << I( 11, don_non ) << '\n';
			iunit << A( 15, "acc_sat:" ) << I( 11, acc_sat ) << '\n';
			iunit << A( 15, "acc_non:" ) << I( 11, acc_non ) << '\n';
			iunit << A( 15, "lig_nb :" ) << I( 11, (*c_ligand)->lig_neighbors ) << std::endl;
			i_ligand++;
			}//for loop
		}

		if ( get_decoystats_flag() ) {
			decoystats_make_pdb_scores( iunit );
			decoystats_write_output_lines( iunit );
		}

		if( decoy_features_ns::get_df_flag() ) {
			decoy_features_ns::print_decoy_features( iunit );
		}

		output_barcode_info( iunit );

		/*
 		if ( get_native_exists() ) {
			FArray1D_float residue_dist(total_residue);
			int orient_start = { 1 };
			int orient_end = { total_residue };

			float local_rms;
			orient_region(total_residue,Eposition,full_coord,native_Eposition,1,
										orient_start,orient_end,1,1,total_residue,local_rms);
			for ( int i = 1; i <= total_residue; ++i ) {
				residue_dist(i) = 0;
				for ( int j = 1; j <= 3; ++j ) {
					float d = Eposition(j,2,i)-native_Eposition(j,2,i);
					residue_dist(i) += d*d;
				}
				residue_dist(i) = std::sqrt(residue_dist(i));
				std::cout << "residue_dist " << i << " " << residue_dist(i) << std::endl;
			}
		}
		*/
		output_residueE( iunit, total_residue, res );
		output_rotamer_stats( iunit );

		//rj allow users to print HB info even if NA are not present, but don't
		//rj double-print the info
		if ( output_hbond_info && !dna_interface ) {
			dna_hbond_info( iunit );
		}

	}

	output_decoy_barcode( iunit );

	if ( ( classical_constraints::BOUNDARY::get_verbose_pdb_output() ) && ( pc_viol > 0.0 ) ){
		classical_constraints::BOUNDARY::output_constraints_status( -2, iunit );
	}

	if ( get_dipolar_exist() ) {
		dipolar_set_verbose(true);
		dipolar_set_verbose_unit( iunit );
		eval_dipolar(Eposition,1,total_residue,res,dipolar_score);
		dipolar_set_verbose_unit( std::cout );
	}

  if ( multi_chain && !design::dna_interface && !design::design_loops_flags::design_loops_hold && !get_enable_ligaa_flag() && !antibody_modeler) {
		output_docking_position( iunit );
		if ( get_docking_output_position_hetatm_flag() ) {
			output_docking_position_hetatm( iunit );
		}
	}

//	std::cout << "** IN OUTPUT_PDB_STATS end **" << SS( fullatom ) << std::endl;

}


//////////////////////////////////////////////////////////////////////////////
/// @begin torsion_diff
///
/// @brief
///
/// @detailed
///
/// @param  angle1 - [in/out]? -
/// @param  angle2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
torsion_diff(
	float angle1,
	float angle2
)
{
	float diff = std::abs(angle1-angle2);
	if ( diff > 180.0f ) {
		diff = std::min( std::abs(angle1-360.0f-angle2), std::abs(angle1+360.0f-angle2) );
	}
	return diff;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_fullcoord_pdb
///
/// @brief
///car output the full_coord array in pdb format
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_fullcoord_pdb( utility::io::orstream & iunit )
{

	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace analyze_interface_ddg_ns;
	using namespace int_fullatom_energies;

	int const max_atom = MAX_ATOM();

	// jk Write header info, if we have any and want it printed
	for ( unsigned i=0; preserve_header && ( i < header_info.size() ); ++i) {
		iunit << header_info[i] << std::endl;
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ( i == 1 && psi(i) == init_psi && omega(i) == init_omega ) ||
				 ( i == total_residue && phi(i) == init_phi ) ||
				 ( phi(i) == init_phi && psi(i) == init_psi && omega(i) == init_omega ) ) {
			for ( int j = 1; j <= max_atom; ++j ) {
				make_pdb_ns::occupancy(j,i) = -1.;
			}
		}
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			loop_set_bfactor(make_pdb_ns::bfactor(j,i),i);
			looprlx_set_bfactor(make_pdb_ns::bfactor(j,i),i,j);
			looprlx_set_occupancy(make_pdb_ns::occupancy(j,i),i,j);
		}
	}

	if ( sasapack_bvalues ) {    // set in get_io_options, options.cc
		sasapack_bfactor( total_residue );
	}

	if ( ala_scan ) {
		if (output_mabo_input) {
			//int count = 1;
			for ( int i = 1; i <= total_residue; ++i ) {
				float tmp;
				if ( aan_orig(i) == aa_gly || aan_orig(i) == aa_pro
						 || aan_orig(i) == aa_cys ) tmp = -9.99;
				else {
					//	++count;
					//	tmp = dgbind_resenergy(1) - dgbind_resenergy(count);
					if ( aan_orig(i) == aa_lys ||
							 aan_orig(i) == aa_his ||
							 aan_orig(i) == aa_arg ||
							 aan_orig(i) == aa_asp ||
							 aan_orig(i) == aa_asn ||
							 aan_orig(i) == aa_ser ||
							 aan_orig(i) == aa_thr ||
							 aan_orig(i) == aa_glu ||
							 aan_orig(i) == aa_gln ) {
						tmp=0.0;
					} else {
						tmp = dgbind_resenergy(1) - dgbind_resenergy(i+1);
					}
					if (tmp<-9.99) tmp=-9.99;
				}
				for ( int j = 1; j <= max_atom; ++j ) {
					make_pdb_ns::occupancy(j,i) = tmp;
				}
			}
		} else {
			for ( int i = 1; i <= total_residue; ++i ) {
				float const tmp = dgbind_resenergy(i+1) - dgbind_resenergy(1);
				for ( int j = 1; j <= max_atom; ++j ) {
					make_pdb_ns::bfactor(j,i) = tmp;
				}
			}
		}

		output_fullatom_pdb(xyz_orig,aan_orig,aav_orig,
		 total_residue,protein_chain,total_domains,domain_end,iunit);
  } else {
		output_fullatom_pdb(full_coord,res,res_variant,total_residue,protein_chain,
		 total_domains,domain_end,iunit);

		output_fullcoord_vpdb( iunit );
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_fullatom_pdb
///
/// @brief
///bk given the coordinates and amino acid sequence for a protein,
///bk output a pdb file.
///
/// @detailed
///car unlike output_fullcoord_pdb, this function can output coordinates
///car from any passed fullatom coordinate array
///
/// @param  xyz - [in/out]? - (xyz,atom#,res#)coordinates
/// @param  aan - [in/out]? - specifies amino acid at each seqpos
/// @param  aav - [in/out]? - amino acid variant at each position
/// @param  nres - [in/out]? - # of residues in the protein
/// @param  chain - [in/out]? - protein chain
/// @param  ndomains - [in/out]? - how many domains?
/// @param  dom_end - [in/out]? - end residues of each domain
/// @param  occ - [in/out]? -
/// @param  bval - [in/out]? -
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car option dependencies:
///car  use_pdb_numbering  - restore numbering/chains of input pdb file
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_fullatom_pdb(
	FArray3Da_float xyz, // (xyz,atom#,res#)coordinates
	FArray1Da_int aan, // specifies amino acid at each seqpos
	FArray1Da_int aav, // amino acid variant at each position
	int const nres, // # of residues in the protein
	char const chain, // protein chain
	int const ndomains, // how many domains?
	FArray1Da_int dom_end, // end residues of each domain
	utility::io::orstream & iunit
)
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace param;
	using namespace pdb;

	xyz.dimension( 3, MAX_ATOM(), nres );
	aan.dimension( nres );
	aav.dimension( nres );
	dom_end.dimension( ndomains );

// local variables
	std::string atomname;
	std::string resname;
	char chain_out = ' '; // for output, depends on mode
	int res_num_out; // ditto
	char insert_let_out = ' '; // ditto
	int atom_number_out; // ditto

	if ( !use_pdb_numbering ) {
		insert_let_out = ' ';
		chain_out = chain;
		if ( chain_out == '_' ) chain_out = ' ';
	}
	atom_number_out = 0;
	int idomain = 1;
	for ( int i = 1; i <= nres; ++i ) {
		if ( use_pdb_numbering ) {
			res_num_out = pdb_res_num(i);
			insert_let_out = pdb_insert_let(i);
			chain_out = res_chain(i);
		} else {
			res_num_out = i;
		}

		name_from_num( aan(i), resname );
		for ( int j = 1, je = natoms( aan(i), aav(i) ); j <= je; ++j ) {
			atom_name_from_atom_num( j, aan(i), aav(i), atomname );
			++atom_number_out;

			iunit << "ATOM  " << I( 5, atom_number_out ) << ' ' << atomname << ' '
					<< resname << ' ' << chain_out << I( 4, res_num_out )
					<< insert_let_out << "   " << F( 8, 3, xyz(1,j,i) )
					<< F( 8, 3, xyz(2,j,i) ) << F( 8, 3, xyz(3,j,i) )
					<< F( 6, 2, make_pdb_ns::occupancy(j,i) ) << ' ' << F( 5, 2, make_pdb_ns::bfactor(j,i) ) << std::endl;
		}
		if ( multi_chain && i == dom_end(idomain) ) {
			iunit << "TER" << std::endl;
			++idomain;
		}

	}
	if ( !multi_chain ) iunit << "TER" << std::endl;

	// added by sheffler jan 22 06
	iunit << make_pdb_ns::nonscored_hetero_atoms;

//            atom23   CA    GLY   L  99 A      x    y    z      occ    bfactor
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_residueE
///
/// @brief
///bk outputs table of residue energies,there are two tables
///          partI : a table of energies for each residue in the protein
///          partII: a table of total measured energies - expected energies
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_residueE(
	utility::io::orstream & iunit,
	int const total_residue,
	FArray1Da_int res
)
{
	using namespace files_paths;
	using namespace fullatom_energies;
	using namespace pdb;
	using namespace pdbstatistics_pack;
	using namespace template_pack;
	using namespace void_ns;

	//residue energies
	output_residueE_partI(iunit,total_residue,res);
	//total measured energies - expected energies
	output_residueE_partII(iunit,total_residue,res);
	//residue-residue energies across docking interface
	output_residueresidueE(iunit,total_residue,res);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_residueE_partII
///
/// @brief
///bk outputs table of residue energies - expected energies
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_residueE_partII(
	utility::io::orstream & iunit,
	int const total_residue,
	FArray1Da_int res
)
{
	using namespace dna_variables;
	using namespace files_paths;
	using namespace fullatom_energies;
	using namespace param;
	using namespace pdb;
	using namespace pdbstatistics_pack;
	using namespace template_pack;
	using namespace void_ns;

	res.dimension( total_residue );

//car local
	int aa,nb,ires;
	std::string res3,chain_out;
	float atr,rep,sol,one,dun,pair,tlj,hb,tot,atrb,atrm,atrs,repb,repm,reps,tljb,
	 tljm,tljs,cbur,cmid,csur;
	double intrares; // hope to eliminate gccdebug precision issue
	atr = 0.0;
	rep = 0.0;
	sol = 0.0;
	one = 0.0;
	dun = 0.0;
	intrares = 0.0;
	pair = 0.0;
	tlj = 0.0;
	hb = 0.0;
	tot = 0.0;
	atrb = 0.0;
	repb = 0.0;
	tljb = 0.0;
	atrm = 0.0;
	repm = 0.0;
	tljm = 0.0;
	atrs = 0.0;
	reps = 0.0;
	tljs = 0.0;
	cbur = 0.0;
	cmid = 0.0;
	csur = 0.0;

	// these arrays only contain values for up to MAX_AUTH_AA (20) residue types
	// if phosphoserine mode, add 1
	// no avg energies for nucleic acids currently exist
	int aa_lim = ( add_pser() ? MAX_AUTH_AA + 1 : MAX_AUTH_AA );

	for ( int i = 1; i <= total_residue; ++i ) {
		nb = neighbors(i);
		aa = res(i);
		if ( aa > aa_lim ) continue;

		atr += (atrenergy(i)-atr_avg(aa,nb));
		rep += (repenergy(i)-rep_avg(aa,nb));
		sol += (solenergy(i)-sol_avg(aa,nb));
		one += (probenergy(i)-one_avg(aa,nb));
		dun += (dunenergy(i)-dun_avg(aa,nb));
		intrares += (double(intraresenergy(i))-double(intrares_avg(aa,nb)));
		pair += (pair_energy(i)-pair_avg(aa,nb));
		tlj += (atrenergy(i)+repenergy(i)-tlj_avg(aa,nb));
		hb += (hbenergy(i)-hb_avg(aa,nb));
		tot += (resenergy(i)-tot_avg(aa,nb));
		if ( nb > 19 ) {
			atrb += (atrenergy(i)-atr_avg(aa,nb));
			repb += (repenergy(i)-rep_avg(aa,nb));
			tljb += (atrenergy(i)+repenergy(i)-tlj_avg(aa,nb));
			cbur += 1.0;
		} else if ( nb > 14 ) {
			atrm += (atrenergy(i)-atr_avg(aa,nb));
			repm += (repenergy(i)-rep_avg(aa,nb));
			tljm += (atrenergy(i)+repenergy(i)-tlj_avg(aa,nb));
			cmid += 1.0;
		} else {
			atrs += (atrenergy(i)-atr_avg(aa,nb));
			reps += (repenergy(i)-rep_avg(aa,nb));
			tljs += (atrenergy(i)+repenergy(i)-tlj_avg(aa,nb));
			csur += 1.0;
		}
	}

	if ( cbur != 0 ) {
		atrb /= cbur;
		repb /= cbur;
		tljb /= cbur;
	}
	if ( cmid != 0 ) {
		atrm /= cmid;
		repm /= cmid;
		tljm /= cmid;
	}
	if ( csur != 0 ) {
		atrs /= csur;
		reps /= csur;
		tljs /= csur;
	}

	iunit << '\n';
	iunit << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "energies-average(in pdb) energies, AND rsd SASA packing score" << '\n';
	if(get_remark_option()){
		iunit << "REMARK ";
		iunit << "res chain aa nb   LJatr    LJrep  LKsol  E_phipsi   Erot  Eintra" <<
			"   Ehbnd   Epair    Elj   Eres   SASApack";
	}
	else {
		iunit << "res chain aa nb   Eatr    Erep    Esol    Eaa     Edun  Eintra" <<
			"   Ehbnd   Epair    Elj   Eres   SASApack";
		iunit << "   res_rms sasaprob";
	}


	FArray1D_float residue_dist( total_residue );
	calc_residue_dist( residue_dist );

	iunit << '\n';

	for ( int i = 1; i <= total_residue; ++i ) {
		nb = neighbors(i);
		aa = res(i);
		if ( aa > aa_lim && !get_enable_ligaa_flag() ) continue;
		name_from_num( aa, res3 );
		chain_out = res_chain(i);
		if ( use_pdb_numbering ) {
			ires = pdb_res_num(i);
		} else {
			ires = i;
		}
		if(get_remark_option()) iunit << "REMARK ";
		iunit << I( 3, ires ) << ' ' << A(3,chain_out) << ' ' <<A(4,res3) << ' ' << I( 2, nb ) <<
			' ' << F( 7, 1, atrenergy(i)-atr_avg(aa,nb) ) <<
			' ' << F( 7, 1, repenergy(i)-rep_avg(aa,nb) ) <<
			' ' << F( 7, 1, solenergy(i)-sol_avg(aa,nb) ) <<
			' ' << F( 7, 1, probenergy(i)-one_avg(aa,nb) ) <<
			' ' << F( 7, 1, dunenergy(i)-dun_avg(aa,nb) ) <<
			' ' << F( 7, 1, intraresenergy(i)-intrares_avg(aa,nb) ) <<
			' ' << F( 7, 1, hbenergy(i)-hb_avg(aa,nb) ) <<
			' ' << F( 7, 1, pair_energy(i)-pair_avg(aa,nb) ) <<
			' ' << F( 7, 1, atrenergy(i)+repenergy(i)-tlj_avg(aa,nb) ) <<
			' ' << F( 7, 1, resenergy(i)-tot_avg(aa,nb) ) ;
		//lin if we output the SASA for all residues, the following if statement
		//lin should be gone !!
		if( param_aa::is_ligand( aa ) ) {
			iunit << " SASA: " << F( 8, 2, rsd_sasa(pr_bin_14,i) ) <<
				" SASA_pack: " << F( 8, 2, rsd_sasa_pack_score(i) ) <<
				" Nat_RMS: "<< F( 8, 2, residue_dist(i) );
		} else {
			iunit << ' ' << F( 8, 2, rsd_sasa_pack_score(i) ) <<
				' ' << F( 8, 2, residue_dist(i) ) <<
				' ' << F( 8, 3, retrieve_rsd_sasa_prob(i) );
		}
		iunit << std::endl;
	}

	iunit << "avgtot    " <<
		' ' << F( 7, 1, atr ) <<
		' ' << F( 7, 1, rep ) <<
		' ' << F( 7, 1, sol ) <<
		' ' << F( 7, 1, one ) <<
		' ' << F( 7, 1, dun ) <<
		' ' << F( 7, 1, intrares ) <<
		' ' << F( 7, 1, hb ) <<
		' ' << F( 7, 1, pair ) <<
		' ' << F( 7, 1, tlj ) <<
		' ' << F( 7, 1, tot ) <<
		' ' << F( 8, 2, avg_sasa_pack_score ) <<
		' ' << "        " <<
		' ' << F( 8, 3, retrieve_avg_sasa_prob() ) << std::endl;

	iunit << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "actual-average(in pdb) energies per residue" << '\n';
	if(get_remark_option()){
		iunit << "REMARK ";
		iunit << "       LJatr   LJrep   Elj" << '\n';
	}
	else
		iunit << "       Eatr   Erep   Elj" << '\n';
	iunit << "buried" << ' ' << F( 7, 1, atrb ) << ' ' <<
	 F( 7, 1, repb ) << ' ' << F( 7, 1, tljb ) << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "middle" << ' ' << F( 7, 1, atrm ) << ' ' <<
	 F( 7, 1, repm ) << ' ' << F( 7, 1, tljm ) << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "surfac" << ' ' << F( 7, 1, atrs ) << ' ' <<
	 F( 7, 1, reps ) << ' ' << F( 7, 1, tljs ) << '\n';
	iunit << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_residueE_partI
///
/// @brief
///bk outputs the first table of residue energies
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_residueE_partI(
	utility::io::orstream & iunit,
	int const total_residue,
	FArray1Da_int res
)
{
	using namespace dna_variables;
	using namespace files_paths;
	using namespace fullatom_energies;
	using namespace pdb;
	using namespace pdbstatistics_pack;
	using namespace template_pack;
	using namespace void_ns;

	res.dimension( total_residue );

//car local
	int ires;
	std::string res3;
	float solvationE,attractiveE,repulsiveE,referenceE,protonationE,dunbrackE,Paa_ppE,
	 sc_bb_hbondE,pairprobE,planeprobE,h2oE,h2ohbE,h2osolE,intra_resE;
	float srbb_hbE,lrbb_hbE, gb_elecE, cstE, elecE, bondangleE;
	std::string conditional_scores = (( get_pH_packing_flag() ) ? "EpH    " : "" );
	if (scorefxns::bond_angle_weight) {
		conditional_scores += " Ebang    ";
	}
  if ( get_simple_elec() ) conditional_scores += " Eelec   ";
	iunit << '\n';
	if(get_remark_option()){
		iunit << "REMARK ";
		iunit << "res aa    LJatr    LJrep   LKsol Eh2o_sol E_phipsi  Erot  Eintra" <<
		" Ehbnd_bb Ehbnd_sc  Epair   Eref    Egb    Eh2o   Eh2o_hb  Ecst  " <<  conditional_scores << "Eres";
	}
	else{
		iunit << "res aa    Eatr     Erep    Esol  Eh2o_sol   Eaa    Edun  Eintra" <<
			" Ehbnd_bb Ehbnd_sc   Epair   Eref   Egb    Eh2o   Eh2o_hb  Ecst  " <<  conditional_scores << "Eres";
	}

	if ( design::dna_interface ) iunit << "   Edna_bs   Edna_bp";

	iunit << '\n';
	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), res3 );
		if ( use_pdb_numbering ) {
			ires = pdb_res_num(i);
		} else {
			ires = i;
		}
		if(get_remark_option()) iunit << "REMARK ";
		iunit << I( 3, ires ) << ' ' << res3 <<
			' ' << F( 7, 1, atrenergy(i) ) <<
			' ' << F( 7, 1, repenergy(i) ) <<
			' ' << F( 7, 1, solenergy(i) ) <<
			' ' << F( 7, 1, wsolenergy(i) ) <<
			' ' << F( 7, 1, probenergy(i) ) <<
			' ' << F( 7, 1, dunenergy(i) ) <<
			' ' << F( 7, 1, intraresenergy(i) ) <<
			' ' << F( 7, 1, hbenergy_sr_bb(i)+hbenergy_lr_bb(i) ) <<
			' ' << F( 7, 1, hbenergy_sc_sc(i)+hbenergy_sc_bb(i) ) <<
			' ' << F( 7, 1, pair_energy(i) ) <<
			' ' << F( 7, 1, unfenergy(i) ) <<
			' ' << F( 7, 1, gb_elecenergy(i) ) <<
			' ' << F( 7, 1, h2oenergy(i) ) <<
			' ' << F( 7, 1, h2ohbenergy(i) ) <<
			' ' << F( 7, 1, cstenergy(i) );
		if ( get_pH_packing_flag() ) {
			iunit << ' ' << F( 7, 1, protonation_energy(i));
		}
		if (scorefxns::bond_angle_weight) {
			iunit << ' ' << F( 7, 1, bondangleenergy(i) );
		}
    if ( get_simple_elec() ) iunit << ' ' << F( 7, 1, elecenergy(i) );
		iunit <<' ' << F( 7, 1, resenergy(i) );

	if ( design::dna_interface ) {
		iunit <<
			' ' << F( 7, 1, param_pack::pack_wts.Wdna_bs()*dna_bs_score(i) ) <<
			' ' << F( 7, 1, param_pack::pack_wts.Wdna_bp()*dna_bp_score(i) );
	}
		iunit << '\n';
	}

	sum_fullatom_energies(total_residue,attractiveE,repulsiveE,solvationE,
	 referenceE,protonationE,dunbrackE,Paa_ppE,pairprobE,planeprobE,h2oE,h2ohbE,
	 h2osolE,srbb_hbE,lrbb_hbE,sc_bb_hbondE,intra_resE,gb_elecE,elecE,cstE,bondangleE);
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "totals" << ' ' <<
		' ' << F( 7, 1, attractiveE ) <<
		' ' << F( 7, 1, repulsiveE ) <<
		' ' << F( 7, 1, solvationE ) <<
		' ' << F( 7, 1, h2osolE ) <<
		' ' << F( 7, 1, Paa_ppE ) <<
		' ' << F( 7, 1, dunbrackE ) <<
		' ' << F( 7, 1, intra_resE ) <<
		' ' << F( 7, 1, srbb_hbE+lrbb_hbE ) <<
		' ' << F( 7, 1, sc_bb_hbondE ) <<
		' ' << F( 7, 1, pairprobE ) <<
		' ' << F( 7, 1, referenceE ) <<
		' ' << F( 7, 1, gb_elecE ) <<
		' ' << F( 7, 1, h2oE ) <<
		' ' << F( 7, 1, h2ohbE ) <<
		' ' << F( 7, 1, cstE );
	if  ( get_pH_packing_flag() ) {
		iunit << ' ' << F( 7, 1, protonationE);
	}
	if (scorefxns::bond_angle_weight) {
		iunit << ' ' << F( 7, 1, bondangleE );
	}
  if ( get_simple_elec() ) iunit << ' ' << F( 7, 1, elecE );
	iunit << ' ' << F( 7, 1, get_fullatom_totalE() );

	if ( design::dna_interface ) {
		iunit <<
			' ' << F( 7, 1, dna_bs_total ) <<
			' ' << F( 7, 1, dna_bp_total );
	}

	iunit << '\n';

	//lin ouput atomic ligaaa energy table
	if( get_enable_ligaa_flag() && cst_set_ns::debug_output ) {
		using namespace enable_ligaa_ns;
		using namespace param_aa;
		using namespace aaproperties_pack;

		iunit << '\n';
		iunit << "Atomic Energy Table of ligaa "<< '\n';
		iunit << "atm_id atm_name  Eatr    Erep    Esol    Ehbnd  SASA5 SASA14   SASApack \n";
		int const aav (1);
		int const aae ( enzyme::ligaaN+param_aa::ligand_aa_vector[1] );
		for( int aa=ligand_aa_vector[1];aa<aae;aa++) {
			name_from_num(aa,res3);
			for( int ii=1, ie=natoms(aa,aav); ii<=ie; ii++ ) {
				iunit << I(4, ii) << ' ' << res3 << "  " << atom_name(ii,aa,aav)
							<< ' ' <<F( 7, 2, ligaa_atrE(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_repE(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_solE(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_hbE(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_sasa5(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_sasa14(ii,has_ligand_no(aa)) )
							<< ' ' <<F( 7, 2, ligaa_sasa_pack(ii,has_ligand_no(aa)) )
							<< std::endl;
			}
		}
	}

	//KMa In order to print the energy terms of aa with ligands or surfaces
	//KMa excluding aa-aa self-interactions and packing terms  2006/1
	if ( get_ligand_flag() )
	{

	iunit << '\n';
	iunit << "Energy interactions of amino-acids with ligands or surfaces only and not aa-aa"<< '\n';
	iunit << "res aa    Elatt   Elrep   Elsol   Elgbn   Elh2o   Elhb   Elp   Elres";

	iunit << '\n';
	for ( int i = 1; i <= total_residue; ++i )
	{
		name_from_num( res(i), res3 );
		if ( use_pdb_numbering )
		{			ires = pdb_res_num(i);		}
		else
		{			ires = i;					}

		if ( latrenergy(i) != 0. || lrepenergy(i)!=0. || lgb_elecenergy(i)!= 0. || lrepenergy(i) != 0.
			 || lh2oenergy(i) != 0. || lhbenergy(i) != 0.   || lpair_energy(i) != 0. )
		{
		iunit << I( 3, ires ) << ' ' << res3 <<
			' ' << F( 7, 1, latrenergy		(i) ) <<
			' ' << F( 7, 1, lrepenergy		(i) ) <<
			' ' << F( 7, 1, lsolenergy		(i) ) <<
			' ' << F( 7, 1, lgb_elecenergy	(i) ) <<
			' ' << F( 7, 1, lh2oenergy		(i) ) <<
			' ' << F( 7, 1, lhbenergy		(i) ) <<
			' ' << F( 7, 1, lpair_energy	(i) ) <<
			' ' << F( 7, 1, lresenergy		(i) ) ;

		iunit << '\n';
		}
	}

	}








}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_rotamer_stats
///
/// @brief
///bk output statistics for how closely the rotamers match the starting rotamers
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_rotamer_stats( utility::io::orstream & iunit )
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace pdb;
	using namespace start;

	if ( !start_initialized ) return;

	FArray2D_int start_rot( MAX_CHI, MAX_RES()() );
	FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );
	int nchi1,nchi12,nchi123,nchi1_correct,nchi12_correct,
	 nchi123_correct,int_nchi1,int_nchi12,int_nchi123,int_nchi1_correct,
	 int_nchi12_correct,int_nchi123_correct,ires;

	FArray2D_float start_chi( MAX_CHI, MAX_RES()() );
	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );
	FArray2D_float offset_chi( MAX_CHI, MAX_RES()() );
	std::string aaname;
	FArray2D_bool same_rot( MAX_CHI, MAX_RES() );
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,
		decoy_chi,decoy_rot);

	get_chi_and_rot_from_coords(total_residue,start_res,start_res_variant,
		start_fullcoord,start_chi,start_rot);

	//bk compute differentials
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		for ( int chi = 1; chi <= MAX_CHI; ++chi ) {
			offset_chi(chi,seqpos) = std::abs(
				subtract_chi_angles(
					decoy_chi(chi,seqpos),start_chi(chi,seqpos),res(seqpos),chi
				)
			);
			if ( decoy_rot(chi,seqpos) == start_rot(chi,seqpos) ) {
				same_rot(chi,seqpos) = true;
			} else {
				same_rot(chi,seqpos) = false;
			}
		}
	}

	nchi1 = 0;
	nchi12 = 0;
	nchi123 = 0;
	nchi1_correct = 0;
	nchi12_correct = 0;
	nchi123_correct = 0;
	int_nchi1 = 0;
	int_nchi12 = 0;
	int_nchi123 = 0;
	int_nchi1_correct = 0;
	int_nchi12_correct = 0;
	int_nchi123_correct = 0;

	//bk calculate what fraction of rotamers are correct
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( start_res(seqpos) == res(seqpos) ) {
			for ( int chi = 1; chi <= nchi(res(seqpos),1); ++chi ) {
				if ( chi == 1 ) {
					++nchi1;
					if ( same_rot(chi,seqpos) ) {
						++nchi1_correct;
					}
				}
				if ( chi == 2 ) {
					++nchi12;
					if ( same_rot(1,seqpos) && same_rot(2,seqpos) ) {
						++nchi12_correct;
					}
				}
				if ( chi == 3 ) {
					++nchi123;
					if ( same_rot(1,seqpos) && same_rot(2,seqpos) && same_rot(3,seqpos) ) {
						++nchi123_correct;
					}
				}
			}
		}
	}

//bk output table of offsets
	iunit << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "decoy chi angles - starting chi angles" << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "res  aa   chi1    chi2    chi3    chi4" << '\n';
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( start_res(seqpos) == res(seqpos) ) { // don't compare different AA
			name_from_num(res(seqpos),aaname);
			if ( use_pdb_numbering ) {
				ires = pdb_res_num(seqpos);
			} else {
				ires = seqpos;
			}
			if(get_remark_option()) iunit << "REMARK ";
			iunit << I( 4, ires ) << ' ' << aaname <<
			 "  " << F( 6, 2, offset_chi(1,seqpos) ) <<
			 "  " << F( 6, 2, offset_chi(2,seqpos) ) <<
			 "  " << F( 6, 2, offset_chi(3,seqpos) ) <<
			 "  " << F( 6, 2, offset_chi(4,seqpos) ) <<
			 "  chi_offsets" << std::endl;
		}
	}
	iunit << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "the fraction of chi1 correct  " <<
	 I( 4, nchi1_correct ) << "  " << I( 4, nchi1 ) << "  " <<
	 F( 6, 2, float(nchi1_correct)/float(nchi1) ) << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "the fraction of chi12 correct  " <<
	 I( 4, nchi12_correct ) << "  " << I( 4, nchi12 ) << "  " <<
	 F( 6, 2, float(nchi12_correct)/float(nchi12) ) << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "the fraction of chi123 correct  " <<
	 I( 4, nchi123_correct ) << "  " << I( 4, nchi123 ) << "  " <<
	 F( 6, 2, float(nchi123_correct)/float(nchi123) ) << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "   " << std::endl;

}


//////////////////////////////////////////////////////////////////////////////
/// @begin output_chi
///
/// @brief
///js output chi angles into decoy
///
/// @detailed
///
/// @param  output_chi - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_chi( utility::io::orstream & iunit )
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace pdb;

	int ires;
	FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );

	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );

	std::string aaname;
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,decoy_chi,decoy_rot);

	iunit << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "absolute decoy chi angles" << '\n';
	if(get_remark_option()) iunit << "REMARK ";
	iunit << "res  aa     chi1      chi2      chi3      chi4" << '\n';
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		name_from_num( res(seqpos), aaname );
		if ( use_pdb_numbering ) {
			ires = pdb_res_num(seqpos);
		} else {
			ires = seqpos;
		}
		if(get_remark_option()) iunit << "REMARK ";
		iunit << I( 4, ires ) << ' ' << aaname <<
		 "  " << F( 8, 3, decoy_chi(1,seqpos) ) <<
		 "  " << F( 8, 3, decoy_chi(2,seqpos) ) <<
		 "  " << F( 8, 3, decoy_chi(3,seqpos) ) <<
		 "  " << F( 8, 3, decoy_chi(4,seqpos) ) << ' ' <<
		 I( 4, decoy_rot(1,seqpos) ) <<
		 I( 4, decoy_rot(2,seqpos) ) <<
		 I( 4, decoy_rot(3,seqpos) ) <<
		 I( 4, decoy_rot(4,seqpos) ) <<
		 "   chi_absolute" << std::endl;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_residueresidueE
///
/// @brief
///   outputs residue pair energies, for use in interface / docking mode
///
/// @detailed
///
/// @param  iunit - [in] - output stream
/// @param  total_residue - [in] - number of residues
/// @param  res - [in] - residue type array
///
/// @global_read fullatom energies, docking partner size
///
/// @global_write none
///
/// @remarks needs pdb_res_label without spaces to play nice with sort.  also needs weights on the total.
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified 17 August 2005
/////////////////////////////////////////////////////////////////////////////////
void
output_residueresidueE(
	utility::io::orstream & iunit,
	int const total_residue,
	FArray1Da_int res
)
{
	using namespace fullatom_energies;
	using namespace pdb;
	using namespace design;
	using namespace docking;
	using namespace files_paths;

	res.dimension( total_residue );

	if ( ( !multi_chain && !get_enable_ligaa_flag() )
			 || design::dna_interface || antibody_modeler ) return;
	if ( get_ligand_flag() ) return;

	iunit << '\n';
	iunit << "Pair energies across interface\n";
	iunit << "res1    aa1 res2    aa2   total    Eatr    Erep    Esol" <<
		"   Ehbnd   Epair     Egb    Ecst  Eplane    Eh2o Eh2o_hb";

	iunit << '\n';
	std::string ires3,jres3;
	for ( int i = part_begin(1); i <= part_end(1); ++i ) {
		name_from_num( res(i), ires3 );

		for ( int j = part_begin(2); j <= part_end(2); ++j ) {
			name_from_num( res(j), jres3 );

			float total_pair = atr_pair(i,j) + rep_pair(i,j) + sol_pair(i,j) + hbond_pair(i,j) +
				pair_pair(i,j) + gb_elec_pair(i,j) + cst_pair(i,j) + plane_plane(i,j) +
				h2o_pair(i,j) + h2o_hb_pair(i,j); // NEEDS WEIGHTS!

			if ( total_pair ) { // ?? use OR instead of TOTAL?

				iunit << A( 1,res_chain(i) ) << I( 5,pdb_res_num(i) ) << A( 1,pdb_insert_let(i) )
							<< " " << ires3 << " "
							<< A( 1,res_chain(j) ) << I( 5,pdb_res_num(j) ) << A( 1,pdb_insert_let(j) )
							<< " " << jres3
							<< " " << F( 7, 2, total_pair )
							<< " " << F( 7, 2, atr_pair(i,j) )
							<< " " << F( 7, 2, rep_pair(i,j) )
							<< " " << F( 7, 2, sol_pair(i,j) )
							<< " " << F( 7, 2, hbond_pair(i,j) )
							<< " " << F( 7, 2, pair_pair(i,j) )
							<< " " << F( 7, 2, gb_elec_pair(i,j) )
							<< " " << F( 7, 2, cst_pair(i,j) )
							<< " " << F( 7, 2, plane_plane(i,j) )
							<< " " << F( 7, 2, h2o_pair(i,j) )
							<< " " << F( 7, 2, h2o_hb_pair(i,j) )
							<< std::endl;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sasapack_bfactor
///
/// @brief
///   assign residue sasapack values to bvalues
///   also assign high bvalues to buried non-hbond polars
///
/// @detailed
///
/// @authors Brian Kuhlman
///
/// @last_modified Feb 2006
/////////////////////////////////////////////////////////////////////////////////

void
sasapack_bfactor(
	int const total_residue
)
{
	using namespace param;
	using namespace hbonds;

	int const max_atom = MAX_ATOM();
	float const cutoff = 0.001;

	// jk Reset decoystats counters
	// note: this clears decoystats output, but this flag
	// is not intended to be used with decoystats anyway
	reset_decoy_scores();
	decoystats_reset_output_lines();

	// call functions needed for finding unsatisfied buried polars
	copy_hbenergies();
	std::list < unsatisfied_buried_polar > uns_list;
	std::list < unsatisfied_buried_group > group_uns_list;
	find_unsatisfied_hbonds(uns_list, group_uns_list);
	report_unsatisfied_hbonds("",uns_list,group_uns_list);

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		for ( int atom = 1; atom <= max_atom; ++atom ) {
			float rsd_sasa_prob = retrieve_rsd_sasa_prob( seqpos );
			rsd_sasa_prob = max( rsd_sasa_prob, cutoff );
			make_pdb_ns::bfactor(atom,seqpos) = -std::log( rsd_sasa_prob ) * 10.0;
			// JSS should there be a factor for buried hbonds?
			// There was vestigal, non-functioning code here that I removed.
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////
void
looprlx_set_bfactor(
	float & bval,
	int i,
	int j
){

	using namespace pdb;

	if( !relax_options::looprlx ) return;

	bval = bvalue( j, i );

}

////////////////////////////////////////////////////////////////////////////////
void
looprlx_set_occupancy(
	float & occ,
	int i,
	int j
){

	using namespace pdb;

	if( !relax_options::looprlx ) return;

	occ = occupancy( j, i );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin alter_pdb_fields
///
/// @brief
/// Go back through and edit an outputted pdb file.
///
/// @detailed
/// Written originally for changing the b-factor column of the pdb output from a
/// designed protein-DNA complex to contain ddg binding energy values for each
/// residue.  (It would be impossible to do so during the initial pdb output,
/// because ddg overwrites the structure and energy arrays and could corrupt the
/// coordinates and statistics.)
///
/// @authors
/// ashworth
////////////////////////////////////////////////////////////////////////////////

void
alter_pdb_fields( std::string filename )
{
	utility::io::izstream pdbfile;
	std::list< std::string > pdblines;

	pdbfile.open( filename );

	if ( !pdbfile ) std::cout << "Can't open " << pdbfile.filename() << std::endl;

	while ( !pdbfile.eof() ) {
		std::string card;
		pdbfile >> bite( 6, card );
		if ( card == "ATOM  " ) {
			std::string bite1, bite2;
			int resi;
			char chain;
			pdbfile >> bite( 15, bite1 ) >> bite ( chain ) >> bite( 4, resi ) >>
			 bite( 34, bite2 ) >> skip;

			int seqpos;
			res_num_from_pdb_res_num_chain( seqpos, resi, chain );

			// this is the ddg_per_residue, which will be replace the b-factor column
			float bfact = int_fullatom_energies::complex_resenergy_res(3,seqpos) -
										int_fullatom_energies::complex_resenergy_res(2,seqpos) -
										int_fullatom_energies::complex_resenergy_res(1,seqpos);

			pdblines.push_back( card + bite1 + chain + I(4,resi) + bite2 +
			 F(6,2,bfact) );

		} else { // not coordinates
			std::string the_rest;
			pdbfile >> bite( the_rest ) >> skip;
			pdblines.push_back( card + the_rest );
		}
	}

	pdbfile.close();
	pdbfile.clear();

	utility::io::ozstream newfile;
	newfile.open( filename ); // should automatically overwrite the old file

	for ( std::list< std::string >::iterator line = pdblines.begin(); line !=
	 pdblines.end(); ++line ) {
		newfile << *line << '\n' ;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_bfactor_to_pdbfile
///
/// @brief
/// Go back through and edit an outputted pdb file.
///
/// @detailed
/// Modified from alter_pdb_fields() method. This allows alteration of the
/// b-factor column of a pdb file, given a FArray1D_float of new values. This
/// can be useful in visualization of per-residue scores of a given structure.
/// This method takes two arguments:
///		std::string filename - pdb file to modify.
///		FArray1D_float data  - data for replacing b-factor. This should be
///			indexed by the resi field of the pdb file (fields 23-26) .
///
/// @authors
/// tex
////////////////////////////////////////////////////////////////////////////////

void
alter_bfactor_pdbfile(
	std::string filename,
	FArray1D_float data
) {

	// Scale the b-factors so that they are in the range [0,100].
	float max_value(-1);
	for ( int i = 1; i <= misc::total_residue; i++ ) {
		if ( data(i) > max_value )
			max_value = data(i);
	}
	for ( int i = 1; i <= misc::total_residue; i++ ) {
		data(i) = std::log( data(i) ) / std::log( max_value );
	}


	utility::io::izstream pdbfile;
	std::list< std::string > pdblines;

	pdbfile.open( filename );

	if ( !pdbfile ) std::cout << "Can't open " << pdbfile.filename() << std::endl;

	while ( !pdbfile.eof() ) {
		std::string card;
		pdbfile >> bite( 6, card );
		if ( card == "ATOM  " ) { // atomic coordinates
			std::string bite1, bite2;
			int resi;
			char chain;
			pdbfile >> bite( 15, bite1 ) >> bite ( chain ) >> bite( 4, resi ) >>
			 bite( 34, bite2 ) >> skip;

			int seqpos;
			res_num_from_pdb_res_num_chain( seqpos, resi, chain );

			float bfact = data(resi);

			pdblines.push_back( card + bite1 + chain + I(4,resi) + bite2 +
			 F(6,2,bfact) );

		} else { // not not atomic coordinates
			std::string the_rest;
			pdbfile >> bite( the_rest ) >> skip;
			pdblines.push_back( card + the_rest );
		}
	}

	pdbfile.close();
	pdbfile.clear();

	utility::io::ozstream newfile;
	newfile.open( filename ); // should automatically overwrite the old file

	for ( std::list< std::string >::iterator line = pdblines.begin(); line !=
	 pdblines.end(); ++line ) {
		newfile << *line << '\n' ;
	}
} // alter_bfactor_pdbfile
