// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#ifndef MURPHP_CLASSES_HH
#define MURPHP_CLASSES_HH

#include <iosfwd>
#include <string>
#include <cmath>

namespace murphp {

  using namespace std;

  // ____________________ Colorable ____________________

  class Colorable {
  private:
    mutable int color;
  public:
    static const int WHITE, BLACK, GRAY, RED;
    Colorable() : color(WHITE) {}
    Colorable(const int c) : color(c) {}
    virtual ~Colorable() {}

    // NB: although this is a setter I have declared it a const fxn
    // since color isn't supposed to be an important field
    void setColor(const int c) const { color = c; }
    const int getColor() const { return color; }

    static const string& toString(const int c);
    static const int fromString(const string& s);

  }; // class Colorable

  // ____________________ Indexable ____________________

  class Indexable {
  private:
    int index;
  public:
    Indexable() : index(0) {}
    Indexable(const int index_) : index(index_) {}
    virtual ~Indexable() {}

    const int getIndex() const { return index; }
    void setIndex(const int index_) { index = index_; }

  }; // class Indexable

  // ____________________ Point2 ____________________


  class Point3;

  class Point2 {
  public:
    double x, y;
    Point2() {}
    Point2(const double x_, const double y_) : x(x_), y(y_) {}

    static const double dist(const Point2& a, const Point2& b);
    const double size() const;
    const Point2 operator*(const double& d) const;
    const Point2 operator/(const double& d) const;

    const bool operator==(const Point2& other) const;
    const bool operator<(const Point2& other) const;
    const bool operator<=(const Point2& other) const;
    const bool operator>(const Point2& other) const;
    const bool operator>=(const Point2& other) const;

    operator Point3() const;

  }; // class Point2

  const Point2 operator+(const Point2& a, const Point2& b);
  const Point2 operator-(const Point2& a, const Point2& b);

  ostream& operator<<(ostream& out, const Point2& p);
  istream& operator>>(istream& in , Point2& p);

  // ____________________ Point3 ____________________


  class Point3 {
  private:

    double m_d[3];

    static const Point3 theZero,theXhat,theYhat,theZhat;

  public:

    Point3() {}

    Point3& operator=(const Point3& rhs) {
      m_d[0] = rhs.m_d[0];
      m_d[1] = rhs.m_d[1];
      m_d[2] = rhs.m_d[2];
      return *this;
    } // operator=

    Point3(const Point3& rhs) {
      (*this) = rhs;
    }

    Point3(const double& d0, const double& d1, const double& d2) {
      m_d[0] = d0;
      m_d[1] = d1;
      m_d[2] = d2;
    }

    static const Point3 rand();

    static const Point3& zero() {
      return theZero;
    } // Point3::zero

    static const Point3& getXhat() {
      return theXhat;
    } // Point3::xhat

    static const Point3& getYhat() {
      return theYhat;
    } // Point3::yhat

    static const Point3& getZhat() {
      return theZhat;
    } // Point3::zhat

    static const Point3& xhat() {
      return getXhat();
    } // Point3::xhat

    static const Point3& yhat() {
      return getYhat();
    } // Point3::yhat

    static const Point3& zhat() {
      return getZhat();
    } // Point3::zhat

    const double& operator[](int i) const {
      return m_d[i];
    }

    double& operator[](int i) {
      return m_d[i];
    }

    const Point3& operator+=(const Point3& rhs) {
      m_d[0] += rhs.m_d[0];
      m_d[1] += rhs.m_d[1];
      m_d[2] += rhs.m_d[2];
      return *this;
    }

    const Point3& operator-=(const Point3& rhs) {
      m_d[0] -= rhs.m_d[0];
      m_d[1] -= rhs.m_d[1];
      m_d[2] -= rhs.m_d[2];
      return *this;
    }

    const Point3& operator*=(const double &d) {
      m_d[0] *= d;
      m_d[1] *= d;
      m_d[2] *= d;
      return *this;
    }

    //const Point3& operator*=(const Matrix3x4& q);

    const Point3& operator/=(const double &d) {
      return (*this) *= 1/d;
    }

    const double size() const {
      double d = 0.0;
      d += m_d[0]*m_d[0];
      d += m_d[1]*m_d[1];
      d += m_d[2]*m_d[2];
      return sqrt(d);
    }

    const Point3& unitize() {
      double d = size();
      if( d == 0 ) {
	(*this) = Point3(1,0,0);
      }
      else {
	(*this) /= d;
      }
      return (*this);
    }

    static const double dot(const Point3& p, const Point3& q) {
      double d = 0.0;
      d += p[0]*q[0];
      d += p[1]*q[1];
      d += p[2]*q[2];
      return d;
    }

    static const double size(const Point3& p) {
      return p.size();
    }

    static const Point3 unit(const Point3& p) {
      return Point3(p).unitize();
    }

    /// Returns the cross product of x and y
    static const Point3 cross(const Point3& x, const Point3& y) {

      Point3 z = Point3(x[1]*y[2] - x[2]*y[1],
			x[2]*y[0] - x[0]*y[2],
			x[0]*y[1] - x[1]*y[0]);

      return z;
    }

    static double dist(const Point3& x, const Point3& y) {
      Point3 z(x);
      z -= y;
      return sqrt(z[0]*z[0] + z[1]*z[1] + z[2]*z[2]);
    }

    static double dist_squared(const Point3& x, const Point3& y) {
      Point3 z(x);
      z -= y;
      return z[0]*z[0] + z[1]*z[1] + z[2]*z[2];
    }

    static const double calculate_angle(const Point3& a, const Point3& b, const Point3& c);

    const Point3 operator-() const {
      return Point3(-m_d[0],-m_d[1],-m_d[2]);
    } // operator-

    const string getR() const;

  }; // class Point3

  inline const Point3 operator+(const Point3& p, const Point3& q) {
    return Point3(p) += q;
  }

  inline const Point3 operator-(const Point3& p, const Point3& q) {
    return Point3(p) -= q;
  }

  inline const Point3 operator*(const Point3& p, const double &d) {
    return Point3(p) *= d;
  }

  inline const Point3 operator/(const Point3& p, const double &d) {
    return Point3(p) *= 1/d;
  }

  inline bool operator==(const Point3& p, const Point3& q ) {
    return
      p[0] == q[0] &&
      p[1] == q[1] &&
      p[2] == q[2];
  } // operator==

  ostream& operator<<(ostream& out, const Point3& p);
  istream& operator>>(istream& in, Point3& p);

} // namespace murphp

#endif // MURPHP_CLASSES_HH
