// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13687 $
//  $Date: 2007-03-23 09:22:54 -0500 (Fri, 23 Mar 2007) $
//  $Author: yiliu $


// Rosetta Headers
#include "nblist.h"
#include "all_atom.h"
#include "all_atom_type.h"
#include "cenlist.h"
#include "constraints.h"
#include "count_pair.h"
#include "constraints.h"
#include "current_pose.h"
#include "fullatom.h"
#include "fullatom_energy.h"
#include "hbonds_ns.h"
#include "jumping_minimize.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "minimize_chi.h"
#include "misc.h"
#include "nblist_ns.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "param_torsion.h"
#include "prof.h"
#include "recover.h"
#include "runlevel.h"
#include "score.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/FArray4D.hh>
//#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>

//Utility Headers
#include <utility/basic_sys_util.hh>

// Namespaces
namespace use_nblist_common {
	bool use_nblist = { false };
}

////////////////////////////////////////////////////////////////////////////////
/// @begin append_atom_maps
///
/// @brief
///
/// @detailed
/// ctsa
///
/// 1) given an atom, identified by residue number and
///    residue atom number, assign this atom a global number
///    and fill in the convertion map between
///    global and residue atom numbers
///
/// 2) for each free torsion angle, associate the lowest
///    numbered atom contained in the rigid body which the torsion
///    torsion angle rotates. store this in structure:
///    torsion2rigid_body_start_map
///
/// atom specification:
/// closest torsion angle to atom in n-term/-chi direction:
///
/// @param  res_num - [in/out]? -
/// @param  res_atm_num - [in/out]? -
/// @param  torsion_res_num - [in/out]? -
/// @param  torsion_type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
append_atom_maps(
	int res_num,
	int res_atm_num,
	int torsion_res_num,
	int torsion_type
)
{
	using namespace nblist;

//------------------------------------------------------------------------------
	++total_atoms;

	// <murphp>
	if( total_atoms > MAX_TOTAL_ATOMS() ) {
		using namespace std;
		cout << "WARNING!!! total_atoms > MAX_TOTAL_ATOMS() - impending doom!!!" << endl;
	}
	// </murphp>

	//// fill in global atm <-> res atm conversion maps:
	////
	global_atm_num2res_num_map(total_atoms) = res_num;
	global_atm_num2res_atm_num_map(total_atoms) = res_atm_num;
	res_atm_num2global_atm_num_map(res_atm_num,res_num) = total_atoms;

	//// special map used to find the start of the rigid body associated with
	//// a torsion angle. Used during minimization functions:
	////
	if ( torsion_res_num > 0 && torsion_type > 0 ) {
		int & maptt( torsion2rigid_body_start_map(torsion_type,torsion_res_num) );
		if ( maptt == undefined_global_atm ) maptt = total_atoms;
	}

}

// <murphp>
////////////////////////////////////////////////////////////////////////////////
/// @begin count_total_atoms
///
/// @brief
///
/// @detailed
///   counts the total number of atoms that will be needed in the nblist arrays
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
count_total_atoms(const bool fullatom,
									const FArray1D_bool& ha,
									const FArray1D_bool& ha2,
									const FArray1D_bool& hn,
									const FArray4D_bool& closest_chi ) {

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace nblist;
	using namespace param;
	using namespace param_aa;
	using namespace param_torsion;
	using namespace runlevel_ns;
	static int const closest_chi_size1 = MAX_CHI; // Dim1 of closest_chi

	int nscatom;
	int atmno,aa,aav;
	int const chi_required_size1 = chi_required.size1();


	int my_total_atoms = 0;
	for ( int j = 1; j <= total_residue; ++j ) {
		aa = res(j);
		aav = res_variant(j);
		int const HNpos_aa = HNpos(aa,aav);

		++my_total_atoms; //append_atom_maps(j,1,j-1,psi_torsion); // N

//bk HN for constraints
		if ( hn(j) ) ++my_total_atoms; // append_atom_maps(j,classical_constraints::constants::H_ATOMNUM,j-1,omega_torsion); // HN

//bk fullatom arrays have a separate HN
		if ( fullatom && HNpos_aa != 0 ) {
			++my_total_atoms; //append_atom_maps(j,HNpos_aa,j-1,omega_torsion); // HN
		}

		++my_total_atoms; //append_atom_maps(j,2,j-1,omega_torsion); // ca(j)
		++my_total_atoms; //append_atom_maps(j,3,j,phi_torsion); // c(j)

//bk HA for constraints
		if ( ha(j) ) ++my_total_atoms; //append_atom_maps(j,classical_constraints::constants::HA_ATOMNUM,j,phi_torsion); // HA(j)
		if ( ha2(j) ) ++my_total_atoms; //append_atom_maps(j,classical_constraints::constants::HA2_ATOMNUM,j,phi_torsion); // HA2(j)

		if ( fullatom ) {
			nscatom = nside(aa,aav);
			//rh - Fix undefined atom problem with variants that do not have the same number
			//     of total atoms
			// Amino acid variants that do not have the same number of amino acids
			// (ex. protonated variant) cause the total_atom count in append_atom_maps()
			// to be improperly set. append_atom_maps does not have any information about
			// amino acid variant, only takes the amino acid and atom number into
			// consideration. For example, if a lys variant with 21 atoms is read into
			// append_atom_map() then if a variant with 22 atoms is later used, the 22nd atom
			// will be undefined in Rosetta. This causes the output of the "write undef_atom"
			// error message and leads to an assert error at
			// "int itmp = ( drv ? ++drv_nb_len(atom1) : ++nb_len(atom1) );"
			int const scoffset = first_scatom( aa, aav ) - 1;
			for ( int k = 1, lcr = chi_required.index(1,5,aa,aav); k <= nscatom;
			 ++k, lcr+=chi_required_size1 ) {
				atmno = k + scoffset;
				if ( atmno != HNpos_aa ) {
					//// ctsa - anything not controled by chi1 (therefore any chi) should
					//// come first (this should just be CB / HCA and maybe some of proline)
					////
					if ( !chi_required[ lcr ] ) { // !chi_required(1,atmno,aa,aav)
						++my_total_atoms; //append_atom_maps(j,atmno,j,phi_torsion);
						 // CB is not controlled by any chi's
					}
				}
			}

			//// ctsa - now progress up the chi angles, only adding those
			//// atoms rotated by that and no other upstream chi
			////
			for ( int chino = 1, lcc1 = closest_chi.index(chino,5,aa,aav),
			 chinoe = nchi(aa,aav); chino <= chinoe; ++chino, ++lcc1 ) {
				for ( int k = 1, lcc = lcc1; k <= nscatom;
				 ++k, lcc+=closest_chi_size1 ) {
					atmno = k + scoffset;
					if ( atmno != HNpos_aa ) {
						if ( closest_chi[ lcc ] ) { // closest_chi(chino,atmno,aa,aav)
							++my_total_atoms;	//append_atom_maps(j,atmno,j,total_bb_torsion+chino);

						}
					}
				}
			}
		} else {
			nscatom = 2; // cb & centroid
			int const k_start = ( aa == aa_gly ? 2 : 1 );
			for ( int k = k_start; k <= nscatom; ++k ) {
				atmno = k + 4;  // CB is atom  #5, CEN is #6
				++my_total_atoms; //append_atom_maps(j,atmno,j,phi_torsion);
			}
		}

		//append_atom_maps(j,4,j,psi_torsion); // o(j)
		++my_total_atoms;

	}

	return my_total_atoms;

} // count_total_atoms

// </murphp>


////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allatom_list
///
/// @brief
///
/// @detailed
///car generates lists of all atoms in the protein
///car atoms must in N->Cterminal order such that all atoms following
///car a torsion angle are found after the C-terminal atom in the bond
///car of the torsion angle.
///
///car WARNING functions in the nblist assume that atom 1 is the N and that
///car this is the first atom in the nblist for any particular residue!!
///
///car must be done after constraints set up
///
/// @global_read
///
/// @global_write
///
/// @remarks
///ctsa Note that in fullatom mode, the atom order now includes
///ctsa atoms from sidechains in chi1->chimax order
///
///car should this be self initializing?  set it up when you're going
///car to minimize or use the nblist?
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_allatom_list()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace nblist;
	using namespace param;
	using namespace param_aa;
	using namespace param_torsion;
	using namespace runlevel_ns;

//car local
	static FArray1D_bool ha( MAX_RES() );
	static FArray1D_bool ha2( MAX_RES() );
	static FArray1D_bool hn( MAX_RES() );
	static FArray4D_bool closest_chi( MAX_CHI, MAX_ATOM(), MAX_AA(), MAX_AA_VARIANTS() );
	static int const closest_chi_size1 = MAX_CHI; // Dim1 of closest_chi
	int nscatom;
	int atmno,aa,aav;
	int const chi_required_size1 = chi_required.size1();

	bool const fullatom = get_fullatom_flag();

//car figure out what protons must be added for constraints...
	ha2 = false;
	ha = false;
	hn = false;
	int np_pairs = classical_constraints::BOUNDARY::np_pairs();
	for ( int i = 1, lpa = -1; i <= np_pairs; ++i ) {
		int const pa1 = classical_constraints::BOUNDARY::pair_atom_index()[ ++lpa ]; // pair_atom_index(1,i)
		if ( pa1 == classical_constraints::constants::H_ATOMNUM ) {
			hn( classical_constraints::BOUNDARY::constraintPair(i,1) ) = true;
		} else if ( pa1 == classical_constraints::constants::HA_ATOMNUM ) {
			ha( classical_constraints::BOUNDARY::constraintPair(i,1) ) = true;
		} else if ( pa1 == classical_constraints::constants::HA2_ATOMNUM ) {
			ha2( classical_constraints::BOUNDARY::constraintPair(i,1) ) = true;
		}
		int const pa2 = classical_constraints::BOUNDARY::pair_atom_index()[ ++lpa ]; // pair_atom_index(2,i)
		if ( pa2 == classical_constraints::constants::H_ATOMNUM ) {
			hn( classical_constraints::BOUNDARY::constraintPair(i,2) ) = true;
		} else if ( pa2 == classical_constraints::constants::HA_ATOMNUM ) {
			ha( classical_constraints::BOUNDARY::constraintPair(i,2) ) = true;
		} else if ( pa2 == classical_constraints::constants::HA2_ATOMNUM ) {
			ha2( classical_constraints::BOUNDARY::constraintPair(i,2) ) = true;
		}
	}

	if ( fullatom ) get_closest_chi(closest_chi);

	// <murphp>
	const int my_total_atoms = count_total_atoms(fullatom,ha,ha2,hn,closest_chi);
  //add a  0.5 atom per residue buffer for atom number changes during design
	const int atom_number_buffer = static_cast<int>( total_residue/2 );

	if ( MAX_TOTAL_ATOMS.initialized() &&  MAX_TOTAL_ATOMS() < my_total_atoms ){
		MAX_TOTAL_ATOMS.assign_if_bigger_nic(my_total_atoms + atom_number_buffer );
	} else if ( !MAX_TOTAL_ATOMS.initialized() ) {
		MAX_TOTAL_ATOMS.assign_if_nic(my_total_atoms + atom_number_buffer);
	}
	//std::cout << "GREPME my_total_atoms = " << my_total_atoms << ", MAX_TOTAL_ATOMS()=" << MAX_TOTAL_ATOMS() << std::endl;
	// </murphp>


	res_atm_num2global_atm_num_map = undefined_global_atm; // not on atomlist
	torsion2rigid_body_start_map = undefined_global_atm;

	total_atoms = 0;
	for ( int j = 1; j <= total_residue; ++j ) {
		aa = res(j);
		aav = res_variant(j);
		int const HNpos_aa = HNpos(aa,aav);

		append_atom_maps(j,1,j-1,psi_torsion); // N

//bk HN for constraints
		if ( hn(j) ) append_atom_maps(j,classical_constraints::constants::H_ATOMNUM,j-1,omega_torsion); // HN

//bk fullatom arrays have a separate HN
		if ( fullatom && HNpos_aa != 0 ) {
			append_atom_maps(j,HNpos_aa,j-1,omega_torsion); // HN
		}

		append_atom_maps(j,2,j-1,omega_torsion); // ca(j)
		append_atom_maps(j,3,j,phi_torsion); // c(j)

//bk HA for constraints
		if ( ha(j) ) append_atom_maps(j,classical_constraints::constants::HA_ATOMNUM,j,phi_torsion); // HA(j)
		if ( ha2(j) ) append_atom_maps(j,classical_constraints::constants::HA2_ATOMNUM,j,phi_torsion); // HA2(j)

		if ( fullatom ) {
			nscatom = nside(aa,aav);
			//rh - Fix undefined atom problem with variants that do not have the same number
			//     of total atoms
			// Amino acid variants that do not have the same number of amino acids
			// (ex. protonated variant) cause the total_atom count in append_atom_maps()
			// to be improperly set. append_atom_maps does not have any information about
			// amino acid variant, only takes the amino acid and atom number into
			// consideration. For example, if a lys variant with 21 atoms is read into
			// append_atom_map() then if a variant with 22 atoms is later used, the 22nd atom
			// will be undefined in Rosetta. This causes the output of the "write undef_atom"
			// error message and leads to an assert error at
			// "int itmp = ( drv ? ++drv_nb_len(atom1) : ++nb_len(atom1) );"
			int const scoffset = first_scatom( aa, aav ) - 1;
			for ( int k = 1, lcr = chi_required.index(1,5,aa,aav); k <= nscatom;
			 ++k, lcr+=chi_required_size1 ) {
				atmno = k + scoffset;
				if ( atmno != HNpos_aa ) {
					//// ctsa - anything not controled by chi1 (therefore any chi) should
					//// come first (this should just be CB / HCA and maybe some of proline)
					////
					if ( !chi_required[ lcr ] ) { // !chi_required(1,atmno,aa,aav)
						append_atom_maps(j,atmno,j,phi_torsion);
						 // CB is not controlled by any chi's
					}
				}
			}

			//// ctsa - now progress up the chi angles, only adding those
			//// atoms rotated by that and no other upstream chi
			////
			for ( int chino = 1, lcc1 = closest_chi.index(chino,5,aa,aav),
			 chinoe = nchi(aa,aav); chino <= chinoe; ++chino, ++lcc1 ) {
				for ( int k = 1, lcc = lcc1; k <= nscatom;
				 ++k, lcc+=closest_chi_size1 ) {
					atmno = k + scoffset;
					if ( atmno != HNpos_aa ) {
						if ( closest_chi[ lcc ] ) { // closest_chi(chino,atmno,aa,aav)
							append_atom_maps(j,atmno,j,total_bb_torsion+chino);
						}
					}
				}
			}
		} else {
			nscatom = 2; // cb & centroid
			int const k_start = ( aa == aa_gly ? 2 : 1 );
			for ( int k = k_start; k <= nscatom; ++k ) {
				atmno = k + 4;  // CB is atom  #5, CEN is #6
				append_atom_maps(j,atmno,j,phi_torsion);
			}
		}

		append_atom_maps(j,4,j,psi_torsion); // o(j)

	}

	// <murphp>

// 	const std::string goodbad = (total_atoms == my_total_atoms ? "GOOD" : "BAD");
// 	if( true ) {
// 		using namespace std;
// 		cout << "GREPME ";
// 		for( int k = 0; k < 10; ++k ) cout << goodbad;
// 		cout << endl;
// 		cout << "GREPME total_atoms    = " << total_atoms << endl;
// 		cout << "GREPME my_total_atoms = " << my_total_atoms << endl;
// 		cout << "GREPME ";
// 		for( int k = 0; k < 10; ++k ) cout << goodbad;
// 		cout << endl;
// 	}

	// </murphp>

	if ( runlevel >= verbose ) {
		std::cout << "total_atoms" << SS( total_atoms ) << std::endl;
		std::cout << "global_atm_num2res_atm_num_map";
		for ( int i = 1; i <= total_atoms; ++i ) {
			std::cout << SS( global_atm_num2res_atm_num_map(i) );
		} std::cout << std::endl;
		std::cout << "global_atm_num2res_num_map";
		for ( int i = 1; i <= total_atoms; ++i ) {
			std::cout << SS( global_atm_num2res_num_map(i) );
		} std::cout << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_nblist
///
/// @brief
///
/// @detailed
///car WARNING: this function assumes that best_nb and nb differ only
///car as described by pair_moved
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_nblist()
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace misc;
	using namespace nblist;
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;
	using numeric::xyzVector_float;

	if ( !get_use_nblist() ) return;

	if ( score_check_current_pose() || pose_flag() ||
			 minimize_check_current_pose() ) {
		std::cout << "STOP: pose and update_nblist" << std::endl;
		std::cerr << "STOP: pose and update_nblist" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		//jmp_update_nblist();
		//return;
	}

	PROF_START( prof::UPDATE_NBLIST );
	static FArray2D_bool nbres( MAX_RES(), MAX_RES() );
	static FArray1D_int nbrs( MAX_RES() );

	int aa1,aa2,type1,type2,aav1,aav2;

	float ca_dis, dist2, cutoff;
	float cendist_ij;
	float cp_weight;

	bool const fullatom = get_fullatom_flag();
	minimize_update_fullcoord(); // copy position/centroid to fullcoord

//car for faster updating, nbs are copied from the nblist for pairs
//car with pair_moved == false;  Residues are copied in blocks. For example
//car when only a single rigid body move or frag move have happened
//car in this case, the pair moved array looks like this:
///    |<---- i axis ----->|
///    o . . . . . . . . . . __
///    F o . . . . . . . . .  ^
///    F F o . . . . . . . .  |
///    F F F o . . . . . . .  |   <----  I1
///    T T T T o . . . . . .  J   <----  I2
///    T T T T T o . . . . .  a
///    T T T T T T o . . . .  x   <----  I3
///    T T T T T T T o . . .  i   <----  I4
///    T T T T T T T F o . .  s
///    T T T T T T T F F o .  |
///    T T T T T T T F F F o  _
///    |<---- i axis ----->|
///      for ( i = 1; i <= total_residue; ++i ) {
///
/// for residue 1,


	FArray2DB_bool const & pair_moved( retrieve_pair_moved() );

//Distance checks
// if cen-cen <144 no atoms is residue can bump
// if ca-ca <49 no atom-atom bumps
// if ca-ca > 60 no atom-centroid bumps
// maybe should always use paircutoffs?
//bk calculate fullatom residue-residue neighborlist
	if ( fullatom ) {
		make_neighbor_info(res,total_residue,full_coord,nbres,nbrs);
	}

	reset_nblist();
	xyzVector_float full_coord_ii_i; // full_coord(1-3,ii,i)
	for ( int i = 1; i <= total_residue; ++i ) {

		int j = 1;
		for ( ; j != total_residue+1  && ! pair_moved(i,j); ++j ) {}
		copy_neighbors( i, 1, j-1, total_residue );
		if ( j > total_residue ) continue;  // next i

		{ // Scope  copy/calculate depending on pair_moved
		aa1 = res(i);
		aav1 = res_variant(i);
		int const natoms_aa1 = natoms(aa1,aav1);
		xyzVector_float const Epos_2i( &Eposition(1,2,i) ); // Eposition(1-3,2,i)

		int copy_start = j+1;

		for ( ; j <= total_residue; ++j ) {

			if ( ! pair_moved(i,j) ) continue;  // next j
			//copy nb for all skipped fixed pairs
			copy_neighbors(i,copy_start,j-1,total_residue);
			copy_start = j+1;   //next copy may start at j+1;

			if ( fullatom ) { // centroid too?
				if ( !nbres(i,j) ) goto L100;
				cendist_ij = cendist(i,j);
			} else {
				cendist_ij = cendist(i,j);
				if ( cendist_ij > cen_dist_cutoff2 ) goto L100;
			}

			{ // Scope
			aa2 = res(j);
			aav2 = res_variant(j);
			int const natoms_aa2 = natoms(aa2,aav2);

			ca_dis = distance_squared( Epos_2i, // Eposition(1-3,2,i)
			 xyzVector_float( &Eposition(1,2,j) ) ); // Eposition(1-3,2,j)

			for ( int ii = 1, iiend = ( fullatom ? natoms_aa1 : 6 ); ii <= iiend; ++ii ) {
				if ( fullatom || aa1 != aa_gly || ii != 5 ) {

   				full_coord_ii_i = &full_coord(1,ii,i); // full_coord(1-3,ii,i);

					type1 = all_atom_type( aa1, aav1, ii, fullatom );

					for ( int jj = 1, jjend = ( fullatom ? natoms_aa2 : 6 ); jj <= jjend; ++jj ) {
						if ( i == j && jj < ii ) goto L50;
						if ( !count_pair( i, ii, aa1, aav1, j, jj, aa2, aav2, fullatom, cp_weight ) )
							goto L50;
//car distance checks
						if ( !fullatom ) {
							if ( aa2 == aa_gly && jj == 5 ) goto L50; // gly cb
//          if ( ii == 6 ) {
// These checks not in vdw_compute, but alluded to
//            if ( jatom != 6 && ca_dis > ca_cen_dist_cutoff2 ) goto L50;
//          } else if ( jj == 6 ) {
//            if ( ca_dis > ca_cen_dist_cutoff2 ) goto L50;
//          } else {       // neither is a centroid
// this check in vdw_compute, but not in deriv
//            if ( ca_dis > ca_dist_cutoff2 ) goto L50;
//          }
						}

						if ( ii == 2 && jj == 2 ) {
							dist2 = ca_dis;
						} else if ( !fullatom && ii == 6 && jj == 6 ) {
							dist2 = cendist_ij;
						} else {
							dist2 = distance_squared( full_coord_ii_i, // full_coord(1-3,ii,i)
							 xyzVector_float( &full_coord(1,jj,j) ) ); // full_coord(1-3,jj,j)
						}

						type2 = all_atom_type(aa2,aav2,jj,fullatom);
						cutoff = all_atom_cutoff(type1,type2,fullatom);
						if ( dist2 < cutoff ) add_to_nblist(i,ii,j,jj,false);
						if ( runlevel >= verbose && dist2 < cutoff ) std::cout << "nb"
						 << SS(i) << SS( ii ) << SS( j ) << SS( jj )
						 << F( 7, 3, dist2 ) << F( 7, 3, cutoff ) << std::endl;
	L50:;
					} // jj

				}
			} // ii
			} // Scope
L100:;
		} // j
		copy_neighbors(i,copy_start,total_residue,total_residue);
		} // Scope
	} // i

	reset_nblist_ends();
	update_cst_nblist();
	if ( runlevel >= verbose ) write_nblist();
	PROF_STOP( prof::UPDATE_NBLIST );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_to_nblist
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  ii - [in/out]? -
/// @param  j - [in/out]? -
/// @param  jj - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_to_nblist(
	int & i,
	int & ii,
	int & j,
	int & jj,
	bool const drv   // true if added only to drv nb list
)
{
	using namespace nblist;
	using namespace param;
	using namespace misc;

	// numbers in total atom list
	int atom1 = res_atm_num2global_atm_num_map(ii,i);
	int atom2 = res_atm_num2global_atm_num_map(jj,j);
	if ( atom1 == undefined_global_atm || atom2 == undefined_global_atm ) {
		std::cout << "write undef_atom" << SS( ii ) << SS( i ) << SS( jj )
							<< SS( j ) << SS( atom1 ) << SS( atom2 ) << SS( res_variant(i) ) << SS( res_variant(j) ) << std::endl;
	}

	int itmp = ( drv ? ++drv_nb_len(atom1) : ++nb_len(atom1) );
	if ( itmp > MAX_NB ) {
#ifdef BOINC
		return;
#endif
		std::cout << "increase MAX_NB in param.cc" << std::endl;
		std::cout << "allatom: " << atom1 << " res: " << i << " atom: "
		 << ii << " has more than MAX_NB neighbors" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	nb(itmp,atom1) = atom2;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin reset_nblist_ends
///
/// @brief
///
/// @detailed
///car set the start and end point in nblist for neighbors that must
///car be considered by fullatom scoring functions
///car ie, eliminate double counting
///car nb_start should be either 0 or 1
///car nb_start & nb_end should always be set such that
///car for atom i with kth neighbor j [j=nb(k,i)]
///car global_atm_num2res_num_map(i) < global_atm_num2res_num_map(j)
///car since neighbors occur in order from n to c
///car  nb_end is always nb_len  (until other atoms are added by drv_update_nblist
///car  nb_start must be determined
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_nblist_ends()
{
	using namespace nblist;

	for ( int i = 1; i <= total_atoms; ++i ) {
		int res1 = global_atm_num2res_num_map(i);
		int j;
		for ( j = 1; j <= nb_len(i); ++j ) {
			int res2 = global_atm_num2res_num_map(nb(j,i));
			if ( res2 > res1 ) goto L100;
		}
L100:
		nb_start(i) = j;
		nb_end(i) = nb_len(i);
	}

}
////////////////////////////////////////////////////////////////////////////////
/// @begin reset_nblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_nblist()
{
	using namespace nblist;

	for ( int i = 1; i <= total_atoms; ++i ) {
		nb_len(i) = 0;
	}

	using namespace hbonds;
  hbond_set.set_nhbonds(0); // 	reset_hblist();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_neighbors
///
/// @brief
///car for atoms in res1,  copy the neighbor list from first_nb to last_nb
///
/// @detailed
///
/// @param  res1 - [in] - res #, aa-type
/// @param  first_res2 - [in] -
/// @param  last_res2 - [in] -
/// @param  total_residue - [in] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_neighbors(
	int const res1, // res #, aa-type
	int const first_res2,
	int const last_res2,
	int const total_residue
)
{
	using namespace nblist;
	using namespace param;
	using namespace runlevel_ns;

//car local
	int j_nb,jend;

	if ( res1 > total_residue ) return;
	if ( first_res2 > total_residue ) return;
	if ( last_res2 < first_res2 ) return;

	int const first_at = res_atm_num2global_atm_num_map(1,res1);
	int const last_at = ( res1 < total_residue ?
	 res_atm_num2global_atm_num_map(1,res1+1)-1 :
	 total_atoms );


	int const first_nb = res_atm_num2global_atm_num_map(1,first_res2);
	int const last_nb = ( last_res2 < total_residue ?
	 res_atm_num2global_atm_num_map(1,last_res2+1)-1 :
	 total_atoms );
	int itmp;

	for ( int i = first_at; i <= last_at; ++i ) {
		jend = best_nb_len(i);
		for ( int j = 1; j <= jend; ++j ) {
			j_nb = best_nb(j,i); // don't assume anything about order
//car potential speed up by assuming neighbor order
//car WARNING!! neighbors are in  order by residue number and atom number,
//car           but not by allatom number!!
			if ( j_nb > last_nb ) goto L200; // outside range
			if ( j_nb < first_nb ) goto L200; // outside range

//car add to nblist...
			itmp = ++nb_len(i); // add to i's nb list
			if ( itmp > MAX_NB ) {
#ifdef BOINC
				return;
#endif
				std::cout << "increase MAX_NB in param.cc" << std::endl;
				std::cout << "allatom: " << i << " res: " << res1 << " atom: " <<
				 global_atm_num2res_atm_num_map(i) <<
				 " has more than MAX_NB neighbors" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			nb(itmp,i) = j_nb;
			if ( runlevel >= verbose ) {
				std::cout << "COPYNB--nb" << SS(i) << SS(itmp) << SS(j_nb) << std::endl;
			}

L200:;
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin copy_nblist
///
/// @brief
///
/// @detailed
///
/// @param  total_atoms - [in/out]? -
/// @param  nb_in - [in/out]? -
/// @param  nb_len_in - [in/out]? -
/// @param  nb_start_in - [in/out]? -
/// @param  nb_end_in - [in/out]? -
/// @param  nb_out - [in/out]? -
/// @param  nb_len_out - [in/out]? -
/// @param  nb_start_out - [in/out]? -
/// @param  nb_end_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_nblist(
	int const total_atoms,
	FArray2Da_int nb_in,
	FArray1Da_int nb_len_in,
	FArray1Da_int nb_start_in,
	FArray1Da_int nb_end_in,
	FArray2Da_int nb_out,
	FArray1Da_int nb_len_out,
	FArray1Da_int nb_start_out,
	FArray1Da_int nb_end_out
)
{
	using namespace nblist;

	nb_in.dimension( MAX_NB, total_atoms );
	nb_len_in.dimension( total_atoms );
	nb_start_in.dimension( total_atoms );
	nb_end_in.dimension( total_atoms );
	nb_out.dimension( MAX_NB, total_atoms );
	nb_len_out.dimension( total_atoms );
	nb_start_out.dimension( total_atoms );
	nb_end_out.dimension( total_atoms );


	for ( int i = 1; i <= total_atoms; ++i ) {
		int const nb_len_in_i = nb_len_in(i);
		for ( int j = 1; j <= nb_len_in_i; ++j ) {
			nb_out(j,i) = nb_in(j,i);
		}
		nb_len_out(i) = nb_len_in_i;
		nb_start_out(i) = nb_start_in(i);
		nb_end_out(i) = nb_end_in(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin accept_best_nblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
accept_best_nblist()
{
	using namespace nblist;

	if ( !get_use_nblist() ) return;
	copy_nblist(total_atoms,nb,nb_len,nb_start,nb_end,best_nb,best_nb_len,
	 best_nb_start,best_nb_end);

	accept_best_hblist();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_best_nblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_best_nblist()
{
	using namespace nblist;

	if ( !get_use_nblist() ) return;
	copy_nblist(total_atoms,best_nb,best_nb_len,best_nb_start,best_nb_end,nb,
	 nb_len,nb_start,nb_end);
	update_cst_nblist();

	retrieve_best_hblist();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_use_nblist
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car turning on the nblist disables rotamer trials permanently
///car when the nblist is turned off, rotamer trials is still off!
///
///car turning on the nblist will overwrite the current pose with the best
///
///pb   jumping only uses the nblist for minimizing
///pb   and so doesnt need the best_nblist stuff
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_use_nblist( bool setting )
{
	//using namespace misc;
	using namespace use_nblist_common;

	if ( setting && !use_nblist && !pose_flag() ) { // PBHACK: will not catch all
		retrieve_best_pose();
		use_nblist = setting;

		score_set_new_pose(); // do not rely on any cached info in best_nb

		update_nblist();
		accept_best_nblist();

	}
	use_nblist = setting;

}

void
pose_set_use_nblist( bool setting )
{
	// not do all the crazy misc stuff
	using namespace use_nblist_common;
	use_nblist = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_use_nblist
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_use_nblist()
{
	using namespace use_nblist_common;

	if ( pose_flag() || minimize_check_current_pose() ||
			 score_check_current_pose() ) {
		// jumping only uses the nblist during minimization
		return ( use_nblist && get_currently_minimizing() );
	} else {
		return use_nblist;
	}
}

bool
get_use_nblist_during_minimization()
{
	using namespace use_nblist_common;

	return use_nblist;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin write_nblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_nblist()
{
	using namespace nblist;

	int res1,at1;
	int res2,at2,neigh,index2;

	for ( int i = 1; i <= total_atoms; ++i ) {
		res1 = global_atm_num2res_num_map(i);
		at1 = global_atm_num2res_atm_num_map(i);
		std::cout << "nblist" << SS( i ) << SS( res1 ) << SS( at1 ) <<
		 SS( nb_len(i) ) << std::endl;
		for ( int j = 1; j <= nb_len(i); ++j ) {
			std::cout << SS( nb(j,i) );
		} std::cout << std::endl;
		for ( int j = 1; j <= nb_len(i); ++j ) {
			neigh = nb(j,i);
			res2 = global_atm_num2res_num_map(neigh);
			at2 = global_atm_num2res_atm_num_map(neigh);
			index2 = res_atm_num2global_atm_num_map(at2,res2);
			if ( index2 != neigh || index2 == undefined_global_atm ) {
				std::cout << "mismatch" << SS( i ) << SS( res1 ) << SS( at1 ) <<
				 SS( nb_len(i) ) << SS( j ) << SS( neigh ) << SS( res2 ) << SS( at2 ) <<
				 SS( index2 ) << std::endl;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_cst_nblist
///
/// @brief
///car note which constraints are on the nblist;
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_cst_nblist()
{
	using namespace nblist::nb_list;
	using namespace classical_constraints::BOUNDARY;

	if ( !classical_constraints::BOUNDARY::get_constraints_exist() ) return;
	bool fullatom = get_fullatom_flag();

	classical_constraints::nblist::cst_nb = false;
	int np_pairs = classical_constraints::BOUNDARY::np_pairs();
	for ( int pair = 1; pair <= np_pairs; ++pair ) {
		int const res1 = constraintPair(pair,1);
		int const res2 = constraintPair(pair,2);
		int const cst_atom1 = pair_atom_index(1,pair);
		int const cst_atom2 = pair_atom_index(2,pair);

//car switch from constraint atom # convention: n,ca,cb,c,o sc-beyond-cb
		int atom1 = cst_atom1;
		if ( cst_atom1 == 4 ) {
			atom1 = 3;
		} else if ( cst_atom1 == 5 ) {
			atom1 = 4;
		} else if ( cst_atom1 == 3 ) {
			atom1 = 5;
		} else if ( cst_atom1 == classical_constraints::constants::CENTROID_ATOMNUM ) {
			atom1 = 6;
		}
		int atom2 = cst_atom2;
		if ( cst_atom2 == 4 ) {
			atom2 = 3;
		} else if ( cst_atom2 == 5 ) {
			atom2 = 4;
		} else if ( cst_atom2 == 3 ) {
			atom2 = 5;
		} else if ( cst_atom2 == classical_constraints::constants::CENTROID_ATOMNUM ) {
			atom2 = 6;
		}

		int const glb_atom1 = res_atm_num2global_atm_num_map(atom1,res1);
		int const glb_atom2 = res_atm_num2global_atm_num_map(atom2,res2);

		if ( glb_atom1 == nblist::undefined_global_atm ||  //constrained atoms
		 glb_atom2 == nblist::undefined_global_atm ) {     //not on atom list:

			if ( cst_pair_satisfies_conditions(pair,fullatom,999) ) {
//all atoms for which constraints are evaluated must be on atom list or derivs
//will not be calculated
				std::cerr << "ERROR:: constrained atom pair not on atom list!!\n";
				std::cerr << "pair " << SS(pair) << " res1" << SS(res1) << " atom1" <<
				 SS(cst_atom1)  << " res2" << SS(res2) << " atom2" << SS(cst_atom2) <<
				 " fullatom" << SS(fullatom) << " global atom1" << SS(glb_atom1) <<
         " global atom2" << SS(glb_atom2) << "\n";
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

		} else {
//car are glb1,glb2 nbs?
			for ( int j_nb = 1; j_nb <= nb_len(glb_atom1) && nb(j_nb,glb_atom1) != glb_atom2; ++j_nb ) {
				if (nb(j_nb,glb_atom1) == glb_atom2 ) {
					classical_constraints::nblist::cst_nb(pair) = true;
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_hblist
///
/// @brief
///bk makes a list of pairs of atoms that are within hydrogen bonding range
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_hblist()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param_aa;
	using numeric::xyzVector_float;

	static float const cutoff = { 20.25 };

	if ( !get_fullatom_flag() ) {
		int const atm2 = 5;
		for ( int seqpos1 = 1; seqpos1 <= total_residue; ++seqpos1 ) {
			if ( res(seqpos1) != aa_pro ) { // Skip prolines
				int const atm1 = HNpos(res(seqpos1),res_variant(seqpos1));
				xyzVector_float const Epos_11( &Eposition(1,atm1,seqpos1) ); // Eposition(1-3,atm1,seqpos1)
				for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
					if ( seqpos2 != seqpos1 ) { //bk calculate distances between backbone Nhs and OC
						float const dist_sq = distance_squared(
						 Epos_11, // Eposition(1-3,atm1,seqpos1)
						 xyzVector_float( &Eposition(1,atm2,seqpos2) ) ); // Eposition(1-3,atm2,seqpos2)

						if ( dist_sq < cutoff ) add_to_hblist( atm1, seqpos1, atm2, seqpos2 );
					}
				}
			}
		}
	}
//$$$
//$$$      } else {       // fullatom hbond neighbor list
//$$$
//$$$         for ( int res1 = 1; res1 <= total_residue; ++res1 ) { // loop through donor residues
//$$$            int aa1 = res(res1);
//$$$            for ( int hnum = 1, hnume = nH_polar(aa1); hnum <= hnume; ++hnum ) { // loop through polar hydrogens
//$$$               hatm = Hpos_polar(hnum,aa1);
//$$$               for ( int res2 = 1; res2 <= total_residue; ++res2 ) { // loop through acceptor residues
//$$$                  if ( res1 == res2 ) goto L150;
//$$$                  int aa2 = res(res2);
//$$$                  for ( int anum = 1, anume = nacceptors(aa2); anum <= anume; ++anum ) { // loop through acceptor atoms
//$$$                     aatm = accpt_pos(anum,aa2);
//$$$                     dis2 =
//$$$                      square( full_coord(1,hatm,res1) - full_coord(1,aatm,res2) ) +
//$$$                      square( full_coord(2,hatm,res1) - full_coord(2,aatm,res2) ) +
//$$$                      square( full_coord(3,hatm,res1) - full_coord(3,aatm,res2) );
//$$$                     if ( dis2 < cutoff ) {
//$$$                        add_to_hblist(hatm,res1,aatm,res2);
//$$$                     }
//$$$                  }
//$$$ L150:;
//$$$               }
//$$$            }
//$$$         }
//$$$      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_to_hblist
///
/// @brief
///
/// @detailed
///
/// @param  donh_atm - [in/out]? -
/// @param  don_res - [in/out]? -
/// @param  act_atm - [in/out]? -
/// @param  act_res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_to_hblist(
	int const donh_atm,
	int const don_res,
	int const act_atm,
	int const act_res
)
{
	using namespace hbonds;

	int const nhbonds = hbond_set.nhbonds();
	hbond_set.hbdonh_atm(nhbonds) = donh_atm;
	hbond_set.hbdon_res(nhbonds) = don_res;
	hbond_set.hbact_atm(nhbonds) = act_atm;
	hbond_set.hbact_res(nhbonds) = act_res;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin accept_best_hblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
accept_best_hblist()
{
	using namespace hbonds;

	best_hbond_set = hbond_set;
	//	copy_hblist(nhbonds,hbdonh_atm,hbdon_res,hbact_atm,hbact_res,best_nhbonds,
	//	 best_hbdonh_atm,best_hbdon_res,best_hbact_atm,best_hbact_res);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_best_hblist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_best_hblist()
{
	using namespace hbonds;

	hbond_set = best_hbond_set;
	//	copy_hblist(best_nhbonds,best_hbdonh_atm,best_hbdon_res,best_hbact_atm,
	//	 best_hbact_res,nhbonds,hbdonh_atm,hbdon_res,hbact_atm,hbact_res);
}

