// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16122 $
//  $Date: 2007-07-19 13:25:34 -0400 (Thu, 19 Jul 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "aa_name_conversion.h"
#include "pdbstats.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_is_backbone.h"
#include "cenlist.h"
#include "count_pair.h"
#include "decoystats.h"
#include "design.h"
#include "DesignMap.h"
#include "dunbrack_pack.h"
#include "etable.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "gb_elec.h"
#include "gb_elec_ns.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "minimize.h"
#include "misc.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "pairenergy.h"
#include "param.h"
#include "param_aa.h"
#include "pdb.h"
#include "pdbstats_ns.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pdbstatistics_pack.h"
#include "planes.h"
#include "planes_ns.h"
#include "pose.h"
#include "pose_io.h"
#include "ramachandran.h"
#include "read_aaproperties.h"
#include "refold.h" // for GL routines
#include "score.h"
#include "silent_input.h"
#include "structure.h"
#include "template_pack.h"
#include "util_vector.h"
#include "vdw.h"
#include "water.h" // for get_hb_env_weight_lin

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


//bk   how to create functions for compiling pdbstatistics using rosetta
//bk   1) create a flag in options.cc - function get_pdbstats_options
//bk   2) create function that counts the statistics, add globals to pdbstats.h,
//bk      add call to count function in function get_pdbstats()
//bk   3) create function that writes out the statistics (use pdbstats.h)
//bk      add call to output function in function output_pdbstats
//bk      output_pdbstats is called after all the proteins have
//bk      been looped through


////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats
///
/// @brief  main function for calling functions that count pdb statistics
///
/// @detailed
///
/// @global_read   obtains flags from pdbstats.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman, 8/14/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats()
{
//bk compile statistics from the pdb

	using namespace pdbstats;

	if ( avg_E ) get_pdbstats_avgE();
	if ( avg_dis ) get_pdbstats_dis();
	if ( hbond_stats ) get_pdbstats_hbonds_and_water();
//chu
	if ( phipsichi ) get_pdbstats_phipsichi();
	if ( dun_rot ) get_pdbstats_dun_rot();
	if ( rep_atm_pair ) get_pdbstats_atom_pair_repE();
//km
	if ( phe_planes ) find_planes_in_sidechains();
	if ( cation_pi ) find_cation_pi();
	if ( carbonyl_o ) find_carbonyl_o();
	if ( rama_stats ) get_rama_stats();
	if ( solv_stats ) get_solvation_stats();
	if ( born_placeholder ) get_born_placeholder();
	if ( sasa_stats ) get_sasa_stats();
	if ( pair_energies) output_pair_energies();
//	if ( env_stats ) get_pdbensemble_stats();
	if ( env_stats ) get_env_stats();

  if ( arom_hbonds ) get_arom_hbond_stats(); // SJF
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_pdbstats
///
/// @brief  calls functions that output compiled statistics
///
/// @detailed
///
/// @global_read  uses flags in pdbstats.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 8/14/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_pdbstats()
{
	using namespace pdbstats;

	if ( avg_E ) output_stats_avgE();
	if ( avg_dis ) output_stats_dis();
	if ( hbond_stats ) output_stats_hbond();
	if ( phipsichi ) output_stats_phipsichi();
	if ( dun_rot ) output_stats_dun_rot();
	if ( rep_atm_pair ) output_stats_atom_pair_repE();
	if ( solv_stats ) output_solv_stats();
	if ( born_placeholder ) output_born_placeholder();
	if ( sasa_stats ) output_sasa_stats();
	if ( env_stats ) output_env_stats(); //output_pdbensemble_stats();
  if ( arom_hbonds ) output_arom_hbond_stats(); // SJF
//	if ( env_stats ) output_pdbensemble_stats();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_hbonds_and_water
///
/// @brief what % of backbone polar groups are hbonding are have room for H2O
///
/// @detailed
///
/// @global_read
///     param.h, misc.h, aaproperties_pack.h, hbonds.h, pdbstats.h
///
/// @global_write
///     pdbstats.h
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 8/22/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_hbonds_and_water()
{
//bk what fraction of polar groups are hydrogen bonding or have room for a water

	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace pdbstats;

//bk how close can a heavyatom come to a water
	int const wdis = { 2 };

//bk maximum number of waters per amino acid
	int const max_waterpos = { 2 };

	FArray3D_float water_position( 3, max_waterpos, MAX_RES()() );
	float dis;
	FArray2D_int hnb( 2, MAX_RES()() );
	FArray2D_bool no_water( max_waterpos, MAX_RES()() );
	FArray2D_bool yes_hbond( max_waterpos, MAX_RES()() );

	set_fullatom_flag(true);

//bk calculte where waters would be if they were hbonding to the backbone atoms
//bk and put coordinates in water_position

	hbond_neighbors(total_residue,res,full_coord,hnb,water_position);

//bk check to see if water could hbond to backbone atom with out clashes
	no_water(1,1) = true;
	no_water(2,1) = true;
	no_water(1,total_residue) = true;
	no_water(2,total_residue) = true;

	for ( int res1 = 2; res1 < total_residue; ++res1 ) {
		for ( int pos = 1; pos <= 2; ++pos ) { // amides and carbonyls
			no_water(pos,res1) = false;
			for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
				if ( res2 != res1 ) {
					int aa2 = res(res2);
					for ( int atm2 = 1, atm2e = nheavyatoms( aa2, res_variant(res2) ); atm2 <= atm2e; ++atm2 ) {
						distance_bk( water_position(1,pos,res1), full_coord(1,atm2,res2), dis );
						if ( dis < wdis ) {
							no_water(pos,res1) = true;
//							std::cout << "waterp" << SS( res1 ) << SS( res2 ) << SS( pos ) <<
//							 SS( atm2 ) << SS( aa2 ) << SS( dis ) << std::endl;
						}
					}
				}
			} // res2
		} // pos
	} // res1

//bk call fullatom energy to check if group is hydrogen bonding with another
//bk protein group

	fullatom_energy_full(res, res_variant, full_coord, total_residue);
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int pos = 1; pos <= 2; ++pos ) {
			yes_hbond(pos,res1) = false;
		}
	}

	for ( int i = 1; i <= hbond_set.nhbonds(); ++i ) {
		if ( atom_base( hbond_set.hbdonh_atm(i), res(hbond_set.hbdon_res(i) ), res_variant(hbond_set.hbdon_res(i))) == 1 )
		 yes_hbond(1,hbond_set.hbdon_res(i)) = true;
		if ( hbond_set.hbact_atm(i) == 4 ) yes_hbond(2,hbond_set.hbact_res(i)) = true;
	}

	std::cout << "the number of hbonds " << hbond_set.nhbonds() << std::endl;

//bk compile counts
	for ( int res1 = 2; res1 < total_residue; ++res1 ) {
		if ( res(res1) != 13 ) { // no prolines
			++numNH;
			if ( !yes_hbond(1,res1) ) {
				++numNH_nhb;
			}
			if ( no_water(1,res1) ) {
				++numNH_nwat;
			}
			if ( (!yes_hbond(1,res1)) && no_water(1,res1) ) {
				++numNH_nhb_nwat;
			}

			std::cout << "amide" << res1 << " hbond " << L( yes_hbond(1,res1) ) <<
			 " wat " << L( no_water(1,res1) ) << std::endl;

		}
		++numO;
		if ( !yes_hbond(2,res1) ) {
			++numO_nhb;
		}
		if ( no_water(2,res1) ) {
			++numO_nwat;
		}
		if ( (!yes_hbond(2,res1)) && no_water(2,res1) ) {
			++numO_nhb_nwat;
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
//\BEGIN get_born_placeholder
//
//\BRIEF	what is the best dist/rad value combo for GB place holders
//
//\FULL
//
//\GLOBAL_READ
//
//
//\GLOBAL_WRITE
//
//\NOTES
//
//\COMMENTERS	Jim Havranek	 9/10/04
//
//\END
////////////////////////////////////////////////////////////////////////////////
void
get_born_placeholder()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace pdb;
	using namespace pdbstats;
	using namespace gb_elec;


	set_fullatom_flag(true);

	std::cout << "Getting real born radii" << std::endl;
// jjh get the real born radii, and transfer to stored array
	gb_get_all_born_radii_dummy(-1);
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		int aa1 = res(res1);
		int aav1 = res_variant(res1);
		for ( int atm1 = 1, atm1e = natoms(aa1,aav1); atm1 <= atm1e; ++atm1 ) {
			real_rb(atm1, res1) = born_rad(atm1, res1);
		}
	}

	std::cout << "Done getting real born radii" << std::endl;
	for ( float work_param = 2.00; work_param <= 3.0001; work_param += 0.01 ) {

		rb_err_accum = 0.0;

		gb_dummy_dist = work_param;

//	std::cout << "Building dummy placeholders" << std::endl;

// Build all the dummy place holders

		//---FArray1D_bool dummy_protamers( MAX_RES()() );
		//---dummy_protamers = false;
		DesignMap local_design_map( misc::total_residue, misc::res );
		//---		gb_build_placeholders(total_residue, full_coord, res, res_variant,dummy_protamers);
		gb_build_placeholders(total_residue, full_coord, res, res_variant,
													local_design_map);
//	std::cout << "Done building dummy placeholders" << std::endl;

// Get born radii with all dummies for other positions

		gb_get_all_born_radii_dummy(0);

// Accumulate the error in born radii

		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
			int aa2 = res(res2);
			int aav2 = res_variant(res2);
			for ( int atm2 = 1, atm2e = natoms(aa2,aav2); atm2 <= atm2e; ++atm2 ) {
//				std::cout << "Real br " << real_rb(atm2, res2) << " dummy " << born_rad(atm2, res2) << std::endl;
				rb_err_accum += ( real_rb(atm2, res2) - born_rad(atm2, res2) ) *
				 ( real_rb(atm2, res2) - born_rad(atm2, res2) );
			}
		}
		std::cout << work_param << " " << rb_err_accum << std::endl;
	}
}


void
output_born_placeholder()
{
	using namespace pdbstats;

	std::cout << rb_err_accum << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_dis
///
/// @brief what are the average distances between atom types
///
/// @detailed
///
/// @global_read
///     param.h, misc.h, aaproperties_pack.h, pdb.h
///
/// @global_write
///     pdbstats.h
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman  8/22/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_dis()
{
//bk what are the average distances between different atom types in the pdb

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace pdb;
	using namespace pdbstats;

	int const min_sep = { 1 }; //bk min separation between residues
//	float const bfactor_cutoff = { 27.0 };
	 // only consider atoms with bfactor below this


	set_fullatom_flag(true);

//bk loop over residues and atoms
	float dis, cp_weight;
	float const rdb = disbin_size;
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = res1+min_sep; res2 <= total_residue; ++res2 ) {
			int aa1 = res(res1);
			int aa2 = res(res2);
			int aav1 = res_variant(res1);
			int aav2 = res_variant(res2);
			for ( int atm1 = 1, atm1e = natoms(aa1,aav1); atm1 <= atm1e; ++atm1 ) {
				int atm_type1 = fullatom_type(atm1,aa1,aav1);
				for ( int atm2 = 1, atm2e = natoms(aa2,aav2); atm2 <= atm2e; ++atm2 ) {
					if ( bvalue(atm1,res1) < 27.0 && bvalue(atm2,res2) < 27.0 &&
					 count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,cp_weight) ) {
						distance_bk(full_coord(1,atm1,res1),full_coord(1,atm2,res2),dis);
						int disbin = static_cast< int >( dis / rdb ) + 1;
						if ( disbin <= ndisbins ) {
							int atm_type2 = fullatom_type(atm2,aa2,aav2);
							if ( atm_type1 <= atm_type2 ) {
								++cnt_dis(atm_type1,atm_type2,disbin);
							} else {
								++cnt_dis(atm_type2,atm_type1,disbin);
							}
						} // bvalue cut
					}
				} // atm2
			} // atm1
		} // res2
	} // res1

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_avgE
///
/// @brief  what are the average energies of aa in diff neighbor env
///
/// @detailed  this function is used to calculate the average energy of each
///       type of amino acid in different neighbor environments.  These
///       statistics are particularly useful for defining good packing
///       energies
///
/// @global_read  param.h, misc.h, template_pack.h, files_paths.h,
///              aaproperties_pack.h, fullatom_energies.h, pdbstats.h
///
/// @global_write pdbstats.h
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman  8/22/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_avgE()
{
	//bk what are the average energies of amino acids in crystal structures

	using namespace fullatom_energies;
	using namespace misc;
	using namespace param;
	using namespace pdbstats;
	using namespace template_pack;

	set_fullatom_flag(true);

	fullatom_energy_full(res, res_variant, full_coord, total_residue);

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		int aa = res(seqpos);
		int nb;
		if ( neighbors(seqpos) < 5 ) {
			nb = 5;
		} else if ( neighbors(seqpos) > 30 ) {
			nb = 30;
		} else {
			nb = neighbors(seqpos);
		}
		sol_n(aa,nb) += solenergy(seqpos);
		pair_n(aa,nb) += pair_energy(seqpos);
		dun_n(aa,nb) += dunenergy(seqpos);
		one_n(aa,nb) += probenergy(seqpos);
		hb_n(aa,nb) += hbenergy(seqpos);
		atr_n(aa,nb) += atrenergy(seqpos);

//bk set the maximum rep energy to 2.0 so that bad apples can't skew the results.
		if ( repenergy(seqpos) < 2.0 ) {
			rep_n(aa,nb) += repenergy(seqpos);
			tlj_n(aa,nb) += atrenergy(seqpos)+repenergy(seqpos);
			tot_n(aa,nb) += resenergy(seqpos);
		} else {
			rep_n(aa,nb) += 2.0;
			tlj_n(aa,nb) += atrenergy(seqpos)+2.0;
			tot_n(aa,nb) += resenergy(seqpos)-repenergy(seqpos)+2.0;
		}
		//chu add intrares_energy, also set the maxium intrares rep energy to 2.0
		if ( intraresenergy(seqpos) < 2.0 ) {
			intrares_n(aa,nb) += intraresenergy(seqpos);
		} else {
			intrares_n(aa,nb) = 2.0;
			tot_n(aa,nb) += 2.0 - intraresenergy(seqpos);
		}
		++cnt_n(aa,nb);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_hbond
///
/// @brief output % of backbone polar groups hbonding with protein or h2o
///
/// @detailed  outputs statistics compiled by
///       'function get_pdbstats_hbonds_and_water'
///
/// @global_read  param.h, design.h, pdbstats.h, files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 8/22/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_stats_hbond()
{
	using namespace pdbstats;

	std::ofstream hs_stream( "hbond_statistics" );
	if ( !hs_stream ) {
		std::cout << "Open failed for file: " << "hbond_statistics" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	hs_stream <<
	 "numNH     numNH_nhb     numNH_nwat   numNH_nhb_nwat frac_nhb_nwat" << std::endl;
	hs_stream << I( 7, numNH ) << space( 4 ) <<
	 I( 7, numNH_nhb ) << space( 4 ) <<
	 I( 7, numNH_nwat ) << space( 4 ) <<
	 I( 7, numNH_nhb_nwat ) << space( 4 ) <<
	 F( 7, 3, float(numNH_nhb_nwat)/float(numNH) ) << std::endl;
	std::cout << std::endl;
	hs_stream <<
	 "numO    numO_nhb    numO_nwat    numO_nhb_nwat frac_nhb_nwat" << std::endl;
	hs_stream << I( 7, numO ) << space( 4 ) <<
	 I( 7, numO_nhb ) << space( 4 ) <<
	 I( 7, numO_nwat ) << space( 4 ) <<
	 I( 7, numO_nhb_nwat ) << space( 4 ) <<
	 F( 7, 3, float(numO_nhb_nwat)/float(numO) ) << std::endl;
	hs_stream.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_dis
///
/// @brief  outputs statistics about the avg distances between atoms
///
/// @detailed outputs statistics counted by
///      'function get_pdbstats_dis'
///
/// @global_read  param.h, pdbstats.h, files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 8/23/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_stats_dis()
{
	using namespace param;
	using namespace pdbstats;

	std::ofstream adfp_stream( "avg_dis_from_pdb" );
	if ( !adfp_stream ) {
		std::cout << "Open failed for file: " << "avg_dis_from_pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	for ( int atm_type1 = 1, atm_type_end = MAX_ATOMTYPES(); atm_type1 <= atm_type_end; ++atm_type1 ) {
		for ( int atm_type2 = 1; atm_type2 <= atm_type_end; ++atm_type2 ) {
			for ( int disbin = 1; disbin <= ndisbins; ++disbin ) {
				float const tot_size = ( disbin - 0.5 ) * disbin_size;
				float const norm_cnt =
				 cnt_dis(atm_type1,atm_type2,disbin) / ( tot_size * tot_size );
				adfp_stream << I( 2, atm_type1 ) << ' ' << I( 2, atm_type2 ) << ' ' <<
				 F( 6, 2, tot_size ) << ' ' << F( 20, 12, norm_cnt ) <<
				 I( 20, cnt_dis(atm_type1,atm_type2,disbin) ) << std::endl;
			}
		}
	}
	adfp_stream.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_avgE
///
/// @brief  output the energy of each amino acid type in diff env.
///
/// @detailed  output statistics compiled by
///       'function get_pdbstats_avgE'
///
/// @global_read  param.h, pdbstats.h,  files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_stats_avgE()
{
	using namespace pdbstats;
	using namespace param;
	float cnt,atr,rep,sol,pair,one,hb,dun,tot,tlj,intrares;

	std::ofstream afp_stream( "avgE_from_pdb" );
	if ( !afp_stream ) {
		std::cout << "Open failed for file: " << "avgE_from_pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	afp_stream << "aa  nb  atr  rep  tlj  sol  pair one  hb  dun  intrares  tot  cnt" << std::endl;
//KMa phospho_ser 2006-01
	for ( int aa = 1; aa <= MAX_AA_PLUS() ; ++aa ) {
		for ( int nb = 5; nb <= 30; ++nb ) {
			cnt = float(cnt_n(aa,nb));
			atr = atr_n(aa,nb)/cnt;
			rep = rep_n(aa,nb)/cnt;
			sol = sol_n(aa,nb)/cnt;
			pair = pair_n(aa,nb)/cnt;
			one = one_n(aa,nb)/cnt;
			hb = hb_n(aa,nb)/cnt;
			dun = dun_n(aa,nb)/cnt;
			intrares = intrares_n(aa,nb)/cnt;
			tot = tot_n(aa,nb)/cnt;
			tlj = tlj_n(aa,nb)/cnt;

			afp_stream << I( 2, aa ) << ' ' << I( 2, nb ) << ' ' <<
				F( 6, 2, atr ) << ' ' <<
				F( 6, 2, rep ) << ' ' <<
				F( 6, 2, tlj ) << ' ' <<
				F( 6, 2, sol ) << ' ' <<
				F( 6, 2, pair ) << ' ' <<
				F( 6, 2, one ) << ' ' <<
				F( 6, 2, hb ) << ' ' <<
				F( 6, 2, dun ) << ' ' <<
				F( 6, 2, intrares) << ' ' <<
				F( 6, 2, tot ) << ' ' <<
				I( 5, cnt_n(aa,nb) ) << std::endl;
		}
	}
	afp_stream.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_phipsichi
///
/// @brief calculates and saves phi,psi,chi
///
/// @detailed
///
/// @global_read  param.h,misc.h
///
/// @global_write  pdbstats.h
///
/// @remarks  unlike many of the pdbstats functions, this one is not meant
///        to compile counts from multiple pdb files
///
/// @references
///
/// @authors  Brian Kuhlman  8/23/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_phipsichi()
{
	using namespace misc;
	using namespace pdbstats;

	ang_from_pdb_wsc(total_residue,res,res_variant,full_coord,input_phi,input_psi,
	 input_chi,input_rot);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_phipsichi
///
/// @brief  writes out phi,psi and chi for each residue position in a protein
///
/// @detailed
///
/// @global_read  param.h, pdbstats.h, misc.h, files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman   8/14/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_stats_phipsichi()
{
	using namespace misc;
	using namespace pdbstats;

	std::string res3;

	std::ofstream pfp_stream( "phipsichi_from_pdb" );
	if ( !pfp_stream ) {
		std::cout << "Open failed for file: " << "phipsichi_from_pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	pfp_stream << "NO#   res     phi     psi      chi1     chi2     chi3" <<
	 "     chi4  r1  r2  r3  r4" << std::endl;
	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), res3 );
		pfp_stream << I( 3, i ) << "   " << res3 <<
		 "  " << F( 7, 2, input_phi(i) ) <<
		 "  " << F( 7, 2, input_psi(i) ) <<
		 "  " << F( 7, 2, input_chi(1,i) ) <<
		 "  " << F( 7, 2, input_chi(2,i) ) <<
		 "  " << F( 7, 2, input_chi(3,i) ) <<
		 "  " << F( 7, 2, input_chi(4,i) ) <<
		 "   " << I( 1, input_rot(1,i) ) <<
		 "   " << I( 1, input_rot(2,i) ) <<
		 "   " << I( 1, input_rot(3,i) ) <<
		 "   " << I( 1, input_rot(4,i) ) << std::endl;
	}
	pfp_stream.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_dun_rot
///
/// @brief  measures angles between closest dunbrack rotamer and chi angles in pdbfile
///
/// @detailed
///
/// @global_read  param.h, param_aa.h, design.h, pdbstats.h, misc.h,
///              dunbrack_pack.h
///
/// @global_write pdbstats.h
///
/// @remarks  key lines in the code are currently commented out, I am not sure why
///
/// @references
///
/// @authors  Brian Kuhlman  8/23/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_dun_rot()
{
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pdbstats;

	FArray1D_int nangles( MAX_CHI );
	int aa, linenum, iphi,ipsi,aav;
//	FArray1D_int drot( 2 );
//	float perc;
//	FArray1D_float dchi( 2 );
//	FArray1D_float dsd( 2 );
	float perc_total;
	FArray2D_float dun_chi_res( MAXANGLES_PER_CHI, MAX_CHI );
	FArray2D_int dun_rot_res( MAXANGLES_PER_CHI, MAX_CHI );
//  float delta_tmp;

	get_pdbstats_phipsichi();
	for ( int i = 1; i <= total_residue; ++i ) {
		get_phibin(input_phi(i),input_psi(i),iphi,ipsi,i,total_residue);
		linenum = 0; //// initialize
		perc_total = 0.0;
		aa = res(i);

		aav = res_variant(i);
		for ( int j = 1; j <= MAX_CHI; ++j ) {
			dun_closest_chi(j,i) = 0.0;
			delta_chi(j,i) = 180.0;
		}
		if ( aa == aa_ala || aa == aa_gly ) { // no rot for Gly and Ala
			for ( int j = 1; j <= MAX_CHI; ++j ) {
				nangles(j) = 1;
				dun_chi_res(1,j) = 0.0;
				dun_rot_res(1,j) = 0;
			}
		} else {
			while ( perc_total < 0.999 ) {
				++linenum;
//					dchi(1) = dun_chi(iphi,ipsi,aa,linenum,1);
//					dchi(2) = dun_chi(iphi,ipsi,aa,linenum,2);
//					drot(1) = dun_rotno(iphi,ipsi,aa,linenum,1);
//					drot(2) = dun_rotno(iphi,ipsi,aa,linenum,2);
//					dsd(1) = dun_sd(iphi,ipsi,aa,linenum,1);
//					dsd(2) = dun_sd(iphi,ipsi,aa,linenum,2);
//					perc = dun_prob(iphi,ipsi,aa,linenum);
//					get_extrachi(aa,aav,dchi,dsd,dun_chi_res,nangles,
//					 extrachi_cutoff+1,drot,dun_rot_res);
//car set neighbors > extrachi_cutoff to guarantee extrachi being used
//					for ( int j = 1; j <= MAX_CHI; ++j ) {
//						for ( int k = 1; k <= nangles(j); ++k ) {
//							angle_between(input_chi(j,i),dun_chi_res(k,j),delta_tmp);
//							if ( delta_tmp < delta_chi(j,i) ) {
//								dun_closest_chi(j,i) = dun_chi_res(k,j);
//								delta_chi(j,i) = delta_tmp;
//							}
//						} // per chi
//					} // one possibility per res
//					perc_total += perc;
			} // all possibilities per res
		}
	} // all the residues

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_dun_rot
///
/// @brief  output angle between real chi and closest dunbrack rotamer
///
/// @detailed
///
/// @global_read  param.h, misc.h, pdbstats.h, files_paths.h
///
/// @global_write  pdbstats.h
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 08/24/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_stats_dun_rot()
{
	using namespace misc;
	using namespace pdbstats;

	std::string res3;

	std::ofstream cdr_stream( "closest_dun_rot" );
	if ( !cdr_stream ) {
		std::cout << "Open failed for file: " << "closest_dun_rot" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	cdr_stream << "NO#   res" <<
	 "  r1    chi1  dun_chi1  delta_chi1" <<
	 "  r2    chi2  dun_chi2  delta_chi2" <<
	 "  r3    chi3  dun_chi3  delta_chi3" <<
	 "  r4    chi4  dun_chi4  delta_chi4" << std::endl;

	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), res3 );
		cdr_stream << I( 3, i ) << space( 4 ) << res3 << "   " <<
		 I( 1, input_rot(1,i) ) << ' ' <<
		 F( 7, 2, input_chi(1,i) ) << "   " <<
		 F( 7, 2, dun_closest_chi(1,i) ) << "    " <<
		 F( 7, 2, delta_chi(1,i) ) << ' ' << "   " <<
		 I( 1, input_rot(2,i) ) << ' ' <<
		 F( 7, 2, input_chi(2,i) ) << "   " <<
		 F( 7, 2, dun_closest_chi(2,i) ) << "    " <<
		 F( 7, 2, delta_chi(2,i) ) << ' ' << "   " <<
		 I( 1, input_rot(3,i) ) << ' ' <<
		 F( 7, 2, input_chi(3,i) ) << "   " <<
		 F( 7, 2, dun_closest_chi(3,i) ) << "    " <<
		 F( 7, 2, delta_chi(3,i) ) << ' ' << "   " <<
		 I( 1, input_rot(4,i) ) << ' ' <<
		 F( 7, 2, input_chi(4,i) ) << "   " <<
		 F( 7, 2, dun_closest_chi(4,i) ) << "    " <<
		 F( 7, 2, delta_chi(4,i) ) << std::endl;
	}
	cdr_stream.close();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbstats_atom_pair_repE
///
/// @brief
///chu what types of atom-pairs are clashing
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_pdbstats_atom_pair_repE()
{

	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace pdb;
	using namespace pdbstats;


	int const min_sep = { 1 }; //bk min separation between residues
//	float const bfactor_cutoff = { 27.0 };
	 // only consider atoms with bfactor below this


	set_fullatom_flag(true);

//bk loop over residues and atoms
	float cp_weight;
	float solE1,solE2,atrE,repE,elecE,d2;
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = res1+min_sep; res2 <= total_residue; ++res2 ) {
			int const aa1 = res(res1);
			int const aa2 = res(res2);
			int const aav1 = res_variant(res1);
			int const aav2 = res_variant(res2);
			for ( int atm1 = 1, atm1e = natoms(aa1,aav1); atm1 <= atm1e; ++atm1 ) {
				for ( int atm2 = 1, atm2e = natoms(aa2,aav2); atm2 <= atm2e; ++atm2 ) {
					if ( bvalue(atm1,res1) < 27.0 && bvalue(atm2,res2) < 27.0 &&
					 count_pair(res1,atm1,aa1,aav1,res2,atm2,aa2,aav2,true,cp_weight) ) {
						int const atm_type1 = fullatom_type(atm1,aa1,aav1);
						int const atm_type2 = fullatom_type(atm2,aa2,aav2);
            float const chrg1 = atomic_charge(atm1,aa1,aav1);
            float const chrg2 = atomic_charge(atm2,aa2,aav2);
            repE = 0.0;
            pairenergy(full_coord(1,atm1,res1),full_coord(1,atm2, res2), atm_type1, atm_type2, chrg1, chrg2,solE1,solE2,atrE,repE,elecE,d2);
						repE *= cp_weight;
						if ( repE > repE_atom_pair_cutoff ) {
							if ( atm_type1 <= atm_type2 ) {
								repE_atom_pair(atm_type1,atm_type2) += repE;
								++repN_atom_pair(atm_type1,atm_type2);
							} else {
								repE_atom_pair(atm_type2,atm_type1) += repE;
								++repN_atom_pair(atm_type2,atm_type1);
							}
						} // bvalue cut
						if ( repE > 0.1 && ( atm_type1 == 20 && atm_type2 == 23 ) ) {
							std::cout << "chu" << ' ' << start_file.substr(0,4) << ' ' <<
							 F( 8, 2, phi(res1) ) << ' ' <<
							 F( 8, 2, psi(res1) ) << ' ' <<
							 F( 8, 2, phi(res2) ) << ' ' <<
							 F( 8, 2, psi(res2) ) << ' ' <<
							 F( 8, 2, omega(res2) ) << ' ' <<
							 F( 8, 2, omega(res1) ) << ' ' <<
							 I( 3, res1 ) << ' ' <<
							 I( 3, res2 ) << ' ' <<
							 I( 3, aa1 ) << ' ' <<
							 I( 3, aa2 ) << ' ' <<
							 I( 3, atm1 ) << ' ' <<
							 I( 3, atm2 ) << ' ' <<
							 I( 3, atm_type1 ) << ' ' <<
							 I( 3, atm_type2 ) << ' ' <<
							 F( 6, 2, repE ) << std::endl;
						}
					}
				}            // atm2
			}               // atm1
		}                  // res2
	}                     // res1

}

//////////////////////////////////////////////////////////////////////////////
/// @begin output_stats_atom_pair_repE
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_stats_atom_pair_repE()
{
	using namespace files_paths;
	using namespace param;
	using namespace pdbstats;

	std::ofstream rc_stream( "repE_count_from_pdb" );
	for ( int atm_type1 = 1, atm_type_end = MAX_ATOMTYPES(); atm_type1 <= atm_type_end; ++atm_type1 ) {
		for ( int atm_type2 = 1; atm_type2 <= atm_type_end; ++atm_type2 ) {
			float avg;
			if ( repE_atom_pair(atm_type1,atm_type2) == 0 ) {
				avg = 0.0;
			} else {
				avg = repE_atom_pair(atm_type1,atm_type2) /
				 repN_atom_pair(atm_type1,atm_type2);
			}
			rc_stream << I( 2, atm_type1 ) << ' ' << I( 2, atm_type2 ) << ' ' <<
			 I( 6, repN_atom_pair(atm_type1,atm_type2) ) << ' ' << F( 7, 3, avg ) <<
			 std::endl;
		}
	}
	rc_stream.close();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_planes_in_sidechains
///
/// @brief
///
/// @detailed
///km  this function finds horizontal and vertical distance between two
///    sidechains, and the angle between the planes through selected
///    sidechain atoms.
///    Sidechains considered are phe, tyr, trp, pro, arg, ile, leu, val, his
///
/// output file is "plane_angle_statistics and contains:
/// res1, res2, aa1, aa2, #neighbors1, #neighbors2, horiz_dist, vert_dist, angle.
///
//// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
find_planes_in_sidechains()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace pdb;
	using namespace planes_ns;
	using numeric::xyzVector_float;

// first and second residues in the pair being counted
	FArray2D_float pair_list( 2, MAX_RES()(), 0.0f );

// local:
	int aa1, aa2, aav1, aav2, relevant_aa1_start, relevant_aa1_end,
	 relevant_aa2_start, relevant_aa2_end, atom_pair_counter;
	xyzVector_float p1;
	xyzVector_float q1;
	xyzVector_float r1;
	xyzVector_float s1;
	xyzVector_float p2;
	xyzVector_float q2;
	xyzVector_float r2;
	xyzVector_float s2;
	float dis;
//	float lj_sep;
	float plane_angle,horiz_distance,vert_distance;
	FArray1D_int angles_distances( number_angle_bins, 0 );
	 // array to store plane angle and distance bin values
	FArray1D_int points_plane1( max_plane );
	FArray1D_int points_plane2( max_plane );

//	int const max_contacts = { 50 };
//	FArray1D_float distances( max_contacts );
//	float smallest_distance;
// to check for burial:
	FArray1D_int neighbors( MAX_RES()() );
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );

	std::ofstream pla_stream( "plane_angle_statistics" );
	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

// initialize:
	int pair_found = 0;

// Loop over all residues and atoms and calculate distances between F,Y,W,H,R,K
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
//		do res2 = res1+1,total_residue ! if you don't want to double count
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
		 // if you want to separate FY from YF
			aa1 = res(res1);
			aa2 = res(res2);
			aav1 = res_variant(res1);
			aav2 = res_variant(res2);

// select aromatic, cation, proline and hydrophobic aa pairs
			if ( ( aa1 == 5 || aa1 == 7 || aa1 == 8 || aa1 == 10 || aa1 == 13 ||
			 aa1 == 15 || aa1 == 18 || aa1 == 19 || aa1 == 20 ) &&
			 ( aa2 == 5 || aa2 == 7 || aa2 == 8 || aa2 == 10 || aa2 == 13 ||
			 aa2 == 15 || aa2 == 18 || aa2 == 19 || aa2 == 20 ) ) {
// Get list of sidechain atoms for each pair
				relevant_sidechain_atoms(aa1,aa2,relevant_aa1_start, relevant_aa1_end,
				 relevant_aa2_start,relevant_aa2_end);

// Start counting number of atom pairs that come within 4A of each other
				atom_pair_counter = 0;
				for ( int atom1 = relevant_aa1_start; atom1 <= relevant_aa1_end;
				 ++atom1 ) { // sidechain atoms
					for ( int atom2 = relevant_aa2_start; atom2 <= relevant_aa2_end;
					 ++atom2 ) { // sidechain atoms
						if ( bvalue(atom1,res1) < 40.0 && bvalue(atom2,res2) < 40.0 ) {
							distance_bk(full_coord(1,atom1,res1),full_coord(1,atom2,res2),dis);
// Can also define a contact as the sum of lj_radii + fudge factor
//							lj_sep = fa_lj_radius(atom1) + fa_lj_radius(atom2);
//							if ( dis < lj_sep + 1.0 ) {
// Or define contacts using cutoff distance:
							if ( dis < 4.2 ) {
								if ( res1 != res2 ) {
									++atom_pair_counter;

// Define how many contacts sidechains should have
									if ( atom_pair_counter >= 2 ) {

// Only include of both residues are buried (use for hydrophobics, comment out
// for aromatics/cations
										if ( neighbors(res1) > 18 && neighbors(res2) > 18 ) {

// skip or check n, n+1 pairs:
//										if ( res2 - res1 == 1 ) { // N, N+1 in sequence

// check to see if this pair of residues is in the list of those already
// calculated, try again if true
											for ( int junk = 1, junk_end = MAX_RES(); junk <= junk_end; ++junk ) {
												if ( ( pair_list(1,junk) == res1 ) &&
												 ( pair_list(2,junk) == res2 ) ) {
													goto L200;
												}
											}
// res1 and res2 have met the criteia of having at least two contacts (<4.2A),
// Choose number and identity of atoms to define plane
											select_plane_points(aa1,aa2,points_plane1, points_plane2);

											for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
												if ( i == points_plane1(1) ) {
													p1 = &full_coord(1,i,res1);
												} else if ( i == points_plane1(2) ) {
													q1 = &full_coord(1,i,res1);
												} else if ( i == points_plane1(3) ) {
													r1 = &full_coord(1,i,res1);
												} else if ( i == points_plane1(4) ) {
													s1 = &full_coord(1,i,res1);
												}
											}
											for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
												if ( j == points_plane2(1) ) {
													p2 = &full_coord(1,j,res2);
												} else if ( j == points_plane2(2) ) {
													q2 = &full_coord(1,j,res2);
												} else if ( j == points_plane2(3) ) {
													r2 = &full_coord(1,j,res2);
												} else if ( j == points_plane2(4) ) {
													s2 = &full_coord(1,j,res2);
												}
											}

// now get plane_angle, vertical and horizontal distances for this pair
											if ( ( aa1 == 5 || aa1 == 7 || aa1 == 13 || aa1 == 15 ||
											 aa1 == 19 || aa1 == 20 ) &&
											 ( aa2 == 5 || aa2 == 7 || aa2 == 13 || aa2 == 15 ||
											 aa2 == 19 || aa2 == 20 ) ) {
												calculate_angle_distance(p1,q1,r1,s1,p2,q2, r2,s2,
												 plane_angle,vert_distance,horiz_distance);
											} else if ( ( aa1 == 5 || aa1 == 8 || aa1 == 10 ||
											 aa1 == 13 || aa1 == 18 || aa1 == 19 || aa1 == 20 ) &&
											 ( aa2 == 5 || aa2 == 8 || aa2 == 10 || aa2 == 13 ||
											 aa2 == 18 || aa2 == 19 || aa2 == 20 ) ) {
												calculate_angle_distance_phobic(p1,q1,r1, p2,q2,r2,
												 plane_angle,vert_distance,horiz_distance);
											}

// add these res pairs to a list so we don't calculate their plane angles
// again if there are multiple atom-atom contacts between the sidechains

											++pair_found;
											pair_list(1,pair_found) = res1;
											pair_list(2,pair_found) = res2;
											if ( res1 != res2 ) {
												pla_stream <<
												 I( 4, res1 ) << "  " <<
												 I( 4, res2 ) << "  " <<
												 I( 4, aa1 ) << "  " <<
												 I( 4, aa2 ) << "  " <<
												 I( 4, neighbors(res1) ) << "  " <<
												 I( 4, neighbors(res2) ) << "  " <<
												 F( 7, 3, plane_angle ) <<
												 F( 7, 3, horiz_distance ) <<
												 F( 7, 3, vert_distance ) << std::endl;
											}
//										} // for n,n+1 in sequence
										} // check for at least two contacting atom pairs
									} // for res1 != res2
								} // dis lt. 4.2A
							} // check for number of neighbors
						} // b-factor cutoff
					} // atom2
				} // atom1
			} // check to make sure restype is phe (or whatever)
L200:;
		} // res2
	} // res1

//	smallest_distance = 5.0;
//	for ( int s = 1; s <= max_contacts; ++s ) {
//		if ( ( distances(s) < smallest_distance ) && ( distances(s) != 0 ) ) {
//			smallest_distance = distances(s);
//		}
//	}

	pla_stream.close();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_cation_pi
///
/// @brief
///km  this function finds the horizontal and vertical distances distances
///    between NZ of lysine and an aromatic sidechain
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
find_cation_pi()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using numeric::xyzVector_float;

// local:
	int aa1,aa2,aav1,aav2;
	xyzVector_float p1(0.);
	xyzVector_float q1(0.);
	xyzVector_float r1(0.);
	xyzVector_float s1(0.);
	xyzVector_float point(0.);
	float dis;
// first and second residues in the pair being counted
	FArray2D_float pair_list( 2, MAX_RES()(), 0.0f );
	float horiz_distance,vert_distance;

	std::ofstream cat_stream( "cation_distance_statistics" );
	set_fullatom_flag(true);

// initialize:
	int pair_found = 0;

// Loop over all residues, calculate distances between F atoms and K(NZ)
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = res1+1; res2 <= total_residue; ++res2 ) {
			aa1 = res(res1);
			aa2 = res(res2);
			aav1 = res_variant(res1);
			aav2 = res_variant(res2);
//Find aa numbers in read_aa_ss.f, atom numbers in read_aaproperties.f
			if ( ( aa1 == 5 || aa1 == 19 || aa1 == 20 ) &&
			 ( aa2 == 9 ) || ( aa2 == 5 || aa2 == 19 || aa2 == 20 ) &&
			 ( aa1 == 9 ) ) {
				for ( int atom1 = 6, atom1e = nheavyatoms(aa1,aav1), atom2e = nheavyatoms(aa2,aav2);
				 atom1 <= atom1e; ++atom1 ) {
					for ( int atom2 = 6; atom2 <= atom2e; ++atom2 ) {
					 // loop through sidechain atoms
						distance_bk( full_coord(1,atom1,res1), full_coord(1,atom2,res2), dis );
						if ( dis < 4.3 ) {

// check to see if this pair of residues is in the list of those already
// calculated, try again if true
							for ( int junk = 1, junk_end = MAX_RES(); junk <= junk_end; ++junk ) {
								if ( ( pair_list(1,junk) == res1 ) &&
								 ( pair_list(2,junk) == res2 ) ) {
									goto L200;
								}
							}

// Res1 and res2 have met the criteria of having at least two contacts (<4.0A),
// Choose number and identity of atoms to define plane
							if ( ( aa1 == 5 || aa1 == 20 ) && aa2 == 9 ) {
								for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
									if ( i == 7 ) {
										p1 = &full_coord(1,i,res1);
									} else if ( i == 10 ) {
										q1 = &full_coord(1,i,res1);
									} else if ( i == 8 ) {
										r1 = &full_coord(1,i,res1);
									} else if ( i == 9 ) {
										s1 = &full_coord(1,i,res1);
									}
								}
								for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
									if ( j == 8 ) { // choose NZ or CE
										point = &full_coord(1,j,res2);
									}
								}
							} else if ( aa1 == 9 && (aa2 == 5 || aa2 == 20) ) {
								for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
									if ( i == 8 ) { // choose NZ or CE
										point = &full_coord(1,i,res1);
									}
								}
								for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
									if ( j == 7 ) {
										p1 = &full_coord(1,j,res2);
									} else if ( j == 10 ) {
										q1 = &full_coord(1,j,res2);
									} else if ( j == 8 ) {
										r1 = &full_coord(1,j,res2);
									} else if ( j == 9 ) {
										s1 = &full_coord(1,j,res2);
									}
								}
							} else if ( aa1 == 19 && aa2 == 9 ) {
								for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
									if ( i == 8 ) {
										p1 = &full_coord(1,i,res1);
									} else if ( i == 14 ) {
										q1 = &full_coord(1,i,res1);
									} else if ( i == 10 ) {
										r1 = &full_coord(1,i,res1);
									} else if ( i == 12 ) {
										s1 = &full_coord(1,i,res1);
									}
								}
								for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
									if ( j == 8 ) { // choose NZ or CE
										point = &full_coord(1,j,res2);
									}
								}
							} else if ( aa1 == 9 && aa2 == 19 ) {
								for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
									if ( i == 8 ) { // choose NZ or CE
										point = &full_coord(1,i,res1);
									}
								}
								for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
									if ( j == 8 ) {
										p1 = &full_coord(1,j,res2);
									} else if ( j == 14 ) {
										q1 = &full_coord(1,j,res2);
									} else if ( j == 10 ) {
										r1 = &full_coord(1,j,res2);
									} else if ( j == 12 ) {
										s1 = &full_coord(1,j,res2);
									}
								}
							}

// now get vertical and horizontal distances for this pair
							calculate_cation_distance(p1,q1,r1,s1,point,vert_distance,
							 horiz_distance);
// add these res pairs to a list so we don't calculate their plane angles
// again if there are multiple atom-atom contacts between the sidechains

							++pair_found;
							pair_list(1,pair_found) = res1;
							pair_list(2,pair_found) = res2;
							if ( res1 != res2 ) {
								cat_stream <<
								 I( 4, aa1 ) << "  " <<
								 I( 4, aa2 ) << "  " <<
								 I( 4, atom1 ) << "  " <<
								 I( 4, atom2 ) << "  " <<
								 F( 7, 3, horiz_distance ) <<
								 F( 7, 3, vert_distance ) << std::endl;
							}
L200:;
						} // dis lt. 4.2A
					} //atom 2
				} // atom1
			} // check ro make sure restype is aromatic/lys
		} // res2
	} // res1

	cat_stream.close();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_carbonyl_o
///
/// @brief
///km  this function finds angle between a planar sidechain and the planar
///    region of the polypeptide chain containing C=O, as well as the
///    distance between the planes.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
find_carbonyl_o()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace planes_ns;
	using numeric::xyzVector_float;

// local:
	int atom2,aa1,aa2,aav1,aav2;
	xyzVector_float p1;
	xyzVector_float q1;
	xyzVector_float r1;
	xyzVector_float s1;
	xyzVector_float p2;
	xyzVector_float q2;
	xyzVector_float r2;
	float dis;
	int relevant_aa1_start,relevant_aa1_end;
	int relevant_aa2_start,relevant_aa2_end;
	FArray1D_int angles_distances( number_angle_bins, 0 );
	 // array to store plane angle and distance bin values
	FArray1D_int points_plane1( max_plane );
	FArray1D_int points_plane2( max_plane );

// first and second residues in the pair being counted
	FArray2D_float pair_list( 2, MAX_RES()(), 0.0f );
	float plane_angle,horiz_distance,vert_distance;
	int atom_pair_counter; //counter to make sure two atom pairs contact

	std::ofstream car_stream( "carbonyl_distance_statistics" );
	set_fullatom_flag(true);

// initialize:
	int pair_found = 0;

// Loop over all residues and atoms and calculate distances between F,Y,W,H,R,K
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
		 // if you want to separate FY from YF
			aa1 = res(res1);
			aa2 = res(res2);
			aav1 = res_variant(res1);
			aav2 = res_variant(res2);

// For planes and c-o bond vector(use n-c-o plane so we can recycle code
// that calculates planes from three points):
			if ( aa1 == 5 || aa1 == 7 || aa1 == 13 || aa1 == 15 || aa1 == 19 ||
			 aa1 == 20 ) {

// Make sure the planar sidechain groups are contacting each other
				relevant_sidechain_atoms(aa1,aa2,relevant_aa1_start, relevant_aa1_end,
				 relevant_aa2_start,relevant_aa2_end);
// Start counting number of atom pairs that come within 4A of each other
				atom_pair_counter = 0;
				for ( int atom1 = relevant_aa1_start; atom1 <= relevant_aa1_end;
				 ++atom1 ) { // sidechain atoms
// Only want to get distance between carbonyl O and plane atoms
					atom2 = 4; //carbonyl O atom
					distance_bk(full_coord(1,atom1,res1),full_coord(1,atom2,res2),dis);
// Define a contact :
					if ( dis < 4.2 ) {
						if ( res1 != res2 ) {
							++atom_pair_counter;
// ** This is where you define how many contacts sidechains should have
							if ( atom_pair_counter >= 1 ) {

// check to see if this pair of residues is in the list of those already
// calculated, try again if true
								for ( int junk = 1, junk_end = MAX_RES(); junk <= junk_end; ++junk ) {
									if ( ( pair_list(1,junk) == res1 ) &&
									 ( pair_list(2,junk) == res2 ) ) {
										goto L200;
									}
								}
// Res1 and res2 have met the criteia of having at least two contacts (<4.2A),
// Choose number and identity of atoms to define plane
								select_plane_points(aa1,aa2,points_plane1, points_plane2);

								for ( int i = 1, ie = natoms(aa1,aav1); i <= ie; ++i ) {
									if ( i == points_plane1(1) ) {
										p1 = &full_coord(1,i,res1);
									} else if ( i == points_plane1(2) ) {
										q1 = &full_coord(1,i,res1);
									} else if ( i == points_plane1(3) ) {
										r1 = &full_coord(1,i,res1);
									} else if ( i == points_plane1(4) ) {
										s1 = &full_coord(1,i,res1);
									}
								}
// aa2 will be backbone plane N-C-O
								for ( int j = 1, je = natoms(aa2,aav2); j <= je; ++j ) {
									if ( j == 1 ) { // backbone N
										p2 = &full_coord(1,j,res2);
									} else if ( j == 3 ) { // backbone C
										q2 = &full_coord(1,j,res2);
									} else if ( j == 4 ) { // backbone O
										r2 = &full_coord(1,j,res2);
									}
								}

// now get plane_angle, vertical and horizontal distances for this pair
								calculate_angle_distance_phobic(p1,q1,r1,p2,q2,r2,plane_angle,
								 vert_distance,horiz_distance);

// add these res pairs to a list so we don't calculate their plane angles
// again if there are multiple atom-atom contacts between the sidechains

								++pair_found;
								pair_list(1,pair_found) = res1;
								pair_list(2,pair_found) = res2;
								if ( res1 != res2 ) {
									car_stream <<
									 I( 4, res1 ) << "  " <<
									 I( 4, res2 ) << "  " <<
									 I( 4, aa1 ) << "  " <<
									 I( 4, aa2 ) << "  " <<
									 F( 7, 3, plane_angle ) <<
									 F( 7, 3, horiz_distance ) <<
									 F( 7, 3, vert_distance ) << std::endl;
								}
							} // check for atom contacts
						} // for res1 ne res2
					} // dis lt. 4.2A
				} // atom1
			} // check to make sure restype is phe (or whatever)
L200:;
		} // res2
	} // res1

	car_stream.close();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calculate_180angle_distance
///
/// @brief
///km this function takes two sets of four points, each defining a
///   plane, gets the cross product of two vectors in each plane,
///   then finds the angle between the planes
///   this is modified to test the idea that using an angle range
///   of 0-180 contains more information than 0-90 (Hunter and Thornton,
///   JMB 1991 218 p.837-846)
///
/// @detailed
///
/// @param   p1 - [in/out]? -
/// @param   q1 - [in/out]? -
/// @param   r1 - [in/out]? -
/// @param   s1 - [in/out]? -
/// @param   p2 - [in/out]? -
/// @param   q2 - [in/out]? -
/// @param   r2 - [in/out]? -
/// @param   s2 - [in/out]? -
/// @param   plane_angle - [in/out]? -
/// @param   vert_distance - [in/out]? -
/// @param   horiz_distance - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calculate_180angle_distance(
	numeric::xyzVector_float const & p1,
	numeric::xyzVector_float const & q1,
	numeric::xyzVector_float const & r1,
	numeric::xyzVector_float const & s1,
	numeric::xyzVector_float const & p2,
	numeric::xyzVector_float const & q2,
	numeric::xyzVector_float const & r2,
	numeric::xyzVector_float const & s2,
	float & plane_angle,
	float & vert_distance,
	float & horiz_distance
)
{
	using namespace misc;
	using namespace param;
	using numeric::xyzVector_float;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

// local
	xyzVector_float vec1_1;
	xyzVector_float vec1_2;
	xyzVector_float vec2_1;
	xyzVector_float vec2_2;
	xyzVector_float cross1;
	xyzVector_float cross2;
	float n1n2;
	xyzVector_float centroid_plane1;
	xyzVector_float centroid_plane2;
	xyzVector_float centroid_vector;
	xyzVector_float vert_vector;
	xyzVector_float horiz_vector;

// get coordinates for center of plane1 and plane2:
	centroid_plane1 = ( p1 + q1 + r1 + s1 ) / 4;
	centroid_plane2 = ( p2 + q2 + r2 + s2 ) / 4;

	centroid_vector = centroid_plane2 - centroid_plane1;

// Get 2 vectors to describe plane, normalized cross product between them.
// Plane1:
	vec1_1 = q1 - p1;
	vec1_2 = s1 - r1;
	cross1 = cross( vec1_1, vec1_2 ).normalize();
// Plane 2:
	vec2_1 = q2 - p2;
	vec2_2 = s2 - r2;
	cross2 = cross( vec2_1, vec2_2 ).normalize();

// Find angle between normal vectors cross1 and cross2:
// plane_angle = (cross1.cross2)/||cross1||||cross2||
// n1n2 = (cross1.cross2)
// l_n1 = ||cross1||
// l_n2 = ||cross2||

	n1n2 = dot( cross1, cross2 );
	assert( ( cross1.length_squared() > 0.0 ) );
	assert( ( cross2.length_squared() > 0.0 ) );
	plane_angle = degrees( std::acos( sin_cos_range( n1n2 ) ) );

// Set angle range from 0 to 90 degrees:
	if ( ( plane_angle > 90.0 ) && ( plane_angle <= 180.0 ) ) {
		plane_angle = ( 180.0 - plane_angle );
	}

	if ( centroid_vector(3) < 0.0 ) {
		plane_angle = ( 180.0 - plane_angle );
	}

// Now get vert_distance and horiz_distance:
	vert_distance = dot( centroid_vector, cross1 );
	vert_vector = vert_distance * cross1;
	vert_distance = std::abs( vert_distance );

	horiz_vector = centroid_vector - vert_vector;
	horiz_distance = horiz_vector.length();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calculate_cation_distance
///
/// @brief
///
/// @detailed
///
/// @param  p1 - [in/out]? -
/// @param  q1 - [in/out]? -
/// @param  r1 - [in/out]? -
/// @param  s1 - [in/out]? -
/// @param  point - [in/out]? -
/// @param  vert_distance - [in/out]? -
/// @param  horiz_distance - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calculate_cation_distance(
	numeric::xyzVector_float const & p1,
	numeric::xyzVector_float const & q1,
	numeric::xyzVector_float const & r1,
	numeric::xyzVector_float const & s1,
	numeric::xyzVector_float const & point,
	float & vert_distance,
	float & horiz_distance
)
{
	using numeric::xyzVector_float;

// local
	xyzVector_float vec1_1;
	xyzVector_float vec1_2;
	xyzVector_float cross1;
	xyzVector_float centroid_plane1;
	xyzVector_float centroid_vector;
	xyzVector_float vert_vector;
	xyzVector_float horiz_vector;

// get coordinates for center of plane1 and plane2:
	centroid_plane1 = ( p1 + q1 + r1 + s1 ) / 4;

	centroid_vector = point - centroid_plane1;

// Get 2 vectors to describe plane, normalized cross product between them.
// Plane1:
	vec1_1 = q1 - p1;
	vec1_2 = s1 - r1;
	cross1 = cross( vec1_1, vec1_2 ).normalize();

// Now get vert_distance and horiz_distance:
	vert_distance = dot( centroid_vector, cross1 );
	vert_vector = vert_distance * cross1;
	vert_distance = std::abs( vert_distance );

	horiz_vector = centroid_vector - vert_vector;
	horiz_distance = horiz_vector.length();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_rama_stats
///
/// @brief
///km writes phi,psi,ss,and rama score of each residue to the logfile
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_rama_stats()
{

	using namespace misc;

// local
	float rama,tmp1,tmp2;

	set_fullatom_flag(true);

	std::ofstream rs_stream( "ramachandran_statistics" );
	rs_stream << "i phi  psi  res secstruct  rama_score" << std::endl;

	for ( int i = 1; i <= total_residue; ++i ) {
		eval_rama_score_residue(res(i),phi(i),psi(i),secstruct(i),rama,tmp1,tmp2);
		rs_stream << I( 5, i ) << F( 9, 3, phi(i) ) << F( 9, 3, psi(i) ) <<
		 I( 4, res(i) ) << ' ' << secstruct(i) << F( 8, 3, rama ) << std::endl;
	}

	rs_stream.close();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_solvation_stats
///
/// @brief
///
/// @detailed
///km This function calculates the parameter sum_solvation for each atom,
///   as in etable.h, which is a measure of burial.  This parameter is a
///   function of the distance between atom1 and atom2, the radius of
///   atom1 and atom2, and a parameter lambda (see the Lazaridis Karplus
///   paper for a description of the solvation model) and is summed for
///   all atom1-atom2 pairs within 8A of each other.  It uses the rosetta
///   definition of a hydrogen bond to separate atoms into those making
///   or not making hydrogen bonds.
///   The counts are binned according to the value for sum_solvation, and
///   it is convenient to make a histogram for each atom type, the peaks
///   correspond to the characteristic degree of "burial" by this measure.
///   This function can also be used to parameterize the LK model by adjusting
///   the parameter fa_lk_dgfree (in etable.h) so that the distributions
///   for the decoys match those for natives.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_solvation_stats()
{
	using namespace aaproperties_pack;
	using namespace etable;
	using namespace hbonds;
	using namespace misc;
	using namespace pdbstats;
	using numeric::constants::f::pi;

// local
	int aa1,aav1,aa2,aav2;
	float dis,dis2,x1;

	float const thresh_dis = { 8.0f };
	float const constant = { 4.0f * pi };

	static bool hbond = { false };
	static bool first_nohb = { true };
	static bool first_hb = { true };
	static bool first_total = { true };

// initialize
	if ( first_nohb ) {
		counted_nohb = 0;
		first_nohb = false;
	}

	if ( first_hb ) {
		counted_hb = 0;
		first_hb = false;
	}

	if ( first_total ) {
		counted_total = 0;
		first_total = false;
	}

	set_fullatom_flag(true);
	float solv_x1 = 0.0;
	float sum_solvation = 0.0;

// Start:
// Loop through each residue, each atom, and calculate
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
		aa1 = res(res1);
		aav1 = res_variant(res1);
		for ( int atom1 = 1, atom1e = natoms(aa1,aav1); atom1 <= atom1e; ++atom1 ) {
			sum_solvation = 0.0; // so sums are for each atom only
			int const atype1 = fullatom_type(atom1,aa1,aav1);

			// Note: this used to loop from 1 to total_residues, instead of from res1 to total_residue.
			//  The old version would have missed half the constributions to the Lazaridis/Karplus term.
			for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
				aa2 = res(res2);
				aav2 = res_variant(res2);
				for ( int atom2 = 1, atom2e = natoms( aa2, aav2 ); atom2 <= atom2e; ++atom2 ) {
					int const atype2 = fullatom_type( atom2, aa2, aav2 );
// Escape clause for atom 1 to itself
					if ( ( res1 == res2 ) && ( atom1 == atom2 ) ) {
						solv_x1 = 0.;
					} else {
						distance_bk( full_coord(1,atom1,res1), full_coord(1,atom2,res2), dis );
						if ( dis < thresh_dis ) {
// We have fa_lk_lambda and fa_lk_volume in etable.h
// Calculate solv_x1 for atom1, then multiply solv_x1 by volume of atom 2,
// then sum this quantity over all atom1-atom2 pairs within 8A

							x1 = square( dis - fa_lj_radius(atype1) ) /
							 square( fa_lk_lambda(atype1) );
							dis2 = dis * dis;
							solv_x1 = ( std::exp(-x1) / ( constant * dis2 ) ) *
							 fa_lk_volume(atype2);

// Now we need to sum up solv_x1 for each atom2 closer than thresh_dis to atom 1

							sum_solvation += solv_x1;
// All done...

						} // distance cutoff at 8A
					} //check for atom 1 to itself
				} // atom2
			} //res2

// Check for hbonds, then write stats to file:

			hbond = false;
			for ( int i = 1; i <= hbond_set.nhbonds(); ++i ) {
				if ( ( hbond_set.hbdon_res(i) == res1 ) && ( hbond_set.hbdonh_atm(i) == atom1 ) ) {
					hbond = true;
				} else if ( ( hbond_set.hbact_res(i) == res1 ) && ( hbond_set.hbact_atm(i) == atom1 ) ) {
					hbond = true;
				}
			}
// Add counts to bin for output_stats:
// Figure out what bin to put counts in:

			float const size = bin_size;
			int const bins = static_cast< int >( sum_solvation / size ) + 1;

			++counted_total(atype1,bins);

			if ( hbond ) {
//			counted_hb(atype1,bins) = counted_hb(atype1,bins)+1
// Exception handling: (get info for what heavy atoms are donating protons)
				if ( (atype1 == 21) && (aa1 == 7) ) { // N in His
					++counted_hb(8,bins);
				} else if ( (atype1 == 21) && (aa1 == 9) ) { // N in Lys
					++counted_hb(10,bins);
				} else if ( (atype1 == 21) && (aa1 == 12) ) { // N in Asn
					++counted_hb(9,bins);
				} else if ( (atype1 == 21) && (aa1 == 14) ) { // N in Gln
					++counted_hb(9,bins);
				} else if ( (atype1 == 21) && (aa1 == 15) ) { // N in Arg
					++counted_hb(11,bins);
				} else if ( (atype1 == 21) && (aa1 == 16) ) { // O in Ser
					++counted_hb(13,bins);
				} else if ( (atype1 == 21) && (aa1 == 17) ) { // O in Thr
					++counted_hb(13,bins);
				} else if ( (atype1 == 21) && (aa1 == 20) ) { // O in Tyr
					++counted_hb(13,bins);
				} else if ( (atype1 == 21) && (aa1 == 19) ) { // N in Trp
					++counted_hb(7,bins);
				} else if ( atype1 == 24 ) { // HN
					++counted_hb(17,bins);
				} else {
					++counted_hb(atype1,bins);
				}
			} else {
				++counted_nohb(atype1,bins);
			}

		} // atom1
	} // res1

}

//////////////////////////////////////////////////////////////////////////////
/// @begin output_solv_stats
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
output_solv_stats()
{
	using namespace param;
	using namespace pdbstats;


	{ // Scope
		std::ofstream on_stream( "output_nohb" );

		on_stream << "atype  counts" << std::endl;
		for ( int atype1 = 1, atype_end = MAX_ATOMTYPES(); atype1 <= atype_end; ++atype1 ) {
			for ( int bins = 1; bins <= number_bins; ++bins ) {
				on_stream << I( 8, atype1 ) << I( 8, counted_nohb(atype1,bins) ) << std::endl;
			}
		}

		on_stream.close();
	}


	{ // Scope
		std::ofstream oh_stream( "output_hb" );

		oh_stream << "atype  counts" << std::endl;
		for ( int atype1 = 1, atype_end = MAX_ATOMTYPES(); atype1 <= atype_end; ++atype1 ) {
			for ( int bins = 1; bins <= number_bins; ++bins ) {
				oh_stream << I( 8, atype1 ) << I( 8, counted_hb(atype1,bins) ) << std::endl;
			}
		}

		oh_stream.close();
	}


	{ // Scope
		std::ofstream ot_stream( "output_total" );

		ot_stream << "atype  counts" << std::endl;
		for ( int atype1 = 1, atype_end = MAX_ATOMTYPES(); atype1 <= atype_end; ++atype1 ) {
			for ( int bins = 1; bins <= number_bins; ++bins ) {
				ot_stream << I( 8, atype1 ) << I( 8, counted_total(atype1,bins) ) << std::endl;
			}
		}

		ot_stream.close();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sasa_stats()
///
/// @brief  measures sasa by residue
///
/// @detailed
///
/// @global_read  param.h, pdbstats.h, misc.h,
///
/// @global_write pdbstats.h
///
/// @remarks
///
/// @references
///
/// @authors  Bin Qian 1/10/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sasa_stats()
{
	using namespace pdbstats;
	using namespace misc;
	using namespace param;

	FArray2D_float atom_sasa( MAX_ATOM()(), MAX_RES()() );

// initialize the per-atom logicals for calculating rms-like scores
	calc_per_atom_sasa( atom_sasa, rsd_sasa, 1.4f, false, false );
	// water probe radius, real sasa,small polar H
	for ( int i = 1; i <= total_residue; ++i ) {
		rsd_sasa_fraction(i) = std::max( 0.0f, std::min( 1.0f, rsd_sasa(i) / rsd_exposed_sasa(res(i)) ) );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_sasa_stats
///
/// @brief  output residue sasa
///
/// @detailed
///
/// @global_read  param.h, misc.h, pdbstats.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian 1/10/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_sasa_stats()
{
	using namespace misc;
	using namespace pdbstats;

	std::string res3;

	std::ofstream cdr_stream( "sasa_stats" );
	if ( !cdr_stream ) {
		std::cout << "Open failed for file: " << "sasa_stats" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	cdr_stream << "NO#   res" <<
	 "       sasa  sasa_fraction" <<  std::endl;

	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), res3 );
		cdr_stream << I( 3, i ) << space( 4 ) << res3 << "   " <<
		 F( 7, 2, rsd_sasa(i) ) << "   " <<
		 F( 7, 2, rsd_sasa_fraction(i) ) << std::endl;
	}
	cdr_stream.close();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin output_pair_energies
///
/// @brief
///  output_pair_energies
///
/// @detailed
///
/// @param
///
/// @global_read fullatom energies
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_pair_energies(
)
{

	using namespace fullatom_energies;
	using namespace files_paths;

	std::string filename = "-";
	stringafteroption( "pair_energy_file", filename, filename );
	if ( filename == "-" ) {
		std::cout << "ERROR:: unable to make pair energy file: " <<
		 " no filename defined" << std::endl;
		std::cout << "use -pair_energy_file option to define finename!" << std::endl;
		return;
	} else {
		std::cout << "pair_energy file: " << score_path << filename << std::endl;
	}

	set_fullatom_flag(true);
	score12();

	utility::io::ozstream pairout( filename );

	if ( !pairout ) {
		std::cout << "Open failed for file: " << pairout.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << "writing pair_energy file: " << pairout.filename() << std::endl;

	pairout << " res_i res_j     atr     rep     sol   hbond      sc" << '\n';
	for( int i = 1; i <= misc::total_residue; ++i ) {
	  for( int j = 1; j <= misc::total_residue; ++j ) {
			if( j >= i ){
				pairout << I( 6, i ) << I( 6, j )
					<< F( 8, 3, atr_pair( i, j ) )
					<< F( 8, 3, rep_pair( i, j ) )
					<< F( 8, 3, sol_pair( i, j ) )
					<< F( 8, 3, hbond_pair( i, j ) )
					<< F( 8, 3, pair_pair( i, j ) )
					<< '\n';
			}
	  }
	}

	pairout.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_env_stats
///
/// @brief
///  get env stats ln(P(aai/nbi)) from a pdb or a list of pdbs
///
/// @detailed
///
/// @param
///
/// @global_read fullatom energies
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_env_stats(
)
{

	using namespace cenlist_ns;
	using namespace misc;
	using namespace pdbstats;
	using namespace pdb;

	int icon;

	compute_CB_environment(); // function is hiding in vdw_compute.cc

	for ( int i = 1; i <= total_residue; ++i ) {
		if( occupancy( 2, i ) > 0 ){
			if ( fcen10(i) >= 31.0f ) fcen10(i) = 30.9999f;
			homolog_env( i, static_cast< int >(fcen10(i)) )++;
			nenvs(i)++;
		}
	}


// get homolog_pair stats
	static double const cen_dist5  = 25.0;
	static double const cen_dist7  = 56.25;
	static double const cen_dist10 = 100.0;

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int kk = 1, lcl = cen_list.index(kk,i), kke = cen12up(i);
		 kk <= kke; ++kk, ++lcl ) {
			int j = cen_list[ lcl ]; // cen_list(kk,i)

			if( occupancy( 2, i ) <= 0.0 || occupancy( 2, j ) <= 0.0 ) continue; // ignore missing density

			if ( j - i > 8 ) {
				float k = cendist(i,j);

				icon = 5;
				if ( k > cen_dist10 ) {
					icon = 4;
				} else {
					if ( k > cen_dist7 ) {
						icon = 3;
					} else {
						if ( k > cen_dist5 ) {
							icon = 2;
						} else {
							icon = 1;
						}
					}
				}

				homolog_pair( icon, i, j )++;
				npairs( i, j )++;

			}
		} // escape for non-AA's
	}


// get the histogram of pair_prob vs sequence seperation of natural pdbs
// de-commissioned
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int kk = 1, lcl = cen_list.index(kk,i), kke = cen12up(i);
		 kk <= kke; ++kk, ++lcl ) {
			int j = cen_list[ lcl ]; // cen_list(kk,i)

			if ( j - i > 8 ) {
				float k = cendist(i,j);

				icon = 5;
				if ( k > cen_dist10 ) {
					icon = 4;
				} else {
					if ( k > cen_dist7 ) {
						icon = 3;
					} else {
						if ( k > cen_dist5 ) {
							icon = 2;
						} else {
							icon = 1;
						}
					}
				}

				int sep = std::abs( i - j );
				pdb_pair( icon, sep )++;

			}
		} // escape for non-AA's
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_env_stats
///
/// @brief
///  output env stats ln(P(aai/nbi)) from a pdb or a list of pdbs
///
/// @detailed
///
/// @param
///
/// @global_read fullatom energies
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_env_stats(
)
{

	using namespace pdbstats;
	using namespace files_paths;


	float c_cen10 = 0.0;
	float c_pair  = 0.0;
	realafteroption( "env_psudo_co_eff",  c_cen10, c_cen10 );
	realafteroption( "pair_psudo_co_eff", c_pair,  c_pair  );

	bool get_prob( truefalseoption( "prob_stats" ) );

	FArray1D_float const & pdb_cen10_prob = structure::BOUNDARY::get_pdb_cen10_prob();
	FArray2D_float const & pdb_pair_prob = structure::BOUNDARY::get_pdb_pair_prob();

	// adding psudo_counts:
	for ( int i = 1; i <= misc::total_residue; ++i )
		for( int j = 1; j <= 30; ++j )
			homolog_env( i, j ) += c_cen10 * pdb_cen10_prob( j );// psudo-counts

	// smoothing and get probabilitys and log ratio
	for ( int i = 1; i <= misc::total_residue; ++i ){
		for( int j = 1; j <= 30; ++j ){
			if( j > 4 && j < 30 ){
				homolog_env( i, j ) = ( homolog_env( i, j-1 )/5.0 +
																3.0/5.0* homolog_env( i, j ) +
																homolog_env( i, j+1 )/5.0 ) /
																( nenvs(i) + c_cen10 );
			//}else	homolog_env( i, j ) /= ( nhomologs + c_cen10 );
			} else homolog_env( i, j ) = 0.0;
			if( !get_prob ){
				//float deno = (nhomologs/30.0 + c_cen10 * pdb_cen10_prob(j))/((nhomologs+c_cen10)*pdb_cen10_prob(j));
				float deno = 1.0;
				if ( homolog_env( i, j ) < 1e-4 )
					homolog_env( i, j ) = 2.0;
				else
					homolog_env( i, j ) =
						-std::log( homolog_env( i, j )/( deno*pdb_cen10_prob(j) ) );
//			if( homolog_env( i, j ) < 1e-10 ) homolog_env( i, j ) = 0.0;
			}
		}
	}

	for( int i = 1; i <= misc::total_residue; i++ ){
		for( int j = 1; j <= misc::total_residue; j++ ){

			for( int icon = 1; icon <= 5; icon++ ){
				int seq_sep = std::abs( i - j );
				if( seq_sep > 8 )
					homolog_pair( icon, i, j ) +=
						c_pair * pdb_pair_prob( seq_sep, icon );// psudo-counts
				homolog_pair( icon, i, j ) /= ( npairs( i,j ) + c_pair );
				if( !get_prob ){
					if( homolog_pair( icon, i, j ) < 1e-4 )
						homolog_pair( icon, i, j ) = 2.0;
					else
						homolog_pair( icon, i, j )
						= -std::log( homolog_pair( icon,i,j )/pdb_pair_prob( seq_sep,icon ));
				}
			}
		}
	}

	std::string filename = "-";
	stringafteroption( "env_stats_file", filename, filename );
	if ( filename == "-" ) {
		std::cout << "ERROR:: unable to make pair energy file: " <<
		 " no filename defined" << std::endl;
		std::cout << "use -env_stats_file option to define finename!" << std::endl;
		return;
	} else {
		std::cout << "env_stats file: " << score_path << filename << std::endl;
	}

	utility::io::ozstream envout( filename );

	if ( !envout ) {
		std::cout << "Open failed for file: " << envout.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << "writing env_stats file: " << envout.filename() << std::endl;

	envout << "    aa    nb ln(P(aa|nb))" << '\n';
	for( int i = 1; i <= misc::total_residue; ++i ) {
	  for( int j = 1; j <= 30; ++j ) {
			envout << I( 6, i ) << I( 6, j )
				<< F( 12, 3, homolog_env( i, j ) )
				<< '\n';
	  }
	}

	envout.close();

	filename = "-";
	stringafteroption( "pair_stats_file", filename, filename );
	if ( filename == "-" ) {
		std::cout << "ERROR:: unable to make pair_stats file: " <<
		 " no filename defined" << std::endl;
		std::cout << "use -pair_stats_file option to define finename!" << std::endl;
		return;
	} else {
		std::cout << "pair_stats file: " << score_path << filename << std::endl;
	}

	utility::io::ozstream pairout( filename );

	if ( !pairout ) {
		std::cout << "Open failed for file: " << pairout.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << "writing pair_stats file: " << pairout.filename() << std::endl;

	pairout << "  icon  resi  resj ln(P(i,j|dij))" << '\n';


	for( int i = 1; i <= misc::total_residue; ++i ){
		for( int j = 1; j <= misc::total_residue; ++j ){
			for( int icon = 1; icon <= 5; ++icon ) {
				pairout << I( 6, icon ) << I( 6, i ) << I( 6, j )
					<< F( 12, 3, homolog_pair( icon, i, j ) )
					<< '\n';
			}
	  }
	}

	pairout.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pdbensemble_stats
///
/// @brief
///  get prior env stats ln(P(nbi)) and pair stats
///  by residue sequential distance from a pdb or a list of pdbs
///
/// @detailed
///
/// @param
///
/// @global_read fullatom energies
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pdbensemble_stats(
)
{

	using namespace cenlist_ns;
	using namespace misc;
	using namespace pdbstats;

	int res_id,icon;

	compute_CB_environment(); // function is hiding in vdw_compute.cc

	for ( int i = 1; i <= total_residue; ++i ) {
// Several checks added to only consider amino acids, not DNA/RNA
		res_id = res(i);
		if ( fcen10(i) >= 31.0f ) fcen10(i) = 30.9999f;
		homolog_env( res_id, static_cast< int >( fcen10(i)) )++;
	}

// get the histogram of pair_prob vs sequence seperation of natural pdbs
	static double const cen_dist5  = 25.0;
	static double const cen_dist7  = 56.25;
	static double const cen_dist10 = 100.0;


	for ( int i = 1; i <= total_residue; ++i ) {
		res_id = res(i);
		for ( int kk = 1, lcl = cen_list.index(kk,i), kke = cen12up(i);
		 kk <= kke; ++kk, ++lcl ) {
			int j = cen_list[ lcl ]; // cen_list(kk,i)

			if ( j - i > 8 ) {
				float k = cendist(i,j);

				icon = 5;
				if ( k > cen_dist10 ) {
					icon = 4;
				} else {
					if ( k > cen_dist7 ) {
						icon = 3;
					} else {
						if ( k > cen_dist5 ) {
							icon = 2;
						} else {
							icon = 1;
						}
					}
				}

				int sep = std::abs( i - j );
				pdb_pair( icon, sep )++;

			}
		} // escape for non-AA's
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_pdbensemble_stats
///
/// @brief
///  output prior env stats ln(P(nbi)) and pair stats
///  by residue sequential distance from a pdb or a list of pdbs
///
/// @detailed
///
/// @param
///
/// @global_read fullatom energies
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_pdbensemble_stats(
)
{

	using namespace pdbstats;
	using namespace files_paths;

	float sum( 0.0 );
	for ( int i = 1; i <= 30; ++i ){
		for( int j = 1; j <= 20; ++j ){
			sum += homolog_env( j, i );
		}
	}

	FArray1D_float pos_env( 30, 0.0 );
	for( int i = 1; i <= 30; ++i ){
		for( int j = 1; j <= 20; ++j )
			pos_env( i ) += homolog_env( j, i );
		pos_env( i ) /= sum;
	}

	for( int i = 1; i <= 30; ++i ){
		if( i > 1 && i < 30 )
			pos_env( i ) = ( pos_env( i-1 )/5.0 +
											3.0/5.0* pos_env( i ) +
											pos_env( i+1 )/5.0
										);
	}


	for( int i = 1; i <= 200; i++ ){
		float sum = 0;
		for( int icon = 1; icon <= 5; icon++ )
			sum += pdb_pair( icon, i );

		for( int icon = 1; icon <= 5; icon++ )
			pdb_pair( icon, i ) /= sum;
	}

	std::string filename = "-";
	stringafteroption( "env_stats_file", filename, filename );
	if ( filename == "-" ) {
		std::cout << "ERROR:: unable to make pair energy file: " <<
		 " no filename defined" << std::endl;
		std::cout << "use -env_stats_file option to define finename!" << std::endl;
		return;
	} else {
		std::cout << "env_stats file: " << score_path << filename << std::endl;
	}

	utility::io::ozstream envout( filename );

	if ( !envout ) {
		std::cout << "Open failed for file: " << envout.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << "writing env_stats file: " << envout.filename() << std::endl;

	envout << "    nb (P(aa|nb))" << '\n';
  for( int j = 1; j <= 30; ++j ) {
		envout << I( 6, j )
			<< F( 12, 5, pos_env( j ) )
			<< '\n';
  }


	for ( int i = 1; i <= 200; i++ ){
		for( int j = 1; j <= 5; j++ ){
			envout << I( 6, i ) << I( 6, j)
				<< F( 12, 3, pdb_pair( j, i ) )
				<< '\n';
		}
	}

	envout.close();

}



////////////////////////////////////////////////////////////////////////////////
/// @begin collect_hb_stats
///
/// @brief Collect hydrogen bond geometries and energies for a list of pdb files
///   for comparing distributions from natives/decoys
///
/// @detailed This is a hack that Phil wrote, which Jack put into pdbstats to be
/// able to dump statistics on the geometries of hydrogen bonds that rosetta evaluates.
/// It causes the import of a lot of pose things into pdbstats, but that seemed the best place.
///
/// If a -cullpdb flag is given it expects the name of a file containing a list of chains
/// of a dunbrack set of pdb files, which it will then look up in "/net/pdb".
/// Otherwise it expects a list of pdb files with full pathnames.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors pbradley snoeyink
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
collect_hb_stats()
{
	using namespace pose_ns;

	// read dunbrack file
	std::vector< std::pair< std::string, char > > files;

	if ( truefalseoption("cullpdb") ) {
		std::ifstream data( stringafteroption("cullpdb").c_str() );
		std::string line, id, pdb;
		char chain;
		getline( data,line);
		while ( getline( data,line) ) {
			std::istringstream is( line );
			is >> id;
			pdb = lowercased( id.substr(0,4) );
			chain = id[4];
			if ( chain == '0' ) chain = ' ';
			std::string const filename( "/net/pdb/"+pdb.substr(1,2)+"/"+pdb+".pdb" );
			files.push_back( std::make_pair( filename, chain ) );
		}
		data.close();
	} else {
		std::ifstream data( stringafteroption("l").c_str() );
		std::string line;
		while ( getline( data,line) ) {
			files.push_back( std::make_pair( line, '-' ) );
		}
		data.close();
	}

  std::cout << "HBSTATS dump contains" << std::endl
<< "HBSTATSstr: pdbName, accres, donres, aatom dhatom datom astruct dstruct" << std::endl
<< "HBSTATS#s:  1- 5: accres#, donres#, aNb dNb, energy 1-5" << std::endl
<< "HBSTATS#s:  6-10: weight W*atractive, W*repulsive, W*solvent, W*Hrepuls 6-10" << std::endl
<< "HBSTATS#s: 11-16: hybrid, dist, theta, psi, chi, outOfPlane 11-16" << std::endl
<< "HBSTATS#s: 17:19, 20:22, 23:25  Hxyz, Dxyz, D2xyz " << std::endl
<< "HBSTATS#s: 26;27: aIsSideChain;dIsSideChain " << std::endl;


	// now loop over pdbs
	for ( int c=0; c< int(files.size()); ++c ) {
		std::string const filename( files[c].first );
		char const chain( files[c].second );

		Pose pose;
		bool const ok
			( pose_from_pdb( pose, filename, true/*fa*/, false/*ideal*/,
											 false/*readallchains*/, chain, true/*skipmiss*/,
											 true /*allowmiss*/ ) );
		if ( !ok ) {
			std::cout << "io failed: " << filename << std::endl;
			continue;
		}


		hbonds::hb_stats_tag = filename+chain;
		hbonds::output_hb_stats = true;

		// this will trigger fill_hbond_arrays
		pose.score( score12 );
		hbonds::output_hb_stats = false;

	}
	exit(0);

}



////////////////////////////////////////////////////////////////////////////////
/// @begin show_hb_stats
///
/// @brief Print hydrogen bond geometries and energies for hbond # hbno
///   for comparing distributions from natives/decoys
///
/// @detailed This is a hack that Phil wrote, which Jack put into pdbstats to be
/// able to dump statistics for the geometries of hydrogen bonds that rosetta evaluates.
///
/// @param res - [in] - residue table for protein
/// @param res_variant - [in] - residue variant table
/// @param xyz - [in] - atom coordinates
/// @param hbno - [in] - number of hydrogen bond we are interested in
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors pbradley snoeyink
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
show_hb_stats(
	FArray1DB_int const & res,
	FArray1DB_int const & res_variant,
	FArray3DB_float const & xyz,
	int const hbno
)
{
	using namespace hbonds;
	using namespace aaproperties_pack;
	using namespace param_aa;
	using numeric::conversions::degrees;

	static bool init( false );
	static bool verbose_output, terse_output;
	if ( !init ) {
		init = true;
		verbose_output = truefalseoption("verbose_output");
		terse_output = truefalseoption("terse_output"); // no output actually
	}

	int dres, ares, dhatm, aatm, type;
	float hbE;

	dres = hbond_set.hbdon_res(hbno);
	ares = hbond_set.hbact_res(hbno);
	dhatm = hbond_set.hbdonh_atm(hbno);
	aatm = hbond_set.hbact_atm(hbno);
	hbE = hbond_set.hbenergies(hbno);
	type = hbond_set.hbtype(hbno);

	int const don_nb( template_pack::neighbors(dres) );
	int const act_nb( template_pack::neighbors(ares) );

	int const
		don_aa( res( dres ) ), don_aav( res_variant( dres ) ),
		acc_aa( res( ares ) ), acc_aav( res_variant( ares ) );

	int const
		datm ( atom_base (dhatm,don_aa,don_aav) ),
		a1atm( atom_base ( aatm,acc_aa,acc_aav) ),
		a2atm( abase2( aatm,acc_aa,acc_aav) );

	bool const d_is_sc = !atom_is_backbone(dhatm,don_aa,don_aav);
	bool const a_is_sc = !atom_is_backbone( aatm,acc_aa,acc_aav);
	float const weight = (d_is_sc || a_is_sc) ? hb_env_dep_burial_lin(act_nb,don_nb) : 1.0;

	// get non-hbond atom-atom terms:
	int const aatm_type ( fullatom_type(  aatm, acc_aa, acc_aav ) );
	int const datm_type ( fullatom_type(  datm, don_aa, don_aav ) );
	int const dhatm_type( fullatom_type( dhatm, don_aa, don_aav ) );

  float const chrg1 = atomic_charge( aatm, acc_aa, acc_aav );
  float const chrg2 = atomic_charge( datm, don_aa, don_aav );

  float sol1(0.0),sol2(0.0),atr(0.0),rep(0.0),d2,Hrep(0.0), elecE(0.0);

  float cp_weight( 1.0 );
  if ( count_pair( ares, aatm, acc_aa, acc_aav, dres, datm, don_aa, don_aav,
                   true, cp_weight ) ) {
    assert( std::abs( cp_weight - 1.0 ) < 1e-3 );
    pairenergy( xyz(1,aatm,ares), xyz(1,datm,dres), aatm_type, datm_type, chrg1, chrg2, sol1,sol2,atr,rep,elecE, d2);
// **
}

	if ( count_pair(ares, aatm, acc_aa, acc_aav, dres, dhatm, don_aa, don_aav,
									true, cp_weight ) ) {
    fast_pairenergy_hydrogens( xyz(1,aatm,ares), xyz(1,dhatm,dres), aatm_type,
                               dhatm_type, chrg1, chrg2, Hrep, elecE, cp_weight );
	}



	//define some vectors for computing stats
	numeric::xyzVector_float
		A ( &xyz(1, aatm,ares) ),
		B( &xyz(1,a1atm,ares) ),
		B2( &xyz(1,a2atm,ares) ),
		H ( &xyz(1,dhatm,dres) ),
		D ( &xyz(1, datm,dres) ),
		Base;

  switch (type) {
    case hbe_RINGSC:
    case hbe_RINGB: // RING uses midpoint between base and base2
			Base = 0.5f * ( B + B2 );
      break;
    case hbe_SP3SC:
    case hbe_SP3B: // SP3 uses base2
			Base = B2;
      break;
    default: // SP2 uses base
			Base = B;
  }


	float const dist( distance( A,H) );

	float const theta
		( degrees( std::acos( dot( ( D - H ).normalized(),
															 ( A - H ).normalized() ) ) ) );

	float const psi
		( degrees( std::acos( dot( ( H  - A ).normalized(),
															 ( Base - A ).normalized() ) ) ) );

	float const chi
		( dihedral( B2, B, A, H ) );

	float out_of_plane( 0.0 );

	if ( (B2-A).length()>0 )  {
		out_of_plane =
			std::abs( 90 - degrees( std::acos( dot( ( H  - A ).normalized(),
																							( cross( B - A, B2 - B) ).normalized() ) ) ) );
	}

	// superimpose coords so that acceptor is at the origin and
	// the acc->atom_base vector points along the x-axis
	// and the acc->abase2 vector lies in the xy plane with positive ycoord

	FArray2D_float dummy(3,3,0.0);
	FArray2D_float Mgl( 4,4 );
	dummy(2,1) = 1.0;
	dummy(1,3) = 1.0;

		FArray1D_float a1_xyz(3);
		if (type == hbe_RINGB || type == hbe_RINGSC) {
		a1_xyz(1) = Base(1);
		a1_xyz(2) = Base(2);
		a1_xyz(3) = Base(3);
		} else {
		a1_xyz(1) = B(1);
		a1_xyz(2) = B(2);
		a1_xyz(3) = B(3);
		}
		get_GL_matrix( xyz( 1, a2atm, ares ),
									 xyz( 1,  aatm, ares ),
									 a1_xyz,
									 dummy(1,1), // (0,1,0)
									 dummy(1,2), // (0,0,0)
									 dummy(1,3), // (1,0,0)
									 Mgl );

	FArray1D_float H_new( 3 ), D_new( 3 ), D2_new( 3 );
	GL_rot( Mgl, xyz( 1, dhatm, dres ), H_new );
	GL_rot( Mgl, xyz( 1,  datm, dres ), D_new );

	// choose a base2 for donors
	int d2atm = (datm==1 ? 2 : ta(2, dhatm, don_aa, don_aav)); // bb=Calpha, else template atom
	GL_rot( Mgl, xyz( 1,  d2atm, dres ), D2_new );


	// original coords
	// energy
	// burial

	if ( !terse_output ) {
		using namespace param_pack;
		std::cout << "HBSTATS " << hb_stats_tag << ' ' <<
			aa_name3(acc_aa) << ' ' << aa_name3(don_aa) << ' ' <<
			atom_name( aatm,acc_aa,acc_aav) << ' ' <<
			atom_name(dhatm,don_aa,don_aav) << ' ' <<
			atom_name( datm,don_aa,don_aav) << ' ' <<
			misc::secstruct(dres) << ' ' <<
      misc::secstruct(ares) << ' ' <<
     	I(6,ares) << ' ' <<
			I(6,dres) << ' ' <<
			I(4,act_nb) << I(4,don_nb) <<
			F(9,3,hbE ) << F(9,3,weight) <<
			F(9,3,pack_wts.Watr() * atr ) <<
			F(9,3,pack_wts.Wrep() * rep ) <<
			F(9,3,pack_wts.Wsol() * ( sol1 + sol2 ) ) <<
			F(9,3,pack_wts.Wrep() * Hrep ) <<
			I(6,type) <<
			F(9,3,dist) << F(9,3,theta) << F(9,3,psi) << F(9,3,chi) <<
			F(9,3,out_of_plane) <<
			F(9,3,H_new(1) ) << F(9,3,H_new(2) ) <<F(9,3,H_new(3) ) <<
			F(9,3,D_new(1) ) << F(9,3,D_new(2) ) <<F(9,3,D_new(3) ) <<
			F(9,3,D2_new(1) ) << F(9,3,D2_new(2) ) <<F(9,3,D2_new(3) ) <<
      ' ' << a_is_sc << ' ' << d_is_sc <<
std::endl;

		if ( verbose_output ) {
			// D,H,A,B,B2
			std::cout <<
				F(9,3,xyz(1,dhatm,dres)) <<
				F(9,3,xyz(2,dhatm,dres)) <<
				F(9,3,xyz(3,dhatm,dres)) <<
				F(9,3,xyz(1,datm,dres)) <<
				F(9,3,xyz(2,datm,dres)) <<
				F(9,3,xyz(3,datm,dres)) <<
				F(9,3,xyz(1,d2atm,dres)) <<
				F(9,3,xyz(2,d2atm,dres)) <<
				F(9,3,xyz(3,d2atm,dres)) <<
				F(9,3,xyz(1,aatm,ares)) <<
				F(9,3,xyz(2,aatm,ares)) <<
				F(9,3,xyz(3,aatm,ares)) <<
				F(9,3,B(1)) <<
				F(9,3,B(2)) <<
				F(9,3,B(3)) <<
				F(9,3,xyz(1,a2atm,ares)) <<
				F(9,3,xyz(2,a2atm,ares)) <<
				F(9,3,xyz(3,a2atm,ares));
		}
	}
}
//////////////////////////END HB_STATS/////////////////// JSS


inline
bool
is_cation_res(
int aa
)
{
  using namespace param_aa;

  return( aa == aa_lys || aa == aa_arg );
}


// Some hbonding residues are not counted (e.g., Ser, Thr), because of
// the ambiguity on placing their hbonding hydrogens in crystallographic
// structures.
inline
bool
is_hbonding_res (
int aa
)
{
  using namespace param_aa;

  return ( aa == aa_asp || aa == aa_glu || aa == aa_lys || aa == aa_asn ||
       aa == aa_gln || aa == aa_arg );
}

inline
bool
is_aromatic_res(
int aa
)
{
  using namespace param_aa;

  return ( aa == aa_phe ); // || aa == aa_tyr || aa == aa_trp );
}


// ///////////////////
// Computes the central point of each ring in the aromatic residue
void
compute_arom_acceptor_pos(
int resi,
FArray2D_float & acceptor_pos
)
{
  using namespace param_aa;
  using namespace misc;

  acceptor_pos.dimension( 3, 2, 0.0f );
  int aa = res( resi );

  assert( is_aromatic_res( aa ) );

  int atmb = 6; int atme = ( aa == aa_trp ? 10 : 11 ); // first ring atoms
  for ( int atm = atmb; atm <= atme; ++atm ) {
    for ( int i = 1; i <= 3; ++i )
      acceptor_pos( i, 1 ) += full_coord( i, atm, resi ) / ( atme - atmb + 1 );
  }

  if ( ! (aa == aa_trp) ) return;

// second ring (trp only)
  for ( int atm = 10; atm <= 13; atm++ ) {
    for ( int i = 1; i <= 3; ++i )
      acceptor_pos( i, 2 ) += full_coord( i, atm, resi ) / ( 13 - 10 + 1 );
  }
}

// Returns the normal vector to an aromatic ring.
// Although trp has two rings, they're coplanar, and therefore have
// the same normal.
void
compute_arom_normal(
int resi,
FArray1D_float & norm
)
{
  using namespace numeric;
  using namespace misc;

  int aa = res( resi );
  assert( is_aromatic_res( aa ) );

//  FArray2D_float acceptor_pos;
//  compute_arom_acceptor_pos( resi, acceptor_pos );

 // vectors connect ring centre to ring atoms
  FArray1D_float v1( 3 ), v2( 3 );
  for( int i=1; i<=3; ++i ) {
    v1( i ) = full_coord( i, 7, resi ) - full_coord( i, 9, resi );
    v2( i ) = full_coord( i, 8, resi ) - full_coord( i, 9, resi );
  }

  norm = cross( v1, v2 );
  norm.normalize();
}

// Return the vector going from the electronegative atom to the hydrogen
void
compute_donor_vector(
int hatom,
int resi,
FArray1D_float & v
)
{
  using namespace numeric;
  using namespace misc;
  using namespace aaproperties_pack;

  int aa = res( resi );

  xyzVector_float hatm_coord(&full_coord(1, hatom, resi)), base_coord(&full_coord(1, atom_base(hatom, aa, 1), resi ));

//  for( int i=1; i <= 3; ++i ) {
//    hatm_coord( i ) = full_coord( i, hatom, resi );
//    base_coord( i ) =  full_coord( i, atom_base( hatom, aa, 1 ), resi );
//  }
  xyzVector_float vec = hatm_coord - base_coord;
  vec.normalize();
  v.dimension( 3 );
  for( int i = 1; i <= 3; ++i ) { v( i ) = vec( i ); }
}

// compute the plane equation: aX + bY + cZ + d = 0 for a phe residue, where a,b,c are contained in plane_norm
inline
void
plane_params(
int ares,
FArray1D_float & plane_norm,
float & d
)
{
  using namespace numeric;
  using namespace misc;

  compute_arom_normal( ares, plane_norm );
  d = -dotprod( full_coord( 1, 6, ares), plane_norm );
}

// find the closest point on the line v1/v2 to the point p.
inline
void
point_line_proj(
FArray1Da_float v1, // one point on the line
FArray1Da_float v2, // second point
FArray1D_float p, // point outside the line
FArray1D_float norm, // normal to the line/p plane
FArray1D_float &proj // return projection point on the v1/v2 line
)
{
  using namespace numeric;

  FArray1D_float v( 3 );
  for( int i=1; i<=3; ++i ) v( i ) = v1( i ) - v2( i );

  FArray1D_float n( 3 );
  n = cross( v, norm ).normalize(); // normal to v within the aromatic plane

  FArray1D_float temp( 3 );
  for( int i=1; i<=3; ++i ) temp( i ) = v1( i ) - p( i );
  proj = p + dotprod( temp, n ); // point on v
}

// find the distance between point xyz and plane aX + bY + cZ + d = 0
inline
float
point_plane_dist(
FArray1D_float plane_norm, // contains a,b,c. Must by normalized!!
float d, // fourth plane parameter
FArray1Da_float point
)
{
  float dist = dotprod( plane_norm, point ) + d;
  return ( dist );
}

// compute the projection of a point on the plane of a phe ring. If that projection falls outside the ring
// (approximated by a cricle of 1.5Ang radius), then return the nearest point to the projection on the edge of the
// ring.
void
nearest_point_on_ring(
int datm,
int dres,
int ares,
FArray1Da_float ring_centre, // the location of the centre of the ring
FArray1D_float &p_proj // the projection of p on the plane
)
{
  using namespace misc;
  using namespace numeric;

  float d; // the offset of the phe plane from the origin
  FArray1D_float plane_norm( 3 ); // normal to the phe plane
  plane_params( ares, plane_norm, d );
  float dist = point_plane_dist( plane_norm, d, full_coord( 1, datm, dres ) );
  p_proj.dimension( 3 );
  for( int i=1; i<=3; ++i ) p_proj( i ) = full_coord( i, datm, dres ) - plane_norm( i ) * dist;

  float dist_ring_centre;
  distance_bk( p_proj, ring_centre, dist_ring_centre );
  if( dist_ring_centre <= 1.4 * 1.4 ) // within the ring radius (approximation to being within the hexagon - actual phe radius=1.42)
    return;

  FArray1D_float dists( 6 ); // distances to p
  int nearest_carb( 6 );
  float nearest_dist( 1000.0 );
  for( int i = 6; i <= 11; ++i ) { // find nearest carb to p_proj
    distance_bk( full_coord( 1, i, ares ), p_proj, dists( i - 5 ) );
    if( dists( i - 5 ) <= nearest_dist ) {
      nearest_carb = i;
      nearest_dist = dists( i - 5 );
    }
  }
  // find second nearest carb
  int neighbour1, neighbour2, second_nearest;
  if( nearest_carb == 6 ) { neighbour1 = 11; neighbour2 = 7; }
  else if( nearest_carb == 11 ) { neighbour1 = 6; neighbour2 = 10; }
  else { neighbour1 = nearest_carb - 1; neighbour2 = nearest_carb + 1; }
  if( dists( neighbour1 - 5 ) <= dists( neighbour2 - 5 ) ) second_nearest = neighbour1;
  else second_nearest = neighbour2;

  FArray1D_float ring_proj( 3 );
  point_line_proj( full_coord( 1, nearest_carb, ares ), full_coord( 1, second_nearest, ares ), p_proj,
                   plane_norm, ring_proj );

  float dist1, dist2;
  distance_bk( ring_proj, full_coord( 1, nearest_carb, ares ), dist1 );
  distance_bk( ring_proj, full_coord( 1, second_nearest, ares ), dist2 );
  if( dist1 >= 1.5 * 1.5 ) // projection lies outside the edge, so it's closest to nearest_carb
    for( int i=1; i<=3; ++i ) p_proj( i ) = full_coord( i, nearest_carb, ares );
  else // projection lies within the edge
    for( int i=1; i<=3; ++i ) p_proj( i ) = ring_proj( i );
}

// Place two points that float directly 1.5Ang above and below the ring centre
void
compute_floating_acceptor(
int ares,
FArray2D_float acoord,
FArray2D_float & floating_acceptor
)
{
  FArray1D_float norm( 3 );
  compute_arom_normal( ares, norm );
  for( int i=1; i<=3; ++i ) {
    floating_acceptor( i, 1 ) = acoord( i, 1 ) + norm( i ) * 1.5;
    floating_acceptor( i, 2 ) = acoord( i, 1 ) - norm( i ) * 1.5;
  }
}

// Return the hbond parameters:
// delta - h - acceptor distance
// theta - angle at the hyd atom (for aromatic acceptors, compared to the ring plane; for
// non arom acceptors, simply the angle formed by the vectors joining the H to the acceptor,
// and the H to the donor)
// phi   - angle at the acceptor atom
void
compute_arom_hbond_params(
int datm, // hyd atom
int dres,
int aatm, // either the atom bound to the acceptor or 0 (if aromatic)
int ares,
FArray1Da_float acoord, // acceptor atom coordinates
float & delta,
float & theta,
float & phi
)
{
  using namespace numeric;
  using namespace numeric::constants::f;
  using namespace pdbstats::arom_hbonds_stats;
  using namespace misc;

  FArray1D_float vecAR( 3 ), vecAH( 3 ), vecDH( 3 );
  FArray1D_float dcoord( 3 );

  for( int i=1; i<=3; ++i ) { dcoord( i ) = full_coord( i, datm, dres ); }

  distance2_bk( dcoord, acoord, delta );

  if( delta >= delta_cutoff2 ) return;

// ALWAYS KEEP AT LEAST ONE OF THE SWITCHES BELOW TURNED OFF!!!
//#define __RING_CENTRE_AND_PERIPHERY__ // the ring centre & any of the periph. carbs are used as potential acceptors (the closest coordinate is used)
//#define __RING_FLAT__ // a projection from the hyd to the ring is used as the acceptor
// o/w only the ring centre is used as the acceptor

#ifdef __RING_CENTRE_AND_PERIPHERY__
  if ( aatm == 0 ) { // aromatic acceptor
// the Phi angle is computed between the normal to the ring and the vector connecting either the ring center or
// any of the peripheral carbons to the h-atom. This is a cheap substitute to computing the angle between the
// ring plane and the h-atom.
    float mindist = delta; int nearest_carbon = 0;
    for( int i = 6; i <= 11; ++i ) { // find nearest aromatic atom
      float d;
      distance2_bk( dcoord, full_coord( 1, i, ares ), d );
      if ( d <= mindist ) {
        mindist = d;
        nearest_carbon = i;
      }
    }
    if ( nearest_carbon != 0 ) // nearest aromatic carbon is not the ring centre
      for( int i = 1; i <= 3; ++i ) { acoord( i ) = full_coord( i, nearest_carbon, ares ); }
  }
#endif

#ifdef __RING_FLAT__
  if( aatm == 0 ) { // aromatic acceptor
// The phi angle is computed between the normal to the ring and the vector connecting the nearest point on the phe
// ring to the h-atom.
    FArray1D_float p_proj( 3 );
    nearest_point_on_ring( datm, dres, ares, acoord, p_proj );
    acoord = p_proj;
  }
#endif

  for ( int i = 1; i <= 3; ++i ) { vecAH( i ) = acoord( i ) - dcoord( i ); }
    vecAH.normalize();

  compute_donor_vector( datm, dres, vecDH );
  if ( aatm == 0 ) { // aromatic acceptor
    FArray1D_float norm( 3 ); // the normal to the plane
    compute_arom_normal( ares, norm );

    theta =  std::acos( dotprod( norm, vecDH ) ) * 180/pi;
    phi = std::acos( dotprod( norm, vecAH ) ) * 180/pi;

    if( theta > 90 ) theta = 180 - theta; // the two sides of the ring are identical
    if( phi > 90 )   phi   = 180 - phi;
  }
  else { // non-aromatic acceptor
   theta = std::acos( dotprod( vecDH, vecAH ) ) * 180/pi;

    for( int i = 1; i <= 3; ++i ) {
      vecAR( i ) = acoord( i ) - full_coord( i, aatm, ares );
    }
    vecAR.normalize();
    phi = std::acos( dotprod( vecAH, vecAR ) ) * 180 / pi;
  }
}

inline
void
print_arom_hbond_entry(
std::string const description,
int const res1,
int const res2,
float const delta2,
float const theta,
float const phi
) {
  std::cout << "AROM_HBOND_STATS ";
  std::cout << description << " " << res1 << " " << res2 << " " << std::sqrt( delta2 ) << " " << theta << " " << phi << std::endl;
}

////////
void
get_arom_hbond_stats()
{
  float theta, delta, phi;
	using namespace cenlist_ns;
	using namespace misc;
	using namespace pdb;
  using namespace pdbstats::arom_hbonds_stats;
  using namespace param_aa;
  using namespace numeric;

  FArray2D_float acceptor1_pos( 3, 2, 0.0f );
  FArray2D_float donor_centroid_pos( 3, 2, 0.0f ); // used only to compute the ring centre of the donor ring

//properties_per_atom_aa_aav
  using namespace aaproperties_pack::properties_per_atom_aa_aav; // for atom_base etc.
//  using namespace aaproperties_pack::intra_cp; // for atom_base
  using namespace aaproperties_pack;

// res1 is treated as a potential acceptor
// res2 is treated as a potential donor
// that's why both iterate over all residues
	for ( int res1 = 1; res1 <= total_residue; ++res1 ) {
    int aa1 = res( res1 );
		//    bool hbond1 = is_hbonding_res( aa1 );
    bool arom1  = is_aromatic_res( aa1 );
    if ( arom1 )
      compute_arom_acceptor_pos( res1, acceptor1_pos );
    for ( int res2 = 1; res2 <= total_residue; ++res2 ) {
      int aa2 = res( res2 );
			//      bool hbond2 = is_hbonding_res( aa2 );
      bool cation2 = is_cation_res( aa2 );
      bool arom2  = is_aromatic_res( aa2 );

// sequence neighbours are not counted
      if ( !arom1 && !arom2 )
        continue;
      if ( std::abs( res1 - res2 ) <= 4 ) continue;
      if ( cendist( res1, res2 ) >= 100.0 )  // 10Ang distance cutoff on centroids
        continue;

     if( arom2 ) {
        int n2e = nH_aromatic( aa2, 1 );
        for( int n2 = 1; n2 <= n2e; ++n2 ) { // all aromatic hyds
          if( arom1 ) { // arom-arom
#define __DISTANCE_BETWEEN_CENTROIDS__
//#define __ACCEPTOR_CENTRE__
#ifdef __ACCEPTOR_CENTRE__
// Hbonding with aromatic center || centroid to centroid distances (see below)
            compute_arom_hbond_params( Hpos_aromatic( n2, aa2, 1 ), res2, 0, res1,
              acceptor1_pos( 1, 1 ), delta, theta, phi );
#endif

#ifdef __DISTANCE_BETWEEN_CENTROIDS__
// override the above-computed delta with the distance between the two centroids
// useful for comparing the statistics to Morozov et al. J Phys Chem B 108, 8489
            compute_arom_hbond_params( Hpos_aromatic( n2, aa2, 1 ), res2, 0, res1,
              acceptor1_pos( 1, 1 ), delta, theta, phi );
            if( delta <= delta_cutoff2 ) { // the reported delta will be the centroid-centroid dist
              compute_arom_acceptor_pos( res2, donor_centroid_pos );
              distance2_bk( donor_centroid_pos( 1, 1 ), acceptor1_pos( 1, 1 ), delta );
              delta_cutoff2 = 36.0f; // override the delta cutoff to allow the cent-cent dist to be reported
            }
#endif

#ifndef __ACCEPTOR_CENTRE__
#ifndef __DISTANCE_BETWEEN_CENTROIDS__
// Hbonding with peripheral carbons
            float delta_tmp, theta_tmp, phi_tmp;
            delta = 10000.0f;
            for(int aro=6; aro<=11; ++aro) {
              compute_arom_hbond_params( Hpos_aromatic( n2, aa2, 1), res2, 0, res1,
                full_coord( 1, aro, res1 ), delta_tmp, theta_tmp, phi_tmp );
              if( delta_tmp <= delta ) {
                delta = delta_tmp; theta = theta_tmp; phi = phi_tmp;
              }
            }

#endif
#endif
            if( res1 == aa_trp ) {
              float delta_tmp, theta_tmp, phi_tmp;
              compute_arom_hbond_params( Hpos_aromatic( n2, aa2, 1 ), res2, 0, res1,
                  acceptor1_pos( 1, 2 ), delta_tmp, theta_tmp, phi_tmp );
              if( delta_tmp < delta ) { delta = delta_tmp; theta = theta_tmp; phi = phi_tmp; }
            }
            if( delta < delta_cutoff2 ) {
              int const phi_b( static_cast<int> (phi / 360 * phi_bins) + 1 );
              int const theta_b( static_cast<int> (theta / 360 * theta_bins) + 1 );

              arom_arom_delta[ static_cast<int> (std::sqrt(delta) / std::sqrt(delta_cutoff2) * delta_bins) + 1] ++;
              print_arom_hbond_entry( "arom-arom", res1, res2, delta, theta, phi );
              if( delta <= short_cutoff2 ) {
                arom_arom_phi_short  [ phi_b ]++;
                arom_arom_theta_short[ theta_b ]++;
              }
							else if ( delta >= vlong_cutoff2 ) {
								arom_arom_phi_vlong  [ phi_b ]++;
								arom_arom_theta_vlong[ theta_b ]++;
							}
							else {
                arom_arom_phi_long  [ phi_b ]++;
                arom_arom_theta_long[ theta_b ]++;
              }
            } // delta
#ifdef __DISTANCE_BETWEEN_CENTROIDS__
            delta_cutoff2 = 3.5 * 3.5; // restore the delta cutoff
#endif
          } //arom-arom
          // non-arom acceptor - arom donor (including bb acceptors)
					int n1e = nacceptors( aa1, 1 );
					for( int n1 = 1; n1 <= n1e; ++n1 ) { // all res1 non-arom acceptors
						int acceptor_atm = accpt_pos( n1, aa1, 1 );
//             if ( acceptor_atm <= 4 )
// bb atoms are ignored for now, but should be considered in future
//               continue;
						int acceptor_base = atom_base( acceptor_atm, aa1, 1 );
						compute_arom_hbond_params( Hpos_aromatic( n2, aa2, 1 ), res2, acceptor_base, res1,
							full_coord( 1, acceptor_atm, res1 ), delta, theta, phi );
						if( delta < delta_cutoff2 ) {
							int const phi_b( static_cast<int> (phi / 360 * phi_bins) + 1 );
							int const theta_b( static_cast<int> (theta / 360 * theta_bins) + 1 );
							if ( acceptor_atm <= 4 ) { // bb atoms
                print_arom_hbond_entry( "nonarom-arom-bb", res1, res2, delta, theta, phi );
								bb_polar_arom_donor_delta[ static_cast<int> (std::sqrt(delta) / std::sqrt(delta_cutoff2) * delta_bins) + 1]++;
								if( delta <= short_cutoff2 ) {
									bb_polar_arom_donor_phi_short  [ phi_b ]++;
									bb_polar_arom_donor_theta_short[ theta_b ]++;
								}
								else if ( delta >= vlong_cutoff2 ) {
									bb_polar_arom_donor_phi_vlong  [ phi_b ]++;
									bb_polar_arom_donor_theta_vlong[ theta_b ]++;
								}
								else {
									bb_polar_arom_donor_phi_long  [ phi_b ]++;
									bb_polar_arom_donor_theta_long[ theta_b ]++;
								}
							} // bb atoms
							else { // sidechain
								polar_arom_donor_delta[ static_cast<int> (std::sqrt(delta) / std::sqrt(delta_cutoff2) * delta_bins) + 1]++;
                print_arom_hbond_entry( "nonarom-arom", res1, res2, delta, theta, phi );
								if( delta <= short_cutoff2 ) {
									polar_arom_donor_phi_short  [ phi_b ]++;
									polar_arom_donor_theta_short[ theta_b ]++;
								}
								else if ( delta >= vlong_cutoff2 ) {
									polar_arom_donor_phi_vlong  [ phi_b ]++;
									polar_arom_donor_theta_vlong[ theta_b ]++;
								}
								else {
									polar_arom_donor_phi_long  [ phi_b ]++;
									polar_arom_donor_theta_long[ theta_b ]++;
								}
							} // sidechain
						} // delta
					} // res1 non arom acceptors
				// nonarom acceptor - arom donor
        } // arom hyd
      } // arom2
      else { // arom acceptor - non-arom donor
        int n2e = nH_polar( aa2, 1 );
        FArray2D_float floating_acceptor( 3, 2 ); // for cations
        if( cation2 ) // compute position of the acceptor as though it's floating above & below the aro ring
          compute_floating_acceptor( res1, acceptor1_pos, floating_acceptor );
        for( int n2 = 2; n2 <= n2e; ++n2 ) { // all donor hydrogens EXCEPT backbone
          int n2_pos = Hpos_polar( n2, aa2, 1 );
          compute_arom_hbond_params( n2_pos, res2, 0, res1, ( cation2 ? floating_acceptor( 1, 1 ) :
                acceptor1_pos( 1, 1 ) ), delta, theta, phi);
          if( res1 == aa_trp || cation2 ) { // 2 rings or floating_acceptor
            float delta_tmp, theta_tmp, phi_tmp;
            compute_arom_hbond_params( n2_pos, res2, 0, res1, ( cation2 ? floating_acceptor( 1, 2 ) :
                acceptor1_pos( 1, 2 ) ), delta_tmp, theta_tmp, phi_tmp );
            if( delta_tmp < delta ) { delta = delta_tmp; theta = theta_tmp; phi = phi_tmp; }
          } // trp || cation2
          if( delta < delta_cutoff2 ) {
            int const phi_b( static_cast<int> (phi / 360 * phi_bins) + 1 );
            int const theta_b( static_cast<int> (theta / 360 * theta_bins) + 1 );

            if ( n2_pos <= 4 ) { // backbone
              print_arom_hbond_entry( "arom-nonarom-bb", res1, res2, delta, theta, phi );
              bb_polar_arom_acceptor_delta[ static_cast<int> (std::sqrt(delta) / std::sqrt(delta_cutoff2) * delta_bins) + 1]++;
              if( delta <= short_cutoff2 ) {
                bb_polar_arom_acceptor_phi_short   [ phi_b ]++;
                bb_polar_arom_acceptor_theta_short [ theta_b ]++;
              }
              else if ( delta >= vlong_cutoff2 ) {
                bb_polar_arom_acceptor_phi_vlong   [ phi_b ]++;
                bb_polar_arom_acceptor_theta_vlong [ theta_b ]++;
              }
              else {
                bb_polar_arom_acceptor_phi_long   [ phi_b ]++;
                bb_polar_arom_acceptor_theta_long [ theta_b ]++;
              }
            } // bb
            else if ( cation2 ) { // hbond2 ) { // sidechain, treat only 'allowed' donors
              polar_arom_acceptor_delta[ static_cast<int> (std::sqrt(delta) / std::sqrt(delta_cutoff2) * delta_bins) + 1]++;
              print_arom_hbond_entry( "arom-cation", res1, res2, delta, theta, phi );
              if( delta <= short_cutoff2 ) {
                polar_arom_acceptor_phi_short   [ phi_b ]++;
                polar_arom_acceptor_theta_short [ theta_b ]++;
              }
              else if ( delta >= vlong_cutoff2 ) {
                polar_arom_acceptor_phi_vlong   [ phi_b ]++;
                polar_arom_acceptor_theta_vlong [ theta_b ]++;
              }
              else {
                polar_arom_acceptor_phi_long   [ phi_b ]++;
                polar_arom_acceptor_theta_long [ theta_b ]++;
              }
            } // sc
          } // delta
        } // donor hyds
      } // arom acceptor - non-arom donor
    } // res2
  } // res1
}

void
output_arom_hbond_stats()
{
  using namespace pdbstats::arom_hbonds_stats;

  std::cout << "AROM_HBOND SUMMARY---------------" << std::endl;
  std::cout << "arom_arom" << std::endl;
  for( int d = 1; d <= delta_bins; ++d )
    std::cout << "delta: " << d * 0.1 << " " <<
        arom_arom_delta[ d ] << std::endl;
  std::cout << "degrees\ttheta short\ttheta long\ttheta vlong\tphi short\tphi long\tphi vlong" << std::endl;
  for( int t = 1; t <= theta_bins; ++t )
    std::cout << t * 360 / theta_bins << '\t' <<
        arom_arom_theta_short[ t ] << '\t' << arom_arom_theta_long[ t ] << '\t' <<
        arom_arom_theta_vlong[ t ] << '\t' << arom_arom_phi_short[ t ] << '\t' <<
        arom_arom_phi_long[ t ] << '\t' << arom_arom_phi_vlong[ t ] << std::endl;
  std::cout << std::endl << std::endl;

  std::cout << "polar_arom_donor sidechain" << std::endl;
  for( int d = 1; d <= delta_bins; ++d )
    std::cout << "delta: " << d  * 0.1 << " " <<
        polar_arom_donor_delta[ d ] << std::endl;
  std::cout << "degrees\ttheta short\ttheta long\ttheta vlong\tphi short\tphi long\tphi vlong" << std::endl;
  for( int t = 1; t <= theta_bins; ++t )
    std::cout << t * 360 / theta_bins << '\t' <<
        polar_arom_donor_theta_short[ t ] << '\t' << polar_arom_donor_theta_long[ t ] << '\t' <<
        polar_arom_donor_theta_vlong[ t ] << '\t' << polar_arom_donor_phi_short[ t ] << '\t' <<
        polar_arom_donor_phi_long[ t ] << '\t' << polar_arom_donor_phi_vlong[ t ] << std::endl;
    std::cout << std::endl << std::endl;

  std::cout << "polar_arom_acceptor sidechain" << std::endl;
  for( int d = 1; d <= delta_bins; ++d )
    std::cout << "delta: " << d  * 0.1 << " " <<
        polar_arom_acceptor_delta[ d ] << std::endl;
  std::cout << "degrees\ttheta short\ttheta long\ttheta vlong\tphi short\tphi long\tphi vlong" << std::endl;
  for( int t = 1; t <= theta_bins; ++t )
    std::cout << t * 360 / theta_bins << '\t' <<
        polar_arom_acceptor_theta_short[ t ] << '\t' << polar_arom_acceptor_theta_long[ t ] << '\t' <<
        polar_arom_acceptor_theta_vlong[ t ] << '\t' << polar_arom_acceptor_phi_short[ t ] << '\t' <<
        polar_arom_acceptor_phi_long[ t ] << '\t' << polar_arom_acceptor_phi_vlong[ t ] << std::endl;
  std::cout << std::endl << std::endl;

  std::cout << "polar_arom_donor backbone" << std::endl;
  for( int d = 1; d <= delta_bins; ++d )
    std::cout << "delta: " << d  * 0.1 << " " <<
        bb_polar_arom_donor_delta[ d ] << std::endl;
  std::cout << "degrees\ttheta short\ttheta long\ttheta vlong\tphi short\tphi long\tphi vlong" << std::endl;
  for( int t = 1; t <= theta_bins; ++t )
    std::cout << t * 360 / theta_bins << '\t' <<
        bb_polar_arom_donor_theta_short[ t ] << '\t' << bb_polar_arom_donor_theta_long[ t ] << '\t' <<
        bb_polar_arom_donor_theta_vlong[ t ] << '\t' << bb_polar_arom_donor_phi_short[ t ] << '\t' <<
        bb_polar_arom_donor_phi_long[ t ] << '\t' << bb_polar_arom_donor_phi_vlong[ t ] << std::endl;
    std::cout << std::endl << std::endl;

  std::cout << "bb_polar_arom_acceptor backbone" << std::endl;
  for( int d = 1; d <= delta_bins; ++d )
    std::cout << "delta: " << d  * 0.1 << " " <<
        bb_polar_arom_acceptor_delta[ d ] << std::endl;
  std::cout << "degrees\ttheta short\ttheta long\ttheta vlong\tphi short\tphi long\tphi vlong" << std::endl;
  for( int t = 1; t <= theta_bins; ++t )
    std::cout << t * 360 / theta_bins << '\t' <<
        bb_polar_arom_acceptor_theta_short[ t ] << '\t' << bb_polar_arom_acceptor_theta_long[ t ] << '\t' <<
        bb_polar_arom_acceptor_theta_vlong[ t ] << '\t' << bb_polar_arom_acceptor_phi_short[ t ] << '\t' <<
        bb_polar_arom_acceptor_phi_long[ t ] << '\t' << bb_polar_arom_acceptor_phi_vlong[ t ] << std::endl;
  std::cout << std::endl << std::endl;
  std::cout << "AROM_HBOND SUMMARY---------------" << std::endl;
}
