// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 01:39:36 -0500 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "planes.h"
#include "aaproperties_pack.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "planes_ns.h"
#include "read_paths.h"
#include "score_ns.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


//==============================================================================
//km  08 April 2003
//km  Draw planes through selected sidechains, finds angles and distances
//km  between them.  Originally developed to capture pi-pi and cation-pi
//km  interactions, as well as optimal packing arrangements, that might
//km  be missed by Lennard-Jones.
//==============================================================================


///////////////////////////////////////////////////////////////////////////////
/// @begin read_plane_table
///
/// @brief
///
/// @detailed
///  This reads in a partially amino-acid specific table generated from
///  ~3500 pdb x-ray structures of < 2.5A resolution.  Dimensions are
///  (aa_pairs)x3x24x28 total bins:
///    3 angle bins of 0-30deg, 30-60deg, 60-90deg
///   28 horizontal distance bins (in plane of res1)
///   24 vertical distance bins (above plane of res1)
///
///@parma
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////////
void
read_plane_table()
{
	using namespace planes_ns;

	int pair_class,angle_bin,horiz_bin,vert_bin;
	float plane_probability;

	utility::io::izstream & table_stream( open_data_file( "plane_data_table_1015.dat" ) );
	for ( int i = 1, ie = max_class*max_angle_bins*max_vert_bins*max_horiz_bins;
	 i <= ie; ++i ) {
		table_stream >> bite( 4, pair_class ) >> bite( 5, angle_bin ) >>
		 bite( 5, vert_bin ) >> bite( 5, horiz_bin ) >>
		 bite( 10, plane_probability ) >> skip;
		Pangle_orientation(pair_class,angle_bin,vert_bin,horiz_bin) =
		 plane_probability;
	}
	table_stream.close();
	table_stream.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin find_plane_orientation
///
/// @brief
/// caluclates plane orientation score (full atom mode only) for a pair
/// of sidechains
///
/// @detailed
/// This function determines if the sidechain pair is making potential
/// cation-pi, pi-pi or hydrophobic packing interactions, by calling
/// select_plane_pair and get_contact_def.  If so, calculate_angle_distance
/// or calculate_angle_distance_phobic, depending on whether the sidechains
/// are making hydrophobic interactions or not, is called to calculate
/// the geometric parameters angle, horiz and vert.
/// The main  function is called from get_trial_energies, which is called from
/// fullatom_rotamer_trials. Pass in aa1,aav1,res1 and coordinates from
/// current rotamer.
///
/// The value passed out is plane_totalE, which is the total plane
/// orientation energy and is defined as the sum of the aromatic-aromatic,
/// aromatic/cation-proline, arginine-aromatic/proline, histadine-aromatic
/// and hydrophobic-hydrophobic interaction energies. The individual
/// components can be examined by setting plane_totalE equal to the
/// component of interest.
///
/// @param  aa1 - [in/out]? -
/// @param  aav1 - [in/out]? -
/// @param  aa2 - [in/out]? -
/// @param  aav2 - [in/out]? -
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  plane_totalE - [in/out]? -
///
/// @global_read - integer - natoms  aaproperties_pack.h
/// @global_read - integer - MAX_ATOM param.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
find_plane_orientation(
	int const aa1,
	int const aav1,
	int const aa2,
	int const aav2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	float & plane_totalE
)
{
	using namespace aaproperties_pack; // fullatom_type, natoms
	using namespace param;  // MAX_ATOM
	using namespace param_aa;
	using namespace planes_ns;
	using namespace scores;
	using numeric::xyzVector_float;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

// local:
	int a_bin, v_bin, h_bin;
	xyzVector_float p1;
	xyzVector_float q1;
	xyzVector_float r1;
	xyzVector_float s1;
	xyzVector_float p2;
	xyzVector_float q2;
	xyzVector_float r2;
	xyzVector_float s2;
	FArray1D_int points_plane1( max_plane );
	FArray1D_int points_plane2( max_plane );
	float dis, a_size, v_size, h_size;
// first and second residues in the pair being counted
	int atom_pair_counter,pair_class;
	float vert_distance,horiz_distance,plane_angle,distance_sep(0.);
//     real centroid_distance,
	bool plane_selected,hydrophobic_selected,cation_selected, proline_selected,
	 histadine_selected;
	int relevant_aa1_start(0),
		relevant_aa1_end(0),
		relevant_aa2_start(0),
		relevant_aa2_end(0),
		contact_def(0);
// local score components:
	float pi_score,hydrophobic_score,cation_score,proline_score, histadine_score;
// Coordinates passed in from get_trial_energies for aa pair:

// check that if we have already met the requirement for x atom
// contacts, jump out of loop so we don't calculate the plane energy
// for a pair more than once

	bool success = false;

	pi_score = 0.0;
	hydrophobic_score = 0.0;
	cation_score = 0.0;
	proline_score = 0.0;
	histadine_score = 0.0;
//	total_score = 0.0;
//	plane_pi_score = 0.0;
//	plane_hydrophobic_score = 0.0;
//	plane_proline_score = 0.0;
//	plane_cation_score = 0.0;
//	plane_histadine_score = 0.0;
	fa_plane_score = 0.0;
	plane_totalE = 0.0;

	pair_class = 0;

	select_pair(aa1,aa2,plane_selected,hydrophobic_selected, cation_selected,
	 proline_selected,histadine_selected);
	if ( ( plane_selected ) || ( hydrophobic_selected ) || ( cation_selected ) ||
			 ( proline_selected ) || ( histadine_selected ) &&
			 !(is_nonnatural(aa1) || is_nonnatural(aa2)) ) {
		get_contact_def(plane_selected,cation_selected, proline_selected,
		 histadine_selected, hydrophobic_selected,contact_def,distance_sep);
		relevant_sidechain_atoms(aa1,aa2,relevant_aa1_start, relevant_aa1_end,
		 relevant_aa2_start,relevant_aa2_end);

// Start counting number of atom pairs that come within 4A of each other
		atom_pair_counter = 0;
		int const natoms_11 = natoms(aa1,aav1);
		int const natoms_22 = natoms(aa2,aav2);
		for ( int atom1 = relevant_aa1_start; atom1 <= relevant_aa1_end;
		 ++atom1 ) { // sidechain atoms
			if ( success ) goto L200;
			float & coord1_11( coord1(1,atom1) );
			for ( int atom2 = relevant_aa2_start; atom2 <= relevant_aa2_end;
			 ++atom2 ) { // sidechain atoms
				if ( success ) goto L200;
				distance_bk(coord1_11,coord2(1,atom2),dis);
				if ( dis < distance_sep ) {
					++atom_pair_counter;

// ** This is where you define how many contacts sidechains should have
					if ( atom_pair_counter >= contact_def ) {
						success = true;

// Res1 and res2 have met the criteia of having at least two contacts (<4.0A),
// Choose number and identity of atoms to define plane

						select_plane_points(aa1,aa2,points_plane1,points_plane2);

						for ( int i = 1; i <= natoms_11; ++i ) {
							if ( i == points_plane1(1) ) {
								p1 = &coord1(1,i);
							} else if ( i == points_plane1(2) ) {
								q1 = &coord1(1,i);
							} else if ( i == points_plane1(3) ) {
								r1 = &coord1(1,i);
							} else if ( i == points_plane1(4) ) {
								s1 = &coord1(1,i);
							}
						}
						for ( int j = 1; j <= natoms_22; ++j ) {
							if ( j == points_plane2(1) ) {
								p2 = &coord2(1,j);
							} else if ( j == points_plane2(2) ) {
								q2 = &coord2(1,j);
							} else if ( j == points_plane2(3) ) {
								r2 = &coord2(1,j);
							} else if ( j == points_plane2(4) ) {
								s2 = &coord2(1,j);
							}
						}

// now get plane_angle, vertical and horizontal distances for this pair

						if ( plane_selected ) {
							calculate_angle_distance(p1,q1,r1,s1,p2,q2,r2,s2,plane_angle,
							 vert_distance,horiz_distance);
						}
						if ( cation_selected ) {
							calculate_angle_distance(p1,q1,r1,s1,p2,q2,r2,s2,plane_angle,
							 vert_distance,horiz_distance);
						}
						if ( proline_selected ) {
							calculate_angle_distance(p1,q1,r1,s1,p2,q2,r2,s2,plane_angle,
							 vert_distance,horiz_distance);
						}
						if ( histadine_selected ) {
							calculate_angle_distance(p1,q1,r1,s1,p2,q2,r2,s2,plane_angle,
							 vert_distance,horiz_distance);
						}
						if ( hydrophobic_selected ) {
							calculate_angle_distance_phobic(p1,q1,r1,p2,q2,r2,plane_angle,
							 vert_distance,horiz_distance);
						}

// Bounds checking (can increase scope of table if desired) :
						if ( vert_distance >= 6.0 ) goto L190;
						if ( horiz_distance >= 7.0 ) goto L190;

// Decide which bin we are in and lookup score:
						a_size = angle_bin_size;
						if (plane_angle == 90.0) {  //this will return a_bin = 4,which doesn't exist
							a_bin = 3;
						} else {
						a_bin = static_cast< int >( plane_angle / a_size ) + 1;
						}

						v_size = vert_bin_size;
						v_bin = static_cast< int >( vert_distance / v_size ) + 1;

						h_size = horiz_bin_size;
						h_bin = static_cast< int >( horiz_distance / h_size ) + 1;

// These are the pairs we are considering, decide which class we are in:

						if ( ( aa1 == 5 && ( aa2 == 5 || aa2 == 19 || aa2 == 20 ) ) ||
						 ( aa1 == 20 && ( aa2 == 19 || aa2 == 20 ) ) ) {
							pair_class = 1; // FF,FW,FY,YW,YY
						} else if ( aa1 == 15 && ( aa2 == 5 || aa2 == 7 || aa2 == 15 ||
						 aa2 == 19 || aa2 == 20 ) ) {
							pair_class = 2; // RF,RH,RR,RW,RY
						} else if ( ( aa1 == 5 || aa1 == 7 || aa1 == 15 || aa1 == 19 ||
						 aa1 == 20 ) && aa2 == 13 ) {
							pair_class = 3; // FP,HP,RP,WP,YP
						} else if ( aa1 == 7 && ( aa2 == 5 || aa2 == 19 || aa2 == 20 ) ) {
							pair_class = 4; // HF,HW,HY
						} else if ( ( aa1 == 20 ) &&
						 ( aa2 == 8 || aa2 == 10 || aa2 == 18 ) ) {
							pair_class = 5; // YI,YL,YV
						} else if ( ( aa1 == 19 ) &&
						 ( aa2 == 8 || aa2 == 10 || aa2 == 18 ) ) {
							pair_class = 6; // WI,WL,WV
						} else if ( aa1 == 10 && aa2 == 8 ) {
							pair_class = 7; // LI
						} else if ( aa1 == 10 && aa2 == 10 ) {
							pair_class = 8; // LL
						} else if ( aa1 == 10 && aa2 == 18 ) {
							pair_class = 9; // LV
						} else if ( aa1 == 10 && aa2 == 5 ) {
							pair_class = 10; // LF
						} else if ( aa1 == 8 && aa2 == 8 ) {
							pair_class = 11; // II
						} else if ( aa1 == 8 && aa2 == 18 ) {
							pair_class = 12; // IV
						} else if ( aa1 == 8 && aa2 == 5 ) {
							pair_class = 13; // IF
						} else if ( aa1 == 18 && aa2 == 18 ) {
							pair_class = 14; // VV
						} else if ( aa1 == 18 && aa2 == 5 ) {
							pair_class = 15; // VF
						}

// saftey check to make sure class was assigned, we don't want to look up
// a value in planes_table_1015.dat that doesn't exist:

						if ( pair_class == 0 ) {  // no class assigned
							std::cout << "aa1, aa2: " << SS( aa1 ) << SS( aa2 ) << std::endl;
							std::cout << "ABORT: unable to assign class for planes" << std::endl;
							utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
						}

// Now look up score in table:
// the commented out lines are to copy the score components to plane_x_score
// in case we want to access them outside this function, can use get_x_score()

						if ( plane_selected ) {
							pi_score = Pangle_orientation(pair_class,
							 a_bin,v_bin,h_bin);
//							plane_pi_score = pi_score;
						} else if ( cation_selected ) {
							cation_score = Pangle_orientation(pair_class,
							 a_bin,v_bin,h_bin);
//							plane_cation_score = cation_score;
						} else if ( proline_selected ) {
							proline_score = Pangle_orientation(pair_class,
							 a_bin,v_bin,h_bin);
//							plane_proline_score = proline_score;
						} else if ( histadine_selected ) {
							histadine_score = Pangle_orientation(pair_class,
							 a_bin,v_bin,h_bin);
//							plane_histadine_score = histadine_score;
						} else if ( hydrophobic_selected ) {
							hydrophobic_score = Pangle_orientation(pair_class,
							 a_bin,v_bin,h_bin);
//							plane_hydrophobic_score = hydrophobic_score;
						}
L190:;
					}   // check for at least two contacting atom pairs
				}   // dis lt. 4.2A
			}   // atom2
		}   // atom1
	}  // check to make sure restype is phe (or whatever)
L200:

//	plane_totalE = plane_pi_score + plane_hydrophobic_score +
//  plane_cation_score + plane_proline_score + plane_histadine_score;

	plane_totalE = pi_score + hydrophobic_score + cation_score +  proline_score +
	 histadine_score;

	fa_plane_score = plane_totalE;
}

//------------------------------------------------------------------------------
// END MAIN
//------------------------------------------------------------------------------


////////////////////////////////////////////////////////////////////////////////
/// @begin select_plane_points
///
/// @brief
///
/// @detailed
///
/// @param  aa1 - [in/out]?
/// @param  aa2 - [in/out]?
/// @param  points_plane1 - [in/out]?
/// @param  points_plane2 - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
select_plane_points(
	int const aa1,
	int const aa2,
	FArray1Da_int points_plane1,
	FArray1Da_int points_plane2
)
{
	int const max_plane = { 4 };

	points_plane1.dimension( max_plane );
	points_plane2.dimension( max_plane );

// Define which points we need for each type of sidechain:
	int const i = aa1;
	if ( i == 5 || i == 20 ) {
		points_plane1(1) = 8;
		points_plane1(2) = 9;
		points_plane1(3) = 7;
		points_plane1(4) = 10;
	} else if ( i == 19 ) {
		points_plane1(1) = 8;
		points_plane1(2) = 14;
		points_plane1(3) = 13;
		points_plane1(4) = 10;
	} else if ( i == 7 ) {
		points_plane1(1) = 7;
		points_plane1(2) = 10;
		points_plane1(3) = 8;
		points_plane1(4) = 9;
	} else if ( i == 13 ) {
		points_plane1(1) = 1;
		points_plane1(2) = 5;
		points_plane1(3) = 2;
		points_plane1(4) = 7;
	} else if ( i == 15 ) {
		points_plane1(1) = 8;
		points_plane1(2) = 10;
		points_plane1(3) = 9;
		points_plane1(4) = 11;
// hydrophobic, 3 points, for small_plane (chi2 doesn't matter)
	} else if ( i == 8 ) {
		points_plane1(1) = 6;
		points_plane1(2) = 7;
		points_plane1(3) = 8;
		points_plane1(4) = 5; // filler
	} else if ( i == 10 ) {
		points_plane1(1) = 6;
		points_plane1(2) = 7;
		points_plane1(3) = 8;
		points_plane1(4) = 5; // filler
	} else if ( i == 18 ) {
		points_plane1(1) = 5;
		points_plane1(2) = 6;
		points_plane1(3) = 7;
		points_plane1(4) = 5; // filler
	}

	int const j = aa2;
	if ( j == 5 || j == 20 ) {
		points_plane2(1) = 8;
		points_plane2(2) = 9;
		points_plane2(3) = 7;
		points_plane2(4) = 10;
	} else if ( j == 19 ) {
		points_plane2(1) = 8;
		points_plane2(2) = 14;
		points_plane2(3) = 13;
		points_plane2(4) = 10;
	} else if ( j == 7 ) {
		points_plane2(1) = 7;
		points_plane2(2) = 10;
		points_plane2(3) = 8;
		points_plane2(4) = 9;
	} else if ( j == 13 ) {
		points_plane2(1) = 1;
		points_plane2(2) = 5;
		points_plane2(3) = 2;
		points_plane2(4) = 7;
	} else if ( j == 15 ) {
		points_plane2(1) = 8;
		points_plane2(2) = 10;
		points_plane2(3) = 9;
		points_plane2(4) = 11;
// hydrophobic, 3 points, for small_plane (chi2 doesn't matter)
	} else if ( j == 8 ) {
		points_plane2(1) = 6;
		points_plane2(2) = 7;
		points_plane2(3) = 8;
		points_plane2(4) = 5; // filler
	} else if ( j == 10 ) {
		points_plane2(1) = 6;
		points_plane2(2) = 7;
		points_plane2(3) = 8;
		points_plane2(4) = 5; // filler
	} else if ( j == 18 ) {
		points_plane2(1) = 5;
		points_plane2(2) = 6;
		points_plane2(3) = 7;
		points_plane2(4) = 2; // filler
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin relevant_sidechain_atoms
///
/// @brief
///
/// @detailed
/// This is to define which atoms in the sidechain we should consider when
/// figuring contacts.  Atoms further along in the sidechain from relevant_aax
/// are counted when figuring the number of contacts a sidechain has with
/// another sidechain.
///
/// @param  aa1 - [in/out]?
/// @param  aa2 - [in/out]?
/// @param  relevant_aa1_start - [in/out]?
/// @param  relevant_aa1_end - [in/out]?
/// @param  relevant_aa2_start - [in/out]?
/// @param  relevant_aa2_end - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
relevant_sidechain_atoms(
	int const aa1,
	int const aa2,
	int & relevant_aa1_start,
	int & relevant_aa1_end,
	int & relevant_aa2_start,
	int & relevant_aa2_end
)
{
	if ( aa1 == 5 || aa1 == 20 ) {      // F,Y
		relevant_aa1_start = 6;
		relevant_aa1_end = 11;
	} else if ( aa1 == 7 ) {            // H
		relevant_aa1_start = 6;
		relevant_aa1_end = 10;
	} else if ( aa1 == 8 ) {            // I
		relevant_aa1_start = 5;
		relevant_aa1_end = 8;
	} else if ( aa1 == 10 ) {           // L
		relevant_aa1_start = 5;
		relevant_aa1_end = 8;
	} else if ( aa1 == 13 ) {           // P
		relevant_aa1_start = 5;
		relevant_aa1_end = 7;
	} else if ( aa1 == 15 ) {           // R
		relevant_aa1_start = 8;
		relevant_aa1_end = 11;
	} else if ( aa1 == 18 ) {           // V
		relevant_aa1_start = 5;
		relevant_aa1_end = 7;
	} else if ( aa1 == 19 ) {           // W
		relevant_aa1_start = 6;
		relevant_aa1_end = 14;
	}

	if ( aa2 == 5 || aa2 == 20 ) {      // F,Y
		relevant_aa2_start = 6;
		relevant_aa2_end = 11;
	} else if ( aa2 == 7 ) {            // H
		relevant_aa2_start = 6;
		relevant_aa2_end = 10;
	} else if ( aa2 == 8 ) {            // I
		relevant_aa2_start = 5;
		relevant_aa2_end = 8;
	} else if ( aa2 == 10 ) {           // L
		relevant_aa2_start = 5;
		relevant_aa2_end = 8;
	} else if ( aa2 == 13 ) {           // P
		relevant_aa2_start = 5;
		relevant_aa2_end = 7;
	} else if ( aa2 == 15 ) {           // R
		relevant_aa2_start = 8;
		relevant_aa2_end = 11;
	} else if ( aa2 == 18 ) {           // V
		relevant_aa2_start = 5;
		relevant_aa2_end = 7;
	} else if ( aa2 == 19 ) {           // W
		relevant_aa2_start = 6;
		relevant_aa2_end = 14;
	}
}

///////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///km this function takes two sets of three points, each defining a
///   plane, gets the cross product of two vectors in each plane,
///   then finds the angle between the planes
///
/// @detailed
///
/// @param  p1 - [in/out]?
/// @param  q1 - [in/out]?
/// @param  r1 - [in/out]?
/// @param  s1 - [in/out]?
/// @param  p2 - [in/out]?
/// @param  q2 - [in/out]?
/// @param  r2 - [in/out]?
/// @param  s2 - [in/out]?
/// @param  plane_angle - [in/out]?
/// @param  vert_distance - [in/out]?
/// @param  horiz_distance - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_angle_distance(
	numeric::xyzVector_float const & p1,
	numeric::xyzVector_float const & q1,
	numeric::xyzVector_float const & r1,
	numeric::xyzVector_float const & s1,
	numeric::xyzVector_float const & p2,
	numeric::xyzVector_float const & q2,
	numeric::xyzVector_float const & r2,
	numeric::xyzVector_float const & s2,
//	float & centroid_distance, //km don't need upstairs for anything as of now...
	float & plane_angle,
	float & vert_distance,
	float & horiz_distance
)
{
	using numeric::xyzVector_float;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

// local
	xyzVector_float vec1_1;
	xyzVector_float vec1_2;
	xyzVector_float vec2_1;
	xyzVector_float vec2_2;
	xyzVector_float cross1;
	xyzVector_float cross2;
	float n1n2;
	xyzVector_float centroid_plane1;
	xyzVector_float centroid_plane2;
	xyzVector_float centroid_vector;
	xyzVector_float vert_vector;
	xyzVector_float horiz_vector;

// get coordinates for center of plane1 and plane2:

	centroid_plane1 = ( p1 + q1 + r1 + s1 ) / 4;
	centroid_plane2 = ( p2 + q2 + r2 + s2 ) / 4;

	centroid_vector = centroid_plane2 - centroid_plane1;

// Get magnitude of centroid vector, in case we want to use this as a cutoff value elsewhere
//	centroid_distance = horiz_vector.length(); //Objexx:SGM Shouldn't this be centroid_vector???

// Get two vectors to describe plane, and normalized cross product between them.
// Plane1:
	vec1_1 = q1 - p1;
	vec1_2 = s1 - r1;
	cross1 = cross( vec1_1, vec1_2 ).normalize();
// Plane 2:
	vec2_1 = q2 - p2;
	vec2_2 = s2 - r2;
	cross2 = cross( vec2_1, vec2_2 ).normalize();

// Find angle between normal vectors cross1 and cross2:
// plane_angle = (cross1.cross2)/||cross1||||cross2||
// n1n2 = (cross1.cross2)
// l_n1 = ||cross1||
// l_n2 = ||cross2||

	n1n2 = dot( cross1, cross2 );
	assert( ( cross1.length_squared() > 0.0 ) );
	assert( ( cross2.length_squared() > 0.0 ) );
	plane_angle = degrees( std::acos( sin_cos_range( n1n2 ) ) );

// Set angle range from 0 to 90 degrees:
	if ( ( plane_angle > 90.0 ) && ( plane_angle <= 180.0 ) ) {
		plane_angle = ( 180.0 - plane_angle );
	}
// Now get vert_distance and horiz_distance:
	vert_distance = dot( centroid_vector, cross1 );
	vert_vector = vert_distance * cross1;
	vert_distance = std::abs( vert_distance );

	horiz_vector = centroid_vector - vert_vector;
	horiz_distance = horiz_vector.length();
}

/////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_angle_distance_phobic
///
/// @brief
///
/// @detailed
///km this function takes two sets of three points, each defining a
///   plane, gets the cross product of two vectors in each plane,
///   then finds the angle between the planes
///   this is the same as calculate_angle_distence except it takes three
///   points instead of four
///
/// @param  p1 - [in/out]?
/// @param  q1 - [in/out]?
/// @param  r1 - [in/out]?
/// @param  s1 - [in/out]?
/// @param  p2 - [in/out]?
/// @param  q2 - [in/out]?
/// @param  r2 - [in/out]?
/// @param  s2 - [in/out]?
/// @param  plane_angle - [in/out]?
/// @param  vert_distance - [in/out]?
/// @param  horiz_distance - [in/out]?
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_angle_distance_phobic(
	numeric::xyzVector_float const & p1,
	numeric::xyzVector_float const & q1,
	numeric::xyzVector_float const & r1,
	numeric::xyzVector_float const & p2,
	numeric::xyzVector_float const & q2,
	numeric::xyzVector_float const & r2,
	float & plane_angle,
	float & vert_distance,
	float & horiz_distance
)
{
	using numeric::xyzVector_float;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

// local
	xyzVector_float vec1_1;
	xyzVector_float vec1_2;
	xyzVector_float vec2_1;
	xyzVector_float vec2_2;
	xyzVector_float cross1;
	xyzVector_float cross2;
	float n1n2;
	xyzVector_float centroid_plane1;
	xyzVector_float centroid_plane2;
	xyzVector_float centroid_vector;
	xyzVector_float vert_vector;
	xyzVector_float horiz_vector;

// get coordinates for center of plane1 and plane2:
	centroid_plane1 = ( p1 + q1 + r1 ) / 3;
	centroid_plane2 = ( p2 + q2 + r2 ) / 3;

	centroid_vector = centroid_plane2 - centroid_plane1;

// Get two vectors to describe plane, normalized cross product between them.
// Plane1:
	vec1_1 = q1 - p1;
	vec1_2 = r1 - p1;
	cross1 = cross( vec1_1, vec1_2 ).normalize();
// Plane 2:
	vec2_1 = q2 - p2;
	vec2_2 = r2 - p2;
	cross2 = cross( vec2_1, vec2_2 ).normalize();

// Find angle between normal vectors cross1 and cross2:
// plane_angle = (cross1.cross2)/||cross1||||cross2||
// n1n2 = (cross1.cross2)
// l_n1 = ||cross1||
// l_n2 = ||cross2||

	n1n2 = dot( cross1, cross2 );
	assert( ( cross1.length_squared() > 0.0 ) );
	assert( ( cross2.length_squared() > 0.0 ) );
	plane_angle = degrees( std::acos( sin_cos_range( n1n2 ) ) );

// Set angle range from 0 to 90 degrees:
	if ( ( plane_angle > 90.0 ) && ( plane_angle <= 180.0 ) ) {
		plane_angle = ( 180.0 - plane_angle );
	}
// Now get vert_distance and horiz_distance:
	vert_distance = dot( centroid_vector, cross1 );
	vert_vector = vert_distance * cross1;
	vert_distance = std::abs( vert_distance );

	horiz_vector = centroid_vector - vert_vector;
	horiz_distance = horiz_vector.length();
}
//------------------------------------------------------------------------------
//km These functions are in case we want to access one of the score components
//   in the rest of rosetta.  In order to do this, plane_x_score will have to
//   be added every where there is plane_total_score.
//------------------------------------------------------------------------------
// function that returns plane_pi_score

//float
//get_plane_pi_score()
//{
//	return scores::plane_pi_score;
//}

//------------------------------------------------------------------------------
// function that returns plane_hydrophobic_score

//float
//get_plane_hydrophobic_score()
//{
//	return scores::plane_hydrophobic_score;
//}

//------------------------------------------------------------------------------
// function that returns plane_cation_score

//float
//get_plane_cation_score()
//{
//	return scores::plane_cation_score;
//}

//------------------------------------------------------------------------------
// function that returns plane_proline_score

//float
//get_plane_proline_score()
//{
//	return scores::plane_proline_score;
//}

//------------------------------------------------------------------------------
// function that returns plane_histadine_score

//float
//get_plane_histadine_score()
//{
//	return scores::plane_histadine_score;
//}

//------------------------------------------------------------------------------
// function that returns plane_total_score

//float
//get_plane_total_score()
//{
//	return scores::plane_total_score;
//}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_contact_def
///
/// @brief
/// function that determines contact definitions and distance cutoffs for
/// amino acid pairs, to be added to atom_pair_counter in Main
///
/// @detailed
///
/// @param  plane_selected - [in/out]?
/// @param  cation_selected - [in/out]?
/// @param  proline_selected - [in/out]?
/// @param  histadine_selected - [in/out]?
/// @param  hydrophobic_selected - [in/out]?
/// @param  contact_def_selected - [in/out]?
/// @param  distance_sep - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_contact_def(
	bool const plane_selected,
	bool const cation_selected,
	bool const proline_selected,
	bool const histadine_selected,
	bool const hydrophobic_selected,
	int & contact_def,
	float & distance_sep
)
{
	if ( plane_selected ) {
		contact_def = 1;
		distance_sep = 4.2;
	} else if ( cation_selected ) {
		contact_def = 1;
		distance_sep = 4.2;
	} else if ( proline_selected ) {
		contact_def = 1;
		distance_sep = 4.2;
	} else if ( histadine_selected ) {
		contact_def = 3;
		distance_sep = 4.1;
	} else if ( hydrophobic_selected ) {
		contact_def = 2;
		distance_sep = 4.2;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin select_pair
///
/// @brief
/// function to select which pairs of aa's we should do this calcuation on
///
/// @detailed
///
/// @param  aa1 - [in/out]?
/// @param  aa2 - [in/out]?
/// @param  plane_selected - [in/out]?
/// @param  hydrophobic_selected - [in/out]?
/// @param  cation_selected - [in/out]?
/// @param  proline_selected - [in/out]?
/// @param  histadine_selected - [in/out]?
///
/// @global_read
///
/// @global_write
///
// @remarks
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
select_pair(
	int const aa1,
	int const aa2,
	bool & plane_selected,
	bool & hydrophobic_selected,
	bool & cation_selected,
	bool & proline_selected,
	bool & histadine_selected
)
{
	plane_selected = false;
	hydrophobic_selected = false;
	cation_selected = false;
	proline_selected = false;
	histadine_selected = false;

// all planar interactions
	if ( ( aa1 == 5 && ( aa2 == 5 || aa2 == 19 || aa2 == 20 ) ) ||
	 ( aa1 == 20 && ( aa2 == 19 || aa2 == 20 ) ) ) {
		plane_selected = true;
	}
// interactions involving cation (arg)
	if ( aa1 == 15 &&
	 ( aa2 == 5 || aa2 == 7 || aa2 == 15 || aa2 == 19 || aa2 == 20 ) ) {
		cation_selected = true;
	}
// interactions invloving proline
	if ( ( aa1 == 5 || aa1 == 7 || aa1 == 15 || aa1 == 19 || aa1 == 20 ) &&
	 aa2 == 13 ) {
		proline_selected = true;
	}
// interactions invloving histadine
	if ( aa1 == 7 && ( aa2 == 5 || aa2 == 19 || aa2 == 20 ) ) {
		histadine_selected = true;
	}

// 0530 take these out of statistics too
	if ( aa1 == 13 && aa2 == 13 ) { // PP
		plane_selected = false;
	} else if ( aa1 == 19 && aa2 == 19 ) { // WW
		plane_selected = false;
	} else if ( aa1 == 7 && aa2 == 7 ) { // HH
		plane_selected = false;
	}

	if ( ( aa1 == 10 ) && ( aa2 == 8 || aa2 == 10 || aa2 == 5 || aa2 == 18 ) ) {
		hydrophobic_selected = true;
	} else if ( aa1 == 8 && ( aa2 == 8 || aa2 == 18 || aa2 == 5 ) ) {
		hydrophobic_selected = true;
	} else if ( aa1 == 18 && ( aa2 == 5 || aa2 == 18 ) ) {
		hydrophobic_selected = true;
	} else if ( aa1 == 20 && ( aa2 == 8 || aa2 == 10 || aa2 == 18 ) ) {
		hydrophobic_selected = true;
	} else if ( aa1 == 19 && ( aa2 == 8 || aa2 == 10 || aa2 == 18 ) ) {
		hydrophobic_selected = true;
	}

}
