// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20497 $
//  $Date: 2008-02-20 18:17:57 -0500 (Wed, 20 Feb 2008) $
//  $Author: csmith $


// Rosetta Headers
#include "after_opts.h"
#include "bond_angle.h"
#include "files_paths.h"
#include "minimize.h"
#include "nblist.h"
#include "pose_backrub_controller.h"
#include "pose_backrub.h"
#include "pose_rotamer_trials.h"
#include "random_numbers.h"
#include "runlevel.h"
#include "score.h"

// Utility Headers
#include <utility/io/ozstream.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// C++ Headers
#include <set>
#include <string>

namespace pose_ns {

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::set_pose
///
/// @brief
/// set the pose object to be manipulated by the Backrub_controller
///
/// @detailed
///
/// @param[in] pose - address of Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::set_pose(
	pose_ns::Pose * pose
)
{
	pose_ = pose;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init
///
/// @brief
/// (re)initialize the Backrub_controller to default values
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init()
{
	//pose_ = NULL;
	//rotcontrol_ ?
	//movie_ ?
	move_allowed_.clear();
	allowed_moves_.clear();
	allowed_phipsi_moves_.clear();
	attempted_moves_.clear();
	accepted_moves_.clear();
	attempted_moves_bond_angle_.init(0, 0, 0);
	accepted_moves_bond_angle_.init(0, 0, 0);
	max_bond_angle_ = numeric::conversions::radians(10.);
	max_angle_disp_2_ = numeric::conversions::radians(40.);
	max_angle_disp_3_ = numeric::conversions::radians(20.);
	max_angle_disp_slope_ = numeric::conversions::radians(-1.);
	max_angle_disp_.clear();
	angle_disp_.clear();
	min_res_ = 2;
	max_res_ = 12;
	only_rot_ = 0.25;
	only_phipsi_ = 0.0;
	only_bb_ = 0.75;
	bb_min_ = false;
	sc_min_ = false;
	allow_pro_ = false;
	moving_residues_.clear();
	initial_bond_angles_.clear();
	running_steps_ = 0;
	running_bond_angle_means_.clear();
	running_bond_angle_variances_.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init_with_args
///
/// @brief
/// initialize the Backrub_controller with command-line arguments
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init_with_args()
{
	init();

	bool const fixdisulf = truefalseoption("norepack_disulf");

	numeric::conversions::to_degrees(max_bond_angle_);
	realafteroption("max_bond_angle", max_bond_angle_, max_bond_angle_);
	numeric::conversions::to_radians(max_bond_angle_);
	numeric::conversions::to_degrees(max_angle_disp_2_);
	realafteroption("max_angle_disp_2", max_angle_disp_2_, max_angle_disp_2_);
	numeric::conversions::to_radians(max_angle_disp_2_);
	numeric::conversions::to_degrees(max_angle_disp_3_);
	realafteroption("max_angle_disp_3", max_angle_disp_3_, max_angle_disp_3_);
	numeric::conversions::to_radians(max_angle_disp_3_);
	numeric::conversions::to_degrees(max_angle_disp_slope_);
	realafteroption("max_angle_disp_slope", max_angle_disp_slope_, max_angle_disp_slope_);
	numeric::conversions::to_radians(max_angle_disp_slope_);
	intafteroption("min_res", min_res_, min_res_);
	intafteroption("max_res", max_res_, max_res_);
	realafteroption("only_rot", only_rot_, only_rot_);
	realafteroption("only_phipsi", only_phipsi_, only_phipsi_);
	std::string phipsi_res(stringafteroption("phipsi_res", ""));
	realafteroption("only_bb", only_bb_, only_bb_);
	bb_min_ = truefalseoption("bb_min");
	sc_min_ = truefalseoption("sc_min");
	allow_pro_ = truefalseoption("backrub_proline");

	int const totres = pose_->total_residue();

	if (files_paths::resfile != "none") {
		pose_read_resfile(*pose_, files_paths::resfile);
	}

	// parse phipsi_res
	unsigned int index_start(0);
	unsigned int index_end(0);

	while (index_end < phipsi_res.length()) {
		// find the next numeric character
		index_start = phipsi_res.find_first_of("0123456789", index_end);
		// if no more numeric characters are found, exit parsing
		if (index_start == std::string::npos) break;
		// find the next non-numeric character
		index_end = phipsi_res.find_first_not_of("0123456789", index_start);
		// if we hit the end of the string, set the index_end to the string end
		if (index_end == std::string::npos) {
			index_end = phipsi_res.length();
		}
		// convert the identified substring into an integer
		std::istringstream intstream(phipsi_res.substr(index_start, index_end - index_start));
		int resnum;
		intstream >> resnum;
		if (resnum >= 1 && resnum <= pose_->total_residue()) {
			allowed_phipsi_moves_.push_back(resnum);
		}
	}

	for (int i = 1; i < totres; i++) {
		if (fixdisulf && pose_->res(i) == param_aa::aa_cys) {
			std::cout << "Fixing backbone/chi at " << param_aa::aa_name3(pose_->res(i))
			          << " " << i << std::endl;
			pose_->set_allow_bb_move(i, false);
			pose_->set_allow_chi_move(i, false);
		}
		/*
		std::cout << param_aa::aa_name3(pose_->res(i)) << " " << i << "\tbackbone: "
		          << (pose_->get_allow_bb_move(i) ? "free" : "fixed") << "\tchi: "
		          << (pose_->get_allow_chi_move(i) ? "free" : "fixed") << std::endl;
		*/
	}

	if (files_paths::resfile != "none") {
		rotcontrol_.init(pose_);

		if (runlevel_ns::runlevel >= runlevel_ns::inform) {
			rotcontrol_.print();
		}
	}

	init_move_vectors();
	init_move_allowed();
	init_allowed_moves();
	init_bond_angle_stat();

	std::cout << "Allowed Backrub Moves (<size>: <start res> ...)" << std::endl;
	for (unsigned int i = min_res_; i <= allowed_moves_.size()+1; i++) {
		std::cout << i << ":";
		for (unsigned int j = 0; j < allowed_moves_[i-2].size(); j++) {
			std::cout << " " << allowed_moves_[i-2][j];
		}
		std::cout << std::endl;
	}

	std::cout << "Allowed Phi/Psi Moves" << std::endl;
	for (unsigned int i = 1; i <= allowed_phipsi_moves_.size(); i++) {
		std::cout << allowed_phipsi_moves_[i] << " ";
	}
	std::cout << std::endl;

	// Code for tracking Davis "classic" backrub statistics
	std::string var_recorder_file(stringafteroption("var_recorder_file", ""));
	if (var_recorder_file.length()) { // && allowed_moves_.size() >= 2 && allowed_moves_[1].size() == 1) {
		var_recorder_.open(var_recorder_file);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init_with_args_no_resfile
///
/// @brief
/// initialize the Backrub_controller with command-line arguments but set
/// allow_bb_move in the code without resfile
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init_with_args_no_resfile()
{
	max_bond_angle_ = numeric::conversions::radians(2.);
	max_angle_disp_2_ = numeric::conversions::radians(40.);
	max_angle_disp_3_ = numeric::conversions::radians(20.);
	max_angle_disp_slope_ = numeric::conversions::radians(-1.);
	min_res_ = 2;
	max_res_ = 12;
	only_rot_ = 0.25;
	only_bb_ = 0.75;
	bb_min_ = false;
	sc_min_ = false;

	bool const fixdisulf = truefalseoption("norepack_disulf");

	numeric::conversions::to_degrees(max_bond_angle_);
	realafteroption("max_bond_angle", max_bond_angle_, max_bond_angle_);
	numeric::conversions::to_radians(max_bond_angle_);
	numeric::conversions::to_degrees(max_angle_disp_2_);
	realafteroption("max_angle_disp_2", max_angle_disp_2_, max_angle_disp_2_);
	numeric::conversions::to_radians(max_angle_disp_2_);
	numeric::conversions::to_degrees(max_angle_disp_3_);
	realafteroption("max_angle_disp_3", max_angle_disp_3_, max_angle_disp_3_);
	numeric::conversions::to_radians(max_angle_disp_3_);
	numeric::conversions::to_degrees(max_angle_disp_slope_);
	realafteroption("max_angle_disp_slope", max_angle_disp_slope_, max_angle_disp_slope_);
	numeric::conversions::to_radians(max_angle_disp_slope_);
	intafteroption("min_res", min_res_, min_res_);
	intafteroption("max_res", max_res_, max_res_);
	realafteroption("only_rot", only_rot_, only_rot_);
	realafteroption("only_bb", only_bb_, only_bb_);
	bb_min_ = truefalseoption("bb_min");
	sc_min_ = truefalseoption("sc_min");

	int const totres = pose_->total_residue();

	for (int i = 1; i < totres; i++) {
		if (pose_->res(i) == param_aa::aa_pro ||
		    (fixdisulf && pose_->res(i) == param_aa::aa_cys)) {
			std::cout << "Fixing backbone/chi at " << param_aa::aa_name3(pose_->res(i))
			          << " " << i << std::endl;
			pose_->set_allow_bb_move(i, false);
			pose_->set_allow_chi_move(i, false);
		}
		/*
		std::cout << param_aa::aa_name3(pose_->res(i)) << " " << i << "\tbackbone: "
		          << (pose_->get_allow_bb_move(i) ? "free" : "fixed") << "\tchi: "
		          << (pose_->get_allow_chi_move(i) ? "free" : "fixed") << std::endl;
		*/
	}
/*
	if (files_paths::resfile != "none") {
		pose_read_resfile(*pose_, files_paths::resfile);
		*/
		rotcontrol_.init(pose_);
		// Keep Monte Carlo from wasting trials on non-moves
		// Also, don't change proline side chains
		for (int i = 1; i < totres; i++) {
			if (rotcontrol_.nrotpos(i) <= 1 || pose_->res(i) == param_aa::aa_pro) {
				pose_->set_allow_chi_move(i, false);
			}
		}
		if (runlevel_ns::runlevel >= runlevel_ns::inform) {
			rotcontrol_.print();
		}
		/*
	}
*/
	init_move_vectors();
	init_move_allowed();
	init_allowed_moves();

	std::cout << "Allowed Backrub Moves (<size>: <start res> ...)" << std::endl;
	for (unsigned int i = min_res_; i <= allowed_moves_.size()+1; i++) {
		std::cout << i << ":";
		for (unsigned int j = 0; j < allowed_moves_[i-2].size(); j++) {
			std::cout << " " << allowed_moves_[i-2][j];
		}
		std::cout << std::endl;
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init_move_vectors
///
/// @brief
/// allocate and initialize diagonal matricies to control/track move pairs
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init_move_vectors()
{
	int const totres = pose_->total_residue();

	move_allowed_.resize(totres);
	attempted_moves_.resize(totres);
	accepted_moves_.resize(totres);
	max_angle_disp_.resize(totres);
	angle_disp_.resize(totres);

	for (int i = 1; i <= totres; i++) {

		move_allowed_[i].resize(max_res_-1, false);
		attempted_moves_[i].resize(max_res_-1);
		accepted_moves_[i].resize(max_res_-1);
		// This doesn't have to be allocated right away
		//max_angle_disp_[i].resize(max_res_-1, max_angle_disp_default_);
		angle_disp_[i].resize(max_res_-1, 0);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init_move_allowed
///
/// @brief
/// initialize lower diagonal matrix of allowed moves
///
/// @detailed
/// This function sets the move_allowed_ matrix to true for moves of size
/// min_res_ to max_res_. In order to be set to true, the starting and ending
/// residues, as well as all intervening residues, must have allow_bb_move set
/// to true in the Pose object. In addition, there can't be any chain breaks
/// along the segment. Lastly, proline residues are not allowed as pivot points.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init_move_allowed()
{
	int const totres = pose_->total_residue();

	utility::vector1<bool> const chain_breaks(pose_chain_breaks(*pose_));

	for (int nres = min_res_; nres <= max_res_; nres++) {
		for (int i = 1; i <= totres-nres+1; i++) {
			bool allowed = true;
//			std::cout << "i " << i << std::endl;
//			std::cout << "pose allow? " << pose_->get_allow_bb_move(i) << std::endl;
			for (int j = i; j <= i + nres - 1; j++) {
//				std::cout << "j " << j << std::endl;
				if (!pose_->get_allow_bb_move(j)) {
//				 std::cout << "disallow j 1" << j  << std::endl;
					allowed = false;
					break;
				}
				if (j < i + nres - 1 && chain_breaks[j]) {
//				 std::cout << "disallow j 2" << j  << std::endl;
					allowed = false;
					break;
				}
			}
			if (!allow_pro_) {
				DesignMap & dm = rotcontrol_.packer_task().get_designmap();
				if (dm.repack_residue(i)) {
					if (dm.get(i, param_aa::aa_pro)) allowed = false;
				} else {
					if (pose_->res(i) == param_aa::aa_pro) allowed = false;
				}
				if (dm.repack_residue(i + nres - 1)) {
					if (dm.get(i + nres - 1, param_aa::aa_pro)) allowed = false;
				} else {
					if (pose_->res(i + nres - 1) == param_aa::aa_pro) allowed = false;
				}
			}

			move_allowed_[i][nres-1] = allowed;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::init_allowed_moves
///
/// @brief
/// initialize list of allowed starting residue positions for backrub moves
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 24, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::init_allowed_moves()
{
	// Bail out if we've already initialized the array
	if (allowed_moves_.size())
		return;

	int const totres = pose_->total_residue();

	allowed_moves_.resize(max_res_-1);

	for (int nres = min_res_; nres <= max_res_; nres++) {
		for (int i = 1; i <= totres-nres+1; i++) {
			if (move_allowed(i, i+nres-1)) {
				allowed_moves_[nres-2].push_back(i);
			}
		}
	}
}

/// @brief initialize data structures for tracking bond angles
/// @authors Colin A. Smith
void
Backrub_controller::init_bond_angle_stat()
{
	using numeric::conversions::radians;

	std::set<int> moving_residue_set;

	for (size_t i = 1; i <= move_allowed_.size(); ++i) {
		for (size_t j = 1; j <= move_allowed_[i].size(); ++j) {
			if (move_allowed_[i][j]) {
				moving_residue_set.insert(i);
				moving_residue_set.insert(i+j);
			}
		}
	}

	int i = 0;
	moving_residues_.clear();
	moving_residues_.resize(moving_residue_set.size());
	for (std::set<int>::iterator iter(moving_residue_set.begin());
	     iter != moving_residue_set.end(); ++iter) {

		moving_residues_[++i] = *(iter);
	}

	initial_bond_angles_.clear();
	initial_bond_angles_.resize(pose_->total_residue());
	running_steps_ = 0;
	running_bond_angle_means_.clear();
	running_bond_angle_means_.resize(pose_->total_residue(), 0);
	running_bond_angle_variances_.clear();
	running_bond_angle_variances_.resize(pose_->total_residue(), 0);

	FArray3D_float const & full_coord(pose_->full_coord());

	for (int i = 1; i <= pose_->total_residue(); ++i) {
		initial_bond_angles_[i] = radians(vec_angle(full_coord(1,1,i), full_coord(1,2,i), full_coord(1,3,i)));
	}
}

/// @brief record statistics for bond angles
/// @authors Colin A. Smith
void
Backrub_controller::record_bond_angle_stat(
	pose_ns::Pose const & pose
)
{
	using numeric::conversions::radians;

	FArray3D_float const & full_coord(pose.full_coord());

	running_steps_++;

	for (size_t i = 1; i <= moving_residues_.size(); ++i) {
		int const seqpos(moving_residues_[i]);
		float const bond_angle(radians(vec_angle(full_coord(1,1,seqpos), full_coord(1,2,seqpos),
																					   full_coord(1,3,seqpos))));
		float const ideal_bond_angle(bond_angle::Theta0_bb[pose.res(seqpos)]);
		float const bond_angle_diff(bond_angle - ideal_bond_angle);
		// The following follows:
		// http://en.wikipedia.org/w/index.php?title=Algorithms_for_calculating_variance&oldid=144077632
		double const delta(bond_angle_diff - running_bond_angle_means_[seqpos]);
		running_bond_angle_means_[seqpos] += delta/running_steps_;
		running_bond_angle_variances_[seqpos] += delta*(bond_angle_diff - running_bond_angle_means_[seqpos]);
		//using numeric::conversions::degrees;
		//std::cout << seqpos << " " << degrees(bond_angle_diff) << " " << degrees(running_bond_angle_means_[seqpos]) << " "
		//					<< degrees(running_bond_angle_variances_[seqpos]) << std::endl;
	}
}

/// @brief print statistics for bond angles
/// @authors Colin A. Smith
void
Backrub_controller::print_bond_angle_stat(
	pose_ns::Pose const & pose,
	pose_ns::Monte_carlo const & mc,
	std::ostream & out
)
{
	using numeric::conversions::degrees;
	using numeric::conversions::radians;
	using std::setw;

	FArray3D_float const & full_coord(pose.full_coord());
	FArray3D_float const & low_full_coord(mc.low_pose().full_coord());
	float const temperature(mc.temperature());
	float const bond_angle_weight(mc.weight_map().get_weight(pose_ns::BOND_ANGLE));

	float mean_initial_bond_angle(0);
	float mean_last_bond_angle(0);
	float mean_low_bond_angle(0);
	float mean_mean_bond_angle(0);

	float mean_initial_bond_angle_diff(0);
	float mean_last_bond_angle_diff(0);
	float mean_low_bond_angle_diff(0);
	float mean_mean_bond_angle_diff(0);

	float mean_sd(0);

	float rms_initial_bond_angle_diff(0);
	float rms_last_bond_angle_diff(0);
	float rms_low_bond_angle_diff(0);
	float rms_mean_bond_angle_diff(0);

	out << "N-CA-C Bond Angle Statistics:" << std::endl;
	out << "Res    Ideal Initial-Ideal    Last-Ideal     Low-Ideal    Mean-Ideal    SD Th SD" << std::endl;

	for (size_t i = 1; i <= moving_residues_.size(); ++i) {
		int const seqpos(moving_residues_[i]);
		float const bond_angle(radians(vec_angle(full_coord(1,1,seqpos), full_coord(1,2,seqpos),
																					   full_coord(1,3,seqpos))));
		float const low_bond_angle(radians(vec_angle(low_full_coord(1,1,seqpos), low_full_coord(1,2,seqpos),
																					       low_full_coord(1,3,seqpos))));
		float const ideal_bond_angle(bond_angle::Theta0_bb[pose.res(seqpos)]);
		float const Ktheta(bond_angle::Ktheta_bb[pose.res(seqpos)]);
		float const sd(std::sqrt(running_bond_angle_variances_[seqpos]/(running_steps_ - 1)));
		float const theoretical_sd(std::sqrt(temperature/(2*bond_angle_weight*Ktheta)));

		out << param_aa::aa_name1(pose.res(seqpos)) << setw(3) << std::left << seqpos << " "
		    << std::right << std::fixed << std::setprecision(2) << std::showpoint
	      << setw(7) << degrees(ideal_bond_angle) << " "
		    << setw(7) << degrees(initial_bond_angles_[seqpos]) << " "
		    << setw(5) << degrees(initial_bond_angles_[seqpos] - ideal_bond_angle) << " "
		    << setw(7) << degrees(bond_angle) << " "
		    << setw(5) << degrees(bond_angle - ideal_bond_angle) << " "
		    << setw(7) << degrees(low_bond_angle) << " "
		    << setw(5) << degrees(low_bond_angle - ideal_bond_angle) << " "
		    << setw(7) << degrees(running_bond_angle_means_[seqpos] + ideal_bond_angle) << " "
		    << setw(5) << degrees(running_bond_angle_means_[seqpos]) << " "
		    << setw(5) << degrees(sd) << " "
		    << setw(5) << degrees(theoretical_sd) << std::endl;

		mean_initial_bond_angle += initial_bond_angles_[seqpos];
		mean_last_bond_angle += bond_angle;
		mean_low_bond_angle += low_bond_angle;
		mean_mean_bond_angle += running_bond_angle_means_[seqpos] + ideal_bond_angle;

		mean_initial_bond_angle_diff += initial_bond_angles_[seqpos] - ideal_bond_angle;
		mean_last_bond_angle_diff += bond_angle - ideal_bond_angle;
		mean_low_bond_angle_diff += low_bond_angle - ideal_bond_angle;
		mean_mean_bond_angle_diff += running_bond_angle_means_[seqpos];

		mean_sd += sd;

		rms_initial_bond_angle_diff += std::pow(initial_bond_angles_[seqpos] - ideal_bond_angle, 2);
		rms_last_bond_angle_diff += std::pow(bond_angle - ideal_bond_angle, 2);
		rms_low_bond_angle_diff += std::pow(low_bond_angle - ideal_bond_angle, 2);
		rms_mean_bond_angle_diff += std::pow(running_bond_angle_means_[seqpos], 2);
	}

	out << "Mean "
	    << std::fixed << std::setprecision(2) << std::showpoint
	    << "        "
		  << setw(7) << degrees(mean_initial_bond_angle/moving_residues_.size()) << " "
		  << setw(5) << degrees(mean_initial_bond_angle_diff/moving_residues_.size()) << " "
		  << setw(7) << degrees(mean_last_bond_angle/moving_residues_.size()) << " "
		  << setw(5) << degrees(mean_last_bond_angle_diff/moving_residues_.size()) << " "
		  << setw(7) << degrees(mean_low_bond_angle/moving_residues_.size()) << " "
		  << setw(5) << degrees(mean_low_bond_angle_diff/moving_residues_.size()) << " "
		  << setw(7) << degrees(mean_mean_bond_angle/moving_residues_.size()) << " "
		  << setw(5) << degrees(mean_mean_bond_angle_diff/moving_residues_.size()) << " "
			<< setw(5) << degrees(mean_sd/moving_residues_.size()) << std::endl;

	out << "RMS  "
	    << std::fixed << std::setprecision(2) << std::showpoint
	    << "        "
		  << "        "
		  << setw(5) << degrees(std::sqrt(rms_initial_bond_angle_diff/moving_residues_.size())) << " "
		  << "        "
		  << setw(5) << degrees(std::sqrt(rms_last_bond_angle_diff/moving_residues_.size())) << " "
		  << "        "
		  << setw(5) << degrees(std::sqrt(rms_low_bond_angle_diff/moving_residues_.size())) << " "
		  << "        "
		  << setw(5) << degrees(std::sqrt(rms_mean_bond_angle_diff/moving_residues_.size())) << std::endl;
}

/// @brief initialize histograms for recording move acceptance statistics
void
Backrub_controller::init_histograms(
	int const size // = 10
)
{
	attempted_moves_bond_angle_.init(0, max_bond_angle_, size);
	accepted_moves_bond_angle_.init(0, max_bond_angle_, size);

	// Figure out how large the per-pair histograms have to be
	float max_max_angle_disp = 0;
	for (size_t i = 0; i < allowed_moves_.size(); i++) {
		for (size_t j = 0; j < allowed_moves_[i].size(); j++) {
			int const start = allowed_moves_[i][j];
			int const end = start + i + 1;
			if (max_angle_disp(start, end) > max_max_angle_disp) {
				max_max_angle_disp = max_angle_disp(start, end);
			}
		}
	}

	// Initialize the per-pair histograms
	for (size_t i = 0; i < allowed_moves_.size(); i++) {
		for (size_t j = 0; j < allowed_moves_[i].size(); j++) {
			int const start = allowed_moves_[i][j];
			int const end = start + i + 1;
			attempted_moves_[start][end-start].init(0, max_max_angle_disp, size);
			accepted_moves_[start][end-start].init(0, max_max_angle_disp, size);
		}
	}
}

void
print_histograms_header(
	histogram<float> const & hist,
	std::ostream & out
)
{
	int const width(7);
	int const precision(1);

	utility::vector0<float> breaks(hist.breaks());

	out << "   Degrees";
	for (size_t i = 1; i < breaks.size(); i++) {
		out << std::setw(width) << std::setprecision(precision) << std::fixed
		    << numeric::conversions::degrees(breaks[i]);
	}
	out << std::endl;
}


/// @brief print a two line summary of attempt/accept statistics
void
print_attempted_accepted_histograms(
	histogram<float> const & attempted,
	histogram<float> const & accepted,
	int num,
	std::ostream & out
)
{
	int const width(7);
	int const precision(3);

	if (num >= 0) {
		out << std::setw(3) << std::left << num << std::right << " Trials";
	} else {
		out << "    Trials";
	}
	for (size_t i = 0; i < attempted.size(); i++) {
		if (attempted.counts(i)) {
			out << std::setw(width) << attempted.counts(i);
		} else {
			out << "       ";
		}
	}
	out << std::endl << "Acc. Ratio";
	for (size_t i = 0; i < attempted.size(); i++) {
		long double const ratio(attempted.counts(i) ? static_cast<long double>(accepted.counts(i)) /
		                                              static_cast<long double>(attempted.counts(i))
		                                            : 0);
		if (attempted.counts(i)) {
			out << std::setw(width) << std::setprecision(precision) << std::fixed << ratio;
		} else {
			out << "       ";
		}
	}
	out << std::endl;
}

/// @brief print a summary move acceptance statistics histograms
void
Backrub_controller::print_histograms(std::ostream & out)
{
	out << "Trial Statistics (N-CA-C Bond Angle Deviation from Ideal):" << std::endl;
	print_histograms_header(attempted_moves_bond_angle_, out);
	print_attempted_accepted_histograms(attempted_moves_bond_angle_, accepted_moves_bond_angle_, -1,
	                                    out);

	out << std::endl << "Trial Statistics (Angular Displacement):" << std::endl;
	bool header = true;
	for (size_t i = 0; i < allowed_moves_.size(); i++) {
		if (allowed_moves_[i].size() == 0) continue;
		histogram<float> attempted_hist;
		histogram<float> accepted_hist;
		for (size_t j = 0; j < allowed_moves_[i].size(); j++) {
			int const start = allowed_moves_[i][j];
			int const end = start + i + 1;
			attempted_hist += attempted_moves_[start][end-start];
			accepted_hist += accepted_moves_[start][end-start];
		}
		if (header) {
			print_histograms_header(attempted_hist, out);
			header = false;
		}
		print_attempted_accepted_histograms(attempted_hist, accepted_hist, i+2, out);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::mc_loop
///
/// @brief
/// run a given number of Backrub trials
///
/// @detailed
///
/// @param[in,out] mc - Monte_carlo object to control/record move acceptance
/// @param[in] ntrials - number of Monte Carlo trials
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::mc_loop(
	pose_ns::Monte_carlo & mc,
	int ntrials
)
{
	for (int i = 0; i < ntrials; i++) {
		trial(*pose_, mc);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::mc_loop
///
/// @brief
/// run a given number of Backrub trials
///
/// @detailed
///
/// @param[in,out] mc - Monte_carlo object to control/record move acceptance
/// @param[in] ntrials - number of Monte Carlo trials
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
Backrub_controller::mc_loop(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	int ntrials
)
{
	for (int i = 0; i < ntrials; i++) {
		trial(pose, mc);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::trial
///
/// @brief
/// perform backrub Monte Carlo move
///
/// @detailed
///
/// @param[in,out] mc - Monte_carlo object to control/record move acceptance
///
/// @global_read
///
/// @global_write
///
/// @return
/// boolean indicating whether the move was accepted or not
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
bool
Backrub_controller::trial(
	pose_ns::Monte_carlo & mc
)
{
	return trial(*pose_, mc);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_controller::trial
///
/// @brief
/// perform backrub Monte Carlo move
///
/// @detailed
///
/// @param[in,out] pose - Pose object to backrub
/// @param[in,out] mc - Monte_carlo object to control/record move acceptance
///
/// @global_read
///
/// @global_write
///
/// @return
/// boolean indicating whether the move was accepted or not
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
bool
Backrub_controller::trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc
)
{
	using namespace param_aa;
	using namespace pose_ns;
	using namespace std;
	using numeric::conversions::degrees;

	int nres = 0;
	float angle = 0;
	float angle_phipsi = 0, angle_type = 0;
	utility::vector0<int> rot_pos;
	utility::vector0<int> rot_num;
	float prev_score = pose.get_0D_score(SCORE);
	float angle_score1 = 0;
	float angle_score2 = 0;
	float max_bond_angle_attempted = 0;
	float bb_min_score = 0;
	float sc_min_score = 0;
	int bb_min_func = 0;
	int sc_min_func = 0;
	bool accepted( true );
	bool gfrag( true );
	FArray3D_float full_coord = pose.full_coord();
	int start = 0, end = 0, seg_size = 0;
	utility::vector1<int> res_init(pose.total_residue());

	for (int i = 1; i <= pose.total_residue(); ++i) {
		res_init[i] = pose.res(i);
	}

	if (rotcontrol_.nrotamers() != 0 && ran3() < only_rot_) {
		// Only do a single rotamer swap
		rot_pos = pose_pick_free_sc(pose, rotcontrol_.nrotpos(), 1, pose.total_residue(), 1, 0);
		rot_num.resize(rot_pos.size());
		if (rotcontrol_.nrotpos(rot_pos[0])) {
			rot_num[0] = rotcontrol_.random_rotnum(rot_pos[0]);
			rotcontrol_.set_rotamer(pose, rot_pos[0], rot_num[0]);
		} else {
			std::cout << "Warning: No rotamers at position " << rot_pos[0] << std::endl;
		}
	} else if (ran3() < only_phipsi_) {
		start = allowed_phipsi_moves_[random_range(1, allowed_phipsi_moves_.size())];

		angle_type = random_range(1, 2);
		angle_phipsi = (ran3() - 0.5)*40;
		float old_angle;

		if (angle_type == 1) {
			old_angle = pose.phi(start);
		  pose.set_phi(start, old_angle + angle_phipsi);
		} else {
			old_angle = pose.psi(start);
			pose.set_psi(start, old_angle + angle_phipsi);
		}
		//std::cout << "Changing angle " << angle_type << " of residue " << start
		//          << " by " << angle_phipsi << std::endl;
	} else {
		init_allowed_moves();
		// Find a random size of the backrub move with an allowed start
		do {
			nres = random_range(min_res_, allowed_moves_.size()+1);
		} while (allowed_moves_[nres-2].size() == 0);

		start = allowed_moves_[nres-2][random_range(0, allowed_moves_[nres-2].size() - 1)];
		end = start + nres - 1;
		seg_size = end - start + 1;

		// Set up the object for bond angle energy evaluation
		backrub_ns::Backrub_eval energyeval(&pose);
		FArray1D_float angles(1, 0.0);
		energyeval.push_back(start, end);
		angle_score1 = energyeval.func(angles, gfrag);

		// Otherwise, do the backrub backbone move
		angle = backrub_select_angle(pose, start, end, -max_bond_angle_, max_bond_angle_, true,
		                             max_angle_disp(start, end), false);
		backrub_rot_full_coord(full_coord, pose.res(), pose.res_variant(), start,
		                       end, angle);
		pose.set_segment_full_coord(seg_size, start, full_coord(1,1,start));

		// Determine the maximum bond angle after the move
		max_bond_angle_attempted = std::abs(bond_angle::Theta0_bb[pose.res(start)] -
		                                    angle_of(pose.full_coord(1,start), pose.full_coord(2,start),
		                                             pose.full_coord(3,start)));
		float const temp_angle = std::abs(bond_angle::Theta0_bb[pose.res(end)] -
		                                  angle_of(pose.full_coord(1,end), pose.full_coord(2,end),
		                                           pose.full_coord(3,end)));
		if (temp_angle > max_bond_angle_attempted) max_bond_angle_attempted = temp_angle;

		angle_score2 = energyeval.func(angles, gfrag);
	}

	// Bond angle minimization using peptide bonds on either side of pivot points (up to 4)
	backrub_ns::Backrub_eval mineval(&pose);
	if (bb_min_ && angle != 0 && end - start > 1) {
		if (start > 1 && move_allowed(start-1, start)) mineval.push_back(start-1, start);
		if (move_allowed(start, start+1)) mineval.push_back(start, start+1);
		if (move_allowed(end-1, end)) mineval.push_back(end-1, end);
		if (end < pose_->total_residue() && move_allowed(end, end+1)) mineval.push_back(end, end+1);
	}
	FArray1D_float minangles(mineval.residue_pairs().size()/2, 0.0);
	if (mineval.residue_pairs().size()) {
		bb_min_score = -mineval.func(minangles, gfrag);
		mineval.minimize(minangles, "dfpmin");
		bb_min_func = get_func_evals();
		bb_min_score += mineval.func(minangles, gfrag);
		for (size_t i = 1; i <= minangles.size(); i++) {
			backrub_rot_full_coord(full_coord, pose.res(), pose.res_variant(),
			                       mineval.residue_pairs()[(i-1)*2], mineval.residue_pairs()[(i-1)*2+1],
		                         minangles(i));
		}
		int const minimize_start = mineval.residues().front();
		int const minimize_size = mineval.residues().back() - minimize_start + 1;
		pose.set_segment_full_coord(minimize_size, minimize_start, full_coord(1,1,minimize_start));
	}

	// Do simultaneous rotamer sampling
	if (rotcontrol_.nrotamers() != 0 && angle != 0) {
		rot_pos = pose_pick_free_sc(pose, rotcontrol_.nrotpos(), start, end, seg_size, only_bb_);
		rot_num.resize(rot_pos.size());

		for (unsigned int i = 0; i < rot_pos.size(); i++) {
			if (rotcontrol_.nrotpos(rot_pos[i])) {
				rot_num[i] = rotcontrol_.random_rotnum(rot_pos[i]);
				rotcontrol_.set_rotamer(pose, rot_pos[i], rot_num[i]);
			} else {
				std::cout << "Warning: No rotamers at position " << rot_pos[0] << std::endl;
			}
		}
	}

	// Minimize side chain angles
	if (sc_min_) {
		if (runlevel_ns::runlevel >= runlevel_ns::inform) sc_min_score = -pose.score(mc.weight_map());
		int const old_runlevel = runlevel_ns::runlevel;
		if (runlevel_ns::runlevel == runlevel_ns::inform) {
			runlevel_ns::runlevel = runlevel_ns::standard;
		}
		minimize_reset();
		minimize_set_vary_phipsi(false);
		minimize_set_vary_omega(false);
		minimize_set_vary_chi(true);
		minimize_set_tolerance(0.05);
		//pose_set_use_nblist(true);
		pose.main_minimize(mc.weight_map(), "dfpmin");
		sc_min_func = get_func_evals();
		minimize_reset();
		runlevel_ns::runlevel = old_runlevel;
		sc_min_score += pose.get_0D_score(SCORE);
	}

	std::stringstream move_type;
	if (angle == 0 && rot_pos.size()) {
		move_type << "backrub_rot";
	} else if (angle_phipsi != 0) {
		move_type << "backrub_phipsi";
	} else {
		int const size_digits = (int)std::floor(std::log10((float)max_res_)) + 1;
		move_type << "backrub_" << std::setw(size_digits) << setfill('0') << seg_size << "_"
		          << rot_pos.size();
		if (angle != 0 && attempted_moves_bond_angle_.size()) {
			attempted_moves_bond_angle_.record(max_bond_angle_attempted);
		}
		if (angle != 0 && attempted_moves_[start][end-start].size()) {
			attempted_moves_[start][end-start].record(std::abs(angle));
		}
	}
	accepted = mc.boltzmann(pose, move_type.str(), false);
	record_bond_angle_stat(pose);

	// gfriedla: write frames to the trajectory whether they're accepted or rejected for proper boltzmann stats
	movie_.write_frame(pose);

	if (accepted) {
		// Update total angular displacement tracking
		if (angle != 0) {
			angle_disp_[start][end-start] += angle;
			for (size_t i = 1; i <= minangles.size(); i++) {
				int const min_start = mineval.residue_pairs()[(i-1)*2];
				int const min_end = mineval.residue_pairs()[(i-1)*2+1];
				angle_disp_[min_start][min_end-min_start] = minangles(i);
			}
			if (accepted_moves_bond_angle_.size()) {
				accepted_moves_bond_angle_.record(max_bond_angle_attempted);
			}
			if (accepted_moves_[start][end-start].size()) {
				accepted_moves_[start][end-start].record(std::abs(angle));
			}
		}
	}

	// Code for tracking Davis "classic" backrub statistics
	if (var_recorder_.good() && allowed_moves_.size() >= 2 && allowed_moves_[1].size() >= 1) {
		int const backrub_start(allowed_moves_[1][0]);
		int const backrub_center(backrub_start + 1);
		int const nchi(aaproperties_pack::nchi(pose.res(backrub_center), pose.res_variant(backrub_center)));
		var_recorder_ << pose.get_0D_score(SCORE) << "\t" << degrees(angle_disp_[backrub_start][2]);
		for (int i = 1; i <= nchi; i++) {
			var_recorder_ << "\t" << pose.chi(i,backrub_center);
		}
		var_recorder_ << std::endl;
	}

	// Bail out early if we don't want to write a line for the output
	if (runlevel_ns::runlevel < runlevel_ns::inform) {
		return accepted;
	}

	// Metropolis/Total Energy Info
	std::cout << fixed << setprecision(1) << showpoint
	          << (accepted ? "A" : "R") << " "
	          << setprecision(2) << setw(9) << prev_score << " "
	          << setw(9) << mc.last_score() << " ";
	// Backrub Move Info
	if (angle != 0) {
		std::cout << setw(3) << start << " " << aa_name3(pose.res(start)) << " "
		          << setw(3) << end << " " << aa_name3(pose.res(end)) << " "
		          << setw(6) << degrees(angle) << " "
		          << setw(6) << degrees(angle_disp_[start][end-start]) << " "
		          << setw(5) << angle_score1 << " "
		          << setw(5) << angle_score2;
	} else if (angle_phipsi != 0) {
	  std::cout << setw(3) << start << " " << aa_name3(pose.res(start)) << " "
		          << "    " << (angle_type == 1 ? "phi" : "psi") << " "
							<< setw(6) << angle_phipsi << "                   ";
	} else {
		std::cout << "                                         ";
	}
	// Backbone Minimization Info
	if (bb_min_) {
		if (bb_min_func > 0) {
			std::cout << " " << setw(5) << bb_min_score << " "
			          << setw(3) << bb_min_func;
		} else {
			std::cout << "          ";
		}
	}
	// Side chain Minimization Info
	if (sc_min_) {
		if (sc_min_func > 0) {
			std::cout << " " << setw(8) << sc_min_score << " "
			          << setw(3) << sc_min_func;
		} else {
			std::cout << "          ";
		}
	}
	// Rotamer Info
	for (size_t i = 0; i < rot_pos.size(); i++) {
		std::cout << " " << aa_name1(res_init[rot_pos[i]]) << rot_pos[i];
		if (res_init[rot_pos[i]] != rotcontrol_.res(rot_pos[i], rot_num[i])) {
			std::cout << aa_name1(rotcontrol_.res(rot_pos[i], rot_num[i]));
		}
	}
	std::cout << std::endl;

	return accepted;
}

void
Backrub_controller::trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int nmoves,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	for (int i = 0; i < nmoves; i++) {

		FArray3D_float full_coord = pose.full_coord();
		int nres;

		init_allowed_moves();
		// Find a random size of the backrub move with an allowed start
		do {
			nres = random_range(min_res_, allowed_moves_.size()+1);
		} while (allowed_moves_[nres-2].size() == 0);

		int const start = allowed_moves_[nres-2][random_range(0, allowed_moves_[nres-2].size() - 1)];
		int const end = start + nres - 1;
		int const seg_size = end - start + 1;

		// Actual backrub move
		float angle = backrub_select_angle(pose, start, end, -max_bond_angle_, max_bond_angle_, true,
																			 max_angle_disp(start, end), false);
		backrub_rot_full_coord(full_coord, pose.res(), pose.res_variant(), start,
													 end, angle);
		pose.set_segment_full_coord(seg_size, start, full_coord(1,1,start));

		// Bond angle minimization using peptide bonds on either side of pivot points (up to 4)
		backrub_ns::Backrub_eval mineval(&pose);
		if (bb_min_ && angle != 0 && end - start > 1) {
			if (start > 1 && move_allowed(start-1, start)) mineval.push_back(start-1, start);
			if (move_allowed(start, start+1)) mineval.push_back(start, start+1);
			if (move_allowed(end-1, end)) mineval.push_back(end-1, end);
			if (end < pose_->total_residue() && move_allowed(end, end+1)) mineval.push_back(end, end+1);
		}
		FArray1D_float minangles(mineval.residue_pairs().size()/2, 0.0);
		if (mineval.residue_pairs().size()) {
			mineval.minimize(minangles, "dfpmin");
			for (size_t i = 1; i <= minangles.size(); i++) {
				backrub_rot_full_coord(full_coord, pose.res(), pose.res_variant(),
															 mineval.residue_pairs()[(i-1)*2], mineval.residue_pairs()[(i-1)*2+1],
															 minangles(i));
			}
			int const minimize_start = mineval.residues().front();
			int const minimize_size = mineval.residues().back() - minimize_start + 1;
			pose.set_segment_full_coord(minimize_size, minimize_start, full_coord(1,1,minimize_start));
		}
	}

	// Full minimize with rotamer trials
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, pose.total_residue(),
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "backrub"+tag_suffix );
}

} // end pose_ns namespace
