// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20497 $
//  $Date: 2008-02-20 18:17:57 -0500 (Wed, 20 Feb 2008) $
//  $Author: csmith $

#ifndef INCLUDED_pose_backrub_controller
#define INCLUDED_pose_backrub_controller

// Rosetta Headers
#include "histogram.h"
#include "pose_backrub.h"
#include "pose_movie.h"
#include "pose_rotamer_controller.h"
#include "pose.h"

// Utility Headers
#include <utility/io/ozstream.fwd.hh>
#include <utility/vector0.hh>
#include <utility/vector1.hh>

namespace pose_ns {

class Backrub_controller {

private:

	pose_ns::Pose * pose_;
	pose_ns::Rotamer_controller rotcontrol_;
	pose_ns::Movie movie_;
	utility::vector1<utility::vector1<bool> > move_allowed_;
	utility::vector0<utility::vector0<int> > allowed_moves_;
	utility::vector1<int> allowed_phipsi_moves_;
	utility::vector1<utility::vector1<histogram<float> > > attempted_moves_;
	utility::vector1<utility::vector1<histogram<float> > > accepted_moves_;
	histogram<float> attempted_moves_bond_angle_;
	histogram<float> accepted_moves_bond_angle_;
	float max_bond_angle_;
	float max_angle_disp_2_;
	float max_angle_disp_3_;
	float max_angle_disp_slope_;
	utility::vector1<utility::vector1<float> > max_angle_disp_;
	utility::vector1<utility::vector1<float> > angle_disp_;
	int min_res_;
	int max_res_;
	float only_rot_;
	float only_phipsi_;
	float only_bb_;
	bool bb_min_;
	bool sc_min_;
	bool allow_pro_;
	utility::vector1<int> moving_residues_;
	utility::vector1<float> initial_bond_angles_;
	int running_steps_;
	utility::vector1<double> running_bond_angle_means_;
	utility::vector1<double> running_bond_angle_variances_;
	utility::io::ozstream var_recorder_;

	void
	init_move_vectors();

	void
	init_move_allowed();

	void
	init_allowed_moves();
	
	void
	init_bond_angle_stat();
	
	void
	record_bond_angle_stat(
		pose_ns::Pose const & pose
	);

public:

	void
	set_pose(
		pose_ns::Pose * pose
	);

	void
	init();

	void
	init_with_args();

	void
	init_with_args_no_resfile();
	void
	init_histograms(
		int const size = 10
	);

	void
	print_bond_angle_stat(
		pose_ns::Pose const & pose,
		pose_ns::Monte_carlo const & mc,
		std::ostream & out
	);
	
	void
	print_histograms(std::ostream & out);

	/// @breif get rotamer controller object for this object
	inline
	pose_ns::Rotamer_controller &
	rotcontrol()
	{
		return rotcontrol_;
	}
	
	/// @breif get the movie object for this object
	inline
	pose_ns::Movie &
	movie()
	{
		return movie_;
	}

	/// @brief get whether a backrub move is allowed for two residues
	inline
	bool
	move_allowed(
		int start,
		int end
	)
	{
		assert(start < end && start > 0 && end <= pose_->total_residue());

		if (end - start + 1 < min_res_ || end - start + 1 > max_res_) return false;

		if ((signed)move_allowed_[start].size() < end - start) return false;

		return move_allowed_[start][end-start];
	}

	/// @brief set whether a backrub move is allowed for two particular residues
	inline
	void
	set_move_allowed(
		int start,
		int end,
		bool value
	)
	{
		assert(start < end && start > 0 && end <= pose_->total_residue());

		if ((signed)move_allowed_[start].size() < end - start) {

			move_allowed_[start].resize(end - start, false);
		}

		if (value == true) {
			if (end - start  + 1 < max_res_) min_res_ = end - start + 1;
			if (end - start  + 1 > max_res_) max_res_ = end - start + 1;
		}

		move_allowed_[start][end - start] = value;

		// Invalidate the allowed moves vector
		allowed_moves_.clear();
	}

	/// @brief get the maximum allowed deviation from bond angle ideality
	inline
	bool
	max_bond_angle()
	{
		return max_bond_angle_;
	}

	/// @brief set the maximum allowed deviation from bond angle ideality
	inline
	void
	set_max_bond_angle(
		float value
	)
	{
		assert(value >= 0);

		max_bond_angle_ = value;
	}

	/// @brief get the default maximum angular displacement for 2 residue backrub moves
	inline
	bool
	max_angle_disp_2()
	{
		return max_angle_disp_2_;
	}

	/// @brief set the default maximum angular displacement for 2 residue backrub moves
	inline
	void
	set_max_angle_disp_2(
		float value
	)
	{
		assert(value >= 0);

		max_angle_disp_2_ = value;
	}

	/// @brief get the default maximum angular displacement for 3 residue backrub moves
	inline
	bool
	max_angle_disp_3()
	{
		return max_angle_disp_3_;
	}

	/// @brief set the default maximum angular displacement for 3 residue backrub moves
	inline
	void
	set_max_angle_disp_3(
		float value
	)
	{
		assert(value >= 0);

		max_angle_disp_3_ = value;
	}

	/// @brief get the default maximum angular displacement adjustment for >3 residue backrub moves
	inline
	bool
	max_angle_disp_slope()
	{
		return max_angle_disp_slope_;
	}

	/// @brief set the default maximum angular displacement adjustment for >3 residue backrub moves
	inline
	void
	set_max_angle_disp_slope(
		float value
	)
	{
		assert(value >= 0);

		max_angle_disp_slope_ = value;
	}

	/// @brief get the maximum angular displacement for two particular residues
	inline
	float
	max_angle_disp(
		int start,
		int end
	)
	{
		assert(start < end && start > 0 && end <= pose_->total_residue());

		if ((signed)max_angle_disp_[start].size() < end - start) {

			if (end - start == 1) {
				return max_angle_disp_2_;
			} else {
				return max_angle_disp_3_ + (end - start - 2) * max_angle_disp_slope_;
			}
		}

		return max_angle_disp_[start][end-start];
	}

	/// @brief set the maximum angular displacement for two particular residues (override default)
	inline
	void
	set_max_angle_disp(
		int start,
		int end,
		float value
	)
	{
		assert(start < end && start > 0 && end <= pose_->total_residue());
		assert(value >= 0);
		int const oldsize = max_angle_disp_[start].size();

		if (oldsize < end - start) {

			max_angle_disp_[start].resize(end - start);
			for (size_t i = oldsize; i < max_angle_disp_[start].size(); i++) {
				if (i == 1) {
					max_angle_disp_[start][i] = max_angle_disp_2_;
				}	else {
					max_angle_disp_[start][i] = max_angle_disp_3_ + (end - start - 2) * max_angle_disp_slope_;
				}
			}
		}

		max_angle_disp_[start][end - start] = value;
	}

	/// @brief get the angular displacement for two particular residues
	inline
	float
	angle_disp(
		int start,
		int end
	)
	{
		assert(start < end && start > 0 && end <= pose_->total_residue());
		assert((signed)angle_disp_.size() >= start);
		assert((signed)angle_disp_[start].size() >= end - start);

		return angle_disp_[start][end-start];
	}

	/// @brief get the minimum number of residues allowed for a backrub move
	inline
	int
	min_res()
	{
		return min_res_;
	}

	/// @brief set the minimum number of residues allowed for a backrub move
	inline
	void
	set_min_res(
		int value
	)
	{
		assert(value >= 2);

		min_res_ = value;
	}

	/// @brief get the maximum number of residues allowed for a backrub move
	inline
	int
	max_res()
	{
		return max_res_;
	}

	/// @brief get the maximum number of residues allowed for a backrub move
	inline
	void
	set_max_res(
		int value
	)
	{
		assert(value >= 2);

		max_res_ = value;
	}

	/// @brief get the probability of only changing rotamers
	inline
	float
	only_rot()
	{
		return only_rot_;
	}

	/// @brief set the probability of only changing rotamers
	inline
	void
	set_only_rot(
		float value
	)
	{
		only_rot_ = value;
	}

	/// @brief get the probability of only changing the backbone
	inline
	float
	only_bb()
	{
		return only_bb_;
	}

	/// @brief set the probability of only changing the backbone
	inline
	void
	set_only_bb(
		float value
	)
	{
		only_bb_ = value;
	}

	/// @brief get boolean indicating whether to backbone minimize
	inline
	bool
	bb_min()
	{
		return bb_min_;
	}

	/// @brief set boolean indicating whether to backbone minimize
	inline
	void
	set_bb_min(
		bool value
	)
	{
		bb_min_ = value;
	}

	/// @brief get boolean indicating whether to sidechain minimize
	inline
	bool
	sc_min()
	{
		return sc_min_;
	}

	/// @brief set boolean indicating whether to sidechain minimize
	inline
	void
	set_sc_min(
		bool value
	)
	{
		sc_min_ = value;
	}

	void
	mc_loop(
		pose_ns::Monte_carlo & mc,
		int ntrials
	);

	void
	mc_loop(
		pose_ns::Pose & pose,
		pose_ns::Monte_carlo & mc,
		int ntrials
	);

	bool
	trial(
		pose_ns::Monte_carlo & mc
	);

	bool
	trial(
		pose_ns::Pose & pose,
		pose_ns::Monte_carlo & mc
	);

	void
	trial(
		pose_ns::Pose & pose,
		pose_ns::Monte_carlo & mc,
		const pose_ns::Score_weight_map & weight_map,
		std::string const & min_type,
		const int nmoves,
		const bool try_rotamers,
		const float energycut,
		std::string const tag_suffix = ""
	);

};

} // end pose_ns namespace

#endif
