// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.13 $
//  $Date: 2005/05/18 07:35:25 $
//  $Author: mentzer $

// Rosetta Headers
#include "pose_loops.h"
#include "after_opts.h"
#include "are_they_neighbors.h"
#include "cenlist.h"
#include "constraints.h"
#include "dock_structure.h"
#include "dock_loops.h"
#include "dock_pivot.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fold_loops.h"
#include "fullatom.h"
#include "initialize.h"
#include "input_pdb.h"
#include "interface.h"
#include "jumping_diagnostics.h"
#include "jumping_loops.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "ligand.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "loops_ns.h"
#include "loops.h"
#include "make_pdb.h"
#include "monte_carlo.h"
#include "misc.h"
#include "minimize.h"
#include "native.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "pose.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "pose_vdw.h"
#include "ramachandran.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "smallmove.h"
#include "status.h"
#include "timer.h"
#include "util_basic.h"
#include "util_vector.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/FArray.io.hh>

// Numeric Headers
//#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

///////////////////////////////////////////////////////////////////////////////
void
pose_loops_main()
{
	using namespace pose_ns;
	using namespace loops_ns;

	set_pose_flag( true );
	set_pose_loop_flag(true);

	// copy from misc
	Pose pose;
	bool const ideal_pos( false );
	bool const coords_init( true );
	pose_from_misc( pose, files_paths::input_fa, ideal_pos, coords_init );

	// define loop regions
	Loops loops;
	bool ok;
	ok = pose_loops_init_from_file( loops );

	if( !ok ) {
		ok = pose_loops_init_from_global_ns( loops );
	}
	if ( !ok ) {
		std::cout << "ERROR -- pose_loops_main: can not retrieve loops info\n"
							<< "exit ...... " << std::endl;
		assert( false );
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << "************************************************\n"
						<< "loops to be modeled:\n" << loops
						<< "************************************************"
						<< std::endl;
	setup_loop_splicing( false );

	// set up fold_tree and allow_bb_move properly
	Fold_tree f;
	pose_loops_build_fold_tree( pose.total_residue(), loops, f );
	pose.set_fold_tree( f );
	std::cout << "Loop_fold_tree: " << f ;
	pose.set_allow_jump_move( false );
	pose_loops_set_allow_bb_move( pose, loops );

	// trim if necessary
	Pose trim_pose;
	Loops trim_loops;
	if( loop_bool::trim ) {
		//trim_template
		FArray1D_bool trim_res( pose.total_residue(), false );
		pose_update_cendist( pose ); // this is required for trim
		pose_loops_select_trim_residues( pose, loops, trim_res );
		pose_loops_trim_template( pose, loops, trim_res, trim_pose, trim_loops );
		//set_allow_*_move correctly after trim
		trim_pose.set_allow_jump_move( false );
		pose_loops_set_allow_bb_move( trim_pose, trim_loops );
		// generate a starting conformation if the loop does not exist
		pose_loops_random_start( trim_pose, trim_loops );
		// fold and refine loops
		pose_fold_loops_with_ccd( trim_pose, trim_loops );
		// trim_reset
		pose_loops_trim_template_reset( pose, loops, trim_res, trim_pose,
			trim_loops );
	} else {
		// generate a starting conformation if the loop does not exist
		pose_loops_random_start( pose, loops );
		// fold and refine loops
		pose_fold_loops_with_ccd( pose, loops );
	}

	// copy back to misc
	pose.copy_to_misc();

	set_pose_flag( false );

	monte_carlo_reset();

}
////////////////////////////////////////////////////////////////////////
void
pose_loops_random_start(
	pose_ns::Pose & pose,
	const pose_ns::Loops & loops
)
{
	using namespace pose_ns;

	for( Loops::const_iterator it = loops.begin(), it_end = loops.end();
			 it != it_end; ++it ) {
		if( it->is_extended() ) {
			int const loop_begin = it->start();
			int const loop_end = it->stop();
			int const loop_size = loop_end - loop_begin + 1;
			int const frag_offset = it->offset();

			pose.insert_ideal_bonds( loop_begin, loop_end );
			pose.set_segment_extended( loop_begin, loop_end );
			// randomly insert 3 mers to generate a starting conformation
			for( int i=1; i<=loop_size-3; ++i ) {
				choose_offset_frag( 3, pose, loop_begin, loop_end, frag_offset );
			}
		}
	}
}
////////////////////////////////////////////////////////////////////////
void
pose_fold_loops_with_ccd(
	pose_ns::Pose & pose,
	const pose_ns::Loops & loops
)
{
	using namespace pose_ns;

	std::cout << "pose_fold_loops_with_ccd" << std::endl;
	const Fold_tree & f( pose.fold_tree() );
	int const num_fold_tree_cutpoint( f.get_num_fold_tree_cutpoint() );
	const FArray1D_int & cutpoint_map( f.get_cutpoint_map() );

	Score_weight_map weight_map;
	FArray1D_float cut_weight( num_fold_tree_cutpoint, 0.0f );

	Pose save_fullatom_pose;
	if( pose.fullatom() ) {
		save_fullatom_pose = pose;
		pose.set_fullatom_flag( false );
	}

	// centroid loop modeling
	std::cout << "perturb loops in centroid mode" << std::endl;
	setup_score_weight_map( weight_map, score4L );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );

	if( truefalseoption("refine_only") ) {
		std::cout << "skip perturb loops in centroid mode\n";

		for( Loops::const_iterator it = loops.begin(), it_end = loops.end();
				 it != it_end; ++it ) {
			int const loop_cut( it->cut() );
			pose.insert_ideal_bonds( loop_cut, loop_cut+1 );
		}

		cut_weight = 1.0f;

		goto L100;
	}

	// for multiple cutpoint
	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		// focus one loop each a time
		Loops one_loop;
		one_loop.add_loop( it );
		// assign proper cut_weight
		int const cutpoint_num = cutpoint_map( it->cut() );
		assert( cutpoint_num > 0 && cutpoint_num <= num_fold_tree_cutpoint );
		cut_weight(cutpoint_num) = 1.0f;
		weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
		// assign allow_bb_move
		pose_loops_set_allow_bb_move( pose, one_loop );
		// ccd perturb
		pose_perturb_one_loop_with_ccd( pose, weight_map, one_loop );
	}

 L100:

	if( save_fullatom_pose.total_residue() == pose.total_residue() ) {
		pose.set_fullatom_flag( true, false );
		pose.recover_sidechain( save_fullatom_pose );
	} else if( files_paths::output_fa ) {
		pose.set_fullatom_flag( true, true );
	} else {
		return;
	}

	std::cout << "refine loops in fullatom mode" << std::endl;
	// refine loops only if in fullatom
	setup_score_weight_map( weight_map, score12 );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight ); // for setup of cut_weight see above sections "refine_only" and "multiple cutpoint"

	pose_loops_set_allow_bb_move( pose, loops );
	pose_refine_loops_with_ccd( pose, weight_map, loops );

	return;
}
/////////////////////////////////////////////////////////////////////////////
void
pose_perturb_one_loop_with_ccd(
	pose_ns::Pose & pose,
	const pose_ns::Score_weight_map & wt_map,
	const pose_ns::Loops & one_loop
)
{
	using namespace pose_ns;

	assert( one_loop.num_loop() == 1 );
	Loops::const_iterator it = one_loop.begin();
	int const loop_begin = it->start();
	int const loop_end = it->stop();
	int const frag_offset = it->offset();

	pose_loops_set_allow_bb_move( pose, one_loop );

	// param for perturbation cycles
	int const one_loop_size( one_loop.loop_size() );
	int  outer_cycles( 3 );
	int  inner_cycles( loops_ns::fast ?
		std::min( 250, one_loop_size*5 ) : std::min( 1000, one_loop_size*20 ) );
	if ( runlevel_ns::benchmark ) {
		outer_cycles = 1;
		inner_cycles = one_loop_size*2;
	}
	float const init_temp( 2.0 ), final_temp( 1.0 );
	float const gamma =
		std::pow( (final_temp/init_temp), (1.0f/(outer_cycles*inner_cycles)) );

	// param for frag insert
	int const size9 = { 9 };
	int const size3 = { 3 };
	int const size1 = { 1 };
	int const cutoff9 = { 16 }; // no 9mer insert if less than this
	int const cutoff3 = { 6 }; // no 3mer insert if less than this
	int const cutoff1 = { 3 }; // no 1mer insertion if less than this

	// mc obj
	Monte_carlo mc( pose, wt_map, init_temp );

	float temperature = init_temp;
	for( int i=1; i<=outer_cycles; ++i ) {
		// recover low
		pose = mc.low_pose();

		for( int j=1; j<=inner_cycles; ++j ) {
			// change temperature
			temperature *= gamma;
			mc.set_temperature( temperature );

			// insert 9 mers if necessary
			if ( one_loop_size > cutoff9 ) {
				choose_offset_frag( size9, pose, loop_begin, loop_end, frag_offset );
				// not necessary to call refold here because it is done through pose.copy_to_misc
				// at the beginning of fast_ccd_loop_closure.
				pose_ccd_close_loops( pose, one_loop );
				pose.main_minimize( wt_map, "linmin" );
				mc.boltzmann( pose );
			} // 9 mer

			// insert 3 mers if necessary
			if ( one_loop_size > cutoff3 ) {
				choose_offset_frag( size3, pose, loop_begin, loop_end, frag_offset );
				// not necessary to call refold here because it is done through pose.copy_to_misc
				// at the beginning of fast_ccd_loop_closure.
				pose_ccd_close_loops( pose, one_loop );
				pose.main_minimize( wt_map, "linmin" );
				mc.boltzmann( pose );
			} // 3 mer

			// insert 1 mer if necessary
			if ( one_loop_size > cutoff1 ) {
				choose_offset_frag( size1, pose, loop_begin, loop_end, frag_offset);
					// not necessary to call refold here because it is done through pose.copy_to_misc
				// at the beginning of fast_ccd_loop_closure.
				pose_ccd_close_loops( pose, one_loop );
				pose.main_minimize( wt_map, "linmin" );
				mc.boltzmann( pose );
			} // 1 mer
		} // inner_cycles
	} // outer_cycles

	pose = mc.low_pose();
}
/////////////////////////////////////////////////////////////////////////////
void
pose_refine_loops_with_ccd(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map & wt_map,
	const pose_ns::Loops & loops
)
{
	using namespace pose_ns;

	//	assert( pose.fullatom() );

	int const loop_size ( loops.loop_size() );

	bool const include_current = { true };
	bool const include_neighbors = { ! loops_ns::fix_natsc };
	// param for small/shear moves
	int const nmoves = { 1 };
	// param for MC cycles
	int   const cycle_number( std::max( 2, static_cast<int>( loop_size*get_looprlx_cycle_ratio())));
//	int   outer_cycles = { 3 };
	int		outer_cycles = 3;
	int   inner_cycles = ( loops_ns::fast ?
		std::min( 20, cycle_number ) : std::min( 200, 10*cycle_number ) );

	if ( runlevel_ns::benchmark ) {
		outer_cycles = 1;
		inner_cycles = 5;
	}

	pose_refine_loops_with_ccd( pose, wt_map, loops, outer_cycles, inner_cycles,
		nmoves, include_current, include_neighbors );
}
///////////////////////////////////////////////////////////////////////////////////
void
pose_refine_loops_with_ccd(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map & wt_map,
	const pose_ns::Loops & loops,
	int const outer_cycles,
	int const inner_cycles,
	int const nmoves,
	bool const include_current,
	bool const include_neighbors
)
{
	using namespace pose_ns;

	bool const fullatom( pose.fullatom() );

	FArray1D_bool allow_repack( pose.total_residue() );
	if ( fullatom ) pose_loops_select_loop_residues( pose, loops, include_neighbors,
		allow_repack);
	int const nres( pose.total_residue() );
	FArray1D_bool save_allow_chi_move( nres );
	if ( fullatom ) {
		for ( int i=1; i<=nres; ++i ) {
			save_allow_chi_move(i) = pose.get_allow_chi_move(i) ;
		}
	}

	if ( fullatom ) { //vats changed the order of calls here,check with Chu, if he is ok with it
		pose.set_allow_chi_move( allow_repack );
		pose.repack( allow_repack, include_current );
		pose.set_allow_chi_move( save_allow_chi_move );
	}
	pose.score( wt_map );

	float const init_temp = { 1.5 };
	float const last_temp = { 0.5 };
	float const gamma =
		std::pow( (last_temp/init_temp), 1.0f/(outer_cycles*inner_cycles) );
	int small_accepted(0), shear_accepted(0), n_trial(0); //diagnostics

	// mc obj
	Monte_carlo mc( pose, wt_map, init_temp );
	//mc.show_scores();

	float temperature = init_temp;

	for ( int i=1, ii=i+3; i<=outer_cycles; ++i, ++ii ) {
		// increase CHAINBREAK weight
		wt_map.set_weight( CHAINBREAK, float(i) ); //chutmp std::pow(float(ii),4.0f)/50.0f );
		mc.set_weight_map( wt_map );
		// recover low
		pose = mc.low_pose();
		// score info
		std::cout << "cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;
		for ( int j=1; j<=inner_cycles; ++j ) {
			temperature *= gamma;
			mc.set_temperature( temperature );
			n_trial++;
			{ // small min trial
				Loops one_loop;
				one_loop.add_loop( loops.one_random_loop() );
				pose_loops_set_allow_bb_move( pose, one_loop );
				//std::cout << "before small_moves :" << pose.score( wt_map ) << std::endl;
				pose_small_moves( pose, nmoves);
				//std::cout << "before ccd:" << pose.score( wt_map ) << std::endl;
				if( one_loop.begin()->stop()  != pose.total_residue() &&
						one_loop.begin()->start() != 1 )
					pose_ccd_close_loops( pose, one_loop);
				//std::cout << "before minimize:" << pose.score( wt_map ) << std::endl;
				pose_loops_set_allow_bb_move( pose, loops );
				if ( fullatom )
					pose.main_minimize( wt_map, "dfpmin" );
				else pose.main_minimize( wt_map, "linmin" );
				//std::cout << "before mc:" << pose.score( wt_map ) << std::endl;
				if( mc.boltzmann( pose ) ) { small_accepted++; }
				//mc.show_scores();
			}

			{ //shear min trial
				Loops one_loop;
				one_loop.add_loop( loops.one_random_loop() );
				pose_loops_set_allow_bb_move( pose, one_loop );
				//std::cout << "before small_moves :" << pose.score( wt_map ) << std::endl;
				pose_shear_moves( pose, nmoves);
				//std::cout << "before ccd:" << pose.score( wt_map ) << std::endl;
				if( one_loop.begin()->stop()  != pose.total_residue() &&
					one_loop.begin()->start() != 1 )
				pose_ccd_close_loops( pose, one_loop );
				//std::cout << "before minimize:" << pose.score( wt_map ) << std::endl;
				pose_loops_set_allow_bb_move( pose, loops );
				if ( fullatom )
					pose.main_minimize( wt_map, "dfpmin" );
				else pose.main_minimize( wt_map, "linmin" );
				//std::cout << "before mc:" << pose.score( wt_map ) << std::endl;
				if( mc.boltzmann( pose ) ) { shear_accepted++; }
				//mc.show_scores();
			}
			if ( fullatom )
				if ( mod(j,20)==0 || j==inner_cycles ) {
					// repack trial
					if (pose.atom_tree()){
						pose.repack( save_allow_chi_move, include_current );//psh changed to save_allow_chi_move from allow_repack
					}
					else {
						pose.repack( allow_repack, include_current );
					}
				pose.score (wt_map );
				mc.boltzmann( pose );
			}
		} // inner cycles

	} // outer cycles

	pose = mc.low_pose();

	std::cout << "pose_refine_loops_with_ccd -- accepted/trial/ratio:\n"
						<< small_accepted << "/" << n_trial << "/"
						<< float(small_accepted)/float(n_trial) << " for small moves\n"
						<< shear_accepted << "/" << n_trial << "/"
						<< float(shear_accepted)/float(n_trial) << " for shear moves"
						<< std::endl;
}
/////////////////////////////////////////////////////////////////////////////
int
pose_small_moves(
	pose_ns::Pose & pose,
	int const num_in
)
{
	// retreive info from pose
	int total_insert;
	FArray1D_int   const & insert_map( pose.get_insert_map( total_insert ) );
	FArray1D_char  const & secstruct( pose.secstruct() );
	FArray1D_int   const & res( pose.res() );

	int nfail = 0;

	// calc num of movable postions
	int nres( 0 );
	for( int j = 1; j <= total_insert; ++j ) {
		int k = insert_map(j);
		if ( secstruct(k) == 'L' && small_move_param::other_max > 0.0 ) ++nres;
		if ( secstruct(k) == 'H' && small_move_param::helix_max > 0.0 ) ++nres;
		if ( secstruct(k) == 'E' && small_move_param::strand_max > 0.0 ) ++nres;
	}
	int num = std::min(num_in,nres/2);

	if ( num < 1 ) {
		if ( nres > 1 ) {
			num = 1;
		} else {
			std::cout << "no movable positions in Pose::small_move" << std::endl;
			return 0;
		}
	}

	// local variables
	std::vector<int> list;
	float small_angle, big_angle; //move magnitude
	float phi_tmp, psi_tmp; // new phi/psi
	float old_rama_score, new_rama_score, tmp1, tmp2; // rama eval
	float boltz_factor, probability, crap; // prob to accept
	// special treatment the first 5 and last 5 residues
	int end = pose.total_residue() - 5;

	for ( int k = 1; k <= num; ++k ) {

	L401:
		nfail++;
		if( nfail > 1000 ) return 0;
		// randomly choose a position to insert
		int j = insert_map( static_cast< int >(ran3()*(total_insert)) + 1 );

		if ( !pose.get_allow_bb_move(j) ) goto L401; // fixed position,choose again

		if ( secstruct(j) == 'H' ) {
			if ( small_move_param::helix_max <= 0.0 ) goto L401; //// skip helix
			big_angle = small_move_param::helix_max;
		} else if ( secstruct(j) == 'E' ) {
			if ( small_move_param::strand_max <= 0.0 ) goto L401; //// skip strand
			big_angle = small_move_param::strand_max;
		} else {
			if ( small_move_param::other_max <= 0.0 ) goto L401; //// skip loop
			big_angle = small_move_param::other_max;
		}
		small_angle = big_angle/2.0;

		//  next three lines skip ends of structure !!
		//  fix a logic error here: the closer to the end, the higher the probability
		//  to skip it; and when it is 1 or total_residue, the probability should be
		//  zero; and the probability distribution should be symmetrical for two ends
		//  residue:    N-   1   2   3   4   5   6
		//  residue:    C-   t  t-1 t-2 t-3 t-4 t-5
		//  prob to skip:    1  0.8 0.6 0.4 0.2 0.0
		//  -- chu

		if ( j <= 5 && static_cast< int >(ran3()*5+1) >= j ) goto L401;
		if ( j > end && static_cast< int >(ran3()*5) + end < j ) goto L401;

		for ( unsigned kk = 0; kk < list.size(); ++kk ) {
			if ( j == list[kk] ) goto L401; // already moved
		}

		// new angles after moves
		phi_tmp = periodic_range( pose.phi(j)-small_angle+ran3()*big_angle, 360.0f );
		psi_tmp = periodic_range( pose.psi(j)-small_angle+ran3()*big_angle, 360.0f );

		// check if rama score is good
		eval_rama_score_residue(res(j),pose.phi(j),pose.psi(j),secstruct(j),old_rama_score,
			tmp1,tmp2);
		eval_rama_score_residue(res(j),phi_tmp,psi_tmp,secstruct(j),new_rama_score,
			tmp1,tmp2);
//    std::cout << SS( j ) << SS( old_rama_score ) << SS( new_rama_score ) <<
//     SS( res(j) ) << SS( pose.phi(j) ) << SS( phi_tmp ) << std::endl;
		if ( new_rama_score > old_rama_score ) {
			boltz_factor = (old_rama_score-new_rama_score)/small_move_param::temp;
			probability = std::exp(std::max(-40.0f,boltz_factor) );
			crap = ran3(); // bug if  ran3 call in if statment
			if ( crap >= probability ) goto L401;
		}

		// make moves and store this position
		pose.set_phi( j, phi_tmp );
		pose.set_psi( j, psi_tmp );
		pose.set_name( j, "-SM-" );
		list.push_back(j);
	}
	return int(list.size());
}
//////////////////////////////////////////////////////////////////////
int
pose_shear_moves(
	pose_ns::Pose & pose,
	int const num_in
)
{
	// retreive info from pose
	int total_insert;
	FArray1D_int   const & insert_map( pose.get_insert_map( total_insert ) );
	FArray1D_char  const & secstruct( pose.secstruct() );
	FArray1D_int   const & res( pose.res() );

	int nfail = 0;

	// calc num of movable postions
	int nres( 0 );
	for( int j = 1; j <= total_insert; ++j ) {
		int k = insert_map(j);
		if ( secstruct(k) == 'L' && small_move_param::other_max > 0.0 ) ++nres;
		if ( secstruct(k) == 'H' && small_move_param::helix_max > 0.0 ) ++nres;
		if ( secstruct(k) == 'E' && small_move_param::strand_max > 0.0 ) ++nres;
	}
	int num = std::min(num_in,nres/2);

	if ( num < 1 ) {
		if ( nres > 1 ) {
			num = 1;
		} else {
			std::cout << "no movable positions in Pose::small_move" << std::endl;
			return 0;
		}
	}

	// local variables
	std::vector<int> list;
	float small_angle, big_angle; //move magnitude
	float phi_tmp, psi_tmp; // new phi/psi
	float old_rama_score, new_rama_score, tmp1, tmp2; // rama eval
	float boltz_factor, probability, crap; // prob to accept
	// special treatment the first 5 and last 5 residues
	int end = pose.total_residue() - 5;

	for ( int k = 1; k <= num; ++k ) {

	L401:
		nfail++;
		if (nfail > 1000) return 0;
		// randomly choose a position to insert
		int j = insert_map( static_cast< int >(ran3()*(total_insert-1)) + 2 );

		if ( !pose.get_allow_bb_move(j) ||
				 !pose.get_allow_bb_move(j-1)) goto L401; //fixed position,choose again

		if ( secstruct(j) == 'H' ) {
			if ( small_move_param::helix_max <= 0.0 ) goto L401; //// skip helix
			big_angle = small_move_param::helix_max;
		} else if ( secstruct(j) == 'E' ) {
			if ( small_move_param::strand_max <= 0.0 ) goto L401; //// skip strand
			big_angle = small_move_param::strand_max;
		} else {
			if ( small_move_param::other_max <= 0.0 ) goto L401; //// skip loop
			big_angle = small_move_param::other_max;
		}
		small_angle = big_angle/2.0;

		//  next three lines skip ends of structure !!
		//  fix a logic error here: the closer to the end, the higher the probability
		//  to skip it; and when it is 1 or total_residue, the probability should be
		//  zero; and the probability distribution should be symmetrical for two ends
		//  residue:    N-   1   2   3   4   5   6
		//  residue:    C-   t  t-1 t-2 t-3 t-4 t-5
		//  prob to skip:    1  0.8 0.6 0.4 0.2 0.0
		//  -- chu

		if ( j <= 5 && static_cast< int >(ran3()*5+1) >= j ) goto L401;
		if ( j > end && static_cast< int >(ran3()*5) + end < j ) goto L401;

		for ( unsigned kk = 0; kk < list.size(); ++kk ) {
			if ( j == list[kk] ) goto L401; // already moved
		}

		// new angles after moves
		float const shear_delta = small_angle+ran3()*big_angle;
		phi_tmp = periodic_range( pose.phi(j)-shear_delta, 360.0f );
		psi_tmp = periodic_range( pose.psi(j-1)+shear_delta, 360.0f );

		// check if rama score is good for res j
		eval_rama_score_residue(res(j),pose.phi(j),pose.psi(j),secstruct(j),old_rama_score,
			tmp1,tmp2);
		eval_rama_score_residue(res(j),phi_tmp,pose.psi(j),secstruct(j),new_rama_score,
			tmp1,tmp2);

		if ( new_rama_score > old_rama_score ) {
			boltz_factor = (old_rama_score-new_rama_score)/small_move_param::temp;
			probability = std::exp(std::max(-40.0f,boltz_factor) );
			crap = ran3(); // bug if  ran3 call in if statment
			if ( crap >= probability ) goto L401;
		}

		// check if rama score is good for res j-1
		eval_rama_score_residue(res(j-1),pose.phi(j-1),pose.psi(j-1),secstruct(j-1),old_rama_score,
			tmp1,tmp2);
		eval_rama_score_residue(res(j-1),pose.phi(j-1),psi_tmp,secstruct(j-1),new_rama_score,
			tmp1,tmp2);

		if ( new_rama_score > old_rama_score ) {
			boltz_factor = (old_rama_score-new_rama_score)/small_move_param::temp;
			probability = std::exp(std::max(-40.0f,boltz_factor) );
			crap = ran3(); // bug if  ran3 call in if statment
			if ( crap >= probability ) goto L401;
		}

		// make moves and store this position
		pose.set_phi( j  , phi_tmp );
		pose.set_psi( j-1, psi_tmp );
		pose.set_name( j  , "-SM-" );
		pose.set_name( j-1, "-SH-" );
		list.push_back(j);
		list.push_back(j-1);
	}
	return int(list.size());
}
///////////////////////////////////////////////////////////////////////////////
void
pose_ccd_close_loops(
	pose_ns::Pose & pose,
	const pose_ns::Loops & loops
)
{
	using namespace pose_ns;

	// param for ccd_closure
	int   const ccd_cycles = { 100 }; // num of cycles of ccd_moves
	float const ccd_tol = { 0.01 }; // criterion for a closed loop
	bool  const rama_check = { false }; // psh turning true
	float const max_rama_score_increase = { 2.0 }; // dummy number when rama_check is false
	float const max_total_delta_helix = { 10.0 }; // max overall angle changes for a helical residue
	float const max_total_delta_strand = { 50.0 }; // ... for a residue in strand
	float const max_total_delta_loop = { 75.0 }; // ... for a residue in loop
	// output for ccd_closure
	float forward_deviation, backward_deviation; // actually loop closure msd, both dirs
	float torsion_delta, rama_delta; // actually torsion and rama score changes, averaged by loop_size

	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		int const loop_begin = it->start();
		int const loop_end = it->stop();
		int const cutpoint = it->cut();
		// ccd close this loop
		fast_ccd_loop_closure( pose, loop_begin, loop_end, cutpoint, ccd_cycles,
			ccd_tol, rama_check, max_rama_score_increase, max_total_delta_helix,
			max_total_delta_strand, max_total_delta_loop, forward_deviation,
			backward_deviation, torsion_delta, rama_delta );
	}
}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_trim_template(
	const pose_ns::Pose & input_pose,
	const pose_ns::Loops & input_loops,
	const FArray1D_bool & trim_res,
	pose_ns::Pose & trim_pose,
	pose_ns::Loops & trim_loops
)
{
	using namespace pose_ns;
	// book keeping between input_pose and trim_pose
	int const nres( input_pose.total_residue() );
	FArray1D_int trim_map( nres, 0 ), trim_map_reverse( nres, 0 );
	int j = 0;
	for( int i=1; i<=nres; ++i ) {
		if( trim_res(i) ) {
			trim_map(i) = ++j;
			trim_map_reverse(j) = i;
		}
	}
	int const trim_total_residue = j;
	// has to set up a tree for trim_pose first in order to call set_phi etc.
	// so choose a simple tree for convenience and will replace it later
	//trim_pose.simple_fold_tree( trim_total_residue );

	// set up the loop obj after trimming.
	assert( trim_loops.num_loop() == 0 );
	for( Loops::const_iterator it=input_loops.begin(), it_end=input_loops.end();
			 it != it_end; ++it ) {
		trim_loops.add_loop( trim_map(it->start()), trim_map(it->stop()),
			trim_map(it->cut()), it->offset()+it->start()-trim_map(it->start()),
			it->is_extended() );
	}
	pose_loops_init_to_global_ns( trim_loops );

	// set fold_tree
	Fold_tree trim_fold_tree;
	pose_loops_build_fold_tree( trim_total_residue, trim_loops, trim_fold_tree );
	trim_pose.set_fold_tree( trim_fold_tree );
	std::cout <<"trim_fold_tree: " << trim_fold_tree;

	// do the dirty copying work
	int i;
	FArray3D_float trim_Epos(3, param::MAX_POS, trim_total_residue);
	FArray3D_float trim_fullcoord(3, param::MAX_ATOM(), trim_total_residue);
	FArray3D_float const & input_Epos( input_pose.Eposition() );
	FArray3D_float const & input_fullcoord( input_pose.full_coord() );

	for( int j=1; j<=trim_total_residue; ++j ) {
		i = trim_map_reverse(j);
		trim_pose.set_phi        ( j, input_pose.phi(i) );
		trim_pose.set_psi        ( j, input_pose.psi(i) );
		trim_pose.set_omega      ( j, input_pose.omega(i) );
		trim_pose.set_secstruct  ( j, input_pose.secstruct(i) );
		trim_pose.set_name       ( j, input_pose.name(i) );
		trim_pose.set_res        ( j, input_pose.res(i) );
		trim_pose.set_res_variant( j, input_pose.res_variant(i) );
		for( int k=1; k<=param::MAX_POS; ++k ) {
			for( int l=1; l<=3; ++l ) {
				trim_Epos(l,k,j) = input_Epos(l,k,i);
			}
		}
	}
	if( input_pose.fullatom() ) {
		trim_pose.set_fullatom_flag(true,false);// set fullatom and do not repack
		for( int j=1; j<=trim_total_residue; ++j ) {
			i = trim_map_reverse(j);
			for( int k=1; k<=param::MAX_ATOM()(); ++k ) {
				for( int l=1; l<=3; ++l ) {
					trim_fullcoord(l,k,j) = input_fullcoord(l,k,i);
				}
			}
		}
	}

	bool const ideal_pos( false ), check_missing( false );
	// this will update jumps.
	trim_pose.set_coords( ideal_pos, trim_Epos, trim_fullcoord, check_missing );
	// Important to call pose_to_misc; see below for explanation.
	pose_to_misc( trim_pose );


	// trim cst
	classical_constraints::BOUNDARY::slice_cst( trim_res, nres );

}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_trim_template_reset(
	pose_ns::Pose & pose,
	const pose_ns::Loops & loops,
	const FArray1D_bool & trim_res,
	const pose_ns::Pose & trim_pose,
	const pose_ns::Loops & trim_loops
)
{
	using namespace pose_ns;

	int const nres = pose.total_residue();
	assert( nres == int(trim_res.size1()) );

	// sidechain could be changed at any position
	if ( pose.fullatom() ) {
		const FArray3D_float & full_cood( trim_pose.full_coord( ) );
		int j = 0;
		for( int i=1; i<=nres; ++i ){
			if ( trim_res(i) ) {
				++j;
				pose.copy_sidechain( i, pose.res(i), pose.res_variant(i),
														 full_cood(1,1,j) );
			}
		}
	}

	// only need to update the backbone/sidechain within the loop regions.
	// and not necessary to update jumps as they should be kept fixed in trim
	for( Loops::const_iterator it=loops.begin(), it2= trim_loops.begin(),
				 it_end= loops.end(); it != it_end; ++it, ++it2 ) {
		assert( (it->start()+it->offset()) == (it2->start()+it2->offset()) );
		assert( it->size() == it2->size() );

		pose.copy_segment( it->size(), trim_pose, it->start(), it2->start(),
			false/*update_jumps*/);
	}

	pose_loops_init_to_global_ns( loops );
	pose_to_misc( pose );


	// reset cst
	classical_constraints::BOUNDARY::restore_cst();
}

///////////////////////////////////////////////////////////////////////////////
/*
void
pose_loops_build_fold_tree(
	int const total_residue,
	const pose_ns::Loops & loops,
	pose_ns::Fold_tree & f
)
{
	using namespace pose_ns;

	f.clear();

	Loops tmp_loops = loops;
	tmp_loops.sequential_order();

	int jump_num = 0;
	for( Loops::const_iterator it=tmp_loops.begin(), it_end=tmp_loops.end(),
				 it_next; it != it_end; ++it ) {
		it_next = it;
		it_next++;
		jump_num++;
		int const jump_start = it->start() - 1;
		int const jump_stop  = it->stop() + 1;
		int const jump_cut   = it->cut();
		int const jump_next_start =
			( it_next == it_end ) ? total_residue : it_next->start()-1;
		f.add_edge( jump_start, jump_stop, jump_num );
		f.add_edge( jump_start, jump_cut,  pose_param::PEPTIDE );
		f.add_edge( jump_cut+1, jump_stop, pose_param::PEPTIDE );
		f.add_edge( jump_stop, jump_next_start, pose_param::PEPTIDE );
	}
	f.add_edge( 1, tmp_loops.begin()->start()-1, pose_param::PEPTIDE );

	f.reorder(1);

}
*/

///////////////////////////////////////////////////////////////////////////////
void
pose_loops_build_fold_tree(
	int const total_residue,
	const pose_ns::Loops & loops,
	pose_ns::Fold_tree & f
)
{
	using namespace pose_ns;

	f.clear();

	Loops tmp_loops = loops;
	tmp_loops.sequential_order();

	int jump_num = 0;
	for( Loops::const_iterator it=tmp_loops.begin(), it_end=tmp_loops.end(),
				 it_next; it != it_end; ++it ) {
		it_next = it;
		it_next++;
		int const jump_start =
			( it->start() == 1 ) ? it->start() : it->start() - 1;
		int const jump_stop  =
			( it->stop() == total_residue ) ? it->stop() : it->stop() + 1;
		int const jump_cut   = it->cut();
		int const jump_next_start =
			( it_next == it_end ) ? total_residue : it_next->start()-1;
		if(  it->start() == 1 ){
			f.add_edge( jump_start, jump_stop, pose_param::PEPTIDE );
			f.add_edge( jump_stop, jump_next_start, pose_param::PEPTIDE );
			continue;
		}else if( it->stop() == total_residue ){
			f.add_edge( jump_start, jump_stop, pose_param::PEPTIDE );
			continue;
		}
		jump_num++;
		f.add_edge( jump_start, jump_stop, jump_num );
		f.add_edge( jump_start, jump_cut,  pose_param::PEPTIDE );
		f.add_edge( jump_cut+1, jump_stop, pose_param::PEPTIDE );
		f.add_edge( jump_stop, jump_next_start, pose_param::PEPTIDE );
	}
	if( tmp_loops.begin()->start() != 1 )
		f.add_edge( 1, tmp_loops.begin()->start()-1, pose_param::PEPTIDE );

	// reorder
	int root;
	if( tmp_loops.begin()->start() == 1 &&
			tmp_loops.begin()->stop() != total_residue )
		root = tmp_loops.begin()->stop()+1;
	else root = 1;
	f.reorder(root);

}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_select_trim_residues(
	pose_ns::Pose const & pose,
	FArray1D_bool & trim_res
)
{
	using namespace pose_ns;

	// pose internal data
	int const nres( pose.total_residue() );
	assert( int(trim_res.size1()) == nres );
	const Fold_tree & fold_tree( pose.fold_tree() );
	const FArray2D_float & cendist( pose.get_2D_score( CENDIST ) );
	// local
	FArray1D_bool keep_this_res( nres, false );
	FArray1D_bool check_this_res( nres, false );
	FArray1D_float cutoff( nres, 0.0f );
	float const STANDARD_CUTOFF = { cenlist_ns::cen_dist_cutoff2 + 25.0f };// 13A^2
	float const EXTENDED_CUTOFF = { STANDARD_CUTOFF + 56.0f }; // 15A^2

	// traverse the tree map
	for( Fold_tree::const_iterator it=fold_tree.begin(), it_end=fold_tree.end();
			 it != it_end; ++it ) {
		bool edge_is_extended = true;
		if ( it->label == pose_param::PEPTIDE  ) {

			// for each edge, first check if it is a loop and whether it is extended
			int const dir ( it->start < it->stop ? 1 : -1 );
			for( int i = it->stop; dir*i > dir*it->start; i-=dir ) {
				if ( ! pose.get_allow_bb_move(i) ) {
					goto L110; // edge is not loop, next edge
				} else if ( edge_is_extended && ( pose.phi(i) != param::init_phi
																					|| pose.psi(i) != param::init_psi ) ){
					edge_is_extended = false;
				}
			}
			// set check_res, trim_res, and cutoff accordingly
			if ( edge_is_extended ) {
				for( int i = it->stop; dir*i > dir*it->start; i-=dir ) {
					check_this_res(i) = false;
					keep_this_res(i) = true;
					cutoff(i) = STANDARD_CUTOFF;
				}
				check_this_res(it->start) = true;
				keep_this_res(it->start) = true;
				cutoff(it->start) = EXTENDED_CUTOFF;
			} else {
				for( int i = it->stop; dir*i >= dir*it->start; i-=dir ) {
					check_this_res(i) = true;
					keep_this_res(i) = true;
					cutoff(i) = STANDARD_CUTOFF;
				}
			}
		}

	L110: continue;
	} // finish traverse the tree

	// now decide what to keep
	for( int i=1; i<=nres; ++i ) {
		if( check_this_res(i) ) {
			for( int j=1; j<=nres; ++j ) {
				if( ! keep_this_res(j) && cendist(i,j) <= cutoff(i) )
					keep_this_res(j) = true;
			}
		}
	}

	trim_res = keep_this_res;
	return;
}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_select_trim_residues(
	pose_ns::Pose const & pose,
	pose_ns::Loops const & loops,
	FArray1D_bool & trim_res
)
{
	using namespace pose_ns;

	// pose internal data
	int const nres( pose.total_residue() );
	assert( int(trim_res.size1()) == nres );
	const FArray2D_float & cendist( pose.get_2D_score( CENDIST ) );
	// local
	FArray1D_bool keep_this_res( nres, false );
	FArray1D_bool check_this_res( nres, false );
	FArray1D_float cutoff( nres, 0.0f );
	float const STANDARD_CUTOFF = { cenlist_ns::cen_dist_cutoff2 + 25.0f };// 13A^2
	float const EXTENDED_CUTOFF = { STANDARD_CUTOFF + 56.0f }; // 15A^2

	// traverse the loops obj
	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		bool edge_is_extended = true;

		for( int i = it->start(); i <= it->stop(); ++i ) {
			assert ( pose.get_allow_bb_move(i) );
			if ( edge_is_extended && ( pose.phi(i) != param::init_phi
																 || pose.psi(i) != param::init_psi ) ){
				edge_is_extended = false;
			}
		}
		// set check_res, trim_res, and cutoff accordingly
		if ( edge_is_extended ) {
			for( int i = it->start() + 1; i <= it->stop() - 1; ++i ) {
				check_this_res(i) = false;
				keep_this_res(i) = true;
				cutoff(i) = STANDARD_CUTOFF;
			}
			check_this_res(it->start()) = true;
			keep_this_res(it->start()) = true;
			cutoff(it->start()) = EXTENDED_CUTOFF;
			check_this_res(it->stop()) = true;
			keep_this_res(it->stop()) = true;
			cutoff(it->stop()) = EXTENDED_CUTOFF;
		} else {
			for( int i = it->start(); i <= it->stop(); ++i ) {
				check_this_res(i) = true;
				keep_this_res(i) = true;
				cutoff(i) = STANDARD_CUTOFF;
			}
		}

	} // finish traverse the loops

	// now decide what to keep
	for( int i=1; i<=nres; ++i ) {
		if( check_this_res(i) ) {
			for( int j=1; j<=nres; ++j ) {
				if( ! keep_this_res(j) && cendist(i,j) <= cutoff(i) )
					keep_this_res(j) = true;
			}
		}
	}

	trim_res = keep_this_res;
	return;
}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_set_allow_bb_move(
	pose_ns::Pose & pose,
	const pose_ns::Loops & loops
)
{
	using namespace pose_ns;

	pose.set_allow_bb_move(false);

	FArray1D_bool allow( pose.total_residue() );
	bool const include_neighbors( false );
	pose_loops_select_loop_residues( pose, loops, include_neighbors, allow );

	pose.set_allow_bb_move( allow );

}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_select_loop_residues(
	const pose_ns::Pose & pose,
	const pose_ns::Loops & loops,
	bool const include_neighbors,
	FArray1D_bool & map // output
)
{
	using namespace pose_ns;

	if ( pose.symmetric() ) {
    pose_loops_select_loop_residues_symm(pose,loops,include_neighbors,map);
    return;
	}

	assert( int(map.size1()) == pose.total_residue() );

	map = false;

	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		for( int i=it->start(); i<=it->stop(); ++i ) {
			map(i) = true;
		}
	}

	if ( !include_neighbors ) return;

	const FArray3D_float & full_coord( pose.full_coord() );
	const FArray1D_int & res( pose.res() );
	float dis2;
	bool neighbors;
	for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
			 it != it_end; ++it ) {
		for( int i=it->start(); i<=it->stop(); ++i ) {
			for ( int j=1; j<=pose.total_residue(); ++j ) {
				if ( map(j) ) continue;
				are_they_neighbors(res(i), res(j), full_coord(1,1,i),
					full_coord(1,1,j), dis2, neighbors);
				if ( neighbors ) map(j) = true;
			}
		}
	}
	return;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_select_loop_residues_symm(
  const pose_ns::Pose & pose,
  const pose_ns::Loops & loops,
  bool const include_neighbors,
  FArray1D_bool & map // output
)
{
  using namespace pose_ns;

 Symmetry_info const & s( pose.symmetry_info() );

  assert( int(map.size1()) == pose.total_residue() );

  map = false;

  for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
       it != it_end; ++it ) {
    for( int i=it->start(); i<=it->stop(); ++i ) {
      if ( s.bb_independent(i) ) map(i) = true;
    }
  }

if ( !include_neighbors ) return;

  const FArray3D_float & full_coord( pose.full_coord() );
  const FArray1D_int & res( pose.res() );
  float dis2;
  bool neighbors;
  for( Loops::const_iterator it=loops.begin(), it_end=loops.end();
       it != it_end; ++it ) {
    for( int i=it->start(); i<=it->stop(); ++i ) {
      for ( int j=1; j<=pose.total_residue(); ++j ) {
        if ( map(j) ) continue;
        are_they_neighbors(res(i), res(j), full_coord(1,1,i),
          full_coord(1,1,j), dis2, neighbors);
        if ( neighbors && s.bb_independent(j) ) map(j) = true;
      }
    }
  }
  return;
}
///////////////////////////////////////////////////////////////////////////////
bool
pose_loops_init_from_file(
	pose_ns::Loops & loops
)
{

	bool ok = false;

	std::string filename;
	stringafteroption( "pose_loops_file", "none", filename );
	if( filename == "none" ) {
		filename = files_paths::start_path + files_paths::start_file
			+ ".pose_loops";
	}

	utility::io::izstream infile( filename.c_str() );
	if ( ! infile ) {
		std::cout << "pose_loops_init_from_file: can not open file " << filename
							<< std::endl;
		return ok;
	}

	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		if ( line[0] == '#' ) continue;
		std::istringstream line_stream( line );
		int begin, end, cut, extended_int;
		bool extended;
		int const offset = 0;
		line_stream >> begin >> end >> cut >> extended_int;
		extended = extended_int != 0 ? true: false;
		if ( cut < begin-1 || cut > end ) {
			cut = (begin-1) + int( ran3()*(end-(begin-1)+1) );
		}
		loops.add_loop( begin, end, cut, offset, extended );
	}

	ok = loops.num_loop() > 0;

	pose_loops_init_to_global_ns(loops);

	if ( ok ) std::cout << "loop(s) defined " << loops;

	return ok;
}
///////////////////////////////////////////////////////////////////////////////
bool
pose_loops_init_from_input(
	pose_ns::Loops & loops,
	std::vector<int> loop_begin,
	std::vector<int> loop_end,
	std::vector<int> cutpoints
)
{
	using namespace misc;

	assert( loops.num_loop() == 0 );
	int num_loop = static_cast<int>( loop_begin.size() );
	bool const 	start_from_extend( false );

	for( int i = 0; i < num_loop; ++i ) {
		int const begin = loop_begin[i];
		int const end = loop_end[i];
		int const cut = cutpoints[i];
		int const offset = 0;
		loops.add_loop( begin, end, cut, offset, start_from_extend );
	}

	return loops.num_loop() > 0;
}
///////////////////////////////////////////////////////////////////////////////
bool
pose_loops_init_from_global_ns(
	pose_ns::Loops & loops
)
{
	using namespace loops_ns;
	using namespace misc;

	assert( loops.num_loop() == 0 );
	bool const 	start_from_extend = truefalseoption("start_from_extend");
	// global loops_ns does not have cutpoint info
	// we just always cut at loop end
	// for extend_loop info, check the commandline option
	for( int i=1; i<=num_loop; ++i ) {
		int const begin = loop_begin(i);
		int const end = loop_end(i);
		int const cut = end;
		int const offset = 0;
		loops.add_loop( begin, end, cut, offset, start_from_extend );
	}

	return loops.num_loop() > 0;
}
///////////////////////////////////////////////////////////////////////////////
void
pose_loops_init_to_global_ns(
	pose_ns::Loops loops /*pass by value as it will be changed locally*/
)
{
	using namespace loops_ns;
	using namespace misc;
	using namespace pose_ns;

	if ( loops.num_loop() == 0 ) return;

	// put loops in sequential order
	loops.sequential_order();

	// reset loop global_ns
	num_loop = 0;
	loop_begin = 0;
	loop_end = 0;
	loop_map = 0;

	//copy correct values to them
	num_loop = loops.num_loop();
	int i = 0;
	for( pose_ns::Loops::const_iterator it = loops.begin(), it_end = loops.end();
			 it != it_end; ++it ) {
		++i;
		loop_begin(i) = it->start();
		loop_end(i) = it->stop();
		if( it->is_extended() ) {
			for ( int j = loop_begin(i); j <= loop_end(i); ++j ) {
				name(j) = "EXTD";
			}
		}
		for ( int j = loop_begin(i); j <= loop_end(i); ++j ) {
			loop_map(j) = i;
		}
	}
	assert( i == num_loop );

}


///////////////////////////////////////////////////////////////////////////////
void
pose_loops_reset_global_ns(
)
{
	using namespace loops_ns;

	// reset loop global_ns
	num_loop = 0;
	loop_begin = 0;
	loop_end = 0;
	loop_map = 0;

}


////////////////////////////////////////////////////////////////////////////////
void
set_loops_ns_fast(
	bool const & setting
)
{

	loops_ns::fast = setting;

}

////////////////////////////////////////////////////////////////////////////////
void
set_loops_ns_fixnatsc(
	bool const & setting
)
{

	loops_ns::fix_natsc = setting;

}
