// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14978 $
//  $Date: 2007-05-16 00:18:39 -0700 (Wed, 16 May 2007) $
//  $Author: sraman $

// Rosetta Headers
#include "pose_docking.h"
#include "after_opts.h"
#include "analyze_interface_ddg.h"
#include "dock_structure.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fragment_class.h"
#include "fragments_pose.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "grid_docking.h"
#include "initialize.h"
#include "interface.h"
#include "jumping_refold.h"
#include "loop_relax.h"
#include "loop_class.h"
#include "loops_ns.h"
#include "loops.h"
#include "make_pdb.h"
#include "monte_carlo.h"
#include "misc.h"
#include "minimize.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_idealize.h"
#include "pose_docking_flexible.h"
#include "pose_io.h"
#include "pose_loops.h"
#include "pose_param.h"
#include "pose_peptide_docking.h"
#include "pose_rotamer_trials.h"
#include "pose_vdw.h"
#include "ramachandran.h"
#include "random_numbers.h"
#include "recover.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "silent_input.h"
#include "smallmove.h"
#include "score.h"
#include "score_ns.h"
#include "start.h"
#include "status.h"
#include "timer.h"
#include "torsion_bbmove_trials.h"
#include "util_basic.h"
#include "util_vector.h"
#include "vdw.h"

// Numeric Headers
#include <numeric/xyzVector.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/vector1.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>


////////////////////
// SJF based on pose_docking_main
/////////
void
pose_docking_peptide_main(
bool & fail
)
{
  using namespace pose_ns;
  using namespace runlevel_ns;

  fail = false;
  set_pose_flag( true );

  Pose peptide_docking_pose;
  Loops loops; // not implemented

  //copy misc to pose
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( get_docking_fullatom_flag() ); //vats HACK remove this !

  pose_from_misc( peptide_docking_pose, fullatom, ideal_pos, coords_init );

  if ( docking::score_only ) {
    std::cout << "pose_peptide_docking score the model only " << std::endl;
  }

  if ( get_docking_silent_input_flag() ) {
    pose_docking_read_silent_input( peptide_docking_pose );
    if ( fail ) return;
  }

  // minimize parameters
  minimize_exclude_sstype( false, false );
  minimize_set_vary_phipsi( true );
  minimize_set_vary_omega( false );
  minimize_set_vary_chi( true );
  minimize_set_vary_rb_trans( false ); // no rb moves allowed for anchored peptide
  minimize_set_vary_rb_angle( false ); //
  minimize_set_local_min( false, 0 );
  pose_set_use_nblist( false ); // small moves cause errors with nblist use

  // docking fold_tree may have been setup in silent_input_file
  bool const dock_tree_init ( peptide_docking_pose.fold_tree().get_num_jump() != 0 );
  if( !dock_tree_init ) setup_peptide_foldtree( peptide_docking_pose );
  if( !docking::score_only ) {
    setup_peptide_bb_dofs( peptide_docking_pose );
    pose_docking_peptide_std_protocol( peptide_docking_pose );
  }


  score_set_default_function( fullatom );

  std::cout << "pose_peptide_docking_score: " << std::endl;
  peptide_docking_pose.score( scorefxn );
  peptide_docking_pose.show_scores(std::cout); std::cout << std::endl;

  pose_docking_calc_interf_energy( peptide_docking_pose, loops );

  // save this pose for silent_output if necessary
  silent_out_save_decoy_pose( peptide_docking_pose );

  peptide_docking_pose.copy_to_misc();

  set_pose_flag( false );
  // hack here to make sure rama score is calculated correctly
  for ( int i=1; i<= docking::n_monomers; ++i ) {
    misc::phi( docking::part_begin(i) ) = 0.0;
    misc::psi( docking::part_end(i) ) = 0.0;
    misc::omega( docking::part_end(i) ) = 0.0;
  }

  monte_carlo_reset();
}

/////
// SJF based on pose_docking_standard_protocol
// but no rigid body moves etc.
// loops not implemented
//////////
void
pose_docking_peptide_std_protocol(
pose_ns::Pose & pose
)
{
  using namespace pose_ns;
  using namespace docking;

//  int const nres = pose.total_residue();

  if ( pose.fullatom() ) {
    pose.set_allow_chi_move( true );
    if ( norepack1 ) {
      for ( int i = part_begin(1); i<= part_end(1); ++i ) {
        pose.set_allow_chi_move(i,false);
      }
    }
    if ( norepack2 ) {
      for ( int i = part_begin(2); i<= part_end(2); ++i ) {
        pose.set_allow_chi_move(i,false);
      }
    }
  }
  if ( get_docking_prepack_mode() ) {
    pose_docking_prepack_protocol( pose );
    return;
  }

  Score_weight_map weight_map;

  Pose pose_save;
  pose_save = pose;

  pose.set_fullatom_flag( false );

  int const cycles( 1000 );
  peptide_move_trial( pose, cycles );
  pose_docking_calc_rmsd( pose );
  docking_store_cenmode_rms();

  if ( get_docking_fullatom_flag() ) {
    // fullatom mode
    std::string min_type = "dfpmin";
    pose.set_fullatom_flag( true, false );
    // copy back the starting sidechain first
    pose.recover_sidechain( pose_save );
    setup_score_weight_map( weight_map, score10d );
    const float temperature( 0.8 );
    Monte_carlo mc( pose, weight_map, temperature );

    // repack interface sidechains
    bool const interface_only(true), include_current(true);
    pose_docking_repack( pose, interface_only, include_current );
    pose.main_minimize( weight_map, min_type );

    // dock_rtmin if requested
    if ( docking::dock_rtmin ) {
      set_docking_interface_pack( interface_only );
      score_set_try_rotamers( true );
      score_set_minimize_rot( true );
      pose.score( weight_map );
      score_set_minimize_rot( false );
      score_set_try_rotamers( false );
    }
    // this is to prevent accidental score increase after repack+rtmin
    mc.boltzmann( pose );
    pose = mc.low_pose();
  }
}


//////////////////////////
// SJF setting up the fold tree for the docked peptide
// The fold tree would allow the peptide's nterm residues to be treated as a tail: held fixed on cterm side but free on nterm
////////////////
void
setup_peptide_foldtree(
pose_ns::Pose & pose
)
{
  using numeric::xyzVector_float;

  int p1_start, p1_end, p2_start, p2_end;
  int anchor_peptide = intafteroption( "anchor_peptide" ); // the cterm residue fromwhich the peptide is held fixed
  float min_dis2( 10000.0 );
  int const nres( pose.total_residue() );

  assert( misc_in_sync( pose ));
  get_complex_boundaries( p1_start, p1_end, p2_start, p2_end );
  assert( nres == p2_end );
  assert( anchor_peptide <= nres && anchor_peptide >= p2_start );

  int jump_pos1( 1 ), jump_pos2( anchor_peptide );

//set the jump position on protein1 to be the closest point to the anchor point on
//the peptide
  xyzVector_float anchor_ca( &pose.Eposition()(1,2, anchor_peptide ));
  for( int i=p1_start; i<=p1_end; ++i ) {
    xyzVector_float i_ca( &pose.Eposition()(1,2,i) );
    float dis2( distance_squared( i_ca, anchor_ca ) );
    if( dis2 < min_dis2 ) {
        min_dis2 = dis2;
        jump_pos1 = i;
    }
  }
  std::cout << "Anchoring jump at residues: " << jump_pos1 << " and " <<
    jump_pos2 << " CA_distance= " << std::sqrt( min_dis2 ) << std::endl;

  pose.one_jump_tree( nres, jump_pos1, jump_pos2, p1_end );
  pose.set_allow_jump_move( false ); // no rb moves allowed for anchored peptides
}

//////////////////////////////
// SJF setup the backbone dofs: allow moves for peptide nterm residues; false for everything else
////////////
void
setup_peptide_bb_dofs(
pose_ns::Pose & pose
)
{
  int p1_start, p1_end, p2_start, p2_end;

  assert( misc_in_sync( pose ));
  int nres( pose.total_residue() );
  int anchor_peptide = intafteroption( "anchor_peptide" );
  get_complex_boundaries( p1_start, p1_end, p2_start, p2_end );
  assert( nres == p2_end );
  assert( anchor_peptide <= nres && anchor_peptide >= p2_start );

  pose.set_allow_bb_move( false );
  for( int i=p2_start; i<=anchor_peptide; ++i ) {
    if( i>=p2_start && i<=anchor_peptide ){
      pose.set_allow_bb_move( i, true );
      pose.set_secstruct(i, 'L');
    }
  }
}

//////////////////////////////////////////////////////////////////////////////
// SJF mc trials over small moves to the peptide residues
// Changes the peptide purturbation size adaptively to achieve ~50% mc acceptance rate, but
// if -docking_local_refine is on, uses very small, non-adaptive small moves
/////
float
peptide_move_trial(
  pose_ns::Pose & pose,
  int const cycles
)
{
  using namespace pose_ns;
  using namespace docking;

  float const temperature( 0.8 );
  int const nmoves = static_cast<int> ( pose.get_total_insert() * 0.5 );

  bool const prev_fa_flag = get_fullatom_flag();
  pose.set_fullatom_flag( false );


  float peptide_pert_size;
  realafteroption( "peptide_pert", ( docking_local_refine ? 0.5 : 3.0 ), peptide_pert_size );

  Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score4d );

  Monte_carlo mc( pose, weight_map, temperature );
  int n_accepted = 0;

  for( int i = 1; i <= cycles; ++i ) {
    set_smallmove_size( 0, 20, peptide_pert_size );
    pose_small_moves( pose, nmoves );
    pose.score( weight_map );
    if( mc.boltzmann( pose ) ) ++n_accepted;

    if( !docking_local_refine ) {
// adaptive changes of the peptide_pert_size to achieve ~50% acceptance rate
      float accpt_rate = (float) n_accepted / (float) i;
//std::cout << n_accepted<< " " << i << " " << accpt_rate << " "<<peptide_pert_size<<std::endl;
      if ( accpt_rate <= 0.5 ) peptide_pert_size *= 0.90;
      else peptide_pert_size *= 1.10;
    }
  }
  pose = mc.low_pose();

  std::cout << "pose_peptide_move_trial -- "
            << n_accepted << " out of " << cycles << " accepted\n";

  std::cout << "last peptide_pert_size used: " << peptide_pert_size << std::endl;

  pose.set_fullatom_flag( prev_fa_flag );

  return n_accepted / cycles;
}

